#
# File: Psad.pm
#
# Purpose: This is the Psad.pm perl module.  It contains functions
#          that are reused by the various psad daemons.
#
# Author: Michael B. Rash (mbr@cipherdyne.com)
#
# Version: 1.0.0-pre1
#
##################################################################
#
# $Id: Psad.pm,v 1.7 2002/09/24 02:06:26 mbr Exp $

package Psad;

use Exporter;
use Text::Wrap;
use Carp;
use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;

@ISA = qw(Exporter);
@EXPORT = qw(buildconf validate_config writepid writecmdline unique_pid
             warn_handler die_handler logr check_commands check_firewall_rules);

$VERSION = '1.0.0-pre1';

### define the main psad directory
my $PSAD_DIR = '/var/log/psad';

### used by logr()
my $SUB_TAB = '    ';

### subroutines ###
sub buildconf() {
    my $configfile = shift;
    my %Config;
    my %Cmds;
    open C, "< $configfile" or croak "  ... @@@ Could not open config file $configfile: $!";
    my @clines = <C>;
    close C;
    for my $cline (@clines) {
        chomp $cline;
        next if ($cline =~ /^\s*#/);
        if ($cline =~ /^(\S+)\s+(.*)\;/) {
            my $varname = $1;
            my $value = $2;
            if ($varname =~ /\w+Cmd$/ && $value =~ m|.*/(.*)|) { ### found a command
                $Cmds{$1} = $value;
            } elsif ($value =~ /\((.*)\)/) {  ### found an array
                my $arr = $1;
                if ($arr =~ /,/) {
                    $Config{$varname} = [split /,\s+/, $arr];
                } else {
                    $Config{$varname} = [split /\s+/, $arr];
                }
            } else {
                $Config{$varname} = $value;
            }
        }
    }
    return \%Config, \%Cmds;
}
### check to make sure all required varables are defined in the config
### this subroutine is passed different variables by each script that
### correspond to only those variables needed be each script).
sub validate_config() {
    my ($psad_conf_file, $varnames_aref, $config_href) = @_;
    for my $var (@$varnames_aref) {
        unless (defined $config_href->{$var}) {  ### the config file did not contain this var
            croak " ... @@@ The config file \"$psad_conf_file\" does not contain the variable\n" .
                "\"$var\".  Exiting!";
        }
    }
    return;
}
### check paths to commands and attempt to correct if any are wrong.
sub check_commands() {
    my $Cmds_href = shift;
    my $caller = $0;
    my @path = qw(/bin /sbin /usr/bin /usr/sbin /usr/local/bin /usr/local/sbin);
    CMD: for my $cmd (keys %$Cmds_href) {
        my $cmd_name = ($Cmds_href->{$cmd} =~ m|.*/(.*)|);
        unless (-x $Cmds_href->{$cmd}) {
            my $found = 0;
            PATH: for my $dir (@path) {
                if (-x "${dir}/${cmd}") {
                    $Cmds_href->{$cmd} = "${dir}/${cmd}";
                    $found = 1;
                    last PATH;
                }
            }
            unless ($found) {
                next CMD if ($cmd eq 'ipchains' || $cmd eq 'iptables');
                croak "\n ... @@@  ($caller): Could not find $cmd anywhere!!!  Please" .
                        " edit the config section to include the path to $cmd.\n";
            }
        }
        unless (-x $Cmds_href->{$cmd}) {
            croak "\n ... @@@  ($caller):  $cmd_name is located at $Cmds_href->{$cmd}" .
                                            " but is not executable by uid: $<\n";
        }
    }
    return;
}
sub check_firewall_rules() {
    my ($fw_msg_search, $emails_aref, $logr_files_aref, $Cmds_href) = @_;
    my @localips;
    ### see which of the two firewall commands we are able
    ### to execute
    my $ipchains_rv = system "$Cmds_href->{'ipchains'} -nL > /dev/null 2>&1";
    ### return value from system call is a 16 bit number,
    ### the high byte is the real return value
    $ipchains_rv >>= 8;
    my $iptables_rv = system "$Cmds_href->{'iptables'} -nL > /dev/null 2>&1";
    $iptables_rv >>= 8;

    my @localips_tmp = `$Cmds_href->{'ifconfig'} -a |$Cmds_href->{'grep'} inet |$Cmds_href->{'grep'} -v grep`;
    push @localips, (split /:/, (split /\s+/, $_)[2])[1] for (@localips_tmp);

    ### prefer iptables so check it first
    if (! $iptables_rv && $ipchains_rv) {  ### iptables works if the return value is 0
        if (&check_iptables_rules($fw_msg_search, \@localips, $logr_files_aref, $Cmds_href)) {
            return 1;
        } else {
            &send_fw_warning($emails_aref, $logr_files_aref->[$#$logr_files_aref], $Cmds_href);
            return 0;
        }
    } elsif (! $ipchains_rv && $iptables_rv) {
        if (&check_ipchains_rules(\@localips, $logr_files_aref, $Cmds_href)) {
            return 1;
        } else {
            &send_fw_warning($emails_aref, $logr_files_aref->[$#$logr_files_aref], $Cmds_href);
            return 0;
        }
    } elsif ($iptables_rv && $ipchains_rv) {
        print " ... @@@  could not run either firewall!\n";
    }
    return;
}
sub check_iptables_rules() {
    my ($fw_msg_search, $localips_aref, $logr_files_aref, $Cmds_href) = @_;
    # target     prot opt source               destination
    # LOG        tcp  --  anywhere             anywhere           tcp flags:SYN,RST,ACK/SYN LOG level warning prefix `DENY '
    # DROP       tcp  --  anywhere             anywhere           tcp flags:SYN,RST,ACK/SYN

    # ACCEPT     tcp  --  0.0.0.0/0            64.44.21.15        tcp dpt:80 flags:0x0216/0x022
    # LOG        tcp  --  0.0.0.0/0            0.0.0.0/0          tcp flags:0x0216/0x022 LOG flags 0 level 4 prefix `DENY '
    # DROP       tcp  --  0.0.0.0/0            0.0.0.0/0          tcp flags:0x0216/0x022
    my @rules = `$Cmds_href->{'iptables'} -nL`;
    my $drop_rule = 0;
    my $drop_tcp = 0;
    my $drop_udp = 0;
    FWPARSE: for my $rule (@rules) {
        next FWPARSE if ($rule =~ /^Chain/ || $rule =~ /^target/);
        if ($rule =~ /^(LOG)\s+(\w+)\s+\S+\s+\S+\s+(\S+)\s.+prefix\s\`(.+)\'/) {
            my ($target, $proto, $dst, $prefix) = ($1, $2, $3, $4);
            if ($target eq 'LOG' && $proto =~ /all/ && $prefix =~ /$fw_msg_search/) {
                # this needs work... see above _two_ rules.
                if (&check_destination($dst, $localips_aref)) {
                    &logr(" ... Your firewall setup looks good.  Unauthorized tcp and/or".
                                      " udp packets will be logged.\n", $logr_files_aref);
                    return 1;
                }
            } elsif ($target eq 'LOG' && $proto =~ /tcp/ && $prefix =~ /$fw_msg_search/) {
                $drop_tcp = 1 if (&check_destination($dst, $localips_aref));
            } elsif ($target eq 'LOG' && $proto =~ /udp/ && $prefix =~ /$fw_msg_search/) {
                $drop_udp = 1 if (&check_destination($dst, $localips_aref));
            }
        }
    }
    if ($drop_tcp && $drop_udp) {
        &logr(" ... Your firewall setup looks good.  Unauthorized tcp".
               " and/or udp packets will be logged.\n", $logr_files_aref);
        return 1;
    } elsif ($drop_tcp) {
        &logr(" ... Your firewall will log unauthorized tcp packets, but not all udp packets. "
            . "Hence psad will be able to detect tcp scans, but not udp ones.\n", $logr_files_aref);
        &logr("\n", $logr_files_aref);
        &logr("       SUGGESTION: After making sure you accept any udp traffic that you need to"   .
              " (such as udp/53 for nameservice) add a rule to log and drop all other udp traffic" .
              " with the following two commands:\n", $logr_files_aref);
        &logr("          # iptables -A INPUT -p udp -j LOG --log-prefix \"DENY \"\n", $logr_files_aref);
        &logr("          # iptables -A INPUT -p udp -j DROP\n", $logr_files_aref);
        return 0;
    } elsif ($drop_tcp) {
        &logr(" ... Your firewall will log unauthorized udp packets, but not all tcp packets." .
              "Hence psad will be able to detect udp scans, but not tcp ones.\n", $logr_files_aref);
        &logr("\n", $logr_files_aref);
        &logr("       SUGGESTION: After making sure you accept any tcp traffic that you need to" .
              " (such as tcp/80 etc.) add a rule to log and drop all other tcp traffic with the" .
              " following two commands:\n", $logr_files_aref);
        &logr("          # iptables -A INPUT -p tcp -j LOG --log-prefix \"DENY \"\n", $logr_files_aref);
        &logr("          # iptables -A INPUT -p tcp -j DROP\n", $logr_files_aref);
        return 0;
    }
    &logr(" ... Your firewall does not include rules that will log dropped/rejected packets."    .
          "  You need to include a default rule that logs packets that have not been accepted"   .
          " by previous rules, and this rule should have a logging prefix of \"drop\", \"deny\"" .
          " or \"reject\".\n", $logr_files_aref);
    &logr("\n", $logr_files_aref);
    &logr("       FOR EXAMPLE:   Suppose that you are running a webserver to which you"   .
          " also need ssh access.  Then an iptables ruleset that is compatible with psad" .
          " could be built with the following commands:\n", $logr_files_aref);
    &logr("\n", $logr_files_aref);
    &logr("              iptables -A INPUT -s 0/0 -d <webserver_ip> 80 -j ACCEPT\n", $logr_files_aref);
    &logr("              iptables -A INPUT -s 0/0 -d <webserver_ip> 22 -j ACCEPT\n", $logr_files_aref);
    &logr("              iptables -A INPUT -j LOG --log-prefix \" DROP\"\n", $logr_files_aref);
    &logr("              iptables -A INPUT -j DENY\n", $logr_files_aref);
    &logr("\n", $logr_files_aref);
    &logr(" ... @@@  Psad will not run without an iptables ruleset that includes rules similar to the" .
          " last two rules above.\n", $logr_files_aref);
    return 0;
}
sub check_ipchains_rules() {
    my ($localips_aref, $logr_files_aref, $Cmds_href) = @_;
    #  target     prot opt     source                destination           ports
    # DENY       tcp  ----l-  anywhere             anywhere              any ->   telnet

    #Chain input (policy ACCEPT):
    #target     prot opt     source                destination           ports
    #ACCEPT     tcp  ------  0.0.0.0/0            0.0.0.0/0             * ->   22
    #DENY       tcp  ----l-  0.0.0.0/0            0.0.0.0/0             * ->   *
    my @rules = `$Cmds_href->{'ipchains'} -nL`;
    FWPARSE: for my $rule (@rules) {
        chomp $rule;
        next FWPARSE if ($rule =~ /^Chain/ || $rule =~ /^target/);
        if ($rule =~ /^(\w+)\s+(\w+)\s+(\S+)\s+\S+\s+(\S+)\s+(\*)\s+\-\>\s+(\*)/) {
            my ($target, $proto, $opt, $dst, $srcpt, $dstpt) = ($1, $2, $3, $4, $5, $6);
            if ($target =~ /drop|reject|deny/i && $proto =~ /all|tcp/ && $opt =~ /....l./) {
                if (&check_destination($dst, $localips_aref)) {
                    &logr(" ... Your firewall setup looks good.  Unauthorized tcp" .
                          " packets will be dropped and logged.\n", $logr_files_aref);
                    return 1;
                }
            }
        } elsif ($rule =~ /^(\w+)\s+(\w+)\s+(\S+)\s+\S+\s+(\S+)\s+(n\/a)/) {  # kernel 2.2.14 (and others) show "n/a" instead of "*"
            my ($target, $proto, $opt, $dst, $ports) = ($1, $2, $3, $4, $5);
            if ($target =~ /drop|reject|deny/i && $proto =~ /all|tcp/ && $opt =~ /....l./) {
                if (&check_destination($dst, $localips_aref)) {
                    &logr(" ... Your firewall setup looks good.  Unauthorized tcp" .
                          " packets will be dropped and logged.\n", $logr_files_aref);
                    return 1;
                }
            }
        }
    }
    &logr(" ... Your firewall does not include rules that will log dropped/rejected" .
          " packets.  Psad will not work with such a firewall setup.\n", $logr_files_aref);
    return 0;
}
sub check_destination() {
    my ($dst, $localips_aref) = @_;
    return 1 if ($dst =~ /0\.0\.0\.0\/0/);
    for my $ip (@$localips_aref) {
        my ($oct) = ($ip =~ /^(\d{1,3}\.\d{1,3})/);
        return 1 if ($dst =~ /^$oct/);
    }
    return 0;
}
sub send_fw_warning() {
    my ($emails_aref, $file, $Cmds_href) = @_;
    for my $e (@$emails_aref) {
        system "$Cmds_href->{'mail'} -s \"psad: firewall setup warning!\" $e < $file";
    }
    unlink $file;
    return;
}
### make sure pid is unique
sub unique_pid() {
    my $pidfile = shift;
    if (-e $pidfile) {
        my $caller = $0;
        open PIDFILE, "< $pidfile";
        my $pid = <PIDFILE>;
        close PIDFILE;
        chomp $pid;
        if (kill 0, $pid) {  # psad is already running
            croak "\n ... @@@  $caller (pid: $pid) is already running!  Exiting.\n\n";
        }
    }
    return;
}
### write the pid to the pid file
sub writepid() {
    my $pidfile = shift;
    my $caller = $0;
    open PIDFILE, "> $pidfile" or croak " ... @@@  $caller: Could not open pidfile $pidfile: $!\n";
    print PIDFILE "$$\n";
    close PIDFILE;
    chmod 0600, $pidfile;
    return;
}
### write command line to cmd file
sub writecmdline() {
    my ($args_aref, $cmdline_file) = @_;
    open CMD, "> $cmdline_file";
    print CMD "@$args_aref\n";
    close CMD;
    chmod 0600, $cmdline_file;
    return;
}
### write all die messages to a logfile
sub die_handler() {
    my $msg = shift;
    my $caller = $0;
    $caller =~ s|^.*/||;
    my @time = split /\s+/, scalar localtime;
    pop @time; shift @time;
    my $time = join ' ', @time;
    open D, ">> ${PSAD_DIR}/${caller}.die";
    print D "$time $caller (pid: $$): $msg";
    close D;
    die $!;
}
### write all warnings to a logfile
sub warn_handler() {
    my $msg = shift;
    my $caller = $0;
    $caller =~ s|^.*/||;
    my @time = split /\s+/, scalar localtime;
    pop @time; shift @time;
    my $time = join ' ', @time;
    open W, ">> ${PSAD_DIR}/${caller}.warn";
    print W "$time $caller (pid: $$): $msg";
    close W;
    return;
}
### logging subroutine that handles multiple filehandles
sub logr() {
    my ($msg, $files_aref) = @_;
    for my $f (@$files_aref) {
        if ($f eq *STDOUT) {
            if (length($msg) > 70) {
                print STDOUT wrap('', $SUB_TAB, $msg);
            } else {
                print STDOUT $msg;
            }
        } elsif ($f eq *STDERR) {
            if (length($msg) > 70) {
                print STDERR wrap('', $SUB_TAB, $msg);
            } else {
                print STDERR $msg;
            }
        } else {
            open F, ">> $f";
            if (length($msg) > 70) {
                print F wrap('', $SUB_TAB, $msg);
            } else {
                print F $msg;
            }
            close F;
        }
    }
    return;
}

1;
__END__

=head1 NAME

Psad.pm - Perl extension for the psad (Port Scan Attack Detector) daemons

=head1 SYNOPSIS

  use Psad;
  writepid()
  writecmdline()
  unique_pid()
  logr()
  check_commands()
  check_firewall_rules()
  $SIG{'__WARN__'} = \&Psad::warn_handler;
  $SIG{'__DIE__'} = \&Psad::die_handler;

=head1 DESCRIPTION

The Psad.pm module contains several subroutines that are used by Port Scan
Attack Detector (psad) daemons.
writepid()  writes process ids to the pid files (e.g. "/var/run/psad.pid").
writecmdline()  writes the psad command line contained within @ARGV[] to the
file "/var/run/psad.cmd".
unique_pid()  makes sure that no other daemon process is already running in
order to guarantee pid uniqueness.
logr()  a logging subroutine that handles printing to multiple file handles.
check_commands()  check paths to commands and warns if unable to find any 
particular command.
check_firewall_rules()  checks the ipchains/iptables firewall ruleset to see 
if the ruleset is compatible with psad (i.e., if it has been configured to
log and drop all packets that are not explicitly allowed through).  The two
signal handlers warn_handler and die_handler write warnings and die messages
to logfiles in /var/log/psad.

=head1 AUTHOR

Michael B. Rash, mbr@cipherdyne.com

=head1 SEE ALSO

psad(8).

