<?php

/******************************
 
Class: Properties
Hierarchical property storage designed for fast caching of properties.
Properties is a singleton (there is only ever one instance).

There are basically just three functions: get, set, commit. The Properties
object is not in charge of actually storing this values, it simply caches them.
It is the PageStorage object's job to store the values.

Every time you need a page property, you should ask the Properties object.

For example:

  $title = $properties->get('title',$page);
  
or, for convenience:

  $title = $page->get('title');
  
Which will do the same thing.
  
If the Properties cache is not older than the file storage, it returns the
cached value. Otherwise, it asks the PageStore for the value, caches it, then
returns the value.

note: the properties 'content' and 'name' are handled by the pagestore and not
by the properties object at all.

******************************/

require_once('Page.php');

class Properties {

var $data = array(
	'default' => array(
		'pagebackend' => 'folder',
#		'userbackend' => 'sqlite, .bamboo/b.users.sqlite',
		'userbackend' => 'none',
		'indexbackend' => 'sqlite, .bamboo/b.index.sqlite',
		'cachedir' => '/tmp/bamboocache',
		'sitename' => 'Untitled Site',
		'siteroot' => '',
		'decorator' => 'default',
		'liburl' => '/bamboo',
		'access-view' => 'anonymous',
		'access-edit' => 'anonymous',
		'title-prefix' => '',
		'static' => true,
		'mtime' => '1',
		'locked' => false,
		'title' => '$name',
		'nav-title' => '$title',
		'type' => 'txt',
		'dir' => false,
		'index' => true,
		'visible' => true,
		'allowed-types' => 'txt',
		'default-lang' => 'en',
		'languages' => 'en',
	)
);

# props which are always global
var $globals = array(
	'cachedir','sitename','siteroot','pagestore','default-lang'
);

# props which cannot be inherited
var $basic = array(
	'name','title','subject','keywords','description','static','dir','mtime','locked','type','visible'
);

# props which might have translations
var $multilingual = array(
	'title', 'subject', 'description', 'keywords',
);

# 
var $proptypes = array(
	'dir' => 'boolean'
);


function Properties($site_config_file) {	
	if (!is_file($site_config_file))
		die("The site configuration file '$site_config_file' does not exist.");
	if (!is_readable($site_config_file))
		die("The site configuration file '$site_config_file' could not be read.");

	# so many roots:
	#   sitedocroot == the filesystem path to the root of the site
	#   docroot     == apache's configuration for DocumentRoot.
	#   siteroot    == the url of the root of this site.
	
	$sitedocroot = dirname($site_config_file);
	$docroot = preg_replace('#/$#','',$_SERVER['DOCUMENT_ROOT']); // trim trailing slash
	if ($sitedocroot == $docroot)
		$siteroot = '';
	else
		$siteroot = preg_replace("'^$docroot'",'',$sitedocroot);
		
	$this->data['site'] = parse_ini_file($site_config_file);
	$site = & $this->data['site'];
	if (!isset($site['siteroot']))
		$site['siteroot'] = $siteroot;
	elseif ($site['siteroot'] == '/')
		$site['siteroot'] = '';
		
	if (!isset($site['sitedocroot']))
		$site['sitedocroot'] = $sitedocroot;
	
	// because i am lazy and don't want to write them out 'prop' => 'true'
	// should be removed in the future.
	$this->globals = array_flip($this->globals);
	$this->basic = array_flip($this->basic);
	$this->multilingual = array_flip($this->multilingual);
}

function &get($prop, &$page, $recursive_count=0) {
	$value = null;
	$ps = &$page->ps;
	$siteroot = $this->getGlobal('siteroot');
	
	if ($this->is_global($prop)) {
		$value = $this->getGlobal($prop);
	}
	
	## handle basic

	$id = $page->path;
	// if memory cache empty, load data
	if ( !isset( $this->data[$id]['basic'] ) ) {
		$this->data[$id]['basic'] = $ps->loadbasic($siteroot,$page->path);
	}
	if ($this->is_multilingual($prop)) {
		if ($value == null) {
			$prop_dot_lang = find_multilingual_prop($prop, $this->data[$id]['basic']);
			if (isset($this->data[$id]['basic'][$prop_dot_lang]))
				$value = &$this->data[$id]['basic'][$prop_dot_lang];
		}
	}
	if ($value == null) {
		if (isset($this->data[$id]['basic'][$prop]))
			$value = &$this->data[$id]['basic'][$prop];
	}
	
	## handle inheritable

	if ($value == null) {
		$id = '';
		$parentid = '';
		$patharray = split('/',$page->path);
		foreach($patharray as $pathelement) {
			if ($pathelement == '')
				$id = '';
			else
				$id = "$id/$pathelement";
			if ( !isset( $this->data[$id]['inherit'] ) ) {
				if ( !isset($this->data[$parentid]['inherit']) ) {
					$this->data[$id]['inherit'] = $ps->loadinherit($siteroot,$id);
				}
				else {
					// with array_merge, the second arg overwrites the first
					// if the key is the same. hence, inheritence.
					$this->data[$id]['inherit'] = array_merge(
						$this->data[$parentid]['inherit'],
						$ps->loadinherit($siteroot,$id)
					);
				}
			}
			$parentid = $id;
		}
		if (isset($this->data[$id]['inherit'][$prop]))
			$value = &$this->data[$id]['inherit'][$prop];
	}
	
	if ($value == null)
		$value = $this->getGlobal($prop); // last ditch effort.
	
	if ($value != null && $value{0} == '$' && $recursive_count < 4) {
		return $this->get(substr($value,1),$page,$recursive_count+1);
	}
	return $value;
}

/**
 * returns all properties which exist for a particular page.
 * returns (for example): 
 * array(
 *   'visible' => array('value'=>'true','inherit'=>true,'type'=>'boolean')
 *   ...
 * ); 
 **/
function getAll(&$page) {
	$id = $page->path;
	$ps = &$page->ps;
	$siteroot = $this->getGlobal('siteroot');
	$ret = array();

	if ( !isset( $this->data[$id]['basic'] ) ) {
		$this->data[$id]['basic'] = $ps->loadbasic($siteroot,$page->path);
	}
	foreach($this->data[$id]['basic'] as $prop => $value) {
		$ret[$prop] = array('value'=>$value,'inherit'=>false);
	}
	
	$id = '';
	$parentid = '';
	$patharray = split('/',$page->path);
	foreach($patharray as $pathelement) {
		if ($pathelement == '')
			$id = '';
		else
			$id = "$id/$pathelement";
		if ( !isset( $this->data[$id]['inherit'] ) ) {
			if ( !isset($this->data[$parentid]['inherit']) ) {
				$this->data[$id]['inherit'] = $ps->loadinherit($siteroot,$id);
			}
			else {
				// with array_merge, the second arg overwrites the first
				// if the key is the same. hence, inheritence.
				$this->data[$id]['inherit'] = array_merge(
					$this->data[$parentid]['inherit'],
					$ps->loadinherit($siteroot,$id)
				);
			}
		}
		$parentid = $id;
	}

	if (is_array($this->data[$id]['inherit'])) {
		foreach($this->data[$id]['inherit'] as $prop => $value) {
			$ret[$prop] = array('value'=>$value,'inherit'=>true);
		}
	}

	return $ret;
}

function set($propname, $data, &$page) {
	$id = $page->path;
	if ($this->is_multilingual($propname))
		$propnamepluslang = $propname . '.' . $_SESSION['lang'];
	else
		$propnamepluslang = $propname;
	
	if ($data == 'true')
		$data = true;
	elseif ($data == 'false')
		$data = false;

	if ($this->is_basic($propname)) {
		$this->data[$id]['basic'][$propnamepluslang] = $data;
	}
	else {
		$this->data[$id]['inherit'][$propnamepluslang] = $data;
	}
}

function commit(&$page) {
	$id = $page->path;
	$ar = array();
	$siteroot = $this->getGlobal('siteroot');
	$basicprops = array_keys($this->basic);
	$basicprops = array_merge($basicprops, array_keys($this->data[$id]['basic']));
	foreach($basicprops as $prop) {
		if ($this->is_multilingual($prop))
			$prop = $prop . '.' . $_SESSION['lang'];
		if (isset($this->data[$id]['basic'][$prop]))
			$ar[$prop] = $this->data[$id]['basic'][$prop];
	}
	
	$err = $page->ps->savebasic($siteroot,$page->path,$ar);

	## TODO: save b.inherit too ##
#	if ($err == '')
#		$err = $page->ps->saveinherit($siteroot,$page->path,$ar);
	return $err;
}

/*
 * returns the value of a global property
 * (the same for all pages)
 */
function getGlobal($propname) {
	$value = isset($this->data['default'][$propname]) ? $this->data['default'][$propname] : null;
	$value = isset($this->data['site'][$propname]) ? $this->data['site'][$propname] : $value;
	return $value;
}

function is_global($prop) {
	return array_key_exists($prop,$this->globals);
}

function is_basic($prop) {
	return array_key_exists($prop,$this->basic);
}

function is_multilingual($prop) {
	return array_key_exists($prop,$this->multilingual);
}

function proptype($prop) {
	if (isset($this->proptypes[$prop]))
		return $this->proptypes[$prop];
	else
		return '';
}

} // end class


/* 
 * tries to guess the filename of the site configuration file
 */
function get_site_config() {
	$docroot = preg_replace('#/$#','',$_SERVER['DOCUMENT_ROOT']); // trim trailing slash
	
	# note: i think that script_url and path_translated will always return
	# bamboo in apache2, and are therefor useless. apache1 is different.
	
	if (isset($_SERVER['REQUEST_URI']))
		$path = $docroot . $_SERVER['REQUEST_URI'];	
	elseif (isset($_SERVER['REDIRECT_URL']))
		$path = $docroot . $_SERVER['REDIRECT_URL'];
	elseif (isset($_SERVER['SCRIPT_URL']))
		$path = $docroot . $_SERVER['SCRIPT_URL'];
	else
		$path = $_SERVER['PATH_TRANSLATED']; // comes with docroot (eg /var/www)
	
	# get only the top level of the path
	# (strip off docroot and subfolders)
	$path = preg_replace("#^$docroot(/.*?)(/.*)?$#", '$1', $path);
	
	if (is_file("$docroot/b.site"))
		return "$docroot/b.site";
	else
		return "$docroot{$path}/b.site";
}

function find_multilingual_prop($prop, $properties) {
	$languages = Lang::priority();
	$lang_len = count( $languages );
	$i = 0;
	$prop_dot_lang = $prop . '.' . $languages[$i];
	while( !isset( $properties[$prop_dot_lang] ) && ( $i < $lang_len ) ) {
		$prop_dot_lang = $prop . '.' . $languages[ $i++ ];
	}
	return $prop_dot_lang;
}

return;
?>
