/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __CDWRITER_H__
#define __CDWRITER_H__

#include "arson.h"

#include <qvariant.h>
#include <qdir.h>

#include "progressdlg.h"
#include "konfig.h"

/*==================================*/
/*	DEFINES
 *==================================*/

#define optSrcdev		"srcdev"
#define optDrv			"drv"
#define optSrcDrv		"srcdrv"
#define optSpeed		"speed"
#define optCdLen		"cdlen"
#define optNice			"nice"
#define optXtraPath		"xpath"
#define optBurnProof	"burnpf"
#define optEject		"eject"
#define optByteSwap		"bytswp"
#define optRipFmt		"ripfmt"
#define optRipDir		"ripdir"
#define optRipDev		"ripdev"
#define optLookup		"ripcdi"
#define optEmail		"email"
#define optSeq			"seq"
#define optIsoLabel		"isolabel"
#define optOverBurn		"overburn"
#define optNoFix		"nofix"
#define optOnTheFly  "onthefly"
#define optDAOCDR    "cdrecorddao"
#define optAudMaster "audiomaster"
#define optCdText    "cdtext"
#define optDrvFlags  "drvflags"
#define optNormal    "normalize"

/*========================================================*/

class ArsonProcessOpts : public ArsonConfig
{
	typedef QMap<QCString,QVariant> PROCESSOPTS;
	
public:
	ArsonProcessOpts (void);

	void addBool (const char *name, bool val);
	void addString (const char *name, const QString &str);
	void addLong (const char *name, long val);

	bool getBool (const char *name, bool deft = false) const;
	QString getString (const char *name, const QString &deft = QString::null) const;
	long getLong (const char *name, long deft = 0) const;

	bool hasOpt (const char *name) const;

private:
	PROCESSOPTS m_opts;
};

/*========================================================*/

class ArsonProcessOutput;
class ArsonProcessMgr;

class ArsonProcessUI
{
public:
	ArsonProcessUI (void);
	virtual ~ArsonProcessUI (void) { }

	//	These all should almost definately be overridden
	virtual void setMaxProgress (long maxp) { m_nMaxProgress = maxp; }
	virtual void setProgress (long prog) { m_nProgress = prog; }
	virtual ArsonProcessOutput *getOutput (void) { return NULL; }
	virtual void output (const QString &str, bool error) = 0;
	virtual void setText (const QString &text) = 0;
	virtual void begin (void) = 0;
	virtual void end (void) = 0;

	//	Utility functions
	void setWriter (ArsonProcessMgr *pWriter) { m_pWriter = pWriter; }
	long maxProgress (void) const { return m_nMaxProgress; }
	long progress (void) const { return m_nProgress; }

	void stepUp (uint distance) { setProgress(m_nProgress + distance); }

	void setProgress (long prog, long maxp) {
		setMaxProgress(maxp);
		setProgress(prog);
	}

protected:
	ArsonProcessMgr *m_pWriter;
	long m_nMaxProgress,
		m_nProgress;
};

/*========================================================*/

class ArsonProcess;

class ArsonProcessMgr
{
public:
	ArsonProcessMgr (ArsonProcessUI *pUI);
	virtual ~ArsonProcessMgr (void);

	void setProcess (ArsonProcess *pProcess);

	virtual void taskComplete (ArsonProcess *ptr);
	virtual void jobComplete (void);
	virtual void begin (const ArsonProcessOpts &opts);
	virtual void abort (void);

	bool isWriting (void) const { return (m_pProcess != NULL); }
	ArsonProcessUI *ui (void) { return m_pUI; }
	ArsonProcess *process (void) { return m_pProcess; }
	const ArsonProcessOpts &opts (void) const { return m_opts; }

protected:
	ArsonProcess *m_pProcess;
	ArsonProcessUI *m_pUI;
	ArsonProcessOpts m_opts;
};

/*========================================================*/

class QComboBox;
class QLineEdit;
class QCheckBox;

class ArsonCdWriterProgress : public ArsonSimpleProgress
{
public:
	ArsonCdWriterProgress (QWidget *parent, const char *name);

protected:
	virtual void processOpts (ArsonProcessOpts &opts);

private:
	QComboBox *m_pSpeed;
	QComboBox *m_pHow;
	QCheckBox *m_pEject;
	QCheckBox *m_pOverburn;
	QCheckBox *m_pNoFix;
};

/*========================================================*/

class QBoxLayout;

class ArsonImgCreateProgress : public ArsonCdWriterProgress
{
public:
	ArsonImgCreateProgress (QWidget *parent, const char *name);

	void setAutoIsoLabel (bool automatic);
	void setIsoLabel (const QString &str);
	QString label (void) const;
	void showLabelEdit (QBoxLayout *pl = NULL);

protected:
	virtual void processOpts (ArsonProcessOpts &opts);
	
private:
	QLineEdit *m_pIsoLabel;
};

/*========================================================*/

class ArsonWriterProcess;

class ArsonCdWriter : public ArsonProcessMgr
{
public:
	ArsonCdWriter (ArsonProcessUI *pUI);
	
//	static uint fixationTime (uint total);

	enum {
		Test = 0,
		Burn,
		Both,
	};

protected:
	virtual ArsonWriterProcess *createWriterProcess (void) = 0;
	
	virtual void taskComplete (ArsonProcess *proc);

	bool writeCd (void);

	enum {
		State_None,
		State_Test,
		State_Burn,
	};

	int m_state;
};

/*========================================================*/

class ArsonCueWriter : public ArsonCdWriter
{
public:
	ArsonCueWriter (ArsonProcessUI *pUI, const char *cuefile = NULL);
	virtual ~ArsonCueWriter (void);

	virtual void begin (const ArsonProcessOpts &opts);

	bool setCueFile (const char *filename);

protected:
	virtual ArsonWriterProcess *createWriterProcess (void);
	
	QDir m_currentDir;
	QString m_cuefile;
};

class ArsonIsoWriter : public ArsonCdWriter
{
public:
	ArsonIsoWriter (ArsonProcessUI *pUI, const char *isofile = NULL);

	virtual void begin (const ArsonProcessOpts &opts);

protected:
	virtual ArsonWriterProcess *createWriterProcess (void);
	
	QString m_isoFile;
};

/*========================================================*/
#endif	/*	__CDWRITER_H__	*/
