/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: source.cc,v 1.50 2005/03/09 02:24:53 pauld Exp $
*/

#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <poll.h>
#include <float.h>
#include <cerrno>
#include <ctime>
#include <cmath>
#include <iomanip>
#include <algorithm>

#include <pbd/lockmonitor.h>
#include <pbd/xml++.h>
#include <pbd/pthread_utils.h>

#include <ardour/source.h>
#include <ardour/filesource.h>
#include <ardour/sndfilesource.h>

#include "i18n.h"

using std::min;
using std::max;

using namespace ARDOUR;
using namespace PBD;

SigC::Signal1<void,Source *> Source::SourceCreated;
pthread_t                    Source::peak_thread;
bool                         Source::have_peak_thread = false;
vector<Source*>              Source::pending_peak_sources;
PBD::Lock                    Source::pending_peak_sources_lock;
int                          Source::peak_request_pipe[2];

bool Source::_build_missing_peakfiles = false;
bool Source::_build_peakfiles = false;

Source::Source (bool announce)
{
	_id = ARDOUR::new_id();
	_use_cnt = 0;
	_peaks_built = false;
	next_peak_clear_should_notify = true;
	peakfile = -1;
	_timestamp = 0;
	_read_data_count = 0;
	_write_data_count = 0;
}

Source::Source (const XMLNode& node) 
{
	_use_cnt = 0;
	_peaks_built = false;
	next_peak_clear_should_notify = true;
	peakfile = -1;
	_timestamp = 0;
	_read_data_count = 0;
	_write_data_count = 0;

	if (set_state (node)) {
		throw failed_constructor();
	}
}

Source::~Source ()
{
	if (peakfile >= 0) {
		close (peakfile);
	}
}

XMLNode&
Source::get_state ()
{
	XMLNode *node = new XMLNode ("Source");
	char buf[64];

	node->add_property ("name", _name);
	snprintf (buf, sizeof(buf)-1, "%llu", _id);
	node->add_property ("id", buf);

	if (_timestamp != 0) {
		snprintf (buf, sizeof (buf), "%ld", _timestamp);
		node->add_property ("timestamp", buf);
	}

	if (_captured_for.length()) {
		node->add_property ("captured-for", _captured_for);
	}

	return *node;
}

int
Source::set_state (const XMLNode& node)
{
	const XMLProperty* prop;

	if ((prop = node.property ("name")) != 0) {
		_name = prop->value();
	} else {
		return -1;
	}
	
	if ((prop = node.property ("id")) != 0) {
		sscanf (prop->value().c_str(), "%llu", &_id);
	} else {
		return -1;
	}

	if ((prop = node.property ("timestamp")) != 0) {
		sscanf (prop->value().c_str(), "%ld", &_timestamp);
	}

	if ((prop = node.property ("captured-for")) != 0) {
		_captured_for = prop->value();
	}

	return 0;
}

/***********************************************************************
  PEAK FILE STUFF
 ***********************************************************************/

void*
Source::peak_thread_work (void* arg)
{
	PBD::ThreadCreated (pthread_self(), X_("Peak"));
	struct pollfd pfd[1];

	LockMonitor lm (pending_peak_sources_lock, __LINE__, __FILE__);

	while (true) {

		pfd[0].fd = peak_request_pipe[0];
		pfd[0].events = POLLIN|POLLERR|POLLHUP;

		pthread_mutex_unlock (pending_peak_sources_lock.mutex());

		if (poll (pfd, 1, -1) < 0) {

			if (errno == EINTR) {
				pthread_mutex_lock (pending_peak_sources_lock.mutex());
				continue;
			}
			
			error << compose (_("poll on peak request pipe failed (%1)"),
					  strerror (errno))
			      << endmsg;
			break;
		}

		if (pfd[0].revents & ~POLLIN) {
			error << _("Error on peak thread request pipe") << endmsg;
			break;
		}

		if (pfd[0].revents & POLLIN) {

			char req;
			
			/* empty the pipe of all current requests */

			while (1) {
				size_t nread = ::read (peak_request_pipe[0], &req, sizeof (req));

				if (nread == 1) {
					switch ((PeakRequest::Type) req) {
					
					case PeakRequest::Build:
						break;
						
					case PeakRequest::Quit:
						pthread_exit_pbd (0);
						/*NOTREACHED*/
						break;
						
					default:
						break;
					}

				} else if (nread == 0) {
					break;
				} else if (errno == EAGAIN) {
					break;
				} else {
					fatal << _("Error reading from peak request pipe") << endmsg;
					/*NOTREACHED*/
				}
			}
		}

		pthread_mutex_lock (pending_peak_sources_lock.mutex());

		while (!pending_peak_sources.empty()) {

			Source* s = pending_peak_sources.front();
			pending_peak_sources.erase (pending_peak_sources.begin());

			pthread_mutex_unlock (pending_peak_sources_lock.mutex());
			s->build_peaks();
			pthread_mutex_lock (pending_peak_sources_lock.mutex());
		}
	}

	pthread_exit_pbd (0);
	/*NOTREACHED*/
	return 0;
}

int
Source::start_peak_thread ()
{
	if (!_build_peakfiles) {
		return 0;
	}

	if (pipe (peak_request_pipe)) {
		error << compose(_("Cannot create transport request signal pipe (%1)"), strerror (errno)) << endmsg;
		return -1;
	}

	if (fcntl (peak_request_pipe[0], F_SETFL, O_NONBLOCK)) {
		error << compose(_("UI: cannot set O_NONBLOCK on peak request pipe (%1)"), strerror (errno)) << endmsg;
		return -1;
	}

	if (fcntl (peak_request_pipe[1], F_SETFL, O_NONBLOCK)) {
		error << compose(_("UI: cannot set O_NONBLOCK on peak request pipe (%1)"), strerror (errno)) << endmsg;
		return -1;
	}

	if (pthread_create_and_store ("peak file builder", &peak_thread, 0, peak_thread_work, 0)) {
		error << _("Source: could not create peak thread") << endmsg;
		return -1;
	}

	have_peak_thread = true;
	return 0;
}

void
Source::stop_peak_thread ()
{
	if (!have_peak_thread) {
		return;
	}

	void* status;

	char c = (char) PeakRequest::Quit;
	::write (peak_request_pipe[1], &c, 1);
	pthread_join (peak_thread, &status);
}

void 
Source::queue_for_peaks (Source& source)
{
	if (have_peak_thread) {

		LockMonitor lm (pending_peak_sources_lock, __LINE__, __FILE__);

		source.next_peak_clear_should_notify = true;

		if (find (pending_peak_sources.begin(),
			  pending_peak_sources.end(),
			  &source) == pending_peak_sources.end()) {
			pending_peak_sources.push_back (&source);
		}

		char c = (char) PeakRequest::Build;
		::write (peak_request_pipe[1], &c, 1);
	}
}

void Source::clear_queue_for_peaks ()
{
	/* this is done to cancel a group of running peak builds */
	if (have_peak_thread) {
		LockMonitor lm (pending_peak_sources_lock, __LINE__, __FILE__);
		pending_peak_sources.clear ();
	}
}


bool
Source::peaks_ready (SigC::Slot0<void> the_slot) const
{
	bool ret;
	LockMonitor lm (_lock, __LINE__, __FILE__);

	/* check to see if the peak data is ready. if not
	   connect the slot while still holding the lock.
	*/

	if (!(ret = _peaks_built)) {
		PeaksReady.connect (the_slot);
	}

	return ret;
}

int
Source::initialize_peakfile (bool newfile, string audio_path)
{
	struct stat statbuf;

	peakpath = peak_path (audio_path);

	if (newfile) {

		if (!_build_peakfiles) {
			return 0;
		}

		_peaks_built = false;

	} else {

		if (stat (peakpath.c_str(), &statbuf)) {
			if (errno != ENOENT) {
				/* it exists in the peaks dir, but there is some kind of error */
				
				error << compose(_("Source: cannot stat peakfile \"%1\""), peakpath) << endmsg;
				return -1;
			}

			/* older sessions stored peaks in the same directory
			   as the audio. so check there as well.
			*/
			
			peakpath = old_peak_path (audio_path);
			
			if (stat (peakpath.c_str(), &statbuf)) {
				
				if (errno == ENOENT) {
					_peaks_built = false;
				} else {
					
					/* it exists in the audio dir , but there is some kind of error */
					
					error << compose(_("Source: cannot stat peakfile \"%1\""), peakpath) << endmsg;
					return -1;
				}
				
			} else {
				/* we found it in the sound dir*/
			}

		} else {
			
			/* we found it in the peaks dir */
			
		}

		if (statbuf.st_size == 0) {
			_peaks_built = false;
		} else {
			// Check if the audio file has changed since the peakfile was built.
			struct stat stat_file;
			int err = stat (audio_path.c_str(), &stat_file);
			
			if (!err && stat_file.st_mtime > statbuf.st_mtime){
				_peaks_built = false;
			} else {
				_peaks_built = true;
			}
		}
	}

	if ((peakfile = ::open (peakpath.c_str(), O_RDWR|O_CREAT, 0664)) < 0) {
		error << compose(_("Source: cannot open peakpath \"%1\" (%2)"), peakpath, strerror (errno)) << endmsg;
		return -1;
	}

	if (!newfile && !_peaks_built && _build_missing_peakfiles && _build_peakfiles) {
		build_peaks_from_scratch ();
	} 

	return 0;
}

int 
Source::read_peaks (PeakData *peaks, jack_nframes_t npeaks, jack_nframes_t start, jack_nframes_t cnt) const
{
	LockMonitor lm (_lock, __LINE__, __FILE__);
	off_t first_peak_byte;
	double scale;
	double expected_peaks;
	double fperpeak;
	PeakData::PeakDatum xmax;
	PeakData::PeakDatum xmin;
	int32_t to_read;
	uint32_t nread;
	jack_nframes_t zero_fill = 0;
	
	// cerr << "\n\nRP: npeaks = " << npeaks << " start = " << start << " cnt = " << cnt << " len = " << _length << endl;
	
	expected_peaks = (cnt / (double) frames_per_peak);
	scale = npeaks/expected_peaks;
	fperpeak = cnt / (double) npeaks;
	
	first_peak_byte = (start / frames_per_peak) * sizeof (PeakData);

	/* fix for near-end-of-file conditions */

	if (start + cnt > _length) {
		jack_nframes_t old = npeaks;
		npeaks = min ((_length - start) / frames_per_peak, npeaks);
		zero_fill = old - npeaks;
	}

	// cerr << "actual/npeaks = " << npeaks << " zf = " << zero_fill << endl;

	if (npeaks == cnt) {

		/* no scaling at all, just get the sample data and duplicate it for
		   both max and min peak values.
		*/
		
		Sample staging[cnt];
		
		if (read_unlocked (staging, start, cnt) != cnt) {
			error << _("cannot read sample data for unscaled peak computation") << endmsg;
			return -1;
		}

		for (jack_nframes_t i = 0; i < npeaks; ++i) {
			peaks[i].max = staging[i];
			peaks[i].min = staging[i];
		}
		
		return 0;
		
	}

	if (scale == 1.0) {
		if ((nread = ::pread (peakfile, peaks, sizeof (PeakData)* npeaks, first_peak_byte)) != sizeof (PeakData) * npeaks) {
			cerr << "Source["
			      << _name
			      << "]: cannot read peaks from peakfile! (read only " 
			      << nread
			      << " not " 
			      << npeaks
			      << "at sample " 
			      << start
			      << " = byte "
			      << first_peak_byte
			      << ')'
			      << endl;
			return -1;
		}

		if (zero_fill) {
			memset (&peaks[npeaks], 0, sizeof (PeakData) * zero_fill);
		}

		return 0;
	}

	uint32_t step;
	double istep;
	jack_nframes_t n;
	uint32_t i, j, nextj, skipj, jbegin, currpeak;
	uint32_t start_byte;
	jack_nframes_t current_frame = start;

	jack_nframes_t start_peak = (jack_nframes_t) ceil(start * scale / frames_per_peak);
	
	if (scale < 1.0) {

		/* the caller wants:

		    - more frames-per-peak (lower resolution) than the peakfile, or to put it another way,
                    - less peaks than the peakfile holds for the same range

		    So, read a block into a staging area, and then downsample from there.
		*/

		uint32_t chunksize = (uint32_t) min (expected_peaks, 4096.0);
		PeakData staging[chunksize];
		
		//ifract = modf (expected_peaks/npeaks, &istep);
		//step = (int32_t) floor (istep);

		istep = expected_peaks/npeaks;
		step = (uint32_t) (istep + 0.5);

		// start at beginning 
		current_frame = 0;
		jack_nframes_t igncnt = 0;

		n = 0;
		skipj = 0;
		currpeak = 0;
		jbegin = 0;
		j = 0;
		
		// calculate starting position
		// and initial values for counters
		// all of this nonsense to prevent jiggling peaks when trimming at
		// the front of a region
		nextj = (int32_t) ((currpeak+1) * istep + 0.5) - jbegin;
		
		while (currpeak < start_peak)
		{
			igncnt = nextj - j;
				
			current_frame += igncnt * frames_per_peak;
			++currpeak;
			j = nextj;
			nextj = (int32_t) ((currpeak+1) * istep + 0.5) - jbegin;
		}

		jbegin += j;
		
		while (n < npeaks) {

			jack_nframes_t tnp;

			start_byte = jbegin * sizeof (PeakData);

			/* don't try to read more peaks than could possibly exist
			   given the current length.
			*/


			tnp = min ((_length - current_frame)/frames_per_peak, (jack_nframes_t) expected_peaks);
			to_read = min (chunksize, tnp);

			if ((nread = ::pread (peakfile, staging, sizeof (PeakData) * to_read, start_byte)) != sizeof (PeakData) * to_read) {
				cerr << "Source["
				     << _name
				     << "]: cannot read peak data from peakfile ("
				     << nread 
				     << " peaks instead of "
				     << to_read
				     << ") ("
				     << strerror (errno)
				     << ')'
				     << " at start_byte = " << start_byte 
				     << " and current frame = " << current_frame 
				     << " _length = " << _length
				     << " expected maxpeaks = " << (_length - current_frame)/frames_per_peak
				     << " npeaks was " << npeaks
				     << endl;
				return -1;
			}
			
			/* adjust back to number of PeakData */

			nread /= sizeof (PeakData);

			j = 0;

			if (nread <= (int32_t) step) {
				
				/* we didn't read enough data to interpolate,
				   so just copy the data we did read into
				   the actual buffer, and then get out of here.
				*/

				while (j < nread && n < npeaks) {
					peaks[n].max = staging[j].max;
					peaks[n].min = staging[j].min;
					j++;
					n++;
				}

				break;
			}

			j = 0;
			nextj = (int32_t) ((currpeak+1) * istep + 0.5) - jbegin;

			if (nread < nextj) {
				nextj = nread;
			}
			
			while ((nextj <= nread) && (n < npeaks))
			{
				igncnt = nextj - j;
				
				xmax = -1.0;
				xmin = 1.0;
				
				for (; j < nextj; ++j) {
					xmax = max (xmax, staging[j].max);
					xmin = min (xmin, staging[j].min);
				}
				
				peaks[n].max = xmax;
				peaks[n].min = xmin;
				
				n++;
				
				current_frame += igncnt * frames_per_peak;

				++currpeak;
				j = nextj;
				nextj = (int32_t) ((currpeak+1) * istep + 0.5) - jbegin;
			}

			jbegin += j;
			expected_peaks -= nread;

		}

		if (zero_fill) {
			//cerr << "at zero, npeaks = " << npeaks << endl;
			memset (&peaks[npeaks], 0, sizeof (PeakData) * zero_fill);
		}

		return 0;

	} else {
		
//		cerr << "scale > 1.0\n";

		/* the caller wants 

		     - less frames-per-peak (more resolution)
		     - more peaks than stored in the Peakfile

		   So, fetch data from the raw source, and generate peak
		   data on the fly.
		*/

		Sample buf[4096];
		int32_t offset = start;

		istep = (double)cnt/(double)npeaks;
		step = (int32_t) (istep + 0.5);
		n = 0;
		nextj = j = 0;
		i = 0;
		currpeak = 0;
		
		start_peak = (jack_nframes_t) (offset / istep);
		currpeak = 0;
		jbegin = 0;

		// set up initial positions
		// all of this nonsense to prevent jiggling peaks when trimming at
		// the front of a region
		nextj = (int32_t) ((currpeak+1) * istep + 0.5) - jbegin;
		
		while (currpeak < start_peak)
		{
			++currpeak;
			j = nextj;
			nextj = (int32_t) ((currpeak+1) * istep + 0.5) - jbegin;
		}

		jbegin += j;

		
		while (n < npeaks) {

			to_read = min ((int32_t) (_length - jbegin), (int32_t) 4096);

			if ((nread = read_unlocked (buf, jbegin, to_read)) < 0) {
				error << compose(_("Source[%1]: peak read - cannot read %2 samples at offset %3"), _name, to_read, jbegin) << endmsg;
				return -1;
			}

			nread /= sizeof (Sample);

			if (nread == 0) {
				memset (&peaks[n], 0, (npeaks - n) * sizeof (PeakData));
				break;
			}

			j = 0;

			nextj = (int32_t) ((currpeak+1) * istep + 0.5) - jbegin;
			//printf ("j is %d  n=%d  npeaks=%d nextj=%d nread=%d  currpeak: %d   offset: %d\n", j, n, npeaks, nextj, nread, currpeak, offset);

			if (nread < nextj) {
				nextj = nread;
			}
			
			while (nextj <= nread && n < npeaks)
			{
				xmax = -1.0;
				xmin = 1.0;

				for (; j < nextj && j < nread; ++j) {
					xmax = max (xmax, buf[j]);
					xmin = min (xmin, buf[j]);
				} 

				peaks[n].min = xmin;
				peaks[n].max = xmax;
				n++;
				++currpeak;

				nextj = (int32_t) ((currpeak+1) * istep + 0.5) - jbegin;
				//printf ("j is %d  n=%d  nextj=%d nread=%d\n", j, n, nextj, nread);
			}

			jbegin += j;
			
			//printf ("j is %d  n=%d  i=%d  nread=%d nextj=%d\n", j, n, i, nread, nextj);
		}
		
		if (zero_fill) {
			memset (&peaks[npeaks], 0, sizeof (PeakData) * zero_fill);
		}

		return 0;
	}

	return -1;
}

#undef DEBUG_PEAK_BUILD

int
Source::build_peaks ()
{
	vector<PeakBuildRecord*> built;
	int status = -1;
	bool pr_signal = false;
	list<PeakBuildRecord*> copy;

	{
		LockMonitor lm (_lock, __LINE__, __FILE__);
		copy = pending_peak_builds;
		pending_peak_builds.clear ();
	}
		

#ifdef DEBUG_PEAK_BUILD
	cerr << "build peaks with " << pending_peak_builds.size() << " requests pending\n";
#endif		

	for (list<PeakBuildRecord *>::iterator i = copy.begin(); i != copy.end(); ++i) {
		
		if ((status = do_build_peak ((*i)->frame, (*i)->cnt)) != 0) { 
			unlink (peakpath.c_str());
			break;
		}
		built.push_back (new PeakBuildRecord (*(*i)));
		delete *i;
	}

	{ 
		LockMonitor lm (_lock, __LINE__, __FILE__);

		if (status == 0) {
			_peaks_built = true;
			
			if (next_peak_clear_should_notify) {
				next_peak_clear_should_notify = false;
				pr_signal = true;
			}
		}
	}

	if (status == 0) {
		for (vector<PeakBuildRecord *>::iterator i = built.begin(); i != built.end(); ++i) {
			PeakRangeReady ((*i)->frame, (*i)->cnt); /* EMIT SIGNAL */
			delete *i;
		}

		if (pr_signal) {
			PeaksReady (); /* EMIT SIGNAL */
		}
	}

	return status;
}

int
Source::do_build_peak (jack_nframes_t first_frame, jack_nframes_t cnt)
{
	jack_nframes_t current_frame;
	Sample buf[frames_per_peak];
	Sample xmin, xmax;
	uint32_t  peaki;
	PeakData* peakbuf;
	jack_nframes_t frames_read;
	jack_nframes_t frames_to_read;
	off_t first_peak_byte;
	int ret = -1;

#ifdef DEBUG_PEAK_BUILD
	cerr << pthread_self() << ": " << _name << ": building peaks for " << first_frame << " to " << first_frame + cnt - 1 << endl;
#endif

	first_peak_byte = (first_frame / frames_per_peak) * sizeof (PeakData);

#ifdef DEBUG_PEAK_BUILD
	cerr << "seeking to " << first_peak_byte << " before writing new peak data\n";
#endif

	current_frame = first_frame;
	peakbuf = new PeakData[(cnt/frames_per_peak)+1];
	peaki = 0;

	while (cnt) {

		frames_to_read = min (frames_per_peak, cnt);

		if ((frames_read = read_unlocked (buf, current_frame, frames_to_read)) != frames_to_read) {
			error << compose(_("%1: could not write read raw data for peak computation (%2)"), _name, strerror (errno)) << endmsg;
			goto out;
		}

		xmin = FLT_MAX;
		xmax = FLT_MIN;

		for (jack_nframes_t n = 0; n < frames_read; ++n) {
			xmax = max (xmax, buf[n]);
			xmin = min (xmin, buf[n]);
		}

		peakbuf[peaki].max = xmax;
		peakbuf[peaki].min = xmin;
		peaki++;

		current_frame += frames_read;
		cnt -= frames_read;
	}

	if (::pwrite (peakfile, peakbuf, sizeof (PeakData) * peaki, first_peak_byte) != sizeof (PeakData) * peaki) {
		error << compose(_("%1: could not write peak file data (%2)"), _name, strerror (errno)) << endmsg;
		goto out;
	}

	ret = 0;

  out:
	delete [] peakbuf;
	return ret;
}

void
Source::build_peaks_from_scratch ()
{
	LockMonitor lp (_lock, __LINE__, __FILE__); 

	next_peak_clear_should_notify = true;
	pending_peak_builds.push_back (new PeakBuildRecord (0, _length));
	queue_for_peaks (*this);
}

bool
Source::file_changed (string path)
{
	struct stat stat_file;
	struct stat stat_peak;

	int e1 = stat (path.c_str(), &stat_file);
	int e2 = stat (peak_path(path).c_str(), &stat_peak);
	
	if (!e1 && !e2 && stat_file.st_mtime > stat_peak.st_mtime){
		return true;
	} else {
		return false;
	}
}

void
Source::use ()
{
	_use_cnt++;
}

void
Source::release ()
{
	if (_use_cnt) --_use_cnt;
}
