// ---------------------------------------------------------------------------
// - Terminal.cpp                                                            -
// - aleph standard library - terminal class implementation                  -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Boolean.hpp"
#include "Terminal.hpp"
#include "Character.hpp"
#include "Exception.hpp"
#include "ctrm.hpp"

namespace aleph {

  // the terminal supported quarks
  static const long QUARK_READ      = String::intern ("read");
  static const long QUARK_EOFP      = String::intern ("eof-p");
  static const long QUARK_WRITE     = String::intern ("write");
  static const long QUARK_GETPP     = String::intern ("get-primary");
  static const long QUARK_GETSP     = String::intern ("get-secondary");
  static const long QUARK_SETPP     = String::intern ("set-primary");
  static const long QUARK_SETSP     = String::intern ("set-secondary");
  static const long QUARK_PUSHB     = String::intern ("pushback");
  static const long QUARK_VALIDP    = String::intern ("valid-p");
  static const long QUARK_READLN    = String::intern ("readln");
  static const long QUARK_WRITELN   = String::intern ("writeln");
  static const long QUARK_NEWLINE   = String::intern ("newline");
  static const long QUARK_READLINE  = String::intern ("readline");
  static const long QUARK_SETEOFIGN = String::intern ("set-eof-ignore");
  static const long QUARK_SETEOFMAP = String::intern ("set-eof-character");

  // default prompts
  static const char* DEFAULT_PROMPT1 = "aleph >";
  static const char* DEFAULT_PROMPT2 = "..... >";
  
  // create a default terminal

  Terminal::Terminal (void) {
    // prompt initialize
    d_prt1 = DEFAULT_PROMPT1;
    d_prt2 = DEFAULT_PROMPT2;
    // save the input stream state
    save ();
  }

  // destroy this terminal

  Terminal::~Terminal (void) {
    // restore terminal state
    restore ();
  }

  // return the class name

  String Terminal::repr (void) const {
    return "Terminal";
  }

  // kill all editing in the current line

  void Terminal::kill (void) {
    long num = d_lbuf.setce ();
    mover (num);
    num = d_lbuf.length ();
    erasel (num);
    d_lbuf.reset ();
  }

  // replace an editing line with another one

  void Terminal::replace (const String& line) {
    kill ();
    d_lbuf.add (line);
    Output::write (line);
  }
    
  // read a line and return a string

  String Terminal::readline (bool pflag) {
    // reset the readline buffer
    d_lbuf.reset ();
    
    // check for the eof
    if (d_eof == true) return eofc;
    
    // turn off canonical mode and reset error mode
    bool canon = nocanon ();
    temode (false);

    // ok, print the prompt
    pflag ? Output::write (d_prt1) : Output::write (d_prt2);
    
    // loop until we have an eol/eof
    while (1) {
      t_word w = wread ();
      
      // check for backspace
      if (InputTerm::isbs (w) == true) {
	bool status = d_lbuf.erase ();
	if (status == true) erasel (1);
	continue;
      }
      
      // check for delete
      if (InputTerm::isdel (w) == true) {
	bool status = d_lbuf.del ();
	if (status == true) del ();
	continue;
      }
      
      // check for insert
      if (InputTerm::isins (w) == true) {
	d_insert = !d_insert;
	d_lbuf.setimode (d_insert);
	continue;
      }
      
      // check for move to begin line
      if (w == InputTerm::WIDE_STDBL) {
	long num = d_lbuf.setcb ();
	movel (num);
	continue;
      }
      
      // check for move to end of line
      if (w == InputTerm::WIDE_STDEL) {
	long num = d_lbuf.setce ();
	mover (num);
	continue;
      }
      
      // check for kill everything
      if (w == InputTerm::WIDE_STDKB) {
	kill ();
	continue;
      }
      
      // check for kill to eol
      if (w == InputTerm::WIDE_STDKL) {
	long num = d_lbuf.setce ();
	mover (num);
	d_lbuf.kill (num);
	erasel (num);
	continue;
      }

      // check for refresh
      if (w == InputTerm::WIDE_STDRF) {
       String rline = pflag ? d_prt1 : d_prt2;
        rline = rline + d_lbuf.tostring ();
        bool status = refresh (rline);
        if (status == true) d_lbuf.setce ();
        continue;
      }

      // check for move to left
      if (InputTerm::isle (w) == true) {
	bool status = d_lbuf.movel ();
	if (status == true) movel  (1);
	continue;
      }
      
      // check for move to right
      if (InputTerm::isri (w) == true) {
	bool status = d_lbuf.mover ();
	if (status == true) mover  (1);
	continue;
      }

      // check for a move down
      if (InputTerm::isdo (w) == true) {
	if (d_cilo.istop () == true) {
	  kill ();
	  continue;
	}
	String* data = dynamic_cast <String*> (d_cilo.getup ());
	if (data == nilp)
	  kill ();
	else
	  replace (*data);
	continue;
      }
      
      // check for a move up
      if (InputTerm::isup (w) == true) {
	if (d_cilo.isbottom () == true) continue;
	String* data = dynamic_cast <String*> (d_cilo.getdown ());
	if (data == nilp)
	  kill ();
	else
	  replace (*data);
	continue;
      }

      // make sure the cursor is reset to the end for a eol/eof
      if (w == InputTerm::WIDE_STDNL) d_lbuf.setce ();
      if (w == InputTerm::WIDE_STDEF) d_lbuf.setce ();
      
      // if we have a wide character here - we ignore it
      if (InputTerm::iswide (w) == true) continue;
      
      // check for printable - if not we reject - we accept eol/eof as well
      if (InputTerm::isprintable (w) == false) continue;
      
      // normal character
      char c = InputTerm::wtoc (w);
      if ((c == eolc) || (c == eofc)) {
	if (d_lbuf.length () > 0) 
	  d_cilo.add (new String (d_lbuf.tostring()));
	d_lbuf.add (c);
	write (eolc);
	break;
      }
      d_lbuf.add (c);
      if (canon == true) insert (c);
    }
    
    // this is it - return the string buffer
    restore ();
    return d_lbuf.tostring ();
  }

  // set the primary prompt

  void Terminal::setpprompt (const String& value) {
    wrlock ();
    d_prt1 = value;
    unlock ();
  }

  // set the seconday prompt

  void Terminal::setsprompt (const String& value) {
    wrlock ();
    d_prt2 = value;
    unlock ();
  }

  // get the primary prompt

  String Terminal::getpprompt (void) const {
    rdlock ();
    String result = d_prt1;
    unlock ();
    return result;
  }

  // get the secondary prompt

  String Terminal::getsprompt (void) const {
    rdlock ();
    String result = d_prt2;
    unlock ();
    return result;
  }

  // create a new terminal in a generic way

  Object* Terminal::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 0) 
      throw Exception ("argument-error", "invalid arguments with terminal"); 
    return new Terminal;
  }

  // apply this terminal with a set of arguments and a quark

  Object* Terminal::apply (Runnable* robj, Nameset* nset, const long quark,
			   Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch derived arguments
    if (quark == QUARK_PUSHB)
      return  InputTerm::apply(robj, nset, quark, argv);
    if (quark == QUARK_READ)  
      return  InputTerm::apply(robj, nset, quark, argv);
    if (quark == QUARK_EOFP)  
      return  InputTerm::apply(robj, nset, quark, argv);
    if (quark == QUARK_READLN) 
      return InputTerm::apply(robj, nset, quark, argv);
    if (quark == QUARK_VALIDP) 
      return InputTerm::apply  (robj, nset, quark, argv);
    if (quark == QUARK_SETEOFIGN) 
      return InputTerm::apply  (robj, nset, quark, argv);
    if (quark == QUARK_SETEOFMAP) 
      return InputTerm::apply  (robj, nset, quark, argv);
    if (quark == QUARK_WRITE)   
      return OutputTerm::apply (robj, nset, quark, argv);
    if (quark == QUARK_WRITELN) 
      return OutputTerm::apply (robj, nset, quark, argv);
    if (quark == QUARK_NEWLINE) 
      return OutputTerm::apply (robj, nset, quark, argv);

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_READLINE) return new String    (readline (true));
      if (quark == QUARK_GETPP)    return new String    (getpprompt ());
      if (quark == QUARK_GETSP)    return new String    (getsprompt ());
    }

    // dispatch one argument
    if (argc == 1) {
      if (quark == QUARK_READLINE) {
	bool flag = argv->getbool (0);
	String* result = new String (readline (flag));
	return result;
      }
      if (quark == QUARK_SETPP) {
	String val = argv->getstring (0);
	setpprompt (val);
	return nilp;
      }
      if (quark == QUARK_SETSP) {
	String val = argv->getstring (0);
	setsprompt (val);
	return nilp;
      }
    }
    
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
