/**********************************************************************
 ** Location class: Manages a location mudobject, providing methods
 **                 for various location related functions
 **
 **
 **    
 **    Reviewed through: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef LOCATION_C
#define LOCATION_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "mudobject.h"
#include "location.h"
#include "individual.h"
#include "player.h"
#include "builder.h"
#include "verb.h"
#include "verb_list.h"
#include "parse.h"
#include "input.h"
#include "lexer.h"
#include "btree.h"
#include "area_dbase.h"
#include "logs.h"
#include "errlog.h"
#include "object_list.h"
#include "mud.h"
#include "global.h"
#include "objtype.h"
#include "utils.h"
#include "door.h"
#include "timespan.h"
#include "moveable.h"
#include "mobile.h"
#include "flags.h"
#include "adminflags.h"
#include "gameflags.h"
#include "locflags.h"
#include "newfuncts.h"
#include "boat.h"
#include "inp_handler.h"
#include "indflags.h"
#include "code.h"
#include "doorflags.h"

char *dir_str[10] = {"North", "South", "East", "West", "Up", "Down", 
                     "Northeast", "Northwest", "Southeast", "Southwest"};

char *fromdir_str[10] = {"north", "south", "east", "west", "above", "below", 
                         "northeast", "northwest", "southeast", "southwest"};

char *terrain_name[] = {"Forest", "Jungle",    "SandDesert", "RockDesert", 
       "Plains",        "Tundra", "SnowField",  "Icesheet",  "Hills",      
       "Mountains",     "HighMountains",        "Swamp",     "Bog",       
       "ThinForest",    "ThickForest",          "City",      "Ocean",     
       "Lake",          "River",                "PavedRoad", "DirtRoad",  
       "Enclosure",  
			NULL};

float terrain_mult[] = {2.0,      2.5,         2.0,          1.5, 
       1.0,             1.5,      2.0,         1.0,          2.0,          3.0,
       5.0,             3.5,      3.0,         1.5,          2.0,          1.5,
       2.0,             1.0,      1.5,         1.0,          1.0,          1.0};

char *lighting_name[] = {"FullDark", "Dark", "Shady", "Average", "Bright", 
                         "Blinding", NULL};

int opposite_exit[10] = {1, 0, 3, 2, 5, 4, 9, 8, 7, 6};

/***********************************************************************
 ** Location (constructor) - initializes the location object
 **
 ** Parameters: the_name - the name of the location to create
 **             the_area - the area the location belongs to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Location::Location(char *the_name, char *the_area)
{
   /* this should be the only place this is set for locations */
   obj_type = OBJ_TYPE_LOCATION;

   /* load the name right when we create this object */
   if (the_name != NULL)
      set_name(the_name);

   /* set the area string to the appropriate area */
   set_area(the_area);

   loc_flags = new Flags(2);

   room_size = 10;
   the_terrain = Plains;
   the_lighting = Average;
   marked = 0;
}


/***********************************************************************
 ** Location (destructor) - prepares the function for destruction
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Location::~Location()
{
   delete loc_flags;
}
      

/***********************************************************************
 ** set_exit - sets an exit to a string that is the index for a location
 **            or mudobject
 **
 ** Parameters: the_dir - the direction we are setting
 **             the_loc - the character index of the location to set it to
 **
 ** Returns:  1 for success
 **          -1 for failure
 **
 ***********************************************************************/

int Location::set_exit(int the_dir, char *the_loc)
{

   if ((the_dir < 0) || (the_dir >= 10))
      return -1;

   if (the_loc == NULL)
      dir[the_dir].truncate(0);
   else
      dir[the_dir] = the_loc;

   return 1;
}

/***********************************************************************
 ** xget_exit - gets the exit if you give the exit string already known
 ** (private)
 **
 ** Parameters: the_exit - the name of the location to get
 **             the_area - the area the location is in we are getting
 **             results - is it a door, dark, door open, etc
 **
 ** Returns: a pointer to the location object that is indicated by the
 **          direction key
 **
 ***********************************************************************/

Location *Location::xget_exit(char *the_exit, char *the_area, int *results)
{
   Object_List *the_dbase;
   MudObject *tmp_obj;
   int tmp_results;

   if (the_exit == NULL)
   {
      *results = 0;
      return NULL;
   }

   the_dbase = mainstruct->get_dbase();

   if ((tmp_obj = the_dbase->get_mudobject(the_area, the_exit)) == NULL)
   {
      *results = 0;
      return NULL;
   }

   if (tmp_obj->get_type() == OBJ_TYPE_DOOR)
   {
      Door *the_door;
      Flags *tmp_doorflags;
      Strings holder;

      the_door = (Door *) tmp_obj;
      if (!the_door->is_visible(this))
      {
	*results = 10;
	return NULL;
      }
      tmp_obj = the_dbase->get_mudobject(the_area, 
                                   the_door->get_link(this, &tmp_results));

      if (tmp_obj == NULL)
      {
         tmp_doorflags = the_door->get_doorflags();

	 // If the door is closed, we have a few checks to perform to see if
	 // it is hidden or not
	 if (the_door->get_door_state() != DOOR_OPEN)
	 {
	   if ((tmp_doorflags->get_flag(DOORFLAG_HIDDENDOOR)) ||
	       ((tmp_doorflags->get_flag(DOORFLAG_HIDEEXITIN)) &&
		(the_door->which_loc(get_name(), get_area()) == 2)) ||
	       ((tmp_doorflags->get_flag(DOORFLAG_HIDEEXITOUT)) &&
		(the_door->which_loc(get_name(), get_area()) == 1)))
	       
	   {
	     *results = 10;
	     return NULL;
	   }
	 }
	 *results = tmp_results;
	 return NULL;
     
      }

   }
   *results = 0;
   return the_dbase->is_location(tmp_obj);
}

/***********************************************************************
 ** get_exit - gets the location matching the key at the given exit
 **
 ** Parameters: the_dir - the direction we are getting
 **             the_area - the area we are in
 **             results - why it returns null if it does
 **
 ** Returns: a pointer to the location object that is indicated by the
 **          direction key
 **
 ***********************************************************************/

Location *Location::get_exit(int the_dir, char *the_area, int *results)
{
   char *tmp_area;
   char holder[(MAXNAMELEN*2)+2];

   if (dir[the_dir].num_char('@') > 0)
   {
      strncpy(holder, dir[the_dir].str_show(), (MAXNAMELEN*2)+1);
      tmp_area = &holder[0];
      while (*tmp_area != '@')
         tmp_area++;
      *tmp_area = '\0';
      tmp_area++;
      return xget_exit(holder, tmp_area, results);
   }
   else
      return xget_exit(dir[the_dir].str_show(), the_area, results);
}


/***********************************************************************
 ** get_custom_exit - gets the exit when it is one of the ExtraExits
 **
 ** Parameters: the_dir - the direction they are going
 **             the_area - the area we are in
 **             results - why it returns null if it does
 **
 ** Returns: a pointer to the location object that is indicated by the
 **          direction key
 **
 ***********************************************************************/

Location *Location::get_custom_exit(char *the_dir, char *the_area, 
                                                           int *results)
{
   Strings exitname;
   Strings exitloc;
   char *tmp_area;
   char holder[(MAXNAMELEN*2)+2];
   int  counter = 1;

   exitname.assign_word(get_extra_dir(), counter);

   if (exitname.str_show() == NULL)
   {
      *results = 0;
      return NULL;
   }

   /* a hidden exit will have a '*' in front of it */
   if (*(exitname.str_show()) == '*')
   {
      counter++;
      exitname.assign_word(get_extra_dir(), counter); 
   }
 
   counter++;
   exitloc.assign_word(get_extra_dir(), counter);
   counter++;

   while ((exitname.str_show() != NULL) && (STRCASECMP(exitname.str_show(), 
                                          the_dir)))
   {
      exitname.assign_word(get_extra_dir(), counter);
      
      if (exitname.str_show() == NULL) 
         break;

      /* if it is a hidden exit (has an * beginning of it */
      if (*(exitname.str_show()) == '*')
      {
         counter++;
         exitname.assign_word(get_extra_dir(), counter); 
      }

      exitloc.assign_word(get_extra_dir(), counter+1);

      counter += 2;
   }

   if ((exitname.str_show() == NULL) || (exitloc.str_show() == NULL))
   {
      *results = 0;
      return NULL;
   }

   if (exitloc.num_char('@') > 0)
   {
      strncpy(holder, exitloc.str_show(), (MAXNAMELEN*2)+1);
      tmp_area = &holder[0];
      while (*tmp_area != '@')
         tmp_area++;
      *tmp_area = '\0';
      tmp_area++;
      return xget_exit(holder, tmp_area, results);
   }
   else
      return xget_exit(exitloc.str_show(), the_area, results);

}


/***********************************************************************
 ** show_location - shows the location to the player, including title,
 **                 description, and later objects/other players
 **
 ** Parameters: the_player - the player to send all the messages to
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int Location::show_location(Player *the_player, int brief)
{
   Flags *tmp_adminflags;
   Flags *tmp_gameflags;

   if (the_player == NULL)
      return -1;

   tmp_adminflags = the_player->get_admflags();

   if (tmp_adminflags->get_flag(ADMINFLAG_SHOWNAMES))
      the_player->send_plr("\n&+GName&+b: &+M%s@%s&*", get_name(), get_area());
   
   if (!is_lit(the_player))
   {
      the_player->send_plr("\nIt is too dark to see anything.\n");
      return 0;
   }

   the_player->send_plr("\n%s\n\n", title.str_show());
 
   if (!brief)
      the_player->send_plr("%s", desc.str_show());

   list_immobile(the_player);
   list_individuals(the_player);
   list_objects(the_player);
   the_player->send_plr("\n");
   
   tmp_gameflags = the_player->get_gameflags();
   if (tmp_gameflags->get_flag(GAMEFLAG_AUTOEXITS))
   {
      show_exits(the_player);
   }
   return 1;
}

/***********************************************************************
 ** show_exit (private) - show this one exit if it exists
 **
 ** Parameters: the_player - the player to send all the messages to
 **             the_exit - the exit to show
 **             exitname - the name of the exit
 **
 ** Returns:  1 if the exit existed
 **           0 if the exit didnt exist
 **
 ***********************************************************************/

int Location::show_exit(Player *the_player, Strings *the_exit, int exitname){
   Location *this_location;
   int      results;

   if (the_exit->str_show() == NULL) 
      return 0;

   if (the_exit->num_char('@') > 0)
      this_location = get_exit(exitname, get_area(), &results);
   else
      this_location = xget_exit(the_exit->str_show(), get_area(), &results);

   if ((this_location != NULL) && (!this_location->is_lit()))
   {
      the_player->send_plr("\t&+W%s&+B:\t&+rDarkness&*\n", 
                                                       dir_str[exitname]);
      return 1;
   }

   if ((this_location == NULL) && (results == 2))
      the_player->send_plr("\t&+W%s&+B:\t&+R[Closed]&*\n", 
                                                        dir_str[exitname]);

   else if ((this_location == NULL) && (results == 4))
      the_player->send_plr("\t&+W%s&+B:\t&+R[No Rope]&*\n", 
                                                        dir_str[exitname]);
   else if ((this_location == NULL) && (results == 10))
      return 0;
   else if ((this_location == NULL) && (results == 5))
      the_player->send_plr("\t&+W%s&+B:\t&+R[Short Rope]&*\n", 
                                                       dir_str[exitname]);
   else if (this_location == NULL)
      the_player->send_plr(
             "\t&+W%s&+B:\t&+R<!Error - Exit not found in database!>&*\n",
                                                         dir_str[exitname]);
   else
      the_player->send_plr("\t&+W%s&+B:\t&+w%s\n", dir_str[exitname], 
                                                 this_location->get_title());
   return 1;
}


/***********************************************************************
 ** show_extra_exits (private) - show the extra exits indicated in the
 **                              extradir attribute
 **
 ** Parameters: the_player - the player to send all the messages to
 **             long_format - display exits in full format or not?
 **             cur_str - the string to put the results into for short
 **             first - is this the first entry, short format
 **
 ** Returns:  num shown if the exit existed
 **           0 if no exits found
 **
 ***********************************************************************/

int Location::show_extra_exits(Player *the_player, int full_format, 
                                       Strings *cur_str, int first){
   Location *this_location;
   int      count = 0;
   Strings  holder;
   int      cur_word = 1;
   Strings  exitname;
   Strings  locname;
   int      results = 0;
   int      invis = 0;

   if ((!full_format) && (cur_str == NULL))
      return 0;

   /* first we get the exit name */
   exitname.assign_word(get_extra_dir(), cur_word);
   cur_word++;
   while (exitname.str_show() != NULL)
   {
      if ((*(exitname.str_show()) == '*'))
      {
         exitname.assign_word(get_extra_dir(), cur_word);
         cur_word++;
         invis = 1;
      }

      exitname.upper(0);

      /* now we get where the exit goes to */
      locname.assign_word(get_extra_dir(), cur_word);

      if (!invis)
      {
         if (locname.str_show() == NULL)
         {
            holder.sprintf(
              "ExtraDir missing second half in location %s, area %s", 
                                               get_name(), get_area()); 
            mainstruct->log_error(holder.str_show(), "show_extra_exits");
            return count;
         }

         if (locname.num_char('@') > 0)
         {
            char *tmp_area;
            char *tmp_loc;
            Strings tmp_locstr;

            tmp_locstr = locname.str_show();
            tmp_loc = tmp_locstr.str_show();
            tmp_area = tmp_loc;
            while ((*tmp_area) && (*tmp_area != '@'))
               tmp_area++;
            if (*tmp_area)
            {
               *tmp_area = '\0';
               tmp_area++;
            }
            this_location = xget_exit(tmp_loc, tmp_area, &results);
         }
         else
            this_location = 
                    xget_exit(locname.str_show(), get_area(), &results);

         if (full_format)
         {
            if ((this_location != NULL) && (!this_location->is_lit()))
            {
               the_player->send_plr("\t&+W%s&+B:\t&+rDarkness&*\n", 
                                                      exitname.str_show());
               return count;
            }
   

            if ((this_location == NULL) && (results == 2))
               the_player->send_plr("\t&+W%s&+B:\t&+R[Closed]&*\n", 
                                                   exitname.str_show());

            else if ((this_location == NULL) && (results == 4))
               the_player->send_plr("\t&+W%s&+B:\t&+R[No Rope]&*\n", 
                                                   exitname.str_show());
            else if ((this_location == NULL) && (results == 10))
               return 0;
            else if ((this_location == NULL) && (results == 5))
               the_player->send_plr("\t&+W%s&+B:\t&+R[Short Rope]&*\n", 
                                                   exitname.str_show());
            else if (this_location == NULL)
               the_player->send_plr(
              "\t&+W%s&+B:\t&+R<!Error - Exit not found in database!>&*\n",
                                             exitname.str_show());
            else
               the_player->send_plr("\t&+W%s&+B:\t&+w%s\n", 
                          exitname.str_show(), this_location->get_title());
            count++;
         }
         else
         {
            if (!first)
               cur_str->str_cat(", ");   

            /* if this is too long, wrap the line */
            if ((cur_str->str_len() + exitname.str_len()) > 
                                                   the_config.max_line_len)
	    {
               cur_str->str_cat("\n");
               the_player->send_plr("\t%s\n", cur_str->str_show());
               cur_str->truncate(0);
	    }  
            cur_str->str_cat(exitname.str_show());
            count++;
            first = 0;
         }
      }
      else
         invis = 0;

      /* prepare for the next round */
      cur_word++;
      exitname.assign_word(get_extra_dir(), cur_word);
      cur_word++;
   }
   return count;
}

/***********************************************************************
 ** show_exits - lists all the exits and where they go to
 **
 ** Parameters: the_player - the player to send all the messages to
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int Location::show_exits(Player *the_player){
   int count = 0;
   Flags    *tmp_gameflags;
   Strings  holder;
   int      i;
   int      first = 1;

   if (the_player == NULL)
      return -1;

   tmp_gameflags = the_player->get_gameflags();

   the_player->send_plr("&+WObvious paths are:&*\n");


   if (tmp_gameflags->get_flag(GAMEFLAG_FULLEXITS))
   {  
      for (i=0; i<10; i++)
         count += show_exit(the_player, &dir[i], i);
      count += show_extra_exits(the_player, 1, NULL, 0);
   }
   else
   {
      for (i=0; i<10; i++)
      {
	 /* if this direction goes somewhere, display it */
         if (dir[i].str_show() != NULL)
	 {
            if (!first)
               holder.str_cat(", ");   

            /* if this is too long, wrap the line */
            if ((holder.str_len() + strlen(dir_str[i])) > 
                                              (unsigned) the_config.max_line_len)
	    {
               holder.str_cat("\n");
               the_player->send_plr("%s", holder.str_show());
               holder.truncate(0);
	    }  
            holder.str_cat(dir_str[i]);
            count++;
            first = 0;
	 }
      }
      count += show_extra_exits(the_player, 0, &holder, first);

      if (count != 0)
         the_player->send_plr("%s\n", holder.str_show());
   }
   if (count == 0)
      the_player->send_plr("None!\n");
   
   the_player->send_plr("\n");
   return 1;
}


/***********************************************************************
 ** send_location - send to all players in the location
 **
 ** Parameters: the_string - the string to send to the players at the loc
 **             not_this - don't send to this player
 **
 ** Returns:  num players sent to if success
 **          -1 if failure
 **
 ***********************************************************************/

int Location::send_location(char *the_string, Individual *not_this){
   return send_location(the_string, not_this, NULL, SEND_ALL);
}

/***********************************************************************
 ** send_location - send to all players in the location depending on the
 **                 flag sent in and whether or not they can see
 **
 ** Parameters: the_string - the string to send to the players at the loc
 **             not_this - don't send to this player
 **             can_see - if we should send this to those that can see
 **                       (CAN_SEE), or can't see (CANT_SEE), or we don't
 **                       care if they can see (SEND_ALL)
 **
 ** Returns:  num players sent to if success
 **          -1 if failure
 **
 ***********************************************************************/

int Location::send_location(char *the_string, Individual *not_this, int can_see){
   return send_location(the_string, not_this, NULL, can_see);
}


/***********************************************************************
 ** send_location - send to all players in the location
 **
 ** Parameters: the_string - the string to send to the players at the loc
 **             not_this - don't send to this player
 **             or_this - don't send to this player either
 **
 ** Returns:  num players sent to if success
 **          -1 if failure
 **
 ***********************************************************************/

int Location::send_location(char *the_string, Individual *not_this, 
                                         Individual *or_this){
   return send_location(the_string, not_this, or_this, SEND_ALL);
}

/***********************************************************************
 ** send_location - send to all players in the location
 **
 ** Parameters: the_string - the string to send to the players at the loc
 **             not_this - don't send to this player
 **
 ** Returns:  num players sent to if success
 **          -1 if failure
 **
 ***********************************************************************/

int Location::send_location(char *the_string, Individual *not_this, Individual *or_this, int can_see)
{
    int count = 0; // the number of players this is sent to
    MudObject  *tmp_obj;
    Player     *player_list;
    Flags      *gameflag;
    
    if ((not_this != NULL) && (not_this->get_type() != OBJ_TYPE_PLAYER))
        not_this = NULL;
    if ((or_this != NULL) && (or_this->get_type() != OBJ_TYPE_PLAYER))
        or_this = NULL;
    
    player_list = mainstruct->get_first_player();
    inventory.reset_current();
    
    tmp_obj = inventory.get_next();
    // go through all players sending the message 
    while (tmp_obj != NULL)
    {
        if (tmp_obj->get_type() == OBJ_TYPE_PLAYER)
        {
            player_list = (Player *) tmp_obj;
            gameflag = player_list->get_gameflags();
            
            if (gameflag->get_flag(GAMEFLAG_ONGAME))
            {
                // if this is not the player we aren't supposed to send this to
                if (((not_this == NULL) || (STRCASECMP(not_this->get_name(), player_list->get_name()))) 
                &&  ((or_this == NULL) || (STRCASECMP(or_this->get_name(), player_list->get_name()))))
                {
                    inventory.push_current();
                    if ((can_see == SEND_ALL) 
                    || (is_lit(player_list) && can_see) || (!is_lit(player_list) && !can_see))
                    {
                        player_list->send_plr(the_string);
                        count++;
                    }
                    inventory.pop_current();
                }
            }
        }
        // go to next player
        tmp_obj = inventory.get_next();
    }
    return count;
}


/***********************************************************************
 ** is_same - are the two locations the same location
 **
 ** Parameters: the_loc - the location we are comparing to
 **
 ** Returns:  1 if the same
 **           0 if different
 **
 ***********************************************************************/

int Location::is_same(Location *the_loc){

   if (the_loc == NULL)
      return 0;

   /* if the location is the same name */
   if (!STRCASECMP(get_name(), the_loc->get_name()))
   {
      /* if this is the same area, then we send them the string */
      if (!STRCASECMP(get_area(), the_loc->get_area()))
      {
         return 1;
      }
         
   } 
   return 0;
}

/***********************************************************************
 ** list_objects - shows all moveable objects at a given location
 **
 ** Parameters: the_player - the player to send the list to
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/
int Location::list_objects(Player *the_player)
{  MudObject *the_object;
   Timespan  the_time;
   Strings   passengers;
   Boat      *the_boat;
   Flags     *adminflags;

   adminflags = the_player->get_admflags();

   /* go through all objects in the location */
   inventory.reset_current();

   /* go through all players, seeing if they are in the room */
   the_object = inventory.get_next();
   while (the_object != NULL)
   {
      if (the_object->is_a_moveable())
      {
         if (the_object->get_type() == OBJ_TYPE_BOAT)
	 {
            the_boat = (Boat *) the_object;

            if (STRNCASECMP(the_boat->get_brief(), "none", 4))
	      if (adminflags->get_flag(ADMINFLAG_SHOWNAMES))
		the_player->send_plr("&+m(%s)&* %s", the_boat->get_name(),
				     the_boat->get_brief());
	      else
		the_player->send_plr("%s", the_boat->get_brief());

            the_boat->get_passengers(the_player, &passengers);
            if (passengers.str_show() != NULL)
	    {
               the_player->send_plr("  The boat contains %s.\n", 
                                                passengers.str_show());
            }
         }
	 else
	 {
            if (STRNCASECMP(((Moveable *) the_object)->get_brief(), "none", 4))
	    {
	      if ((the_object->get_type() != OBJ_TYPE_ROPE) ||
		  (!((Rope *) the_object)->get_itemflags()->
		   get_flag(ITEMFLAG_ISTIED)))
	      {
		if (adminflags->get_flag(ADMINFLAG_SHOWNAMES))
		  the_player->send_plr("&+m(%s)&* %s", the_object->get_name(),
				     ((Moveable *) the_object)->get_brief());
		else
		  the_player->send_plr("%s", ((Moveable *) the_object)->
                                                             get_brief());
	      }
	    }
	 }
      }
      the_object = inventory.get_next();
   }
   return 1;
}

/***********************************************************************
 ** list_immobile - shows all immobile objects at a given location
 **
 ** Parameters: the_player - the player to send the list to
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int Location::list_immobile(Player *the_player)
{
   MudObject *the_object;
   Flags     *tmp_admflags;
   Flags     *tmp_doorflags;

   /* go through all objects in the location */
   inventory.reset_current();

   tmp_admflags = the_player->get_admflags();
   /* go through all players, seeing if they are in the room */
   the_object = inventory.get_next();

   while (the_object != NULL)
   {
      if (the_object->get_type() == OBJ_TYPE_DOOR)
      {
	if (((Door *) the_object)->is_visible(this))
	{
	  tmp_doorflags = ((Door *) the_object)->get_doorflags();
	  if ((!tmp_doorflags->get_flag(DOORFLAG_HIDDENDOOR)) ||
	      (((Door *) the_object)->get_door_state() == DOOR_OPEN)) 
            the_player->send_plr("%s", 
				 ((Door *) the_object)->get_brief(this));
	}
      }

      else if ((the_object->get_type() == OBJ_TYPE_MARKER) &&
          (tmp_admflags->get_flag(ADMINFLAG_SHOWMARKERS)))
      {
         the_player->send_plr("&+g[&+bMARKER&+g]&* %s\n", 
                                                the_object->get_title());
      }

      the_object = inventory.get_next();
   }
   return 1;
}


/***********************************************************************
 ** list_individuals - shows all individuals at a given location
 **
 ** Parameters: the_player - the player to send the list to
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int Location::list_individuals(Player *the_player){
    
    MudObject *the_object;
    Individual *the_ind;
    Timespan   the_time;
    
    /* go through all objects in the location */
    inventory.reset_current();
    
    /* go through all players in the room */
    the_object = inventory.get_next();
    while (the_object != NULL)
    {
        if (the_object->is_an_individual())
        {
            the_ind = (Individual *) the_object;
            
            if ((the_ind->get_type() != OBJ_TYPE_PLAYER) 
            ||  (the_player != (Player *) the_ind)
            ||  ((the_ind->is_vis_game() || the_ind->is_vis_admin()) 
            && !(((Player *)the_ind)->get_admflags())->get_flag(ADMINFLAG_ADMINISTRATOR)))
            {
                if ((the_ind->get_indflags())->get_flag(INDFLAG_CORPSE))
                {
                    the_player->send_plr("A corpse lies here, slowly rotting away.\n");
                }
                else
                {
                    the_player->send_plr("%s", the_ind->get_brief());
                }
            }
        }
        the_object = inventory.get_next();
    }
    return 1;
}


/***********************************************************************
 ** list_terrain_types - lists the possible terrain types to the builder
 **
 ** Parameters: the_builder - the builder to send this to
 **
 ** Returns:  the number displayed
 **
 ***********************************************************************/

int Location::list_terrain_types(Builder *the_builder)
{
   int     i = 0;
   int     newline = 1;
   Strings the_line;

   the_builder->send_bldr("Terrain types: ");
   while (terrain_name[i] != NULL)
   {
      if ((int) (the_line.str_len() + 15 + strlen(terrain_name[i])) > 
                                                  the_config.max_line_len)
      {
         the_builder->send_bldr("%s,\n", the_line.str_show());
         the_line = "               ";
         newline = 1;
      }
      if (!newline)
         the_line.str_cat(", ");
      else
         newline = 0;

      the_line.str_cat(terrain_name[i]);
      i++;
   }
   the_builder->send_bldr("%s\n", the_line.str_show());
   return i;
}


/***********************************************************************
 ** list_lighting_types - lists the possible lighting types to the builder
 **
 ** Parameters: the_builder - the builder to send this to
 **
 ** Returns:  the number displayed
 **
 ***********************************************************************/

int Location::list_lighting_types(Builder *the_builder)
{
   int     i = 0;
   int     newline = 1;
   Strings the_line;

   the_builder->send_bldr("Lighting types: ");
   while (lighting_name[i] != NULL)
   {
      if ((int) (the_line.str_len() + 16 + strlen(lighting_name[i])) > 
                                                  the_config.max_line_len)
      {
         the_builder->send_bldr("%s,\n", the_line.str_show());
         the_line = "                ";
         newline = 1;
      }
      if (!newline)
         the_line.str_cat(", ");
      else
         newline = 0;

      the_line.str_cat(lighting_name[i]);
      i++;
   }
   the_builder->send_bldr("%s\n", the_line.str_show());
   return i;
}



/***********************************************************************
 ** describe - describes the location to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Location::describe(Builder *the_builder)
{
   int i;
   the_builder->send_bldr("\n&+GLocation: \t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_builder->send_bldr("&+GSub-Area: \t&+M%s&*\n", get_location());
   the_builder->send_bldr("&+GClones: \t&+g%s&*\n", get_clones());
   the_builder->send_bldr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_builder->send_bldr("&+GRoomSize: \t&+w%d&*\n", get_room_size());
   the_builder->send_bldr("&+GTerrain: \t&+g%s&*\n", 
                                              terrain_name[get_terrain()]);
   the_builder->send_bldr("&+GLighting: \t&+g%s&*\n", 
                                              lighting_name[get_lighting()]);

   for (i=0; i<10; i++)
   {
      the_builder->send_bldr("&+G%s: %s&+M%s&*\n", dir_str[i],
			     ((i < 6) ? "\t\t" : "\t"), 
               ((dir[i].str_show() != NULL) ? dir[i].str_show() : "--"));
   }
   the_builder->send_bldr("&+GExtraDir: \t&+g%s&*\n", get_extra_dir());

   the_builder->send_bldr("&+GListen:&*\n%s\n", get_listen());
   the_builder->send_bldr("&+GSmell:&*\n%s\n", get_smell());
   
   if (loc_flags->get_flag(LOCFLAG_SEASONAL))
   {
      the_builder->send_bldr("&+GDesc1 (Summer):&*\n%s\n", get_desc(0));
      the_builder->send_bldr("&+GDesc2 (Winter):&*\n%s\n", get_desc(1));
   }
   else
   {
      the_builder->send_bldr("&+GDesc:&*\n%s\n", get_desc(0));      
   }

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the location to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Location::describe(Player *the_player)
{
   int i;

   the_player->send_plr("\n&+GLocation: \t&+M%s&*\n", get_name());
   the_player->send_plr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_player->send_plr("&+GSub-Area: \t&+M%s&*\n", get_location());
   the_player->send_plr("&+GClones: \t&+g%s&*\n", get_clones());
   the_player->send_plr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_player->send_plr("&+GTerrain: \t&+g%s&*\n", 
                                              terrain_name[get_terrain()]);
   the_player->send_plr("&+GLighting: \t&+g%s&*\n", 
                                              lighting_name[get_lighting()]);

   for (i=0; i<10; i++)
      the_player->send_plr("&+G%s: \t\t&+M%s&*\n", dir_str[i], 
               ((dir[i].str_show() != NULL) ? dir[i].str_show() : "--"));

   the_player->send_plr("&+GExtraDir: \t&+g%s&*\n", get_extra_dir());
   the_player->send_plr("&+GListen:&*\n%s\n", get_listen());
   the_player->send_plr("&+GSmell:&*\n%s\n", get_smell());

   if (loc_flags->get_flag(LOCFLAG_SEASONAL))
   {
      the_player->send_plr("&+GDesc1 (Summer):&*\n%s\n", get_desc(0));
      the_player->send_plr("&+GDesc2 (Winter):&*\n%s\n", get_desc(1));
   }
   else
   {
      the_player->send_plr("&+GDesc:&*\n%s\n", get_desc(0));      
   }
   the_player->send_plr("&+YSize: \t\t\t&+W%d&*\n", get_mem_size());

   list_specials(the_player);

   the_player->send_plr("\n");
}


/***********************************************************************
 ** write_object - writes the location to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Location::write_object(FILE *the_file, int build_format)
{
   int i;

   fprintf(the_file, "\nloc %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   write_mudobject_attrib(the_file);

   for (i=0; i<10; i++)
      fprintf(the_file, "%s ", 
                ((dir[i].str_show() == NULL) ? "0" : dir[i].str_show())); 

   fprintf(the_file, "\n^%s^\n", (get_extra_dir() == NULL) ? "" :
                                                          get_extra_dir());
   fprintf(the_file, "^%s^\n", (get_listen() == NULL) ? "" : get_listen());
   fprintf(the_file, "^%s^\n", (get_smell() == NULL) ? "" : get_smell());
   fprintf(the_file, "^%s^\n", (get_desc(1) == NULL) ? "" : get_desc(1));
   fprintf(the_file, "^%s^\n", (get_desc(2) == NULL) ? "" : get_desc(2));
   fprintf(the_file, "%d\n", get_room_size());
   fprintf(the_file, "%d\n", (int) get_terrain());
   fprintf(the_file, "%d\n", (int) get_lighting());
   loc_flags->write_flag(the_file);
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Location::set_attrib(Builder *the_builder, Parse *the_parsed){
   int   i;
   Flags *tmp_adminflags;

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
       send_bldr("You can set the following attributes on a location.\n"
                  "    north, south, east, west, up, down, northeast,\n"
                  "    nortwest, southeast, southwest, extradir, title,\n"
                  "    desc1, desc2, desc3, listen, smell, locflags,\n"
                  "    clones, subarea, roomsize, terrain, lighting,\n"
                  "    and specials\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "title", 
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a title to set to.\n");
         return -1;
      }
      set_title(the_parsed->get_speech());
      the_builder->send_bldr("Title on %s set to: %s\n", get_name(), 
                                                        get_title());
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "desc1", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_desc(the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "desc2", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_extra_desc(the_builder, 0);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "desc3", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_extra_desc(the_builder, 1);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "listen", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_listen(the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "smell", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_smell(the_builder);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "extradir", 
                               strlen(the_parsed->get_target1())))
   {
 
      if (the_parsed->get_speech() == NULL)
      {
		set_extra_dir(NULL);
		the_builder->send_bldr("ExtraDir on %s cleared.\n", get_name());
		return 1;
      }
      set_extra_dir(the_parsed->get_speech());
      the_builder->send_bldr("ExtraDir on %s set to: %s\n", get_name(), 
                                                        get_extra_dir());
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "roomsize",
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as RoomSize.\n");
         return -1;
      }

      set_room_size(atoi(the_parsed->get_speech()));
      the_builder->send_bldr("RoomSize set to %d on location object %s.\n",
                                          get_room_size(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "terrain",
                               strlen(the_parsed->get_target1())))
   {
      int results;

      if ((the_parsed->get_speech() == NULL) || 
                                (strlen(the_parsed->get_speech()) == 0))
      {
         the_builder->send_bldr("You need to specify a terrain type.\n");
         list_terrain_types(the_builder);
         return -1;
      }

      if ((results = find_pos_in_list(terrain_name, the_parsed->get_speech()))
                                                           == -1)
      {
         the_builder->send_bldr("That is not a valid terrain type.\n");
         list_terrain_types(the_builder);
         return -1;
      }

      set_terrain((terrain_type) results);
      the_builder->
            send_bldr("Terrain set to '&+g%s&*' on location object %s.\n",
                                   terrain_name[get_terrain()], get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "lighting",
                               strlen(the_parsed->get_target1())))
   {
      int results;

      if ((the_parsed->get_speech() == NULL) ||
                                (strlen(the_parsed->get_speech()) == 0))
      {
         the_builder->send_bldr("You need to specify a lighting type.\n");
         list_lighting_types(the_builder);
         return -1;
      }

      if ((results = find_pos_in_list(lighting_name, the_parsed->get_speech()))
                                                           == -1)
      {
         the_builder->send_bldr("That is not a valid lighting type.\n");
         list_lighting_types(the_builder);
         return -1;
      }

      set_lighting((light_type) results);
      the_builder->
            send_bldr("Lighting set to '&+g%s&*' on location object %s.\n",
                                   lighting_name[get_lighting()], get_name());
      return 1;
   }

   for (i=0; i<10; i++)
   { 
      if (!STRNCASECMP(the_parsed->get_target1(), dir_str[i], 
                               strlen(the_parsed->get_target1())))
      { 
         Strings new_loc;

         new_loc.assign_word(the_parsed->get_speech(), 1);
         if (new_loc.str_show() == NULL)
	 {
            set_exit(i, NULL);
            the_builder->send_bldr("%s exit on %s cleared.\n", dir_str[i],
                                                  get_name());
         }
         else
         {
            if (set_exit(i, new_loc.str_show()) <= 0)
	    {
               the_builder->send_bldr("Error setting exit!\n");
               return -1;
	    }
            the_builder->send_bldr("%s exit on %s set to: %s\n", dir_str[i], 
                                           get_name(), new_loc.str_show());

            tmp_adminflags = the_builder->get_adminflags();
            if (tmp_adminflags->get_flag(ADMINFLAG_AUTOASSOCIATE))
	    {
               auto_associate(the_builder, i, new_loc.str_show());
	    }
         }
         set_modified(1);
         return 1;
      }
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "locflags", 
                               strlen(the_parsed->get_target1())))
   {
      Flags *tmp_locflags;
      int flagnum;
      Strings holder;

      tmp_locflags = get_locflags();
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("Set which location flag?\n");
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);
     
      if ((flagnum = 
          tmp_locflags->get_by_name(holder.str_show(), locflagnames)) == -1)
      {
         the_builder->send_bldr("That is not a location flag.\n");
         return -1;
      }

      holder.assign_word(the_parsed->get_speech(), 2);
      if (holder.str_show() == NULL)
      {
         the_builder->send_bldr("Set that flag to what?\n"
                                "Valid choices are: On or Off\n");
         return -1;
      }

      if (holder.str_n_cmp("On", holder.str_len()))
         tmp_locflags->set_flag(flagnum);
      else if (holder.str_n_cmp("Off", holder.str_len()))
         tmp_locflags->clr_flag(flagnum);
      else
      {
         the_builder->send_bldr("That is not a valid setting.\n"
                                "Valid choices are: On or Off\n");
         return -1;
      }
      the_builder->send_bldr("Flag &+M%s&* has been set to: %s\n",
             locflagnames[flagnum], (tmp_locflags->get_flag(flagnum)) ? 
			     "&+GOn&*" : "&+ROff&*");
      return 1;
 
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "specials", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_special(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "clones",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_clones(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "subarea", 
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a subarea to set to.\n");
         return -1;
      }
      set_loc_str(the_parsed->get_speech());
      the_builder->send_bldr("Subarea on %s set to: %s\n", get_name(), 
                                                        get_location());
      set_modified(1);
      return 1;
   }

   the_builder->send_bldr("The attribute '%s' is not a location attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}

/***********************************************************************
 ** set_title - sets the title of the mudobject
 **
 ** Parameters: the_title - the title string to set it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Location::set_title(char *the_title)
{
   if (the_title == NULL)
      return -1;

   title = the_title;
   if (title.str_len() > MAXTITLELEN)
      title.truncate(MAXTITLELEN);

   return 1;
}


/***********************************************************************
 ** copy_object - copies the location to an object of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Location::copy_object(Entity *copy_obj)
{
   Location *copy_from;
   int      i;

   if (copy_obj->get_type() != OBJ_TYPE_LOCATION)
      return 0;

   copy_from = (Location *) copy_obj;

   /******* set the mudobject attributes *****/
   copy_mudobject_attrib((MudObject *) copy_from);

   /****** set the location attributes ******/

   for (i=0; i<10; i++)
      set_exit(i, copy_from->get_exit(i));

   set_extra_dir(copy_from->get_extra_dir());   
   set_desc(1, copy_from->get_desc(1));
   set_desc(2, copy_from->get_desc(2));
   set_listen(copy_from->get_listen());
   set_smell(copy_from->get_smell());
   set_room_size(copy_from->get_room_size());
   set_terrain(copy_from->get_terrain());
   set_lighting(copy_from->get_lighting());

   loc_flags->copy_flags(copy_from->get_locflags());

   return 1;
}


/***********************************************************************
 ** get_exit - gets the exit string
 **
 ** Parameters: the_dir - the direction we are getting
 **
 ** Returns: a pointer to the char string
 **
 ***********************************************************************/

char *Location::get_exit(int the_dir)
{
   return dir[the_dir].str_show();
}


/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Location *Location::operator = (Location *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}


/***********************************************************************
 ** get_locflags - gets the pointer to the location flags
 **
 ** Parameters: Nothing
 **
 ** Returns: a pointer to the location flags
 **
 ***********************************************************************/

Flags *Location::get_locflags(void)
{
   return loc_flags;
}


/***********************************************************************
 ** is_lit - is this location lit by something?
 **
 ** Parameters: Nothing
 **
 ** Returns: 1 for lit, 0 for not
 **
 ***********************************************************************/

int Location::is_lit(void)
{
   return is_lit(NULL);
}

int Location::is_lit(Individual *the_ind)
{
   Flags *indflags = NULL;

   if (the_ind != NULL)
      indflags = the_ind->get_indflags();

   if (contains_lit())
      return 1;

   if ((mainstruct->get_tod() == Night) &&
       (loc_flags->get_flag(LOCFLAG_OUTSIDE)))
   {
      if ((indflags != NULL) && (indflags->get_flag(INDFLAG_NIGHTVISION)))
         return 1;
      return 0;
   }

   if (((mainstruct->get_tod() == Dusk) || 
        (mainstruct->get_tod() == Dawn)) &&
       (loc_flags->get_flag(LOCFLAG_LITATDAY)))
   {
      if ((indflags != NULL) && (indflags->get_flag(INDFLAG_NIGHTVISION)))
         return 1;
      return 0;
   }

   if (get_lighting() > Dark)
      return 1;
   if ((get_lighting() > FullDark) && ((indflags != NULL) && 
                                 (indflags->get_flag(INDFLAG_NIGHTVISION))))
      return 1;
 
   return 0;
}


/***********************************************************************
 ** read_location_attrib - reads in location attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Location::read_location_attrib(FILE *read_file, char *areaname, 
                                                      ErrLog *error_log)
{
   token_record *the_token;
   Strings      holder;
   Strings       tmp_char;    /* used to get exits */
   int           i;           /* used to loop */
   char          *temp_desc;

   /* loop through all directions, setting their exits. If we find an @
      after we get the exit, make sure we add the area to the exit */
   for (i = 0; i < 10; i++)
   {
      the_token = get_token(read_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         tmp_char = the_token->the_string;
         set_exit(i, tmp_char.str_show());
      }
   }

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in mudobject %s, area %s", get_name(),
		     areaname);
      error_log->log_err(holder.str_show(), "read_location_attrib");
      return -1;
   }

   /* get the extradir */
   the_token = get_token(read_file, '^');
   set_extra_dir(the_token->the_string);

   /* set the listen desc */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_listen(temp_desc);
   delete temp_desc;

   /* set the smell desc */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_smell(temp_desc);
   delete temp_desc;

   /* set the first extra desc */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_desc(1, temp_desc);
   delete temp_desc;

   /* set the second extra desc */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_desc(2, temp_desc);
   delete temp_desc;

   /* Set room size */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute roomsize in location %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "read_location_attrib");
      return -1;
   }
   set_room_size(atoi(the_token->the_string));

   /* Set terrain size */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute terrain in location %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "read_location_attrib");
      return -1;
   }
   set_terrain((terrain_type) atoi(the_token->the_string));

   /* Set lighting size */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute lighting in location %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "read_location_attrib");
      return -1;
   }
   set_lighting((light_type) atoi(the_token->the_string));

   /* read in the location flags */
   if (loc_flags->read_flags(read_file, error_log) <= 0)
   {
      holder.sprintf("Error reading locflags for location '%s@%s'", get_name(),
                                                                 get_area()); 
      error_log->log_err(holder.str_show(), "read_location_attrib");
      return -1;
   }   

   return 1;
}


/***********************************************************************
 ** get_random_exit - gets a random useable exit and returns the string
 **
 ** Parameter: holder_str - where to place the exit name
 **
 ** Returns:  A pointer to the direction string if exit found, NULL
 **           if no useable exit found
 **
 ***********************************************************************/

char *Location::get_random_exit(Strings *holder_str)
{
   char *dir_name[] = {"north", "south", "east", "west", "up", "down", 
                       "ne", "nw", "se", "sw"};
   int  rand_num;
   int  i;
   char *valid_dir[10];
   int  num_valid = 0;
   int  results = 0;

   for (i=0; i<10; i++)
   {
      if (get_exit(i, get_area(), &results) != NULL)
      {
         valid_dir[num_valid] = dir_name[i];
         num_valid++;
      }
   }
   if (num_valid == 0)
   {
      holder_str->truncate(0);
      return NULL;
   }

   rand_num = (int) (((float) num_valid) * (rand()/(RAND_MAX+1.0)));   
   holder_str->str_copy(valid_dir[rand_num]);
   return holder_str->str_show();
}


/***********************************************************************
 ** set_attrib_extra_desc - for set attrib command, sets the extra
 **                         description attributes
 **
 ** Parameters: the_builder - who is setting this desc
 **             the_desc - which desc to set
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Location::set_attrib_extra_desc(Builder *the_builder, int the_desc)
{
   if (the_builder->get_long_input(&extra_desc[the_desc]) < 0)
   {
      the_builder->send_bldr("Error reading in input, failed!\n");
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** set_attrib_listen - for set attrib command, sets the listen attrib
 **
 ** Parameters: the_builder - who is setting this desc
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Location::set_attrib_listen(Builder *the_builder)
{
   if (the_builder->get_long_input(&listen) < 0)
   {
      the_builder->send_bldr("Error reading in input, failed!\n");
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** set_attrib_smell - for set attrib command, sets the smell attrib
 **
 ** Parameters: the_builder - who is setting this desc
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Location::set_attrib_smell(Builder *the_builder)
{
   if (the_builder->get_long_input(&smell) < 0)
   {
      the_builder->send_bldr("Error reading in input, failed!\n");
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** get_desc - gets the location description, the desc depending on
 **            certain flag settings and environment settings
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the description string
 **
 ***********************************************************************/

char *Location::get_desc()
{
   if (loc_flags->get_flag(LOCFLAG_SEASONAL))
   {
      if (mainstruct->is_summer())
         return desc.str_show();
      else
         return extra_desc[0].str_show();
   }
   else
   {
      return desc.str_show();
   }
}


/***********************************************************************
 ** get_desc - gets the location description, the desc depending on
 **            the number passed in
 **
 ** Parameters: the_num - the number of the desc to get, 0-2
 **
 ** Returns: a pointer to the description string
 **
 ***********************************************************************/

char *Location::get_desc(int the_num)
{
   if ((the_num > 2) || (the_num < 0))
      return NULL;

   if (the_num == 0)
      return desc.str_show();

   return extra_desc[the_num - 1].str_show();
}


/***********************************************************************
 ** set_desc - sets the descriptions, depending on the number passed in
 **
 ** Parameters: the_num - the number of the desc to get, 0-2
 **             the_str - the string name to assign to it
 **
 ** Returns: a pointer to the description string
 **
 ***********************************************************************/

void Location::set_desc(int the_num, char *the_str)
{
   if ((the_num > 2) || (the_num < 0))
      return;

   if (the_num == 0)
      desc = the_str;
   else
   {
      extra_desc[the_num - 1] = the_str;
   }
}


/***********************************************************************
 ** set_listen - sets the string displayed when the player listens
 **
 ** Parameters: the_str - the string name to assign to it
 **
 ***********************************************************************/

void Location::set_listen(char *the_str)
{
   listen = the_str;
}


/***********************************************************************
 ** get_listen - gets the string displayed when the player listens
 **
 ** Returns: a pointer to the listen string
 **
 ***********************************************************************/

char *Location::get_listen()
{
   return listen.str_show();
}


/***********************************************************************
 ** set_smell - sets the string displayed when the player smells
 **
 ** Parameters: the_str - the string name to assign to it
 **
 ***********************************************************************/

void Location::set_smell(char *the_str)
{
   smell = the_str;
}


/***********************************************************************
 ** get_smell - gets the string displayed when the player smells
 **
 ** Returns: a pointer to the smell string
 **
 ***********************************************************************/

char *Location::get_smell()
{
   return smell.str_show();
}


/***********************************************************************
 ** set_room_size - sets the size of the room across
 **
 ** Parameters: new_size - the new size to set the room to
 **
 ***********************************************************************/

void Location::set_room_size(int new_size)
{
   if (new_size <= 0)
      room_size = 1;
   else
      room_size = new_size;
}


/***********************************************************************
 ** get_room_size - gets the size of the room across
 **
 ** Returns: the size of the room
 **
 ***********************************************************************/

int Location::get_room_size()
{
   return room_size;
}


/***********************************************************************
 ** get_terrain_mult - gets the terrain multiple for this location
 **
 ***********************************************************************/

float Location::get_terrain_mult()
{
   return terrain_mult[(int) get_terrain()];
}


/***********************************************************************
 ** set_terrain - sets the terrain type for this room
 **
 ** Parameters: new_terrain - the new terrain setting to set this to
 **
 ***********************************************************************/

void Location::set_terrain(terrain_type new_terrain)
{
   the_terrain = new_terrain;
}


/***********************************************************************
 ** get_terrain - gets the terrain type for this room
 **
 ***********************************************************************/

terrain_type Location::get_terrain()
{
   return the_terrain;
}


/***********************************************************************
 ** set_lighting - sets the lighting (during the day) for this room
 **
 ** Parameters: new_terrain - the new terrain setting to set this to
 **
 ***********************************************************************/

void Location::set_lighting(light_type new_lighting)
{
   the_lighting = new_lighting;
}


/***********************************************************************
 ** get_lighting - gets the lighting (during the day) for this room
 **
 ***********************************************************************/

light_type Location::get_lighting()
{
   return the_lighting;
}


/***********************************************************************
 ** set_extra_dir - sets the extra directions for this room
 **
 ** Parameters: the_str - the string name to assign to it
 **
 ***********************************************************************/

void Location::set_extra_dir(char *the_str)
{
  if (the_str == NULL)
    extra_dir.truncate(0);
  else
    extra_dir = the_str;
}


/***********************************************************************
 ** auto_associate - tries to set the opposite exit of the one we just
 **                  set on the location we just set this exit to (confused?)
 **
 ** Parameters: the_builder - the builder who did the exit setting
 **             the_exit - the exit they set this to
 **             new_loc - the new location we are setting this to
 **
 ** Returns: 1 for success, 0 for not associated, -1 for fatal error
 **
 ***********************************************************************/

int Location::auto_associate(Builder *the_builder, int the_exit, char *new_loc)
{
   Entity      *tmp_obj;
   Location    *the_loc;
   Strings     the_locname;
   Strings     reverse_loc;
   Inp_Handler *tmp_handler;
   Strings     *pop_loc;
   Area_Dbase  *the_area;

   if ((the_area = the_builder->get_area()) == NULL)
   {
      the_builder->
           send_bldr("You should not see this!  Inform a coder immediately!\n");
      return -1;
   }

   the_locname = new_loc;
   if (the_locname.num_char('@') != 0) 
   {
      char *tmp_str;

      tmp_str = the_locname.str_show();
      while (*tmp_str != '@')
         tmp_str++;

      *tmp_str = '\0';
      tmp_str++;
      if (STRCASECMP(tmp_str, the_area->get_areaname()))
      {
         the_builder->send_bldr(
                "Autoassociate &+Rerror&*: Exit location in another area.\n"
                "You will need to associate the return exit manually.\n");
         return 0;
         
      }
   }

   if ((tmp_obj = the_area->get_areaobject(the_locname.str_show())) == NULL)
   {
      the_builder->send_bldr
                  ("Autoassociate &+Rerror&*: Could not find location '%s'\n",
                                                      the_locname.str_show());
      return 0;
   }

   switch(tmp_obj->get_type())
   {
      case OBJ_TYPE_DOOR:
         the_builder->send_bldr("Autoassociate doors not supported yet.\n");
         return 0;
         break;

      case OBJ_TYPE_LOCATION:
         reverse_loc.sprintf("%s@%s", get_name(), get_area());
         the_loc = (Location *) tmp_obj;
         break;

      default:
         the_builder->send_bldr
            ("Autoassociate &+Rerror&*: Object '%s' not a location or door.\n", 
                                                      the_locname.str_show());
         return 0;
   }

   tmp_handler = the_builder->get_input_handler();
   if (the_loc->get_exit(opposite_exit[the_exit]) != NULL)
   {
      the_builder->send_bldr
                  ("Return exit '%s' for location '%s@%s' already set to '%s'\n",
                    dir_str[opposite_exit[the_exit]], the_loc->get_name(), 
                the_loc->get_area(), the_loc->get_exit(opposite_exit[the_exit]));
      tmp_handler->push_input_handler(confirm_assoc_replace, 
                                    "Would you like to replace it? (y/n) ", 0);
      pop_loc = new Strings();
      pop_loc->sprintf("%s@%s", the_loc->get_name(), the_loc->get_area());
      tmp_handler->add_pop_function(replace_exit, 
                the_builder->get_name(), dir_str[opposite_exit[the_exit]], 
                reverse_loc.str_show(), pop_loc);

      return 1;
   }
   else
   {
      the_loc->set_exit(opposite_exit[the_exit], reverse_loc.str_show());
      the_builder->send_bldr("Exit %s on location %s set to %s.\n", 
            dir_str[opposite_exit[the_exit]], the_loc->get_name(),  
                                                       reverse_loc.str_show());
   } 
   return 1;
}


/***********************************************************************
 ** get_extra_dir - gets the extra directions from this room
 **
 ** Returns: a pointer to the smell string
 **
 ***********************************************************************/

char *Location::get_extra_dir()
{
   return extra_dir.str_show();
}


/***********************************************************************
 ** dir_to_num - converts the direction string to the direction number
 **
 ** Returns: a pointer to the smell string
 **
 ***********************************************************************/

int Location::dir_to_num(char *the_dir)
{
   int i;
   int found = 0;

   for (i=0; i<10; i++)
   {
      if (!STRCASECMP(dir_str[i], the_dir))
      {
         found = 1;
         break;
      }
   }   

   if (!found)
      return -1;

   return i;
}


/***********************************************************************
 ** short_dir_to_num - converts the direction string in abbreviated format
 **                    to the direction number
 **
 ** Returns: a pointer to the smell string
 **
 ***********************************************************************/

int Location::short_dir_to_num(char *the_dir)
{ 
   int i;

   if (*the_dir == 'n')
   {
      if (*(the_dir+1) == 'w')
         return NORTHWEST;
      else if (*(the_dir+1) == 'e')
         return NORTHEAST;
      else
         return NORTH;
   }
   else if (*the_dir == 's')
   {
      if (*(the_dir+1) == 'w')
         return SOUTHWEST;
      else if (*(the_dir+1) == 'e')
         return SOUTHEAST;
      else
         return SOUTH;
   }
   else if (*the_dir == 'e')
   {
      return EAST;
   }
   else if (*the_dir == 'w')
   {
      return WEST;
   }
   else if (*the_dir == 'u')
   {
      return UP;
   }
   else if (*the_dir == 'd')
   {
      return DOWN;
   }
   else
   {
      return -1;
   }  

   return i;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Location::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Location::get_mem_size_dynamic()
{
   int  size = 0;
   int  i;

   size += loc_flags->get_mem_size();

   for (i=0; i<10; i++)
   {
      size += dir[i].get_mem_size_dynamic();
   }

   size += extra_dir.get_mem_size_dynamic();

   size += extra_desc[0].get_mem_size_dynamic();
   size += extra_desc[1].get_mem_size_dynamic();

   size += listen.get_mem_size_dynamic();
   size += smell.get_mem_size_dynamic();

   size += get_mem_size_mudobj();
   size += get_mem_size_entity();

   return size;
}


/***********************************************************************
 ** confirm_assoc_replace - confirms that the builder wants to replace the
 **                         exit with a new exit
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int confirm_assoc_replace(MudObject *the_user, char *the_input) 
{
   Builder *the_builder;
   Strings builder_input;
   Inp_Handler *tmp_handler;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return -1;

   the_builder = (Builder *) the_user;

   tmp_handler = the_builder->get_input_handler();

   builder_input = the_input;
   builder_input.remove_newline();

   if (builder_input.str_len() <= 0)
   {
      the_builder->send_bldr("\nPlease select yes or no.\n\n");
      return -1;
   }

   if (!STRNCASECMP("yes", builder_input.str_show(), builder_input.str_len()))
   {
      tmp_handler->pop_input_handler();
      return 1; 
   }
   else if (!STRNCASECMP("no", builder_input.str_show(), 
                                                    builder_input.str_len()))
   {
      tmp_handler->remove_pop_function();
      tmp_handler->pop_input_handler();
      return 1; 
   }
   else
   {
      the_builder->send_bldr("\nPlease select yes or no.\n\n");
   }
   return -1;
}


/***********************************************************************
 ** replace_exit - changes the exit that is already there to the new
 **                exit.  Used as a pop function to execute when the
 **                player says yes they want to change it
 **
 ** Parameters: the_handler - the input handler this is a part of
 **             builder_name - the builder who is replacing the exit
 **             the_exit - the exit we are changing
 **             reverse_loc - the location we need to set the exit to for return
 **             return_loc - the location on which we need to set the exit of
 **
 ***********************************************************************/

int replace_exit(Inp_Handler *the_handler, char *builder_name, 
               char *the_exit, char *reverse_loc, Strings *return_loc)
{
   Builder    *the_builder;
   Strings    holder;   
   Entity     *tmp_obj;
   Location   *the_loc;
   Area_Dbase *the_area;
   int        exit_num;

   if ((the_builder = mainstruct->get_builder(builder_name)) == NULL)
   {
      holder.sprintf("Tried to get builder %s, not on.", builder_name);
      mainstruct->log_error(holder.str_show(), "replace_exit");
      return -1;
   }

   if ((the_area = the_builder->get_area()) == NULL)
   {
      mainstruct->log_error("Tried to get builder area, none found!", 
                                                             "replace_exit");
      return -1;
   }

   if (return_loc->num_char('@'))
   {
      char *tmp_char;
  
      tmp_char = return_loc->str_show();
      while (*tmp_char != '@')
         tmp_char++;

      *tmp_char = '\0';
   }

   if ((tmp_obj = the_area->get_areaobject(return_loc->str_show())) == NULL)
   {
      holder.sprintf("Return loc %s doesn't seem to exist in loaded area!", 
                                                 return_loc->str_show());
      mainstruct->log_error(holder.str_show(), "replace_exit");
      return -1;
   }

   if (tmp_obj->get_type() != OBJ_TYPE_LOCATION)
   {
      holder.sprintf("Return loc %s not a location!", 
                                                 return_loc->str_show());
      mainstruct->log_error(holder.str_show(), "replace_exit");
      return -1;
   }

   the_loc = (Location *) tmp_obj;

   exit_num = the_loc->dir_to_num(the_exit);

   the_loc->set_exit(exit_num, reverse_loc);
   the_builder->send_bldr("Exit %s on location %s set to %s.\n", 
                       dir_str[exit_num], the_loc->get_name(), reverse_loc);
   the_handler = NULL;
   return 1;
}


/***********************************************************************
 ** cleanup_marks - cleans up the marker settings set to 1 on locations 
 **                 back to 0
 **
 ** Parameters: distance - the distance to apply this special, based on
 **                        the size of subsequent rooms
 **             from_dir - which direction the special has come from
 **
 ** Returns: 1 for success, -1 for failed
 **
 ***********************************************************************/

int Location::cleanup_marks(int distance, int from_dir)
{
   int      dist_traveled;
   int      i;
   Location *new_loc;
   int      results = 0;

   /* if it is marked 1, we reset it */
   if (marked == 1)
      marked = 0;

   /* if they are casting from this room, we assume they stand in the middle
      so dist to the next room is only half the room size */
   if (from_dir == 10)
      dist_traveled = (room_size / 2);
   else
      dist_traveled = room_size;


   /* now we execute this function for surrounding rooms if applicable, but
      first see if we have enough distance to go furthur since it has to
      traverse the entire room (unless they are casting in this room) */   
   if (dist_traveled >= distance)
      return 1;

   for (i=0; i <= SOUTHWEST; i++)
   {   
      if ((i != from_dir) &&
          ((new_loc = get_exit(i, get_area(), &results)) != NULL))
      {
         new_loc->cleanup_marks((distance - dist_traveled), opposite_exit[i]);
      }
   }
   return 1;
}

/***********************************************************************
 ** apply_special - applies the special to a location and then if there
 **                 is still enough distance remaining, fans out from that
 **                 room as well
 **
 ** Parameters: the_user - the user running this special, if any
 **             the_special - the special to execute in this room
 **             distance - the distance to apply this special, based on
 **                        the size of subsequent rooms
 **             from_dir - which direction the special has come from
 **
 ** Returns: 1 for success, 0 for not run, -1 for failed
 **
 ***********************************************************************/

int Location::apply_special(Individual *the_user, Specials *the_special, 
                                                int distance, int from_dir)
{
   int         dist_traveled;
   int         i;
   special_env environment;
   in_params   fill_param;
   Location    *new_loc;
   int         results = 0;

   /* if the special has already been run here, don't run it again */
   if (marked)
      return 0;

   /* if they are casting from this room, we assume they stand in the middle
      so dist to the next room is only half the room size */
   if (from_dir == 10)
      dist_traveled = (room_size / 2);
   else
      dist_traveled = room_size;


   /* here we apply the special, if this is not the first room to get this */
   if (from_dir != 10)
   {
      fill_param.primary_obj = this;
      fill_param.secondary_obj = NULL;
      fill_param.this_obj = the_user;

      environment.trig_used = "onapply";
      environment.target_str = fromdir_str[from_dir];
      environment.exec_vars = NULL;
      the_special->run_special(the_user, &fill_param, &environment);
   }

   marked = 2;

   /* now we execute this function for surrounding rooms if applicable, but
      first see if we have enough distance to go furthur since it has to
      traverse the entire room (unless they are casting in this room) */   
   if (dist_traveled >= distance)
   {
      marked = 1;
      return 1;
   }

   for (i=0; i <= SOUTHWEST; i++)
   {   
      if ((i != from_dir) &&
          ((new_loc = get_exit(i, get_area(), &results)) != NULL))
      {
         new_loc->apply_special(the_user, the_special, 
                               (distance - dist_traveled), opposite_exit[i]);
      }
   }
   marked = 1;
   return 1;
}


#endif








