/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.treedisplay;

import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import java.awt.Point;
import java.util.Iterator;

/**
 * Defines an interface for that classes wanting to layout trees for
 * TreeDisplay should implement. Uses a TreeModel as the tree to
 * layout. Classes that want to listen for changes in the layout
 * should implement the {@link javax.swing.event.TreeModelListener
 * TreeModelListener} interface and register themselves with layout
 * objects using the {@link #addTreeModelListener} method.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */
public interface TreeLayoutModel extends TreeModelListener {
	/**
	 * Get the model that is being layed out.
	 *
	 * @return a <code>TreeModel</code> value
	 */
	public TreeModel getModel();

	/**
	 * Set the tree to be layed out.
	 *
	 * @param m a <code>TreeModel</code> value
	 */
	public void setModel(TreeModel m);

	/**
	 * Get the class used as the TreeLayoutNode.
	 *
	 * @return a <code>Class</code> value
	 */
	public Class getTreeLayoutNodeClass();

	/**
	 * Set the TreeLayoutNode class that should be used when doing
	 * layouts.
	 *
	 * @param c a <code>Class</code> value
	 */
	public void setTreeLayoutNodeClass(Class c);

	/**
	 * Create a tree layout node.
	 */
	public TreeLayoutNode createNode(Object o);

	/**
	 * Set the orientation of the tree.
	 */
	public void setOrientation(int o) throws TreeDisplayException;

	/**
	 * Get the orientation of the tree.
	 */
	public int getOrientation();

	/**
	 * Is the given orientation supported?
	 */
	public boolean supportsOrientation(int o);

	/**
	 * Perform a layout.
	 */
	public void layout();

	/**
	 * Get the root node of the layout model.
	 *
	 * @return a <code>TreeLayoutNode</code> value
	 */
	public TreeLayoutNode getRoot();

	/**
	 * Get the width of the layout model.
	 *
	 * @return an <code>int</code> value
	 */
	public int getWidth();

	/**
	 * Get the height of the layout mode.
	 *
	 * @return an <code>int</code> value
	 */
	public int getHeight();

	/**
	 * Get the layout node that contains the point (x, y).
	 *
	 * @param x an <code>int</code> value
	 * @param y an <code>int</code> value
	 * @return a <code>TreeLayoutNode</code> value
	 */
	public TreeLayoutNode getNode(int x, int y);

	/**
	 * Get the layout node that contains the Point p.
	 *
	 * @param p a <code>Point</code> value
	 * @return a <code>TreeLayoutNode</code> value
	 */
	public TreeLayoutNode getNode(Point p);

	/**
	 * Get the layout node that contains the Object data.
	 *
	 * @param data an <code>Object</code> value
	 * @return a <code>TreeLayoutNode</code> value
	 */
	public TreeLayoutNode getNode(Object data);

	/**
	 * Get an Iterator that can be used for iterating over all
	 * layout nodes.
	 *
	 * @return a <code>Iterator</code> value
	 */
	public Iterator getNodes();

	/**
	 * Get an Iterator that can be used for iterating over all layout
	 * edges.
	 *
	 * @return a <code>Iterator</code> value
	 */
	public Iterator getEdges();

	/**
	 * Get the TreePath that correponds to the given TreeLayoutNode.
	 *
	 * @param node a <code>TreeLayoutNode</code> value
	 * @return a <code>TreePath</code> value
	 */
	public TreePath getTreePath(TreeLayoutNode node);

	/**
	 * Add a TreeModelListener. Objects using a TreeLayoutModel can
	 * add themselves as a listener and will then be notified whenever
	 * the layout changes.
	 *
	 * @param l a <code>TreeModelListener</code> value
	 */
	public void addTreeModelListener(TreeModelListener l);

	/**
	 * Remove a TreeModelListener.
	 *
	 * @param l a <code>TreeModelListener</code> value
	 */
	public void removeTreeModelListener(TreeModelListener l);

	//
	// methods implementing TreeModelListener
	//

	/**
	 * {@link javax.swing.event.TreeModelListener#treeNodesChanged}
	 *
	 * @param e a <code>TreeModelEvent</code> value
	 */
	public void treeNodesChanged(TreeModelEvent e);

	/**
	 * {@link javax.swing.event.TreeModelListener#treeNodesInserted}
	 *
	 * @param e a <code>TreeModelEvent</code> value
	 */
	public void treeNodesInserted(TreeModelEvent e);

	/**
	 * {@link javax.swing.event.TreeModelListener#treeNodesRemoved}
	 *
	 * @param e a <code>TreeModelEvent</code> value
	 */
	public void treeNodesRemoved(TreeModelEvent e);

	/**
	 * {@link javax.swing.event.TreeModelListener#treeStructureChanged}
	 *
	 * @param e a <code>TreeModelEvent</code> value
	 */
	public void treeStructureChanged(TreeModelEvent e);
}// TreeLayoutModel
