/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.explain;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.sql.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;

import com.redhat.rhdb.vise.PlannerOptions;

/**
 * The <code>Explain</code> class provides a nice interface to
 * PostgreSQL's EXPLAIN command.
 *
 * @author Liam Stewart
 * @author <a href="mailto:fnasser@redhat.com">Fernando Nasser</a>
 * @version 1.2.0
 */
public class Explain extends Observable implements Serializable
{
	// ATTENTION:
	// Make sure your changes are backward compatible.
	// Otherwise change the fingerprint, but them VISE will
	// refuse to read plans saved with previous versions
	static final long serialVersionUID = -480744932039659764L;

	private String query;		// The query (to be) examined
	private String popts;		// Planner option set commands used
	private String plan;		// A summary of the planner choosen plan in ascii
	private String dump;		// Dump of the plan tree in ascii
	private ExplainTree tree;	// Parsed plan tree (which is displayed)
	
	private transient ExplainComponent expcomp;
	private transient EventListenerList listenerList;

	// Temporary data for sharing among threads,
	// cancelling operations, etc.	
	private transient Connection con;
	private transient boolean analyze;
	private transient ExplainParser exp;
	private transient PlannerOptions newpopts;

	/**
	 * Creates a new <code>Explain</code> instance.
	 */
	public Explain() {
		popts = query = plan = dump = "";
		tree = null;
		newpopts = null;
		listenerList = new EventListenerList();
		expcomp = new ExplainComponent(this);
	}

	/**
	 * Creates a new <code>Explain</code> instance.
	 *
	 * @param query the query to explain
	 */
	public Explain(String query) {
		this();
		this.query = query;
	}

	/**
	 * Loads a new <code>Explain</code> instance.
	 */
	private void readObject(ObjectInputStream in)
	throws IOException, ClassNotFoundException {
		// This loads query, plan, dump and tree
		in.defaultReadObject();
		// Now we initialize the rest ourselves
		listenerList = new EventListenerList();
		expcomp = new ExplainComponent(this);
        // For compatibility with older save plans
        if (popts == null)
            popts = "";
    }

	/**
	 * Get the query to explain.
	 *
	 * @return a <code>String</code> value
	 */
	public String getQuery() {
		return query;
	}
	
	/**
	 * Set the query to explain. A side-effect of setting the
	 * query is the invalidation of current plan, dump, and ExplainTree
	 * (i.e.: if their accessors did not return null previously, they
	 * will return null until explain() is called).
	 *
	 * @param q a <code>String</code> value
	 */
	public void setQuery(String q) {
		invalidate();

		this.query = q;

		changed();
	}

	/**
	 * Set planner options to be set before the query is run
	 *
	 * @param q a <code>PlannerOptions</code> object
	 */
	 public void setPlannerOptions(PlannerOptions opts) {
	 	// This is saved for the next run,
		// when we do update the one associated
		// with the computed plan
	 	this.newpopts = opts;
	}

	/**
	 * Gets the planner option set commands used to obtain the plan.
	 *
	 * @return a <code>String</code> value
	 */
	public String getPlannerOptions() {
		return popts;
	}

	/**
	 * Gets the new planner option set commands to be used when
	 * explain is executed.
	 *
	 * @return a <code>String</code> value
	 */
	public String getNewPlannerOptions() {
		return newpopts.toString();
	}

	/**
	 * Gets the set commands to be used to restore the defaults
	 *
	 * @return a <code>String</code> value
	 */
	public String getResetCommands() {
		return newpopts.getResetCommands();
	}

	/**
	 * Gets the normal EXPLAIN output.
	 *
	 * @return a <code>String</code> value
	 */
	public String getOutputExplain() {
		return plan;
	}

	/**
	 * Gets the EXPLAIN VERBOSE output.
	 *
	 * @return a <code>String</code> value
	 */
	public String getOutputExplainVerbose() {
		return dump;
	}

	/**
	 * Gets the ExplainTree created by calling the explain method.
	 *
	 * @return an <code>ExplainTree</code> value
	 */
	public ExplainTree getExplainTree() {
		return tree;
	}

	/**
	 * Run an actual SQL EXPLAIN (ie: send an EXPLAIN to the backend).
	 * An ExplainTree is generated as the result of the execution of
	 * this method; it can be obtained at a later time by calling the
	 * getExplainTree method. Doesn't try EXPLAIN ANALYZE the query.
	 *
	 * @param con The connection to use for sending the EXPLAIN
	 * command.
	 * @return nothing. Use the getExplainTree operation to
	 * retrieve the tree produced by invoking this method.
	 *
	 * @exception ExplainException if an error occurs
	 */
	public void explain(Connection con) throws ExplainException
	{
		/* don't try to explain analyze by default */
		explain(con, false);
	}

	/**
	 * Run an actual SQL EXPLAIN (ie: send an EXPLAIN to the backend).
	 * An ExplainTree is generated as the result of the execution of
	 * this method; it can be obtained at a later time by calling the
	 * getExplainTree method. If an EXPLAIN ANALYZE is asked for and
	 * is possible, the Connection will be taken out of auto-commit
	 * mode. If it was already out of auto-commit mode, a rollback
	 * will be performed. A rollback will be performed after the
	 * explanation since EXPLAIN ANALYZE actually <i>runs</i> the
	 * query and the auto-commit state will be restored.
	 *
	 * @param con The connection to use for sending the EXPLAIN
	 * command.
	 * @param analyze Try to EXPLAIN ANALYZE the query?
	 * @return nothin. Use the getExplainTree method to retrieve
	 * the explain tree generated by invoking this method.
	 *
	 * @exception ExplainException if an error occurs
	 */
	public void explain(Connection con, boolean analyze) throws ExplainException {

		if (con == null || query == null)
			throw new ExplainException("Null connection or query");
			
		try {
			if (con.isClosed())
				throw new ExplainException("Connection is closed");
						
			String newquery = checkQuery();
			if (newquery.trim().length() == 0)
				throw new ExplainException("Empty query");
			setQuery(newquery);

			DatabaseMetaData dbmd = con.getMetaData();
			String dbver = dbmd.getDatabaseProductVersion();

			if (dbver.startsWith("7.0") ||
				dbver.startsWith("7.1") ||
				dbver.startsWith("7.2") ||
				dbver.startsWith("7.3") ||
				dbver.startsWith("7.4"))
			{
				// Older versions do not have ANALYZE
				if ((dbver.startsWith("7.0") || dbver.startsWith("7.1")) && analyze)
				{
					throw new ExplainException("ANALYZE option of EXPLAIN unsupported on this database version");
				}
				
				// Mark application as running
				ExplainEvent runningEvent = new ExplainEvent(expcomp, "running");
				EventQueue queue = Toolkit.getDefaultToolkit().getSystemEventQueue();
				queue.postEvent(runningEvent);
				
				// save arguments for thread use
				this.con = con;
				this.analyze = analyze;
			
				ExplainThread t = new ExplainThread(Explain.this);
				t.start();
			} else {
				throw new ExplainException("Unsupported database version");
			}
		} catch (SQLException ex) {
			throw new ExplainException("SQL error: " + ex.getMessage());
		}
	}
	
	class ExplainThread extends Thread {
		Explain explain;

		public ExplainThread(Explain explain) {
			this.explain = explain;
		}
		
		public void run() {
			
			try {
				DatabaseMetaData dbmd = explain.con.getMetaData();
				String dbver = dbmd.getDatabaseProductVersion();

				if (dbver.startsWith("7.0") ||
					dbver.startsWith("7.1") ||
					dbver.startsWith("7.2"))
				{
					explain.exp = new ExplainParserV7(explain);
					explain.exp.explain(explain.con, explain.analyze);
					changed();
				}
				else if (dbver.startsWith("7.3") ||
						dbver.startsWith("7.4"))
				{
					explain.exp = new ExplainParserV73(explain);
					explain.exp.explain(explain.con, explain.analyze);
					changed();
				}
			} catch (SQLException ex) {
				// We can't throw an exception from a thread, so we generate
				// an error event instead
				ExplainEvent errorEvent = new ExplainEvent(expcomp, "error",
								new ExplainException("SQL error: " + ex.getMessage()));
				EventQueue queue = Toolkit.getDefaultToolkit().getSystemEventQueue();
				queue.postEvent(errorEvent);
			} catch (ExplainException ex) {
				// Try a rollback to recover the connection
				try {
					con.rollback();
				} catch (SQLException sqlex) {
				}
				// We can't throw an exception from a thread, so we generate
				// an error event instead
				ExplainEvent errorEvent = new ExplainEvent(expcomp, "error", ex);
				EventQueue queue = Toolkit.getDefaultToolkit().getSystemEventQueue();
				queue.postEvent(errorEvent);
			} finally {
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						// Mark application as idle
						ExplainEvent idleEvent = new ExplainEvent(expcomp, "idle");
						EventQueue queue = Toolkit.getDefaultToolkit().getSystemEventQueue();
						queue.postEvent(idleEvent);
					}
				});
			}
		}
	}//ExplainThread
	
	/**
	 * Adds a explain listener
	 * @param listener the listener to add
	 */
	 public void addExplainListener(ExplainListener listener)
	 {
	 	listenerList.add(ExplainListener.class, listener);
	 }
	
	/**
	 * Removes a explain listener
	 * @param listener the listener to remove
	 */
	 public void removeExplainListener(ExplainListener listener)
	 {
	 	listenerList.remove(ExplainListener.class, listener);
	 }
	 
	 class ExplainComponent extends JComponent {
		Explain explain;

		public ExplainComponent(Explain explain) {
			this.explain = explain;
		}
	 
	 	public void processEvent(AWTEvent event)
	 	{
	 		if (event instanceof ExplainEvent)
			{
	 			EventListener[] listeners = explain.listenerList.getListeners(ExplainListener.class);
				
				for (int i = 0; i < listeners.length; i++)
					((ExplainListener)listeners[i]).statusChanged((ExplainEvent)event);
			} else {
				super.processEvent(event);
			}
		}
	}//ExplainComponent
	
	public void cancel() {
		exp.cancel();
	}

	//
	// private methods
	//
	
	/**
	 * Sets the ExplainTree.
	 *
	 * @param s an <code>ExplainTree</code> value
	 */
	protected void setExplainTree(ExplainTree s) {
		tree = s;
	}

	/**
	 * Sets the planner options used.
	 *
	 */
	protected void setOptions() {
		// Now the planner options requested
		// have indeed been used
		popts = newpopts.toString();
	}

	/**
	 * Sets the query dump.
	 *
	 * @param s a <code>String</code> value
	 */
	protected void setDump(String s) {
		dump = s;
	}
	
	/**
	 * Sets the raw plan tree.
	 *
	 * @param s a <code>String</code> value
	 */
	protected void setPlan(String s) {
		plan = s;
	}
	
	/* invalidate the ExplainTree, the dump, and the plan */
	private void invalidate() {
		tree = null;
		dump = null;
		plan = null;
		popts = "";
	}

	/* set our changed flag (inherited from Observable) and notify
	 * observers that we've changed */
	private void changed() {
		setChanged();
		notifyObservers();
	}

	/* check the query for sane-ness */
	private String checkQuery() throws ExplainException
	{
		StringBuffer buf = new StringBuffer(query);
		StringTokenizer strtok, strtok2;
		String str, str2, q;
		boolean done = false;
		int i;
		char ch;

		// do comment removal until got first query or end of text
		final int NORMAL = 0;
		final int STRING = 1;
		final int COMMENT_C = 2;
		final int COMMENT_SQL = 3;

		int level = 0;
		int start = -1;
		int mode = NORMAL;
		int cut = -1;
		boolean gotcomment = false;
		i = 0;

		// do comment removal until get to end of first query
		while (i < buf.length() && !done)
		{
			ch = buf.charAt(i);

			switch (ch)
			{
				case '-':
					if (i+1 < buf.length() &&
						buf.charAt(i+1) == '-'
						&& mode == NORMAL)
					{
						mode = COMMENT_SQL;
						start = i;
						i += 2;
					}
					else
					{
						i++;
					}
					break;
				case '\n':
					// fall through
				case '\r':
					if (mode == COMMENT_SQL)
					{
						// delete all but the newline
						buf.delete(start, i);
						mode = NORMAL;
						i = start + 1;
					}
					else
					{
						i++;
					}
					break;
				case '/':
					if (i+1 < buf.length() &&
						buf.charAt(i+1) == '*' &&
						(mode == NORMAL ||
						 mode == COMMENT_C))
					{
						mode = COMMENT_C;
						if (level == 0)
							start = i;
						level++;
						i += 2;
					}
					else
					{
						i++;
					}
					break;
				case '*':
					if (i+1 < buf.length() &&
						buf.charAt(i+1) == '/' &&
						mode == COMMENT_C)
					{
						i += 2;
						level--;
						if (level == 0)
						{
							buf.delete(start, i);
							i = start;
							mode = NORMAL;
						}
					}
					else
					{
						i++;
					}
					break;
				case '\'':
					if (mode == NORMAL)
					{
						mode = STRING;
					}
					else if (mode == STRING)
					{
						if (buf.charAt(i-1) != '\\' ||
							(buf.charAt(i-1) == '\\' &&
							 buf.charAt(i-2) == '\\'))
							mode = NORMAL;
					}
					i++;
					break;
				case ';':
					if (mode == NORMAL)
					{
						cut = i + 1;
						done = true;
					}
					i++;
					break;
				default:
					i++;
			}
		}

		if (mode == COMMENT_SQL)
		{
			buf = buf.delete(start, buf.length());
		}

		if (cut != -1)
		{
			q = buf.substring(0, cut);
		}
		else
		{
			q = buf.toString();
		}

		// unbalanced c-style comments -- let PostgreSQL deal with it
		if (level != 0)
		{
			return query;
		}

		// is the query not a SELECT, UPDATE, INSERT, or DELETE?
		strtok = new StringTokenizer(q);
		if (strtok.countTokens() >= 1)
		{
			while (strtok.hasMoreTokens() && !done)
			{
				str = strtok.nextToken();
				strtok2 = new StringTokenizer(str, "(", true);
				while (strtok2.hasMoreTokens())
				{
					str2 = strtok2.nextToken();

					if (str2.equals("("))
					{
						continue;
					}
					else if (str2.equalsIgnoreCase("select") ||
							 str2.equalsIgnoreCase("update") ||
							 str2.equalsIgnoreCase("insert") ||
							 str2.equalsIgnoreCase("delete"))
					{
						done = true;
					}
					else
					{
						throw new ExplainException("Statement must be a SELECT, UPDATE, INSERT, or DELETE.");
					}
				}
			}
		}

		return q;
	}
}// Explain
