/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
*/

package com.sap.dbtech.jdbc.translators;

import java.math.*;
import java.sql.*;
import com.sap.dbtech.util.*;
import com.sap.dbtech.vsp00.*;
import com.sap.dbtech.jdbc.DriverSapDB;
import com.sap.dbtech.jdbc.exceptions.*;
import java.text.*;
import java.util.*;

/**
 *
 */
class NumericTranslator
    extends BinaryDataTranslator
{
    protected int frac;
    protected boolean isFloatingPoint = false;
    private static DecimalFormat doubleInputFormat;

    /**
     * NumericTranslator constructor comment.
     * @param mode int
     * @param ioType int
     * @param len int
     * @param ioLen int
     * @param bufpos int
     */
    protected
        NumericTranslator(
        int mode,
        int ioType,
        int dataType,
        int len,
        int frac,
        int ioLen,
        int bufpos)
    {
        super(mode, ioType, dataType, len, ioLen, bufpos);
        switch (dataType) {
            case DataType.Dfloat:
            case DataType.Dvfloat:{
                /* more digits are unreliable anyway */
                frac = 38;
                isFloatingPoint=true;
                break;
            }

        }
        this.frac = frac;
    }
    /**
     *
     * @return java.math.BigDecimal
     * @param scale int
     */
    public java.math.BigDecimal getBigDecimal (
        int scale,
        SQLParamController controller,
        StructuredMem mem) throws SQLException
    {
        BigDecimal result = null;
        try {
          switch (this.checkDefineByte (controller, mem)) {
            case (DBTechTranslator.nullDefineByte):{
              return result;
            }
            case (DBTechTranslator.specialNullValueDefineByte):{
              throw new SQLExceptionSapDB( MessageTranslator.translate(MessageKey.ERROR_CONVERSIONSpecialNullValue),"", -10811);
            }
          }
          result = VDNNumber.number2BigDecimal (
              mem.getBytes (this.bufpos, this.physicalLength - 1));
          result = result.setScale ( scale);
          return result;
        } catch(java.lang.ArithmeticException exc){
          throw this.newParseException (result+" scale: "+scale, null);
        }
    }

    /**
     *
     * @return java.math.BigDecimal
     * @param scale int
     */
    public java.math.BigDecimal getBigDecimal (
        SQLParamController controller,
        StructuredMem mem) throws SQLException
    {
        BigDecimal result = null;
        try {
          switch (this.checkDefineByte (controller, mem)) {
            case (DBTechTranslator.nullDefineByte):{
              return result;
            }
            case (DBTechTranslator.specialNullValueDefineByte):{
              throw new SQLExceptionSapDB( MessageTranslator.translate(MessageKey.ERROR_CONVERSIONSpecialNullValue),"", -10811);
            }
          }
          result = VDNNumber.number2BigDecimal (
                  mem.getBytes (this.bufpos, this.physicalLength - 1));
          if (! isFloatingPoint)
            result = result.setScale ( this.frac);
          return result;
        } catch(java.lang.ArithmeticException exc){
          throw this.newParseException (result+" scale: "+this.frac, null);
        }
    }

    /**
     *
     */
    public boolean getBoolean (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        int intResult = this.getInt (controller, mem);
        return intResult != 0;
    }
    /**
     *
     * @return java.sql.Date
     */
    public byte getByte (
        SQLParamController controller,
        StructuredMem mem) throws SQLException
    {
        return (byte) this.getLong (controller, mem);
    }
    /**
     *
     * @return java.sql.Date
     */
    public double getDouble (
        SQLParamController controller,
        StructuredMem mem) throws SQLException
    {
        switch (this.checkDefineByte (controller, mem)) {
          case (DBTechTranslator.nullDefineByte):{
            return 0;
          }
          case (DBTechTranslator.specialNullValueDefineByte):{
            return Double.NaN;
          }
        }
        return this.bigDecimal2Double (
                 VDNNumber.number2BigDecimal (
                   mem.getBytes (this.bufpos, this.physicalLength - 1)
                 )
               );
    }

    /**
     *
     * @return java.sql.Date
     */
    public float getFloat (
        SQLParamController controller,
        StructuredMem mem) throws SQLException
    {
        switch (this.checkDefineByte (controller, mem)) {
          case (DBTechTranslator.nullDefineByte):{
            return 0;
          }
          case (DBTechTranslator.specialNullValueDefineByte):{
            return Float.NaN;
          }
        }
        return this.bigDecimal2Float (
                 VDNNumber.number2BigDecimal (
                   mem.getBytes (this.bufpos, this.physicalLength - 1)
                 )
               );
    }
    /**
     *
     * @return java.sql.Date
     */
    public int getInt (
        SQLParamController controller,
        StructuredMem mem) throws SQLException
    {
        return (int) this.getLong (controller, mem);
    }
    /**
     *
     * @return java.sql.Date
     */
    public long getLong (
        SQLParamController controller,
        StructuredMem mem)
    throws SQLException
    {
        long result = 0;
        byte [] number;

        switch (this.checkDefineByte (controller, mem)) {
          case (DBTechTranslator.nullDefineByte):{
            return result;
          }
          case (DBTechTranslator.specialNullValueDefineByte):{
            throw new SQLExceptionSapDB( MessageTranslator.translate(MessageKey.ERROR_CONVERSIONSpecialNullValue),"", -10811);
          }
        }
        number = mem.getBytes (this.bufpos, this.physicalLength - 1);
        result = VDNNumber.number2long (number);
        return result;
    }
    /**
     *
     * @return java.sql.Object
     */
    public Object getObject (
        SQLParamController controller,
        StructuredMem mem) throws SQLException
    {
        if(this.isNull(controller, mem)) {
            return null;
        }
        switch (this.dataType) {
        case DataType.Dfloat:
            if(this.logicalLength <15)
                return new Float(this.getFloat(controller, mem));
            if (this.logicalLength < 16)
                return new Double(Float.toString(this.getFloat (controller, mem)));
            else
                return new Double(this.getDouble (controller, mem));
        case DataType.Dinteger:
            return new Integer(this.getInt(controller, mem)); // isnull catched before
        default:
            return this.getBigDecimal (controller, mem);
      }
    }
    /**
     *
     * @return int
     */
    public int
        getPrecision ()
    {
        int result;

        switch (this.dataType) {
            case DataType.Dfixed:
                result = this.logicalLength;
                break;
            default:
                result = this.logicalLength;
                break;
        }
        return result;
    }
    /**
     *
     * @return int
     */
    public int
        getScale ()
    {
        int result;

        switch (this.dataType) {
            case DataType.Dfixed:
                result = this.frac;
                break;
            case DataType.Dsmallint:
            case DataType.Dinteger:
                result = 0;
                break;
            default:
                result = 0;
                break;
        }
        return result;
    }
    /**
     *
     * @return java.sql.Date
     */
    public short getShort (
        SQLParamController controller,
        StructuredMem mem) throws SQLException
    {
        return (short) this.getLong (controller, mem);
    }
    /**
     *
     * @param value double
     */
    public Object transBigDecimalForInput (
        BigDecimal value)
    {
        if (value == null) {
            return null;
        }
        else {
            byte [] number;

            value.setScale (this.frac, BigDecimal.ROUND_HALF_DOWN);
            number = VDNNumber.bigDecimal2number (value);
            return number;
        }
    }
    /**
     *
     * @param value double
     */
    public Object transDoubleForInput (
        double value)
    throws SQLException
    {
        try {

            BigDecimal bigD=new BigDecimal(doubleInputFormat.format(value));
            byte [] number;

            if (this.dataType == com.sap.dbtech.vsp00.DataType.Dfixed){
                bigD = bigD.setScale (this.frac, BigDecimal.ROUND_HALF_DOWN);
            }
            number = VDNNumber.bigDecimal2number (bigD, 16);
            return number;
        } catch(NumberFormatException numberFormatEx) {
            // Detect nasty double values, and throw an SQL exception.
            if(Double.isInfinite(value) || Double.isNaN(value)) {
                throw new SQLExceptionSapDB(MessageTranslator.translate
                                        (MessageKey.ERROR_SPECIAL_NUMBER_UNSUPPORTED,
                                         Double.toString(value)));
            } else {
                throw numberFormatEx;
            }
        }
    }



    /**
     *
     * @param value double
     */
    public Object transFloatForInput (
        float value)
    throws SQLException
    {
        try {
            byte [] number;
            BigDecimal bigD;
            bigD = new BigDecimal (doubleInputFormat.format((double)value));
            if (this.dataType == com.sap.dbtech.vsp00.DataType.Dfixed){
                bigD = bigD.setScale (this.frac, BigDecimal.ROUND_HALF_DOWN);
            }
            number = VDNNumber.bigDecimal2number (bigD, 14);
            return number;
        } catch(NumberFormatException numberFormatEx) {
            // Detect nasty double values, and throw an SQL exception.
            if(Float.isInfinite(value) || Float.isNaN(value)) {
                throw new SQLExceptionSapDB(MessageTranslator.translate
                                        (MessageKey.ERROR_SPECIAL_NUMBER_UNSUPPORTED,
                                         Float.toString(value)));
            } else {
                throw numberFormatEx;
            }

        }
    }
    /**
     *
     * @param value double
     */
    public Object transIntForInput (
        int value)
    throws SQLException
    {
        byte [] number;

        number = VDNNumber.long2number (value);
        return number;
    }
    /**
     *
     * @param value double
     */
    public Object transLongForInput (
        long value)
    throws SQLException
    {
        byte [] number;

        number = VDNNumber.long2number (value);
        return number;
    }
    /**
     *
     * @param value double
     */
    public Object transShortForInput (
        short value)
    throws SQLException
    {
        byte [] number;

        number = VDNNumber.long2number (value);
        return number;
    }
    /**
     *
     * @param obj java.lang.Object
     */
    public Object transSpecificForInput (
        Object obj)
    throws SQLException
    {
        Object result = null;

        if (obj == null) {
            return null;
        }
        if (obj instanceof BigDecimal) {
            result = this.transBigDecimalForInput ((BigDecimal) obj);
        }
        else if (obj instanceof Boolean) {
            result = this.transBooleanForInput (((Boolean) obj).booleanValue());
        }
        else if (obj instanceof Byte) {
            result = this.transByteForInput (((Byte) obj).byteValue ());
        }
        else if (obj instanceof Double) {
            result = this.transDoubleForInput (((Double) obj).doubleValue ());
        }
        else if (obj instanceof Float) {
            result = this.transFloatForInput (((Float) obj).floatValue ());
        }
        else if (obj instanceof Integer) {
            result = this.transIntForInput (((Integer) obj).intValue ());
        }
        else if (obj instanceof Long) {
            result = this.transLongForInput (((Long) obj).longValue ());
        }
        else if (obj instanceof Short) {
            result = this.transShortForInput (((Short) obj).shortValue ());
        }
        return result;
    }

    public String getString (
            SQLParamController controller,
            StructuredMem mem) throws SQLException
    {
        switch (this.checkDefineByte (controller, mem)) {
          case (DBTechTranslator.nullDefineByte):{
            return null;
          }
          case (DBTechTranslator.specialNullValueDefineByte):{
            return ("NaN");
          }
        }
        return VDNNumber.number2string
            (mem.getBytes (this.bufpos, this.physicalLength - 1),
             (dataType!=DataType.Dfloat && dataType!=DataType.Dvfloat),
             this.logicalLength,
             this.frac);
    }


    /**
     *
     * @param value double
     */
    public Object transStringForInput (
        String value)
    throws SQLException
    {
        BigDecimal decimal;

        if (value == null) {
            return null;
        }
        try {
            decimal = new BigDecimal (value.trim ());
        }
        catch (NumberFormatException exc) {
            throw this.newParseException (value, null);
        }
        catch (StringIndexOutOfBoundsException exc) {
            throw this.newParseException (value, null);
        }
        return this.transBigDecimalForInput (decimal);
    }

    static {
        DecimalFormatSymbols dfs=new DecimalFormatSymbols(Locale.US);
        if (DriverSapDB.currentVMVersion <= DriverSapDB.JAVA_1_2)
          doubleInputFormat=new DecimalFormat("###############################################################.000000000000000000000000000000000000000000000000000000000000000", dfs);
        else
          doubleInputFormat=new DecimalFormat("0.0000000000000000000000000000000000000E00", dfs);
        doubleInputFormat.setGroupingSize(Integer.MAX_VALUE);
    }
}
