﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/logs/CloudWatchLogs_EXPORTS.h>
#include <aws/logs/model/OpenSearchResourceStatus.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace CloudWatchLogs {
namespace Model {

/**
 * <p>This structure contains information about the OpenSearch Service data source
 * used for this integration. This data source was created as part of the
 * integration setup. An OpenSearch Service data source defines the source and
 * destination for OpenSearch Service queries. It includes the role required to
 * execute queries and write to collections.</p> <p>For more information about
 * OpenSearch Service data sources , see <a
 * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/direct-query-s3-creating.html">Creating
 * OpenSearch Service data source integrations with Amazon S3.</a> </p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/OpenSearchDataSource">AWS
 * API Reference</a></p>
 */
class OpenSearchDataSource {
 public:
  AWS_CLOUDWATCHLOGS_API OpenSearchDataSource() = default;
  AWS_CLOUDWATCHLOGS_API OpenSearchDataSource(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLOUDWATCHLOGS_API OpenSearchDataSource& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLOUDWATCHLOGS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the OpenSearch Service data source.</p>
   */
  inline const Aws::String& GetDataSourceName() const { return m_dataSourceName; }
  inline bool DataSourceNameHasBeenSet() const { return m_dataSourceNameHasBeenSet; }
  template <typename DataSourceNameT = Aws::String>
  void SetDataSourceName(DataSourceNameT&& value) {
    m_dataSourceNameHasBeenSet = true;
    m_dataSourceName = std::forward<DataSourceNameT>(value);
  }
  template <typename DataSourceNameT = Aws::String>
  OpenSearchDataSource& WithDataSourceName(DataSourceNameT&& value) {
    SetDataSourceName(std::forward<DataSourceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This structure contains information about the status of this OpenSearch
   * Service resource.</p>
   */
  inline const OpenSearchResourceStatus& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = OpenSearchResourceStatus>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = OpenSearchResourceStatus>
  OpenSearchDataSource& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dataSourceName;

  OpenSearchResourceStatus m_status;
  bool m_dataSourceNameHasBeenSet = false;
  bool m_statusHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchLogs
}  // namespace Aws
