#!/usr/bin/env python
# -*- coding: utf8 -*-

"""
    This file as a hole is published under the MIT license:

    ✂----------------------------------------------------------------------

    The MIT License (MIT)

    Copyright (c) 2013-2016 Ned Batchelder, Stephan Hennig, Werner Lemberg,
    Guenter Milde, Sander van Geloven, Georg Pfeiffer, Gisbert W. Selke,
    Tobias Wendorf, Lukas Sommer.

    Permission is hereby granted, free of charge, to any person obtaining
    a copy of this software and associated documentation files (the
    "Software"), to deal in the Software without restriction, including
    without limitation the rights to use, copy, modify, merge, publish,
    distribute, sublicense, and/or sell copies of the Software, and to
    permit persons to whom the Software is furnished to do so, subject to
    the following conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
    OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
    WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

    ✂----------------------------------------------------------------------

    This file contains public domain code from Ned Batchelder:
    “Ned Batchelder, July 2007. This Python code is in the public domain.”
    See the Hyphenator class for details. Modifications from Lukas
    Sommer (2016), published under the MIT license.

    It contains patterns that are derived from the Trennmuster project,
    that publishes its files under the MIT license.
    Copyright (c) 2013-2014 Stephan Hennig, Werner Lemberg, Guenter Milde,
    Sander van Geloven, Georg Pfeiffer, Gisbert W. Selke, Tobias Wendorf.
    See the GermanLigatureSupport class for details.

    The rest of the code is from Lukas Sommer (2016), published under
    the MIT license.
"""

import re
import scribus

# This file is in UTF8. It uses also really quite some characters outside
# ASCII and even outside BMP. As sometimes the encoding makes problems
# (Scribus script console for example), we assert here that the encoding
# is interpreted correctly
if ord("a") != 97:
    raise RuntimeError("Source code encoding problem 01.")

if len("a") != 1:
    raise RuntimeError("Source code encoding problem 02.")

if ord(u"\xE4") != 0xE4:
    raise RuntimeError("Source code encoding problem 03.")

if len(u"\xE4") != 1:
    raise RuntimeError("Source code encoding problem 04.")

if ord(u"\u00E4") != 0xE4:
    raise RuntimeError("Source code encoding problem 05.")

if len(u"\u00E4") != 1:
    raise RuntimeError("Source code encoding problem 06.")

if ord(u"\U000000E4") != 0xE4:
    raise RuntimeError("Source code encoding problem 07.")

if len(u"\U000000E4") != 1:
    raise RuntimeError("Source code encoding problem 08.")

if ord(u"ä") != 0xE4:
    raise RuntimeError("Source code encoding problem 09.")

if len(u"ä") != 1:
    raise RuntimeError("Source code encoding problem 10.")

if ord(u"\u1E9E") != 0x1E9E:
    raise RuntimeError("Source code encoding problem 11.")

if len(u"\u1E9E") != 1:
    raise RuntimeError("Source code encoding problem 12.")

if ord(u"\U00001E9E") != 0x1E9E:
    raise RuntimeError("Source code encoding problem 13.")

if len(u"\U00001E9E") != 1:
    raise RuntimeError("Source code encoding problem 14.")

if ord(u"ẞ") != 0x1E9E:
    raise RuntimeError("Source code encoding problem 15.")

if len(u"ẞ") != 1:
    raise RuntimeError("Source code encoding problem 16.")

if ord(u"\U0001F404"[0]) != 0x1F404 and ord(u"\U0001F404"[0]) != 0xD83D:
    raise RuntimeError("Source code encoding problem 17.")

if len(u"\U0001F404") != 1 and len(u"\U0001F404") != 2:
    raise RuntimeError("Source code encoding problem 18.")

if ord(u"🐄"[0]) != 0x1F404 and ord(u"🐄"[0]) != 0xD83D:
    raise RuntimeError("Source code encoding problem 19.")

if len(u"🐄") != 1 and len(u"🐄") != 2:
    raise RuntimeError("Source code encoding problem 20.")


__version__ = '0.1'


class Hyphenator:
    """An implementation of Frank Liams hyphenation algorithm in
    Python.

    Original code by Ned Batchelder from
    http://nedbatchelder.com/code/modules/hyphenate.html
    Original license:
    “Ned Batchelder, July 2007. This Python code is in the public domain.”

    Differently as the original implementation, this implementation
    adds Unicode support. On the other hand, some other features
    of the original implementation are not available.
    """
    def __init__(self, patterns):
        """Precondition: “patterns” is of type “unicode”.
        Postcondition: Constructs a hyphenator object for the
        given patterns.
        """
        if type(patterns) is not str:
            raise TypeError("The “patterns” parameter must be of type "
                            "“unicode”, but it isn’t.")
        self.tree = {}
        for pattern in patterns.split():
            self._insert_pattern(pattern)

    #
    def _insert_pattern(self, pattern):
        # Convert the a pattern like 'a1bc3d4' into a string of chars 'abcd'
        # and a list of points [ 0, 1, 0, 3, 4 ].
        chars = re.sub(u'[0-9]', u'', pattern)
        points = [int(d or 0) for d in re.split(u"[^0-9]", pattern)]
        # Insert the pattern into the tree.  Each character finds a dict
        # another level down in the tree, and leaf nodes have the list of
        # points.
        t = self.tree
        for c in chars:
            if c not in t:
                t[c] = {}
            t = t[c]
        t[None] = points

    def hyphenate_word(self, word):
        """ Precondition: “word” is of type “unicode”.
            Postcondition: Given a word, returns a list of pieces of
            type “unicode”, broken at the possible
            hyphenation points. Note that patterns are typically
            lower-case-only, so you have to convert “word” to
            lower-case before calling this function (otherwise
            the word might get wrong hyphenation because the
            upper-case-letters are not recognized).
        """
        if type(word) is not str:
            raise TypeError("The word must have the data type “unicode”, "
                            "but it doesn’t.")
        else:
            work = '.' + word.lower() + '.'
            points = [0] * (len(work) + 1)
            for i in range(len(work)):
                t = self.tree
                for c in work[i:]:
                    if c in t:
                        t = t[c]
                        if None in t:
                            p = t[None]
                            for j in range(len(p)):
                                points[i + j] = max(points[i + j], p[j])
                    else:
                        break
            # No hyphens in the first two chars or the last two.
            # But we comment this code out because it is not
            # necessary for ligature setting.
            # points[1] = points[2] = points[-2] = points[-3] = 0
            # But it is necessary to do at least this, just to avoid empty
            # pieces. Otherwise, the pattern “von1” would lead for the
            # word “von” to the result “[u"von", u""]” which is not intended.
            points[1] = 0
            points[-2] = 0
        # Examine the points to build the pieces list.
        pieces = ['']
        for c, p in zip(word, points[2:]):
            pieces[-1] += c
            if p % 2:
                pieces.append('')
        return pieces


class GermanLigatureSupport:
    """Provides support for german ligature setting. The pattern is derived
    from the word list of the Trennmuster project. Also get_word_characters()
    is derived from the Trennmuster project. For more information about
    the Trennmuster project: http://projekte.dante.de/Trennmuster

    License of the Trennmuster project:

    The MIT License (MIT)

    Copyright (c) 2013-2014 Stephan Hennig, Werner Lemberg, Guenter Milde,
    Sander van Geloven, Georg Pfeiffer, Gisbert W. Selke, Tobias Wendorf

    Permission is hereby granted, free of charge, to any person obtaining
    a copy of this software and associated documentation files (the
    "Software"), to deal in the Software without restriction, including
    without limitation the rights to use, copy, modify, merge, publish,
    distribute, sublicense, and/or sell copies of the Software, and to
    permit persons to whom the Software is furnished to do so, subject to
    the following conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
    LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
    OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
    WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE."""

    def __init__(self):
        return

    def simple_case_fold_for_lookup(self, my_unicode_string):
        """Before applying the hyphenation algorithm to a string, some
        “folding” has to be done. The german word “auffallend” has the
        ligature ["auf", "fallend"]. If it is the first
        word of a sentence, than it is written with capital letter
         “Auffallend”. The “case” (the
        fact that a letter is a small letter or a capital
        letter) does not matter. You can read
        more about this topic in the Unicode standard:
        3.13 Default Case Algorithms → Caseless matching
        The pattern uses lower case. So we have to map all
        upper case letters in a string to
        lower case letters before applying the
        hyphenation algorithm. Unicode describes
        “full case folding” and “simple case folding”.
        “full case folding” converts for example
        both lowercase ß and uppercase ẞ to ss: it
        maps one original character to two
        substitution characters. “simple case folding”
        leaves lowercase ß as is, and converts
        uppercase ẞ to lowercase ß. I think the
        only relevant application of this “one-to-many
        mapping” for the german language is the sharp
        s. As the pattern is generated for
        both (normal german with ß and swiss
        german without ß but with ss), this “one-to-many
        folding” is not necessary. A simple
        toLowercase() with additional mapping
        of the lowercase long s (ſ) to the lowercase
        normal s should be enough.

        Preconditions: my_unicode_string is of type “unicode”.
        Postconditions: Returns a “unicode” value
        that corresponds to my_unicode_string, but has
        mapped uppercase characters to lowercase
        characters – or at least these that are important
        for our patterns. The mapping is guaranteed
        to be a one-to-one mapping of Unicode Scalar
        Values. That means that one Unicode Scalar
        Value is replaced by exactly one other
        Unicode Scalar Value. So the count of
        Unicode Scalar Values of the return value is equal
        to the count of Unicode Scalar Values of
        my_unicode_string. (Note that the count of code
        units might change between input and output
        if you do not use UTF32.)
        WARNING This function must be kept
        in synch with isWordCharacter().
        """
        if type(my_unicode_string) is not str:
            raise TypeError("The “my_unicode_string” parameter must be of "
                            "type “unicode”, but it isn’t.")
        return my_unicode_string.lower().replace("ſ", "s")

    """
    The following pattern has been generated using the Trennmuster project:
    http://projekte.dante.de/Trennmuster/
    This project provides a quite long list of german words with its hyphenation
    points and its morpheme boundaries, prefix boundaries, suffix boundaries…
    This can be used to generate ligature patterns.

    The current pattern is generated from the Trennmuster project at the
    following state: Sat, 10 Sep 2016 09:52:30 +0000
    http://repo.or.cz/wortliste.git/commit/c21d2f93e9ece64e47de9fc59e9115a0540c89a0

    Instructions for Linux shell:

    1.) Create a work directory
    > mkdir workdir
    > cd workdir

    2.) Clone the git repository of Trennmuster
    Preconditions: “git” is available.
    > git clone --depth 1 git://repo.or.cz/wortliste.git
    > git clone git://repo.or.cz/wortliste.git
    > cd wortliste
    > git config core.DeltaBaseCacheLimit 200m
    > cd ..
    For details, see http://projekte.dante.de/Trennmuster/RepoHaupt (german).

    3.) Create a directory for the data
    > mkdir datadir
    > cd datadir

    4.) Combine both, “wortliste” and “pre-1901” in one single file
    > cat ../wortliste/wortliste ../wortliste/pre-1901 > full-list.utf8
    The other available word lists currently do not provide information
    about morpheme boundaries, so it does not make sense to use them.

    5.) Preprocess it
    Preconditions: “python 2.7” is available.
    > python ../wortliste/skripte/python/ligatures/prepare-ligature-wordlist.py \
       full-list.utf8 patgen-input.utf8 preprocess-log.utf8

    6.) Convert to ISO 8859-15
    Preconditions: “iconv” is available.
    > iconv -f UTF-8 -t ISO_8859-15 patgen-input.utf8 > patgen-input.iso-8859-15
    Patgen only accepts single-byte encodings, and ISO 8859-15 is an encoding
    that covers all necessary characters and is the one that is also used
    by “german.tr” in the Trennmuster project.

    7.) Generate the pattern
    > sh ../wortliste/skripte/make-full-pattern.sh patgen-input.iso-8859-15 \
      ../wortliste/daten/german.tr
    This takes about 20 minutes.

    8.) Convert the pattern to UTF8
    > iconv -f ISO_8859-15 -t UTF-8 pattern.8 > ligature-pattern.utf8.txt
    """
    __germanLigaturePatterns = (u"""
.ab1a
.ab3b
.abel2
.ab3l
.ab1or
.abs2
.ab5test
.ab3u
.ach2
.ack2
.ada4
.ade4
.ad2r
.af2
.aft2
.ag2a
.agen2
.ag4r
.ag2u
.akro4
.akt2a
.al4
.ali4
.alt3s
.alz2
.am4
.amt2s1
.an1
.ana1c
.and2
.an3dri
.ang4
.angs2
.angst3
.ank2
.an3s
.ans2p
.an3the
.anti5ge
.an3z2
.aos4
.ap1p
.ap3po
.ar4
.ark2
.arm3ac
.art3ei
.art5erh
.arto2
.as4
.asbe2
.at4
.au2
.au4f3
.au4sc
.au4sp
.aust2
.aus5tes
.au4v
.äo2
.ärme4
.ba4
.bak4
.baus4
.be3erb
.bege4
.bel2a
.be3lab
.bel4f
.ben4
.bens2
.ber4
.berg3a
.berg5eb
.berg3r
.be7ringe
.bes4
.be2ve
.bi3na
.bio3
.bit1a
.bo4
.bor4
.bu4
.ce2
.ch4
.che4
.dab2
.da1l
.dar3i
.darm1
.datei3
.de4b
.de1i
.de4los
.de1mi
.den2
.de1o
.de2pa
.de3ran
.der4en
.der2n
.der4wi
.de1s
.de3sk
.de1te
.di3el
.dien4e
.dien6st
.dienst7ad
.dio2
.do2
.dorf1
.dro4
.dy2
.dys3
.eb2
.eben3d
.ed4
.efer2
.eg4
.ehe3
.eif2
.ei4gen
.eins4
.ei2s
.ei3sch
.eis3s
.ei4s3t
.ei2t2
.ei1z
.el6
.elb3s
.elch3t
.ele4
.els2
.em2
.emm2
.em4s
.en2d
.en3dem
.end3er
.en4d1r
.end3s
.en2f
.en2g
.enge2
.enn2
.en2q
.en2r
.en2s
.en2t1
.ent3io
.ent3r
.erbt2
.er4dar
.er4din
.er1e
.erf2
.er3g
.er6hart
.er1i
.er4lac
.er8lang.
.ern2
.er3neu
.er8stein
.er1t2
.er4z3el
.es4
.est2
.est6e
.eß1
.et4
.ett4
.eu4g
.eu1tr
.ex1te
.fa4
.fe4
.fel4
.fer4
.fern3t
.fi4
.fie4
.fil4
.fin4
.fon2
.fra2
.fus6
.ga4
.gd2
.ge1bu
.gege2
.ge6het.
.ge1hu
.gel2
.geld1
.ge3lu
.ge3me
.ge3nar
.gen2e
.ge3nec
.ge3nei
.gen4fe
.gen2o
.ge3nor
.gen2r
.gen3te
.ge3nun
.ge1r4
.ge3re
.ge7riere
.ges4
.ge5sint
.ge3u
.gie4
.gin2
.go2
.ha4
.haf2
.haft5s
.hal4
.han4
.hau2
.hau4s
.haut1
.he4
.hef2
.hel2
.hen2
.her3an
.her6ber
.her6bra
.her5inn
.her6man
.her6rin
.her4zi
.her7zog.
.hi4
.hin3u
.hip3s
.ho4
.hof1e
.hom3b
.ide2
.ig2
.ih2
.il1
.im2a4
.im4k
.imm4
.im1pe
.in3dem
.in3des
.indi5ge
.in3e
.in1g2
.inge2
.in4geb
.ink2
.inn2e
.in4sel
.in3ser
.in3sk
.in1z
.is2
.isch2
.is4f
.it2
.iv2
.ka4
.kal2
.kal4k
.kal4o
.kan4
.kat2
.kat3io
.kat4m
.ke4
.kel2
.ken2
.ker4
.ki4
.klang3
.ko4g
.ko1i
.ko4k
.kom1m
.kopf1
.ko4s
.kose3
.kro2
.ku4b
.kus2
.la4
.lan2
.lau4s
.lär2
.le4
.lei4
.leich4
.len2
.ler2
.li4
.licht7ers
.lin2
.lit4
.lo4
.log3in
.lor2
.los3s2
.lub2
.lun4
.lut2
.ly4
.ma4
.mal4e
.man2
.mas4
.me4
.mee4
.mel2
.men2
.mer4
.mes4sp
.mi4
.min2
.mo4
.mor4
.ms4
.mund3a
.mus2
.na2
.nach3s
.nat2
.näs3c
.ne4
.nea4
.nen2
.ner2
.nicht5e
.nie4
.nik2
.nin2
.no4
.nus2
.nuss1
.ob1a
.obe4
.oben3
.ob1l
.och2
.of2e
.of3fer
.oh4
.ohr3s
.on2k
.or4
.oran2
.ort2
.orts3e
.os2
.ost6alg
.ost3el
.ost5end
.ost3ri
.ot2
.ot3a
.ot4f
.ot4m
.ott4
.ox4
.öd2
.öl3b
.öl1l
.öl1z
.pa2
.par4k
.par4l
.par4m
.pat4
.pel2
.pen2
.per4l
.pf4
.ph2
.pi2
.pier4
.po4
.por4
.ps2
.ra4
.rab2
.raf4
.ran2
.rang3t
.ranzu4
.re1b
.reb3s
.re3cha
.rei4
.rein3t
.reis6ei
.rem4
.re4mag
.re1mi
.ren2
.re1q
.res2
.re2sc
.re3tri
.ri4
.rie4
.rif4
.rik4
.rin4
.ro4
.roll4e
.rom2
.rom4a
.ror2
.ruhr1
.run2
.sa4
.sab4
.sal4
.san4
.sch4
.schen4
.se4
.sen2
.ser4
.sie4
.sig2
.sim2
.so3ge
.son2
.st4
.stan4
.ste2
.sten2
.ster4
.stie6
.ta4
.tab2
.tage4s3
.tal3la
.tan4
.tank3l
.tar4
.tat3ta
.tat1u
.te4
.tei2
.tele3b
.tem4
.ten2
.ter4
.tes2
.test3r
.tex2
.ti4
.til2
.to4
.tode4
.ton3s
.tör3c
.tra6
.tre4
.trie4
.tro2
.ts4
.tum4
.tur4
.ty2
.ub2
.uer2
.ufer1
.uh2
.ul2
.ull2
.um1
.un1a
.un1d
.un1g
.un1k
.un1n
.un3s
.un1z
.ur1
.ur2an
.ur3b
.ur2i
.urin3t
.uro1
.ur3om
.urs2
.ur3t
.ur3z
.us2
.ut2
.va2
.val4
.vel2
.ven2
.ver3be
.ver3d
.ver3g
.ver3t
.ver5v
.vil2
.walt2
.weg5s
.wei2
.wen4
.wer4
.wert7her
.wider7tes
.wie3f
.wil2
.wo2h
.wort5en
.yo2
.zahn3
.ze2
.zer1n
.ziel3u
.zin4
.zo2
aa2
a1ab
a1ace
aa3do
aal1di
aale3t
aal1t
2a1an
aan1g
2a3aq
2aar
aar3a
aar3b
aar3f
aar3g4
aar3k2
aar3ma
aar3n
aar3r
aar3s
aar3ta
aar3th
aar3tr
aar3z
1aas
aas1t
aat3h
aat3r
aat4s1
2a3au
a1ä
a2b
aba4
2abab
a4b2ak
2abal
ab2am
ab2an
abar4
ab1au
abau4t
a3b1ä
a4bän
ab2äu
ab4bas
ab2c
1abd
2abe.
4a3bec
abe1e
ab1eil
2a4b2e2l
abe3lis
a4ben
1abent
aben1z
2aber
ab1erk
aber2n
ab1er4r
ab1erz
2abes
ab3esse
ab1eß
2a3b2et
a5betr
2abew
1abf
3abg
3a4bh
2abi
4abil
ab1ins
a3b1ir
abi1st
3ab1it
1abk
ab1l
1abla
ab3lag
1ablä
3ablei
a4blen
a4bles
2ablet
2ab2lo
1a4blö
2ablu
1abn
3abo.
ab2of
a3bog
1a4bon
abor3s
abo3se
2abot
2abö
ab1r
abra2
ab3rec
a4bri
2abro
2abrö
1absc
ab3s2i
1ab3s2p
abs2t2
1absta
ab3ste
ab1t2
abte2
1abtei
3abtr
2abu
abu3b
ab1ur3
2abü
1abw
2aby
3ab3z
2ac.
2aca
2acc
2ach.
ach1a
ach3as
ach3au
2achb
ach6bars
2ache
ach1ec
ache3f
ach1eh
ach1ei
ach3erh
ach3erl
ach3erw
2achf
ach1l
ach3lö
2ach1m
ach1n
ach1ob
ach3ö
2ach1r
2achsc
achs4el
6achsens
achs4i
ach3sin
ach4tak
acht3in
ach2t1o
ach4tru
acht6sal
ach1uf
ach3ü
2achv
4ach1w
ach3z
a4ci
4ack.
acke2s
ack2mu
ackmus3
ack2sp
2acl
a2co
2a1cu
a2d
4ad.
4adab
ad2ag
ad1ama
ad1an
3ad3ap
4ad2ar3
ad1au
4ada4v
1adä
ad1c
1add
ad2dis
ad2e
2ade.
2adeg
ade3k
ade2l
4aden
aden3ti
ade1ro
4ade1s2
ade3sp
2adf
adge3
4adh
4adi
adie4
a3dikt
adio1s
adi4p
3adj
ad2ke
ad3len
ad3me
3admi
4admu
ad3na
ad2ne
ad2ob
ado1m
1adop
2adp
2adq
2ad1rec
ad1rei
a3drom
ad1run
2ads2
ad2se
ad3st
ad3sz
2ad2t1
adt3h
1adv
2ae2
a1eb
a3ed
a1ei
ael2s
a3em
aeo3
a1ep
a1erh
3aer2o1
aes4
a1ex
2afa
af1ab
a3fag
a5fah
a3fak
af1an
af1au
2a3fä
2afe
af1ec
af2f
aff4a
af3fam
af3far
af3fr
af4fro
aff2s
2afi2
2af1l
a5flu
2afo
af2ob
a4foh
afon2
afo1s
2afra
af1rau
af1rä
af1re
2afro
af1rö
af3s2a
af3sh
af2si
af2s1p
afs2t
2af2t
aft1a
aft1r
aft3re
af3ur
2afü
af3z
4ag.
2aga
ag1ab
ag1ad
ag1am
ag1ar
ag1au
aga2v
ag2d1
a2ge
2age.
age1b
age1g
a4geh
age1i
age2l1e
age2n
2agen.
2ages
a4ge2sa
ages3i
age2s3p
age4spe
age4s3ti
a4get
agge2r
3aggr
ag4gu
ag1id
agie4
2agl
ag1la
ag2lan
2agm
ag2ma
agner4s
ag2ni
a1gons
a2grä
a2g3re
a2g3ri
a5gru
2ags
ag3s2ah
ag3seid
ag5seri
ags8porta
ag4stät
ag3ste
ag4stei
ag3stö
2agt
ag2th
2agu
2ah.
a1han
a1hä
ahdi3
2ahe
ahe1g
ahe3in
ahe1l
ah1erh
ahe1s
a1h2i
ah2j
ah2l
ahl3a
ahl1ei
ahl3erd
ahl3erh
ahl3sz
ah2m
ahme1i
ah4n
ahn1a
ahner4e
ahn3ern
aho2
1a4hor
ah1os
a2h3ö
ahr1a
ahr2e
ahre4s
ahr4tri
2ahs
ah1se
ah2te
aht1ex
ah2t3r
aht3s6
aht1z
ah1wä
a3hy
ah3z
ai1a4
aian1
2aib
aib1l
ai1bu
aid2s
aids1t
aien3
ai1f
aig4
ai1gl
ai2j
ai2ko
ai1ku
ai1lä
ai2le
ail3lis
ail3s
ail1t
aim2
ai1ma
ai2man
ai1mo
2ain
ain2a
ai3nac
a1ind
ain2e
ain3f
a3in1g
ain2m
ain2r
ain3sp
air5l
ai3schw
ais2e2
ait2
ai1ta
ai2tre
ai2w
a2k
2ak.
2akab
ak4ad2
2akal
2a3kam
2akar
ak4at
ak2au
1akaz
a3kä
2akb
2akc
2akd
2ake
ak1em
ak1ent
aket3z
2aki
a3kil
ak1ins
ak2ka
2akl
2ako
ako3k
ako3l
2akr
ak1res
ak2ri
a4kro
2ak2s
ak3sc
ak3sh
ak3so
ak3ste
ak2t
2akta
2aktb
akt2er
2aktik
2aktm
akt1r
2aktsi
2aktsp
2aktst
2aktw
a4kum
2akun
2akur
3akut
4a3kü
1akz
4al.
2ala.
al3abo
alach2
al1ad
al1af
al1age
a2lai
al1am
alami3s
al1ana
al1ans
al1anz
2al3arr
al1asi
al1ass
alat3s
al1au
al3auf
al3aug
al1äm
al1är
al1äu
alb3erw
alb1l
alb3ru
alb3s
al1c
al2dac
al1dat
alde4
ald3erl
ald3ern
al2dr
4ale
al1eb
al1ec
ale4d
al1ef
al1eh
al1ei
a1leic
al3ein
a1lek
al1el
a3lenas
al3ends
al1epo
al1erb
al1erf
al1erg
al1erh
al3erl
al1err
al1ert
ale4s4
al1esk
al1eu
al3fe
alf2r
2alg.
al3gas
3algi
al2gli
1algo
3algor
al3h
a3lief
a1l2imb
al1ind
al1ins
alin4sc
a1lip
a2lis
alk1le
1alkoh
alk3s
al3kür
all1ap
all1ar
al1läu
3allee
alle4g
4allin
al1lip
al1lit
al1loh
al4lok
all3öse
al2lu
alm3ast
al2me
al3n
al1ob
3aloe
4alog
alo2gr
al1ont
a2l1ö
3alp.
3alpe.
al3per
3alph
2al3pr
al3r
2als.
al3ska
al1skl
als2te
al5sterb
al2t1ak
alt1an
al2tat
alt3erf
4alth
alt1op
alt3rad
alt3reu
alt3ric
alt1ro
alt4stü
al1tur
2alty
alu3b
alu3f
alu3g
al1umb
al1ur
alut3s
4aly2
2alz
alz4erk
al2zw
am2
2am.
2ama.
a3mac
a3mag
2amah
ama3k
a1mal.
a1mals
2aman2
a2mane
am3ap
2amar
a1max
2a2mä
a3män
am3d
2ame.
2amel
amen2d
amen1e
amer2a
am3erf
amer2u
ame1s
2a1mir
2amis
a1miss
2amit
ami3tak
2am3l
am4m
2amm.
2ammal
ammer6li
2amml
amm3unt
amo2
a2mol
am3ord
a4mö
am3öl
amp2
am3pas
2ampe.
2ampen
amp4f
2ample
2am4po
am3ra
am3rö
am4sc
am3sp
am3str
ams4w
1amt.
amt1a
amt1ä
amt3ern
amt3r
2amu
am3unt
a3mü
am4wa
2an.
an2a
2ana.
2anab
an3abb
2anad
2anak
an3alg
2anam
anama3r
anama3z
2anan
an3ang
an3ant
ana5po
an3ara
an3arb
2anas
an4atm
an3au
1anäs
1anb
3anbi
2anbu
2an2c
ancen3
4and.
3an3d2ac
and3anz
an2de
and3ei
ande2l
ander4e
ande2s
and1ex
2andi
an2dor
an3dran
an2dri
an4drom
and4sas
and4spa
ands5paß
an2du
and1ur
2an4dy
ane2
2ane.
an3ec
an2ei.
an3eif
1aneig
3anek
2anen
4aner.
an1erk
4anern
anex1
1anf
2anf.
2anfab
3anfä
2anfi
anf2t
anf2u
4ang.
1angab
an2g1ar
an3gau
1angeb
ang1ei
ang3erf
ang3erw
ang3erz
2angh
4angie
ang1la
ang1lä
ang1n
ang3ra
1angri
4angs.
angt4
angun1
1anh
an6halt.
an8halts.
an4haus
2an2i
4anie
ani1g2
anik2e
ani2l
3anim
an3ind
an3ins
an3ion
anis2t
a1niv
4an2j
2ank.
ank1ak
an2ke
2anken
an3ket
an2ki
ank1no
ank1ra
ank1rä
2ank1s
anks2p
2ankt2
an2ku
3ankü
1anl
2anlad
2anmo
1anmu
2ann.
1annah
an2nas
2anne
an2net
ann2g
an2nie
an2no
an3noc
2anns
ann4sto
2annt
a2no
2ano.
ano1b
1anod
a3noi
an1or
2anos
2anö
4anpr
1an3r
2anrö
4ans.
an3s2ä
1ansc
ans4ga
2ansk
an3skr
ans1pa
ans2t
an3sta
an3str
an3s2z
4ant.
an2tag
ant1ar
an2tau
an2tä
an3tät
ante2
2ante.
3an3tei
2antem
an2ten
4anten.
2anter
2antes
ant2g
an2th
ant2he
an3thei
2anti
3antib
ant2l
an2to
anton2
3an1tr
4ant2s
ant3st
1antw
2a2nu
anus1s
2anv
1anw
2anwi
2anz.
2anzab
an2zä
2anzb
2anzd
1anzei
2an2zen
2anzes
2anzf
2anzg
2anzh
anz1in
2anzk
2anzm
2anzp
2anzr
2anzs
2anzt
1anzü
2anzv
2anzw
an2zwa
an2zwi
2ao
ao3b
aof2
ao2l
ao1m
a1op
a1or2
ao2s3
aost2
ao3ts
a1ö
2ap.
2apa
apa3b
a1pä
a2pe
ap2f
1apfel
2apfes
a3pfl
a1phä
a1phr
2api
2apl
apo1d
apo1k
2apo2l
ap2po
ap2pu
2a1pr
a2pri
ap2so
apt2
ap1to
2apu
ar2
2ar.
ara4b
2ar3abb
ar3abf
ar3abt
arad2
ar3adr
ar3al
2aran
ar3anf
ar3ang
ar3ans
ara3nu
ar3anw
ar3anz
2ara1p
ar3ap1p
ar3ar
a2ras
2ar3au
ara1ve
ar3är
1arb
2arb.
4ar4ba
arb1au
arb3ein
2arbek
2arben
2ar3ber
4ar3bes
ar3bet
4arbi
ar3bis
2arbl
2arbo
2arbr
arb3re
2arb3s2
2arbt
2arbu
arb3un
3arc
4archl
4ar3chr
arch2t
2ard
ar3dam
ar3deb
ar3dir
ard1r
2are.
are1b
a1rec
ar3eff
are3g
are1h
ar3ehr
2arei
ar3eid
a1reih
ar3ein
ar3ele
2aren.
aren3so
aren3zi
are1r2a
ar3erg
ar3erh
ar3erk
ar3erl
ar3ert
are1te
are3u
2arf
ar3fi
ar3fl
arf1ra
ar3fu
ar4fus
4arg.
ar3ge
ar4gi
arg3l
2arg4o
arg4r
ar3gru
2ar3h
ar5he
2ari
ari4at
ar3ide
arie3c
ar3im
4arin
arin3st
ar3int
ar3inw
a1riv
ark1ar
ark3aue
2ark1l
ar3kn
2arko
ar4kor
ar4kri
ark3sa
ark3she
ark2t
ark4tre
ar3kul
ar4let
ar4lin
arma1l
ar3mang
ar3mas
arma3sc
ar3m2ä
ar3med
arm1eg
arm1ei
ar3min
ar3mit
4armü
ar4nas
arn2el
ar3od
ar3of
a2rom
aron2
ar3op
ar3or
aro2t
ar3ö
2arp
ar4pat
ar3ph
ar3pi
2arr
arr1ad
arre2g
2ars.
2arsa
ars2h
2ar3s2i
ar4s3k
ar4so
ar3sta
ar3su
ar3tag
art2e
2artei
artei5le
artel4
ar3tex
art1he
art2i
2arto
ar3tol
art3ram
art3ren
2arts
ar3tue
2aru
ar3uh
ar3ums
a2r3ü
2ar3v
ar4vi
ar5vo
ar4win
2ary
2arza
2arze
2arzi
ar3zie
3arzn
ar3zo
1arz2t
arzt1r
2arzu
arz1w
ar3zwä
2as.
a2sa
a3sai
as1am
as3aug
a2sä
a4sca
a2sche
2ascht
2ase
a3see
as1ef
a1seg
a3seh
as1eie
ase2l
ase2m
as1emi
a4sen
asens2
as1ent
asen3ta
as2er
aser1t
a4ses
ase2t
as1eta
a1seu
2asg
2asim
a2sin
2asis
as2k
as3kan
aska3s
aski2
as3kl
as3ko
as3kr
as2m
as3me
as3mu
as3ne
as1of
a2son
as1or
as1ped
a2sph
as2po
as2s
ass2ab
as3saf
ass6aus.
as3sc
ass2e
ass3ein
assen3t
ass2i
as1ski
ass1p
as3sta
ass1ti
ass1to
as3str
as3stu
as4sü
a3stas
as1tep
ast2er
2asti
ast3orc
a1str
as2t1re
ast5roll
as1tür
2asu
as2ur
3asyl
a4s2z
aß1er
aß1ti
2at
4at.
4ata
at1abe
at1abr
ata1f
at4ag
at1akt
at3ank
at1apf
at2as
a3tass
at1au
at1än
4atb
at2c
4ate.
a1tee
at1eig
a4tel
ate3le
3ateli
a2te4n
atens4
ate2r
atern2
ater3st
4ates
at2eu
at2ex
4ath.
at3hag
ath3in
3athl
at2hu
4ati4
atie3b
atien2
atinen3
at4is
at2la
4atli
4atlo
3at2m
4atma
4atmus
ato2m
3atomk
4ator
a1torh
at1ort
a1tow
at1rä
at1re
at1rom
at1rot
a3tru
at3rü
4at2s
at3sä
at4säm
at4sche
ats1o
ats1p
ats3tät
at2t
3attac
att1ak
4atte.
at4tec
att2el
at3tell
at4ti
att1rä
att3s2
att4u
at3tur
atu2
a3tub
atur1m
atur1t
at2z
atz3elt
atz1er
atz1in
atz1th
atz1w
a2u
2au.
2au1a
auan1
2aub
au2b1al
au2bau
aub3b
aub2si
2auc
auchs4p
au3co
au1da
au3deg
au2dr
au1du
2aue
au1eb
au1el
au2er
aue2s
au1esc
au2fa
auf1an
3aufber
2aufe.
2aufeh
4aufen.
auf1er
auf3ind
au3fis
auf4ler
1aufn
auft2
2auft.
au1fu
6aufzeit
2aug
au3g2ar
4augeh
au1gel
au1gem
aug2er
au1ges
aug3g
au3gla
au1go
4augr
2auh
au3ha
au1hi
auh1r
2aui
au3in
2auj
au1ka
2aul
au1la
au1lä
au1lo
aul3p
aul1ti
au1lu
au1lü
4aum
aum1er1
aum3eri
au1min
aum1m
aum1o
aum1p
aum3s2
2aun
aun2a
aun2e
au3neb
au1o
2aup4
2aur4
au1rei
au1rin
au1roc
au1rö
a4us
aus1ah
4ausc
au3schl
au3schw
1ausd
2ause.
2ausen
aus3erb
aus3erf
aus3erp
1ausf
1ausg
aus3k
1ausl
au1so.
1ausr
1auss2
aus3seg
aus1si
auss4s
aus3st
2aust.
aust2a
2auste
au5stein
aus3tie
aust2o
aus1tr
1ausü
1ausz
auß2
2aut.
au1tal
au3tanz
au2t1äu
2autb
2aute
aut1el
aut3erh
au3term
2autg
au1tie
1auto
auto3f
au1tr
aut3roc
2auts2
aut1t
au3tur
2auu
2au3w
2aux
2auz
auz2w
2a1ü
av2a
a3vera
4avouir
2aw
a1wä
a1wo
a1wu
a1wü
2ax.
ax2a
ax3ab
ax3al
ax3an
ax5ans
2axd
2ax2e
ax2ha
axi1d
axi1l
axi1v
ax2w
2ay
ay1anz
ay2m
ay1of
ays2
ay1ste
az4
3azal
a5zei
a1zep
a2zi
azi3g
azin3s
azi3p
azi3v
a2zo
a4zy
ä1am
äan2
ä1ar
ä2b
äbel1
äb1l
äb2s
äb2t
äche1e
äch1li
äch4s
äch4t
ächt4e
äch2z
äck2e
äck1n
äcks2
äde2l
äde2r
ä1de1s
äd1ia
ädis2
ä2dr
äd2s
äd4t
2äe
ä1em
äer3v
ä1ex
äf2f
äff1le
äf1l
äf2s
äf2t
äft2e
äft4s
ä2g
äg1a
1ägä
ägd2
äg1da
äge1i
ägen4e
äge2r
äge3s
äg1l
äg2n
ä3gna
ägs2
äg3sta
äg3str
1ägy
äh1a
äh2d
2ähe
äh1ein
äh1in
ähl1a
ähl2e4
ähl3ebe
ähls2
2ähm
äh1ma
äh2me
äh2n
äh3na
ähn2r
2ähr
äh1rob
ähr3sa
2ähs2
2äht
äh1w
2äi
ä1im
ä1j
2ä2k
äko3
äl2
ä1la
älbe2
äll1a
älp1
ä1lu
älz2e
äm2
2äma
ämer2s
2äm3l
ämpf4e
ämt4e
ä2n4
2än.
ä3na
än5c
2äne
äne3s
änf3l
änft2
2äng2e
änge2r
2ängl
äng4s
2äni
änk2e
änk1l
änk2s
än5l
änn2e
2äns
äns1a
äns1c
äns2e
änte3le
2änz
äo1
äon2
äo3s2
äp2
ä1pa
äps1c
ä1pu
1äq
äqui1
2är.
är3a2
är1ä
2är2b
är4bel
ärb1l
är1c
ärde4
2äre2
är1ei
är1el
är1ene
är1eng
är2f2
är3fl
är2g
är3ga
är1int
är2k2
är3ka
är4ke
ärke3t
är3kl
är4kli
ärkom1
är1ma
ärm3arm
ärme3t
är1mis
ärm2s
är1mu
är1of3
är1op
är2ri
2är2s
ärsch5li
är3se
är4ste
ärt4e
ärt4n
ärt4s1
är2z
är3zi
äs2
ä2sa
2äse
äse3g
äse1h
äse1i
äsel2
äsen2
äse3ref
äser4ei
äse3r2i
äse1t
2ä1si
äs3ko
ä3skr
äs3m
ä2s3p
2äss
äss1c
äss2e
äss3erk
ä1sta
äst2e
äste3m
1ästh
äs3w
äß1erk
äß2r
ät2
ät3a4
äte1h
äte1i
äte3l
ät3emp
ät3epi
äti2
ät3id
ät3l
ät3n
ät3ob
ät3r
ät4se
äts1i
äts1or
äts1p
äts1t
äts3te
ät3tis
ät3w
ät3zei
ä2u
äu2b
äube2
äub1l
äu2d
äude3
2äuf
1äu2g
2äul
2äum
äum1p
äumpf4
äum4s1
2äun2
2äu2r2
äure1
äu2s
2äus.
äus2e
äuse3n
äus2s
äuss1c
1äuß
äut2
äu2z
ä1v
1äx
ä1z
â2t
1ba2
4ba.
4babf
b1abs
bab2w
bach4sc
bacht4e
back3s2
bade1i
bade3l
4b1ad2m
b1adr
2b1af
bag4
3bah
4bai
bak1er
bak1i
bak1l
bak1r
bal2
bal1da
4bale
4balis
balk4a
ball3ei
bal4s
bal3tem
4b1am
ban2
4ban.
ban4a
b2and
band1a
4bane
b1ang2
4bani
3ban4k
bank1a
4b3anl
b1anna
4b1an3s
4b1ant
ban3tr
4banw
b1an3z
2bao
2bap
bar3ast
bar3at
2barb
4bard
2barel
ba3rep
2barf
2barg
bar3ga
bar4gen
2barh
2bari
bar3ins
bar3k
2barki
2barko
2barkr
2barl
2barn
2barp
2bar3sc
bar6sche
2barsj
2barsk
4barsta
4barstä
bar3tis
bar3t2r
2barv
2barw
2barz2
bar3zw
bas4
3bass
bast2
3baß
2bat
3bata
bat3en4t
4bati
4batj
bat2o
bau3b
bau3cha
bau1d
3baue
bau3erg
bau3erl
bau3erw
bau1f
b2aufo
bau1g2
bau1h
b2auk
bau1l
3baum
bau3men
bau1r
bau1s
bau3s4k
bau1ta
bau3z
2ba3w
b1ax
1bä
b2äc
bäch1l
b2äd
4bäe
b2äl
b2är
bär3b
b2äs
2b1b2
b3ba
bbau3er
bb3b
bbbe2
b4be.
b2bel
bbe2n
bben3s
bben2t
b3bep
b2ber
bbin2
b4bot
2b1c
bch2
b3che
4b1d2
bde2
b3dec
b2der
bde1s
bde3st
2be.
be3al
3beam
1bear
be3as
beb2
b1ebb
2bebeg
be1bel
be1ber
2bebet
1bebt
be1bu
be2c
2becc
1bech
be3ci
3be1d
b1edel
bee2
be1eh
be1en
be1erl
be1ert
be1eta
3beete
be3f2
2befäl
b1eff
be1g2
be3ga
2begel
1begi
1begl
2bego
1be1h4
be3he
3behö
1bei
bei3b
2beid
b1eier
bei1f2
bei4fus
bei4fuß
bei1g
3beih
bei1k
bei1l2
bei1m
b1eime
bei3nam
be1ind
be1inh
bei1r
bei1s2
beit2s
beit4st
bei1z
be1k
be2ki
2bekk
3bekl
2bekr
2bel.
be3las
bel3b
bel1d
be1le
2bele.
bel1en
be2ler
bel3ere
bel3f
be1l2i
2belis
be2liz
bel1k
bel3lam
4beln
bel3ö
4bels
bel1se
bel1sz
2belt
bel1ta
bel1tr
bel4u
belun4
bel3v
4belw
be1m
2bem.
be3me
b1emp
2ben
be3nabe
ben3ar
ben3da.
b2ene
3bened
be1nei
ben2eu
be1ni
ben3k
ben1n
beno2
be2nor
b2ens
benst2
b1entb
ben3ti
ben1t4r
b1ents
benz2
3benzo
2bep
be1q
4ber.
be1ra2
ber3am
2berb2
berber4
ber1d
be1re
2bere.
ber2ec
ber4ei.
ber3eiw
2berem
2be2ren
2berer
2bere2s
ber3esc
4berf2
1berg.
berg3af
berg3as
4berh
ber4har
ber4hor
be1rie
2berin
ber4in.
be5ringt
ber6inne
be1rip
2beris
ber3iss
2berm
ber3mi
2bern
bern2a
b1ernt
2bero
be3rose
be2rö
ber3r
2bers.
ber3st4a
4bert2
ber3tab
ber2tä
ber3til
2berus
ber3v
2berw
ber3z
b2e1s
2bes.
be3sak
1besc
be2sen
be4sens
bes1er
be2s1id
be3slo
bess4
b3esst.
bes1sz
be6stein
best2i
bes3tin
be4s3tol
best2r
bes5trau
be4strä
2bestu
bes3tur
be2sum
2besz
be3sze
beß2
1bet
be1ta
be2tam
be2tap
beta2s
be4ta3te
be3thö
be1tit
be3tri
bet3sc
1beu4
be1um
be1ur
2beü
1b2ew
3bewa
b1ex
be3z
3bezi
4bezim
4bé
4b3f4
2b3g2
bgas3
bge1
bgel2
bge3n
bge3o
bge3s
2b1h2
b2har
b3hä
bho2
bhol3f
6bi.
4bia
bi1ak
biar4
1bib2
4biba
2bic
2bid
2bien.
bi1enn
bien3s
bie4s
3bie4t
bi1f
4big
bi1ga
bige2
4bij
bi2jo
bik2a
bi1kar
bil2a
bi1lab
bi1lat
bil3au
3bil2d
bi1lin
1bill
b1illu
bil3m
4bi1lo
4bilö
2bin.
b1inb
2bin2e
2b1inf
b1inh
2bini
bin2n
bi3no
b1int
1bio
bi4o1d
bio3g
bio1i
bio1n
bio1r
bi4s2a
bi1se
bis2m
b3iso
bi2sp
bis2s4c
bis1t
bi1sta
bis2tr
bi2stü
biß2c
2b2ita
bi1tic
3bitt
2biu
4bi1v
4biw
2biz2
bi1ze
4b1j
bjekt3o
2b1k2
bl2
bla4
2b3lac
b1lad
b1lag
blan2
b2last
1blat
b1late
b2latt
blau1b
b1laus
b1law
2b1län
3blät
4ble.
b2lea
b1leb
3blec
2bled
b1lee
b3leg
2bleh
2b1leid
b1lein
blei3sc
ble3l
4b2ler
ble3sc
ble3sz
b2let
2blich
3blick
bli4m
blin2
bling4
b2lis
1b2lit
b3lite
bli3to
3b2loc
b1los
3b2lum
3blut
blut1o
3blü
2b1m
bma2
4b1n2
b3na
bni4
bnis1c
1bo2
4bo.
bo3au
b1ob
bob1l
bob1r
bod2
bo3di
2boe
bo3ef
2b1of
bo3fe
boge2n
3boh2
4boj
2bo3k
bol3au
bol3c
bol3f
3bolik
bol3v
3bom
4bo3ma
bombe2
4bo3mo
bon2a
bond1e
bo4n1e
2b3onk
bon4m
3bo4o
2b1op2
bo3pr
bor2
bor3b
bor4d
bord1r
bor3m
bor1ma
b1ort
bort5rat
2bos.
bos3p
b2o4t
boten4e
bot2st
3b2ox
3boys.
3bö
4böf
4b1öl
bör4
2b1p4
2b1q
1br4
2br.
2bra.
3brac
b3rad
bra3f
b2rah
2brai
b2rak
brast2
b1rat.
brat3er
b1ratg
brat4sc
brau3g
b3raum
brau2t
brä2
2bre.
3brea
bre2c
6b5rechte
b1re2d
2b1ref
2breg
brei2
b1reif
b1rek
3b4rem4
bren2
b3rep
bre2s2
b1re3se
bre2t
bre2z
bri4
brie4
5brief
b1riem
4brien
brik1t
b1ritt
3b2roc
b1roh1
4broi
bro2k
b1rol
b2ro4s
bro2t
brot1t
b1rou
3b2rö
3b2ruc
brun2
b1rund
brus2
brust1
3brü
2bry
4b1s
bsa4
bs1ad
b2sal
b4samt
bs1än
b3säu
b3sc
bsch2
b4schan
b4schef
b4sco
bs4cu
bse2
bs1eb
bs1ent
bs1er
bs3er3in
bser2v
b3set
b2sim
b3sit
b4ski
bs4ku
b4sl
bs3m
b2s1of
bs2ort
b2sö
bs2pi
bs2pl
bs3s2
bst1ab
bs2t1ak
bst3ank
bst1as
bs1tät
bs2te
bst1er
b3steu
bst1h
b3stic
b2stip
b3sto
b3stö
bs2tr
b2s3trä
b4s3treu
bst1ro
b3stü
b4stüb
b2s1un
b3sy
bs2zep
bs2zi
4bt
b1ta2
btast3r
b1tä
b1tei
b2teig
b2tem
b2ten
b2ter
b2tes
bt1h
b1ti
b1to2
b1tö
b1t2r
btran2
bts2
bt3ste
b1tu
b1tü
bt4wi
bu2
bub2
bubi3
bu3br
3buc
1bud
1buf4
2bug
bu3ge
bu3gr
1buh
2bul2
4bul.
4bule
bull3a
bull1t
4buls
b3umk
1bumm
bum2s
1bund
bunde4
b1une
2bun2g4
b3ungn
bun2s
bun2t
2buq
1bur2
bur4b
b2urg
burg1a
burg1l
bur4i
bur4l
bur4n
burts1
2bury
bus3cha
3busi
bus1p
1buss
bus4sa
bus4sä
bus5term
bus1tr
bus1un
1buß
but1al
4bute
bu3th
but2l
1but2t
2buz
bu3zo
1bü
bügel3e
büh4
2bül
3bür
bür4g
bürge2
2büt
2b3v
4b1w
2b2y
6by.
by1a
by2i
by1j
by1k
by1le
by1m
by1n
by3p2
by1ra
by1rä
bys2
by3sa
by1si
by1st
1byt
by2te
by1th
4b1z2
b3zei
bzeit1
b3zen
b3zg
b2zi
b3zu
b3zü
c2a2
2ca.
1cab
ca3bl
cad2
1ca4f
cal2
calf1
cal3t
1cam
can4
cap2
car4
car5sh
1cart
cas4
ca3se
2cat2
1cau
cä2
c1b
c2c
cd2
c2di
c1do
2ce.
ce1b
ce4d
2cef
ce1g
cei2
2cel.
cel2l
cem2
ce3me
ce4na
cen3an
2cenn
cen2t
cen1ta
2ceo
ce1op
ce1q
2ce2r2
ces2
ce3sh
cet4
ce1u
c1f
c1g
c2h
4ch.
cha4
chab4b
ch3ab3i
ch1ack
ch1ag
ch1ak
chal5tie
ch3amer
3chanc
ch1ang
chang3h
2chant
2chanz
1chao
2char.
1chara
3charta
chau1b
chau3sa
2chä
ch1äh
ch1ärm
ch1äs
3châ
2chb
4chc
2chd
2che.
ch3eben
ch3echt
1chef
3chef.
3chefs
che1g
2chei
ch1eie
ch1eil
ch1eim
2chel
2chema
1chemi
che4n
chen3ti
cher3a
1cherc
ch3erfü
cher3m
2ches
ch3ess
ch1e2ta
che1te
2cheu
ch3ex1
2chf
2chg
2chic
3chines
ch1ins
ch1int
ch1inv
2chio
3chip
chi3re
1chiru
chi3sta
2chj
chl4
chlan2
2chle
ch3lein
ch1ler
2chm
ch3mas
ch3mus.
2chn4
chne4
ch4ner
chner8ei.
ch1off
chof4s
ch1oh
chol4
cho3m
ch1orc
ch1ori
ch2os
cho3v
ch3öl
1chör
2chp
chr4
2chra
ch1rad
ch1re3s
ch1rh
2chrif
ch1rin
2chrit
1chron
2chs
chst3ri
2cht
2chu
ch1ums
ch1unf
ch1urs
2chü
2chw
2ci2
cine2
cin2n
cin2q
c1int
3cip
cis4
3cit
c1j
4c4k
ck1a
ck1ä
ck1e2h
ck1ei
ck2ere
ck1erh
ck3erke
ck1err
ck1ese
ck3f
ck3he
ck1id
ck1in
ck1la
ck1lä
ck1leb
ck1lei
ck3lim
ck1lin
ck1lis
ck1lo
ck1na
ck1nä
ck1ni
ck1o
ck3ö
ck1q
ck1r
ck3rü
cks2al
ck1sen
ck1so
ck2spa
cks3pen
ck1tal
ck3ther
ck1t2i
ck1tr
ckt3s
ck1tü
ck1um
ck1up
c2l2
clan3
cli4
clin2
clip3a
clo8pädie
3clu
cm2
c1me
c4mi
c1mu
1co2
4co.
3coa
2cob
co3be
cod2
coff4
2cok
co3ku
2col4
3com2
4com.
com4b
com4mo
com3pe
com4ple
4co3mu
con1
con2d
con2fe
con6fise
con2ni
co3no
con2q
con2r
con4sta
con2su
con3tu
co3pr
cor2b
cor2n
cor4p
cor2s
4cos
2cot
co3tä
cour2
co4v
cover2
cover4b
2cp
c2r2
cra2
1cre2
2cree
cro1c
crom4
2cs4
c1se
c1str
2c2t
cti2
ction3
c2u2
4cud
2cul
cum4b
3cup3
cur4
2cus
c1ve
1cy2
c1z2
2da.
da1a
d1a3b
d2abä
d2abe
d3aben
d3abi
da4bit
da4bo
da4bre
d2ab2rü
d1ac
da1ca
dach3a
4d3achse
d4acht.
d1ad
dad2r
da3du
d1af2
da1fe
da4fo
d1ag2
da3ge
da1h
da2ha
da2ho
d4ai
da1in
da3kri
2dal
dal2a
d1alar
dal2b2
dal3bl
dal2k
d1all
dal2m
d1alp
d1alt
2dam.
2dama
1dame.
da1mi
1damm
d1amma
d1ammä
3d2amp
dampf8erf
2dams
d1amt
d1ana
d1and2
1dandy
d1anei
d1ang
d2ango
1dan2k
dank1o
dank2w
d1ann
3dann.
d1ans
2dant
dan1ta
2d1anz
d2anze
2d1ap
da1pa
da3pe
d2aph
da2por
dap1p
2daq
dar3a
dar4d
darf4
dar3g4
dark2
d1arka
darm1i
dar3n
dar3o
dar3s
2d1art
dar3ta
dar3tu
dar3u
1darum
d1arz
da1s2
2das.
da3sa
2dasc
da5se
da3sh
3das1s
das3se
d1asy
3daß
da2t2
daten1
d3atl
4datm
4d3atta
1dau4e
dauer3e
d1auf
d3aug
dau2p
2d1aus
da1we
d1ax
da3z
1dä2
3däc
d1äg
2d1äh
2däl
2d1ämt
2dän
d1änd
d1äng
2d1äp
2däq
2där
där3b
där1t
d1ärz
4d1äu
2däx
d1b4
d3ben
dber2
d2bol
d1ca
d1ch
d1cu
2d1d2
d3dä
d2de2
ddha3
d2dy
2de.
de2ad
de3at
d2e1b2
2debe
d3eben
1debü
3debüt
1dec
de2ca
2de1ch
3deck
de1d
dede2
dee2
de1eb
de1eg
deen3g
de1er
de1es
def2
de2fau
1defek
d1eff
3de2fi
2de3fl
2defr
de1g2
de4guss
deh2
de3ha
de3he
1dehn
d1ehr
d1ei4
1d2eic
d2eim
de1ind
de3inse
d3eint
de1k
de2kal
de2kan
de2kl
1dekod
1de4kor
de2kre
2dela
del4ade
del3aug
del1än
del1d
de1leb
del1ec
de3lege
de3lein
de1les
del2fi
del4gar
del2i
de1lie
3delik
del1k
dell3eb
del2lö
del1ob
de2lö
del4ph
dels3en
del2so
del2s1p
delt2
del1tr
de3luf
de3luk
2dema
dem4ar
dem5ents
de2met
demie3
de1min
1demo
de3mon
de1mot
2d1emp
dem3s
d2emu
de2mut
de2mü
4den.
de2nac
de1n2am
dend2
den3def
2dene
den3end
den4es
4denf
d3enge.
3denke
den3kel
denk3li
1denkm
de2not
den4s3en
den4sko
den6s5tau
den3tat
den3teu
den1th
den3tie
den1tr
2deo
de2o3b
deo1l
2depf
1depo
2depoc
de4pri
dept2
de2pu
de1q
de1ram
der3an
der3ap
der3asi
de1rat
der3bee
der2bl
d1erbs
derb2t
der3c
der3eis
de1reo
de3rep
der3ero
derf2
d4erfi
d2erh
d4erha
de3ring
der3k2
der5le.
d2erm
der1ni
de1ros
4dert
dert5end
der3the
der3ti
de2ru
der3v
der3z
de2s1a
de3sac
de4sam
des3an
3desast
des1än
des3b
1descl
de3see
de2s1eh
de2sei
des3eil
d1esel
des3elt
de3sem
des3eng
de3senk
des3erm
de2si
de3sid
des1in
des3n
de2s1o
de3so.
de2s1p
de3spe
dess2
dess4e
des3sel
dest5alt
de3stel
dest4he
de4sto
dest5rat
de3stri
de2s3u
det4
de1tal
de1tan
1detek
de1th
de1ti
de2tro
de1tu
d1etw
1deu
de1un
2deur
de1url
3deut
de2vi
de4von
1dex
de2zi
2dezu
de4zu.
2dezü
d3f2
2d1g2
d3ga4
dgas3tr
dge3m
dge2t
dget3e
2d1h2
d2hab
d3he
dher2b
d2his
di2
2di.
dia1b
3diag
diat2
dia1v
1dic
3dich
di3chl
dick1s
did2
d1ide
2didy
1dieb
diebe2
die3li
dien1d
4diens.
dienst5r
die4r
dies1c
2dig.
2dige4
di3gene
1digi
dig1l
3dig2m
1dign
2digs
dik2
di3kar
dil2
dils3
dim2
2d1imb
di3mo
2dimp
2din2a
d1ind
2din2e
d1inf
d1inh
2dini
d1init
2dinn
d3inner
2d1ins
din4sk
d1int
2dio.
dio1b
di4od
2diof
dion3i
dion3s2
dior2
2dios
3dip
1dir
2d3ira
d1irl
dis1a
2disch
1disk
dis2l
1dis1p
di3sper
d3isr
dis3sen
dis3sim
dis1so
dist2
di3sta
dis1to
dis1tr
di3su
3disz
dit3erl
dit3erm
dit3ers
dit3r
dit3s
dit1tr
dit1u
dit1zi
1div
di4vi
3divis
2diw
diz2
d1j
d1k2
2d1l2
dlap2
d2lat
d2l4e
d3lea
d3leb
d3leg
d3leh
d3lei
d3lek
d3lert
d3leu
d2lig
d3liga
dl3m
d3lo2
dl3s
d2lu
d3lud
d3luf
d1m2
dmen2
dmin2
d3mini
2dn2
d1na
d3nä
d1ne
d1ni
dnis1t
d3niv
d3no
d1nö
d3nus
d3nü
2do.
do3a
2d1ob
d2oba
do3bo
1doc
2doch
do1chi
3docu
do3d
do2de
2doe
d1of
do1ga
do1ge
3dogm
1dok
4doka
4dokr
do1lei
3doll2
1dolm
2doma
domen1
d2o2n2
donau1
3donn
don3s
don3te
do2o
dop2
do1pa
d1ope
d1opf
do1ph
3dopi
3dopp
dor4
2dor.
d2ora
dora5b
do1rat
2dorc
d1ord
3dorf
dorf1a
4dorff.
2d1org
dort1h
do2s4
do3sa
3dose
do3sp
dost3an
dot2h
dot3o
do1tr
do3un
2dov
do1wi
dows1
3dox
do2z
d1ö
2d3öl3
3d2ör4
dös1c
d3p4
dpass3
d1q
1dr4
2dra.
2d1rad
drag4
3d2ram
d3ramp
d1rand
2drar
2dras
d1rast
2drat
drat3z
4d1rauc
d2rauf
4draup
2drä.
2dräe
d2räh
d1rät
d1räu
dre2
4dre.
d2rea
d3real
3d2reck
2d1ref
2dreg
3dreh
dreh1r
4d1reic
drei1g
drei1l
drei6lic
drei1t
2drek
d1ren
4d3rep
dres4
dre3se
dre3st
2d3rez
2dré
4d1rh
2dria
4dric
d1rieg
drieg3l
2drik
4d1rind
d1rip
2d1ris
3drit
4d1ritu
4dritz
2dro.
2d1rob
2d1roc
d1rod
d2rog
3drohe
3d2rohu
2droi
2drol
d1roll
2dro2n
d1rose
d1rost
4d1ro2t
d1rou
d1ro2v
d3rub
d2ruc
d1rud
4druf
4druh
d2ruh.
4drund
4d3rut
d2rü
d1s
2ds.
d4s1amt
d2san
ds3assi
d2sau
ds1än
ds2äu
d3s2ce
dsch2
d4schin
ds2co
d2scr
dse2
ds1eb
dsee1
ds1ef
ds1eh
d2sehe
d3s2eig
d2sein
d2s1eng
d2s1ent
d2sera
ds1erf
ds1erk
ds1err
ds1eta
d2sex
d3s2ha
d3sho
d4shor
d2sid
d2s1im
d2sind
ds2inf
d3s2kan
ds3ke
d3skel
d4sli
ds3m
d2sop
ds1ori
d2sö
d5spal
ds3part
ds1pas
d2s1pat
d2spä
d2s1ph
ds2por
d3spri
d2spro
ds1s2
dst4
ds3tabe
d4stag
ds1tas
ds3tauf
d4s3täti
d2ste
d3stei
d4steil
d5stell
d4stem
d4sten
d3stern
ds1th
ds2til
ds1tis
d4stoch
d5strei
ds1ums
d2sun
d2sz
d1ta2
d3t2ac
d3t2ag
d3t2as
dt2ax
d1tä
d3tea
d1tee
d1tei
d1tel
dte3na
dte3rei
d1term
d1teu
d2thei
dtho4
dt5l
d3to
d4tob
dt2op
d3tö
dt1r
dt3s2
dt3t2
d1tu
d3t2ur
d1tü
dt3w
d3ty
dt1z
du2
2du.
1duf2
2d1ufe
d1uh
2dui
1duk
2duku
1dul
dult2
dul1tr
d1umb
2d1umd
d1um1e
2dumf
2dumk
2duml
d2ump
2dumr
2d1ums
d1umv
dun2
2d1und
d1unf
dung4
d1ungl
d1uni
1dunk
dunst3r
3duo
du3pr
1dur2
dur4b
3durc
durch3
2d1url
dus2
duse2
dus3k
duss2
2dust
du3sta
7dustria.
duß2
1duz
1dü
2düb
dür4
d1v2
d3vo
2d1w
d3waf
d3wal
d3wan
dwest1
d2wic
d2wil
d2y
2dy.
dy1c
dy3d
3dyn
dy1ne
dys1
2d3z2
dzu2g
2e1a
e3a4b
e4abi
ea1gat
ea3kr
eak3s
eakt2
eak3ta
e2al
eal3ent
eali4
eal1li
eal1ti
eal3u
eam3a
eam3b
e2ame
eam3o
eams2
eamt2
ean3ar
e3anb
e2ane
e3ang
ean2n
e3ant
ea3nu
ea2p
e3arb
eare2
ear3ene
ear3ma
e3aro
ear3ra
e2a4s
e3asc
east4
eas3z
eater1
e3ath
eat3s
e4au.
eau2b
e3aue
e3auf
eau1g
e2aus.
eau1sc
eaus1s
e2av
e3ä
2eba
e2b2ak
eb3ber
2e1be3a
ebe1b
2ebec
2ebed
2ebeg
ebe2i
e5beku
ebe2lo
ebe2n
2eber
e2bers
e1beru
4ebes
ebe4s3an
e3besc
2e3bet
2e3bew
e1bez
e1bi
2e1bl
eb2laß
eb3ler
eb2leu
e3blie
eb1lo
eb2lu
e2bos
ebot2
2eb2r
eb2s
eb4sche
ebs1i
ebs3in
ebs1o
ebs1p
ebs3pa
eb4stät
ebs5temp
ebs1th
ebs1ti
ebs3tot
eb3str
ebs1u
2ebu
e3bur
e3bus
e1but
eb2z
2eca
2e1ce
ech1ä
ech3b
2eche
e3chef
ech1ei
ech5erzi
e1chip
ech3m
e2cho
ech1ob
echo1l
ech3tab
echt4ei
ech1uh
ech1w
eckar1
eck4har
eck1n
eck4sen
eck4s3ta
2eckt
2e3cl
2eco
e2con
e1cr
2ect
e1d2a
e4da.
e3dau
e3dä
ed2d
ed2e
e1deb
e1def
e1dei
e2de2l
eden2e
eden4se
eden4s3p
e2de2r
edert2
eder3ta
e3desi
edi4
e3dif
2edip
e3dir
e2dit
e3div
e2diz
ed2n
ed2o
e1d2ö
ed2r
ed2sa
eds2ä
ed2seh
ed2s1es
ed2s1o
ed2s1p
ed2s3tr
ed2su
e1du
e3due
edy1
4ee
ee3a
ee1b2
ee1c
e1eck
eede1s
ee1di
ee1do
eed3s2
ee1e
e1eff
ee1g4
ee1hu
e1ei2
ee1k
eel2e
e3elek
ee1leo
eem2
e2e3ma
e1emb
ee3me
eemen2
ee1min
e3emp
ee1nad
ee1nä
e1end
een1da
een2g
ee2ni
een3s
e1ent
een4wi
ee1ny
e1enz
e2ep
ee2pe
ee3po
ee1q
eer3as
ee1rat
e1erbt
e1erd
e2ere
eer1ei
eer3eng
eer2es3
ee2ri
ee1ro
ee1rö
eer2ös
eert2
eer3ti
eer2u
e1erz
ee1s4
ee3sh
ee3sp
ee3st
ee1t4a
ee1u
ee3un
e3ex
ee3z
e1f
2ef.
2efa
ef1ad
e3fah
e2fak
ef1ana
ef1ar
e3farb
e3fas
e3faß
2efä
2efe
ef1eb
e2fek
efe4l
efel3ei
ef1em
ef1ene
efe2r
efe3sc
e3feu
efex1
2eff.
3effek
1effi
2efi
e3fib
ef1id
efil4
ef1ins
ef2ke
2efl
e3f2lu
ef1n
2e5f2o
efon1t
e3fö
2efr
e3fra
e3fris
ef1rol
ef1rom
ef2ru
ef2rü
efs2
ef3so
ef3sp
ef1tel
2efu
2efü
ef3z
e1ga
ega3d
eg1am
eg1d
e3gei
e1gelä
ege2lo
eg2en
egen1a
e2genf
e2geni
ege2ra
ege3rad
e3gesc
e3gese
eges3to
eges3tr
ege2t
ege1u
e3gew
eg2fri
2egi
e1gis
e4giss
2e1gl
eg2mu
eg1nä
eg3ni
e1go
e1gö
eg4ri
e3gro
egs2ag
eg4s3an
eg3se
eg4sei
egs2el
egs3er1
eg3si
eg4sin
eg4sk
eg4spa
eg4sto
eg2th
egung4
e3gü
2e1ha
eh1ach
e3h2ah
eh2al
e2hap
eh1arm
e2harz
e3haut
e2hav
e1hä
e1hec
ehef4
eh1eff
e1hei
ehe3k
e1helf
eh1elt
e3henk
1ehep
2eher
eher2b
eh1erf
eh1erl
e3herr2
e3herzt
2e1hi
eh3im
e3hir
eh1lam
eh1lä
ehl3ein
ehl2er
ehl1or
ehl2se
2ehm
eh1ma
eh1mo
2ehn
eh1na
eh1ni
eh1no
2e1ho
eh1oly
e3hö
ehr1a
ehr1ä
ehr1ec
ehre3s
ehr1ob
eh1roc
ehr1of
ehr3sch
ehs2
eh1se
eh3sh
eh3sp
eh1ste
2eht
eh1ta
eh1tel
eh1ti
eht2r
eh1tu
eh2tun
2ehu
e1huh
e1hul
eh1ums
e1hup
e1hus
e1hut
e1hü
eh3üb
eh1w
e3hy
eh3z
2ei3a4
eian1
ei2bau
ei1bie
ei1bus
eib3ute
ei1ce
e2id
eid1a
eid4ein
eid3err
2eidn
eid3st
ei1due
eie4
ei1eb
ei1ec
ei1el
ei1emi
4eien
eien3s
ei1erz
ei3esk
ei3fest
ei1fla
1eifr
2eig.
2ei1ga
ei3g2ar
2eigä
2eigeb
2eigel
4eigeno
2eiger
2eiges
2eigew
2eigi
ei1gie
1eign
2eigre
2eigrö
2eigru
2eigrü
2eigs2
2eigt
2eigu
4eih
ei1ka
eik4am
eik2ar
ei1k2i
eik2l
ei1kla
ei1ku
e2il
2eil.
ei1lä
2eilb
2eil1c
eil3d2
ei1leg
eil3f2
ei3lic
ei1lie
ei1lis
ei1lit
eil1l
eil3m2
ei1lo
ei3l2ö
eil1se
eil3z
2eim.
eima2
eim1ag
ei1mal
eim3all
eim5alp
ei1mas
eim3m
ei1mo
eim1or
2eimp
eim3sa
ei1mu
e4in.
ein1a
ein1ä
ein3dr
2eindu
ein3ebe
ein5erbe
2einfo
ein3g2
e1init
e2in1k
3einkä
ein3n2
2eino
ein1ob
ei3not
1einri
3einsat
5einschä
1einu
e4inver
ei1o2
eip2
ei3pf
ei1q
2eir
ei1re
e1irr
ei3serv
eis3erw
eis3he
ei1sit
eis3ke
eis3me
eisof3
eis3sac
eiss6lun
ei1sto
e2iß
eiß4lun
e2it
eit1ab
2eitä
eit1h
eit3t2
eit1um
ei1tü
eit1z
ei1v2
ei3zent
eiz1in
eiz3z
2e1j
e2je
e3ju
ek2
e1ka
1ekd
6ekdote.
4ekdoti
2ekdö
e1ke
e3kee
e3ket
e3key
e1ki
e3kif
ek4k
e1kl
e3klas
e3klo
e3ko
e3kra
e3krei
eks1p
ek3st
2ekt
ekt3at
ekt3erf
ekt3erz
ekt1ob
ekts2
2eku
e1kuc
e3kum
1ekz
2el.
e1la
el3abu
el3ader
e3ladu
el1af
e2l1ak
el1am
e2lame
el1ana
el1ans
el1ant
e2l1anz
2elao
e2l1ap
el1ar
el3ari
e4las.
el1asi
el1asp
ela3su
2elat
e3lats
elau4
e2law
e4lay
e1lä
e2läug
elber4g
el2bis
el1c
eld3erh
eld5erst
el1di
el3die
el1do
elds2
el1dy
4ele.
el1ech
e1legu
e3leh
4elei
e2lei.
e3leid
el5eier.
el5eiern
el1ein
e3lein.
e3leine
el4eint
e3leis
e3leit
1e2lek
ele3ki
el1el
ele3la
ele3ler
1e4lem2
ele3ma
el1emb
el1emi
el1emp
2elen.
el1ent
elen3z
e2lep
2eler
eler2a
e2l1erd
el1erf
el1erg
e2l1erk
el1erl
el3ernä
el1err
e2lers
el1erw
el1ess
e2let
el1eta
el1eti
2e1lev
el1evo
e2l1ex
4eley
1elf.
2elfa
elf2er
elf2l
1elfm
1elft
elg2
el3h
el4har
el4hup
e3lic
el1id
2elie
e4lien
e1lif
2e2lig
e1lil
elin3a
el1ind
el3inst
el1ita
e1litu
e1liz
2elk
el3ki
el3ko
elks2
elk3sc
el1ku
el3kü
2ell
el1lac
el1lad
el1lag
el3lame
el3lang
el2lat
el3läd
el1län
ell3ein
el3leit
el4ler
el4lig
el1lo
el4log
ell2ö
ell3sp
ellzu4g
elm2a
elm2e
elm3ein
el3mo
2eln
el1ni
2e1lo
e4l1oa
el1obe
elo1m
e2l1or
e3lore
elo2s
el2öf
el3p4
el1ses
el5stern
el1tat
elte4
el1tee
elter4b
3eltern
el1the
el1ti
el1tri
el1tro
elt3sa
elt3se
elt3ska
el1tur
2elu
e1lud
el1uf
e3lui
e1lul
el1um
el1ur
e1lus
el3use
el3uto
e1lü
el3va
e2ly2
4em.
e1ma
em3ab
e3mac
e3mal
em1alk
em3anf
em1ano
e2m1ans
em1app
e3mar4
2emas
2emaß
e3max
2e3mä
1embo
e3mec
e3med
e1mei
e2men
emen2t
ement3h
emer2g
em1erl
em1erw
1emeti
e1meu
em1id
emik2
2e1mil
2emin
emin4a
em1int
3emiss
emi3tr
em2ki
em3li
em3lu
em2m
emmen3t
em1n
em2ni
emo1k
e1mol
e1mon
e3mö
e4möl
em2p
em3pa
em8pathie
2empe
1emp1f4
em3pfl
emp3ler
em2s
em3so
em3sp
ems2t
ems3tr
emt2
em1th
e1mu
1e2mul
e3mus
2emü
e1my
6en.
2ena4
en1ack
en3agi
e1nai
en3ak
en1al2
en2alg
enal4l
enal4m
en1am
en3anz
e2n3ap
enap1p
e2nar
en1as
e1nase
enat4
en3atl
e4n3att
en3aug
e2naut
en1ax
en1az
en1ä
e3näc
2enb
en1c
en3chi
1ency
2enda
en3d2ac
en1dat
en3desh
end3ess
en3desw
en1die
en1din
en3dir
en1do
end4ort
end3s4l
end3s2p
end3sz
end1t
en1due
en3dus
en3d2ü
en1dy
4ene.
en3eben
en1ec
e1neig
e4neige
en1e2kl
en1el
en1ema
en1eng
en1ent
en4entr
en1ep
en1erb
en1erd
en3erei
en4erfr
1energ
en1erk
en1erl
en3ermo
en3erne
en1ers
en1ert
en3eru
en1erw
2enes
en3esc
en3ess
ene2t
en1eta
en1ev
2enf
enf2a
en3fe
enft2
enf2u
en1g
1engad
1engag
eng2al
enger4a
eng2i
en3go
1engp
2engr
eng2t
2engü
2en3h
2eni2
e1nic
e1niet
en3iko
en3ima
en1imi
e1nip
e2nisc
enit2
e3niv
2enj
en3je
en2kli
enk2ü
2enl
2enm
en3nab
e1no
en1oa
en1o2b
e2n1oc
en1oli
en1olm
eno2m
e2nom.
e2nome
e2noms
en1op
en1or
en1ost
e1nö
en1öd
2enp
2enr
4ens.
2ensa
en3sac
en3saf
en2sau
2ensc
ense2
en3sek
en3sel
1ensem
en3sema
ens3ere
en2sid
en3skor
2en2so
en3so.
en3son
en3spo
ens4por
2enst
ens4tak
enst5alt
en4s3tät
ens4tel
ens5test
ens5trie
enst2ü
2enta
ent2ag
enta3go
ent4ark
en1tas
1entd
en1tee
ent2el
en3terr
1entf
4entfo
2entfö
1entga
3entgeg
en1thr
ent3id
1entla
1entn
en1to
ent3os
ent3rol
1entsc
1entso
3ent3w
4entwet
1entz
en1u
e2nua
e3numm
e2nun
e3nut
e1nüg
enü3st
2env
2enw
en3zare
en1zeh
enz3erg
enz3erk
enz3l
4eo
e1o2b1
eo3bi
eo3bl
eo3bo
eo3bu
eo1c
eo1da
e1of
e2ofe
e3of3f
eo1g
e1oh
eo1ha
eo1ho
eo1k
e1oly
eo1m
eon2h
eo1o
e3ope
e3o2pf
e1or2
e3ord
eor4g
e2orge
e3ort2
eo1s
eo1t2
eo1w
e3ö
e1p2
2ep4a2
e3pas
e2pec
2eper
e3pers
e2pes
epe3ta
e3pf4
e3ph
epi1
e2pia
1epid
e3pir
1epis
2epist
1epit
e3pl
1epoc
e3poi
e3por
e2pot
2ep4p
epp1l
2e3p4r
e4ps
e4pt
ept4u
e3pun
e3py
2eq
4er.
er1a
e2rab
er2a3ba
era3ber
e2rach
e2rad
e4radmi
e4r3adr
eraf4
era3fa
era3ges
e2rai
er3aic
e2rak
e3rake
er3akt
era3leb
er3alke
er3all
eran3dr
er3apf
er3apr
erar4
era4sp
e1rass
er4ast
2erat
e2rati
era1tr
er3att
erau4
e1raub
er3aue
er3aug
e3raum
era1w
e1rä2
er1äh
er1äm
e2r1äs
eräte1
erät3s
er1ätz
erb2au
erb2e
2erber
2erbew
2erbo
2erbü
2er1c
erch2o
4erd.
1erdb
2erdec
erd3eng
erd3erw
1erdg
er1di
erd5l
erd3st
2erdy
2ere.
e2r1eb
ere3ber
e1rec
e3rech
er3echs
er1eck
e1red
er1edi
e1ref
er1eff
e3reg
er1eh
ere4i
6erei.
erei1b
e2reig
4e1reih
2erein
e4reins
er3eis.
er5eisar
er3eisf
erei5str
er1el
ere2le
2ere2m
er1ema
4eren
e1renk
e1renn
erenz7end
e1repa
er1epe
2ere2r
er1erh
er1ers
2e2res
er1ess
er1eß
ere2t
er1eta
er1eti
er1eul
e3re2v
ere2z
2erfam
4erf2e
4erfl
er3f4r
4erfüh
4erfür
4erfüs
4erfüß
erg3are
4ergebi
3ergebn
4ergebü
4ergeha
4ergehä
ergel6
6er3gem
2ergen
erge3ru
6erges
6ergew
4ergl
erg3ler
4ergrem
4ergru
erg3s
ergs2o
er3gu
e2r3h
1erhab
2erhai
4erhall
4erhals
2erham
2erhan
2erhar
2erhas
2erhau
2erhäu
3erhebu
2erhei
2erher
2erhin
4erhöhe
2erhu
4erhü
2eri
4eric
er1id
erien1
erien3e
4erin.
er1inb
er1ini
er1in3k
er1inl
er1int
4eris
e1rit
e2rite
2erj
2erkaj
er3kar
erken2
2erket
2er3ki
1erklä
2erkli
er3klo
2erkon
2erkö
2erkre
erkt2
erk1ti
erk1to
erk1tr
4erl.
2erlag
3erlaub
3erlebn
4erlei.
er4ler.
4erli
4erln
2erlo
er1m2
2erma
ermen2
erm3ers
2ermo
erm3t
er3mü
4ern.
er3nä
2ernd
3erneue
erno2
e2r1o2b
er1of
erof3f
ero1g
ero3ge
e1roh
ero2i
er1oly
er3onk
ero1ny
e2roo
er1op
e1rou
er1ox
2erök
er1ös
4erpa
4erpf
er3ph
4erpl
4errec
4erreis
4erren
2errh
4erro
2erru
2errü
4ers.
ers2a
ers4ana
ersch2
ers2el
4ersh
ers2i
er3sk
4erstä
4erstr
4erstu
er3swi
2ert.
2erta
ert1ab
er1tal
er1tar
er1tat
ert3erf
er4tes
er3thi
er1thr
er2tie
er4tin
er1tis
er1to
ert2ra
2ertre
4ertru
ert3s2e
ert5s2p
2ertu
er1tue
er3tun.
2eru
e1ruc
e1rud
eruf4s
e3rui
er1um
e1rumm
er1und
erung4
e1rup
er3up.
er3ur
e4rus
er3uz
e3rüm
4erv
3erweck
2erweg
2erwes
2erza
2erzei
er3zel
er3zem
2erzet
2er3zu
2erzü
4erzy
6es.
es2ac
es1ad
e3sai
e3sal
e4salt
es2amm
es2an
es3ant
es2as
e3sat
es3ato
es1av
es1ax
es2äu
es2bie
e1sc
es2ce
esch2
esch4n
e3scho
e3schr
es2cl
e4sco
e2scu
e1se
es1ebe
es1ehr
e2sein
e3sek
es2el
e2sel.
e3selbe
ese2le
ese2m
esen4ac
esen1o
e2sent
e3seq
ese3r4at
ese3rei
eser1u
eser2v
es1eta
2esh
e1si
e3sic
es1il
e3sit
es2kat
e4s3kl
e4sky
es3li
es3m
e1so
e3s2oh
eson2
e2sop
e3sorp
es2ort
eso4t
e3s2ö
2esp
es1pat
es2pe
e2s1ph
es2pi
es2po
e3spor
es4pot
e3spu
4ess.
1essay
2es2sä
es3säu
2essc
es4sens
8essenziell.
ess3erg
2essk
ess4lin
2esso
es1soh
es1soz
2essp
ess1pa
ess3tab
es4su
essub2
essub4j
es2sü
est1ak
es2tan
e1star
e4starb
est1as
es2te
est5eink
e1stel
e4sten
est3eng
est5erha
est3ess
e3stil
e2stip
est1ob
est3ori
e1stu
est3ums
e3s2tü
e1su
e3sul
es1um
es1ur
2esw
e3sy
e2sz
es2zi
e2ß
eß4bl
eß1el
eßer2e
eß1erg
eß4ki
eß6kopf.
eß4men
eß2pu
eß3ri
eß1ti
et2
2et.
e2tab2
et3abd
et4abl
e3tac
et4ag
2etal
e3talb
e3tals
et3ami
e3tan.
et3ano
et3ant
et3anw
et3auk
et4ax
e1tä
et3äh
et4chi
2ete
e3tea
e3tec
ete2e
e1tei
e3teil
et3ein
e3tek
et3ele
e1temp
e2ten
etend2
e1term
ete2s
e3teu
et4fri
et3g
2eth.
et3hai
et3hal
2et3hä
e5thea
1ethi
3ethn
eti2
et3id
e2tik
et3imp
et3inf
etin2g
et3inh
et3ini
et3ins
e1tip
et3l
et4lin
et3me
et3mu
et3ne
et3ni
et3obe
et3of
etof3f
e2ton.
e1tone
e4toner
e2tons
e1tonu
eto1ph
et3ord
et3org
e3tö
2e1tr
e3trai
e4traum
e3trä
et3rec
e3tref
et3rel
et3res
e3treu
etrieg4
etsch3w
et3se
et4sh
et3si
et3so
et3s2p
et3str
et3su
et4sum
2ett
ett1a
et3tag
ett2as
et4th
etto3r
ett1ra
ett1ro
ett1um
e1tul
et3ups
e1tur
e1tus
e1tü
et3üb
et3w
2etz
etze4
et3zei
etzes1
eu1a2
euan1
eu3b2
euber4
2euc
euch2
2eud
eu2dä
eu4den
eude1s
eue1b
eu1eng
eu1ent
euer3ei
eu3ernt
eu4erst
2euf4
euf5ar
eu2fer
eu1fu
eug1a
euge2n
eug2er
eug3g
eu1go
eug1r
eugre3
eugs2
eug3sp
eu1h
eu1in
1euk
eul2
eu1la
eu1li
eu1lo
2eum2
eum3in
eu1mo
eum4sc
eum4s1p
eum3st
eu3mu
2euna
eun2e
e3ung
e5union.
eunk2
eu1o
eu1p
euper4
eu3pf
eura2
eur3ast
eur1d
2e2ur2e
eu1rei
eur3ens
eur4er
eur1m
1euro
eur4sau
eur1t
eu1ru
eus2
eu1sel
eu1si
eu1so
eu3sp
eu1sto
eu1str
eut2e
euter2
1eutha
eut2o
eu3tor
eu2z
eu3zel
euz1w
e3ü
e1v2
evan2
eva2s3
2eve
e4ven
e2via
e2vie
e2vl
e3vo
e2von
2evor
e4vot
e1w
e3we
ewei2
ewer4
ewert4
e3wir
ew4s
2ex.
2exas
ex3at
2exc
2exd
ex1er
2exes
2exik
ex1in
1exis
ex1l
3exp
2exs
2exta
1extr
2extu
2extv
2exu
2exy
2ey
ey1c
ey3da
eye1l
ey1k
ey1l
ey1m
ey1na
ey3p
ey1ru
ey1s2
ey3sa
e1z
ez2a
ez2ä
ezei2
ezen2
e4zent.
e4zente
e3zep
e2zer
e2zes
e3zess.
e5zessiv
ez2m
ez2o1f
e3zon
e3zu
e4zue
ezu2g
e4zu2m
e3zü
e3z2w
ez2z
é2c
é1h
é1lu
é2r
é1t
é2te
é1u
é1v
é2z2
1fa2
2fa.
4f1abl
f1abs
3fac
fach3i
f1ader
fa3ec
2faf
fa3fa
f4ah
fahl5ent
fahr5tes
fa3ki
f3aktio
3f2akto
3f2al2
4falg
fall5ent
fall5erk
2fam.
2fame
famen2
2famie
f1amt
2fanb
fan2f
fan2g
fang1t
f1an3k
f1anp
6fantil.
6fantile
f1an1z
4f1ap
f2ar4
farb3er
farb3l
farr3s
4f3art
f3arz
fas4a
f1astr
f1aße
2f3at2
f1auf
f3aug
f1ausb
faus2t
faust3r
f4av
3fax
1fä
fähr1u
3fäl
f1älte
2fäq
f1ärm
f1ärz
2fät
4fäx
2f1b2
fbe1
fber2
2f1c
2f1d2
fde2
fdie4
4fe.
fe1b
f2ech
f1eck
3fe2d
fe1em
fee1t
fef2
3feh2
3f4eie
f1eing
f1einw
f1eis
fe2l
fel1a
1fel2d
felde2
f3elek
fel1er
fel3f
fe3lie
3fell.
3felle
3fells
fel1o
fel4s3oh
felt2
fel1tr
fel1tu
fem2
f1emp
2fen
fen1a
fe4ne
fen3s2a
fen3s2c
1fensi
4fensic
fens2t2
f1ent
fen1ta
fen1te
fen1z
fer4ant
fer1ä
fer2d2e3
f2ere
4feree
fer3eis
fe1rel
ferg4
f4ergr
4ferin
fe3ring
fer3mi
2ferno
fe3rol
fe2rö
f4erra
fer3re
fers2p
f2ert
fer3tie
1ferts
1fertt
f2erz
fe2s2
4fes.
fe3sa
fess2
fest1a
fest3ei
fest1r
fes3w
fet4
f1e2ta
fe1tie
fett1r
feu2
1feue
f1ex
2fé
4ff
ffab2s
ff1ad
ff1ans
ff1ar
ff3ark
ff3arm
ff4art
ff1au
f2fär
ff2e4
ff3ei
f3feld
ffel1l
ff3emi
f3fensi
ffer4sc
ff3f2
ffi2
f4fic
f1fis
ff1lag
ff1loc
ff2lö
ff3löc
f1flu
f1flü
f1fr
ff2rä
ff4ri
ff1ro
ffs2am
ff4schr
ff2s1p
fft2
fft3h
ff1ti
ffus5s
2f1g2
f3ga
fge1
fge3b
2f3h2
2fi.
fia3
fia4s
fi2b
fi3bet
fich2
fid2
2fidi
1fiel
fien3
2fier
fi1erf
2fif
fi1go
2fih
2fij
2fik
fik1o
fik1r
fik1te
3fil
4filag
fil1an
6filaufb
fil1da
4filig
fil1l
2fim
fi1ma
1fin2
2fin.
3fina
4fine
f1inf
fing4
fings2
4finitu
fin3sc
f1int
4fio4
3fir
4fi1ra
fis4
fisch3a
fisch3o
fisch3w
f3iso
2fitea
4fitenn
fit1o
2fitou
2fitr
2fi1tu
2fi1v
fi2z
2fize
2f1j
3f2jo
2f1k2
fka3k
fl4
fla4
f1lad
flag2
flan2
f3lap
f3lats
3f2läc
2f3läd
f3län
f1läu
fle4
f3leb
2f1lein
flekt2
f2lex
2flé
f2lig
1f2lim
f1lin
flo4
3flop
f2lor
1f2lot
f2löc
2f3löf
3fluc
1flug
flus6
flut1o
f2lü
f3lüd
f3lüm
f2ly
4f1m2
fmen2
2fn2
f3na
f1nä
f1ni
f1no
f1nü
3fo
4fo.
4foa
4f1ob
fo1bu
4foce
4fo1d
4f1of1
fo1ha
4foi
fok2
4fo1la
fol4g
fol4k
fol4t
4fom
fo1ma
fo4n
fon1di
fons2
fon1te
4f1op
for4
fo1rad
4forc
6f3org
form3ag
form3ei
fort1
fort3s
4fos
fot2
4fo1ta
4fote
fo1tel
4foth
foto3
4fo1tr
4fov
4fow
f1ox
4foz
1fö
4föf
4f1ök
2f1öl
för4
2f3p2
fpen2
2f1q
fr2
frach2
2f1rad
f1rah
fran2
f1rand
f3rap
2frar
fra1sc
2f1rat
3frau.
2f3rau3c
frä2
3f2rän
2fre.
f1rec
2f1red
2f1ref
2freg
f1reic
frei1g
frei3k2
f1rek
f1rep
fres4
f1rest
3freu
frie4
f2rist
2froa
fro1b
2f1roc
1f2ron
fron1d
f1rot
3frös2
f1ru
fruch2
früh1m
4f1s
f2s1al
f2san
fs3ar
f2s1as
fsau2
f2sauf
f2saus
f2saut
fs2än
f3sc
f4sca
f4sce
f4schan
f4schef
f4scr
fse2
fs1eb
fs1ehr
f2s1ent
f2s1er
f3serv
fs1eta
f2s1id
fs3ke
f3skie
fs3m
f2s1o
f3sol
f3spann
f2s1pas
fs1pel
f2s1ph
f3spl
f3s2por
f2spre
f2spro
fs3s2
f3stal
fs1tät
f4stäti
f3stel
f3stern
fs1th
f2s1tic
f2stip
fs1tis
f3st2r
f4s3tres
f4stüte
f2sty
f2s3un
f3sy
f2sz
2ft
f1ta
ft1af
f3t2ag
ft1an
ft1ar
f3tat
ft3att
f3tä
f4t1äu
f3tec
ft1ed
fte3g
ft1eh
f1tei
ft1eig
ft1ein
ft3eis
fte3m
f1temp
ft1ent
ft1erk
ft1erl
f1term
ft1eti
ft1ex1
ft1h
f2t3hei
fti2
f1tis
f1to
ft1of3
f3ton1
ft1op
ft1or
ft1ot
f1tö
f1tr
f3trai
f3tran
ft3ric
ft1ro
f3t2ru
ft2s1
ft4sam
ft3s2c
ft3sel
fts3i
ft3st
ft5stie
ft6stier
ft3sü
ft1t
ft1url
f1tü
ft3wo
ft3z2
fu2
2fu.
3fuc
fuch2
3fug
1f2uh2
ful2
2f1um
fun2
f1unf
2fung4
f1ungl
f1uni
fun4k
funk1r
funk3tu
f1unm
f2ur
4fur.
4furs
fus4
1fuss
fuss1p
3fuß
fuß1er
3fut2
2fuz
3fü
4füb
füd2
füh2
fün4
fünf3l
für3b
2f1v
2f1w
fy2
2f1z2
f3zei
fzeit7end
f2zi
f3zu
f3zw
4ga.
ga4b
2gabf
2gabg
g4a5bi
gab2o
g1ab2r
gab3ra
2gabw
ga5by
2gabz
ga1ch
2gad2
ga3deb
4gae
ga1fl
ga1fu
2gag
3gag.
2ga1h
2gai
2ga3k
gal2
2gal.
gala3d
g1alau
gal3c
g1alg
2gall
gal3la
gall4e
gal3m
2g1alp
g1alta
g1alu
g2ama
ga1man
gamm2
2g4amo
2g1amt
4gan.
2g1ana
2ganb
2gand2
2gane
2ganf
gan2g
4gangeb
gang4sp
gang3u
2ganh
2gani
2g1ank
2ganl
g2ano
2ganr
2gans
g1ansi
g1anst
2gant
4ganw
2g1anz
2gap
ga1pa
2gar.
g2ara
2garb
gar4c
1gar4d
2gari
2g1arm
gar4o3
gar4t
g1arti
ga1ru
g1arz
ga2s
gas1al
3gase
gas3m
gas4mu
g4asse.
g4assen
gas3sp
gas3ste
gast3el
gas4t3rä
gas1tu
gas3z
gat2a
ga1tab
2gatm
2gato
ga1t2r
2g1auf
g3aug
g2auk
2g3aus
g1aut
2ga1v
2ga1w
1gä2
2gäne
2g1äp
g1ärz
4g3b2
g4by
2g1c
2gd
gd2ad
gdel6
g1di
gd3l
g1do
g2don
g1dö
gd1r
gd3s2
gd1t
g1du
g3dy
4ge.
1ge3a
2gean
1geä
1geb
ge3b2a
3gebä
gebe4am
ge1bel
ge1bes
3gebi
ge5bigk
ge3ble
gebot4
3gebüh
ge1c
1ge1d
4ged.
1ge1e2
ge3ec
ge2es
1gef2
ge3fa
2gefar
3gefec
2g1eff
3gefü
1geg2
ge1gel
ge1gem
ge1ger
ge1ges
ge1gu
3geh
ge1he
ge2her
geh1m
gei2
2g1eif
geig2
g2eil
2gein
ge2int
ge1ir
geis2
g1eise
geist3r
1ge3j
2gej.
1ge3k
2gel.
2gela.
4gelas.
4gelba
gel5bert
2gelbl
gelb3ra
ge1le4
2gele.
4gelei.
gel3el
gelen2
gel3ers
4geles.
4geless
4gel3f
2gelg
2gelh
ge1li
2gelik
gel1im
2gelin
4gelk
4gel1l4
gel3m
4geln
2gelo.
2gelor
2gelp
2gelr
4gels
gel3s2a
gels2p
gels2t
gel3ste
gel1sz
gel1ta
gel3t2r
2gelus
4gel3z2
ge1m6
2gem.
3gemei
gemen2
ge2mon
ge3mos
g3emp
ge3mu
5gemü
2gen.
ge1na
gen1ac
gen3al
ge2nat
gen4aug
ge1nä
2genb
2gend
gen2d1r
4genec
gen3ern
gen6erwe
2geng
2genh
1geni
3genie
ge1nis
gen3k2
2genm
gen1n
1genom
ge2nov
2genp
2genr
2gens
gen4sam
gen3sz
2gent
gen3tal
gen1t2h
gen3tr
ge1nu
2genv
2genw
2genz
genz2e
gen3zin
genz1t
gen3zu
1geo
2geoo
geo1pa
ge1ou
1geö
1g2e3p
3gepu
1ge1q
4ger.
ge1ra2
ge3rann
3gerä
ge3räu
4geräx
ger1d
ge3rec
ge1rei
ge3remp
ger3ent
ge1ret
2gerg
4gerin.
ge3ring
g1erkl
2gerl
ger1no
g1ernt
ge1ro
1gerol
ge1r2ö
ger3re
2gers
gers2p
ger4sto
ger2u
ger3um
ger3v
g3erwer
ger3z
g1erzä
g4e1s
2ges.
2gesb
1ge3s2c
1gese
ges3elt
ges1er
2gesf
ges2i
2gesk
ge3s2p
2gesr
ges3ser
ges3ste
2gest.
6gestan.
6gestani
6gestans
ge3ste
ge4ster
ges1th
ges3z
1ge1t
2get.
ge3tag
g1etap
2geti.
ge3to
getrie4
4get3s
ge3t4u
get3z
2g1e1ul
1geü
ge3v
1gew
g1ex
gey2
1ge3z
2gé
2g1f2
g3fä
g3fes
g3fet
2g1g2
g3ga
g3gä
g3geb
g3gef
g3gela
g3gep
g3gew
g3glo
g2go
g3gr
g4gri
g2gy
2g1h
gh2a
g2hai
g2har
gh2e
gheit4
g2het
g3hi
g4hie
gh3l
gh1n
ghs2
gh1w
gi2
gia2
3gib2
4gibi
gie3be
gie3d
gie3f
gie3g
gie3h
gie3i
gie3l2
gie3m
gien2e
gie3res
gies2
gie3st
3gieß
gie3t4
3gif
gift5s
2gig
2gik
gi3ko
gi3kr
3gil2
gi4m
4gin.
2gina
g1ind
4gin2e
4g1inf
gin2g
g1inh
g1ins
2g1int
4gio
3gip4
gis2
g3isel
2giss
gi3sta
3giß
git2
2g1j
2g1k2
g3ka
g3ki
g2kok
g3kr
gl2
4gl.
gla4
2g1lab
glad2
2g1lag
1glan2
2gland
1glas
glas3k
glas1p
glas3ti
3glat
glau2
1g2laub
g1lauf
1glä
g1läd
g1läß
2g1läuf
gl3b
gl4e
2gle.
gle3a
g1leb
g1lec
2g1led
g1leg
glei4
2g1lein
gleit3r
2g1len
2gler
g1lese
2gleu
g1lev
gl3h
g1li
1g2lid
g2lie
2glif
2glig
2g2lik
2glil
g2lim
2glin
2g2lis
g3list
1g2lit
g3lite
2g2liz
g1lo
2g2loa
g2lob
g2loc
g3loch
2g2lo1k
1glom
2glop
g2lor
g2lot
2glöw
g1luc
2gluf
2g1luk
g1lus
3glü
g1lüg
3g2ly2
2g1m2
gman2
gma1ri
g2mat
g2me
g3mel
gmen2s
g3mes
gn2
3gnad
gnal3er
1gnä
2gne4
g1nel
g2nen
gn3g
2g1ni4
g2nie
g3nis
gno2
g2nom
g1not
g1num
gn3z
4go.
2go2a
2gob
go2ba
gobe2
2goe
g1of1
gof3f
2gog
2g1oh
2go4i
2gola
2gole
gol4f
2golo
gol3s
4gom
go1ma
gon2a
go1nau
gon2d
2goni
2go1p
go2pos
g3opt
gor2
2gor.
gor4b
2gord
2gort
go2s2
gos3i
gost2
go3ste
go4te
4got2h
go4ti
go1tr
gott5erg
3gou
4gow
2göd
2g1öl
gör4
2g3p2
g4pon
g1q
1g2r4
gra4
g4rab
g3radl
g3rah
2grai
g3rak
2gral
gramm7end
grar3
grar1e
grass4
2g3rau1b
grau3f
g3raum
grau3sc
grä2
gräs3c
g3räu
gre2
4gre.
g4reb
2g3rec
g3rede
8gredienze
8gredienzi
2g4ree4
2g3ref
g3rege
gre5gist
grei2
2g3reic
2g3reih
g3reim
g3rein
g3reit
g3rek
gren2
g3renn
grenz5ei
grenz3w
gres4
g3re3se
4gressu
gre3st
2g3ret
2g3rev
2gria
g3ric
2grid
g3riese
3grif4
grin2
g3rind
g3ring
g3rinn
g4rip
gro2
4gro.
2gro3c
gro3i
gron4
gro3no
g3rose
g4ross
g4roß
g4rot
gro4u
2grov
3grö
4g3röh
3grub
g3rui
g3rum
3g4rup
2g3rut
2g3rüc
3g4rün
2g2s
gsa4
g4s3ab
gs1ac
g3sack
gs1ad
gs1af
gs1ag
gs1ahn
g4s3ak
gs1alb
gs1alm
gs1amb
g3sani
gs3ant
gs3ar
gs1as
g3sat
gs3ato
gs1ax
gs1ä
g3säu
g3sc
g4sca
g4sce
gsch2
g4schef
g4sco
gs3d
gs1e4
g3see
gs2eh
gs2eid
g3s2eil
g3s2eis
g3seni
g4sen2t
g4ser
g3sere
g3seri
g4ses
g4seu
gs3he
gs1i4
g3sic
g3sig
gs3ik
g3sil
gs3in
gs3is
g3sit
gs3ita
gs3k
g4s3l
g4s3m
g4sn
gs1o4
gsof3
g3sol
g3son
g4s3op
g5sorge
g3soz
gs1p
g3spek
g4spl
g3spor
gs6port.
g6sporto
g4s3pru
gs3r
gsrat4
gs3s2
gs1tab
g4stag
g3stan
g3star
gs1tä
g5stein
g3stel
gs3term
gst3err
gs3test
g3steu
gs1th
gs2thy
g3stim
g3stir
g3sto
g4stoch
g4stod
g4s1top
g4s1tor
gs1tot
gs1tr
gst2ra
gst5reit
gst3rit
gst3ros
gs1tu
g6s3tub
g3stun
gs1tü
gs1u
gs3un
g4sw
g3sy
gs3z
4gt2
g1ta
g1tä
g1tei
g1tenn
g1tep
g1term
g1terr
g3tex
g1t1h
gt3he
g1ti
gt3l
g1to
g2ton
g3tou
g1tr
gt3t
g3tu
g3tü
gt3w
gu4
3guc
g1uf
g1uh
1gum
4gum4e
3gum2m
gum4p
2gun
gun2e
g1unf
g1ungl
gun2h
g1uni
g3unk
gun2n
3g2un4s
gun2t
gur2
g1url
gurt3r
gurt3s
gus3a
2gusc
gus4s
guss1o
gus4t
gust3ab
gust3en
3guß1
3gut
gut1a
gut3erh
gut1h
gut3s
gut4sa
gut1t
gut3z
1gü
2güb
gür3c
2g1v
2g1w
g2wam
g2wie
1gyn2
2gyn.
gyp2
2g3z2
ha2
4ha.
hab2a
hab2e
hab2i
h1abr
h1abs
ha3bus
h4abz
h1adle
h1adr
2hae
ha3fa
haf2e
h1affä
haf3fl
haft4s
hafts3p
h2agg
h1ah
h2ai
hai1l
h2aj
2ha3kl
hal2
4hal.
hal3ang
h1alar
halb1r
hal3ch
2hale
halen3t
hal4f
hal3ge
h2ali
4hall1a
halo3
2h1alp
hal4te
4haltin
4haltis
hal3v
h1amt
2han.
2hana
h2anbe
h2an4d
hand3r
hand3s
2hane
han2f1
2hani
han2k
hank1t
han4n
2hano
han4so
ha3nu
2hanz
h1ap
3h2ape
ha3ph
har4
h2ara
2hari
h1arm.
h2arme
hart4e
h1arti
h2as2
2has.
2hasa
hash3
has3k
has4v
h1aße
h3atl
hat4m
hat1r
hat3tr
h1audi
hauf4li
hau5flie
h1aufm
h1aufs
h3aug
h1aukt
hau2mi
h2ause
haus3ti
hau2t
haut3s
4ha3z
hä2
h1äff
häl3c
häl3g
2häni
här1d
här2p
h1ärz
häs5chen
2häug
häus1c
häus5le
2h1b2
h2bara
h4barka
h2baro
h2barr
h4bars.
h2bart
h3bat
h2bek
h3ben
h3bo
2h1c
h2coc
2h1d2
hde2
hde4k
hden2
2he.
2hea
he3al
he3as
heb3eis
hech2
hed2
he4dem
he1det
he2e2l
he1e2m
he1eng
he1et
he2f
he3fab
hef1ei
hefe1m
hef3erm
hef3ing
hef1l
1heft
hehl2
he2i2
2hei.
h4eib
2heie
2h1eif
h1eig
2heii
heim1p
2hein
hein3d
hein3er
hein1t
2heio
hei3str
heit2s1
2heiu
h1eiw
3heiz
he2k4
he3kl
he3kr
hekt3a
2hel.
hel1an
he2l3au
hel1ec
he2leg
he3lege
hel3ers
hel4fe
hel1la
hell3au
hel3lei
hel2m
hel2or
he2lö
2hels
he2m
2hema
he3mag
he3man
1hemd
2heme
1hemm
h3emp
2he2n
hen1a
hen1da
hend2s
4hene
hen3ebe
hen3end
hen1et
4heni
henst2
hen3str
hen1t2a
hen1te
hen3tie
hen1tr
h1ents
hen3u
hen3z2
4he2o
h1e2pi
hept2
he2r
2hera
her3ab
heran1
her3ar
her3at
he3rä
herb4
herber4
her8beste.
her8besten
her8bester
her8bestes
her3bl
her2c
h2ere
he3rec
her4eif
her7eises
h1erfo
her4for
her4fri
2herif
her4kas
h1erke
her4kes
2herm
her4mo1
2h2ern
her3ob
h1erö
her8richt.
her8richts
her6rinn
2hers.
her4so
hert2
her1ta
her3tr
her4zap
h1erzä
her2zo
her2zö
her2z1w
2hes.
2hese
hesen1
hess4
hesse2
he3stro
heß2
heter2
he3tint
1hetz
h2eu
heu3err
heu1g
3heu1sc
4heuse
he2wi
hex2a
2h3f2
hfell1
2h1g4
h3ga
hgas1
hge1
h3gel
h3gu
2h3h2
hhau2
h4hold
hi2
4hi.
hi4a2
2hic
hicht5er
2hid
h2id4e
hi3di
2hie.
hie4b
hie3g
hie3h
hien2e
hie4r
hier3i
hier4l
hiers2
hiet4
hi3fa
2hig
hi4h
hi3in
hik1r
hik1t
hil3a
hil2b
hil2d
hil4f
hilo5
h2im2
2h1imb
h4imm
h3impe
hin2en3
h1inf
hin1ge
hin1gl
h1inh
hin2i
hink4
hinn2
hin3s
hin4so
hin2t3a
hin2te
hin1z
2hio2
2hip
hip1h
hip1i
hip1o
hip1s
h2ir
hirm1a
hirn1
hir4r
his2
hi3sp
hiß4
hit2a
hi3th
hit2i
hit2l
1hit2z
hitz2e
hi3un
hi3ur
hi3ver
hiv1o
h1j
2h1k2
h2kent
h2keu
2hl
hlab2
h1lad
hl2ag
h1lai
hl3an.
hlan4d
hl1anz
h1las
h1lat
hlau1b
h3laub.
h3laus.
h1laut
h1law
h1lay
h3läche
h3läd
h1länd
hl1är
h1läs
h1läß
h1läu
hl3b2
hlber4
hl1c
hl1d2
hle3a
h1leb
h1led
hle3e
hle3ga
hlei2f
hle3inh
h3leit
hle3l
hle3ma
hlen1n
hlen5tes
hlen3ti
hl3entr
hl4ere
hl1erg
hl3ernä
hle3run
hl1erw
hle3sc
hle3ta
hle3tr
hle3v
hl3f2
hl3g
hlicht6w
h2lig
h1likö
hl1ind
h1lini
h1list
hl1k
hl1l2
hlm2
hl1n
hlo2
h1loc
hl1op
h2lor
hlos4
h1los.
h1losi
hl2ö
h2lös
hl3p
hl4sar
hl1se
hls3ka
hl3slo
hl3str
hl2su
hlt2
hl1ta
hl1th
hl1ti
hl1tr
hl1tu
h1luk
h3lumpe
h1lüf
hl3v
hl1z
2hm
h1mad
hma3ge
h1mak
h1man
h3mann
hm1ant
h1mar
h1mas
h3maß
h2mäc
h2mäh
h2mäl
hm2e
hme3d
hme1e
h1mein
h3meist
hme3la
hme3le
hme3li
hme3ric
h1mes
hme1st
h2met
hme3ta
hme3te
h1mex
h1mil
h1minz
h2miß
h1mon
h3mop
h1mot
hm2ö
hm1p2
hm2s
hm3sa
hms1p
h1mul
h1my
2hn
hnach4b
hnam4
h1nami
h1nas
h1nati
hn1äh
hn1är
h1näs
hn1c
hnd2
hn1da
hn1di
hn3dr
hn2e
hne2e
hn3ef
hne3g
hn3eig
hn3ein
hnen1t
hner3ei
hn3ero
hn3ersa
hn3ex
hnf2
hn3g
hnick2
hni3co
hni4k
hn1im
hnin2
hnit4
h1niv
hnk2
hn1n
h1not
hn3s2a
hn3se
hns2p
hn1ta
hn3ti
hn1to
hn1tr
hnts2
h2nul
h1num
hn1unf
hn3z2
2ho.
ho2b
ho2c
hoch3
2hocy
2hod
2ho2e
ho3er
ho4f
hof1a
hof3fes
hof1l
hof1o
hof1r
ho1g
ho3ge
hoh2
ho1ha
2hoi
ho2k2
2hol.
hol3a
hol3b
hol3ei
hol3g
holl2
hol3op
hol3s
2hols.
hol1tr
hol3v
h1olym
1holz
ho2m2
h2on
ho1nad
hon2d
hon3di
hon4e
hon1to
2hoo
ho1on
2hop2
ho1pa
h1ope
ho1ph
h1or2an
h1ord
hor3di
2hore
hor2f
h1or2g
hor1ma
hor1me
hor2n
hort3s
ho2s
2hos.
ho3sek
ho4sen
hos4g
ho3si
ho3sl
2hosö
ho1sta
2hot.
ho1tom
4ho1tr
2hot3s2
2hou
ho2v
ho4vi
2ho2w1
h1ox
2hoz
1h2ö
4hö.
höch3l
3höh
hör3b
hör3f
hör3g
hör5l
hör3m
hör3r
hös1c
h3öst
h3p2
hpa2
h1q
4hr
hr3ad
hr1ag
h1r4ah
hr3ass
h1rat
hrau2
h1räu
hr3b
hr1c
hr3da
hr1di
hr2du
hre2
h1rech
h1red
h1ref
hre3gis
hrei2b
h1reic
hr3eig
h3r4eini
hrei3th
h1reiz
h1rep
hr2erk
hr4ern
hr2erw
hr4erz
hres3
hrest2
hr1eta
hr1eu
h1rev
hr3g2
hr4har
hri4
hrie4
h3riesl
hr1int
h2ris
h5ritter
hr3k
hr3l
hr3m2
hr1n
hro2
h3rock.
h1rog
h1roh
h1rol
h2rom
h2ron
hro4p
h1rou
h1rös
hr3r4
hr2s1ac
hr4s3and
hr2sau
hr2sen
hr3sena
hr4sh
hr2sin
hrs3k
hr2s1of3
hr4spro
hrst2
hr4s3tri
hr2su
hr2sz
hr1ta
hrt3ab
hr2tan
hrt3ric
hrt4sa
hrt2se
hrt2sp
hr1tu
hrt1un
hr1ums
hr3v
hr3z2
4hs
h3sache
hs3acht
hs1ad
h4samt
h2san
h3sani
h2s1as
h2sauf
h2säh
h1sc
h4schan
hs2cr
hse2
hs1ec
h1see
h1seg
h2s1ehr
h1sei
hs1eie
hsel1l
h1sem
hs1erf
hs1erg
h1seri
hs1erk
hs1erl
hs1ern
hs4erne
hs3erz
h1sess
h1s2ext
h1si
h2s1id
h2sig
h3sign
hs2im
h2s3ing
hs2kal
h3skand
hs3ke
hs2ler
hs3m
h1so
h2s1of
hsof3f
h2sop
h2spac
h2s1par
h2spat
hs1pel
h2s1per
h2s1ph
h2sprä
h2spro
hs1s2
h1sta
hs3tabl
h2staf
hst3alt
h3stan
hs1tau
h4stäu
hst4e
h1stec
h1stei
h1stel
hs3temp
h3sterb
hs3term
hs1th
hst2he
h1sti
hs1tie
h2stit
h1sto
h2s1tol
h2s1tor
h1str
hst3ran
hs1tu
h1stun
hs1u
4ht
ht1a4
ht2ag
ht4akt.
ht4akte
h1tal
ht3an.
h1tanz
h2tar
h1t2as
ht3at
h1tate
h4tax
ht1ä
h1tän
h1tät
ht1ec
ht1eff
hte3g
ht1ehe
h1tei
h4t3eilz
ht1eim
ht1ein
ht1eis
ht1eke
ht3elas
h3teleg
hte3lie
ht3elit
htel1l
ht3engl
h1tenn
ht3enta
h1tepp
ht3erfü
ht3ergr
ht1erh
ht5erleu
h3termi
h3terra
ht3erre
ht5erspa
ht3erst
ht6erste
ht1erz
ht1ese
ht1ess
ht1eth
ht1eu
htfünf3
ht1h4
ht4heil
ht3hen
hther2
h1thes
h1thü
h1tie
h1til
ht1im
ht1in
ht2int
h1tip
h1tis
h1tit
ht5la
h1to
h2t1ob
htod1
h2tol
ht1oly
h2tope
ht1or
h1töp
h1tr
ht1rak
htran2
ht3rand
h2t1ras
h2t1rat
ht1rau
h4traub
ht1rec
h3treck
h2tref
ht1rei
ht1rel
h2t1res
ht3rieg
h2t3rin
ht1ro
h2tros
h2troß
ht2rot
h2trü
ht2sah
ht4s3an
ht4sein
ht2sen
hts3end
hts3k
hts1o
ht2sp
ht4spin
hts3tät
hts3tem
hts2ti
hts5trau
ht4stri
ht4s3tur
ht4stür
ht1t
h3tue
h1tum
h1tur
ht1urs
h1tus
h1tut
h1tü
ht3w
htz2
ht3za
ht3zä
ht1zen
ht1zer
ht1zin
hu2
2hua
hub1a
hub3b
hub1ei
hub1en
hub1l
hub1r
huh3a
huh1i
huh3m
h1uhu
2huk
huk1i
huko3
hukt2
hul3a
hul1ei
hul3eng
hul1er
hul3in
hul1k
hul1l
hul1s
hul3ste
hul1tu
hul3v
hul1z
2hum.
hum4a
hum4b
1humo
hum2p
hu4n2
h1una
hund2e
hun4f
hung4
2h1uni
hun4l
2hur
hur3g4
hus2
4hus.
hus3h
hus4s
huss3a
hut2c
hut3s
hut4z
hutz3er
hu3w
h2ü
hüb4
h3über
h3übu
hüg2
2hüm
hüpp1
2hür.
2h1v
hver2n
2hw
hwan3s
hwei2
h1weib
h1weih
hweis4s
hwen2
hwer3b
hwer3g
hwer3m
h2wirr
hy2
hydro3
1hym
1hyp
2hyr
4hyt
2h1z2
h2zeh
h2zen
h2zes
h2zi
h3zie
h3zim
h3zin
h3zip
h3zir
h3zis
h3zu
h3zw
iab2
ia3be
ia1ber
iab3t
iad2
ia3da
ia3di
ia1do
ia1e
ia1f2
ia1g
ia3ge
i2ago
ia1h2
ia1j
ia3ka
ia4ke
ia3kl
i3akt
ia3ku
ial1a
ia3labo
ial3d
ial3erm
ial1et
i1alia
ial1k
ial1l
ial3m
ia1lo2
ial1se
ial1t2
ial3u
ial3z2
iam4
i3amp
ia2n
ian3alt
iand2
ian2e
ian3eb
ia3net
i3ang
ian2s1p
ianz1t
ia1o
ia1p
ia1q
i3ara
ia1ro
iar3r
ia1s
ia3sa
ia3sh
ia3s4k
ias4m
iast2
ia2ta
i3at2h
ia1the
ia1tre
ia1tro
i1au
ia3un
iaus1
ia1vi
ia1w
ia1z
iä2
i3äm
i1äp
iät3s2
i3bac
i3bak
i2bar
ib1art
ib1auf
i3bä
i1bea
i1bec
ib1eig
ib1eis
i1bek
ibe2l
ibe2n
iben1a
ib1ep
i1beru
i1besi
i2bet.
i1bez
ibi2k1
ibi2s
i1bla
i1blä
i1blo
i1blu
ib2n
ib2o
i4bö
i2bra
ib1rau
i2brä
ib1ren
i2bric
i2bro
ib2rü
ib3sa
ib2se
ib4s3te
ib3unk
ibus1c
i4busi
i4bussc
i2büb
4ic
ical3
i1car.
ice1
ich1a
ich1ä
iche2
ich1ei
i2chen
ich1l
ich2le
i3ch4lo
ich3m
ich1ni
ich1ra
ich3rei
ich3rit
ich1ru
ich2sa
ich2s1i
ich4spo
ich4sti
icht3in
ich1w
i4ci
ic1in
ic5kos
ick1sk
ickt2
i1cl
ico3b
2id.
2ida4
id2ab
i1d2ac
id1an
idar3b
id1au
id2d
id2e
2ide.
i2dec
1idee
id3ei
ide3k
2idel
idel2ä
ide1li
ide3me
id3emu
i1denk
ideo1
ide2pa
2ider
ide1rö
2ides
ide5sa
ide3si
ide3so
i3deu
ide3v
id4ga
id2har
2idia
i3dik
i1dim
1i4dio
id2lin
id2n
id2o
ido1b
2idoo
i1dot
2idr
id1rin
id2s1p
id3str
id1t2
1idy
4ie2
ie3a
ie3ba
ie5bei
ie3bil
ieb1l
ie3bos
ie3bra
ieb3re
ie4b3rü
ieb4sto
ie3bü
ie3da
ied3an
ie3e
ief1ak
ief1an
ief1ei
ief1f
ief2i
ie3fid
ief1l
ie3ga
ie3gelä
iege3t
ie3gi
ieg4lin
ieg3lo
ieg3r
ie3gra
ieg4s1c
ieg4se
ieg4st
ie3ha
ie3hä
ieh1in
ieh1m
ieh3n
ie3ho
i1ei
i3eie
ie3im
ie3ind
ie3j
ie3ka
ie3kl
ie3kr
ie3ku
iel1a
ie3lä
iel1d
iel1ec
i4elen
iel4erw
iele3v
iel3f
ie1lie
iel1li
ie3lo
ie3lö
iel1se
iels2p
iel1sz
iel1ta
iel1tr
ie3lü
ie3ly
iel1z
ie3ma
ie5mah
iem2e
ie3mei
iem4h
ie3mi
ie3mo
ie3mu
ie4mun
ie3nac
ien1ag
ie3n2am
ie3nat
ien1da
ien1eb
ie3neu
ien3g
ien3in
ien1k
ieno2
ien2s
ien3sa
ien3sä
ien3sc
iens2e
ien3si
ien3s2k
ien3s2p
ienst5er
iens4tr
ienst5rä
ien3sz
ien3tab
ien3tä
ien1tr
ien1zi
ie3o4
ie3p
ie4pe
iep3to
ie3q
ier3a
ie5raum
ie5räu
ie3rei
ier3eis
ier4erb
ierf2
ier5g
ie3rie
ie3rit
ierk4
ier5la
ier3m
ie3roh
ier3r
ier4s3eh
ier4sk
ier3so
ier4spe
ier3sta
ier3ste
ier1ta
ier1ti
ie3rüc
ier3v
ier2ze
i2es
ies3an
ie3sä
ie3sc
iesen3k
iesen3s
ie3so
ie3sp
ies2s
iess3ti
ie3sta
ie3stel
ie3str
ie3stu
iet3a
ie3tab
iet3erh
iet3ert
iet3ho
i3ethy
ie3tie
ie3tit
iet3ös
ie3tr
ie4tri
iet3ru
iet4se
iet3z2
ie3um
ie3un
ieu1s
ie3v
ie3wa
ie3wi
i1ex
ie3ze
ie3zi
2if
i2f1ab
i2fad
i3fah
if1ar
if3arm
i2f1au
i2fazi
i1fed
if1ef
i1fei
if2e4n
if1erg
if1erh
i2fex
if2f
iff4e
iff2s1
iff4st
ifi2
i3fin
if1l
if2la
i1f2lä
if2le
if2lu
i1flü
if1n
i1fra
i3frag
i1fre
if3reif
if2s
if3sa
if3se
if3si
if3sp
if2t
ift1a
ifte4
if3tre
ift3ri
ift3sp
ifts2t
ift3sz
i1fu
2ig
iga3g
i2gan
ig1art
iga3s
ig3att
ig1d2
ige3ber
i3gefä
ige1g
i1gei
ig1ein
i1gel
i2gelt
ige3m
i1gema
i2gen
ige2na
i3geno
ige2ra
i3geric
ig1erz
i3gesc
i3gese
i1gesp
i3gew
ig2fr
i1gil
i1gle
ig2mu
ig2na
ig2ne
ig2nu
i3go2n
i2gona
ig4ra
i2grä
ig3rei
ig3sa
igs2ag
ig3sä
ig4schr
ig3s2o
ig3sp
ig4spa
ig4spi
ig3sta
ig3stei
ig4s1to
ig3str
ig4stre
ig3s2tü
igung4
i2gü
i1gy
2i1h
i3ha
i4h1am
i4har
i3hä
i2he
ihe1e
i3hef
i3hei
i3helf
ih1elt
i3hera
ihe3re
i3her2r
ih1l
ih1m
ih1n
i3ho
ih1r
ih1ti
ih1um.
i2hung
ih1w
ii3h
i1im
i1in
i1j
i2ji
1ijs
i3jü
2ik
i2kab
i2k1ak
ik3amt
ik1ang
ik1ano
ik1art
ika3sc
ik3att
i2k1au
ik1är
ike2
ik1ei
ik1ens
ik1ere
ik1erf
ik1erh
ik1erl
ik1eta
i1kic
ik1in
ik1l
i1k2lä
ik3lö
i1klu
ik1n
iko3b
ik1off
ikon4z
ik2o3p
ik1ori
ikot1t
ik1q
ik1rä
ik1res
ikro1
ik1s
ik3sa
iks2z
ik1tab
ikt3erk
ikt1re
ikt2u
ik1tü
ik1uh
i2kum
i2kü
il3abs
il1ac
ila3f
i1lag
i2la3h
il1ak
il3ama
il1ans
i1las
il1asp
il1au
i1laub
il4aufb
i2l1är
2ilb2
il1dat
il2de
ild3ent
ild2er
il1di
ild1o
il2dr
2ile
i1leb
il1ec
i3lehr
i1lei
il1ein
il1el
ile3li
ile3na
ilen2i
il1ent
il1erf
il1erg
il1erh
il1err
i2les
ile3se
ilett2
i1leu
i1lev
il3fa
ilfe3s
il2fi
il2fl
ilf1le
ilf1re
ilf4s1
il3ges
il2gl
il3gr
2ilh
i4lichs
ilien3
i3lif
ili4g
ilig1a
ilik4
il1ind
i3linie
il3init
i1link
il1ins
il1ipp
il2j
il1k
ill2a
ill4an
ill2e
il1lei
ill4fr
ill2k
il3lu
il2m
ilm1au
ilm1ei
il1n
2ilo
i2l1ob
il2of
il2oh
ilo1k
il2op
i2l1or
i2lot
ilot3s
ilo1w
il3p
ilt2
il1ta
il1th
il2ti
il3tie
il2to
ilung4
il1ur
2ilv
ilz1er
i1mac
ima3d
imal3m
i1mann
im3aren
im1arm
ima1s
im4at
imat3s
ima3z
imär3s
im3ber
im2bi
im2bo
im1d
4ime
ime4d
i1mei
im1ein
imel2
im2en
i2mer
im1erf
imer2s
im1erz
ime3sh
i3met
imi3a
im1ind
im1inf
im1ins
imi3se
im2man
im4me
imm3ent
im1mis
1im3mo
im4mod
im4mö
im1n
i2mog
im1op
i2mö
1imp2
im2pal
im3par
im3pi
im3pla
im3po
3impor
im3pu
im4ro
im2se
im3ser
im2so
im3sph
im1ti
imt3s
im1tu
2im2u
i2mun
im3unt
i1my
i2n
2in.
2ina
in3ab
in1ac
in4acht
in1ad
in3ae
in4a1f
in4alp
in1am
i3namen
i4nan
in3ann
in1ap
ina5po
i4nar
in3arb
in2are
ina3res
ina1sc
in3att
in3au2
2inä
in1äh
2in1c
in4ci
2ind.
2inda4
in1d2ac
in1dar
2indä
in2deb
in2dep
indes4t
1in3dex
in3dic
in1do
in2dol
in1dö
in4drec
in2dro
ind3se
ind3sp
3indus
in3d2ü
4i4ne
in4e5len
in1eng
ine3p
in3erbi
in2erh
in1erk
in3ertr
in2et
in1eu
ine3un
inf4
1info.
1infos
in2fra
2ing
ing1af
ing1ag
ing1al
ing1ar
ing3at
ing4sam
ings6por
ings3pr
ing5stä
ing2th
1inhab
2inhar
2inhau
2inhe
ini3b
in2id
2inie
2inig
ini3kl
ini3kr
2inis4
ini3se
in1it2
3inkarn
1inkas
in2kav
ink3ent
ink3na
inks3t
ink3tie
ink3tis
ink2ü
4inn.
in2nen
in2nes
in2net
4innl
1innta
2ino
ino1b
in2ol
ino1n
in1or
ino1s
ino3t
ino3v
ino1w
in1öd
2inp
2ins.
in2san
2insä
insch2
2inse.
4insed
in3seli
in3sem
2insen
in3s2i
2insk
ins2t
3instal
in4s3tät
4inste
3instit
in3str
in2su
1insuf
2insy
in3s2z
2in1ta
in3tä
1integ
in3tent
in1th
in3tie
in1t2o
in3tor
2in1t2r
int3ren
int4ri
int3s
in3tu
in1u4
in3unz
2inw
inz2e
inzel3l
in3zem
in2zer
inz2i
in3z2u
io3a
io3bo
io1c
i3od
io1da
io3e
io1f
iof2l
io1g
io1h
io1k
io2kl
io1la
io1lei
io1m
io2mat
i2on2
ion3au
ion3d2
ionen1
io3nik
ion3k
ion3n
ions1
ions3a
ions3el
ion4spe
ion4spi
ion4stä
ion3t
ion3z
io1p
i5o2pf
io1q
4ior
ior1c
ior4e
ior2g
io1r2h
ios2
io1sc
io3sh
io4so.
io1st
io1sz
iot2
io1ta
io1tel
ioten1
io3th
io3to
io1tr
io1w
i3ox
iön1
i1pa
ip2an
i3par
i1pä
ip1ein
ipf2
iph2
i2plem
i2plen
ip4m
ipo1c
ip2p
2ipp.
2ippik
2ippis
2ip4po
2ipps
2ippu
i1pr
ip2s
ipt2a
ipt2u
ir2
4ir.
2ir4a4
4i1rad
i1rau
i1raz
2i1rä
ir3äh
ir3bu
ir4e
i3reg
i3rei
2irek
iren2e
ire1t
i2reu
ir3fl
ir4g4
irgs3
2irig
ir4k
irk3l
ir4l
irm1ag
irm1ei
irm1o
2irn
ir3nad
i1roc
i1roh
1iron
iro4s
i1rö
ir3ra
ir4re3ge
ir3ri
irs2
irsch3r
irsch3w
ir3sh
ir3sin
irt2s1t
2iru4
irus1
i2rut
i3rü
2is
i2sa
i3sac
i4sach
i3sai
i4sam.
isam3b
is1amt
i4san
isan1g
i2sä
is1än
4isc
isch3eh
isch3ei
isch1l
isch3re
isch3ru
isch3wu
i2scr
ise1c
ise1d
ise1e
ise1g
i1segl
iseh2
is2el
ise3lit
i1seme
i2sen
isen3a
isen1k
isen3s
is1erg
is1erm
i2ses
is1ess
is2et
i2set.
is3etat
is2eu
is3ho
is1id
isie4
i3siev
i1sil
i1sim
i1sinn
isi3st
is3kel
is3ker
is3kr
is6kusse
is2ler
is3lig
is3lo
is2me
is3met
is2mo
is3na
is1of
3isol
is2on
is1org
3i2sot
is1pa
is2per
is1ph
i4s1pic
is1pu
is4saa
iss1ac
is3sag
is3sai
is3sal
is2säl
is4s3che
is3senk
iss3erf
isser4t
is1sit
iss3l
is1soc
is1sor
is1soz
is4s3per
is4stec
iss1tr
is4stri
is4sub
iss1um
4ist
ist3ac
is2tal
is3tang
ist4e4
i1stel
is1th
is2ti
i3stic
i1stil
ist2o
istör3
ist2r
is3tras
i1s2tro
is1tur
i1suf
i3suk
isum1
is3urt
i2sü
is2z
is3ze
i2ß
iß1ers
iß3l
iß3ri
2it.
i2tab
it1ab.
it1abs
i2tad
i3tag
ita2l
ital3l
it1alt
it1an
it3anr
it1apo
it1are
it1art
it1au
i3tauc
i2tauf
i2taut
2itä
it1äs
it4dem
2ite
it1eff
ite3g
it1ei
i3t2eic
it2eil
4i2tel
itele4
ite3leg
ite3li
iten3s
iten1z
ite3p
it2erö
it1ess
i3tetr
i2t1ex
it1hil
itho8logische
iti2m
it1in
it2inn
ition4
it3iss
itler4s
it3lu
it3m
ito1c
i1tod
it3of3
i3tonu
it1op
it3org
2i1tr
it1rad
it1raf
i3tram
it1ras
it1rau
it1räu
it1re
i2tref
i2tro
it1rom
it1ruc
it1run
it2rü
it2sa
its1ag
it2s1e
its3er1
its1o
its3tem
2itt
it2ta
it4tag
it2tä
it4tel
itt3hä
it1t2he
itt4seh
itt4sei
itt2sp
itt4sti
it1uh
it1ums
it4ung
i3tül
it3wo
i2ty1
2itz
it2z2er
itz3erg
itz3erl
it2z1w
4i1u2
i2um
ium1a
ium1i
ium3n
iur2g
ius1
i3ü
iv1ad
iv1ak
ival1l
iv1am
iv1an
iv1ef
iv1elt
iv1ene
iv1ent
iv1erf
iv1erh
iv1erl
i2ves
iv1ex
ivil1l
iv1t
iv1ur
2i1w
i3wä
i3we
i3wo
2ix
ixt2
ix3ta
2iz
i2za
i3zac
iz1ag
iz1ap3
iz1au
ize1c
izei1c
izei3g
izen2z
i2zer
iz1erl
ize1t
i2zis
iz3n
i2zor
iz4sä
iz1th
i2z1w
iz2z
iz3zw
j2a
2ja.
2ja3b2
3jag
jahr3ei
jahr4s3
ja3j
jal2
2jami
2jan2
janit4
jas2
ja3sa
ja1sc
ja1so
ja1st
jat2
ja1ti
2jau.
ja2wol
3jä
jäh1l
je2
2je.
jean4
jed2
4jektir
jekt3r
jel2
je4ma
je4p
2je4s2
jet3a
jet3h
jet3r
jet3s
jet3u
je4v
2jew
ji4
2jia
jo2
2jo.
job1
jon2
jord2
2js
j2u2
jugend3
jul2
jum4
jun2
jung3s2
jur4
jus4t
jut2
jute1
2j2v
ka2
4ka.
k1aa
ka3ar
kab4b
2kabn
2k1abo
ka3bra
k1abs
k1abt
2kabw
2kabz
k2ad
k3ada4
k4ade
2kade.
4kaden.
4kadenz
kade2r
2kadi
k3adm
k3adr
2kae
kaf2
2ka3fe
ka3fl
ka3fr
k2ag2
ka3ga
ka3ha
kai1sc
ka3kl
2kal.
2kala
kalab2
kal1d
kal3eri
2kalg
kal2k
kalk1a
kal1lä
kal1lo
k1allt
ka3log
kal1os
2kals2
kal1se
kal3tra
kam4
ka3mei
3kamer
5kamm
k2amt
4kan.
1kana
kan3as
3kanä
kan2d4
4kanda
2kan2e
kan3k2
2kanl
2kanna
1kanon
2k1ans
k5antenn
k1anth
kant3s
kanz4l
kanz1t
kao1
1kap
3kape
2kara.
2karas
kard2
ka3ref
ka3rei
kar4fu
k2ar4g
k2ar3k
2k1arm
3karn
kar3p
kar4pf
k2ars
kart2
k2arta
kar3tal
2k1arti
kar3uf
k2arw
kar4wen
kas2
4kas.
kas4m
ka3so
kas4s2
ka4ste
kat1an
3kateg
2kati
kat3se
kau2
kau4f
kauf1o
kauf4s3a
kauf4sp
kaufs7tem
kau3g
k1au4s
k2aus.
kau3ta
kau3tr
2kay
1kä2
3käf
k1äh
k1ämi
k1änd
käs3c
3käu
2k3b2
kbe1
2k3c
2k1d2
k3dä
kde2
4ke.
ke1b
keber2
k1ec
ke2da
ke2de
kee2
2k1ef
1ke2g
2ke3ga
kege2
ke3gr
3keh2
ke1her
kehr2s
kehrs3o
3kei
k1eic
4k1eig
4k1ein
ke1ind
4k1eis
keit2
keits1
ke2l
2kel.
kel1a
kel2ag
keld4
4kele
kel1er
kel3g
kell4
kel1la
kel3lei
kel1li
2keln
kel1o
2kels
kel1se
kel3sk
k4elt
kel1ta
kel1tr
kel1z
k1emp
ke2n
4ken.
2kena
ken3au
ken1da
ken3dr
ken3end
ken3ern
1kenn
kenn2a
kenn2e
ken1ni
3kennt
2kenr
ken3s2k
kens4te
ken3sz
k3ente.
k1entg
ken1th
ken3ti
k1entl
ken1tr
k1ents
ken3z
ke1o2
4kep
ke2r
4ker.
ker1ä
ker3f
k3ergeb
ke3ring
kerk2
k2erko
k3erleb
ker2n
k3erneu
kern5eur
k1ero
ker1oo
k3eros
ker3r
kerz2
k1erz.
k1erzi
2kes.
ke2s3a
ke2se
1keß
4ket.
ket3a
ket3eb
keten3t
ket3ha
kett1h
1keu
ke1up
k1ex
2k1f2
2k1g2
kge1
k2gos
2k1h
k2hac
k2ham
k2hart
k4hee
4ki.
2ki1a
ki2ad
ki3ak
ki1be
ki1bu
4kicl
ki1da
2k1ide
ki1di
2kie4
kiel3o
kier2s
ki1f2
ki3fa
2kig
ki3h
2ki1k2
1kil2
2ki1la
ki1lä
2ki1le
2ki1li
kilo3
2kim
ki1ma
ki1me
ki1mi
ki3nat
kinde2
k1indi
k1indu
1kine2
4king
kinge2
king3s
k2ini3
k2inn
kin2o3
4kins
k1inse
k1inst
4k1int
kio4
ki1or
ki2pe
ki1pi
3kir
ki1ra
ki1re
4kiri
4kirt
4kis.
ki3sa
2kisc
ki3si
kis2p
2kistl
kit2
kitz2l
ki1v2
4kiz
ki1ze
2k1j
2k1k2
kkab2
kke2
k4kei
k3kor
k2kub
k2kug
kl4
4kl.
2k1lac
k1lad
2klade
k2lam
klan2
1klar
k1last
klau1d
klaus4t
2k1läd
klär3c
kle4
4kle.
2k1led
k1leg
2kleh
k2leid
k3leit
2klen
k1leng
k2les
1k2let
2k1leu
k1lie
3klien
3klif
2klig
3k2lim
klin2
3klina
3klip
k1liz
klo3br
k1loc
k3loh
k3lok
klo1sc
2klose
2klöc
2klöf
k2löst
k2löt
klung4
2k1lux
k1lüc
k2ly
2k1m2
kman2
1kn4
3knab
kne2
k2neb
knie3g
kno4
k1nor
k1nov
3knö
k1num
k1nus
3k2nü
1ko
4ko.
4k1oas
2ko2b
kob1la
kob4s
3koc
ko2d
ko3e
k1ofe
koff4
ko2g
ko3ge
ko3gr
koh4
ko1hab
kohl2e
ko2ho
ko2j
ko2k2
2koka
ko3kan
ko1ki
ko2l
ko3le
kol4k1
kol4m
kolo2
ko3log
2kom.
ko4me
kom4mas
2ko2mo
3komp
k2on
2kon.
kon1d
kon2i
kon3s4
kon3tu
3konz
kon5ze
kop2
4kop.
2kopa
ko4pe
kopf5err
2ko1ph
4kor2a
kor4b
kor5bac
korb3r
k1orc
kor4d
4kori
kor4k
kor2m
kor2n
korn1a
korn3ä
korn3o
kor4p
2korpi
kor2s
2kor2t
k2o2s2
2kos.
2ko3sc
ko5sin
ko4sk
kos3p
ko2ta
4koth
ko1t2r
kot3s
4k1ou
ko3un
4kov
ko3va
ko2vi
2kow
2k1ox
2koz
1kö
k2öf2
k1öl2
kör4p
2k1p2
kpa2
2kq
k1qua
1kr2
4k1rad
2k1rah
2krai
k1rats
krau2
2k2raz
k2räc
2kräd
k2rän
k1räum
4kre.
k1real
3kreb4
4k1rec
2krede
2k1ref
4kreg
krei4
k1reic
k1reif
k1reih
3kre2m
kre3mi
4kren
k1resu
k1rev
2krez
2k1rh
kri4
2krib
2kric
k2rieg
k1ries
2krip
k1risi
2krol
k2ron
kro1ph
kro1sc
2krot2
4k1rou
krum2
k2rü
2ks
ks3ab
k2sal
k4s1amt
k2san
ks2ana
ks3ar
k2sau
k3sau.
k3saue
k2säh
k2s1äl
k3s2än
k1sc
ksch2
ks1eb
ks1ec
ks1ei
k1s2eid
ks2eif
k1sel
ks2end
k2sent
k1seq
ks1er
ks2ere
k1seri
kser2v
k1si
k2s1id
k2s1in
ks3is
ks3ke
ks3li
k4s3m
ks3n
k3son
k2sop
k2s1or
ks1pa
k3s2pat
k2spä
k3spe
k2s1ph
ks2por
ks3s4
k1st4
k2s1tal
k4s3tanz
k3stat
ks3tat.
ks2te
k2stet
k3steu
ks1tie
k4stier
ks1tis
k2stit
k2s1tor
k4strop
k2s1tum
k2s1tur
k2stüt
k2sty
ks1u
ks3un
4kt
kt1abs
kt1ad
kt1akt
kt1am
kt1an
kt3ank
kt1ar
kt3are
kt3ars
k1ta2s
ktat3te
kt3au
kta2v
k1tä
kt1äu
k2tea
k1tei
kt1ein
kte3li
kte4n
kten1e
kt1ent
kten3z
kt3erfo
kt1erg
kt1erh
kte1ru
k2tex
kt1h
kti2
kt3ing
kt1ini
ktion4
k1tit
k1toc
kt1of3
kto1l
ktor3k
k3tran
kt1ras
kt1rau
kt1res
ktro3me
ktro3s
kt1run
kt1s
kt3s4a
kt3sä
kts2el
kt3ser
kt2s1o
kts2p
kts2t
kts2z
kt1t2
kt1ums
ktur3t
kt3w
kt3z
ku2
2ku.
ku3be
2kud4
ku3dr
1kug2
kuge2l
k1uhr
2kui
kul4
2kule
2kulin
2kulo
2kulö
4kulp
2kulu
kum2e
k3uml
kum2p
kum4sa
kum2sp
kun2
k2una
3kunf
2kung4
kun4m
kun4s
kunst3
4k1up.
3kur2
kur3ans
kur3ei
kur3g
kur3m
kurs1c
kur4sp
kur4str
k1urt
2kus.
2kuso
2kusr
4kusse.
4kussen
3kussiv
kus1ta
2kusv
2kusw
1kü
3küc
3küh
2kül
kül3s
2küne
kür4s
3kürz
2k3v
2k1w
2ky2
3kyb
ky3li
ky3m
3ky4p
2k1z2
kze2
k3zu
la2
4la.
la3ab
l1ab
l2ab.
l2ab2a
l3aben
la3bew
2labl
2labn
l2ab2o
l4abor
7laborations
labor3t
2labr
2labs
la3bur
2labw
2labz
2lach.
lacks2
l1ada
2ladd
lad2i
l1adl
2lad2m
2l3adr
1l2adu
2laf
laf3fes
laf3s
laf3t2
la4g
2lagg
lag3l
lag2m
lag3ma
lag1ob
lah2
2laho
lais1t
lake2
l2akk
la3kl
lak3tu
2l1al2
l2ama
2lami
lamm1a
1lammf
2lamn
l2am4p
l1amt
lamt2s
la3nac
l1anal
2lanc
1l2an2d
land3a
land5erw
land5erz
land3l
land3rü
land6spo
lan2ek
2lanf
lang3s2
2lanh
l2anhe
4lanl
2lanm
l1ann
2lano
2l1anp
4lans
l1ansä
4lant
lan3tel
lanti2
l3antw
4lanw
4lanz
lan2z1w
la3or
2lapa
2l1apf
l1apo
la3pr
2lar
lar3an
lar5ba
lar3ei
lar3ene
lar3f
larg4
lar3gl
lar3ini
lar3k
lar3l
lar3n
lar3re
lar3s
l1art
lar3ta
lar3th
l3arti
lar3tr
la1ru
la3rus
lar3ze
2lasd
2lash
2la4sp
las3sei
3lasser
2lassü
las2to
las3tur
4lasu
1laß
laß1th
la3tab
lat2e
la4th
lat2i
2latin
2latm
lat2o
lat1ra
latt3in
latt1r
lat4z
4lau.
lau2b1r
laubs3t
lau2f
1laufb
lau4fo
1laug
lau1gl
lau1h
lau1ma
1laun
l2aus.
lau2sa
2lausf
2lausg
4lausw
lau1ta
laut3s
2lav
la3vo
4lawe
la4z
lä1a
2läe
2läf
2l1ähn
l1ämt
länd3l
2länz
lär3b
lär2c
lär3k
lär3sc
lär2t
l1ärz
lä2s
läs3c
2läse4
2lät
2läub
2läuc
2läue
lä2z
2l1b
l3bac
lbau3c
lbdi3
l3beg
lb3eink
lb3eise
l2bem
l3bep
l3best
l3bet
lb1e2ta
l3bez
lbi4
lb1id
lb1ins
lb2lat
l3blä
lb1li
lb2lo
lb2lu
lb2or
l2bou
lb1rea
l2brec
lb2ru
lb2rü
lb2s
lb3sa
lb3se
lb4sh
lb3si
lb4sk
lb3so
lb3sp
lbs2t
lbst3ac
lbst3ei
lb4sto
l3bud
lb1uf
l2bum
l3bü
2lc
l1ca
lch2au
lch1le
lch1li
lch1m
lch1n
lcho4
lch1ob
lch1r
lch3s
lch3ü
lch1w
lch3z
l4ci
l1cl
l2cu
4ld
lda4
ldab1
l1d2ac
ld3ack
ld1ah
ld1ak
ld1al
l1dam
ld1an1
ld3ari
l3darl
ld1arm
ld1au
ld1är
ld2bre
ld3d
lde3g
lde4h
l1del
lde3lis
l1dem
ld1emi
l3depo
ld2erl
ld1erp
lde5sa
ld1ess
lde3str
l1dez
l1dia
l1d1id
l3dien
l1dif
l2din
l3dio
l1dis
ld3l
ld1n
ld2ne
l1don
l1dor
ld1ori
l1d2ö
ld1r
ld3rea
ld2ros
l3d2ru
l2ds
ld3s2a
ld3ska
ld3s2t
ld1t2
ld3tu
l1due
ld1um
le2
4le.
le3ak
lear3t
1leas
le3ba
leben2
lebens3
leb3re
2lech
lecht4e
le3die
2lee.
le3ei
2lees
4lef2
le3fe
l1eff
le3fl
leg1ab
leg1as
lege1b
lege3h
leg2g
le3g4r
1leh2
le3ha
le3hei
le3her
lehr2e
2lehs
2leht
lei3bel
lei5berg
l2eic
l2eid
2leien
leif1a
leif3s
2l1eig
lei2h
lei1l2
l2ein.
l2eind
l2eine
2leinf
l1einn
leins2
l3einsa
l2einu
le4is
leiss5er
l2eit
leit7ersc
leit3s
leits4t
le3kr
2lel
l2ela
le3lag
le3lä
l1elf
l2eli
l3ell
lel1tr
2lem
4lema
le6mark.
lem1au
le4mä
4leme
le3met
lem1o
lem4p
lem3s
4len.
2len1a
len3ab
le4n3an
4lenb
len2cy
4lend
lend2r
len3dro
l4endur
4lene
len1ed
len3end
4lenh
len3kli
lenk3lo
4lenm
4leno
le3nov
4lenp
4lenr
4lens
len3ska
lens2p
len3sz
4lent
len1ta
len1tr
4lenw
l1enzy
le3of
leo2m
leo4p
le3pa
l3epi
1lept
l2era
ler3as
ler1ä
ler3bu
le3r2ec
l3ereig
ler3eim
ler3eis
ler3f
l1erfo
l2erfr
l2erfü
l3ergeb
ler5gu
2lerik
l2erka
l1erke
l2erko
l4erlei
2lernä
l4erne
2l1erö
l2erra
ler4sto
le3rüc
ler3v
l4erwa
l1erz
2les.
les2am
les2bi
les2e
l1esel
le3s2h
le3s2k
les2n
2lesse
les2t6
lester6i
le3sti
le3sto
le3stö
le3str
les2z
le3tex
4leth
let4i
4lets
letts2
leu2
2leud
le3uf
1leuk
l1eul
1l2eut
le3vis
2levo
lex1t
le3z
2lé
2l1f
l3fah
l2fast
l2fe
lf1ec
lfe1e
lfe1g
l3fei
lf3einh
l3fel
lfe3le
lfe1m
lf2en
lfe1na
lf1erl
l3fes
lfe3sc
l3feu
lf1f
lf3lan
l3flä
lf1led
lf1lo
l3flu
lf2ru
lf2rü
lf2sa
lf2spe
lf2s1ti
lf2su
lf2tr
lf1tu
l3fu
2l1g
lga3d
l2gam
l3gau
l4gäu
lge2br
l3gef
l3gei
lgen2a
l4gers
l2geti
l2gi
l3git
l2go
l3goa
lgo1l
l3got
lg4p
l3g4ro
lgs1t
2l1h2
l3ha
l4hard
lher2b
li2
4li.
2l2ia
li3ak
li4at
li3bal
1lic
4lica
lich2t
2lick
lid2
l1ido
li1don
lid3sc
lie4
4lie.
1lied
3liefer
2lien
lien1a
lien3s
lier2e
2lierz
lies1c
4lieu
li4f4
2lifec
2lig.
2lige
li3gei
lig3ers
li3gr
li4gre
2ligs2
2lii
2lika
li3kli
4liko
lik2sp
likt1o
lik2u
lil2
li3lan
li3lau
li3lit
lim2a
lima1c
lima3g
lima1l
l1imb
li3me
lim2m
1limo
li3mor
4lin.
4lin2a
lin4ar
2linda
lind2b
2lindu
4l1inf
2lingh
lings1
2l1inh
lini4
4lini.
l3inj
link2s
2lino
lin2q
l2insa
l3insel
linsen3
l1inst
l3insu
l1insz
2l1int
lin3tes
4l1inv
4linw
lin2zo
4lio
li3pf
li3pr
1liq
li3rem
lis2
4lis.
2lisa
li3sal
lis4b
2lisc
li3scho
li3sek
2lisi
2lisk
l3isl
lis4s2
li3sy
lit2
lit4a
4litä
li4te
3lit4h
lit3r
lit1s
lit3sa
lit3st
lit3sz
2lity
lit4z
4litz.
4litzc
2liu
lium1
2liv
liv2e
li3w
2lix
lizei3t
2l1j
4lk
l1ka
l3kale
lk1alp
lk2an
l3kap
l3kasu
lk2au
l1kee
l1kei
lk4er
lker2e
l1kess
l3ket
l1ki
l3kit
l1kl
lk2no
l2koh
l3kol
lko2v
l3kr
lk1roc
l4k1ru
lk2s1
lk4sp
lk4stä
lkt2
lk1tü
lk2ü
l2kür
lk2wa
4ll
ll2a3br
ll3acht
ll1aff
lla3kr
ll2al
ll1am
ll3ama
ll2anw
ll1anz
ll1arm
ll3aufg
ll3aufk
ll1aus
ll1är
ll3b2
ll1c
ll3d2
lle3ber
l1lec
ll1ech
lle3d
ll1ef
l1leic
l1leid
ll1eim
ll3eise
l2len
lle4na
ll3endl
llen3dr
ll3endu
llen5tes
l2ler
ller1d
ll3erfa
ller6gen
ll3ernt
ll3ertr
ll1ex
ll3f2
ll3g2
l1lib
l1lief
ll1ind
l1litz
l3lize
ll1k
ll3l2
llm2
ll3mo
lln2
ll1ob
llo2d
ll1ol
ll1opf
l4l1or
llo2t
ll1ou
ll3p
ll2säu
ll1se
ll2s1es
ll3sk
ll2spr
llt2
ll1ta
ll1th
ll1ti
ll1tr
llt3s2
ll1tu
l4lun
ll1urs
ll3v
l2ly
l3lym
ll3z2
llzu3ge
2l1m
lma4
lm3ab
l3mac
lm1ad
l3mag
lm1aka
lm1am
lman2s
lm1apf
lm1art
l3mä
lmäh3
lm1äst
lm1c
lm1d2
l3med
lm1ef
l3mel
lmen4sc
lm1e2p
l2mer
lm1erl
lm1erz
lm1id
l2mig
lm1ind
lm1ins
l2mis
l3mit
lm1m
lm1n
l3mol
l3mom
lm1orc
l3mö
l4möl
lm1p
lmpf4
lm3se
lms2t
lm3ste
lm3str
lm3s2z
lmt2
lm1th
lm1ti
lm1tu
l3mus
2ln
l1na
l1n2ä
lnd2
l1n2e
l2nen
l3no
l1nu
l1nü
4lo.
lo2b
3lob.
2l1obl
l2obr
lo2di
2loe
2l1of
lof3f
lo4fo
3logi
6logiert
lo2gn
1logo
logs2
log3t
loh2e
2l1ohr
1lok
lok3r
3loks
2lol
lo1la
l1oly
2lom
lo2mä
lom4b
lo2mi
lo2mo
lo1mu
2lo4n
3long
l1onk
lon2n
lon1o
lo2o
4lop
lo3pa
l1ope
lo2po
lop2pr
lor2an
3lorb
l1orc
l1or4d
lo2re
l1or2g
lor1m
lor3p
lor2q
lort2
1lo2se
lo3sec
lo3shi
los3ka
los3m
los3ste
los3ta
los3to
los3t2r
2loß
l2ot
lot1a
lo1tä
lo2t4e
lo2th
lo1tr
lot2s
lotz3k
lo2v
lo3vo
2low
lo1wi
1loy
2loz
1lö
2löck
2löd
2l3öfe
l1öhr
2l1öl3
l2ölu
2löp
lör4
2löß
2löz
2l1p
l3p4ar
lpar4k
l2pe
l3peg
l3pes
lp4f
l2ph2
l3phä
l3phi
lp1ho
l3phr
l3pht
l3phy
l2pi
lpin3s
lp4m
lpo2
lp4r
l3prei
l3pro
lpt2
lp1tr
l3q
2l1r2
lrat4s
l4raun
lre1s
l3rh
l3ro2
lrut4
l3rü
4ls
ls1ad
l3s2al
ls1amb
l4samp
l2sanf
l2sang
l2sank
l2sann
l3sarg
l3sark
l2sas
l3sat
l2sau2
ls3aug
l3s2äm
ls2äug
ls1äus
l1sc
l4schen
l2sco
lse4
ls1eb
ls1ec
l1see
l1seg
l1sei
l2sein
l1sel
ls2ele
l2s1eli
ls1emb
l2sent
ls1er
lser2v
ls1eta
l1sex
l3s2hi
l1si
l2s1id
l2simp
l2sind
ls2kal
ls3ke
l3skele
ls2kle
l4sko
l4sky
ls2log
l1so
l2sop
ls3ort.
l3sos
l1s2ö
l2spac
l2s1ph
ls2po
l3spul
ls3s2
l1st2a
lstab4
ls3tabl
ls2taf
ls1tas
l4s3täti
l1stec
l1stei
l3stel
l4sten
ls3terr
l3steu
l1sti
ls1tis
l2stit
l1sto
l4stoch
ls1tor
l1str
l3stra
ls2trä
l1stu
l1su
ls1um
l2sun
l3sy
2lt
lt1abs
lta3d
lta2g4
ltal2
l1tal.
l1tale
lt1alg
lt1am
l1tann
l1tanz
lt1ara
l1tas
lt3ato
lt1au
l1tä
ltde2
lte3g
lt1eh
l1tei
lt1eig
lt1ein
lt1eis
l1tel
lt1elt
lte3mi
l1temp
l1tenn
l1teno
lten4sa
lten4se
lt3erde
lt2erg
lte3rie
l3term
lter2n
lt2erö
lt1esk
lt4est
lte3str
l1teu
l3texa
lt3g
lt1h
lt3hag
l2thei
lt2hu
lt3hun
l1tic
ltie4
l3tier.
lti1k
lti1l
lti3na
lti1r
lti1t
lti3u
lt3l
l1to
lt1of
lt3org
lt1ori
l2tos
l3tou
l2tow
l1tö
lt1öl
lt1ös
lt1öt
ltra1l
l3trane
l6trans.
l1trap
lt1räu
lt1re
lt3ris
lt1rol
lt1rö
lt1ruc
l1t2rü
lts2eh
lt2sei
lt2sel
lt1so
lt4s3ort
lt2spe
lt4s3tab
lt2sti
lt1t2
l1tug
lt1uh
lt1um
ltur1m
ltur3z
l1tü
lt3w
lt1z
lu2
2lu.
2lub1
lub3s
luch2
luck2s
lu3do
lues3
1luf4
4l1ufe
2luff
luft1
lug1a
lug1eb
lug3erp
lug3g
lug3l
lug1n
lug3r
lug3se
lug3sp
l1uh
lu3he
lul2
lu3lö
lum2b
lum4bi
l1umd
lumen3
l3umk
lum2n
l2ump
l1ums
l1umw
2lun
l1una
l3unf
lun2g
4lung.
lung4sc
l1uni
lun2k
l1uns
2luo
lu3pl
lur1an
l3urk
l3url
l1urn
lu3rö
l3ur1t
lu4s2
2lus.
4lusc
lus3k
lus3p
lus4s
luss1c
luss3er
luss3or
luss3p
1lust
lust1o
lust3re
lus3u
2luß1
2lut.
lut1a
lute4
lut1eg
lut3erg
lut5ersa
lut1of
lut1or
lut1r
lut5schl
lut1t
lut3z
luz2
lü2
2lüb
3lübd
lück4e
lücker3
lüh1l
lüt2
2l1v2
l2va
l2ves
l2vi
l3vil
l3vo
4l3w
l4wang
l4wau
2lx
1ly
2ly.
ly1a
ly1ch
ly1et
ly1l
lym2
ly1mo
ly1n2
3lync
ly3pr
lys2
3lyt
ly1th
ly1u
ly1z
2lz
lza2
l2z1ag
l2z1ap
l2z1äp
l2z1är
lze2
l3zei
l1zel
lz3enth
lz1erh
l3zett
l1zi
lz1id
lz3l
l2zof
l2zor
lzt2
lz1tep
lz1th
lz1ti
lz1u2fe
lz1ur
lz1w
lz2wis
l2zwu
4ma.
2maa
2m1ab
ma3bal
m2a3be
ma1ber
ma1bes
m3abf
ma3bi
ma3bli
m2a3b2r
ma3bu
ma3bü
1mache
mach2t
macht7ers
mack4st
m1adm
mad2s
ma1en
ma1er
ma3fe
ma1fr
ma1fu
1mag2
ma1gar
2m1agg
ma1gl
2m1ago
ma1gor
3mah4
ma1he
mahl2s
mai1d
mai5sone
mai1v
3maj
mak2
ma3kar
ma3kat
ma3kon
m1akt
mal2ag
mal1ak
mal3at
2mal3b
mal1da
mal2er
4mal1k
mal1li
m4al3p
2malq
mal1ta
mal3ut
mal3va
2malw
ma4n
1manag
m3anal
man3an
m1anat
man2d2
man3dr
man3ers
m2anf
mang2
3m4angel
m1angr
m2anh
2manin
3manip
man2n
1mann.
2mano
1manö
mant3s
2manz
m1anza
2mao
2maph
2mapr
4mar.
4marag
m3arb
2mare
mar4g2
3mark
4markl
mar4kr
mar4li
marm2
2mar4o
2marr
mar4sp
mar4su
1marx
m1ar4z
mas2
4mas.
ma3sa
ma3schm
2mase
2masp
mas4q
1maß
m3atel
mat3erd
mat3l
m4atme
2mato
mat3se
mat3sp
mat3st
mat3tag
mat3tal
mat3t2r
mat3url
mau4
1maue
m1auf
1maul
ma3un
1maur
maus2e
ma1v
ma1w
may1t
2ma2z
ma4ze
1mä2
2mä.
2m1ähn
4mäi
4m1änd
2mäo
4m1äp
mär3g
3mäs
3mäß
mät4
mäus1c
2m1b2
m2bab
m3bac
mba3p
m3bat
m3bau
m4be.
mbe2e
m3beg
m3beh
m3bei
m3bek
m4bek.
m4beke
mben1t
m3bere
m3bet
m3bew
m3bez
m4bik
m3bil
m4bit
m4bi3z
m2bl
mble3i
mbol3l
mbol3s
m4bra.
m4bris
m4bug
m3bun
m3bur
m3but
2mc
m3ca
mc3ca
m3ch
m1cr
2md
m1da2
m2dan
m2d1as
md3ato
md1är
mde2
m1def
md1erl
m3di
mdien4
md3le
m1do
md3s2e
md1t2
md1um
md3w
4me.
me3a
me1b2
mecht2
1me2d
me1ef
mee1m
me1ene
mee1ru
m1eff
me1g4
mega3s
3meh2
mei2
m1eif
m1eig
1m2eil2
mein4d
mein6har
m2einu
m2eist
me1k
me2l
me3lag
mel4ant
me3lä
1mel2d2
melde3i
3meldu
m3elek
me3leu
melt2
mel1ta
mel1tr
mel2z
mem2
me3me
m1emis
m1emp
m1e2mu
me2n
2men.
me4na
men3ab
men1ac
men3an
men3ar
men3au
2mend
2mene
2menh
men1k
2menl
2men3r
m2ens
menst2
mens8werth
m4en2ta
men1tr
men3tri
men1z
4meo
me1ou
m1epi
me1q
me2r
me4ra
meral2
mera5lin
mera3um
mer3d
me3rec
me3rel
mer3f
m3ergän
mer2k
m1erne
2mers
mer3sm
mers2p
mer1th
me3rüc
merz4en
mer2zi
mer4zu.
2mes.
me2sa
2mesb
mes2e
me2si
3mes2s
mes4sa
mes4sä
mess1o
2mest
me1sto
me3s2z
3meß1
meß4ban
meß8buch.
meß8buchs
meß4kun
meß8platz.
meß4rau
meß6schn
meß3u
meßun1
1me2t
2met.
me3taf
meta3g
me3tar
me3tau
2meti
me3tim
3metro
m1ex
mex1t
2m3f2
mfahr4t
2m3g2
mge2he
mge3l
mge1r
2m3h
mher2r
mi2
4mi.
2mia
mi3ab3
mi3ano
mi3ar
mi3b
mi4bi
mic1e
4mid
2mie.
mie3b
mie3c
mie3dr
mie3fa
mie3l
mie3m
mie3no
mien3s
mier3z
mie1s
mie4t
miet3i
mie4z
mi3fu
1migr
4mik.
mik1an
mi3ko
mil2
mi3lag
milch1
mild4s
mi3leg
1mill
2milz
mi3ma2
m1im2m
2mimp
4min.
4min2a
mi3nac
min5anze
4minb
3mind.
minde2
m1indu
4minf
4ming
min2ga
min1ge
ming2h
ming3s
ming2w
2mini.
minik2
mini3t
min2o
min3of
m1inse
1minü
4mio
mi3os
mi3p
mi3q
2mir
mi3re
mir3s
mis2
mis3a
misch3w
mi3see
mi3sei
mi3so
mi3sp
mis4s
mi3sto
mi3str
1miß
miß1e
1m2it
mita2
mit1ak
2mitä
mit1ed
mit1er
5mitg
mit3h
2miti
mit3s2
mit5sa
mit1ta
mit3tan
mitt2e
mit1u
4mit1z
2miu
mi3w
3mix
miz2
mi3ze
2m1j
2m3k2
mkin2
2m1l2
m3la
ml3c
m3leb
m3leh
m3lei
m4let
m4lig
m2lis
m3list
m3lo
ml3p
ml3s
m2lu
4mm
m1ma
mma3a
mm1ak
mm1al
mm1ans
mm1anz
mm1art
mm2as
mma1se
mma1st
m2mata
mm1au
mm1äu
mm3b
mm1d
mme2
mm1ef
mm1ein
mm3eise
mmel1l
mm2ene
m2mens
mmen5tal
mmer3a
mme4re
mmer3k
m1merz
mmes3a
mme3sc
mm1eu
mm1id
mmie4
mmi3m
mm1inb
mm1inf
mm1inh
mm1ins
mm1int
mmi3r
mmi3sc
mmi3st
mm2le
mm1m
mm1na
mm3ni
mm2nu
m1mo
mmo2p
mm1p
mmpf4
mm2s2
mm3sa
mm3si
mm3sp
mm3sta
mm3sti
mm3str
mm1ti
m1mu
mm2un
mmun1z
m2my
2mn2
m3nac
m1nä
mne2
m2nen
m1ni
mni1p
m1no
m1nö
m1nu
2mo.
2moa
1mob
mo2c
3m2o2d2
model2
mode5ric
mode1s
4modo
4mody
2moe
m1o4f
4mog.
mog3al
moge2
1m2o2h2
2moi
2mok
mok2k
m2ol
1mom4
mo3ma
m2on2
2mon.
mo1nau
mond1r
4mone
7monnaie.
mons2
mon3sa
mon3se
mont2a
mon3th
mo1ny
2monz
1m2o2o
2mop
mo1pa
m1ope
mo1ph
mor2
1mora
mor3ar
mor4d
mord3a
mor3f
morgen5s
2mori
mor4r
mort2
mor3tr
mor3z
2mos
mo4ska
4mosp
mos3s
moster4
mot2
4mo1tr
mo2v
mo3ve
mo3w
2m1ox
1mö
4mö.
3möb
möd2l
4m3öf
3mög
2mök
2m1öl
mör3f
m1ört
4mp
m1pa
mpan2
mpar4b
m3paß
m2paz
m1pä
m2ped
m2pel
mpel4mu
mp1ene
mpf1ef
mpf3erf
mpf3erg
mpf3erp
mpf3err
mpf3erz
mp2f3l
mpf1or
mp1hos
mp2l
m2plu
mpo3b
mpo1m
m3p4r
m8pressionisti
mps2
mp3sc
mp1str
mp3ta
m1q
2m1r4
mre1b
mre2g
m3ro2
m2rö
4m1s
m2san
ms1as
m3sat
m2sau
m3sä
m4s1än
m3sc
msch2
m4sco
mse2
m2s1ed
m3see
ms1ef
m2sein
m3sek
m3sem
m2sen
ms1ene
m2ser
ms1erf
ms2erh
m3sic
m3sign
m3sit
ms3ke
m3s2ki
ms3m
m3sol
ms1ori
ms1ped
ms2por
m2spot
m2spro
ms3s4
mst2
m4stag
ms3tal
m3steh
m3stel
m3steng
m2stet
m3steu
m2stit
m3sto
m3stu
m3stü
ms1ums
m2sun
m2sü
m3süc
m3sy
m2sz
4mt
m1ta
mt1ab
mt1ak
mt1ar
m1tä
m3tea
m1tei
m3teil
mt1ein
mt1elt
m1temp
m1tep
mt1erf
mt1erg
mt1erl
mt1ers
mt3ert
mt1eta
mt1eu
mt1h
mt3ho
mti2
mt1in
mt3l
mt2ne
m1to
m1tö
mt1öl
mt1ös
m1tr
mt1ro
mt2sa
mt2s1e
mt3s2ka
mt1sor
mt4stät
mt1t
mt1um
mt1urt
m1tü
mt3w
mt3z
mu4
2mud
3muf2
m1uh
mul4
3muld
multi1
m4um4b
3mumi
m1ums
mum2s1p
m1unf
mung4
m3ungeb
2mungs
mur1uf
m4us
mus1a
1musc
4musen
1musi
mus2k
mus3kr
mus1o
2musp
must4
mus3to
muße3
mut2h
mut3sc
mut4str
2mutt.
1mü
2müb
2müc
3müd
3müh
mühl1a
3mül
3mün4
5müthigt
2m1v
m3ve
m3vo
mvoll1
2m3w
mweg4s
my2
2my.
my4a
my3kl
myl2
my3la
myn2
myo3
my4s
2m3z2
mzin2
4na.
na2a
2n1ab
n2aba
n3abd
na3bes
n4abi
nab2rü
4n3abs2
nab3u
1nac
nach6bar
3nachf
3nachm
3nachsc
nach3sp
3nacht
nacht7ers
3nachw
2nada
nade4l
2n1adl
2n3adm
2nado
n1adr
2nae
na1es
2n3af
nag2a
na1gar
2n1agg
n1agi
3n2ah
nah1a
n3ahn
n3aho
nai2
n1air
2n1ak
na3ko
na3kr
n3akt
na3kul
na3kun
2nal
nal1a
nal3am
n2al1d
nal3ent
nal3erm
nal1et
nal1ex
nalf2
nalg2
n2ali
nal1ku
nal1la
nal3lei
nal3m
nal1s
nalt2
nal3tan
nal1tr
2nama
2namb
3name.
1n2amen
namen4s3
n3amer
na3mn
n2amo
n1amp
2n1amt
namt2s
n1a2n
n3an.
2nana
2nanb
2nand2
nan1eu
4nanf
2nan1g
2nanh
2nan3k
2n3anl
3n2ann
n3anna
2nano
2nanp
n3anr
2nans
2nant
n3anw
2nanz
nanz3l
nanz1t
na1ot
na4p
1napf
n1apfe
n1ar
n2ar1a
2nard
na1rei
nar3ent
2nari
na1rin
n2ar4k
n3arm
2n3art
n4arta
na3sä
na1sit
nas4s1c
2nast
nasyl2
1naß
2nat.
2nate
2natm
nat2o
2nats1
nat3st
1natu
n1au
nau1d
2nauf
nau1ma
2nausb
nau1sc
2nausd
naus2e
2nausf
2nausg
n2auso
2nausr
2nausw
na1w
2nay
1na2z
nazi3t
nä2
2näb
1n4äc
n1äf
1näg
3nähe
4n1ähn
2näl
4n1äm
2n3än
4näq
2när
när4s
när1t
n1äs
1näß
2näu
2n3b4
n5ba
n4be.
nbe2in
nbe1n
nber2
nbi2
n5bod
n4büre
2nc
n1ca
nc3ab
n1cel
n2ces
n2chi
nch1m
nck2
n3cl
n3cr
n1cu
4nd
nda1f
nd2a4g
n1dah
nd1ak
nda1ma
nd3arr
n1dars
n1das
nd3att
nd1au
nda1v
nd1c
nd3d
n1deal
nde3alt
nde1e
nde3g
n1degu
nde3k
nde4län
ndel1l
ndel4sa
n2dem.
ndens4e
nden3sk
nde3o
n2der
nder3m
nder5ste
nd2es
n2des.
nde2se
nde3sig
nde4spe
ndes3s
nde3v
n1dia.
n1diä
n3diens
n2dier
n1dif
n1dik
n1dim
n1dio
nd4ir
n1dit
n1diz
nd4lac
nd1n
nd3ni
n2dog
ndo3ge
n3dok
ndo2mi
n2don
n2dop
nd1or
ndo3st
n2dot
ndo3z
nd1rat
nd1rau
n2d1re
nd1rif
n2drog
n2d1rö
n3dru
n4drui
n2d1run
nd2sor
nd4sper
nd2spr
nds3zi
nd1tep
nd1th
nd1ti
nd1t2r
n1dua
4ne.
ne2a
ne3am
ne3art
ne3at
ne3au
1neb2
nebe2l
ne1ber
ne1bes
n1ebn
2nebu
ne2c
3nece
neck2a
1neckt
ne1di
ne3ein
ne3eis
nee1n2
nee1r2
nee1t
n1ef2
ne2g4
ne3gä
n1egg
n1ego
3negr
3neh2
4neha
4nehä
4n1ehe
ne1her
4nehi
ne2hin
5nehm
4neho
n1ehr
ne1hu
n1ei
2nei.
nei2d
2neie
nei2g
3neigu
nein1d
2neis
n1eks
nekt2
ne2l
n1ele
ne3leu
nel2i
ne3lie
ne3lin
3nelk
nel2l
nel3lö
ne3l2o
2nem2
n1emb
n1emi
n2emo
n3emp
n1ems
ne2n
2nen.
nen3a
ne3nac
n2enc
n1endb
n1endd
nen3dep
n1endf
n1endg
n1endh
n1endk
n1endl
n1endp
n1endt
n1endw
ne4ne2
nen1eb
nen1i
n2en3k2
1nenn
nens2e
nen3sk
2nent
nent2a
n1entb
nen5tier
n1entl
n1ents
ne4n3u
n2en1z2
ne2o1
neo3b
neo2p
neo3r
ne2p
ne3par
ne3pu
2ne2r
ner3af
ner4al
ner3an
ner3ap
nerb2
n3erbe.
n3erben
n3erbie
n2erbr
ner1de
n1erf
n1erg
n1erh
n3erhö
ne3rit
n1erlö
n1ermi
n1ernä
n1ernt
n1err
nerrieg5
n1ersa
nert2
3nerv
n1erz
n2e2s
2nes.
ne4sal
ne3san
ne3säng
ne3s2c
nes1ev
ne3sh
ne3ska
nes1of
ne3sol
nes1or
nes1pa
nes2s
2nessb
n3essi
ne3sta
1neste
nest2l
n3estr
2neß3
net3ak
ne3tal
net3an
n1etat
net3au
net3emp
n1eth
net3ha
2neti
ne1tor
net3sc
net3ta
n3etu
3netz
netz1i
1ne2u
3neub
neu3c
neu1d
3neue
neu3erf
neu3erk
neu3ers
neu3erw
neu1f
neu1g2
neu1k
neu1l
n2eun
n1eup
neur2
neu1s
neu1te
3neuv
ne2v
n2e3vi
ne4vol
2n2ew
n1ex
nex1t
2ney
2ne2z
ne3ze
2n1f
nf1ak
nfalt4
n3f2ang
nf4ar
n3f2ä
n3fel
n4fer.
n4fers
n3fes
n4fex
nf1f2
n3fi
nf2l
nf2o
n4fora
nfo1s
nf2r
n3frag
nfra1t
nfrau4
nf3s2
nf2t1r
nft4st
nf1u
nf2z
nf4zi
2ng
n1ga4
n2g1ac
ng1ad
ng1ak
n2gal
n3gala
n3gale
ng1am
ng2amb
n2gamm
ng1and
ng1ant
n3gard
ng4aus
ng1av
n2g1äl
ng1d2
n3geb
n1gec
n1gei
ng1ein
n6gelaus
n1gele
n4gelic
nge4lis
nge4los
n2gelt
n1gem
ng2en
ngens2
ngen3ze
ng1erg
nge3rie
ng3erse
n1ges
n3gesi
nges2t
ng2fe
ngg3s
ng2häu
ng1id
n1gin
ng1lac
ng1län
n2glic
n1gloc
ng3m
ng1ne
ng3ni
ng1no
n2go
ng1ope
ng1opf
ng1or
n3got
ng3rai
ng3rat
ng3roc
ngs3an
ngs3au
ng4scr
ngs3eh
ng4som
ngs3pa
ng4stan
ng4stä
n2gue
n2gum
n1gy
2n1h2
n2hac
n4ham.
n3hau2
n3he
nho4
n2hud
2ni.
nia4
nib2
ni1bi
ni1cen
nich2
nich3s
ni1cr
2nicu
2n1id4
n2idi
ni2e
2nie.
nie3b
nie3c
3nied
nie3fr
niel2
nie3la
nie4r
1nies
2nies.
4niestr
1nieß
4nif
nife4
4nig
nig1a
ni2ge
n3igel
ni2gl
ni1go
nig3r
nig4sp
ni2h
2ni2k
ni4ke
nik3ing
ni3kli
niko4
nik1r
nik1t2
n2il
ni1la
nil3d
nil2i
nim2b
ni1mo
n3imp
2n1in
ni3n2ac
nin1al
ni3nä
n2inb
nin1de
n3inv
4nio
nio4r
ni1ra
nir4w
ni2s
ni4sa
nis1e
1nish
2nisi
nis1in
nis2ku
nis2mi
1nisn
nis1or
nis1p
nis3pe
1niss2
2nissa
nis3sk
nis1sz
ni3sta
nis3tha
nis1to
nis1u
1niß
2ni2t
n2ita
nitan2
nit4o
3nitr
nit3s
nit2t
nitt5erk
nitt4sa
nitt4s3c
2niu
nium1
ni1v2
1nixe
n1j
n2jas
n1ka
nk1abr
nk3ad
nk2al
nk3alg
nk1ang
nk1apf
n2kare
nk3art.
nk2as
n2k3aus
n2käh
n2k1äp
nke1d
nk1ei.
n1keil
n1kell
nk2er
nk3erfa
nk4erg
nk3erle
n1ket
n1ki
n3kic
nk1inh
nk1ins
n3kist
nkkom1
n1kl
n3klag
nk1lis
nk1lit
nk3m
nk2na
nk1nes
n2k1nis
n3kon
nk1ope
nk1ori
nk1ort
n2köl
nk2rab
nk1rät
nk1ro
nk3roc
n2krö
nk1sen
nks2t
nks4ti
nk3s2z
nk1tel
nkt3it
nk1tor
nk1tr
nkt3ric
nk1tü
n1ku
nk1ums
2n3l2
nlan2
nlau2
nle4
n3m2
nmen2
n1na
nna3ber
nnach4b
nn1all
nna1ma
n2nan
n2nau
n1n2ä
nnä3l
nn1c
nn1d
n2nec
n4ne3g
n1neig
n1nek
n2nel
nn2ens
nnen3ta
nn4ergr
nn2erh
nne3ros
nn3erwa
nnes1e
n1ne4st
n1net
nn2eu
n1n2ex
nn1g4
nn2hi
nn2i2
n1nic
nni3l
n1nim
nni4s
n2niss
n1niv
nn1n
n1no2
nn1obl
n2noc
nn1op
nn1ori
n1nö
nn2sa
nn4sam
nn3sen
nn3s2p
nn4s3pe
nnst2
nns3tat
nn2stö
nn1ta
nn1tel
nn2th
nn1tr
nnu4
nn1uf
n1num
n2nun
nn1unf
nn1ur
n1nü
nnvoll4
n1nym
nn1z
4no.
2nob
n2obel
n1obh
n1ob2s
no1bu
2n1of
nof3f
no3g
2n1oh
n2o1he
no3j
2no1k
4noke
n1okk
nok2l
4nol
no1la
nol3c
no1li
no2lin
nol3s
n1o4ly
1nom2
2noma
no1man
4nomet
2nomf
nomie3
2nomp
2no1mu
no2my
2non
n1onk
2nop
n2o3pa
no1ped
no1phi
n1opp
3noppt
n1ops
no1q
nor2
2nor.
n2orak
nor4d
nord1r
nor4e
nor4f
2nori
no3rin
3norm
nor3s
2n1ort
nor4w
nos4
no3sa
no1sc
no3sh
2nost
n1osth
nost2r
no2su
not1ei
not3in
not3op
no2tr
3nots
4nott4
2noty
no2v
nover4s
no1wa
n3ox
2noz
no3zi
2nöd
2nöf
n1ök
2n1öl
1n2öt
n3p4
npa4
npf2
n4plo
npo2
n1q
4n1r4
n2regr
nrei2
nre3sz
n3rh
nro2
n3rö
n3ru
2n1s
n3sabo
ns1ad
ns1agi
n3sai
ns3ak
n2sang
n2sann
n3s2arg
ns3ari
nsa3rin
ns4ath
n3sau.
n3saue
n2saus
n3sause
n2saut
ns2ax
n3säl
n2s1än
ns2äug
n2s1äus
ns3b
n6schobe
n4s3chor
nsch7werd
n2sco
ns1eb
ns2e2h
nseh5ere
nseh1r
ns6eins.
ns1eke
n4sekti
n2se2l
n4sel.
n3selb
ns2ele
n2sem
n3semi
n2sene
n2sent
n3seq
n2serb
n3seren
ns1erf
ns1erg
ns1erk
ns3erle
ns3erne
ns1erö
ns1ers
ns3ertr
n3serv
ns1erw
n2ses
n3sess
n2s1eu
ns2ex
ns3hor
n4si.
n3sic
ns1id
n4sier
n3sig
n2simp
nsinn2
nsinns3
n4s3int
n3skal
ns3ke
n3skel
ns3m
n3smara
ns3n
n2so.
n3soc
ns1od
ns1of
n2sof3f
n3soh
n3sol
ns4om
nson2s
nsor3b
ns2or1c
ns3ort.
n3sos
n3sou
n3sow
n3soz
ns2pac
n3spe
ns2pei
ns4pen
n4speri
n4spers
n2s3ph
ns2pi
n2spo
n2sprä
n4s3prie
n4spro
ns3s2
nsse2
ns3tabl
ns3tag
nst1ak
ns3tale
nst3ane
n3star
n4s3tat.
n4s3tate
nste2
nst3eif
n5sterbe
nst5erge
n3steri
n5sterne
n5sterns
n3s2teu
ns3them
n3stic
n4stilg
n3stim
n3stit
nst5opfe
n4strie
ns4trun
nst3ut
n2sty
n3suf
ns2um
n3sump
n2s1un
n2sur
ns1urs
n3sy
n2sz
ns3zent
ns3zie
2nt
nt2a1c
nta4f
n3tag.
n3tagn
n3tags
n3tagt
n1tai
n2take
nt4a2l
n1tam
nt1ant
n5tante.
n1tanz
n1tap
nt2ar3b
nt1ark
nt2arm
nta3rot
nt1art
nta2s
nt4at
n1tau
n1tä
n3täg
nt1äm
n2täte
n2t1äu
nte3au
nt1eb
nte1e
n1tege
nt1eh
nte1he
n1tei
nt1ein
n1tek
n1tele
nte3me
n1temp
nt2en
n1tend
n1tep
nte2r
nt2erh
n4teri
nt2erk
nt2er3m
nt2ers
nte3sa
nt1ess
n1the
nt1ho
nt2hu
n2t2i
nti1d
n3tierh
nti1kl
nti1ma
ntim3p
nt3inf
nt3inh
n3tint
n1tip
nti1pe
nti3sa
n4tiu
nti1z
nt3l
nt4lit
nt3m
nt3ni
nto3b
nto2mo
nt3orie
n2tos
nto2t
n3tou
n1tö
n2trab
n1trak
ntran2
n2trap
nt4raum
nt1rea
nt1rec
nt3reif
nt2rep
nt3ric
nt3rieg
nt3rin
n1trio
n1trop
n2tros
n1trö
nt3rüm
nts2ah
nt3san
nt4sang
nt1so
nts2p
nt4s3par
nts2t
nt2sto
nt3str
nt3t2
n1tu
n2tul
n1tü
n3ty2
nt3z
nu2
nuar3
1nud
2nue2
2nu3f2
nu4fe
1nug
n1uh
nu3hi
2nui
3nuk
1nul2
n2um.
num1a
2n3umb
2numl
1numm
2n1ums
2n3umz
n1une
nung4
n1ungl
n1uni
nun2m
2nur1
nu3re
nur4i
nur4m
2nus.
4nusg
nu3skr
nu3spo
nuss3er
nus4su
nus1t
3nuß1
1nut
nut1a
n3uto
nut3r
nu3v
2nüb
3nüc
nü3fu
nü1ges
nür3b
nür3c
nür3r
1nüs
1nüß
1nüt
n3v
n4vers.
nvor3
2n3w
n4wang
n4war.
nwei2
ny2
2ny.
ny3b
nym2
ny3r
2nz
nza2
n2z1ag
nz1al
n2zan
n2zar
n3zar.
n2z1au
n2zän
n2z1är
nz3b
n1zec
nze4d
nzel3a
nzel3la
nzen3so
n1zent
nzen1z
n1zep
nz1erh
nz1erl
n1zerr
nzes2
nze3sk
nz1eta
nz3g
nz1id
n1zif
nzig4s
n2zin1g
nz1ini
nz1int
n1ziv
nz2le
nz3n
n2zof
n2zop
n2zor
nzo2s
nz3s
nz3temp
nz1th
nz1ti
nz1wa
n2z1wä
n2zwet
n2zwir
n2zwö
n2z1wu
nz1z
o1a
2oa.
o2ad
oad4s
o2ah
oa1in
oak2e
oak1l
oal2
o2ala
oal3g
oan2n
o2a2s
o3asi
o3ass
o1ä
2ob.
o2bab
obach4
o3bak
ob2al
obal3l
ob2am
ob2ar
o3b2as
2o3b2ä
2ob2b
ob2e
2obe.
2obea
obe1b
o1bec
2obef
o3befe
ob3ein
2oben
obe4na
obend2
obens4e
1oberf
ober3in
obe3se
2o3bev
o2bew
2ob2i2
o3bit
ob3ite
3obj
o1bl
ob1lei
ob1li
2o3b2lo
2o3b2lö
ob2n
2o2bo
o3bon
ob1or
2obö
ob1rei
2ob2rü
ob3s2h
ob3sk
obs2p
ob2sta
2obu
2o3bü
2o3by
oca3s
o1ce2
och1a
ochan1
och1ec
och1ei
och1l
och1m
och1n
och1o
och3ö
och1r
och1s
ocht2
och1w
ock5ente
ock2er
ock3sa
ock1sz
o1cl
o2cod
o1cr
o3cy
od2a
o1dar
od2d
ode2ga
ode4l3ag
odel2s
o4den
oden1e
oden3g
ode2r
o1derm
ode3sp
odes3tr
o1dia
o2din
4odisti
od2ka
4od2n
o8dramatis
od1t2r
4o1du
o2dun
od2w
4o1dy
o1e2
o5eb
o2ec
oen4d
o2es
o2et4
2ofa
of1ag
ofa3l
ofa3m
of2ang
of1au
2ofä
of1är
of1eb
of1ec
of1ed
o1fee
of1ei
2ofer
of1erb
of1et
of1eun
of1fi
of4fil
off1in
of4fir
1of3fiz
off1l
of4fo
off1r
off3sh
off3si
off3sp
of4fü
2o1fi
o2fi1b
ofi3be
o2fic
o2fi1g
ofi1k2
ofi3lig
o2fi1s
2o1fl
of1le
of1li
of2lö
2ofo
4ofö
2o1fr
of1rä
of2s1
of4sam
of3sä
ofsof3
ofs2tr
2oft2
of2te
of1th
2o1fu
2og2
oga2
og3ab
og3ang
og3ans
o3geb
og3ei
ogel1d2
ogel1i
ogen2i
ogen1t
oger2
o2ge2t
o3geti
og3f
og3ins
og3li
o1glo
og3mu
o1gn
og3ori
og3rau
ogs1
ogs2c
og3si
og3sp
og3ste
og3th
og3to
o2gu
og3w
2oha
oh1alk
o1hä
oh1eis
ohen2g
ohen3s
oh1ert
oh1erz
o2hes
2o1hi
2ohl
ohl1a
ohle3b
ohle3c
ohl1ei
ohl3erh
ohle3s
oh1lo
ohl1or
ohls2
ohl3se
ohm4
oh1ma
oh2n
ohn1ac
oh3nat
ohn1d
1ohnm
ohn1o
ohn3sk
ohn2th
2oho
ohol3e
ohol3t
oh1op
2ohö
oh3öl
ohr3a
ohr2sc
ohrt4r
oh1s
oht2
oh1ta
oh2u
oh1w
2o1hy
oh3z
2oi
o1im
o1in
oin1de
oin3do
o3in1g
o2ink
oin4t
oi1th
oi4w
o1j
o2ja
oja1m
o3jet
ok2a
okab2
oka3bl
okal1l
o3kap
o2kar.
oka3st
ok3d
oke2
o2kel
o1kl
ok1lau
ok1lä
o2kle
ok2n
ok2o2
o2kod
oko3ri
oko3t
2o3kr
oks1p
oks2t
okt2
o1kul
ok2z
ol2
2ola
o3lal
ol3ant
olar3m
ol4as
2o1lä
ol3ät
olb2
ol3ba
ol3bi
2old
old1am
olde2
ol4dr
2ole
o1leb
ol3ef
ol3eie
ol3eis
oler2
ol3exz
ol3fr
ol4fra
olge1g
olg3r
ol3h
olien1
o1lif
olik2
o1lim
ol3ka
ol3ko
olk1re
oll1ac
oll1ak
oll3ans
oll1ec
oll3ein
ollen3k
oll3erw
ol4lo
oll1ov
olls2
oll3sp
ol3ma
ol3mi
2o1lo
olo1k
olo1p
o2lor
ol3org
ol3ort
o2los
ol3r
ol3sä
ol4sh
ol3si
ols2k
ol3sp
ol4ster
ol3tag
ol3t2h
ol3t2r
ol3tu
ol3unt
o2ly
oly3ph
o3lys
olz1a
olz3ern
2om
o1mac
om1all
o2man.
om1art
omaso3
om1a2sy
oma1th
om1au
o2män
o2mäu
om2b
om3bel
om3bi
om3bl
om3br
om4buc
om1d
om1ebe
o1mec
om1ef
om1ei
ome3la
omen4sc
omen2t
om1erh
ome3ru
om1erz
o3met
omiet3
o3mil
o1mim
om1ind
om3ing
om1ins
o2mis.
o2mita
om3meng
om3ment
om1mi
om3mis
o4m2n
om3ne
3omni
om3no
om3nu
o1mo
om1oa
o2mo3g
o4mol
om1org
om1ori
o2mö
omp4
om1pe
om1pf
om1ph
om3pl
ompo2
oms2
om4so
omt2
om1th
o1mus
o1mut
o1my
2ona
onage1
onal3l
onal1t
onal1z
on3ann
on3ant
on2an2z
onanz5ei
on1ap
ona2r
on3arb
onas3
onat4s
onau3z
o1nav
on3äh
on4bo
on1ch
2ond
on1do
on2dom
on2du
on3duk
ond4w
o4ne1h
on3ein
one4n
on3ends
onen3g
onen3s
on1erb
on1erd
on3erke
on1erö
on3ers
on2eu
onf2
on3fen
on3fl
on3fr
2ong
on3ga
on1ge
ong2k
ong2l
ong4r
ong3s
on3gü
o2ni
on2id
onie3g
oni1fe
on3ing
on2inn
onis4
onk2
1onke
3onkel
on1ker
on3ki
2onn
onn2e
on1nu
ono1c
on3oke
o3non
on1ope
on1orc
on3ord
ono1s
ono1t2
on2rad3
ons1a
on4sam
onse2
onsen2
on2sin
ons3ing
ons1p
ons3pin
onst2a
ons3tie
onst2r
on3sur
onta4
on2tä
ont3end
ont3erl
on2th
ontie4
on3tin
on1tra
on4trad
ont1ri
ontro2
ont3s
on3ums
o2nus
1ony
on1z2
on3za
onzer2
on3zi
oo1be
oo1h
oo2k
ool3c
oo2m
oom3b
oon2k
o1op
oo1pa
o1or
oor3da
oor2g
oo4s
oos3s
oo2t
oot1a
oot1ei
oot2l
oot2st
oo2v
o1ö
2op.
o2paa
opab2
opa3de
o2pa3ge
o2pak
op1akt
o1pan
o2pa1p
opa1ra
opa1re
o1park
opa3s
opa1sc
opa1st
o2pau
opa1v
opa1ze
o1pec
op3ef
o3pek
opel2
2open
1opera
2opf.
opf3a
o1p2fä
o2pfe
opf3erd
opf3la
op1flü
2oph2
o2pi
opia2
opier3u
o3pil
opin2
o3pit
2opl
op1lag
op1le
op1li
2opo
op2pa
op2pl
1oppo
2oppt
2o1pr
op3res
o2pros
op3sc
1opsi
op1so
op1sz
1opt2
op1th
2opy
4or.
or1a
2ora.
2orab
or3ab3b
orad2
or3adr
o1raf
or2a3g
1orake
or3alm
ora2m
orama1
or3ami
o2ran
oran3dr
2orar
ora1sc
or3att
2orau2
orauf4h
or3aut
2oraw
or1änd
or1ät
orbar4
or3bec
or3bei
or3bel
or3bew
or2bit
orb2l
or3bo
orb3s
2or2ca
or3cha
orch2l
or2d
2ord.
2orda
ord1am
orde4
or3dem
ord3eng
ord3ing
ord1ir
1ordn
ord3st
2ordw
4ore
o1rec
or1eck
o1red
or1eff
or3eig
or3ein
o1rek
or3ele
or3emu
o2ren
o3renn
o1rep
or1er
orer1i
or3erk
or1eth
or1eu
2orf
or2fli
2org.
org4a
or3geg
2orgen
or3ges
org4h
2orgi
or3gla
or3gle
or2gn
2or3gr
2orgs
org2t
org4w
2orh
or3ho
o2ri
2oric
3orient
o3ries
4oril
4orin
orin2t
2oris
2oriu
ork2a
ork3ar
or3kat
or3ket
ork2h
or3kl
or3kn
ork2r
ork2s
2or3l
2orm
orma2
orm3asp
orm3erg
orm3ord
2orn
or3neb
or2nis
or3niss
or2n2o1
4oro
or3ob
or1oly
o1rom
oron4
or1ope
or1opf
oro4t
o1rou
2orö
2orp
or2pe
or2pi
or4pu
2orq
2orr4
or3rh
or3ri
2ors2
ors4a
or3sh
or3si
or3sk
or2ta
ort1ak
ort3an
ort3ere
ort3erf
ort3erl
ort3ev
ort3off
or2tö
ort3räu
ort1um
2oru
o3r2uf
or3uh
orum2s
or3un
or3ur
or3v
2orw
or1z
or3zu
2os.
o2s2a
osa3b
os3ad
osa3f
osal2
osa3lin
osa3mu
osa3r
osa3s
osa3t
os2c
osch3ar
osch3le
os4do
2ose
ose1g
os1ei
ose1in
os1erk
o1sex
os2hi
o1sho
2osi
osin1g
o3sit.
2o3s2k
o4s3kr
os3ku
1os2m
os2n
o3s2o
o2so.
osol3
o4sov
o4sö
os2p
os3pec
2os2s
oss3ac
oss3and
os3sc
os4se
oss3enk
oss3enz
oss2er
oss5erei
oss3erf
ossof3
oss1or
oss3p
oss1t
os3stü
os2t
ost1a
ost4art
ost1ei
oster3e
oster3n
ost5erwe
ost3eur
ost1or
ost3ran
ost1rä
ost1re
ost3roc
ost3rot
ost3uf
ost1un
os1um
o3sü
o3sy
o1s2ze
2oß
oß1el
oß1enk
oß1enz
oß1ere
oßof3
oß3r
oß1t
ota2
ot1abi
ot1ah
ot1ant
o1tap
otar3te
ot1ast
ot1au
ot2ax
ot1ä
otär4
ot4bur
ote4
o1teg
ot1eib
ot1eic
ot1eis
o1tek
otel1a
ot1emi
ot2em3p2
ot3entr
o1terr
ot1erw
o1tes2
o1th
ot1hel
ot1hi
otho4
ot1hos
o2thu
2o2t2i4
ot3im
ot2o
oto1b
oto1c
oto1l
ot3opf
otor3k
oto1s
oto1t
ot1ö
o1tra
o3tran
ot1rat
ot2rau
ot1re
o1tri
ot3rin
ot1roc
ot2sat
ot3sch
ots2d
ots2e
ots1o
ots1p
ot4spa
ot2spr
ot3stra
ott1a
ot2t3an3
otte4
ottes3
ott2o
ot1tra
ott1re
ot1tru
o1tub
ot2und
ot1url
o1tü
ot2wel
otz2d
otz2ki
ot3zu
o2u2
ouf2
oug2
o3uh
2o3um
2oun
o3unds
oun2g
oun4t
ou3pa
our2g
our3ga
our2m
ous2
out3s
o1ü
o1v
o2v2a
oven2s
ove3s
o3vid
o3vio
o3vol
o2vu
o1war
o1wat
o1wä
o1wec
o2win
o3wo
ow1t
o4x
2ox.
ox2a
2oxk
ox1l
ox1t
ox3ta
ox2te
ox3ti
1oxy
oy2
oy3s
o1z4
o2za
o4zä
1ozea
o3zei
ozen2
ozes4
o2zi
ozon1a
ór2
órd2
ö2b
öbe2
öbel3i
öben1
öb2m
öb2s
öch2
öchs2
öchs5l
öde2
öd3la
öd3ni
öd2st
1ödu
1ö2f
öf2f
3öffn
öf1l
ö2g
ögen2
ögens1
ög1l
öhen1e
ö2k2
öko3
3öl.
öl1a
öl2b
öl3ba
öl3bi
öl3bo
öl3br
öl1ei
öl1em
ölf1ei
ölf2er
öl1in
ölk1le
öll1an
öl1lä
ölle3m
3ölm
öl1s2
öl3sa
2ölst
öl1ta
öl1tep
öl1tu
1ölu
ölz2w
öm4
2ö2n
ön1d
ön2e
öne1b
ön1g
öni2
ön4n2
ön2s
ön3sc
önt2
ön1ta
öo1
öo2t
2öp4
öpf3l
öpp2
ö2r2
ör3a
örb2e
örb3l
ör3da
örden1
ör3di
örd4lin
ör3ec
ör3ei
ör3el
ör3em
ör3ene
ör3ent
ör3ere
ör3erf
ör3erg
ör3erl
ör4erz
öre2s
örf1l
ör3fr
ör3fu
ör3ga
örge2l
ör3gr
ör3h
ör3im
ör3ku
ör4le
ör3ma
ör3me
örn2e
ör3o
ör3q
ör3ro
ör3ru
ör3sc
ör4schi
örs2e
ör3si
ör3sk
ör3sta
ört2e
ör3tei
ör3tri
ört2s
ör3tu
ör3une
ör3v
ör3z
ö2s
2ös.
2ösc
ösch3ei
ösch1m
2öse
öse1g
ös1ei
öse3str
2ösl
ös2o3
ös2s
öss1c
ös2t
öst1a
4ösu
öß2r
ö2t
öt3a
öten1e
öte2r
öt1r
öt2sc
ött2w
öt2z
öwen3
özes4
4pa.
1paa
2pab
pa3ba
pa3be
2pace
pa1cen
p2ad
1padd
p3adr
paf2
pa1fe
pa1fr
1pag2
pa1h
3pai
1pak
pa3ka
pa3kl
pa3ko
pal2
1pala
palat4
1palä
4pal3b
1pale
4pal3g
pa1lig
pal4m
palm1o
4palt
palt3r
pa1m
pam5s
pa2n
pan3at
pand4
pan2da
pan3de
pank2
pan2n
4pannu
pan2s2
pan3sl
pant2
3panz2
1pap
papier7end
papo2
2paq
3parad
par3aff
para3go
par3akt
par3b
1parc
4par3d
pa1rek
2parer
2par3g
par6gel.
1park.
park3am
1parke
par4kr
park1s
parm2
par3n
par1ne
1paro
2parp
2parr
par3ra
par3ru
par3sc
part4
4parta
1partn
1party
par1z
par3zi
par3zw
pa2s4
4pas.
pa1sem
pa3sp
pat2
2pat.
pat3a
2patel
paten4t
1path
4pato
1patr
1pau2
p1auf
pa3uni
p3ausz
pa1ve
pa1w
pa3zo
1päc
1päd
päd1er
1pär
1päs
pät3eh
pät3ent
pät3h
pät3s
2p3b
p1c
p5ch
2p1d2
pde2
pe2
4pe.
2pea
p1eb
pech1e
ped3l
pe3f2
p1eff
peg4
pei2
2peic
peichel5
pe3im
2peis
1peit2
2pej
2pek
pe3kr
pekt4i
pekt2s
4pel.
pel1a
pel3ax
peld2
pel1er
pel3f
pel3ink
pel4ins
1pell.
pel1l2a
pel5lan
pel1lä
pell4e
pel3leb
pel3lei
pel3lin
pel3lit
4peln
4pels
pel3sp
pel1ta
pel1tr
pel3v
pe3m
pe4n
4pen.
pen2d2
2pene
pen1ed
2peng
pen3it
pen1k
pen1o
pens2
1pen2su
pen3sz
2pent
pen1te
pen1tr
pen1z
1pep
2per.
per2a2
per3as
per3au
per1ä
per6ben.
per1d
2pere
2perg
per2ga
6periali
per2id
2perie
1perio
2periu
3perlen
1perlh
per3nic
2per2r
perr3an
per4rä
per2se
3per4so
1perü
per3v
per4zen
pes2
2pes.
pe3sp
pess2
2pesv
1pet
petri3
2petu
peu2
pe3v
2pez
4pf.
pf1ab
p2fad
pf1ai
pf1ak
pf1am
pfang4s
pf1ans
pf3are
p2f1au
pf1är
pf1äu
pfe2
4pfe.
pf1eim
pf1ein
pf1ep
pfer5a
pfer2c
pfer3r
pf1f2
p1fie
pf1inn
pf1ins
p1fit
pf2l
pf1lam
pf3lä
pf4leg
pf3lei
pf4ler
pf3lo
pf3lö
pf4lü
pf1n
pf3om
pf1ra
pf1re
pf1ris
1pfro
2pfs2
pf3sa
pf3se
pf3sl
pf3sz
2pft2
pf1th
pf1ti
pf1tu
pf3z
4p1g2
p3gu
4ph.
4phab
3phag
phan4s
1phas
p1hau
2phär
phe2
2phe.
p1hei
phen1d
phen1e
phen3s
ph1ers
p2hi
4phi.
4phia
4phi4d
phi4g
phik1a
4phin
4phis
ph1m
ph1ni
1pho
pho3m
p3hop
pho2t
ph2r
2phro
4phs
ph2t2
3phtho
4phu
p1hü
1phy
pia1b
pia1l
pia1m
pia1r
pia1t
pi2c
pi3chl
2pici
2pid2
2pi2e
pie3d
pie3fr
piege2
pie4l
piel1l
pie3m
pie4r
pi4gona
2pi4k
piko4
1pil4
2pile
3pilg
2pin2d
pin2e
pin3gen
ping3s
ping1t
pin2s
1pinse
pin2z
pi2o
3pip
pipe3
pi2r
pis2
pi2sc
2piso
pi2st
2pit2
pitz2e
2piu
pi2v
1pix
piz1in
p1j
p2je
2p3k
p4ke
pkur3
1pl2
2pl.
pla4
p1lab
plan1z
plas4
plau4
3play
3plä
2ple.
2ple3c
plen4
2pler
2ples
pli4
2plif
2plig
2plis
plut4
ply1
4p3m2
p1na
p1nä
pneu2
p1nu
1po
2po.
2po1b2
po5ba
2poc
3pock
po1da
3poe
2po1g
2poh
po2i2
5poie
3poin
3pol
pol3au
pol3g
pol3li
polo3p
pol3sc
pol4t
pom2
po1man
pon2
2ponn
2pons
3pont2
poo2
po1ob
p2o2p
popa4
pop1ak
pop1ar
po3ph
po3pt
po1rau
por3d
po1rei
porf2
2porn
por3s
por2t
3porto.
3portos
pos2
2pos.
3posi
8positums
2posp
po1str
post3ra
pot1ar
pot2h
pot2t
pott1r
pot1u
poxi1
2po3z
pöl2
pöst4l
2pp
p1pa
pp1ab
p2pad
p2pal
p2pan
ppar4
p2pat
pp1au
ppe3e
ppe3g
pp1ein
ppel1z
p2p2en
p3pendi
ppen3t
p2per
pper3r
pp1f
pp3fe
p2p1h2
ppin4
pp1lad
p2plan
pp1lä
p2ple
pp1lei
pp1oh
p4pol
pp1p2
p1pr
p2p3ra
p2prä
pp3ren
p2pri
p3pro
p4ps2
pp3sa
pp1se
pp1si
pp3sy
ppt2
pp1ta
pp1tel
pp1ti
pp1tr
p2py1
p2r4
prach7tes
1prak
1prax
1präd
1präg
1präl
3präm
3prä3s
prä3t
pre2
4pre.
2prec
1pred
pree1
2p3reg
1prei4
3preis
preis3c
2preiz
prem4
pren2
1pres4
press4e
1preß
pre3v
pri4
2pric
prin2
4pring
prings4
1prinz
prit3e
1pro
pro4be
pro2bi
2proc
3pro1d
3prog
3proj
pro1m
3pro1p
2pross
pro1st
2proß
prot4e
2prott
2prö
2pru
prun2
prung5sa
1prüf
2prüh
2prün
4ps.
p4sal
p3sat
p1sc
2pse
p1see
p3sh
4psi
ps1id
ps3k
ps2l
p3sp
ps1pu
2pst
p1sta
p1stea
p1stel
p1sti
ps4w
3p2sy
4psys
ps2ze
2pt
pt1a2
pt2ab
pt3at
p1tä
pte4
pt3ec
pt1ei
ptem2
pt1eng
pt1ent
pt1ep
pt3erei
pt1erw
pt1erz
pt1h
ptho4
p3tief
pt1in
pt3l
1pto
p2tor
pt1r
pt3s2
pt4sl
pt1t
pt1um
pt1urs
pt3w
3p2ty
pt3z2
pu2
1pub2
2puc
3pud
1puf2
p1uh
pul2
2pule
pul3in
pul4s
puls1t
3pul4v
2pulw
1pum
pum2p
pump3l
1pun4
2pund
punk2
1pup
2pur
pur1m
pur3r
3putz
1püf
pül3l
pül3ste
pür3g
p3v
2p3w
py3d
py1e2
1pyl
py1s
p1z2
p3za
p2zi
pzu2
2qi
qi1g
qu2
quan2
2que.
quen2
2quiem
qum2
4ra.
2r1aa
ra1ar4
r1ab
ra3bal
r2abä
r4ab2er
ra3bew
2r4abi
ra3bit
r2able
rabs2
r3abw
ra3by
ra2c
2rac.
r3acet
rach3b
racht5rä
r2ack
r2ad
1radar
rade2a
3radel
1radf
1radh
radio1
4rado
rad3t
2rae
r2a2f
raf3ar
raf1er
raf1r
rage3ru
rages4
2ragg
ra4gi
ra1gle
rag4stä
ra2h
ra3her
3r2ahm
2ra3ho
2raht
r2ai
rain2h
r3air
2rak.
ra3kab
ra3kal
r4ake
ra3kin
ra3kla
2rako
ra3kos
ra3kot
ra3kr
r3akro
2rakt
r3akz
2r2al
r3alar
ral1de
ra1lex
r4ali
ra1lib
r3alk.
rall2e
ral3lin
r3alm.
ra1log
ral3sk
ral3su
r3alt
ral1th
2rama
ra1man
ra1mar
2rame
ra1mee
r2ami
ra1mil
r2amm
ramm5ers
ram3ste
2r1amt
ramt2s
ra1mu
2ran.
4rana
ran3ade
4ranc
ran2d
r4anda
r4ande
rand3er
1randi
rand3s
2r2ane
ra1net
2ranf
rangs2
2ranh
2rani
r3anil
4r3anl
r3anm
r4anmu
rann2
2ra1no
r1anp
ran2q
r3anr
2rans
ran4spa
2rant
ran1ta
r5antei
r1anth
4ranu
2ranw
r2ap
ra3per
2rapr
rap2s
r1aq
4r1ar
rar3a
r3arb
rar3d
ra1rei
rar3f2
rar1in
ra1ro
rar3r
rar3tr
r3arz
r2as
2r4as.
ras4a
ra3schw
2rasd
ras2e
ra1sed
ra2set
ra1sig
ras4mu
rass2s
rast3ei
ras3tri
2rasy
2raß
raß2s
2rat1a
rat3m
rat2o
rat2r
rat3sc
rat4stä
rat3ste
2ratum
2ratz
2rau.
1raub.
rau4e
2raufb
2rauff
2raufl
raug2
rau3ga
1rau2m
raum3ag
rau3mel
raum1i
ra4un
1raup
2raur
2rausb
1rausc
2rausf
2rausg
2rauss
2rausv
2rausw
2rauto
raut3s
ra2vo
2raw
2rax
raxis1
r1axt
4ray
r2az
rächs1
3r2äd
rä1di
2räf
rä1fe
rä1fi
rä1fr
2räg
rä1hi
rä3ka
rä1li
4räm
rä1mis
rä1mo
2räq
2r1är4
rä1ra
rä1ro
r3ärz
rä1se
räte1s
räte1t
1rätse
2rätz
2räue
räus2
2räuß
2räut
2räx
4r1b
r3ba
rb1ab
r4b1ade
rb4al
r4bals
rb2am
r4ban3g
rbar4g
r4bari
r5barin
rb1art
r5bas
rb1auf
r3bä
rb3b
rb1ech
r3beck
r2bee
r2befa
r3beh
rbeid2
r2beka
r3bela
r2belä
r3benu
r3benü
rbe4ran
rber2e
r4bergs
rb1erl
rbes2
r2besa
rbe3se
r2bet.
r2beta
rbe3to
r2beur
rbi2
r3bil
r2binn
r3bio
rbit2
rb2la
r2b3lan
r8blasser
r4b3last
r3blat
rb3ler
r2bleu
r3b2lö
rb2o
r2bol
r3bot
r3b2r
rb2s
rbs1o
rb4stä
rb3stre
rbs3tri
rb1th
rb2u
r3bud
r3bus
r3bü
r4büb
rby1
r3byt
2rc
r1ca
r2ce
rce3r
rch1au
rch1ei
r3chem
r2chi
r5chif
rch1l
r3chlo
rch3lö
rch3m
rch1r
rch1s2
rch3sp
rch1t2a
rch1w
rck2sc
r1cl
r3co
r1cr
r2cy
4rd
r1da
r4da.
r3d2ac
rd1ak
rd1al
r2dame
rd1ant
rd1ara
r3darl
r3dat
rdär4
rd3d
r1deb
r3dec
rd2ei.
rd1elb
rdels2
rdem6
rde1mi
r1denk
r1dep
rde6pend
rder4er
rd3ernt
rdert2
rdes2
r3desi
rde3sp
r2d1ex
r1dez
r3dic
r3dif
r2dig
r3dik
rd3inf
rd1inn
rd1iri
rd1ita
rd3l
rd2lo
rd4loc
rd2n
rd3na
r1do4
r3dor
rd1os
rd3oss
r3dow
r3dr
rd1rat
r2drau
r2drin
rd2rö
rd2sän
rd3s2k
rd3sz
rd1t2
r1du
r4dunge
r3dur
rd2wa
r1dy
4re.
2re2am
reas3
re2b1a
re1bel
2rebi
re2b1l
reb3ra
re4bu
re2bü
r2ech
rech3ar
re1ci
2reck.
reck1l
re4cl
2redd
1rede
2redi
re1dim
red2m
re2dun
4ree.
1reed
re1em
re1eng
re1er
ref2
re3fa
1re3fe
3refer
2reff
2refi
3re3fl
3refo
1reg
re2ga
4regef
re2ge2l
regel3ä
re3gene
4r1eg2g
3regi
re2gl
reg2n
re2go
re3gr
2reh
reh3ent
reh1l4
reh1o
4r2ei.
2r2eie
r1eig
rei3ga
rei3ger
rei1gl
2reik
reil2
rei1la
1reim
rei1mi
r1ein
rein4al
2reing
r3einr
rein8s7tre
re1inv
rei1r
rei1sc
reisch4
reis3m
reis5tro
rei3tal
reits2
2reiv
2reiw
2rej
re2ke
2reki
2rekk
re1ku
rel2
4rel.
re3la
2relb
re1leg
2relek
re3lev
1relie
3relig
re1lik
re1lis
r2els
re3lu
2rem
rem2d
rem1ei
r2emi4
re1min
re2mo
remp4l
rems1c
re2m2u
4ren.
re2nac
1re3nai
re4n3an
re3nat
ren2d
2rend.
ren3dam
ren3die
ren3dr
2rene
re1neg
re2nei
ren3end
ren4ers
ren2eu
2reng
2re2ni
2renko
r2enn
4renna
re2nom
rens2p
ren3tal
ren3tel
ren5term
r3entf
r1entg
r3enthä
ren7tier.
ren3tis
r1entl
ren1tr
r1ents
r3entsc
4r2enz
renz5erf
renz5ers
re2ob
3repar
re2per
2re2pi
1repo
2repoc
3repor
2r1epos
2repp
1re2pu
re3put
2req
4rer
r1erb
r2erba
r2erbr
r1erd
r1erek
rer1ep
r1erf
r1erg
r3ergeb
re1rie
r1erk
r2erko
r1erl
r3erla
r4erlag
r3erleb
rer6lei.
r1erm
r1ernä
r1erne
r1erns
r1ernt
r2ero
r1erö
r1erre
r1erri
r1ersa
r2erse
r1ert2
r2erte
r1erw
r3erz
4res.
re4sa
2resc
re2sem
2resen
re3sis
1resol
1reson
res2s
res3sc
2resse
ress5erw
res4so
2ressu
r2est
re1sta
re1str
re2sum
2resz
2reß
reß1e
re1tai
re1tal
ret3au
ret3c
2rete
re2tr
re3tre
re3tus
reu1f
re2un
r1eur
1reuse
reut4l
reu4z
r1evid
re3vis
4rew
r3ex1
2rey
1rez
4rezi
4ré
2r1f
r3fa
r4fahra
rfalls3
r4f1ant
r3fä
rf1erg
r3feu
rf1f
r3fie
rfin4s
r2flan
r3flä
rf2lö
r3flü
rfo2
rfolg4s
r3fö
r3fre
rf2ru
rf2rü
rf2s1ä
rf2su
rf2ta
rf3tau
rft2r
rf1tu
rf2u
r2furt
r5fü
2r1g
rga4
rg1ad
r3gag
rg1ah
rg1ak
rg1all
r3gang
r4gans
rg1ap
rg3art.
r3gas.
rg1ask
rgas4m
r3gä
rg5b
rg1d2
rge4an
rge2bl
r3gef
rg2el
r3gel2d
r4ge4l3er
r4genen
rgen4z3w
r2gern
rge4tap
r2ge2ti
r3gew
rg3g
r2gi
r3gin
r3gir
r3giss
r3git
rgleichs7
rg2log
rg3m
r2g1na
r2g1no
rg1oa
rg1ob
r4gon
rg3op
rg1or
rg1öd
rg3ral
rg3res
rg3rin
r3gro
rg3rüs
rg3sä
rg3se
rg3s4i
rg3s2p
rgs2t
rg3su
r2gu
r4gun
r3gü
rg3w
r1h2
2r3ha
rhal4
rhan2
rhand4l
r4hard.
rhau2
2r3hä
rhe2
2rhi
rho4
r3hö
r3hu
rhus5s
2r3hü
ri2
2ria
ri3anna
rib2
1ribs
1ric
4rich.
4riche
4richs
rich4t
4rick
4rid
rid3el
rid2g
4rie.
rie3ber
2riefm
rief1r
rie3h
rie3i
riel2
rie3la
2rien
rie3nad
rien3s
2rier
rie3ra
rier1d
rie3ro
rif1a
rif1ei
ri4f1o
rif3s
2rift
2rig
4rig.
4rige
rig2g
rig1l
ri3gr
ri3he
2rik
ri3kol
ri3kop
ri3lat
ril3lin
ri3lo
r4imb
ri3mes
rimm4
rim2s
rim4sc
2rin.
2rina
r1ind
r3indu
2rin2e
rine1i
rine3m
rine3s
r1inf
r2infr
rin2g
ring1l
r1inh
ri3nitä
4rin2k
r5innenm
r3inner
r1innr
r1innu
r1ins
r4ins.
r4inspi
r1inv
2rio
rio1b
ri3pa
ri3pe
r1ir
ris2
ris4a
ris4b
2risc
risch3o
2rise
3risik
ri3sko
ris4m
r3iso
ri4s3p
3riss
riss1t
rist5ers
3riß1
r2it2
2rita
r3ital
rit3ant
rits2
rit3sp
ritt1r
rit3w
4riu
ri3wi
rix1
2riz
2r1j
2r1k
r3kab
rk1all
r3k2am
r3kap
r3kas
rkauf2
r3kee
rken3te
r2ker
r3kerz
r3kla
rk2las
rk2lau
r3klä
rk1lis
rk2lö
rk2lu
rk2na
rk2ni
r3ko
r4kob
rko4g
r5komm
rk1ori
r4kot
r3k2ö
r3kr
rk1räu
rk1rin
rk2se
rk3shi
rk2sp
rk4stec
rk4stoc
rkt3eng
rkt3erf
rkt3ers
rkt3erw
rkt3erz
rkt1o
rkt1r
rkt3ri
rk2um
r3kup
r2kus
rkus1t
2r1l
rl2ab
r5lad
r3lag
r2lak
r3lamp
r3lan
r6langs.
r4langv
rl1ar
r2l1asc
r3lau2
r4l3aug
r5lä
rlber2
rl2bo
rl1c
r3l2e4
r4lec
r2len
r4leng
rle5th
rl3f
rl3h
r3l2i
r4lind
r4lins
r4lith
r2litz
rlk2
r3l2o2
r4lon
r4loo
r5lor
r4lot.
rl2ö
rlös3s
rls2a
rl2spr
rl1ste
rl2sto
rl1t
rl2te
rl3tep
rl2to
r3l2u
r3lü
rlz2
2rm
r3mac
r2maf
rma3ges
rm1ald
rm1ank
r3mann.
rm1ans
rm1anz
rm3aph
rma1ri
r1mas
r3maß
rmat2o
r2maus
r1max
rm3b
rm3d2
rme3le
r2meli
r2mem
rm2ene
rme3nes
r2mer.
r1merk
rm1erl
rm1erp
rme1st
rmet3t
rmi4
r3mik
r3mil
rm1im3
r2min
r2mis.
rm3lu
rm1m
rm3n
r3mob
rmo3des
r2mo3h
r2mom
rm1ori
rmo1s
r3mot
r5mö
rm1p
rms2
rm3sa
rm3se
rm3sta
rmt2a
rm1ti
rm1tu
r3mus
4rn
r1na2
r3n2ade
rn1all
r2nan
rn2and
rn3ani
rn3ari
rn1ast
r4naß
rn3att
r2nau
rn3aug
r1nä
rn3c
rnd2
rn1da
rn1di
rn1do
rn3dr
rn1du
rne2b
r4neg
rn2eh
r1n2eid
r1neig
rn1ema
rn1ene
rn1ep
rn1erk
rn1erl
rn1ert
r2ness
r1n4e2t
rnet3t
rn2eu
rne3uf
r2neur
rn3g4
rn2har
r1nim
r1nip
r1nis
r2nisa
r2nism
rnis4s
r2nist
r3niv
rnk2
rn1n2
rn2oh
rn1op
rn1or
rn1ost
r3not
r1nov
rn1ö
rn3s2a
rn3s2ä
rn4sche
rnse4
rn3s2p
rn3s2u
rn3s2z
rnt2
rn1ta
rn2ten
rn1th
rn3ti
rn1to
rn1tr
r1num
r3numm
rn1ur
r1nü
r3ny
rn1z2
2ro.
ro2ban
ro2beh
ro4bü
ro1ca
roch2
ro1ci
2rockn
2roe
2rof
ro1fe
2rog.
1ro2ga
roge2n
1rogg
2rogs
roh1l
r1ohm
2rohn
3rohr
ro3ir
2r2ok2
ro1ka
ro1ki
ro1lab
2rold
role4
ro1lei
ro1lex
ro1lig
roll4en
ro3lo
2rol3s
4romat
rom3ber
ro2med
ro1mi
2romm
rom2p
r2on
ron2a
ro1nau
2roni
1ronn
rons2
2ron2t
ront1u
ro1nu
ron2z
ro1of
4rop
ro1pei
ro2pf
2r1or
r4ora
ror3al
ror2g
ror1m
rort2s
ro1ru
ror2ü
ros4
ro3san
4rosc
ro1sei
rosen1
ross1c
ro3st2a
ro2st2e
ros5um
ro2ta
rot3au
rot3s
rots2o
1roul
ro3unt
3rout
ro5vers.
ro1wa
ro1we
ro1wi
ro3za
röd2
2r3öf
1röh
r1ök
1röl
2röl.
r1ölp
1römi
rön1c
r1ör
rös1c
rö3si
1rösl
1rötu
2r1p4
r3pa2
rper3in
r3pf4
r2ph
r3pl
r4pli
r3po2
r4poo
r3pr
r2ps
rps1t
r3pu
r5pum
2r1q
r3quo
2r1r
rr2ab
r4rac
r3rag
r2rai
rran2
rr2ar
r2rass
r3rau
r4raut
r3rä
rr1äm
rrb2
rr1c
rr1d
rr2e
r3rea
r3rec
r3red
r2ree
r3reg
r3rei
r3re1l
r2ren
r3renn
r3rep
rre3pe
rrer2s
r3rese
rre2st
r4restr
rre2t
r3rev
rr3g
r4rha
rr2hos
r2rhö
rr2i
rrie4
r2rit
rr2j
rr3m2
rrn3au
rr1ni
rr2o
rr3obs
r4rog
rror3d
rror1t
rro2t
rr2sa
rr4sche
rr2st
rr1th
rr2the
rr1tu
rr2u
r3ruh
r3rum
r3rup
r3r2ü
rr3v
r4ry
rr3z2
2r1s
r3sabo
rs1ad
r3sagen
rs2al
r2salb
r2sald
r4samp
r4s1amt
rs2an
rs3ana
r2s3ang
rs3ar
r3sax
r3sä
r4säm
r3schü
r2sco
rs1ebe
r3se2e
rs1ef
r3sek
r2sell
rs2end2
r2sep
rse4ph
rs1epi
rs1erb
rs1ere
rs1erö
rs1ers
rs1eta
rs2ext
r3s2hav
r3shir
r3sho
rs2hor
r7sierteste
r5sierth
r3sis
rs2ka
rs3kas
r4s3ke
rs2kl
r4skr
r4sku
r4sky
rs3le
rs3li
rs3mu
r3sno
r2so.
r2son
rson4e
r2sop
rsor4g
rs3ort.
r2sos
r3sp
r4spara
r4s1ph
rs2pl
rs2po
r4spun
r4sput
rs3r
rs3s2
rsse2
rst3abl
r4stant
r3stat
rs2te
r6st5eing
r4sten
r3sterb
rst3erl
r3stern
rst3erw
r4stet.
r3steu
rst1h
r3stie
r3stim
r2stin
rst3ing
r3stink
r2stip
r2stit
r2s1tot
r3s2tr
rst3ran
r6strang
r4strun
r3stu
r3stü
r2sty
rsuch2
r2sum.
r2sumf
r4sus
r3sy
4rt
rta4
r2tab
rt2abl
rt1abs
rt1ad
rt3ae
r3taf
rtag4
rt3akr
rt3albe
rt1all
rt1alp
r2t1am
rt2ame
rt1an
rt2anb
r1tap
rt1ar
r3tari
r1tas
r3tast
rt3att
r1tau
r4tauft
r1tä
rt1ärm
rt1ärz
r3tec
rte1d
rte1e
r1tee.
r1tees
rte3g
rt1ein
rtei3na
rt4eind
rtei1s
r3teleg
rte3li
rtel3li
rtem4e
r1temp
rten1e
r3tenö
rten3s4
rt3ents
rten3z
r1tepp
rt3erei
rt1erg
rt1erh
rt1erk
rt3erla
r1term
rt3ernä
r3terr
rt1ers
rt4ersp
rt1erz
r2tes
rte3sk
rtex2t
rtge3h
rt4heil
rt1hel
rt1hi
rt2hie
rth1l
r1t2hu
r3thun
rti2
r1tic
r3tief
r2tim
rt1ima
r2tin
rt3inf
rt4is
rt2nä
rt1ob
r1toc
r2toi
r1tol
rto1pf
rt1or
r3tor.
rto1ta
r3tou
r1tö
r1tr
rt1rak
rt1rec
r3trie
rt3ris
r3trop
rt2ruh
r4ts
rt5sac
rt2seh
rts2el
rt3sex
rts1o
rt2spa
rt2spr
rts4tie
rt1t
rt3tr
r3tuc
r1tum
rt2una
rt2une
rt1up
r1tur
rt1urt
r3tut
r1tü
rt3w
rty1l
rt3z2
ru2
ruar3
rub2
rude2
1ru4f
ruf1f
ruf2s
rufs1p
ruf3spa
rug3r
1ruh
r1uhr
1ruin
r1umd
rumen2
r3umf
r3umg
r3uml
ru4n
r1una
run2d
rund1a
r2unde
rund3er
2rundi
r1unf
run2h
r1uni
r3unio
run2k
r1unl
r1unm
4run2n
r1unse
2r3unt
run1zi
ru3pl
r1ur1
rur2g
2rus.
2ruse
rus2l
ru3s2p
rus2s
rus3sc
rus4se
russ1p
1rut
2ruta
rut1el
2rut3h
rut1o
rut1r
2ruz2
1rü4
2rüb
rück1s
rüh1la
2rüm
4rümm
rün3f
rün3sc
rün3z
2rüp
rüt4
2r1v
r2van
r2vat
r2veg
r2veh
rv2el2
r3velo
r2ven
rven1e
rven1z
r2vern
r8versität.
r2ves
r4vet
r2vi
r4vig
r3vill
r2vl
r3v2o
r4voi
rvo3le
r4vu
2r3w
rwei2
rwelt4s
r4wem
r4wins
rwo2
r5wö
rx4s
2ry
ry1m
ry2me
rys2
2rz
rz2an
rz3ant
rz3as
rzäh2
rz2än
rz2bo
r1ze2
r3zei2
r4zele
rzel4l
rzell4a
r2zen
rz1eng
r3zent
r4zents
r4zentw
r4zentz
r2z1erf
rz1erg
rz1erk
r2z1erl
r3zers
r2zes
rz1ess
rz1id
r1zie
r1zif
r3ziga
rzi3l
r3zim
r2zing
r1zink
r3zins
rz2ir
r1zit
rzo2
rz2of
r3zon
r1zö
rz2te
r3zuk
rz1wa
r2zwä
r3z2wec
r2zwir
1sa2
4sa.
3s2aa
2s1ab
sa5b2ä
sab4bat
sa4be
s3abi
5sabot
s3abs
sach3th
s2ack
4sad
s1ada
sad2d
s1adm
s1adr
2s1aff
sa3gan
sag2e
sa3ges
2s1agg
2sagi
s1agr
3sags
3sagt
3sagu
s2ai4
4sai.
saik1
2s1ak2
4sakt
2sal.
sal2a
s1alar
3salä
sal2b
sal3bl
2sale
2salg
2s2ali
sal1id
2s1alk
4s1all
2salm
3s2alo
4salom
2s2alp
2salr
2sals
2s1alt
sal1th
sal2v
2salw
3sam
4s1ama
4samb
s1amei
s3ameri
samm2
s1amma
sammen3
4s1amn
s1amp
4samph
sam4t
s1an
2s3anb
3s2and
sand3ri
sand3s
2sane
2sanfo
3sang.
san1ge
4sangr
s3anh
2s3anl
2sanm
2sano
2s3anp
2sanr
2s3ans
sans2k
4sant
2s3anw
2s3anz
2s1ap
sap1p
sa3pr
2s1aq
2s1ar4
s4ar.
s2ara2
4s3arb
s2ark
s3arm
s3arr
s2ars
sar5ta
sar5u
2sas.
s1asi
2s1asp
6sast
sa3stu
s3asy
3saß
sat2
2sat.
sat3ant
2sate.
2s3atem
2saten
2sates
2s3ath
2sati
2s3atl
4sato
sat3st
s3atta
s3attr
3sat4z
satz3en
s1au
3sauc
2saud
2s3aufb
sau2g
sau2l
3saum
sauna3
2s3ausb
sau1sc
2s3ausw
2sauß
2savi
sa3vie
5savouir
sa3w
1sä
3s2äb
3s2äc
3s2äg
s1äh
2s3ähn
4s1ält
2s1äm
sä1ma
4s1änd
s3äp
2säq
2s1är
3s2ärg
3s2ät
4säuß
4s1b2
s3ba
sbe1
sbe2i
s2bek.
sber2
s3bl
s3bu
s3bü
2sc.
2scab
2s1cac
4scap
2scar
2s1ce
4sch.
3schaf
s2chal
s2cham
sch3ana
4schanc
sch3ans
4schao
s3chara
sch3arm
3s2chat
s2chau
2schb
4schc
sch2e
4sche.
2schef
sch3ei.
4schem.
4schemi
4schen.
4schend
4schens
4scher.
scher3c
4sches
2schet
2schh
2schig
1schim
4schiru
2schk
s4chl
sch4lag
4schlöc
1schn
sch3nis
4schonn
1schop
2schör
sch3rom
4schron
4schs2
sch1se
sch3sk
4scht
sch1t2a
sch1ti
scht2o
scht2r
3schul
sch4web
sch3z
4s1cl
2scoa
2scoc
1scop
1scr
s3cu
4s1d2
sde2
s3def
sde4k
sde3s
sde4si
s3di
sdien4e
sdi3st
sdi3v
s3do2
4se.
2sea
4se1b
s1eben
se1ca
sech2
2s1echo
2s1echt
s1eck
se1die
se1dis
1seea
see1d
see1h
see3ig
1se2el
see3lin
see2lo
see3n2e
see3ra
see3r2e
se1erk
se1erö
see3s
see1t
4sef
s3eff
se3f2l
s2eg2
s3egal
se1gem
se1ges
se2gl
1segm
1segn
1seh
seh1a
se3häu
seh1ei
seh1in
seh1l
sehr2
seh1ri
seh1s
seh1t
s3eid.
sei2de
2seie
1seif
2s1eig
1seih
s2ei2m
s1ein
3s2ein.
2seinb
2seind
2seine
sein4e3i
2seinf
se3info
2seing
2s3einh
2seini
2seink
2seinl
2seinr
4seinsp
4seinst
se5instr
2seint
2seinw
2s3einz
2s1eis2
3s2eit
1sek
2seka
4seke
2sekl
4seko
sekom1
se2kr
se1kul
se2kün
sel1a
se2lau
se3läu
s2elb
2sel1d2
se3leb
sel1ec
2s3eleg
se1lei
sel3ers
2sel3f
sel4he
4s1elix
2selk
sel1la
sel3leb
4selm
s2els
sel1se
sel1sz
4selt
sel1ta
sel1tr
1se4ly
sem2
2sem.
4semb
s2eme
1semin
1semm
2s1emp
se2n
4sen.
sen3ac
sen1ad
2senb
2send.
sen1da
3sendet
s1endl
sen3d2r
2sener
se3net
2senh
sen1im
2senm
se3not
se3nov
2senr
senst2
4sensz
sent2a
sen3tal
sen3ter
sen3tig
6sentiru
sen3tis
sen1tr
s1ents
se4n3u
2senw
sen3zin
sen3zw
2sepo
s1epos
4ser.
ser3ad
ser3al
ser3äus
serb2
s3erbe.
ser2bi
2serc
2sere
4sere.
se1rea
se3reic
ser3eim
4serem
ser3enk
4serer
4seres
se3rest
2serf
s2erfr
s1erfü
2serg
s1ergä
s2ergr
4s1erh
se2ri
2serin
2serk2
s1erkl
2ser5l
4serm
4sern
s1ernä
s3erneu
s3ernt
2sero
se1rol
s1eros
s1erot
2serö
2serp
2ser3r
4sers
4serth
2sertr
1serts
2sertu
4sertw
s4eru
se2ruh
ser2um
se3rund
se3rü
2servo
2serw
2s1erz
2ses.
ses2a
se3see
se2sel
se3sk
ses2ka
s1essa
sess4m
ses3z
1seß
1set
4set4a
se1tan
s1e2tap
2setä
2sete
se1ter
4s1eth
4seti
se1to
2setr
set3s
se1tu
2s1ety
3setz
3seuc
se1un
2seus
s1ex
1sex.
s2exi
4sexp
sext2r
2sey2
2sez
4s3f4
sfall5er
sfall3t
sfi2
4s3g4
sga6
sgang4s
sge1
sgel2
sge2o
sger2
sges2
2s1h
4sh.
sh2a
s2hak
shalt2
4s3han
4shä
sh2e4
sher2b
sher2r
sher4sc
s2him
s2hir
sh1ma
sh2n
sho4
4shoc
4shof
4shom
3shop
3show
4s5hö
sh2r2
4shs
sh1t
si2
2si.
4sia2
sia3la
2sib
1sic
side2
2siduu
2sidy
4sie.
3sied
sie3f
1sieg
siege2
siege4s
1sieh
2siel
4sien
sien3s
2sier
sie4s
3sieu
4si4f2
2sig.
sig1a
2sige
sig2l
3signi
2sigs2
sig4st
4si4k
sik1ab
sik1ä
sik1el
siko3
sik1r
sik3s2
sik1t2
sil2
2sil.
2sili
sil3m
2sil3s
2sima
si3me
2s1imm
simme3
sim2s
2sin.
sin4a
s1ind
2sine
s1inf
4sing.
sin2g1a
sin2gr
sing3sa
2s1inh
2s1ini
s1inq
2s1ins
2s1int
sin3teg
2s1inv
4si4o
sion4
sions3
sio2p
si3pe
si3rin
3siru
2sis
si4sa
4sisc
sis1e
4s3i4so
sis1p
siss2
sis3z
s2it
2sit.
4sita
2sitä
2sitb
2sitf
2sitg
2sith
2siti
2sitl
2sitp
2sitr
2sits
sit2u
2sitv
2sitw
sit2z
2siu
si3un
2siv
siv1o
2siz
2s3j
2s1k
4sk.
sk4a
4s3kab
4s3kam
4skana
4skanä
3skanda
s3kann
4skas
4skateg
4s3kä
4skb
s2ke
s3kenn
3skep
3ski.
s3kic
s4kie
s2kif
ski1g
3skik
s2kip
4skir
ski1s
3skiz
sk2l
4s3klas
3s2klav
4s3k2n
sk2o
s2kog
4skoh
4skol
s3kom
4skon
sko2p
3skop.
sko2t
4s3kö
sk1q
4s3kra
skre4
4skro
4sks
4sk1t2
s3kub
s3kug
3skulp
s3kun
s3kup
s3kü
2s1l2
s2lak
3slal
s3lan
s2laus
s3lä
sl3b
sle3bes
sle4g
s3lei
sler3s
sliga1
s2li4m
3s2lip
sli3r
slo2
s3loc
3slow
s3lus
2s1m4
s3mac
sma3d
s3mal
sma1la
sman2
s3mä
s3mee
s3mei
s3meng
smen2s
smer4
s3mil
s3mit
s2mu
s3must
s3mut
s3mü
2s1n2
snab2
s2nen
s2nos
4snot
s3nu
s3nü
s2ny
so2
4s1ob
s2o3ba
so3bea
1sod
so3et
s1of4e
so3fer
3soft
3sog
so3ga
so4gen.
so4ges
so3gl
s1ohe
so3hi
1sohl
2s3ohng
2s1ohr
3soj
so4k
s2ol
so3lan
1sold
sol3ei
solo1
sol3s
2s3oly
3som4
so3mit
so3mo
so4n
2sonad
son4an
2sone
son3end
3song3
s1onk
3sonn
3sono
son3sä
son2s1o
son1t2
3so3o
s1op
s2opa
so3po
3sopr
sop1s
sor2
2sor.
sor3a
s1orc
4s1ord
2sore
2s1orga
s3orgi
so3rh
2sori
s1orie
2sor3k
sor3m
2sorr
2sor3s
2sort.
2sorts
1so3sc
so3se
s3osk
3soss
s1ost
s1osz
3soß
so3t2
so4te
so1to
so1tr
sound1
so3unt
s1out
sou3te
3sov
s1ove
so3vi
1sow
so3wa
so3we
so3wi
2s1ox
s2oz
s1öd
2söf
2s1ök
s1öl
2sön1
s1ös
1sp4
2sp.
spa2
4spaa
s1pac
2spack
2spak
4spala
4spalä
s2pan
3spannu
s3pano
3s4pant
2s3pap
2s1para
s4parka
2sparl
spar3m
2sparo
5s6parten
4spartn
4sparty
3s2paß
3spat.
4spatr
2spau
s1pav
s2paz
2späd
3späh
2spär
2späs
2spe.
3speic
4s3pein
8spektivi
4s3pensi
4sperat
3sperg
4sperle
sper2m
2spero
sper4r
2s1pers
4spet
3spez
4s3pf
2spha
3sphär
3spi
s4pie
4spier
4s3pil
s2pinn
4spip
s2pir
4spis
s1pist
6spizien
2spl
4s3pla
4splä
3s2pli
4s3plu
2s1pn
spo2
4s3pod
4spoe
2s3pog
s2poi
4s3pok
4spol
2spop
sport3r
4s3pos
4s3pote
s2pott
2spr.
s2prac
2sprak
s2pran
2sprax
4spräm
4spräs
3sprec
4spred
s2pren
4spres
3spring
4sprinz
s2prit
4sprob
4sprod
4sprog
4sproj
4sprop
3spross
2sproz
3sprö
3spru
2sprüf
4s1ps
2spt
2spub
2spud
3spuk
3spule
2spum
2spun
2spup
3spur
2s1put
4s1py
2s3q
4s1r4
s3rak
sre2b
sre1l
sre3m
sre1ta
sro2
s3rü
4ss
s2sa
s4sab
ss2a3ba
ssa5ber
ss4ag
ss1aj
ss1alb
ss2ant
s4sap
s4sar
ss2are
s5sarg
s4sas
ss3att
ss2ä
s1sc
s3sca
s4sco
ss1ec
sse1d
sse1ec
sseh2a
sse1hu
ss4eind
ss4e3int
sse4ku
ssel1l
ssen4ac
ssen3g
sse5rat.
sse5rate
sse5rats
sser1d
ss2erf
ss3erfü
ss1erö
ss3erse
sser3ti
ss3estr
sse1ta
sse3v
ss3hi
s1sieu
s3skala
ss3ke
ss3m
ss1oa
s3sod
ss1off
sso3m
s4sov
s1sö
s2söl
ss2pin
ss1pis
ss2po
s3spri
s2spro
s3s2pü
ss3s4
s1st
sst2a
s5stad
s3stern
s2stet
s3s2teu
s3stil
ss1tis
ss3trit
s3strö
s2sty
ss2ur
4st.
sta2
4sta.
3staa
2stabb
4s3tabel
4stabit
2stabl
st2ac
3s2tad
3staff
2stag
3stah
2stak
s1tal.
2s1talb
2s1tale
stal2i
2stalk
st1alp
st1alr
1stan
3stan2d
2stanf
2stanl
2st1ann
2stanw
2stapo
4s1tari
s2tars
2stas
sta3str
1stat
2statb
3stati
s4tau.
s2taub
2stauf
2staug
s4taur
2staus
3staus.
st1ave
4stax
3stä
5stäb
4stäg
4stält
4stänz
4stäp
5stär
4stäus
2stb
2stc
st3ch
2std
4ste.
2steam
4stechn
st1edi
4s1tee
ste2f
3steg
steg3r
1steh
ste2i4
st1eid
3steig
2s3teil
3steilh
2stel.
4s1tele
ste3li
s2tel2l
stell5än
2stels
2stem
4stem.
3stemm
2sten
st3ends
ste4p
st1epo
2ste2r
4ster.
st3erfü
st2erg
ster6loh
4stes
stes1e
stes3ta
st1eta
1steu
3steue
st1eun
st1ev
ste2w
4stex
2stf
2stg
4sth
s2t3ha
st1her
st1hi
st3ho
sti4
2stia
2stib
2stie.
4stief.
2stien
3stif
4stig
2stik
1stim
s2tin
st1inb
2stint
2stio
s1tip
2stip.
4stipp.
st3ira
st1iri
4stis
st1ita
4stite
1stitu
2stiv
2stj
2stk
stkom1
2stl
st3la
stle4
st3lu
4stm
2stn
2sto.
sto2b
st1obl
2s1tod
s2tode
3stof
6stoffir
4stok
4stole
sto1m2
sto2ma
st1omn
4s1ton
s2tone
4s1too
sto2p
2stopo
2stor.
2store
2storg
2stori
2stors
3stoß
4stou
4sto1w
2stoz
1stö
4stöch
2s1töl
2stön
3s2tör
3stöß
2stöt
2stp
2stq
stra4
2s1trac
2strad
5s2traf
4strag
1strah
4strahi
4s3trai
2strak
2stral
stran2
4s3trank
4strans
1strap
3s2tras
3straß
4straum
s2träf
4sträg
s2trän
4sträne
4stref
4streib
3streif
2strep
s2tria
2s1trib
2strig
3s2trik
4stris
2striu
2stro.
s2trof
1stroh
3strom.
2stron
2stros
st4ross
1strö
4ströp
1stru
4strub
s2trud
3struk
s2trum
2strun
4strup
1strü
2st3s2
stsee1
2st1t2
st2u2
5s2tub
2stuc
3stud
4s1tue
3stuf
2s1tug
3stuh
2stum2s
stum4sc
2stumt
4stun.
1stund
2s1tune
2stung
st3uni
4stunn
4stuns
4stunt
s2tup
2st3url
2s3turn
4st3urt
3sturz
2stus
1stut
1stü
2stüch
s2tück
3stüh
2stür.
4s1türe
2stütc
2stv
2st3w
3styl
2st3z
stzu2g
su2
3su4b3
4subi
5subv
3suc
s1uf
1sug
sugge4
2s1uh
suhr4
sul2
3sul.
sult2
sum1a
s2ume
s3umfa
2s3umfe
3sum2m
sum1o
s2ump
2sums
s3umsa
s3umst
sum1t
s1una
sund5erh
sun3dr
sunds4
s1unf
s1ungl
sung4s
4s1uni
s1uns
2sunt
3s2up
sur2d
sur2f
s1urk
2s1url
4s1ur1t
2sus1
sus2a
sus2s
2s3ut
1sü
4süb
süd3
2süme
3sün
3süs
3süß
4s3v2
2s1w
s3we
swei2
4swie
4swil
4swis
4swit
s3wo
1s4y2
4sy.
sy4f
2syl1
sym3
sy3me
syn3
sy4p
3sy4s
4sys.
s1z2
4sz.
4s3za
2s3zä
4s3zei
4s3zel
szen2
3szene
4szent
2szer
s2zes
s2zeß
szi4
4szie
s2zin
s3zins
s3zir
4s3zu
4s3zü
4s3zw
ß3a2
ßal2
ßan1
ßar4
ß1ä
ß1b2
ß3ba
ßbe1
ßbe2i
ßbe2to
ß1c
ß1d2
ßde2g
ßdi3s
ße2
ß1eb
ß1ec
ß1ef
ß1eg
ß1ei
ßel1a
ß1emp
ßen3a
ßen3se
ßen3sz
ß1entl
ßen1tr
ßer1d
ß1erf
ß1erö
ß1erw
ß1ess
ß1estr
ß1ex
ß1f2
ßfi2
ß3g2
ßge2bl
ßgel2
ßger2
ß1h2
ßher2
ßig1a
ßig4s
ß1im1
ß1in
ß1j
ß1k2
ß3kl
ß4kopfs
ß1l2
ßler3s
ß2lig
ß3m2
ßman2
ß1n2
ß2ne
ß3nef
ß1o
ßor2
ß1ö
ß1p4
ß3pul
ß1q
ß1r2
ßre1s
ß2ri
ß1s2
ß3sa
ß4schnü
ß3schu
ßse2
ßser2
ß3sk
ß3sp
ß3st
ß1ta
ß1tä
ß1tei
ß1tel
ß2tem
ß2ten
ßtent2
ß2tes
ßt1h
ßti2
ßt1in
ß1to
ß1tö
ß1tr
ßts2
ß1tu
ß1tü
ß1uf
ß1uh
ß1um
ß1una
ß1uni
ß1ü
ß1v
ß1w
ß1z2
ßzu2g
4ta.
2taa
ta3an
1t2aba
tab3an
t1abb
2tabf
2tabi
t1abit
t3abn
ta3bo
t3abt
1tabu
ta3by
2t1ac
ta2ca
tad2
t1ada
ta3dat
tad3d
t3adr
tad4t3
2tae
1taf
2tafa
2tafet
2taff
t1afg
2tafl
t1afr
2ta1fu
1tag
3taga
tag1ei
4t3agent
tage1r
tage4sp
3tagf
2t1agg
ta3gla
3tagm
ta2gn
2t1ago
2tagr
tags1
tag4san
tag4st
3tagu
3tagv
2tah
t2ai2
tain4m
ta1ins
1tak
2t1aka
t1akk
ta3ko
t3aktiv
takt1o
t2aktu
tal1an
tal3au
2talbe
tal3d
tal3end
tal6ents
tal2ga
t2a2li
2tal2l
tall2ö
tall3s
t1alm.
2ta1lo
ta3loc
tal1or
tals3en
2talt
tal1t2r
tam4b
t2amen
t1amer
ta3mir
ta1mo
t1ampl
t2ams
2t1amt
ta1mu
t1ana
tan3ab
t1anä
2t2and
tand4ar
tand1r
tand4st
2tanh
t2anho
2t4ani
1t2an2k
tan3kri
tan3kü
2t3anl
t1anm
t1anna
1tanne
t1ano
t1anse
t1ansi
t1ansp
t1anza
1tanzg
t1anzu
tanz1w
t2ape
ta1ph
t2appe
ta1pu
2tar.
tar3ap
tar3bl
2tarc
2tare
tar3ens
tar3er
tar3g4
ta3ric
3tarif
tar3ins
2tark
tark2l
t2arl
t1arm
tar3ma
1tarn
tar3s
2tart
t1arti
t4ar3tis
t1arz
2tas.
1tasc
2tase
ta4si
tas4m
t1a2sp
t3assi
tast2r
2ta2sy
ta3tal
tat1an
3tatb
t4ate
2tat1ei
tat1er
ta1the
2tati
t3atl
t4atm
ta2to
4tatt
tatt3an
tau2br
2taud
t1auf
3taufe.
tau3f4li
2taufn
tau1fr
2taufw
1tau2g
2t3auge
t1auk
1taum
2taur
2t1ausb
1tau4sc
t2ause
t3ausg
t1ausk
2t3auss
t1ausw
2tausz
2tauu
2tav
ta1ve
3tax
4t1axt
tä2
2täb
2täd
1t2äf
1täg
täl3c
t1ält
2täm
t1ämt
tänd3l
2täng
t1ängs
1tänz
t1äp
4tär
tär3b
tär1d
tär3g
tär3r
tär3s
t2ät
2tät.
tät4s
4tätt
2täug
1täus
2t3b2
tbe1
tbe2i
tben4
tber4
tblock5e
t1c
t3ca
t5ce
t3cha
tch2i
tch1l
tch1w
t3cl
t3co
t3cr
2t3d2
tde4k
t4dem.
tde1t
tdi4
4te.
1te2a
2teak
2te3al
3team5
2te3an
2teau
te1b
t2ebb
t1eben
1tec
2tec.
t2ech
2te1cha
2teche
3techn
2teck
teck2e
t1ecu
2teei
te1em
tee1mi
teen1
te1erw
tee1t
tef2
tefe4
te2g
t1egg
te3gl
teg4o
1tegre
1tegri
2tei.
2tei1b
2teie
2teif
tei1gl
2teik2
1teil
teil1a
3teilc
3teiln
2tei1m2
4tein
t2ein.
t2eine
tein3ec
t3einge
t3einkü
t3einla
tein4se
tei1ra
2te4is
t1eis.
t1eisb
tei1sc
tei1z
te2ke
te2ki
tekt2
te1ku
te2l
2tel.
tel1ac
te3lad
te3lan
tel1au
tel3d2
tel1ec
3telef
tele3g
tel3ehr
3telema
2telen
tel3eng
1telep
t3elf.
tel3fe
2teli
tel3inn
tel3ins
telk2
tel3lau
tel1lä
tel3leb
tell4er
tell2i
tel3lin
tel3m
2telo
te3los
te4lost
tel1ö
tel1se
tel3s2k
telt2
tel1ta
tel1tr
telz2
te2m
2tem.
te3mas
tem1ei
te3mit
tem3ma
tem1or
1tempo
tem3s
te2n
2ten.
ten1a
ten3an
ten3au
ten3ä
t3endal
ten3deb
ten3def
ten3deg
ten3del
t3endf
t1endl
2t6endo
t3endp
2ten3d2r
ten3dsc
2tendu
tene4
ten1eb
ten1ed
ten4ei.
ten3end
ten3ern
ten3ero
4tenf
t1eng.
teng2a
t3engla
ten1im
ten3in
ten1k
tenn2
1tenni
3tenor.
1tenö
t2ens
t3ensem
tens2p
tens3th
ten3tal
t1entb
ten3tel
ten3tim
ten3tis
t1entl
ten1tr
t1ents
te4nu
ten3zu
te2o
te3ob
te4ph
t1e2pi
te3po
te4poc
tep3t
4ter.
t2era
ter3ac
ter3ap
ter3as
ter3c
ter1d
t3erde.
terd2s
ter3eif
te1rek
ter3end
ter3f
t3erfol
t4erfr
2ter3g2
t1ergu
te2rit
ter3la
ter3ma
ter6mine
2tern
ter3net
ter3neu
ter3nos
t2ero
te1rom
t1e2r1ö
t2erp
t4erra
3terras
ter3re
2ters
t2erse
terst4
t4erst.
t4ersti
ter3sto
t4erstü
t2ert4
ter3tan
ter3tei
ter3tel
ter1th
ter3tie
ter1uf
ter3v
terz2a
t1erzb
ter1zi
6tes.
tes3ac
tes3ak
te2se
te3seg
t1esel
te3spen
t1essa
t2est
tes3tät
1testb
t3estri
3tests
te2su
2tet
te3tab
t1etap
te1tau
te1tee
te1tel
te1tie
te1tis
te1to
tetra3
te1tu
teu2
1teuf
te1un
te2va
te2vi
1tex
2t1exa
2t1exe
2t1exi
2texp
tex1tr
t1exz
te3z
2t3f2
t4faß
2t1g2
t3gan
tgebe2
tgel2
tge1r2
t3gl
t3go
t3gr
2th.
t1h2a4
t2hag
1thal.
3thalp
t2har
t2hat
t3hau2
t1hä
1t4hät
2thd
th2e2
2the.
3t4hea
2t1heb
t1hef
t1hei
1theil
t4heils
2theim
2the3in
4t3heit
1t2he4k
1theme
2then
3theo
t3herab
t1herg
1ther2m
ther4sc
t1herz
1these
t1hess
t2heu
1theue
2thf
2thi
thic5k
t1hiel
t3him
t1hin
t1hir
th1lä
th2le
2th2m2
2tho
t1hoc
tho3chr
t1hof
t3hoh
t2holo
t2hon
tho3s
t3hot
t3hou
2thö
th2r2
2ths
thsel4
th2s1t
t1hu
t3hub
1t2hum
1thur
t1hü
2t3hül
1thüm
2th1w
2thz
2ti.
tia4
ti1ag
ti1alk
ti1all
ti1am
4tiat
2tib
tibe2
4tich
ti1chr
t1id
tid3end
ti1do
2tie.
tieb4
tie3d
3tief.
4tiefel
3tiefs
tieg4
tie3i
4tiel
tiel3a
tien3s
3tierc
3tiersc
tier5tel
2ties
2tieß
ti1eu
tif2
ti1fe
ti1fr
ti2g
4tig.
t3igel
ti3gene
tige2w
ti4go
ti3gr
tigs2
ti3h2
ti3is
tik2el
ti2klu
ti4k3rei
tiks2
tik1t
ti2le
1til2g
til3ger
3tilgu
til1lo
til3m
ti2lö
til1s
til1t
til3v
tim2
ti1maf
ti1mil
2tin.
t3ind
tin2e
tine1i
t1inf
ting1a
tin2gi
tin2g1l
ting3s2
t1inh
t1init
t1inj
tin2k
t1inka
tink1l
ti3nom
t1ins
tin4spa
t1int
3tinu
4tinua
4tinuo
t1inv
2tio
tio2c
3tip.
2tipe
ti1per
3tipp
3tips
ti1ra
tire1r
ti1ret
ti1rh
2tis.
tisch3w
2ti1se2
tis4fa
ti3sk
t3isl
ti1so
t3isr
2tiss
t1iß
2tit.
tit2a
1tit2e4
3titel
2titen
ti1ter
ti1th
ti3t2r
tium2
2tiv
tiv1o
tiv1r
tiz3z
2t1j
t3ja
t4ja.
2t3k2
4t1l
tl2a
tlan2
tlap2
t3lä
tl2e
t3lebe
t3lei
t3lek
tle3me
t2len
tler3a
t3lese
t3let
t4ley
t3li
t4lieb.
tli5f
tlings3
tli3sc
t3lo
t3lö
t2lu
t3luf
t4lun
tlung4
t3lus
2t1m2
t3ma2
tmal2
t3mä
t3mei
tmen2
tment3
t3mer
t3mi
t4mig
tmon3ta
t3most
t3mot
t3mus
t3mü
t3my
2t1n2
tnach4b
t3nad
t2nan
t3näh
t2nes
tness1
tni4
t3nö
t3nu
4to.
to3at
to3bat
t1obs
1tobt
t1ochs
1tocht
to3co
t2od
tode2
tod1er
2to1di
to1do
tod3u
2tof2
toff3s
to1g
to3ge
1togg
to3gi
to1ho
to3ja
to1k2
to1lau
2told
1toler
2tolo
2tolt
to1man
tom1en
to6mere.
tom1m
tom1t
tom3u
2tonc
ton1d
ton2e3h
2tonei
ton3ig
2tonk
1tonn
3tonst.
ton5stel
1tont
4tontei
ton1to
2tontu
2tonum
to1ny
to3om
t1ope
1to4po
2to4pt
top1te
to1q
2torc
2t1ord2
tor3da
t2ordi
2t1org
tor3gen
2tori
to3ric
tor3int
tor2k
tor3ka
tor1m
2torn
to2rö
tort2
t1ort.
tor3ta
to2ru
2tory
tos2
to3sa
to1sc
to3sh
to5sth
to1tr
tots2
2tott
1t2ou
2tou.
to3un
2tous
to3v
to1wa
3tox
to3z
1töch
2töf
4t1ök
töl2
1tölp
1tön
tön1c
2tön4l
1töpf
tör3b
tör3g
tör3p
tör3s
2tös
t1öst
1töt
2t3p4
tper4l
tpf4
t1q
tr4
2tr.
2tra.
2traa
2traä
t3rad.
2trafa
1trag
2trahm
1t4rai
3train
t1rake
2trakl
2tra3ko
2tral
tral1l
tra1na
2trand
2trane
2trang
3trans
tra1sc
2trasi
2trasp
tra2st
2traß
trat4s
2trauc
t2raue
t2rauf
2traup
2trav
2tra3z
1trä2
3träg
2träh
3träne
2träts
2träus
2träuß
2tre.
t1reak
2tre2b
2trec
t1rech
t2reck
3treck.
t3red
1tref
3treff
2trefl
2trefo
2t3re2g
tre5gist
trei2
1t2reib
2treif
2t1reig
2t1reih
t1rein
2t1reis
2treit
t1reiz
t1rek
2trel
t4re2m
tren2
3trend
t3rendi
t1rent
2t1repe
t3repo
1trepp
t4repr
t1rese
t1rest
1tret
tret3r
t1rett
t2reu
2t1rev
2trez
t1rh
t2ri
tria2
tri3co
3trieb.
3triebs
trie3c
trie3d
2trieg
tri3enn
2trigi
tri3gl
1tril
tri3ma
t3rind
2tring
t1rinn
tri3tiu
t3ritu
t3riv
2tro.
tro1b
1troc
tro1ch
4trock.
2trod
2troh
tro1ha
tro1he
2trok
2trol
tro1li
2trome
tro2mi
2troms
3tropf
3tropi
2tropo
tro3r
t1rosa
tro1sc
2trose
tro3sm
2trou
2trov
2t5röc
2tröh
4tröm
1tröp
2t1röss
1tröt
1trub
4truf
truf1t
1trug
4truk
trum2
t1rumä
trums1
t1rund
1trunk
3t2rup
2t1rus
t3ruß
2trut
2t3rüc
t2rüg
1trümm
2try
2ts
ts3ab
t3sac
ts1ad
ts1ahn
t3sai
ts5alben
ts2ame
t4samp
t4s1amt
t2san
tsan1g
t3sani
ts3ari
t2s1as
ts3att
t2sau
t3sau.
t3säf
t2säh
t2s1än
ts1äus
ts4bur
t1sc
t3schan
t6schart
tsch5aus
t3schä
t4schen
tsch4li
t3schü
ts2cor
t1se2
ts1eb
t3seg
ts1eh
t3seil
t5sein.
t3sek
t3seme
ts1eng
t3sens
t2s1ent
t2s1ep
t2s1er
t3serv
t6s5essen
t3set
ts1eti
t2sex
t1si
t3sic
t2s1id
t2sind
t2s1ir
t3sit
t3skala
ts3ke
t1ski.
t4sko
ts3li
t4s3m
t4sn
tsof3
ts1off
t1sol
ts1ori
ts3ort.
t3sos
t1s2ouv
t1soz
t2spac
t3spal
ts1par
ts4pare
t2spat
t2spä
t3spei
t3spek
ts1per
t2s1ph
ts1pis
ts3ple
t2spo
t3spon
t3s2por
t4sprei
ts3s4
t1st4
t4s3tabe
t2staf
t4stag
t4stas
t4s3tat.
t4s3täti
t2stea
t3stein
ts2tel
ts1tep
ts3term
t3stern
t3stero
ts3terr
t5s2teu
ts3th
ts1tie
t3stil
t3stim
ts1tis
t4s3tit
ts3toch
t4stoi
t3stop
t2s1tor
ts3trad
t2strä
t4stren
ts1tri
ts2tro
t4strop
t3stu
ts3tuns
t3stü
ts1tür
t4sty
ts1u
ts3un
t3sy
4tt
tt1ab
tt2ac
tt1ad
tt2ag
ttag5ess
tt1ah
tt1am
tt1ap
t3tari
tt1art
t1tat
t1tä
t3tät
tt4bu
tt1ebe
t1tei
tt1eif
t3teig
tt1ein
tt1eis
ttel3eb
ttel1o
ttel1z
ttens2
tte2r
tt2erg
tt2erö
tter4ra
tte2s
ttes3ä
tte3sc
ttes1o
tt2ga
tthätig8keit.
tt2häu
t3thr
tt3igi
ttik2
tti2l
t3t2int
tt2le
tto1b
tto1d
tto1l
tto1m
tto1s
tto1w
t1tö
ttran2
tt1rea
tt1ren
t3trog
tt1rud
tt4scha
tt2sen
tts1p
tt2spe
tt2spr
tt4s3tät
tts3tem
tts3ter
tt1sz
tt1t2
tt1uf
t1tut
t1tü
tt4wil
tt1z2
2tu.
2tua
tu2b2
tuba3b
1tuc
2tuck
2tu2d
tudien3
2tuel
2tuf2
t3ufer
2tuh
2tui
2tuk
t3ukr
tul2
2tulin
2tumb
t3umf
2t3umg
2t1umh
t3umk
1tumo
2t3umr
t3umsat
2t3umt
2t1umw
t3umz
1tun.
t1una
2t1und
tun2en
t3unf
2tun2g
t3unga
tung4s
3tunn
t1uno
t3uns
2t3unt
1tu4p
2t3up.
tu2r
2tur.
2tura
tur1ag
2turbe
2tur1c
tur1d
2ture
tur1er
turg2
tur3ga
tur3ki
tur1o
tur3r
2turs
2turz
tu1sta
2tutz
2tuv
2tuw
1tux
2tüb
tüb3l
1tüch
tück2s
1tüf
2tüh
2tümp
1tür.
tür3c
1tür3g
1tür3s
2türz
1tütc
1tüte
2tütz
2t3v2
tver3g
4t1w
twei2
t3wi
t2wo2
t3wu
t3wü
1t2y
2ty.
ty3d
ty5e
ty1g
ty1k
2tyl
2tym
ty1mu
ty1n
3ty4p
typ1t
2tyr
ty1rä
2ty1s
ty1z
2tz
t2za
tz1ag
tz1al
tz1au
t2z1ä
tz1ec
tz1eie
tz1eis
t3zeit
t3zell
tz2ene
tzen3ga
tzen3ta
tz3entg
tz3ents
tzes3
tzig4s
t3zim
tz1int
t3zir
t3zol
t3zon
t2zor
t1z2ö
tz1ti
t3zuh
tzüg2
tz1wä
tz1wi
tz1wu
tz1z
2ua
u1ab
ua2c
uack3
ua3ku
ual1a
ual1ä
ual1d
ual2g
ual1k
ual1tr
u1am
ua1ma
uam3p
ua1n4a
uan2d
ua2r
uar3s
uas2
uasi3
uat2
uate4
uat3h
ua1tin
u3au
uä2
u1äm
u1äu
2ub
u2bab
uba3k
u2ban
ub2bi
u1bec
u1be2e
u1beg
ub3eins
u1bema
uben3so
ub2er
u3berat
uber3b
u4bern
u4bers
u1bez
ub4i2
u4bis
ub1läu
u1blö
ub1lu
ub3lun
u2bob
u4bö
u2b1rit
ub2rü
ub2san
ub4sauf
ub4s3che
ub2so
ub2sp
ubs1pa
ubst2
ub2s3z
u2büb
ub3z
4u2c
uch1a
u4chan
uch1ä
uch1ec
uch1ei
u3chem
uche4r
uch1il
uch1in
uch1la
uch1le
uch1li
uch1lo
uch3m
uch3n
uch1op
uch1r
ucht3re
uch3ü
uch1w
u1cit
uck3erl
uck4err
uck4sti
2ud2
uda3d
u3dam
u2dat
ud4e
ude1e
ude3i
ude2l
uden3s
uden4te
u1de2p
uder2e
u1dez
ud3h
udo1b
ud3on
ud3ra
u3dru
ud3we
2ue
ueb2
ued2
ueen4
u1ef
ue1g
u1ei
ue2k
uel2l
uel1t
u1emi
ue2n
uen3a
uen1eb
uen2gl
uen3k2
ue2r
uer3a
uer1ä
uer3b
u3erbau
uer1d
uere4
uer4ei.
uer3eim
ueren2
uer3ent
uer3esk
uer5g2
u3erhal
uer3m
u3ernan
uern3st
uer1o
u1erö
u1erri
uert2
uer3tei
uer1th
uer3v
u1eta
ue4te
u1eti
ue1to
u1ex1
4uf.
uf1ab
u3fah
uf1ak
ufall2
uf1ane
uf3ar
u3fas
uf1au
uf1är
uf3äs
uf1äß
uf1ei
ufe3la
ufel3se
uf1em
u3fenst
ufe3rat
uf1erh
uf1eß
uf1et
2uff
uf1fl
u3fic
uf1id
uf1l
uf1n
uf1ori
uf1r
u2fru
uf3sä
uf4so
uf2spo
uf4s3tab
ufs3ter
2uft
ufte4
uft3erd
uft3erg
uf1tie
uft3s2
uf1tu
uf3z
2ug
uga4
ug2abe
u4gabte
ug1ak
ug1ap
ug1ar
ug1au
ug1d2
u3gef
ug1ei
u2geig
ugen3g
u3gep
ug1erf
uge3rie
ug1erl
ug2et
ug3f
ug2g
ugge2l
ug1id
u2g1la
u2g1lä
u2glic
ug3lis
u2g3lo
u2glu
ug3m
ug4men
u2gn
ug1nä
ugo3
ug3om
ug1o4p
u2g1or
u2g3rä
u2g3reg
u4g3reis
u2g3res
ug3rie
ug3ro
u2grol
u2grou
ug3rüs
ug3sau
ug3sei
ug3sek
ug3sim
ug3span
ugs4por
ug4spr
ug4spu
ugs4tan
ugs2to
ug3s2tr
ug3stu
ug3s2tü
u1gui
u2gum
u1h2a
u1hä
uh2d
2uhe
uhe1g
uhe1h
u1hei
uhe3k
uhe3ma
uhe3ra
uher2r
uhe1s
2uhi
uh1la
uh1lä
uhl3ent
uhls4
uh1ma
uh2me
uh1mi
u1ho
u3hö
uhr1a
uhr3er
uh1ri
uh2r1o
uhr1t
uhs2
uh1se
uh3sp
u1hü
uh1w
2ui
ui2l2
uin1g
ui3no
uin4tu
ui4s
uis2e
uis2l
uit4
ui1v
u1j
u2jol
uk2a
u2kaly
u1kau
uk2e
u1kera
u1kern
u1ki
uk2k
2uk2l
u1kla
u1klo
u3klot
u1klö
uk2n
uk2ö
uk2r
ukt1in
ukt3ori
ukt3r
u1ku
uk2ü
uk2z
ul1am
u3land
ulan2e
ul1äm
ulb2
ul1c
ul1die
ul1do
ul2dr
uld2se
2ule
u1led
u1lei
ul1el
ul1erh
uler2s
ules3t
ul1eta
ulf4
ul3fe
ul1id
ulik1t
ul1ins
u1liz
ul3ko
ul1ku
ull2m
ull1s
ulm2e
ul1n
ulo2
ul1op
ul1or
ul2ö
ul4p1h
ul3sus
4ulta
ul1tal
ult2an
ult1ar
ulte2
ult2h
ul3ton
ul1tro
ult3s
ul2v
ulz2
u1mac
um1ad
um1ak
uma1li
um1all
um1am
um1an1g
um1anz
um1ap
um1art
u1ma4s
u1mate
u1max
u2m1äh
umär4
um1ärg
um3b
um2bo
um4bus
um1d2
u1mec
um1ef
u1mei
um1ein
u3meis
umen1e
u4ment
umer2a
um1erf
um1erg
um1erl
um1erw
1umf
1umg
um1ide
um1ill
um1ind
um1inh
u2mins
um1ir
1umk
1um3l
2umm
um3mas
um4mes
um2mu
um1n
2umo
u3mol
u1mon
u2mor
ump2fa
ump2fi
umpf4li
um2ph
1umr
2ums.
2umsab
um4sam
um4s3an
1umsat
ums1er
2umsf
um2sim
2umsk
um2s1pe
umt2
um2tat
um1tel
um1th
um1ti
um1tu
um3um
um2un
um1ur
u1mus
1umz
3umzug
u2n
2un.
2una.
1u4nab
3unabh
un2a3br
u3n1ac
una1ga
un3an1
un1ap
2una1s
un3at
una1ta
u3nä
un1c
und3ak
1undd
und3erf
und3erö
und3erz
1undf
2undg
und3l
1undn
undo2
un3dot
2un2d1r
4unds.
und3sa
2undsc
und3sp
und3st
und1t
und1um
1undv
und4weg
1undz
un1e
2une.
un3ein
un3eis
un2emi
unen1e
uner3g
un3erz.
u3net
unf2
un5fa
unft4s
unf3z
2ung
un2gar
unge1b
3un3gena
3un3get
3ungew
ung5h
3un1gl
1unglü
ungs1
ung4sa
un2hof
u3nic
1unif
unik2
1unio
un3isl
uni1so
3u4niv
2unk
unk1a
unk1es
unk1n
unkom1
unk1s2
unk3si
un4meh
unn1ad
unn2e
un1o
1unr
uns2
unsch5el
un4sere
1un3si
un3sk
un3s4p
uns4t
2unsy
2unsz
1unt
un1ta
un1th
2un3ti
un4tig
un1to
un1t4r
unt3s
2untu
un2tum
2un1u4
u3num
u3nut
u3nü
unvoll3
1unw
2unwä
2unz2
un3za
2uo
u1ob
uo1d
u3of3
uor1c
uor2g
up2
u1pa
1upd
uper3
u2pfe
u2pf1i
1upg
u1ph
u2phr
up4i
u4plet
u1pr
upra1
up3si
upt3a
upt3erf
upt3erg
upt1o
upt3r
up3z
u1q
2ur.
ur2aba
u3rad
u3rah
ura3k
ur1am
ur3amt
ur1ana
uran3at
ur3anf
ur3ang
ur2anh
ur1an5s
ur1anz
ura1sc
ur3a2sp
ur3att
ur1au
ur3auf
2u1rä
ur1än
urb2
ur3bet
ur2bi
ur2bo3
urd2
ur3dam
ur3dir
ure3b
u1rec
u1red
ure1e
u1ref
ur1eff
ur2eg
ure2n
uren1e
ur1ep
ur1erh
ure2s
ure2t
ur1eta
ur2eth
2urf
urf4spr
ur4gi
urg3s
ur3h
u1rhy
uri3mu
ur1ini
ur3ins
ur1int
urio4
urk2
ur2ke
ur3kel
ur2ki
ur3kl
urk3t
ur3l
ur5la
ur4lige
ur1m2ei
urmen2
ur3mi
ur1mo
ur2n
ur3nam
ur3nat
ur3nä
urn2e
ur3no
2uro
uro3b
uro1c
uro1d
ur1off
uro1g
uro4kr
uro1m
ur3or
ur2os
uro1sc
2urö
ur3p
ur4pat
ur4pi
ur4pu
2ur2r
ur3ra
urre1c
ur3ro
ur4rou
ur3sac
ur3sla
ur2s1of
ur3son
ur4spa
urst4r
ur2su
urt2
ur1ta
ur3tei
ur1tem
ur3th
ur1tie
urt3sc
ur1tu
u3ruc
u1rui
ur3va
ur3ver
ur3vi
u3ryt
ur2z
urz1a
urz1ec
urz1o
urzt2
urz1w
2us
4us.
us3af
us1agi
us2al
us3alta
us1amb
u4samt
us3ark
us3art
u2s1as
us3ate
us3chec
u3schei
usch3mü
u4schun
usch5wer
usch7witz.
u3scr
us2eb
use3be
us1ec
u1see
us1ei
u1seid
usen2d
usen4tu
use2r
use3ran
us1erk
us1erl
us1ers
us1erw
us1ese
use3ta
us5f
u1sim
uski2
us3ko
us3m
u3soa
us3oc
us1oh
us1ou
u1sov
us3part
us1pas
us1pat
u3spek
us1pen
u2s1ph
us1pic
u2spo
us2por
u2sprä
us1pu
us3sa
us4sak
uss3aue
us5sende
us5sendu
uss3erf
uss3k
us1so
ussof3
uss3tab
uss3tät
uss3tic
uss3tor
uss1tü
us2sü
us3tag
u1stal
us1tar
us1tas
us1tau
u1stel
usten3t
ust3erl
us1thü
us1tic
us1tob
us1tor
u1stu
us1tur
us3ty
us1umd
us2ur
3usus
u2sü
u2s3z
2uß
ußen1
ußof3
uß3ri
uß1u
2ut
uta2
u3taf
ut1alt
ut2ama
ut2ami
ut1ap
ut1ar
ut1är
u1tät
ut1ed
ute3g
ut1ei.
ut1eie
ut1ein
ut1ela
u2tem
u2ten
ute3n2a
uten1e
u2ter
ute3ras
ut2es
u1teue
ut2ev
ut1ex1
ut1hel
ut1hil
uth1l
ut1ho
uti2
u1tit
ut3ne
uto1
uto3c
utor2a
uto3s
u1tö
u2t1öd
u2töl
4utr
u1tras
ut1rea
ut2s
ut3sa
ut4sab
ut3se
ut4sh
ut3sk
uts2ka
ut3so
ut3sp
ut4spa
ut3st
utt4l
ut2to1
utto3r
ut1tr
utts2
ut1tu
ut1une
ut2z
utz3eng
utz3g
utz1in
utz1w
2uu2
u1uf
uum1
u1uni
uur1
u1ü
u1v2
u2vi
u3vie
u1w
u3wä
2ux
ux2e
ux4p
uxt2
2u1z
uz3as
u3zei
u2zen
u2zer
uzer2n
u2zes
uz2li
uz1ot
u3zu
u3zü
uz1we
1ü2b
üb1ä
2übc
2üb2d
üb4e2
über3
über5h
üb1l
üb2le
üb1ro
üb2st
2üc
üch1l
üch2s
üchs1c
üch2t
ücht4e
ück1er
ück3eri
ück1l
ück4spe
ü4d
üd3a2
üd3d
üden2
üd3o
üd1r
üd3s2
üd3t4
üe2
ü2f
üf1a
üf1ä
üf1ei
üf1erg
üf2f
üf3i
üfin3
üf1l
üf2t
üft2l
üg2e2
üge3g
üge4l
ügel1a
üge3s
üg4g
üg1l
üg3s
üg4st
üh1a
üh3b
üh1ei
üh1eng
üh1ent
üh1er
üh3i
üh1lam
üh1lä
ühl2er
üh1li
ühmen2
üh1mi
üh3mo
ühn2s
üh1o
ühr2e
üh1rei
ühr3ei.
üh3rent
üh3ro
ühr3t4a
üh1s2
üht2a
üh1tau
üh3temp
üh1w
ük2
ü1ka
ü2l2
ül3a
ül3b
ül3ef
ü3lei
ülf1l
ülf1r
ülf4s
ül3h
ül3k
üll1ad
üll1au
ülls2
ül3m
ül3ö
üm1id
üm1in
üm2m
üm1u
2ü2n2
ün3a
ünd3ler
ün3do
ünd3s
ün3ei
ünf1a
ünf1ei
ün3fl
ünf3li
ün3gü
ün3ker
ün3ko
ün3r
ün3se
üns2l
ün3sp
ün3str
ünte2
ün3tee
ün3tö
ün3zo
üp2
üpel1
üpf3l
ü1pu
ü2r2
ür3a
ür3bo
üre2
ür3ei
ür3el
ür4fr
ürg3eng
ür4gi
ür3gr
ür3h
ür4k2e
ür3kl
ür3na
üro3
ürom2
üror4
ürr2
ür3ra
ür3sc
ür3se
ür3si
ür3so
ürt2
ürtel1
ür3v
ür3za
ür4ze
ü2s2
üsen1
üse1s
ü3sp
üss1c
üss2e2
üt1al
üte1d
üte3g
üte3m
ütent2
üten3z2
üt1r
üt2s1
ütt2g
üt2z
ütz2e
ü1v
2va.
v1ab
va3br
va1f2
3vag
vag2a
4vai
va4k2
4vald
val2s
va1ma
2vang
van2n
2vap
vas2
va1st
v4a2t
vat1a
vat3h
vatik2
vat1in
vat3m
vat1r
vat3s2
vat3t
vat1u
v1au
v1b
v1c
v1d2
ve2
4ve.
2vea
ve3ar
veau1s
2ve3b
ve3c
2vef
ve3gr
veh2
ve3ha
veil2
v1ein
veit4
veits1
2vel
ve3lan
vel1au
v1ele
ve3lei
velt4
2vem2
ve3ma
ve3mu
4ven.
4vend
ven3k
1venö
ven2t
vent3ag
2veo
ve3of
2vep
ver1
4ver.
1ver3a
ve3rad
ver3b2l
verd2
ver3da
veren2
verf2
3verg2
ver3m
ver4mon
vern2
ver3r
ver3sta
vert4
ver2tü
ver3u
ver3v
ver4ve
3verw
ver3z
ve3sa
ve3sc
ve3se
ves2t2
2vet4
ve3ta
ve3tr
2vev
2ve3w
4vey
v1f2
v1g
v1h
vi4
2vi.
via2
4vianti
2vic
vid3st
vie4
vier1d
3view
2vii
3vik
vil1a
vil1eh
vil3in
vil1se
v1im
vin2
v1ind
ving5
v1int
vinz3l
1vio2
3vir
vis2
vis4z
viv2
vize3
2v1k
v1l2
v1m4
2vn2
v1ne
1vo2
4vo.
vo3a
4v1ob
vo3bl
vo3br
voge2
vogel1
v2ol4
voll3ar
voll7auf.
voll5end
4vo3m
von1
v1op
vor1
vor3a
vor3b
vor3da
vor3di
vor3ec
vo3reg
vor3gi
vorm2
vormen4
vor3p
vor3t2
vor3z
4vos
voß2
vot2a
5voti
vot1t
2vou
völ4
2vös2
4v1p
2vr2
v1ra
v1re
v2ree
v1ro
4vs2
v1sc
v1se2
v1si
v1so
v1sta
v1su
v1sz
4vt
v1ta
v1tä
v1to
v1tö
v1tr
vue3
vul2
v1ü
v1v
v1w
2vy1
v1z
w2a4
4wa.
1waa
wab2
wa5ba
wa5bo
waffel3
1wag
3wagen
1wah
wahl5ent
wai2
waib2
2wak
wal2bu
wal2c
1wald
wal2k
wal2m1
wals2
wal2t
walt3a
walt4st
3walz
wam4
wan2
2wana
wand1a4
w3an3f
2wang
wang2e
wang4s1
1wann
wan3se
wanz5end
war4
2war.
1ware
ware1i
3wareke
ware1l
waren1
wart4e
1was2
wasch3l
wasi3
wass4
wasse4
1wäh
1wäl
2wäng
wär4
1wäs
w1äu
w1b
w3bi
w2boy
2w1c
w1do
we2
4we.
we4b
web1a
webe2r
web1l
2webs
4websa
web3se
wee2
weed3
1we4g
weg1a
weg5ersc
weg3l
weg1n
weg1o
weg3r
weg3s
wegs2a
wegs2t
1weh
weh1l
weh1m
weich3l
2weie
wei2er
wei1k
3weil
wei1m
wei4n
wein4s3a
wei1p
wei1sc
weis4se
weis4s3p
wei1ta
wei2t1r
wel2
wel3b
well2s
welt1
welt3r
we3me
we3mo
wen2d
wen2g
wen3ge
wen2i
wenk3ri
we4r2
2wer3a
1wer4bu
werd2
werde3i
5werdens
1werdu
2werg
werg3el
werg1o
wer3i
wer4in.
wer1io
1werk.
1werke
wer3l
wer3ma
wer3me
wern4h
wer3ni
2werp
wer3r
2wers
wert3a
wert3ei
wert5erm
wert1o
wert3ri
wer3v
we4s2
1wese2
wesens3
wes3h
west1a
west3ei
west1o
west1r
1wet
4wete
4weto
2wets
wett3s
4we3u
4we3v
4wey
w1g
w3ho
w2i2
3wid2
2wieb2
wie3m
wien2e
wien1t
wie4st
4wig
wig4s
2wik
1wil2d
wil2f
wil2h
wim4
wimm3u
win4
4win.
wind3ec
2wing
4wini
winn7ersc
wint2
1wir
wire1
wirk1l
wis4
3wiss
wit2
2wits
4witzd
w1k
2wl
2w1m
wns2a
wn3sh
wn1t
wo2a
wo2b
1woc
wo4d
wo4f
3woh4
wo1he
1wolf
wolf4s1
won2
wo2r4
woren2
wort3a
wort1r
wo3si
wo2v
wo2z
1wöc
wöl4
wör4
2w1p
1wr
wrest4
w1ro
2ws
ws2e
w3s2h
ws2k
w4ski
w3sky
ws1s
wsser2
w1sti
2wt
w1ta
w1term
w1ti
wton2
w3tons.
w1tr
w4u2
1wuc
wuch2
wul2
wul4f
wuls2
wul3se
wun4
wund4e
wung3r
wur2
4wur.
wur4f
wur3g
wur4l
1wurst
wus2
1wut3
wut4h
1wüh
wür4
wüte2
2ww
2w1z
x1a
xa3by
x2a3d
xa1fl
x2a1g2
x2an
xan2d
x3ann
x4ant2
x4anz
x2as
x1b
x1ca
x1ce
x1ch
x1cl
x1d
x2du
xe4
x1eges
x3egete
x1ele
x1em
x2eme
x2ems
x2en
xen1k
xen3s2
xen1t
xer2
xer3a
x2ere
xer3s2
x1eu
x1ex
x1f
x1g
x2gl
x1h
xi1c
xich2
xid3em
x1ido
xie3l
xi1g
xi2k
xil1l
xil3u
xin3s2
xio2
xi1r
xis1e
xi1si
xis1o
xi2sp
xis3s
x1itu
xi1ze
x1j
x1k
xkal2
xkon1
xl2
x1lä
x1le
x1m
x1n
xo1d
xo2du
x1oe
xo1p
x1or2
x1ö
x3p
xpan4
x1q
x1ri
x2ro
x3s2
x2t1a
xt2ane
xt2ant
xt2as
xt1ä
x1tät
xt1ed
xt1ein
xt1el
x1tens
xt1erf
x1term
xt1ev
xt1h
x2ti2
xt1ill
xt1o
xtra3b2
x2trag
x1trak
xt1ran
xtra1t
x1trau
x2tre
xt1re1c
x1tru
xt3s2
xt1t
x2tu
xt1um
xt1un
xt3w
xt1z
xu2
xual1l
x1un
xus3
xuss2
x1ve
x1w
x3wer
xy2g
xy1m
xy1r
xy1t
x1z
y1ab
1yac
ya3g
ya1la
y1am
yan4m
ya1ra
y1art
ya2s
yat2
y1ät
y1b2
y2bi
y1ca
yco3b
y1cr
y2d2
y3dec
y3deu
y3do
ydro1
y1ed
y1ei
y2el
yen2
yern1
yes2
y1est
y3f2
yg2
y1ga
y1ge
y1gl
y1go
y3gr
y1gu
y3h
yhr2
y1in2
yk2
y1ki
ykol3
ykot2
y1ku
yl1am
yl4ante
y1lä
yl1c
yl1d
yl1em
yl1es
yl2l
yl3lan
yl4lo
yl3m
ylo1g
yl3s
yl2t
y1m4a
y1m4e
ym1m
ym2n
yn1
yn2a2
ynk2
y1n2u
yo1c
yo1k
yom4
yon4
yo1p
yor2
y1ou
y2p
y3pa
yp1ab
yp1an
y3pä
yp2e
ypen3
yper1t
y3pho
yp1id
yp1in
yp1l
y4po3
y4ps
yp1th
yp1um
yr2
y1rau
y1rei
y3rig
yrin2
yro1
yro2p
yrr4
y2sa
ys2c
ys2e1
y3s2h
y4sl
ys3la
y1s2o
ys2pa
ys2s
yst2e4
yst2h
y1sty
y1s2z
y1ta
yt2h
yto1s
y1tr
y1tu
yu2
yur2
y1v
y3w
yz2
y2za
1za
4za.
2z1ab
2zach
z1achs
2z1ad
2z1af
za2g
3z2ah
2z3ak
za3ka
z5akk
2zal
z2ali
2z1am
z1an
2z3anf
2z3angs
3z2an2k
z3anl
z2ano
2z3anr
zap2
z1ar
za1res
2zarm
z2aro
2zarti
2z1as2
zast2
2z1aß
2z3at4
za1te
za1ti
zau2
3zaub
z1auf
z3aug
3zaun
2z1aut
2za3v
1zä
z1äc
3z2äh
zä1hei
2z1äm
z1än
z1ärg
z1ärm
2z1äus
2zäuß
4z1b2
z3be1
zbe2i
zbe2to
z2bol
2z1c
z3ch
2z1d2
zde2
zdi3st
z3dü
4ze.
2ze2a
ze3ad
ze3am
ze3au
2ze1b
z1eben
2z1echo
ze2d
ze3di
ze1e
2zef
z1eff
2ze1g
3zehe
zehen1
3zehnm
1zei
zei1b
zei1d
2zeie
z2ei1f2
zei2g
zei1k
zeil2
zei1la
zei1m
4z1ein
z2e1ind
ze3inse
zei1r
zei1s2
zei3sk
zeiss4
zeiß2
zei2t
zeit3er
zeit3ri
zei1z
ze2l
4zel.
zel1a
zel1ä
zel1d
zel1er
zel1in
zel2l
zel3lau
zel1lä
zel3läu
3zeller
3zellig
zel3lö
zelm4
zel1o
zels2
zel1sk
zel1sz
zel1tr
2zem4
ze1mi
4z1emp
ze2n
4zen.
2zena
zen1ac
2zend
4zene
4zeng
z4engl
zen1k
2zenn
zen1ne
4zensio
zens2p
zen5s6tr
zen3tal
zen1th
zen3tis
z2entn
8zentralit
zent3s
4zenz
zenz3er
zen2zw
2zeo
zeo1l
ze4ph
ze2pi
2zer.
zer1d
ze1rek
2zerg
z1ergä
z3ergeb
z2erhe
z3erhö
2zerk2
z2erko
zer3m
zer4mat
z2ern
zer2ne
zern3eb
z1erö
2z1erq
zer2ru
z1ersa
z3erste
zert3ag
zert4an
zer2to
zert5rau
2z1erw
z2erwü
z1erz
ze2s
ze3s4c
zes1e
ze4s3p
zes2s
zess1c
zess5end
zess1t
ze3sta
zes1tr
zeß3
1zet
2zet.
2zeta
2zett.
2zetts
3zeu2
zeug3r
z1ex1
2zey
2zez
2z3f4
2z1g2
zgang5
zge2ni
zger2a
z2gere
zge5rinn
z2germ
zger2s1
zgie4
4z1h2
zher2
zi2
2zi.
2zi4a2
2zi3b
1zic
zich2
zid1r
zie4
3zieh
5ziehs
5zieht
1ziel.
2zien
4zierk
zie5sc
2zig
zig2e
zi3go
zi3h
2zil4
zi3la
zi3lit
1zim2
zi3me
3zimm
2zimp
zimt3
2z1ind
zin2e
zin3ei
z1inf
z1inh
zin2n
zin2s
zin3sc
zin3sta
z1inv
zi4o
zio1d
2zipr
3ziproz
1zir
zirk2
zirk4s
z1irr
zis4
z3iso
ziss4
zi3sz
ziß4
2zit.
2zitä
zite4
zit1o
zit2r
2ziu
ziv2
zi3z
z1j
4z1k2
zkom1
2z1l2
z4laue
z4laus
z3leb
z3led
z3leg
z3leh
z3lein
z3leis
zle3t
z2li3p
z2lu
z3luf
z2ly
4z1m2
2z1n2
znach4b
z2ne
znei1
z3nel
z3neu
1zo
2zo.
z1oas
z1ob
2zo1d
zo2e
z1of
zof3f
zo2g
3zogen.
2zogi
2zogl
zog4s
3zogst.
3zogt.
4z1oh
zo1ha
zo1k
2zol.
2zola
2zole
zom4
zo1ma
zonal2
zon3s
zont3er
zo2o
zoo3f
zoo3t
zoo3v
2z1ope
zo3pi
z1or2
3zorn
2zort
4zo1s2
z1osz
zot2
1zög
z1ök
z1öl
z2öll
2zöls
2zön
2z1p4
z1q
2z1r2
4z1s2
z3sa
zse2
z3sh
z3sk
4zt
z1ta2
zt1ane
z2t1au
z1tä
z3tee
zte3g
z1tei
zt3eins
z1t2el
zte3ma
zt1ent
zte3o
z1term
zte3str
z1teu
zt1h
z2thei
z1ther
zt3ho
z2tin
z1to4
z1tö
z1tr
zt1rec
zt2ru
zt3s
zt1t
z1tu
z1tü
zt1z
1zu1
zu3a
3zu3b2
5zuc
zuch2
zuck4
zud4
zue2
zu2el
zuer1
3zuf2
3zug
zu2gar
zug3g
zu2glö
zugs1
zug4stu
zug1un
z1uhr
zu3i
3zul2
3zum
4z1um.
zumen2
zum2i
4zumr
z1ums
2zun
zu3na
3zu3n2e
3zunf
zung4
z1ungl
z1u3ni
z1uns
zu3nu
zur4a2
zu3re
2z1urk
2z1url
z1urs
z1ur1t
3zu3s2
zusch4
3zut4
zu3u
zu3v
3zuw
zuz2
1zü2
2züb
3züg
3zün
zünd4le
2z3v
1zw2
z1wac
2zwag
2z1wal
zwand1
z1war
2zweg
2zweh
2zweil
2z1wel
2z1wen2
2z1wer4
2z1wes
z2wic
zwie3g
zwie3l
zwil1
2zwiss
z2wit
2z1wo
z1wör
z1wur
2z1wü
3zy2
4zy.
4zylb
4zyle
2zz
zz4a
zza3b
zza1m
z1ze
z3zei
z3zet
zz1id
zzi3s
zz2l
zz3la
z1z2ö
z3zu
zzug4
    """)

    def patterns(self):
        """Invariant: patterns() is of type “unicode”.
        Invariant: patterns() provides patterns for the hyphenation algorithm.
        These patterns do not break at hyphenation points, but break where
        ligatures have to be suppressed in german texts. Example:
        auffallend → auf fallend
        The patterns are all lower case.
        """
        return GermanLigatureSupport.__germanLigaturePatterns

    def get_word_characters(self):
        """Invariants: Returns a “unicode” value that have to be
        considered as word characters when using the patterns.

        This is the content of the file “german.tr” of the Trennmuster
        project (both capital letters and small letters), adding LATIN
        CAPITAL LETTER SHARP S (ẞ) and LATIN SMALL LETTER LONG S (ſ). So it
        contains all characters that occurred in the original word list from
        which the patterns are generated. (So it is likely
        that is contains more characters than the patterns itself.)

        It contains also at least for the pattern itself (not necessarily
        the hole word list) every character to represent every normalization
        form.

        NOTE There is no special treatment for Unicode normalization forms.

        WARNING This function must be kept in synch with
        simple_case_fold_for_lookup() and also with
        InstructionProvider.get_instructions() which does some Unicode
        normalization.
        """
        return (# from german.tr
                u"aAäÄâÂàÀáÁãÃåÅæÆbBcCçÇdDeEéÉèÈëËêÊfFgGhHiIíÍìÌîÎjJkKlLmMnN"
                u"ñÑoOóÓòÒôÔöÖøØœŒpPqQrRsSšŠßtTuUúÚüÜvVwWxXyYÿŸzZžŽ"
                # U+1E9E LATIN CAPITAL LETTER SHARP S
                u"ẞ"
                # U+017F LATIN SMALL LETTER LONG S
                u"ſ"
                # combining characters
                # (normalization support, canonical decomposition)
                u"\u030C\u0302\u0308\u0301\u0300\u030A\u0303\u0327")


class InstructionProvider(GermanLigatureSupport):
    def __init__(self):
        GermanLigatureSupport.__init__(self)
        self._myHyphenator = Hyphenator(self.patterns())

    def get_instructions(self, my_word):
        """ Get instructions for a single german word “my_word”.
        Precondition: my_word is of type “unicode”.
        Postcondition: Returns a list with as many elements as my_word has
        elements. Each element is either True, False or None. True means
        that at this index position, a ZWNJ has to be introduced, making
        all characters starting from this index position shifting to the
        right. False means that at this index there is a ZWNJ and this
        ZWNJ has to be removed. None means that nothing has to be done.
        Of course, the list might be empty if my_word is empty. my_word
        is supposed to contain a single word, and not various words with
        whitespace. This function handles correctly the soft hyphen
        U+00AD and the following characters with canonical decomposition:
        šâäéóöü
        :rtype: list
        """
        if type(my_word) is not str:
            raise TypeError("myWord must be of type “unicode”, but it isn’t.")
        stripped_word = u""
        stripped_word_index = []
        folded_word = self.simple_case_fold_for_lookup(my_word)
        # Handle normalization…
        #
        # The string is yet folded, so only small letters need to be
        # handled. We substitute the decomposed form by the composed
        # form (that is used in the pattern). To avoid a different
        # character count, we introduce a soft hyphen. This is okay,
        # because the soft hyphen will be ignored later anyway, and
        # ZWNJ are always inserted before normal characters and never
        # before soft hyphens. We handle only the minimum of
        # canonical composition that really occurs in our pattern.
        #
        # WARNING: This must be kept in synch with
        # GermanLigatureSupport.get_word_characters().
        folded_word = folded_word.\
            replace(u"s\u030C", u"š\u00AD").\
            replace(u"a\u0302", u"â\u00AD").\
            replace(u"a\u0308", u"ä\u00AD").\
            replace(u"e\u0301", u"é\u00AD").\
            replace(u"o\u0301", u"ó\u00AD").\
            replace(u"o\u0308", u"ö\u00AD").\
            replace(u"u\u0308", u"ü\u00AD")
        for my_index in range(len(folded_word)):
            my_character = folded_word[my_index]
            # Do not copy SOFT HYPHEN and ZERO WIDTH NON JOINER
            if my_character not in u"\u00AD\u200C":
                stripped_word += my_character
                stripped_word_index.append(my_index)
        hyphenated_word = self._myHyphenator.hyphenate_word(stripped_word)
        # correct_zwnj_positions will contain the indexes of alphabetic
        # characters before which a ZWNJ
        # should be (index relative to my_word).
        correct_zwnj_positions = []
        i = 0
        # For all partial strings but the last one:
        for j in range(len(hyphenated_word) - 1):
            i += len(hyphenated_word[j])
            correct_zwnj_positions.append(stripped_word_index[i])
        # Make a list with one element for each element in my_word.
        # None means “do nothing”, True means “insert a ZWNJ here,
        # and False means “delete this ZWNJ”.
        my_status_list = []
        # First step: remove all ZWNJ, leave the rest without change.
        for my_character in my_word:
            if my_character == u"\u200C":
                my_status_list.append(False)
            else:
                my_status_list.append(None)
        # Second step: cancel the remove where not necessary and
        # add ZWNJ where necessary.
        for i in correct_zwnj_positions:
            if my_status_list[i - 1] == False:
                my_status_list[i - 1] = None
            else:
                my_status_list[i] = True
        return my_status_list


def is_bmp_scalar_only(my_string):
    """In Python 2, the data type “unicode” is supposed to be a unicode
    string and is a sequence of code units. The encoding form might be
    either UTF-16 or UTF-32, depending on the compile time options of the
    python compiler. What does this mean? Example: The character U+0001F404
    COW might be represented either as a sequence of two UTF-16 surrogate
    pairs (string length: 2) or as a single UTF32 code unit
    (string length: 1). If you want to know if you have to deal
    with surrogates or not, you can use this function. It works
    on both UTF-16 and UTF32 sequences.

    Preconditions: my_string is of type “unicode”

    Postconditions: Returns “false” if the string contains at least one code
    unit that is not a Unicode scalar value or is not within the
    Basic Multilingual Plane. Returns “true” otherwise. Quoting from the
    Unicode standard:
    “D76 Unicode scalar value: Any Unicode code point except high-surrogate
    and low-surrogate code points. As a result of this definition, the set of
    Unicode scalar values consists of the ranges 0 to D7FF (16) and E000 (16)
    to 10FFFF (16), inclusive.”
    """
    if type(my_string) is not str:
        raise TypeError(
            "“my_string” must be of type “unicode”, but it isn’t.")
    return re.search(u"[^\u0000-\uD7FF\uE000-\uFFFF]", my_string) is None


def get_affected_text_objects():
    """Preconditions: import scribus
       Postconditions: Returns a list. The list contains the unique object
       identifier string of each affected text frame in the currently active
       document (if any). If there is a set of objects selected, than only
       the text frames within this set are affected. (This might result in
       an empty list if only non-text-frame objects are currently selected.)
       If there is no currently active document, than the list is empty."""
    my_return_value = []
    if scribus.haveDoc() > 0:
        for i in range(scribus.selectionCount()):
            my_object = scribus.getSelectedObject(i)
            if scribus.getObjectType(my_object) == "TextFrame":
                my_return_value.append(my_object)
    return my_return_value


class StoryInterface:
    """An interface to the content of a “story” in scribus. A story is
    the common text content that is shared by various linked text frames.

    Some of Scribus’ script API is about stories, other parts are about
    the text frames, using different indexes which leads sometime to
    unexpected behaviour. This class offers a simple interface to the
    story content. It hasn’t many functions, but it’s a consistent
    interface and avoids unexpected side effects. It works as expected
    independently of the current text selection, but it might change
    the current text selection. It takes care of the necessary encoding
    and decoding when dealing with scribus, while its public interface
    exposes only functions that work with the data type “unicode”.

    Note: Objects in Scribus (Text frames, images, geometric forms…)
    are supposed to have a unique identifier. However, there is bug
    https://bugs.scribus.net/view.php?id=11926 that allows non-unique
    identifiers. The interaction of StoryInterface with this bug
    has not been tested.
    """

    def __init__(self, text_frame_identifier):
        """Precondition: text_frame_identifier has the data type that is
        used for identifiers for scribus objects and is not empty.
        Postcondition: The object is created."""
        # We assume that the data type of the Scribus object identifier
        # is “str”.
        if type(text_frame_identifier) is not str:
            raise TypeError(
                "argument “text_frame_identifier” has wrong data type.")
        if text_frame_identifier == "":
            raise TypeError(
                "argument “text_frame_identifier” might not have "
                "empty content.")
        self.__identifier = text_frame_identifier

    def read_text(self, first, count):
        """Precondition: The object with the unique identifier “textFrame”
        (constructor argument) currently exists in the current document,
        and it refers to a text frame. “first” and “count” are non-negative
        integers. The requested range exists really.
        Postcondition: Returns a value of type “unicode” that contains the
        requested text range (the total number of “count” indexes, starting at
        index “first”). Note that the indexes are the indexes provides by
        scribus. This may not be unicode characters, but UTF16 code units, and
        if you choose half a surrogate pair, scribus will silently add the
        missing half surrogate pair. The indexes does not refer to the actual
        text content of “textFrame”, but to the content of the underlying
        “story”, that means the common text content that is shared between
        this text frame and all linked text frames. Note that this function
        will (likely) change the current text selection of the story."""
        if (type(first) is not int) or (type(count) is not int):
            raise TypeError("Both arguments, “first” and “count”, must be "
                            "integers, but they aren’t.")
        if (first < 0) or (count < 0):
            raise IndexError("Both arguments, “first” and “count”, must"
                             "not be negative, but they are.")
        if scribus.getObjectType(self.__identifier) != "TextFrame":
            raise RuntimeError("The argument “textFrame” that was given in "
                               "the constructor does currently not refer to "
                               "a text frame in the current document.")
        # If count is 0, scribus.selectText will select nothing. But when
        # nothing is selected, scribus.getAllText will not return an empty
        # string, but the hole content of the story. That is not what we
        # expect, so we have to catch this case manually.
        if count == 0:
            if first >= self.length():
                raise IndexError("“first” is out of range.")
            return u""
        scribus.selectText(first, count, self.__identifier)
        return scribus.getAllText(self.__identifier)

    def delete_text(self, first, count):
        """Precondition: The object with the unique identifier “textFrame”
        (constructor argument) currently exists in the current document,
        and it refers to a text frame. “first” and “count” are non-negative
        integers. The requested range exists really.
        Postcondition: Starting from index “first”, the number of
        “count” indexes are removed. Note that the indexes are the
        indexes provides by
        scribus. This may not be unicode characters, but UTF16 code units, and
        if you choose half a surrogate pair, scribus will silently add the
        missing half surrogate pair. The indexes does not refer to the actual
        text content of “textFrame”, but to the content of the underlying
        “story”, that means the common text content that is shared between
        this text frame and all linked text frames. Note that this function
        will (likely) change the current text selection of the story."""
        if (type(first) is not int) or (type(count) is not int):
            raise TypeError("Both arguments, “first” and “count”, must be "
                            "integers, but they aren’t.")
        if (first < 0) or (count < 0):
            raise IndexError("Both arguments, “first” and “count”, must"
                             "not be negative, but they are.")
        if scribus.getObjectType(self.__identifier) != "TextFrame":
            raise RuntimeError("The argument “textFrame” that was given in "
                               "the constructor does currently not refer to "
                               "a text frame in the current document.")
        # If count is 0, scribus.selectText will select nothing. But when
        # nothing is selected, scribus.deleteText will delete the
        # hole content of the story. That is not what we
        # expect, so we have to catch this case manually.
        if count == 0:
            if first >= self.length():
                raise IndexError("“first” is out of range.")
            return
        scribus.selectText(first, count, self.__identifier)
        return scribus.deleteText(self.__identifier)

    def insert_text(self, text, first):
        """Precondition: The object with the unique identifier “textFrame”
        (constructor argument) currently exists in the current document,
        and it refers to a text frame. “first” is a non-negative
        integer ≤ length(). “text” is of type “unicode”.
        Postcondition: The text is inserted at the given index position.

        Note that the indexes are the indexes provided by
        scribus. This may not be unicode characters, but UTF16 code units,
        and if you choose half a surrogate pair, scribus will insert between
        them, making the unicode text string invalid.
        0The indexes does not refer to the actual
        text content of “textFrame”, but to the content of the underlying
        “story”, that means the common text content that is shared between
        this text frame and all linked text frames. Note that this function
        will (likely) change the current text selection of the story."""
        if (type(first) is not int) or (type(text) is not str):
            raise TypeError("“first” must be “integer” and “text” must "
                            "be “unicode”, but they aren’t.")
        if first < 0:
            # We have to check for “first < 0” here, because scribus
            # would accept “first == -1” as valid, but we do not
            # want this behaviour here.
            raise IndexError("“first” must"
                             "not be negative, but it is.")
        if scribus.getObjectType(self.__identifier) != "TextFrame":
            raise RuntimeError("The argument “textFrame” that was given in "
                               "the constructor does currently not refer to "
                               "a text frame in the current document.")
        scribus.insertText(
            text,
            first,
            self.__identifier)

    def length(self):
        """Precondition: The object with the unique identifier “textFrame”
        (constructor argument) currently exists in the current document,
        and it refers to a text frame.
        Postcondition: Returns an integer that represents the length
        of the text. The measurement unit of the length is provided
        by Scribus. It is in Scribus 1.5.2 not “Unicode Scalar Values”,
        but “UTF16 code units”.
        """
        return scribus.getTextLength(self.__identifier)


def show_messagebox(
        caption,
        message,
        icon=scribus.ICON_NONE,
        button1=scribus.BUTTON_OK,
        button2=scribus.BUTTON_NONE,
        button3=scribus.BUTTON_NONE):
    """Shows a message box. Use this function instead of the original
    function to get more exceptions for wrong argument types and be
    forced to use unicode string. (This function uses an explicit conversion
    command for the unicode string instead of relying to the default
    encoding.)

    Preconditions: “caption” and “message” are of type “unicode”. icon,
    button1, button2 and button3 are either not used or of type int.
    Postcondition: Calls show_messagebox and returns the result."""
    if type(caption) is not str:
        raise TypeError("“caption” must be of type “unicode”, but it isn’t.")
    if type(message) is not str:
        raise TypeError("“message” must be of type “unicode”, but it isn’t.")
    if type(icon) is not int:
        raise TypeError("“icon” must be of type “int”, but it isn’t.")
    if type(button1) is not int:
        raise TypeError("“button1” must be of type “int”, but it isn’t.")
    if type(button2) is not int:
        raise TypeError("“button2” must be of type “int”, but it isn’t.")
    if type(button3) is not int:
        raise TypeError("“button3” must be of type “int”, but it isn’t.")
    return scribus.messageBox(
        caption,
        message,
        icon,
        button1,
        button2,
        button3)


def do_ligature_setting():
    """Do the ligature setting for the stories behind the selected text
    frames.

    Preconditions: Scribus is available.

    Postconditions: Does the ligature setting."""
    def ligature_setting_for_story(identifier, provider):
        interface = StoryInterface(identifier)
        used_characters = provider.get_word_characters() + u"\u00AD\u200C"
        i = 0
        while i < interface.length():
            temp = interface.read_text(i, 1)
            if temp in used_characters:
                # So temp is guaranteed to contain exclusively Unicode
                # Scalar Values inside the BMP. This is because
                # used_characters does so. This is tested (has
                # to be tested!) before this local function is
                # called.
                # Also my_word will have this characteristic.
                my_word = temp
                story_index = i
                i += 1
                while i < interface.length():
                    temp = interface.read_text(i, 1)
                    if temp in used_characters:
                        my_word += temp
                        i += 1
                    else:
                        break
                # It might be better to ignore words, that have in their
                # character style not the german language attribute, but
                # another language attribute. However, I do not know how
                # to get this working.
                instruction_list = provider.get_instructions(my_word)
                for j in range(len(instruction_list)):
                    # Scribus indexes are UFT16 code units.
                    # The instructions use as indexes either UFT16 code units
                    # or UTF32 code units, depending of the Python build.
                    # However, we have made sure earlier that only Unicode
                    # Scalar Values inside BMP are in used_characters, so
                    # it is guaranteed that here the indexes are the same,
                    # whether it is UTF16 or UTF32. So we can safely use
                    # these indexes to do our modifications in Scribus.
                    if instruction_list[j] == True:
                        # Insert here a ZWNJ.
                        interface.insert_text(u"\u200C", story_index)
                        i += 1
                        story_index += 2
                    elif instruction_list[j] == False:
                        # Delete here a ZWNJ.
                        interface.delete_text(story_index, 1)
                        i -= 1
                    else:
                        # Don’t make any modifications here.
                        story_index += 1
            else:
                i += 1

    german_locale = \
        re.search(u"^de(u|@|_|$)", scribus.getGuiLanguage()) is not None

    if scribus.haveDoc() <= 0:
        if german_locale:
            show_messagebox(
                u"Ligatursatz",
                u"Kein Dokument gefunden.")
        else:
            show_messagebox(
                u"Ligature setting",
                u"No document found.")
        return
    all_my_objects = get_affected_text_objects()
    if len(all_my_objects) <= 0:
        if german_locale:
            show_messagebox(
                u"Ligatursatz",
                u"Kein Textrahmen ausgewählt.")
        else:
            show_messagebox(
                u"Ligature setting",
                u"No text frame selected.")
        return
    if german_locale:
        result = show_messagebox(
            u"Ligatursatz",
            u"<b>Einführung</b>"
                u"<br/>"
                u"Einige Buchstaben\xADkombinationen sind eine "
                u"Heraus\xADforderung für die Gestaltung von "
                u"Schrift\xADarten. Ein gutes Beispiel ist die "
                u"f\u2011i\u2011Kombi\xADna\xADtion. Das "
                u"kleine\xA0f ist oft aus\xADladend gestaltet, und so "
                u"ent\xADsteht eine unschöne Lücke zwischen dem kleinen\xA0f "
                u"und dem kleinen\xA0i. Viele Schriftarten lösen dieses "
                u"Problem durch Ligaturen. Die Buch\xADstaben werden "
                u"ver\xADschmolzen; so wird die Lücke ver\xADmieden und das "
                u"Schrift\xADbild wirkt harmonischer. Im Deutschen gibt es "
                u"nun viele zusammen\xADgesetzte Wörter. Ligaturen "
                u"erschweren hier aber das Lesen. Des\xADhalb wird bei "
                u"deutschen Texten <i>in solchen Fällen</i> keine "
                u"Ligatur gesetzt."
                u"<br/>"
                u"<br/>"
                u"<b>Umsetzung</b>"
                u"<br/>"
                u"Ligaturen werden überall dort gesetzt, wo sie nicht "
                u"durch einen Binde\xADhemmer (Unicode-Zeichen U+200C) "
                u"unter\xADdrückt werden. Dieses kleine Skript arbeitet "
                u"sich nun durch alle ausgewählten und alle mit ihnen "
                u"verketteten Textrahmen und ergänzt und entfernt "
                u"Bindehemmer gemäß den Regeln für deutschen "
                u"Ligatursatz. Dieses Skript basiert "
                u"auf einem Wörter\xADbuch, in dem für viele Wörter der "
                u"passende Ligatur\xADsatz hinter\xADlegt ist. Bei Wörtern, "
                u"die nicht im Wörter\xADbuch ver\xADzeichnet sind, "
                u"ver\xADsucht das Skript, den passenden "
                u"Ligatur\xADsatz zu „erraten“. "
                u"Das Skript setzt Binde\xADhemmer an allen "
                u"Morphem\xADgrenzen, und zwar unabhängig davon, ob die "
                u"unter\xADdrückte Ligatur gebräuchlich ist oder in der "
                u"dort verwendeten Schrift\xADart überhaupt vorkommt. Somit "
                u"wird beim Wort „Auflage“ die häufig anzutreffende "
                u"fl\u2011Ligatur durch einen Binde\xADhemmer unterdrückt. Aber "
                u"auch beim Wort „Aufgabe“ wird zwischen\xA0f und\xA0g ein "
                u"Binde\xADhemmer gesetzt, obwohl eine fg\u2011Ligatur nicht "
                u"gebräuchlich ist. Dieses Vor\xADgehen hat den Nach\xADteil, "
                u"dass wohl viele Binde\xADhemmer gesetzt werden, die "
                u"keinen Effekt haben, weil die Schrift\xADart diese "
                u"Ligaturen überhaupt nicht enthält. Dieses Vor\xADgehen "
                u"hat aber den Vor\xADteil, dass es im Grund\xADsatz für alle "
                u"Schrift\xADarten funktioniert – auch dann, wenn "
                u"Schmuck\xADligaturen ver\xADwendet werden oder die "
                u"Schrift\xADart exotische Ligaturen ent\xADhält.",
                scribus.ICON_NONE,
                scribus.BUTTON_OK,
                scribus.BUTTON_CANCEL)
    else:
        result = show_messagebox(
            u"Ligature setting",
            u"<b>Introduction</b>"
                u"<br/>"
                u"Some character combinations are a challenge for type "
                u"design. A good example is the f-i\xA0combination. The "
                u"small\xA0f has often a spreading design, so there is an "
                u"unpleasant gap between the small\xA0f and the small\xA0i. "
                u"Many fonts solve this problem by using ligatures. The "
                u"glyphs are melt together and the typeface is more "
                u"harmonious. But in German, there are many compound "
                u"words. Now ligatures make reading more difficult. "
                u"Therefore <i>in such cases</i> ligatures are not used "
                u"in German."
                u"<br/>"
                u"<br/>"
                u"<b>Implementation</b>"
                u"<br/>"
                u"Ligatures are used everywhere, except they are "
                u"suppressed with a zero\xA0width non-joiner (Unicode "
                u"character U+200C). This little script walks through "
                u"all selected text frames and all text frames linked "
                u"with them, and completes and removes zero\xA0width "
                u"non-joiner following the rules for German ligatures. "
                u"This script is based on a dictionary "
                u"that provides the fitting ligature setting "
                u"for many German words. For words that are not in the "
                u"dictionary, the script tries to guess the fitting "
                u"ligature setting. The script inserts zero\xA0width "
                u"non-joiners on all morpheme boundaries, regardless of "
                u"whether the suppressed ligature is common or available "
                u"at all in the current font face. So for the word "
                u"“Auflage” the common fl\xA0ligature is suppressed by a "
                u"zero\xA0width non-joiner. But also in the word “Aufgabe” "
                u"a zero\xA0width non-joiner is inserted between\xA0f and\xA0g "
                u"even though an fg ligature is not common. This "
                u"approach has the disadvantage that probably many "
                u"zero\xA0width non-joiners are inserted that will have no "
                u"effect because the font face does not even contain "
                u"these ligatures. But this approach has the advantage "
                u"that is works in principle for all font faces, also "
                u"if discretionary ligatures are used or if the "
                u"font face contains exotic ligatures.",
            scribus.ICON_NONE,
            scribus.BUTTON_OK,
            scribus.BUTTON_CANCEL)
    if result == 4194304:
        return
    if result != 1024:
        raise AssertionError(
            "Expected message result be 1024 or 4194304, but it isn’t.")
    my_provider = InstructionProvider()
    # Make sure that get_word_characters() has only BMP values.
    word_characters = my_provider.get_word_characters()
    for char in word_characters:
        if not is_bmp_scalar_only(char):
            raise AssertionError("get_word_characters() should return only "
                                 "BMP values, but it doesn’t.")
    try:
        scribus.setRedraw(False)
        for my_textframe in all_my_objects:
            # For each text frame in our list, we do the ligature setting
            # for the hole story (means, also for all linked text
            # frames). Result: If the user has selected various linked
            # text frames of the same story, than we do all the same
            # work various times. That’s quite inefficient, but however
            # I don’t know how to prevent this.
            ligature_setting_for_story(my_textframe, my_provider)
            # Clean the text selection for this Scribus object
            scribus.selectText(0, 0, my_textframe)
        # Show final message (if no exceptions have been raised so far)…
        if german_locale:
            show_messagebox(
                u"Ligatursatz",
                u"Der Ligatursatz ist abgeschlossen.")
        else:
            show_messagebox(
                u"Ligature setting",
                u"Ligature setting has finished.")
    finally:
        # Make sure that redraw gets enabled again.
        scribus.setRedraw(True)
        scribus.redrawAll()

if __name__ == '__main__':
    do_ligature_setting()
