/* $NetBSD: fsu_mount.c,v 1.9 2009/07/13 16:03:54 stacktic Exp $ */

/*
 * Copyright (c) 2008,2009 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/mount.h>
#include <sys/stat.h>

#if HAVE_NBCOMPAT_H
#include <nbcompat.h>
#endif

#include <err.h>
#include <errno.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <rump/ukfs.h>

#include <fsu_utils.h>

#include "fsu_mount.h"

#include "filesystems.h"
#include "fsu_alias.h"

#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

#ifndef RUMP_LIBDIR
#define RUMP_LIBDIR "/usr/lib"
#endif

#define add_arg(a) mntd.mntd_argv[mntd.mntd_argc++] = (a)
struct mount_data_s {
	fsu_fs_t *mntd_fs;
	char mntd_canon_dev[MAXPATHLEN];
	char mntd_canon_dir[MAXPATHLEN];
	int mntd_flags;
	int mntd_argc;
	char *mntd_argv[15];
	char *mntd_cwd;
};

static struct ukfs *mount_alias(struct fsu_fsalias_s *, char *);
static struct ukfs *mount_fstype(fsu_fs_t *, char *, char *, char *);
static char *fsdevice, *fstype;
_Bool fsu_readonly = false;
_Bool fsu_blockfs = false;

/*
 * Tries to mount an image.
 * if the fstype is not given try every supported types.
 */
struct ukfs
*fsu_mount(int *argc, char **argv[])
{
	struct ukfs *ukfs;
	fsu_fs_t *fst;
	struct fsu_fsalias_s *alias;
	int idx;
	char ch, *mntopts, afsdev[MAXPATHLEN], *path;
	struct stat sb;
#ifdef WITH_SYSPUFFS
	const char options[] = "f:o:p:t:";
	char *puffsexec = NULL;
#else
	const char options[] = "f:o:t:";
#endif

	alias = NULL;
	mntopts = fstype = fsdevice = NULL;
	fst = NULL;
	ukfs = NULL;

	ukfs_init();
	ukfs_modload_dir(RUMP_LIBDIR);

	/*
	 * [-o mnt_args] [[-t] fstype] [ -p puffsexec] fsdevice
	 */
	while ((ch = getopt(*argc, *argv, options)) != -1) {
		switch (ch) {
		case 'f':
			if (fsdevice == NULL) {
				if (stat(optarg, &sb) == 0 &&
                                    realpath(optarg, afsdev) != NULL) {
					fsdevice = strdup(afsdev);
                                        fsu_blockfs = true;
                                } else
					fsdevice = optarg;
			}
			break;
		case 'o':
			if (mntopts == NULL) {
				mntopts = optarg;
				setenv("FSU_MNTOPTS", optarg, 1);
			}
			break;
#ifdef WITH_SYSPUFFS
		case 'p':
			if (puffsexec == NULL) {
				puffsexec = optarg;
				for (fst = fslist; fst->fs_name != NULL; ++fst)
					if (fst->fs_name == MOUNT_PUFFS)
						break;

				if (fstype == NULL)
					fstype = strdup(MOUNT_PUFFS);
				else if (strcmp(fstype, MOUNT_PUFFS) != 0) {
					warnx("-p is only for puffs");
					return NULL;
				}
			}
			break;
#endif
		case 't':
			if (fstype == NULL)
				fstype = optarg;
			break;
		case '?':
		default:
			errno = 0;
			return NULL;
		}
	}
	idx = optind;
#ifdef __linux__
	optind = 1;
#else
	optind = optreset = 1;
#endif
	if (mntopts == NULL)
		mntopts = getenv("FSU_MNTOPTS");

	if (fsu_readonly) {
		if (mntopts == NULL)
			mntopts = strdup("ro");
		else {
			char *tmp;

			tmp = malloc(strlen(mntopts) + 4);
			snprintf(tmp, strlen(mntopts) + 4, "%s,ro", mntopts);
			mntopts = tmp;
		}
	}

	if (fstype == NULL && fst == NULL)
		fstype = getenv("FSU_FSTYPE");

	if (fstype != NULL && fst == NULL) {
		for (fst = fslist; fst->fs_name != NULL; ++fst)
			if (strcmp(fstype, fst->fs_name) == 0)
				break;

		if (fst->fs_name == NULL) {
			fprintf(stderr, "%s: filesystem not supported\n",
				fstype);
			return NULL;
		}
	}
	if (fst == NULL && idx < *argc) {
		if (strcmp((*argv)[idx], "--") == 0) {
			errno = 0;
			return NULL;
		}

		for (fst = fslist; fst->fs_name != NULL; ++fst) {
			if (fst->fs_flags & FS_NO_AUTO)
				continue;
			if (strcmp((*argv)[idx], fst->fs_name) == 0)
				break;
		}
		if (fst->fs_name != NULL)
			idx++;
		else
			fst = NULL;
	}

       	if (fsdevice == NULL) {
		fsdevice = getenv("FSU_DEVICE");
		if (fsdevice == NULL) {
			if (idx < *argc && strcmp((*argv)[idx], "--") != 0) {
				if (stat((*argv)[idx], &sb) == 0 &&
                                    realpath((*argv)[idx], afsdev) != NULL) {
					fsdevice = strdup(afsdev);
                                        fsu_blockfs = true;
					++idx;
				} else
					fsdevice = (*argv)[idx++];
			} else {
				errno = 0;
				return NULL;
			}
		}

		build_alias_list();
		alias = get_alias(fsdevice);
		if (alias != NULL) {
			ukfs = mount_alias(alias, mntopts);
			fstype = alias->fsa_type;
			fsdevice = alias->fsa_path;
			goto out;
		}
		free_alias_list();
	}
#ifdef WITH_SYSPUFFS
	ukfs = mount_fstype(fst, fsdevice, mntopts, puffsexec);
#else
	ukfs = mount_fstype(fst, fsdevice, mntopts, NULL);
#endif
out:
	if ((*argv)[idx] != NULL && strcmp((*argv)[idx], "--") == 0)
		++idx;

	if (--idx > 0) {
		(*argv)[idx] = (*argv)[0];
		*argv += idx;
		*argc -= idx;
#ifdef __linux__
		optind = 1;
#else
		optind = optreset = 1;
#endif
	}

	path = getenv("FSU_CWD");
	if (path != NULL) {
		if (ukfs_chdir(ukfs, path) != 0) {
			fprintf(stderr, "could not chdir to %s\n", path);
			ukfs_release(ukfs, 0);
			return NULL;
		}
	}
#ifdef __linux__
	optind = 1;
#else
	optind = optreset = 1;
#endif
	if (alias != NULL)
		free_alias_list();
	return ukfs;
}

static struct ukfs
*mount_fstype(fsu_fs_t *fs, char *fsdev, char *mntopts, char *puffsexec)
{
	struct mount_data_s mntd;
	struct ukfs *ukfs;
	int rv;

	mntd.mntd_fs = fs;
	mntd.mntd_argc = mntd.mntd_flags = 0;
	ukfs = NULL;

	/* setting up the argv array */
	mntd.mntd_argv[mntd.mntd_argc++] = strdup(getprogname());

#ifdef WITH_SYSPUFFS
	if (puffsexec != NULL && fs->fs_name == MOUNT_PUFFS)
		mntd.mntd_argv[mntd.mntd_argc++] = puffsexec;
#endif

	if (mntopts != NULL) {
		mntd.mntd_argv[mntd.mntd_argc++] = strdup("-o");
		mntd.mntd_argv[mntd.mntd_argc++] = mntopts;
	}

	mntd.mntd_argv[mntd.mntd_argc++] = fsdev;
	mntd.mntd_argv[mntd.mntd_argc++] = strdup("/");
	mntd.mntd_argv[mntd.mntd_argc] = NULL;

	/* filesystem given */
	if (fs != NULL) {
#ifdef __linux__
		optind = 1;
#else
		optind = optreset = 1;
#endif
		rv = fs->fs_parseargs(mntd.mntd_argc, mntd.mntd_argv,
				      fs->fs_args, &mntd.mntd_flags,
				      mntd.mntd_canon_dev, mntd.mntd_canon_dir);
#ifdef __linux__
		optind = 1;
#else
		optind = optreset = 1;
#endif
		if (rv != 0)
			return NULL;
#ifdef HAVE_UKFS_DISKLABEL
                if (fsu_blockfs) {
                        int part = UKFS_PARTITION_NONE;
                        ukfs_partition_probe(mntd.mntd_canon_dev, &part);
                        ukfs = ukfs_mount_disk(fs->fs_name,
                                mntd.mntd_canon_dev, part, mntd.mntd_canon_dir,
                                mntd.mntd_flags, fs->fs_args,
                                fs->fs_args_size);
                } else
#endif
                        ukfs = ukfs_mount(fs->fs_name, mntd.mntd_canon_dev,
				  mntd.mntd_canon_dir, mntd.mntd_flags,
				  fs->fs_args, fs->fs_args_size);
		if (ukfs == NULL)
			fprintf(stderr, "%s is not a valid %s image\n",
				fsdev, fs->fs_name);
#ifdef WITH_SMBFS
		if (strcmp(fs->fs_name, MOUNT_SMBFS) == 0) {
			extern struct smb_ctx sctx;
			smb_ctx_done(&sctx);
		}
#endif
		goto out;
	}

	/* filesystem not given (auto detection) */
	for (fs = fslist; fs->fs_name != NULL; ++fs) {
		if (fs->fs_flags & FS_NO_AUTO)
			continue;
		mntd.mntd_flags = 0;
#ifdef __linux__
		optind = 1;
#else
		optind = optreset = 1;
#endif
		rv = fs->fs_parseargs(mntd.mntd_argc, mntd.mntd_argv,
				      fs->fs_args, &mntd.mntd_flags,
				      mntd.mntd_canon_dev, mntd.mntd_canon_dir);
#ifdef __linux__
		optind = 1;
#else
		optind = optreset = 1;
#endif
		if (rv != 0)
			continue;

#ifdef HAVE_UKFS_DISKLABEL
                if (fsu_blockfs) {
                        int part = UKFS_PARTITION_NONE;
                        ukfs_partition_probe(mntd.mntd_canon_dev, &part);
                        ukfs = ukfs_mount_disk(fs->fs_name,
                            mntd.mntd_canon_dev, part, mntd.mntd_canon_dir,
                            mntd.mntd_flags, fs->fs_args,
                            fs->fs_args_size);
                } else
#endif
                        ukfs = ukfs_mount(fs->fs_name, mntd.mntd_canon_dev,
                            mntd.mntd_canon_dir, mntd.mntd_flags,
                            fs->fs_args, fs->fs_args_size);
		if (ukfs != NULL)
			break;
	}

out:
	if (fs->fs_name != NULL)
		fstype = strdup(fs->fs_name);

	return ukfs;
}

static struct ukfs
*mount_alias(struct fsu_fsalias_s *al, char *mntopts)
{
	struct mount_data_s mntd;
	struct ukfs *ukfs;
	fsu_fs_t *cur;
	int rv;

	ukfs = NULL;
	mntd.mntd_argc = mntd.mntd_flags = 0;

	mntd.mntd_argv[mntd.mntd_argc++] = strdup(getprogname());

#ifdef WITH_SYSPUFFS
	if (al->fsa_puffsexec != NULL)
		mntd.mntd_argv[mntd.mntd_argc++] = strdup(al->fsa_puffsexec);
#endif

	if (al->fsa_mntopt != NULL) {
		mntd.mntd_argv[mntd.mntd_argc++] = strdup("-o");
		mntd.mntd_argv[mntd.mntd_argc++] = al->fsa_mntopt;
		setenv("FSU_MNTOPTS", al->fsa_mntopt, 1);
	}

	if (mntopts != NULL) {
		mntd.mntd_argv[mntd.mntd_argc++] = strdup("-o");
		mntd.mntd_argv[mntd.mntd_argc++] = mntopts;
		setenv("FSU_MNTOPTS", mntopts, 1);
	}

	mntd.mntd_argv[mntd.mntd_argc++] = al->fsa_path;
	mntd.mntd_argv[mntd.mntd_argc++] = strdup("/");
	mntd.mntd_argv[mntd.mntd_argc] = NULL;

	for (cur = fslist; cur->fs_name != NULL; ++cur)
		if (strcmp(cur->fs_name, al->fsa_type) == 0)
			break;

	if (cur->fs_name != NULL) {
		mntd.mntd_fs = cur;
		rv = cur->fs_parseargs(mntd.mntd_argc, mntd.mntd_argv,
				       cur->fs_args, &mntd.mntd_flags,
				       mntd.mntd_canon_dev,
				       mntd.mntd_canon_dir);
		if (rv == 0) {
#ifdef HAVE_UKFS_DISKLABEL
                        if (fsu_blockfs) {
                                int part = UKFS_PARTITION_NONE;
                                ukfs_partition_probe(mntd.mntd_canon_dev,
                                    &part);
                                ukfs = ukfs_mount_disk(cur->fs_name,
                                    mntd.mntd_canon_dev, part,
                                    mntd.mntd_canon_dir,
                                    mntd.mntd_flags, cur->fs_args,
                                    cur->fs_args_size);
                        } else
#endif
                                ukfs = ukfs_mount(cur->fs_name,
                                    mntd.mntd_canon_dev, mntd.mntd_canon_dir,
                                    mntd.mntd_flags, cur->fs_args,
                                    cur->fs_args_size);
                }
	}

	if (cur->fs_name != NULL)
		fstype = strdup(cur->fs_name);
	return ukfs;

}

void
fsu_unmount(struct ukfs *ukfs)
{

	if (ukfs == NULL)
		return;
	ukfs_release(ukfs, 0);
}

const char
*fsu_get_device(void)
{

	return fsdevice;
}

const char
*fsu_get_fstype(void)
{

	return fstype;
}

const char
*fsu_mount_usage(void)
{

#ifdef WITH_SYSPUFFS
	return "[-o mnt_args] [[-t] fstype] [-p puffs_exec] [-f] fsdevice";
#else
	return "[-o mnt_args] [[-t] fstype] [-f] fsdevice";
#endif
}
