/*-
 * Copyright (c) 2014 Jared D. McNeill <jmcneill@invisible.ca>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/drvctlio.h>

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <dev/biovar.h>

enum {
	STATE_OK = 0,
	STATE_WARNING,
	STATE_CRITICAL,
	STATE_UNKNOWN
};

static int drvctl_fd;
static int bioctl_fd;
static int status = STATE_OK;
static char output[4096];

static void
panic(const char *msg)
{
	printf("%s\n", msg);
	exit(STATE_UNKNOWN);
}

static int
drvctl_walk(const char *devname, void (*fn)(const char *))
{
	struct devlistargs laa;
	size_t children;
	int i;

	snprintf(laa.l_devname, sizeof(laa.l_devname), "%s", devname);
	laa.l_childname = NULL;
	laa.l_children = 0;

	if (ioctl(drvctl_fd, DRVLISTDEV, &laa) == -1)
		panic("DRVLISTDEV failed");

retry:
	children = laa.l_children;
	laa.l_childname = calloc(children, sizeof(laa.l_childname[0]));
	if (laa.l_childname == NULL)
		panic("calloc failed");
	if (ioctl(drvctl_fd, DRVLISTDEV, &laa) == -1)
		panic("DRVLISTDEV failed");
	if (laa.l_children != children) {
		free(laa.l_childname);
		goto retry;
	}

	for (i = 0; i < laa.l_children; i++) {
		fn(laa.l_childname[i]);
		drvctl_walk(laa.l_childname[i], fn);
	}

	free(laa.l_childname);
}

static const char *
bio_status2str(int status)
{
	const char *s;

	switch (status) {
	case BIOC_SVONLINE:
		s = BIOC_SVONLINE_S;
		break;
	case BIOC_SVOFFLINE:
		s = BIOC_SVOFFLINE_S;
		break;
	case BIOC_SVDEGRADED:
		s = BIOC_SVDEGRADED_S;
		break;
	case BIOC_SVBUILDING:
		s = BIOC_SVBUILDING_S;
		break;
	case BIOC_SVREBUILD:
		s = BIOC_SVREBUILD_S;
		break;
	case BIOC_SVMIGRATING:
		s = BIOC_SVMIGRATING_S;
		break;
	case BIOC_SVSCRUB:
		s = BIOC_SVSCRUB_S;
		break;
	case BIOC_SVCHECKING:
		s = BIOC_SVCHECKING_S;
		break;
	case BIOC_SVINVALID:
	default:
		s = BIOC_SVINVALID_S;
		break;
	}

	return s;
}

static void
bio_probe(const char *devname)
{
	struct bio_locate bl;
	struct bioc_inq bi;
	struct bioc_vol bv;
	int v;

	memset(&bl, 0, sizeof(bl));
	bl.bl_name = strdup(devname);
	if (ioctl(bioctl_fd, BIOCLOCATE, &bl) == -1) {
		free(bl.bl_name);
		return;
	}
	free(bl.bl_name);

	memset(&bi, 0, sizeof(bi));
	bi.bi_cookie = bl.bl_cookie;
	if (ioctl(bioctl_fd, BIOCINQ, &bi) == -1)
		panic("BIOCINQ failed");

	for (v = 0; v < bi.bi_novol; v++) {
		memset(&bv, 0, sizeof(bv));
		bv.bv_cookie = bl.bl_cookie;
		bv.bv_volid = v;

		if (ioctl(bioctl_fd, BIOCVOL, &bv) == -1)
			panic("BIOCVOL failed");

		snprintf(output, sizeof(output), "%s %s", output, bv.bv_dev);

		switch (bv.bv_status) {
		case BIOC_SVINVALID:
		case BIOC_SVOFFLINE:
			printf("FAILED %s ", bv.bv_dev);
			status = STATE_CRITICAL;
			break;
		case BIOC_SVONLINE:
			break;
		default:
			printf("WARNING %s %s ",
			    bio_status2str(bv.bv_status), bv.bv_dev);
			if (status < STATE_WARNING)
				status = STATE_WARNING;
			break;
		}
	}
}

int
main(int argc, char *argv[])
{
	drvctl_fd = open(DRVCTLDEV, O_RDONLY);
	if (drvctl_fd == -1)
		panic("failed to open " DRVCTLDEV);
	bioctl_fd = open("/dev/bio", O_RDONLY);
	if (bioctl_fd == -1)
		panic("failed to open /dev/bio");

	drvctl_walk("", bio_probe);

	if (status == STATE_OK) {
		printf("OK %s\n", output);
	} else {
		printf("\n");
	}

	close(bioctl_fd);
	close(drvctl_fd);

	return status;
}
