#! /bin/sh

# $NetBSD: t_netpgpverify.sh,v 1.1.12.1 2014/11/25 07:55:25 snj Exp $

# Copyright (c) 2012 Alistair Crooks <agc@NetBSD.org>
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

atf_test_case netpgpverify_case

netpgpverify_rsa_head() {
	atf_set "descr" "RSA signatures for netpgpverify"
}

netpgpverify_rsa_body() {
	uudecode << EOF
begin-base64 644 b.gpg
xA0DAAgBG2jc/MBZaCMBy4tiAAAAAABQUk9HPXAKU1JDUz1wYXJzZS5jCldBUk5T
PTUKTUtNQU49bm8KQ1BQRkxBR1MrPS1nIC1PMApMREZMQUdTKz0tZyAtTzAKCi5p
bmNsdWRlIDxic2QucHJvZy5taz4KCnQ6ICR7UFJPR30KCS4vJHtQUk9HfSBncGdz
aWduZWQtYS5ncGcKwsBiBAABCAAWBQJQTZOaBQMAAAAACRAbaNz8wFloIwAABTcH
/RhxktWLH3Cw6YkC/4Wcm9yq6flgiKQsB0VUaVgmmOfWnLxcowyH5ISENo/J6qhT
S8Z0B2lAkRzBhXbfSe8GoESc/NsXTuOIvdjZEy7FBkl9Lumuqp1IlmoXh3RgPv5Z
gcJ+uDCV958uXAxLn017xN/CVKi5yBiNWWk/mOJuI8FCiQCyFVQ3dOXcFU/gf4iZ
YkH6OMX6eNpLQXp+s1ar6ZqpFE09QEoiFuIiDas+V6g0IG1c+PgP+TOlO5ztaKjx
XxejP0Thtft0T+AKTANVrtx+aTseVt4CR3jBt0n4CJjTTRQwnIYxGML3ddgMXSPT
0c4J/4dwyMqdDuhby//52Nw=
====
EOF
	cat >det << EOF
To Do
=====
tests with -k sig
detached sigs
DSA

Done
====
basics
localise pgp_read_packets
fix lint
WARNS=5
lib man page
prog man page
do we do it statically linked as well?
multiple files in netpgpverify
EOF
	uudecode << EOF
begin-base64 644 det.sig
iQEcBAABAgAGBQJQf8gBAAoJEBto3PzAWWgjbLoH/i9MZSv2l9lJ5O+bfGmRSUxe
rGeAM0dq//xi+AABByDEWNYORq2I6CO6k2eVPv7YbNE2Whn7qXA6U7LotM7Lt61l
MCLRhuTv9RViV1qzsy+Z8VAQmhERAcG/qA5JSQbzjuVYhg5FexpiVmxgG9L8Z/Oj
4pU9zIId1vH7CvAcy/mfSyM8Ih8bmsrgs9b1wzMPLQKl0v0jtcrEcnwMtscXa9Mw
X66bHxCHtRyGcWsHU8Ab2fJCvsCbBBMtg3JjZ7+VgDoZqPqBgIkhmppqC05vU5QR
leNo8g9OwiJ6fMtdwTMsFkxqjjVKk5//X/6jQd/mf55XcxLOw4UK4ev5ADExnOU=
====
EOF
	cat >jj.asc << EOF
-----BEGIN PGP SIGNED MESSAGE-----
Hash: SHA1

1. tag & 0x3f
2. len

one pass (tag 4)
========
b version:3
b sig type
b hash alg
b pubkey alg
8b keyid

literal data (tag 11)
=============
b binary/text
b length
c string
L mtime
text
-----BEGIN PGP SIGNATURE-----
Version: GnuPG v1.4.11 (NetBSD)

iQEcBAEBAgAGBQJQaIZcAAoJEBto3PzAWWgj678IALbDHon3Rm6qUhn7k1TFT6D3
yi/jzf3piSJGsgUg2wEghs175edC/cJK3lG9Gx/3/uQq06R9g37nVRX8I0sK7yT2
XgR+RHoGh/b+CQxdRNC+ub5QoNb8LcmCb/MQGq2KK9otSExiy4WMUP4K1DblaK5L
+Hg4VTooMot1NVqyFSoB2aZauXc2F4ZVh5q0fn8w5GEw45P+AUUbmzpgbLwXbl+I
tMsX54V1dxyDcCYUs0xUH/VxJUQEeIlDbCOXYMbCVtggYRqKksTr+u/riw/Llnql
jQdq5rBRW1SlD7Ll6z/LF2WBJOWtHzp4qbnBGSq5uB1q37H3mWL28f1tL//TUjM=
=EX8W
-----END PGP SIGNATURE-----
EOF
	uudecode << EOF
begin-base64 644 pubring.gpg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====
EOF
	cat >NetBSD-6.0_hashes.asc << EOF
-----BEGIN PGP SIGNED MESSAGE-----
Hash: SHA1



The following is a complete list of distribution files for NetBSD-6.0

This includes all binary distributions, kernels, install images,
source tarballs, release notes and related files.  Multiple hash
formats are used for each file.

Signatures will be made available for any additional install media as
they are published.

UPDATE: the macppc ISO was replaced, the mac68k and source ISOs were added,
and a new signature was generated for this hashes file on 15 October 2012.

						NetBSD Security-Officer
						security-officer@NetBSD.org

SHA1 (NetBSD-6.0/CHANGES) = 44ed658e4f8c813d67613497afdc6656470aab00
SHA512 (NetBSD-6.0/CHANGES) = ababa97f9674bfaa4d941515cff2b98a1c254221394e10f3060f456474f6888f5e79aa6480bebb6f69dc1e682cd8f4c450feee91c0da1a2ed1fc805bd1dec273
RMD160 (NetBSD-6.0/CHANGES) = 2a0ed7457dfe13229bb72cfde66668ffff4ec7ff
MD5 (NetBSD-6.0/CHANGES) = 63363144027cac2345030caf2d934d05
SHA1 (NetBSD-6.0/CHANGES-6.0) = 159eef1002f7e904954fb3bd9490ac2e5391fdbb
SHA512 (NetBSD-6.0/CHANGES-6.0) = f09bf320629f6db65a469c4220eebd8db668a81fcc1a8d41b9fb06bbadbcdafd76c9d6372d8d9e5258af8424abf648d1114da2eef3ae9b396642b9b3aba434ec
RMD160 (NetBSD-6.0/CHANGES-6.0) = b03df64c3b4081c56578066c23a24db14ae935e1
MD5 (NetBSD-6.0/CHANGES-6.0) = 92a82839d67702799f45da9167b0a60e
SHA1 (NetBSD-6.0/CHANGES.prev) = 25bdfe4a49acc75ee3a59ca04dc3b3c7c71a4563
SHA512 (NetBSD-6.0/CHANGES.prev) = 73057475c27c98d729dc97f6203f89f39459405c74f7abd7b18d4382f7c5ad6c3f598989cecdd5013ba4b77d66e694c05e3fcd51fd9776fde908513ea70e63d5
RMD160 (NetBSD-6.0/CHANGES.prev) = 38f959a7c9d0c8da19eeb1740126b04dbb71a90e
MD5 (NetBSD-6.0/CHANGES.prev) = 0a50635a033237b439f5d7ef4c29b802
SHA1 (NetBSD-6.0/LAST_MINUTE) = 49f8a218ff6657e0d30449a87b7c4f65ccf6115b
SHA512 (NetBSD-6.0/LAST_MINUTE) = d6b47cdd24a6be60d68f867c361106da69dea5f9b7f4f9764d1bd0f706e8fc9d604a6b89b95eb2829dba78df936f824be3dec21318da495705796a1882ee9277
RMD160 (NetBSD-6.0/LAST_MINUTE) = ba07328d66f7955cc9fcb402e461871bfcf2aad3
MD5 (NetBSD-6.0/LAST_MINUTE) = fb247c08580991ef0cc5f126b62f3b6e
SHA1 (NetBSD-6.0/README.files) = 183b12bef627768c595ab7df30bc038952166809
SHA512 (NetBSD-6.0/README.files) = 9aacd7e1042c7579309ed9ec1ee0fb017803f967be04e51f08e78fd56537ceae6ee79ed0bd3250994e4c16dd7de6995c1be3a301aeb95a500bbc77f6915ecb31
RMD160 (NetBSD-6.0/README.files) = 61301ea3ffe6a3e1c415ce00cfe6d22b2d544365
MD5 (NetBSD-6.0/README.files) = 7e47e7eb92db7cb8b8ce39ceb5986847
SHA1 (NetBSD-6.0/acorn26/binary/kernel/netbsd-FOURMEG.gz) = e127c41d5e332c75e79f5ca0e72ba250ef3c739a
SHA512 (NetBSD-6.0/acorn26/binary/kernel/netbsd-FOURMEG.gz) = 7487273a4289d2f1dd69fb0230c263ac4fd198aac5399de043e1416834ea4b0bb4970129886223b9c5090ea9526dd765e6152f89ccfa58d4c8ba95597a845a34
RMD160 (NetBSD-6.0/acorn26/binary/kernel/netbsd-FOURMEG.gz) = 09dcd3b8f9be85b99ed094fefc6a88e27be9b91d
MD5 (NetBSD-6.0/acorn26/binary/kernel/netbsd-FOURMEG.gz) = 5aef629404b3d65f15bc44554b948145
SHA1 (NetBSD-6.0/acorn26/binary/kernel/netbsd-GENERIC.gz) = c6362b83dbc496509c4e539cca33924b14318804
SHA512 (NetBSD-6.0/acorn26/binary/kernel/netbsd-GENERIC.gz) = da75edae5fcd456028c5699d3872ac6e9c041aafb37141cfbf6b5f374f5c69d21979d23709f7e82a9198580e6f5c3c50060b9d1cd3dcac74c1e2530131c5e892
RMD160 (NetBSD-6.0/acorn26/binary/kernel/netbsd-GENERIC.gz) = 6caa2f5e65c871664b1af2dd2a5eceb8b64f2212
MD5 (NetBSD-6.0/acorn26/binary/kernel/netbsd-GENERIC.gz) = b14bd857ffbe97a3601f91bc0a1b388d
SHA1 (NetBSD-6.0/acorn26/binary/sets/games.tgz) = e1d2bb091d8202962133475bc83edb55230b9008
SHA512 (NetBSD-6.0/acorn26/binary/sets/games.tgz) = a147fe6d0419d53d5a428802d0bb6c9fa8a84a76c96d7d6d7c0d399e5af3a69240116463549a83d53387b0214ba0207796d7c10228d92a684fe7ff0ece3aec16
RMD160 (NetBSD-6.0/acorn26/binary/sets/games.tgz) = 2be515c495548ba5135ff42a75a8ebbfc35ea776
MD5 (NetBSD-6.0/acorn26/binary/sets/games.tgz) = 9059a6f2cd41cb58827709c9e9a1649b
SHA1 (NetBSD-6.0/acorn26/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/acorn26/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/acorn26/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/acorn26/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/acorn26/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/acorn26/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/acorn26/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/acorn26/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/acorn26/binary/sets/xbase.tgz) = 875fd641f969ec00f854ca1334d084947450c740
SHA512 (NetBSD-6.0/acorn26/binary/sets/xbase.tgz) = 75d0c9e9685b6b7bb706df7427c8b52f59c4fc4d0e725b02ee18024f75a546dc41de5f1a5d8552a8d2c84bdf87856e5857b0f12914801ca7b36b69536d617b4e
RMD160 (NetBSD-6.0/acorn26/binary/sets/xbase.tgz) = 1a4d099d59b2bd882e99f1b2df1c7bf22b890d48
MD5 (NetBSD-6.0/acorn26/binary/sets/xbase.tgz) = 5eb9c573c5a0c96701f268a3470a58b9
SHA1 (NetBSD-6.0/acorn26/binary/sets/xcomp.tgz) = 8d7469a85b7433c89690cce74e6b3f4e66cf0647
SHA512 (NetBSD-6.0/acorn26/binary/sets/xcomp.tgz) = 411304cf66e300a7f162128203099bb4d6eebc4a93ba18ed7caf79f30b31cc7ffcede85c4e1bad02c3f5f8a27d009fb71f3ce5862c1d8aa477f39a25c535fdaa
RMD160 (NetBSD-6.0/acorn26/binary/sets/xcomp.tgz) = 822d26e0ee08b72ca8b991f0709a8c1bf0399208
MD5 (NetBSD-6.0/acorn26/binary/sets/xcomp.tgz) = 5af361f0334d28119d703064278b3ed3
SHA1 (NetBSD-6.0/acorn26/binary/sets/xetc.tgz) = 222d14c205f1018cc5a86eb03214d18cda2abb82
SHA512 (NetBSD-6.0/acorn26/binary/sets/xetc.tgz) = fb82261996aa772fd8a9022eadebf899ee04ee70229ee4fe07758ca14937bdb34752c8218eb4c5b722b6208f719891493bbc141372fcfa503de83e0a61aa7387
RMD160 (NetBSD-6.0/acorn26/binary/sets/xetc.tgz) = 511ebfe5076f361b7f7ca03a265d16052a16d259
MD5 (NetBSD-6.0/acorn26/binary/sets/xetc.tgz) = 165cffbb6d2c79663198d80c82afac1e
SHA1 (NetBSD-6.0/acorn26/binary/sets/xfont.tgz) = ce0e141beeffeb5b470478e9887e4127da85de3c
SHA512 (NetBSD-6.0/acorn26/binary/sets/xfont.tgz) = 5d98354ff77103793d56af3192ab5f9afcde62208791976064ac33a149f994e0227dd5dd28ed544eeac5ea504e5212d5fb9e7d05e59e5a48a2fe4062e34e3106
RMD160 (NetBSD-6.0/acorn26/binary/sets/xfont.tgz) = 4d09f5e82fec875ef0ea328809465e7a98f2ebfc
MD5 (NetBSD-6.0/acorn26/binary/sets/xfont.tgz) = be26b3890261302420b09976a1d38144
SHA1 (NetBSD-6.0/acorn26/binary/sets/xserver.tgz) = 4ccfef1d49179603286042088bec0218c6b2031d
SHA512 (NetBSD-6.0/acorn26/binary/sets/xserver.tgz) = 809fc0a9065c1b9e7cd17aeb0d41cfd3158efd6c585556af01c4a3c93f0df3646a2ff77c6994449a703a408c0988fc1aa68cf34c7e297a3056d73ded57034c12
RMD160 (NetBSD-6.0/acorn26/binary/sets/xserver.tgz) = 403604404261764c74f10fab774c4653fd20cc9b
MD5 (NetBSD-6.0/acorn26/binary/sets/xserver.tgz) = 914f909b6878cef16a92765a8cbb52ca
SHA1 (NetBSD-6.0/acorn26/binary/sets/kern-FOURMEG.tgz) = 3e74d7ca4f0ad181d8f37a28fc179fe4fdc3521f
SHA512 (NetBSD-6.0/acorn26/binary/sets/kern-FOURMEG.tgz) = 794689c5f293ed453fe424f3e1cd45bdf6127a08da0e6ded9b6f3c8afe0af6f65b1ca1aca96db5b8a4891102f38b2895aca82bba0dce8ee1389f2eb439ce11b7
RMD160 (NetBSD-6.0/acorn26/binary/sets/kern-FOURMEG.tgz) = 66e7db127f67cf318789d74d7bdf5a57fdb922ed
MD5 (NetBSD-6.0/acorn26/binary/sets/kern-FOURMEG.tgz) = f26ed42c971ee7ce64682f5fa5a5c2b5
SHA1 (NetBSD-6.0/acorn26/binary/sets/base.tgz) = 48cf8b5b14a8a61a85bf8bfa1ae739ef599a17e6
SHA512 (NetBSD-6.0/acorn26/binary/sets/base.tgz) = f68d2d3e80b88fad8ff98a9bfdac7cc039d51e766130d3cade364a83bede6c6984c0cbcc56588f81760e6754ebbb08ad444fe4dd006e8ef91a3bc4e66c777e4a
RMD160 (NetBSD-6.0/acorn26/binary/sets/base.tgz) = 090a8ef5abfede25635b86e1ed989bd8c7b988e5
MD5 (NetBSD-6.0/acorn26/binary/sets/base.tgz) = 6ea39b172012faf2bdbcb861d0de443c
SHA1 (NetBSD-6.0/acorn26/binary/sets/comp.tgz) = 8d6222fb0ba80750737719e3c9eb4d42aef1c92e
SHA512 (NetBSD-6.0/acorn26/binary/sets/comp.tgz) = eeacb62b5075abfd130a4ee4bcc8b3ada6fe2e3b4ad95cfa65fb45e61956092368c9e7dd90b7fbfefd7d96162890d23872d3bc4ad92488d92a50738d59629f9c
RMD160 (NetBSD-6.0/acorn26/binary/sets/comp.tgz) = adc77379342aa005884eb58a65a8d57d07e63853
MD5 (NetBSD-6.0/acorn26/binary/sets/comp.tgz) = a660a76cb874c828690f5a128ff61501
SHA1 (NetBSD-6.0/acorn26/binary/sets/etc.tgz) = 183e6af0c9ddf7c62218f3616129ed1223e0fd0b
SHA512 (NetBSD-6.0/acorn26/binary/sets/etc.tgz) = 0c9967126ddf9cdfdf568b6a7183bd9a17a119798f318f511ec9b9351770ffb8a394d57ccf09ce1e38b4e56d836f4ab61c943e81d932007e8726819733b0b108
RMD160 (NetBSD-6.0/acorn26/binary/sets/etc.tgz) = 92fab0f295c28c53f693444918b12bc026596a3f
MD5 (NetBSD-6.0/acorn26/binary/sets/etc.tgz) = 1c4124013d68bbf2b85d3025aab1a0a9
SHA1 (NetBSD-6.0/acorn26/binary/sets/kern-GENERIC.tgz) = 86de3875a3df7b400d86d2f97ea02067dac49288
SHA512 (NetBSD-6.0/acorn26/binary/sets/kern-GENERIC.tgz) = c3eb9e9e696c7f166c9da1acdbf14b314863f68a5af5786c9d79c5e2ffd9c70ad366a6cec7ff0858ffae2ab8e028c0224b091de5bfcfad6a4ae5327709c7981e
RMD160 (NetBSD-6.0/acorn26/binary/sets/kern-GENERIC.tgz) = f2b59e92e18402c2e96a4b254fa18f21b339629c
MD5 (NetBSD-6.0/acorn26/binary/sets/kern-GENERIC.tgz) = 107061f8599c58b8a982c7f6b96e0fd3
SHA1 (NetBSD-6.0/acorn26/binary/sets/modules.tgz) = 2e72e54d98a0bae829f2fdd53a9a927a4269fe83
SHA512 (NetBSD-6.0/acorn26/binary/sets/modules.tgz) = 560d27968ab9740b7fec62e5afefddc5d6db9ebb73b79c38c34ab4d609fc4cfebe8bef7e05eb5fdacc17502236ae95675e2db87e5a8c258a3633f93635fe5aae
RMD160 (NetBSD-6.0/acorn26/binary/sets/modules.tgz) = eee9e83fc9e9936b93f382beb4f619c0d7010cd8
MD5 (NetBSD-6.0/acorn26/binary/sets/modules.tgz) = 3e0985108f65114c721c3672279d28cb
SHA1 (NetBSD-6.0/acorn26/binary/sets/tests.tgz) = 72b14d964822d8a2e03c4d3855c33b2d5a02bcf5
SHA512 (NetBSD-6.0/acorn26/binary/sets/tests.tgz) = 1b87e9174bb22df4c6b894fb731c53df03af87e6dbb8b2488fec832721c3ef3508c7dac35a2f51f144c6e726a8fff61d6f66d6a5c354c11ca2f2c8fe31a7d7a8
RMD160 (NetBSD-6.0/acorn26/binary/sets/tests.tgz) = aa26cbefa7b2891c0d2e51a5f59c123c223721ff
MD5 (NetBSD-6.0/acorn26/binary/sets/tests.tgz) = a69e9d96707beccb4149aed72d701bfa
SHA1 (NetBSD-6.0/acorn26/binary/sets/text.tgz) = b4974a12425354e1d6dbe8e1e7def8ca2c8bea5a
SHA512 (NetBSD-6.0/acorn26/binary/sets/text.tgz) = 360652de2f694807759927e57beff0e60fae7855d4ab55a574e8c2d9cab25bf76bf1b880dd419c4bef6339dc72a16959f563e964644d64a81029354c87fb1b08
RMD160 (NetBSD-6.0/acorn26/binary/sets/text.tgz) = e40abdac843b3ac9055fdacff0c74d259609886d
MD5 (NetBSD-6.0/acorn26/binary/sets/text.tgz) = e1ad13174b20a114fd74863b8a5a0399
SHA1 (NetBSD-6.0/acorn26/installation/misc/netbsd-INSTALL.gz) = e7fda95700ee86c95cb2135c841840b815319eb5
SHA512 (NetBSD-6.0/acorn26/installation/misc/netbsd-INSTALL.gz) = 79c810a6e1f7221aacd1e7b7edca4064ec6c3b7b99bbbe7282bdec0354c9387f9170019c7e6df5ec049497d8b0a69c07aaf485dfd286069437c16b7a44dd04ab
RMD160 (NetBSD-6.0/acorn26/installation/misc/netbsd-INSTALL.gz) = 68c551c4746c765068cc39e81eff40d13065ac19
MD5 (NetBSD-6.0/acorn26/installation/misc/netbsd-INSTALL.gz) = 2f3714cfea4b8ace15469ca877ddcaf2
SHA1 (NetBSD-6.0/acorn26/installation/misc/boot26,ffa) = 983591033b4f5301e5c3f9696b3e9037aaa11047
SHA512 (NetBSD-6.0/acorn26/installation/misc/boot26,ffa) = 16ef2baf8d00639c68dafbf42a563815ca6a0cc51c1083820c47d10c06422abfa8257d25be675024bf674d70def6747cb70a45d2a8451551e0b7ac6915057bfe
RMD160 (NetBSD-6.0/acorn26/installation/misc/boot26,ffa) = a517e705c94bf87b617648aee6542e8194ad29b4
MD5 (NetBSD-6.0/acorn26/installation/misc/boot26,ffa) = ed0fd36ea9ed9893250d9f588d5c358c
SHA1 (NetBSD-6.0/acorn26/installation/misc/netbsd-INSTALL.symbols.gz) = 3eca116cb41df17471b9e24e453c3164bc2f6fee
SHA512 (NetBSD-6.0/acorn26/installation/misc/netbsd-INSTALL.symbols.gz) = e2fce3e17958cd1865c584fa3a97a061f8fef09a8793f61018144e753d4667026349747ac7d9dfa6ad70afe2c4dbfeaef1f5718430947958f34785858e2424e3
RMD160 (NetBSD-6.0/acorn26/installation/misc/netbsd-INSTALL.symbols.gz) = 9f25df06373241ff5dfaafa250311a4bc5020197
MD5 (NetBSD-6.0/acorn26/installation/misc/netbsd-INSTALL.symbols.gz) = 4daa298789745793c9f9f1a03c4ecdb5
SHA1 (NetBSD-6.0/acorn26/INSTALL.html) = e1896519f52cf8b3574c2ddad19d135e837ff05b
SHA512 (NetBSD-6.0/acorn26/INSTALL.html) = 9951385e876eed0126a9706c5c075453fd27ef0eb8fc42fa1bdca0e100a04d5f2d269145040eaffcfb683ef63f51a574254ce02cd3d0d3449ef867205e55c49a
RMD160 (NetBSD-6.0/acorn26/INSTALL.html) = 4e81e718d1526b61a1ebfc3dde631cc1c4f1998d
MD5 (NetBSD-6.0/acorn26/INSTALL.html) = e77700615f6d71545079d4f98be4fda3
SHA1 (NetBSD-6.0/acorn26/INSTALL.more) = 31ce17bd0e1a3eb8956f06af4a2635a8cea3693e
SHA512 (NetBSD-6.0/acorn26/INSTALL.more) = c9693333dbbde859f3f8e1f5cb571e43eeac2a1727ce7e4681b4dacda1e145a54bacbb16346bed669a0f83e3592e7b824b66d0e0a2a0f39ab41891a0bee78ca5
RMD160 (NetBSD-6.0/acorn26/INSTALL.more) = 12606307f5f4314a84d4401c5297bcf701fc1dcb
MD5 (NetBSD-6.0/acorn26/INSTALL.more) = 2104dbaa19182d2de7416aa62b7de537
SHA1 (NetBSD-6.0/acorn26/INSTALL.ps) = 61b3e6838c01429478fd78f84fc380a320f1b2ba
SHA512 (NetBSD-6.0/acorn26/INSTALL.ps) = d96d21356a59e59ab344d4e4b720e9425b3e7a3f926e8e7d453f33c907eff12c9dfdb8a97133d77bafc30a7fdf5fb63ab529a73facbad6ac8f9a36c63a236db5
RMD160 (NetBSD-6.0/acorn26/INSTALL.ps) = 50c971d098fc714fb2efe2f3e2cf49351a0f8fc5
MD5 (NetBSD-6.0/acorn26/INSTALL.ps) = f0d32affe70e695f4bcbe2df304f6a65
SHA1 (NetBSD-6.0/acorn26/INSTALL.txt) = 00ac475ec70626dc1041211ecd0e5c7e213af948
SHA512 (NetBSD-6.0/acorn26/INSTALL.txt) = 816c536f0caacbf292b9b54e812dc474299ee6a1084682321039bb1d3fd725523d76666fc14c772951ca45409550a25b5b74cefa0bf3f7539a17995a56df9756
RMD160 (NetBSD-6.0/acorn26/INSTALL.txt) = c4618e23cd501a7520502285a24dc8813e1a4913
MD5 (NetBSD-6.0/acorn26/INSTALL.txt) = f02c846c660d531f87f85271e991c77b
SHA1 (NetBSD-6.0/acorn32/binary/kernel/netbsd-GENERIC.gz) = 8d2b1f19b2ba2231aba8e20c14ab5fd83dde59ae
SHA512 (NetBSD-6.0/acorn32/binary/kernel/netbsd-GENERIC.gz) = 364ad36c66788c822a3cbfd0ddffafc6d19f9bb82144be988848970b7e0b6d11c2afaf4784f2132affe6ca052c69a4595181b1c37363637bb54f849e5bf4a1dd
RMD160 (NetBSD-6.0/acorn32/binary/kernel/netbsd-GENERIC.gz) = 02d7f52cc0c15ff1039f3767cfa3840e5551e03a
MD5 (NetBSD-6.0/acorn32/binary/kernel/netbsd-GENERIC.gz) = 96f504edccc8d4f5657c1837cdfd2aaa
SHA1 (NetBSD-6.0/acorn32/binary/kernel/netbsd-NC.gz) = fc98fecc5548fde9220873e791d4d4846cefe4b3
SHA512 (NetBSD-6.0/acorn32/binary/kernel/netbsd-NC.gz) = ff106589ca7b58c2ef3146a93be72b73987c221694400f3a2256728e618a7b486f32020f031e68edade5d13b3b9584a18ca88ce3fed71f06f9b494b3cb6eb844
RMD160 (NetBSD-6.0/acorn32/binary/kernel/netbsd-NC.gz) = 4ff74a916d3283eb80e77739fd31dcda4890658b
MD5 (NetBSD-6.0/acorn32/binary/kernel/netbsd-NC.gz) = f3a826b7eea3f009868a5ad419b69c3c
SHA1 (NetBSD-6.0/acorn32/binary/sets/games.tgz) = e1d2bb091d8202962133475bc83edb55230b9008
SHA512 (NetBSD-6.0/acorn32/binary/sets/games.tgz) = a147fe6d0419d53d5a428802d0bb6c9fa8a84a76c96d7d6d7c0d399e5af3a69240116463549a83d53387b0214ba0207796d7c10228d92a684fe7ff0ece3aec16
RMD160 (NetBSD-6.0/acorn32/binary/sets/games.tgz) = 2be515c495548ba5135ff42a75a8ebbfc35ea776
MD5 (NetBSD-6.0/acorn32/binary/sets/games.tgz) = 9059a6f2cd41cb58827709c9e9a1649b
SHA1 (NetBSD-6.0/acorn32/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/acorn32/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/acorn32/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/acorn32/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/acorn32/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/acorn32/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/acorn32/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/acorn32/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/acorn32/binary/sets/kern-GENERIC.tgz) = 0ec8993d446ad847bf7a6154a19c638d944bd24b
SHA512 (NetBSD-6.0/acorn32/binary/sets/kern-GENERIC.tgz) = bed0345f83d42ee18b1cb1fab95e6780a217edda00c0c548aeb99323327493d1720b615d746a4e364a5217f963f24ae55142fc943a0d95aec65d8ddff2d36045
RMD160 (NetBSD-6.0/acorn32/binary/sets/kern-GENERIC.tgz) = 2f27e6f036ad4fb08171f93e21324b9f901b781a
MD5 (NetBSD-6.0/acorn32/binary/sets/kern-GENERIC.tgz) = 5eed762a46bf55d66491dab47bd40f23
SHA1 (NetBSD-6.0/acorn32/binary/sets/base.tgz) = 519435bdcdffa41f24c56796558ad43ba6c89f31
SHA512 (NetBSD-6.0/acorn32/binary/sets/base.tgz) = b7a3f4a5627790133104674c25e434d16a0e706006352f0032a4672f624c7fbf37b0678f72f593c5e17dea469afc03e09ac914eb4a02acb47b9a5673bd7e9777
RMD160 (NetBSD-6.0/acorn32/binary/sets/base.tgz) = caf168bcd8c374896c8a0408e3d34e9c7e6f2c7b
MD5 (NetBSD-6.0/acorn32/binary/sets/base.tgz) = b221228a61e59d66f1b457f5dcb1b5cc
SHA1 (NetBSD-6.0/acorn32/binary/sets/comp.tgz) = 54b3cdd58c6db224ec42033f1f40842e55625e9f
SHA512 (NetBSD-6.0/acorn32/binary/sets/comp.tgz) = a32e6f39ea01c783e1ff603c8774130698db0056923ef4c029f29f0df20f6b63885f33a50586774bc7c12b392f278344afd3aca42112bed8a3cd14d6ef72ff7e
RMD160 (NetBSD-6.0/acorn32/binary/sets/comp.tgz) = cde7f725c376e8c0633d821357033e742358bb28
MD5 (NetBSD-6.0/acorn32/binary/sets/comp.tgz) = c3866ce0cbc35fc4ffc58b14cc72ed31
SHA1 (NetBSD-6.0/acorn32/binary/sets/etc.tgz) = e7c005ca92edbf1b8225f951b95b5938a94ee651
SHA512 (NetBSD-6.0/acorn32/binary/sets/etc.tgz) = 01b47e91a0969eb2c3d7fd7f73563e98201ab401b9069b6d5e64e878757761410b8a181fcd04b141168c851d466e6d4e1b48b540e654d1831195deecbe5c23f7
RMD160 (NetBSD-6.0/acorn32/binary/sets/etc.tgz) = f654da0c3fa14a8d8dc3a0cdb977d98b07b0d4ac
MD5 (NetBSD-6.0/acorn32/binary/sets/etc.tgz) = 295b42a630082e4db876d35a7e5a204c
SHA1 (NetBSD-6.0/acorn32/binary/sets/kern-NC.tgz) = e0744de8e7bc2a5808e603563cd8201de0601323
SHA512 (NetBSD-6.0/acorn32/binary/sets/kern-NC.tgz) = 93c2fda5dd5a16eab210f74c0cf749fd013ec3488d6424fea84d404b2db5555a449a6a0ce9d936d7e18ec93216b1fa7a914e951dba3f223ac6b4de27d1f8cd32
RMD160 (NetBSD-6.0/acorn32/binary/sets/kern-NC.tgz) = b0be18f3134075df50daec32dc87e30f370f7531
MD5 (NetBSD-6.0/acorn32/binary/sets/kern-NC.tgz) = 5b8eb4b95fa6151bfbcaf0f72a6f6abb
SHA1 (NetBSD-6.0/acorn32/binary/sets/modules.tgz) = 7df712c3742771ef53b42065426e7761020cf439
SHA512 (NetBSD-6.0/acorn32/binary/sets/modules.tgz) = 3c682c61bca3059470ea6ceaf5e40314cc1876b9b183910692a1664a975aacae619c2214695a92f68463142940640042ba48f9acb39488fddfa757ffb74eb6e4
RMD160 (NetBSD-6.0/acorn32/binary/sets/modules.tgz) = 35fc06c8194cb14f2258ce805c1e4fb8097669d0
MD5 (NetBSD-6.0/acorn32/binary/sets/modules.tgz) = ffbd55bff72382bbc82342bacf6840ba
SHA1 (NetBSD-6.0/acorn32/binary/sets/tests.tgz) = 68dae6dac916379bb48be79e8b2fe49898c4cfe5
SHA512 (NetBSD-6.0/acorn32/binary/sets/tests.tgz) = b06a5ac96d76e0521e2d6666518a1549975a4d11767264c7d3b0d3dd8afe8aaa8e4f3113095d18d74146ee7e4c17a8d8a180bb4367c9aff6359b60ec06107970
RMD160 (NetBSD-6.0/acorn32/binary/sets/tests.tgz) = 5a4c05ecc975a7ee61eda422a7dc7a5b1084424f
MD5 (NetBSD-6.0/acorn32/binary/sets/tests.tgz) = ef0313ce1e3aec16c96660fc0b8e691b
SHA1 (NetBSD-6.0/acorn32/binary/sets/text.tgz) = bb5835aee28d492add5e601c1323c4c1f0ac09e0
SHA512 (NetBSD-6.0/acorn32/binary/sets/text.tgz) = ac45662324b717ccc775d4f4adfaf6623e8383fea5c5d4177fe21f8799a2a7534faff515c979f85cac6de86f166a0fcbc1021a8b8ff2a2c787fd341a4e6a6c91
RMD160 (NetBSD-6.0/acorn32/binary/sets/text.tgz) = 96fbbe37b32d596fad17af067c28acf08eaaaa57
MD5 (NetBSD-6.0/acorn32/binary/sets/text.tgz) = 1ff71742549a26794970d5aaef2c5f09
SHA1 (NetBSD-6.0/acorn32/binary/sets/xbase.tgz) = 150aad78ea2477c5c1b395bda15bae8f0f72d36c
SHA512 (NetBSD-6.0/acorn32/binary/sets/xbase.tgz) = 7ff70ee14b5943899517b4a7b2422ba1c62468399f55905e595d50de15fcfa2e89281a917c96d6d36e2dec519162227f7e6809c9a1da475c68982216d9acb242
RMD160 (NetBSD-6.0/acorn32/binary/sets/xbase.tgz) = 0411d8686275c3d1146e42ddb69a0080bc82d950
MD5 (NetBSD-6.0/acorn32/binary/sets/xbase.tgz) = 97aec1ef3a6ef744c85a7eaa6c403c1a
SHA1 (NetBSD-6.0/acorn32/binary/sets/xcomp.tgz) = 9847900750d33556792e7b62e3e0ffc3f3ea625f
SHA512 (NetBSD-6.0/acorn32/binary/sets/xcomp.tgz) = 010fc75bcd0355d1ea09cd8ddba94b16fab7224bec4aba26d07bf43f259b47b564fe008e6c477ce4a41161ae76dbd26d89f894ec5bf2ea7193cab2965d96de50
RMD160 (NetBSD-6.0/acorn32/binary/sets/xcomp.tgz) = 975dbb1ec0a8f32cee4bfe55b4adb9f617d917be
MD5 (NetBSD-6.0/acorn32/binary/sets/xcomp.tgz) = 7e761d7288845752bf05af5ac943d158
SHA1 (NetBSD-6.0/acorn32/binary/sets/xetc.tgz) = 6b7815869c1c48a2efb0dfcdaae373700d5da8d9
SHA512 (NetBSD-6.0/acorn32/binary/sets/xetc.tgz) = 42cbbc48a006498d46221eff6403acceeaf20ee5cb4a4f368b74e5b4599cccbd54f3d83b4a55359622599184f376f6bea2f2ad0feb364ce7d0293c3fb9d5b32d
RMD160 (NetBSD-6.0/acorn32/binary/sets/xetc.tgz) = a60ea2e898741a2d009ca8c6c621fbcae1f1efce
MD5 (NetBSD-6.0/acorn32/binary/sets/xetc.tgz) = 16cc8b76db930a1eb7b009e7776e2881
SHA1 (NetBSD-6.0/acorn32/binary/sets/xfont.tgz) = 558af1104a11439ab8c868728c55027dd5862ce8
SHA512 (NetBSD-6.0/acorn32/binary/sets/xfont.tgz) = fb28abf27b6cadadc9c0ca82df3025e22d4e0661dcc0488083c7339c5372acf0adea8edc848b7cea50980d971aa24f81bbda068b4b4bed6ac4cf26cc4f5b35bf
RMD160 (NetBSD-6.0/acorn32/binary/sets/xfont.tgz) = f6df63aefaa5c459b2c86ae79fc7b07253424c97
MD5 (NetBSD-6.0/acorn32/binary/sets/xfont.tgz) = 498abc855e6717ab51d5d0be8fa5662d
SHA1 (NetBSD-6.0/acorn32/binary/sets/xserver.tgz) = 292d81e44c40c73e8341d59175695b649f839baa
SHA512 (NetBSD-6.0/acorn32/binary/sets/xserver.tgz) = 3593eeeda7a9f55ff29622c2a95672c746f9235e007b3d8e81e33829fa655d0f6bc8ba90f6ef53baad1b25789fd656d5d8ea14152a9b41abf1a1bfbd2b11937f
RMD160 (NetBSD-6.0/acorn32/binary/sets/xserver.tgz) = e0e2425da086891061ddae334fd8e98d77c519c9
MD5 (NetBSD-6.0/acorn32/binary/sets/xserver.tgz) = bd36f768d1488956ea90a9f13ac8929c
SHA1 (NetBSD-6.0/acorn32/installation/kernel/netbsd-INSTALL.symbols.gz) = b155be9bb81a3c15abe2b6732d48e8bfe53abc53
SHA512 (NetBSD-6.0/acorn32/installation/kernel/netbsd-INSTALL.symbols.gz) = a3a1c068acbe2d634da3442aac4de7107dfcaf3634be291f9c3285aeb9fa680e59acb9273774e6317f71d6f9be2e1851a0e0f60b3cc83972f3e185f3e8237522
RMD160 (NetBSD-6.0/acorn32/installation/kernel/netbsd-INSTALL.symbols.gz) = 3ac12cb07a5e8b9810946c489b9a4c3b9aecdcbc
MD5 (NetBSD-6.0/acorn32/installation/kernel/netbsd-INSTALL.symbols.gz) = 1fe3c151480781a09a6f642ab87b9495
SHA1 (NetBSD-6.0/acorn32/installation/kernel/netbsd-INSTALL.gz) = dc38de07bef2736068ee9cd3103f87ea6938fd6d
SHA512 (NetBSD-6.0/acorn32/installation/kernel/netbsd-INSTALL.gz) = 11b29549b4b8232380eb6db4bf89a979e9b2a2726a3c7ff51ab059fd4f8b6a979d2cb7fe6d0c1802e345c445989655339a9a8b8d4a10f728ac7a24bae9f1830f
RMD160 (NetBSD-6.0/acorn32/installation/kernel/netbsd-INSTALL.gz) = 605df2071d2e590be8a4d0f83c9a55a140066bc5
MD5 (NetBSD-6.0/acorn32/installation/kernel/netbsd-INSTALL.gz) = dbc61ce2c5adf5f8e18441138aa4ae5a
SHA1 (NetBSD-6.0/acorn32/installation/misc/BtNetBSD.spk) = 12dac3d0059f8eef736a378bb6e026166694c1a5
SHA512 (NetBSD-6.0/acorn32/installation/misc/BtNetBSD.spk) = f54991c3e27613ac735b1a10d5ffcc102799417a68415b74c9557657c0e714fd7faa381a3945991bd80b49cefa2ec01395d88d0e935d063ff91174a8e12887ee
RMD160 (NetBSD-6.0/acorn32/installation/misc/BtNetBSD.spk) = 02e5c2a9d751d8236ebbd0d4151f687fdd896735
MD5 (NetBSD-6.0/acorn32/installation/misc/BtNetBSD.spk) = 4b3907f57b03d49765f92ffff178ef40
SHA1 (NetBSD-6.0/acorn32/installation/misc/ramdisk.fs.gz) = 73ae7358fb27f8d227e253b6f43e5fcbdd7cb5f3
SHA512 (NetBSD-6.0/acorn32/installation/misc/ramdisk.fs.gz) = 3c68e80766ae3603b289f11469c2c80ba1ffb5eb30f323fee016d992d7d226bf1984f487ee486cb3d0b1739bf45ae3ad52ddeb5f90ff2d0abd08cafc1a181ccf
RMD160 (NetBSD-6.0/acorn32/installation/misc/ramdisk.fs.gz) = 3320c3a2a2d0e60d55b4612811d1870a4e8b9ec1
MD5 (NetBSD-6.0/acorn32/installation/misc/ramdisk.fs.gz) = 70f0aee2f9cfc56744c50b3bed23b5a5
SHA1 (NetBSD-6.0/acorn32/installation/misc/boot32,ffa) = 1a8c3e3a8950e68d6fd3d22ccf985814f67e00c1
SHA512 (NetBSD-6.0/acorn32/installation/misc/boot32,ffa) = 2f7091d17fc4c910846f5cdc8cd267eb7d4056e281175c00407ab958548c62af4936394a04bde6549223fc874df72c5c46b0d11ae32ea6970c17326b0cedc6af
RMD160 (NetBSD-6.0/acorn32/installation/misc/boot32,ffa) = 5821041d9dff2485cba9ab49e289a0bd479fc63d
MD5 (NetBSD-6.0/acorn32/installation/misc/boot32,ffa) = 0f03b946d6479db2aaf8088dba4632ed
SHA1 (NetBSD-6.0/acorn32/installation/misc/splug.bas) = 19015ba8a9efd73fea65fe180596271640893d1b
SHA512 (NetBSD-6.0/acorn32/installation/misc/splug.bas) = ef38a1da79c93ff798c858f02d3d13e88d24bd0b56808ddb71b3dd577203195a60f601aa52e37892ab164ef7dd0cd490995ac9cd689fe33282fb49435f4c4d56
RMD160 (NetBSD-6.0/acorn32/installation/misc/splug.bas) = 3080fc8d5ff946754a33bc41e7ad83cbe1eaa0b5
MD5 (NetBSD-6.0/acorn32/installation/misc/splug.bas) = 5596dc5df580ceb656d040393e778087
SHA1 (NetBSD-6.0/acorn32/INSTALL.html) = d16bd043227b34d9a9c3c69f380c6c42b50d080c
SHA512 (NetBSD-6.0/acorn32/INSTALL.html) = 629401693cb91992f46b839a34bf4c4ac704194fd8aa6be3aa51e405920b697fe961fa7320eef7a4cf804dad96891b38ca57c61ef3c0c6096bb6bc71f58e55a9
RMD160 (NetBSD-6.0/acorn32/INSTALL.html) = 222b9b590667b0a6dfe2717bb6dc94e95a485174
MD5 (NetBSD-6.0/acorn32/INSTALL.html) = 07466bc8646d08bee6c7030d00cc9173
SHA1 (NetBSD-6.0/acorn32/INSTALL.more) = 011db6fc8ab7b724dc66eafb3b712e9202bb2174
SHA512 (NetBSD-6.0/acorn32/INSTALL.more) = d13335bfe65882608362fda44f89091113e3bffb53893f15bb747aaa72eefbb24ca3fd3d72017cfc446ba49ad7c00f34436e859f71727f9e6a182766a1810a13
RMD160 (NetBSD-6.0/acorn32/INSTALL.more) = aaa800047583e64496187a80e54d0ac3bf4c851c
MD5 (NetBSD-6.0/acorn32/INSTALL.more) = c45c7b7fd7522e41a977ffa6a58b5a4d
SHA1 (NetBSD-6.0/acorn32/INSTALL.ps) = 78d2daf3255e7bc61b5682d5fcfd853d7dd25883
SHA512 (NetBSD-6.0/acorn32/INSTALL.ps) = 4aec0769a5827735e13ab61dd80b5c7254561786afe0c95ff76796a8b29ae2284c064a1a1c523120366e37197bcbfd3d588985dbaae62c9848a4669d8c797714
RMD160 (NetBSD-6.0/acorn32/INSTALL.ps) = 764dc8cc82d5623174886f98f77d808650ef90d1
MD5 (NetBSD-6.0/acorn32/INSTALL.ps) = fc4371f1451a03c4ea429fe029e46d9c
SHA1 (NetBSD-6.0/acorn32/INSTALL.txt) = 9d1ed61f6916645ffa30e66b23da2d0d24e18582
SHA512 (NetBSD-6.0/acorn32/INSTALL.txt) = 10763c9164aeb2d4070f1677b0f97ca8263ec1b9d9c017fbcf348ce302d7d846fa669b741031eef4b12c27a7d077f6af9073f7668244b1b4daa39ab426d79ec7
RMD160 (NetBSD-6.0/acorn32/INSTALL.txt) = 755fbe335d76c9b549fae4d01120cbe00290e8ca
MD5 (NetBSD-6.0/acorn32/INSTALL.txt) = 1a6f36bd9f77e0773aeae95de9ceed8f
SHA1 (NetBSD-6.0/acorn32/prep.RISCOS) = 269ee721985cbc44a8056ca4804f8b2d70d52a3e
SHA512 (NetBSD-6.0/acorn32/prep.RISCOS) = 54923d8dc59e6a3852ceab25adbced3e545de19458c412552ef0cb47534303d812e8860662025b1aa9ddf9a99bb139f4be75656ef109546a328ba57985ec2cfa
RMD160 (NetBSD-6.0/acorn32/prep.RISCOS) = 8e3acd93d35d85062996b061cff5d7293f46ee1f
MD5 (NetBSD-6.0/acorn32/prep.RISCOS) = 3c0e290b44f735728abd3bda8bd96a0f
SHA1 (NetBSD-6.0/algor/binary/kernel/netbsd-P4032.gz) = e9df71ffe487dd98c1aae0c6e79fbf8e4780340a
SHA512 (NetBSD-6.0/algor/binary/kernel/netbsd-P4032.gz) = 48673b13075fc7694dccb1a31b6121281c5c0e85c5d7107a4f9414e9888c19f07133a07e43400495dd9e11757acc2fa2269458ae4b091900772c49a944624465
RMD160 (NetBSD-6.0/algor/binary/kernel/netbsd-P4032.gz) = 85b0c6fde5bd0f4a02f71c30ccaf35d9bc29ac19
MD5 (NetBSD-6.0/algor/binary/kernel/netbsd-P4032.gz) = 2f041466a193e263576e80bfca3ce8ef
SHA1 (NetBSD-6.0/algor/binary/kernel/netbsd-P5064.gz) = f230eefee689dff40a0aca1983191c2e21c7dbde
SHA512 (NetBSD-6.0/algor/binary/kernel/netbsd-P5064.gz) = eb7fea73f29d2141f508743785ee865bd5f7ce23477fc1bf256e40a0f47de76a7cdb22bfac0cb9dce94c0e1694b1c84b1760db13ab1b34248a6e04367817214d
RMD160 (NetBSD-6.0/algor/binary/kernel/netbsd-P5064.gz) = 355d23c8cbde9c1a92bf5f1eb4e1f1890603e7f7
MD5 (NetBSD-6.0/algor/binary/kernel/netbsd-P5064.gz) = 5ebc249debbe1797d915529f1d469f8b
SHA1 (NetBSD-6.0/algor/binary/sets/games.tgz) = 27d9f98fecb3495e51939df9f107e81e527af26f
SHA512 (NetBSD-6.0/algor/binary/sets/games.tgz) = 46bb4e10313cfffaaf4f7fa2157474ef940f94bcc9f84b46f829c7615bbf45e4e01edee7b7efb84bc28561e0712c37cfc737f9c2a0d9a73b67c80c8aa0f3a9af
RMD160 (NetBSD-6.0/algor/binary/sets/games.tgz) = 947feea90ec2698a65d555cb78a29a001b086589
MD5 (NetBSD-6.0/algor/binary/sets/games.tgz) = 355f00978fed8e8da6a251e591be5387
SHA1 (NetBSD-6.0/algor/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/algor/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/algor/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/algor/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/algor/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/algor/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/algor/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/algor/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/algor/binary/sets/xbase.tgz) = 76468fcada5a9e26652aa9f0f9f3c94829a317e9
SHA512 (NetBSD-6.0/algor/binary/sets/xbase.tgz) = b2adcfc2dedef30b21b885c8fc757d0479541744a280a9997fbeaa9c881009ab4b0019e0001a540a8eb6db5f84db87b365e420bb2d4270af209821bdb080369c
RMD160 (NetBSD-6.0/algor/binary/sets/xbase.tgz) = 1e0074d03d9fbb3373e0a427b25cc7097bf6d3c8
MD5 (NetBSD-6.0/algor/binary/sets/xbase.tgz) = a8f29ec2e3768405942f97af3dd08ee7
SHA1 (NetBSD-6.0/algor/binary/sets/xcomp.tgz) = d8ac89f3f6ff20abb0adabc49227afdb74aa2040
SHA512 (NetBSD-6.0/algor/binary/sets/xcomp.tgz) = 4f03213fb061dae9421747209628d06ede6bbb71a7560a9e1a54796b907f1620146a33fdce52974227a21acedcc210b99e15c4fdd36432eba60c1041a92d8cee
RMD160 (NetBSD-6.0/algor/binary/sets/xcomp.tgz) = 6be5e72b9b57c9bdf8ed8ea127e2a73830feeecb
MD5 (NetBSD-6.0/algor/binary/sets/xcomp.tgz) = 59a7ea141587417be38ab1f6056cc780
SHA1 (NetBSD-6.0/algor/binary/sets/xetc.tgz) = b9f6b1a54ffab37b11f4c6e717b481cd087fb609
SHA512 (NetBSD-6.0/algor/binary/sets/xetc.tgz) = 72f35e65f3a90967e250a2006d500619b3824f7f5c3d7494554f153ec5cda5b28c445f28c34a1e4463f40da5855e8d4257726411796967fe37763d90cf7fe461
RMD160 (NetBSD-6.0/algor/binary/sets/xetc.tgz) = 1169c246f91189fd0c4ae86ac27a92f5c2edda3e
MD5 (NetBSD-6.0/algor/binary/sets/xetc.tgz) = 8f3c52cc5af1cafaf7b0d0f884a5b537
SHA1 (NetBSD-6.0/algor/binary/sets/xfont.tgz) = d06848571034f3b7fc975082f1de37b912acb18c
SHA512 (NetBSD-6.0/algor/binary/sets/xfont.tgz) = 66a1013c935bee5e7e5d24325a527522306bcded7ced63790f2e58831b3ed91176a1b9510b2159be63504d5167d48d534919927ffc6e225ae2bb2487f90a9ede
RMD160 (NetBSD-6.0/algor/binary/sets/xfont.tgz) = c9502843a5d4dd7e996e8d158c519cf7f8575cbd
MD5 (NetBSD-6.0/algor/binary/sets/xfont.tgz) = e812899176eac2bf65a6d949ac3f3424
SHA1 (NetBSD-6.0/algor/binary/sets/xserver.tgz) = ff805022f409202f3346c45475c7ec0f8187218e
SHA512 (NetBSD-6.0/algor/binary/sets/xserver.tgz) = f2bb3810a5572b519980126a3d934f63b5ab5c2fcf9fb8166c556bd3cc14cfaa151d582086bb1506ac9ac7e21d4e70a5cd5c3faed96212d8a14a0b5bafdcc4c1
RMD160 (NetBSD-6.0/algor/binary/sets/xserver.tgz) = d5ae336337cc77f7f55f542b3457918d6fc0efe8
MD5 (NetBSD-6.0/algor/binary/sets/xserver.tgz) = 92447a1560887d337d22e2deed0670a6
SHA1 (NetBSD-6.0/algor/binary/sets/kern-P4032.tgz) = 810ebe688c1817dcfd7cab153a6d9997900555b8
SHA512 (NetBSD-6.0/algor/binary/sets/kern-P4032.tgz) = 8846047a1a93d3ba853e235b2169286e09f5955d3116bd2bdb18f0d252c8b0ae0d310f3ccf6d30244d089f7b077579e8246ddb27f32ddcee26e6b7b7b2266aaa
RMD160 (NetBSD-6.0/algor/binary/sets/kern-P4032.tgz) = 039bc25b47bc29648ebb86fad5fb52852badc0fc
MD5 (NetBSD-6.0/algor/binary/sets/kern-P4032.tgz) = f8011c18eab0acde582df5ddda0d3a57
SHA1 (NetBSD-6.0/algor/binary/sets/base.tgz) = 1cacc62a3d246592b148a70bef11a751b63a4810
SHA512 (NetBSD-6.0/algor/binary/sets/base.tgz) = cd448e4f0533c50daf74ed21467409013d512e128b177c9b9ab47771d3d2cffcbe403ba207309af77eb6712d316d9db2923205765be8d9045e2b2216f94774d4
RMD160 (NetBSD-6.0/algor/binary/sets/base.tgz) = c78efd1403b1cb3af77f5a0cbddbd095505d4f65
MD5 (NetBSD-6.0/algor/binary/sets/base.tgz) = 33b467da96b628c9b68cd57209d353b7
SHA1 (NetBSD-6.0/algor/binary/sets/comp.tgz) = 3e955d4a87fe115e37da729b9d2474a775614d30
SHA512 (NetBSD-6.0/algor/binary/sets/comp.tgz) = 12ce5ac677ba9f44944f57859218d668d3d2bc3323ac5793545ca7de849480dc093f03226fed4dac42bf7a4ee69fc998b4a5e2c62607d754e0b58a3cc73d3d9d
RMD160 (NetBSD-6.0/algor/binary/sets/comp.tgz) = 106fbdfddd7c9bd13d1b5d2111b39d4338bd8c10
MD5 (NetBSD-6.0/algor/binary/sets/comp.tgz) = 8d493a09f9e5580f830f15a3c0b12d84
SHA1 (NetBSD-6.0/algor/binary/sets/etc.tgz) = c0511cfd28c1b22e71166507a0f564c24015f025
SHA512 (NetBSD-6.0/algor/binary/sets/etc.tgz) = d47fee208d0d59cf0edbbf2c75bbe404a176f66f0058db93ba553fffdc31d5b5ed6c2ba2df505f7b1576cd06cc27de5da3dfda1230927c2ac603a326aff688b2
RMD160 (NetBSD-6.0/algor/binary/sets/etc.tgz) = 09d069cea8b5a133736d483097bcf24649f0d52a
MD5 (NetBSD-6.0/algor/binary/sets/etc.tgz) = 4ce17fdd45c77f0d8632a836463db036
SHA1 (NetBSD-6.0/algor/binary/sets/kern-P5064.tgz) = 1fa8786b3038a1bc218f5e8280ae3871f5687e4c
SHA512 (NetBSD-6.0/algor/binary/sets/kern-P5064.tgz) = 0977b6911d05ee98e53841af3acff6b5f751c93841c96f3869cfad0e978596204cb22d3a1608842c36d24a50f16a35d0ec00288d56fd69e6cd33d8b2b1dc37c6
RMD160 (NetBSD-6.0/algor/binary/sets/kern-P5064.tgz) = 8fd50961f224a94e272a15df2061f226d0d2456a
MD5 (NetBSD-6.0/algor/binary/sets/kern-P5064.tgz) = 7932f7ce7a89995425a3e97319de27b7
SHA1 (NetBSD-6.0/algor/binary/sets/modules.tgz) = 3627b369e3392221946d517d950da8bb6c3b6170
SHA512 (NetBSD-6.0/algor/binary/sets/modules.tgz) = bbdaa688d423fce2ba3513727b4fd0758cb3c6b6db3147b09319990270380b4cf8feede38a2b57757095627356d6955a30d647d770d67a2bf97e66f0673e4770
RMD160 (NetBSD-6.0/algor/binary/sets/modules.tgz) = 5040267ebae4a2069ad1b9168b31d568d2f81fca
MD5 (NetBSD-6.0/algor/binary/sets/modules.tgz) = d3e9a884b3cef9a6ee2dd6b8c715901a
SHA1 (NetBSD-6.0/algor/binary/sets/tests.tgz) = 76544bff809ab691a739a3ffebc3988b1473fdc2
SHA512 (NetBSD-6.0/algor/binary/sets/tests.tgz) = 5e5709ec49a4e20f253ed5686f9fc62300b3c0aa70bc2cf036a41130803c578cc692b89fc234334723bc39f74cb0a8bd9e9be88cfa692213554713766c9a3b7b
RMD160 (NetBSD-6.0/algor/binary/sets/tests.tgz) = 66eb344c33b3c463c3a6c3af96af263d8cb6dfe0
MD5 (NetBSD-6.0/algor/binary/sets/tests.tgz) = 3ae196d57949ee2cb682ab871c9ef316
SHA1 (NetBSD-6.0/algor/binary/sets/text.tgz) = fe2e0c01c5af66987eab5bb59b06fba32c006be2
SHA512 (NetBSD-6.0/algor/binary/sets/text.tgz) = 3fdd84b91bc54bb3f045bf068d01b4b67fe35a62deffdf8b5964f5dc978d55b58b5b592c9fc9a05d40b305914ce7465981b4d55ec95571203c54f240a2bdff4c
RMD160 (NetBSD-6.0/algor/binary/sets/text.tgz) = 420dc3d206809ea2438b82c912a041637e4d0949
MD5 (NetBSD-6.0/algor/binary/sets/text.tgz) = b421f4428b8e52bb19e997954cb1d0ca
SHA1 (NetBSD-6.0/alpha/binary/kernel/netbsd-GENERIC.MP.gz) = 20bd351669d8b8c5e6ed9726567516aa2b20e101
SHA512 (NetBSD-6.0/alpha/binary/kernel/netbsd-GENERIC.MP.gz) = 16699cd9ff46de1282439235429a63aa38ad6eae8197f037e7f8bd2324ff696a04d4b2008723edb8bdae8e55f139024c55e3858670a3db4f6ff317fef3d39919
RMD160 (NetBSD-6.0/alpha/binary/kernel/netbsd-GENERIC.MP.gz) = 5897524a8bc830e819894c17803a141c65b4f417
MD5 (NetBSD-6.0/alpha/binary/kernel/netbsd-GENERIC.MP.gz) = 0ab46adc2ba2d37bfaac5fe5296fe616
SHA1 (NetBSD-6.0/alpha/binary/kernel/netbsd-GENERIC.gz) = daee92c8428d90796700da80702277f38afd7c78
SHA512 (NetBSD-6.0/alpha/binary/kernel/netbsd-GENERIC.gz) = 658bcdb3fe7c015e0de0d6fd795e9a9033618eb193c6e81b179b978836c93cd3b6a09f79ff4405f831f81e003bd190e91534112e9a923d10c3fb094478f4421a
RMD160 (NetBSD-6.0/alpha/binary/kernel/netbsd-GENERIC.gz) = 2bbdd4f11b998dbe08e2076e895785e90f17b192
MD5 (NetBSD-6.0/alpha/binary/kernel/netbsd-GENERIC.gz) = 12d571967266dbdce4d9c0b02cda4718
SHA1 (NetBSD-6.0/alpha/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/alpha/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/alpha/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/alpha/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/alpha/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/alpha/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/alpha/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/alpha/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/alpha/binary/sets/kern-GENERIC.MP.tgz) = bef91c02bdb6ebd5e578bae73acbcbc5d9fcb47b
SHA512 (NetBSD-6.0/alpha/binary/sets/kern-GENERIC.MP.tgz) = 0800ef43e76c838a4f91535fbad30adee37a4be94d01220ce3d0ca898204ef4b9140d753853faa29944ef1f0b7feff2a54551633ed10293751d0fb5c2134c858
RMD160 (NetBSD-6.0/alpha/binary/sets/kern-GENERIC.MP.tgz) = bbc5e568ee1036473614f45ce443ae6129c2afb1
MD5 (NetBSD-6.0/alpha/binary/sets/kern-GENERIC.MP.tgz) = 255686b1fa7b97a53599c7759fad4843
SHA1 (NetBSD-6.0/alpha/binary/sets/base.tgz) = d6a62bc7f515a3cf470fe4298045346f79dbdfb8
SHA512 (NetBSD-6.0/alpha/binary/sets/base.tgz) = b3eee92239be76b17cf36cf49d845123dd26853ec162d2f340338fff172af3ca20d6c85025e9a5fbc085d8522d331ecec488acc034c757f7a44d885b8f31ec41
RMD160 (NetBSD-6.0/alpha/binary/sets/base.tgz) = 447c0754e06c36c2844563722dfd727938b7fdb5
MD5 (NetBSD-6.0/alpha/binary/sets/base.tgz) = e898499ce51685436186e3be5227184a
SHA1 (NetBSD-6.0/alpha/binary/sets/comp.tgz) = 3cd2356501ebeb4190de3596453b045f694f69de
SHA512 (NetBSD-6.0/alpha/binary/sets/comp.tgz) = 51a526038f6e9cf7d787b86997abee47e9735a56f02215cd8f3260e50d2d3bd7a17d11b2f80b2010fd52640aab0afe074fa8997c3522ccf1a8463854bab9ee0b
RMD160 (NetBSD-6.0/alpha/binary/sets/comp.tgz) = 5fd982783cb934f260a96a0f85279b07deb2b505
MD5 (NetBSD-6.0/alpha/binary/sets/comp.tgz) = c5deb3e315f875050362df56411badcf
SHA1 (NetBSD-6.0/alpha/binary/sets/etc.tgz) = b525757e8a68fee704bc4845c411c6e9ab06db01
SHA512 (NetBSD-6.0/alpha/binary/sets/etc.tgz) = 4c881736688636b7376dc7f255266f606678d7fe57a9a787275dc78bbafb8c0f36948bc6b06ea853bd7d1e2dc3de90693fc518c3ffe57aa831777229d910b8c3
RMD160 (NetBSD-6.0/alpha/binary/sets/etc.tgz) = bbacdc2a3adc49d3772faa1e60227d3f20357220
MD5 (NetBSD-6.0/alpha/binary/sets/etc.tgz) = 5975a4d5767d0d68fd4371e32f1ebd40
SHA1 (NetBSD-6.0/alpha/binary/sets/games.tgz) = f3c81c83cd5dd29ff44eeff54a39bc22e1739b43
SHA512 (NetBSD-6.0/alpha/binary/sets/games.tgz) = f773dfef7f2449a7ea804e756181ea622428becff7cd86e227f8d9c33a343ea097a191c08b73f258e236695ac37df6af80ffcc930ae75a3f6bc4a542a0797673
RMD160 (NetBSD-6.0/alpha/binary/sets/games.tgz) = 079affe9e062a8f418e15fc69dae6b19722d8e43
MD5 (NetBSD-6.0/alpha/binary/sets/games.tgz) = 1bdf16fbef7c6ee4bc1e8f6e99cbe649
SHA1 (NetBSD-6.0/alpha/binary/sets/kern-GENERIC.tgz) = e6e00d7a0d440b435c65b34ba6e5ea156f123f18
SHA512 (NetBSD-6.0/alpha/binary/sets/kern-GENERIC.tgz) = b5a99efe112a3d00b4108dc632a2413c0edc72c8a4fdd4daba1c974b5f1233d48e21964e4e1f7588430d01b0813767861bc0a700910a1e288493fccf269255f3
RMD160 (NetBSD-6.0/alpha/binary/sets/kern-GENERIC.tgz) = ccba42305acbf2080751f1b35c23a8bd2464d1c6
MD5 (NetBSD-6.0/alpha/binary/sets/kern-GENERIC.tgz) = b05f2ce6966e37b9dd3018930753f14c
SHA1 (NetBSD-6.0/alpha/binary/sets/modules.tgz) = 2ed2d5d8a1054fac920b44123147d1f445a8cd20
SHA512 (NetBSD-6.0/alpha/binary/sets/modules.tgz) = e85332336dd7091fecb0d48aa772ec0f9890fae90a71bf3e79178cc319a32c7320e7e6357d4355292335809d12499875357583179199f9b0c94dedcce2bffee8
RMD160 (NetBSD-6.0/alpha/binary/sets/modules.tgz) = 286d34676fdeb368fe4146a636878026b678afad
MD5 (NetBSD-6.0/alpha/binary/sets/modules.tgz) = 31ddea9a696f752f01db0c8910252dbe
SHA1 (NetBSD-6.0/alpha/binary/sets/tests.tgz) = 5a71181696e124159891257ed04551ca0fac96ed
SHA512 (NetBSD-6.0/alpha/binary/sets/tests.tgz) = a62eb137237ed94f16b759f4b836932caf06f4fa4215126db879918e38b993349962b6cd63dc06b2516cccb2604208d12df0b599ae64c4b5bfacd8d51618be0c
RMD160 (NetBSD-6.0/alpha/binary/sets/tests.tgz) = baaa5152926a07e890f3c6ab2e2ba0123e132d62
MD5 (NetBSD-6.0/alpha/binary/sets/tests.tgz) = a3bfffe6cbdb12c847a397c474996dbd
SHA1 (NetBSD-6.0/alpha/binary/sets/text.tgz) = 4087ff757867de3d1e48c7edcd0a1f63879141a2
SHA512 (NetBSD-6.0/alpha/binary/sets/text.tgz) = 396175d3caa9afb10f69ca6724fe1544b685863edb272a36e46c6164f8a5ce8099d9ce98c988fc691e61cafa0741053b31f46af6d87873e7473ee2c7db05c5e2
RMD160 (NetBSD-6.0/alpha/binary/sets/text.tgz) = 84877daedd720eaae66d03670b5391586cb5818d
MD5 (NetBSD-6.0/alpha/binary/sets/text.tgz) = 3ad58e3441e75733d2bd172cc03d96c2
SHA1 (NetBSD-6.0/alpha/binary/sets/xbase.tgz) = 8cb5878605114e42a100af9f6000da692936b458
SHA512 (NetBSD-6.0/alpha/binary/sets/xbase.tgz) = 55901d2b29117e70ee9f60bca845a291f73c827b82a6d79d2035e8e861cc6d3553e9b698a2573af458ba7ff7f0d66c1b904e0510aac5391a768a7e85c8dcafa5
RMD160 (NetBSD-6.0/alpha/binary/sets/xbase.tgz) = 27a6b757c2c136eb48d35e996bebae1154691620
MD5 (NetBSD-6.0/alpha/binary/sets/xbase.tgz) = 7599e1da98dc669a5915742f46819445
SHA1 (NetBSD-6.0/alpha/binary/sets/xcomp.tgz) = 0f7cf0e0611671bcc8763c66b6ddb96ba2b84cbc
SHA512 (NetBSD-6.0/alpha/binary/sets/xcomp.tgz) = cabd0398fbb72d9d2c833a56c5ff77e5fcef535dd4ebb7ecf482fc278c6a05d9bdc4d0e443b2eaa9c1ba2ba6044cb066b9c10de8377bf754c3b2fde1925e90e8
RMD160 (NetBSD-6.0/alpha/binary/sets/xcomp.tgz) = 0e6f7dfd1b78eed02bfedd4e354017d4ca193d3f
MD5 (NetBSD-6.0/alpha/binary/sets/xcomp.tgz) = f830daacbaca8ab9bc135239ee1568ec
SHA1 (NetBSD-6.0/alpha/binary/sets/xetc.tgz) = 392c70eb80215d3165d545e40212a8b5d0e071e1
SHA512 (NetBSD-6.0/alpha/binary/sets/xetc.tgz) = 54d20a3d9023cb964eac1ea822cab4648ee70d4ddd4e45779077f9ee7bfeb895f06adf2ec5e6f63869c4746d89aaefad17837afe43023631a204b1e536b15447
RMD160 (NetBSD-6.0/alpha/binary/sets/xetc.tgz) = 6308c2a5326f2e8dc958195e9c98e151c9727b06
MD5 (NetBSD-6.0/alpha/binary/sets/xetc.tgz) = a591745a497f419dd0d277d43c2e7e1c
SHA1 (NetBSD-6.0/alpha/binary/sets/xfont.tgz) = da7a98e196e0630e68572a298db959c1ad9e5cac
SHA512 (NetBSD-6.0/alpha/binary/sets/xfont.tgz) = 6304ec4f69e76faaaea07f0ee720bc90dbbcb620d8980dedb6966767cd58be9ba2c8c4a7f977f4a4b9293f40dc267f7d1a92ced290e704ea6cab7f512a2e21eb
RMD160 (NetBSD-6.0/alpha/binary/sets/xfont.tgz) = 51c4aeaae821cb352d3249f2894bbd4b90884e5b
MD5 (NetBSD-6.0/alpha/binary/sets/xfont.tgz) = 18fdd4a9bed7c69c840cd8591c17670a
SHA1 (NetBSD-6.0/alpha/binary/sets/xserver.tgz) = 32e0112e7e65de8beda18c41e9ddd18db5d586c9
SHA512 (NetBSD-6.0/alpha/binary/sets/xserver.tgz) = 82587fb8cd0719eb39bc14f0e8bbf8a943d6ac710a9f4bed1ac8ea16443cf84a4f378a5dd8f077d5f8ce9213da74d6444a9da10b872f074ab45da8abc89135a5
RMD160 (NetBSD-6.0/alpha/binary/sets/xserver.tgz) = 60936683b7027ff514f411e63c1767828afed830
MD5 (NetBSD-6.0/alpha/binary/sets/xserver.tgz) = d8b3888a02b4e6f2198d67e731296e4a
SHA1 (NetBSD-6.0/alpha/installation/diskimage/README.files) = 8fef6343776d407c097ffbb0622b509aa3b5d6ed
SHA512 (NetBSD-6.0/alpha/installation/diskimage/README.files) = 0a37602db16239255dd32a5b6585a23780d9c7568e7bcdcd4f67f4cc5869ce772deda68df2428b6a8ae4e7e2e4e449f7b643d1fc5d929efa0409466a50cfaf74
RMD160 (NetBSD-6.0/alpha/installation/diskimage/README.files) = ff4ac5a982e0c75bcc3a466b27b003da73573026
MD5 (NetBSD-6.0/alpha/installation/diskimage/README.files) = 47f2c0383d6fc4ac6c8b25c348fa992c
SHA1 (NetBSD-6.0/alpha/installation/diskimage/cdhdtape) = 560f896de7f7b1e2f8cb908048b3ebaeb6636263
SHA512 (NetBSD-6.0/alpha/installation/diskimage/cdhdtape) = 10d6bd96b74b6df96c35533d14237b771d5f3f449120ea7062aff44960e8c0480048d607139ec2358961d93bba17afef407d0e775513c262e44249224ac54834
RMD160 (NetBSD-6.0/alpha/installation/diskimage/cdhdtape) = 69eadfe52a7b4b3fe412190b7d2fd5fb83f7fe6a
MD5 (NetBSD-6.0/alpha/installation/diskimage/cdhdtape) = 87f184344a501c90af4f7bcff11622ee
SHA1 (NetBSD-6.0/alpha/installation/floppy/README.files) = 8fef6343776d407c097ffbb0622b509aa3b5d6ed
SHA512 (NetBSD-6.0/alpha/installation/floppy/README.files) = 0a37602db16239255dd32a5b6585a23780d9c7568e7bcdcd4f67f4cc5869ce772deda68df2428b6a8ae4e7e2e4e449f7b643d1fc5d929efa0409466a50cfaf74
RMD160 (NetBSD-6.0/alpha/installation/floppy/README.files) = ff4ac5a982e0c75bcc3a466b27b003da73573026
MD5 (NetBSD-6.0/alpha/installation/floppy/README.files) = 47f2c0383d6fc4ac6c8b25c348fa992c
SHA1 (NetBSD-6.0/alpha/installation/floppy/disk1of3) = 4d855acf87e1e1064b06c2232178d75a13c33fd3
SHA512 (NetBSD-6.0/alpha/installation/floppy/disk1of3) = 9e86017634026e3e90a50c84a8f57badb1b7cf223aca9a643fae2bf186e443dc9eb5380cba764c89d64b7288078ca5b086b5a37b97fc47a4cc0ecfaad755af67
RMD160 (NetBSD-6.0/alpha/installation/floppy/disk1of3) = 74abd218ad7f0299822e154ac2b420b258c83aad
MD5 (NetBSD-6.0/alpha/installation/floppy/disk1of3) = 5a2d6e79a6714cf358c9ee8755e0aedf
SHA1 (NetBSD-6.0/alpha/installation/floppy/disk2of3) = 4c8d4ef59efd9aeaceaef12aba4acb5c80ac3be2
SHA512 (NetBSD-6.0/alpha/installation/floppy/disk2of3) = 72fcf5173d3b29eca98afcc9dd1c0ca172cf8509c9f449356086fcef28acff3641140745df32db394b43281a1421000959631cf55185673e430a792d026e4eb3
RMD160 (NetBSD-6.0/alpha/installation/floppy/disk2of3) = 02ff0fdc28cb7ac47eae6869620ec5f801324654
MD5 (NetBSD-6.0/alpha/installation/floppy/disk2of3) = 4f86d12bbbb50c2c06103f2002c88be9
SHA1 (NetBSD-6.0/alpha/installation/floppy/disk3of3) = 4f0a721ea8eb063d964c3ec31e035f41febad205
SHA512 (NetBSD-6.0/alpha/installation/floppy/disk3of3) = f3c66c64760246cde2e99999deac1a4dba5caa47603491de0edd47badafc5f6a08b7108e62a3e2c44d4cca55f17df48e1bc41fee948bca15d7dc0874ed321b18
RMD160 (NetBSD-6.0/alpha/installation/floppy/disk3of3) = 1ca8581e23cc086d8bf93b39bf1efb34c8964d32
MD5 (NetBSD-6.0/alpha/installation/floppy/disk3of3) = e558bf919697872fc6f5bfb8ef697f8f
SHA1 (NetBSD-6.0/alpha/installation/instkernel/netbsd.symbols.gz) = 72358f5496a38794927cca6f55addc09e74983bd
SHA512 (NetBSD-6.0/alpha/installation/instkernel/netbsd.symbols.gz) = f4fde67472e0cea2069a5d8c1a614429486c275a3ff38fd2f9c216a79749b366c79c898e9e4d58b4a2826c8b724d76b6de6b744d4eba058169ffb88b428b7790
RMD160 (NetBSD-6.0/alpha/installation/instkernel/netbsd.symbols.gz) = 0b8b2fda5075cf789139581258d33e55e7edc18b
MD5 (NetBSD-6.0/alpha/installation/instkernel/netbsd.symbols.gz) = 2f1474645c7c61a79af6046b873d0c53
SHA1 (NetBSD-6.0/alpha/installation/instkernel/README.files) = 8fef6343776d407c097ffbb0622b509aa3b5d6ed
SHA512 (NetBSD-6.0/alpha/installation/instkernel/README.files) = 0a37602db16239255dd32a5b6585a23780d9c7568e7bcdcd4f67f4cc5869ce772deda68df2428b6a8ae4e7e2e4e449f7b643d1fc5d929efa0409466a50cfaf74
RMD160 (NetBSD-6.0/alpha/installation/instkernel/README.files) = ff4ac5a982e0c75bcc3a466b27b003da73573026
MD5 (NetBSD-6.0/alpha/installation/instkernel/README.files) = 47f2c0383d6fc4ac6c8b25c348fa992c
SHA1 (NetBSD-6.0/alpha/installation/instkernel/netbsd.gz) = 3057e0db687c65ad00bcc9433a8e1e8e8853b0f8
SHA512 (NetBSD-6.0/alpha/installation/instkernel/netbsd.gz) = 72c72707e34bf282fec601e885eaa9dbfafc34dc44c97d1e100e439620ed5f3751e2b629b2b1ff40b229a29b1b217627f5fb70a198d571c0478d13d2c1f9d5e2
RMD160 (NetBSD-6.0/alpha/installation/instkernel/netbsd.gz) = bb22d8b6c7113aad71720a489ab71dbe938e1475
MD5 (NetBSD-6.0/alpha/installation/instkernel/netbsd.gz) = 9fece65592c8b7decdbff32513650306
SHA1 (NetBSD-6.0/alpha/installation/netboot/netboot) = 17cf3e7f848aa8a853e322e977163547830f7357
SHA512 (NetBSD-6.0/alpha/installation/netboot/netboot) = eb78138cdb1d289e4f39af61e24a2d0a42b5807c6869e921ffbcaf720902362f0c702ece18e1bbbb599d50b31b09ecebcaf9df5786f337ee511c0900e1bcb9cf
RMD160 (NetBSD-6.0/alpha/installation/netboot/netboot) = 3228d2be31a224636f15cfa0f5c24cebea1d4b64
MD5 (NetBSD-6.0/alpha/installation/netboot/netboot) = 2a71b40ce0eebc66564f796f681592b2
SHA1 (NetBSD-6.0/alpha/INSTALL.html) = 6984fd17504c01e2c1dd83e7dbdf34bdcc7b6b1e
SHA512 (NetBSD-6.0/alpha/INSTALL.html) = d4b63b33e62764fa6e008f5961599abf0ceb486aee59c55deb0e2f62f5e3efb0a7ec667217f72f553460d1931fff86c209345c6eccb688f67822e19f78f2b35b
RMD160 (NetBSD-6.0/alpha/INSTALL.html) = 3d6597686d7079ca4857f5afe7fada4ddae2e4db
MD5 (NetBSD-6.0/alpha/INSTALL.html) = c00f5ce1cc724dddfbe201deb54605d3
SHA1 (NetBSD-6.0/alpha/INSTALL.more) = b670f381aeda31fa7a1c861aa90c8856a144460c
SHA512 (NetBSD-6.0/alpha/INSTALL.more) = e8343ca62e59534c74a2cb3404092b71a771390943c5738c10cec908be2b202b4caafa09629b5086c0119f8c867a294fb759a76dc80705583f3db7115ad41b1c
RMD160 (NetBSD-6.0/alpha/INSTALL.more) = 6236acfbb416a7f45499c9ba097b0ebc7df0ddc0
MD5 (NetBSD-6.0/alpha/INSTALL.more) = c0f41cb950b8c2133e1c3d6005ed478b
SHA1 (NetBSD-6.0/alpha/INSTALL.ps) = 902a55dd7d1b3cc6489f730e76fd8a656188ca3c
SHA512 (NetBSD-6.0/alpha/INSTALL.ps) = 612f6a50dd02a6ef525948faaa4f77c6b60e6ea6b73fbf43fe28891523b0ba715c4c513991bbd99b36a1842c5bea3a4f59c31abcb7141af3ca77419d5ac7a399
RMD160 (NetBSD-6.0/alpha/INSTALL.ps) = 95af61540f44c6ad5bde6110bb988b30144cac0e
MD5 (NetBSD-6.0/alpha/INSTALL.ps) = d5cb827162bea74a99a75c4ec658cb13
SHA1 (NetBSD-6.0/alpha/INSTALL.txt) = 00428f4fee211a107142bd3c673ac0e1bf82f8bb
SHA512 (NetBSD-6.0/alpha/INSTALL.txt) = acc36a8ca1e2e171db9a482d5b5914cde68c2ba917478c91cbe4c6b1f14f0e46b6fc5393a6597b928b2fe043afe9373e92bc50eddaccad205b1513c9cfa5b89b
RMD160 (NetBSD-6.0/alpha/INSTALL.txt) = c340c3eafed3235732e7bd6de8bd2e5798279a32
MD5 (NetBSD-6.0/alpha/INSTALL.txt) = bebe671881b51664213b2b74a352ccea
SHA1 (NetBSD-6.0/amd64/binary/kernel/netbsd-GENERIC.symbols.gz) = cb8fb452336026da954a431064ac908983c73c56
SHA512 (NetBSD-6.0/amd64/binary/kernel/netbsd-GENERIC.symbols.gz) = 5ac462c2bbe07c1fc5bb719af9b8a048278c914e25c3e63a5908d04eed9b9ed3b3324af677f5910065c569e7eef9ab92fb7b23dc27a06361065193736170beb9
RMD160 (NetBSD-6.0/amd64/binary/kernel/netbsd-GENERIC.symbols.gz) = f294d6c837951cc2179aabe993db92be0507b73b
MD5 (NetBSD-6.0/amd64/binary/kernel/netbsd-GENERIC.symbols.gz) = d5c7cba8ce43359a50bc6b41e1a59ac8
SHA1 (NetBSD-6.0/amd64/binary/kernel/netbsd-GENERIC.gz) = 0e1a51e79c8ed82328856cb4e9e1ca1f77f96ad6
SHA512 (NetBSD-6.0/amd64/binary/kernel/netbsd-GENERIC.gz) = 71822434adf5610fc6bf77371f0ea65403cbfac39194435c6fd0dcd62cfd214da5d9bd8c9a06ae300b81425339ecdca1e373e91d38870a56f1910e9e4b2f0dcb
RMD160 (NetBSD-6.0/amd64/binary/kernel/netbsd-GENERIC.gz) = 2344b490341441b4a08608454269d1bb55db13fa
MD5 (NetBSD-6.0/amd64/binary/kernel/netbsd-GENERIC.gz) = fdba9cfe2198dbc3403ba5c9b251147e
SHA1 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL.symbols.gz) = b275c23887a6d7522777f6c70da7191ae2d3b64d
SHA512 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL.symbols.gz) = 30f57b1aa47d0c8dfae72063810dffda248ad25e03e432221861c4de98088eba502637675cf4d400809c6d2d45a17ee2b3ce87be4007fa8a2b86fb507d13426b
RMD160 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL.symbols.gz) = c98d06602cce33614079d4ac9049235aa71601f5
MD5 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL.symbols.gz) = 5d2e2deeac92159eefa0c14bedb999e5
SHA1 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL.gz) = 22859d8f0aa148474db11e8493c1c1c5ca88a4bf
SHA512 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL.gz) = 5aad6a3e174330803b12b3c5566f422c96d6755e520b2bda951e7efd73f8f4bfbc0b902d7964688650717630355af96d51fed90fac5bbfe343f3c06d3354f5e6
RMD160 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL.gz) = 25124d4395c714580bcc3a2f8574df3d9a5e5ef4
MD5 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL.gz) = 09691319c1029293d76f2b51048cd7d8
SHA1 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL_XEN3_DOMU.gz) = 1ef0dce70ce636d55eb7cb5973f335729d17826a
SHA512 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL_XEN3_DOMU.gz) = 49a644ba39cd11ad2fd3c1aa27b7f2999019613bbb4731ee2098b484eee74d740baf4f67ddb7b4cdaf7f710e507ed719bc6840d704acf5ed80ce419fb8bfae1b
RMD160 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL_XEN3_DOMU.gz) = 459fb44892f496a9c7b445e837d875ddc1a7f056
MD5 (NetBSD-6.0/amd64/binary/kernel/netbsd-INSTALL_XEN3_DOMU.gz) = 5902c6bc85313724e355e95a06e968a0
SHA1 (NetBSD-6.0/amd64/binary/kernel/netbsd-XEN3_DOM0.gz) = 0f4ec7742c4c53d99cbc4b645931758d0a7ebe2a
SHA512 (NetBSD-6.0/amd64/binary/kernel/netbsd-XEN3_DOM0.gz) = fa1e39362e515e808831bc81751231062ebe4987eb1790eb87cab7490697372472fac67df5d49fd41b1d6d3fbecc93f64d7c433d357c13a626636feda14f4a7e
RMD160 (NetBSD-6.0/amd64/binary/kernel/netbsd-XEN3_DOM0.gz) = 2322a8960ba58563ea8507a0073c03934d8f58ad
MD5 (NetBSD-6.0/amd64/binary/kernel/netbsd-XEN3_DOM0.gz) = 31ea20a739a3e9caad0f4af0da531a0d
SHA1 (NetBSD-6.0/amd64/binary/kernel/netbsd-XEN3_DOMU.gz) = 30bfc0edb5ec053c327cfab318877bcf633e3d26
SHA512 (NetBSD-6.0/amd64/binary/kernel/netbsd-XEN3_DOMU.gz) = 7244306522782443df4e807ebf627eab0f6f994c3c1de9cde73781b7b8d16a2b3e7d1ab578c03bd3b2d6df19dc20485146ecb9ff920e2860a0858158509a4229
RMD160 (NetBSD-6.0/amd64/binary/kernel/netbsd-XEN3_DOMU.gz) = bbae752f9fc5b0a9725e12aef5da652699ba614b
MD5 (NetBSD-6.0/amd64/binary/kernel/netbsd-XEN3_DOMU.gz) = 72cc8998d1dd79b0471e4924a1ee9cbf
SHA1 (NetBSD-6.0/amd64/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/amd64/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/amd64/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/amd64/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/amd64/binary/sets/kern-GENERIC.tgz) = 8880ef6795951dbca5379bffb7040b31e75e8217
SHA512 (NetBSD-6.0/amd64/binary/sets/kern-GENERIC.tgz) = e1a58551a91511f0fa86ab4395a8901154e75bae344045200c1794d759cbd82c2441703123ee0c5d42561f08cb192e2bb39c57f4cb61b8fb970b579525fb1952
RMD160 (NetBSD-6.0/amd64/binary/sets/kern-GENERIC.tgz) = be4c1cef72f8b3971890c549d908d03cf9ccae74
MD5 (NetBSD-6.0/amd64/binary/sets/kern-GENERIC.tgz) = 60e67b64337a09d0d00bbe70c0ef43f6
SHA1 (NetBSD-6.0/amd64/binary/sets/base.tgz) = 41f139d30b18839b0486e238bdcb1a08e084ac0f
SHA512 (NetBSD-6.0/amd64/binary/sets/base.tgz) = 136fec9ccc141fbf39c4667f09f45ed3a7f07183c315819c391e3c47b67574d4278762e9be6ac7e325e7ebd2929dd59d1d8aa84411fb435baccf5b7d71c1e58b
RMD160 (NetBSD-6.0/amd64/binary/sets/base.tgz) = 38903209c490bb8094061951fffc5ba7bf2cce24
MD5 (NetBSD-6.0/amd64/binary/sets/base.tgz) = c4d69c1e245e95ee1ef891a998536f30
SHA1 (NetBSD-6.0/amd64/binary/sets/comp.tgz) = 5d0d34f4debbf611578852fdc549197357f574c2
SHA512 (NetBSD-6.0/amd64/binary/sets/comp.tgz) = 96cd27a092b2cbabc0f6f053d651ae94b6ec65b2c9f4812c6efd432c9ea9a321d38f8e88e6dba03ca7e6d3fe8dc2b43c3846880966964d6841d6cacb87f5c4a0
RMD160 (NetBSD-6.0/amd64/binary/sets/comp.tgz) = 0fc9f9cef2839291204f399014f730a321ff846a
MD5 (NetBSD-6.0/amd64/binary/sets/comp.tgz) = 1f6de4df6174d0e919b343bb0426ba92
SHA1 (NetBSD-6.0/amd64/binary/sets/etc.tgz) = 23e42cd803c314f35f61713d7ce3d8ae6f978036
SHA512 (NetBSD-6.0/amd64/binary/sets/etc.tgz) = 3aa2017bfd9d81b51371b31ae2eb4fa613c3cc90c532aabf1e76df938c312d1894f1b5313869a93fbd4463b903bb390ea5cf485e870d421ff0580240a04505e3
RMD160 (NetBSD-6.0/amd64/binary/sets/etc.tgz) = d8d98a46df49ec9a28cda63da60c2fd8b2f5440c
MD5 (NetBSD-6.0/amd64/binary/sets/etc.tgz) = d4d5351a9ee4c89d46d03025363d0af7
SHA1 (NetBSD-6.0/amd64/binary/sets/games.tgz) = 535c8d580229e63485dbc9215ff8994babd19ba4
SHA512 (NetBSD-6.0/amd64/binary/sets/games.tgz) = abc07cbb728d3b4148ca824f3d37c0fd789f22548f92c8d03c78cd04969828d6fb3e7396654e4709bfdc82783e5c17e3998372149b76a7bdfb186fd417a4e1cd
RMD160 (NetBSD-6.0/amd64/binary/sets/games.tgz) = 80ca9419dae0b955ae39ba879b269eb27d34b0f9
MD5 (NetBSD-6.0/amd64/binary/sets/games.tgz) = 1f7b98cfd037ac9a6b2487b1d3e2397a
SHA1 (NetBSD-6.0/amd64/binary/sets/kern-XEN3_DOM0.tgz) = 622ca31060c3d803bff339d53321e3b5cbe56fcf
SHA512 (NetBSD-6.0/amd64/binary/sets/kern-XEN3_DOM0.tgz) = 9943d8752f75569a2cb6611f2eb87a552cf98c92040cf2c0ec699ff0380b3b5395a7b2c951fcdbc6ef74d36073b7b066dc37ee1d60a3463836820fee7f15e5fe
RMD160 (NetBSD-6.0/amd64/binary/sets/kern-XEN3_DOM0.tgz) = 4627daedf28168da5994ad430aaac7b069785d42
MD5 (NetBSD-6.0/amd64/binary/sets/kern-XEN3_DOM0.tgz) = 3c473eed7a2c400bc22e743e6f2fad74
SHA1 (NetBSD-6.0/amd64/binary/sets/kern-XEN3_DOMU.tgz) = d29bb910d1ecf61700d54790c7376f2729dd6a71
SHA512 (NetBSD-6.0/amd64/binary/sets/kern-XEN3_DOMU.tgz) = e714d759a0635e3375ce56f945fa5df011b5037502ce0602d74db1a9d3ddf86144b13b9a2a503c231c931de998a645b8562b5254bc8e68c9537a7dea366957da
RMD160 (NetBSD-6.0/amd64/binary/sets/kern-XEN3_DOMU.tgz) = adb7515022b355b9b2f61a318d1fea6734e8859a
MD5 (NetBSD-6.0/amd64/binary/sets/kern-XEN3_DOMU.tgz) = 3beec0bff32cc7c78bcadb6f90b0455f
SHA1 (NetBSD-6.0/amd64/binary/sets/man.tgz) = f583717a7705475c144e1b66e93b3c9c6d8e3293
SHA512 (NetBSD-6.0/amd64/binary/sets/man.tgz) = 92f6dbefacff42d2a6c317158aee801e8d4f3791704a792ce0761faf5425c516166a9769e7f3b81abd282cc9a60e0ff07c4350ac889fe66aec794f359a27452b
RMD160 (NetBSD-6.0/amd64/binary/sets/man.tgz) = 02ffd712d7b6d265e8c566ccbfcaa34cc9916eda
MD5 (NetBSD-6.0/amd64/binary/sets/man.tgz) = 59adc1ce49347d61ee6eea024e2ea2a2
SHA1 (NetBSD-6.0/amd64/binary/sets/modules.tgz) = 46a78ea373e41e73dabc27ae802b395b7348c419
SHA512 (NetBSD-6.0/amd64/binary/sets/modules.tgz) = 8dc00ff46c87d04b72f2e4fa0b2b290ab326b907f88f4d2667e63aa0390756df4c45ed5b9e36bec6e60e472a8162b9ddc55eea89b6ed65bd2f6cf5d760f611b9
RMD160 (NetBSD-6.0/amd64/binary/sets/modules.tgz) = b2d93414e9e22b4b55dd73fb347aefe4b6a11d10
MD5 (NetBSD-6.0/amd64/binary/sets/modules.tgz) = aa84f46e76b3f543ac9d8ea315869092
SHA1 (NetBSD-6.0/amd64/binary/sets/tests.tgz) = cdcfc53423e8cd984258bc8635490bd815576322
SHA512 (NetBSD-6.0/amd64/binary/sets/tests.tgz) = d11520f52dd889947a298a2428e4b81b840ef939ee82a63fb39ac30fa59918348a6546598b9b6f28268677f7bebf9e022f8ba568a47e3f9ff801ff08666c1366
RMD160 (NetBSD-6.0/amd64/binary/sets/tests.tgz) = 7fb2dbecae8a377eca305f74075e9f4c4f326455
MD5 (NetBSD-6.0/amd64/binary/sets/tests.tgz) = 4c8e0463db9e784b47d92aa43bd7ec82
SHA1 (NetBSD-6.0/amd64/binary/sets/text.tgz) = 2ad7dfb1cf3e39da165e6d0d356444a3dda315ab
SHA512 (NetBSD-6.0/amd64/binary/sets/text.tgz) = 77140ca96473762ba22737bdf1dcd74fbe7dea97778c2da7b8b015daa2af47c20a9663e5eee092a8b353cb36caf599efc5975b7572692cb5c8b77bcde7bb8fff
RMD160 (NetBSD-6.0/amd64/binary/sets/text.tgz) = bdb4eb7e3081ec2bbd4b1eaaacd0cd96dfe92be1
MD5 (NetBSD-6.0/amd64/binary/sets/text.tgz) = 3c9ac230578345961050fd8adbb15023
SHA1 (NetBSD-6.0/amd64/binary/sets/xbase.tgz) = c88f6ae63e7acba75831ae5a19cfb8151f961780
SHA512 (NetBSD-6.0/amd64/binary/sets/xbase.tgz) = be81153fc620a58e00ab7e1069dfa80567e9d641fc4ab6256512074d5cebb7a561fa8596de8f985022ef79ec0330ee317f2a2628a7236882f28a1ea9475d809f
RMD160 (NetBSD-6.0/amd64/binary/sets/xbase.tgz) = 9e961b920bdc6432e1e0bfcad9adfb8dfc872cfb
MD5 (NetBSD-6.0/amd64/binary/sets/xbase.tgz) = b98819c3afd401940a578ced15418ab6
SHA1 (NetBSD-6.0/amd64/binary/sets/xcomp.tgz) = 7f6d2efa3a5e3a5ea2729020a8ad175929353995
SHA512 (NetBSD-6.0/amd64/binary/sets/xcomp.tgz) = 35652d685c9581d20f198d13de8b34764c9a67e808416f006b98793f6ded279111d2bbeee6ad453a8416ba3259a5d7967d0327aa7e7109b7651d923bde31a8bd
RMD160 (NetBSD-6.0/amd64/binary/sets/xcomp.tgz) = 0b3a659fdca9ed423aec4e799f739ad6ed3a2b7b
MD5 (NetBSD-6.0/amd64/binary/sets/xcomp.tgz) = 72711d2c486caca7a8db94316c7704ad
SHA1 (NetBSD-6.0/amd64/binary/sets/xetc.tgz) = d2a37199497c292f9a780bb5a77a14dd481bc0db
SHA512 (NetBSD-6.0/amd64/binary/sets/xetc.tgz) = e8fe86ecaa56d0a016ad52024d956d8e77d690b4ccbb24b060d1c824f5e67979b2067ea6635361f25e80160e6abe3d4bce7ac586107fb673c12d65e9d9bbc4bf
RMD160 (NetBSD-6.0/amd64/binary/sets/xetc.tgz) = e566d0c03e66e9365ad71c0057dd86bcc8663f96
MD5 (NetBSD-6.0/amd64/binary/sets/xetc.tgz) = 6a1af564a85ba3996757440dfed4df94
SHA1 (NetBSD-6.0/amd64/binary/sets/xfont.tgz) = 425d64efdeed5bf9f8038b9f0d61cdfd0b8679ba
SHA512 (NetBSD-6.0/amd64/binary/sets/xfont.tgz) = c6e1b0829edb2e78302cf352880d4ee3cb6f6a5187f8082b0a0df3d49bd32f57cb0df450b850044cc0437da2f0ad75249cbf0777295d90ab38a9e1e13c786e08
RMD160 (NetBSD-6.0/amd64/binary/sets/xfont.tgz) = d0f3520bdaad4cdbd90b3fe32e663c543153c0d0
MD5 (NetBSD-6.0/amd64/binary/sets/xfont.tgz) = be05eeb3078d0fbb93b97130d49c5982
SHA1 (NetBSD-6.0/amd64/binary/sets/xserver.tgz) = bc339a86b1e7d42e667a03561dd66835b0e39828
SHA512 (NetBSD-6.0/amd64/binary/sets/xserver.tgz) = 00739626f1c78297988d1696e369b7ec933119ee7f76d57c426f83a1efba9414b7a1258258ab86b6b79a1277a327ac3ab0e2177bcaa396332ecc5221c6bf8954
RMD160 (NetBSD-6.0/amd64/binary/sets/xserver.tgz) = 9adf580fd382fa5edc147da77b23410830e6ef9b
MD5 (NetBSD-6.0/amd64/binary/sets/xserver.tgz) = b30af2c69ca833f25c0b163fea4496d8
SHA1 (NetBSD-6.0/amd64/installation/cdrom/boot-com.iso) = 0aeb15a62bfd640903525b517b5cb254fbb8f694
SHA512 (NetBSD-6.0/amd64/installation/cdrom/boot-com.iso) = c3998bedff5f4320e3b14b3bc07f7e348f45a106a1f78d738934fbce564b209eb9bce72f91bfbd4f95247a2eb4fb69366d3a8a4f7a597be3bfae889858f1badc
RMD160 (NetBSD-6.0/amd64/installation/cdrom/boot-com.iso) = 90e7d9c68c959a8c3455fca88ac7fdc5af40589d
MD5 (NetBSD-6.0/amd64/installation/cdrom/boot-com.iso) = 5551d8cd6c5675eda32c80de279860ee
SHA1 (NetBSD-6.0/amd64/installation/cdrom/boot.iso) = 12323858ce09cfce6d3b294bb6d33a3ecd599400
SHA512 (NetBSD-6.0/amd64/installation/cdrom/boot.iso) = 536021f6159e5412a9b7be41967e75cdaafb3fcb319ce7c712af6bac22cee2b07f185500fb77505f5a4018a0fa5e619a5e00dc11aee59babfc1dd3c69081a0c0
RMD160 (NetBSD-6.0/amd64/installation/cdrom/boot.iso) = a66e6ff744cb27ed79f58824945806b4ddc5e743
MD5 (NetBSD-6.0/amd64/installation/cdrom/boot.iso) = b4ae730ff5fcc86885c266665183dd35
SHA1 (NetBSD-6.0/amd64/installation/floppy/boot-com1.fs) = 7773242845d56dae16e957ba3dcde54db2d44ad9
SHA512 (NetBSD-6.0/amd64/installation/floppy/boot-com1.fs) = 14a0d9a1d7b6d893ee962b6c0e73cd2f44cdef9aa0c8d814465eb9e30bbee00fffd332a962d86430a04e13fb8b111fc857b5790648d27bbbe6eaac3fb0f6a0c5
RMD160 (NetBSD-6.0/amd64/installation/floppy/boot-com1.fs) = 911fbf259c7ede76a08956e0e38ea8dba1522ae6
MD5 (NetBSD-6.0/amd64/installation/floppy/boot-com1.fs) = 18052eb95118b57ee28cff9cd78a0fab
SHA1 (NetBSD-6.0/amd64/installation/floppy/boot-com2.fs) = 5a1705082aa0fdbe9df2deeb8bfcde27f62e39c9
SHA512 (NetBSD-6.0/amd64/installation/floppy/boot-com2.fs) = 57a01c68902c568a9e061d588b3f96857de3ebe05eab55fffc38a843bb1e25d3c1962aeb78d711a5c70ee2cc7abd5717abc4b4224794803019cd1e81c9afa547
RMD160 (NetBSD-6.0/amd64/installation/floppy/boot-com2.fs) = 03c5214a031d1cee57c50888bad984f59a0ec275
MD5 (NetBSD-6.0/amd64/installation/floppy/boot-com2.fs) = 9d14a976045f475ae3a96cb7ed04bae6
SHA1 (NetBSD-6.0/amd64/installation/floppy/boot-com3.fs) = 80a4a45120aad34e9cc46b8f2fbba73977e395da
SHA512 (NetBSD-6.0/amd64/installation/floppy/boot-com3.fs) = db431a6d199f53a3f367e759ed4fcb117a7b664dd9d5060677a7b3ef60853d3c6d31f5c4e6c54b33df6e4223a8aa7e27c3cb56f777205b58b7239980da5f58d5
RMD160 (NetBSD-6.0/amd64/installation/floppy/boot-com3.fs) = fd6ca1f2110722e28189179a1e6ae1724c1b0c3d
MD5 (NetBSD-6.0/amd64/installation/floppy/boot-com3.fs) = cf7c49437a2e74cef7d07e5ecefe26f9
SHA1 (NetBSD-6.0/amd64/installation/floppy/boot-com4.fs) = 611b759a6b95830057cea83517f8065fa131d328
SHA512 (NetBSD-6.0/amd64/installation/floppy/boot-com4.fs) = 8d0229d04d365efe365a1dc84a5d05090290d684fda96cb637fd8989e162a6cd90c9a9409a80740356cdbe39cb81dcf498874db475b4c2aa443ec3087675611c
RMD160 (NetBSD-6.0/amd64/installation/floppy/boot-com4.fs) = f7fda0f8502ab8255fc63de3de54c92d05ce6a17
MD5 (NetBSD-6.0/amd64/installation/floppy/boot-com4.fs) = 4e1c8e49e8592e5e7a13e4118df3a13c
SHA1 (NetBSD-6.0/amd64/installation/floppy/boot-com5.fs) = 671ad9865cf186c69739279463e408504e89c072
SHA512 (NetBSD-6.0/amd64/installation/floppy/boot-com5.fs) = 1a4329afefd19b3dbf37a12cb287aa771dfabdeeaf6b0302158eae0cde68d970a94a9643b4b18798654473c6eb577c8e555188a7c7776e6cccc05241a7b47c7f
RMD160 (NetBSD-6.0/amd64/installation/floppy/boot-com5.fs) = 9e7000ef7b4a971936e74eb1858b0f74816d42f3
MD5 (NetBSD-6.0/amd64/installation/floppy/boot-com5.fs) = 52824ec1c65bc87f196fe4fcd1887cb2
SHA1 (NetBSD-6.0/amd64/installation/floppy/boot1.fs) = 8393070ff7b79fb60d3aa6bf9675c5bce42b7340
SHA512 (NetBSD-6.0/amd64/installation/floppy/boot1.fs) = 092f3177383ff4bb4f51fdca19069306f75bb0b86674e2862c34374bdc2da633008805c0cdd2bd352f0da12a2cd529e19086323f0c74d1fed89b2a0d65066bba
RMD160 (NetBSD-6.0/amd64/installation/floppy/boot1.fs) = 9b24c6e8042942c96ad46c65443e48824c2b1fa8
MD5 (NetBSD-6.0/amd64/installation/floppy/boot1.fs) = ad5246fc6d7ce6840762ce87d281e0fc
SHA1 (NetBSD-6.0/amd64/installation/floppy/boot2.fs) = 5a1705082aa0fdbe9df2deeb8bfcde27f62e39c9
SHA512 (NetBSD-6.0/amd64/installation/floppy/boot2.fs) = 57a01c68902c568a9e061d588b3f96857de3ebe05eab55fffc38a843bb1e25d3c1962aeb78d711a5c70ee2cc7abd5717abc4b4224794803019cd1e81c9afa547
RMD160 (NetBSD-6.0/amd64/installation/floppy/boot2.fs) = 03c5214a031d1cee57c50888bad984f59a0ec275
MD5 (NetBSD-6.0/amd64/installation/floppy/boot2.fs) = 9d14a976045f475ae3a96cb7ed04bae6
SHA1 (NetBSD-6.0/amd64/installation/floppy/boot3.fs) = 80a4a45120aad34e9cc46b8f2fbba73977e395da
SHA512 (NetBSD-6.0/amd64/installation/floppy/boot3.fs) = db431a6d199f53a3f367e759ed4fcb117a7b664dd9d5060677a7b3ef60853d3c6d31f5c4e6c54b33df6e4223a8aa7e27c3cb56f777205b58b7239980da5f58d5
RMD160 (NetBSD-6.0/amd64/installation/floppy/boot3.fs) = fd6ca1f2110722e28189179a1e6ae1724c1b0c3d
MD5 (NetBSD-6.0/amd64/installation/floppy/boot3.fs) = cf7c49437a2e74cef7d07e5ecefe26f9
SHA1 (NetBSD-6.0/amd64/installation/floppy/boot4.fs) = 611b759a6b95830057cea83517f8065fa131d328
SHA512 (NetBSD-6.0/amd64/installation/floppy/boot4.fs) = 8d0229d04d365efe365a1dc84a5d05090290d684fda96cb637fd8989e162a6cd90c9a9409a80740356cdbe39cb81dcf498874db475b4c2aa443ec3087675611c
RMD160 (NetBSD-6.0/amd64/installation/floppy/boot4.fs) = f7fda0f8502ab8255fc63de3de54c92d05ce6a17
MD5 (NetBSD-6.0/amd64/installation/floppy/boot4.fs) = 4e1c8e49e8592e5e7a13e4118df3a13c
SHA1 (NetBSD-6.0/amd64/installation/floppy/boot5.fs) = 671ad9865cf186c69739279463e408504e89c072
SHA512 (NetBSD-6.0/amd64/installation/floppy/boot5.fs) = 1a4329afefd19b3dbf37a12cb287aa771dfabdeeaf6b0302158eae0cde68d970a94a9643b4b18798654473c6eb577c8e555188a7c7776e6cccc05241a7b47c7f
RMD160 (NetBSD-6.0/amd64/installation/floppy/boot5.fs) = 9e7000ef7b4a971936e74eb1858b0f74816d42f3
MD5 (NetBSD-6.0/amd64/installation/floppy/boot5.fs) = 52824ec1c65bc87f196fe4fcd1887cb2
SHA1 (NetBSD-6.0/amd64/installation/miniroot/miniroot.kmod) = dabc250e1cead594f5dea847da52d9ba2933b310
SHA512 (NetBSD-6.0/amd64/installation/miniroot/miniroot.kmod) = a4dd87defdb602de2329e7397c503b075c1c373af8a0547e843040029aaf34595e117f11f9e184a69765386ed66283f3a9c994d2159e473792b4f49d45991cdb
RMD160 (NetBSD-6.0/amd64/installation/miniroot/miniroot.kmod) = 92d16ace79c31254694e0b401abb19e659f7879f
MD5 (NetBSD-6.0/amd64/installation/miniroot/miniroot.kmod) = df6ec580ac9753bd17f1e9aaada0be34
SHA1 (NetBSD-6.0/amd64/installation/misc/pxeboot_ia32.bin) = 96287819b198b35ceaf13400e6852a472b0e86e1
SHA512 (NetBSD-6.0/amd64/installation/misc/pxeboot_ia32.bin) = cad69e2c71ec9879adbcec03468172aa1c2b7dd2ae4d4b5315a7013a39a24667faa8610ab663d96d5d8229e74f8150a0afda03d3e308a54e67fb555bffba821e
RMD160 (NetBSD-6.0/amd64/installation/misc/pxeboot_ia32.bin) = e226d78864e488c284063c1eeca1431babd8934e
MD5 (NetBSD-6.0/amd64/installation/misc/pxeboot_ia32.bin) = a6cf18820ed6a4d823ae117148272279
SHA1 (NetBSD-6.0/amd64/INSTALL.html) = 2ba66a1c4d70a9376ae64ba56e97acf97cb14064
SHA512 (NetBSD-6.0/amd64/INSTALL.html) = 6fbece25fed2ab136a77f592975808e919260963a9be83b2d4ddbee6b9b24319fb08c6316f262fe208aba81e50d1156db6ad8e36974fecd6955608d807da4667
RMD160 (NetBSD-6.0/amd64/INSTALL.html) = e4976765905456206ba0906f3d7aee383e80b5dd
MD5 (NetBSD-6.0/amd64/INSTALL.html) = 9a1d1f0005ff232d9501ebe159d122b7
SHA1 (NetBSD-6.0/amd64/INSTALL.more) = d1da705f0adc32b67801928824379ea45ddb70fd
SHA512 (NetBSD-6.0/amd64/INSTALL.more) = f5e6d604726edc5759dcc63e32c046fc0f952b9e4357e331b3864d1074954db46f0a7295b63f6f3a2c5ecda4ad3fcfc568a6dfc7dc91db5559a768ad2f6fd6dd
RMD160 (NetBSD-6.0/amd64/INSTALL.more) = 4b06e7cde85cdd8fef618f254e8a9136ff25e766
MD5 (NetBSD-6.0/amd64/INSTALL.more) = c1d511b8ec407155635d1df1d3698af9
SHA1 (NetBSD-6.0/amd64/INSTALL.ps) = dc0b32b40a813365cab508de06d4280a4ebe8496
SHA512 (NetBSD-6.0/amd64/INSTALL.ps) = 62ad1a41b0101222de2a4aea45c32ed7a02a4b06d1fd86b1c4d35f6ac5605a3562ca21b815db8e27d67e4584a5e155bbf8d5dda842e165379f4efd962c19a4a8
RMD160 (NetBSD-6.0/amd64/INSTALL.ps) = fc6b89c1155481439b4a4a66357b925227720d97
MD5 (NetBSD-6.0/amd64/INSTALL.ps) = 833ee159c8aeb1456ad514df2b4da716
SHA1 (NetBSD-6.0/amd64/INSTALL.txt) = fd054d29f49e793dec119508a0466e243d54d05d
SHA512 (NetBSD-6.0/amd64/INSTALL.txt) = c5e9a67c9bcd990c16a13390db60ff432f4640434234dc69a412d2ebd57fd333ace7ea59dc5b3acc815ceef9077cd76cb26dc5d6fcc5ebaa5fc95a511928bd21
RMD160 (NetBSD-6.0/amd64/INSTALL.txt) = 271a2919d0ff6fddc1bb771716e67a7319795f7c
MD5 (NetBSD-6.0/amd64/INSTALL.txt) = 2de264a9a927f1f3513674ec16ac4d45
SHA1 (NetBSD-6.0/amiga/binary/kernel/netbsd-GENERIC.gz) = ad3ce973bb574d84022cbc071997dca9377e1be7
SHA512 (NetBSD-6.0/amiga/binary/kernel/netbsd-GENERIC.gz) = 23e227becd8364657d767347b251ca8f47efe9397b9a0cadec90aeb9e90372dc49f143026ca678d933acde69c9be780b50a589ce0cf164f61af8d4eb641f5c64
RMD160 (NetBSD-6.0/amiga/binary/kernel/netbsd-GENERIC.gz) = 65ccd49cd01c0981f4b4a65914d59731492a9876
MD5 (NetBSD-6.0/amiga/binary/kernel/netbsd-GENERIC.gz) = acb579f6a97b9a5909bebf16f2742b0a
SHA1 (NetBSD-6.0/amiga/binary/kernel/netbsd-INSTALL.gz) = c3904eede4df412ea9a7db90f34009cdaee611dc
SHA512 (NetBSD-6.0/amiga/binary/kernel/netbsd-INSTALL.gz) = 6087a06debbf2f6820f5c450495680b0f5356d2e5036e1a7787b02f9cd5c8c43da24069ac0cb14bf5b09ca5ff122905d0c26c044c361c27202b4aa9a561cfeba
RMD160 (NetBSD-6.0/amiga/binary/kernel/netbsd-INSTALL.gz) = 8dcb009d9495e2c1f768e7900b28a263cafb1610
MD5 (NetBSD-6.0/amiga/binary/kernel/netbsd-INSTALL.gz) = 635f24c98bd4bd94f42bf3daa9090b1c
SHA1 (NetBSD-6.0/amiga/binary/kernel/netbsd-ISDN.gz) = c91b1e76e6f8d7b327728eb7110995968e56cb80
SHA512 (NetBSD-6.0/amiga/binary/kernel/netbsd-ISDN.gz) = d1ba0cda42d6eaf9815f39a52429767c00769d671faaf7144c830f1d06dc5571c39e0e823a7764fbdfaad10a1f516385ed1d3d54c4053a913e8399e3a1d37084
RMD160 (NetBSD-6.0/amiga/binary/kernel/netbsd-ISDN.gz) = 32dfb7318589e9e2445f781c1344f7b418b1b9e3
MD5 (NetBSD-6.0/amiga/binary/kernel/netbsd-ISDN.gz) = efdc155fef394563fed3a0c32d90a291
SHA1 (NetBSD-6.0/amiga/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/amiga/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/amiga/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/amiga/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/amiga/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/amiga/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/amiga/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/amiga/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/amiga/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/amiga/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/amiga/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/amiga/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/amiga/binary/sets/xcomp.tgz) = e13548b8974f5d55e98caf17b2b4bbbcf2430370
SHA512 (NetBSD-6.0/amiga/binary/sets/xcomp.tgz) = e4b13c798df04ffef88e72100e2db5af400e5ae255cd7a1ca2907736b8138a59045b6f3bd8531b5753d0b87eadb1b5f801043a929bb25d1611e4be693c1da954
RMD160 (NetBSD-6.0/amiga/binary/sets/xcomp.tgz) = 3b1f4e26ddd57e69afd9709553c1bf70d6e29ea0
MD5 (NetBSD-6.0/amiga/binary/sets/xcomp.tgz) = 894bb9aa05b164d9923682165d708335
SHA1 (NetBSD-6.0/amiga/binary/sets/xetc.tgz) = d2ca34bd668e108cbbfd4b45720823663339f8bd
SHA512 (NetBSD-6.0/amiga/binary/sets/xetc.tgz) = 431ca465f300e7eb5d4dbd44c43e412a296fb4d3129217e752f86a720e8a8bdf736c9add05a45518a4f3e6cffa32381ef61e9186d360b68a5f8c64d87e385fb5
RMD160 (NetBSD-6.0/amiga/binary/sets/xetc.tgz) = 4c5d664679856b02b694534cd6afdf20597e6d53
MD5 (NetBSD-6.0/amiga/binary/sets/xetc.tgz) = 72bd67556ac14ea0ce147374af0ba624
SHA1 (NetBSD-6.0/amiga/binary/sets/xfont.tgz) = 24ca4cb77a59246b1b05e0a91fed21c63259d12b
SHA512 (NetBSD-6.0/amiga/binary/sets/xfont.tgz) = 8dc297fb6205a995767e68e77cc21169234dfb66a86bbcd2e1828fd32efeda0a25c9f2a9b1a39b97fe8d7b24f8cd5cf6160859af6be2259876f39ee4e76c405b
RMD160 (NetBSD-6.0/amiga/binary/sets/xfont.tgz) = d2f134959940e6667aca185f01e388c05dc6bc85
MD5 (NetBSD-6.0/amiga/binary/sets/xfont.tgz) = 3c96c1f0a13cd7d277ca35622ac9c058
SHA1 (NetBSD-6.0/amiga/binary/sets/kern-GENERIC.tgz) = 295ba9f28d2b11af8b7ea93c0b4cec03e40002a7
SHA512 (NetBSD-6.0/amiga/binary/sets/kern-GENERIC.tgz) = b762b6592db8439df6c8d7cfd26f509425c779dcb3d27face8aea6c50dec314c15124b6623152969d5a0998c6991050c5c18a4f547ff5027c8ef56e4d4b5baf3
RMD160 (NetBSD-6.0/amiga/binary/sets/kern-GENERIC.tgz) = 566f5c6520f253fbd5c7043931b275811853e289
MD5 (NetBSD-6.0/amiga/binary/sets/kern-GENERIC.tgz) = 3ef7c2157fd42c60a0c1c4c898855219
SHA1 (NetBSD-6.0/amiga/binary/sets/base.tgz) = 863379dd24489824d8140df08e3f5ca5bb63c35a
SHA512 (NetBSD-6.0/amiga/binary/sets/base.tgz) = 845fec5935b19d810896af3451bff4f2eb6b3b02b285e8e7651c3be32c60d2abacbbb614e3c11b96cf33497fbe6505458f9a292bb0393c54a764449b9b5dff65
RMD160 (NetBSD-6.0/amiga/binary/sets/base.tgz) = 5c87d3799986d70963b0e02416ff7648602d31de
MD5 (NetBSD-6.0/amiga/binary/sets/base.tgz) = c8b7075f670666c9492a46c44e4632de
SHA1 (NetBSD-6.0/amiga/binary/sets/comp.tgz) = fad62d4d7c47d956a450773b817a838f30497212
SHA512 (NetBSD-6.0/amiga/binary/sets/comp.tgz) = c8c85b7a59df5930e8cde1715b03c2ae7646d40902e2c402292308f8eba81132a155120215c763a68dd7f37ef489d786ee73911849b69cc9c9e3ba1e86de1bb9
RMD160 (NetBSD-6.0/amiga/binary/sets/comp.tgz) = 46162e89fb8c66089ccf0523e3b3483834284284
MD5 (NetBSD-6.0/amiga/binary/sets/comp.tgz) = f107b1d62d1535a4dc1052752837aecd
SHA1 (NetBSD-6.0/amiga/binary/sets/etc.tgz) = 7a2e1bc8aa53aeea37bb71ae598886da762888c4
SHA512 (NetBSD-6.0/amiga/binary/sets/etc.tgz) = 80d275882cc7df12a3ba35a9e101e1f4597c3cd16864026013be90f0a8e8ef2ab405b5b29dc39ed3e114fac9986dc05efb661cc4361cf4d05d4e6cb25462ae45
RMD160 (NetBSD-6.0/amiga/binary/sets/etc.tgz) = 3b71a2cfee1368a862a75f44d249b85059f34bc6
MD5 (NetBSD-6.0/amiga/binary/sets/etc.tgz) = e8fba82eea6cdabed070234fde1acf2b
SHA1 (NetBSD-6.0/amiga/binary/sets/modules.tgz) = 412c02a1fe2bc42ffaa13e3b5cfe30ab89ca8066
SHA512 (NetBSD-6.0/amiga/binary/sets/modules.tgz) = 0fcda39f953bfbeaae0480d14a4e36724296ff88185985aec0863d5730048da38dd37f5f609e4b085a4776fbf139e8ca3acd693e7a5827f5fa7f51b2006b1447
RMD160 (NetBSD-6.0/amiga/binary/sets/modules.tgz) = 44f4e35271aa128a88c14913b82edc82f93ffb57
MD5 (NetBSD-6.0/amiga/binary/sets/modules.tgz) = c0a7fed3fc40edbb00bc24975c10c011
SHA1 (NetBSD-6.0/amiga/binary/sets/tests.tgz) = 3d5677828808ee833c1c65cf5ff5d1cf1f66611f
SHA512 (NetBSD-6.0/amiga/binary/sets/tests.tgz) = 4d868f712f2e0e3e9b56e6f346934b71eb174d97ff197883f3e3d4923c3dabda651bb3fa3d172fcd32ee5d07b659e7639cff6d9c9450e605174d8a09a6ac730a
RMD160 (NetBSD-6.0/amiga/binary/sets/tests.tgz) = 662e3dcc455114628676304b277a852a49597104
MD5 (NetBSD-6.0/amiga/binary/sets/tests.tgz) = 84a17f9c5c7bb88e34f42e3a7cbdc490
SHA1 (NetBSD-6.0/amiga/binary/sets/text.tgz) = 8153acee4263203531a66cdf71215db94d745bd9
SHA512 (NetBSD-6.0/amiga/binary/sets/text.tgz) = 3ca801f17e44f701543140841123196e132f920d7ac5859dc2a5133da08b73f4d730cbfc62fee5ba9bac6b0d96cf4985b951aad33cf59d5470d7b99cadaf51dd
RMD160 (NetBSD-6.0/amiga/binary/sets/text.tgz) = 5820be807d773054a2c25d0c0009141ddf4a3d5c
MD5 (NetBSD-6.0/amiga/binary/sets/text.tgz) = 6640874a003b751e38fe8ecd14f001d7
SHA1 (NetBSD-6.0/amiga/binary/sets/xbase.tgz) = c2df6e9a1c3bdae97964577118ae358efe3e3f38
SHA512 (NetBSD-6.0/amiga/binary/sets/xbase.tgz) = 4751bb14d6a5da31d95ab7e5cef52fcba99b8e13b8379f7fcff8aafe8116c4a036553adf4e6e9bed1642cf39b8e3de6a4e29a37ea48d1363dad853ef3d198604
RMD160 (NetBSD-6.0/amiga/binary/sets/xbase.tgz) = cbeb269f4126adf0e3e9894888047f70a42f1d6d
MD5 (NetBSD-6.0/amiga/binary/sets/xbase.tgz) = 03727cb263b9f50dfec22ac09d5d0543
SHA1 (NetBSD-6.0/amiga/binary/sets/xserver.tgz) = 3e746f532be1a2ea1d5d38894bf0be416021412a
SHA512 (NetBSD-6.0/amiga/binary/sets/xserver.tgz) = b90fcec5aea23879b3c39aa516e1c658064d35e8dc8172fac9f722b9c08f71a73f3a33d84dc618c88bf368156e9e4bf220f3cbd110f66bd213838fef73af50f3
RMD160 (NetBSD-6.0/amiga/binary/sets/xserver.tgz) = c50619861d88fbc22413aca3bad38cb33065b590
MD5 (NetBSD-6.0/amiga/binary/sets/xserver.tgz) = 5ac22d2440ed272bbebec827a0134acc
SHA1 (NetBSD-6.0/amiga/installation/miniroot/miniroot.fs.gz) = d83dc1458c18c0625f767f00066eff80547ee71f
SHA512 (NetBSD-6.0/amiga/installation/miniroot/miniroot.fs.gz) = 415b4c824c4efdd1c8fd597c6c8222c26d77e56f6f36bcb65d9e7eb3882c20f3c452f05b4541a0832590e53a67b12423d21196626fcc7c1cad4992648a98e136
RMD160 (NetBSD-6.0/amiga/installation/miniroot/miniroot.fs.gz) = 4cf219c14c3c367d95ff4aa4cf04dba379ee16c7
MD5 (NetBSD-6.0/amiga/installation/miniroot/miniroot.fs.gz) = 1bb2a3735737d2af8155407157cb2120
SHA1 (NetBSD-6.0/amiga/installation/misc/loadbsd-2.14) = 32e902d420c2ba8498854bd26387369b09efabb6
SHA512 (NetBSD-6.0/amiga/installation/misc/loadbsd-2.14) = 3fe58857d83d88f1fc7f6e29e8ee079be25326abad3a25ba5f64e3429f22f08ef102551e15c4275fe973d5536a5d459edea384fdd2293f68fe6d293801fba85c
RMD160 (NetBSD-6.0/amiga/installation/misc/loadbsd-2.14) = 17a84971cee95ff8bf1b64d5071bd1fdbdbc5a99
MD5 (NetBSD-6.0/amiga/installation/misc/loadbsd-2.14) = 1f841716409b236df90d9c5d63332edf
SHA1 (NetBSD-6.0/amiga/installation/misc/loadbsd) = 2c8e314ee812e159ea479bacf09c47a2a4911e17
SHA512 (NetBSD-6.0/amiga/installation/misc/loadbsd) = 881d3187366e2736e65e22bae93f24afd0b34b841b733898d3d172d98084f1729b19a6277445601da35471e3583e98b86008262b1a77dbb8bcc37e0c03d07b32
RMD160 (NetBSD-6.0/amiga/installation/misc/loadbsd) = 00249c76e9a57ed5e58efbdc1001f782595c3918
MD5 (NetBSD-6.0/amiga/installation/misc/loadbsd) = c1cc983600053a7de29feb09d703754d
SHA1 (NetBSD-6.0/amiga/installation/misc/device-streams.tar.gz) = 5b96fab4bbb49a985481dc5c1e2830e994bd88ec
SHA512 (NetBSD-6.0/amiga/installation/misc/device-streams.tar.gz) = 88e35b476d60dd988e8056263fc4f86893c5a7c2c3125934db8d728e3b679554b229b2adc3babf8c2ff21af46530eea1ff30e4b6af550b331ad22bd29e18434f
RMD160 (NetBSD-6.0/amiga/installation/misc/device-streams.tar.gz) = d0444d1e967a624fde15e4526f9c7f673566492d
MD5 (NetBSD-6.0/amiga/installation/misc/device-streams.tar.gz) = 6a3eed6278587660ed543c7449ee571e
SHA1 (NetBSD-6.0/amiga/installation/misc/runbootblock.README) = 2d62005b98533a4fffd7d0445ac8cb7c5ac11c59
SHA512 (NetBSD-6.0/amiga/installation/misc/runbootblock.README) = 9f071ff7ac0d2eabc77d43faeee48cba3a1883c957b046fad4d7d3237f41d2752a1e010a5d9f5324f90b81bc9968afe425d320e6742ac8f5e081056d5e133baf
RMD160 (NetBSD-6.0/amiga/installation/misc/runbootblock.README) = 872a2169c248b016d6782a6b9c4208c67a346d12
MD5 (NetBSD-6.0/amiga/installation/misc/runbootblock.README) = c60fe181cd9b255e294bfeb4be411181
SHA1 (NetBSD-6.0/amiga/installation/misc/loadbsd-2.16) = eb660b987186a4e3a4eccb740dbb9e061a4b7216
SHA512 (NetBSD-6.0/amiga/installation/misc/loadbsd-2.16) = 0f4565e2de70eaaf1d973da2b4a988e794b4608b2b0fa9c84dd05c82ade92fc4b75339e94e4a593e051ab3a364611fd7dbc2b48dccd4906f7b42fb665f7d711c
RMD160 (NetBSD-6.0/amiga/installation/misc/loadbsd-2.16) = a071897afbdd11655a21df5a89c6e49b6e8a3574
MD5 (NetBSD-6.0/amiga/installation/misc/loadbsd-2.16) = 8e67fb434887e942fe63cf308195b18f
SHA1 (NetBSD-6.0/amiga/installation/misc/rdbinfo) = e413a8a5631dfedb4b1634c709836d1f92d9b08c
SHA512 (NetBSD-6.0/amiga/installation/misc/rdbinfo) = 912ddb9bd1ba1af07e1072fee9a09f22124792abfc8f4f178a5df66dad17c39211c52d2a581111338a523f17a0dd58315d798659e901f14084d62a8c890e0a44
RMD160 (NetBSD-6.0/amiga/installation/misc/rdbinfo) = 8fb8b41e39f6c2d48b6b46be0433dfc45c353621
MD5 (NetBSD-6.0/amiga/installation/misc/rdbinfo) = 027c4bfce639d3cb88d617b0e85c0c4c
SHA1 (NetBSD-6.0/amiga/installation/misc/runbootblock) = 598d688badd592a838dc260ce36b84c7835e72c0
SHA512 (NetBSD-6.0/amiga/installation/misc/runbootblock) = d4cf9fe94244f1a832a6e474ff777ae7b901e86e0aabbbceb356c42f533dadb370686080b8e64993b9e07502310a4140fd393db5ddbdcd7a9e44c86b20f4c0a2
RMD160 (NetBSD-6.0/amiga/installation/misc/runbootblock) = 2457b64f16ee58d22bacdf407ef35a1e9d244e1f
MD5 (NetBSD-6.0/amiga/installation/misc/runbootblock) = d4d059566ad067e4d60259b389a5a9c3
SHA1 (NetBSD-6.0/amiga/installation/misc/xstreamtodev) = eaf846a0d81754fb778b070926c495720cecffac
SHA512 (NetBSD-6.0/amiga/installation/misc/xstreamtodev) = 873e7ffa5686a69b4a0ad68626f5e32176c03fc6231072531b36af3d0c0c055ced63e6c7f5cb0f3c6568f47720f31b76140aeede3f3dd9ddc8b639c6b3bdd5bf
RMD160 (NetBSD-6.0/amiga/installation/misc/xstreamtodev) = 16910f7f243ca8ae74b8ff90d89163edbd4c3fbf
MD5 (NetBSD-6.0/amiga/installation/misc/xstreamtodev) = a9e2e97f7945cf34da63aef17bed0f3f
SHA1 (NetBSD-6.0/amiga/INSTALL.html) = 42ae087edd4c644275cb6a351bfc0801a23f7b41
SHA512 (NetBSD-6.0/amiga/INSTALL.html) = e3f4389bff567192811d94c90b1df6d35d379774173ee935b2ebacf6ebc04edee9d270a71cf1e312bb52cfdfdf3ea7fa50f3cc6058bef88051f6da2953faefe0
RMD160 (NetBSD-6.0/amiga/INSTALL.html) = 941e3e7501290a6a596c10bb9e9dce71fd68f01f
MD5 (NetBSD-6.0/amiga/INSTALL.html) = ca5a2dcf7c5a927f0f89edabd084d59f
SHA1 (NetBSD-6.0/amiga/INSTALL.more) = 2ef7c56d3b3789f858d82b96f8b955801feb402a
SHA512 (NetBSD-6.0/amiga/INSTALL.more) = e1f87c081f267bb5bda5f1f07467e4081b3289cc1b5eead1ce26e954acf6704ad86ac95fb8958130f0a5b198eca80b967c7089cdc35ba4399cc79ab822df2ec2
RMD160 (NetBSD-6.0/amiga/INSTALL.more) = 55a90159f299abe126ea37f0becea2f692fb1595
MD5 (NetBSD-6.0/amiga/INSTALL.more) = 1357d2cc09f7d5429c5790dd32592032
SHA1 (NetBSD-6.0/amiga/INSTALL.ps) = b4a7420ef6502bd98009c6f7b79053a076881f28
SHA512 (NetBSD-6.0/amiga/INSTALL.ps) = 295f43cd3ba4b64deec0652bab61802cc37a7c3e75cab305cd0e12a682822a137abcd0915115709e1147e952b87bbb128e53251625288ba58068a52e491dd0da
RMD160 (NetBSD-6.0/amiga/INSTALL.ps) = 07a4ea8478343a0f6a1e935eca5133ab116b5779
MD5 (NetBSD-6.0/amiga/INSTALL.ps) = f07315266b0fd45fe58f107dda2177f8
SHA1 (NetBSD-6.0/amiga/INSTALL.txt) = 6f5c90d805c69d5c75efb96b3b579d49bb123a6b
SHA512 (NetBSD-6.0/amiga/INSTALL.txt) = 2fba02db405672d44df143421547a2291272a56f6d6e37b5da1f0396739ac55c695178c2e115a5af8574d6c323c8adc2747073e799b59d4e886191ce1baeb5cd
RMD160 (NetBSD-6.0/amiga/INSTALL.txt) = 18db9a1a6e4af5935ed15d37c218b0309309c95c
MD5 (NetBSD-6.0/amiga/INSTALL.txt) = 23deef4d8fd55d404915f004ba0fde9b
SHA1 (NetBSD-6.0/amigappc/binary/kernel/netbsd-GENERIC.gz) = 1fb8ac599ecb67083a0505f7767600b5a00e6164
SHA512 (NetBSD-6.0/amigappc/binary/kernel/netbsd-GENERIC.gz) = a41b53fecaa5077d80a2a6191073785c87ef703a4cd17e45ea1064e64c923fbf9d97edcc1c4e58ed205ea85f070f80a992198943a9384f4af473a11058c56d4f
RMD160 (NetBSD-6.0/amigappc/binary/kernel/netbsd-GENERIC.gz) = b9f459e21e09ac314b2a02a2ec1a5ef0841af3d0
MD5 (NetBSD-6.0/amigappc/binary/kernel/netbsd-GENERIC.gz) = 8c59d3085078659f9fd319d69ea1bc0a
SHA1 (NetBSD-6.0/amigappc/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/amigappc/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/amigappc/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/amigappc/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/amigappc/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/amigappc/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/amigappc/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/amigappc/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/amigappc/binary/sets/kern-GENERIC.tgz) = 8cfda7d67d00c0344fc8b7e15c14367c9dd3c0d0
SHA512 (NetBSD-6.0/amigappc/binary/sets/kern-GENERIC.tgz) = 00ecdc9511f389035fdd0d5786ec5b16ea52722cf390b75b11c0d0c5504d96bdfbaf6b19237e752aba9d530ed80cdaeadf18a7d0079c22a1560644fa8fb2ffc9
RMD160 (NetBSD-6.0/amigappc/binary/sets/kern-GENERIC.tgz) = 0508773016c4dcf8b88d238235c66fc61785c0fe
MD5 (NetBSD-6.0/amigappc/binary/sets/kern-GENERIC.tgz) = ada4ef952407ea1d320765955f626aed
SHA1 (NetBSD-6.0/amigappc/binary/sets/base.tgz) = 69692adf17c4631804d6e78e96868bf22a0f80e4
SHA512 (NetBSD-6.0/amigappc/binary/sets/base.tgz) = bc3f89b3201c184f38bed2dcd3e74ce01a8c9b5cdba1065bfe5bc0eadf0ed71fd73bcb9d0add77ab7d2cfc4848f40c482b0e2cbf50771577716f8a344eda5a6b
RMD160 (NetBSD-6.0/amigappc/binary/sets/base.tgz) = 3cb20928324c3bf484a05a04d47afa4f587d5d52
MD5 (NetBSD-6.0/amigappc/binary/sets/base.tgz) = 00d35c074b0f69e81494f42cd43ccecd
SHA1 (NetBSD-6.0/amigappc/binary/sets/comp.tgz) = db98cb19c33216dafdd0ab26cbfa2a4f0cc685e7
SHA512 (NetBSD-6.0/amigappc/binary/sets/comp.tgz) = 7dc6b1ed24deecd9f29aae5867bd2d89c9df3f443d522315f541eae77fde420d15b63233694043595a0f12491af5e24699598acc45ed05d9622911869f79ceb9
RMD160 (NetBSD-6.0/amigappc/binary/sets/comp.tgz) = 776b79b7b476cc3b4caee7712070864d35779f66
MD5 (NetBSD-6.0/amigappc/binary/sets/comp.tgz) = 385fe9939b6bb28a37cdf9cc3958dba8
SHA1 (NetBSD-6.0/amigappc/binary/sets/etc.tgz) = f424a289be42a94c875757d9f83756d7e9286f20
SHA512 (NetBSD-6.0/amigappc/binary/sets/etc.tgz) = b736da20bee72bb1962acce7cb8cd51e851f541ec79766d2258f1322ae843b0879208a6f69d6dda9ecc74a4bf7739fed626b8205432dd63f630b04803418307c
RMD160 (NetBSD-6.0/amigappc/binary/sets/etc.tgz) = 0f163aa3a9450f45b40760786373089e52dae0bd
MD5 (NetBSD-6.0/amigappc/binary/sets/etc.tgz) = c2cb43cbeb231d24b3864c7882da8edb
SHA1 (NetBSD-6.0/amigappc/binary/sets/games.tgz) = d10c319a009f9859de37a5d5bb994d29669dce16
SHA512 (NetBSD-6.0/amigappc/binary/sets/games.tgz) = 59e00cdf1b42e285db9cd29ab195b693f1989cd1480e644787307ea722a1a695e70708ad8fa6d0afbbb51de4ca8a4b81b550e2f1745d0140a47e1b6a7002a236
RMD160 (NetBSD-6.0/amigappc/binary/sets/games.tgz) = 0baf65f6198b6f1c40960b30424a3ad44c1eca2f
MD5 (NetBSD-6.0/amigappc/binary/sets/games.tgz) = 304dc86d68c23c7f30c30e4f092b6982
SHA1 (NetBSD-6.0/amigappc/binary/sets/modules.tgz) = 2a05a8cf525b6b0323f54840bb42e069a36f1988
SHA512 (NetBSD-6.0/amigappc/binary/sets/modules.tgz) = fc33e9c609d38775e7004d2715158178d08508f8a890831270191b7f32fae8a2d2b56e74e59d4f5d37574827cf85eda28e15f88ac1544a1ceec008105107dcb5
RMD160 (NetBSD-6.0/amigappc/binary/sets/modules.tgz) = 931d81973e92a161da6468aeeb0b7a486fc96511
MD5 (NetBSD-6.0/amigappc/binary/sets/modules.tgz) = 13f7ee97c852a71b51c91dd8f659b202
SHA1 (NetBSD-6.0/amigappc/binary/sets/tests.tgz) = eee320cd38b5396a305b912cd43ad4f40dcb0e98
SHA512 (NetBSD-6.0/amigappc/binary/sets/tests.tgz) = 3c2377b75fb69924c646904f67e3762bddbd5fe40214df319d205f412e7377e7695d5b648545f22e9b42696cb3fd3ddb9bde4ed7b3ad1d48ea32fbbf13356354
RMD160 (NetBSD-6.0/amigappc/binary/sets/tests.tgz) = dd0ea2b9c6f23f409ced129c5dc847c582b572fe
MD5 (NetBSD-6.0/amigappc/binary/sets/tests.tgz) = 2f0dc32c95f23026c06b50c946dce965
SHA1 (NetBSD-6.0/amigappc/binary/sets/text.tgz) = 7b28a25a69ba9d2870c31c302be38d0c12b11486
SHA512 (NetBSD-6.0/amigappc/binary/sets/text.tgz) = 77dcaa1c134ac4f346bf782c5c20ea4304b42a15a60a1321e944e59665946d1b3ddfcd9eba919f6ea4ca947dd8df63a9162adfea363e01758ffa5bf8e7f49e12
RMD160 (NetBSD-6.0/amigappc/binary/sets/text.tgz) = 438d720340cbd19f86ac8449dd705d125958cdbf
MD5 (NetBSD-6.0/amigappc/binary/sets/text.tgz) = 3fc18e1ff4fc50d561b478b39fe7f855
SHA1 (NetBSD-6.0/amigappc/binary/sets/xbase.tgz) = 11ec98aff57d5a8f8cabcc0a41eb7e41c2cd750e
SHA512 (NetBSD-6.0/amigappc/binary/sets/xbase.tgz) = c9c39d05fa60bc2a0e1a4aed89ecfafa47c203652f23b971fa9b3d23df291fbeb4735dd4621a81a46eff276c80410ee36492162e5282bf19bc9df4bf889d4fb8
RMD160 (NetBSD-6.0/amigappc/binary/sets/xbase.tgz) = 4d1a32acb55c107327f1605a6a7d3379e3e3fbc1
MD5 (NetBSD-6.0/amigappc/binary/sets/xbase.tgz) = 0a959da0b8e03e2686da275c8311addf
SHA1 (NetBSD-6.0/amigappc/binary/sets/xcomp.tgz) = 5a450d9cb090214e14b45d59ee87d7786da4afcd
SHA512 (NetBSD-6.0/amigappc/binary/sets/xcomp.tgz) = 2478cf90ce45e16d8854bfc2aad28d102aafb53f9b2fd39756bff0970c0546f52a3d31be85e581ae78e982b06f17a8de77eea776d7d8e0770e00a0e198c06bbe
RMD160 (NetBSD-6.0/amigappc/binary/sets/xcomp.tgz) = 82323c0e4b56e79891087934eed15d1472646bec
MD5 (NetBSD-6.0/amigappc/binary/sets/xcomp.tgz) = 096eaba339adb8b492dc1ce5de1dada0
SHA1 (NetBSD-6.0/amigappc/binary/sets/xetc.tgz) = 98fd5a6cb8eab73e8453afe8626198945e0cd1e4
SHA512 (NetBSD-6.0/amigappc/binary/sets/xetc.tgz) = 13b0c70af019c1e2bf14c8df794f497c317cc94e23012f61b88e7386327cd43c6a7b00a04ed4d65975849db9978fa63b401a2d1a49f0d476e17e4d5123d66e8b
RMD160 (NetBSD-6.0/amigappc/binary/sets/xetc.tgz) = ce7639ef1eebd525f0f6631e4aa80a5d44a9ba5e
MD5 (NetBSD-6.0/amigappc/binary/sets/xetc.tgz) = 2bdf6c00be2e1b9420a8d6fd6ab9e285
SHA1 (NetBSD-6.0/amigappc/binary/sets/xfont.tgz) = f87acd8ea1747d825ad1038f8984bfd9cc5ed3c8
SHA512 (NetBSD-6.0/amigappc/binary/sets/xfont.tgz) = bd4c6c5913572cfaff4c3cdd66d2e745605d4f8c70f46e25951c109700c4a05a4b8d1202beb90b7b0cb5ba7b37bdcb3bc31a5ddba8737b7133fa3c2239312f62
RMD160 (NetBSD-6.0/amigappc/binary/sets/xfont.tgz) = 6ff6daa8eff23cd8f49f60ac7adf82afaf029a02
MD5 (NetBSD-6.0/amigappc/binary/sets/xfont.tgz) = e9eb507d880069bb39bca1e12ca61f05
SHA1 (NetBSD-6.0/amigappc/binary/sets/xserver.tgz) = 5dca2b2f9ebbdd6d659e06bdb69987fe02278968
SHA512 (NetBSD-6.0/amigappc/binary/sets/xserver.tgz) = e2e87ac352cbadc9e0590b7ddc402de6dc4ee3fb26a504265aa1a706ef53135fb80d0d3e149e94825485fd121add254d68a88ead521949f55da7d9e252192575
RMD160 (NetBSD-6.0/amigappc/binary/sets/xserver.tgz) = 94ab84cf7ee91393ef0630b7b3110a4846940dfa
MD5 (NetBSD-6.0/amigappc/binary/sets/xserver.tgz) = 6ca7773bb022c8842ce6a0a4a0f23c80
SHA1 (NetBSD-6.0/arc/binary/kernel/netbsd-RAMDISK.symbols.gz) = f4e479e3fec7462469600527ae1b9aecd1c86bcd
SHA512 (NetBSD-6.0/arc/binary/kernel/netbsd-RAMDISK.symbols.gz) = 8cd3e3d089a53b56fdd47ab4cda09b42bf12c92074699a3764f4fdbb23c58425d68f0ee8e862987386d9c8107e0e115daf39bd35058b4e51d93c4dc7d5756d04
RMD160 (NetBSD-6.0/arc/binary/kernel/netbsd-RAMDISK.symbols.gz) = 79dd220dcba4f30fd629e744993e3d8617d2c3ac
MD5 (NetBSD-6.0/arc/binary/kernel/netbsd-RAMDISK.symbols.gz) = 2ffba67d5ec969424a75c71e5f04c77f
SHA1 (NetBSD-6.0/arc/binary/kernel/netbsd-GENERIC.gz) = c34a668c8d951cd778d973a0066223bd4ea98133
SHA512 (NetBSD-6.0/arc/binary/kernel/netbsd-GENERIC.gz) = 47685372141fd6fae114fcbcc57016c6896f96705be116020802c6e6a7493cbea9ecf62dbbfc89c49f0f90d42aee1f06b9d6ed3eb679b0b1e3e2705c3ff0ec1c
RMD160 (NetBSD-6.0/arc/binary/kernel/netbsd-GENERIC.gz) = d46270739b1bc29e7e361805d4f23a28ced95975
MD5 (NetBSD-6.0/arc/binary/kernel/netbsd-GENERIC.gz) = 046973b3c11b35cb476d07db96419743
SHA1 (NetBSD-6.0/arc/binary/kernel/netbsd-RAMDISK.gz) = 01c3eb7eabe2423ff73da425627b361ff99bbe21
SHA512 (NetBSD-6.0/arc/binary/kernel/netbsd-RAMDISK.gz) = ae090ee3642160ed2ab38f6cc1487b545cce31320c8561bb4af5501dc757f24997798a7556a2048cd7f7e0da9ad3415e458273245317bcacafa0d9b76f4326a8
RMD160 (NetBSD-6.0/arc/binary/kernel/netbsd-RAMDISK.gz) = 9bd6a441f8fbbf01f78b536d75bb8cfdaa56f8cd
MD5 (NetBSD-6.0/arc/binary/kernel/netbsd-RAMDISK.gz) = 6103f4c89298eab52773ed2a6fa59de6
SHA1 (NetBSD-6.0/arc/binary/sets/games.tgz) = 27d9f98fecb3495e51939df9f107e81e527af26f
SHA512 (NetBSD-6.0/arc/binary/sets/games.tgz) = 46bb4e10313cfffaaf4f7fa2157474ef940f94bcc9f84b46f829c7615bbf45e4e01edee7b7efb84bc28561e0712c37cfc737f9c2a0d9a73b67c80c8aa0f3a9af
RMD160 (NetBSD-6.0/arc/binary/sets/games.tgz) = 947feea90ec2698a65d555cb78a29a001b086589
MD5 (NetBSD-6.0/arc/binary/sets/games.tgz) = 355f00978fed8e8da6a251e591be5387
SHA1 (NetBSD-6.0/arc/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/arc/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/arc/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/arc/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/arc/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/arc/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/arc/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/arc/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/arc/binary/sets/xbase.tgz) = 76468fcada5a9e26652aa9f0f9f3c94829a317e9
SHA512 (NetBSD-6.0/arc/binary/sets/xbase.tgz) = b2adcfc2dedef30b21b885c8fc757d0479541744a280a9997fbeaa9c881009ab4b0019e0001a540a8eb6db5f84db87b365e420bb2d4270af209821bdb080369c
RMD160 (NetBSD-6.0/arc/binary/sets/xbase.tgz) = 1e0074d03d9fbb3373e0a427b25cc7097bf6d3c8
MD5 (NetBSD-6.0/arc/binary/sets/xbase.tgz) = a8f29ec2e3768405942f97af3dd08ee7
SHA1 (NetBSD-6.0/arc/binary/sets/xcomp.tgz) = d8ac89f3f6ff20abb0adabc49227afdb74aa2040
SHA512 (NetBSD-6.0/arc/binary/sets/xcomp.tgz) = 4f03213fb061dae9421747209628d06ede6bbb71a7560a9e1a54796b907f1620146a33fdce52974227a21acedcc210b99e15c4fdd36432eba60c1041a92d8cee
RMD160 (NetBSD-6.0/arc/binary/sets/xcomp.tgz) = 6be5e72b9b57c9bdf8ed8ea127e2a73830feeecb
MD5 (NetBSD-6.0/arc/binary/sets/xcomp.tgz) = 59a7ea141587417be38ab1f6056cc780
SHA1 (NetBSD-6.0/arc/binary/sets/xetc.tgz) = b9f6b1a54ffab37b11f4c6e717b481cd087fb609
SHA512 (NetBSD-6.0/arc/binary/sets/xetc.tgz) = 72f35e65f3a90967e250a2006d500619b3824f7f5c3d7494554f153ec5cda5b28c445f28c34a1e4463f40da5855e8d4257726411796967fe37763d90cf7fe461
RMD160 (NetBSD-6.0/arc/binary/sets/xetc.tgz) = 1169c246f91189fd0c4ae86ac27a92f5c2edda3e
MD5 (NetBSD-6.0/arc/binary/sets/xetc.tgz) = 8f3c52cc5af1cafaf7b0d0f884a5b537
SHA1 (NetBSD-6.0/arc/binary/sets/xfont.tgz) = d06848571034f3b7fc975082f1de37b912acb18c
SHA512 (NetBSD-6.0/arc/binary/sets/xfont.tgz) = 66a1013c935bee5e7e5d24325a527522306bcded7ced63790f2e58831b3ed91176a1b9510b2159be63504d5167d48d534919927ffc6e225ae2bb2487f90a9ede
RMD160 (NetBSD-6.0/arc/binary/sets/xfont.tgz) = c9502843a5d4dd7e996e8d158c519cf7f8575cbd
MD5 (NetBSD-6.0/arc/binary/sets/xfont.tgz) = e812899176eac2bf65a6d949ac3f3424
SHA1 (NetBSD-6.0/arc/binary/sets/xserver.tgz) = ff805022f409202f3346c45475c7ec0f8187218e
SHA512 (NetBSD-6.0/arc/binary/sets/xserver.tgz) = f2bb3810a5572b519980126a3d934f63b5ab5c2fcf9fb8166c556bd3cc14cfaa151d582086bb1506ac9ac7e21d4e70a5cd5c3faed96212d8a14a0b5bafdcc4c1
RMD160 (NetBSD-6.0/arc/binary/sets/xserver.tgz) = d5ae336337cc77f7f55f542b3457918d6fc0efe8
MD5 (NetBSD-6.0/arc/binary/sets/xserver.tgz) = 92447a1560887d337d22e2deed0670a6
SHA1 (NetBSD-6.0/arc/binary/sets/kern-GENERIC.tgz) = a49dc5701671d377df81d0dc82ebca225e8aff4a
SHA512 (NetBSD-6.0/arc/binary/sets/kern-GENERIC.tgz) = 75bf73c2b22aac1948b5c0fd6ca0a0d240fc398baf7dfe4bbb155f44c8e3bf47d618fb5ee97215728baa471c25dddc285643064acbfed88fcc33f463ec36162a
RMD160 (NetBSD-6.0/arc/binary/sets/kern-GENERIC.tgz) = 49075dea0d02de7d508afb65fa4c4591f7ab1b0a
MD5 (NetBSD-6.0/arc/binary/sets/kern-GENERIC.tgz) = 3dd1b55b9d8aaa84ce5630c649d6d17e
SHA1 (NetBSD-6.0/arc/binary/sets/base.tgz) = 5c25c736260c32dc5e61db84d779f979421417c4
SHA512 (NetBSD-6.0/arc/binary/sets/base.tgz) = 7bccf72e05a792158f7aa8617b270d860221b935c3c092da8961b35277931500f5e6e8a572536b7371e66dbe7368b17f62512783240dce9aa2990af50c60fc1e
RMD160 (NetBSD-6.0/arc/binary/sets/base.tgz) = c36e7a0c39a18646de9363196aa56dd0c4a6ea6b
MD5 (NetBSD-6.0/arc/binary/sets/base.tgz) = 97e2836870812371a84cf44790cf6f16
SHA1 (NetBSD-6.0/arc/binary/sets/comp.tgz) = 52ac953130a36ae6bba2139c7892d6ccbab0f0cf
SHA512 (NetBSD-6.0/arc/binary/sets/comp.tgz) = a09590c181019592ffb4341620eeaaae1f25e971ee975ac337c96cdc5dc6ceb2074dcba6a5dcc723c30e5db3d32148146eb6c1eb38390d0162aff4d5071434c6
RMD160 (NetBSD-6.0/arc/binary/sets/comp.tgz) = e9c5ae0abd43a2ac743f62a498ad8bb0dde61ed2
MD5 (NetBSD-6.0/arc/binary/sets/comp.tgz) = 2eebbc2b1b8bbe21166a0d860bea9dd5
SHA1 (NetBSD-6.0/arc/binary/sets/etc.tgz) = cfa766b3b33eeae2aa28312e0ca6e434e2573487
SHA512 (NetBSD-6.0/arc/binary/sets/etc.tgz) = ef3806802fa813b1bb60674c032f31e5a140b89953aabe715d050e7c820413ae991de791e96d8aecaa1aea8989f6f0268726ad5dc62e42ef06ca6670fb6744ba
RMD160 (NetBSD-6.0/arc/binary/sets/etc.tgz) = 395630144133efd05b0fbe596773c5ea4218aa8c
MD5 (NetBSD-6.0/arc/binary/sets/etc.tgz) = af810d6fb839f2953bba9844501cdcab
SHA1 (NetBSD-6.0/arc/binary/sets/modules.tgz) = cfd582b6f65203d1916ed6655d02f5aa6ad5ba88
SHA512 (NetBSD-6.0/arc/binary/sets/modules.tgz) = 86be394fe232856d0de502fb1569fd03de91a81ad782389414ed75618d63d97d8abebc14fd44843414d865071f51940c248d076a57524e8ef91c791ee98620c9
RMD160 (NetBSD-6.0/arc/binary/sets/modules.tgz) = c63e9a0caaf005f8c03275c8faf582d95b3c6700
MD5 (NetBSD-6.0/arc/binary/sets/modules.tgz) = cd4a56dcabef37fbf7a3fb7ee95ea738
SHA1 (NetBSD-6.0/arc/binary/sets/tests.tgz) = 25fca6e0ee3c0c05615fb5753d78584e0c3fd5ca
SHA512 (NetBSD-6.0/arc/binary/sets/tests.tgz) = 54fd09bdf390959e4a795af7efa34cf463f0d41676a87bceae7e8278f97693549bea10d7de099c0f3af34f58ea76820ecd357894d9c7e5bccc65d2fa9736dd40
RMD160 (NetBSD-6.0/arc/binary/sets/tests.tgz) = baf5b98a820547d80922493f59c48028345306da
MD5 (NetBSD-6.0/arc/binary/sets/tests.tgz) = 601e2f0d37d00e07a671ed2cb823e68e
SHA1 (NetBSD-6.0/arc/binary/sets/text.tgz) = fe43613098e27d16d74e9c21e0b18cc5e932e673
SHA512 (NetBSD-6.0/arc/binary/sets/text.tgz) = 4935e883c8328121eb941726a41487dc9048a9d3265460e9730465a3d1b8283e4bd01ad2c220607848f898f178dac0738c6bb6402243bcc338e5f933a6c99f06
RMD160 (NetBSD-6.0/arc/binary/sets/text.tgz) = b57da314266be1fdf31856f5ad5a52703016b61e
MD5 (NetBSD-6.0/arc/binary/sets/text.tgz) = 9236a9d595f3a96234ba9156ed3437e2
SHA1 (NetBSD-6.0/arc/installation/boot) = 889f35138bde05f74032980aa2ca7c17442c4324
SHA512 (NetBSD-6.0/arc/installation/boot) = 9a1ccabb379ab44784c481e749f633cf4235c03e22c24f2a837e6fcbe28b23f7bde8a7ed596dde93996bdc5efbf639ed6a073fd848eef2b0d80e932ba360c9e5
RMD160 (NetBSD-6.0/arc/installation/boot) = 369fb440abfca5f2f4340740265f61a9d2ca8df7
MD5 (NetBSD-6.0/arc/installation/boot) = 36b7e6f86ddc8aede5c9ea0160939cb2
SHA1 (NetBSD-6.0/arc/INSTALL.html) = 0cf4b01d7175a1eb6dd2734e412c3bed0fb0967d
SHA512 (NetBSD-6.0/arc/INSTALL.html) = 68a51695f6fbe96610354b253741efcd7f52bd543351a3f3b734398264f7b75c03e72ee731be917b2e34a0041c621107177efb57ba9abe6158456e19526845db
RMD160 (NetBSD-6.0/arc/INSTALL.html) = e174f77ca32ded5ae5363e88ca33ec17df6b95ae
MD5 (NetBSD-6.0/arc/INSTALL.html) = 94eb6340dc1e3d6ae7b539e3ca3744a4
SHA1 (NetBSD-6.0/arc/INSTALL.more) = 6146391a310edb4727d4b5a61793d535c0983c0d
SHA512 (NetBSD-6.0/arc/INSTALL.more) = 12e1ebc08bd3fb259caafd6b75e1120daba07fb4ad201f513459e5e36ad96790a26076cf154b34332a54192cbda13bf0b85847478a2443114522225f1adbf95d
RMD160 (NetBSD-6.0/arc/INSTALL.more) = 52a7433c4a3fbb3fa6ca52117f9f902a96748dee
MD5 (NetBSD-6.0/arc/INSTALL.more) = 9a6a3c4c52c723151ac81fc40567f625
SHA1 (NetBSD-6.0/arc/INSTALL.ps) = ed6ce1da53cd979819a5080666fef04c62c3cf60
SHA512 (NetBSD-6.0/arc/INSTALL.ps) = 8ea3bef19288069a145d6873030895c960811c8dbcff956101f425a9ce0a57701f0f0579c925248b5ecc9f7ac110ddcb18b8041ed74ffd69c22b598ce8d89788
RMD160 (NetBSD-6.0/arc/INSTALL.ps) = e1c8d2d8ef5b2d170f659f14bf124063d8d34c22
MD5 (NetBSD-6.0/arc/INSTALL.ps) = f97b10a669be7ec3f9603d5f936fe9fe
SHA1 (NetBSD-6.0/arc/INSTALL.txt) = 799c2e5d3f8a8d4c93e0ba1e0e738dffbec98504
SHA512 (NetBSD-6.0/arc/INSTALL.txt) = ed89213d84a4ce5ca43b8b7ed35b6e2e731c77205518b8c48258b18567ecc6e2150e81c2d2a4ca0b3c2bbf8059218d0cf918b168c2f55b305815f89ce3cb45c7
RMD160 (NetBSD-6.0/arc/INSTALL.txt) = d9be36fb053f75805f95439723d18875e6d0cd5c
MD5 (NetBSD-6.0/arc/INSTALL.txt) = 56f07f0f3cd11f592b56f5978d23c533
SHA1 (NetBSD-6.0/atari/binary/kernel/netbsd-MILAN-ISAIDE.gz) = abe68fd4d9212fee55be19e05105bed9ee51687b
SHA512 (NetBSD-6.0/atari/binary/kernel/netbsd-MILAN-ISAIDE.gz) = 759476c2983250d38267853ca369a432b968e9d8187ae8476d5e2931667518abc63fadef50c6bcf55299866c019d74790e8268b58f433c6413c03cfe1460d8a0
RMD160 (NetBSD-6.0/atari/binary/kernel/netbsd-MILAN-ISAIDE.gz) = defe3e00a5198a97320f53a1460eb7601f2d5a9d
MD5 (NetBSD-6.0/atari/binary/kernel/netbsd-MILAN-ISAIDE.gz) = fadc1a2fcec83bae67e6e70f1ee8d4c6
SHA1 (NetBSD-6.0/atari/binary/kernel/netbsd-ATARITT.gz) = 9d2e04ce73a5a24cfb10801c0ccf659242602dd3
SHA512 (NetBSD-6.0/atari/binary/kernel/netbsd-ATARITT.gz) = 27b28318b371707290e3b8482dc0e98e32c6d5b7fb4d38e3bddda7bc5e1b953ae0651b680cf7ea9a296018b5a0ac883cb59d4da3c02db1baa67e151da783f6ed
RMD160 (NetBSD-6.0/atari/binary/kernel/netbsd-ATARITT.gz) = 77d47ef664e710823cef2633c92bde054a970552
MD5 (NetBSD-6.0/atari/binary/kernel/netbsd-ATARITT.gz) = f99c55c64e84c84de3991fa4f21897c5
SHA1 (NetBSD-6.0/atari/binary/kernel/netbsd-FALCON.gz) = 9c0e5bc223f81e24108dde3428ee8833e557d891
SHA512 (NetBSD-6.0/atari/binary/kernel/netbsd-FALCON.gz) = 6147ba07227f85a3ba2ee1e78449aa2c4e0ad708e9cc000f456a74a69f6443f099e200f2dd3602ceb453dffe7b41c90c32f45ff63d44d59cd0ce24ba68f3ffa1
RMD160 (NetBSD-6.0/atari/binary/kernel/netbsd-FALCON.gz) = b2c8d8f4742da6ccf1e0ec3811160d9ce9606baf
MD5 (NetBSD-6.0/atari/binary/kernel/netbsd-FALCON.gz) = 2189a16e1ee3e1586f691490fe6bfc9a
SHA1 (NetBSD-6.0/atari/binary/kernel/netbsd-HADES.gz) = c4f3101449a14e294394b13eacff10053078ad4d
SHA512 (NetBSD-6.0/atari/binary/kernel/netbsd-HADES.gz) = bef4589fa86a33df94fdb36038e93528984629682c2827ed4dad582ae17cfea37b9841608dacc95fd8a55c57c40a81b8d24d7afd253dd46d7af970e0927518bc
RMD160 (NetBSD-6.0/atari/binary/kernel/netbsd-HADES.gz) = 821bc772711ece35be32f1fadac8fa31a10041ac
MD5 (NetBSD-6.0/atari/binary/kernel/netbsd-HADES.gz) = 82b69e45fbf4740774a338f62c987dec
SHA1 (NetBSD-6.0/atari/binary/kernel/netbsd-MILAN-PCIIDE.gz) = 4e056fe210771d7d97ad303cc4d92fed64d883f2
SHA512 (NetBSD-6.0/atari/binary/kernel/netbsd-MILAN-PCIIDE.gz) = 2c303e9421be0eae122cf55c2b5205b4c2129e6bf3033372f8fca93ad410f92137db6463ad9b75db53526dccb2b9c8a3da23d04357a6a3663aa58972c39184ad
RMD160 (NetBSD-6.0/atari/binary/kernel/netbsd-MILAN-PCIIDE.gz) = 878a29fec55b0429c47f0932facf25139721c88d
MD5 (NetBSD-6.0/atari/binary/kernel/netbsd-MILAN-PCIIDE.gz) = b2e52ec52845325caf8fce89fcf5b034
SHA1 (NetBSD-6.0/atari/binary/kernel/netbsd-SMALL030.gz) = e6b4dad71f39674e21a0baa6cd5bbc61872786e1
SHA512 (NetBSD-6.0/atari/binary/kernel/netbsd-SMALL030.gz) = 856e3435c5ae454b145bb16a679da93cc966cd7ee75ab8cce77f216d4070b9e3f1a0d2bf15ae9eda28f48a0eaeeae590559888eef477ab4c72badcaed7b7484e
RMD160 (NetBSD-6.0/atari/binary/kernel/netbsd-SMALL030.gz) = 449713cd4e5c43656648e76a59b0228cbf51de82
MD5 (NetBSD-6.0/atari/binary/kernel/netbsd-SMALL030.gz) = d38e0f0b5090a8d0d6552092c241e105
SHA1 (NetBSD-6.0/atari/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/atari/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/atari/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/atari/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/atari/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/atari/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/atari/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/atari/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/atari/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/atari/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/atari/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/atari/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/atari/binary/sets/xserver.tgz) = 1d20fd039546a66b428b854c659ea5e2ee9f8114
SHA512 (NetBSD-6.0/atari/binary/sets/xserver.tgz) = 091d09049d0af5555feec02065234e281677aeb3ecbab5a19413d924a8b58dc7f64401ef68e68bc52d8b6eeb73e091d351c45805374b8bf276b0181415968561
RMD160 (NetBSD-6.0/atari/binary/sets/xserver.tgz) = feef5d52278d23637d7b8ba77639bfd96918e90e
MD5 (NetBSD-6.0/atari/binary/sets/xserver.tgz) = b2bd6057c374376388766fc8667eb9a9
SHA1 (NetBSD-6.0/atari/binary/sets/kern-ATARITT.tgz) = 589b0e48516d961e8961847f441bf7dbcbb67700
SHA512 (NetBSD-6.0/atari/binary/sets/kern-ATARITT.tgz) = 00ad9fefd5c3fb87f45b20612582b818def61c0f08483f923622a19e94090feb7ba1e39dc8d3b4b7d57e70c60aacaaeee966e1a6a788aff9fbbedc6848e52cd4
RMD160 (NetBSD-6.0/atari/binary/sets/kern-ATARITT.tgz) = 510465a0b2f0e19a4873fa87fa66d7c47ec8226e
MD5 (NetBSD-6.0/atari/binary/sets/kern-ATARITT.tgz) = cea983b428fc041a9ad553b25e6de431
SHA1 (NetBSD-6.0/atari/binary/sets/base.tgz) = 948b83244fe16e439af84cee9dd48f555ff2c479
SHA512 (NetBSD-6.0/atari/binary/sets/base.tgz) = ee2c41fa904f75740c358f85bc97c36537d6a0f252bfb4ca3ec60098109ca07b8be38e8474ea7355e557793fe726160e1f0d10de2769720432a8cb0896bdcf7f
RMD160 (NetBSD-6.0/atari/binary/sets/base.tgz) = d08e1a2d3eb8629aafe49fe89e9c00c65c97967b
MD5 (NetBSD-6.0/atari/binary/sets/base.tgz) = 07d14a63f5591f86d0cb7728174ad6b1
SHA1 (NetBSD-6.0/atari/binary/sets/comp.tgz) = 081806be5308d18e9910a9f3394bb15fec7111bb
SHA512 (NetBSD-6.0/atari/binary/sets/comp.tgz) = 17e0a84c5ccefcbee5807012b93f413fa90d6fdb55bc91020cc5265528eb4331a4e2d5f951ac457b0fbf24ce2a97f7dde29a2cfda946edc5db2bf7c574289a75
RMD160 (NetBSD-6.0/atari/binary/sets/comp.tgz) = e2f1fbee6aa542a8b2d80cb2eb4b593fa13a4dae
MD5 (NetBSD-6.0/atari/binary/sets/comp.tgz) = 46cdf183c47138a86c07a43cc4c365d7
SHA1 (NetBSD-6.0/atari/binary/sets/etc.tgz) = 7e13a9c53246d90a86a0db1a3e38ade70705402d
SHA512 (NetBSD-6.0/atari/binary/sets/etc.tgz) = d751e14266428861f1ef067f02c8e4f79047e3c482f9b89ca44e880ea71ef66de3792d03f4b3704fc98bd8c9ff17cbdb42fa4c4cf3dccff9a94e88662a9035bb
RMD160 (NetBSD-6.0/atari/binary/sets/etc.tgz) = f09fa0ba9766e3cbaf7eb0f59f32dac370d37f14
MD5 (NetBSD-6.0/atari/binary/sets/etc.tgz) = 42096354bae512677af5f6d5c9a8484c
SHA1 (NetBSD-6.0/atari/binary/sets/kern-MILAN-ISAIDE.tgz) = c623eff071500b740c875a6984194d2826ab66c1
SHA512 (NetBSD-6.0/atari/binary/sets/kern-MILAN-ISAIDE.tgz) = 25f68ad0b3a910cad6f8d8a7cc232a3e6e71ea381fa8cefd342ff3c18c869abdcdfa517984c54ffcdf72e7393b79aeff78467137d6729447042233a55297bfe7
RMD160 (NetBSD-6.0/atari/binary/sets/kern-MILAN-ISAIDE.tgz) = 16cc868818532530f4441eed778a26b98ba06190
MD5 (NetBSD-6.0/atari/binary/sets/kern-MILAN-ISAIDE.tgz) = cc65ece6a5e66a5c1f0aa67a24bb1726
SHA1 (NetBSD-6.0/atari/binary/sets/kern-FALCON.tgz) = 4e7e28bad7354a7bb7fdde4f608340c4907b4453
SHA512 (NetBSD-6.0/atari/binary/sets/kern-FALCON.tgz) = 24d5df6a74be2fa65016ab03eb11bcd6dbab11ebc33b0ab936685345ff54fe518b6b87171c93c4afab54b0374fb9d5ffa29390a34dd218b400e985dfeae8d147
RMD160 (NetBSD-6.0/atari/binary/sets/kern-FALCON.tgz) = 82c44f12a090c11dd2b7de11b8b9ae95fa74ed08
MD5 (NetBSD-6.0/atari/binary/sets/kern-FALCON.tgz) = fea878ea06b74eef20f60e192539f033
SHA1 (NetBSD-6.0/atari/binary/sets/kern-HADES.tgz) = 24a828836e47ffe647f8a16b1fd0469a9ee62f03
SHA512 (NetBSD-6.0/atari/binary/sets/kern-HADES.tgz) = 8bbc2397d01594b89515abea27fa58e9ce7dbda5d2f9468b5bc46576d858395c11e4e1d6a9c25dd9b7d5ddb7ac2c2fd2208d95274f9ce7519fb470cbf08da2ae
RMD160 (NetBSD-6.0/atari/binary/sets/kern-HADES.tgz) = ae93fdba82eecea06486ad88927f25fe1013803d
MD5 (NetBSD-6.0/atari/binary/sets/kern-HADES.tgz) = 244b09181178cc0b51c79ad7568f4d9e
SHA1 (NetBSD-6.0/atari/binary/sets/kern-MILAN-PCIIDE.tgz) = 0269a5f553ad95565fb5092460fbefb4567afa27
SHA512 (NetBSD-6.0/atari/binary/sets/kern-MILAN-PCIIDE.tgz) = cd74cbc52c0357b9af3290dd02362e5561202c14d58bad0d65b96c84361be2e774c9c58b5e38ca9523a7e796d778330cf257f8e2482bf8153e7a43c167e7c392
RMD160 (NetBSD-6.0/atari/binary/sets/kern-MILAN-PCIIDE.tgz) = 87ac890b25b9bacc28ce600476add0d6fd8aba03
MD5 (NetBSD-6.0/atari/binary/sets/kern-MILAN-PCIIDE.tgz) = f18391129e4495470783e0a1d8467a81
SHA1 (NetBSD-6.0/atari/binary/sets/kern-SMALL030.tgz) = fff933858b1ef2c086ce99631195c2e89714b540
SHA512 (NetBSD-6.0/atari/binary/sets/kern-SMALL030.tgz) = 40458f1b8455fcd8fb3d01dc52815294705ab3752a5952ce93de01c5b7fb553d5aa0aaff4d9124d46e43d318ed6289c336e0a73470d7471fe4989877cae347a7
RMD160 (NetBSD-6.0/atari/binary/sets/kern-SMALL030.tgz) = 08e49adb3414a2c4490b143865dcf2f0ab3ff337
MD5 (NetBSD-6.0/atari/binary/sets/kern-SMALL030.tgz) = f0aee544f1a3c894e71ff07e1ab135a7
SHA1 (NetBSD-6.0/atari/binary/sets/modules.tgz) = 8ea8801011d50b52e637ee0cb1b94fdbf79b9de4
SHA512 (NetBSD-6.0/atari/binary/sets/modules.tgz) = 6f002bf0695f00d06ce7f43f3087d6c06949b5df8e33baf0b8b30ac18e72700445dcfeb74b51c16fab32e63e537c10bf696bfdb31ab8b08d25c9189075cd3673
RMD160 (NetBSD-6.0/atari/binary/sets/modules.tgz) = 2bf65897ea1432bac2666df1b221821a5a0642a5
MD5 (NetBSD-6.0/atari/binary/sets/modules.tgz) = 4dda2688a56fa0774342c63c367c8ec6
SHA1 (NetBSD-6.0/atari/binary/sets/tests.tgz) = 88a32113d4d6d34b29a94e69c0e4510b51e7e059
SHA512 (NetBSD-6.0/atari/binary/sets/tests.tgz) = 53fa49988eca04a409dad693f26d5445cdafa9ff69ac918583849d9c6726278b9f0d2fd0da23c05e73d23453970eaadb7b233d951b430c1b7b3ca1b05bab5a24
RMD160 (NetBSD-6.0/atari/binary/sets/tests.tgz) = ab9f0395fd5d0d1a732000d07e71efe3860f9a32
MD5 (NetBSD-6.0/atari/binary/sets/tests.tgz) = 213a94d65400a854c2260e98f6150df5
SHA1 (NetBSD-6.0/atari/binary/sets/text.tgz) = de8e33381d0714b33aab422c2aa5841352941712
SHA512 (NetBSD-6.0/atari/binary/sets/text.tgz) = bf15a081319d8ce62cc14f465bce3b204bc9a5f57f7ed9d163cf71a9422eeec5ff0e84de7c43d0fc503e9422283e2f99e1e48ee5ccb9224effdc2e3ba70f31db
RMD160 (NetBSD-6.0/atari/binary/sets/text.tgz) = 387ade3ddcc90bd7b6174deeb24bd8cc6a818e46
MD5 (NetBSD-6.0/atari/binary/sets/text.tgz) = 937c56219d0ec6a96aef98533e10b569
SHA1 (NetBSD-6.0/atari/binary/sets/xbase.tgz) = 07c1478cf8a0e4b7ee4b5e49841966fe17f3d797
SHA512 (NetBSD-6.0/atari/binary/sets/xbase.tgz) = 54d845bc341dc68debf004659d2dcd753604de16e896bceabe6587bfa2934bd5b81b19936c3635cb61dbdee4b4d571603659cd6ae8206ccdaa345f3779e4a83c
RMD160 (NetBSD-6.0/atari/binary/sets/xbase.tgz) = 6a3aba94b4acef9e311f61c1acc96b33c8707ea8
MD5 (NetBSD-6.0/atari/binary/sets/xbase.tgz) = e02f145dcfd9f7d1c7be166b28559094
SHA1 (NetBSD-6.0/atari/binary/sets/xcomp.tgz) = cea8de3b7820897ee8270ae3b591fb602e0028d9
SHA512 (NetBSD-6.0/atari/binary/sets/xcomp.tgz) = 9e6344200d3ba54e29db576b9f7c0c1d2465cbda8b8a05d015596e2b162c207453cfe0ddc7f9d55bd8463bf2c200c2dd43b4ec34c30635fc0923a0d31c2a8ff8
RMD160 (NetBSD-6.0/atari/binary/sets/xcomp.tgz) = e64e955294f627c722087a4d1e62a6d646028a37
MD5 (NetBSD-6.0/atari/binary/sets/xcomp.tgz) = 870efcab1ee55d57351c55ac3c412dda
SHA1 (NetBSD-6.0/atari/binary/sets/xetc.tgz) = 64dca98d1263e3d78ef7e240c9975e9077356ce0
SHA512 (NetBSD-6.0/atari/binary/sets/xetc.tgz) = 1e76598c460681b43f2c5b19f2f4af9efeed6ce16aa8bb4d34f99083ee46cf6c5c31c0aeab974a723a3338c3a9ab1a50ad273061f9f02870d9741ec8809ce0cb
RMD160 (NetBSD-6.0/atari/binary/sets/xetc.tgz) = f147db5da2954155dbc754963d02f309ab41e923
MD5 (NetBSD-6.0/atari/binary/sets/xetc.tgz) = d23248d3d92350124d8e9e869b28a5f7
SHA1 (NetBSD-6.0/atari/binary/sets/xfont.tgz) = ad9fb5958debad82e953bc2b1e7e53292d70463b
SHA512 (NetBSD-6.0/atari/binary/sets/xfont.tgz) = 789529bdf38abda3b00e1db3cffc12272af8ec7f7a52855e0ef5af09ac7cea8639b407265ee6082809127b3bd1b673d686d967c987be6ac12be3825f9531933e
RMD160 (NetBSD-6.0/atari/binary/sets/xfont.tgz) = f060722e795db40bec2c283ecbb5f7bdbb688ce0
MD5 (NetBSD-6.0/atari/binary/sets/xfont.tgz) = 1b83f33dd3e43db09d0b55502a7b347a
SHA1 (NetBSD-6.0/atari/installation/miniroot/prepare.fs.gz) = c4e92cdcb08838b9fa5dc32fb1bd923dd2b97291
SHA512 (NetBSD-6.0/atari/installation/miniroot/prepare.fs.gz) = 4c25bd0fa0c584b09c2b0a2f9a4a67e942d9a16d9646f1364c71b8bf24326902fae74dce17ee60c81261777ac27e69c2ce4988779ac604a6ac6ebbd4636698d6
RMD160 (NetBSD-6.0/atari/installation/miniroot/prepare.fs.gz) = 247f6e13edeb6272ceeed462d8729d5e27db000b
MD5 (NetBSD-6.0/atari/installation/miniroot/prepare.fs.gz) = e99c7d5da695d99c717db4585cac009c
SHA1 (NetBSD-6.0/atari/installation/miniroot/sysinst.fs.gz) = 353039d699408734197b1a5ecc2490310269e23f
SHA512 (NetBSD-6.0/atari/installation/miniroot/sysinst.fs.gz) = bd8f6ce4115a98b026fc906c1259ce8ea91d51b97ac44190362a03154ad12c05419d848daf27749a17f0e255b9cb01d2ccae81e75f634825865b95032e5e634f
RMD160 (NetBSD-6.0/atari/installation/miniroot/sysinst.fs.gz) = 3f866900d10efc89f1f22e8ab6099f26d79cb5f4
MD5 (NetBSD-6.0/atari/installation/miniroot/sysinst.fs.gz) = 4fabfb561929af7e667fdf39a64e58dd
SHA1 (NetBSD-6.0/atari/installation/misc/file2swp.ttp) = 9aabfba1673d710445ce4a151e070cd00a364795
SHA512 (NetBSD-6.0/atari/installation/misc/file2swp.ttp) = d522703bab622aad1f125bf076fa58ea8a309c34425d188df8130298f5af0ccf8d0b0980ddbb90bff86e73cf11d082081f5d0cbe5e8dba68fdc87c9fbdbe2da4
RMD160 (NetBSD-6.0/atari/installation/misc/file2swp.ttp) = 5bc12b2214b90eefc315f50321b072f3359443eb
MD5 (NetBSD-6.0/atari/installation/misc/file2swp.ttp) = b2e39863389322ab0834a30a58fbe553
SHA1 (NetBSD-6.0/atari/installation/misc/aptck.ttp) = 9a1b1b99bd6b50281cb1a1416f854b1d1e0909c4
SHA512 (NetBSD-6.0/atari/installation/misc/aptck.ttp) = 90fedb40ba1ada8a28b228c0ad34ce9e400b3f9985d69b52345df1e29c7b9b2fd0ceafe1ae91cab0095f3f24ec8bbb274e2f514e351b39eeb8591d3c5ec46d03
RMD160 (NetBSD-6.0/atari/installation/misc/aptck.ttp) = e399308e499e2e859eba1fc190208537421e407a
MD5 (NetBSD-6.0/atari/installation/misc/aptck.ttp) = 5061a34d4cd283af586c5f382b546a82
SHA1 (NetBSD-6.0/atari/installation/misc/chg_pid.ttp) = a989a7318087ebccdb92ea9c84e81a8fcb1d2bd3
SHA512 (NetBSD-6.0/atari/installation/misc/chg_pid.ttp) = 15dca31b0555023ba6e2b04e6bca504adddcc56177f6224a2757d8784ded4a53fd5cf68b3dd5e794b46bcfba69d199dd430aa9379ed565a7aff4c360855cc4cd
RMD160 (NetBSD-6.0/atari/installation/misc/chg_pid.ttp) = fef62601213fdafa60099375eaed9ea23163e165
MD5 (NetBSD-6.0/atari/installation/misc/chg_pid.ttp) = c414114be1b45d1b8abcd41091fb0c6d
SHA1 (NetBSD-6.0/atari/installation/misc/rawwrite.ttp) = 18c663ff37bf90130bc6eeefd3aabfcd632c654c
SHA512 (NetBSD-6.0/atari/installation/misc/rawwrite.ttp) = 259903e1e3439b3a3d128a18a597ccb6ccec507bc6b1c449445e7e00c4e0d37e436ba2be0cdb71a7a756c30c2a6406cffd3bef0631b2083a2f0402ab322dbf96
RMD160 (NetBSD-6.0/atari/installation/misc/rawwrite.ttp) = 3b66c3e7f84325e3be39c15e2a597d2bbd233a79
MD5 (NetBSD-6.0/atari/installation/misc/rawwrite.ttp) = 20458ba9d4f2fae37607648aeb92bf71
SHA1 (NetBSD-6.0/atari/installation/misc/gzip.ttp) = 2f3c87cac323a1641622ed960d7ca3df6d8e92ce
SHA512 (NetBSD-6.0/atari/installation/misc/gzip.ttp) = 37be09560542fd55afebe8f3a07a76d4121a49c5d8771107a6065dbe55c3d8e33ca25326ffdecb38ceaba0e5b6af33f78f92ecf2600169779b5d7282f74335a2
RMD160 (NetBSD-6.0/atari/installation/misc/gzip.ttp) = 8e9dbdb381f2b7ebaf01597bca73ba920e18427f
MD5 (NetBSD-6.0/atari/installation/misc/gzip.ttp) = e27e5b3c585ad0a96daaaa810b56f5ef
SHA1 (NetBSD-6.0/atari/installation/misc/loadbsd.ttp) = c7f08ef85b1b3120061b3b651456cd23f8937c16
SHA512 (NetBSD-6.0/atari/installation/misc/loadbsd.ttp) = 2b645062a84fe5892c3d424d181dd6ad11f134c4a6728b4ca9a67f1912642daabdfaecdf7af985b4d4907d93adc1b72ed86a5cf9b57de42e9213582ec89a8723
RMD160 (NetBSD-6.0/atari/installation/misc/loadbsd.ttp) = 802c240f7069d79c0432b28dead7884bc4a06d38
MD5 (NetBSD-6.0/atari/installation/misc/loadbsd.ttp) = e5e92b220890c5ca942d0c878fd090ca
SHA1 (NetBSD-6.0/atari/INSTALL.html) = f71f76494a6bf37c8aefe74e698838c1a03b14c0
SHA512 (NetBSD-6.0/atari/INSTALL.html) = a12b9739f7918dd22bb0434c3d1c41c5031454fd6dbcb53df4eca669b9b4bc918c1f8df379bcac46204664dc521a049b5c39b4151f207a4a22b19b436917fd35
RMD160 (NetBSD-6.0/atari/INSTALL.html) = f634ad4adfaa1002f352d703118aba204cbe4f60
MD5 (NetBSD-6.0/atari/INSTALL.html) = 7eac9b0603de1d7d718b994c92b7a229
SHA1 (NetBSD-6.0/atari/INSTALL.more) = 6ba2460611e6f92531b9ae9d28f4c68d302b9dc5
SHA512 (NetBSD-6.0/atari/INSTALL.more) = 718766711a583a0539790a3fa7046f845a15f53dd1cd1f0ddadea939158256b3fbf5dced1a88ea592ded2685f0fda81984d6cb137f26f7b543aded6c7c1ca6a8
RMD160 (NetBSD-6.0/atari/INSTALL.more) = 8bdd7c42fa93913bd14d5ab80457394e289f8c03
MD5 (NetBSD-6.0/atari/INSTALL.more) = 19adcdce47ce4b9070326b433afe5a3a
SHA1 (NetBSD-6.0/atari/INSTALL.ps) = 2d9874be06bf612885e3f19fc37635b0c421f52e
SHA512 (NetBSD-6.0/atari/INSTALL.ps) = 0dd3a09f0a91403f412aa3650847c3512db2ac3d36d0133ac924db74d07b59607c9f57df7fd0df4a3652adbba4842f6b6afa01c07cc4a06c36623ad9c60a9ca3
RMD160 (NetBSD-6.0/atari/INSTALL.ps) = 83f619a939ef01654f3e3bce7f7ddea0b88828bc
MD5 (NetBSD-6.0/atari/INSTALL.ps) = 73fd3b04168f78f2de31b91190851ba9
SHA1 (NetBSD-6.0/atari/INSTALL.txt) = 711ce58ce059429925a6e08ad178569c5e00a00d
SHA512 (NetBSD-6.0/atari/INSTALL.txt) = b837d76a7e0d754c09fff90845906ec6a1dc5ed77832b91442d5a9ec3608f5e6785782ff1debe3b145f23ba382c5472abec75f49055859bc715caf713941cf41
RMD160 (NetBSD-6.0/atari/INSTALL.txt) = ecd373463a1cfd5b5399cb3ca0204362ddb22be2
MD5 (NetBSD-6.0/atari/INSTALL.txt) = f6b564115bd3ae1a28f79dd879fe8ce6
SHA1 (NetBSD-6.0/bebox/binary/kernel/netbsd-INSTALL.symbols.gz) = bdb282bf1fce2a0c48cf661a68b16fb7663859df
SHA512 (NetBSD-6.0/bebox/binary/kernel/netbsd-INSTALL.symbols.gz) = c476616f879c94a8135889c4691ca927896c74950c16543039a74e4befd918619845f31e519cb452bd215d8ca4fa1acd697f9867810f4961d517f7803f7e2c33
RMD160 (NetBSD-6.0/bebox/binary/kernel/netbsd-INSTALL.symbols.gz) = dbb18f3fb59483723599306093a562a45f23392b
MD5 (NetBSD-6.0/bebox/binary/kernel/netbsd-INSTALL.symbols.gz) = 582ae8bf879f7bc42d24bf114a5d6a5a
SHA1 (NetBSD-6.0/bebox/binary/kernel/netbsd-GENERIC.gz) = ecfa395d442d86acbc8bbd080c04a8137012138c
SHA512 (NetBSD-6.0/bebox/binary/kernel/netbsd-GENERIC.gz) = b48b213c4ffd06180d1272a65e27ec0ed539e731e07aea29d31f6c939fef99e1a965c42e2fdd99daad1e8ee543c99a02c420fbcf6531d47e961e7b295150fdb9
RMD160 (NetBSD-6.0/bebox/binary/kernel/netbsd-GENERIC.gz) = 0b01e797fe821429603d9252035e7425f388046e
MD5 (NetBSD-6.0/bebox/binary/kernel/netbsd-GENERIC.gz) = 22289b3d7d60df27fd0ed86731ab77bf
SHA1 (NetBSD-6.0/bebox/binary/kernel/netbsd-INSTALL.gz) = 8721973ccd6399bf563bdb266d9e5316e1ef98ed
SHA512 (NetBSD-6.0/bebox/binary/kernel/netbsd-INSTALL.gz) = 5390ebe49a1d50d2ba42e529c8fb2c23606e4650cd3225783809ee4dee6013f60cf7dbb347fabf48f3d7ce98ededef37cfc8d5de5c07ad7462dacc668243f429
RMD160 (NetBSD-6.0/bebox/binary/kernel/netbsd-INSTALL.gz) = 8228a5df5239737e7ba039b84303c4cd491e26f6
MD5 (NetBSD-6.0/bebox/binary/kernel/netbsd-INSTALL.gz) = 9165b1f4313cc543868d7ab7f6eff5ad
SHA1 (NetBSD-6.0/bebox/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/bebox/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/bebox/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/bebox/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/bebox/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/bebox/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/bebox/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/bebox/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/bebox/binary/sets/kern-GENERIC.tgz) = d4e8dde09eeb9886e1d84573d136ea92db147201
SHA512 (NetBSD-6.0/bebox/binary/sets/kern-GENERIC.tgz) = bb375fbcd379a02ccb0f770ae22aeb02f90fda95be2864f085bd46ad58c670e5fda2792b9af8acb5972398a4056ef4ca8d95367f20f57e1258401d1b02d29abd
RMD160 (NetBSD-6.0/bebox/binary/sets/kern-GENERIC.tgz) = 2726cf7d2e1fa0278bb78a1f1d21e3a6846bec27
MD5 (NetBSD-6.0/bebox/binary/sets/kern-GENERIC.tgz) = 17e1bdb7465cc8aff08f41a81a37913e
SHA1 (NetBSD-6.0/bebox/binary/sets/base.tgz) = cb3438d686bc37c67ef3506e7b5f14af708d0f69
SHA512 (NetBSD-6.0/bebox/binary/sets/base.tgz) = caaf5501269c91599fe8a6315a14bd33776549df0d82d34e516b846df3ace79c1b3a77ef745da87fca1bcc822201765366a373c2d9e29195b4f76bc954f6903e
RMD160 (NetBSD-6.0/bebox/binary/sets/base.tgz) = 9eaed756de31f7ef1f390895400ff57d90b6936f
MD5 (NetBSD-6.0/bebox/binary/sets/base.tgz) = 844ca078de04b6965afc9d6840712a87
SHA1 (NetBSD-6.0/bebox/binary/sets/comp.tgz) = fa35a56729cf7a8957a24482e5855629b63da9ca
SHA512 (NetBSD-6.0/bebox/binary/sets/comp.tgz) = dae46ff9a1b0ec32444bf93501e5ce5a003556ec6e057c1a85ef63d6f18fab00370fb5c8c6dda14da7222032f26aded43ff2ba4619df58f13d773f3c55a3203a
RMD160 (NetBSD-6.0/bebox/binary/sets/comp.tgz) = e23f0ffdecc624bd584b1bdafcc1919c0a6b4bee
MD5 (NetBSD-6.0/bebox/binary/sets/comp.tgz) = e24aedf775e35faa7b0be771def8e964
SHA1 (NetBSD-6.0/bebox/binary/sets/etc.tgz) = 7955187dd8c102acefc90e649d70bf87334b8c15
SHA512 (NetBSD-6.0/bebox/binary/sets/etc.tgz) = 3dbc921b6f58296fecb96ac7a0497f4de1f84179128afa3fe2f0a9260399934f659c53346ab55dbf387f19e348e4cf441506b2b92a72f0a2de71d59701289e34
RMD160 (NetBSD-6.0/bebox/binary/sets/etc.tgz) = 7426566e042dd69397282ea85d9999952d57fde4
MD5 (NetBSD-6.0/bebox/binary/sets/etc.tgz) = 8980eb4fe0883bbcf7d2001bc9d705a4
SHA1 (NetBSD-6.0/bebox/binary/sets/games.tgz) = 780b002a35471789a68ee7be17362c60e96fa5fd
SHA512 (NetBSD-6.0/bebox/binary/sets/games.tgz) = 954365033c924762b524f0f1e4732216b4c661697336b0947943f05a1d1e0ecc12c1679c1c5d7572e1a15a7296bd7b14d31749917ef03a2b097b94bfa0f16837
RMD160 (NetBSD-6.0/bebox/binary/sets/games.tgz) = 9b237eefee5dc43fb3fffca5a2aa390fb2c3031f
MD5 (NetBSD-6.0/bebox/binary/sets/games.tgz) = 337e6739c192296cac92b610d0ad6945
SHA1 (NetBSD-6.0/bebox/binary/sets/modules.tgz) = 14ad320114a249b1ed676832524f773dae772798
SHA512 (NetBSD-6.0/bebox/binary/sets/modules.tgz) = 1aa3a619fdff5824c904495352248fe990130b9ce9d9a3cd443afd2f088acf7048150de54f1cc304ccf08587fd91f7614521a0b9be96fc7768c06e0a80ed72ac
RMD160 (NetBSD-6.0/bebox/binary/sets/modules.tgz) = d1df79e00026a19f724c6115055c611c07d592cb
MD5 (NetBSD-6.0/bebox/binary/sets/modules.tgz) = 8dc1d2dbc89e5e6aa3e04b0033cc4a22
SHA1 (NetBSD-6.0/bebox/binary/sets/tests.tgz) = 979d26c8e2e5283c416a2e4f4b603acdb43756a0
SHA512 (NetBSD-6.0/bebox/binary/sets/tests.tgz) = e726187887b9429ce3bd39fb51e7703792f288c47eb9c0f20b85e1a2be4f6448bc3e73919c5d28c28f400ada5b0b024b647079a3670f0a2153aff9b7f5738f13
RMD160 (NetBSD-6.0/bebox/binary/sets/tests.tgz) = 5cd1c8a42ae7a6ef706de0fcae5292db026efa9c
MD5 (NetBSD-6.0/bebox/binary/sets/tests.tgz) = 122dc0ea7ebc7ca924301e406ef6e002
SHA1 (NetBSD-6.0/bebox/binary/sets/text.tgz) = 30f7a17a6d9b03c7ce09bce6341895ac27c2f7b4
SHA512 (NetBSD-6.0/bebox/binary/sets/text.tgz) = e258a9a741918d6bd8de1e10cff771541b1f1191af776101ff38c722fb037d44d51879d45a5a90342c7e2c21d36cbf5a9a6f5f1a047e61bb18609385381fc3ae
RMD160 (NetBSD-6.0/bebox/binary/sets/text.tgz) = 6dee36ded12f84c00d7c82912da8755b0d04d497
MD5 (NetBSD-6.0/bebox/binary/sets/text.tgz) = e1e3ed8f872b4dd51b555e9100ed6173
SHA1 (NetBSD-6.0/bebox/binary/sets/xbase.tgz) = 03f73933d3719ef967af4a003282e1d56453e3f5
SHA512 (NetBSD-6.0/bebox/binary/sets/xbase.tgz) = d90e373c143773a93e430058628bc65161e6c25c725fcb279ac0765edcee8e9cab4ff55fa1d99ea66ccb6ca06133145eaaa77f2c26a082c27a53e2e10a19cfdb
RMD160 (NetBSD-6.0/bebox/binary/sets/xbase.tgz) = 0fe0422c310a56e55e2d5e4b85e8d8a41daaaf4e
MD5 (NetBSD-6.0/bebox/binary/sets/xbase.tgz) = 7120010c15d5f84927963c780c9197e9
SHA1 (NetBSD-6.0/bebox/binary/sets/xcomp.tgz) = 1c7522ec79f661a84912a5213ed835d400e108f7
SHA512 (NetBSD-6.0/bebox/binary/sets/xcomp.tgz) = bfc0d388d18e14cfe99e6bcbda22e4b11e5082b0acf5b314c6f0275823138c35d1a16c3ab3e3fc27e5a04228d6380ef3c2f45dc807715db2c1de689e5674a397
RMD160 (NetBSD-6.0/bebox/binary/sets/xcomp.tgz) = 23c8e8bb0e2f0708d2bd464468196f4d74359302
MD5 (NetBSD-6.0/bebox/binary/sets/xcomp.tgz) = b6fb09a621ed2138ed674838c72ec64a
SHA1 (NetBSD-6.0/bebox/binary/sets/xetc.tgz) = a14b23868bf88e59f5a845bd86cf4af61522f482
SHA512 (NetBSD-6.0/bebox/binary/sets/xetc.tgz) = a2aee6ee11a89b3fe3b5834502dfd1a50aec7bdcc0f424b3e9d19400370b922258d77a979dae0bd87f8e55df8da9b98f8bf88a73792b1ca6a8915e5cd08777aa
RMD160 (NetBSD-6.0/bebox/binary/sets/xetc.tgz) = bf37bf09d79777922f344d6b52f69d50c708b64c
MD5 (NetBSD-6.0/bebox/binary/sets/xetc.tgz) = 837bbc31ba2a81d2610adf225551a61a
SHA1 (NetBSD-6.0/bebox/binary/sets/xfont.tgz) = e44a6f29bfb3a947ca7d916c7556386a900cb3ee
SHA512 (NetBSD-6.0/bebox/binary/sets/xfont.tgz) = 9ca97fbe1d40c26938f5a84f09bf273f132b40a82f3ef548295901840155ca45c68ce0d0d980f7fe63bd8acb57be487fcc5f81e61ae4f9765151333a84636a43
RMD160 (NetBSD-6.0/bebox/binary/sets/xfont.tgz) = e9e153af690a96523a3b7af0a314e1104fa694af
MD5 (NetBSD-6.0/bebox/binary/sets/xfont.tgz) = 1323ead1d7e9f69f6790c1eb836ec117
SHA1 (NetBSD-6.0/bebox/binary/sets/xserver.tgz) = 05c93b9d2b3774b6cb71b9d829f5d21860f549ae
SHA512 (NetBSD-6.0/bebox/binary/sets/xserver.tgz) = 28c97ffdd74c9a2d1d060ce356fd6951d377b61b7fa691c01a150472b082dd027b2be909508ee3370ec59d89bd9c5e12914b3f34feccc6c7904a9ce49869d7d8
RMD160 (NetBSD-6.0/bebox/binary/sets/xserver.tgz) = 52564d70781876abda7f16a678791c8b3753f53b
MD5 (NetBSD-6.0/bebox/binary/sets/xserver.tgz) = aeca26a6ae798bcf849e7bd4df423614
SHA1 (NetBSD-6.0/bebox/installation/floppy/boot_com0.fs) = 1fc015fd100d0518a57e3e7a30e336be1000c098
SHA512 (NetBSD-6.0/bebox/installation/floppy/boot_com0.fs) = 674ae651d28163eae5f02f78e427d7291e4b977fff90036b1d6870f572e63b1963ef83453ff9da1d49ebb707ff918c0f64889bc0f7d17072690aadd65837e14a
RMD160 (NetBSD-6.0/bebox/installation/floppy/boot_com0.fs) = 1f59e43c9e6b08ccd63506a5179fdd2a365151b8
MD5 (NetBSD-6.0/bebox/installation/floppy/boot_com0.fs) = 300dd58c4641cf1fda257074b674937c
SHA1 (NetBSD-6.0/bebox/installation/floppy/boot.fs) = 8a5770fb10efc7d7c453138ca866b93e2287607a
SHA512 (NetBSD-6.0/bebox/installation/floppy/boot.fs) = ffbb187efcdeaaf5018827b619abe00ee42aedabab188a443b494c3a5ab94d21fea4761409d9eb23b5a37d5daee602baa838b1d052e1f66b5380a09ed446fb89
RMD160 (NetBSD-6.0/bebox/installation/floppy/boot.fs) = 58430893606678b17f132c3f3559dd4d30a7246d
MD5 (NetBSD-6.0/bebox/installation/floppy/boot.fs) = f807ed8a7b25aa9fc9c08637a5188dbf
SHA1 (NetBSD-6.0/bebox/installation/floppy/boot_vga.fs) = a922c42dd39babfb12443b7d122526f207969577
SHA512 (NetBSD-6.0/bebox/installation/floppy/boot_vga.fs) = 4f547ce92d72d826253aafab92f9df42b5dba05031ab4d8537fcd40291a703b06eb2e972729c8e703f4e4e0f75d2a0ee2e2f69e8f5d66735f2932089bf1ec7fc
RMD160 (NetBSD-6.0/bebox/installation/floppy/boot_vga.fs) = c999c7705d258c4b294f0905ba519ec86b7f34f3
MD5 (NetBSD-6.0/bebox/installation/floppy/boot_vga.fs) = 343aba7dd2584f3945d1fb539a72dc82
SHA1 (NetBSD-6.0/bebox/installation/floppy/kernel1.fs) = 8af8e65d2919686db6121c68347b8541991e373a
SHA512 (NetBSD-6.0/bebox/installation/floppy/kernel1.fs) = 91e8df83298ea05c0bec3920d491ff2277d2342a803f84b7393a160ae4a108b84b7d21a0fbf2875d9248287bf047d075e4d33c582a6083ad276e974139635172
RMD160 (NetBSD-6.0/bebox/installation/floppy/kernel1.fs) = 169667f00c760604c7302d9299448769c8e07b9f
MD5 (NetBSD-6.0/bebox/installation/floppy/kernel1.fs) = 84f875ee3ca1633777ae9db0acd4e8ae
SHA1 (NetBSD-6.0/bebox/installation/floppy/kernel2.fs) = 12e804736be0c389da64cd888c3ebfeeec2a1078
SHA512 (NetBSD-6.0/bebox/installation/floppy/kernel2.fs) = 151b541d723ea1c09bc93cd2cc6abc6a04d0adff1b20c24f53c5b0a8b72462614e64912c6b21dfc9bf4676e2a60a03cab40b44fa23f7dd70183116b4b55f6725
RMD160 (NetBSD-6.0/bebox/installation/floppy/kernel2.fs) = 159915fd391a03f34782322760d31e1966669b50
MD5 (NetBSD-6.0/bebox/installation/floppy/kernel2.fs) = 229b4ef55392a3b8c25b6b53691b5fbb
SHA1 (NetBSD-6.0/bebox/INSTALL.html) = 836f306d6806f97de5e27bce454acd5882b63090
SHA512 (NetBSD-6.0/bebox/INSTALL.html) = f903a8d35baabfc1ecf5107a6b321930a4b6966d25e8064c6a14d19ddcf67906fbfe5f227487e4120a876dfdad3ebf4664d3f22a14bed7d970bf3f29ffd2c7b2
RMD160 (NetBSD-6.0/bebox/INSTALL.html) = 4ad2f27fa58d87d0a1b22c6282894af983f4e89a
MD5 (NetBSD-6.0/bebox/INSTALL.html) = 8015ad9791e90ca1c944d1ecaab10a88
SHA1 (NetBSD-6.0/bebox/INSTALL.more) = f00ff910b68e2f4f98bfe3b08c6713a5d27c308c
SHA512 (NetBSD-6.0/bebox/INSTALL.more) = 2ab02d1a9d69c28d6b0248fd75c3395b5bc6fdefd099c1c8f199c083fc87383478818a152842f749bab553a6420df0c1427820c2b893a47cce693a65db91d175
RMD160 (NetBSD-6.0/bebox/INSTALL.more) = 357216417da8c0e9fa268285ae401c9b289f56a0
MD5 (NetBSD-6.0/bebox/INSTALL.more) = 55e950718919e06393d4eec6eeb33feb
SHA1 (NetBSD-6.0/bebox/INSTALL.ps) = 76f8926725f4b4b2c67518ba8a08c7804d8455a1
SHA512 (NetBSD-6.0/bebox/INSTALL.ps) = b730f63d8d471706dfb44484b7adb22374a51ca8e506f9b848ff7936149bf3f7b1b335b3e061b4c89904f53556bdd9bffba99a9275a5c78299705600adad6eee
RMD160 (NetBSD-6.0/bebox/INSTALL.ps) = 4edf6341d842d3626be2f6e51b9fb1e8e2e2331b
MD5 (NetBSD-6.0/bebox/INSTALL.ps) = 09cc4abbf4039ab00e41994c0e274550
SHA1 (NetBSD-6.0/bebox/INSTALL.txt) = f826c7a4f2f3500f258f40bb89fd5d996527e6a6
SHA512 (NetBSD-6.0/bebox/INSTALL.txt) = c8cc1e6e07c502ec0d6d807a67de6ffc943d260e2538d512d4322199e49e70fdcf01cb389455ea6d48655c4a246fa31dedf6c18ec58ef58c6f9b17de068b3944
RMD160 (NetBSD-6.0/bebox/INSTALL.txt) = 17fb9bcbdd888e110926df401169d8ead0aa1dca
MD5 (NetBSD-6.0/bebox/INSTALL.txt) = f256d74bbee88c7bb349992fd11f975d
SHA1 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.ABLE.symbols.gz) = 6e64a8b0dec1e052750bd03c3e240e2e4678a0d7
SHA512 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.ABLE.symbols.gz) = bcbaa47e0c29054669a3fccdade1fd64b75934f33cf34ce59675190fd62074920973675f85dc4a6a4973898df0c41b71bab47cde0bd438db97c56a13e3e5fe14
RMD160 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.ABLE.symbols.gz) = 18c790be97dc94277543ab409b0c46a186fc8ded
MD5 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.ABLE.symbols.gz) = fd246c1c2231766dfcd5307328d6791f
SHA1 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.ABLE.gz) = 837fee4fd025b1b6302e59feb9d0996a5a69069d
SHA512 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.ABLE.gz) = 8a24960901173b604e26992c2dc3b401eb20787952f6b6c0fa070b10c0d3b5a188241b9353c4ef405c4e9864cf200bf97edf692c45e2eede38b858a890fe7b47
RMD160 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.ABLE.gz) = 68c9d5f9f4278bbcd6431b34d62a4971e953767a
MD5 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.ABLE.gz) = 2f4ae7ec3badbee7e9739207599e1edf
SHA1 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.aout.gz) = 572497febedc95796b46a3bc0aecad740db1da14
SHA512 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.aout.gz) = 549c46b80cec59bc05d9c06f03fea4cfe1789c0ca51f2a95ab6f37df7b2dd6450f056e8bd60499ae8d96547c36f2ddb3367910d52b397d2ed5188e25424fe174
RMD160 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.aout.gz) = cbd8947668d7cf78e2074436ef9d0a046a919870
MD5 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.aout.gz) = 82285aa28d3e972d574d369dc25b5f50
SHA1 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.gz) = 54e76627b1fd3ee4c32046f4105f5809408f881f
SHA512 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.gz) = ef934d136eccb13da91a086e43c915a52f8206f8fda3237034f9bc03d73fa0147466efd58e231217be5179a8c1e6bc288c55b0c56bc47adb0f7bd45aa89be64d
RMD160 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.gz) = 39d74bb0008f44f71e73ae7deabaa0795b331564
MD5 (NetBSD-6.0/cats/binary/kernel/netbsd-GENERIC.gz) = 11fb2ce363dc7e39ca3a62b11d2aa62f
SHA1 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.ABLE.gz) = c55e66950e096d68620ed1fc3296f1d557ed45eb
SHA512 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.ABLE.gz) = 32253268b69d201a700dea2a855e966bf9860321e94b99507960b5e6bf870f18f856e4a7e425a236752e48bbf5875ceae38627d2d49e9a63faf0377f75c002c1
RMD160 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.ABLE.gz) = 7f7bbdcb761251ba78d769e045a53a73e67c2788
MD5 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.ABLE.gz) = 93e3ff78c78c67a40de296421d949551
SHA1 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.aout.symbols.gz) = 56742c573641715c2ad92e38a0f3206e4b6b3629
SHA512 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.aout.symbols.gz) = 41ec7eef2073b8f6fd9e34fbe9670cdaafd65b1712397db855eb44212b17a96addcd31ae62479fac09ff73cb650fa9859aab905c727305f28a6841c30c7df54c
RMD160 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.aout.symbols.gz) = c52d4eb8b33f9562c241dc296f9d9c06ee685e58
MD5 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.aout.symbols.gz) = c51c51101225f4f260e1ec92aa64f689
SHA1 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.aout.gz) = 58e8d3c14ca2714bbb91003f533ae5f967f524b4
SHA512 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.aout.gz) = e2c5f771f02b6d7896841bf5a25efc16b34830ff375e9be1f3911225313c664b7f3a0ac023e5928c2d5fb7ab584f9bde029480b84ce67bd67ee862673cf45705
RMD160 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.aout.gz) = 4282d861ac62fa542e4496cbfaca087476d60597
MD5 (NetBSD-6.0/cats/binary/kernel/netbsd-INSTALL.aout.gz) = e64438671861843fdc0fd4c9f324f04a
SHA1 (NetBSD-6.0/cats/binary/sets/games.tgz) = e1d2bb091d8202962133475bc83edb55230b9008
SHA512 (NetBSD-6.0/cats/binary/sets/games.tgz) = a147fe6d0419d53d5a428802d0bb6c9fa8a84a76c96d7d6d7c0d399e5af3a69240116463549a83d53387b0214ba0207796d7c10228d92a684fe7ff0ece3aec16
RMD160 (NetBSD-6.0/cats/binary/sets/games.tgz) = 2be515c495548ba5135ff42a75a8ebbfc35ea776
MD5 (NetBSD-6.0/cats/binary/sets/games.tgz) = 9059a6f2cd41cb58827709c9e9a1649b
SHA1 (NetBSD-6.0/cats/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/cats/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/cats/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/cats/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/cats/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/cats/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/cats/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/cats/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/cats/binary/sets/kern-GENERIC.tgz) = 5ca14afa2e8a62cb54d2ae62e3516ffc10b4568b
SHA512 (NetBSD-6.0/cats/binary/sets/kern-GENERIC.tgz) = e0189ff9d057ac445ef5ea45155f0a7045b876f7ddbd912b45fb3ee7f2c8bf83a46d167d9a916214d6cc1de9b825dcf91f35384bd3690f885acb44be5356c625
RMD160 (NetBSD-6.0/cats/binary/sets/kern-GENERIC.tgz) = 5324b2a4cfd27b7aeee8c2bc2cbb6041669e0bff
MD5 (NetBSD-6.0/cats/binary/sets/kern-GENERIC.tgz) = 6825664fe2f18e25fe7cb307b8f2923f
SHA1 (NetBSD-6.0/cats/binary/sets/base.tgz) = c458481bd58c1d1dca0329bc28de71409c1e4aba
SHA512 (NetBSD-6.0/cats/binary/sets/base.tgz) = 52550865b944057be1c4de1f30cb1ef8bf1b126258a0ef2ff7d2108b562b211149cc05b8397fb7c6142a5276e2636192c54cffbf8d424d528c0bce462f155d49
RMD160 (NetBSD-6.0/cats/binary/sets/base.tgz) = 7977e28aedfcc3e97ad9e748c6d3c7abd4aa831c
MD5 (NetBSD-6.0/cats/binary/sets/base.tgz) = 2a41824b43d9f6cfee26ea6fe3a04368
SHA1 (NetBSD-6.0/cats/binary/sets/comp.tgz) = b725c4d09fc5f7ee54c5d800e4c2e12b3af6f669
SHA512 (NetBSD-6.0/cats/binary/sets/comp.tgz) = e36e73f377ca33d2a76ca25973464ee54c70bafc05fea574fa2edc2060ad93be714a9ee6cb7173b52a75a77b47b49e50dd5b7803fd84afbd6be19b858147d551
RMD160 (NetBSD-6.0/cats/binary/sets/comp.tgz) = fcc12fec265d939130a40cd2b2d73bc5a97ab1fa
MD5 (NetBSD-6.0/cats/binary/sets/comp.tgz) = 33321cea969922b1a6ec122d03546470
SHA1 (NetBSD-6.0/cats/binary/sets/etc.tgz) = 89dc5b13c09b5325742e8e4ca8a22857b29a2bcf
SHA512 (NetBSD-6.0/cats/binary/sets/etc.tgz) = f18e1f34fd7b909422e431fa16738b047afa451ea9e7e77bbbcf4fc465835ad4b3b5b145efd2a609130d057e5adab868e544806ad7584947dfb82440e976c6f0
RMD160 (NetBSD-6.0/cats/binary/sets/etc.tgz) = 81e6058380507732dcc4157e10c849d896fc51aa
MD5 (NetBSD-6.0/cats/binary/sets/etc.tgz) = da8349c012e9f029cef0f909cc466973
SHA1 (NetBSD-6.0/cats/binary/sets/kern-GENERIC.ABLE.tgz) = c9adc81c86dda4d994d79fd37b7622475310fda5
SHA512 (NetBSD-6.0/cats/binary/sets/kern-GENERIC.ABLE.tgz) = 5a2c270a700ae0f433afba336f4270c1a9f0646ddc0839318638946f38f23b150775d9f6e2cb91d08ce327f4c2057377e5c27e525277f5c001d1c03fad852ce8
RMD160 (NetBSD-6.0/cats/binary/sets/kern-GENERIC.ABLE.tgz) = 7c83c750bff69191dc40b6bf34d801cc082f4970
MD5 (NetBSD-6.0/cats/binary/sets/kern-GENERIC.ABLE.tgz) = 18c1f0b06270fb64a47ffdb95edb7f0e
SHA1 (NetBSD-6.0/cats/binary/sets/modules.tgz) = fd876e3d182e55ff19b6121b07b8e69fa1d0df29
SHA512 (NetBSD-6.0/cats/binary/sets/modules.tgz) = a9a55cb34e467094be7e72a443b8a449078efaab3adc8a9b1496e45ea6c9fc7d5b3cc7523b30fade9263cb2f90fe9e18825bc6d5acd415a3fbea1c444ec0ed5d
RMD160 (NetBSD-6.0/cats/binary/sets/modules.tgz) = 8c6829a3b9d46c091203f11ae7809fe1087e8fbc
MD5 (NetBSD-6.0/cats/binary/sets/modules.tgz) = fec5cad3f45abd455761405b2b352a1a
SHA1 (NetBSD-6.0/cats/binary/sets/tests.tgz) = bdfb9591316f5c17b8acfff63b0ab895cb9fe3e8
SHA512 (NetBSD-6.0/cats/binary/sets/tests.tgz) = aaf8d4b6e428dbd0a162cf09d5bbb3273450b4ddacd6cdcb9ba2175cde5276fc30173466a1e394ca2fadb3e92ee7637d9e8fdfdc7f6e59436b9178efc1806b4c
RMD160 (NetBSD-6.0/cats/binary/sets/tests.tgz) = cf4cb18da6b8c46c0648d6c33556dca95035b024
MD5 (NetBSD-6.0/cats/binary/sets/tests.tgz) = 5f83acde92619b5eb23f3b21c3bdfbed
SHA1 (NetBSD-6.0/cats/binary/sets/text.tgz) = 4434e6febe84969a66d92fdfa147aec4a6f18549
SHA512 (NetBSD-6.0/cats/binary/sets/text.tgz) = 41206832dff7217dc0f61fa8ed39c7b824bd7f781318317520aaf3b190cf5d8588b5c06a8252fa8318ded7df84a104b9718dbbc4e6d2e3c62c822d79b9be4d3f
RMD160 (NetBSD-6.0/cats/binary/sets/text.tgz) = d7a3c7498b28f79cb6044f4483f6b4970c2d265d
MD5 (NetBSD-6.0/cats/binary/sets/text.tgz) = dbd9391020e7272a288c0c30002cad8a
SHA1 (NetBSD-6.0/cats/binary/sets/xbase.tgz) = ae3b8c01b219ca930ad23427b88601c98fce9ae9
SHA512 (NetBSD-6.0/cats/binary/sets/xbase.tgz) = a06f8a61a8c61df4aa2f9e0efdaede96e56765552d642ab3f048afeeada1716fbb5a0a5b3f2d511e7a146fa7d173ad0b59d5707a7c9b29216c014ec3911d3d37
RMD160 (NetBSD-6.0/cats/binary/sets/xbase.tgz) = 032ae59bb4c894f973b3fba09b9c72762f4be5f2
MD5 (NetBSD-6.0/cats/binary/sets/xbase.tgz) = d021cc89a93acc430e5cad14758b1ef3
SHA1 (NetBSD-6.0/cats/binary/sets/xcomp.tgz) = 2daa1f374f82e66d62900af2252f72a06f366ebc
SHA512 (NetBSD-6.0/cats/binary/sets/xcomp.tgz) = 803d5501dfcda968a4f098378989cc8186bda8f9dc28af6b9bc591301edfd0dd194922a0d661c75f3ced1fcdbb700e61eeb7bbf5b95ee6ec1111cb8bb9eec933
RMD160 (NetBSD-6.0/cats/binary/sets/xcomp.tgz) = 4bf6721bd75641fef50dce9cc5781b6991acd0fa
MD5 (NetBSD-6.0/cats/binary/sets/xcomp.tgz) = c78fd7987bf26697dcb6447c6818c8f5
SHA1 (NetBSD-6.0/cats/binary/sets/xetc.tgz) = 081116c5590415004358c063c13f77a32c0fa5ba
SHA512 (NetBSD-6.0/cats/binary/sets/xetc.tgz) = 129cc55a79bee0d1d821c137fa27ae1fdaa82d71cecbbecd555e7d8ab649dfccbb1c2086575e97ff66bd40a50a34ab5b45233ceb15bc9770044bc776c646d86b
RMD160 (NetBSD-6.0/cats/binary/sets/xetc.tgz) = eb721c7066e0088a6be10b2214fa7227d2b95043
MD5 (NetBSD-6.0/cats/binary/sets/xetc.tgz) = 7904920faa3acddfb9b635b88789a440
SHA1 (NetBSD-6.0/cats/binary/sets/xfont.tgz) = f6547d11ac8302fda8b2b992e99f5d8c03ef9f0b
SHA512 (NetBSD-6.0/cats/binary/sets/xfont.tgz) = c38be1439bd2a0b9a10978e1833a5ad243f3d3ed557bb05033aa6c76d1320851663e8819987b4a77c8f4f123820d82cfa950e8e486986686030aa9a40b7577ae
RMD160 (NetBSD-6.0/cats/binary/sets/xfont.tgz) = 401d3e9883c381410c7e45ee3cb1ab72e9fd54a7
MD5 (NetBSD-6.0/cats/binary/sets/xfont.tgz) = d154b32a82c67ebba30762daad5a37e9
SHA1 (NetBSD-6.0/cats/binary/sets/xserver.tgz) = 941b1622f43738a9a31860ad6464f2fff35e9f1b
SHA512 (NetBSD-6.0/cats/binary/sets/xserver.tgz) = dbbd41022a86bfd59958a96f821256bec29793952b4adfff10e3974d437e1317b8312662359a27a961d76f4c0ac7e7af9482603d6ab918668174818f9b1c1008
RMD160 (NetBSD-6.0/cats/binary/sets/xserver.tgz) = 4fb11053b3a0ad7775efdb502bbf8e1198dc0348
MD5 (NetBSD-6.0/cats/binary/sets/xserver.tgz) = 20c811752f066b472b1f578431f25d42
SHA1 (NetBSD-6.0/cats/INSTALL.html) = cbe385200e2572a8325f1559ff0b00914b6892f8
SHA512 (NetBSD-6.0/cats/INSTALL.html) = a62d6cbd6f300d4287f899e76526282ac04a538aedd201754876f8ea906eb3c486e8580b7d373eb6436ffb6c6b0d47273a3999ebfb852b1cecd7d9714931b7af
RMD160 (NetBSD-6.0/cats/INSTALL.html) = 8fd593263624ff1e7a45b7102956094a71f16576
MD5 (NetBSD-6.0/cats/INSTALL.html) = 50947ae742e64480da9f0f3964789ee0
SHA1 (NetBSD-6.0/cats/INSTALL.more) = 8343db2afce040778ea31ae38be9700674dc5f02
SHA512 (NetBSD-6.0/cats/INSTALL.more) = d40356dbe2c728dc51d6cd0efe3857a27ec9b39e2a03a01efe67733debda54b58e768218eba79dd20f97614e4a8c1138362c65acdef1aec922e157bb638f2a18
RMD160 (NetBSD-6.0/cats/INSTALL.more) = 76588eb4c081058584fc7bd3d350f896177bec31
MD5 (NetBSD-6.0/cats/INSTALL.more) = a912e620c4899d40ffaf15a00b033920
SHA1 (NetBSD-6.0/cats/INSTALL.ps) = 06fff86ec204f92418e0050417a548da63a2ddad
SHA512 (NetBSD-6.0/cats/INSTALL.ps) = 31a7ca304e80c80942d5c3b1f055251194981e7e98096fbca7c1d56f6617e5aae082a44702fcbda9cec006823d093973bdf24f1af369c532d72706b5c44243f2
RMD160 (NetBSD-6.0/cats/INSTALL.ps) = 80c4749fc0e3d9144cae101e3e83709b509eb541
MD5 (NetBSD-6.0/cats/INSTALL.ps) = aa6a7a310a23fcf771ab49fcec71a77d
SHA1 (NetBSD-6.0/cats/INSTALL.txt) = 59e12295d3e1819f7ff76701baeb4541e637ef88
SHA512 (NetBSD-6.0/cats/INSTALL.txt) = a710c598d300605c2d2f373f12f8b9b2366dd28f6cf84ccedc3c6ac3efa410e2c0a3e454489824c4a65467b0947231a6f280a4ea2cb3582bdd3d532ed994e63e
RMD160 (NetBSD-6.0/cats/INSTALL.txt) = 5f0042bf269b86bb159915585707077a98a5f06d
MD5 (NetBSD-6.0/cats/INSTALL.txt) = 8cb2eb0a3f5249382607fb963ca82313
SHA1 (NetBSD-6.0/cesfic/binary/kernel/netbsd-GENERIC.gz) = 5d2bcce895dd7d88e0d5600b6ca55a4b216bd368
SHA512 (NetBSD-6.0/cesfic/binary/kernel/netbsd-GENERIC.gz) = 85e01da3fe9ebb93fb7ae3f028297972bf4538117430185bf0885b46352f68df08fe025d91569ee0da2e17e8b83a4de99e1f3d56d2c5cb31ede3beb366afa157
RMD160 (NetBSD-6.0/cesfic/binary/kernel/netbsd-GENERIC.gz) = 7c946881e8b671248896e5e620f8125426911450
MD5 (NetBSD-6.0/cesfic/binary/kernel/netbsd-GENERIC.gz) = 39a8c5037a6266d0bb6df8fdb5c14c7d
SHA1 (NetBSD-6.0/cesfic/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/cesfic/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/cesfic/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/cesfic/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/cesfic/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/cesfic/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/cesfic/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/cesfic/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/cesfic/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/cesfic/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/cesfic/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/cesfic/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/cesfic/binary/sets/xserver.tgz) = 1d20fd039546a66b428b854c659ea5e2ee9f8114
SHA512 (NetBSD-6.0/cesfic/binary/sets/xserver.tgz) = 091d09049d0af5555feec02065234e281677aeb3ecbab5a19413d924a8b58dc7f64401ef68e68bc52d8b6eeb73e091d351c45805374b8bf276b0181415968561
RMD160 (NetBSD-6.0/cesfic/binary/sets/xserver.tgz) = feef5d52278d23637d7b8ba77639bfd96918e90e
MD5 (NetBSD-6.0/cesfic/binary/sets/xserver.tgz) = b2bd6057c374376388766fc8667eb9a9
SHA1 (NetBSD-6.0/cesfic/binary/sets/kern-GENERIC.tgz) = 682508a28054a48a7242a494917ac4889bc67992
SHA512 (NetBSD-6.0/cesfic/binary/sets/kern-GENERIC.tgz) = 0225b3aa055ad8d4c5b3029cea2459cc27e5670cd4aa50529c3490f93692273fabf6fb1d47336ca82e7b9b97c9cd6eaa6da29f1c71076c36664952cd2e8f92cb
RMD160 (NetBSD-6.0/cesfic/binary/sets/kern-GENERIC.tgz) = 39e19612b53e389e4da7ab24ad463d19da13f296
MD5 (NetBSD-6.0/cesfic/binary/sets/kern-GENERIC.tgz) = ae69c0b436f251484d22512f3207263f
SHA1 (NetBSD-6.0/cesfic/binary/sets/base.tgz) = df22763e9478973636b3651d21f5cc5c5b065515
SHA512 (NetBSD-6.0/cesfic/binary/sets/base.tgz) = bfdab8e7b9dcbd14d7e6321edbfbf06c45eafc0c85dcace6879ff485d45262706dd29a165c9d89a041618244977763af4c0f0c93d48ac39c96c5a4446122d2b9
RMD160 (NetBSD-6.0/cesfic/binary/sets/base.tgz) = 1359d35509d93695dd0cfda9adead93875618ddd
MD5 (NetBSD-6.0/cesfic/binary/sets/base.tgz) = 232f3d5ccd51c66577dc50e25c149476
SHA1 (NetBSD-6.0/cesfic/binary/sets/comp.tgz) = dbd27c5388fd2ccfa90ceaf819f5fb3b86bac78e
SHA512 (NetBSD-6.0/cesfic/binary/sets/comp.tgz) = 0657745dfedf6283919628d26665d30374807c78e049cdf5d9ce3d3dca54275ccaa992b5c21ef5c850decd1def118cb14c1e89a2bbdbe55b905e0f8100e259ef
RMD160 (NetBSD-6.0/cesfic/binary/sets/comp.tgz) = b0bdc0fd2ca77bdadb6d03a465333bd9dca2a016
MD5 (NetBSD-6.0/cesfic/binary/sets/comp.tgz) = 8ff8a532ccd1d24e93b4ca53a41d1efa
SHA1 (NetBSD-6.0/cesfic/binary/sets/etc.tgz) = 20b9dc0339ed74a93dc82fc1b60f9b3fbf6b4b8d
SHA512 (NetBSD-6.0/cesfic/binary/sets/etc.tgz) = 593ef36e7f37c6153b8068a2f8b433a6bbe821a2bfcad7ea6ce4e30bf5611172aa94e82b7aca57c141d84381c597306779e5023cab4ff28fa415b6b6bac69e3d
RMD160 (NetBSD-6.0/cesfic/binary/sets/etc.tgz) = cd6bcc3d96f38f8e9a00cfedbeb4463a27708d9d
MD5 (NetBSD-6.0/cesfic/binary/sets/etc.tgz) = 343a0c8c930807f71f37e7116bf8f391
SHA1 (NetBSD-6.0/cesfic/binary/sets/modules.tgz) = d5191e8899d7345c88bae6f65c1fd249b02a0063
SHA512 (NetBSD-6.0/cesfic/binary/sets/modules.tgz) = 22e74e63cdc763effdcb58c7d549c0811dedee37953a9669dca09010dde364c6ba7440a29ad1e1e634a7c44b69cbca65afe006b9f564bd10bc4f0852cb55dc75
RMD160 (NetBSD-6.0/cesfic/binary/sets/modules.tgz) = 5ad1d1c039de7da6dd1183ea6f4e9c319931dc86
MD5 (NetBSD-6.0/cesfic/binary/sets/modules.tgz) = c0dda9a6803a213f3bca675728056dfd
SHA1 (NetBSD-6.0/cesfic/binary/sets/tests.tgz) = fdf867451cd371c25672448a1dbc7c204e8534b7
SHA512 (NetBSD-6.0/cesfic/binary/sets/tests.tgz) = 9f137af1458c138a40dd2cf293e58deaa93a7145457127ddddabad1c23f380e36950d21cc6051bc5a4b101a632b9610a74cc3fc4bf6129e29e57ee8fd9d2da6b
RMD160 (NetBSD-6.0/cesfic/binary/sets/tests.tgz) = 0c27b80fb02f5ca0dfea6854d0df2818df52b9e1
MD5 (NetBSD-6.0/cesfic/binary/sets/tests.tgz) = 2cb77946d9b305a0e03a6ac5fed04ff7
SHA1 (NetBSD-6.0/cesfic/binary/sets/text.tgz) = d9f8aaf67b6ffd9828d447b5a6b208ea5bffcbd3
SHA512 (NetBSD-6.0/cesfic/binary/sets/text.tgz) = 931dc3aec206ff95df08c36ed09a01caa2b10bdff34affc723e7ea0c6d2cdfc493d02b9875568ec9276c2d35a74cc782ec8c8cee10ad62879831b1d728cd3d62
RMD160 (NetBSD-6.0/cesfic/binary/sets/text.tgz) = cec72f88379d8c6b30730568eb653de4a07f8c8d
MD5 (NetBSD-6.0/cesfic/binary/sets/text.tgz) = 1daceefaf9afa99dfb01cde29485cfe2
SHA1 (NetBSD-6.0/cesfic/binary/sets/xbase.tgz) = d052985aaeadbeaad2aab5aab60ef68f5527a9eb
SHA512 (NetBSD-6.0/cesfic/binary/sets/xbase.tgz) = 83089a0b043d1413d865ea766cf1c50da9d1c12165e05bfc72f26b4d4513549119cc716ffebda1fd82b20ff1766c579aa000b7c10b4f29fcca49f0ff9f05177b
RMD160 (NetBSD-6.0/cesfic/binary/sets/xbase.tgz) = 3a8239f2ecb4e13e2352e3b09c3496f51dd12fb4
MD5 (NetBSD-6.0/cesfic/binary/sets/xbase.tgz) = 61c565148bfbdd631981e7d42845c147
SHA1 (NetBSD-6.0/cesfic/binary/sets/xcomp.tgz) = f3eda836d68a0b648fec7072373ae0b7267ab7d4
SHA512 (NetBSD-6.0/cesfic/binary/sets/xcomp.tgz) = c448fa8a9665c6cd19aa05eae8456a06e0301fed427eff4d20f4a337e398c4346b285bea3c46924eada81d2bd1c34bcfa91cfd36d15eeb8ca6e9f7606514f131
RMD160 (NetBSD-6.0/cesfic/binary/sets/xcomp.tgz) = 68333053bfff6afe3fa5e7555e58039ca42b679e
MD5 (NetBSD-6.0/cesfic/binary/sets/xcomp.tgz) = c318f3138aabcb6453f77a6dfa4778c7
SHA1 (NetBSD-6.0/cesfic/binary/sets/xetc.tgz) = 5965357819e9e3e64a9cf0963595a5870699cd82
SHA512 (NetBSD-6.0/cesfic/binary/sets/xetc.tgz) = da1f214b6de2a4a1ef227c608b741a076a7d4ba491131fadd33ebbf0782fd6c0d1ade406b4aa92c973c9f87ee70b2dff20d3a710df08201aac3c1a9ad339fe31
RMD160 (NetBSD-6.0/cesfic/binary/sets/xetc.tgz) = 7151d3746cae1d748a566356431ca8ba4a5b51f3
MD5 (NetBSD-6.0/cesfic/binary/sets/xetc.tgz) = 616f7da77033039c3b6972c33223215f
SHA1 (NetBSD-6.0/cesfic/binary/sets/xfont.tgz) = 46a06412b08381288aeb30ddc5b5a30d09530fcf
SHA512 (NetBSD-6.0/cesfic/binary/sets/xfont.tgz) = b5c5c517924de3ae3bc0fd094c0be19501668f4ad25feae8768c5c17065ff90b9b60760afa47ee93909fb92c4c12a818a95f0744e988a04e20be51f0d5dd4d66
RMD160 (NetBSD-6.0/cesfic/binary/sets/xfont.tgz) = b58231d11d9ea95811dd0a6f548b6bdb18eae61e
MD5 (NetBSD-6.0/cesfic/binary/sets/xfont.tgz) = e52bf6c25b8e5fcef899329b68856e5e
SHA1 (NetBSD-6.0/cobalt/binary/kernel/netbsd-GENERIC.gz) = c9618dcb53044599e6397eac6bfd5dbd6f106cf0
SHA512 (NetBSD-6.0/cobalt/binary/kernel/netbsd-GENERIC.gz) = f46c57b34ffc0f0140094384a10ea7c3d0140f315fceed6b02a1ec4770d6114eb1e995020fd056e03af71185f0448fde86829f5921b961a903bf0c275ff730e3
RMD160 (NetBSD-6.0/cobalt/binary/kernel/netbsd-GENERIC.gz) = 4f69bb230b7c20fc591af0ff0e788e0f9a6dfff7
MD5 (NetBSD-6.0/cobalt/binary/kernel/netbsd-GENERIC.gz) = 4ef49f4e446b38c2dcecf61dbca8fa25
SHA1 (NetBSD-6.0/cobalt/binary/kernel/netbsd-INSTALL.gz) = 8c6edfcb94e9c0bfe670a78e6599dbba4787ef6b
SHA512 (NetBSD-6.0/cobalt/binary/kernel/netbsd-INSTALL.gz) = aa2dd1067f3fe253b43eb3a1ec0c8119a717c43236f1df4d72a4ac5667de310ac904bbca89588b9438a3aacdee5bb3d569e170dab17487ec5c6d9c148086085f
RMD160 (NetBSD-6.0/cobalt/binary/kernel/netbsd-INSTALL.gz) = 54a79af6fbce4c15b3415137f67a35e0d5e34daa
MD5 (NetBSD-6.0/cobalt/binary/kernel/netbsd-INSTALL.gz) = 112e15f9d41a31dbe6eb6864d62633ef
SHA1 (NetBSD-6.0/cobalt/binary/kernel/netbsd-RAMDISK.gz) = 4075b370e08ffe708669556c584e79073a9280b0
SHA512 (NetBSD-6.0/cobalt/binary/kernel/netbsd-RAMDISK.gz) = ae5eb10935acf635e7a3abf985b86fbe9a57a071dea8ef7fe08ff5fd887901331d603e61d1bfeba330489534c4accae61e8c63fcc0cbb4a248e8fffa53d59fec
RMD160 (NetBSD-6.0/cobalt/binary/kernel/netbsd-RAMDISK.gz) = 9537810097f1bfe7c0cb7a4154639c8374a17c63
MD5 (NetBSD-6.0/cobalt/binary/kernel/netbsd-RAMDISK.gz) = 0a477e1b678ab3c2906eef7d9082d2a5
SHA1 (NetBSD-6.0/cobalt/binary/sets/games.tgz) = 27d9f98fecb3495e51939df9f107e81e527af26f
SHA512 (NetBSD-6.0/cobalt/binary/sets/games.tgz) = 46bb4e10313cfffaaf4f7fa2157474ef940f94bcc9f84b46f829c7615bbf45e4e01edee7b7efb84bc28561e0712c37cfc737f9c2a0d9a73b67c80c8aa0f3a9af
RMD160 (NetBSD-6.0/cobalt/binary/sets/games.tgz) = 947feea90ec2698a65d555cb78a29a001b086589
MD5 (NetBSD-6.0/cobalt/binary/sets/games.tgz) = 355f00978fed8e8da6a251e591be5387
SHA1 (NetBSD-6.0/cobalt/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/cobalt/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/cobalt/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/cobalt/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/cobalt/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/cobalt/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/cobalt/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/cobalt/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/cobalt/binary/sets/xbase.tgz) = 76468fcada5a9e26652aa9f0f9f3c94829a317e9
SHA512 (NetBSD-6.0/cobalt/binary/sets/xbase.tgz) = b2adcfc2dedef30b21b885c8fc757d0479541744a280a9997fbeaa9c881009ab4b0019e0001a540a8eb6db5f84db87b365e420bb2d4270af209821bdb080369c
RMD160 (NetBSD-6.0/cobalt/binary/sets/xbase.tgz) = 1e0074d03d9fbb3373e0a427b25cc7097bf6d3c8
MD5 (NetBSD-6.0/cobalt/binary/sets/xbase.tgz) = a8f29ec2e3768405942f97af3dd08ee7
SHA1 (NetBSD-6.0/cobalt/binary/sets/xcomp.tgz) = d8ac89f3f6ff20abb0adabc49227afdb74aa2040
SHA512 (NetBSD-6.0/cobalt/binary/sets/xcomp.tgz) = 4f03213fb061dae9421747209628d06ede6bbb71a7560a9e1a54796b907f1620146a33fdce52974227a21acedcc210b99e15c4fdd36432eba60c1041a92d8cee
RMD160 (NetBSD-6.0/cobalt/binary/sets/xcomp.tgz) = 6be5e72b9b57c9bdf8ed8ea127e2a73830feeecb
MD5 (NetBSD-6.0/cobalt/binary/sets/xcomp.tgz) = 59a7ea141587417be38ab1f6056cc780
SHA1 (NetBSD-6.0/cobalt/binary/sets/xetc.tgz) = b9f6b1a54ffab37b11f4c6e717b481cd087fb609
SHA512 (NetBSD-6.0/cobalt/binary/sets/xetc.tgz) = 72f35e65f3a90967e250a2006d500619b3824f7f5c3d7494554f153ec5cda5b28c445f28c34a1e4463f40da5855e8d4257726411796967fe37763d90cf7fe461
RMD160 (NetBSD-6.0/cobalt/binary/sets/xetc.tgz) = 1169c246f91189fd0c4ae86ac27a92f5c2edda3e
MD5 (NetBSD-6.0/cobalt/binary/sets/xetc.tgz) = 8f3c52cc5af1cafaf7b0d0f884a5b537
SHA1 (NetBSD-6.0/cobalt/binary/sets/xfont.tgz) = d06848571034f3b7fc975082f1de37b912acb18c
SHA512 (NetBSD-6.0/cobalt/binary/sets/xfont.tgz) = 66a1013c935bee5e7e5d24325a527522306bcded7ced63790f2e58831b3ed91176a1b9510b2159be63504d5167d48d534919927ffc6e225ae2bb2487f90a9ede
RMD160 (NetBSD-6.0/cobalt/binary/sets/xfont.tgz) = c9502843a5d4dd7e996e8d158c519cf7f8575cbd
MD5 (NetBSD-6.0/cobalt/binary/sets/xfont.tgz) = e812899176eac2bf65a6d949ac3f3424
SHA1 (NetBSD-6.0/cobalt/binary/sets/xserver.tgz) = ff805022f409202f3346c45475c7ec0f8187218e
SHA512 (NetBSD-6.0/cobalt/binary/sets/xserver.tgz) = f2bb3810a5572b519980126a3d934f63b5ab5c2fcf9fb8166c556bd3cc14cfaa151d582086bb1506ac9ac7e21d4e70a5cd5c3faed96212d8a14a0b5bafdcc4c1
RMD160 (NetBSD-6.0/cobalt/binary/sets/xserver.tgz) = d5ae336337cc77f7f55f542b3457918d6fc0efe8
MD5 (NetBSD-6.0/cobalt/binary/sets/xserver.tgz) = 92447a1560887d337d22e2deed0670a6
SHA1 (NetBSD-6.0/cobalt/binary/sets/kern-GENERIC.tgz) = d085ff7c5cd7f260cc1aaca4c2e54c26c948d893
SHA512 (NetBSD-6.0/cobalt/binary/sets/kern-GENERIC.tgz) = 0c1ad1014af64d059a5ce8261e46534e0eaed8702440b93f5109c19ed9fb0b2dbf37a23cc28f1a688ca45f9e721dd0f4a1ee6771990a835a459cd633ff0f4887
RMD160 (NetBSD-6.0/cobalt/binary/sets/kern-GENERIC.tgz) = cfecc943b0e501f53558c87be282a948469c57fe
MD5 (NetBSD-6.0/cobalt/binary/sets/kern-GENERIC.tgz) = 630b91f62d422855f30b33590119e9d7
SHA1 (NetBSD-6.0/cobalt/binary/sets/base.tgz) = 90870f3189063cb2c4a2aa81d1c68d835e48ee25
SHA512 (NetBSD-6.0/cobalt/binary/sets/base.tgz) = 9a962443ad380ef6d4dce623bb41e6b1aecd452b0adfc6e51d1c17610829f17a1b02cf5be2e539eeafcc7ff47ae79f8f7a367eebd74ea6cb70e0fc57eebbf27b
RMD160 (NetBSD-6.0/cobalt/binary/sets/base.tgz) = 12111c0c67c4f8fbb3c54c2c3480e73caee6f33c
MD5 (NetBSD-6.0/cobalt/binary/sets/base.tgz) = d7aba51f2552fb2e19af9adbd673dff9
SHA1 (NetBSD-6.0/cobalt/binary/sets/comp.tgz) = f22b026d0b9c1e7725e4db4b9abbda1160a9c80a
SHA512 (NetBSD-6.0/cobalt/binary/sets/comp.tgz) = a0a6f6185d30e28c4db98189134bcb5e2dc9820c748911e36483b84fb598c5b122c10863ccff82a2f2556734004fab2c9ac59cbc1530ec79ec535e4e426d4e5c
RMD160 (NetBSD-6.0/cobalt/binary/sets/comp.tgz) = ed9d28519a8cc55a2806d6c1df0953c535075b1d
MD5 (NetBSD-6.0/cobalt/binary/sets/comp.tgz) = 44112b73f63f4cddda4ac8ab8f5ba1f5
SHA1 (NetBSD-6.0/cobalt/binary/sets/etc.tgz) = 7109941b0df321286aaf226ee48222fbc7ad692f
SHA512 (NetBSD-6.0/cobalt/binary/sets/etc.tgz) = 1d18f21beb41dd07c5744b127209b5de8f5fd2fbbd109c3c65dbe73126e2a987e1e1d08a68668f597ecce1d0e9875fe349bf89d74d3add850a3c4554808cda07
RMD160 (NetBSD-6.0/cobalt/binary/sets/etc.tgz) = 1687c63ca2d75377354754bb425c02bd6907c445
MD5 (NetBSD-6.0/cobalt/binary/sets/etc.tgz) = 3b36b16b387597ceb2233eaa72d3bea6
SHA1 (NetBSD-6.0/cobalt/binary/sets/kern-INSTALL.tgz) = cfefd1fef2db13b66730678c9df067ab208158ad
SHA512 (NetBSD-6.0/cobalt/binary/sets/kern-INSTALL.tgz) = 2a62524f994a5034d60f679b1d30eafe60dafb35216458055ad9e25f3c7053f4556f1edd233754e0e4eb4a79373fb814c28e832b67d9de69035513b819d5080f
RMD160 (NetBSD-6.0/cobalt/binary/sets/kern-INSTALL.tgz) = 894035c7e551dfee7ae53cd1e7b358876fef240f
MD5 (NetBSD-6.0/cobalt/binary/sets/kern-INSTALL.tgz) = c6f35e3463064c86be5b3229fdf23ce1
SHA1 (NetBSD-6.0/cobalt/binary/sets/modules.tgz) = eda182e3418c72ab2d0bb33efa014d77be461b73
SHA512 (NetBSD-6.0/cobalt/binary/sets/modules.tgz) = 52a988323ace3452cdd3c8db6d03e26f15b3d9570314c8e8efd80169676d36209a8dee6736d4a1aed53beb3ad3ecd6fe8ea9d4b3b0a85417a6540cc4f6ca2154
RMD160 (NetBSD-6.0/cobalt/binary/sets/modules.tgz) = 0610022e6cce6e576d3bbc6f3bca284d36eff821
MD5 (NetBSD-6.0/cobalt/binary/sets/modules.tgz) = 2850ca4ea69cbac5274f2307a48394f1
SHA1 (NetBSD-6.0/cobalt/binary/sets/tests.tgz) = ae62656c92b1883296087eec3534d7542356c79e
SHA512 (NetBSD-6.0/cobalt/binary/sets/tests.tgz) = 35a22f1ede3db17eaa2c29645c5675fe5a2c4fd6deefc5d372cbdf1ac61c592fe7acf65486a8c576d3ee8651ddb3afa33e39e644de18d2831d25c89aa74c734b
RMD160 (NetBSD-6.0/cobalt/binary/sets/tests.tgz) = fba7399b23a918470dcd108a422bb4f998a94a42
MD5 (NetBSD-6.0/cobalt/binary/sets/tests.tgz) = 21dee182d4f02df926ce52688347af72
SHA1 (NetBSD-6.0/cobalt/binary/sets/text.tgz) = 58049b832c3722ddb05f8eca622f77ff8e4e8a4f
SHA512 (NetBSD-6.0/cobalt/binary/sets/text.tgz) = 4eb3919068d26f193fcca2a30e27152421be198b313fa9700612a0d9d094d67d329b2e72e70b02c4789c1430378269f35f0758042faf60c26a18e293947c7db9
RMD160 (NetBSD-6.0/cobalt/binary/sets/text.tgz) = a2eaa36b8c306e0cd6a9e17218aad574d4fc3508
MD5 (NetBSD-6.0/cobalt/binary/sets/text.tgz) = e0f6aa8df8d17719f265922d2aebc787
SHA1 (NetBSD-6.0/cobalt/installation/boot.gz) = 59e1785a9e2ea35df01f4726b62f6fbbc74c56df
SHA512 (NetBSD-6.0/cobalt/installation/boot.gz) = ad2baef23c54d600d49e7777b924c7d79c8e7e3f74107e9fcd939b16811b31979391ce8f3ebeb4b31810fef2cec13df063d31e4b3a5b5c57b550156b796a4410
RMD160 (NetBSD-6.0/cobalt/installation/boot.gz) = 9ac9dbfb3932a92e2b4e03664bdb8dbed4fc0e71
MD5 (NetBSD-6.0/cobalt/installation/boot.gz) = f4b10cb6244d765eb7a9a6a3ef1d09ba
SHA1 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.bin.gz) = 971d8de48c6c6be81f723ada4b7c509dd152b86c
SHA512 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.bin.gz) = ac63feaca53a1abad181877bec95098e06e89343cd783f309ed3d223a80005d49fd5b3420a36c8e2894389373a9a8d4a023f134e1ff847f240864b8f529adfbd
RMD160 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.bin.gz) = 5abfe7b82ce9b81c69818cbb533a9d7e69f01081
MD5 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.bin.gz) = 6ed9526e1fc60db1cdab3dfedb2a4ffe
SHA1 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC.bin.gz) = 7950c0154a603614e8aaa75da019ffe266fe36de
SHA512 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC.bin.gz) = 2421894a9dda95a17a30b6a4d87e350f7bd17f467362255eb22c8a56e0ac1e54592136d0b0d218c16a14afb8684bb8eaf11442217af1c217ea4588d6290c761e
RMD160 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC.bin.gz) = 326f220c689b47fb30e91d483a009246993da43e
MD5 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC.bin.gz) = acaf2f6174037777dc0dfc175da86403
SHA1 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC.gz) = 85e46b6fa4ab45b810f6c3f4b3268cba0f895cb4
SHA512 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC.gz) = d3f2e7306c8b8ea0969bb68e5888ba16436632479a6555a4fffe2479c4cea1dce048c2d18b6653fe1946b9a9bd2d2df1da172d71df45c6eca59a1c91e81f51a9
RMD160 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC.gz) = 31820c3a9b7305792ca4246aaa0d8e5474c2a8ba
MD5 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC.gz) = b48de4452de2d94b74d617ba30b529f8
SHA1 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.symbols.gz) = 83040bbf6395952a68fbfde410db2729c8d814c9
SHA512 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.symbols.gz) = c48a2bb6dc2bac588b3e9690dbeed3b85361439e40e3410220c0b9fcb70e2e6a2bdffd24ddb428f740895ef34c3e56955f2877509441be0c158e97407656da24
RMD160 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.symbols.gz) = a6f75d1e098e0bebb9da8f6149114de13416b210
MD5 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.symbols.gz) = d72d4a1170c9f40c41c951eb2f40dd9c
SHA1 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.gz) = 98765e3c532799a669420e9b6a89b22033059d66
SHA512 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.gz) = 34a64fc5eb79bec00adf55f9788139730c5f35bb8e05f0b00899776836aa0dc7c372482ed5cb9341fa7354d5f8c76ed6b879ac516210d89e33794efbccf40127
RMD160 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.gz) = 5a4349f94c8afe0d7ad4c6ca4b02d766ddfc63b3
MD5 (NetBSD-6.0/dreamcast/binary/kernel/netbsd-GENERIC_MD.gz) = 7c4659ca64c7f31b205e8d1ce4166597
SHA1 (NetBSD-6.0/dreamcast/binary/sets/games.tgz) = 2040b4af557a62f2ade5fc9fcfe4cab3df88c7e2
SHA512 (NetBSD-6.0/dreamcast/binary/sets/games.tgz) = 8b5071337dc844042537c24c75660e61c9709cd10cbbeab059a84da6d482260b73365ff3870f85060c130a291b6f02be174d288127ca62ce4d291a6982192317
RMD160 (NetBSD-6.0/dreamcast/binary/sets/games.tgz) = 28acfa5a131d52df7c157e2edca1dd0f20e774dc
MD5 (NetBSD-6.0/dreamcast/binary/sets/games.tgz) = d82c46dbc0021fd4694e464042936052
SHA1 (NetBSD-6.0/dreamcast/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/dreamcast/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/dreamcast/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/dreamcast/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/dreamcast/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/dreamcast/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/dreamcast/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/dreamcast/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/dreamcast/binary/sets/xcomp.tgz) = 53aa1e2a910e8eed09dc07f8b724804f53fe5e2f
SHA512 (NetBSD-6.0/dreamcast/binary/sets/xcomp.tgz) = 24645da85419528015cdaefd0c835741fa801392e795b363bdfc39558e75e7f384511c0668d12e2440114b7c543c7532aaec438048f08d35c54c9c7a748f20c0
RMD160 (NetBSD-6.0/dreamcast/binary/sets/xcomp.tgz) = 60a461ffc1f1562274110d65c72a541ee131cd21
MD5 (NetBSD-6.0/dreamcast/binary/sets/xcomp.tgz) = f1d2650a8d7a4ecd60fda8e1aaf8c09b
SHA1 (NetBSD-6.0/dreamcast/binary/sets/xetc.tgz) = 1636a6fe3e01f6d8a07a19cd204a6807c8caa4c7
SHA512 (NetBSD-6.0/dreamcast/binary/sets/xetc.tgz) = 9cea4ba4af287dd57e591d4269f2f2582010250316f323746ac56b12cefcf2590b93fbb0197c665f61b1df41dcca62d17fb6ad80d4b1feb8ac8265e91e15074f
RMD160 (NetBSD-6.0/dreamcast/binary/sets/xetc.tgz) = 5f68e3f96f7bb9ae07fccdb1d058e4b1307b4b29
MD5 (NetBSD-6.0/dreamcast/binary/sets/xetc.tgz) = da84d97230787283e215a4ee1c4053ad
SHA1 (NetBSD-6.0/dreamcast/binary/sets/xfont.tgz) = 97439ccf6840b18b1e9885f5504a401bf1a93650
SHA512 (NetBSD-6.0/dreamcast/binary/sets/xfont.tgz) = 3d15541f7d7ae2128f44355bf110415adfa4e5182829259d2c6f7a4217feff99e236d13c5bb7bbeb724ff1a09ac1f58f02bdd6eef528bc416ec3118db6fc7d45
RMD160 (NetBSD-6.0/dreamcast/binary/sets/xfont.tgz) = cdc9fcdb5fd5032803d5bba18db58f3ea6d62c85
MD5 (NetBSD-6.0/dreamcast/binary/sets/xfont.tgz) = a8466a0477c6c443eafe99e9d1c61131
SHA1 (NetBSD-6.0/dreamcast/binary/sets/kern-GENERIC.tgz) = 8f11a046a1e6277a5f90f44142735f39375f949c
SHA512 (NetBSD-6.0/dreamcast/binary/sets/kern-GENERIC.tgz) = 12786eb317effcd2f7f108a139a95b99c6026e75370853d3761292d431063859dfff26ab4a2bdf5c8bfcfbda4deaed9025583b30367e00074dc875d759f1cf76
RMD160 (NetBSD-6.0/dreamcast/binary/sets/kern-GENERIC.tgz) = 464ed36bd11cfa0b181632d2b1c54f34a511fd14
MD5 (NetBSD-6.0/dreamcast/binary/sets/kern-GENERIC.tgz) = 1cd64692f608f13541288f3ff263fa82
SHA1 (NetBSD-6.0/dreamcast/binary/sets/base.tgz) = ce23182c9b35fcba35fce0e3f12578a988a8ecc7
SHA512 (NetBSD-6.0/dreamcast/binary/sets/base.tgz) = 20dc7d634449bb7e57ae55458aa0f2d53c5f03f92df3df5a6429c1f3192871ed2fb8ec64700a045591d308a31baa3212c8a7ee6f3bfcccc05974557bc2e1ebad
RMD160 (NetBSD-6.0/dreamcast/binary/sets/base.tgz) = 7fbdce2370f91ad3aa3abec889333b59755b0f4b
MD5 (NetBSD-6.0/dreamcast/binary/sets/base.tgz) = 3aee642fc36860e63bf85392888ab2a2
SHA1 (NetBSD-6.0/dreamcast/binary/sets/comp.tgz) = c288c3890b514eb00673399571a6d22d4abf7608
SHA512 (NetBSD-6.0/dreamcast/binary/sets/comp.tgz) = cfbe2ddaf47453f9b20a74890e3585fd82a23d25489378b41276848019971f1e740e00203418db16e1832777b96d3b16eeeae1a4fcdf9551b93582395c5a3838
RMD160 (NetBSD-6.0/dreamcast/binary/sets/comp.tgz) = b79a42001602eb84e650fe34a25ac20e5773703e
MD5 (NetBSD-6.0/dreamcast/binary/sets/comp.tgz) = 57451cb40d16c7d6dfb960eb4848754e
SHA1 (NetBSD-6.0/dreamcast/binary/sets/etc.tgz) = df82ad0e59f00bfa96ca079e1dac2b0f772a6b1b
SHA512 (NetBSD-6.0/dreamcast/binary/sets/etc.tgz) = 87d552690ee356eb3a82c763ddf3f44ed57a5d0d8616974570ba21a2901371e6ee6e54d0b25cb5607466f01cbabca86fe096572f395c74aa6dd98be45e4c7ee8
RMD160 (NetBSD-6.0/dreamcast/binary/sets/etc.tgz) = bbc9f48831c1483ffe9c9aab353995e162a5a385
MD5 (NetBSD-6.0/dreamcast/binary/sets/etc.tgz) = f0881f75efa980e58cea0c0d60f55656
SHA1 (NetBSD-6.0/dreamcast/binary/sets/modules.tgz) = 243ab8d8f5d3375fbd17efb668c1abe45d0d1487
SHA512 (NetBSD-6.0/dreamcast/binary/sets/modules.tgz) = 193c158f2d2f64c3b571c6e10d97bdbca141faf5ee1a970e3c1e194970e2723e74d8372e0110b52d11e210e860102ebf24012a2267a0f98dedeceb0cc60429f8
RMD160 (NetBSD-6.0/dreamcast/binary/sets/modules.tgz) = 788e45cc0d8a03191c14f3fe0d429ae331d96895
MD5 (NetBSD-6.0/dreamcast/binary/sets/modules.tgz) = 83acdebfad50fc820d420b4e74c87762
SHA1 (NetBSD-6.0/dreamcast/binary/sets/tests.tgz) = 263b82b9d13d080e46cd27e4a6e5df9d43e6f74e
SHA512 (NetBSD-6.0/dreamcast/binary/sets/tests.tgz) = ffc4a4348c68f8b06f1387d60e1d5b1bcc1b42ec6d255cd267dea08aef4874591dfc95f69ce416333679cdbc124540cb9b92e58ad9f522e6f34eafa2f6dd9ef2
RMD160 (NetBSD-6.0/dreamcast/binary/sets/tests.tgz) = 2518cdf17a8f82dcf39525e2928ef52e46e32325
MD5 (NetBSD-6.0/dreamcast/binary/sets/tests.tgz) = 2f0af9ef7e65f9afb0185e305a206183
SHA1 (NetBSD-6.0/dreamcast/binary/sets/text.tgz) = 577c45d37aefc65d2787bd989002bd3748440b56
SHA512 (NetBSD-6.0/dreamcast/binary/sets/text.tgz) = f1a7e6beac2f881e8b9d356491e4b09d7b92b0a0ed33462575db59ce8effae6ca2a1989e4fa9b78fc5d012b118b2f99a2e2fcd7505a0ca80eb61684d5bceb096
RMD160 (NetBSD-6.0/dreamcast/binary/sets/text.tgz) = 2688ae8578acf1b6b8937abf4d05cb09e7e2ce8d
MD5 (NetBSD-6.0/dreamcast/binary/sets/text.tgz) = 0cc81795b0f4ddd432f3dfa01eb2a530
SHA1 (NetBSD-6.0/dreamcast/binary/sets/xbase.tgz) = 7305b994f65ffa7d67a9f99d0146e705f81ec723
SHA512 (NetBSD-6.0/dreamcast/binary/sets/xbase.tgz) = 904f0c21201d51b58c366a984eb20191c620287b77b12b520003d48fb0ce1a3ae10fa871c3514598a74071ce0cc1321ff781b96195a13e81c75d35cdc7958030
RMD160 (NetBSD-6.0/dreamcast/binary/sets/xbase.tgz) = faa8760bbee9d3b2a21e3b62fddb641faf939ee9
MD5 (NetBSD-6.0/dreamcast/binary/sets/xbase.tgz) = 655c33ebf6f37bd90b1921b220dff2ac
SHA1 (NetBSD-6.0/dreamcast/binary/sets/xserver.tgz) = df506c92cf396d2d59ee212c03a10f44129f51a3
SHA512 (NetBSD-6.0/dreamcast/binary/sets/xserver.tgz) = ee8b15abf9e8115ffc50deeb372e9c7dfe9412f8e94d65b3ed78f78fba63e9282b668c8fc2fb2a22437f950adafd1dedf2e9baf395fa945d04eeb50db0d1f5cd
RMD160 (NetBSD-6.0/dreamcast/binary/sets/xserver.tgz) = 73bcaffdf98d0a5c4d5c22c97c5476c1b5901eec
MD5 (NetBSD-6.0/dreamcast/binary/sets/xserver.tgz) = c5e16ae2cf663c0373fbb82f64f4f48b
SHA1 (NetBSD-6.0/emips/binary/kernel/netbsd-INSTALL.symbols.gz) = 7ffd5f42e92a4ae32ca78766815db12dbfd4c4f6
SHA512 (NetBSD-6.0/emips/binary/kernel/netbsd-INSTALL.symbols.gz) = f7a50a9ee148ac9566dbc3722b14d904c02a3cbf285ea8efb10a24b8a93db514abbf825dbde48a8c494da6b02f48408b9414a6d8a45ed61f334afd2dd148aa71
RMD160 (NetBSD-6.0/emips/binary/kernel/netbsd-INSTALL.symbols.gz) = e03b53e71c192a3de6d720b0502e0597f2ab0aee
MD5 (NetBSD-6.0/emips/binary/kernel/netbsd-INSTALL.symbols.gz) = 014979ca9f7b5feb3664479cd34dd2d3
SHA1 (NetBSD-6.0/emips/binary/kernel/netbsd-GENERIC.gz) = 663be1ef5f9177715d8a299d3e610b21a942dc16
SHA512 (NetBSD-6.0/emips/binary/kernel/netbsd-GENERIC.gz) = 006a5189536650946aeddb6763997bab1cafe82b578777674f75df8346d6bd01467d532de7be3c5ed0664d9b85e4da90a90871f6e15fa5166af29f1e87d9d7ea
RMD160 (NetBSD-6.0/emips/binary/kernel/netbsd-GENERIC.gz) = 15cbdb609c3925081aac2e84ebc104775601fad5
MD5 (NetBSD-6.0/emips/binary/kernel/netbsd-GENERIC.gz) = 71af44cf65da1b8ecdadbbf037f07ec5
SHA1 (NetBSD-6.0/emips/binary/kernel/netbsd-INSTALL.gz) = 4f40fd5aa8028bd956dfc51a5e61be77ca742adc
SHA512 (NetBSD-6.0/emips/binary/kernel/netbsd-INSTALL.gz) = 9fb9cfe8aa5d8ea350b24ad80a207d9439141914bd4e3371297b9971969c96ca6899b45f099e1e02ed4c8ffbea75d421b22af7d09b5b91c2f435df176229b4c8
RMD160 (NetBSD-6.0/emips/binary/kernel/netbsd-INSTALL.gz) = b269d2a87391b9640988493e3493edd114e7e8b2
MD5 (NetBSD-6.0/emips/binary/kernel/netbsd-INSTALL.gz) = 59724cb6c92366dc47fa0b4ecf2fe22f
SHA1 (NetBSD-6.0/emips/binary/kernel/nfsnetbsd-GENERIC.gz) = 5dc90ad559e682bd427b6165f760bb09ba676e7d
SHA512 (NetBSD-6.0/emips/binary/kernel/nfsnetbsd-GENERIC.gz) = 4992a1e1bd6cc1dc21821a6151aa3561fda5e369ab6057575cd70d63503a98a8d9da257956486f2f58996b0aaabf20266c16033c1f069e3dcc21f582f74006f8
RMD160 (NetBSD-6.0/emips/binary/kernel/nfsnetbsd-GENERIC.gz) = e7f4efd86d048e9f9857bb3cbf136a3e461b9463
MD5 (NetBSD-6.0/emips/binary/kernel/nfsnetbsd-GENERIC.gz) = 66abd685dbc87ed5282e1b55a6b7bfbd
SHA1 (NetBSD-6.0/emips/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/emips/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/emips/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/emips/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/emips/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/emips/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/emips/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/emips/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/emips/binary/sets/kern-GENERIC.tgz) = ed3903192d38367303cf5210725c33ba123c216d
SHA512 (NetBSD-6.0/emips/binary/sets/kern-GENERIC.tgz) = 62687e8a1af317454e30a65dafd1b54eb444db9a27abf987ba17bfaf4b9796623f279011c2102500563c9a862fa983c5e48a9c94afdedeee15414840dcd328fb
RMD160 (NetBSD-6.0/emips/binary/sets/kern-GENERIC.tgz) = 070bb8b5cab57c375fad13304165f8825e6647d5
MD5 (NetBSD-6.0/emips/binary/sets/kern-GENERIC.tgz) = 732ac8530d63fe2f3eea6e2def05ab4b
SHA1 (NetBSD-6.0/emips/binary/sets/base.tgz) = d058920475032b263b2020af3230894da8ded663
SHA512 (NetBSD-6.0/emips/binary/sets/base.tgz) = 606b27579e80fca77e82a385dfda2e03bac6d8563186ccc98d993cd532bc79da93883cf71f850380cafb31107efb424585a4aa5946f54bbc36aae5d9d9495a44
RMD160 (NetBSD-6.0/emips/binary/sets/base.tgz) = 2663d48750530861b5e9514902b9cc44e8a4162a
MD5 (NetBSD-6.0/emips/binary/sets/base.tgz) = 22ff23e09d0502d77de5fb289ad53427
SHA1 (NetBSD-6.0/emips/binary/sets/comp.tgz) = 5c25affcdce77ce7a77755c866316a9db89272a1
SHA512 (NetBSD-6.0/emips/binary/sets/comp.tgz) = d2b09aa5b72624ad3f25fa1438a8feef1e02b8f9ff064dd1d05a621f29a9790b0029baf652ab5c0ae2c3811532208235288c6614f62c378bc6b1b4a5494fd06d
RMD160 (NetBSD-6.0/emips/binary/sets/comp.tgz) = 2fd5cfc13841a1188580c6584f78bf8262752da9
MD5 (NetBSD-6.0/emips/binary/sets/comp.tgz) = dc12cdcce6751fd44291a95a06cf97aa
SHA1 (NetBSD-6.0/emips/binary/sets/etc.tgz) = 6bf20c929df89dfe6764df1f4ab15b6aeb45d4cb
SHA512 (NetBSD-6.0/emips/binary/sets/etc.tgz) = 4ee6668a884bb4b4c3ccc4370a7ac70ed14f32b7856219fa375f200ad259087400e93fe144bad4650318cbe9af4133e8d8d8f79cd0d91aeb24df117117ac54bd
RMD160 (NetBSD-6.0/emips/binary/sets/etc.tgz) = 9081112a95ee958fce83e79f687b97ef847587b4
MD5 (NetBSD-6.0/emips/binary/sets/etc.tgz) = 912ab9bd3949946dc2af9522581118fe
SHA1 (NetBSD-6.0/emips/binary/sets/games.tgz) = ec034031c6b1ea012c36910f2e901b032b0aaebe
SHA512 (NetBSD-6.0/emips/binary/sets/games.tgz) = e9bd506e1f6a6c17c0505db90f7da49795dc63905c9962e01a69224a1cf22312d9e4c68d815e9c487f21fd0791bac3a65a5e5f7bce46198694f673522b057d96
RMD160 (NetBSD-6.0/emips/binary/sets/games.tgz) = 393d511e3432d93932f2b3d2308d6a16b7cce764
MD5 (NetBSD-6.0/emips/binary/sets/games.tgz) = 71d23c31e025ff3b1eece7b3cb170abe
SHA1 (NetBSD-6.0/emips/binary/sets/modules.tgz) = f9aef1ecd6e2ee16d7ddfdad51da034bbdcd8aa6
SHA512 (NetBSD-6.0/emips/binary/sets/modules.tgz) = abaa6c62593d85131f691a3125e055970a8c3fdd7d8103db6e501dc2d2379c9dd16f9a819e14ac20959011d8946c40a8dc59ee0b7ead8a8f25c20a6475696bc6
RMD160 (NetBSD-6.0/emips/binary/sets/modules.tgz) = bca8a2cdb2ced4804cc2e0543981b1db671732e1
MD5 (NetBSD-6.0/emips/binary/sets/modules.tgz) = c98d2b8fab6184f74545c6fa54897a35
SHA1 (NetBSD-6.0/emips/binary/sets/tests.tgz) = 05a68659aec8758adb269e6f898607e1fc3f9089
SHA512 (NetBSD-6.0/emips/binary/sets/tests.tgz) = 1434cd321d5f37e6482645a11ce4f203ea3406fcea550103e1881bc68b1e8760697e5e6207ad01d4dba2d1b71c2ed0488bbb0c01dec2b28a6a74e3529ab81a8a
RMD160 (NetBSD-6.0/emips/binary/sets/tests.tgz) = dc76faefeb42e73abb5727e46bdb46ce7a737be3
MD5 (NetBSD-6.0/emips/binary/sets/tests.tgz) = 1ef426de07b570c1d9affeb44e452225
SHA1 (NetBSD-6.0/emips/binary/sets/text.tgz) = 96fc1b82a283a2bf4cdd76168ee3cefa73191be8
SHA512 (NetBSD-6.0/emips/binary/sets/text.tgz) = 7cfdb8107f4d0bfe28bf41c15b2fe7f9c0aa0d075cdb7fdb0b4566ed512da3123eff94410bf563c6a37aa498d83ed9cc7b9e16d54c8bf33605fc60bf1357c4c4
RMD160 (NetBSD-6.0/emips/binary/sets/text.tgz) = 1e1ad1dd678eb777938e2320522e275a5751475c
MD5 (NetBSD-6.0/emips/binary/sets/text.tgz) = 11f296781f98a8ae6aac39f40cb45805
SHA1 (NetBSD-6.0/emips/binary/sets/xbase.tgz) = 535b4c29666ef2de43bb6017102e9181576255ea
SHA512 (NetBSD-6.0/emips/binary/sets/xbase.tgz) = 31656ac187e0adf03eff3f896876510bb7624bb18789094abf22a59008e9d7b8d848cc1703e15b799fa2b4cd9e108f1d29c5625f1ae9f012edf8cf35912e4590
RMD160 (NetBSD-6.0/emips/binary/sets/xbase.tgz) = c8b8ebfb26aaecc931b8d8a17625eb91f904fcfb
MD5 (NetBSD-6.0/emips/binary/sets/xbase.tgz) = 46641b24d37899b78ef7a83559bfc3b0
SHA1 (NetBSD-6.0/emips/binary/sets/xcomp.tgz) = 6cddeb4f503fb424f9a06645ef46701288799401
SHA512 (NetBSD-6.0/emips/binary/sets/xcomp.tgz) = 45f779894389850776fdd4785165f6bf07dfcd46d408199eb352492536b254fa229b9d4e99a716657f210c4772feea966814329f3f2aaa729112c9b8f9d4e50a
RMD160 (NetBSD-6.0/emips/binary/sets/xcomp.tgz) = 9203f95dc810dbbd873aa54234dada2961a6d309
MD5 (NetBSD-6.0/emips/binary/sets/xcomp.tgz) = cbb4c61d832ebb4aa97c99484e9109b5
SHA1 (NetBSD-6.0/emips/binary/sets/xetc.tgz) = 7db74837c8e702a831451e4660c1f8670e674003
SHA512 (NetBSD-6.0/emips/binary/sets/xetc.tgz) = d68c233f0e4dcf5961bfc214fa10188d5fcb954e103ad610b76964c8e832e5758283bb28ef3eba448cde496d610b488f7958fb391371e1230a7b6127c267dcc5
RMD160 (NetBSD-6.0/emips/binary/sets/xetc.tgz) = 2d5fd67c9ef6ddef6c3f5fdc845ea23f2b0e3b57
MD5 (NetBSD-6.0/emips/binary/sets/xetc.tgz) = 98767dc7c3a45e641592807be04f1a64
SHA1 (NetBSD-6.0/emips/binary/sets/xfont.tgz) = 6f8acfd3feb8f6421cfea08d8cfe514ad7bc7acc
SHA512 (NetBSD-6.0/emips/binary/sets/xfont.tgz) = 2b4d59176fdef5055b0c5d8ef7677d6a1ae823d86d5955a2c33be046ba7bd8bfad85ae21e8acb11a50cba250d21bf0f2bb8c43aaa5d7d0767998dfbbfa54d014
RMD160 (NetBSD-6.0/emips/binary/sets/xfont.tgz) = 34878c9b7c1380c42f57eadea7369b687d0e45bc
MD5 (NetBSD-6.0/emips/binary/sets/xfont.tgz) = 540f04b4c6acb6e1b6364303388f82ed
SHA1 (NetBSD-6.0/emips/binary/sets/xserver.tgz) = a85da9c68ded22e61f2234e696b0bd61739f4a08
SHA512 (NetBSD-6.0/emips/binary/sets/xserver.tgz) = fd1bf84b2f34befbae2dc34e3fc5af3048c93c185d8cf5e8afe91e324bba7005d14d9cd73db5e09bee783116b9a09b01b650a9097df07a335fc81a2bc569c2b1
RMD160 (NetBSD-6.0/emips/binary/sets/xserver.tgz) = f066886778a33191aaed1a30ac8494b6deb152c5
MD5 (NetBSD-6.0/emips/binary/sets/xserver.tgz) = fc4fa9292f46a325f74c4abc922fd215
SHA1 (NetBSD-6.0/emips/installation/diskimage/diskimage.gz) = 9c4d0f6ee9e3251ef720acdb06e20683c55a3db7
SHA512 (NetBSD-6.0/emips/installation/diskimage/diskimage.gz) = 5b9c5d72e5e4bfcae27bf51c2ec9fa849b019b475d6677be8c49c31f18eaff6e559d80790ba6d5e606169215e46e04322c4d12ed0386e76635fdebe380cadd28
RMD160 (NetBSD-6.0/emips/installation/diskimage/diskimage.gz) = e7373c5f20e4728fef9951dfc1beffb70390f651
MD5 (NetBSD-6.0/emips/installation/diskimage/diskimage.gz) = 4f2e15b3d203006588a75670b4d3e75c
SHA1 (NetBSD-6.0/emips/installation/miniroot/miniroot.fs.gz) = 4075299a2ef6a4368234ed1e378c2931b0df6376
SHA512 (NetBSD-6.0/emips/installation/miniroot/miniroot.fs.gz) = cee815c8a985125fd2c1a9a4811542371ade25fa7f35b4cc4da1783186146d4620d1910a24b92beb02d1cf836e9cfc2921ff9d8c81ecb4babf585b1ec3601496
RMD160 (NetBSD-6.0/emips/installation/miniroot/miniroot.fs.gz) = 1701f681ccb15b5232abc07550de79d3380af96c
MD5 (NetBSD-6.0/emips/installation/miniroot/miniroot.fs.gz) = 5cb7df17380c5bd1d56b28d6061332f8
SHA1 (NetBSD-6.0/emips/installation/netboot/diskimage.tgz) = 04f571f04a2227b98df91f97023e860af233d2c8
SHA512 (NetBSD-6.0/emips/installation/netboot/diskimage.tgz) = e5c7354014f23996e0e5c331103b9936387b9c6402491041727b69353f6af08b57cf429b51ee0c28e76e3f4506479b0559dac18b05a009a3afca76ad2edbd29d
RMD160 (NetBSD-6.0/emips/installation/netboot/diskimage.tgz) = c57777b29a2c589549a906fd0f425c56366a7075
MD5 (NetBSD-6.0/emips/installation/netboot/diskimage.tgz) = 9d1bfe38f9ec7d4ed86173bb157abf52
SHA1 (NetBSD-6.0/emips/INSTALL.html) = bad639d8582318241d268e529e8e024159dc5997
SHA512 (NetBSD-6.0/emips/INSTALL.html) = 94627da8cf355e0f7b3a597e2e22be0b151a9c4e40301c7c0b67c1877c34abff57476a5c177035c032cfc1d9b8f9d38aa919c27f1f3c15d3edb3403d18e71b44
RMD160 (NetBSD-6.0/emips/INSTALL.html) = f2b00377e6a6d060e9027d78d2278445ff6b4d75
MD5 (NetBSD-6.0/emips/INSTALL.html) = 825b11bed2394322021f65e43627a391
SHA1 (NetBSD-6.0/emips/INSTALL.more) = 9906d8f0c74d8538aa2e7d1361800cbbc2db2d29
SHA512 (NetBSD-6.0/emips/INSTALL.more) = e8419f85e6beef4bf18f6bc916141ae9bca52eb6f56e2ae7133fc771ca1033cf2172a644e4415be0019868004278605f0e93c42f8f4f60d1e2fecf8c06a51a90
RMD160 (NetBSD-6.0/emips/INSTALL.more) = 70fa5e023134732afc81b3622480328ae9f5d449
MD5 (NetBSD-6.0/emips/INSTALL.more) = 0cff06f0f5c3bac3eaf4bf4336494217
SHA1 (NetBSD-6.0/emips/INSTALL.ps) = 5783005eb38711a9ffb248fb5bdf2987912f26fc
SHA512 (NetBSD-6.0/emips/INSTALL.ps) = a27d36ced381f22565ac6ee290d4ef43de834977ca8be3d96bf758005a6fe0875aa03f703da08b7dfd5fdbd48c19c77ef5263f8571d0f5f983ff7c1d08088873
RMD160 (NetBSD-6.0/emips/INSTALL.ps) = b8e28009dd02ff9c1f98d5fadadc6455c262238e
MD5 (NetBSD-6.0/emips/INSTALL.ps) = c16b7dcef0674aaf0dd240381961eaa5
SHA1 (NetBSD-6.0/emips/INSTALL.txt) = 0d166d27be7166a76ad58779f1272a1a1542e952
SHA512 (NetBSD-6.0/emips/INSTALL.txt) = ba74f2486c3bee1b30fc39edae6ad38b0abcd9610afdad172a49590aa0c1fd8a2256d185a699baf6bea9e40d9fec7d1e56c17d77858d9ebde027419cc044b8dd
RMD160 (NetBSD-6.0/emips/INSTALL.txt) = 729887c61690c966497811da66852780d0d3eec5
MD5 (NetBSD-6.0/emips/INSTALL.txt) = 8a28948bd5d45a30ca79556be75dbd69
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_TEAMASA_NPWR_sd0_flash_0x00080000.gz) = e8620ef3bbba89642fcce48ed01e16f0783e889e
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_TEAMASA_NPWR_sd0_flash_0x00080000.gz) = 055cdfd0872737d68054fa4c2125d131e6ee3f60135be817da7fe111adb2290c4ad8d968b4cbefc67658c3e9f3dfdf99caaec320f572a00a790604c93fa73408
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_TEAMASA_NPWR_sd0_flash_0x00080000.gz) = db49ed4cceac70d18a8a6b335f20923d07c8ad8f
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_TEAMASA_NPWR_sd0_flash_0x00080000.gz) = 12617c0444b2a570637a45339937c77f
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_ADI_BRH_sd0_flash_0x00140000.gz) = 3160f03e833b96020804361b1d98c0cffd8c5345
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_ADI_BRH_sd0_flash_0x00140000.gz) = e5d7ceae7a4b4446002170c84a44dcd65cdbb2458c889582ff3070a20580ff31d2f497b07c320217b55f75e6b343f689b9312d172dd3880bd82808e6cdb995c9
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_ADI_BRH_sd0_flash_0x00140000.gz) = 41f31d747ec5265903490d300bce1e6c34ef32ca
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_ADI_BRH_sd0_flash_0x00140000.gz) = efec631edb7870b4cc3ab1cec8d98810
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_ADI_BRH_wd0_flash_0x00140000.gz) = e1c927a0718d63609fe15d9dc9c265ab0172ddc5
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_ADI_BRH_wd0_flash_0x00140000.gz) = 5418c2ebffd11c30028d867561e26dc5857167698c88f5d9eb30af3944fdab4950ec2f6953999689c03cf684aaada038d6f7f07494988dba9467a814fcfe1d3f
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_ADI_BRH_wd0_flash_0x00140000.gz) = 05680bce8a448c3fec2149d6c89dced13166f34e
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_ADI_BRH_wd0_flash_0x00140000.gz) = 36d8336012f5603e1b30565061fd8c3a
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_CP3100_sd0_flash_0xf0080000.gz) = 9419b71634e5ff228357a37651061a1c4c2c2ebe
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_CP3100_sd0_flash_0xf0080000.gz) = b17b66b1b62c16d6a0add67d55f82a6943557abbbcb8001b328770dcd2191b1b55881dca2919d7f76381aceb1ed53ae81ff002317e99d02fbfc2817d9b366661
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_CP3100_sd0_flash_0xf0080000.gz) = dcf913149707315fda181d6dd6967b68fe61fd22
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_CP3100_sd0_flash_0xf0080000.gz) = 4f81965829024ad764d293456cd96fbd
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_CP3100_wd0_flash_0xf0080000.gz) = 748d01d1b8796881f853c31ce98e0dbf473bcf8e
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_CP3100_wd0_flash_0xf0080000.gz) = 898de3034e164cbef89c0f09c76decf48c4291c7c0d077134ba1695cf0c9287d658e61c82cb2d3d46dce986014d2e52d84f0600dab0f99db996623eed7ad93ea
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_CP3100_wd0_flash_0xf0080000.gz) = 452633be8258bf3df07d6e9461785a8f3ac09f98
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_CP3100_wd0_flash_0xf0080000.gz) = f6644ec6b9e7cbeb624d57e00e764b5e
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_GEMINI_flash_0x01600000.gz) = 2dece6cc86980856d901df11995ecb0a5524451b
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_GEMINI_flash_0x01600000.gz) = 270b8d528bed206bfee4cfc63ae9a04f69c282d52b2beae8cbc8298872d2ff0e50e00d849c6e435d472845a0f6b8601b94a0fe52e63c61e7b5c534b9d830d252
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_GEMINI_flash_0x01600000.gz) = fbd959fffd37010a92a76026469affec9bbe95da
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_GEMINI_flash_0x01600000.gz) = adaae55226bd3fc806962ae70e9281dd
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80310_sd0_flash_0x00080000.gz) = 97ec6f4a472d0b73f1d3b8d284f57debab190c78
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80310_sd0_flash_0x00080000.gz) = c52a596cd2ffb52c1321b2e8006a590186dddde671f4325067e546338eac4d405c7d5fe559ba6d187a1c26eba4c45d233505a434a98ee11b88b315db0a42d63d
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80310_sd0_flash_0x00080000.gz) = 3305a23227d275c25307ab7b999f5e0c0560195b
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80310_sd0_flash_0x00080000.gz) = fd76ab094a19b1f66c9d21df4cb69700
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80310_wd0_flash_0x00080000.gz) = 0ed362fdba88789a5d26d00e0e8285f75587f5c2
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80310_wd0_flash_0x00080000.gz) = 78110414c2dc51bc378c3a1d44914b4f5dffd6b1d6d0a749576e2d7d5474e7856a2235641fe310fb87b3789bbf495b16459f9c773bdab19161e65af671ae6bec
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80310_wd0_flash_0x00080000.gz) = 4d958d9f772b214efbf9932a00970375a6fa7e57
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80310_wd0_flash_0x00080000.gz) = ec5386e862bee8da4d46a124584e2cee
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80321_sd0_flash_0xf0080000.gz) = 037aa337f876321d75654dfd576dbc0c4c301e0e
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80321_sd0_flash_0xf0080000.gz) = e033668a417f453ca35b2b43633a7eb9160765eb42ba1b86ea61d5fd4cb5136123a69194d58b3bddc51fd7e319bae1d70028e1215a1e8f4e98270bf0b4eabfa2
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80321_sd0_flash_0xf0080000.gz) = a8e8d6ea67cf26c422d638794675626e7261b3ea
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80321_sd0_flash_0xf0080000.gz) = 2b3bb43bdb7fcd4999377eaa3b994c48
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80321_wd0_flash_0xf0080000.gz) = 6d8870ad00ed9f496738dd5cf7f670b5b43d2f34
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80321_wd0_flash_0xf0080000.gz) = 414a11de38ff565307fbe599b87b0f59ddf91db1882b5e246774ac23f6af49411ae33de4145a0d8e10755d6e78d85c930ecd0705804a25ad10e3957765e565c1
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80321_wd0_flash_0xf0080000.gz) = 27959615599b74b946e9e8483c2284aa3f80ffa7
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_IQ80321_wd0_flash_0xf0080000.gz) = cc0f0b0621d6d931f6dec494c7bb2305
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_SMDK2800_flash_0x00000000.gz) = 09617a28f2da331a17efe6cbdd4df7367f8ff98a
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_SMDK2800_flash_0x00000000.gz) = 9324e9fe0d4756cac25b33f8b6592f2a1a25d12b0f3752447cf660c21a3b2e93eaeb3e41a15d91c2edd827e2f5318a241c58129cff998d8f0a0a7a6a6bbc598f
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_SMDK2800_flash_0x00000000.gz) = c2e0b7b74df9090350d1afc52a15465b77eea3b9
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_SMDK2800_flash_0x00000000.gz) = ee28fbd00b44705a33d6ca6e6982507f
SHA1 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_TS7200_wd0_flash_0x60660000.gz) = 3036d8f2318c69d791b710a06d2e94c5c563fcac
SHA512 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_TS7200_wd0_flash_0x60660000.gz) = e8f2cab3b75e3f4ecfd2ed85b624c8c643eab3cd32a8d0398148f49923622df598071fbb2fef6b5687c8273762d197992a03b1d8465d96630202b78592199a1f
RMD160 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_TS7200_wd0_flash_0x60660000.gz) = ead79fd1f78230bd30f851b78dd13b670a791863
MD5 (NetBSD-6.0/evbarm/binary/gzimg/gzimg_TS7200_wd0_flash_0x60660000.gz) = 1e25872ad616b30406dab15aa8c20b22
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.bin.gz) = e673789f88d60407c85c744a64382c102b91e983
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.bin.gz) = 43aa01312bac8975f2f66cfc47e9cbdcbfe37eb37f2076c221e5c118521e987f18c1b14bf2d94b390942820e5f6218cfa1eba67a1818f8723481d434f9fea00c
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.bin.gz) = 7c45cf94c28c04db6df14decfede69e83bc0cf4c
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.bin.gz) = 49da3170f0fe2fb304e1131bf77d4c3d
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.bin.gz) = 8f057cc764ab0cc64689c2df8652f9db24e20004
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.bin.gz) = 537b67fb3baf1f5363dec0079a7bf64c0ed7d5ec9b2a383362ca04ce3f746011798d12fe10ff6e4a77ccde35aa66e8cb06e689dcd7f19b5233999ffd7a2f817b
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.bin.gz) = 3119fe0017084fdb8a97ba14a7f6258450c6b938
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.bin.gz) = 755aa54a7e8e2091f3f3e7c37f34f45c
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.gz) = f133ab0dedf14635b824f7a0a7d55451933e23be
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.gz) = 56d8d397af893debd28cf4283759213f8c3171647fb4df80b570991792d2addf4473d5ac0ac8a4e4dc4a668035ee37d3331cc3b6d5e029876760d4a6e0aab459
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.gz) = 59b03c3a53d5b54cef8240612a803d1ea61486f9
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.gz) = 9e800486bef03915eec15ca90ea163fe
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.srec.gz) = 18f4bd985378baf25a9e28ffeaf95423b4bf85c7
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.srec.gz) = 9a535fc733582c7e8b59784c117f6714a96a1be2e9f65be2074731532511458a97b344359992de681bd7d7980c63dcbe98e7cef4bd58cd4cd3a313cefe822814
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.srec.gz) = 0e3554f0de32a39d19c1e674084e6b42f7ddfde8
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ADI_BRH.srec.gz) = 22db840fa81c95919f08d2cb7ef81450
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.bin.gz) = 5e3cd42ec6e6af1203ea8921eb1979183164c8b4
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.bin.gz) = 21cc95b8e28d50eab2b20e5eda315781ba95c34a4e4f8d1c5907ddd81ed4b236dd100b3bdb957456e3a713211f5de11a0f55cc6516a58dab3d7212b23de95120
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.bin.gz) = 5a1f1e22d7a29bf8205c1a3550fd78d2cdbe1d38
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.bin.gz) = 61edb94110b4be5c9d8bc9ee660dbd41
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.gz) = 19c96d4bf28919a4f30e55838b0859e197d0759f
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.gz) = 823dd6b45920bae6c2817091c106af7d7fd4ffc965f64c68da520fcdd70b8bd341765d33f1adf06574baaa79395a501b58aa64b8b83628209f3a08560b90c0d5
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.gz) = 30566487cbea824734bd4d9061f5a9df30875b07
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.gz) = b35c13395ed69e7931eefff8e42f1198
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.srec.gz) = a7d4841b78d684f17658819d150d7d16b135786c
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.srec.gz) = e196c13406146baa4b8d923b4406d8e3b2a451a7cfb174a4c652b8a702b0d619d30705bc998acbe8a037fca1c8f2fda53aee9ce8d4a52722cb11c40bf1288f75
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.srec.gz) = 78cd23cb694a0062858e5e3bb2959d794176c2db
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-CP3100.srec.gz) = b4e62ffbd5091a667a7fb61c31ff578e
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GEMINI.bin.gz) = 6f6bf5de9d82b058100eb089aca61b9bd59aaf61
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GEMINI.bin.gz) = f91cf6eada8a84ce9c0036a4c0b22e550d71bf20fcb67d6b409d252e7db2c707aa104ae68a998cf87e98e8a47e3cda2cfa67f19c6792d3cf4f32bee59f3e2c48
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GEMINI.bin.gz) = b7ec88f6f6604e897b2450223e791758e1c278d9
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GEMINI.bin.gz) = fb61d11fbdd49a4d826896bc45bad653
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GEMINI.gz) = d44a7af68b2dffe73d6a28a742a15a64ad1c7f17
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GEMINI.gz) = 244d5a9e1b36b7a18f77a1689d444375f733e4abe0a13047300b815633fe76af3bf595544078f89b1de302619e10f6d19245adeabd177fafeba93b7889dd34d8
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GEMINI.gz) = cad1e137c7d8bdb92221cb76679bee0ecb78f7a7
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GEMINI.gz) = d5a82888cc0477c064dc3ae9dfb29110
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GUMSTIX.bin.gz) = af321ff02cea6f05295ed48d63145f2557065034
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GUMSTIX.bin.gz) = 103b4e272b4ffd89e2705edcde9ecdff2d1b11bb4e4f6c46a2cfed6ea8bae9386bb357dda1fb04ca558f30d201171f2ec2aaae84b39b792715f142ab83772988
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GUMSTIX.bin.gz) = 69a190021189d71376d88493fbe9bceb5c1bb50a
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GUMSTIX.bin.gz) = cddd2da663644297ba2737081b26fab8
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GUMSTIX.gz) = b7ec166d07220620ec458e776df34ce0f954a861
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GUMSTIX.gz) = 8b38c79a2e2533d85733cc5dd176f808036e3d04f91134fab57110aa2c827f26487cbee563fe85cfb0facd6a463c7069d5d22aafe90c1803a8f0b7df2fe526fe
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GUMSTIX.gz) = d71b9dc8f5d59f8b126ce83dc06df5f97a009195
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-GUMSTIX.gz) = a8fe08eb1fd48f4870f1f736c4a72cb0
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-HDL_G.bin.gz) = 33e9946d8fc4834f9c74647b322a79425d3d6c52
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-HDL_G.bin.gz) = 354db08b656e913a0803051b861cc77d90c3c7e170362488c9940c3d0dae047e7a9d92fec348f92d951ea4093196b9f0b210d9d0cf04b9d842494c45b6feda5f
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-HDL_G.bin.gz) = 393631ffd8b30a1e3dd5ecccbf950bb1a75759a3
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-HDL_G.bin.gz) = e650bd0cf7ad90361db3a5cd788553e9
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-HDL_G.gz) = eeccc5e10b4cc5b8900e9dca97222f0329c5ee67
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-HDL_G.gz) = 3ef842acc43f885398224dfb7c696d2334aecce84aecf7ea11d9464332e62fb796b45b302a097fd3a1ec4fe48ea92a029a8301f9d97899a51cea19c3524e39cb
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-HDL_G.gz) = b402a5e445815908643d27c5b69b7e055f2487ed
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-HDL_G.gz) = a74d0d8c03f2d840e118654bada47b03
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SHEEVAPLUG.gz) = 9380dcaafde1fc16b90858e04d0804b159a18b6a
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SHEEVAPLUG.gz) = 20e027261b82d2188342fc3e10bdf1c2531c9012b4c0a5fea9d480c70bf00f118f0a0487646f541a63422d1acddf4abaabb34ea0a9161bb8f429708fedd18e63
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SHEEVAPLUG.gz) = 23e93e206ce3d710bdb9f4e57c342aaac073ff70
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SHEEVAPLUG.gz) = 2101cfda9d3ee129a2768bab982c7d4d
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-RPI.gz) = 1558752bbd6b362decdd3caef1d05b8c80dcf43b
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-RPI.gz) = 15dedb25bd8759aa736fd0ce444c4c782a65ec6cdd64e3fe424e1665d3e937df88b375dae773dda973a66264b90e10d0aa89e03cc217a8afbf4d79a23cce635e
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-RPI.gz) = e64628b2e9cc699bb0a48172cc4422e9e50a624b
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-RPI.gz) = 2224734e36d6bc4a139efdad6cc86978
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.gz) = 2c4862e6fdd14328dee5191b7e007fd7f2e626e3
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.gz) = 259b9964b3d10a4b49a413ad6355c16960a538b02f19bf4c0e07fcfc51699ba9df0a0b6342e6f67b9366302026bade0bf46a53356dfe42342e06f6fcbaf64dcb
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.gz) = afcc94417f4da4b84269420db2b0952565bbee93
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.gz) = 6010c0404590c4264ff824a9a57c0e04
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.srec.gz) = 0226f405675defd77c9dcdda8916e5b372cd206d
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.srec.gz) = 0213a4b6784f018154bc11a148fcd6eec14ab26bc7b714d087c845db361776a6d37ff11aab7b1c1218e7a143e7fb12a85fff77f93d97832fa12d4ac39764691c
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.srec.gz) = c5132bac29060129efd95b4d9beb4eee87e0b6a8
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-INTEGRATOR.srec.gz) = d176d5afd6c40c0cbfd4f64a19e39a25
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.bin.gz) = d105689d85a45fc834acd60c920d1420782e47ed
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.bin.gz) = bbe9fdeebed0c150622bd216013971c27646a4496f4ba64a8dcb0748e91c402a03441a0ee7b78341107f4d92f81771f41299b52d4142c6fd93e345f46a9319d8
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.bin.gz) = ee05e9b64e9039aaee667dfdd71934a2b029b22a
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.bin.gz) = 993d3e5e2595d0da326ec142a45b2750
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.gz) = 7bacc9ecbca2a680a72fa9d71d0f622302fd94ec
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.gz) = f12ed4db4fae72630b905b3edd5f6eaa87f84db7a13d908559f7c1bab054054440dedf6806b8ae6ce873bb09bf2a4cf69d89529f21abbc476ac50274d9a71741
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.gz) = 360190a9de654cf05c1aa9a6edc908079fa2c237
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.gz) = 73b8043c242bb9739d216f3f8da7d3a5
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.srec.gz) = 2c976175ede9683b8a8ec5b8c3711df84001300b
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.srec.gz) = c683332d8fb46e88dd0dc5d9b56a4fa942fe9cc7cb79a005ce59609caa9d357d7dc5126feae3b181f63009980a9bf1700130d086b7b7103a8ac199c2a25090dc
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.srec.gz) = a1cde3a2841db820d89ffaa32232ce83fd5f44f9
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80310.srec.gz) = 811e47b38fb8e8c1c7f228f0bbab9a74
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.bin.gz) = 862eb1a7ee9de8e3b3525e7d7d26d2cc5ba9bc59
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.bin.gz) = 884b2229b6a78748f61fdba23fe88a623a05c974d71e711821d902fea63e258fea523b9d9c1044ce5afd1d941a2e73d09649a6abc6d38ba0c263bbb4700dcfc1
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.bin.gz) = f3c4d5b5ec632c62410643b9e12960a40b2074ab
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.bin.gz) = 31e7f8b3e40c22ade7cc22d01d95b532
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.gz) = a60ac00d16384a828eef8a837fb21e00b53a8a4a
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.gz) = 54bceeae1a8b27e59a3767284017552f717cefafc71e68b6c74a40429131709ecfd79726737d2dbaefc30e70d09fdab893e4651d125c85f10d9017b1c9e1db7d
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.gz) = f5ce2ac5312c8fecbd072965df9bc2eb0c340d5b
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.gz) = dc8d921c26170e31f92bf29b1ce1a183
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.srec.gz) = 6fc69281083cfff4fa9fe78ab260619b92972167
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.srec.gz) = aaacf4b2b3f5ac015eddf58f6e5757c061b94ad57b8dc850cd5285fa4e111aaacfa2dbc0c9cc3b4e7a866a2fe2b55b900a33fed48e8587b50c6dede0dc66cd9f
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.srec.gz) = d63585bed43debdbb05df1eb44130fc50e2edfb9
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IQ80321.srec.gz) = db12176962530274584befedc2794d71
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IXM1200.gz) = cbf69533e9704756eed91dc42ab880300452cbf2
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IXM1200.gz) = 104cbab6498100ebf4e91b89055568f8e770f33abf331787d061ad000c506caf54ca9cd9afe2d6c16b6874502e10e3b20c0fdb97607f18d2ffb59718b2df62c8
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IXM1200.gz) = 24af829fe0a4c696a78dc11eeb9a6830c1faded6
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-IXM1200.gz) = e3634e55cff56a10538e3139c4eaf83a
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.bin.gz) = 3921a70d245903a0c95e4944b835f8b05326278d
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.bin.gz) = 49bf1c9de5e10f21603f8eb6a45b7ae29bda1fa9ba8fd4b0c2a07e498af6bfc818be5d59e384ae3afbe6a0aa8000ea494f2a431067d5f32e553d09c4346b488a
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.bin.gz) = 506b209365f4650bf205b3c989504665e6a464da
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.bin.gz) = 4d00574f0fccd038ce96a75e40119f54
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.gz) = ac288980ec3e5615d33d18bbb5b20b863b64d4e5
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.gz) = c96f2aac73d0e62c35d76c6a467f5b3b4ecd8461504a0ebbba456868f5ac1a42b6ff7c80aa2717bdeea242d79b940cd3584184da0fe8a07a79e2ac2f7bdc96bc
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.gz) = 02fd101de84eb4c6aa40bfa7cc477305246a7610
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.gz) = d4f8e1de25f4050d4b357cc9222799a6
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.srec.gz) = 10bb46c7b4e78af1cf9978c76759694a10f5b2b5
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.srec.gz) = 75b010e6766552928518ebda6daca8a979dd9a2168e932621c63271607c48ced148b11ed29cb14a11a1905f4a30a3613a63a41a5743b5857b346501b5516c3b0
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.srec.gz) = c18f791fe071f039783b42800302d62d407245fe
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MINI2440.srec.gz) = a749256c48d3b8ac02d6ddca5fc669bc
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MV2120.bin.gz) = 5f240a5493136d85a3b3ce856856a5781873834a
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MV2120.bin.gz) = 19b95311ee9fe25d64d8eff569a02b68672348a1c7372bb99f9f07421e390ccaa0e1a3c65b6d8a67baae1b34859b5ae2ae0d1471c4c1bce0f249765ee085dd03
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MV2120.bin.gz) = b0536ea2075ddf15ba68cd3597aca512d25404c5
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MV2120.bin.gz) = 0174a07da16cafc0426a75dd4c9affbc
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MV2120.gz) = f59146a75e366ba624eaf77f9582cd7200f56530
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MV2120.gz) = d68a8a5c29cbcab54673232481a17bb901d64d15de7df630c1657494cb70494436a0763a38ab31b12e3f273c8e8404f8515d29f2f290c99d25372d46763ab6ec
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MV2120.gz) = 2cf27b32a9bfd684b5f4013c9d25b8837aab0a08
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-MV2120.gz) = f623986c6286c02be5144d832c0fd3c2
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.bin.gz) = dc5760fca180affcb51da10f190dffbfd1b56cd1
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.bin.gz) = 56d1f1d5cf1faecbd845eb2a4a8fb88ac6d54808c87a6b39c70249231a532c7b7d858d5d89981f5feb726d93bfab384c6e5b8697a020e1c7ab495c13a5580e6f
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.bin.gz) = 9be387154227f881c0ed38529c7a7a6916a59433
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.bin.gz) = 338ce4a5cb70414c375f3ef8cc9385d6
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SHEEVAPLUG.bin.gz) = 9d9cab4789700632fb90b593893a81015b51f619
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SHEEVAPLUG.bin.gz) = b1632b6bf5b3e6ebe3b8f311d8de3218e42e40bfabeabb2ddfd99285b31a0451e4ee4267d0dfbae1618c023b52bbeea5b76aea7162d4bf108f74538b5aa535b3
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SHEEVAPLUG.bin.gz) = b6861bc0a0266cf68f5d91e6e3fb8c7da61690c5
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SHEEVAPLUG.bin.gz) = 9da615e9589e8b82d9a7e30c680d87d1
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.bin.gz) = c7ed735255266257fb135c4abc2187199970ad3c
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.bin.gz) = fac2299142fe0f8b45f7a433b756d1e48e66940155cf5dd53bfd6bd2827d66fbf43c08306a0cf5d69e25ece03253ce5f806498cc925786d55973f95bc6079c7f
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.bin.gz) = c3a03ab1bcd188d70b889c6384417025abe0f459
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.bin.gz) = bc3474cddb53421b463c1172621fd278
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.gz) = 0b08bd196892dff6fc1997afb9110e3f756f3124
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.gz) = 77b0d553364442904b9b08a3db76c70a7306b23b1b9f71a495eac9b294165da023d9ded9f8ee86f6e0ef336b514d83461cefc63885f754d4f6e03ea208842326
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.gz) = c0be26eb32c1d60435e1fea7521fee95660772fb
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.gz) = 79e74c16602e752a034f0d93be2ca62b
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.srec.gz) = ecffff46a8c528208595aa47d6739be05f4be815
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.srec.gz) = 532657ee54b385c79c7ef9768044d4d4012f16f54eb0fba2b743cadd66bc8288b00fac818503faf00db6f28a54f2bf51c7bc7f6ae8d5562e84a60149af119e63
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.srec.gz) = 749c2ad2c37bc095893a6ae9bf2b614af1effeb6
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2410.srec.gz) = bfebb327393531018e6b9f6e66738805
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.bin.gz) = cbc1ebd6e56937148dae2c2d968add4d8b3df600
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.bin.gz) = 463210825ff997e4cb1a7dcb36ed0d0f352ce8ab72ce55fdadfdb03f38d18fd524452fc2129942cf78d2911bf7be4875df238a03f8108ce2708944ed8a9682f9
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.bin.gz) = de526c5e982f7173406cd0b2fffcea1dd4fc9a7e
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.bin.gz) = 9c8cdb5067ff1bee4fe4705cc1cfebad
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.gz) = feeb3ef6e66284fa727a9feb4f3d7c45fedd0ca6
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.gz) = 9ec41f1cbabe0c960134267ec18fe6f1c5d8062489735a85bd3b1b8d9ee2acbbbe5f32c8e8e416ba3bdd55e563dc25c8224500702bbc5038e7bc5f91be5fed05
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.gz) = d03357b46454c1ffef1b1e3524c8b56a61336a06
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.gz) = 39dfa81264cd40fa3e0baf2db3f2fb8f
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.srec.gz) = e7908babcfe4f8e9400a83351028545d87c111d1
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.srec.gz) = 3e13a586fde15f0fff6be7719975eb9f2fb1c4778de5a19b30b208b050e2b8549de137c5820b0c233af3bb32d4b0dcbcb7d50d4012c6418146c78358f11ad194
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.srec.gz) = d8a910ca9c3b14abb626d94cce73ffee1d951258
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-SMDK2800.srec.gz) = 43252ffe4ce1a12b526af0b37aa7a2c2
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.bin.gz) = 1eaa16791cd37e16ece1fd2a3f84a9f1b744de4a
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.bin.gz) = c12fe5137c903af5a4291fa06d95408e1a554ff70834036766a971693edb8164dbfecd8783ced92f548a42eb354da31e658e6ea3b347cb41d7551a85ff14b441
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.bin.gz) = a630baa7e17c1dd8e67f5d925fcfa782301efd09
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.bin.gz) = 8bf004134779adbfd47e710810cd541d
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.gz) = 8f750421edb291b7231b9ebb2df970f5da937ba5
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.gz) = fbd4cc574e78cbbdd68754790777507f8b6b6288e05cb5261423f21545ff0825c9be03bc1ec663bf6a5552ac6686cff980ed1f2e4716a4690132058107a6fcf3
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.gz) = dc315d5e0fb0cadb02954e75fb78670244ad1878
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.gz) = a0a2a1a1e5fa5e566b076fd1925f5a0b
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.srec.gz) = 27a486313bf959dbdc89279b8da39ecb7ebf7e6d
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.srec.gz) = 4dca299d4f9a90a2bffe3ffea6b5262e6cd248b9abae0fbd838e6bd83502fd573428952180f2dbd77284907fd7b1ffd1ef0c7524897dce49ab05c8c3aeec3aa3
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.srec.gz) = e06dcfeb9376bb44f0444e71afd03c51754ad533
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TEAMASA_NPWR.srec.gz) = 0478aad2e41f791604d7d9e33efc16c1
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TISDP2420.bin.gz) = a9c56c83c032a539bf23ac6250412bdd71a996a8
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TISDP2420.bin.gz) = d61d6f3e0a937039f13182d03ef1bbce0ff4f04a0b59e60096fbffd34f810d8e254d8e42b456a2be03bd84ce127cbe40e12c7864c85e4bf963ec9761ae322b9c
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TISDP2420.bin.gz) = fdcb7c9bf1b49107c95e7c0731a6ca97922dfceb
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TISDP2420.bin.gz) = 23acbd88065f394218ead9a00dbc6daa
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TISDP2420.gz) = 2d4f5baf51625ecb75fc4d043d248ea326c13102
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TISDP2420.gz) = 2bb556b33c74bc49b82693a4b66437498f7630709923aa54ae0cfbb0af50f5d28db62d062eb234e6c5b33bfb081490e9cbd687a899fece106756c79abe325a85
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TISDP2420.gz) = 84c7b73fd2c20914727a86e8aeeecba0e3b54ce9
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TISDP2420.gz) = c1ef3736981fd98a6f6636a8785066ed
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TS7200.bin.gz) = 80dea7855e77c34fa5a6df8fb657f23b02777a71
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TS7200.bin.gz) = 673533b24668f44c09364ddb22f1fe3e058b5aacb2c544b0fe2c70eba9823e8aefcddcd02f6df8fabd74b5283e585b30ae33501e02b7edb7bc05cadcf8deaf07
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TS7200.bin.gz) = ba834fcc21f7b5210f10388a209172e2a1554944
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TS7200.bin.gz) = 6228219fac300e28fee657b3d640d2cb
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TS7200.gz) = 5d321821199ef390140e42f5de3aadbef178ec7d
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TS7200.gz) = e5c98ca85a4a926860f11a84abbfeadbfdf2ded1f7ba8fe4f852e48188c16878016f66bd79585eef8d8cf1afa28be748ca144cddb742029c27b97051224fd41c
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TS7200.gz) = ce45d65a2eeb0e8b3af0ce02b09a1dfeeea56d3e
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TS7200.gz) = 9b77d5ea8b37b0550d4610e53bcce56b
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TWINTAIL.bin.gz) = cc608ec28ab6439d45fdcaceb2608d00b20f5739
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TWINTAIL.bin.gz) = 18d2c25112a185bf5b7e23a7c30ebd183a29f16cbc79e11327fb086efd8f04faaa1fdee8ecfda8de6e437eea0ef79f788ee89d4388aa37fbc88331ba3a8cfd4b
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TWINTAIL.bin.gz) = 85e82f3918141510d0f3b9741fb10e15616d38c8
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TWINTAIL.bin.gz) = 62e6180d8bc9b2ed8ffd209115083506
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TWINTAIL.gz) = 784f311793c86ea06397c5e35f93833f1711c92e
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TWINTAIL.gz) = 3d481b051578603414849bf1e43bc84cca28049991640d3c41af3bf1f223623e46200c872cb29e3f98ad2bfcad5c377c1b489f074ac390f4ac9a04b4260e8c77
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TWINTAIL.gz) = 17e6e1253799aac23cb810aa24e81c060115d311
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-TWINTAIL.gz) = 823ed2d54f115836527299f9542b1531
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-epe0-TS7200.bin.gz) = 0d27b6fb7baf68c25f5e37005c6864cb82e075e2
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-epe0-TS7200.bin.gz) = 70d04bcb2896f6e27c0278189d57549f9c3a286e43a827a704255487095123fb61b00b155750fd007491e42f5498d4fb207e9ff40d32354d1bb8b4adea67db1b
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-epe0-TS7200.bin.gz) = e2850dff5c5d6a2dcb975ed69d9cdc3dab59304f
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-epe0-TS7200.bin.gz) = 1b5cd32ef196ccff17b3a3bfc3a790f1
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-epe0-TS7200.gz) = 2778315d331ffc62b829685c27668514472f3e19
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-epe0-TS7200.gz) = 1fb7e68c7599c6174abce47d01c5376bd37e966342c5b377e0375e050200737f507c34e76653377a787bd9ea11ce526c0e52fa01009d10ccb4e124b77f59ac33
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-epe0-TS7200.gz) = 08efa696270c86ab0b2ccecec2f65fd0f6ed07e4
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-epe0-TS7200.gz) = 103a3dadd04cf056386a2b7bbd0d666e
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.bin.gz) = 11f132f4c6e60f09f84e8cc536c38d125336373e
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.bin.gz) = 0a6442acb206b96f9450a80a1a02f5420fea62d96464695490a5efdd63b11c30ab38e2ba270fbb4125386a5b3bf452ae44427f75fb5dd675202e862301cf28ca
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.bin.gz) = 14a2ee2c2477084bd47369e3ab5bff9ec5163a2a
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.bin.gz) = 13646782aa47578a9ad12f7c07907633
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.gz) = 41474e92c5c2634a3ea25bba4510dffc8d2e399d
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.gz) = 269cfa9fdab9f28e0931d20e6ff68284ef6e578d11c1c2b23fb272d9e82b12b19fddec101e24032b479c15ce8042efd2265678cf7a3e951e6b2429deb2de2b11
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.gz) = b080ed55c674ec69554713838ef050eba2a195a9
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.gz) = 37ec6bc374177e34ac79e774e15cc679
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.srec.gz) = 98d97a466e67c472d2b778cabb6ff6196b379079
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.srec.gz) = 8ea14e0483850adef7d6bc614527739e9f3d1880a03522ca6d7c3df1e38222200983ff4fca413b9aa2c2e6d90efd5170aa7b43be8ea68eb9c8cb8f024c9954f4
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.srec.gz) = b88e4587b48064fc6719bf3e776bc3163efad390
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-ADI_BRH.srec.gz) = 5b59dd3b2796db425638472b47970280
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.srec.gz) = 77140597384a41679467c0cb3814f4524f5a7fbf
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.srec.gz) = 104d1a827d7a622c77139c4a5f6f36a205d708ffa3f3b438ba139981aaca00342288fca6adc82f00603d0210e05e9187fd4fabe3a79b3ab0baa5899a72ce6253
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.srec.gz) = 3c1e9ccfe1c9df8c07976c82348191273a09024e
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.srec.gz) = 4a503d1c30776dc77a7dc3ba3e160f37
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.gz) = f74be1c541128bf2c1bc8992058df64b48d1fb97
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.gz) = cf97c8ab92e45baf4fe64c2d5722391cc35569c035143371cdb746c9a3704954a06fc336c50af51f9f0b198a5e05d6e7e3a3e5b5257ed214489fc124525b8104
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.gz) = fd503fa3df024d8aa2397a060e986074453d8ed4
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-INTEGRATOR.gz) = 240f52b880f6c81b4e67314404fc5a97
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.gz) = 55a7ccaf693a8f72322bd4e898b4ecae4a8e0c80
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.gz) = fd40bcb076b661407034b45baf50731dfcdd9ff9f78b4768b0c300fc6e0187204fba1e9491c8ced8a12e77070a3f599608d7c2c649222a142efd0b799e414499
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.gz) = 6e69674afd923ffa4e320a7c1cdde0224959b3ad
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.gz) = f783d0328ed41dac449058e80fa81dde
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.bin.gz) = cd7ca6c6ba8134af29a1cb3c4db19e80753b5cda
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.bin.gz) = f1b0bcc810fa29ddd4102477ea422457aca2e037964a284f1eab1b2e31c4c408c5d495eb489c7b4b5c2c578998a9643f68e8577670b9b04e177b5147dc6b17dc
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.bin.gz) = 2e7c3d70ad38408b2e24ef3a6e3a5c22c8724706
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.bin.gz) = 693dfec6cb311df0ba890e55c610121e
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.gz) = a5ed3ef72be67e59c81f1701c630e420c19df6c1
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.gz) = b9b4dfb8d4aaabaf3346d5af652fb4d79b15c8ea43eba94bf3ee3b0a9f05bd129a050ede2b4c127f14c3df1149f7a9f965499d09a55640eae83151c42f054dcc
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.gz) = 12ea272d03d51081eee0ab7858e05bd77c21a86d
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.gz) = 114b5ae7efcf519aa2f969e827ef03a8
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.srec.gz) = 01d52551442d858c9e8b9404311a0c3e421083a5
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.srec.gz) = f74951ab8e5b2445643c046bee3d34c67310399a84db9b0c2e1c3fb7f460eb1d863f520c6703f51f270d6cd1bdd8ed0569bd32fda709e6f62a4ec9a8088418be
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.srec.gz) = 9baca8969db5db172716ad6b6e9154c42c0816ec
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-fxp0-IQ80310.srec.gz) = 60528f9ba917f0412d4c9026c58dba5e
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-mvgbe0-MV2120.bin.gz) = 1d0c2b24a143a451f4ea9e11f4580e45ecab2237
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-mvgbe0-MV2120.bin.gz) = 367845db7dac4bb80eab23acb6d40c193ac2c53239deca54acca2adc5c1161240755bc9dbf9ea4082b48b83eea70f4b01909c46f26f4221df8af74cf64524cd5
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-mvgbe0-MV2120.bin.gz) = 5bb353dba0b6a32a0916359fdaa5b06a260de51c
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-mvgbe0-MV2120.bin.gz) = 56c549b0fe6d221922b59b7ae00c6ca6
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-mvgbe0-MV2120.gz) = 72017ac40eb0de30cee8ffc340e9bb7f33cc40bc
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-mvgbe0-MV2120.gz) = 836362bfbeb34fc0040b9a3492012e1dbea1cf8c1b7bc6afa45c59e784caa1d07be1b829d10e807d1701d71d6350a3d1a5e63ff71d85cb3c37fe8b9b4f65c47f
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-mvgbe0-MV2120.gz) = d74682aaee589c7ca089a49bbaa6029175d13c3c
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-mvgbe0-MV2120.gz) = 6712477894ae747784d281b4ea595131
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ne0-TWINTAIL.bin.gz) = 0044ec7b51299209fe8a15443aace1984350d1c3
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ne0-TWINTAIL.bin.gz) = becc7276686635e6b10b03d8374f1a24526066692cdfcc72d556ce07a0a06ac44144ed011451b4542105211d9a9d0b24f408d193a932bf161a52cebcb49a6e06
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ne0-TWINTAIL.bin.gz) = 35e63f0805242303ded144ee50f429ed6b165797
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ne0-TWINTAIL.bin.gz) = 56670b2148acf612495ae5a30662bedb
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ne0-TWINTAIL.gz) = 3e68e8b0d100356489f0a5b027c19139ef9e97ea
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ne0-TWINTAIL.gz) = 6e87c01ae9a15702a15b2d6eb36a7a4ddaccd3ae0ff9188b3aca9f3c48207443829e2b57cbdbbb01b958cec1f8416e326547380928c928d0fe67a66adff9d2d9
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ne0-TWINTAIL.gz) = 50496c450bc32fd7da2e16674eff08df7e3ffbc9
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-ne0-TWINTAIL.gz) = 2d40cabe8960a91cb7b995cac28c016c
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.bin.gz) = 4fce330a36c78eb1de452bc0df57a84b5df02ddd
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.bin.gz) = d021074a1083f3cdf55ac1a1ef9a1e93268b418e36c01bc3a401d52d418f8402c19cba1c06a380c86c0b9f5f552cdf7cafeea19b2f03dfd46aad733ac7095479
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.bin.gz) = b573e7bcafe3e1d1fc25aa34177b9c5d845095f9
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.bin.gz) = 5035eae6a7bdbf82b39960661bbc1548
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.gz) = 3e5e3eaa9d205254d39746b8cc9c4291fc1a8aae
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.gz) = c454baa740a1e8e7e16c027ca4da512731a8cee8d265bd7ca7110fecb1019cff0e5b0db423aface71e0228639c4a7465108bf3fe69b9d711bd5f5113c4a0a6b3
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.gz) = 1aa0d312767044b044041a1d9431b7d4b4841bef
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.gz) = ccc8b7e6d1627a24e5ee493d3d769681
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.srec.gz) = 9e3be145e73c346fc467de425aa5ae84c8e86dcb
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.srec.gz) = c7f1566fa1665b6f81a36e0c4f7e00f1ccb08af73698d5d938fc26770c65f5485da4c106cde8aa8518fc312208c58182e4a9e6bd92f2cf6c3b7ffd0c822e8fe8
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.srec.gz) = 5446c67277640bac0811d35a413b2c9747d2f1f6
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-ADI_BRH.srec.gz) = 9072dfcce2a2736faeff274a61d6b378
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.bin.gz) = a8aa9bf7afaf2468e38fd8f9f3e2fbf1957c47e6
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.bin.gz) = c40af33b0a32888b18dc3e05cb747269ccdeb38f7fea3457be91688e44ee256754d8b31489b9cb6495af63bdfc6a6ea16a6f800b70f5f567d85130ca50ead974
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.bin.gz) = b0b41be5da00dd6a4c84277bfe39975a4cfe31a0
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.bin.gz) = b63766732a4710924581351a1778f71f
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.gz) = 6a53fe270036359318eb84b30bfd04f089cc6e6e
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.gz) = 8c61a2d1a08ae6c889304a8a1335d0b40d092f4f7a1ebc0a14fb721efe1d56fa7c3c3a43c65249754f0bd1baacae5f16a6ff497c6146c6c1fb805aeb14d07f65
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.gz) = 1de7507b9869a86de2468c9200d728f6a379148d
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.gz) = d878c13f9a7706e55848df3b09563620
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.srec.gz) = 4a762c7e63648a321e727cff47e99a01570ed1ee
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.srec.gz) = 19e1b9284fe2a46a5952c379400bb1d74117b3b8d2aea91e6820c1356f4e6761add87d462ec0a14a545534d1e17f211e505f539ff681f39eafa10b0893b13038
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.srec.gz) = f4b31cb80b5da3de1ab654f7287e544ca9e761cf
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-CP3100.srec.gz) = aa4640c78d3499c1bd1b6ade6b5564df
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.bin.gz) = d297e9f63876e259acadf43d735d26c70648cfea
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.bin.gz) = bba032dbe3a47aa40f45e232a06c38edf352515983e9896b56a28510fc031edb12a3b462ea799bcbe5e3dd2ca7585729c40e8203a02347aaf27f367ebc027cf8
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.bin.gz) = 3c2d8df7cda72b357e291b47454859098d5a71b5
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.bin.gz) = 2cb06ddf778f331df715f39fe0443219
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.gz) = 0ed7595924e6d03491d2bd7d32d42d00f96a3834
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.gz) = 0e33713d7943731f3430b74e4a72cb98ace2878790ad7dc2d185f0e8a825c2a9db9c3194418bf91ebb30c4f87659d1de36706b57349a38d85e584456071d2d84
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.gz) = fef4dbb06bdf1299ffd2719ef5c147f51704ae0d
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.gz) = 142bf1b95c4e72d91063e29379b858a5
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.srec.gz) = 4a12c83305d58e03cd8c49f05cc6b32660939700
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.srec.gz) = c7e132403f37ca9514a8bcd8c8a94de04fb63a2319b1510db5ce9f14d60ff45778228e4c6d9ca5b042bd16ea26396e035a31f948908bbff152a7bf9529b3854a
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.srec.gz) = 306ccfdb478dae1136303199e84f54f1e65a63f9
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-INTEGRATOR.srec.gz) = 4805ccd3adcb1ff438b7c4d900416284
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.bin.gz) = f3f8c21960efb2c67418c29fef978e98b472a813
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.bin.gz) = 7b37f0c129e92372d4f170ab4106c2bcbdce9628c03a22ee1a265ee73263e78e3befbb712ea1f241b2e2b0e0b54bca3a9b3f7628927a18ee28551e867e526bf0
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.bin.gz) = 237204cf97f62dbb37ec28d9a12158f3d7ec01ef
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.bin.gz) = ed5f31fe08e40a01da35e68a8ce24da5
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.gz) = 2aee3ee787581f73d3e667aaa72e5211a316db75
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.gz) = 6a131f56fb801174037294c32a30244a500fa4bcacfccc65b2e1904b63450cd0ef0656dfadb4eaf1db651b8378070b97a93ab2df053641324c345766b4b043ae
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.gz) = f4a9d9ec37f3a9bb98055681735399a914673fb4
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.gz) = 25bbb323800e83e721976fcf4b5d42c1
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.srec.gz) = 766410442e0892a471e2dd4420209f6974ef16ca
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.srec.gz) = ce977a4a9c87431fc57805ee22efe908bb79a344c6a1d2f3c75b974db3b54626908fd25e524a4ed3f55327877c97f78e9496515fd6c4677fd98b3c273ccf1c2e
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.srec.gz) = 093855469bff467f4744c3bf678d55d9eb08b951
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80310.srec.gz) = fead543e57b7c2f2a458d451b6c59f08
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.bin.gz) = 257ee7c14d678bfbe2511fbd47d72f6c8c461752
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.bin.gz) = db6a78dd47b3298d8f66fedb46e00b6c269e3ea389aa276480c947f606690002c95e7cc2306fe7f243117da3894ca7f7ce9b3bbc12e2cbed4d17b0ece02dddec
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.bin.gz) = 542ae9874a3223e014f50d542c9a677d0a0efdf1
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.bin.gz) = a70cbdb6278eab864d1bf3b288c3392b
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.srec.gz) = 697561a2e82361d2d92775821ea4cd03e163c217
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.srec.gz) = c1959211704f091e865ee26560496424974c0dc3ad3085df21fe51b21c95a56d7f024f833728f9748b02f4ef409a24454ae731d071764921b0fe57df38a3ed3d
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.srec.gz) = 7d95cff40a30f44d91013c8bf2e13a540470e5cd
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-IQ80321.srec.gz) = 7cb2d5067df40372c016211fee729a1c
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.bin.gz) = dc0f06b24ab21f8188a4e3f0321246acf541854c
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.bin.gz) = 13a84014dbb366e32bd715279e46c67a00f09cd40eacdc0fa91cd0e8a7e3e4862110161ca56be8024d3a0cd14f2a17c539111b366e91a27c65c5c47a741dc4e2
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.bin.gz) = 79967e4ee6c3b5038007f8deb9b6ec9172bcd4dd
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.bin.gz) = c5caa593ac363dd9dbf8f44de4c06099
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.gz) = 6f5c0d944bdcf0b3dfa76f60e6b5035dedc94c00
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.gz) = cd56585b5fd7a8d4c15453966942d48c538004a142daf8b0f30d26637e82e0483dba20680a303bd561341b0c35aa4a537f6a24764bd96ca8102fb018d57b22c4
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.gz) = f0c30e5955068a64ce24eb6d96793fd4cabebba5
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.gz) = de797d7956e6fd360b2f4028d05e7608
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.srec.gz) = 0689f7b9a0ff0140a28fdcce4d8cd206d566c680
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.srec.gz) = eafe6cf5e618cf43054145f925f154d99acaffc7eff8d4be2bb801de61c27193debb983412b45c4b14b5dc834633452f89c170a895078ccd31f00a5a0685661e
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.srec.gz) = b07d921cd736ac79d88e8f0e041db636c198faad
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sd0-TEAMASA_NPWR.srec.gz) = 146f7101d810d5b401dc3b320a2f89bc
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sm0-GUMSTIX.bin.gz) = 393c30bf2cdc7bd75e749dd8b2537bd77dc91a65
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sm0-GUMSTIX.bin.gz) = 44d1609a6be448f0df6d7f733a9fa544fdc673da11e0b64b66174a9889b4bd6c069c69a312853954c0c631eb036f37ce56beac486e347e600be01de6c9c3a73d
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sm0-GUMSTIX.bin.gz) = b7b4482aa76347a87722d9eb34662af96fe94e00
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sm0-GUMSTIX.bin.gz) = 6bc478629045ffcd392adc0f5c9d133a
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sm0-GUMSTIX.gz) = 1ae50b9a6c3554474481cfe3c37dc88bc29cc436
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sm0-GUMSTIX.gz) = c1827ec89039136799b21418e0df33992f4ee4c166c130c34865f22cab0601fde056078892a76075714bb3ff09dd1edc54f33464c0dd8854f303398b6357b09d
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sm0-GUMSTIX.gz) = 823391035b738182fdee19bfc807eb2f09573f1d
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-sm0-GUMSTIX.gz) = 46159d039abfeb7e9dc00177ba1f9271
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-smsh0-GUMSTIX.bin.gz) = c7bb680f57a438d65351ba2e665b0c98b979fb80
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-smsh0-GUMSTIX.bin.gz) = c8cc5ec9d35ef1c8bb0db010e358f331335fc69db09f67110a92eea46eb275950f2ea5cb4e9f88ea2ed80561a5e31d88ae6f76e5582c448f3df265dc6141be9a
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-smsh0-GUMSTIX.bin.gz) = cfa0506e9de7a97955bf05bb9d89275ae65bd77e
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-smsh0-GUMSTIX.bin.gz) = ed0673b11f604500c60263dd3dca3991
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-smsh0-GUMSTIX.gz) = 6a22b3a0b0baeb17201a15a1e46b6a1bc22345f5
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-smsh0-GUMSTIX.gz) = 1a44e1fc731b0099fc726ba2362c5967f0073ada7d036656d5f60df865217349b5304bf05c7699662f582ac2519f2db64ee774bfe58c1780824ad04c8ddc8390
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-smsh0-GUMSTIX.gz) = e5872e39e97e9a9e1d72da6d35d5a8013e168b59
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-smsh0-GUMSTIX.gz) = 2d76c6a78af5dbc9a1e5200819e53511
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.bin.gz) = 298b05e33ef85a4b937c2a5e84a48022e388d0f7
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.bin.gz) = 3d709cc48eee7fad9cd986542430e23a4b81c84e549d2f0f9210df57a38a9d36eb3abe1bc2ccf2c66d9ec84575bdc9c1b55981f3686de5ff857b55f86d9ab661
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.bin.gz) = c2d4ab46b1b90b55bc40c74d31f287487f781142
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.bin.gz) = 91e2b298632e8fa0ebebad32ef0df715
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.gz) = a63011e3f96507e9585442178de4c0e41338cf87
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.gz) = 76b99b424e4f8cc4f47b0ba7ed8b4679a5d49ea3a022a84d2cd74be396bc497f481fb3fa16e23deac876d5dcbfc9a29f745bb1672d2844fce15252db989c7677
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.gz) = 8cddff412a5b879ce5b17a5a22b09603ae9f9324
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.gz) = 85c7096f583e4f9f3841d7f663de4999
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.srec.gz) = 83cec666fdaa7fd2d5fb6de32461615839932cc6
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.srec.gz) = ec118ec8fba3c15ba1b6725a1b9778d14c4b4fc6f624d7979c281d6195271df3bbcc6668eaf0bb25b6089243c5d7b05e43ca6247e3d72a548abfff0f31f14f10
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.srec.gz) = 6741604d5336591c02e6a9fce7cd093f7e0703fb
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-ADI_BRH.srec.gz) = e98cf8ea7ce8704d98880ca519bde38f
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.bin.gz) = faaa399c0796bde9609425f73ebf4b7436516a09
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.bin.gz) = c61821bf159dca3ef32e69aff1f935a9de0228e5e0568afc74dfaa5e91902fdb0b13589e70af2871b3c7afbe2cc3777be64e9bf52798534addd6338ef6fb2095
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.bin.gz) = 081460e1657d3f542615c9b63d2bd8638d4f64bb
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.bin.gz) = f79d3833d46191e99e7617029c981cf8
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.gz) = e56e0fbbbe556a6741af58398b3d7d116f3ce469
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.gz) = ebdbee8f4ffed5e2ffbb7890a8e693299e441aa70aa5a7249f1b468373970bc29d2284d82bf638cbbc258ee8b500ba5b3244b83fedcff1e1465218d9fc76575c
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.gz) = 881461289bc9ee95132cd5997c19c3838aaa6c39
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.gz) = c5d42e48168ba69e065be3f44d2e2e09
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.srec.gz) = 1fb817599a91b10187301c3f2333dc6484451556
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.srec.gz) = 19f6d85d50d1fdd5fbabe1f60b67992acaa80e0afa44ab7d94f31e095592b7852f2c8f42227bd9d90de9698211085a69aa86d4076100711f96e59bd446317164
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.srec.gz) = bd13c2dec6d22ca3e1d245159b0e765ff03301bb
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-CP3100.srec.gz) = 916df9db9485676f33a5bfd017e4cd2a
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.bin.gz) = 2c81eb3878a53e36a554b1f4e91c5011ce4ba507
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.bin.gz) = 2aa85113e7bf69371c8027929857e4c35623f07a16db83bf1cd3824053776e0daf33ab8ddd2e954b938fc33424ea044101c6f9c6bd500a32b030f28ffe40049e
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.bin.gz) = 1e1187fd6a831f7e9dfc546280681b924c03b36e
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.bin.gz) = a767b383bfbe980c1da78e744144dbd3
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.gz) = e60bf72e33a1948bfdf1d84a209b1ab6e4ec6522
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.gz) = 9459f4a7532ac2d2890c538a3325c217d1ba468db8e8d4c7d2fae8e7a4ba256e0b0453e5757e1e2ecfffa5f1382d843289279d471562911d56215f2f3ee73eac
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.gz) = 95de4a223120d3c34a7862edb781345652904ffc
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.gz) = b7113d5899d418131a2d376711e32379
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.srec.gz) = cc193aac830dabd4793f1fb6722cadc6fed0bdd4
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.srec.gz) = 05a81920bb52518267ce4046aa79824a763dca9643146646bc6746d16dab1854140d28dabb5f7ba1ce9cfa1cea3a2e9d76df3bdcf227da89ea883011c464a579
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.srec.gz) = d458f2b8a4afddecde3bad093576b0b6a17b2f3f
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80310.srec.gz) = 9d205882f4befbe25707f009910f95b2
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.bin.gz) = 78884cf87f77551ec0e47bf3d629b41ca5a4420e
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.bin.gz) = 094252727a03ca3dce207c5c92959ba21a888bddf502d5a9ee9709bfbdf9eba5d60dae5793ca6ab5dacf720d475499710d70b36622b97c1449badc5ba85e82e1
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.bin.gz) = 4d7b190601413c8c89ed39d8dee44abbd388d8ad
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.bin.gz) = 6aed071b41867489457065cc3bddb0bb
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.gz) = 771add734839295eaeb51bb341d43cdeb64733b9
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.gz) = 1ad26279054ef65ab4c4c37ed2443087a04d530641cd005b7d02dba395bad6770270b6f4e5d9ee522fde9ffcf3e4842e704b4fac2896eff2e73db39cd4c7e96d
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.gz) = 578a063f106ea04a6e8133cfa3c0eb94b0decac3
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.gz) = bc6fd4df95278bf2e327a3b85a5528eb
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.srec.gz) = 1b77033ac9b0bba5b829868fc3a826cc2b8bfc3e
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.srec.gz) = f14f41fd0600a3502383ffcedc902975372ee3ac03ff8e5ff156e93230eb8572798b00a6049a1cbe6c09d53002a07d6242064068b030a8e588355c0c67508e4c
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.srec.gz) = 5360066de4b445f57064f767cd41c16d2104da1a
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-IQ80321.srec.gz) = c4197fe93de106e7b927570baa374dbc
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-TS7200.bin.gz) = 70a3d642aa8ea8d1a3e9df143a9e453aa1f6284b
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-TS7200.bin.gz) = 391ac58855b9c24a8dbf8baaab4e9b4efd956eddf343a0bdea8351e1469187798e36ea875d6f4d91b5ba2be0eba5b9b02f33a6eded8fe6d727d3703950d554d8
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-TS7200.bin.gz) = 7f860a8346d99c80a3cd48694244040d976283ef
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-TS7200.bin.gz) = 379072e809792e50b4015b9577360574
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-TS7200.gz) = 0616f7ca1f3ef9c60e4c34aa459b40fb72277025
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-TS7200.gz) = 8530ac25385210421ab23391925fee75ffd81e369c6b9773034e850519577ec69ba74493a22b08835ff847251ed16bbadc5e86c7956ae88c48e5bd2fff61417c
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-TS7200.gz) = 18dd6a88210d336fb61e3dcd0826fe50c8d13f94
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0-TS7200.gz) = 10be5c2ffaf39f29b3948473e78ea1a8
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0e-MV2120.bin.gz) = 3ca17c6882a1519488a77a45207f3ed8ea19fa23
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0e-MV2120.bin.gz) = bd41dc8a2e268be47c2900ee5564bbbcd2ad6635f7ab99ff059efb71c27c277cb9da8ebfc34eff5d17839f345867ab9d478d0d027e7980fff4fc3e7494b1ad4b
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0e-MV2120.bin.gz) = daeda9ae3e3cbbf3f5a7a7c6381d13a30070ccaa
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0e-MV2120.bin.gz) = 6eccdeef46d96a5ed0267484262bdee2
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0e-MV2120.gz) = 219f483fe3afb49cdefa8b1865cbbc69730090de
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0e-MV2120.gz) = 937fb99f921b629c1fbecbd60a4de24fe1c55ea70211b03ac15d2b20ae925122a6e3d3b43e34cc8ca6fa9b1d8a1f500ef29ea55f72b1a80a8f52ad43ed6fe1e1
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0e-MV2120.gz) = 437b6861a5f1288ea6553790feb1de02293ed3b4
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0e-MV2120.gz) = 4d05c9e5325526185f0db4a6a11edc92
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0f-MV2120.bin.gz) = a2d678ea1b57438872bb33f8e3f5ab311680ba55
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0f-MV2120.bin.gz) = adc42e2e13bdef6fc24f27cc4cff4363e0287350ba2afd7681e8f3f027865cd7f58e3e5d4fe59131870b34e8e6a8721a935adfb4930957525a24c02cd4c2ceb3
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0f-MV2120.bin.gz) = bc32af49bdcc8ae9226c97ae03f4306efb1b646e
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0f-MV2120.bin.gz) = 344ed7e1145103dd1f747176fa09a993
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0f-MV2120.gz) = 98a9af8c3b74a45940b587383545587a300274cf
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0f-MV2120.gz) = ef6b06092f0a9ada2774b3b172ed405d82f9fb3ea0c3c53c7da7d493f9c4fdd92124970af972342e3134852d9575a80c05f0ab22fcff70b52dfa4007b31c5a10
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0f-MV2120.gz) = 1558754fb8015e4564b65823fa61d74e7ee2758f
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0f-MV2120.gz) = 4c2d594388f5d5ada250a5e9137a2dd1
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0g-MV2120.bin.gz) = 655147b472960f9a2f0addaec681344d62344d52
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0g-MV2120.bin.gz) = 91448fa3831d263781cc4a54b9103ef3fe99f7f71d1b4d5f25ffdc049ad34e9ab84fd38e7703ec55f1875278dc94e808b984b6d6610a0a310b9cb633205a864b
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0g-MV2120.bin.gz) = d985973ac1155c6506fd2dc36208da2c89c1c010
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0g-MV2120.bin.gz) = def424287a7d6e0c9dc9878cb6af8d9b
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0g-MV2120.gz) = ecd70279e4ba36c40e8aa14b9d2705942e48e0de
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0g-MV2120.gz) = 1f4fad54a9a843e66fca15c0ab3de33388b5c94006a051eea03cb3078fce4a671aae802eeb94b410a47b4ea291c7bdac0100359fdd5405fdcc2b0d8db3aad9df
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0g-MV2120.gz) = 16f2789c94f147c86ee94427a2bb0cee353d91bf
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0g-MV2120.gz) = 9df82b31b91bceb714ebd8a7036eec8f
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0h-MV2120.gz) = 7f65f7ce1167b8b0bea66a821a28f8a328f960bc
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0h-MV2120.gz) = 7d98ecd33f11bfcdc1e89d02df391b5c7fb4d66bd66af2713913b9ad8a6de264fb18f64020f009e21f7fe7fadf6cfeadcf72b195be731c5365745d2119a22c08
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0h-MV2120.gz) = d84424851ce6a0cefd0881650870f5c902a50e37
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0h-MV2120.gz) = 2b7ec0cac876ff4758fdb4cfe7e32d7d
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0h-MV2120.bin.gz) = 92236b446b36023cd70765c10fa61fbe3bd2414f
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0h-MV2120.bin.gz) = 2882f6b905d11bd0019d4657586412e93c566abf48d10702af36ccac9f688886fbad3d8ef62c3342bb7ddbfaf72f2cf7510d6635f9388187bd7932c814aea288
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0h-MV2120.bin.gz) = 7bb29d6cd1f68788fdf3a98d198ca7d98e264c7b
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wd0h-MV2120.bin.gz) = a1d749d80636959956efface64861885
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.bin.gz) = b3937345ad5103a5331f8f3d9217f6c73b7a2e28
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.bin.gz) = 1ddce72f72e631cd8af38d326ff826ac52005cda9974dcf87c33370adf04f90e7a9025177948ef2ddf2f78e259c535bdd5ce2d9da2541f4ca4cc8d57a927afa6
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.bin.gz) = adeb55b34d1468ac0fa1ac03ddeb6d86749a4994
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.bin.gz) = c10dbd3dd8b711e0e431146585d862c8
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.gz) = cf1dad35f06a51619d75518386fb7f0f60330734
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.gz) = 085729350310674f285fe2ca0fb41f63701eb1f8f38da50329483d2dd1e4644a84c20815f94cecadf686d925a808eac57a986338aac762e4fe9080b2ad5adf99
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.gz) = c51e997b08e72d5e9a1a4953dfca874d4d86d4f6
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.gz) = aa4e8b576991e29425d66131a6737800
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.srec.gz) = 6823aedd0b5469af8bdc05b9fde2fa1bd5133e48
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.srec.gz) = 1b372e4d5cc54171ae399d5d55558e8cc99234c08172c7cd2c83185ec2de4a326aac04a800a050be1d1ef42b60dcc4b20ae25b5fd1f08f707a09926268207031
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.srec.gz) = e072eb111fb0bb71c30fc8ad457e1d2711591887
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-CP3100.srec.gz) = db7f9d23f27d533266eda56715e2adfc
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.bin.gz) = fd211bf40ed4f68e9f05bb5f6da85eeeb92e35d3
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.bin.gz) = 1f8dcdc7f7d06a42111745ac2754644cc5796552417f7a0a6d80fb4a7ffa667c6ab577e76f84e2f497d367b244f578b5a1c1d882c558929b12470d02026a3b3b
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.bin.gz) = e853cff382e03161c0354f83f100c67779ecc7bb
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.bin.gz) = 75e9772c889f60ae2baf766aff24f997
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.gz) = 03973311d8dc84c5522524968506a70f3e69e93d
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.gz) = 0582b69eaa266f9874ad02fb21fb7e70fbebe3952d8d2826bc6d58f23cd4d043ff3971775ecd1a12b0a828ff7266677f40c495bf394a2f33c41d372fa1ef7668
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.gz) = 8e8b7d0e5619ca58a50f6ad1f15b2764f9a3570b
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.gz) = ccfc004433b54dc119d556d198e611d6
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.srec.gz) = 72b16c8dea4c9391cb9282e2c1f91e3ba30d9522
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.srec.gz) = 94d123f6a62639d3849b60441d19e49359a922881b8818c11a41cf43ffaccea3e30b5a001e6b96ffea0f6863ca5ed6465d4ae6ea9ec6e91f95c9e498deb44105
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.srec.gz) = 4c94b472797fe5fdfc6bfc8427abe598f7ffbe21
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-IQ80321.srec.gz) = 3c722e71ccd200b5d113f3d5d3b177d0
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.bin.gz) = fc8ab0bc9eb0d8956ade1baa5651d278852ee08a
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.bin.gz) = 00c22223dd564fb484a1a477efca1a4af97bd910a88b722cd3ef05a55f4c56dc90ca0eebb6563ef73106bbe1e3ff1e9c2f51e14ac1ad5277035a1beed8250651
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.bin.gz) = 081ee767155378a6cefb29064e13c46e4de45508
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.bin.gz) = 7c1846c7207b9760e716581a07a5d366
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.gz) = 72d1968438f4d0b4c063d5b85009669416b39892
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.gz) = cfccacff88e457bf30f0336db8882c63e03fa8ac8eb7d2ae87b7a6bd0eb36b5708eaa512aba0df46ebcd0a89cd7732a0ccadeab0776a54eb4e61ed32bec40df8
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.gz) = 3fdc54921cf7a3e81ae3be6854847d0026720467
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.gz) = 74ef3b97327e4591d8a3d6acb59c6c3e
SHA1 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.srec.gz) = 26020265778b152f20cd35c43dd0a8e8be644f74
SHA512 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.srec.gz) = 672d1ca404b4dabe7791bbe1506b47e64445c047f92c3eac1810b8deb25e23e6a1e554acc88a307e4f85850df967e521f36c7ffb90da7325f7a3d76aba3494b8
RMD160 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.srec.gz) = 2e0466a32e150882f148c40d8096b53849a7bfc6
MD5 (NetBSD-6.0/evbarm/binary/kernel/netbsd-wm0-TEAMASA_NPWR.srec.gz) = 9b35a2416410acc9d4fac08926ffa38c
SHA1 (NetBSD-6.0/evbarm/binary/sets/games.tgz) = e1d2bb091d8202962133475bc83edb55230b9008
SHA512 (NetBSD-6.0/evbarm/binary/sets/games.tgz) = a147fe6d0419d53d5a428802d0bb6c9fa8a84a76c96d7d6d7c0d399e5af3a69240116463549a83d53387b0214ba0207796d7c10228d92a684fe7ff0ece3aec16
RMD160 (NetBSD-6.0/evbarm/binary/sets/games.tgz) = 2be515c495548ba5135ff42a75a8ebbfc35ea776
MD5 (NetBSD-6.0/evbarm/binary/sets/games.tgz) = 9059a6f2cd41cb58827709c9e9a1649b
SHA1 (NetBSD-6.0/evbarm/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/evbarm/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/evbarm/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/evbarm/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/evbarm/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/evbarm/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/evbarm/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/evbarm/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/evbarm/binary/sets/xbase.tgz) = 875fd641f969ec00f854ca1334d084947450c740
SHA512 (NetBSD-6.0/evbarm/binary/sets/xbase.tgz) = 75d0c9e9685b6b7bb706df7427c8b52f59c4fc4d0e725b02ee18024f75a546dc41de5f1a5d8552a8d2c84bdf87856e5857b0f12914801ca7b36b69536d617b4e
RMD160 (NetBSD-6.0/evbarm/binary/sets/xbase.tgz) = 1a4d099d59b2bd882e99f1b2df1c7bf22b890d48
MD5 (NetBSD-6.0/evbarm/binary/sets/xbase.tgz) = 5eb9c573c5a0c96701f268a3470a58b9
SHA1 (NetBSD-6.0/evbarm/binary/sets/xcomp.tgz) = 8d7469a85b7433c89690cce74e6b3f4e66cf0647
SHA512 (NetBSD-6.0/evbarm/binary/sets/xcomp.tgz) = 411304cf66e300a7f162128203099bb4d6eebc4a93ba18ed7caf79f30b31cc7ffcede85c4e1bad02c3f5f8a27d009fb71f3ce5862c1d8aa477f39a25c535fdaa
RMD160 (NetBSD-6.0/evbarm/binary/sets/xcomp.tgz) = 822d26e0ee08b72ca8b991f0709a8c1bf0399208
MD5 (NetBSD-6.0/evbarm/binary/sets/xcomp.tgz) = 5af361f0334d28119d703064278b3ed3
SHA1 (NetBSD-6.0/evbarm/binary/sets/xetc.tgz) = 222d14c205f1018cc5a86eb03214d18cda2abb82
SHA512 (NetBSD-6.0/evbarm/binary/sets/xetc.tgz) = fb82261996aa772fd8a9022eadebf899ee04ee70229ee4fe07758ca14937bdb34752c8218eb4c5b722b6208f719891493bbc141372fcfa503de83e0a61aa7387
RMD160 (NetBSD-6.0/evbarm/binary/sets/xetc.tgz) = 511ebfe5076f361b7f7ca03a265d16052a16d259
MD5 (NetBSD-6.0/evbarm/binary/sets/xetc.tgz) = 165cffbb6d2c79663198d80c82afac1e
SHA1 (NetBSD-6.0/evbarm/binary/sets/xfont.tgz) = ce0e141beeffeb5b470478e9887e4127da85de3c
SHA512 (NetBSD-6.0/evbarm/binary/sets/xfont.tgz) = 5d98354ff77103793d56af3192ab5f9afcde62208791976064ac33a149f994e0227dd5dd28ed544eeac5ea504e5212d5fb9e7d05e59e5a48a2fe4062e34e3106
RMD160 (NetBSD-6.0/evbarm/binary/sets/xfont.tgz) = 4d09f5e82fec875ef0ea328809465e7a98f2ebfc
MD5 (NetBSD-6.0/evbarm/binary/sets/xfont.tgz) = be26b3890261302420b09976a1d38144
SHA1 (NetBSD-6.0/evbarm/binary/sets/xserver.tgz) = 4ccfef1d49179603286042088bec0218c6b2031d
SHA512 (NetBSD-6.0/evbarm/binary/sets/xserver.tgz) = 809fc0a9065c1b9e7cd17aeb0d41cfd3158efd6c585556af01c4a3c93f0df3646a2ff77c6994449a703a408c0988fc1aa68cf34c7e297a3056d73ded57034c12
RMD160 (NetBSD-6.0/evbarm/binary/sets/xserver.tgz) = 403604404261764c74f10fab774c4653fd20cc9b
MD5 (NetBSD-6.0/evbarm/binary/sets/xserver.tgz) = 914f909b6878cef16a92765a8cbb52ca
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-ADI_BRH.tgz) = d9cf46427b7ceed2cbe9e6bb2b62a5d461799474
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-ADI_BRH.tgz) = f2beff9ed4d712db8eb3bf1d9e8f60942e3fab2a834b73fed76ac79bd012fccbbb6698e04edd7ce7dba7802852386bdd2996dadc2c8a04a409fae303dace1104
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-ADI_BRH.tgz) = 443b7c69a5f5abeaa3dedd3b6213486af56eeaef
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-ADI_BRH.tgz) = 6e3e8b794a16eb8a079342ec0b1af212
SHA1 (NetBSD-6.0/evbarm/binary/sets/base.tgz) = 74a00851a474a252991743274528e6272f50dd57
SHA512 (NetBSD-6.0/evbarm/binary/sets/base.tgz) = 154a8230676785921f3d6819d25a2900fe860291ebc2e4891fe94b039871e1515199a4c0e6fe3bb5aba4f9ee802a977e985f87206a1688e4d7e0ad24e1069eee
RMD160 (NetBSD-6.0/evbarm/binary/sets/base.tgz) = c6d3c3b83b8a2a56870929a9d616393e6d19e9fc
MD5 (NetBSD-6.0/evbarm/binary/sets/base.tgz) = 2e26124352843379db4fcc7ab680ec5f
SHA1 (NetBSD-6.0/evbarm/binary/sets/comp.tgz) = 4004f24b5f208575da97b542db46731015cb7573
SHA512 (NetBSD-6.0/evbarm/binary/sets/comp.tgz) = a0950dab5cbb258ee2f29f2d35173e00a5cfd79ef30b27605782b9f22b8564e05d9633ca6886178c1b1b0cdd6a8112d3a9f663211804e554ac63ac691ff3f9f7
RMD160 (NetBSD-6.0/evbarm/binary/sets/comp.tgz) = e8a618972f78271ba8a957f4f626b632d1b03ae1
MD5 (NetBSD-6.0/evbarm/binary/sets/comp.tgz) = ae2425114d29b9fd2136342ce730baf0
SHA1 (NetBSD-6.0/evbarm/binary/sets/etc.tgz) = e3f3386182b99812eb3fe35daec274b0da1aaa1c
SHA512 (NetBSD-6.0/evbarm/binary/sets/etc.tgz) = 46337fca0954acb3b4e0e6af3b6ecd473002d2f482468ac98c62ee7ff7bd84d90ec9c2bb13d29c5ae608e625f3075471af1da22f8a4254d4e8efc9ab8dcff21d
RMD160 (NetBSD-6.0/evbarm/binary/sets/etc.tgz) = 098b5563540c880d69e0ba9aa793ec9ba6f33a10
MD5 (NetBSD-6.0/evbarm/binary/sets/etc.tgz) = d3f4709b5264a9a8bd820648330a9653
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-GUMSTIX.tgz) = 8a21525ca51f643747145c529add5e1de487f0fc
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-GUMSTIX.tgz) = f0f9d6e4d8f34b446e511db5a0df6b3d81b20334169caa79790cbbf34a8eaaae0f9da332559b3cb78a00fa05b807b9e3a118474146c69e75d1519c46420003c1
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-GUMSTIX.tgz) = 47002447a76cd76c4a2687923682dff473ec25dc
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-GUMSTIX.tgz) = 1ad356834d9467703647793950ccd9a5
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-CP3100.tgz) = 4baaacdf8b2ae99e945eabb70c425fd5952eca48
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-CP3100.tgz) = 467818af6caa1068437b691fdbc77e11617808fa44e2b3caf3634f7cc8bec1cca7f3a46325825d9427cbfd113eba01a8d3a4d78da832290ff40aa60964c4b42c
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-CP3100.tgz) = b2a46e69e6b4ce2264eb78a2151274796b224dba
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-CP3100.tgz) = 1528b53a9a8a622972ed1e6891535619
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-GEMINI.tgz) = 38ac63a06193c3ad7814b39a11d911da20db6ca4
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-GEMINI.tgz) = c05553408f0fde7a1a428faac350de7f969396f9e8caef6e83a269875743472e52435d6bcb1ad5531f9622c1289aebd419cdb02ad83c92d4797e1b4b952e7fdf
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-GEMINI.tgz) = 5f8d006417864f9609d691652fc03f37f1d12769
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-GEMINI.tgz) = 0767217ae126cf86a65cb17bdbeb884f
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-INTEGRATOR.tgz) = a18100b11fa96e4cd1eb1e36831f14c3d17da04c
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-INTEGRATOR.tgz) = d8bb0260e9e2b687bce84053cb3c88ef6ca3eaf12b79337746e62be94555233c8a1302dc104359b2aa6eb64eac8d633d0640d533688af4c4971a8ea095b96940
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-INTEGRATOR.tgz) = b8b57516a6d2c5e62d57dd9437fde58485989d09
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-INTEGRATOR.tgz) = 7f80ec30a784dddfe53492ecdf78646d
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-HDL_G.tgz) = 6e95c2e313c04d228a65da51f252791fd1476827
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-HDL_G.tgz) = 380861d0bafe812937f004ed00054b6fbcc1dc581fbddf3f745101ec39a57759dbc0a33f0f1e8429a9630febd7ae69e1fc88c75d39c196234260bda58641edd3
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-HDL_G.tgz) = 5262674ba10ad272c62440d4f61034dad9be072d
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-HDL_G.tgz) = 1af966fc2df6c623f4147d3cc648d22d
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-TEAMASA_NPWR.tgz) = 239e37769ab837541fc445e7a74a9bc02c74f0d3
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-TEAMASA_NPWR.tgz) = c3d61466da0066d0a61d1fa3052e54f10c19f2f20fe8342c64f0dc6365d95602e47c1e1722d242219e698c824cb4c6dd823b54915eb98f9c6af8a6119920fd6c
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-TEAMASA_NPWR.tgz) = a4be2875ed76751028c167d6a0c2efecdf3b79d3
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-TEAMASA_NPWR.tgz) = 72220c0d721aae18cf77bc235bb03cfb
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-IQ80310.tgz) = dda069ed4e7ad7df46bc23e51a6ef2013af45dc4
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-IQ80310.tgz) = 91bd0716398c20a01510a6fe340891d1712da51882e2019bcb7e5852c5b3ac706aa36cc0ebe3f1c1db4b4318624f767118b9ab549a38f253c477b4b25acb84fa
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-IQ80310.tgz) = bf180d150abaedea9aad9265bb4965197a587490
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-IQ80310.tgz) = c00c7a69803c130c2f7d34fb3d91610e
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-IQ80321.tgz) = 2d3162d5072ec3659964ff812b3595ff82830fbb
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-IQ80321.tgz) = 199bd6e9a8ee5547f4b589c2172400dc1d8204ff2c75ac021004aecc816f5840b04f3e8408e43d64722e829acf654069a161750b733af045648ab5647071a113
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-IQ80321.tgz) = 22d77cc3665909c23cbbd57dda56f412c683b75a
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-IQ80321.tgz) = 2bc4f1eb1a0355fdc0dc886c13c895ab
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-IXM1200.tgz) = eaedad32158dd867e95063f5bb56c66ac8a8f3c3
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-IXM1200.tgz) = a8c1730348c3b3aac15b11c2cb598f6a667b9528748e4361baa33ee7a899aadc43366cab3ac0d6a4305e605287952a8ba6db210f4a15fb3158bded3da4071a67
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-IXM1200.tgz) = 206d96f4304bb755159fb72884977e53bd608c34
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-IXM1200.tgz) = 2e6747257e639f5fe31d23d1834ee9c6
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-MINI2440.tgz) = bf2d80624e97f1964673908ef28ded8091568b21
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-MINI2440.tgz) = 1669f39e3caab537018880d4e93d6010860689f37e60f9d17ceb0a7209442053b1f461b683b6ba201dfd1f0c8641b8650d1561985e8fd47fb80aa3c71d2ee938
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-MINI2440.tgz) = be7274304c9c2bb18fe71bbeb9368719ca155cbb
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-MINI2440.tgz) = 1c1547b85d29dae79c76f4d7ee727b31
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-MV2120.tgz) = 363e695b0fd04709143647a1f07b22a8029c24b6
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-MV2120.tgz) = 99cbcf9b5969385787e852546f3e5685e721c7ff5fb2f898e4d9f29e2bb7640bd08e38cf38eba4cfbfc5d2c1ba3fabcc074328b5ec9bcd13faee233e9a5c10f5
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-MV2120.tgz) = 7dcee51999f376d080942e2f46cb936dc32385b9
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-MV2120.tgz) = 82c6e07c9bb44d8c06d4d4ef7228fc1f
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-RPI.tgz) = d01b81e07470d09296ca9e2e07eff84ef739865d
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-RPI.tgz) = df740174545dd2a34f6692b7a15b329c93bf1496b32b70270e339f91cbd6e9c980d92a4da0ea5a8e825eb39cf6b4514ade0d2c78846128ee424fba595e175748
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-RPI.tgz) = fb98d2b03e66057f4e002505e26af67d23da171b
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-RPI.tgz) = a07be9e2696fa18bae91363005cd09b0
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-SHEEVAPLUG.tgz) = 0f11f8084b40c9f41ef77dc6ae2ec257e819f5c4
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-SHEEVAPLUG.tgz) = 8444ff6b4f85c4a8569d51147c83583b37a1dee134c01bb1991666f02d1fd674d80899ac9d11dbe0457a6a1246028a2898444ef0070367edf66dcdde5118d6e8
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-SHEEVAPLUG.tgz) = 0f00bd92326287b2db8c825f6e7a93c34e8d7c63
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-SHEEVAPLUG.tgz) = 3221d316f1518d5bc0680bf7b1179a4f
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-SMDK2410.tgz) = 93da07d5884e0c20544d001c5f9fb49893f5055f
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-SMDK2410.tgz) = 69d65444df4a31c129ac2b2f216d6c0c27e84243f3c1870d0e7f133afe6c95e419f1e907b4a4f0ce05162be24ac149616da16d0f840700b39393bf090e1421f1
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-SMDK2410.tgz) = 0e515ec91a4186de8cae2243fb079348e91acaab
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-SMDK2410.tgz) = 6a86551a8997ef91ea427022568a658a
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-SMDK2800.tgz) = 2cdcb47ec0f5173ef5b2dd3ba319dfe10dad37e9
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-SMDK2800.tgz) = 106b6c4241c7d4d51f77f2dba13c4977fbf09e8c3b2c4762d9b4de224d04c94a94bd192b89bb3e10a5a52c27141518a0429d09fa39023c9bb59569de7bcc897b
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-SMDK2800.tgz) = 84e4387b95d8731489c37d13bfd9eb68f323b19b
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-SMDK2800.tgz) = 00ed7a1a639ecc17b6f627bc5e7b2862
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-TISDP2420.tgz) = 11e7dfd918fa822c19633b2c0d7139fe3e5b4cbc
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-TISDP2420.tgz) = f5e0b25240b828e657de6818580bfb5dd47c3d1da978a44f60cbd0fdd86ac824f09922f4ab8ac8af8eb63c32ceb726ce2835498f649828fdae867fca4991a412
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-TISDP2420.tgz) = defb0cc263198d9ab3f0fc3e2702d6a2405de15e
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-TISDP2420.tgz) = 00fd1b9a1cc66dc3eff9fe90f85aca6b
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-TS7200.tgz) = 6e0547db5e16621220bb45150a22b25123022be1
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-TS7200.tgz) = e507e5475501cf9b96d1a7f88d5fd8783f089c756a36c6975bdfa0a33ecec441d1c15c06b8387ad5a0cda0a7ce2e18ab731365f8efef4fe8d615da1dd8f8cf52
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-TS7200.tgz) = 9b9bc14c01328090ec8b855f30690202a2e3e639
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-TS7200.tgz) = 1086e9efedf641d90a2fdef271d69944
SHA1 (NetBSD-6.0/evbarm/binary/sets/kern-TWINTAIL.tgz) = 6587a6dbfe1479c459d670942e9abed58c2077b8
SHA512 (NetBSD-6.0/evbarm/binary/sets/kern-TWINTAIL.tgz) = 4f2978dada20c19a2b2a35dcee406fdbced5caad7c46b6c733c2c24b2952202d872f666d2ac139baf5b0ff98092a5509594eb8fb70627a25a268fdefa36f1162
RMD160 (NetBSD-6.0/evbarm/binary/sets/kern-TWINTAIL.tgz) = 49bee3aea5f5b88b79344836c779d428fcd0d19f
MD5 (NetBSD-6.0/evbarm/binary/sets/kern-TWINTAIL.tgz) = c70d7c187e58dd8415522bce16bb12de
SHA1 (NetBSD-6.0/evbarm/binary/sets/modules.tgz) = f9a8894103100366dbd4343aecc6452a35c2fae0
SHA512 (NetBSD-6.0/evbarm/binary/sets/modules.tgz) = ac2fdf8cc9bff05a5f66749e4dd96a2f0ed4d143acd490e6f8b3826a9b9538918a8fec0045d40f6f418aab80372b23729410997fd81a05f9d9aad27c2dc3589a
RMD160 (NetBSD-6.0/evbarm/binary/sets/modules.tgz) = 35204da96f307f853b28c487b0b78c478cbe99c6
MD5 (NetBSD-6.0/evbarm/binary/sets/modules.tgz) = b6886a6197725ed21d53bae4a089fc1d
SHA1 (NetBSD-6.0/evbarm/binary/sets/tests.tgz) = 971ffc9e0521617dc678b9e9923371af2e9fca64
SHA512 (NetBSD-6.0/evbarm/binary/sets/tests.tgz) = 55013634b7d0e607d847ef69ea6a72b7f939830caf8e08dc55c78b04a27897490210377bf1c5b1216778d4ffca44145a0fabddd8ffc449acc93f40877a71daea
RMD160 (NetBSD-6.0/evbarm/binary/sets/tests.tgz) = c9be008f99e4d72d0260139d3ece0db37454d842
MD5 (NetBSD-6.0/evbarm/binary/sets/tests.tgz) = e1bbc8e25aabbda881eb290efc4b25e2
SHA1 (NetBSD-6.0/evbarm/binary/sets/text.tgz) = 08a253fdb7a0118ada516bcfd1293100728240c8
SHA512 (NetBSD-6.0/evbarm/binary/sets/text.tgz) = 41eb29b2f82b18d916d1fa2e858d45fff23f613120c53256c7f7f3dcf7e31ddede9f954a143d9d8c3ec95cd967f46861a1385382634028d61eb0883bebb294b2
RMD160 (NetBSD-6.0/evbarm/binary/sets/text.tgz) = b2cd2ecb39f44039e90a2465047b5510d0de2eb2
MD5 (NetBSD-6.0/evbarm/binary/sets/text.tgz) = b5a8566f96d102342467dde0a2ffeeb7
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.symbols.gz) = ea2b03a807cb44578f36e3f34d58a0435434f421
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.symbols.gz) = f8d97b757b0d0d82f5e4ff14afa6abd1d41ed35e3c97684ade36a0dbb277cfcb55a6f9ef656b1983507247b07f164ecb407d0973d57d1533b2a1e1364ccd51ce
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.symbols.gz) = 33c2edc444787b5477a459d6426d80ec2ac8c9c1
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.symbols.gz) = b06f95b94e9fc3ff03d7c02485d9226e
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.bin.gz) = 72839e1353d343d2ba5e449273aea745a980cb50
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.bin.gz) = 9f8019273ef9b3f46a70e6faa97a580e6ae1d0d820974de24033c55a729fff2d2780d5096964673f97e26b3af86835c5087fd0b09a345e15c29403bc1f467c3d
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.bin.gz) = 81b228fe339897de8d47c2cf5555c39eb891193a
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.bin.gz) = 662f1773c969157b42f70b72bd72c9f8
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.gz) = 257f98f33f8429d27486ea526b045f31cc4b9ab0
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.gz) = b2b2aff23b6be4c84ef07f1c3ac698f9554a13a54c8fd49ee12b0e175d09da2f116799260083e6a87a8b4e761a4a66ad8d3f02fb6fcec9f3f4b928271cea12a2
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.gz) = 0f5a2cdc5951921440a7147912e722a481c1360d
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.gz) = 117ff4be0dd35bb5cfaff3414c27a3f1
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.srec.gz) = 1fa80c74c4b2f44a9db4f1d5e0be62a7fd6f68e3
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.srec.gz) = 9d291a02fc808baaa9ffe3b259239d64758aa394b6fd782b4220ab47eec1106a33e9972d06ac07fbc1632923b12a9dee9574bfacf5a2dcc096490aedebaa7309
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.srec.gz) = cf28c82c673429f391bcfc37493f07489ab8e260
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-ADI_BRH_INSTALL.srec.gz) = 6a743321a49a841146309dd71a2b7782
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.symbols.gz) = 1d15f8fd20b400aac41fda396b615742593f14c2
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.symbols.gz) = 8a2ef21088d568841b740d0c316f513a98ac3f45f1c0d994dc199b4e2529836da832b8409368295b4b326de2f4defdef34f1829e3e4fb7217b39c858ab0eb09b
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.symbols.gz) = 2202821e57d5506b266456b70f18ff6a1506487e
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.symbols.gz) = 026ab51e016618bcc9f1f1bf363492ce
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.bin.gz) = feeb79b4d6334a0f6a87ca2c63d195e7039caae7
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.bin.gz) = f76735623ac3295690df2eabeefeb39c8cd3e5d85fd09833f9e58b8aa1bbec65f462e8a1edd683a75e5ac2188a50e3ef9f574b00dcbe6f7ffb0c02c4aaaafe26
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.bin.gz) = 5344ba62b8815bff63195fb12c5804194c5b74f7
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.bin.gz) = 1bf8ae796a35caa01e0332fb0c193e2c
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.gz) = d1d88ea80333147d4b73f6c20d5f0cf9af362acf
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.gz) = e525b3d43fe6ff0d7ffe83e546b80cdc1892140e3dad8cc409312a544f9ea111869e085fdfb736a1dde3d029d39e84443f06f0175d28a5b21a1a2967c82d904b
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.gz) = bae57fad021b76fc1a9e9c796b9b6df84b3a8814
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.gz) = eb56d7e3dbbfffd56cdd5b995690e1d8
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.srec.gz) = b0b4f43f1f4f8bcd5ca1a9a0d2877fdce3397998
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.srec.gz) = 50c8e9381dfff94d56c7330b217ab079fd3daaa77935dea557d321eca5e5ca59866e59b59b9ac498b0b4a71410dd776c8068f9ee2ee3ed646a39029ba1d6e9da
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.srec.gz) = 9c88faf5c7de521d9fa0be09a4d9b3b9206a2dcf
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-CP3100_INSTALL.srec.gz) = 558c67513391c3825047d26f08d48997
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.bin.gz) = 7e8c3f055b239755fecc8aa462e54b8b89a9e8c9
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.bin.gz) = 6977a4d96f1b1cb16ccfe4b15678b40b6627b01385fe6961487817f7ebf84d470277fad37c53ea7bb48e7c19e7b37dd8be3152a6e0274e6f8d6bee69bbb7feab
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.bin.gz) = 28fee55b727cc25187df3981d6fbe2bfa64b9e4c
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.bin.gz) = ecb5118258da0bdb24ec8ab9ac6faa0c
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.gz) = 0af690fd4f7fa383ddca7ee5f63bdc698d6ec5a3
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.gz) = 7be43506576a17872997db65441e436b51f1fe3bcc9c42eeea4fd19809f8d3f61d2cb9027ad6cc7e2a26fa3261482b9ff1b7e00878b5bc18ba4e7130919e8dc9
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.gz) = 46c65905cec0fc54bacb408d57af312598bf8502
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.gz) = 28ef106d2bffe72fe879c19ef7f25741
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.srec.gz) = f3646a6435283a79f0580692798694d31d8b672b
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.srec.gz) = 4b85ad5c44a4076715d43f51aa94e33ddf7f1e6645c7dbbd490ec8d8b305766f3e6a6facdc5c87a20be3865a827e531810e288eeab2184155d83cbc062a0e14a
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.srec.gz) = 7f87fafe296706dae3a75441b4f6fe26f52cace7
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.srec.gz) = b81ffbbeb626dde373c9c54e4cb15b03
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.symbols.gz) = 0ac81f8038480354e6fafbfba5da86ab333f85e7
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.symbols.gz) = 116b9be7c9e3135d1d1847e084a10e5bb2dc2a95b4810a2e268fee9e6e5b3a5f157b2f34b94a2e03256d959b2bb10d01760347398dc81bda5d6ce64aeac1c2a1
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.symbols.gz) = 4cd6b326352d9398035a2c7eb157c34604da6709
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.symbols.gz) = 938ba4475e1668e7fa02cd727e237339
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.symbols.gz) = 9634080097f581271b2e0aa1bc2f87b10c7c57f1
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.symbols.gz) = fb85279a670488e7e8b83c91188bf075363ab8278335c1d3a06411effe829cd8a47cfd269aed5e9a598414fbeb56165eb0ad27550f6f6838d2c9ae9847525e99
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.symbols.gz) = 81c648d753322e87068041c7bdc30dea925ec2ce
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GEMINI_INSTALL.symbols.gz) = e4b9a0b3e230fb2cb6a6e2ad3162b1f7
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.bin.gz) = 5c60fde2ead5db357827530b6b899e3236fbf814
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.bin.gz) = 953fda76c9d47d8af51e578049e98a5f169cf141a057ff1933bccec43ce89b419182f533fd105462af3c09a0a843e8e766f96d2ebcce4cf404166d80ab6b1f08
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.bin.gz) = 34e148a900e77f07aa63f0fbbf181bcbc79e2026
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.bin.gz) = 60aed2122f98e633e78e0f19a3724881
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.gz) = d94e8dcfc14fd6928b3d2c784cb38f0f55b009c3
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.gz) = 9ef172a0aabcffae7abe5312415da7d407975905484f69ffe73dda2389f0230a39ab2d03e9ca42005db662ca2f6060a4a6ed0db4b5cbf23676a35af38d23a416
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.gz) = bd13d39ee7366ba6ab2a1ee1b6094322a5b894fa
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.gz) = 9c16afcb3ce11d9923247d310c23875b
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.srec.gz) = 380390e6eba24b258a8eed603561f05e2a1cdfd6
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.srec.gz) = 519683b9ee12d3dcce26182bd8a46c27d9f6285debe96df2b88d46d0eff372deb2342a1823216da2dd0c28f90732858e7e8a4d9d6305d5e0b82b5345dce92112
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.srec.gz) = ad5c85bb6d641596e2b0ae7dbdb6f235b758546f
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.srec.gz) = e2612b861cab60aa64008ab686f570bd
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.symbols.gz) = 91f9c530da00c48d4b22b7fd98916ba4528f77a8
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.symbols.gz) = cefcbae785c5fb72ef5548c16a8ae083a5be19fd27f386c1eaa723c7cd296dbf4c4e774150bf3017a5ae52eecff0aef125f7ebce87f73fd930a019759e0ce7a8
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.symbols.gz) = e91be60bcee907ee6259c16e54ce1e3e79f7937b
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-GUMSTIX_INSTALL.symbols.gz) = 088c9eeabf75406c634a15511972829e
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.bin.gz) = 3fb32f0f8c981cf7d0012870d9c5019f06b844b9
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.bin.gz) = 7b7ecc024c36ffe8d57ffb7f53f25e9a3aaa203605ef955c3d94b47627bdf63cb0ac49a8ab230f5f7d5c07c4a56aca94787e861b381af4c84acf48c8b6040b4e
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.bin.gz) = f0faa1b8b1d48e6c736959211a1581221e92de24
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.bin.gz) = 66e3c14d2f927cd248e0b07882892fdf
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.gz) = 90ef61bc7a306fae7f41e20a6c615e37b7d51bb1
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.gz) = 98bcba4e6a5ea6260f42e44b78010bf5b02475a12c198c5898a6e4342bdbfb0207a5b7ee17ebb7cda0859057abe0e5442a41e7fed2c9b78b566e67c7897c21c1
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.gz) = a0fb23b82b04fcd0817a38d5b1b2da620bf23ccf
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.gz) = 546a0231797a48e08fcba615fbb5dc5c
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.srec.gz) = d234b22bf784c64b812dcf9afeba9f106c969e08
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.srec.gz) = f7f78ffeb87a42943c8271cc0bb7d9a3805569fe0cc75f62359ea827cd44ae48f425bdecae341f1a947a10f35df32ce12155105f333b97c765d48db2b1f4c0f7
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.srec.gz) = 218875a0a61876726f5790366ba2748acaf63996
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-INTEGRATOR_INSTALL.srec.gz) = 63e3b095ce7b3512ab0fcdc562cff952
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.bin.gz) = 9b24a0420c1b2b406fd2b6d8fc7f25a3dca984ca
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.bin.gz) = 5a46d91f646c354c5bf39ae4c15cf45b2ea205328f859a3006bfef815a40b71109b748e6efe91b85ce692663db36c2b6f08e6a43b4ee843c5d1dbb2eb095a7ac
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.bin.gz) = 4a6cb6eb7612c34d1b3c68ab7cb56fe32e45b7f4
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.bin.gz) = 86823cc4ac5a09992501014bf88c877b
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.gz) = a4c5e90d5a9fc6c4b127fd2ba479b27228b64f98
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.gz) = 98f9d8a9bb4c6b29be7002dc4aba375b89a7e98d23fe77634c26efcf4bfec3bec46917718f4c245a9ad8a480db0205224bf0946962ad5b748e04f2b170645344
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.gz) = c9756d9b19051c3f34a748eeca4fc283784f205c
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.gz) = edc01ffccbf9a6db1d1b40878927b492
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.srec.gz) = 1d2d7369ea182d3cbe36b6cf0929bd26e48e7799
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.srec.gz) = 2164f0f36fbdf41d2fb37ca8a19beffcfdbcfee8a5629350e28a94a105e9b9707fa44b99cf122c102e073084d09905a75a69a8da2f263882e89e9192d66c232b
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.srec.gz) = 91147de829c8e594800eb713e55330fb2160c062
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.srec.gz) = 1ac108b4534ab6529420b5e6eb4596ee
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.symbols.gz) = a594571814d5fd54f060a324f961c8728207414c
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.symbols.gz) = 6a55dbba559e4675123270036ef2b05d877463c0101628e50d6f84d12035784a76e010e469840b46cf56291e5d27fa5569457cf83fe7d5a5f128ef6777d4b0fa
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.symbols.gz) = 54553929ec55ca2d3e1a34163f79019ce8d21f68
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80310_INSTALL.symbols.gz) = 796f55a92c0f4a29a920251d51a90d9d
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.bin.gz) = 8667913b1247f23e940f945fc1d3c6d7b42e3bc1
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.bin.gz) = 30df01772cc961320c0600225caad50145a9f84b9fa5689ed61c59b136c4489e4f6ce3f1075e53c46335f955ed2dd94e662ba2674c56a0171f33e3ace70af2c8
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.bin.gz) = b6c98b6c38bfb10e36bf119c49749cd5c6be20b0
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.bin.gz) = 9a3b14a1e64592301beaa6c7581e5f3f
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.gz) = e619f80715bb394b1fe67cc05f45cf0e076fc3d5
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.gz) = b131081790eb4e2a82ac2cbd9726b8239f8ec91e25567606411bbcb4c9cb0676f8f15d1daca89623e29d6045fb4e6b6edc40ec7bff4d0646cd06313c8a827672
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.gz) = f791654aa8cf1ee8d9d71aef03ac0f904345465f
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.gz) = 42b4d81e7d76bf98705e97927f1a6fed
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.srec.gz) = 418c6e4fdae62a9dffc0cc52300b3308d4c39726
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.srec.gz) = c6025b5ca7555b4870026cf2a9b23a000bf3f3bdac4a20347593f431062ccfa8b43c29dd1caf17f95787ef685066509e409164c302fcbb40cc18307fec372442
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.srec.gz) = fcaa8176ac524a456a5e75dfcffadbeafc133187
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.srec.gz) = 0674c1b44f421bd847dddac9e87e1221
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.symbols.gz) = 579e6ecfc351e91ba0bcf36918bd3fb781b7c1e7
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.symbols.gz) = 34074f57b8eeca79deb3f2a070ea824be2fddae7e8b5a2d8bedab2b4dd4796f2eb1baab06cdf95b53da4b43cbd8e8fa873dc0986325282dea9902ffe9fe521da
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.symbols.gz) = 139e713201e03fa17e31803834d898d1ef636bb6
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-IQ80321_INSTALL.symbols.gz) = d08f0319e5ccb9d142f20f81a1a4d0f9
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.bin.gz) = eb490e7ec36f92c871934f2cd7c1fcb2339ad2a7
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.bin.gz) = 7be26fc8e01142f873d0e2ca89a414bb4d2dd332dabfec1f7cd126ba2eaf0d803abde4f62c627b583fe2a57b93bd1669fc669666185ce1d5cdbeb47b88690ca6
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.bin.gz) = a93d35c0251e92ac7562cc6138419fefb6fca84b
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.bin.gz) = dc54e455b762156bdfd6a0fb2d14fe2f
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.gz) = 2567dce09ac1764f78ec074e4f5644a87ac83c8c
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.gz) = 6beb220906314c4ac93a058a4994104ec2dccefcca4e8c2a0be6d3399536b4765e49294b275dacfe20e39c7cccd81b88c9521b2b273835001cc6b131237b20d2
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.gz) = bc87bf5b540e6847689016b2301ad48aa637a3eb
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.gz) = c96abcd45f88d52e5826c9419e895e49
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.srec.gz) = 82575c4f3f6344ff9fcd848f30364fd1b65eed82
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.srec.gz) = f1c6e41724db95eefe005e2a4f3defa81b7d8319521ad2897fb976cc8fd31af9f9399bb0df03e8b910d72baf25384112efad9699a111c6a0ebc2e8d79dc63f72
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.srec.gz) = 82c356d5a3916d82b941cfca334f55586b30805b
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.srec.gz) = 16ff2a11bc97e2bddfa4ecda023be897
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.symbols.gz) = a8cf19b83ab6b0aa48e6486a2ef5717c14025933
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.symbols.gz) = 3c9919130d4470ff778d77119b4f17132002de6229a1d0f4f2024e1bbe8e68de7cfe93a4da18835c111dad1de50bbc4e5e0e50a89e71757acc2da888afcf944a
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.symbols.gz) = 53339f2e886137c722353c7e4641fb79fb4160c9
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-MINI2440_INSTALL.symbols.gz) = 2125d332a50f2382dc9ea2fa1c76fff6
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.bin.gz) = 7f7533adb7f744376eda30bed953e0c24925a7ac
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.bin.gz) = 6df01180c26dea9f40e427414375f872d65be05f1541d887fedf8d98f22e3f98354ca15aa058c1cba0d0e4d30974b37787855c1a9ee36f7b702ad42c31ef69ab
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.bin.gz) = 1c6192b2841123a6a7b34f46530a49cf731fe8b0
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.bin.gz) = 7080e515fcdbe3d4d976da0bd0db1356
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.gz) = 009c609c69eb87be8129cb8cca61732493771053
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.gz) = 1aef3bb8e1f66280e2478e207618ef125b02fee796f1d31bc9c8ab5e28f132bd5aa5c6e78cf1dd2e63a72cfe814264db394c74a2c1f04ae139dbd3d592bca8ce
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.gz) = 6b7f00ce3de221960905acde44c93679d12318b0
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.gz) = cb78419cf4b8862517c4cdb83f09edcb
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.srec.gz) = 5405155f29eab526e4b489e156a3a6a84b694c04
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.srec.gz) = e4ddb650bed5716c2a6070a88104ead0119e3e41a6ad61d0a1ab944966fd5e29d3c17b83061940c81788583d833b3b90e2b3fca30a2291bb7e99be7acfc52eb4
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.srec.gz) = d28c777e9f79b74d39d9cebeb55582bf6554bdf4
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.srec.gz) = e33527c137440f91dc481372e55b5153
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.symbols.gz) = 0adc4d2b9af7fed43f2cb2a6d461b0b67dca64ff
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.symbols.gz) = fac69e001c5383808c41901f250e53267cee4df0738cf2fda17675057f397d33fd2be817b4235ba4ff47c3ad4c667388fe94e935372702729e38a39d95932bf1
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.symbols.gz) = 08cb50f087ee2c1edcb6bb8b0a10fd9436ee5291
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TEAMASA_NPWR_INSTALL.symbols.gz) = 303fa88bab1f84ce1e5e7392b63d335f
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.bin.gz) = 6d96ba7627a743040710559958052fe5ad516e07
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.bin.gz) = c8980e5a5fcbf6d16a94b273537d0b7e684d46621d46cb52b1c2672e67c864bc4651a09fed96356e441d209f57c5eba10696d9d82a3aba7bca24781ef6e6d570
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.bin.gz) = 8faf196aecee5eab9fe5a256daf1bdeee5f485e0
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.bin.gz) = fa021fd649cc68d99ed50414c562937e
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.gz) = 7084ab7948465cd7fb832a191991166d460bcbe9
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.gz) = a6e2dd918a3965573e3417e24984ff5d4c546a9bade8d7b68a930e695a76243aa4acb89eaede1ad8ea07280a836fbdc3200b7cade29486e1a1783734ad3164aa
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.gz) = eb9e46f17f5a0d0d296fcf3983fc7cac086d139d
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.gz) = ad30409880e74a4c9aa9be2f0cd67ba4
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.srec.gz) = 5dce945d53df7aa098712e966a08d8c4a11d5615
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.srec.gz) = 13270d86a94b74829fdd691cd97b8afbebf397fd68d9e8ecc8dcce611b3eca79666e5da303d75ad0af56f762b65d3d487ef1317685538b6962ff5b482e57c954
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.srec.gz) = dbb6b76ac38227dc2b5e0dcf35dbbc187a0f9cec
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.srec.gz) = 573eddbddcc0e213758ee25fd7f3d6e0
SHA1 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.symbols.gz) = 7a5bcaaa2847b816c0568f28647ee07a18650c38
SHA512 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.symbols.gz) = e7fa9c30cef40ab421ed36421f9c92bac2fb21c88077d4dd176a55b3b82cb94ff1e912b35d6504e5d37ca737378da3eae0f64e7721a023e7af596a037f0970a0
RMD160 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.symbols.gz) = dbe42d8b278149bae03d553e4d8a2676544b53ba
MD5 (NetBSD-6.0/evbarm/installation/instkernel/netbsd-TS7200_INSTALL.symbols.gz) = 28d969c46d89d77b04fdc532621a9c5b
SHA1 (NetBSD-6.0/evbarm/installation/bootmini2440) = 322711938832b81edf3a30de085fe84c118fffdd
SHA512 (NetBSD-6.0/evbarm/installation/bootmini2440) = f985790660b66ab4843a2e938e67bb7b1856e6acfee877e00bcadaf945cae7f85dd5f5ea2557f331d142c2f79ff129ac505a9f8ac276650ed49ae86b202cdd1f
RMD160 (NetBSD-6.0/evbarm/installation/bootmini2440) = 6549e1c02eec982d3bfbd42317bc0c49bd7a3fcc
MD5 (NetBSD-6.0/evbarm/installation/bootmini2440) = ab71c605b7165325bc2efffbf8366e01
SHA1 (NetBSD-6.0/evbarm/INSTALL.html) = 364e208c39e0aa0c1a873702629363049749555b
SHA512 (NetBSD-6.0/evbarm/INSTALL.html) = 0d23a02d74624eceee1b8908f49ca677ab4c260725798614faf67cfcf396aa0862aaf5296ab68315b3a1b110a998cd9fc08f36630aa3202ffb18db915e6bc5cf
RMD160 (NetBSD-6.0/evbarm/INSTALL.html) = 5e46ad234a6311ea55f157da0978fd0ca4543ee1
MD5 (NetBSD-6.0/evbarm/INSTALL.html) = 4ddc104e853a1b0a9ff34bdeffdb849a
SHA1 (NetBSD-6.0/evbarm/INSTALL.more) = 49005be72eacdfd5e58240ea40c272362f1794ca
SHA512 (NetBSD-6.0/evbarm/INSTALL.more) = 29b0240cd5e8b957fd6f0cb26858f856f2ca1e5bc664d7617138519d38788f442bd0e2045f506eb49b6fb07200e3251694679c7e9079e152e022e4c36d69339c
RMD160 (NetBSD-6.0/evbarm/INSTALL.more) = b2227c9abf8c42edad582429b388b46292b2b642
MD5 (NetBSD-6.0/evbarm/INSTALL.more) = 8f7edc27634f260c5216ce2ac1c7ab20
SHA1 (NetBSD-6.0/evbarm/INSTALL.ps) = be244573aa70f6ea8bd6afef50d3129e95ecb232
SHA512 (NetBSD-6.0/evbarm/INSTALL.ps) = b4dcdf490b1e2fdfb72a21fbd00922eef14da81665f56e69d5fce4458222a3108c27fc793a1507e9f6f3b3555c236899f06863282b3a0dcf061ddd81451e0f82
RMD160 (NetBSD-6.0/evbarm/INSTALL.ps) = 38340ef793c940f555c637fb63ed603e2813b6b0
MD5 (NetBSD-6.0/evbarm/INSTALL.ps) = 0702cfb1c18290c2f6b3990ec2b9b838
SHA1 (NetBSD-6.0/evbarm/INSTALL.txt) = 84d0cf406199e9e160201946ef76c5544f859094
SHA512 (NetBSD-6.0/evbarm/INSTALL.txt) = 779245d5eac9f6aad487d0ef8c09ad8ed1ccd1ab6b315d5436d681422a1ca23376304d9f5f36e0010d38a26947c2d67b214f7fe06207b52ec4ca1d1abf996280
RMD160 (NetBSD-6.0/evbarm/INSTALL.txt) = f21886bd1f37f1557a890ce294fc842a4a9322ba
MD5 (NetBSD-6.0/evbarm/INSTALL.txt) = 4e8f849ebf04f5af7feb401bd8b828ac
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-MALTA32.gz) = 837f5c347644ad20116892b91a170bd9a177cf25
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-MALTA32.gz) = 83ef6ec1fea23841a6a03d5f91d0c00f8dae56d78276b04dc56f3be1121950c8b21cdc3b4f580929f85e0b7dca9f5635cbc248cd9bdf018bd019837953165796
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-MALTA32.gz) = da9f38ccb4f1a09ce9567d148115538eedddfd64
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-MALTA32.gz) = 94e33f6853b4a1d9253b01310a9d31e8
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-MALTA64.gz) = 1194bf50c6a2a1cccfd7017ec100e18840cab60a
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-MALTA64.gz) = 05939d55299eb8c98f89c0cced1f090fc47568c81be3c4aaf32325ce4a087b44b1c6945bb8b9a12384ae588e9dfb1ad0ca1c701e2f0603a672bf3ea1d2bad1d0
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-MALTA64.gz) = ceecace87ff5c72a8bf0551d03e7f24297c8210e
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-MALTA64.gz) = f70db6b8696d6db6538471a0389aba6a
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-XLSATX32.gz) = 50baddbb54cecc95e5aacae73100b35b1a50808a
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-XLSATX32.gz) = 937b7632d22c4e22bdacdad4bb473b1c4537cccb35f97d1bb08b80d64c6228a410f59310ba336d19e62e0bf38f3ccb9c5e1c27aed8d769ef66657de06070b273
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-XLSATX32.gz) = 97d67324ca36d259e51a80f89b3ff8c32aeb0936
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-XLSATX32.gz) = 2c17d4f86050689342b031332bae427b
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-XLSATX64.gz) = 1e1c1a9cd9c7bedace03b3c25a736014d7053adf
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-XLSATX64.gz) = dd0689962b451e442a48a35c35f426d4428aad93a22e6e2d33dd0033125eeb8e763ed31ee13410a6f7b7d337a46c351a736605adf49e301e09df14096fffbf41
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-XLSATX64.gz) = c94021afffc7e21001cd3f045063b725fcbbdb96
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/kernel/netbsd-XLSATX64.gz) = 7ed50990347095eced85286bac3894ad
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-MALTA32.tgz) = 8aef48b2afd3a5c2daaf9298b1377e5fb48a4775
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-MALTA32.tgz) = 769777c570e41cef669cadd1caca734ca55f71382656b3d229bf32811e22fcac10c3258032e79ad190d9e34b4a4168269b0a44352209f9321d3b3d18e1d920ef
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-MALTA32.tgz) = 27fd132ecb4aef492c8194a598641b0d57df788f
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-MALTA32.tgz) = 547ed14fd7607348ac12d50d6e142625
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/base.tgz) = 321d0a48e96415413f6c76fe94cfae0b763c0ab4
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/base.tgz) = e3306d01022a478928ebf33894603aa4788983b58b73beb1cdd1592cc5d545bb3eefcc07ef381f7be24099bf5100914d65f5575c9ccc43279332972d5e84c01c
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/base.tgz) = 822520b0721dbc6ff7402acf65a7078df4ee495b
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/base.tgz) = 838db1b3837f00c46b9b7f8e2ceddee5
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/comp.tgz) = 9e1125bb9734db8e861c085c3f4baeefed5bc5ca
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/comp.tgz) = 405ffdb54d1ad4f0cb56b07ade8ef1cf26235526e143f62622870e9733d9f860405519b355129bfad7ebe659c56358c73dcc941bf0554a89ef210a70e69c2855
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/comp.tgz) = 13b1ab9f477f0d22afd1dc11d7a0ce04ed87e247
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/comp.tgz) = 880a15da9d452aba11a9c3e4b8c32c44
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/etc.tgz) = 793af95b8423250427980e55e1b40e7d2d946dd7
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/etc.tgz) = 9c9dafc30b85ced01a296fa4464bed9c8bbed77b359307a637e236646880d93aa3b17c857423f26c286732686b00855131a359295fbb22c4f4d4271aaf21fd23
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/etc.tgz) = dc21febaa97b3f196ddf788541e93081919ac05d
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/etc.tgz) = affff84911016dc75d8dcc556ed1cd90
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/games.tgz) = 6a9922b70c13cccb7e2c3b2d2b4f63f4d6ba8b83
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/games.tgz) = e17192d20cebcb6583f76e3badf382c2734bdccb596b5a7c7104c28bd6d96419a59a642c4eac1e246d821d8cdf211e8b00aeb7043a439edfa6cf9e5554441007
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/games.tgz) = f93c4992616fc210ae24c58b64f8607ca6f3d1af
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/games.tgz) = 2186988f1072dc9be8a8e1ea77bc1906
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-MALTA64.tgz) = 6cff22a0f2e961617cd4eaf3bb571a7b7f4845a0
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-MALTA64.tgz) = 05b431e02e4e858f79103bc1950c4da7fc87d476d46479a8f0c2a9850b96391d8a55cedc0fa3b47da4857d7f91a9acacc08369e2121a7d47af3bda5712ad333f
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-MALTA64.tgz) = e1048547f5851837f476010b3c5866f527542465
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-MALTA64.tgz) = 3ccf1d8db71de9c182c81d99689aab8d
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-XLSATX32.tgz) = 4400d1601558ff59a7d90028bd91e00c4b67e230
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-XLSATX32.tgz) = a16e8b5d34e07c97fdda6b09e41a67d48fcf56b4276e9e52c2a2d8ce1eac5d21abb542afb67991f7b2bf87261c8134f8e522d59375ccd251f4e853d3ed877b71
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-XLSATX32.tgz) = a7ea06c9ce95d00548c7dc9a3403f873d3f42749
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-XLSATX32.tgz) = a6c542f79813324f5ddaf80a9a12a531
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-XLSATX64.tgz) = 47cc48be2b1edf22d48fb3d45c2ebcf1e6b84655
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-XLSATX64.tgz) = ce9378b66f327c84e004d075e0033e195c1e649acc7f668d154b5608fa71070cfed74b1eedea404cead8c92a9f9f332fb5f2407a14d80175d4f2e7f3115962d7
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-XLSATX64.tgz) = 64602c29c9cad9767bdc771464f766d4c4bc7a14
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/kern-XLSATX64.tgz) = f55ccc413a2546e5db4c66b38095dfb2
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/modules.tgz) = 150d99ebadd4c626bf48d9dd772584df18d0f161
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/modules.tgz) = eb3e15f4dd50d6eb59ffec96245ca440abdc2427f7cd14f6405ce979f9ee13b5d40ed30d0fc544cd92e5c8dd49b2cd033cafa44f32baef8c434734665c7b35d2
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/modules.tgz) = 74a564f4270f8b8df9c0f207512ccbf02cd4436e
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/modules.tgz) = 6f2c6d66540b3994a5400f1b3a7c0e06
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/tests.tgz) = 607a8a23fb7447a1bd6363feb446cdfc7110ac7f
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/tests.tgz) = babcea43b781105dea81af3797bce6f77bbf2766a28104c45aa7f43952d0a0a7e3c981927d4b2745ed46b290ff4db7d43b5a2620b82d7185a55e06156604432f
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/tests.tgz) = 90dc0855b2b85584cc08f07d55645bd98f04b40b
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/tests.tgz) = c0dc4879f5bf2e13462fe11f1afb9755
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/text.tgz) = a38aef75824fe9c86579b8a320ec645e1079224d
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/text.tgz) = 39f42cbd084867ba177e3f3c7eda0ae19040eac2ee4dca991caa172211ba499081f720f16511b967fe6e4529f78e446cf9cf8ab89b595bca41aff3e7017b2028
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/text.tgz) = af58fb15683143196193ee0ad443a9fc94b6f8b7
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/text.tgz) = 8716d1b4d2cb02c015e20e87650021ed
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xbase.tgz) = f22a560c757bc506a118ce9803f68e148fbcf5f8
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xbase.tgz) = 5ee7c1db10aaf6a67fe6e9026057266340e901a2a4042d126862bb3ebc1b26dc824a3d47febaf2c38738b10e563feb903c1020ef98dd34a34ed6ab1811e1dd0a
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xbase.tgz) = 4dd1d30eed29fad5ad00d08f8655850ba2966beb
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xbase.tgz) = 83fe206984459147b084ce31f5712ac5
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xcomp.tgz) = 6a62de4e61ca7fb51120a2d949f0c6a76728c795
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xcomp.tgz) = 1a4d6159f2afb79ea54d7b50515d33585ac70a41da0dce13790d9acc1bc83636eedf83284aa76c4b9be2d51e25000fcb000842d4491edc1f0032e6e68f591359
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xcomp.tgz) = b5d253b8ed963d2aa4542c177199f8bf56a7413f
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xcomp.tgz) = 9ab19bfd18fe8afea9ec8b987eb3f745
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xetc.tgz) = b3ca1c7e85d0832091fc391717c914889f745a13
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xetc.tgz) = e386fa04d66afe2b4817c6e77d6ae351e7a10265099395285e15ff5fb8c41d169de511b3bdc71d9b8fd144a0153262b068b6b837b846310e255bfea1a06ac5e6
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xetc.tgz) = d7cd1c13d547ec0fe4ad59ee92be9499cbeb53ed
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xetc.tgz) = 8165fb3d8f4f335f32f391d6f6ba414c
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xfont.tgz) = c3b40e7a439d60c27022fc1cea5e58fff68f2da3
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xfont.tgz) = 5a7e564dc9cc41d3704722a461ffe7ada8011510e3db08c2a209afc462a69986261ec02d41e792f1e1a305300f4d23216cb07c231c461df5edb2fb54916d2438
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xfont.tgz) = 03066eeaa22fac20cb69be43939f870a313519d2
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xfont.tgz) = 46e16e5288c250f4f2394bf709df039b
SHA1 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xserver.tgz) = 13d61af721933a9484ea722fcf5a151900b07d55
SHA512 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xserver.tgz) = 1372c13478ca7da2c5ba2b3d56f2f09f2bd764f2c345a9e4b8c2fa81fd101d557d53b16637b7644676431cc57b481d22d7f702c1a24d7c6456c04b75422f9485
RMD160 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xserver.tgz) = 3cc8603545553cf13c0e41525d4f71df2f04a243
MD5 (NetBSD-6.0/evbmips-mips64eb/binary/sets/xserver.tgz) = 599284f9e42a94928802db5b36d6c8db
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.srec.gz) = f94948277528ab8356fbcb1f447ea3f88da0dbff
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.srec.gz) = 37025250657daad79c15d79ade2f1e815b551bb9541d852e0e0b208b6964b2e7bcff4eab477ff9dc8c62b23a061681c6076684f41d50ffac2b9ca1c9b85eb657
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.srec.gz) = e0f5ead94d76d0131e9b6351a1b32066c0e37004
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.srec.gz) = f07bb28b2742e51e8a130a1633813342
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.gz) = ce135463449caf04fe5915edf2689e53f7d34450
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.gz) = e30a6963cdee29fc634232bad314c1093df462a1fc2751908ff1f6118165b1f4cbc7fa3fb780c7483b26dbed2339f562d75618ed355388ca4a7ea8da182f5718
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.gz) = 726d74693693f22e261de851efc11f489af83d5e
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.gz) = cf0519b0dc9ab365ee268ee0cf667cd3
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.symbols.gz) = 7d61c5ea450b2348bbf829f17801d1444142030a
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.symbols.gz) = be8fadc2f971b5ceae719fe76f121a1d394826c0509d0052eb681ec3398e8944a8866c45ed213bc1a07ccc0306ede8ec3c28c1415a26bfc92721fe8a90ec8c86
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.symbols.gz) = 8be481d57e200549bddd24d27e640fc192758aa8
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA32.symbols.gz) = 23dc546c19890b5bfe2981a4be93cf61
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.gz) = 774d5ac21ea9f3b0d356a7de041c89afedd85cb4
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.gz) = 228797b395c498e531f359c59d8a79f336138923e95881c1506c9b1a772865b6a7df96be8a05e62a8f3a3b791e208173e35bc6ac1ad29d8c62b388620d3bb828
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.gz) = d84a8c78f5f81c897066771fabb70dd447d423ab
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.gz) = 9392b5ff6a21a0f043f4ba6e9ff14a00
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.srec.gz) = 028bf1fb8a8e7f1733eecb45b7016b387660558c
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.srec.gz) = c7d3178f2dc591e1d39e8e04aaae780cdd38c190d850d0b71a0c69eca7a8dc29b8adb300e106a236e9fbd7ea9e85dc070bc5ae6baf6bc742aff4dbafca91c00e
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.srec.gz) = b4f76a19a3119f6b35830db195d0db3b40b962d7
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.srec.gz) = 6f1c87fa36192f6cce9fe900576de781
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.symbols.gz) = 05ebd0d64d73f452fbc31f9a7945fc1503bfa23d
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.symbols.gz) = 839eafde2ed98b6be776ad75e591c23a295aae726a2d384fa1c2f9e52333f1d5ba14c21473ad5d7c51098608d04b055cc163e743485556ac30c75a5a913f0422
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.symbols.gz) = 6f1dbafcaf7ca5df8c47a61b5f6d8805a209d7eb
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_MALTA64.symbols.gz) = cebaa3e1af8c79ccb42a92f25140c653
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.gz) = 039e0a1c46bd768a34bd4522d2523018429e6987
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.gz) = 4914b5675a4b3858c3feee1eed653a75f0708b84c12d80a82715502fc9e2761eef8d23735ccdff6c596a93fa3e01ee69dcdc39cd44b5e3876722134ee90ef31b
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.gz) = c0cb5df60249fca57ac322b2cb8fcc6f90645698
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.gz) = ee03b9d28419f05275ee663b8b9d0ac9
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.srec.gz) = e831083d34e0200ed793aee4a890a80f518e1ab1
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.srec.gz) = 7cd35c5f5ed1ee14dbabfa9c7d89a6ed592e37b66676c1a523422d24830a28507afea99d6408dbe436d59233703257684a9cc765370c98dc23de78f3931b50c3
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.srec.gz) = e6de2239f84cf7cacf8ae6b1bb62326483a1c62e
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.srec.gz) = d7de63125741840ff89a6dff9fda38c8
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.symbols.gz) = 5aa1511398191686462abbd5e20ab6e6b56fb325
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.symbols.gz) = bd4c5f4b969c6c5bbb5d424c21430a3c1cbdc1f6d40a60842402ffe2d812393e0556ef6b7fade839a117a0ec601359127fd2957d35bf7468217c9adcaca53e70
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.symbols.gz) = 6a2fd527f96c49423f58f6f8363e0582e2b894e9
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX32.symbols.gz) = d0654f9677e051fc34fed0d81a27e528
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.gz) = 39ea7d6f681981f11fbdd9e63473e4dd69cc2f9c
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.gz) = 41378a129484896d4598dcfd29b1f65cc462c5746d4793e973b7912ad10cf3f1e487701f4d3267ba7615bb6fbe9390a356f1d89146e8f90d7bc8aa57ae575dec
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.gz) = cd77d4e08b77ede06fdcacf53cac8360b115b11d
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.gz) = d736cdad40ba2ae3beba484dd5a5ce05
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.srec.gz) = a2fd46154978cde95b379f110407c52ae0c688e1
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.srec.gz) = c778ee1c88dc9aed20d7113233f43660258d1f7616e611d8e8df413b9db1e73039fe422a9ddc8c02039baea5bf5434f660fed57148f13ee641e339f9277f2c51
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.srec.gz) = e98d08b85f6aae19176ad570f061f4ac14c42629
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.srec.gz) = c2de85181edf54b18fbe495d65faf32a
SHA1 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.symbols.gz) = 2808091cbd1d32e9f38e7c38e7f2a0aa8c66e617
SHA512 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.symbols.gz) = 7c4157a809dec6f292fafd1fa21744932e8b0add59d2f3ea62c4536d31e14c6b8a24633354cb4918dd38fc88695d373d356dda7e93dfb77f83c6a6fcf6e70b54
RMD160 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.symbols.gz) = a31fb2125afada861428f3b416218cef7a641caf
MD5 (NetBSD-6.0/evbmips-mips64eb/installation/netbsd-INSTALL_XLSATX64.symbols.gz) = d727b2cd38e49bed46bba3a1d33f9b97
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-GDIUM64.gz) = 938493a6f165b0a66e87d0ced332116b3e5fa1e7
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-GDIUM64.gz) = 05906bce297394f48bddff8ea2a92f527ca5a673c1ddad6ee776eaeeb5298c562e01b533288add57bf39239c9cb02e6e8f3569c4430b658bf99b0bac61245aaf
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-GDIUM64.gz) = a2d803c0597759fcaebb669d3f4ccb7d008d2e9b
MD5 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-GDIUM64.gz) = e1f26f59ce4ff562a5c8e6f8bd511a0b
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-LOONGSON.gz) = b717b1950f7e2ee2745b74e32c9f6cfc7d7ca15a
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-LOONGSON.gz) = 47916c7dc402df96876307fe31e21de43b5dcd6a57425d2454ba44593c03b0852b6dc1c5822880590a14a91717e7478314e6ac0a099004596f4d256ba3427e4b
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-LOONGSON.gz) = a1c0e2c2ea412fa4a49fce8cc1a366d9d373a5d0
MD5 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-LOONGSON.gz) = 7da6b1ffa558b1b211fe3702c5837e54
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-MALTA32.gz) = 7c79ae79c6672c73ac03b88dab24103a01a86658
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-MALTA32.gz) = de76f96e25bb798dfc28c0e186676833392c47a47e9eaa9c0883f9ac43f89946addd61ebfd3c91a9556eee4bb218f483053513190242abe472ddc79131224081
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-MALTA32.gz) = 325db86c0e99e01418f70575d75c3285ad586bad
MD5 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-MALTA32.gz) = 67b5fcff9d18ad156d0c028ab16ba081
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-MALTA64.gz) = 8c7904ebdd030a94458f13ac7c11df832929cc86
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-MALTA64.gz) = a51547232434584769c0a8ea108bff1425e5e96f1989224b3420863ebeded4a3cb272b6873e3fe860aef3658982bcde844d86fcb46d79ce647b6969e6758c978
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-MALTA64.gz) = 656bc08ab196ab9e4638804a5ca840327a8a31a4
MD5 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-MALTA64.gz) = 8cad946d5e2a395f26d8f18eba137ada
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-P5064-64.gz) = a228a6982bb8c5cf468860e09fd9b17b4307f356
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-P5064-64.gz) = db4187479c550a71c23f9bde6cc8dc50d2eb758293b293c7444a786ab30becbae9e150c12ef3fd55a42df1b88efdb937a7fffc48457c20c5441b082e0c0e2745
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-P5064-64.gz) = 8d9bc3f9d1ea60e3322fac87a3c9ccd43a54055d
MD5 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-P5064-64.gz) = c318ae2623b38a290cce71191c6c6afd
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-XLSATX32.gz) = 2ef72f56c819a2bc0fd4d48e463161ebef9551f2
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-XLSATX32.gz) = 4b80ebf478d77d91205ab2d89d2459317c4b61bbae18646c2c201fb4ba1ee2978f5be3a87ad33c83d9b139d3ffa077a4bcb3e315c16fb68f116b802dd2217637
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-XLSATX32.gz) = b5a20d970f5ad8c177c6244b1e527ea04c705cce
MD5 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-XLSATX32.gz) = 34ceee3acf4435335a43c594dceff98e
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-XLSATX64.gz) = 5bf585188ba90ab0231e99247967eacfff520cc8
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-XLSATX64.gz) = 20e39f09067084fccb2587a05a3820e50816fb3f097e8e89cb07091c937fe97aa27593948fa3466f9a5d6835a8bd1bfe31268a2414c6c8694fb9fac156e68f98
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-XLSATX64.gz) = 96ebcf3117c2a1f60b50ba1de3852898083393d4
MD5 (NetBSD-6.0/evbmips-mips64el/binary/kernel/netbsd-XLSATX64.gz) = 335908a4efd612fbc51487db37af3eb0
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-GDIUM64.tgz) = a09a36dad31ab7a766bcddea3fb08807d93819bd
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-GDIUM64.tgz) = e3876e7f4164e117ada834e6769fd3357d144fe36779a197573a8ea735821150fbc6c4b6c2b9ea917b14b23a795e4db7e069cb8e121ac3fe9ae5e0296400d065
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-GDIUM64.tgz) = 945ea08a5fdad4d8ac8504212712b40aa01d1443
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-GDIUM64.tgz) = 2fcaffaee00d469cb7f2756fdcfd2fa6
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/base.tgz) = e7550ddcf5edc7c3c27f53b7e6b4636f744108c7
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/base.tgz) = 0096eb300e8a71093cbe3e84830ff048af73e85cbbad1986390b8838c50ab6bd63cdc201062514aa63529c2043e1a2b2428cb2b50fa2bcb9dfd4ed11e42febdc
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/base.tgz) = a5509e8f5a4e5aadcf057710489fa4105dc4dcbc
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/base.tgz) = 76f45332e5fae1c999dba2da97f92834
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/comp.tgz) = 2475e5bc61cb7084e06f111d6e24e3ef4527ab0e
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/comp.tgz) = 5ca83f2b42cd3145d3fa5c88e72dec3509f52f63bd0e5ac26f752a18065473ea298bc0e88cb0b954a06d3d7eba5a9afdee3087feb06079e9c76cd5a0ae3a6759
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/comp.tgz) = 8bf71a3552e3e9388c85d5426530e58d2ff15f9f
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/comp.tgz) = e8dcb0d4b7b5acbf8f4603be69f4e9b1
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/etc.tgz) = 106b0102ae77486072a61db71016443e4a63af9e
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/etc.tgz) = 386948f2aa4dd140bf17c9aa0a8bd0b41096603497dbb9792454fa3772724312a3a2ddd3e69a7b1a710e6339f799b5174b16f8f91ef123ebde548eb47aee4e90
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/etc.tgz) = c704f1a292af8fbfe37a9a61654f8e8873b6cf49
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/etc.tgz) = ea16b2ab565706b8038b459e0519936e
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/games.tgz) = fbb547d701d95fa7c9fedb8ce1616fa73dc0a962
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/games.tgz) = 9b0c4338c3227f974305624f0ba8756836f156f3d3f9bbb1ec3188ca9ec2de344d5ff3f8b9969180e2c871cc0b09b12b9e6311e33ca548d59f4596a04224f14e
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/games.tgz) = ae8ec9dca5729ee4203ff86e10ebb16379ccf5f5
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/games.tgz) = 18b0244267ae2ea671632562ebf493c2
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-LOONGSON.tgz) = 8da37bef5b86301e6e5741969aaab713752c2866
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-LOONGSON.tgz) = be868f49e960bdee048cc1a717d17563c081afdf2a258007312f306f14470159943877ef29d0c3c783b32682074c299aeb2b48a7623bdc4af2adaf82ee9b52e1
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-LOONGSON.tgz) = 10f65602b7dab9305df83c26e670221efe1b452c
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-LOONGSON.tgz) = caffb25555770c6474b23e6cd028e82e
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-MALTA32.tgz) = 7e5d386cee62920ecd932af45ff4a854234edc8a
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-MALTA32.tgz) = f5e9bb4886260c3eb0b18e29999047b1f39692be5cd5cbeb58638604efae408f167cbccb81d2fc6c601f20515955b48035113cc2cb2e6d30088cd9750fa64465
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-MALTA32.tgz) = e0a36f2148ed78c4eda1c26e5ef0045ff6b97e22
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-MALTA32.tgz) = 3ccd62f53f80b0eabb9baa2a29a691ac
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-MALTA64.tgz) = 225382c44befe15c46b05f2f052718bd80dd9a7d
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-MALTA64.tgz) = 1c1a13c0b73098d14468cf12d7744f79696773698adf94e622afd95e436560fc4934d3b02390187647a9d97effbed9162542149fbe06628cf38aff197faaafd6
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-MALTA64.tgz) = e5ae37af03461f04dc6550141e4c2301dce51ff2
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-MALTA64.tgz) = 672129c4f8f0315c9b374ac5651f82cc
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-P5064-64.tgz) = a41b86f2235180b688882c8eb2959edcb48c81ca
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-P5064-64.tgz) = 9db6c584144a249006a671894e010fa0674e62e6abc60d43441462d1f1478bf4b1f9c6d9b46f14a1e9ffc3f7641bc206f5ed95c90b405e57f3f40e54d74aec92
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-P5064-64.tgz) = a7f59b498f286aeffe614b4ae43461d8a859f758
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-P5064-64.tgz) = 5dc76b6c8b5b49b757dd4a5dfa370420
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-XLSATX32.tgz) = 9cdf4993097b04929dbad4b76c5bbf0cd6796c7d
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-XLSATX32.tgz) = 171bcfbb92a1c1ca96dc14707604998eb4c5760179e9781da9b0ad18ad9695ea44d49c5cdd9ac9250a7a60e08b336daefa9a6efc2f172c9a70d10417e4643b0d
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-XLSATX32.tgz) = a1a517153943866b07fb5850644d1079721fd131
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-XLSATX32.tgz) = f68253bb53367876e463884c2a8eaabf
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-XLSATX64.tgz) = 73dcbbb4ffbd263a8ab68a5144b16cda35e14272
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-XLSATX64.tgz) = bee28f3a052cad0a8624729675f28d42847b7545234f98e141c180e851a3f2052236606c512f1a7515c7ba115417eb8e8a9ceba2224a1c5915ea44f25cd99abb
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-XLSATX64.tgz) = ba17bd15f5e3b131ed77687817b58f8a1730232c
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/kern-XLSATX64.tgz) = 0a2ea3efdd4858019a64cf67994cbc36
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/modules.tgz) = 4ce7dd5ecba275042042fc283bf93453fc57b2fa
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/modules.tgz) = b90e1c87a5ad7981bf51e9fd33b2614d6d092c10c6d67a242ebd814276dd116ac660d9801ad02a5b9b09ce58e3c2aca802322c97ea73c9e270f88d1162e7ae24
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/modules.tgz) = 73d91a2e36f2ec78fc3aa1f2c2af0d2a2d9174a8
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/modules.tgz) = ead882dfd50f9421b51a5fc5deb0151e
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/tests.tgz) = 441100d5b3275b38c206eeb86a9153084f37e613
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/tests.tgz) = e3f33662a577ad2ec975b432424b5d2c2b3b2b742d5e2ce37043cb1820ba225b185638c5c88a34fe05af9fa0083db918074cbb8f091354c35e8f2fd121f93dea
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/tests.tgz) = 64e119a7b1ed5eec0151ebe0ac8c6d2f362e8dc0
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/tests.tgz) = 0d6eedfa2f9aae4c8f0055ca1a358c66
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/text.tgz) = b7f32924a9d2c54824fb19b1bb3158b2996a16d1
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/text.tgz) = 00b6a2831c439c9a5bb07023fecbd7ac99285b2f08a2eb2ba00ace828eaa4cb5fdbd12c339170b9235713d9936683837214389ebfeb52c9bc4be07bcc0df0e15
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/text.tgz) = 6f72bbbef2e15bc63b99ae7419a6a68f3a82b7e5
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/text.tgz) = ec982ff4007c1082127af777e978bc85
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/xbase.tgz) = 2e87de8a350d32c3529222c0195e8b89a03f1192
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/xbase.tgz) = 65ba278254dcef89f1cc464487921995413a5400cf0a3388eb987416486f96f510c91569b6c3145d687e05eedc93be79bf54374c67fb33ffc57eb2b89902d096
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/xbase.tgz) = 7acd24fa398777d99977da6d6080c89ab8a75ef3
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/xbase.tgz) = 5535460ce0de58bbbd1aa49741edc463
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/xcomp.tgz) = 70a07fcea399d1032660b185c2d9ac6486da8fbc
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/xcomp.tgz) = a803322c59fef449066d1d2e42beec42587e40f6134f5c62d475440a01fbf93a91aa9bfae96b89eb6394e2f20d925fb85dc23754bf8600feb40cc98a9beb844e
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/xcomp.tgz) = 12a13cd5fa3a63ddc24e4f4bfc6d35ce99dad31d
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/xcomp.tgz) = 0b455b3722b1429e0a134317b8dbdea2
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/xetc.tgz) = a95c9eb9995be3639417ed3098f7ef36baa0d729
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/xetc.tgz) = be4544652b01b9161ba2f27461ed723378712d0b34058ba5a16de3cc366a5b551f0e8e7b740404eebbee7ae656e7745281b7d329068bb1aedf8f757ed06e9f98
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/xetc.tgz) = b7e1bac15f5d4dda393d5e07e2b26d3c3877a01d
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/xetc.tgz) = ee2771391cc229fb9bea94f455e841e4
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/xfont.tgz) = b985c741b487ae0e65ca36e318c558c147d1eca5
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/xfont.tgz) = 3da2aa9b1cb7906f515d0c02d67a8aa994a0022189dab11428a28040f57f5ae3cfb758cc7676837aca3479bfcc1e1551afe592417b59ad79b80a6392d6f34929
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/xfont.tgz) = bccdaaa5821219a03bd67a57320311b265a19e94
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/xfont.tgz) = 60eaa14a01f682577ab5b1774fefd7b8
SHA1 (NetBSD-6.0/evbmips-mips64el/binary/sets/xserver.tgz) = 30134e060bc2ad1df81dfb637ec9d7e35bbc4932
SHA512 (NetBSD-6.0/evbmips-mips64el/binary/sets/xserver.tgz) = 94e7986f96ec8b41653db1b4c05d0237c614678b43f1a52ab3b40c5627ffea2c7e91ae3e154391b964dc36edbd60aaed767427bd50eae5306beb0fead97c0346
RMD160 (NetBSD-6.0/evbmips-mips64el/binary/sets/xserver.tgz) = 2423334e389526cac76e76fa0fa38fd671e0d3c2
MD5 (NetBSD-6.0/evbmips-mips64el/binary/sets/xserver.tgz) = 508453785f0150d7115610f4ab1f0ae9
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.srec.gz) = 3d7cbad15200cb0e18ed529e8d65768503bd0a96
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.srec.gz) = c549dc552ae2b10bd3d8b6b716d6a107e5ef6988a49b2ae1292b40608d217cafa8bbb77974c3e8fe9291820bd534b0713c1136a09c173c282384f2805cabad1c
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.srec.gz) = eb53ea3f18593ad0a8d23ab92b8d78e590709cae
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.srec.gz) = 6c0fbfaa9cafb5799820bf14e057a5fe
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.gz) = 5f370ee3290556f0df9346c1c279b4cd239815bd
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.gz) = 8a7770a9dca3bd6bb59525c7f5340c9347d7c315e6f54c719d5f46312e0ff25d1fd6ba0e31d881ee680d904534a71f7258c49c16e706a46cb6ff0ada537630fc
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.gz) = ed1b7878ae50846b27db943ff3f82d0be2e4bd05
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.gz) = d935eea2ed1da5225fd8af96482ecc08
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.symbols.gz) = 0f65b3f9550c07786d711dfa98c426f3601e0f01
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.symbols.gz) = b37c854b5a20cd45da6e1fb4132a0fc8bf1234ff08294168249a3a38e8463a9c52fc9863af80787c10ae62e1bb17be087cab605478f5fb29dd615f8168ea9180
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.symbols.gz) = 1365f99730e4ab317d1d35e73f2eb5592680ece6
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_LOONGSON.symbols.gz) = af18fc0829c787d47e6cdc7e6146413f
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.gz) = 24b04245cb3bb8808c2643f6efa74677e18425c9
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.gz) = 4d76662f4aaa9fbdb3bc3e807eb716804f4ac070e8b0e6def5e9c4c816ed456894bad40a3b51ff8f68a21b05bee08de02e7fb1ee1ece8e625884dbb2bccd9109
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.gz) = cdff1dcb0aecc09f926915055769081acab42201
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.gz) = 06cca864f41f90c5f0bd6d95a57ecc5c
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.srec.gz) = 5a432e451fb73617b8377b4f479a6eaf611bac29
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.srec.gz) = 3f92a16819d8452f7d105e19d803985832f89e1d6c09b067d6ffe4522c7e924707d9411535ccf838cdec3b7fd1fa17cd7cfbb6e2d3bd369690dcd9094f5e10f3
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.srec.gz) = 71a1e5b455df4458e1bb4ec9d45b4043bebc5788
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.srec.gz) = 98534235eaeb7626a485458779a0e6a8
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.symbols.gz) = 2f3ed468ae1a3a4082bfd5412bc11ee8d8cb90f9
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.symbols.gz) = fc3b4ac13b3c951037e30132d9178dd79635cb78adea2b97046ef6d8a8486bf9b12e43d26897f32f7db967fddd048c177acbb5585b573bc2b9c80374b59c133d
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.symbols.gz) = 5e863f2e95a8623056dc32ea5ba0265876f102df
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA32.symbols.gz) = e84090937754b1d91357b759679a2ae4
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.gz) = 3e9f23c9e3f15de38a7e0908f0253d1f23e49a43
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.gz) = 57cf422b28f5442c105e23fae81119e993b543e3eb2c344ab5334e8edb1018c54f3b39245d1bc4091b8de4e28fe78e357dfef31bfa51de174342ba881ef9cf45
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.gz) = 9950fb85a0100463810fee9d0ab79d4297e94401
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.gz) = 571361834195d0048cfb4c2bb13faf10
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.srec.gz) = 03a6a6428275c9f01f514031f5c288739a081850
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.srec.gz) = 4165164e3830c5dd897107c64cd6dcbb27edc31e32bf6cf7305fe74024c467a81e703ed1be94071c62cf467eab89ecd66b3b72876fc1d8fece4792dbb8384255
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.srec.gz) = 68586dd9c3e823a26a5439907f088772a5939b88
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.srec.gz) = 37ec696958709fadbfe7e08fcac3bb57
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.symbols.gz) = bdc6c366da273e056df2ea2bd9d92813b61f5b36
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.symbols.gz) = 574c80f478e5f87eae6602a8e8b6d52df19061199f2e78d05bc6e662daca5f9b6aef26dd4b543d24e557b78aa2d5f0a5e80aa9cdfca5fbcdfc8a4da7a84b95d0
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.symbols.gz) = 4f8ba01f1ba53c8b8230a974d7c8aa45a9667915
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_MALTA64.symbols.gz) = 5e726ecc87a4406b10e5c11cbf745a7e
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.gz) = a21c899aa2389958474b94a057e7356da840589f
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.gz) = d092b4d17356ed34a9a7a79838b76a487c35d99b26671f1ddfadb85db810bda74b3db64f7734e30114429cd8fad87f13c04ca462c6d3bb56dc1f78855d7b9def
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.gz) = ae46da84fdb8a17e60c91ef88049989eed666a96
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.gz) = 1ec7762cd142ac3eacfad76ebd7f018d
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.srec.gz) = b76ec124d80dbda9a425bd14737ff88105d4c053
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.srec.gz) = eb446974d31976ddac9313a0a45ffc084dea1bfff75d18e6544f16cab3333dcaf6e62cff59b0cf88070985c1ab7a4c0285cdd0b54eb75d73f538d45a80be9256
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.srec.gz) = 1eae8b53f9ad020c3a39d67502e74888059125f1
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.srec.gz) = 3c2a22221dab7153f7eb41daec3a4de3
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.symbols.gz) = 0be09076adaccfe13864e71c9f1ed8085a515401
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.symbols.gz) = 7bd72d9d584872ca346e2ee6eeebc8e8fd06a7e655acf621ad1b849b440ff66a33bf71650738a074732845c2bde24acb08e1b457e08abde3234d2972da370773
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.symbols.gz) = cfc81eef00d46e0c1deea04c9364f64e87d637db
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX32.symbols.gz) = 11e04b25355c8c60ba97c142c21ba8f3
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.gz) = a68f5f892312208526869378614a6c38f0ce7ac1
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.gz) = 7b298f7fe80f77e2283c407af4736c643e3b0d0710d43bbe809dda43660344caa948ebb07727ec80cf19065eb1a30ef24c68d0249d2cbc9598833c14f36a04fb
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.gz) = 431a3a8bf7ea9aa86b7c82075a2c6272ea9f4938
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.gz) = d1cfbf1aea636ec7f9b36760eea32d08
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.srec.gz) = e766bb842b698937ab7252c0bf6f325f1ec2c7b6
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.srec.gz) = 832e277c2366f3379f104fe68aa927b3c7358c40df62f393419ae9f2fe30a989a532f4d4cac8dd821ea07c5707246329990e64ded734cad1c4e755a59cb59402
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.srec.gz) = 44af060baa21f3c09fa3e283cde72a7171f6e6f2
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.srec.gz) = 058b4dc450b1e2e5fc14136e4ccf3743
SHA1 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.symbols.gz) = 5130e72edc460e5bc620f9101c6b260db4417ef3
SHA512 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.symbols.gz) = 53360173909b40e40ac018fa0f5211a74970d3429c85c182f59a6ee4ce782f54d707f2d6205d9bb5f4fd5b100b074babec12bbaf178f74b867cfba5964aa2ec2
RMD160 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.symbols.gz) = db94d2cc9bfc2eb2fd44e87d47be4b273aa7533e
MD5 (NetBSD-6.0/evbmips-mips64el/installation/netbsd-INSTALL_XLSATX64.symbols.gz) = 37e4e9464364847ad2327d6352019047
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-aumac0-ALCHEMY.gz) = c8b43df25bfc2e9171d6ddaa73b436f63a9c7085
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-aumac0-ALCHEMY.gz) = 73d80744cb6154342f5c3d36f8849c2a9686c1318fea913b840eb97e3177f1b225f4cb9178a9e15b4cadae35579c2258b38d180d1aedfb0e9ffa00543b1cd8e6
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-aumac0-ALCHEMY.gz) = c8e8dd4c41e4e88fd7bc0c34420df32a00a941b6
MD5 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-aumac0-ALCHEMY.gz) = 53e030127285e17b67640e9fd98af1a6
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-ALCHEMY.gz) = 9f738766737db8cb0f2404c4952bb75cbf407669
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-ALCHEMY.gz) = 1298e944b4d8495905e6649476cb8e1bf49719abf2d2fb3ae55f5e1416a2af3234310047926fc261264bb12b8c40d09159a42cd978297cd6ee4197249a34561e
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-ALCHEMY.gz) = 53f5c1646d7811b99d2a6bade48712647802264e
MD5 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-ALCHEMY.gz) = 7315ee909933e50ada51ab5652ad0d2f
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DB120.gz) = ff83b1a41048bcef21a723c356d5e762d1ae697f
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DB120.gz) = 400430bfab26962e450f73640ca169fdf4f748ff34c63dac1edaab893f58a3494a81ecefa807bd80cdf846213d004dcedeeb0b57e785436488c26a66c06ed685
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DB120.gz) = aa36f1124c77777325cf30badccc02f2353e9065
MD5 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DB120.gz) = 8e6dfcd4eaa37591d2648cfbead5b650
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DBAU1500.gz) = 5e4747aafb2621b40c0adb250613225090d8f219
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DBAU1500.gz) = a24d3164c5f9eac497b338e99fd04881e9fb3ea1e25844c4d9927383f306b9d0fe16e26f61e33ca7fcc47be9e6b1a04b2cc6b397f51187e49c5683714ee4880f
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DBAU1500.gz) = c90b133fd12b1327cbf8cd0229df26d3b71774dd
MD5 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DBAU1500.gz) = b691ce9b6e0e31b0945fb7ff041e336d
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DBAU1550.gz) = 62ac90542124d3b65582b7c5667e545fbd7b1145
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DBAU1550.gz) = 29622cc0d471a80ab095fda13f2d06ac32d4757d2a2ce4c4c7a54b490b6445ee86c702007de54ac9fe81515566d7eca3805f2d1c4d842a49d8c752df4d09099c
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DBAU1550.gz) = b624724146136eaf312d9be3cead0eb8f55445bc
MD5 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-DBAU1550.gz) = 0d30e656a06c9c5ab86c9097e08b2ca8
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-MALTA.gz) = 83ac7456a68f04283e651f472160025d9af2b256
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-MALTA.gz) = 1c36d9ef350ae385688a4300af725d1b2ccc94ca3f6eded04d54ca2dcfceb40d5ef8db88610428051598e7a1504340ab86bfda25d96bfc3b454aa74f0f35ece7
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-MALTA.gz) = 59acc8c3c660bf2256966b58cce5df78cc6a4da3
MD5 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-MALTA.gz) = a64c723ecc8ff566fc859d155bf288a5
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-RB433UAH.gz) = 8af25f51ed2e87eaa234c4bf43a0c8adefa3c267
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-RB433UAH.gz) = ecfff13379b5e6a657670f69aec3ce471a593505e0f6284f80c0b4f3c4c903bc9bb7d9cf0c9385e0b0f5288cc141bf5c21e56250ae17107f53ca38152fc59a13
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-RB433UAH.gz) = 38d8fcad12171bb0209eaef94e137483d6d3f5e5
MD5 (NetBSD-6.0/evbmips-mipseb/binary/kernel/netbsd-RB433UAH.gz) = da92ce6119755ebd89fe27c24f452aea
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/games.tgz) = f56d3e5181634f6a3d7908dcb002ebcb1c5aaff0
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/games.tgz) = 0cff50ea13e34fe338edf70da6a28af061580feb4cb0773828adefa631ebd471a96568a3b30ea0f13f3bb116856a1b6aad8a5a52443346f52e4907f4caabc5f1
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/games.tgz) = 1538910578dbb8eee2931200fac0ccda44758e5f
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/games.tgz) = 2bd5bd0ce72df5cf92767b8c3f7cc208
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/xcomp.tgz) = c8f5ff5ea9f9bbe231f92d8b7f3bfcbd2897069f
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/xcomp.tgz) = c7590075195019078b68ec5f87e36479ae2e7b7749c523df4ef21c9d881c343cda9f5c2bba897dd313329a76fe5c18e920d92827878d98ce08a7dc776352cd02
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/xcomp.tgz) = bc60e267cb6e11b4a7305f8b7074084e179e0ff5
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/xcomp.tgz) = 8e58ee51667def3bd015f79769c0d932
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/xfont.tgz) = c551bd13179b9508c2e3a7e19cec323e8ba21844
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/xfont.tgz) = 764527f2f3f4686759fcae3fd093f4e00f34e1249994126221a6f9afe73d4cf8ddfbe8b8311798da2ed38c52466a44bb26ea08ad5aab1c08eebcc9019b4223f7
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/xfont.tgz) = b3a365bedf8b4a20f3920d0e33b9a42fce288953
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/xfont.tgz) = adac8939da213a0195cc1d351f979199
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-ALCHEMY.tgz) = 0adc8f742c596217127185e8f9e4db43896d9299
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-ALCHEMY.tgz) = 35865aa4e39ecbf9c8f4e48ac85b804b571b1488de729cbbc5663deec75bdbf90a229ce99473975ea46715d502442497a596fa35ed89e5d3602c3b1221db6676
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-ALCHEMY.tgz) = 8a4fea3817d615408966528475869669efc69701
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-ALCHEMY.tgz) = 94e9d6a99aa8ad762dfd71d5c2c5a410
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/base.tgz) = 152709728189dfceee8db96c9b793229d80514ca
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/base.tgz) = fab7d3d262ebc5bff7c6edd891b967dc10f4c274b33da21a874ef874ccd1f1917afef30cccc27efce5869296e9bcec95584fd6109a48abd94b958b231a5e2fbe
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/base.tgz) = e52f8333537d524fd1580d0b07c9c63414fe680b
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/base.tgz) = b2dae13bf91a2929948ee84429598111
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/comp.tgz) = b04647e2ea51827ceff965350bff9290cb4e1da7
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/comp.tgz) = 15b6bda11ff6d85a0e1696a2fbd174d1b060e606f52e7447d6c25a48de41f03e99e8601f098c6604608e3a056096c8b17ea6865802db353a018860a3c563d387
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/comp.tgz) = 0ca46bc47c48ba36c8e56b9131abd9d61d5c7ae6
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/comp.tgz) = 5917382e90e63ec66e687546760a8532
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/etc.tgz) = e55b4d17ca25e42c4a0652a40106d64eda9f8096
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/etc.tgz) = e6286e75f5ec6591a53162c332be61b976e4e0386d429aa675cce295ed8fdb90d62287b244eecd0e0341235a637bbd593f50164dc8f589ac4f969991d2a3f0f9
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/etc.tgz) = b45e18e794d732b11dc8eb3ef733c2dfdbd0155a
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/etc.tgz) = 2a4d24ad39ee26ad3bf2a6c9d86381a2
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DBAU1500.tgz) = d6c5b3f1074c70e13e1388d68a1932cc11b6a3ff
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DBAU1500.tgz) = 6ed5b668ff52a75aeadbf11b1096822b97f01af47d8b6b171343feab7e10258cc02b04e151f37995bc85cf6a7c562c4404181c8484a3aa3d593ba9a2c934cd04
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DBAU1500.tgz) = 3f5750e9b576ed3e05624052d5f69103eb32c61d
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DBAU1500.tgz) = a4513305f6d3e69cc52c47047d89de2a
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DB120.tgz) = 2f08fca9d13f265f4d560740a55bebf8b115dc37
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DB120.tgz) = a4a2c6e1711da7562520ee189131e499aa65ec114f713cccd7ccb94f55d9e845a253607ec2219284df1422ba5492a0f24004b8088601dc807aaa2a925e14bcab
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DB120.tgz) = 018c1be32ee04724b86fa2ea173adbe460f63e50
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DB120.tgz) = d89deab24227d2179a680db2c9477357
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DBAU1550.tgz) = efd95e1f98f8ebc28f7621e5696b7ac148091aa4
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DBAU1550.tgz) = c67276ab70bdfef9c8a9b9ace43ae950a686aa630f0bd11fb06c533cc463faf41ad1bb89d5a6f2cd6d4d04914ada697da20ec19924720aeeb4d5d1282393cad6
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DBAU1550.tgz) = d91cfd72640e117cf952ea3978af6ad106fbed88
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-DBAU1550.tgz) = 1e914a5acbbc55fdab0a70d059171d99
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-MALTA.tgz) = a5da116a64f3b9e840cd568391f7f755a1ca3f14
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-MALTA.tgz) = aef284a501e6b2cd16293a5689c5c798dc6bfdc0879fe2cf212b43a4f9d33e9f7dc863f7020c6eda08c8958e713412e5b531e0af799c4695d327c4b66206030e
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-MALTA.tgz) = ed3afb30019e8ac95a9a7348c9099827de561edc
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-MALTA.tgz) = c85bd4c68c3458f9581c45a240f5e1ab
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-RB433UAH.tgz) = 64f38dca64142a0a8426e216cb8edd0884551603
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-RB433UAH.tgz) = b0904e89e80ee6b16f9a97a36974ac5e8a4a7ca8dda9e5f4b27de218fd824d9515ad446574337139aab25fb290c6a93a177f4ddbae7b7a92c8daf5586c87f727
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-RB433UAH.tgz) = b5e20de5b71f3f0880f714b49e0c39a54d19736e
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/kern-RB433UAH.tgz) = 687ac6220a984373b976c5e73cf23240
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/modules.tgz) = f46c0e5db2b77d898161bf6e3ebac55b05413047
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/modules.tgz) = 04dfd0379e10923c47b740c5ceed84f69beecf5a72f581885e5c4dddcf9d241499578e6b7a435a119ef8be74909878620fdb700e6638ae119da59c635883aae7
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/modules.tgz) = 1b674a6cc66d8c1362d66acc69f99f1e0fd70bd9
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/modules.tgz) = a3500877590cea3b09db38130581d10d
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/tests.tgz) = c0a06d6671f0141559bfc80c076fe6f0b2d87003
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/tests.tgz) = 114f7f7e299614296fe2b272a172b9c1d77da42dfe09064c6d9fe9a6450322a38a3c1fc403348aa536d2039884899200591b59d653a3a7d89d22ea22b1750b29
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/tests.tgz) = 4833a7fb848401385159dd129da45d96a9a7b180
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/tests.tgz) = fb54298f8aa3423bcf7bc43b3fea488e
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/text.tgz) = 91921b0be07535e75e97288e49ee2bc0b3c9949b
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/text.tgz) = 336ae450c79b99cf2f2c64a9b378125904c2b7f2d2b6de8eb0f8cd5a2b735cc7a55fe69b599daed3e3e8ef9c51bc6044d994e6dcda85adba659204b92724b814
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/text.tgz) = 3e1e8f989f25749485c0ce9471dedc141cb73726
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/text.tgz) = ff542bf96a0884faa5df5e30d2ac2ff5
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/xbase.tgz) = 6c4272f0ef26a4f32eb393bb479f62b2b7142bc5
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/xbase.tgz) = 568a69f2077f011b98792eb1f71a8b7bf11251f42ca8a082559fd3d836a6afcb7eb05e77331b1cd8b996731c4fddd4ddc8abc6b82cc3c4d9460827e52dde424b
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/xbase.tgz) = 4bcb45e19d2da3ccb065977c5cbdf640144afb8f
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/xbase.tgz) = f7cc64d9cdf50d1c85923eb835bc6dcd
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/xetc.tgz) = 19444f8d1799d782bc72d2684588b1479ee5976b
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/xetc.tgz) = 0075efc99fa5c63bfce8806d5bc083e3e68bb6e9dc885b382353dff0bf55b85a27608318deeeb504833cc5abcedd5bacb45b943bc3beba42e202d08ed368962d
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/xetc.tgz) = 32b296247a2f343a49473e7d54987c6d87229a38
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/xetc.tgz) = 7d63fd2d8fdbb123792d61e4fa3ece2f
SHA1 (NetBSD-6.0/evbmips-mipseb/binary/sets/xserver.tgz) = 2f38506d78bb7be30d5cd8719f3c137dc28b7f92
SHA512 (NetBSD-6.0/evbmips-mipseb/binary/sets/xserver.tgz) = 3364dfa7db892eb0c1c92ff50c1e321687300a2cb572da25e55ff501d958201fc3f2d916681778de0879ea20cb31ccf30766038863e0b97e4ca52365b5dcbf51
RMD160 (NetBSD-6.0/evbmips-mipseb/binary/sets/xserver.tgz) = 9a26ea480f7ff2ce07a648e0004dcabcadb8432e
MD5 (NetBSD-6.0/evbmips-mipseb/binary/sets/xserver.tgz) = 00b808e0f27ff7924b22f2c74b1f7121
SHA1 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.srec.gz) = a05b722b902910be1bda3a582d3ebc850048ced1
SHA512 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.srec.gz) = fcde1c3163e2973f213350595dd38d202524e4f125f14ff48b68f9d569f10788c8796123e051ae72e64b786452f60ac7079ae59390333b63252a8991ee1a1786
RMD160 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.srec.gz) = 2d4a3c604fec21934e0bf2c4e987c8978aa8bce3
MD5 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.srec.gz) = ecd58d8ad5bf459ed7022e72580b9942
SHA1 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.gz) = fde4c37b9543ae150854af55c5c34243e1316c05
SHA512 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.gz) = b72c9f6516e3ff03e64c6598e153893d6beb5c64997488571bf3aa4547f2f251b78721fb4a23892470c8b352b0ba3a6c95f2e0c8e450918533ed1b9cbebf90ef
RMD160 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.gz) = c37277dcb3eff20bf03c9319d4a6766d1f869225
MD5 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.gz) = d6f55d866675b4e51178d7c7daeb7b6c
SHA1 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.symbols.gz) = baff18cdf467ce42a54eb77e9774c524c9ba347d
SHA512 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.symbols.gz) = 530df115006eef8bf3f3ea78c1bc1b8a37e90dc460fc8c3cfbeb70422bcb0a9ef52fd116b5b9ae3a9eec83fe60cd3a32fa0933ac0589e2c68c831978e7e988b7
RMD160 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.symbols.gz) = e4ed20740fecd9b75d635d99567d3b67f5ae5cf6
MD5 (NetBSD-6.0/evbmips-mipseb/installation/netbsd-INSTALL_MALTA.symbols.gz) = 6c54f00839d4980e95252520ce7a3814
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-aumac0-ALCHEMY.gz) = 134fb77bafabb2f5cdc53713cedd145c6695b5c2
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-aumac0-ALCHEMY.gz) = f21c2b6185b6ddfcca9f1873239ca3855165898d537f4ceb6fec7818ae157ac95b143f559c77be261d4fa3c427e0579d5ecdcdd6047944d0faa94681b8375c16
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-aumac0-ALCHEMY.gz) = 003133387cb76ab3a8afcb59683cd6d117e4dd0a
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-aumac0-ALCHEMY.gz) = c99637f4bab91983011482c5cd8f84e7
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-ALCHEMY.gz) = 87f20bbd047d0c7b793c4ad98ecb01128565e2c6
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-ALCHEMY.gz) = 25a0f40f69c8fa908265a4e757cb64c074f47442b3f9e4d6051e7024042d2685b4e01bf3b0081d603bd633523b8f28c3570b93f5886a4021986b03131175854c
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-ALCHEMY.gz) = 4f0ded0dec635ab37d6e43e5198a3eebb9d5432f
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-ALCHEMY.gz) = 886795f8573d66ada3b5bf8f8443cbc4
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-CPMBR1400.gz) = f90b68d7d1fb37b69896657f4060f0dc1c196951
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-CPMBR1400.gz) = a639799a0431ab22cc468f7168358416ff6095a6015cfd783406b24a6e62f1dc38a978c2c4875cef999654c3f0fa2f8cdadaed5d0283fa05bc56db9ff89d4c68
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-CPMBR1400.gz) = 5265e5e99e360048b07d997d3bf0f73b806f905c
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-CPMBR1400.gz) = 5447c227f2147295b74c3cd9a62efa60
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-DBAU1500.gz) = c298c37feb22bee4f641ede6c29b82f55a15fb43
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-DBAU1500.gz) = 9ac3727e8f9777042fe7aab1613e910da72d07fd5b8277349f2b538f620f6ac34cfe5972fa72a91275cf99ce0e98e062bd559b169eab6cf9700d0065262d2d98
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-DBAU1500.gz) = 45fbbb985b1bd77331e018f10df8d95d17c12af1
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-DBAU1500.gz) = 3391b8ea4799c856e260069f7a4a0761
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-DBAU1550.gz) = 2a32d5bda59467bf957d1dba8f4c716c553436bc
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-DBAU1550.gz) = f5b499dd62317cda4483cc70f65e56d7a8a6be0095dfa1a3836158b6615066237ecc13b3ca66a11545a47a938ec2f2c54c27c8613d16f931ba0b3a8bdbabd2dd
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-DBAU1550.gz) = 52319f5ea1ba91f0e3d762df4580873ae8015f3f
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-DBAU1550.gz) = e55c5185cd7d35ddf3a8aaca93be4598
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-GDIUM.gz) = cd1ba889150b543a42b1abc72bed2da50a4705ee
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-GDIUM.gz) = ba267fc4d44531b95167b3321c4b6f2885902a8a19332c647ea325ba2ab37e7d48a1753d2ebe3991dde1ea286455b1062e56242b0d7804b9db51bd906fc6fa46
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-GDIUM.gz) = c4fc7d36e9e1b9d22f04c85db96894267551ce4a
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-GDIUM.gz) = 53fbc4ff1eabc437ff5546fa12df92df
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-MALTA.gz) = 18cb97fb9b5fa551a4230460d66174bec28a7017
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-MALTA.gz) = 5887e65233675d5584c5f8b9d14ab70b0368ad9c9d6e085470f63f3a37a82d5adfab7306866fa55303cb73668ab8b836082609da77eb5ebb5fb7b05fcf36ad03
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-MALTA.gz) = 326fb5286d584548044d264ce8650cf9b13e3901
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-MALTA.gz) = 06f4d46d9d802dac0954390934246fa3
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-MTX-1.gz) = 95b45facdc9fa0b69709e7e5505e3849eb3c318d
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-MTX-1.gz) = abf3ee530bf7a515fb2f4d93ff50d5ead425232171eaf33ecd9fdc56fb0ee37e976aa918fb3a153b0c8d8b7b51cf288bca1acf7a9a61402d149818e83ff70296
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-MTX-1.gz) = eeb4eac3e382e41631e8f3f8806e2c3f10355a4e
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-MTX-1.gz) = 36da2e3383b17e5bf06844c414dc239a
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-OMSAL400.gz) = 44841ceb6f1af159ff6056ef89884da1f3c7cb3a
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-OMSAL400.gz) = 7ed282421374238ff02c9ca3e413165be0d0b16e1011cdab9d4af9f9b3b868b05d5832a8f24ea396222035ebf46ea3b02b2eb49cf378f93f26d807192a5b6458
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-OMSAL400.gz) = 7b039a86fd581c56153a113366efd278414f9921
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-OMSAL400.gz) = a8bcd078458637f91753faae2c99c8ac
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P4032.gz) = e14fd9558a2cf7405398aec0f3df97f846ccbbdd
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P4032.gz) = 4640b39a386e0223ff255ddb9796820a249e34e25eb2f550cae3ca36d2a3af7eeb2f0133d084323b964b8aa3db93db462db2bd3debdd78388e71eedcce37a48d
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P4032.gz) = 845c8595b29846b3ad87272048e0d5ea18413c2a
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P4032.gz) = c2e071d630599cb393bc13e4db60a187
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P5064.gz) = 11fecb91f2c90740271dcc51a7fe5f78cfd35281
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P5064.gz) = 076c38f8684254a744cb14d592b5f0404d5a7944cbd486b2b6da4c4a1057057029f5f1949f007ad5255d97466ca5ea1b5af047566823526b7e7804280b0206f9
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P5064.gz) = c2e78b4d2da205e8af4b8bffe97e71b78649917c
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P5064.gz) = ae9f3bd2236f924b501fc6d0ae0b42c0
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P6032.gz) = cd8af99e05c16d79b574f176d0229bc363c3f094
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P6032.gz) = f1375dd13b2950afed3a0b3c9217a0630da47ce52fe467140217a2b29706e68354018d2e5c4b0ab7b16a9af119a66b6430cdca1298275dfd1d9b22eb65cd9ea3
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P6032.gz) = 0af7124d320813134c90c509027396730e0bb758
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-P6032.gz) = 277e59364f2ed7160337ec3d20060a9b
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-reth0-CPMBR1400.gz) = 55565c142f55fac5076d21a61b0a5ac8cad002ee
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-reth0-CPMBR1400.gz) = 8db0765cc0d2affa6cf5490d766f577f202c89d6694fb8e57f81c076257242ad47439fa9e3d46c738d2e1dc1bc306f951fff54211f48c4bcf3c27ce672db67bd
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-reth0-CPMBR1400.gz) = 015a00aebbe3f9ff27bbc38e50863794829c34a5
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-reth0-CPMBR1400.gz) = 994571e2665c04bfc5ed50709b09d74e
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-sd0a-CPMBR1400.gz) = 7c23499487a89cd84aa8b4a429b097e8ace4c0bb
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-sd0a-CPMBR1400.gz) = 1a517e528904e379b39f740f0335c93de12e1b9398fa0564ebfd702fa064b75dab85bcbbd44f2f5baca767dcc61cd79bc7415648ed346e2e4d15ff3c2c278bb8
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-sd0a-CPMBR1400.gz) = 95ca7bbed50714a1aa9694362565c4219305a9a8
MD5 (NetBSD-6.0/evbmips-mipsel/binary/kernel/netbsd-sd0a-CPMBR1400.gz) = df9de39483767ed081aeb4bfe5d87f23
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/games.tgz) = 27d9f98fecb3495e51939df9f107e81e527af26f
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/games.tgz) = 46bb4e10313cfffaaf4f7fa2157474ef940f94bcc9f84b46f829c7615bbf45e4e01edee7b7efb84bc28561e0712c37cfc737f9c2a0d9a73b67c80c8aa0f3a9af
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/games.tgz) = 947feea90ec2698a65d555cb78a29a001b086589
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/games.tgz) = 355f00978fed8e8da6a251e591be5387
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/xcomp.tgz) = d8ac89f3f6ff20abb0adabc49227afdb74aa2040
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/xcomp.tgz) = 4f03213fb061dae9421747209628d06ede6bbb71a7560a9e1a54796b907f1620146a33fdce52974227a21acedcc210b99e15c4fdd36432eba60c1041a92d8cee
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/xcomp.tgz) = 6be5e72b9b57c9bdf8ed8ea127e2a73830feeecb
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/xcomp.tgz) = 59a7ea141587417be38ab1f6056cc780
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/xetc.tgz) = b9f6b1a54ffab37b11f4c6e717b481cd087fb609
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/xetc.tgz) = 72f35e65f3a90967e250a2006d500619b3824f7f5c3d7494554f153ec5cda5b28c445f28c34a1e4463f40da5855e8d4257726411796967fe37763d90cf7fe461
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/xetc.tgz) = 1169c246f91189fd0c4ae86ac27a92f5c2edda3e
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/xetc.tgz) = 8f3c52cc5af1cafaf7b0d0f884a5b537
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/xfont.tgz) = d06848571034f3b7fc975082f1de37b912acb18c
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/xfont.tgz) = 66a1013c935bee5e7e5d24325a527522306bcded7ced63790f2e58831b3ed91176a1b9510b2159be63504d5167d48d534919927ffc6e225ae2bb2487f90a9ede
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/xfont.tgz) = c9502843a5d4dd7e996e8d158c519cf7f8575cbd
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/xfont.tgz) = e812899176eac2bf65a6d949ac3f3424
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-ALCHEMY.tgz) = 5835e5112e44bae1346420ffc0f3e09a2e4f6117
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-ALCHEMY.tgz) = c3b2ad6f9678403cb3a6e0b8dbd05b6717aceac5472eac0c2bad86dbeb30b509f7a8b865daa499279bd524a10b075da08f6fa9736994cdc4633b0f67b0ac22b6
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-ALCHEMY.tgz) = 3b7c2006143846db5c3eab27c2f881dc22ef5e05
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-ALCHEMY.tgz) = da923fa9431ec9e4349f58d50bea5adc
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/base.tgz) = f0e2ba7bc4f19f19c71a28de51c4359ac56a6a44
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/base.tgz) = fbf1b00664f9a8530afad453e1af05ff39741d87187d25e3e5242c06fa013e6f6027d40944ef82fbc12c0ced24e90824a135bdff8ecfd69b04f2ffd03aa8d065
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/base.tgz) = 581b658d659de51923f79c189fb4dc75201aff64
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/base.tgz) = 185928a3ba27f23a06356b0b0df0ff4a
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/comp.tgz) = 66f0609c617456757b6563e94e2feaa2f788d2df
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/comp.tgz) = 83bc3d7127bbf0bb9d86f015d4cded042e1c0cdb75efdab00fd7941a7d96e8b54893b54a36b489d221dcbc17348473ba73287708f031b2ef37e0288074201400
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/comp.tgz) = d8d32c55c767525effee5ea75186373d6c253873
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/comp.tgz) = e56af26f877a19649c558c9d3ced6e6c
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/etc.tgz) = df85d74c9f2767c9be3de45d693f13547ff9e3da
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/etc.tgz) = e06ac6ef086e5951de773f02fe83c16ca1804611dda15b1dfcc0a44e81c3e6e6acf06b94480b0f643ef63e1a90c8eea81b2af4302e82801c033705418201e66a
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/etc.tgz) = 16cb173c32cd73373937f89eab796e5bc987ec96
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/etc.tgz) = 81f8caccfb4f464cf105f293393f6c78
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-CPMBR1400.tgz) = 1da339adf6e178fd648cf87cc4d2b5e1ad0bcc2a
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-CPMBR1400.tgz) = eb2c935de15c5622d5752c949b86965ec587b5c36c5f51ae2b801a9e20482772d1d728563aa70b299ec0316bc71f7754a0c4fb2e9795c01effcafa9123bf2029
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-CPMBR1400.tgz) = 3dd811b1a249023ffa47e31969f4086b9e156a15
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-CPMBR1400.tgz) = bd1349e129ae6367bf0e3532935c2317
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-DBAU1500.tgz) = 3f3199d94860f1a97269132ec110acb983133723
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-DBAU1500.tgz) = 1a1ed387d0902cad78598dba5c0a088e231594d16b4f49ac73c082f5c05f972b1a8e5ab5f28b97b23d696297cc9f334f1c8bff2daeb8a386aebf1930e3587007
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-DBAU1500.tgz) = d7f81fc610d0f09a26d14a082c9a143283d8594f
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-DBAU1500.tgz) = 54e6afe80b72e44b94d994d2465d63f1
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-DBAU1550.tgz) = 22c9cf65d89b9042f2b47316b99368dc0cad4d45
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-DBAU1550.tgz) = 3d2e737026c76a192d4461234be70622682563a14584ba8da5272be018a711dd108fc71f93d8251f8851246f8998c768f4a2f6e81b7985482dccf1fc166b5138
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-DBAU1550.tgz) = bde90234bc61e0105d5b7d1b23c23cae81461f84
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-DBAU1550.tgz) = 2905ebd9467c3a2ca17f776044ac9b80
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-GDIUM.tgz) = d7085296fbaf3357d9b3bdfc107c0632b7855d24
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-GDIUM.tgz) = 8841e72b1c725be15c37489f42254fc84feecfc2bc90050a4360ffc248ec14dce16dfe41ed8950f198c6a13fe78f287ed53d5545a9c0de775161de52994c1c21
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-GDIUM.tgz) = 49552df226923fde73b20818fcc3b88859095252
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-GDIUM.tgz) = e24f2b056cbf42955f24fc45075f2836
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-MALTA.tgz) = bbd9390fc5aee7d91c0fe0651503360f32fa6498
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-MALTA.tgz) = c09a032170d82e55a1c32fc468e6656509c99e9ee2a661255ed92d663c0605e1836afa2a098b40cbb9f6ca16854973733f6aacafafd129ca752c3ef569434f76
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-MALTA.tgz) = ff39a003e6a67114cbf0cac19c38f958cd07cc14
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-MALTA.tgz) = f9b7bf0cc95c4da411d9342dd2ebe8b8
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-MTX-1.tgz) = 5d4fb7abbffb58499873b02451e2d87e3818cd3f
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-MTX-1.tgz) = 2801b511740ce3310fb21ed8d1d1bba46ec2565721bbea394e1e1683c05984a88b2346b6949e90dcc737929e2919fb752661582984b27ea4c726ca6c70fdcff7
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-MTX-1.tgz) = 5b55476b98c8b0610bb7b03b15a457640b03df95
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-MTX-1.tgz) = f185c47ceea774af477cdc675e5da827
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-OMSAL400.tgz) = 854aac9898f3c0552fb0b4491d45dae21dc084b3
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-OMSAL400.tgz) = eb6aa82deeb2fa15ea59198d3ba10275eb73b85a902af353442138df9b66307ac2a595a61824301e56bdcf3fddea03414fd7c312564d039fb8d75fd9159cde58
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-OMSAL400.tgz) = 0fc4d0bf3a667e4fae2f60a504a4032751a5bb84
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-OMSAL400.tgz) = 53a48a5e4731a1160347ad6c655ccf5b
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P4032.tgz) = 9e8cb0ee684033f08ba0b4bda5a4eb16eac71fb6
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P4032.tgz) = 100f84c1ca2763695d90d8b65822c0ce4156c5c51236be33e395dcb7e68c806f979f7afc50c3c76457c82f225df32c44b7cf42c9c8a183469ead0b512a5880f2
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P4032.tgz) = bdda2cb2f34b84c0d46a8dfb7ef72bb6f3562018
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P4032.tgz) = 36f64c61f21f88c7d6067d2e9b1476ab
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P5064.tgz) = 8d6f635c7b19fe4b4ae3f81b8e55782ff929a275
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P5064.tgz) = 4382b4df12950d3ee601efd85e13a0b3aa4e41ec9ab7f1bb5024d1b198ee8c588592c2b50099354f1164aedb585ffd7e9dd123516f32f1f30c32cf78e77fc70f
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P5064.tgz) = b0bccd6abe423b957c452d1d42f40796f42533d4
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P5064.tgz) = b9659cc8f1fba16994a78e42ba17d9ed
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P6032.tgz) = 86427980342dd947c57c838152ae64d15cff5142
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P6032.tgz) = c86734213f246621075f8b35f51997d4abfc722502a4b8a3006d8224b1808871fb9103e539ab7ebf730dbb94377c3f75cd1596a3f7d3fda2505cca879c4608b8
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P6032.tgz) = 75aa0da6324608e7055e923ba1004f5427c88770
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/kern-P6032.tgz) = abcfe72de3e084c79528616c061d70ad
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/modules.tgz) = 66831501dc768e732c9b93cd102bd84f078ea1a6
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/modules.tgz) = 6b50ee54d67e9f57f125ac1eb0f1cf849f566a2ca4055032f708d8bda121b8ba96274c5fca21deec4753a3bbd64e18ffa66fc0548b363f95b5c0d0d4f2690dcd
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/modules.tgz) = 1db5ef74be63618d39d44716face7ca5cbafb86f
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/modules.tgz) = 47d708c50d3fbb3f170883f337a84a3c
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/tests.tgz) = b87c52edef5dae8c13e4982f742d107c6bde617a
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/tests.tgz) = a227c03925059ca58390063e75c934e9bebda7f2944478302ffca4f53a48bf991314c4bf837c4e833aa700424e154cc62526f76c53f23ac7281457606e975c20
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/tests.tgz) = ea25f7257b61f3530435a9dc66123831d59e6c43
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/tests.tgz) = 42da5bef4121a88ef7ba8e89bbd065d6
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/text.tgz) = 38e7f089801576d8a7370546c8f6dfee8938c54d
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/text.tgz) = 4694ab32a63c7a00b2da8921e278d14ac83dd114beec338716a98cfbdbe7c06ed07abe0d6a10b317b50a9deeff4f4049c487458bfa9cef043b92905e9f4828e7
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/text.tgz) = 6b4ed0dc8220b79963b2cbabfcbe1ae449cb4509
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/text.tgz) = bfbab9eee77986403da3513d836616bf
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/xbase.tgz) = 71f5a25bb86649bdbeab0146bb32c76329afa9c6
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/xbase.tgz) = 70afd5fdc176eda270259b113d7f28cc92fe2037d7bc84cc69a2bb32ea05b23bf26c10f834fa9d59be9900a2ef780b55e5e181d6b1a88ff51ccf35488d28fa22
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/xbase.tgz) = 95a5dcc67edbe81dccf5f90e0125e84a6f267cfe
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/xbase.tgz) = b0a603b80fceb0f5f0c02786d87542cc
SHA1 (NetBSD-6.0/evbmips-mipsel/binary/sets/xserver.tgz) = 9be61698fe883646972e3441d1ea02e16fa351d2
SHA512 (NetBSD-6.0/evbmips-mipsel/binary/sets/xserver.tgz) = 17d9d25171859208b00a7e64dc74eeaf4a2533cb86d7be138227aa731a09654c007bc4245817177dc2308d693c1792ad247bc0692de79e3b0dd8e5d2257c7246
RMD160 (NetBSD-6.0/evbmips-mipsel/binary/sets/xserver.tgz) = 168f6c36213258b1f7f3019bc47434a58cf68a54
MD5 (NetBSD-6.0/evbmips-mipsel/binary/sets/xserver.tgz) = fdf224dd21749c1a2eb06f6ebde96338
SHA1 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.srec.gz) = 1148ef0c3e4745f529e78b33e048e77a2357b107
SHA512 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.srec.gz) = 127eac81b94185bea670ef1766abe38506310147c058be948614bb472cd97444402c65eef69375c39d6bad518b617a2fcfae314095c94c9263b322e061ffda0b
RMD160 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.srec.gz) = 88a878858ddcfbdcc82d30efe6db396e12324245
MD5 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.srec.gz) = e0fd06b18d51bbed126e558fa66a0124
SHA1 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.gz) = 388abbcf174f34d8615308f4d9387534d0dd4973
SHA512 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.gz) = 6aa9bcccacbb410b5f0fc8fc905e78bde3af777913ea3c2035cc6f9a6e4a436fefb7a3afa987403d16a6f1e7151cec117391ba0d146679e7c0c4c9ab1f9727dd
RMD160 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.gz) = de9ce82a62f7f24c1c78b0bde362f0e4e1d66293
MD5 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.gz) = b90209535b8a3e40f5d020a99bba9813
SHA1 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.symbols.gz) = b1199797080df5d8669b35bcde9172d628bee188
SHA512 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.symbols.gz) = 3b4d915b17ba81bfe9b4f9768841c8a140359ea07121d88b2016f0712f42de965f5cf37dca483ba718d47743da6c300b38499ab5c93ff1978d10622e203f84db
RMD160 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.symbols.gz) = cdb363ed9995287373ce565c3a6a0a4c0441e9cd
MD5 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.symbols.gz) = 16a4265a582c47946a7bb8ad40f93856
SHA1 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.symbols.gz) = 067b86e211cec25f5b7b826139195679a7b13ad8
SHA512 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.symbols.gz) = 7d4006e0da241ee8b44572a510dcdc33d14cd543d46022faa855b1a554832309d0ca756702a0c7eb551d0d2b77643bf1905153c59cfabd9879e8057d4d9ba89d
RMD160 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.symbols.gz) = 432610d592a1e11d09d2c6de9bcfa6eb8021e2cf
MD5 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_MALTA.symbols.gz) = 4c03720d979460cbb6b5311c54214642
SHA1 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.gz) = bf85a74b3aba2ec80e8aa7b595e34ed729b9beba
SHA512 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.gz) = efe3a15153df52551b5542dcceb8c41c9ac74a56cb9e5e4fb2d531d0d933d6ea3f8555831e8e9b707d724ae15cc27b79c790232fa9c5ad142f71b6a15ec61e1d
RMD160 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.gz) = 4299849406f323f0f65a40f74734a1cd34206bb8
MD5 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.gz) = 54ed1ba8d8009ba512e9019f905c0368
SHA1 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.srec.gz) = 2f72f6fe9f6a716994fc2dd68ae51c6401c7389e
SHA512 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.srec.gz) = 4d8d6a4eebdda98b6a8074ea33ee54f729bff3e8e1f8c5c47699819a3cd29d329853bf0083024f9ba3a8893a2255651fe00232408433de8654308d27ba72560d
RMD160 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.srec.gz) = b05f15e41052f8ab490d7cc00367eb5ce2430bde
MD5 (NetBSD-6.0/evbmips-mipsel/installation/netbsd-INSTALL_OMSAL400.srec.gz) = 1b2e3da6a2b2dfea79a9669561f04413
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EV64260.img.gz) = 963fb1dfd634ec77183e0e19e63add5d6f515a46
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EV64260.img.gz) = 670195305b6eba8bc6cd2b5a05bcae746dffd3a9bcac873d61bc659893de97f35dca7707d073d9fd2e55058f99f83102364bcaa64d0fa9954aba2bd42eaf297a
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EV64260.img.gz) = 4ff261bfda6f964c67273e2c6cad05577ff39cc8
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EV64260.img.gz) = b3cbc16d9f2d3870fb9c137fa21b7625
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EV64260.gz) = f74b0a7d7afc324d7b463780fc6cbcee71030e28
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EV64260.gz) = 894aa697fd0f0968e534db42528be520c74adec169dcbd2d3603cc262d33e23092253494b072f7239f1542b6237128d8c5069616e28e51422dc69d0bca117d1b
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EV64260.gz) = 7d00ed85fbb611948f16e513072f2a307fa46d83
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EV64260.gz) = fce468eda12d673785cf7b16c7f4355c
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EXPLORA451.img.gz) = 0dcdbfd5853cd5ee3649e1bcdfe16e3133128dce
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EXPLORA451.img.gz) = 0654c6b68a52bdee61715fb8b2b1af775f4c7b4a034d11ad7d4cf9f31cae4b8fb327a78dbabfaefcf6d68352ef8be2b4e71d053c97c0c59a24250358fbf47589
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EXPLORA451.img.gz) = e307e29dbdc57b0843e54caf02092c38dcb11cae
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EXPLORA451.img.gz) = 444e406bfe16f2653bf60273db88dc3a
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EXPLORA451.gz) = 54c5de8e61a39b7114fab8a4a28036ae9d44090e
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EXPLORA451.gz) = 092f45dab2ddc2cfb79b1c1d649f3a86f169a380c01b7016737bff28b4404a47871546c33e89fe43a6db0255556a2a1cfb2bf8cfd725a852af2c19dea35462bb
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EXPLORA451.gz) = 30903e7cf5db4e4f57334f71240da21f6e0fbad2
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-EXPLORA451.gz) = 229f310637a0c65b47a078f2a6f0df82
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.img.gz) = 5046cdf7029475b3c2d6f8fb0eb9ac766d55d568
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.img.gz) = db6f4a419ea019f46205cbb75f76bb8a35bc9d6cee83f9da3b776abb3f83334a60fcad2ef6de4041c0ed3b0689cc259789ca2517dca0e101fd46ede807846d20
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.img.gz) = 42f2c254a99df74b42ae21c1a89d42ca1d8dc105
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.img.gz) = c7b61fa28cd76c1c694fad7becf124c5
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.gz) = caf9b0a77c6adfa8d6cf1effad83b107ea9c5976
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.gz) = 14db94aae558e9e3311e667343468625f309cb71dafc7610f029e6b6e27267ad6169ce21320d6fc512bef2cba49c3f1919d396fbaccbeebe49e8e53dcab865b0
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.gz) = 3a12875aed7924a33df8290b73576ecfbc8dd752
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.gz) = 20580d5bd36fc097ce824bf9915fae05
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-MPC8536DS.gz) = 54b88ef9d752b74a27f85affe9d162bcb43009b2
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-MPC8536DS.gz) = 42903a0d5f74e45d00cb48a4cca92c164e26097791048f2cc1c2b7a1d8b243cebe9f64a3daba9542c12b077017e59c1d0d6c8a8c11ea120855f0f40cc7ff8fb0
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-MPC8536DS.gz) = fd7ac6dae359227cb757ae1eadf39b1fe08e43d3
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-MPC8536DS.gz) = 505ede4a04c4726a1121a17a531a74d4
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.symbols.gz) = 9c6ab2ce83f2ca09cc8789a9431638f7fccdd64a
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.symbols.gz) = 8ffdfde54ed60c89f642b91c6e434fb4ed4de6ed91009df3bd70556e73f6eb5cc28b7bb3090823cd8671e052c5d52bb15eaf8409e779cbbbf4ee6776580b67e7
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.symbols.gz) = d52adf8dfa375abc8e4d0cfd7a414afc7b44aef9
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_OPENBLOCKS266.symbols.gz) = 6bf07ac6f2c54f9b46171b57492b6c4d
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.gz) = c4999c62d6e7c7c8351dba57c34ed0213d2d374e
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.gz) = 2acf7209a186bb3c52c54e49660c76629fdc8e80e73e1b77beb23e6b56c7acd1d985c958f152c8692303c41d3a4bdcb753ba00c232a772919ce106b1c439d697
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.gz) = 9a5d70bdff7b5208eb2f7f6bc2df3b72ce770588
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.gz) = 777ada452ccd51471ab700f4a6f9e652
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.img.gz) = 2a5de49c3e1b5a2f4455ea780f85d185b26845fb
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.img.gz) = f1f740403f47dc84dd3935611f4e1f71d860c202baeb8b53c23c4c4c6543f9b903cfd559465f7a5c3e1e59a8604f0194af556ebbc7000592cd532c7895b25287
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.img.gz) = e974201a8d50e1c9a77bc34a4df289e94a58e966
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.img.gz) = f710c52befd5c32ae49e90b7f74f8e77
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.symbols.gz) = 89c1b110f3a260ae41093f65323a11028f37826b
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.symbols.gz) = 30c988a3a2b647a1c93a95ab3bc7fcf0428f89dcce44152f81c2a58af49f5e6c95817650f5affca8a83ea0318d913a0ba20c28dfb6b7cb3027dce2c5561ece52
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.symbols.gz) = 4838d9d57fdcd79da40dbf3fdbbf09e77118ff2d
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-INSTALL_WALNUT.symbols.gz) = 5b9c0173413ff799fe5140db7ab9c13b
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-MPC8548CDS.gz) = fa0461c820b70c326f5f2867ff081cc88d357aa4
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-MPC8548CDS.gz) = 436d573a0b468d6794a9fc16955bca56a7b726c3e0e306f05e6965bcbd109ebf71fff060b8afcda9782043bd5389e961f1bb191d9b9dd209b385e64e448eb63f
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-MPC8548CDS.gz) = 01ccef4bca33dfd5dbc63d901d9b006a930858ed
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-MPC8548CDS.gz) = 9c552da04cd50465e8451c12ce98ac4c
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS200.gz) = d8431c30e8c1e34eb2b721d51f10e2429b542cf9
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS200.gz) = 1d111341a7439e63ec210bc09c168cd613b114222241cba494f19e6bf582a0085a8894e1a6a8842b7b1c6f8ee5f2c8b39238b9dd714727a053637cbb59a1f77e
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS200.gz) = 47a3f7ebc1980ce1817112520f13396d6efa2d44
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS200.gz) = 4875531aefe44295429c200a8dbbf3d6
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS200.img.gz) = e7e23c8fbe61ca6ef52822a4fb9336be7168abba
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS200.img.gz) = 6c78a517557651386736d048efb9953b4df3350b411291fae3029074ad0dd961f128ebcac4e6f14dd56e53d43465994972ca29572fbcdcd8793f713df74a3e5c
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS200.img.gz) = 04ea401821e981bff21fd61d8503608fde540bb8
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS200.img.gz) = 3e5f6b82afebe7b1ab5de3fca0bd75f0
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS266.gz) = 22aed20110feacb4634479f3080db81e3a511e30
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS266.gz) = e0934031d3b963fe5edbb5944b92a57c8f4ba4b496d8c5ab6aed555012a53f73bfcaba7eebcc1669bd6a308603360804242b7c37e92c022872c524c159c92f54
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS266.gz) = ad1b890f8ef6418a1ed2858dadafc7e99729b6bd
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS266.gz) = e935df1037b38d1a903359f63aca6df4
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS266.img.gz) = a3b023c3bf29f930fc5ff0e0c074e54e23141963
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS266.img.gz) = 3d1c0b86676c2bdd861f7e8e4e20d6a34d1a04079d908b6d2ef5404b1d00ec1f394a8b13fe5d4b4906c996d1912777e0f52ec367ad6e1c9db64cbd22931347c0
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS266.img.gz) = abdf86f9c182d23740543843d59e4dfcb05e59cb
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-OPENBLOCKS266.img.gz) = 84ce6f797d900ad639313c55e64178b2
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-P2020DS.gz) = 8cc216c33fcb3f59904067c3e9294f0369d6f1e9
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-P2020DS.gz) = 6bb4dd7dba52e6a6f4dcb76aa5218ab893deb64d57a87048ed5dbf882c13e92646ca89a037ebcba5211407363a9365d2bc9296397322dac95374e3f03bd88e51
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-P2020DS.gz) = d8f03ff91019223b562ee52da049a4f71facf379
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-P2020DS.gz) = 958385f31a4f6b1a234652faa95c6c95
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-P2020RDB.gz) = 9140563a57548b77f93eee35e2615f4a377612d3
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-P2020RDB.gz) = 1fbddd9a55f6669c1f2f48dd96b7147cae93a2d5642b262a81669fa4a2670d465cd09347a6f9b7751baa338935114bc2ba2e67622e2f524d9793628274f92625
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-P2020RDB.gz) = 3ad3d010c5c0f34a69d7b49c7c1d639f7c5c3618
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-P2020RDB.gz) = 094787fd9d22021a1b63422895976203
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-PMPPC.gz) = 8f72cbe7df21613d6b796b579ced2911efbe66e3
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-PMPPC.gz) = f4f11445454376ec9f83f9e3072421a8ccb7bee9b1aaf01aee70a6c044e0de1156a5fbe28b1243e8efb334adeb70621e8eab875c44f590fd0276114a26d8db15
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-PMPPC.gz) = ff1f745891b1c4a02836982de15575afe78c16eb
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-PMPPC.gz) = 87d0847ec4707e24c58146eb0475d1aa
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-RB800.gz) = 139b9c2b87b112ab2d6498e726360504d0adedd1
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-RB800.gz) = 8ffc6ac565bf37ee69f8736d7c86738dd69ab7873b77ec062a12a442851559e4c94c903f76e649a21984efd4b98ccaa97333177be635807e6dec8a17bd421880
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-RB800.gz) = b6724d690aef60f63c4b1339eb6ce514e0886151
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-RB800.gz) = 2df5233415f6f20f2af441b4b54379b5
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-WALNUT.gz) = d5e489629cc8d1b2903e541b944a123cfc0937bb
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-WALNUT.gz) = 77d3eda4223f4cf9120a7e84e8c61917749462734c5adc70d95e3108a1b151e0ab7518092d37ddceead9e9047297ef47f4712de7eb0dc05f4190ab60e775d313
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-WALNUT.gz) = 15568ccc03d6a06ebdaf407b6d2b4ac07985a0f1
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-WALNUT.gz) = 11d3fcb67c3ec63c5c43ab5bfe06c0b1
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-WALNUT.img.gz) = f001160c675e3d0f73978743c88bf61561a7b10a
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-WALNUT.img.gz) = a85072ff3b1a1d8b9e35b40b1bb49f5065bad003f603add8330de38a71c19028ed54e377534e4e901281ad5eb65dcf73e8657a39bffac9645e6fee6bce7ac391
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-WALNUT.img.gz) = 31480f15986b52f445cefa8aa4e056b6fcb74f1b
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-WALNUT.img.gz) = a0719a86fd17f9ddfa5131e58f77622d
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-esata-P2020DS.gz) = 50b925468b1ace37856fbfbd4fe5363ed187b4e6
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-esata-P2020DS.gz) = f8cde43dbaf1744d7dec3437beb43be1b73178fb95038771a52315b5b54f4ae369c9261c35b710be563b0bc9be07e4801100e012c8313c564f3bf66845082c31
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-esata-P2020DS.gz) = cff81475ea7ebe293bbefc4088d5dcb76906108d
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-esata-P2020DS.gz) = d79a18ae8aa921fbd0df460cd5e550a3
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-sd0a-P2020DS.gz) = 1c75ca32cb0267f982916a5fb2740483e5aa9f7b
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-sd0a-P2020DS.gz) = 99d1b928ee2674f7dd22a10ea6c3cb7b24f5bdd4961a5315ad310e2189cf4508bde00d9b752cf3eb6843c3a461f748deeda4f395206a4973be55165f6f17c9ff
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-sd0a-P2020DS.gz) = 24d2b7dab4018de8de5feb65f23cda0e8b4c1502
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-sd0a-P2020DS.gz) = 6fc3a2ba5e17d4ecef5cb52b482761ff
SHA1 (NetBSD-6.0/evbppc/binary/kernel/netbsd-sd0a-P2020RDB.gz) = 342dcb5cc506d3272737f3b2c0fd16b7221ef429
SHA512 (NetBSD-6.0/evbppc/binary/kernel/netbsd-sd0a-P2020RDB.gz) = c74ccc79f91ac3780093676aa88cb544da9cd17abaee1c1d2692acef50c5f13c4ff1e6ea4c334c43242f20460ca6f08406c1514f4b84cb7206900192699b2e09
RMD160 (NetBSD-6.0/evbppc/binary/kernel/netbsd-sd0a-P2020RDB.gz) = db206fd4eaec146d07e71794320de0e680bed506
MD5 (NetBSD-6.0/evbppc/binary/kernel/netbsd-sd0a-P2020RDB.gz) = 6a7ec6e8e8a7a2f75e60073154234efa
SHA1 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-MPC8536DS.gz) = 3c9abf574a3c5dd2c38ef5af3433ce2123a30192
SHA512 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-MPC8536DS.gz) = e58dd95eadafef8b242ce0894ec809b1d73063f7142cb0d6ce42dd03f5a71032ef3d31ef09e19f9cd6594b5bf65806473f560e17f0d505f5491c3651d6f9e930
RMD160 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-MPC8536DS.gz) = a419d3af2732cc85c1fe0ca06322e0996a7946cb
MD5 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-MPC8536DS.gz) = c900656ec6ae14a67385ea265639a500
SHA1 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-MPC8548CDS.gz) = 14a02ead1d8e5963e96ddd3a5b3b2625202d0011
SHA512 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-MPC8548CDS.gz) = c51940733928beb1f9beb280ecc642dd867317be5d0ca36ed9e0e01b1c8b2a1d788658bdb822e4b7be358fca34fce4620a9e2afb10a265007824efb965dfeaf6
RMD160 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-MPC8548CDS.gz) = fbf09f5e31f3b61a3cb4f1cde889a2c91635cfd3
MD5 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-MPC8548CDS.gz) = a4094ff843bdcfcd143957a0fd15a01e
SHA1 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-P2020DS.gz) = d5125242574b57126e0a9ed0b2c9e18ba47ba7dd
SHA512 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-P2020DS.gz) = af9b230b2940352e71cba43854b29418e2d1c6a51cfbac7cfcc801a4c1ecbc0605d055fe673fae7f244a421058073f921f1cbc8ce55b3131017138396f2071ac
RMD160 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-P2020DS.gz) = 7e25cb6632e7d744f60572c578fe34ea1acecae8
MD5 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-P2020DS.gz) = 3659db43dbb405a80a675b7f2840c7df
SHA1 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-P2020RDB.gz) = 9a55d38b7e6f8c5b274ab45f20794feab61a6264
SHA512 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-P2020RDB.gz) = 15ee2dfc4c476a54f0483facb66aec9b2b743775df471353ca1e443522119af238bebf7a986413d6cfac237aec8242180499b23fa85b85017d20179561025ab8
RMD160 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-P2020RDB.gz) = dceb778c7ea33542379775b8481122d7e55f9e98
MD5 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-P2020RDB.gz) = 39f5865216f1e2e37984109a0cf2c708
SHA1 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-RB800.gz) = ea5d21d649ce2a6c3502b58446cbe747433dcf12
SHA512 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-RB800.gz) = 8748659a4a7e79a6b4b00dcb90ed2cf8213c138d4df7f93e6e53bee63f0c8f6c9a837f9522c9dba0580226eaa756473640226d1f4edb688a5b04807cb588730c
RMD160 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-RB800.gz) = 9fe62c16553ca3e5955d12ca7e59cd0336bba55d
MD5 (NetBSD-6.0/evbppc/binary/kernel/nfsnetbsd-RB800.gz) = b4a5d575b02cf078c5c31542e5b4954f
SHA1 (NetBSD-6.0/evbppc/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/evbppc/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/evbppc/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/evbppc/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/evbppc/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/evbppc/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/evbppc/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/evbppc/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/evbppc/binary/sets/xbase.tgz) = b7c8146146ebc15191c1bf53f51f8f980fe8c816
SHA512 (NetBSD-6.0/evbppc/binary/sets/xbase.tgz) = 091f5d00414a9ad478811bfb0e953f2a204869bc5e310790886a59c11953974cf7068cd9a96c8823856ab2687ffa2ac31e97494f47338bfe31d80217382c59f1
RMD160 (NetBSD-6.0/evbppc/binary/sets/xbase.tgz) = 4bd1bfdc7c1e1619de63fb83276c21039b945357
MD5 (NetBSD-6.0/evbppc/binary/sets/xbase.tgz) = e621806afa0fcc9cdcef0f9b3c7dfcbd
SHA1 (NetBSD-6.0/evbppc/binary/sets/xcomp.tgz) = 960dd33b61e19dec92096a54791aca1cdf56ce54
SHA512 (NetBSD-6.0/evbppc/binary/sets/xcomp.tgz) = 111dd50041e4008aeadb898c3d08f9995c5c13a41ccc20993b4ad51c81d7f6ee3b4dab16158ac7436d085b80e4a643b236a9c97edd3e915125a1466154538420
RMD160 (NetBSD-6.0/evbppc/binary/sets/xcomp.tgz) = 0cf8b9a8d1489a32ba0b5a277d90e429c40650fb
MD5 (NetBSD-6.0/evbppc/binary/sets/xcomp.tgz) = 4ba4077ca2c0edf71813165fcf5321fb
SHA1 (NetBSD-6.0/evbppc/binary/sets/xfont.tgz) = 4711fcbade0238d60b7b89cf0d7873b03d7ed1f7
SHA512 (NetBSD-6.0/evbppc/binary/sets/xfont.tgz) = c29130f55f536f602253a7f92f073ebf6ba02baea09f8011530bbf7e03ae96594334f0c60b29a8bceaffc3eb8b152db47548e38a3c8a14abc5fe35bc5d1d7d93
RMD160 (NetBSD-6.0/evbppc/binary/sets/xfont.tgz) = 3b01af0f8e34367988d559fe0fbb0acbbafb0b2b
MD5 (NetBSD-6.0/evbppc/binary/sets/xfont.tgz) = 6ba15ed12217b726d9129db8e942a048
SHA1 (NetBSD-6.0/evbppc/binary/sets/xserver.tgz) = 05aac7e617e0cac05a88062fef29bb459da7a1d1
SHA512 (NetBSD-6.0/evbppc/binary/sets/xserver.tgz) = 19fb5af5d6afd78839a22e5c0694a492047b5200c569cbf58c676ff32ec250b21edcfe2ed00b65092fdac374d1ab9eaaed6113c69bc03009cf458aedc6014913
RMD160 (NetBSD-6.0/evbppc/binary/sets/xserver.tgz) = 774d2a1072b8d122790ad189a6d2ada2ad99e7cf
MD5 (NetBSD-6.0/evbppc/binary/sets/xserver.tgz) = 3daf1477cadf3593e9ba45e3703fe8a5
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-EV64260.tgz) = f49583bb3bdb33b9eb3b0b4136668772da20f45b
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-EV64260.tgz) = ba5c1437d9a233dc818880e72de7746dced45081a2fe998ca922c51eaf404bce9f42f7f763a611485bb518e50c1e4da5a6906d0b28d01b9cd3c9f1be4d8c2761
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-EV64260.tgz) = f477b6948cae222002fa140de299929d9d9807c5
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-EV64260.tgz) = 168a599ee54d18f0eab734e5717b2984
SHA1 (NetBSD-6.0/evbppc/binary/sets/base.tgz) = 660713e719fb1be5824c7b5f573730b9d1408922
SHA512 (NetBSD-6.0/evbppc/binary/sets/base.tgz) = 366e7065fad6b8e8a7e593fa5790eaaba11a8ad49006730fc01fba2f1d7908a6ac1ebfc88a4cf7731b1d31b3e21dd5d0aa5c5433cd4f01ce95254735da15c772
RMD160 (NetBSD-6.0/evbppc/binary/sets/base.tgz) = a460662933495846dc34b422fbf2f2f3b7141068
MD5 (NetBSD-6.0/evbppc/binary/sets/base.tgz) = f527c4156acc659412d01a00756d57f3
SHA1 (NetBSD-6.0/evbppc/binary/sets/comp.tgz) = c9caa7ece23746ea1423f1cb05924d7aafb69081
SHA512 (NetBSD-6.0/evbppc/binary/sets/comp.tgz) = d60f9bbefcaa3331646da3e9bd2738cf11e357807ce64619cd48db62cd3c72469f39113b04883771c610c63f74e574941add1d83ee2cce2a7b13e85e844b9eaf
RMD160 (NetBSD-6.0/evbppc/binary/sets/comp.tgz) = 2db97b320a8befeb352b84a44e32b3844648b85a
MD5 (NetBSD-6.0/evbppc/binary/sets/comp.tgz) = ab787a55ed8b6765619f75f613914955
SHA1 (NetBSD-6.0/evbppc/binary/sets/etc.tgz) = 01f15943e5ac9e3d4885c4391f332b3367a96483
SHA512 (NetBSD-6.0/evbppc/binary/sets/etc.tgz) = 3d69692f01bf9b9dd36e719af4f2e99d7c9aac0b92925e2751b0ff8ff7651e37f948d18765fda1d1841fe8d97a7869674d584423594ad1ca48da8b4c35e0a3ec
RMD160 (NetBSD-6.0/evbppc/binary/sets/etc.tgz) = 6422793912a3843dafb20baa3c9ecdc10305262a
MD5 (NetBSD-6.0/evbppc/binary/sets/etc.tgz) = 789c56228f05a0cf67d1917ae04862f5
SHA1 (NetBSD-6.0/evbppc/binary/sets/games.tgz) = e7bde1f51127db8c3a7d15ec931c915806661f0b
SHA512 (NetBSD-6.0/evbppc/binary/sets/games.tgz) = f39fcc09a9b81a604a29c968dc74264a42c46ccc122c9ac01030e6ca01c28967fb7e20d8fa182f2ba07ade376a14a802c2abd06154c754f907a9879b7e7ab52e
RMD160 (NetBSD-6.0/evbppc/binary/sets/games.tgz) = a09ee1c7811eaa0ed3aa1ca0f85e1279b096fc46
MD5 (NetBSD-6.0/evbppc/binary/sets/games.tgz) = 81e8837aedbfcea8d6957f192e54545f
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-OPENBLOCKS200.tgz) = 561ca3ad31c44deca4b197207d2a2e6a92ed4bd4
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-OPENBLOCKS200.tgz) = b47b23e9b6c3efdc389ce3b47218eea8da190734075985e7ad009a10addcba48e565efabbc3b11c32e220d6059fe5114ae77351bc72549e8b8f37be98dcba27b
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-OPENBLOCKS200.tgz) = b2058d6a4e4fe11ebb19f429347a6ed69a1b014a
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-OPENBLOCKS200.tgz) = b2ea5966c8110677fd107fd4029242f1
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-EXPLORA451.tgz) = 7a6fe1de68c0fffedc5918ffabeb87ee71ad3da1
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-EXPLORA451.tgz) = 2e30f89fc77fd9bcd9310c4d12379245a18aed68b661c037593870c86ea806edb62f32a433f46e4f4c205c2eddfd86c353c3ca970a887a324b4d9eaf952b537d
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-EXPLORA451.tgz) = d68c12c132889e9cb48e003811e9c2de7fe41aa3
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-EXPLORA451.tgz) = b5ba93f49378f17fa86d91832a97d63b
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-MPC8536DS.tgz) = cb9e2b4befa9231d5c6bcf52460189d6ad18fe85
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-MPC8536DS.tgz) = 937bdfd98369d438c25a0839e50cac22602b3766db5f359d31825445c156fbe94de24956584b0668bf5c1b09565733c5dc4e9dec8fb5c034f5764edbf8241bff
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-MPC8536DS.tgz) = 63389f49fc87afc7227318374fa8fbb130c1f076
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-MPC8536DS.tgz) = b4ccb088690ba31893f554c22874c25c
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-MPC8548CDS.tgz) = f17cacf88747cab418fed98c8f2d4341d883c45c
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-MPC8548CDS.tgz) = 13adbfaa5c03713aba913d4bb08f6c8f654b3af210164d111d4c5825bd49ebc7ea8b0b4e5dabecfe685cc9ea9bbdbd0c41cbbe3e3a4bf9d497186c0cc15ed3b7
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-MPC8548CDS.tgz) = 5e2198d8b2c62f629a2e41ff422f50f0d90b2ca2
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-MPC8548CDS.tgz) = efdf3c8e143e54c672ba669c5b2211c7
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-OPENBLOCKS266.tgz) = 3dff6cb4cc77e770b17e56dc132fbb3ecafa65f1
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-OPENBLOCKS266.tgz) = 6309b5240c155074f7f0ad04684043e26886283d81369faa2c5d5ae202a82c5fad980c1101cb5f03fbf50e61b78fa87cf5ce633ab354b90ae0f267c4fc03dd92
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-OPENBLOCKS266.tgz) = 22a0af77fbc2dc62e11aabc36b92701e12a23624
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-OPENBLOCKS266.tgz) = 776cb54622eca1a207cef956b8fe9a2f
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-P2020DS.tgz) = 6ce50d8dc26e08d660edc8daf9b4c35e2a03b005
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-P2020DS.tgz) = b04c1630a1bcf4ac4855e4242c5cd12b82c846b371b6257abb99ba4b28026c546ae914d68ed31ecfc02d952aef81bd8fa50879c698735d453bb0a8ed72131fd0
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-P2020DS.tgz) = 16938708031722094448b4a613dce8e2acbf8c63
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-P2020DS.tgz) = 94ba27899a9ee3756ec7084e362ca968
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-P2020RDB.tgz) = b0bbfe5b10d1596af93eec4960120ebb23d00411
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-P2020RDB.tgz) = 536a382b932643d9e4a65dca9f4601edce766799da3e291484e9460291ce4168b5cfc59a39bb1169e69065bb65d1ff8f56caa98651d643bad9f33ef35b219d0f
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-P2020RDB.tgz) = ea8619e5c65fa02151fb23136e03ee8889f68ab4
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-P2020RDB.tgz) = b14ef5c3eff1d4c89afb1e9d883aa3d4
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-PMPPC.tgz) = 56e8d8a4c533163eaf18a2b9cb988e0a61be8174
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-PMPPC.tgz) = 87f5c77334788e59662c95dd073e3fa8a680f840ccab27e839fb405cc942941d5c2459cfa4d93a02179ab27fc6e75f8d3c957c3c95e5e271a404f64ec08a0b19
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-PMPPC.tgz) = 7160a35f527dc620a2e67e17a22e8cbdffd8fdfe
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-PMPPC.tgz) = d0459860eae7ad936e2b318a2cbfd7ac
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-RB800.tgz) = 79b942c2076a9219525ec9da7b25da3a21488675
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-RB800.tgz) = 0c9462c4519ed0fc1b378a7abb23c3a89768d6fe128c726e2847c6daf576757492438d4cab244e32f1bd7601f1a3e808d79f874b3480e22abff4a45b9cc5a8fb
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-RB800.tgz) = 6c7309b72d5a61879c3151d2065cb70361ad5c70
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-RB800.tgz) = 9951a8e7708ca609ec9abfb3b313f03b
SHA1 (NetBSD-6.0/evbppc/binary/sets/kern-WALNUT.tgz) = 0a60e6f9a6fbaa8477b05baae64d4b405c9c252d
SHA512 (NetBSD-6.0/evbppc/binary/sets/kern-WALNUT.tgz) = cf5e16c71fbe6366c41503578254a5be9bacf01cab691f3667c6b3b5ec3413cfd30f3dd00152a7c0dea73effa0f46fde1070df5aa29db02c021bc60a83d2ebc9
RMD160 (NetBSD-6.0/evbppc/binary/sets/kern-WALNUT.tgz) = 54ade37dbefe29a6de73720d7900ce68b7dff1f3
MD5 (NetBSD-6.0/evbppc/binary/sets/kern-WALNUT.tgz) = 4f92053a06ab8e494ae5068657a449ee
SHA1 (NetBSD-6.0/evbppc/binary/sets/modules.tgz) = 3297a716bfbfdb870519304912c1f0ba834ef318
SHA512 (NetBSD-6.0/evbppc/binary/sets/modules.tgz) = 586c1502a8fcb99323ba75ca343fbb8204fb80a00de8b09aa6722bc4d7bb142dd35bf023f11ef981f5ede707dec673dc1040578677216cb0aab2e1d9d871263d
RMD160 (NetBSD-6.0/evbppc/binary/sets/modules.tgz) = bb87c0bfbb41190443d888876e2f6329f8c5a0c6
MD5 (NetBSD-6.0/evbppc/binary/sets/modules.tgz) = be483e96c1f3eb3a155b17fe7097d398
SHA1 (NetBSD-6.0/evbppc/binary/sets/tests.tgz) = a9c5bab23ec89b45bc96550385064ca681c1aaf7
SHA512 (NetBSD-6.0/evbppc/binary/sets/tests.tgz) = a629539a7f807f4cf5651329769924fbd42102b4e9f96b4aaef254d1237e0ff4b6a55a9a1968bd6aaecf5f4e4db667fe59b8e2afa8232b80af4372fc436eed18
RMD160 (NetBSD-6.0/evbppc/binary/sets/tests.tgz) = 8f786bbb44342927ecaee51314d2492b228ca507
MD5 (NetBSD-6.0/evbppc/binary/sets/tests.tgz) = 6d2a4b03364dadacd276c5e908c6053d
SHA1 (NetBSD-6.0/evbppc/binary/sets/text.tgz) = 9b944538cc5e87133cecd712523216a783ba97d7
SHA512 (NetBSD-6.0/evbppc/binary/sets/text.tgz) = 7d87e8bfe246ce56e93257beac1808fd00160911acc22b5abcfc4de9125182d982e69709ff8e7f5dabd2454024cbf2969d836cfec0bc8b314f09143654648e99
RMD160 (NetBSD-6.0/evbppc/binary/sets/text.tgz) = cf2dc7ca65a246453309df0e94e804c5a04b0719
MD5 (NetBSD-6.0/evbppc/binary/sets/text.tgz) = e8220cff6b33bd3cb4e91d50573b43ff
SHA1 (NetBSD-6.0/evbppc/binary/sets/xetc.tgz) = b6aafacecaf76d99a4c3c428e33a23633419f52d
SHA512 (NetBSD-6.0/evbppc/binary/sets/xetc.tgz) = 9257fc58e039f2afc1ae7f46a7ed2ecdd1a13e67a76960490e89efb8fb12b44100a465270469058071249e0864346ff4dca825786552a4a8ddc804cc7a90dd33
RMD160 (NetBSD-6.0/evbppc/binary/sets/xetc.tgz) = 6c17517837ddebee9482883db92d1542a89835b4
MD5 (NetBSD-6.0/evbppc/binary/sets/xetc.tgz) = 77d81348f7721a447c8548aac150659b
SHA1 (NetBSD-6.0/evbppc/INSTALL.html) = d106438243f94987f5d126102e6324f6c1ad274c
SHA512 (NetBSD-6.0/evbppc/INSTALL.html) = d83cf719d3601674fe54b1fff57a8916d342ca9924dd38eaf3f6be43a3d109b85213d4217a5ede1f54734644d1e062437a63f428063cc3bf66166c6131860cc8
RMD160 (NetBSD-6.0/evbppc/INSTALL.html) = 991e52340a15583f9b28560cacf1d10a5e8a6617
MD5 (NetBSD-6.0/evbppc/INSTALL.html) = ab505dcacfa6b669a830b026c43d929d
SHA1 (NetBSD-6.0/evbppc/INSTALL.more) = 375545e5f7e921c1635e3538cfa3f615f58ba82a
SHA512 (NetBSD-6.0/evbppc/INSTALL.more) = d2c28cbf4417afcfcb33f363f3c234403f60c3899cee910767c9dcb7c8f39ef2f1dcc84779e67c5ad386a162ba886c37dfc64d4c7e8a70a3b0fcba1286a5a059
RMD160 (NetBSD-6.0/evbppc/INSTALL.more) = af295999442791298fff266b84d6ab1a92c3cbe1
MD5 (NetBSD-6.0/evbppc/INSTALL.more) = b21374e21e9e0599b62595c1aca77cce
SHA1 (NetBSD-6.0/evbppc/INSTALL.ps) = ec4c228a542dd84139f5eb00ff550be759020a10
SHA512 (NetBSD-6.0/evbppc/INSTALL.ps) = f784251812449ee49a2a2f69b54e20aaad818e0c07c220553da4d1c34676f843c7f60472c750a3dbe952725aad68e5abdf5dc19bb4d3b90cb742d8fcacf1334a
RMD160 (NetBSD-6.0/evbppc/INSTALL.ps) = 525b9a28bed1478b2c2ff4a94da78aaae125bcc7
MD5 (NetBSD-6.0/evbppc/INSTALL.ps) = cfa9f4c9a173353430edc81cfe8a37af
SHA1 (NetBSD-6.0/evbppc/INSTALL.txt) = fc26b26f28fb8eeeb5bb990fc72d9c3d57067350
SHA512 (NetBSD-6.0/evbppc/INSTALL.txt) = 1bd1da4343b0ab19075682a1ea38e33c7a0e11aa2253bae18a19dfb82c99c2b21b7e74e28f0952cbe6c3499a00bf412e998309b52c0c6fffafc868fd5a370e43
RMD160 (NetBSD-6.0/evbppc/INSTALL.txt) = 9d44c454ba9249c02ac948603f1305342bedf9a5
MD5 (NetBSD-6.0/evbppc/INSTALL.txt) = 44b21d90e58f137302ac1c96f78914ef
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-COMPUTEX7750.gz) = 99907a8b29592f20a683481b351d6c3b629ae74d
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-COMPUTEX7750.gz) = d25c300a0bef7432042f22856f899b60993f739522f62d7c1450da81c6774d01769d54f83b1039ed7b97af8ea4cabd954f8017b77b35d0d1a09c47baa09f0cae
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-COMPUTEX7750.gz) = cca19d946d47159fc455f36067dbd3cfee7f0f5f
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-COMPUTEX7750.gz) = 2556834458fa65b30784812d6defbce0
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-COMPUTEXEVB.gz) = b4fcb725319fdf75cb8fbfe3cc385f0546536032
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-COMPUTEXEVB.gz) = 3ae3fa47d5e760564ba5309b5bcbd4ca32ec52bf80ede5764811779acf73c206643cdfa9b0020b300ee70d75af820aa28ba798663ea962b16af935df79c9e4df
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-COMPUTEXEVB.gz) = 697977f6961f8dc0b2204532edc875fc23dd309b
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-COMPUTEXEVB.gz) = a163b83287053245cd74d74a2c66a13b
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-CQREEKSH3.gz) = 3bc3953e940f68f2edadd93f79c8ef7c841c4000
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-CQREEKSH3.gz) = 532a1d96716c85a6b77ebcbeb19560f2e402cbf68a50feafa6452117b477285ca05a0c86dd073bf9c237b6156fa59b42c05fa5c4029739b4fdb277a0c5238e09
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-CQREEKSH3.gz) = dad35f588cb96a63d46bad58b6f4e565402ecc3e
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-CQREEKSH3.gz) = dfdbab13faaee507abaa31e6b1819e30
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-KZSH401.gz) = 6bc4eb9bc72f6ab2412a0f4db523f849955d12e6
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-KZSH401.gz) = 80208e8df1f7c9fdc60ba1558c6f7182d65d6f96f7d57afc3d41887908dfee3598db6e82e827d2a9f14106ec262c4428fcf1a2046dc847c7993cabd9e6e668e6
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-KZSH401.gz) = e16349b9d4c834165922b78175d291af77caa0d9
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/kernel/netbsd-KZSH401.gz) = f95e0f3c93cff1ad0ea5667fdaba435b
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/games.tgz) = 37e15b0950aaaf1af36427ae5dd5866c1879b6b3
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/games.tgz) = e674abb1b1966acbc581a8f8b3e37841f48656e66f3302077cbb2de443487912b99521336ec5dc096fa3c7769b2d7623e1d1720a079407ec1861d32159a72ebc
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/games.tgz) = 36070071e8315d2352be1b3e563ce7ef7625d184
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/games.tgz) = 0016145b46957b95dd158b4441dfb010
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xbase.tgz) = 564bce6dd7c37b5e941e41d50f363c133fdf8f97
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xbase.tgz) = 23b3306254c789331542e63974f9d7cdc54d99ba2cd67cf6a1c97f09afa4f6cf6aa8798e37b7c38f4a55afd1572154f291fa1bf2e2a1fc87165fd6bd7f89fb80
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xbase.tgz) = 476fbd8301bec0795a01552728ada44204d2dbe6
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xbase.tgz) = 9fbbbdb3acfa4ccd9f0b309ea0ed0700
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xcomp.tgz) = 526a278a13833073e6231e879abdec85400ca888
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xcomp.tgz) = d2081a7536e3d7fa27a363afd8f6eb5fc3de4606f11b4857be2f1df59934da9d3d16e0df3a6edf4cdf52c8f579e0e4c06f07de0959b5213c21be1a8fdc2ee134
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xcomp.tgz) = 4a767ff08e9bcf98f886849cabebd1300397256f
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xcomp.tgz) = a23a2d8be9298497b4b6d39fda330bf4
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xetc.tgz) = 93a5cfbbfdc641f91ab363d4b65d1158d71c0cfc
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xetc.tgz) = 85a67734801e81e129a37b73ae62211459fb16d4e52a945ef21a4bb908dd7d1d775233956086dd3d59af45bcea24fbd6f54a2a9175756d242b57e07d10767548
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xetc.tgz) = 6fb055d0e1b6264bf1ec396236d903218832377a
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xetc.tgz) = af954b56e624948145b91aa451d77392
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xfont.tgz) = 6139c05273b6feeecdfcd11cbe0da91256d12a11
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xfont.tgz) = bb849454ede88d39ca60d6c513c833c2b29ea548adedb9f76870839174a3daa838d38a6ad701c78bfc164e690c5ebda9c9d00d4c23ad816df9f1962313afdbf2
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xfont.tgz) = ea8f123f9f78b429b77ff15c924803da1da2068d
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xfont.tgz) = a87cb6769c37ae20dbf0dd70cef93d11
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xserver.tgz) = 6cd4fdb0ff00ef5d89b47e68e538c86e395b2054
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xserver.tgz) = e3b575cab95e297a05a26a03eeff0cf843d665ebdfd103846ad544bef1d04f93aa20ceab09b1dfaae8f59d89ebb1cbef400b7521bdb41f7542f23156540aa4c0
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xserver.tgz) = 7e52b98276b4d097f7babe918e6768ae9d7945db
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/xserver.tgz) = a505ab4ce0801aabe21fe6038b63809f
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-CQREEKSH3.tgz) = 126932a5bc72bd20874c71d76047894c6a7fedb4
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-CQREEKSH3.tgz) = 89727c156d25912c17f801e7cab55d487eff7abc0a9168f490f77313d38eed5a37bde3c0ffbad4db49cc70c335b1147df1408ded59c4b5aeec9a0557a47e7b8f
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-CQREEKSH3.tgz) = 427056f8e924c6346b86a52b73dc8f0e9a98e44b
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-CQREEKSH3.tgz) = 07b32973dbbb2d0a7bf8f397349c032e
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/base.tgz) = 3dc230ebfc3669475edb5f69a61191cf0310b438
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/base.tgz) = e60bf3c63a76db118e7646ddc582150558df14cfde8cb8a6fa74f06bdfcb1f920e6015091ef4763be13072da77e2243b69a8e2739cc8945ec388b8f7fdb2fe50
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/base.tgz) = 5b347ce9da19b214d2069c8dc029cd0448af298d
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/base.tgz) = 79849fe1cbe830a98d8bc0a7db8f4e62
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/comp.tgz) = 882b629fd6de1ec706834199fb45736bb599bbd9
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/comp.tgz) = 700640c27a7453cb2ca1b5c29a2e0558a9cce8f21e4c44e17659bff25c4727aa578f85d3bdbf6da6173c80cac33955ec1731288160b1870fbf92a5df0ed258c1
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/comp.tgz) = 0c3da5c9c5ad1e9d8835040b230e89192f8cc98f
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/comp.tgz) = ae57e406883549f546048ff2480c5707
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/etc.tgz) = 5ef345b014742f576164cd85dc6744b87a4da775
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/etc.tgz) = 998a737bb0964b9bb61798310ae4f979c9e8518f90c482817bcaff1251036e51bdefb39a6b9595f2e313c0df79d44b387c960878582d7721c551882f2a3aba14
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/etc.tgz) = 64d587adda384831dccfee1548c9c09ed5c36f44
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/etc.tgz) = 495f4d62d00649132f218565bddeb50b
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-COMPUTEX7750.tgz) = 364b983f7368d62a6f35a63945026cd8c19c3bf6
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-COMPUTEX7750.tgz) = 8341464294654062caf3c3cfdc1ece9bff1cb6642b9873cebc4520e4ff9a0ed27e5e0106a488ee323d41d77b0bed99d7ad450c35d9d2f805ab18b3e3a941ea71
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-COMPUTEX7750.tgz) = c84e2532d3efe0b47b0db0c7471f9f78b0fd6dde
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-COMPUTEX7750.tgz) = bf341bdb90b335be676be766b8949217
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-COMPUTEXEVB.tgz) = fd664ded3626416bfbabf99bc73dac4a2bac909e
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-COMPUTEXEVB.tgz) = 3adcebf6f3acda9c84666cc1ce39a75302776e0f640cc02024d1ae2a7fb4ef7df14bfbb32e808f4a30fb482fdf181670c87327c0d5ded1145ee0f6bda268e369
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-COMPUTEXEVB.tgz) = f21c49e3f0a4a5fe70070da938296750a222e0f4
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-COMPUTEXEVB.tgz) = 39ababb0c8156e51b2c9b3854286a72f
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-KZSH401.tgz) = 58b48cb56251a87dfc59abd0ae25e8cb0e0ec7bb
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-KZSH401.tgz) = 4c3c9efd94bf1b9080d7355143d40f2796841448a92bf25c6f8b49772dec595915c8d7aa091afffab8df6f2eec40685c5a008ea65abe3ab32f2da6a7000de523
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-KZSH401.tgz) = 5931d576a61a240dacf5dfdaaeab6ac8a435b17e
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/kern-KZSH401.tgz) = d4dfbdffb2514e61fbfb4d7d39962091
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/modules.tgz) = e5078cfd24cfc29249d14a6a6f84de8d27bf1153
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/modules.tgz) = bc00fa4158aa2606925bd8f71698e543a5628739dcaa953ae99c0218fb8d3d1b70c78134cc5554353c612ec7be6fb4efe2f57b28a91a54c87266a9f7450eb38d
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/modules.tgz) = d35c1aa762a049440ce3095ae19f14cece89102f
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/modules.tgz) = 09a7af7e7ec408df3bd4541aa159ee66
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/tests.tgz) = bc89ec5fa64ca7df4c9b4a748cdf2ea437bbbf11
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/tests.tgz) = 011d3773e6363fccfb41d424a51d455d5508cdfb8da226a3a08670c0f5f96a613f0fa5b9818b634d750ab728a8e1546b3c8823b37c91317eb1d0b1cbd85cb0e6
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/tests.tgz) = b5534365eea74dcca99a5d5568c41b3ce79109c2
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/tests.tgz) = 324430decd8312ba548bff34d8197b5a
SHA1 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/text.tgz) = 8cbc530ce508c246c6bfca4abc898f139da4cb39
SHA512 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/text.tgz) = 6cb69d7b7019a82cc1db4326cff681916eee736081c55e08d8a7ff163db75deb91ce2b6feed78fd5775224d3aa24d74d0ca594535447826c03f6613792c9871b
RMD160 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/text.tgz) = 85f073b017e79c9f8069912fe27ac96b7ad84cc8
MD5 (NetBSD-6.0/evbsh3-sh3eb/binary/sets/text.tgz) = 72ac1c599c8f41bbe40cad9c17afb550
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-AP_MS104_SH4.bin.gz) = 156a43383fb461877ca595a24fe433c41bb53505
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-AP_MS104_SH4.bin.gz) = 74da9a5e48ac2fcc998176c4a8ae90f6d82e4269b071af2922210d23075c5c8b9b41ee63e9a4a7eddf5edb98a7b1ff32d07ea78a3b1ca95f533bc69a108ecb4b
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-AP_MS104_SH4.bin.gz) = bb94e484f2bfba2253bc36a6724491e54d94739c
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-AP_MS104_SH4.bin.gz) = 02ffe820f66f7d380b2a35522ea3d438
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-AP_MS104_SH4.gz) = 0056d5cf3c88e73792a091ea2aea6ccdfa8ae908
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-AP_MS104_SH4.gz) = 678c13171474bc55453f9a69467b8d85365e6a61e371e212e6e03a75193170bbd033575a46450b11c31724e56eeae77991a225aca564f1e1ca27e625f0dca191
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-AP_MS104_SH4.gz) = 5303158d3acafe0290bbbbff80db24fb2ca0ec28
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-AP_MS104_SH4.gz) = 30c622231e6b9daac8ed290bdeebd164
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LAN.bin.gz) = 16ae6982b9df3f6eb4a035477fd951edb3d300af
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LAN.bin.gz) = c1ca446d125f788e8a155d7069247c3d63daf8ac9c99da2ad9f3b7d56e11b4491444f0ff0fdb2f4ff3722834d9e811f38621d4aa22d2f15434beb2d7033a0355
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LAN.bin.gz) = 032c5396f55b50ad489e595c91c08b680d045154
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LAN.bin.gz) = 47885af19837e332b23aea912dd25c3c
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LAN.gz) = fbe844ab7e33356c35da0a05e841a4e01e4adfa0
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LAN.gz) = abd3144c85d886bf6ed4be0315fe8a6f027d7c31be94d4252fc005c243abe8ac92de6fbb6a5459c0ead867de119ec83caaf6a078c6a6544dfc8c0a98661ed11a
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LAN.gz) = a2e21620fb48f5eac2314187bbafd65cfd80cbcc
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LAN.gz) = c3681279fd8bb4637c6552ba2be82ea6
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LSR.bin.gz) = 3abb9efd6ce8a3dc1fe36eda3e3544d40b8fadaa
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LSR.bin.gz) = 881ed27f7a5b433a6189fa71445db10f9525396c230b1c28a6bdaca291ac658ea15b479cb0bea31a9f1e97ce3d348ec10fbb590c46543a3beb0284b3b12cacfb
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LSR.bin.gz) = fae500b91381408aa48f2cbbd2f10ff4f7f10744
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LSR.bin.gz) = 5b3c31bbb5f5ef0a547728d1d38fe0ad
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LSR.gz) = e108054983abed92e290fb262e94a13afa9d9863
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LSR.gz) = d7c0232734a6cca9d6e077da1b287dc2dbea5c195efd838de60a90121f3625b56978327fd28630d04249faf4f780a54a9154fdab5bf52226a1e7744d417d9d91
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LSR.gz) = c1f2c381a07c03b405656f033475ca9faede3ba2
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/kernel/netbsd-T_SH7706LSR.gz) = 8c1c70f2f1a89d30184699d3a72d690f
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/games.tgz) = 2040b4af557a62f2ade5fc9fcfe4cab3df88c7e2
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/games.tgz) = 8b5071337dc844042537c24c75660e61c9709cd10cbbeab059a84da6d482260b73365ff3870f85060c130a291b6f02be174d288127ca62ce4d291a6982192317
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/games.tgz) = 28acfa5a131d52df7c157e2edca1dd0f20e774dc
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/games.tgz) = d82c46dbc0021fd4694e464042936052
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xcomp.tgz) = 53aa1e2a910e8eed09dc07f8b724804f53fe5e2f
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xcomp.tgz) = 24645da85419528015cdaefd0c835741fa801392e795b363bdfc39558e75e7f384511c0668d12e2440114b7c543c7532aaec438048f08d35c54c9c7a748f20c0
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xcomp.tgz) = 60a461ffc1f1562274110d65c72a541ee131cd21
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xcomp.tgz) = f1d2650a8d7a4ecd60fda8e1aaf8c09b
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xetc.tgz) = 1636a6fe3e01f6d8a07a19cd204a6807c8caa4c7
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xetc.tgz) = 9cea4ba4af287dd57e591d4269f2f2582010250316f323746ac56b12cefcf2590b93fbb0197c665f61b1df41dcca62d17fb6ad80d4b1feb8ac8265e91e15074f
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xetc.tgz) = 5f68e3f96f7bb9ae07fccdb1d058e4b1307b4b29
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xetc.tgz) = da84d97230787283e215a4ee1c4053ad
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xfont.tgz) = 97439ccf6840b18b1e9885f5504a401bf1a93650
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xfont.tgz) = 3d15541f7d7ae2128f44355bf110415adfa4e5182829259d2c6f7a4217feff99e236d13c5bb7bbeb724ff1a09ac1f58f02bdd6eef528bc416ec3118db6fc7d45
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xfont.tgz) = cdc9fcdb5fd5032803d5bba18db58f3ea6d62c85
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xfont.tgz) = a8466a0477c6c443eafe99e9d1c61131
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/base.tgz) = 125d939841f4c0d2d66af02a7cafcb788e62f49e
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/base.tgz) = 768d8f497c4f735be3799406807852393df11f432b5cc9b1ec69ebe56953f1592a48794629c6cf556eb7bbfed2fd8c0c308b91ca9349256d28ce475a75c44ffb
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/base.tgz) = 94a40312cf09aa8867a6dedb8cdb8b3fbdc00f76
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/base.tgz) = 2f9a062d6116d6c89893fc2d0568521c
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/comp.tgz) = b1c32d11d282303c29854fa152b78be77361dfcb
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/comp.tgz) = 715ed5bd87827f3a84d78b28e855c28ea91944b02c1020551d1fa171239547c7b131f11149fae355ae75aae3059c1cbdbaefaa8a22e6daecf26dd045193bd424
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/comp.tgz) = 25c1503f1c807c8edb6521d85a26c45130b066e1
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/comp.tgz) = 5b355abbc1cd023a56a5d91b06b00883
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/etc.tgz) = 9e2708dd9d1dfabdb39ed6979adb744ab09c72af
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/etc.tgz) = c13a2812104b2f36cf94129d638cdffdd62bb86928eb1478306e205745ea5dc56038e40051da26a94cf56a25a6acd46c86cec2e4e04680715089dc8591d84e0c
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/etc.tgz) = 741003e38321059470fcb7bd727ee4c05add965c
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/etc.tgz) = 474217b3b8eb1c86fba55ed465eb0bff
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/modules.tgz) = a6704d748c69ac31bfc6277d625ba6ecce318deb
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/modules.tgz) = ccccacf090cf2a8dcef8df2c43b99f188d6a8898d30546d2f69341819e9feac6f2fd92e0f14086c45e9afcd462a6e4320ef3fc13bde002f3c697094d2d98144e
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/modules.tgz) = 68a529c0d25d89b79bc283b265fb5fd355328d2f
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/modules.tgz) = d7d56e99bb5ba43313e55abf65901ef0
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/tests.tgz) = 68b5f95807399dc8a592abdd911d015938152e1f
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/tests.tgz) = 2798d89e86057ced5d3e12c7771db2e94a96bd6c09fb49ec5803a56d0c484ea0614a1ea6a8a8647ec4427ee34fc191968e9a4025ca1257d072adcd536bc7c3f2
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/tests.tgz) = 6a3f335d038da0dcb81227f0b909399ddecdb0c3
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/tests.tgz) = 7de6b24e3df249a77cdc4c002259ef30
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-AP_MS104_SH4.tgz) = 74f70ee0502eab199c5d1a5855698772627c4ac7
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-AP_MS104_SH4.tgz) = 76656c9b809e842a7db95daa3a5210f845dbc1a6cc2c05dfcd2d31b4404178ab6c3f9dce8705c4c1f1841ee6d75121fec557ffe25660fa24b492e4ee3f4dde4d
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-AP_MS104_SH4.tgz) = d4e2fc853980b624a04b6a4f895a65ea99c3fa6f
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-AP_MS104_SH4.tgz) = cbf232a6d90bbb23a2ff524dc95d0765
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-T_SH7706LAN.tgz) = bd3d26a06d12126b4bcfd594ec43b5b1b2b141ad
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-T_SH7706LAN.tgz) = 7eaeccc298861c058298a99c1f77c4bdc201a4dd11704fa3033177412893e3934f4f6018ba26604c0da00d1544f610af3e606d40b647d25f4d758ebd3803e3f0
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-T_SH7706LAN.tgz) = 802229436fd22dd4213b8ee964c06d0e264032cc
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-T_SH7706LAN.tgz) = d0f7d8401f8a43421267722847806897
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-T_SH7706LSR.tgz) = 07c0d72f0d893ce0812faf6c6a5c98d7b3b58035
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-T_SH7706LSR.tgz) = 8c724d951a2492d879318720c6e768b86094ffbb935c6107ec0bc42fd225e45ec2ac301bd6bad4e2d8df1b59fdf10f0c68df542369ef57c2e2b4ab537b39efe4
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-T_SH7706LSR.tgz) = ce3c2734ca7c81d61f7794add983ac3d40d6c0f1
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/kern-T_SH7706LSR.tgz) = 9efdd97f65905080d86975228272f5ad
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/text.tgz) = ec93765a5c45be2526dd74721c706cf2349cbcad
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/text.tgz) = ca1d8ae749a5cae5b06f568bd4d6cfb9b0af76848cac05b3d6bd6e1bbecbf84a76213064de51cd4ffff3062b5ee039fb7a99b8336474b39110d298dde616a51f
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/text.tgz) = 55c63f02b460a593dcca374f26d03b9595b6a20a
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/text.tgz) = a7d639318aac350f9f955f97035a8fd4
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xbase.tgz) = 4cb422608de78100a7b3bb2113f334951a6d8602
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xbase.tgz) = acbe1e8859d013a1756d97db9bb194d7ead04db13102f45437f1e3054308471ca2c2498cc0c60c4ac8f6431e58ccf7ad4bae62a9d8e2b9857689af4c0f637de2
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xbase.tgz) = 94f03b6f83e22818d2c98c726a4065680049385a
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xbase.tgz) = 7e4b59a51385b2d05aa15b0d98179e7c
SHA1 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xserver.tgz) = 1423504dda963a794f5710462bd0e3b0ad2a23ba
SHA512 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xserver.tgz) = 36e50002ccd00c6e945a3e9f49f3f210a3bdd5c493aad4932cb81d30ffcc22ab1d11403bfa04859153148bec4a1c57d6f8d2206dc8ee37dfb2ccb23914144c8c
RMD160 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xserver.tgz) = 23d438695d917fcf7ece054eeea841fe02bf268c
MD5 (NetBSD-6.0/evbsh3-sh3el/binary/sets/xserver.tgz) = b526fe22ecba0abf60efad839e21f4d7
SHA1 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.symbols.gz) = 1a3978b4d00f62cb2ff6277dcdbf79fa3f9bf2eb
SHA512 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.symbols.gz) = b504aaa9957cb2b08b5656c1c38e35697d38b3fb8f7fe2dcd2942bc0dcbf7d7fd6cfa1ebd79f536a8cb8e86eb1b69a3676afb3c01f9cdd50c2a61a8fa7f6fcb6
RMD160 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.symbols.gz) = 48c1c3a15270f558fe3b30cbf745a790e51acff6
MD5 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.symbols.gz) = e2e15163e4bae081865da32d01514e6f
SHA1 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.bin.gz) = 873515f4532952c89f5a9f6d91a3f7fa7f948490
SHA512 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.bin.gz) = c19c7a649d4723f255d79005c6da3caa022857e7a7dfee1afdcaf7db9138d777eebec9a2715032e461ec10129403bbdbb25a5bde1eb84efc2a1e7dc044921149
RMD160 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.bin.gz) = 18f832108e5af4f1db1314b3d619950f22660c32
MD5 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.bin.gz) = 5c15217d997a48ae98f97ef5bc1d9f37
SHA1 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.gz) = 86b21fd531bdc7e87b16064ae739b5dcb3c92e81
SHA512 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.gz) = 4e2413b98a091922a7b468169345e4c5090b87264b7232105b0684c2511c78d44a0b0b72f9a8c4ed37bdfb1f24826f87347f883074225fcdd715cadbbf6e8b2b
RMD160 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.gz) = 882aff308401e9ef5a895233e0759a69840da2f7
MD5 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-AP_MS104_SH4_INSTALL.gz) = fa60323505d7a47e47b73dc44e394e47
SHA1 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.symbols.gz) = 349e56359f7f6dec8f0ae526b3a789516a9e4f88
SHA512 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.symbols.gz) = c1c075a4ac229add650e83aa77c6d7efe562793645fe7ca868234f53bd79a112f2198905255f0f03e8bdf7a2e44904146907aaf67da9e41652faa7014686ba08
RMD160 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.symbols.gz) = 6c87f32161c4c2d529cb6a8026cc481dd1d42139
MD5 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.symbols.gz) = b64153e036a04f448159999e4f96b863
SHA1 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.bin.gz) = 23b1af2b9b450007d33c9c68946d6650f8121077
SHA512 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.bin.gz) = 51a3cb3daaba5d099524325af6a4f32ed164bff6118e88c1e4a5f33ff8a69df456833e5a041396179660af053bd0d1c6a43b5f72ad524be4e61ae9e833222c99
RMD160 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.bin.gz) = e5bcd565add34239cf0c2083fad873dd12f4e13d
MD5 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.bin.gz) = 54fde28b5da7660812ff14933dabc902
SHA1 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.gz) = 26f7c3703bac98f6350f5a2423f5d9df4a46bbb3
SHA512 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.gz) = a0a4afa1c58fe9ae74fc03a3780ce16d10f6f00ed6db84e713e2fa51d3b6fb697e281f412eed74d30206edf624ed3bb585b7706252cee2ea37b85de9ed7fe2e4
RMD160 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.gz) = 6e9be1eba2d6db27781fbcc035d794758b4dd444
MD5 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LAN_INSTALL.gz) = d2d661e75be21be7987c548ee785236a
SHA1 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.bin.gz) = 9f3fabe29ffe68e56430d35f894893491a16c169
SHA512 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.bin.gz) = 8ca978143e29fed1916beccb35493521ad70772cc7d4093dd2d2d0bee973e06275abd8167a072db9d79f28d2cba1394d7fd940baaf8d55c84622db60ecce735b
RMD160 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.bin.gz) = 3ecea6d66ed1fc3adb83b5e68d7b5d03ee2ab8ca
MD5 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.bin.gz) = 853aef3ffec3749a8e387c0b8cf9c183
SHA1 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.gz) = 5ea3158a1ddb01af1b0ad97d2788d1e70341faa6
SHA512 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.gz) = 9d2e7c9da144cc79317b2e40f737b09f5320b90c920d7927db332ed3638c68b5cfbdda3f8dbefcb74330e98bcd9163d9003666b842ef74aada7b2b609de25b1a
RMD160 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.gz) = 1f2d464e7b2c50a3c33b0a7d2076ff44627e476d
MD5 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.gz) = 79f028dbbb91d2c182523ef5d1a26b1e
SHA1 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.symbols.gz) = e1393c12d538858fc67c3dbb850099d363802af8
SHA512 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.symbols.gz) = 1a2d80f70a65e38f036a652bcad7774bbc11df1ef51d96868c988dadd25e07c96c2c19d29095b8cebadc6460b2ed5fd141254313d8e3c3f968098cfc136a1b95
RMD160 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.symbols.gz) = 50d94667fa3c0d7f63e9a6e202967ca006f2f00d
MD5 (NetBSD-6.0/evbsh3-sh3el/installation/instkernel/netbsd-T_SH7706LSR_INSTALL.symbols.gz) = bd8daee983f82ce040f96f6f929f273d
SHA1 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-RAMDISK.symbols.gz) = 25a30d5745b993e2102a3786fe1a6f39f470ce45
SHA512 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-RAMDISK.symbols.gz) = 2c6541b1f613443ffd5f7b331a19ec7e8426783232e3fa9241e621ed96ced5e030fcde14e27c1127e336d74f60e21908f2ff4240285df516a5315c37d5e39943
RMD160 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-RAMDISK.symbols.gz) = 5736b4fd760d469af93d177f831ea83e63fcee43
MD5 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-RAMDISK.symbols.gz) = f9ae07f7f20a790a2bba49ed48e9f50b
SHA1 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-GENERIC.gz) = 4c6d1486cc99844354d97e2b1b2ebd6fe061ecc6
SHA512 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-GENERIC.gz) = 6f5e544688a50385b3c519e4929508d663e646457c832cd715d0e05ef5f1ef38a40206c90e95356e17cdef5eede141f10371abab306b534087afca9b65536061
RMD160 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-GENERIC.gz) = 81246f09144ee348e548fa9b7cee0d410ec98f38
MD5 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-GENERIC.gz) = c861b19c9c0ac6c97350564aebddbe77
SHA1 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-RAMDISK.gz) = 95a9a9ba6a613ba5be0afdd1457f7243f01d7efd
SHA512 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-RAMDISK.gz) = 8a36cc3704bc474e9ca4542c8cedcf59dfb93b71a2a5ad44d4c605f059cc515742921eb6de4b64382735a09ae9a72c1fc14e08107480a5090f4319d196a304c3
RMD160 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-RAMDISK.gz) = 5577d47f6a4429aef2e64789a3ea604718275d7c
MD5 (NetBSD-6.0/ews4800mips/binary/kernel/netbsd-RAMDISK.gz) = cc1a04eff2275ed68971e5799b9d8af9
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/ews4800mips/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/ews4800mips/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/kern-GENERIC.tgz) = 81a3b610f413d0a6be35c6de2b7047dc4dd1cbf3
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/kern-GENERIC.tgz) = c9716b08912af531dc5304baac333f34ddc0899bca52dc53185bc48d22217613c03533bed4d70faa01423f0a253d2d03ecc6e1c1704430b28eedaec74110cbab
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/kern-GENERIC.tgz) = 5cd01bbebf5245014d265995f6f96e699dd4bad8
MD5 (NetBSD-6.0/ews4800mips/binary/sets/kern-GENERIC.tgz) = 5a01f783440f099de1bd80cc3abeb52b
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/base.tgz) = 7b4b71623002896a52c0a7c9efe856a0c95de7b1
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/base.tgz) = 97361d892e6c9618569c1022f7b2c2e3485cdf3f1db8b3da1fadd8051a9241dd8baae01c89ffc4b2209651df13be64e8121fa34e1e9373b877add6a4c497e72e
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/base.tgz) = 93372c7049c0a931f1f408d764c3a25600944d7f
MD5 (NetBSD-6.0/ews4800mips/binary/sets/base.tgz) = 0ba9f6bb4c14ce56fb8b4c24d28c5b54
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/comp.tgz) = fb67d7d7410b3489167d90ac9eab8739d5981b69
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/comp.tgz) = 123e4214744b4b319f93de9443465a1185a755ad34053ef82ed74b5b859ab2ba08655528b98110f7b90973c2f6fe6cc4567d83117e215078819d5558e4795829
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/comp.tgz) = 2949a78034cb32eaf732d7127b21990ef697247e
MD5 (NetBSD-6.0/ews4800mips/binary/sets/comp.tgz) = 1eaf1d6e18c59de98043f4e477aef4d2
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/etc.tgz) = 9112e98e3b0e6214758b386e4d49dddff2c53fa8
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/etc.tgz) = f6e54b49bd6585c117474ca5a04c9efecebb63f9c074b5e81fc73bb8c4508d21695f8b6689aa518b0a156c1d0fc3653c5d7487e7f373c5eec679bd7d5b80bbc0
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/etc.tgz) = d797fc90bbdef32b5b9d024d4363acec2abbf787
MD5 (NetBSD-6.0/ews4800mips/binary/sets/etc.tgz) = eb8800ef4e55b63b58066f1f85001efc
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/games.tgz) = e4e2e655f93f3594f4db543839016077023fd1ab
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/games.tgz) = 377f737cae00cc7d9ccc2eae825e68d08a9f6d09474f39294406699ff7d9ca5a0a4347004759b6da9e3a4fb46f3e7ea14c90183bd0ee983cd3625bafb1712e23
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/games.tgz) = 96e4da250eec674b1f3a795026593364798b6a7f
MD5 (NetBSD-6.0/ews4800mips/binary/sets/games.tgz) = cefe2cbb2de8304de93b3fdb34c664ec
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/modules.tgz) = 5179ba1557042dd0edf4b0dcb43feeb51a51fd9c
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/modules.tgz) = c1f026f1e1ef0dfb74347a9f846971ec05d9c9cd27d9480b4dca4105d7f87c39678c8c76557a8bec407b985087c2371f47856cb81912ccf0e8e219d8f2b1ba4f
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/modules.tgz) = 8cacd504f4fce3b4547758a812d28fc41f1c29a4
MD5 (NetBSD-6.0/ews4800mips/binary/sets/modules.tgz) = 17f5259f27b6b9e35e05d38e32a47bc7
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/tests.tgz) = 05f79c3300425fa9c41fb0f6d4b67bbd22143c43
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/tests.tgz) = 5202c4bf6086643686bf4fef3927bedb17262acba8c62bfad92db632d6b545a9ffba8eafbb237f8076beba9da3f496f2da068031745bbad9c042d84b954d7bc2
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/tests.tgz) = 7dceaf484da2d9856cbea453914309af8f75faff
MD5 (NetBSD-6.0/ews4800mips/binary/sets/tests.tgz) = 379a301eaec46b70599b92ff3d69e568
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/text.tgz) = 591abf75de4e52fa3c188ff87fe5fbf28f541766
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/text.tgz) = 0fa9434b5447025c84b3fccbf175b25fe975e88fe711126a9b0625fa070bcfc05c6c962dc59a57ecec38e97e85455911bdc59c411cb6ea9ba2232256191fea4b
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/text.tgz) = 8ccae27fd5067d3e60148abc0751ed5cfb1fb10a
MD5 (NetBSD-6.0/ews4800mips/binary/sets/text.tgz) = 93c20890d37a4c4b7aacb3cdd534070c
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/xbase.tgz) = 5516e5fd661a6aa8886689fbeab06f0d6ef315bc
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/xbase.tgz) = 16f7b97341977b5b7d1629598d8cc250b0fd361f01b2b65afff93dcc97fcd81e3df2b20b42ef37902b64a20fa5a96c48407eac6f0faa8f6467bddfdfa6dcc975
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/xbase.tgz) = 55c47dd7eed35b436fb033ce22281a13d3c016fb
MD5 (NetBSD-6.0/ews4800mips/binary/sets/xbase.tgz) = 949eb603afb4e09b5c53facf3ffcba03
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/xcomp.tgz) = c20568ce72c1ab65b9b2b7d6063d4b330c67e0dd
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/xcomp.tgz) = 5ce8a9f1749abed8c536fabd262ca65297d851a26dd108bf917171ae38dc82dda3ecc4315b5d532e8a393778ed566394c3b7a0cd4d8140f4e9ad275371360504
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/xcomp.tgz) = 138ba52cf36085ea08c8f9022fce482e2b119b1e
MD5 (NetBSD-6.0/ews4800mips/binary/sets/xcomp.tgz) = 9267462e8d885667f784d3e444574bab
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/xetc.tgz) = 0b6040849347e71dc6694909b064c46858ef7267
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/xetc.tgz) = 94cf5fe5f971595d5e17d182aa5fd5cae717fb030336783ec18b34bc054da2b183c184e948b75dda13b1b31468bdadf228682c586e2c1687f4acca2c3545da04
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/xetc.tgz) = f90d3564bda34f368372300e7e0f1dbc716f0761
MD5 (NetBSD-6.0/ews4800mips/binary/sets/xetc.tgz) = 08fbdb1532ed26f606faba4a71052f15
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/xfont.tgz) = 1c83cc716a88a16e480948fb712c55617dac4840
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/xfont.tgz) = b1913d99dc60c680f2fc82aa8d2ff0970cc7c5a712ca412445334cc2b6136993b6fc58c27aecf12d24455bd8ed5e434e4b6e7067b87f8df73960c5b090f4e624
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/xfont.tgz) = a079720a15cdc7659a8efc5a3ef574814993f517
MD5 (NetBSD-6.0/ews4800mips/binary/sets/xfont.tgz) = 1ceec721c52139d9b94d42631fb45b5b
SHA1 (NetBSD-6.0/ews4800mips/binary/sets/xserver.tgz) = 5acda874d4e98472f3bde4d22ee5c76f8e63753d
SHA512 (NetBSD-6.0/ews4800mips/binary/sets/xserver.tgz) = 044bc28fb73dff4c2c50482adc2b69ec1b0b880b40ca25ea528be078b8f1d84e9e797b5238ea5ce0103aed54f5b32d0ff96ab67661156e28e9969aebe33a548b
RMD160 (NetBSD-6.0/ews4800mips/binary/sets/xserver.tgz) = df7ea47ab25dbe8529abb8f5dc1d49cd996d6cdc
MD5 (NetBSD-6.0/ews4800mips/binary/sets/xserver.tgz) = 20191a4f5c5a6cca80c760bfd8836183
SHA1 (NetBSD-6.0/ews4800mips/installation/boot/boot-RAMDISK.gz) = c0d874d928968bc07c4378652d9df32f5ea0367b
SHA512 (NetBSD-6.0/ews4800mips/installation/boot/boot-RAMDISK.gz) = 2cfedab16bbb540c131f2518ed74ee2692d96d8a11d008a11279d4c4b95cdf52fb76baf81823e49d3037f1f5230c1194a473f9e0fd37b36025a14988f6e49de3
RMD160 (NetBSD-6.0/ews4800mips/installation/boot/boot-RAMDISK.gz) = ee3f845cb765fca243be9bb84f17162c7a865304
MD5 (NetBSD-6.0/ews4800mips/installation/boot/boot-RAMDISK.gz) = 42ccd2d2f905ccadd7497ccd4ff91996
SHA1 (NetBSD-6.0/ews4800mips/installation/boot/boot) = df1c0cdcdea86e38dface2a89e2fe26dc7df108f
SHA512 (NetBSD-6.0/ews4800mips/installation/boot/boot) = 6007e2ed816488fc00fca543ab2541f5db95f7a6ad44d4dbde05f4a397fcbf667a41290f7fab5249d6a2f80870a6c04a6bf061d8a25c91b496cbb02d5699b29e
RMD160 (NetBSD-6.0/ews4800mips/installation/boot/boot) = d321a217c04aafeb530f5d3eb29954eda9fb5838
MD5 (NetBSD-6.0/ews4800mips/installation/boot/boot) = f39fbdcdde61cea48a1e10e426480550
SHA1 (NetBSD-6.0/ews4800mips/installation/boot/boot.coff) = e0b297b3888ffdfd4077e67124efa2444194cdc7
SHA512 (NetBSD-6.0/ews4800mips/installation/boot/boot.coff) = 2c465d178b1c8b4688249410c55b6224b06436dc0b42f09028bd1ce7352c2542749a9e3b685b92f670d3401d6b3b207dfbbf549e19ba38bfe3522f09e08f9429
RMD160 (NetBSD-6.0/ews4800mips/installation/boot/boot.coff) = 7e7161d03b7f04a55c35dfd8c34a2a5debed56c7
MD5 (NetBSD-6.0/ews4800mips/installation/boot/boot.coff) = 6bc92c0ddea36ee0816580d8cce6167d
SHA1 (NetBSD-6.0/ews4800mips/installation/floppy/boot1.fs) = e706f91cc1733ab264473d3f1a887285ca79f013
SHA512 (NetBSD-6.0/ews4800mips/installation/floppy/boot1.fs) = b946e58e4d1318d5be27e7bfc943b464309c02c63b936f3d11f57a519988d8847b172f30e2d61dad13bd672325dd2dc9ad502ddc5ed0dd49ca5583c1c51c1fed
RMD160 (NetBSD-6.0/ews4800mips/installation/floppy/boot1.fs) = 16d59bb5fd830cf9551e4184bc338b7395bef5f7
MD5 (NetBSD-6.0/ews4800mips/installation/floppy/boot1.fs) = 1a09cfb0f06ab364b4a5b55754474310
SHA1 (NetBSD-6.0/ews4800mips/installation/floppy/boot2.fs) = 73d13ad2676d8db24c4c53e59be5cd53a70e7dee
SHA512 (NetBSD-6.0/ews4800mips/installation/floppy/boot2.fs) = 4841e1aeb787ece67e3d172c2a614655746beab89f2b09a7566d36979b57d5793d0a308122182e1ca4e052257501f460d0e3d692ef701a1c869041b95e452c14
RMD160 (NetBSD-6.0/ews4800mips/installation/floppy/boot2.fs) = 3fac30b411e6650a18a85afdca71695e7ecf2d4b
MD5 (NetBSD-6.0/ews4800mips/installation/floppy/boot2.fs) = 4f501c25c15a582ed08e3e211c040506
SHA1 (NetBSD-6.0/ews4800mips/installation/floppy/boot3.fs) = 32e2072f8bcef77633e24ec29215c45d2a448c5e
SHA512 (NetBSD-6.0/ews4800mips/installation/floppy/boot3.fs) = 31f812f795fcecef8d5acf1511e0ccf8a498a51a199cef9384e4f197ecb28409fe9dfa7ccc66ea88fdcdfb7c11bbf2a908f3afda7aba16d4adbce172c48a291c
RMD160 (NetBSD-6.0/ews4800mips/installation/floppy/boot3.fs) = baa412f7b18b72d89a2f7468b5d10bf76aa1a437
MD5 (NetBSD-6.0/ews4800mips/installation/floppy/boot3.fs) = e0a8da5b5ed80730a3c4ff8ec316f22b
SHA1 (NetBSD-6.0/ews4800mips/INSTALL.html) = a2b19549f5348d2f1d52980adce6b6dd1c5fa58c
SHA512 (NetBSD-6.0/ews4800mips/INSTALL.html) = f70a62de3ff88606a2193b8bf01036b6770079bc8e5c7912c82e223d81671f8e2bc6dddbcd1022fce4352bccee3a1c19c90a2b1fa72bd12bf7edcf4dd42fb101
RMD160 (NetBSD-6.0/ews4800mips/INSTALL.html) = 500263b32d3fd4f6459496b7e9d250d0d048c79c
MD5 (NetBSD-6.0/ews4800mips/INSTALL.html) = a341d2e552eea54e2887ab795e752d24
SHA1 (NetBSD-6.0/ews4800mips/INSTALL.more) = 112fc313673a7ec94f1c6a7ab3ba79d232e0b866
SHA512 (NetBSD-6.0/ews4800mips/INSTALL.more) = 61df394591fc7b7554c82ced0edec2bd62e7a7e98e0203cc908c6fa48f961062a42148b856f2046f9a3d5262de7d7cc392c15e346b0bea8ec5ef2717760dc30f
RMD160 (NetBSD-6.0/ews4800mips/INSTALL.more) = e92e60dd50dfc2ed354ed18f01cd412f62257367
MD5 (NetBSD-6.0/ews4800mips/INSTALL.more) = f6ff574342525c12f5e039e524ab23cf
SHA1 (NetBSD-6.0/ews4800mips/INSTALL.ps) = a5e622805b2d9d7f96b2aae0b819b7607335d8fc
SHA512 (NetBSD-6.0/ews4800mips/INSTALL.ps) = c8cfeb73d5c4f033877795597dc9edd1dab7902a9753b09127bd9e1875d55ead42746d06b3b31b6ddecc19d876495201736b9ef6c242c357d0bca6d74aac9dc3
RMD160 (NetBSD-6.0/ews4800mips/INSTALL.ps) = 07d50c1fde4cd58f0db754495c39509f69e1dbbe
MD5 (NetBSD-6.0/ews4800mips/INSTALL.ps) = 831c1f71243c24726c508a45dfa97dbc
SHA1 (NetBSD-6.0/ews4800mips/INSTALL.txt) = 6528e1b0d61bf2daab21a1ad8b36dfaf56ef8289
SHA512 (NetBSD-6.0/ews4800mips/INSTALL.txt) = f2f1d34cc0d64c9764604de75197554e49095f305042d0d729da16a0a8edfa4792ce647d3321072af4c1980367270b7097275b32bcacf841409ba4eaa6e97edd
RMD160 (NetBSD-6.0/ews4800mips/INSTALL.txt) = 5311f44e35a9074f47aec723c579e2c04e9928ea
MD5 (NetBSD-6.0/ews4800mips/INSTALL.txt) = 1e6dfee2b6a42e87aa56dbf9f345174c
SHA1 (NetBSD-6.0/hp300/binary/kernel/netbsd-RAMDISK.symbols.gz) = 3e4f61e16b7b76ffbda7ab2b2826ed6e47f55cac
SHA512 (NetBSD-6.0/hp300/binary/kernel/netbsd-RAMDISK.symbols.gz) = 7c0389792e3d8722cd6187619978b413194ec8aebc17fab04ea75f37207c1c74bcf6f44a70beb5ca6712382b745d37d6e1ea3c87e1e93cf03c5891b583f5b513
RMD160 (NetBSD-6.0/hp300/binary/kernel/netbsd-RAMDISK.symbols.gz) = 94f4e2e55d48069e13e05c43ee7688999a9d8488
MD5 (NetBSD-6.0/hp300/binary/kernel/netbsd-RAMDISK.symbols.gz) = a1f4d88bc66a8924f563b00b4c194823
SHA1 (NetBSD-6.0/hp300/binary/kernel/netbsd-GENERIC.gz) = 70a5a051fb4b94008c3598cf7432e40048ac15b5
SHA512 (NetBSD-6.0/hp300/binary/kernel/netbsd-GENERIC.gz) = 342651536f367f4adcae4435afd623af0818e56362ba8f8c9376b49a0f2c57af2ca568edd32954744792904aa1fdf98392d398d75b5c8cc4f30ac456fdedb62d
RMD160 (NetBSD-6.0/hp300/binary/kernel/netbsd-GENERIC.gz) = ce09eea6cbeaf12e2fcd4aabcdb2047e48c686ac
MD5 (NetBSD-6.0/hp300/binary/kernel/netbsd-GENERIC.gz) = d1b8cd8ebde9c0bf97177f7cc8615bbf
SHA1 (NetBSD-6.0/hp300/binary/kernel/netbsd-RAMDISK.gz) = 653c4beb8dc20a0c8d66e9cee85225fc221b5e8e
SHA512 (NetBSD-6.0/hp300/binary/kernel/netbsd-RAMDISK.gz) = 62259ecdcaf08723230e0fcd0c9738fa3b4e42954e114146a7bfc11ab775adf2c36a7924fc9532ffeb5b50da52c9789162b56fd991511922a074f109edf22212
RMD160 (NetBSD-6.0/hp300/binary/kernel/netbsd-RAMDISK.gz) = 634dcaef5fd936621863005e583978fff3f4638c
MD5 (NetBSD-6.0/hp300/binary/kernel/netbsd-RAMDISK.gz) = b4cfe5ee1d9fcffa55595ceceb2cf7dd
SHA1 (NetBSD-6.0/hp300/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/hp300/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/hp300/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/hp300/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/hp300/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/hp300/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/hp300/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/hp300/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/hp300/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/hp300/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/hp300/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/hp300/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/hp300/binary/sets/xbase.tgz) = ffc6dc889560371042dd12a5486a9a2c499b86a9
SHA512 (NetBSD-6.0/hp300/binary/sets/xbase.tgz) = 9aa2ba4d703a13020acf8543d1efcde76b6f0f4b411be20f804709b3802e537be1e651eba469830c4fd4777a379ea9936200a3a3eb4b1a702c04bb81aafcc723
RMD160 (NetBSD-6.0/hp300/binary/sets/xbase.tgz) = 3ecf53e7414ef859a47fe5afc108cf3bfa82fb1d
MD5 (NetBSD-6.0/hp300/binary/sets/xbase.tgz) = a6af002ce158cf72eb5c6a7cde2c0989
SHA1 (NetBSD-6.0/hp300/binary/sets/kern-GENERIC.tgz) = 7ab2f7be438f52926aee3c94e07951b7eb288775
SHA512 (NetBSD-6.0/hp300/binary/sets/kern-GENERIC.tgz) = a4fdc0d2604b8f00790d8a3766bca90391cde8f0f564b7409738f151355f2bc1bc512a25629ee45ba7321a8791f983dda206124a7280ab082f7b255d50b57df2
RMD160 (NetBSD-6.0/hp300/binary/sets/kern-GENERIC.tgz) = 85283b11855a8e69be87b21678328f4a675ca02e
MD5 (NetBSD-6.0/hp300/binary/sets/kern-GENERIC.tgz) = 64374db1afce82099588ba9afc920440
SHA1 (NetBSD-6.0/hp300/binary/sets/base.tgz) = 06b4b615865b0d72c01ca5676c7755af13eaec39
SHA512 (NetBSD-6.0/hp300/binary/sets/base.tgz) = 326b28bb808cfc55a3f25766ba747483c7b1ec585b56ef8deaabbb62e3a6adbee3af61eeadf34ea1b6ff41a379e267bfea7516ee2bd80684e3228913c11f771f
RMD160 (NetBSD-6.0/hp300/binary/sets/base.tgz) = e1bffbda1eb605d10b7eb6e607971d53ba6bf26a
MD5 (NetBSD-6.0/hp300/binary/sets/base.tgz) = 23ecb1b54aa4c4f5d8a3217a1ef01ca0
SHA1 (NetBSD-6.0/hp300/binary/sets/comp.tgz) = 3031c0b6b9903720a6b1fe865c366faf35e35af9
SHA512 (NetBSD-6.0/hp300/binary/sets/comp.tgz) = 37ec2ee919d9d087dd0690cfce76d6be60abc618c3939ab32941ee6c0990c9d5de75f0035f56d2c501d0c5f6e016098bacd502eeaf0849c091dff2ec1ec52cb4
RMD160 (NetBSD-6.0/hp300/binary/sets/comp.tgz) = 2b934c679fb8e8c0f79a2dcec65e9ffb1ee3477f
MD5 (NetBSD-6.0/hp300/binary/sets/comp.tgz) = 5db470910b6991bd50d1003122e50bec
SHA1 (NetBSD-6.0/hp300/binary/sets/etc.tgz) = 6f8edda4999c8a4343fc2bc279241b9550ea490c
SHA512 (NetBSD-6.0/hp300/binary/sets/etc.tgz) = 562d88456cedf5e071d8fa1d93a8dce22f687ffdac77354129d67db4f701cae1af8108a27d9bbd3fae6035f29cea3438d08954e4e784e0a4a103bbc97362179d
RMD160 (NetBSD-6.0/hp300/binary/sets/etc.tgz) = 924ef5e6623a78f068d2be3eb41d7acd6dc767d4
MD5 (NetBSD-6.0/hp300/binary/sets/etc.tgz) = 21fd073754ac797a337928786b03c811
SHA1 (NetBSD-6.0/hp300/binary/sets/modules.tgz) = 83594fc109e081cdb189cc607f9f0e2bdbd17149
SHA512 (NetBSD-6.0/hp300/binary/sets/modules.tgz) = 0741d29e6d7a9ab7bdcd0f7d24cfdce277787bb4e4677035222b34a66570a245279b257b1adb1a0e174af3009ba8f54db7801238b754365efee73af0c1aedef7
RMD160 (NetBSD-6.0/hp300/binary/sets/modules.tgz) = bdf66666ad75847756c3fc123fdcc44ce959e1af
MD5 (NetBSD-6.0/hp300/binary/sets/modules.tgz) = 28892a09bb10b455457f5f581f57f2f0
SHA1 (NetBSD-6.0/hp300/binary/sets/tests.tgz) = a5ef542440872976bdd13a62aefa0d14d61c26c0
SHA512 (NetBSD-6.0/hp300/binary/sets/tests.tgz) = c3f7eec4bdf7dee4746ce7c6f234fb0fd98b97cf4d8eaf6b3ac2f90c3e9ca5cb8cdf35427310135ff7e90f81f584ac3e42c5e7ccfdf2de03c07bca5b5768e078
RMD160 (NetBSD-6.0/hp300/binary/sets/tests.tgz) = 1088670cc9cf63a9232b07ead39467b939007ec5
MD5 (NetBSD-6.0/hp300/binary/sets/tests.tgz) = afa98cf6d10be6bd60a4122ad26f69b6
SHA1 (NetBSD-6.0/hp300/binary/sets/text.tgz) = 34cbcecdf88d3b20397c210adaef51a7aeebdbf1
SHA512 (NetBSD-6.0/hp300/binary/sets/text.tgz) = 9718fde476237be7d04c9e3bc0fef0ac97fe275d6c98cc58f42e4266a7501b8a95e0edf8b46e1a33b4aef67222731978be2fe427d52993a0fc8ea30185411c5c
RMD160 (NetBSD-6.0/hp300/binary/sets/text.tgz) = c96e584c3615a0f4e9eb12a8542b0fa8e60e1f1f
MD5 (NetBSD-6.0/hp300/binary/sets/text.tgz) = 3dc6892231a62bff94ab46648015667f
SHA1 (NetBSD-6.0/hp300/binary/sets/xcomp.tgz) = 55de1ba9512bb95b3678c1569840e740e6f5b556
SHA512 (NetBSD-6.0/hp300/binary/sets/xcomp.tgz) = 83011012f7ce5cd11e58e12f068cfbb5ee4987d40aeae1026503899cdb9d5cf4d4edd6497b2ef0a3e75894d6bcf962fc3492572b51a505fae38ce9d12e4dc36b
RMD160 (NetBSD-6.0/hp300/binary/sets/xcomp.tgz) = d696a85c3da7325eb570111f63f9d3517d47572f
MD5 (NetBSD-6.0/hp300/binary/sets/xcomp.tgz) = 5729b37859a9265fbbfe184d66daaa9b
SHA1 (NetBSD-6.0/hp300/binary/sets/xetc.tgz) = 64f0dab5e32092b4ea7dc9352efd06fd648f5f4b
SHA512 (NetBSD-6.0/hp300/binary/sets/xetc.tgz) = 6fdac8545bf826d9a263cef4354914842986159aae07cd49ffbdda2cefeff6906f3f369141e0f54a213f4530b34160ac420c91c0fa9638354f244c1a625c8ffe
RMD160 (NetBSD-6.0/hp300/binary/sets/xetc.tgz) = b1167b7794746ee9e9828f6b5d3a34efee1e0fe4
MD5 (NetBSD-6.0/hp300/binary/sets/xetc.tgz) = fd2d95f88081017c690d585af9cb52fa
SHA1 (NetBSD-6.0/hp300/binary/sets/xfont.tgz) = 7ff182435282d7ae2377257437bf5f2ead69bf6a
SHA512 (NetBSD-6.0/hp300/binary/sets/xfont.tgz) = aec8b80b466469fd5ab3d851f46cdeccb90b708401d15bd97301001640888b1d6cb50d4b5df95e95516e74e2b644e9b940be085ecd174a7b165ec6557b15fd40
RMD160 (NetBSD-6.0/hp300/binary/sets/xfont.tgz) = 1de0aef2c305fd7dd1ed69d5c14ada0f1b709bc7
MD5 (NetBSD-6.0/hp300/binary/sets/xfont.tgz) = cc29580d8a56cca694e732c04ec99480
SHA1 (NetBSD-6.0/hp300/binary/sets/xserver.tgz) = 2c9dda0fd8c1e7212d505f36e5c89ad2468efe52
SHA512 (NetBSD-6.0/hp300/binary/sets/xserver.tgz) = 412e82089215b7a52c4fd12ad4df99d3c2a0ed39207e41b102163ae7a627fd765f6b3b93cf0d197d2122d50a2196f17ff1bcfbdcc99c44f03e4b4cc80ee95455
RMD160 (NetBSD-6.0/hp300/binary/sets/xserver.tgz) = 133cec24a054a44d8137770ae19faeae2d454a43
MD5 (NetBSD-6.0/hp300/binary/sets/xserver.tgz) = 5f4ca5d0340f3616df2a4514e9bee2e3
SHA1 (NetBSD-6.0/hp300/installation/miniroot/miniroot.fs.gz) = 2bffdc8e8f2ad0c26c428349eb4e04adcaaaf0da
SHA512 (NetBSD-6.0/hp300/installation/miniroot/miniroot.fs.gz) = 6a9135af4681c6b56877b2beff8a5e3e5bd2ae07283f1c78ad2b52d7c56db82779daf57c8c1939aba1dbab03be129372a2d1255f7c23692f548d684bcab2825f
RMD160 (NetBSD-6.0/hp300/installation/miniroot/miniroot.fs.gz) = f81c0d294b7e481ea7f881e4ee89dfe9090e30bd
MD5 (NetBSD-6.0/hp300/installation/miniroot/miniroot.fs.gz) = 6e487378f7e4b0cff281e3d2270e0a80
SHA1 (NetBSD-6.0/hp300/installation/misc/HP-IB.geometry) = 1f0f75de1ee7d8e3181a5d6aa66773677755a884
SHA512 (NetBSD-6.0/hp300/installation/misc/HP-IB.geometry) = 92d43dab00d4075e373ef3bb391cbfc404778b79b12da342eecc67a3c1a6cd1bcdf6da79ce48dc7262fe8b7d58f62317d4c2175ae67e3b7c69bdab6708509e50
RMD160 (NetBSD-6.0/hp300/installation/misc/HP-IB.geometry) = c98d08dbec0a38097b07350201358ce795207abb
MD5 (NetBSD-6.0/hp300/installation/misc/HP-IB.geometry) = 5d37ef7fbd5c7cf0fc9be6ca42e13766
SHA1 (NetBSD-6.0/hp300/installation/misc/SYS_INST) = a49379b1d8420e60551f6255fef597495bfa6ed1
SHA512 (NetBSD-6.0/hp300/installation/misc/SYS_INST) = 6c56496dbd3c76c120c9b4acec61a2e91bce64c9c640cc7d2d1aeb7865f57aff311b71526e82eae87b92d5938ce56264d427dfa02b434ce1859ff179e64cfb6f
RMD160 (NetBSD-6.0/hp300/installation/misc/SYS_INST) = 827a7056af385da9b3d3adaea74d39e819dc0041
MD5 (NetBSD-6.0/hp300/installation/misc/SYS_INST) = f396854e44d13f00a55e32217d907781
SHA1 (NetBSD-6.0/hp300/installation/misc/SYS_UBOOT) = 2939db5d57267e58c09ce33be7a91e4842cd8152
SHA512 (NetBSD-6.0/hp300/installation/misc/SYS_UBOOT) = d609c8e747839742255b0826d0024adb2c9fd6b454e4ab6af71f371c2e437fea20f74eaeec52446d1eeff113edb7a4335a435bd9493d4356f2c6fdabfff6a99b
RMD160 (NetBSD-6.0/hp300/installation/misc/SYS_UBOOT) = 94935b4c88578641336c28fca15951e613869a4d
MD5 (NetBSD-6.0/hp300/installation/misc/SYS_UBOOT) = ca853b39c14fb1cf61fb9a8291e86c0f
SHA1 (NetBSD-6.0/hp300/INSTALL.html) = 8e1c3fbcbc18743ea7a5e551efdb53f84615c5f6
SHA512 (NetBSD-6.0/hp300/INSTALL.html) = 03450311591c740dd9a59f7d59c82d347f216141a7d297b2aceec341ba73afaa39fe7a97c7d929e93e48db1a38d3e8e82fd5df6036ca1387a9fdcb67e5f1860e
RMD160 (NetBSD-6.0/hp300/INSTALL.html) = 74b4c33c7bdf6739b4fab6d95811c4a6e705f441
MD5 (NetBSD-6.0/hp300/INSTALL.html) = b432699922bc91b44886f23df7cf19f3
SHA1 (NetBSD-6.0/hp300/INSTALL.more) = 1cf75d206beed7a64d40e96fbe895f74a70d1f16
SHA512 (NetBSD-6.0/hp300/INSTALL.more) = d4ba1700ff39cce32ce327e810d4e87d40d942209a323beee44697e44e541acb0fe44e66d16277a0cda05cbebd0051e6c7573269868b1e7574e6645c678449ff
RMD160 (NetBSD-6.0/hp300/INSTALL.more) = 89b542e3ef218607f3e65413c3965b7f2333187f
MD5 (NetBSD-6.0/hp300/INSTALL.more) = 624bd59da093361cdaaab5c1de5cecba
SHA1 (NetBSD-6.0/hp300/INSTALL.ps) = 382b1bee832f5d64618348011bbafa088821422f
SHA512 (NetBSD-6.0/hp300/INSTALL.ps) = d986b290afd180501692aca87592c0c71f4a79a2ab774b4b5f28a2fec714547732964125e11ac047afdabbe5f2aec9c44c40d4abbe064716368d012c88a696e6
RMD160 (NetBSD-6.0/hp300/INSTALL.ps) = 484e2cd1d11eccb452fe023cf442225f60381c75
MD5 (NetBSD-6.0/hp300/INSTALL.ps) = 532f28fb7f53079a80b851a5ada9cdf5
SHA1 (NetBSD-6.0/hp300/INSTALL.txt) = 5f6a19f1fcbe957565baf1fae076513a127af37c
SHA512 (NetBSD-6.0/hp300/INSTALL.txt) = 39744f1a32fa5869dac98bdf458d4c43b55857ea7881f3f66e437c496840707fe2a587be8c38bf1d61593c7d3603d231df56f8110be97a65f6a551d4aa75c1f0
RMD160 (NetBSD-6.0/hp300/INSTALL.txt) = 026e4872a0fc52ba4d08907a6907cc72a814c577
MD5 (NetBSD-6.0/hp300/INSTALL.txt) = 025fee7db55908a9ba7a358d3598a074
SHA1 (NetBSD-6.0/hp700/binary/kernel/netbsd-RAMDISK.symbols.gz) = 3b574a0a7fa12dccfe7ec7a74f3c6d44b3de73d8
SHA512 (NetBSD-6.0/hp700/binary/kernel/netbsd-RAMDISK.symbols.gz) = 3de19c62127f98cc12b223d296bf8c69c262c7cc79e4eb4975eb313c96882f4df8d8bc0ff64a7c57b7e7d30896b49935c5475203a369a8f2e134622dcc29d3cd
RMD160 (NetBSD-6.0/hp700/binary/kernel/netbsd-RAMDISK.symbols.gz) = 9c58c704e1a31f946467144840d8eeb207137fbb
MD5 (NetBSD-6.0/hp700/binary/kernel/netbsd-RAMDISK.symbols.gz) = de4a876f7ef0c5aa9c566e4f976723c5
SHA1 (NetBSD-6.0/hp700/binary/kernel/netbsd-GENERIC.gz) = b9b1dddb8855998b36022d03cc9c49ce5e6ed212
SHA512 (NetBSD-6.0/hp700/binary/kernel/netbsd-GENERIC.gz) = 39bcd475712ece695379203c3ec20e2bada75f8b6b6c6f8f80a8826c65ff112a6e2cc8e8c18a5547fa4fa2e2bb7704d6f5b303b2c600f05cadd9dff1da997904
RMD160 (NetBSD-6.0/hp700/binary/kernel/netbsd-GENERIC.gz) = 0b67d5990ab7491b64a591e99053470eadcb13a6
MD5 (NetBSD-6.0/hp700/binary/kernel/netbsd-GENERIC.gz) = 2090d459fd5d513c070c401d72b3ca74
SHA1 (NetBSD-6.0/hp700/binary/kernel/netbsd-RAMDISK.gz) = 38bb542016466c8e1a7c8535520729cc5b9726b8
SHA512 (NetBSD-6.0/hp700/binary/kernel/netbsd-RAMDISK.gz) = 49d747023260ee1b61b59e0a14ff413c97b1daaf5cc20ae916b42d83b99f7438465afed77185c7ef0154ac762edfa93acc8733574fed66e68b58db11c34ce72f
RMD160 (NetBSD-6.0/hp700/binary/kernel/netbsd-RAMDISK.gz) = 268e5e4322fb04246e0a1f8df5788e8530cb88c7
MD5 (NetBSD-6.0/hp700/binary/kernel/netbsd-RAMDISK.gz) = 4175ca432943eb0d4cae87e5eafd3316
SHA1 (NetBSD-6.0/hp700/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/hp700/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/hp700/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/hp700/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/hp700/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/hp700/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/hp700/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/hp700/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/hp700/binary/sets/kern-GENERIC.tgz) = a7b11dacf977418c4a57b400530c123c99447a66
SHA512 (NetBSD-6.0/hp700/binary/sets/kern-GENERIC.tgz) = 071f2e512f843bd91f68e757682e5414600220833b8269be06956d29e61f0383a5f187bdf9a9414a8d64edc6ed9808c8df468fabca174f9079954991c6582300
RMD160 (NetBSD-6.0/hp700/binary/sets/kern-GENERIC.tgz) = be81b5bc0509f3c9a10888b8f7fac2982871320a
MD5 (NetBSD-6.0/hp700/binary/sets/kern-GENERIC.tgz) = 313aa8cbee11f72da2bb608cb35f7a29
SHA1 (NetBSD-6.0/hp700/binary/sets/base.tgz) = 4cc8adc7171679140335cc44ecc8f4f338f47ba5
SHA512 (NetBSD-6.0/hp700/binary/sets/base.tgz) = 48e68c7d28cbc15733f8677d7d139897350c389f7032e78e06d6cbb8b9debc09bde2aa52e7226ef58660da81c238e960e5794689a3f3e19f5eca40f532117542
RMD160 (NetBSD-6.0/hp700/binary/sets/base.tgz) = 3582b7f0c2852854cccad70a2e8f372a7acf0369
MD5 (NetBSD-6.0/hp700/binary/sets/base.tgz) = f8166ed58eec05489e3857d9e09469ad
SHA1 (NetBSD-6.0/hp700/binary/sets/comp.tgz) = d9f4674db21772a521c5eb24e5fbe4ae2656c5d6
SHA512 (NetBSD-6.0/hp700/binary/sets/comp.tgz) = caa488b17380cc2cc8f35ed9703bcac5c778fd597f8f607b26328c58b6b64dec7a784b22335a5c144f31fddf6d03a67d0f2c473ed7498e4f1ada17570f2893aa
RMD160 (NetBSD-6.0/hp700/binary/sets/comp.tgz) = b625cf9eaa395a2844cea44b1904ce9358c862b8
MD5 (NetBSD-6.0/hp700/binary/sets/comp.tgz) = 07a825a8852f558425856a06e5677bfd
SHA1 (NetBSD-6.0/hp700/binary/sets/etc.tgz) = 0576313db33f67f3208c996a1819d19b5f0bd2e3
SHA512 (NetBSD-6.0/hp700/binary/sets/etc.tgz) = 4d1eb5b04fd24f34112c49829b68ea0b7c841fc8b465e363ed4e79d58cd448ddcc6a493bae49cc9a4a4e9bb92314ddc032c5ed060f48b6de60e5045e981809af
RMD160 (NetBSD-6.0/hp700/binary/sets/etc.tgz) = eedfb6a8724ef33f4bc3ef4e08c1ec6f868d548b
MD5 (NetBSD-6.0/hp700/binary/sets/etc.tgz) = 8c95ad14875e9a87428ac7f6cf3a944e
SHA1 (NetBSD-6.0/hp700/binary/sets/games.tgz) = 1be2d6b3d863032e376b31b2b4489a65acd09d94
SHA512 (NetBSD-6.0/hp700/binary/sets/games.tgz) = 322569b0060cc67b16d15a79b8c813e548ea4908ec18fdb91472a4ac5b03bc8abcbba2e5bfed3ed400cbc84ecc82a5edf087f54c06bc682c276a15b516f4dacd
RMD160 (NetBSD-6.0/hp700/binary/sets/games.tgz) = b4d4f8cbddbca828beadbd87ef4f20f1b23d7421
MD5 (NetBSD-6.0/hp700/binary/sets/games.tgz) = 235dc6818c1a24a788301264c292ca2a
SHA1 (NetBSD-6.0/hp700/binary/sets/modules.tgz) = 1b4ecd74bda33a325888994e03e67327a959a381
SHA512 (NetBSD-6.0/hp700/binary/sets/modules.tgz) = 7f77163eb7973139a1e6a2c4aaf3147b94e5f1f977ee642c08b1ad912861d810d953c71ac9db5c5b74c9f9b691a48faeccb4192b099f6ba3cc7fb41093f55533
RMD160 (NetBSD-6.0/hp700/binary/sets/modules.tgz) = e6f669845fcac20f9921602a26bfed3d83c0b453
MD5 (NetBSD-6.0/hp700/binary/sets/modules.tgz) = d5fdd2b3f62edab0ea9e9652c2274b01
SHA1 (NetBSD-6.0/hp700/binary/sets/tests.tgz) = 4a3701895f22ba6d97662a345fa9c14bda977335
SHA512 (NetBSD-6.0/hp700/binary/sets/tests.tgz) = 9ec93c9ec1b2732690f2a3d07cce15569b5ff57a2cabe9489ad28e7b52ccbc0382795267abb6d30a7db9b38f3b7b6a0c6016ea175b1d91f24b50922e3098cc4a
RMD160 (NetBSD-6.0/hp700/binary/sets/tests.tgz) = f62f1798af8d3df9d9653487c543be4411fb2037
MD5 (NetBSD-6.0/hp700/binary/sets/tests.tgz) = 764d8df608cc1e9c23923eb7292f1962
SHA1 (NetBSD-6.0/hp700/binary/sets/text.tgz) = 12fe20e8ab174d0f6bcd8ec6be600ed0e0cca782
SHA512 (NetBSD-6.0/hp700/binary/sets/text.tgz) = b56b29228d79ba548c3ad908f9c81cf7e427bc72e607820438c6154e256f5bb4bd102aab7c31e86b241ecc42f824420aa7969e6ac418250c344eeffd1280abdc
RMD160 (NetBSD-6.0/hp700/binary/sets/text.tgz) = 1e16f9ea1efaee41bc10b045105a1d0dc4f0d17c
MD5 (NetBSD-6.0/hp700/binary/sets/text.tgz) = 09610caef8a531f7c490920d43a7c3d0
SHA1 (NetBSD-6.0/hp700/binary/sets/xbase.tgz) = 0c5e93ea81c0f09063e5c30f42c9d652aa9b68cc
SHA512 (NetBSD-6.0/hp700/binary/sets/xbase.tgz) = 03f48b4ef5dc3f869866f00214d7d5482f03d2d838ceb05e0003484d8835f05b484d998debf24d801721dbb155aa5632a9a1e6d3a830d6faddc552c4500ac3e7
RMD160 (NetBSD-6.0/hp700/binary/sets/xbase.tgz) = 8661bbe49a27e86810bee528b84947926e0a2e65
MD5 (NetBSD-6.0/hp700/binary/sets/xbase.tgz) = 21b4506d3d4f407c33d8c152ca8ef304
SHA1 (NetBSD-6.0/hp700/binary/sets/xcomp.tgz) = a70481db9baba9f60cf23c52e72dac446888df2d
SHA512 (NetBSD-6.0/hp700/binary/sets/xcomp.tgz) = 07c5ba6aa0ce0da2c5c2a3987c7c20226e4ee3158578a7ff70609bbfafc83becdd6a6185fd2c0b4a441f7d92c2cf5be3c767252c11f8fae07068f3ddeb48382e
RMD160 (NetBSD-6.0/hp700/binary/sets/xcomp.tgz) = 2d0179f816fb0a90c931f0733cbfad6dfe3f4d17
MD5 (NetBSD-6.0/hp700/binary/sets/xcomp.tgz) = 84ddb46d08c60fab91553d60f848e21d
SHA1 (NetBSD-6.0/hp700/binary/sets/xetc.tgz) = 0ba461866c920a4fb82c50eede51ec242527a428
SHA512 (NetBSD-6.0/hp700/binary/sets/xetc.tgz) = 1e84618c1e253df70f3801c93b1276ab63df818b231afb21a6c34089fd53cc95c9314a8ff14270d622a231c7fb963473e152f013e1cc2f338e7f231eabf524af
RMD160 (NetBSD-6.0/hp700/binary/sets/xetc.tgz) = a98382606c9775e89131fc737d791b9c4c8334d6
MD5 (NetBSD-6.0/hp700/binary/sets/xetc.tgz) = 626d88e448091f14845bf3a565d330ec
SHA1 (NetBSD-6.0/hp700/binary/sets/xfont.tgz) = b70a4c27223f81cf49a62a85c3addc59b52f4b48
SHA512 (NetBSD-6.0/hp700/binary/sets/xfont.tgz) = 82d94e80e98de78d71d5bae5af20b2755cadbaca48d548b2575fba4a52b2f35e09b71efb4fc8f0f1eadb5282e449807745feeb1d121275e9457eebe93722deb1
RMD160 (NetBSD-6.0/hp700/binary/sets/xfont.tgz) = ec9a08a3c3e1563d8c4bd9a86d0e7fa38dcfa944
MD5 (NetBSD-6.0/hp700/binary/sets/xfont.tgz) = 0bd2c8bc1760f957a2dec6126148ccb3
SHA1 (NetBSD-6.0/hp700/binary/sets/xserver.tgz) = 892de89d8c46c05b44affea346312132bdc9d7b8
SHA512 (NetBSD-6.0/hp700/binary/sets/xserver.tgz) = 18f1705c9e448b34e240f0f215fd618f62af2c1efa59fc9456a28df4fd477a6da26af82f7a99c359edd312abfedfc7b3058f651472a17d339f672db872ade390
RMD160 (NetBSD-6.0/hp700/binary/sets/xserver.tgz) = dadd7b1915e43f92a618b65f739d91fc47cb17f4
MD5 (NetBSD-6.0/hp700/binary/sets/xserver.tgz) = d0ce54a0227edeea0ed0572c32f7382e
SHA1 (NetBSD-6.0/hp700/installation/netinstall.lif) = e4f530ba8005c2a02ab8cc5f6f2f9bc266ed7d27
SHA512 (NetBSD-6.0/hp700/installation/netinstall.lif) = 9cb62f32608d400472276a255cc25c872406cd9a2113a787dd9735759a3d05a1de97c2b53daeeb35b4ab07e3e016aaea0bcc2da165c3c07633ac716900383354
RMD160 (NetBSD-6.0/hp700/installation/netinstall.lif) = 00807d44c30717d4b524a94e209ae8881d9894b8
MD5 (NetBSD-6.0/hp700/installation/netinstall.lif) = 070bcd9b95c8c1f02cb3e233bbfd997d
SHA1 (NetBSD-6.0/hp700/installation/SYSNBSD) = 0458f52af240895100db4e2c15e685a9a8af00da
SHA512 (NetBSD-6.0/hp700/installation/SYSNBSD) = 6acc8321830dbb38cbbbe88bba88c6f0bbdfcfe60380f11fbe92724b0defcb8dbf3ea9207e7522fb90b95a0a401114a751cdef01ddcd5d354cb7ec058c83f98a
RMD160 (NetBSD-6.0/hp700/installation/SYSNBSD) = 2c9d5b050abe290f785eaf9fd1fb746ec0c2d310
MD5 (NetBSD-6.0/hp700/installation/SYSNBSD) = d6cb9aa32688c2b571b6b62f7141d23d
SHA1 (NetBSD-6.0/hp700/INSTALL.html) = caa84e344b51c67646142496becf7b3580bdd769
SHA512 (NetBSD-6.0/hp700/INSTALL.html) = 87e4eaa6077a464832397ddf9eb59ca95949c655d9897fb630899274176ded45b22f4da21ebbfad688515b7cb63a7aca9e5fa8b003d4c81208c0c514ddaa9d3f
RMD160 (NetBSD-6.0/hp700/INSTALL.html) = e46593839ae31aa17259f36ae4f71ee9cf56d317
MD5 (NetBSD-6.0/hp700/INSTALL.html) = 5b63b1d8b0e76a1626ced50e760b6a24
SHA1 (NetBSD-6.0/hp700/INSTALL.more) = bec1e3da9f724159b4a5b52e7bde3948e92573a4
SHA512 (NetBSD-6.0/hp700/INSTALL.more) = 14a618c17e946dfc8c537f48b3f8c225ccf0484acdd1311555c7273a720a867f57b0f72786bf1c5ce874c0a8dc122d6452e476f6b3b5cfd2fdcf3e7b90c20b0a
RMD160 (NetBSD-6.0/hp700/INSTALL.more) = 3cf2ba4b0c27b4e83b2275ffe8b0736c536b40b0
MD5 (NetBSD-6.0/hp700/INSTALL.more) = 211474596cbc940278e9edfbf8ab7d23
SHA1 (NetBSD-6.0/hp700/INSTALL.ps) = 31767f9d14e0018f59f4bf89233f3a546d53234b
SHA512 (NetBSD-6.0/hp700/INSTALL.ps) = 7f1ef8344a35e0aacf8246212f9ef07fe7727048de0a7b24030fdf7698e736152623e6d5d02c32bffbceb3c0337aeac719c523711f838118f0d31316d46c6fb1
RMD160 (NetBSD-6.0/hp700/INSTALL.ps) = bca575e3df8abf0c7f8ff70e614fe52a9c39f645
MD5 (NetBSD-6.0/hp700/INSTALL.ps) = 644eaeb94d45f3183b9cae5f0b39d974
SHA1 (NetBSD-6.0/hp700/INSTALL.txt) = 546cea2f82e5c548de6f6978f569d3dcf65476ce
SHA512 (NetBSD-6.0/hp700/INSTALL.txt) = 4d0d5e6363680e11da17d445fe7156b709e11d28f3226d1374e3612e68ba9c445bd2ec01e4422b523ea2d4197fc8a8b465da0c75401b9560ced43f217931b558
RMD160 (NetBSD-6.0/hp700/INSTALL.txt) = b7536cf8d4f664d03d213978619724a94a16b1d5
MD5 (NetBSD-6.0/hp700/INSTALL.txt) = 241f27fd87ec0932db94a067465d30b0
SHA1 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-JORNADA720.gz) = 7a39f220d6af810e25b4799685aab25a226b9b91
SHA512 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-JORNADA720.gz) = 4b2d4125cf737229264e73bd830c4def0ee33dba71cc5f68abf3e8e936dc89fb02f48a0d821754d2427fe1f72595dd266cd9cfc369c778f50f48573fb2c0bbe1
RMD160 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-JORNADA720.gz) = 129136e4544bb8fcee75e1addd1fee16c522e416
MD5 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-JORNADA720.gz) = 6ed126d914383f7ca0c3269960e885db
SHA1 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-IPAQ.gz) = 7750f05c52fc8460f3ad8b271ae2d9e723c24ebe
SHA512 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-IPAQ.gz) = 485f621f65f80fa02b8fbc08a876aa135d3e4c541ccc6d13ebb3bf119b0628dc5a540ba168a6e908b84422450829dd5a98915a145d629e893930ed40d9e3dfde
RMD160 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-IPAQ.gz) = ee2d243a0fd633e6dac497b28b420b4a8157879f
MD5 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-IPAQ.gz) = 563942f00d685c8618fb5bfa2a6a893b
SHA1 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-WZERO3.gz) = e6f2cede1c69a3f41582f1a8f2a5187948512248
SHA512 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-WZERO3.gz) = 8cc9f93b55a9a64341b266f7d3b8c6d1051f84a36f8aa398b24404d8aa57a16e883b1e7215b7cd235764b4507e04259525333f28369aa95efa39713e17bef07b
RMD160 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-WZERO3.gz) = 536bc6153a0c83339c95e38b87a87bc98e738cb5
MD5 (NetBSD-6.0/hpcarm/binary/kernel/netbsd-WZERO3.gz) = c4092d8e84f02fa75ea88436d404f1b1
SHA1 (NetBSD-6.0/hpcarm/binary/sets/games.tgz) = e1d2bb091d8202962133475bc83edb55230b9008
SHA512 (NetBSD-6.0/hpcarm/binary/sets/games.tgz) = a147fe6d0419d53d5a428802d0bb6c9fa8a84a76c96d7d6d7c0d399e5af3a69240116463549a83d53387b0214ba0207796d7c10228d92a684fe7ff0ece3aec16
RMD160 (NetBSD-6.0/hpcarm/binary/sets/games.tgz) = 2be515c495548ba5135ff42a75a8ebbfc35ea776
MD5 (NetBSD-6.0/hpcarm/binary/sets/games.tgz) = 9059a6f2cd41cb58827709c9e9a1649b
SHA1 (NetBSD-6.0/hpcarm/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/hpcarm/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/hpcarm/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/hpcarm/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/hpcarm/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/hpcarm/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/hpcarm/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/hpcarm/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/hpcarm/binary/sets/kern-IPAQ.tgz) = c2bc2e5d8606cbd122fc825031025e2946f06e40
SHA512 (NetBSD-6.0/hpcarm/binary/sets/kern-IPAQ.tgz) = 52e8c28012dca6e4cc1bab2cce4443a6600a9cd42b3e1e027e715792e794f34baab32300da9ad58fbc23ecb2a22e7787a1d181577317a0dc6fd457ba66f11fc6
RMD160 (NetBSD-6.0/hpcarm/binary/sets/kern-IPAQ.tgz) = 7be149eb7b493fd76dfe0dfc533315458fb0bfea
MD5 (NetBSD-6.0/hpcarm/binary/sets/kern-IPAQ.tgz) = 0d6a168f2d92d06962b581d06f807d46
SHA1 (NetBSD-6.0/hpcarm/binary/sets/base.tgz) = 080c8331106fce332cc5092a8332c2ad603e2596
SHA512 (NetBSD-6.0/hpcarm/binary/sets/base.tgz) = e22024be4e644ae872d6cf538a1210f8ec2aefeadfb47f6497e3892de2b617109a082650e1c1b16006cb84312c1550d3e98610ca63a36a083a073c55a08a3ab2
RMD160 (NetBSD-6.0/hpcarm/binary/sets/base.tgz) = dc0f2bef3b562a81901faf28d738e34168c3b812
MD5 (NetBSD-6.0/hpcarm/binary/sets/base.tgz) = 0e05338048fb790920168e820e0f5771
SHA1 (NetBSD-6.0/hpcarm/binary/sets/comp.tgz) = 8ae72e40483c13b3ea15aebb3efbcb5cedcf921e
SHA512 (NetBSD-6.0/hpcarm/binary/sets/comp.tgz) = 3ee22dfba60196fe19897537c3139dff0b50d1aa6f6c27a024dcdbdc810a9dccd2c32480f73c0ffb0e4dcc4261e03ca6c565a875e291c963011467bc119f292c
RMD160 (NetBSD-6.0/hpcarm/binary/sets/comp.tgz) = 8bbf9baef503fd64c7676821abad991027f42ccf
MD5 (NetBSD-6.0/hpcarm/binary/sets/comp.tgz) = 84a5d47bbb69b46673a2b2a53ffe3854
SHA1 (NetBSD-6.0/hpcarm/binary/sets/etc.tgz) = cbd3aeb7eee9da4d67693adee922e65835ad8908
SHA512 (NetBSD-6.0/hpcarm/binary/sets/etc.tgz) = 96f02aae1ff4fe6096261449dabe1c749ddf179e809b936393bd3b1aea3d708b878ea9c6fcededf0820216bc3f5a4862da04cd9ad5cd4991094bfc309421dc4f
RMD160 (NetBSD-6.0/hpcarm/binary/sets/etc.tgz) = b3f5dbf2f1a092a0fed4f7ff661635387bdfb7cc
MD5 (NetBSD-6.0/hpcarm/binary/sets/etc.tgz) = 940a5d84fd6546eb8ef1fc0fc2af9f12
SHA1 (NetBSD-6.0/hpcarm/binary/sets/kern-JORNADA720.tgz) = fbe60c24963e700dbcded7ed9d6a006fce937c4d
SHA512 (NetBSD-6.0/hpcarm/binary/sets/kern-JORNADA720.tgz) = 0ff4a883f983d768355dab3a124bb1ccc3418d7a61bd8dbfc571089b934153eda2b9bcedf4d655e61c680c60b50772a40322804f86accb96acbd11dbd0159aab
RMD160 (NetBSD-6.0/hpcarm/binary/sets/kern-JORNADA720.tgz) = 010eea6fd1f9645480ed8d96e67a7a87766e6097
MD5 (NetBSD-6.0/hpcarm/binary/sets/kern-JORNADA720.tgz) = 8a1eb406cc9cb944a0dc11ade6d139d9
SHA1 (NetBSD-6.0/hpcarm/binary/sets/kern-WZERO3.tgz) = 5a30317da0f7f2c6f1998a8c53350da39a390bff
SHA512 (NetBSD-6.0/hpcarm/binary/sets/kern-WZERO3.tgz) = 6908dea8428f52a08f481105b602fd2b93318207b5142ccec5104a897a9ad5b10ec25db1a5849074c179e064c7c418cebccbc529609873b031dde5ef1659ec6e
RMD160 (NetBSD-6.0/hpcarm/binary/sets/kern-WZERO3.tgz) = db2ec740804a2c22b4d9d5f05f7b20748206fcca
MD5 (NetBSD-6.0/hpcarm/binary/sets/kern-WZERO3.tgz) = b745687a92452be38b0d2cdb625175e8
SHA1 (NetBSD-6.0/hpcarm/binary/sets/modules.tgz) = cd6c4ccc2c87d573cdb5a92a8806292c5063bd28
SHA512 (NetBSD-6.0/hpcarm/binary/sets/modules.tgz) = 66efd8e33cb118fcb7e6094a55dce87db0b38f03398688e19eaa29a88a61075ca4456cfbcf262e4ad372d1a85d31f1a230e79a09a60d9d0315ee2691f836f408
RMD160 (NetBSD-6.0/hpcarm/binary/sets/modules.tgz) = 1a4f4a21d35c183594080c4c2c6aeb63d0f07ec0
MD5 (NetBSD-6.0/hpcarm/binary/sets/modules.tgz) = 0d07dc5c45004e2442e6da94a40e2082
SHA1 (NetBSD-6.0/hpcarm/binary/sets/tests.tgz) = d77efdefa2b3df022767062ce3e5b37ee33aca9f
SHA512 (NetBSD-6.0/hpcarm/binary/sets/tests.tgz) = b2827bef6684fe45b1ca7dc690a30787a0033b46013281c0d12dc182939a32287579ea0ed6dfdda5ac357cb69a1bfccffcce2cbc991a3b47f0cf714df45f511c
RMD160 (NetBSD-6.0/hpcarm/binary/sets/tests.tgz) = c51c78a413d9b8d240db9965edd6732158319bec
MD5 (NetBSD-6.0/hpcarm/binary/sets/tests.tgz) = a157cb333c867f1efce0c0009db0c596
SHA1 (NetBSD-6.0/hpcarm/binary/sets/text.tgz) = 5e276b4e097bd1ab37a48ba8ac4591d784311152
SHA512 (NetBSD-6.0/hpcarm/binary/sets/text.tgz) = f7d62b4ac928028f37cc0ad711a12e4f5d4a2b77d2e108ce8716194052e6796bf5fb54ce77fa148aa257165f3d71f2e30d562d3dc809052020b85daaa5ba9fcb
RMD160 (NetBSD-6.0/hpcarm/binary/sets/text.tgz) = 806a5e59b90062c827a23ca33be22cda14a2e0c6
MD5 (NetBSD-6.0/hpcarm/binary/sets/text.tgz) = e4ed04614c1a767e663fc73d647f5b7a
SHA1 (NetBSD-6.0/hpcarm/binary/sets/xbase.tgz) = 56daf552141d6161e0b08d5918e351f9cf5f783b
SHA512 (NetBSD-6.0/hpcarm/binary/sets/xbase.tgz) = ccc570735b9636dad693a87964efa1c6d5b14a15e8e9ea28b0ebbcf97db04a671a771d6163a62f78d9979357f7fb0d5543591ac584781cfb47778228a268d723
RMD160 (NetBSD-6.0/hpcarm/binary/sets/xbase.tgz) = 134800ce03d324aa4090e1ce235cc5a6e5ba9f2b
MD5 (NetBSD-6.0/hpcarm/binary/sets/xbase.tgz) = e3afaa495833b5492470b03d432bc773
SHA1 (NetBSD-6.0/hpcarm/binary/sets/xcomp.tgz) = 380508f5e8d94d66e301a6f8fddbddddaf2eb15d
SHA512 (NetBSD-6.0/hpcarm/binary/sets/xcomp.tgz) = 4860c2274b0d4d3b1188d915c90b7e183672527e6734023c78694df3f507d2e920e5af9374353f7581c8f0e22462e8af30aff7e989146de61c0c7b9de7cff93c
RMD160 (NetBSD-6.0/hpcarm/binary/sets/xcomp.tgz) = 2f0f86a40b6887699ea5949b994007a9bb6ad876
MD5 (NetBSD-6.0/hpcarm/binary/sets/xcomp.tgz) = 8f917121a2bbc2df46c6ab3db5f7c276
SHA1 (NetBSD-6.0/hpcarm/binary/sets/xetc.tgz) = 4109069bd859ebc949dede730589472e53e9e73a
SHA512 (NetBSD-6.0/hpcarm/binary/sets/xetc.tgz) = 260ee8565cb3b0fc496ff20856cc718585d36d380a85a5793c0e6502e57b683e132a3904ddb5990ce670ad9069eb999c2a41e427d5bb4ce96dd7f526427dfecf
RMD160 (NetBSD-6.0/hpcarm/binary/sets/xetc.tgz) = 24731afa16c748e4129111adbc446931859ac8b5
MD5 (NetBSD-6.0/hpcarm/binary/sets/xetc.tgz) = c5aa4d6e7bda3c04ff21c49fa7a88349
SHA1 (NetBSD-6.0/hpcarm/binary/sets/xfont.tgz) = 93475dc493faa6b2df35bede901aa170b6f9e9af
SHA512 (NetBSD-6.0/hpcarm/binary/sets/xfont.tgz) = 182e2f9f19c09b4a10c5591d2d9a85a151263ed9773ecc581fd4ab155f5342b364b2fd15c3ce7e82ba9b760ef567f0ccaa8fd51fa4ad3e70dbc63a429ec544b4
RMD160 (NetBSD-6.0/hpcarm/binary/sets/xfont.tgz) = ea5190cce3fb48693746b233b2173595263d0b64
MD5 (NetBSD-6.0/hpcarm/binary/sets/xfont.tgz) = e866b30099f57e267e8be054446c259b
SHA1 (NetBSD-6.0/hpcarm/binary/sets/xserver.tgz) = 33ea7cf47464441ccaa2779544df524406479446
SHA512 (NetBSD-6.0/hpcarm/binary/sets/xserver.tgz) = 718cb878630d6a3d8e2a4c8c0ca427e36f6772017810df557ff9bb0a688445af10f4087662e91631ec07a0a83e3007fa016c485950b3753c50bfdd3cc219ccd1
RMD160 (NetBSD-6.0/hpcarm/binary/sets/xserver.tgz) = 09ce10d8e2254deac07ece3f8625d6c3bf2c4513
MD5 (NetBSD-6.0/hpcarm/binary/sets/xserver.tgz) = a36d535d398b0d6e6e66da91d03ddb6a
SHA1 (NetBSD-6.0/hpcarm/installation/miniroot/miniroot.fs.gz) = add38fbef1c9a42872427fc3ae856cc490dba491
SHA512 (NetBSD-6.0/hpcarm/installation/miniroot/miniroot.fs.gz) = 1ac653efe632dd53dc6bf8f1abda479a4bdecf57b4aadce9b289381a8608601e7444404bac4b4187ea5eb87a8e56affafb06b30c5f27264797b8811b4b4a61ce
RMD160 (NetBSD-6.0/hpcarm/installation/miniroot/miniroot.fs.gz) = d36e893b023bc07b2ae23e65891cd8edd74ae60c
MD5 (NetBSD-6.0/hpcarm/installation/miniroot/miniroot.fs.gz) = d790021f968d07d34a34acb3fdc18a3c
SHA1 (NetBSD-6.0/hpcarm/installation/netbsd-IPAQ.gz) = 640e6d70fa752f3f1e6b415d79bfa26eb524d49f
SHA512 (NetBSD-6.0/hpcarm/installation/netbsd-IPAQ.gz) = 0266c16c91cd0edf3fad56c8806d8736edcb70138486a8596afd2ad58a89846ed319973d4383c6dc240e894a23658effd390012c6bb55d768a3980c5cb9bfa34
RMD160 (NetBSD-6.0/hpcarm/installation/netbsd-IPAQ.gz) = 8b57f335f3ce186509ff88c07bc85c27716b542d
MD5 (NetBSD-6.0/hpcarm/installation/netbsd-IPAQ.gz) = 4963aae14a8da8a218ec0f43110e4329
SHA1 (NetBSD-6.0/hpcarm/installation/hpcboot.exe) = f62f1b2bbf14dd5ee1159d987365122bbfa061d4
SHA512 (NetBSD-6.0/hpcarm/installation/hpcboot.exe) = b1ad5471eec903659a2fe6ff79e631ea40d940e447126a6d8d6b46166dadd7cbbbea4285927968134d2ff493fe66a698d97374febf991ee9ecd4d295f06c92b8
RMD160 (NetBSD-6.0/hpcarm/installation/hpcboot.exe) = fefa32b139995cd0b71c59d09cfb225fce08dbc8
MD5 (NetBSD-6.0/hpcarm/installation/hpcboot.exe) = e8da90c935c719d600bed4f07046950f
SHA1 (NetBSD-6.0/hpcarm/installation/netbsd-JORNADA720.symbols.gz) = 04eac0a968c96a038388c18816b735fd0fda5c2e
SHA512 (NetBSD-6.0/hpcarm/installation/netbsd-JORNADA720.symbols.gz) = 61357b623b75c75df8f2c5318c82f3300183de81003900fb10a1caa566cd9cbc90563277a5d0738b0a7ed52934cfb40d08b4c7af428b06bca90dd907b89673ad
RMD160 (NetBSD-6.0/hpcarm/installation/netbsd-JORNADA720.symbols.gz) = 43b52d529a62cbfeacdae60b54d9e0fece7a1ab0
MD5 (NetBSD-6.0/hpcarm/installation/netbsd-JORNADA720.symbols.gz) = acfc91c707d81e9c84b3fb7f6626f9e5
SHA1 (NetBSD-6.0/hpcarm/installation/netbsd-IPAQ.symbols.gz) = 48c1a0db5c6a2e7d69b1bcf77763e88adadf6373
SHA512 (NetBSD-6.0/hpcarm/installation/netbsd-IPAQ.symbols.gz) = de5ef76b391472a2c18637f36f58a904d693085bfd9108676a1c0476712ab94b72ac41628a65a9c0bfcd5f33c9e141540906f63cfa5614ab8ea2f3f6867b9b1d
RMD160 (NetBSD-6.0/hpcarm/installation/netbsd-IPAQ.symbols.gz) = 383e05d6255afda312b4b81fdaffe4e4a9c83dbf
MD5 (NetBSD-6.0/hpcarm/installation/netbsd-IPAQ.symbols.gz) = 4b4a3474459c93e727e4f20559139464
SHA1 (NetBSD-6.0/hpcarm/installation/netbsd-JORNADA720.gz) = 70818132f816688fa6d74e13544e9e6f07faaddb
SHA512 (NetBSD-6.0/hpcarm/installation/netbsd-JORNADA720.gz) = 0b9b7718b30b88b81fb66cdc32ddee57ac7fd6c90c1ecb6a422f84513df70d594b48badb8b71f00b3ef8d90f544c01439bef4942a51eace3d1fc5dbca1461ced
RMD160 (NetBSD-6.0/hpcarm/installation/netbsd-JORNADA720.gz) = bd193cf9815e6f8a60ca9df77eb1537913c2daa0
MD5 (NetBSD-6.0/hpcarm/installation/netbsd-JORNADA720.gz) = 39c8584c95a8c2197165b102a1548a5c
SHA1 (NetBSD-6.0/hpcarm/installation/netbsd-WZERO3.symbols.gz) = 2afaf89d89ef9c6538960f64b36802beb3e74472
SHA512 (NetBSD-6.0/hpcarm/installation/netbsd-WZERO3.symbols.gz) = 53c9ddac6c56fa5e1c77323483b129b0b1ebe88b558f0e50346d17d54477db1187831c6c8e0e2a231b9473dfcf8d285672f69cbca8197e6fa889f8285836b38f
RMD160 (NetBSD-6.0/hpcarm/installation/netbsd-WZERO3.symbols.gz) = 9dfb61526eb859fd999c8c05b257d436a6d209f3
MD5 (NetBSD-6.0/hpcarm/installation/netbsd-WZERO3.symbols.gz) = c8aa20ace8e4938a05386d8c02f1cb6c
SHA1 (NetBSD-6.0/hpcarm/installation/netbsd-WZERO3.gz) = ded59dfdb078cd978a9a74670d43899ef5186b39
SHA512 (NetBSD-6.0/hpcarm/installation/netbsd-WZERO3.gz) = f93d2e0c80f5b9f8d2f1d851515b2b50f8f518caf7e0b8b71dfdb002a2f97c1da80f7a7d7263228a0d5400267d0be91ee649b558eb4fea8dac4fe07f23b24138
RMD160 (NetBSD-6.0/hpcarm/installation/netbsd-WZERO3.gz) = 99920bf26d5c1697ed030c151c630b125e3b6cf1
MD5 (NetBSD-6.0/hpcarm/installation/netbsd-WZERO3.gz) = c958ba2807a0e03d47f98fa66fcbccef
SHA1 (NetBSD-6.0/hpcarm/INSTALL.html) = 90f13483a6a7cefd8261806546700202df5e89ee
SHA512 (NetBSD-6.0/hpcarm/INSTALL.html) = f7b3a19235b72a4a19c27380e6e22f89e4ce58864a8ef3b492b05aff47cd017a0157adf96bf1d04ef4d6c9b1f4f24980afad91f2644e74fccac06274a35bc75b
RMD160 (NetBSD-6.0/hpcarm/INSTALL.html) = 6228411bb6d6018e59a1fd74f2887aefed5ae8fe
MD5 (NetBSD-6.0/hpcarm/INSTALL.html) = 3d21d24e5a8b5a1ed45afd2f0e7c25e0
SHA1 (NetBSD-6.0/hpcarm/INSTALL.more) = bab711c6d2c3ef110b778fa3bb18a690e3350071
SHA512 (NetBSD-6.0/hpcarm/INSTALL.more) = 85c117ec4b92633391497f62723320999e714a68f0a9b008508fd1b884f77729235859c2ed0ff9539af6615eb452cd87a9bf9f10456ae53fc057b5a3d665dbb5
RMD160 (NetBSD-6.0/hpcarm/INSTALL.more) = 5283c45bcd291d03e3d2f579eaccdc78708a2dc6
MD5 (NetBSD-6.0/hpcarm/INSTALL.more) = fd547cbcd59d803d4f5ef38f619fca0c
SHA1 (NetBSD-6.0/hpcarm/INSTALL.ps) = 6ce0b3584e4b45bbed722cb8bddb66e3deedc88a
SHA512 (NetBSD-6.0/hpcarm/INSTALL.ps) = 3d766bafb31be6edb83b4ada23bd1eec923f7297f6b4cf51adc247a160749c5488a99282ce7e4380b4c27025581901c6f5d7d774c799fdb23f8410c43c8e4b0b
RMD160 (NetBSD-6.0/hpcarm/INSTALL.ps) = e8c042d83705b3b63bcb793d856307b95ed31521
MD5 (NetBSD-6.0/hpcarm/INSTALL.ps) = 1a50f108a265984cdd0f0206c83daa35
SHA1 (NetBSD-6.0/hpcarm/INSTALL.txt) = 06a8b26e2de81570eaaa71593a67b1e4c517590a
SHA512 (NetBSD-6.0/hpcarm/INSTALL.txt) = 4fa7355a943c7e48925ab6db1c281ee9cc7e674095c7eea355d270d15afbb73d45891ce07a4b6104b1fcb0dfaceb3af48db6cb134070077fa4f1bcab453e83eb
RMD160 (NetBSD-6.0/hpcarm/INSTALL.txt) = 3bb0f03d782104ec4792423f3d640e8e7adc29b8
MD5 (NetBSD-6.0/hpcarm/INSTALL.txt) = 68c0576d6d8b54ebfbb0728b7a73438b
SHA1 (NetBSD-6.0/hpcmips/binary/kernel/netbsd-GENERIC.gz) = b038fa721af9faeddea84add1762439db98cf908
SHA512 (NetBSD-6.0/hpcmips/binary/kernel/netbsd-GENERIC.gz) = 6d550d0b1797734c7340d8632930c78be193f397591767e2eed674423ad469dda333206a8f539adfba9d0d81ee70cda7c01de58cfcc4e3ca0440cd6615ba8167
RMD160 (NetBSD-6.0/hpcmips/binary/kernel/netbsd-GENERIC.gz) = a4627fac6aad12dd9a2f04c3aa90fc8c61699d4f
MD5 (NetBSD-6.0/hpcmips/binary/kernel/netbsd-GENERIC.gz) = 6b0e8c6ff949fbea7ff99cbbed4df778
SHA1 (NetBSD-6.0/hpcmips/binary/kernel/netbsd-TX3912.gz) = be377f94448f7026050fbd885d0eeb2ed6a1e84f
SHA512 (NetBSD-6.0/hpcmips/binary/kernel/netbsd-TX3912.gz) = 3855f4e59d507281b41d8363e8d39ea229bd41fd1cc6090704727dcd97bb75f3c265f68a0cfe84a10292e55c7d58a266f4d27705a1fb9218d34e1727210692f0
RMD160 (NetBSD-6.0/hpcmips/binary/kernel/netbsd-TX3912.gz) = 686c703228a5ace7ddb3dcbb232bdb4ece47b6f1
MD5 (NetBSD-6.0/hpcmips/binary/kernel/netbsd-TX3912.gz) = e462f4f6a2cbc6d8eb20e49cbcfed56f
SHA1 (NetBSD-6.0/hpcmips/binary/sets/games.tgz) = 27d9f98fecb3495e51939df9f107e81e527af26f
SHA512 (NetBSD-6.0/hpcmips/binary/sets/games.tgz) = 46bb4e10313cfffaaf4f7fa2157474ef940f94bcc9f84b46f829c7615bbf45e4e01edee7b7efb84bc28561e0712c37cfc737f9c2a0d9a73b67c80c8aa0f3a9af
RMD160 (NetBSD-6.0/hpcmips/binary/sets/games.tgz) = 947feea90ec2698a65d555cb78a29a001b086589
MD5 (NetBSD-6.0/hpcmips/binary/sets/games.tgz) = 355f00978fed8e8da6a251e591be5387
SHA1 (NetBSD-6.0/hpcmips/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/hpcmips/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/hpcmips/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/hpcmips/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/hpcmips/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/hpcmips/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/hpcmips/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/hpcmips/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/hpcmips/binary/sets/xcomp.tgz) = d8ac89f3f6ff20abb0adabc49227afdb74aa2040
SHA512 (NetBSD-6.0/hpcmips/binary/sets/xcomp.tgz) = 4f03213fb061dae9421747209628d06ede6bbb71a7560a9e1a54796b907f1620146a33fdce52974227a21acedcc210b99e15c4fdd36432eba60c1041a92d8cee
RMD160 (NetBSD-6.0/hpcmips/binary/sets/xcomp.tgz) = 6be5e72b9b57c9bdf8ed8ea127e2a73830feeecb
MD5 (NetBSD-6.0/hpcmips/binary/sets/xcomp.tgz) = 59a7ea141587417be38ab1f6056cc780
SHA1 (NetBSD-6.0/hpcmips/binary/sets/xetc.tgz) = b9f6b1a54ffab37b11f4c6e717b481cd087fb609
SHA512 (NetBSD-6.0/hpcmips/binary/sets/xetc.tgz) = 72f35e65f3a90967e250a2006d500619b3824f7f5c3d7494554f153ec5cda5b28c445f28c34a1e4463f40da5855e8d4257726411796967fe37763d90cf7fe461
RMD160 (NetBSD-6.0/hpcmips/binary/sets/xetc.tgz) = 1169c246f91189fd0c4ae86ac27a92f5c2edda3e
MD5 (NetBSD-6.0/hpcmips/binary/sets/xetc.tgz) = 8f3c52cc5af1cafaf7b0d0f884a5b537
SHA1 (NetBSD-6.0/hpcmips/binary/sets/xfont.tgz) = d06848571034f3b7fc975082f1de37b912acb18c
SHA512 (NetBSD-6.0/hpcmips/binary/sets/xfont.tgz) = 66a1013c935bee5e7e5d24325a527522306bcded7ced63790f2e58831b3ed91176a1b9510b2159be63504d5167d48d534919927ffc6e225ae2bb2487f90a9ede
RMD160 (NetBSD-6.0/hpcmips/binary/sets/xfont.tgz) = c9502843a5d4dd7e996e8d158c519cf7f8575cbd
MD5 (NetBSD-6.0/hpcmips/binary/sets/xfont.tgz) = e812899176eac2bf65a6d949ac3f3424
SHA1 (NetBSD-6.0/hpcmips/binary/sets/kern-GENERIC.tgz) = 5a49a4163b7d841104dd4082113ba253ca95f946
SHA512 (NetBSD-6.0/hpcmips/binary/sets/kern-GENERIC.tgz) = 3856b27c3d200e162fb88b7e094e38d843a91f67dadb56b31bd66c18daddd2c0aed58dbea87d43a3d530d3ece68bac562c20d623b613bdd4855fb86bc4794463
RMD160 (NetBSD-6.0/hpcmips/binary/sets/kern-GENERIC.tgz) = cbd699ebfe796b39eee403ac3831765f5871e789
MD5 (NetBSD-6.0/hpcmips/binary/sets/kern-GENERIC.tgz) = 22031cc2c2f3a4a6002fe20cff53264c
SHA1 (NetBSD-6.0/hpcmips/binary/sets/base.tgz) = ff9051176546bd2cacea52fee85fcef472e0fc39
SHA512 (NetBSD-6.0/hpcmips/binary/sets/base.tgz) = 2a770bede2e0f7af3a0d8c2a1f3871bd72c1f67dbe5aed9f6d1cf12b78c8676c819dfd99d3c7cbfc1794c993e8cf059f9e72b764f8d25a5f1ad3406c42169d6d
RMD160 (NetBSD-6.0/hpcmips/binary/sets/base.tgz) = c0d7aa660957624cda33c738d6026870f14fb823
MD5 (NetBSD-6.0/hpcmips/binary/sets/base.tgz) = b79936c5e1c4ff6a86270da2931b1c54
SHA1 (NetBSD-6.0/hpcmips/binary/sets/comp.tgz) = 32515af667967de5ad8dbca2ad993c62c9794b80
SHA512 (NetBSD-6.0/hpcmips/binary/sets/comp.tgz) = 5604e8af6fe737b81128a976cd53cc5f5b287f04de2b21fdbc8bfffa1b521ceae4f7244ad8b98580ca77af7c43b75623e2e9ab8772bd23e8e3c7d297b6c54bfa
RMD160 (NetBSD-6.0/hpcmips/binary/sets/comp.tgz) = b9a5cf74d2c16f7870081a27a3862459c101b690
MD5 (NetBSD-6.0/hpcmips/binary/sets/comp.tgz) = 4265c60c89150dc4c7de32c0c3f4aeac
SHA1 (NetBSD-6.0/hpcmips/binary/sets/etc.tgz) = 68b265bdfcdf09e792bcdfe69656516b44d37b66
SHA512 (NetBSD-6.0/hpcmips/binary/sets/etc.tgz) = af2d02e2aa9a03df02dd8f0a8840e7d185e18b57e8086f0eeb5b7a1349e02433bd7ef1c83f77d5962f90350b71fb4610932e29d723f1cf4488c930048ab43dd1
RMD160 (NetBSD-6.0/hpcmips/binary/sets/etc.tgz) = 3a5d433571b7a9c44fa0ffca6efd6df514101b15
MD5 (NetBSD-6.0/hpcmips/binary/sets/etc.tgz) = 295c108b6f0e7a9dd8f7cc142e87dbd9
SHA1 (NetBSD-6.0/hpcmips/binary/sets/kern-TX3912.tgz) = e1ff4c2bdcfe12d8c26b87801f28970661e5e109
SHA512 (NetBSD-6.0/hpcmips/binary/sets/kern-TX3912.tgz) = 42d069cd3e055b33ffd04b0e99f3db75a45c80cf8cfffb0a4cc08fd3fafb9e9e8c8c6205cf827689a91080c75250c9e753c5c4f1657427287bd567244e29891a
RMD160 (NetBSD-6.0/hpcmips/binary/sets/kern-TX3912.tgz) = d5d7c8d6ab8d3cec287bd0ede9c658f5e196471d
MD5 (NetBSD-6.0/hpcmips/binary/sets/kern-TX3912.tgz) = d9aa987b53c079063b97e947c2b561ed
SHA1 (NetBSD-6.0/hpcmips/binary/sets/modules.tgz) = fd655bd27b90d1cd9ebb2f64a2b744e5f57178bc
SHA512 (NetBSD-6.0/hpcmips/binary/sets/modules.tgz) = c7b1ba107094c528d6edc3c3fa68962882af4fa207bb6b323e6230ebd09f0d7555abc34a4f76bc7f6c7fbf393258089a3a11d9ed7284edd0a9deb44bd8b6b1a7
RMD160 (NetBSD-6.0/hpcmips/binary/sets/modules.tgz) = 13356c7a4e3f09123b31fbd42bcef53ebb10ed61
MD5 (NetBSD-6.0/hpcmips/binary/sets/modules.tgz) = b820dfc2e2d9420fa72bd887eb9ab505
SHA1 (NetBSD-6.0/hpcmips/binary/sets/tests.tgz) = 12c5aab5f1f4ab6124d3cb5e8598359b5f893652
SHA512 (NetBSD-6.0/hpcmips/binary/sets/tests.tgz) = e7656a5f8bb8104576f8218fcbc48c5fdf4d4cfdf407c6805421f1101471c95ac6e1f83444c13ffd1cd82d3b51d4da0d8fc726795b0765363b700d5054cfd553
RMD160 (NetBSD-6.0/hpcmips/binary/sets/tests.tgz) = 8abc82efc30f6bde5b48bd7778a3774fcd993ae4
MD5 (NetBSD-6.0/hpcmips/binary/sets/tests.tgz) = d1cd20123adb795808586fef4825bc29
SHA1 (NetBSD-6.0/hpcmips/binary/sets/text.tgz) = 66f261af8763d48f3c6e62511b0660801b89115a
SHA512 (NetBSD-6.0/hpcmips/binary/sets/text.tgz) = f7428c73f23bfaec5ddad225487c3e569479c537231c2c7ce46869c6debe07a0bf2f139cecd1622230337871a706d6da3891759bd75166d1ae700e85c014dcca
RMD160 (NetBSD-6.0/hpcmips/binary/sets/text.tgz) = 8735b6f7b4e4f04f0db4736cce5cced1b12a4cfb
MD5 (NetBSD-6.0/hpcmips/binary/sets/text.tgz) = 69d9e8e32932f359964186ee95b11c42
SHA1 (NetBSD-6.0/hpcmips/binary/sets/xbase.tgz) = 0299ef346075a8066e40486b3de09de5148940a3
SHA512 (NetBSD-6.0/hpcmips/binary/sets/xbase.tgz) = 077e6b5023eeb6ccbbdd3ebe6b4f677d8a2ac2e76ff235951fe3cfba327f87149e9dd46abedf0963ef465fa829589a802cd0b1ad76a49b983f1766a881c648c4
RMD160 (NetBSD-6.0/hpcmips/binary/sets/xbase.tgz) = 3c7a528dfb109df9525b7f41dd3c1a0ef7ce8bda
MD5 (NetBSD-6.0/hpcmips/binary/sets/xbase.tgz) = 24312b9cd85e739ec15b4b53ad6955f6
SHA1 (NetBSD-6.0/hpcmips/binary/sets/xserver.tgz) = 9faebdfa18d6846a47ee49174a2b510265cf26c0
SHA512 (NetBSD-6.0/hpcmips/binary/sets/xserver.tgz) = 367f12b49e0ab6793075bed49a4830d8d4ca08eca3d93489c90e55201a4e5d7c1454aaea447684309552c5ca3a695abb800fe9da35ec129f1fc11f80f3d904bd
RMD160 (NetBSD-6.0/hpcmips/binary/sets/xserver.tgz) = ece37a2b4b5096401d6add543d13dc136cabe85d
MD5 (NetBSD-6.0/hpcmips/binary/sets/xserver.tgz) = 3c0b67d9f4dc00af6b197724b41b572e
SHA1 (NetBSD-6.0/hpcmips/installation/miniroot/miniroot.fs.gz) = 1e1ce1cba7ab9b704105a3d52f7c5b17773e9ba5
SHA512 (NetBSD-6.0/hpcmips/installation/miniroot/miniroot.fs.gz) = f1746402d265c51ed693fd30961d3dd7da011a30ad47ccdc709a2a9a5dd665dd884f91de6d56ecab947d8c958bb572a4a19b8c446ec6644fd2f3b3096b14e615
RMD160 (NetBSD-6.0/hpcmips/installation/miniroot/miniroot.fs.gz) = 15102213a728cb35bcac2a49f93b496b8f9aa838
MD5 (NetBSD-6.0/hpcmips/installation/miniroot/miniroot.fs.gz) = d910bba08c66b4454242edd78b73cebc
SHA1 (NetBSD-6.0/hpcmips/installation/netbsd-TX3912.gz) = b1cb178b2c1edd93e4d03ab3205c07c701c1a7ce
SHA512 (NetBSD-6.0/hpcmips/installation/netbsd-TX3912.gz) = b982379b9ed3f09895be90c2dcaff7c041e249102bbf72f6ff41bd9100e28990095563db8de6c6dbd40accf3b11f9e0848c2b1bff5dcf0ab9688e65d5229d39a
RMD160 (NetBSD-6.0/hpcmips/installation/netbsd-TX3912.gz) = 6923c07471d3beb90bdb466c1cb76f03ac65ad00
MD5 (NetBSD-6.0/hpcmips/installation/netbsd-TX3912.gz) = 8860562df651f9ca3adf61dd62718dba
SHA1 (NetBSD-6.0/hpcmips/installation/hpcboot.exe) = 70e5234558257f49964fcb28a19f544670791c47
SHA512 (NetBSD-6.0/hpcmips/installation/hpcboot.exe) = bcde10547db4de0623863c93a733706edc1096e2535ff6a7574e86ccab25144c0c653d80ccfb1989eaf685e0b0710eab41f34cc2c45a7bbdc10bb9f845326102
RMD160 (NetBSD-6.0/hpcmips/installation/hpcboot.exe) = ec4e5dfcbb787ba57de55209e12cbbf891c29f7d
MD5 (NetBSD-6.0/hpcmips/installation/hpcboot.exe) = 9c4073db4cb8cc5a534f709731e271bf
SHA1 (NetBSD-6.0/hpcmips/installation/netbsd-TX3912.symbols.gz) = 6aea661d601c817cef6d1283efe84a23e98bfef8
SHA512 (NetBSD-6.0/hpcmips/installation/netbsd-TX3912.symbols.gz) = 0ae88640bdd00cfd0f1d908bab889b72f71173a376d26159e5bd9221e4241a867f4050bb9c349bf4225ecc4b82ae558002b3303e7c03e98dff7f72f57b08a9fc
RMD160 (NetBSD-6.0/hpcmips/installation/netbsd-TX3912.symbols.gz) = 1ddaff9ebe6775a9b1f98e3d592c9e38f944a5f3
MD5 (NetBSD-6.0/hpcmips/installation/netbsd-TX3912.symbols.gz) = ccf4219237474d2c8e2fa02fe33f2e9b
SHA1 (NetBSD-6.0/hpcmips/installation/netbsd.gz) = 75ecc93d0f321a819aca2fafbf58a92651c2ef8e
SHA512 (NetBSD-6.0/hpcmips/installation/netbsd.gz) = 94ef6d49c91ac5ffba39ada8854d3a7be535bdcc7a5121a4941255a30a8c9d68acaebcb1e99ad804360910a45562e3c17b307819ed4f97c1f9a4900c765ac06c
RMD160 (NetBSD-6.0/hpcmips/installation/netbsd.gz) = 10470fb18a31416543c016667bcbcaab1b454297
MD5 (NetBSD-6.0/hpcmips/installation/netbsd.gz) = 1acf8ad6849c07fb17c0646acf7e76c2
SHA1 (NetBSD-6.0/hpcmips/installation/netbsd.symbols.gz) = 8a2bfae03f350dc2c9d83709c8255b8ee4eb9e15
SHA512 (NetBSD-6.0/hpcmips/installation/netbsd.symbols.gz) = c4917814e14d1163139be52927a5a59d422304289239eae15e91495a9964a16e005865f231b3647964bdfed2658d955ef983fccda7775616df92487e10f36b53
RMD160 (NetBSD-6.0/hpcmips/installation/netbsd.symbols.gz) = b353f7752d20b78218082d129e4ed5b6702ee2a2
MD5 (NetBSD-6.0/hpcmips/installation/netbsd.symbols.gz) = 453e0c8ae44500b483151a83e1689a2f
SHA1 (NetBSD-6.0/hpcmips/installation/pbsdboot.exe) = 3b0a8ac9c0d4c6e7fef95970e81f1ae42462f405
SHA512 (NetBSD-6.0/hpcmips/installation/pbsdboot.exe) = 28b6b8b3ac983d5fa9ae9dd28127981367720c3754a88fd0b08e404093c657df3df93bceae63835fdaf021fd608002464411818ef3e9becf100e0e7ffd94fcbc
RMD160 (NetBSD-6.0/hpcmips/installation/pbsdboot.exe) = d713fda3478b92d9dc71347122a1fd7bac6cab9b
MD5 (NetBSD-6.0/hpcmips/installation/pbsdboot.exe) = 5c184fc062316a94d8408adf0884bee2
SHA1 (NetBSD-6.0/hpcmips/installation/pbsdboot1.exe) = e937c9bf2759e58c32523ff19c45feeb5e90fe77
SHA512 (NetBSD-6.0/hpcmips/installation/pbsdboot1.exe) = 65e044958194d15fd58b222dba5470343068ffbe968d43819b11744c711bf75083d1e61df0e433ac846629d8d0ccd14ccd0388b58424a123d4018f9731b6415d
RMD160 (NetBSD-6.0/hpcmips/installation/pbsdboot1.exe) = 9e072b9c080bcfa4213b36fe9d85ac684de46f06
MD5 (NetBSD-6.0/hpcmips/installation/pbsdboot1.exe) = 4d6c719c61a98aae297369c41e3991a8
SHA1 (NetBSD-6.0/hpcmips/INSTALL.html) = 34d4a448b23da2e50007d37ba33938d78baf8b03
SHA512 (NetBSD-6.0/hpcmips/INSTALL.html) = 508dd90bf6b4543b31108ec82b96a8ef765ffd29d2fcfbb463e380bd8cff6909fcad9673a8c7307b8a524347c20c117a94220dc743813028785520f1739b8d89
RMD160 (NetBSD-6.0/hpcmips/INSTALL.html) = 3e111666257f4bcbc22fba2a542559733ab56f28
MD5 (NetBSD-6.0/hpcmips/INSTALL.html) = cea8b96d283b5ddec4b1be43b327a986
SHA1 (NetBSD-6.0/hpcmips/INSTALL.more) = d4dc5bdab669d3e4a97c6e07e3a4e6634a171103
SHA512 (NetBSD-6.0/hpcmips/INSTALL.more) = 90924423731cdaeca599d151e56ef996a4a217432ffe1791b3bf7ad2ea3e025ae031d9e237377d1cd7fbab33a6e8e666909a77bb4867be3aac99340e4292d140
RMD160 (NetBSD-6.0/hpcmips/INSTALL.more) = d8517223e07059e28737972d4cc4ebee900a308f
MD5 (NetBSD-6.0/hpcmips/INSTALL.more) = 6b918c9fc18e093fdae5e4c61fe2fe3e
SHA1 (NetBSD-6.0/hpcmips/INSTALL.ps) = eb7befb368fa0e792db5c250f1cf7cfa9d21d529
SHA512 (NetBSD-6.0/hpcmips/INSTALL.ps) = 42f376e89f871f1a1f223d12816ffa887891818897a591d7b6ac0495c088559eac383449569dda24bfae563991e775c89b55dbec27aac2449ce5363c02581ec9
RMD160 (NetBSD-6.0/hpcmips/INSTALL.ps) = 718659993cfe1a81d63fcb9b7717e551c7213944
MD5 (NetBSD-6.0/hpcmips/INSTALL.ps) = 660ff8cf43dcbeacb63a10a2ac4442cd
SHA1 (NetBSD-6.0/hpcmips/INSTALL.txt) = 93686fc2958437cddbb8e0ab34efa7a153b16939
SHA512 (NetBSD-6.0/hpcmips/INSTALL.txt) = f71addf8728978547ec7d560a4590e392cfb55bd2d06b9987def48a08373fa27fdc7d2d1bfd241d6c9d8b22297b180b434043ffbbe86281b56edb9d4b714a567
RMD160 (NetBSD-6.0/hpcmips/INSTALL.txt) = 0da553b2c528d6766776dfae2005bd6ff531e905
MD5 (NetBSD-6.0/hpcmips/INSTALL.txt) = 3c60451e18599c7e2d8bc6a79c6fc8e9
SHA1 (NetBSD-6.0/hpcsh/binary/kernel/netbsd-GENERIC.gz) = 0a43926fc91b70cb80d51c78394a2fa54fd98ae1
SHA512 (NetBSD-6.0/hpcsh/binary/kernel/netbsd-GENERIC.gz) = 42ba58126c60285a642156c27dfa9d89009b41739a832898f876abe6387edac7c36f08e71028c326ef617a29020e525d779def3e583381645b5a1fc6ba6413b3
RMD160 (NetBSD-6.0/hpcsh/binary/kernel/netbsd-GENERIC.gz) = 0d01f666a4b6fd2df148381e442ee85aa8afc635
MD5 (NetBSD-6.0/hpcsh/binary/kernel/netbsd-GENERIC.gz) = de0c019c15358bd1673689f590e64529
SHA1 (NetBSD-6.0/hpcsh/binary/kernel/netbsd-HPW650PA.gz) = c0359ab834228b4b0af9fd92ec0870cfa424fcab
SHA512 (NetBSD-6.0/hpcsh/binary/kernel/netbsd-HPW650PA.gz) = 6e2d1f10d4eda5725c7293c179162091fb9f04630f14bf37506e49700178b1baee5594ccf70fa3ede42ef702bb512016d9edef3a23cd0777fd40f961a43ba021
RMD160 (NetBSD-6.0/hpcsh/binary/kernel/netbsd-HPW650PA.gz) = f6b944beeaf6fda516165b2196b429db28741e96
MD5 (NetBSD-6.0/hpcsh/binary/kernel/netbsd-HPW650PA.gz) = 7dc95c93d4b9b5a94e566c9f5565bd74
SHA1 (NetBSD-6.0/hpcsh/binary/sets/games.tgz) = 2040b4af557a62f2ade5fc9fcfe4cab3df88c7e2
SHA512 (NetBSD-6.0/hpcsh/binary/sets/games.tgz) = 8b5071337dc844042537c24c75660e61c9709cd10cbbeab059a84da6d482260b73365ff3870f85060c130a291b6f02be174d288127ca62ce4d291a6982192317
RMD160 (NetBSD-6.0/hpcsh/binary/sets/games.tgz) = 28acfa5a131d52df7c157e2edca1dd0f20e774dc
MD5 (NetBSD-6.0/hpcsh/binary/sets/games.tgz) = d82c46dbc0021fd4694e464042936052
SHA1 (NetBSD-6.0/hpcsh/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/hpcsh/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/hpcsh/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/hpcsh/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/hpcsh/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/hpcsh/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/hpcsh/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/hpcsh/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/hpcsh/binary/sets/xcomp.tgz) = 53aa1e2a910e8eed09dc07f8b724804f53fe5e2f
SHA512 (NetBSD-6.0/hpcsh/binary/sets/xcomp.tgz) = 24645da85419528015cdaefd0c835741fa801392e795b363bdfc39558e75e7f384511c0668d12e2440114b7c543c7532aaec438048f08d35c54c9c7a748f20c0
RMD160 (NetBSD-6.0/hpcsh/binary/sets/xcomp.tgz) = 60a461ffc1f1562274110d65c72a541ee131cd21
MD5 (NetBSD-6.0/hpcsh/binary/sets/xcomp.tgz) = f1d2650a8d7a4ecd60fda8e1aaf8c09b
SHA1 (NetBSD-6.0/hpcsh/binary/sets/xetc.tgz) = 1636a6fe3e01f6d8a07a19cd204a6807c8caa4c7
SHA512 (NetBSD-6.0/hpcsh/binary/sets/xetc.tgz) = 9cea4ba4af287dd57e591d4269f2f2582010250316f323746ac56b12cefcf2590b93fbb0197c665f61b1df41dcca62d17fb6ad80d4b1feb8ac8265e91e15074f
RMD160 (NetBSD-6.0/hpcsh/binary/sets/xetc.tgz) = 5f68e3f96f7bb9ae07fccdb1d058e4b1307b4b29
MD5 (NetBSD-6.0/hpcsh/binary/sets/xetc.tgz) = da84d97230787283e215a4ee1c4053ad
SHA1 (NetBSD-6.0/hpcsh/binary/sets/xfont.tgz) = 97439ccf6840b18b1e9885f5504a401bf1a93650
SHA512 (NetBSD-6.0/hpcsh/binary/sets/xfont.tgz) = 3d15541f7d7ae2128f44355bf110415adfa4e5182829259d2c6f7a4217feff99e236d13c5bb7bbeb724ff1a09ac1f58f02bdd6eef528bc416ec3118db6fc7d45
RMD160 (NetBSD-6.0/hpcsh/binary/sets/xfont.tgz) = cdc9fcdb5fd5032803d5bba18db58f3ea6d62c85
MD5 (NetBSD-6.0/hpcsh/binary/sets/xfont.tgz) = a8466a0477c6c443eafe99e9d1c61131
SHA1 (NetBSD-6.0/hpcsh/binary/sets/kern-GENERIC.tgz) = 567f596fd760416c4805a60ccdc39634394b368a
SHA512 (NetBSD-6.0/hpcsh/binary/sets/kern-GENERIC.tgz) = 517a450a02d80e4ae14463b5abcbb4e887d404ac5fd97a2d70092d9d53b6117af874171db26e1b43712b81f6f5f7c23d34ef4d6ee60a911c1288f07ed86675ba
RMD160 (NetBSD-6.0/hpcsh/binary/sets/kern-GENERIC.tgz) = 9f9a50ba10b38fd0a92e4671a61dc968ed206c49
MD5 (NetBSD-6.0/hpcsh/binary/sets/kern-GENERIC.tgz) = 43c86c80541870470199c27117d78bdd
SHA1 (NetBSD-6.0/hpcsh/binary/sets/base.tgz) = 09469ee791c205b746f290108fb4e0fd2d2e157c
SHA512 (NetBSD-6.0/hpcsh/binary/sets/base.tgz) = b4f946a01df921d1b6b83db51e4da06934be1c23d5bd8602d357c4716524a944abd24f9053c130121a05a0a73b06124b1fb8d2e433b461e7122a6f7994b70d4f
RMD160 (NetBSD-6.0/hpcsh/binary/sets/base.tgz) = 9ba69deac2b406c07e38117048bc439865eb8061
MD5 (NetBSD-6.0/hpcsh/binary/sets/base.tgz) = 7a62d7baa2d993d1de3d579d7076fdb2
SHA1 (NetBSD-6.0/hpcsh/binary/sets/comp.tgz) = 2487c8289908b3f46b47dfe16a78d041d159e4a9
SHA512 (NetBSD-6.0/hpcsh/binary/sets/comp.tgz) = 9e89c2f701cd69fc74aeb6ab03304f4bb51d73b6ff6c75b93906d2b05e42639e1da36dbce39a37519070a991bf5efe9ae93b29b6ae20da42fe2b01e951a51cf1
RMD160 (NetBSD-6.0/hpcsh/binary/sets/comp.tgz) = 3843c49d31f966f8d41b2b07407d5952eb788d18
MD5 (NetBSD-6.0/hpcsh/binary/sets/comp.tgz) = 313b69abf3e1a78923f1a7c5c6f171b4
SHA1 (NetBSD-6.0/hpcsh/binary/sets/etc.tgz) = 0ede2593731365cb6501ca69865e1fdcfa72717c
SHA512 (NetBSD-6.0/hpcsh/binary/sets/etc.tgz) = 612da4fc764826c7f374c846d7712c4142d1cd041eb13944c578054fec342a2cb435c8815f9c477afc70a5655e35e9fb9eb1d01ee6c57218c944c6e11f18ffd7
RMD160 (NetBSD-6.0/hpcsh/binary/sets/etc.tgz) = e226671156547333000c60e32b9b24376cc63885
MD5 (NetBSD-6.0/hpcsh/binary/sets/etc.tgz) = 420c5afbc0c1dd6c1d7a085979bab643
SHA1 (NetBSD-6.0/hpcsh/binary/sets/kern-HPW650PA.tgz) = 5534a47bdec7e7e6ce7d5474d3b159ffebe72cd1
SHA512 (NetBSD-6.0/hpcsh/binary/sets/kern-HPW650PA.tgz) = a7d789cafbe113131a7a93700ec06da4256ab5f93513d2ce96e0ed715a7dd99e866a89a907336a761ff53e53c20ca325b4a569198b75ca806e810395638a0ebd
RMD160 (NetBSD-6.0/hpcsh/binary/sets/kern-HPW650PA.tgz) = 2dcb3e3e54ce471cd2185469e4f788aeef5c6539
MD5 (NetBSD-6.0/hpcsh/binary/sets/kern-HPW650PA.tgz) = e146321b865991498743de2b41a3c034
SHA1 (NetBSD-6.0/hpcsh/binary/sets/modules.tgz) = 36ccb3afb83508f713aa2aabd8c66990a7bee974
SHA512 (NetBSD-6.0/hpcsh/binary/sets/modules.tgz) = a9cc7bdcc7e3cb7aab35d329095b0b67cb46b13850f9897433539ce2836b96d38ba69ec95515ae8988a80864f865546dbb619352261f8e413ac21580c78a7be4
RMD160 (NetBSD-6.0/hpcsh/binary/sets/modules.tgz) = 29b45abec58e7be9e801cf6a2be74bcc56df595e
MD5 (NetBSD-6.0/hpcsh/binary/sets/modules.tgz) = 82ca10605dffe603d39ad994e305b53e
SHA1 (NetBSD-6.0/hpcsh/binary/sets/tests.tgz) = d7eb3b59410ff4f341e9dd65c9995e160684fada
SHA512 (NetBSD-6.0/hpcsh/binary/sets/tests.tgz) = 443ff8b932876a515b75ac9b4eb1e225d2d09bd7dfc531f6e19f812e659de4644d0aac96ca74e73aa877bffbaddf13d308905f28cfe23d48a6cef1decf9b820b
RMD160 (NetBSD-6.0/hpcsh/binary/sets/tests.tgz) = 7030e2b09ddbe4bd952524d58f0916b2079a4bb1
MD5 (NetBSD-6.0/hpcsh/binary/sets/tests.tgz) = 9a95f9e73414b2c3ad7acbb2241178fe
SHA1 (NetBSD-6.0/hpcsh/binary/sets/text.tgz) = 028f0e33f75acdbab625786cec067f630aa63e4a
SHA512 (NetBSD-6.0/hpcsh/binary/sets/text.tgz) = 434ec2e2627f4068e75db33cf4ad353ae1c7437c21d4aa0e861f06fb5eb1d85bae5ca15f7c1091017f76adb8b517a6e7860c5a406d157cf4485c1e6acfedfdda
RMD160 (NetBSD-6.0/hpcsh/binary/sets/text.tgz) = 37213e71f1fe2e1d8222d4b64ea6cbbbd2e40a4f
MD5 (NetBSD-6.0/hpcsh/binary/sets/text.tgz) = da797fc1c503a808baea4b873b4575eb
SHA1 (NetBSD-6.0/hpcsh/binary/sets/xbase.tgz) = 3c17cc530069599ced85ef999be6012422973217
SHA512 (NetBSD-6.0/hpcsh/binary/sets/xbase.tgz) = 3dd6421f3248867b6015fcc420b29dcdbe82506c75b7bdd9698dc55b1a93e54d078cc93a68015ccf4bae00dcc96695702acde53f0a47f521116081f44255e205
RMD160 (NetBSD-6.0/hpcsh/binary/sets/xbase.tgz) = f3a5a0df1ced2758ac1016c970034a8a7e9f16c2
MD5 (NetBSD-6.0/hpcsh/binary/sets/xbase.tgz) = fef3052b786c27898ebfcab88012729d
SHA1 (NetBSD-6.0/hpcsh/binary/sets/xserver.tgz) = e518391c0fa08ecefd59c19049aa8b6e05427683
SHA512 (NetBSD-6.0/hpcsh/binary/sets/xserver.tgz) = 85fe307f64bf7b5eb0300293426a377ddde46de15d7cdc7365eea65651d5b952cee395107dc6d73c55b19d43b19c47bac3bdc253d251c5f98858cd2d0aee8cad
RMD160 (NetBSD-6.0/hpcsh/binary/sets/xserver.tgz) = 04274aa41e6f003c492c4d27e8a7182694a98386
MD5 (NetBSD-6.0/hpcsh/binary/sets/xserver.tgz) = 3fb668ea3e2d13574cd78b5a59cc40df
SHA1 (NetBSD-6.0/hpcsh/installation/miniroot/miniroot.fs.gz) = d65c961e5411c64f1110f55036f20c59b3a4ef75
SHA512 (NetBSD-6.0/hpcsh/installation/miniroot/miniroot.fs.gz) = 95a54cd59939533feb467c23085569f35f2ac246a205825acb84ad8c144ae0c6cda888844270f74d452c5bbf4bd7056e9ae62e56f22c598f21a2cb92dcd8499b
RMD160 (NetBSD-6.0/hpcsh/installation/miniroot/miniroot.fs.gz) = 10cd9e01b234129588477829523bed1ea19164f2
MD5 (NetBSD-6.0/hpcsh/installation/miniroot/miniroot.fs.gz) = f5098b166d3da210356a284e484315e5
SHA1 (NetBSD-6.0/hpcsh/installation/netbsd.symbols.gz) = 1a92b94312e5e4cdf5c088342d52040948fefc6c
SHA512 (NetBSD-6.0/hpcsh/installation/netbsd.symbols.gz) = c0ca68315dee6c402bcd7a0180808d4a1c9fab799466d76a8fb460ae110c22fee7ceae5dbac451cb7a0d7ddb59d3af49f3965de88201778c7b14513f2ba1e61a
RMD160 (NetBSD-6.0/hpcsh/installation/netbsd.symbols.gz) = d9c41a7a101b96727c040c0fdd992ba57999fc89
MD5 (NetBSD-6.0/hpcsh/installation/netbsd.symbols.gz) = 67c48072e6d7c01c9c35c60071d1e371
SHA1 (NetBSD-6.0/hpcsh/installation/hpcboot-sh3.exe) = 567ca586f3c3604143e61ee4b644faed466cf60c
SHA512 (NetBSD-6.0/hpcsh/installation/hpcboot-sh3.exe) = bbbda6cb7c671816e07ac1ba9a05f57632f501019f9e7e2b3a373aa2b55bd56d600f7a6353fe433201284b0ff43908bc988f12568d95b251e8b89d5d5656b244
RMD160 (NetBSD-6.0/hpcsh/installation/hpcboot-sh3.exe) = cac5d9056182928594f5c15b2c5d8a6bb54e6654
MD5 (NetBSD-6.0/hpcsh/installation/hpcboot-sh3.exe) = af41643f4b488f837514418c34d06eee
SHA1 (NetBSD-6.0/hpcsh/installation/hpcboot-sh4.exe) = 3b04443ede94fb5345be01440dfac5c600e46683
SHA512 (NetBSD-6.0/hpcsh/installation/hpcboot-sh4.exe) = 60ae23af771f81a472d5ea84b5df140f9b2c2678753ca1878d51157366c9d04253a41b19a6bb5ef08ce755ec82b9fa67f82209fa6a0e49eb507e4ba8118dbc3e
RMD160 (NetBSD-6.0/hpcsh/installation/hpcboot-sh4.exe) = 69a4509e0e173a94a2720f1469a00dd424e19ee4
MD5 (NetBSD-6.0/hpcsh/installation/hpcboot-sh4.exe) = d259e7fcc3c329c95448f96b7fda6920
SHA1 (NetBSD-6.0/hpcsh/installation/netbsd.gz) = 77bc4656ee9ce29522e56beeedf26cdec6df77ef
SHA512 (NetBSD-6.0/hpcsh/installation/netbsd.gz) = 98ba42e9f3d8be016785ef5a807d009f6f8cf46a7c9c4abd8ad860fca5db37bbae22bdfe54bcd8b5963b1d0aa8610fca21402817ceac6b6e59a2c8cf7e986f21
RMD160 (NetBSD-6.0/hpcsh/installation/netbsd.gz) = 705cf57dfbf1a1ee4e3629d1d82a60b8aa8cd6b6
MD5 (NetBSD-6.0/hpcsh/installation/netbsd.gz) = 75c72e83a03c4d9f316f28e81c428c2b
SHA1 (NetBSD-6.0/hpcsh/INSTALL.html) = 8ab73c33b0ecde2e0b0beea372a7996e866553ef
SHA512 (NetBSD-6.0/hpcsh/INSTALL.html) = 7d7f628cd37890c8db79d76dcc9d87984b6ded522ccb2cb5670aa4a40114b1b323cb78b1b7274bb4bd27daefa25c3ee1dd0c6611736222ee546d0dd895cd6e52
RMD160 (NetBSD-6.0/hpcsh/INSTALL.html) = 1e3edb91c943f9e1aa17315a592f5c68e7d66487
MD5 (NetBSD-6.0/hpcsh/INSTALL.html) = 7b6883d6de2349156019286bbf61d14f
SHA1 (NetBSD-6.0/hpcsh/INSTALL.more) = 29137bc092201f10bb2f2725d72299b5e5475b5b
SHA512 (NetBSD-6.0/hpcsh/INSTALL.more) = 6a8737896c43bed9befbd131f8b13fc6070008f13bf9a9e798cc5bcb2210271f92c8b4138b20e716b5f48aeb478bc7ca9be66347d8636fdecd7bebb013b35ebc
RMD160 (NetBSD-6.0/hpcsh/INSTALL.more) = b50c579d04464e3781f5bf7dcb4b69327d79a9a9
MD5 (NetBSD-6.0/hpcsh/INSTALL.more) = 4f08e258b707193fea1797d2915df90f
SHA1 (NetBSD-6.0/hpcsh/INSTALL.ps) = bc5b3e51e6a6f8a09ccac7db7f1ac85366921911
SHA512 (NetBSD-6.0/hpcsh/INSTALL.ps) = 3945a024ba07da3ff29847c33f4061c8395113eaadf053b603b1391c3b994ee23c271442d90b8e9cfe104aee458348422068b5e704f8c89f2cd248e248414f36
RMD160 (NetBSD-6.0/hpcsh/INSTALL.ps) = fadfccdee249158cab774c1bae128f6889f9bbd1
MD5 (NetBSD-6.0/hpcsh/INSTALL.ps) = 01cdb6201ba834eb25dd70d0cd7dd827
SHA1 (NetBSD-6.0/hpcsh/INSTALL.txt) = b291d1be7635867dda1fe5ffb3f24d41b43107ff
SHA512 (NetBSD-6.0/hpcsh/INSTALL.txt) = 7a5ecbef64ad55680dcaa3e58319a5019096e1fef9ff24de2a8401c29b4006520dc724ff740c14f8bf56cc4fca1bf26a5525040c63d0de3559cfdc7f34710e2c
RMD160 (NetBSD-6.0/hpcsh/INSTALL.txt) = a2e4347c690cb982b5097800f74a3729dc6c2628
MD5 (NetBSD-6.0/hpcsh/INSTALL.txt) = 39020135702f1291d2691258a598ef52
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-GENERIC.symbols.gz) = 747df3225196d826d632a750770850510c725c3d
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-GENERIC.symbols.gz) = bf7546bf878dc86c2579e4660adfe0dfc2e7aa06416526ba2ce70425fee806f92bd7e61d2211aa8f302e167f77dd5e61674a1144274be7a3b9a14c13f2a758f0
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-GENERIC.symbols.gz) = e2aef01c044902e8685f72d3f0bba4b0ed6857ce
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-GENERIC.symbols.gz) = 71aca09f9e5bd22fd42144fbadabcda5
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-GENERIC.gz) = d680282ec53e51ed2f2df312764748ef2dbce31f
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-GENERIC.gz) = b9c3bf1bd9a442e5a7e7f515561771436b4c43e44cf454db37c94070de292aa6c00ab0a95957f80f71ffaa517a60dc1651bcb4452853f35fd541cbca2df87139
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-GENERIC.gz) = 60783b17fa32bdef611c8c8c793248f2494f8b4f
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-GENERIC.gz) = 780873f1a06cad1f8e2644f2827e6b8b
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL.symbols.gz) = 986e2b589d49d58105c82c1a3bf8d4292f5ec1c0
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL.symbols.gz) = 0919e2e21da9aace7144d82119ecf8097548ad7179b5f8b495b72b6ad75544c72d462526d7625a2b42fbf9c68a436b2d233ca7e06367f5ddf5be3386d67659ac
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL.symbols.gz) = 4abf1e4dd5c1df9578a256eb7fc9bfc6de174f60
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL.symbols.gz) = acd73bddb752df740099d651da32881d
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL.gz) = 88fdedc8bea8f6e139947e6e4a8922324a2073a2
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL.gz) = 0f13507f05a4dc76faa937fbb9dc8c5107b55e2a7dbcd97a6fd2dede7c2dff1f16ff0b908433b826125a2e1f5c4b9d14b9a809e920b92acd88968588b8129f1a
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL.gz) = 0b7be220f16b74210af24ce8bbb42402909ad847
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL.gz) = 85d626d678c978ad387238dfc76e20a9
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL_XEN3PAE_DOMU.gz) = 82987dd8d3d035c3741ec54e567c797aa527b85e
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL_XEN3PAE_DOMU.gz) = 43c9d6c594b997834e1bfadf63a20fe0735947419a1e86e9e06bc97a7953738219eba2072f9a7b1fa6b23870287be413650145234d2c58166138bb00bd0136a2
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL_XEN3PAE_DOMU.gz) = c0c2edf5e75878f0a07a2d7f971d020a16bd33dc
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL_XEN3PAE_DOMU.gz) = a6a8ef4a7c0da213a9821e7f2d0e18c8
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL_XEN3_DOMU.gz) = 67b61d551fd822e4578b9a8d54bc4a08a39f8b34
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL_XEN3_DOMU.gz) = bfdb3e7ab7a5456434117156bd187d1f111eeedbc7a2d568266fecfeab33bf847fcc4b72e6edffe8015841b87aa04645c1c0442ce521c8965cd7007326b38e78
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL_XEN3_DOMU.gz) = c5378820ecdb9baf20fabded1f5caa4a1d6a3021
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-INSTALL_XEN3_DOMU.gz) = a09e8070066fb225a26b0904422e1704
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-MONOLITHIC.gz) = 205b7f5a5c457a2db4c997c75c02d1f55a0692e7
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-MONOLITHIC.gz) = 68953183a437b13d94af72acbc231553e6aa57f4e3acbe432e710ad452317b974509a9ae83ea0f350d24164de4d1ae6a3d8d4a221e31c61734b67df026b93b03
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-MONOLITHIC.gz) = 11fd88405a800b36b7b030991e632d17992f5e15
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-MONOLITHIC.gz) = f746f0b2fcebfb329cdd45181f2acb74
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3PAE_DOM0.gz) = 6c7943423d026377f0d9308e8aac75d52f08907c
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3PAE_DOM0.gz) = a2fea33cd4aaebb2c2e69d8790ea40b0c2f00e8be3a823ca93274e534a751fe04efdc9d5ed7fc15897c42c9cc7d987ee645f50897cf6ef55383c448885a324f5
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3PAE_DOM0.gz) = 948c072d5d1424578e2ff5bc3bc7bca39364f8e0
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3PAE_DOM0.gz) = 1d4aafba1a470802f7f2c7e10e98ca52
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3PAE_DOMU.gz) = 2ae71d2f1c2b512d124e2d92d431ab5b2c1f05a7
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3PAE_DOMU.gz) = 9b553c5b145d2b29b9032416bc01f5c43bbad80f2c6feb1e96c881e4d1dd77c02740413cefbbd4fb6cd247cb75fd24959d5a60d3210020548236e3a14a82a65a
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3PAE_DOMU.gz) = 2b884a96bb6cdc3059448104c7ddafc31be7c188
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3PAE_DOMU.gz) = dba74b4cb61cf1606556492f6f1b213c
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3_DOM0.gz) = 7b21a86bc29b1573539105b83e6f847124afee15
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3_DOM0.gz) = 9f0a07816ba899e5d003750e1320c6f83c66bee70b71b1356980e64019a4eeff2ee105eea33dc31822c614609782e3143c6575afa7cd9dbfdf05c119ab2cb92c
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3_DOM0.gz) = e976db5aa9e2e5e1109f804913e5a8f075708f85
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3_DOM0.gz) = b8ec7670143458921b7dd71f05251257
SHA1 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3_DOMU.gz) = f528d474640a5bda692eb50859c208f2277ca61f
SHA512 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3_DOMU.gz) = 24f2b7317b51e12e15084d1aad1177a63d37be3ecc3777de4f7b2b93076e4c980a4667c7013db8bdb500e033e745adb665c45aa842d4422494d67beb3f2f4442
RMD160 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3_DOMU.gz) = 373525999967fd37d620cea1b0b38692ea102d64
MD5 (NetBSD-6.0/i386/binary/kernel/netbsd-XEN3_DOMU.gz) = f76afc8d70580acebe22d4daba31df3b
SHA1 (NetBSD-6.0/i386/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/i386/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/i386/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/i386/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/i386/binary/sets/kern-GENERIC.tgz) = dfbdb16d93f42cafac4314e173c3698abb2b2249
SHA512 (NetBSD-6.0/i386/binary/sets/kern-GENERIC.tgz) = 572284b4e2daab363f9f89f6bb9b80428f6d048fbb70120541b7cff881ed7fd8145b2a5dcafb63160a4292d286e57a3191154267167cedb7641f1ee3098072fc
RMD160 (NetBSD-6.0/i386/binary/sets/kern-GENERIC.tgz) = e663f76167f352828f007b3271408d0c4d1f6735
MD5 (NetBSD-6.0/i386/binary/sets/kern-GENERIC.tgz) = 43a3993a20de2d5c58aa9a1aa90dc667
SHA1 (NetBSD-6.0/i386/binary/sets/base.tgz) = 5847817e7b75483373fe51fb6b465404cb74e1ad
SHA512 (NetBSD-6.0/i386/binary/sets/base.tgz) = 927acc412e1372e2205d4c90fe017b2812bbd6036823b6a18bfcc33513279bdf9f02731b874ad73f614da46194bdacd525a8bf9e301c5842f3f6475fa27507be
RMD160 (NetBSD-6.0/i386/binary/sets/base.tgz) = b36be3ba67b7b2209c5baf2fe314ddc2a7254054
MD5 (NetBSD-6.0/i386/binary/sets/base.tgz) = c643ea74ea234056b1f741a561790c1f
SHA1 (NetBSD-6.0/i386/binary/sets/comp.tgz) = 0f22e69dce5b6678e1ab117b875c7b28f2a1224a
SHA512 (NetBSD-6.0/i386/binary/sets/comp.tgz) = 99a73445be3ac1354a3ec94c7b4d231d3fe72bc2bc7cc020c9dd4c74d5fa06f1a7e681f48a2dbf3e90f0d8e611d51e6524883a1a4edf1494607259bc1c49a99b
RMD160 (NetBSD-6.0/i386/binary/sets/comp.tgz) = 1bae0e183624d0dc957e24eb52041e2b3e42492c
MD5 (NetBSD-6.0/i386/binary/sets/comp.tgz) = 5cff8ef6491c3b9270310558ca8360db
SHA1 (NetBSD-6.0/i386/binary/sets/etc.tgz) = 00df7a05fddcdbbf9e0ee430c634fd1e1da2a2db
SHA512 (NetBSD-6.0/i386/binary/sets/etc.tgz) = 0100afa02f2c0f09bcbe513b282b7e23124b94d5d77816f88c7b927e0625bedde3020b0c34e03f26da5d53941cdb0aac1398d2e15c9b4bc6cbb4e46e4bf8786a
RMD160 (NetBSD-6.0/i386/binary/sets/etc.tgz) = 978efa1d91dea561eaa148f4da679a88e1730c7e
MD5 (NetBSD-6.0/i386/binary/sets/etc.tgz) = 3f174862437df05c82293f93a42a52f3
SHA1 (NetBSD-6.0/i386/binary/sets/games.tgz) = eb79bb969242401a2573c8cb861e399f374ffdce
SHA512 (NetBSD-6.0/i386/binary/sets/games.tgz) = 537eaab987a71933b63673477c276cf3b558f4dd46331ea224f5c0f2b0ac387fb591a7880b1055225746b9c46fcb4ff0f8b72382a629abc6af6315ee038c0d0d
RMD160 (NetBSD-6.0/i386/binary/sets/games.tgz) = cfbc8842fcc975fdff71c6fc472396d533d8aa5c
MD5 (NetBSD-6.0/i386/binary/sets/games.tgz) = df096416d676f1a44710711d7c895329
SHA1 (NetBSD-6.0/i386/binary/sets/kern-XEN3PAE_DOM0.tgz) = 714c9b6c0a41973c06ea62a97babb7000ea1245e
SHA512 (NetBSD-6.0/i386/binary/sets/kern-XEN3PAE_DOM0.tgz) = 97529a4c90a36ad65bec57dca59bc80f724981c4358d32195e3b4e614761fae96a79cab552fe0388e7c8e57da581a2b07a84a44d12d32f23e6ce45c771493ccc
RMD160 (NetBSD-6.0/i386/binary/sets/kern-XEN3PAE_DOM0.tgz) = 0171b60434e21cf6d29d38300c05988d19ad1783
MD5 (NetBSD-6.0/i386/binary/sets/kern-XEN3PAE_DOM0.tgz) = 6baad66e20421f3fb0af25e4b95f8a6d
SHA1 (NetBSD-6.0/i386/binary/sets/kern-MONOLITHIC.tgz) = 52d1f2b561194201e67abf1ba2c1726d72e97984
SHA512 (NetBSD-6.0/i386/binary/sets/kern-MONOLITHIC.tgz) = 19aa162f30e144882f2924be4b864821e109991ca16027c4016d54baf812945c8231f423d141091f08ecf8dc9310a74fcbae43b6ea18578afd1bef3346772cf6
RMD160 (NetBSD-6.0/i386/binary/sets/kern-MONOLITHIC.tgz) = ca3d3b0cb59e5d5ba2089b6cf250c62c079b6855
MD5 (NetBSD-6.0/i386/binary/sets/kern-MONOLITHIC.tgz) = 56528935806a8a2ec9f48c6dbfd9c175
SHA1 (NetBSD-6.0/i386/binary/sets/modules.tgz) = a6d22e796d8b1f344b43bcf38fbb79e53ffdce13
SHA512 (NetBSD-6.0/i386/binary/sets/modules.tgz) = 9bbf31b0e1c1192e9ff0ff38f0978c4c2972a82638e63f5f023bd7263ff9386d36cfd666e55e62a6bf3bf3e8c47571252b3508e485ec86830caaa89c66c19fe7
RMD160 (NetBSD-6.0/i386/binary/sets/modules.tgz) = e3dcc86e04adaa6f347d51801422854b35fcedfb
MD5 (NetBSD-6.0/i386/binary/sets/modules.tgz) = 71908d43cbba617e2ff14e3fffa3dae0
SHA1 (NetBSD-6.0/i386/binary/sets/man.tgz) = f526dfe015359f1181af9cf3dc6061960289451f
SHA512 (NetBSD-6.0/i386/binary/sets/man.tgz) = 75dc7781f8531f09bee6f1696561d036b12f08a66a44c4e56b11986e6d91df213ca8b4c6e59da9a2168038c94322c6cfdcdeb52125f5fb03c5b84cdb9522bd7a
RMD160 (NetBSD-6.0/i386/binary/sets/man.tgz) = 1f8551b79ca0342aaf8df0c6f2a77e7337cb7672
MD5 (NetBSD-6.0/i386/binary/sets/man.tgz) = 6114b3fadf0a5e3bf8fde0e7bf0101fc
SHA1 (NetBSD-6.0/i386/binary/sets/kern-XEN3PAE_DOMU.tgz) = 4e24d5f9641c69962f0e9124a3a0082074db7117
SHA512 (NetBSD-6.0/i386/binary/sets/kern-XEN3PAE_DOMU.tgz) = cbbd3c74d74bcd20c8fe2dfa5052b6886a5d25bf8c7aa595638655af790072bd3067e8555201849c2f3d0ec67465f41721c4a8aa14fb060231683de850fd6e99
RMD160 (NetBSD-6.0/i386/binary/sets/kern-XEN3PAE_DOMU.tgz) = 56610e7d2c9e2844ce0516255bbf76b330d23f80
MD5 (NetBSD-6.0/i386/binary/sets/kern-XEN3PAE_DOMU.tgz) = e79dd5d5ca1fc640045efccb106e6a01
SHA1 (NetBSD-6.0/i386/binary/sets/kern-XEN3_DOM0.tgz) = 9c8c6c75aec356a4b6591bd5ce5fcf0e76536c41
SHA512 (NetBSD-6.0/i386/binary/sets/kern-XEN3_DOM0.tgz) = 2acd400b61f87b32e41daf0cb89b512d887f222310cab9be25f2b030480b720042c86abb68ae81ba73875a2c70dfc01184e1e771ad671e98383ce9c8357b4e8e
RMD160 (NetBSD-6.0/i386/binary/sets/kern-XEN3_DOM0.tgz) = ac64d2c3ff6ff7352d43f4f275cfc07a509e40de
MD5 (NetBSD-6.0/i386/binary/sets/kern-XEN3_DOM0.tgz) = 5541acebbb837f4882be868d86bc2b22
SHA1 (NetBSD-6.0/i386/binary/sets/kern-XEN3_DOMU.tgz) = 01432bab26f65297c0739bc3eb4740bd984d67ab
SHA512 (NetBSD-6.0/i386/binary/sets/kern-XEN3_DOMU.tgz) = bd7221f127923ebbd3099589b5bfd332cadbdcf2ad29870b9eee66b8241d1ad50c4d9481647d41d9ceba1e7071735538acd19d458d78e666d2b8ae1b097f993b
RMD160 (NetBSD-6.0/i386/binary/sets/kern-XEN3_DOMU.tgz) = f9c5fd62ac6b11f4cb1ab8eb648f4ee51209ff17
MD5 (NetBSD-6.0/i386/binary/sets/kern-XEN3_DOMU.tgz) = fead463ca4403cdf358b55114421e53e
SHA1 (NetBSD-6.0/i386/binary/sets/tests.tgz) = 65e60f978857f50c36bf4066ad82acee0ef06af4
SHA512 (NetBSD-6.0/i386/binary/sets/tests.tgz) = 714220aecacd2640160768ae00089cb25b5d3b5cf29f715e32257889159699566fe8d9b3bbcfa7bb6e2864ee1945dbfc61b48a66e20463025774616d91cc5392
RMD160 (NetBSD-6.0/i386/binary/sets/tests.tgz) = 83c54132cfa9e7b0b5352466070bba098576ff86
MD5 (NetBSD-6.0/i386/binary/sets/tests.tgz) = 1cf15f457ca67156d7dad0f7723ed259
SHA1 (NetBSD-6.0/i386/binary/sets/text.tgz) = 0e426ee26f13d2523798fa8c09e934725087dc5e
SHA512 (NetBSD-6.0/i386/binary/sets/text.tgz) = 2809f127e3baa3e6ebdb181c31620b68d01e90208bd191db99aa9657c91db9a97b061cdfdc2d7a39fbedf2288067f2ed885d6515fdcf8e5b5a68dc754e9b3e30
RMD160 (NetBSD-6.0/i386/binary/sets/text.tgz) = b30ff1dd8f2cfe414939980d4be3b61f39c04353
MD5 (NetBSD-6.0/i386/binary/sets/text.tgz) = 2667947b26dffe05f3de7d1e2c8906d7
SHA1 (NetBSD-6.0/i386/binary/sets/xbase.tgz) = 934f64b717d722045717691a55fb9d295f83b4e3
SHA512 (NetBSD-6.0/i386/binary/sets/xbase.tgz) = 385cf3a944f4a6ef7271220f0367f12d123d9a2b84ebd3ca46181279f7cf5d544a946df3e10ea52f1e07638c5bc5d0b766fd63bed4e08b468f6ea5b0a75e17de
RMD160 (NetBSD-6.0/i386/binary/sets/xbase.tgz) = 04203a7b8bb5cd1a41aeeb2b92697677cfe58e98
MD5 (NetBSD-6.0/i386/binary/sets/xbase.tgz) = ec9f6a167fd05c459f1fdf8cc889785d
SHA1 (NetBSD-6.0/i386/binary/sets/xcomp.tgz) = 7bc2f4e10afaa599959832c62f024d9f2f45bc10
SHA512 (NetBSD-6.0/i386/binary/sets/xcomp.tgz) = 148bfb83357133e5377aeee82fdae29e0b90913ba59b1f51264108ca70010d55bd4adc4ce0ba7611a83b5f3f13307f196f724f3cb3a170d7f2caef801b79cc38
RMD160 (NetBSD-6.0/i386/binary/sets/xcomp.tgz) = be8c61b5890936fc93dfb88c02c5f17f79f3cc73
MD5 (NetBSD-6.0/i386/binary/sets/xcomp.tgz) = 9c25f1bf64f97c778e3c87b6e0e72871
SHA1 (NetBSD-6.0/i386/binary/sets/xetc.tgz) = ea4db636b3919906a00e7b161b2c5f62fe7b4fa6
SHA512 (NetBSD-6.0/i386/binary/sets/xetc.tgz) = a59442b3ab58c33774197f9bb28a96836b4c519c750270bd880d166250523f8119a68a9de6b6861f76cdf713da9682772e754075c965c56aa260b52ed837943e
RMD160 (NetBSD-6.0/i386/binary/sets/xetc.tgz) = 28e3ce094b557af508e1ec82ef46ebb01eb68509
MD5 (NetBSD-6.0/i386/binary/sets/xetc.tgz) = 2a521f73ba4bd630070b01f24c8647d6
SHA1 (NetBSD-6.0/i386/binary/sets/xfont.tgz) = a7a41c82046568a695ba0afeb81d11bf92e67950
SHA512 (NetBSD-6.0/i386/binary/sets/xfont.tgz) = f46f4c6b4c61bbcbde85333cc714654902263c92e70035545caba3f3e2d5029cb074c8e427c85ea83b8478b19f32b93c4735a7adf64cc75609a4acd1118d0416
RMD160 (NetBSD-6.0/i386/binary/sets/xfont.tgz) = 8863108bc2cc456cd1298b5317f519c5c6b3147a
MD5 (NetBSD-6.0/i386/binary/sets/xfont.tgz) = 36e968072dae462efddd6014f9f44272
SHA1 (NetBSD-6.0/i386/binary/sets/xserver.tgz) = 27fd7ca36d5a3ca94b66eb5fd81ceeaed1648fe0
SHA512 (NetBSD-6.0/i386/binary/sets/xserver.tgz) = 6a0d364672e4513dbf4c76c1cada80f84e774efa85c3e1fe4865ab26333765fab1db8ce3b590682a3e2c63895165e0d694c3e6717789b2f0ec87defa608930dd
RMD160 (NetBSD-6.0/i386/binary/sets/xserver.tgz) = 7521930788a2410a2842767b9c2886af83fccad7
MD5 (NetBSD-6.0/i386/binary/sets/xserver.tgz) = ce3565a1b5e8af09e54ef8488edb14df
SHA1 (NetBSD-6.0/i386/installation/cdrom/boot-com.iso) = bcfbd8c6288248e55d78704802dfdbbd1d20519a
SHA512 (NetBSD-6.0/i386/installation/cdrom/boot-com.iso) = 9531877436cf8853db4dbd96672cb13a0784a1eef2a90ab43aeaae421f70bb6bdab7d5c10095f186b733d04ee1184087ff7b5d26ec7ef77d0783ee0f5c0d21f3
RMD160 (NetBSD-6.0/i386/installation/cdrom/boot-com.iso) = 46205ca4d924c43e85b01b7750274f31fa2ad170
MD5 (NetBSD-6.0/i386/installation/cdrom/boot-com.iso) = da3da9b120520e631bc5c17dfb70bd27
SHA1 (NetBSD-6.0/i386/installation/cdrom/boot.iso) = 146e96103568f2ab3e98c1a0c7b98f89a93ac474
SHA512 (NetBSD-6.0/i386/installation/cdrom/boot.iso) = 5073e008276c0fe9a4c3231bea37b2534e40d20b17d32a5bb30e9811a0578feae08a810a808201f44f74ab7258a58c5c1dfa1cc59ad0d4d41eff02f6772dade5
RMD160 (NetBSD-6.0/i386/installation/cdrom/boot.iso) = 54ff36aedaa5266fb788b1b3f74dee0799c432f7
MD5 (NetBSD-6.0/i386/installation/cdrom/boot.iso) = 6d1fae4935084e853fa152e09b472be5
SHA1 (NetBSD-6.0/i386/installation/floppy/boot-com1.fs) = d4891595c9155d7598f1718cce43ed2fde9d4aa8
SHA512 (NetBSD-6.0/i386/installation/floppy/boot-com1.fs) = 89d2af1f41e20deba16c1abfdc4709f353021710ae33ff40f9f4f20dd4c875db855f2bddf9af444f49dfb2d1d1caf0a2f3eab7bced113c45708da590092c9b04
RMD160 (NetBSD-6.0/i386/installation/floppy/boot-com1.fs) = f2d8d259d8abaed540079cb0301fd111462bc0ba
MD5 (NetBSD-6.0/i386/installation/floppy/boot-com1.fs) = 320d7fcd247eee78495d4bc56205ba9f
SHA1 (NetBSD-6.0/i386/installation/floppy/boot-com2.fs) = 25012df4758eca3d307004fe1b448bfd1002b6ba
SHA512 (NetBSD-6.0/i386/installation/floppy/boot-com2.fs) = 3ad3778e665bbb8a0ea8e39d9c6a9144a79fc158aeef0af4bac90ee98561beb460b2f3993c929f27a1f26170be53d153faa0ec523e402bb1143d9cec5a54ac91
RMD160 (NetBSD-6.0/i386/installation/floppy/boot-com2.fs) = e76b6f6a667e4548f8efdce8fded360a9ab7ce00
MD5 (NetBSD-6.0/i386/installation/floppy/boot-com2.fs) = 1c731748f3af6300b39311051870dd5d
SHA1 (NetBSD-6.0/i386/installation/floppy/boot-com3.fs) = 00f812e968818ac87e2f0217b9aa2f67cf1826b2
SHA512 (NetBSD-6.0/i386/installation/floppy/boot-com3.fs) = ac293ed38646e83851e74e70451893b301fe935fd4ee91fbe2f66d4a49583763ce3a5a123f05e38d0806e6b51c6db01dce467908f7ac49447b872e1a06364e74
RMD160 (NetBSD-6.0/i386/installation/floppy/boot-com3.fs) = cd261aa83970e5699763ec792aebbc4bcedae712
MD5 (NetBSD-6.0/i386/installation/floppy/boot-com3.fs) = a9073f83732b763206b32fd46c166fab
SHA1 (NetBSD-6.0/i386/installation/floppy/boot-com4.fs) = 6d9f052a4ba66b7a8d818418c7548a96b358189c
SHA512 (NetBSD-6.0/i386/installation/floppy/boot-com4.fs) = 86909e41525fe7744c3e10f523b7bd2651a80a000906e7040bae8fadeed8f2ee00e6d39998aab110a4aedbe9af95aa15eff7a8fd5781b6d979545c7751c54aea
RMD160 (NetBSD-6.0/i386/installation/floppy/boot-com4.fs) = 64980af391f001e6b67d6b603fd67e3d424a48a4
MD5 (NetBSD-6.0/i386/installation/floppy/boot-com4.fs) = 9868c40fe2e4e26b0db5eb1a24d4b84b
SHA1 (NetBSD-6.0/i386/installation/floppy/boot-com5.fs) = e138754c8978462f24c07a3e61be0085592ab1be
SHA512 (NetBSD-6.0/i386/installation/floppy/boot-com5.fs) = 785ae7b940e77d108d20f4c8f9625db138334144e740ccf15949e4434572115fdcafecce25c3ea9c0e0d260e7d4c526a860bacda09ab8b2823d5d48a3a220b1a
RMD160 (NetBSD-6.0/i386/installation/floppy/boot-com5.fs) = 9eae2380cd42784b68dba1da19eb8e7a2f37940e
MD5 (NetBSD-6.0/i386/installation/floppy/boot-com5.fs) = 6f1b2546ecddd49db5fae090706ced45
SHA1 (NetBSD-6.0/i386/installation/floppy/boot1.fs) = 955c193dc080b69fe111652972bb1a4a5fb88b70
SHA512 (NetBSD-6.0/i386/installation/floppy/boot1.fs) = bea5bb410069dc463f5d670ca33a8040c6bc741ee30e90c93492ce5110334f0e87cc519571f35f2962bc18de0352c3f50002ceb8346f4aed84069b60c61740a8
RMD160 (NetBSD-6.0/i386/installation/floppy/boot1.fs) = 3828a790e9743011d6b2f257f79deecee3686c4b
MD5 (NetBSD-6.0/i386/installation/floppy/boot1.fs) = 83997622af685d71b47e687ee0226d5c
SHA1 (NetBSD-6.0/i386/installation/floppy/boot2.fs) = 25012df4758eca3d307004fe1b448bfd1002b6ba
SHA512 (NetBSD-6.0/i386/installation/floppy/boot2.fs) = 3ad3778e665bbb8a0ea8e39d9c6a9144a79fc158aeef0af4bac90ee98561beb460b2f3993c929f27a1f26170be53d153faa0ec523e402bb1143d9cec5a54ac91
RMD160 (NetBSD-6.0/i386/installation/floppy/boot2.fs) = e76b6f6a667e4548f8efdce8fded360a9ab7ce00
MD5 (NetBSD-6.0/i386/installation/floppy/boot2.fs) = 1c731748f3af6300b39311051870dd5d
SHA1 (NetBSD-6.0/i386/installation/floppy/boot3.fs) = 00f812e968818ac87e2f0217b9aa2f67cf1826b2
SHA512 (NetBSD-6.0/i386/installation/floppy/boot3.fs) = ac293ed38646e83851e74e70451893b301fe935fd4ee91fbe2f66d4a49583763ce3a5a123f05e38d0806e6b51c6db01dce467908f7ac49447b872e1a06364e74
RMD160 (NetBSD-6.0/i386/installation/floppy/boot3.fs) = cd261aa83970e5699763ec792aebbc4bcedae712
MD5 (NetBSD-6.0/i386/installation/floppy/boot3.fs) = a9073f83732b763206b32fd46c166fab
SHA1 (NetBSD-6.0/i386/installation/floppy/boot4.fs) = 6d9f052a4ba66b7a8d818418c7548a96b358189c
SHA512 (NetBSD-6.0/i386/installation/floppy/boot4.fs) = 86909e41525fe7744c3e10f523b7bd2651a80a000906e7040bae8fadeed8f2ee00e6d39998aab110a4aedbe9af95aa15eff7a8fd5781b6d979545c7751c54aea
RMD160 (NetBSD-6.0/i386/installation/floppy/boot4.fs) = 64980af391f001e6b67d6b603fd67e3d424a48a4
MD5 (NetBSD-6.0/i386/installation/floppy/boot4.fs) = 9868c40fe2e4e26b0db5eb1a24d4b84b
SHA1 (NetBSD-6.0/i386/installation/floppy/boot5.fs) = e138754c8978462f24c07a3e61be0085592ab1be
SHA512 (NetBSD-6.0/i386/installation/floppy/boot5.fs) = 785ae7b940e77d108d20f4c8f9625db138334144e740ccf15949e4434572115fdcafecce25c3ea9c0e0d260e7d4c526a860bacda09ab8b2823d5d48a3a220b1a
RMD160 (NetBSD-6.0/i386/installation/floppy/boot5.fs) = 9eae2380cd42784b68dba1da19eb8e7a2f37940e
MD5 (NetBSD-6.0/i386/installation/floppy/boot5.fs) = 6f1b2546ecddd49db5fae090706ced45
SHA1 (NetBSD-6.0/i386/installation/miniroot/miniroot.kmod) = 3949ed3ca0384f45c40b59eef585566e947fc150
SHA512 (NetBSD-6.0/i386/installation/miniroot/miniroot.kmod) = e91e8d4aa8ef1f81945c495df4b43f8bcc669fa1f0183c5d0092d526889d985432d6f318418a42910596f1f44cc0d235d32a0b5e3aa1572198182ff1d90188c1
RMD160 (NetBSD-6.0/i386/installation/miniroot/miniroot.kmod) = 2d67e98155d79210ae2349fb13937724f79cef47
MD5 (NetBSD-6.0/i386/installation/miniroot/miniroot.kmod) = 2fbac91d26cec04141c2c79a18aa839e
SHA1 (NetBSD-6.0/i386/installation/misc/pfdisktc.zip) = e3dedc5211120fdb8259075541a6bf4fdb5a0c5b
SHA512 (NetBSD-6.0/i386/installation/misc/pfdisktc.zip) = 2f4daa1b52b4f40be868ee04faf9b55f17df64ca03ed0b6577f2f1680692226b68d84c0a9f59fd0b08085219555c25e52e447053a6e03589957718b954aadeaf
RMD160 (NetBSD-6.0/i386/installation/misc/pfdisktc.zip) = 29ed32ef8b4390bd1545f9337f16994a154c20e6
MD5 (NetBSD-6.0/i386/installation/misc/pfdisktc.zip) = 33c938480de81999f82a3ba3fca0b233
SHA1 (NetBSD-6.0/i386/installation/misc/dosboot.com) = 0c89f2a01913cbd69002b821273fe196ff835f89
SHA512 (NetBSD-6.0/i386/installation/misc/dosboot.com) = 661bb1a057d23a8dbd8f33216e590439e83b8cbc1ee7988980de68890148c8d86b9e58254be366afdc92b8238cf1d304ce972b518e4656ed9a378fa5eb7e03e6
RMD160 (NetBSD-6.0/i386/installation/misc/dosboot.com) = 11182a8c2c18dc561e9b07f84895e21f81d42aeb
MD5 (NetBSD-6.0/i386/installation/misc/dosboot.com) = 8d32a69f5a7c200404b5e6747eb38397
SHA1 (NetBSD-6.0/i386/installation/misc/pfdisk.doc) = 9886ea575769f46c49251fe1b096c860457e996b
SHA512 (NetBSD-6.0/i386/installation/misc/pfdisk.doc) = a34e971ec8a0d1eb5107ddfa2a5a6c7eb028ec58235983658ccee2e3b42bd2e40e2ff4504a2b6aa16668a918e17f0d1866ca4ef904df114f30674dd9f8e32c17
RMD160 (NetBSD-6.0/i386/installation/misc/pfdisk.doc) = 9e6ddcbeab7d6894f3173efbc9b1067d40d79409
MD5 (NetBSD-6.0/i386/installation/misc/pfdisk.doc) = 7c9091037c42ecd0d1fb67275af0650a
SHA1 (NetBSD-6.0/i386/installation/misc/pfdisk.exe) = e780cbcd893d446cc9d07950e9b9462d9a02e439
SHA512 (NetBSD-6.0/i386/installation/misc/pfdisk.exe) = 798a00ddfd76ffcae1dd427ebcd9eac6ef4fc54bb0eeeba36ecca3903b9de785ff6113e3ee0960ca695d9a8196dbdd3f7434530e50dd0208d5b872e989038c3b
RMD160 (NetBSD-6.0/i386/installation/misc/pfdisk.exe) = 4569b5259fb5d13606e521a4d85a0748afa92670
MD5 (NetBSD-6.0/i386/installation/misc/pfdisk.exe) = 6e470b18a39a3b7bf7c8fe7c34241da7
SHA1 (NetBSD-6.0/i386/installation/misc/pxeboot_ia32.bin) = a1c0c4f7137ad0c4131aec154c3cdc7fd62c4ad6
SHA512 (NetBSD-6.0/i386/installation/misc/pxeboot_ia32.bin) = 4ef5af523ac4d164885c4ce7ea8286303ce1a60d6225730d98b14f6b96c56c1227bf8bbd4d4ee77ff30645e5f0375aea1f879fc1f61de1263bc5f68e470abe28
RMD160 (NetBSD-6.0/i386/installation/misc/pxeboot_ia32.bin) = 732aa68a238219b0fa20ac73fed5cdd59ba4b775
MD5 (NetBSD-6.0/i386/installation/misc/pxeboot_ia32.bin) = d56818a2c96b381e6b907ef7ee5c146d
SHA1 (NetBSD-6.0/i386/installation/misc/rawr32.exe) = b43acce43bab52a4d7a5cca69b5c448700c3084c
SHA512 (NetBSD-6.0/i386/installation/misc/rawr32.exe) = c8e466f2a57083a0cd577720f31ec6d84096680e89c824b633f168ccde962be28cc3ec8019cf136a12437b3df92b872a3600266857c4e5c43f12c6989ce3490c
RMD160 (NetBSD-6.0/i386/installation/misc/rawr32.exe) = b0cafd5f6dab34b71cbd5885821188bb8479bdc8
MD5 (NetBSD-6.0/i386/installation/misc/rawr32.exe) = a38e68ebd955604d1b4e137c8eb63dbc
SHA1 (NetBSD-6.0/i386/installation/misc/rawrite.c) = 45e127dcd24e3a5666d1d4fff54349ca9bf5c077
SHA512 (NetBSD-6.0/i386/installation/misc/rawrite.c) = 05315a055d1b95bb409c986e3a4faa0ffaf407a239ca32638b8df05757af480cd409abf141cc9b695bf2f61ececbb0de07737a9a927e8c9d9e9f78c39a54c9a1
RMD160 (NetBSD-6.0/i386/installation/misc/rawrite.c) = 424511ff8fe6118c147cda137803fbc3bdea34a0
MD5 (NetBSD-6.0/i386/installation/misc/rawrite.c) = 606e8042f2343028bbe2fbb7db391c53
SHA1 (NetBSD-6.0/i386/installation/misc/rawrite.doc) = a5d3f4916c427ec4ec93d3b1e044f8141ed83dd8
SHA512 (NetBSD-6.0/i386/installation/misc/rawrite.doc) = ccc3a2b7daadd7527e048de382bdf37580fc8069f5ad2e9f5f8feec5e50374ad2c8a8cbd293f38fffddb08a84573f13ed1152695f64a9c4b745c6e6dae793401
RMD160 (NetBSD-6.0/i386/installation/misc/rawrite.doc) = 892d37aa0401f9d42effec57944c007e7a6d530d
MD5 (NetBSD-6.0/i386/installation/misc/rawrite.doc) = 0006cfba390f738eb0d4f6c2fcafb7c6
SHA1 (NetBSD-6.0/i386/installation/misc/rawrite.exe) = 81152a9688b5fdce26ba09c1685ffa842e31c83d
SHA512 (NetBSD-6.0/i386/installation/misc/rawrite.exe) = 76ca20eca1b087c8d583b23c52734ef6db49f15d5bbbc4da0eef301b2b46fad5b86c063e4d868bf3db41c625e022c4604759e1a1bc50c9caff2665eaecf11920
RMD160 (NetBSD-6.0/i386/installation/misc/rawrite.exe) = 2b5707df5c5a7a84b8e80fafb03442fd29ca3eac
MD5 (NetBSD-6.0/i386/installation/misc/rawrite.exe) = 7dac6ea496014d32dbce6c50feb254ca
SHA1 (NetBSD-6.0/i386/INSTALL.html) = 6b92ae0724869e0b33123151562db113ff87310d
SHA512 (NetBSD-6.0/i386/INSTALL.html) = a3541d32c0c954af91d5af22c7b1d134096a4d87f61baa9e76d192aed5461f5a44bca260dfabc3cf1979b9736958ea2d4467a5f98023bf816ea9915c6ad81786
RMD160 (NetBSD-6.0/i386/INSTALL.html) = beba55314f919265ffba84528a2ed7b2ebc2ef7d
MD5 (NetBSD-6.0/i386/INSTALL.html) = 5597002f70f1f3e1905becc1a21b5046
SHA1 (NetBSD-6.0/i386/INSTALL.more) = fd226e8ade97b7071048c7f3a71d68b8f2fbc440
SHA512 (NetBSD-6.0/i386/INSTALL.more) = df5bfc36c307a943d5def074292619b8b9452b7d8ad612feb6e3aed63534c5698701fa0008e007f6b08a742141643b2d9d82d3386a1194421c556a499e12b1b6
RMD160 (NetBSD-6.0/i386/INSTALL.more) = 811bda03f587e977180da84b2ee3a1715e059a68
MD5 (NetBSD-6.0/i386/INSTALL.more) = a080d66a0f33055ae14ac59dd13b6c11
SHA1 (NetBSD-6.0/i386/INSTALL.ps) = 81b57f86901189d2f513553abf30858d95ce11d0
SHA512 (NetBSD-6.0/i386/INSTALL.ps) = b2fcab6677f9949755c9aa702719c0572c7a7198c926e8d50103e513f2c44ec92b11147488eea8a1f0e5f04c3e024480cde2c47e538137b99f80df27f9deab91
RMD160 (NetBSD-6.0/i386/INSTALL.ps) = f7e87587ad8249b5f71b89988802cbff68e645cf
MD5 (NetBSD-6.0/i386/INSTALL.ps) = 134248944ca4bca697eeb3aaa49c1dc6
SHA1 (NetBSD-6.0/i386/INSTALL.txt) = a1072904f9c7baa1f3b9136a12e7b8fe7821f757
SHA512 (NetBSD-6.0/i386/INSTALL.txt) = 4229cbe31deb065a67fd10b737e4acb0784f9a8b31546e4c9d8b41fd002e81a9d48acf08a455dfa488669358d19cc1daa5c329cd3ccad5bfa1471432656fc70f
RMD160 (NetBSD-6.0/i386/INSTALL.txt) = 2dee238856637d6990f33f3aac177f34478c6ff1
MD5 (NetBSD-6.0/i386/INSTALL.txt) = 92702556c5a08e156b8eae602d975fdf
SHA1 (NetBSD-6.0/ibmnws/binary/kernel/netbsd-GENERIC.symbols.gz) = c4b53dcc495fbb67f30084776444088c6fa69af6
SHA512 (NetBSD-6.0/ibmnws/binary/kernel/netbsd-GENERIC.symbols.gz) = 08dc9f41dbc5f88e4e4d718880f8ac072ee9386daf727cd717b7276c88443dbe194781b062cd29409a964737cb8b27f0b158dfefbcb1b4226f9bc40deedd1f19
RMD160 (NetBSD-6.0/ibmnws/binary/kernel/netbsd-GENERIC.symbols.gz) = 5d66efd39a08359a525748ecd06b42705c53028f
MD5 (NetBSD-6.0/ibmnws/binary/kernel/netbsd-GENERIC.symbols.gz) = e819b2d7af2bddcf3d4cf935724695f0
SHA1 (NetBSD-6.0/ibmnws/binary/kernel/netbsd-GENERIC.gz) = 0494a909caf1ca5dec73b1f5aada6ef099653a7d
SHA512 (NetBSD-6.0/ibmnws/binary/kernel/netbsd-GENERIC.gz) = 5ca8f4fec7a5b89369934e081f7316c4e482ef7090ed8904f0985e941e20bf42fae8396d7da4151b78234568b0a735667b56f7b480fa37d1bba36b594d3d71f6
RMD160 (NetBSD-6.0/ibmnws/binary/kernel/netbsd-GENERIC.gz) = 2ff1c7a5c604f56f85ff3ade294e3cb55c053ac3
MD5 (NetBSD-6.0/ibmnws/binary/kernel/netbsd-GENERIC.gz) = d4f4dfb62bac44f16e9bd16c7df8a501
SHA1 (NetBSD-6.0/ibmnws/binary/sets/games.tgz) = fbf69640b362df2446d7414ac3a13db100c289eb
SHA512 (NetBSD-6.0/ibmnws/binary/sets/games.tgz) = d86b41d79aea4204ccf45306937a5ec45629e37873a5956c0280fd1d54cf4a808542d097f9d978fd3277791554ce8a6fb4d0ccd46e30210549d5cce1145bb499
RMD160 (NetBSD-6.0/ibmnws/binary/sets/games.tgz) = fd2aed1eb51f4db522fb5163e8fde4c040b55d30
MD5 (NetBSD-6.0/ibmnws/binary/sets/games.tgz) = 997ae5fb42196a787df89873be321e47
SHA1 (NetBSD-6.0/ibmnws/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/ibmnws/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/ibmnws/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/ibmnws/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/ibmnws/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/ibmnws/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/ibmnws/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/ibmnws/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/ibmnws/binary/sets/xbase.tgz) = b7c8146146ebc15191c1bf53f51f8f980fe8c816
SHA512 (NetBSD-6.0/ibmnws/binary/sets/xbase.tgz) = 091f5d00414a9ad478811bfb0e953f2a204869bc5e310790886a59c11953974cf7068cd9a96c8823856ab2687ffa2ac31e97494f47338bfe31d80217382c59f1
RMD160 (NetBSD-6.0/ibmnws/binary/sets/xbase.tgz) = 4bd1bfdc7c1e1619de63fb83276c21039b945357
MD5 (NetBSD-6.0/ibmnws/binary/sets/xbase.tgz) = e621806afa0fcc9cdcef0f9b3c7dfcbd
SHA1 (NetBSD-6.0/ibmnws/binary/sets/xcomp.tgz) = 960dd33b61e19dec92096a54791aca1cdf56ce54
SHA512 (NetBSD-6.0/ibmnws/binary/sets/xcomp.tgz) = 111dd50041e4008aeadb898c3d08f9995c5c13a41ccc20993b4ad51c81d7f6ee3b4dab16158ac7436d085b80e4a643b236a9c97edd3e915125a1466154538420
RMD160 (NetBSD-6.0/ibmnws/binary/sets/xcomp.tgz) = 0cf8b9a8d1489a32ba0b5a277d90e429c40650fb
MD5 (NetBSD-6.0/ibmnws/binary/sets/xcomp.tgz) = 4ba4077ca2c0edf71813165fcf5321fb
SHA1 (NetBSD-6.0/ibmnws/binary/sets/xfont.tgz) = 4711fcbade0238d60b7b89cf0d7873b03d7ed1f7
SHA512 (NetBSD-6.0/ibmnws/binary/sets/xfont.tgz) = c29130f55f536f602253a7f92f073ebf6ba02baea09f8011530bbf7e03ae96594334f0c60b29a8bceaffc3eb8b152db47548e38a3c8a14abc5fe35bc5d1d7d93
RMD160 (NetBSD-6.0/ibmnws/binary/sets/xfont.tgz) = 3b01af0f8e34367988d559fe0fbb0acbbafb0b2b
MD5 (NetBSD-6.0/ibmnws/binary/sets/xfont.tgz) = 6ba15ed12217b726d9129db8e942a048
SHA1 (NetBSD-6.0/ibmnws/binary/sets/xserver.tgz) = 05aac7e617e0cac05a88062fef29bb459da7a1d1
SHA512 (NetBSD-6.0/ibmnws/binary/sets/xserver.tgz) = 19fb5af5d6afd78839a22e5c0694a492047b5200c569cbf58c676ff32ec250b21edcfe2ed00b65092fdac374d1ab9eaaed6113c69bc03009cf458aedc6014913
RMD160 (NetBSD-6.0/ibmnws/binary/sets/xserver.tgz) = 774d2a1072b8d122790ad189a6d2ada2ad99e7cf
MD5 (NetBSD-6.0/ibmnws/binary/sets/xserver.tgz) = 3daf1477cadf3593e9ba45e3703fe8a5
SHA1 (NetBSD-6.0/ibmnws/binary/sets/kern-GENERIC.tgz) = c9804814e9b76392c467295e0700798f7305d9ea
SHA512 (NetBSD-6.0/ibmnws/binary/sets/kern-GENERIC.tgz) = dbe66f41c0023cf48726763c0accca7d17146888cef03f6bcd8356f48bca2a5685e60224e3369fcf4ebc524843a9e3a2df558199a4b2b1a43cc7b22931c52c5b
RMD160 (NetBSD-6.0/ibmnws/binary/sets/kern-GENERIC.tgz) = e054c430a723e57798b9f13d852d6be6895a6b18
MD5 (NetBSD-6.0/ibmnws/binary/sets/kern-GENERIC.tgz) = 9bb6d0f70b522f8bdfe0da0855fe12b2
SHA1 (NetBSD-6.0/ibmnws/binary/sets/base.tgz) = 150b9b1ed403c611d17b3bf258f7bf064f3f76c6
SHA512 (NetBSD-6.0/ibmnws/binary/sets/base.tgz) = 7c9eb4dbb9e770395ea516b27dc37c86a34d10437bfbc59bd6b55ebd9a3565e470ee7553a238e69012c7eb19f1fafe3bb94d750c642c179190566d0b97ba79f7
RMD160 (NetBSD-6.0/ibmnws/binary/sets/base.tgz) = 6f959eb6b88fa607c19c9ab5e0b82d172960676e
MD5 (NetBSD-6.0/ibmnws/binary/sets/base.tgz) = 1d2ccb914175ffb1a6e90e787914e08d
SHA1 (NetBSD-6.0/ibmnws/binary/sets/comp.tgz) = 449a6e0ccfaa3b9105912d05d4204242bab79635
SHA512 (NetBSD-6.0/ibmnws/binary/sets/comp.tgz) = 5ceb6d751f429f466ae15f98244e82f32461f7f0bf546c982a1e35bcfc4c6c845a121ecf07b387189fa4173d672e0291a0d396010f5c6ab9130f3264ef26f36d
RMD160 (NetBSD-6.0/ibmnws/binary/sets/comp.tgz) = 6d544592799a77914a724215f8514f0d22bae2b2
MD5 (NetBSD-6.0/ibmnws/binary/sets/comp.tgz) = 84f35afba48084d9eed192a484616382
SHA1 (NetBSD-6.0/ibmnws/binary/sets/etc.tgz) = 4ec9d5606ead158c85791c298c829a030aaa6acd
SHA512 (NetBSD-6.0/ibmnws/binary/sets/etc.tgz) = 2786ad3a1aca388352f2c4fa1da7da624d9762c4d8c7deec54441bd3ce95c79cb2ccd6fd15619a224987f9726c66d5a88b825e57b69b68aac7033ce80c96af3e
RMD160 (NetBSD-6.0/ibmnws/binary/sets/etc.tgz) = 25c6401fec22c9ec3f2bc6c3d33127b9222def9b
MD5 (NetBSD-6.0/ibmnws/binary/sets/etc.tgz) = 9f83cc04c872ec0e355e2528387cdcac
SHA1 (NetBSD-6.0/ibmnws/binary/sets/modules.tgz) = fef0894685186ef94fc691e71a8a366fe7bb66e0
SHA512 (NetBSD-6.0/ibmnws/binary/sets/modules.tgz) = 0051ecab124fbe7a4a8005ae4fddc9b4be520e6be5c34cd1811f6c280e35f10d79914756f5106a840177b7cc33f39db337c3f461f00669c1838a6ac0c2204ee5
RMD160 (NetBSD-6.0/ibmnws/binary/sets/modules.tgz) = fce29d3683f7e0eec8e0b45c94bba924bad6b5d7
MD5 (NetBSD-6.0/ibmnws/binary/sets/modules.tgz) = a47b110fc0e721d6849bb6fa4d3a7ab3
SHA1 (NetBSD-6.0/ibmnws/binary/sets/tests.tgz) = 3eb8ca45b13b0c43f1d9e10322b8b8469859ffac
SHA512 (NetBSD-6.0/ibmnws/binary/sets/tests.tgz) = d9d2bc61ad2f5ba8ca9bd4b53c348b01fc4f7dc7417d14fe7c6c3f68b2a0ff32221bda546b9ca26a0b998ee72a393e3231de17c94bd1e72a0bc6365dba42ea5f
RMD160 (NetBSD-6.0/ibmnws/binary/sets/tests.tgz) = 8348b43e8543ad7ab8ba46978930fd7bef910c6d
MD5 (NetBSD-6.0/ibmnws/binary/sets/tests.tgz) = 5cb30f9ed33e774f3c1bc14cea070f34
SHA1 (NetBSD-6.0/ibmnws/binary/sets/text.tgz) = 38ca726175c32d5ef01309254f51584001f2888b
SHA512 (NetBSD-6.0/ibmnws/binary/sets/text.tgz) = 78102b1494c3e0d92ee40eaed00cdf6a4cd9e6d5dcfe2f523442fdce7f46d9de2e63c9ee5e0ba90fbae71509518ed2d212a0eab357f0a235ad8b4339b8f74bc6
RMD160 (NetBSD-6.0/ibmnws/binary/sets/text.tgz) = 9532b62eed56ed6308d0b564552a77239ddcc406
MD5 (NetBSD-6.0/ibmnws/binary/sets/text.tgz) = 25b993723182c6c3fc4c4c7d5544a7c0
SHA1 (NetBSD-6.0/ibmnws/binary/sets/xetc.tgz) = 6068a07bd5602c1e57dcce29ce0a58ab8d6fc63f
SHA512 (NetBSD-6.0/ibmnws/binary/sets/xetc.tgz) = 2bc55ed596339c00a303d972145c4621db7f5781f7f4bf7bd8cd6ca634cd397f8b603c83b2f607f9a65c73cc237b16db0fa3ac30f28ed898d555224355741f4f
RMD160 (NetBSD-6.0/ibmnws/binary/sets/xetc.tgz) = f5b7f09b378d534eb6436d76a135192dae3cae0a
MD5 (NetBSD-6.0/ibmnws/binary/sets/xetc.tgz) = 1c3134612470159dcad9c60e9e0664d9
SHA1 (NetBSD-6.0/images/NetBSD-6.0-amd64-install.img.gz) = 0fb40298c03dc94850911ccf296f74b4daee6ab1
SHA512 (NetBSD-6.0/images/NetBSD-6.0-amd64-install.img.gz) = e0cf82777d65c1f8fa2776a154f297f8cadcca5c8ea8064adf0ac87b51f5999af8208fe284e95478d87f45d68ece4c63e82611d624514b96a9e236bfbe2aa56d
RMD160 (NetBSD-6.0/images/NetBSD-6.0-amd64-install.img.gz) = 0f38bf9ac003bdc7b7a73a1321af35f77e498de4
MD5 (NetBSD-6.0/images/NetBSD-6.0-amd64-install.img.gz) = 05aa2e9948e5f613ac8942f04f5703b3
SHA1 (NetBSD-6.0/images/NetBSD-6.0-acorn26.iso) = 3d532f8e1c0d8c70ac0204a0e30352946d5b4cf8
SHA512 (NetBSD-6.0/images/NetBSD-6.0-acorn26.iso) = faac85c94da6ff71b98f6ee934aee21d972c52b4ecf1a1bd39609b31ac80956fed51442fa34e423c432fe6a2957768df5db91585c0edd385cbd4e13f067eedfe
RMD160 (NetBSD-6.0/images/NetBSD-6.0-acorn26.iso) = db40e23e1cd76af703d18f9e9fadfb7bad0bdbc0
MD5 (NetBSD-6.0/images/NetBSD-6.0-acorn26.iso) = c3adc16971c37d3319e62fe83dbc6fe9
SHA1 (NetBSD-6.0/images/NetBSD-6.0-acorn32.iso) = 1b8406a3ddc779f865e4edca16d6e5cbb0b33728
SHA512 (NetBSD-6.0/images/NetBSD-6.0-acorn32.iso) = 8c15a060921d1339eb814456955753badacc1db25f84f471ae66106f78f3ebc22f017ec1ff9af71cf1c59ad9a669943210967105e2669ba9093596e6b2009020
RMD160 (NetBSD-6.0/images/NetBSD-6.0-acorn32.iso) = 58b72463d91d486398514eac65f3cebb2bbcb0db
MD5 (NetBSD-6.0/images/NetBSD-6.0-acorn32.iso) = eef67583eee51a578d4f6ce69864fd25
SHA1 (NetBSD-6.0/images/NetBSD-6.0-alpha.iso) = ff793eb5257ba6f7c3ca97049cab973dafc70785
SHA512 (NetBSD-6.0/images/NetBSD-6.0-alpha.iso) = e2fcfb789c6c0bf801cb02b1a4f9d133902a9d78be3458b19e12b8050c0686cfae37d2ad4a5661ffc3cddb089e7079d0bfe5b0c7fddbc0ad7c43abc250f21db4
RMD160 (NetBSD-6.0/images/NetBSD-6.0-alpha.iso) = 50a4b3f98483a0ff94e2c6c4d93615069d2c4953
MD5 (NetBSD-6.0/images/NetBSD-6.0-alpha.iso) = eacfcb881765bac2c8fc45e1650091a1
SHA1 (NetBSD-6.0/images/NetBSD-6.0-dreamcast.iso) = dd89f98a8bcd0d8bdc9861682227cd1e48031db3
SHA512 (NetBSD-6.0/images/NetBSD-6.0-dreamcast.iso) = f18e432d93110b9cafe06c1a5bb13a27b028a5a773bc9ddfc59685737b2939f4ebf30de6e8e7fc5d6bca2f16b576af48cd2aa62ebd791b28d499b008cb5f11db
RMD160 (NetBSD-6.0/images/NetBSD-6.0-dreamcast.iso) = 763baa9e578be882dd458c8879f42745cf6ff2f2
MD5 (NetBSD-6.0/images/NetBSD-6.0-dreamcast.iso) = 19d1400e0b92a906c4f01cae132e5d22
SHA1 (NetBSD-6.0/images/NetBSD-6.0-amd64.iso) = 78d8ca5aa25eef0bdab5c794550e7c43637169a5
SHA512 (NetBSD-6.0/images/NetBSD-6.0-amd64.iso) = f4900e8effcb1f5b5767873976dbb231a07346f48df20f56a8441942a91ecca0b6ea1309e01ea6a67b0513e61ebe3b8501b436063676558f119735f6d35592b6
RMD160 (NetBSD-6.0/images/NetBSD-6.0-amd64.iso) = 76a6c030ca86c7d5abdfe91568a543d3bad04ffa
MD5 (NetBSD-6.0/images/NetBSD-6.0-amd64.iso) = 9155899df5ebc1a69c7f13ef004677a0
SHA1 (NetBSD-6.0/images/NetBSD-6.0-amiga.iso) = 7a81516c832299ddcfa8b364422e82a63b3869df
SHA512 (NetBSD-6.0/images/NetBSD-6.0-amiga.iso) = 6695416ad0dbe2bf03bb75df8b40b04cec9179d2f60aec0b1a5e8f56ad12995911158251cb78c67d62d60324e8708c85c02b84ba0fcffd3fcf73a69aaa497b0b
RMD160 (NetBSD-6.0/images/NetBSD-6.0-amiga.iso) = 72c96741b1fad68368043e16d21ff533490b8275
MD5 (NetBSD-6.0/images/NetBSD-6.0-amiga.iso) = 6b214d3da92f2d84210ac00c4eb9650b
SHA1 (NetBSD-6.0/images/NetBSD-6.0-arc.iso) = 09dd2c85bf2772333ec299e45dec5c016dc22d7e
SHA512 (NetBSD-6.0/images/NetBSD-6.0-arc.iso) = 6eaab9510f8ea92a175115377924dfb7806be38c78d12920a86e6284bc59bf0711adbb5b46809eb48fa8c299d8c9683b777ad16fe78c3e4760adc6ff605941e6
RMD160 (NetBSD-6.0/images/NetBSD-6.0-arc.iso) = d4f2d0301da83d93a18cce50be4a878cb5955202
MD5 (NetBSD-6.0/images/NetBSD-6.0-arc.iso) = a4c2332dc064e7d8cbccb921af5c2022
SHA1 (NetBSD-6.0/images/NetBSD-6.0-atari.iso) = 9976598eb68ec6bb90329161c3ffad985e2b3c35
SHA512 (NetBSD-6.0/images/NetBSD-6.0-atari.iso) = ac4951dcf3f148925c19db7bc7027db69549ac90e57db25994c7de5959943476eff7eeb0c490ca7b3df4b6f7bc0604fb5515c337e776a384d5b6a8169e92bf47
RMD160 (NetBSD-6.0/images/NetBSD-6.0-atari.iso) = 35d389ebb1ea02751e819cd76750a1116d472b1f
MD5 (NetBSD-6.0/images/NetBSD-6.0-atari.iso) = 44c8de8283cccb8a576242adcad878fc
SHA1 (NetBSD-6.0/images/NetBSD-6.0-cats.iso) = a1f0554b7193949055ee9cb35eae529893bfc8a8
SHA512 (NetBSD-6.0/images/NetBSD-6.0-cats.iso) = 7a38bfe0e1ff3948579dbcceed822634e3352e690911052dcf25f9b8746f457f1acb80830b6ca22637c559948eb8fb1eab8c9434225f13c000204879b9edc16c
RMD160 (NetBSD-6.0/images/NetBSD-6.0-cats.iso) = 2116103ecfe4c450432b7c1bad8787aaa6149a51
MD5 (NetBSD-6.0/images/NetBSD-6.0-cats.iso) = ebcfe064bb40fc87487654d7f8ccc15a
SHA1 (NetBSD-6.0/images/NetBSD-6.0-cobalt.iso) = d5e6a8be8cfcef7362542c2ecbfe7e0173845061
SHA512 (NetBSD-6.0/images/NetBSD-6.0-cobalt.iso) = 28181b68559432a4c566bcc9f71a5ed2b00a5d7118073d114ce4085f39d2139ce50e6b843beb46a59ab4e55aab4ead1e2fb952610f633fb77ae87edb5d2c3d76
RMD160 (NetBSD-6.0/images/NetBSD-6.0-cobalt.iso) = ddcff88c659b8f0190e66ed2df4014395bf819e7
MD5 (NetBSD-6.0/images/NetBSD-6.0-cobalt.iso) = 5975ccac715d36432aa101e6c1c745e4
SHA1 (NetBSD-6.0/images/NetBSD-6.0-emips.iso) = 0c9a09ee73ef6d90af39aca9513b539a79e7c505
SHA512 (NetBSD-6.0/images/NetBSD-6.0-emips.iso) = a0ec2048ab0c44ac69722b79b56308781e9f298fc3366bd628d4cfeebd8891e08091e8d86a576be50ccb12b1ae48263f86c8f0bf42d743d040a2407fb21039de
RMD160 (NetBSD-6.0/images/NetBSD-6.0-emips.iso) = 2d0e11edc64ceb8af1fc7239a9390868e1febfae
MD5 (NetBSD-6.0/images/NetBSD-6.0-emips.iso) = d5dbd826114f5e530d171761462718c6
SHA1 (NetBSD-6.0/images/NetBSD-6.0-evbarm.iso) = 95e132f2d72a36eadf98b59d6883cac167d22887
SHA512 (NetBSD-6.0/images/NetBSD-6.0-evbarm.iso) = 5c3a43f9609687ac5e5f5d5f21cdc798dd75f0d89bfe3a4d46ee151f60e13a90e2059d9906efcc5831f5e7ef0c267c3615737da03cb781eeff39dcff518b4114
RMD160 (NetBSD-6.0/images/NetBSD-6.0-evbarm.iso) = 99c9115a98e45e96730163fde4fc90f37ef11e26
MD5 (NetBSD-6.0/images/NetBSD-6.0-evbarm.iso) = 1bb998b53e8a82a28fdefecd4c2d559b
SHA1 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mips64eb.iso) = 0ffea2aa4fb1fc1d58b87259e56080caa9fd6996
SHA512 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mips64eb.iso) = 2d3a7807156a465c9b2d8ee5d23951f5415984c2bc0cd8a7e06102399ac4b89d247380542efe1148d5bf71bb945c4de005fc64df7ecf2cdc9ce63481a2f10d07
RMD160 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mips64eb.iso) = 1de400a05e285d41af9eb1fcb4164932ecee0888
MD5 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mips64eb.iso) = 62143f6b35eee23d62a01b9cd05e17bf
SHA1 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mips64el.iso) = 5b0f9688d925754bd79d28226f29dc0057c634b0
SHA512 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mips64el.iso) = d4dda568787efe15faa1c00b604c9c262c579ae86f4abd07dcdb796112fa9ca41122efc9c3beedf2281ab1859e11e2ede536d3a7d679a31590021acca4b32324
RMD160 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mips64el.iso) = 9cb45c46d6ce9d9cabcee0f660fc1a646f849190
MD5 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mips64el.iso) = 320f53af1cf67bf37bf6d962a49aa688
SHA1 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mipseb.iso) = c9894866972a80c2ff9032d21339e3ca59c8d0dd
SHA512 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mipseb.iso) = 04a89998d1cd8e70b045479265ce82b3733aa630d536c349e12b4d12e3aae123bd29cc0a7a6930fe07dc121a89e0327d14c2b78987542f064b67287268300830
RMD160 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mipseb.iso) = 13cc4119e07eb9a8a26666cbf6ae983001cbf39c
MD5 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mipseb.iso) = a5af80295a5bce15ae2cb3c585891d7e
SHA1 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mipsel.iso) = 09155023a46f527048b17f00f554d64b73fb1d40
SHA512 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mipsel.iso) = 21108a854d7f8ee48d18f8d1fc74886350d1d95ca2eef044f5314570619f108f471c768f93b99b917b887fca4a1d5bfb9f858929c88672faf505ccc6493eef61
RMD160 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mipsel.iso) = c6c4eddd0acb7b311e6bc40a9ef7acea4c8792d9
MD5 (NetBSD-6.0/images/NetBSD-6.0-evbmips-mipsel.iso) = 37b00a36438893f30b98de7327628d48
SHA1 (NetBSD-6.0/images/NetBSD-6.0-evbppc.iso) = bba761cfabb03d27837a21a61e681c0a39b82b50
SHA512 (NetBSD-6.0/images/NetBSD-6.0-evbppc.iso) = 615d5ba030e1d19c56aaf9152640e652c6aa3ab3609d357b859df61a874125c551ae912f01179f861bae6fa0b09cbb899e9ad689dd62ec7edcd1caec3157d5ca
RMD160 (NetBSD-6.0/images/NetBSD-6.0-evbppc.iso) = 4cba6606c2fc9febb684665512c89ff8106790bd
MD5 (NetBSD-6.0/images/NetBSD-6.0-evbppc.iso) = 45b10af76ae7128c4619864237e703fc
SHA1 (NetBSD-6.0/images/NetBSD-6.0-evbsh3-sh3eb.iso) = 51da3d8f99c84a9e12bab81442a0249624550532
SHA512 (NetBSD-6.0/images/NetBSD-6.0-evbsh3-sh3eb.iso) = c804c8f1457178bffa1e5b7989a5818381f4911a8fe18a2c75ef9c02b866a9f53be17926faaa4a9460852e482d7a304a13424b8fd6b8013cae15696ebcc24bea
RMD160 (NetBSD-6.0/images/NetBSD-6.0-evbsh3-sh3eb.iso) = 05f04f2d8e64a4508034424bbd9f6adf795675a5
MD5 (NetBSD-6.0/images/NetBSD-6.0-evbsh3-sh3eb.iso) = c9358e63b62edc277bb8e630627fa49d
SHA1 (NetBSD-6.0/images/NetBSD-6.0-evbsh3-sh3el.iso) = 89dd396a870a8d08bb1ab2a09f9e8b24bb55bb53
SHA512 (NetBSD-6.0/images/NetBSD-6.0-evbsh3-sh3el.iso) = d9d2018111fc190ab0602e87572e210572bebf2185df101265f29d4ac8bfed8ca671038131eefe125ffacef63924e1274c34fa248b88d01764d138144e768b2b
RMD160 (NetBSD-6.0/images/NetBSD-6.0-evbsh3-sh3el.iso) = ccc5e6f669147c71f9272cae12d63bb270407748
MD5 (NetBSD-6.0/images/NetBSD-6.0-evbsh3-sh3el.iso) = d33272ecf641d0c4ced897c2fb4d0ae1
SHA1 (NetBSD-6.0/images/NetBSD-6.0-ews4800mips.iso) = 029c1d1910dd234ad3b33f8a53557a56779dde86
SHA512 (NetBSD-6.0/images/NetBSD-6.0-ews4800mips.iso) = 09e0dec1e67c44a29cea4eae8c2e7840906c8cc6881331b720905c71dfd0f8050210568d2e5ad6a325a827a08e1e440d84980cb4b515d76501616c7e3f104cf2
RMD160 (NetBSD-6.0/images/NetBSD-6.0-ews4800mips.iso) = bd7da2f0e68a488d351eab9cda26f11ca2ddaa35
MD5 (NetBSD-6.0/images/NetBSD-6.0-ews4800mips.iso) = 1286a5646c2f93fc128c8fabb2244bd1
SHA1 (NetBSD-6.0/images/NetBSD-6.0-hp300.iso) = 3b62c2c6037a1009b5d417e6f09dfdda2781f12a
SHA512 (NetBSD-6.0/images/NetBSD-6.0-hp300.iso) = 2a3d668fde1b5098933d0fbd9fe9100092043a81389f6f7b457a3425dfca5cca52382a29434309baae791c908855b1aed99d32f290723172c62f439f68e4fe5d
RMD160 (NetBSD-6.0/images/NetBSD-6.0-hp300.iso) = 8314b4f992aff06157d9dae5951cce3d6083e6cf
MD5 (NetBSD-6.0/images/NetBSD-6.0-hp300.iso) = 144eb416cc2563df1330cecf26c19f2c
SHA1 (NetBSD-6.0/images/NetBSD-6.0-hp700.iso) = d2840fa102cff8b3d5825304818c98ba2ffdcbe5
SHA512 (NetBSD-6.0/images/NetBSD-6.0-hp700.iso) = 322461efc0e1648c8e4e16c93421831d14937b5ca93f2a6f25405882c2de40482315ee9e4c914a2e9206cc25eade89687a2a248bc60ea1145626272067fdecba
RMD160 (NetBSD-6.0/images/NetBSD-6.0-hp700.iso) = 11dad584942591d9c719eab0248dba8f71ba8ec8
MD5 (NetBSD-6.0/images/NetBSD-6.0-hp700.iso) = 93522e82b2562f1f889f74153deccf30
SHA1 (NetBSD-6.0/images/NetBSD-6.0-hpcarm.iso) = 5afcc737502c47e2a4efcf6a897ecbdee400d3aa
SHA512 (NetBSD-6.0/images/NetBSD-6.0-hpcarm.iso) = 32bd828c5ef1b50b6a9cdac2a419ef80522625bedd2bbb9f60b8076b7f9bef969af19fe28da9b56b6f552e008700b9fa586e1159706aa6eb4c998b2dcf89384f
RMD160 (NetBSD-6.0/images/NetBSD-6.0-hpcarm.iso) = 9a609fb57d4c4a2f8fd9aecee475eddaf7e26da7
MD5 (NetBSD-6.0/images/NetBSD-6.0-hpcarm.iso) = 7b8bbdc89300d63043e233de6d56584b
SHA1 (NetBSD-6.0/images/NetBSD-6.0-hpcmips.iso) = 203b9953e02574d71e49277b847dfd4b253199cb
SHA512 (NetBSD-6.0/images/NetBSD-6.0-hpcmips.iso) = 72a2e1eb4f9398c3779a64226c8e442e9628edcc0601b9e0c835af9e2ac31d77217b731c831d8d9bbb2c9322ef28b61194281beb0074e39aaa747610fa065ca1
RMD160 (NetBSD-6.0/images/NetBSD-6.0-hpcmips.iso) = b384e1bd9b5e0644e1d98f68270a82b47e1e0103
MD5 (NetBSD-6.0/images/NetBSD-6.0-hpcmips.iso) = fc5caa7af29622771e2451debd59f75a
SHA1 (NetBSD-6.0/images/NetBSD-6.0-hpcsh.iso) = 1b3ad4689058b56f012dcdf2ec78d046c7469d0c
SHA512 (NetBSD-6.0/images/NetBSD-6.0-hpcsh.iso) = f2d4686d5560a5ebadbd0f235033bb18db4b291ac9080635e67a8b42328108f0944c69e81ff9daaf28820f0ee98d32c0faba0ba3d397053670f4b2312ba7fdf2
RMD160 (NetBSD-6.0/images/NetBSD-6.0-hpcsh.iso) = a39e04b9cdcfb2d21fe9ee262756b05f122893b7
MD5 (NetBSD-6.0/images/NetBSD-6.0-hpcsh.iso) = ca44003562262d681f9eb5e6e3c1f25f
SHA1 (NetBSD-6.0/images/NetBSD-6.0-i386-install.img.gz) = 43e6d3f9d0ae352b8eaec597e3ab48590a8dd62c
SHA512 (NetBSD-6.0/images/NetBSD-6.0-i386-install.img.gz) = 2911ec984eb0b74ffc89dea948d2f0ec6570cf0a13790be8ce31bfe7d90cbcf132d9b7fa27bc3202d16426cf15c192c3543db7d108455f375aebd291e5162827
RMD160 (NetBSD-6.0/images/NetBSD-6.0-i386-install.img.gz) = 7da2b6c329c7e19a96d7473bed4fed2847a9e7dd
MD5 (NetBSD-6.0/images/NetBSD-6.0-i386-install.img.gz) = 9db5437e777175f3d9fee3370070cca1
SHA1 (NetBSD-6.0/images/NetBSD-6.0-i386.iso) = ccc8c4d50aaa0e43724df3b3c3afa0e130164053
SHA512 (NetBSD-6.0/images/NetBSD-6.0-i386.iso) = e71e4dc1ab41720e04f34ae8f5daccd5441f58efe983683574578b4f6c3019b53060e25e4e4880792f3012e47acfeb7eb0f571d01910435db6a4ac541ede8fef
RMD160 (NetBSD-6.0/images/NetBSD-6.0-i386.iso) = 4aefd7279ff09f228506f66ac5e91195c081e31c
MD5 (NetBSD-6.0/images/NetBSD-6.0-i386.iso) = 9b7aca4464fdd5dbeea6d6d85c5ba64f
SHA1 (NetBSD-6.0/images/NetBSD-6.0-ibmnws.iso) = 766ffac66f62d90b017d02b9f571b27ee002eb30
SHA512 (NetBSD-6.0/images/NetBSD-6.0-ibmnws.iso) = ba2e04ac3173b765d98a72796f643f2c7fe2fbc9eae9f7a85402e65d731c3a089df5a685ad98603e6e44c72956507bd77939d855f4820bcf0aaa1e705be68ec4
RMD160 (NetBSD-6.0/images/NetBSD-6.0-ibmnws.iso) = f6c480b79e05eb34818b5f018750c4dbfceb7b38
MD5 (NetBSD-6.0/images/NetBSD-6.0-ibmnws.iso) = 8cbc5b4e8fd0cd9d6cbf6d77f1285f3f
SHA1 (NetBSD-6.0/images/NetBSD-6.0-luna68k.iso) = fb688537b94273c74c52027a5bd0e9f0f1d96381
SHA512 (NetBSD-6.0/images/NetBSD-6.0-luna68k.iso) = 62911e352841866cdadd4f9ebf909cfe496a03ac4f4452f83ed7a96890448a527a0afc3ab199bcc13a12760bbf287b5b367b96ac340d148d522e759db20a7c5c
RMD160 (NetBSD-6.0/images/NetBSD-6.0-luna68k.iso) = ad5c1f6747d72a83816e13c37eb97cc4b6b0bbfb
MD5 (NetBSD-6.0/images/NetBSD-6.0-luna68k.iso) = 5373e0ffd08922c1e8e5a1f66908d2b7
SHA1 (NetBSD-6.0/images/NetBSD-6.0-mac68k.iso) = 2216006dda929d7b66bb2c92549378362f772b52
SHA512 (NetBSD-6.0/images/NetBSD-6.0-mac68k.iso) = 26d539ab284db402e00b48bddcdd726e7869bd28f717771d45cb6e5934fb069ec98f789a2baab8b7fd76aa35db5f9992f58094b9e3e936c83901e277a4bbe6b8
RMD160 (NetBSD-6.0/images/NetBSD-6.0-mac68k.iso) = b8cf8d1c1ccbb739ce29c01896a4aa55ad66469c
MD5 (NetBSD-6.0/images/NetBSD-6.0-mac68k.iso) = 4e06fdc52d82cc401d3b02c4d0232733
SHA1 (NetBSD-6.0/images/NetBSD-6.0-macppc.iso) = a8697c11dbd8b3811f1fcc8562a9d11a35f4431b
SHA512 (NetBSD-6.0/images/NetBSD-6.0-macppc.iso) = ba31ab8d00ac0038dc6509661eb39b41d7cafc268b17e9ff75809aac3fc4978fe338b2ee4c2ff59991ae93c11c32cfe019e7309fc507562f9c276172e63492e4
RMD160 (NetBSD-6.0/images/NetBSD-6.0-macppc.iso) = 36f6b8b12eba9cb39f412ce2cba71ab88a4445fe
MD5 (NetBSD-6.0/images/NetBSD-6.0-macppc.iso) = ca246495fb1465b57eb93c826063a20d
SHA1 (NetBSD-6.0/images/NetBSD-6.0-mipsco.iso) = 5f3a351f960e81c829d7bb8f6924b0b912006c05
SHA512 (NetBSD-6.0/images/NetBSD-6.0-mipsco.iso) = e04111c293c6a3a6df5433aaa8d7883c7f88bf03c868ddccb05ae7c243d9a880e54329d4d995cc5bfc6539fdb7120207463f54c3a4a5c0a317bd12173e0fac18
RMD160 (NetBSD-6.0/images/NetBSD-6.0-mipsco.iso) = 85552cf47b7eab35619385444e0d76ce8c6ed887
MD5 (NetBSD-6.0/images/NetBSD-6.0-mipsco.iso) = 7e684c533513710039e0fa96ba70b51c
SHA1 (NetBSD-6.0/images/NetBSD-6.0-mvme68k.iso) = 48ed08c58863636dcf49ad5728812def96cf82eb
SHA512 (NetBSD-6.0/images/NetBSD-6.0-mvme68k.iso) = ded376941ef8437fcf37f7ca801f4e9996b4adcd97d9ccc7a16d30c8045880d4e7a894ab5cd44000bb3895950c007669207bfd58bdbde3a11b2a123e5b747c47
RMD160 (NetBSD-6.0/images/NetBSD-6.0-mvme68k.iso) = cfb7ee4fc9422ebf6dc0c7d73b59dc2b161d58ef
MD5 (NetBSD-6.0/images/NetBSD-6.0-mvme68k.iso) = 9ee62757053b2cb7f7a1d6ab82db929d
SHA1 (NetBSD-6.0/images/NetBSD-6.0-mvmeppc.iso) = 5b74dd3e1d76f697d089ad1a769ef8aba17330ba
SHA512 (NetBSD-6.0/images/NetBSD-6.0-mvmeppc.iso) = a007559063e269c4f27597f27c3bf6dda97f34a33d7e750c2a6fb36bacc0a64b12a0011b0570ae71b5977735de5f1d58de553e5a89ea4f61b6c09fc34e3989ea
RMD160 (NetBSD-6.0/images/NetBSD-6.0-mvmeppc.iso) = 9d26c663310b2555b8dc544e9b286a7ab80e8bd3
MD5 (NetBSD-6.0/images/NetBSD-6.0-mvmeppc.iso) = 3c8ab738e662b3b7f1b481a4712c4525
SHA1 (NetBSD-6.0/images/NetBSD-6.0-news68k.iso) = b40de418e4f4140f31895100674dc69e1b3c2f80
SHA512 (NetBSD-6.0/images/NetBSD-6.0-news68k.iso) = bfca4085850abfe577482870397af0c964574cb7b426903c5634a747026e1db835fe6db0dde4d8a320438b1f136dc7aa2dd0bd1e05d664c92ddd307b3ad718e9
RMD160 (NetBSD-6.0/images/NetBSD-6.0-news68k.iso) = 00df7a8b50bb5b4f45b56107db94a559b1571458
MD5 (NetBSD-6.0/images/NetBSD-6.0-news68k.iso) = a9a9524f9d696a5104d2b9f157f8a416
SHA1 (NetBSD-6.0/images/NetBSD-6.0-newsmips.iso) = 91c6ba0265d2ab04f60358dbf8fcb388c81e674f
SHA512 (NetBSD-6.0/images/NetBSD-6.0-newsmips.iso) = 182904027a2e24e0869e0c177915c2d99be87ea8994298f7106940bfde95feacb1ab07e17f0535adaea35d63eb6ec014378bb2777b0a3dc4e15f362038050460
RMD160 (NetBSD-6.0/images/NetBSD-6.0-newsmips.iso) = 93bedba32f25a7d6f658e396de5b7eb98ec7654a
MD5 (NetBSD-6.0/images/NetBSD-6.0-newsmips.iso) = c6a0377dfb49f663d7bf657589a68c48
SHA1 (NetBSD-6.0/images/NetBSD-6.0-ofppc.iso) = 1b0927effa1f68779ec787536f67447574f50674
SHA512 (NetBSD-6.0/images/NetBSD-6.0-ofppc.iso) = 7cfa755bab97c165f5ed60063d4fde79d90fa4a786cf13d32617f324759719175b909806aac34df32b39a5a443348f020787bce43f4d7600ba9bc65ec00e111a
RMD160 (NetBSD-6.0/images/NetBSD-6.0-ofppc.iso) = aab50e82812ffe6b4216d9c9a44787d9f0f667b8
MD5 (NetBSD-6.0/images/NetBSD-6.0-ofppc.iso) = 97772de08fb8020d661192e622e3cf21
SHA1 (NetBSD-6.0/images/NetBSD-6.0-pmax.iso) = 5293cfe1a34eced76bd825d7eeef7ee55af98299
SHA512 (NetBSD-6.0/images/NetBSD-6.0-pmax.iso) = 681eb46ded55a111f6ac2f5baa9ebe4f3f6c252ecd20270377b466305402a97f6be47db192e5a05920a17ecbf03c03bb169161bcb75b4813af5a2e10b1becd53
RMD160 (NetBSD-6.0/images/NetBSD-6.0-pmax.iso) = b61b7ee3525e26f990f4276d689d23469b4639a1
MD5 (NetBSD-6.0/images/NetBSD-6.0-pmax.iso) = 4211afdc8ed6225b395f65a089347ebb
SHA1 (NetBSD-6.0/images/NetBSD-6.0-prep.iso) = e1702b470594f9b5955bba78d9179eec12176e9d
SHA512 (NetBSD-6.0/images/NetBSD-6.0-prep.iso) = 552a43f2a255b844a0a39db0e352dcf04b7df73cbe6eb4851a2bbfd1f5bcdf12d4742ef9f596052fd678ff4adae4c92a1eda71acccaa53c8bf310cf5900dbaa2
RMD160 (NetBSD-6.0/images/NetBSD-6.0-prep.iso) = 382ff5006df921cb66fc8ab2bf78544c1dc501ae
MD5 (NetBSD-6.0/images/NetBSD-6.0-prep.iso) = 2cacc13c35c1e406dfe0aec549f97eee
SHA1 (NetBSD-6.0/images/NetBSD-6.0-sandpoint.iso) = 6d866b095f838011b6132d443b602d4b97a644e0
SHA512 (NetBSD-6.0/images/NetBSD-6.0-sandpoint.iso) = 6e6a0eaa9de07b142c43a749bc6257a410f9c7f801c450cfec17fa30c0f7b0f5a925d1c73705a4d2d567451fe5c88304c6e85c4f1ba1c0127b07f9bd0ed452c5
RMD160 (NetBSD-6.0/images/NetBSD-6.0-sandpoint.iso) = a2d052a86ed3bc1a751de334b737cdd1e1661f8b
MD5 (NetBSD-6.0/images/NetBSD-6.0-sandpoint.iso) = 8a137ad061aba60a2bcffa8b9c9fa370
SHA1 (NetBSD-6.0/images/NetBSD-6.0-sgimips.iso) = eee6aecc9d478d101b20756b5438e858d43f2700
SHA512 (NetBSD-6.0/images/NetBSD-6.0-sgimips.iso) = 3979fa422cd562e6dcfa83eefc072d66380ce1ec10f5534c394c08bebcc977d58d5c27f0f41efe1555570be9eb93e33143df0c1fb1cec47b29f3f8961853132e
RMD160 (NetBSD-6.0/images/NetBSD-6.0-sgimips.iso) = b20388701c9428de38cf7cd5afa7464bb0d5a07c
MD5 (NetBSD-6.0/images/NetBSD-6.0-sgimips.iso) = 73c4e4f6723ef82859be32abb34b9367
SHA1 (NetBSD-6.0/images/NetBSD-6.0-shark.iso) = d331eaa8be85d8a112c277ac018d9d0cc88d8b39
SHA512 (NetBSD-6.0/images/NetBSD-6.0-shark.iso) = 91aacdd54e4ae41cbf71d4a91533bf66ef9dacaf7d71ffa81f0fabec5e98f3f5685f62b75ceac6db51a3da160a4c3e92fe3a5469165fa92c23d76d4e9aebe9a2
RMD160 (NetBSD-6.0/images/NetBSD-6.0-shark.iso) = 88a248ed7456f4c08fa9581026c555df29c05daf
MD5 (NetBSD-6.0/images/NetBSD-6.0-shark.iso) = ea424a950e7168c79a2b80152504fe61
SHA1 (NetBSD-6.0/images/NetBSD-6.0-source.iso) = 09749966ddaadad296254adec70d4f4c01f20fdf
SHA512 (NetBSD-6.0/images/NetBSD-6.0-source.iso) = d45abaf7551b96232228024a7f08d19943a8912a2383c456a66aff8bde1dc4ec60a3dc084ea4801c6c4691d714e257672f87436d7070a939fbf0fb5b2119854e
RMD160 (NetBSD-6.0/images/NetBSD-6.0-source.iso) = 002f87e50d6113c7a2820a4888b8484c1848bfa7
MD5 (NetBSD-6.0/images/NetBSD-6.0-source.iso) = 4d63a05600813e5588720da5328967ac
SHA1 (NetBSD-6.0/images/NetBSD-6.0-sparc.iso) = c60cdabd739d287e3424f8a6e2b671b7b41a944a
SHA512 (NetBSD-6.0/images/NetBSD-6.0-sparc.iso) = 1dd5305519efa2c90341088e26b66b38426ce5e42a066277605abaa0e85322a9af2031ada8faf1e303e7f10f8b29e8765673de14a5a04c4e5f2f6001b7c52241
RMD160 (NetBSD-6.0/images/NetBSD-6.0-sparc.iso) = ba9dc6b22a648f9b0af3c44eed518401b20e50ab
MD5 (NetBSD-6.0/images/NetBSD-6.0-sparc.iso) = ec7578864447eb816413264ad512b017
SHA1 (NetBSD-6.0/images/NetBSD-6.0-sparc64.iso) = a6a5475eeb3e88d520a5bbed7d62f0067c5fe65d
SHA512 (NetBSD-6.0/images/NetBSD-6.0-sparc64.iso) = 50f36f089ee8c27190512d16ec50661db4136c257c947439d670893fe610a9b07d577a7ab8b9f4d9cd6766e9daa8b619aa3a0f6f95e3abe43a4955dbf9d2d401
RMD160 (NetBSD-6.0/images/NetBSD-6.0-sparc64.iso) = b8dd4236c01dbe061cc0e0e20e71b805aee8b998
MD5 (NetBSD-6.0/images/NetBSD-6.0-sparc64.iso) = dd2dd0625585e969262fd0c99751c64e
SHA1 (NetBSD-6.0/images/NetBSD-6.0-sun2.iso) = 1c0755c7c8a094769b4c1b50566fa05e4848e113
SHA512 (NetBSD-6.0/images/NetBSD-6.0-sun2.iso) = d37ddcca3ac60bba6e8547c8bdd2029d5ee39517eff1b90534dd02eacb5e5e36b3be9a06b86386d451ee89c2408c151fb55f0427d3ba682b4076dc3f5c8925a8
RMD160 (NetBSD-6.0/images/NetBSD-6.0-sun2.iso) = 4a20723c243cd7dd43c8eb4272971072d21a2efc
MD5 (NetBSD-6.0/images/NetBSD-6.0-sun2.iso) = c2f9f999646eef2bcf2c9bdb6d2e860d
SHA1 (NetBSD-6.0/images/NetBSD-6.0-sun3.iso) = db35e360979a3c38984afb91309387fa9a415bf0
SHA512 (NetBSD-6.0/images/NetBSD-6.0-sun3.iso) = c4a8fde6c486ee8b23a485d6b2947a1620a2f7582df13a255dc8f6fd2860dd93b0e1c2dc3b56387adad49b935999c70cdb293f25f87c2f227af35e158fe6a7b5
RMD160 (NetBSD-6.0/images/NetBSD-6.0-sun3.iso) = 0ff6f3798f945c797b2ccae534cbece05e3a4d34
MD5 (NetBSD-6.0/images/NetBSD-6.0-sun3.iso) = 042f8191d0aa3b4ee986408c1bcce04a
SHA1 (NetBSD-6.0/images/NetBSD-6.0-vax.iso) = 8f43de5ed15d0178faa9b79763c7564d479711c8
SHA512 (NetBSD-6.0/images/NetBSD-6.0-vax.iso) = afa099641b4684f021c3a94f3a8624c3aad92a8083555ad3bf78028fcb530d08b9b778d96635bbe098c88e6e08be2733d01c4dfca8933d4120e7571dbc715a88
RMD160 (NetBSD-6.0/images/NetBSD-6.0-vax.iso) = 09800ce94b2007f7c51850e9db87270e0123e9f7
MD5 (NetBSD-6.0/images/NetBSD-6.0-vax.iso) = 0830c9dc45aa84e08cf97580708cc60e
SHA1 (NetBSD-6.0/images/NetBSD-6.0-x68k.iso) = 022558b77ecb7c5b6a35aa0e79c6aaab56031d16
SHA512 (NetBSD-6.0/images/NetBSD-6.0-x68k.iso) = 0cf2986fa2477df206a46bc19f566c6cbe8712ad14fbf40d9d4c20aa104a65eaeb93d73c86ac101125d0c5b5619f53a7296d032f56c15930baf496853b604902
RMD160 (NetBSD-6.0/images/NetBSD-6.0-x68k.iso) = 5becbb0b0c10a69dfaf131910da0c72a208beeb3
MD5 (NetBSD-6.0/images/NetBSD-6.0-x68k.iso) = 6f7c0d6afb962d18f61cfafc78f38b12
SHA1 (NetBSD-6.0/images/NetBSD-6.0-zaurus.iso) = 5c78cce50c6f26c64456228695ff51b12b8cd44f
SHA512 (NetBSD-6.0/images/NetBSD-6.0-zaurus.iso) = 8a0849c505a0db1ed89fb066f6814f32e9998017cfd684ac38b9027d94ba4a2ebec496ba1ee194e0f98d176820077ea50c5a45cb85b6f24a1d1a2b14e9a0dd1a
RMD160 (NetBSD-6.0/images/NetBSD-6.0-zaurus.iso) = f505c50d0049b053abd5fce076e0e67dcd13c2aa
MD5 (NetBSD-6.0/images/NetBSD-6.0-zaurus.iso) = 5e3a023090f8b8df40a7372dc1749b81
SHA1 (NetBSD-6.0/iyonix/binary/kernel/netbsd-GENERIC.gz) = 235873272a098d18934590d26ab5c636859262ad
SHA512 (NetBSD-6.0/iyonix/binary/kernel/netbsd-GENERIC.gz) = 0a4f296210a18785815dc51d191c97aa069403e2e0460038f6dc59a65a97f39d9345abf0e793a12dc562068a7eb0d825881030969ea43a48775704958700ee86
RMD160 (NetBSD-6.0/iyonix/binary/kernel/netbsd-GENERIC.gz) = 1c5a2e39c1876ee7b46a9ee9751da1d418c2a607
MD5 (NetBSD-6.0/iyonix/binary/kernel/netbsd-GENERIC.gz) = 5f2302d30d7dcfa644d62c656eeb33e3
SHA1 (NetBSD-6.0/iyonix/binary/sets/games.tgz) = e1d2bb091d8202962133475bc83edb55230b9008
SHA512 (NetBSD-6.0/iyonix/binary/sets/games.tgz) = a147fe6d0419d53d5a428802d0bb6c9fa8a84a76c96d7d6d7c0d399e5af3a69240116463549a83d53387b0214ba0207796d7c10228d92a684fe7ff0ece3aec16
RMD160 (NetBSD-6.0/iyonix/binary/sets/games.tgz) = 2be515c495548ba5135ff42a75a8ebbfc35ea776
MD5 (NetBSD-6.0/iyonix/binary/sets/games.tgz) = 9059a6f2cd41cb58827709c9e9a1649b
SHA1 (NetBSD-6.0/iyonix/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/iyonix/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/iyonix/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/iyonix/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/iyonix/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/iyonix/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/iyonix/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/iyonix/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/iyonix/binary/sets/xbase.tgz) = 875fd641f969ec00f854ca1334d084947450c740
SHA512 (NetBSD-6.0/iyonix/binary/sets/xbase.tgz) = 75d0c9e9685b6b7bb706df7427c8b52f59c4fc4d0e725b02ee18024f75a546dc41de5f1a5d8552a8d2c84bdf87856e5857b0f12914801ca7b36b69536d617b4e
RMD160 (NetBSD-6.0/iyonix/binary/sets/xbase.tgz) = 1a4d099d59b2bd882e99f1b2df1c7bf22b890d48
MD5 (NetBSD-6.0/iyonix/binary/sets/xbase.tgz) = 5eb9c573c5a0c96701f268a3470a58b9
SHA1 (NetBSD-6.0/iyonix/binary/sets/xcomp.tgz) = 8d7469a85b7433c89690cce74e6b3f4e66cf0647
SHA512 (NetBSD-6.0/iyonix/binary/sets/xcomp.tgz) = 411304cf66e300a7f162128203099bb4d6eebc4a93ba18ed7caf79f30b31cc7ffcede85c4e1bad02c3f5f8a27d009fb71f3ce5862c1d8aa477f39a25c535fdaa
RMD160 (NetBSD-6.0/iyonix/binary/sets/xcomp.tgz) = 822d26e0ee08b72ca8b991f0709a8c1bf0399208
MD5 (NetBSD-6.0/iyonix/binary/sets/xcomp.tgz) = 5af361f0334d28119d703064278b3ed3
SHA1 (NetBSD-6.0/iyonix/binary/sets/xetc.tgz) = 222d14c205f1018cc5a86eb03214d18cda2abb82
SHA512 (NetBSD-6.0/iyonix/binary/sets/xetc.tgz) = fb82261996aa772fd8a9022eadebf899ee04ee70229ee4fe07758ca14937bdb34752c8218eb4c5b722b6208f719891493bbc141372fcfa503de83e0a61aa7387
RMD160 (NetBSD-6.0/iyonix/binary/sets/xetc.tgz) = 511ebfe5076f361b7f7ca03a265d16052a16d259
MD5 (NetBSD-6.0/iyonix/binary/sets/xetc.tgz) = 165cffbb6d2c79663198d80c82afac1e
SHA1 (NetBSD-6.0/iyonix/binary/sets/xfont.tgz) = ce0e141beeffeb5b470478e9887e4127da85de3c
SHA512 (NetBSD-6.0/iyonix/binary/sets/xfont.tgz) = 5d98354ff77103793d56af3192ab5f9afcde62208791976064ac33a149f994e0227dd5dd28ed544eeac5ea504e5212d5fb9e7d05e59e5a48a2fe4062e34e3106
RMD160 (NetBSD-6.0/iyonix/binary/sets/xfont.tgz) = 4d09f5e82fec875ef0ea328809465e7a98f2ebfc
MD5 (NetBSD-6.0/iyonix/binary/sets/xfont.tgz) = be26b3890261302420b09976a1d38144
SHA1 (NetBSD-6.0/iyonix/binary/sets/xserver.tgz) = 4ccfef1d49179603286042088bec0218c6b2031d
SHA512 (NetBSD-6.0/iyonix/binary/sets/xserver.tgz) = 809fc0a9065c1b9e7cd17aeb0d41cfd3158efd6c585556af01c4a3c93f0df3646a2ff77c6994449a703a408c0988fc1aa68cf34c7e297a3056d73ded57034c12
RMD160 (NetBSD-6.0/iyonix/binary/sets/xserver.tgz) = 403604404261764c74f10fab774c4653fd20cc9b
MD5 (NetBSD-6.0/iyonix/binary/sets/xserver.tgz) = 914f909b6878cef16a92765a8cbb52ca
SHA1 (NetBSD-6.0/iyonix/binary/sets/kern-GENERIC.tgz) = 13354b2863a67ec38b91a8b8cab2ae5081cf68cf
SHA512 (NetBSD-6.0/iyonix/binary/sets/kern-GENERIC.tgz) = 02ad11423d70de2c4b1e937bfdb6348af4025c026513f4f05768f2344997a4414896f10b72af056410dcc62d8760449c2dd47ff2bd37d0e4011d45a1fe0e98b1
RMD160 (NetBSD-6.0/iyonix/binary/sets/kern-GENERIC.tgz) = 4a98f89230f9faaf839d5b607097d7612aa6741e
MD5 (NetBSD-6.0/iyonix/binary/sets/kern-GENERIC.tgz) = ecfbf309ba59a5ed39e46ed54b8173e4
SHA1 (NetBSD-6.0/iyonix/binary/sets/base.tgz) = 697554f5182bff135cb94ba8c99b29d719aeefc1
SHA512 (NetBSD-6.0/iyonix/binary/sets/base.tgz) = 52b73ea3de1d96f706a48b02e7cc6234576d5bad72828ef34e61929c460ea839f6051e3a650880307a86757f35265c2a33780bbe4640d301f70a4ebae06c7e8b
RMD160 (NetBSD-6.0/iyonix/binary/sets/base.tgz) = 7e7a5a24095a326e8998ba95c291c5380a89600f
MD5 (NetBSD-6.0/iyonix/binary/sets/base.tgz) = 74682becd88ad5b73b5434940f1e8436
SHA1 (NetBSD-6.0/iyonix/binary/sets/comp.tgz) = ab05d7d706899d1a555b6ec101f6d13d7f13ad97
SHA512 (NetBSD-6.0/iyonix/binary/sets/comp.tgz) = 246d589d8d3c37375496ddd3e15da6a5bb101bc1405c13ba2751818555eaad37c403e27437326bac2ff3c20763c95a49df0bfdb324faf3110c7ec01d7676b78e
RMD160 (NetBSD-6.0/iyonix/binary/sets/comp.tgz) = a5835f803a5c74d6484e7171085460b803705e22
MD5 (NetBSD-6.0/iyonix/binary/sets/comp.tgz) = 53c3ef4ec81bc6b1f89e15081ad4bb48
SHA1 (NetBSD-6.0/iyonix/binary/sets/etc.tgz) = 7294ce630333a9bea7392c5083118cc68bcb5732
SHA512 (NetBSD-6.0/iyonix/binary/sets/etc.tgz) = f1699017e37109605eecfb44174327a974f1a7a741e16f67f387170ddd1b2c661c58bd9b6dc0795153e828ef52d16cc45b37743cb86be98623da38736938061b
RMD160 (NetBSD-6.0/iyonix/binary/sets/etc.tgz) = 1a9cdec0a39de8c3d0b2ec63b7c674bd597e67e0
MD5 (NetBSD-6.0/iyonix/binary/sets/etc.tgz) = d655e4ef20778005efe66b02cdd5f471
SHA1 (NetBSD-6.0/iyonix/binary/sets/modules.tgz) = 2689f2acb9a977df7801394cd49f6f84d93fab8a
SHA512 (NetBSD-6.0/iyonix/binary/sets/modules.tgz) = 853553a5ae0acde4b24a80372c1fa71777ede655a76c60f1d2bc835df50fc2ff03270a5edca72c492fe07a3208a8680a10277b83fab4e7a3f064ded4c91cd151
RMD160 (NetBSD-6.0/iyonix/binary/sets/modules.tgz) = 4d2d0ba3492b751accda0a03daef7cba859f223f
MD5 (NetBSD-6.0/iyonix/binary/sets/modules.tgz) = 129dd8d6331eacc63934568b1467edc9
SHA1 (NetBSD-6.0/iyonix/binary/sets/tests.tgz) = b54fb7a814d7bebe4a40a00c17253d0bc6ca4544
SHA512 (NetBSD-6.0/iyonix/binary/sets/tests.tgz) = c99e6c4b80fc2d6a0f236f324f844a562db128876a5b89ced62097a732b273fb6d9af8fde55f3e949768882cb31fab2918cd731f1259b67a1b2548030417c2de
RMD160 (NetBSD-6.0/iyonix/binary/sets/tests.tgz) = 61ee29ddcfd0268fb7ef37829aeb75e3152b76d8
MD5 (NetBSD-6.0/iyonix/binary/sets/tests.tgz) = ce8b49fc2bfd31965b5b989de17d5cd1
SHA1 (NetBSD-6.0/iyonix/binary/sets/text.tgz) = 6f119b3eb99d94a4aa41b5bde28856b3daa83d84
SHA512 (NetBSD-6.0/iyonix/binary/sets/text.tgz) = 03a29fcc5ba8ce7d6ab590d61bc0a3f4dbb9bbce89d4e8cefc00cf398fa92e2363f3e3fa5da4cc17d3bd9195434c5a951550c6091b1168614a6a0b2831f3f881
RMD160 (NetBSD-6.0/iyonix/binary/sets/text.tgz) = 2a1331121dc41105accbf9b491bcd4aaa3589907
MD5 (NetBSD-6.0/iyonix/binary/sets/text.tgz) = 9bd5612ff2228f1bfee64c0f6a872fe4
SHA1 (NetBSD-6.0/landisk/binary/kernel/netbsd-INSTALL.symbols.gz) = d9440af1a917beaff6598eec266db86853625b08
SHA512 (NetBSD-6.0/landisk/binary/kernel/netbsd-INSTALL.symbols.gz) = 982b13dc871a9e562e191ff6d156f202cef18174b66e18adac8e77037fd4edbd29848dfbfec15f94744375b7967bdbd0c5bac0414eac22b78c5ea1fae91f0ed6
RMD160 (NetBSD-6.0/landisk/binary/kernel/netbsd-INSTALL.symbols.gz) = 6320116f0640f069a52a4c52196ba45a306b594f
MD5 (NetBSD-6.0/landisk/binary/kernel/netbsd-INSTALL.symbols.gz) = b6d766477c4b69e1e3003d68a28d742a
SHA1 (NetBSD-6.0/landisk/binary/kernel/netbsd-GENERIC.gz) = 2f1a4b4a1b4b88ab919ce3addcf2310da10c1c12
SHA512 (NetBSD-6.0/landisk/binary/kernel/netbsd-GENERIC.gz) = 21fe1b84d9c5c85f3c9d963ce9497d6b8250a3c5789dcf387612089aea424a728503c9adc5e047d8887959be84a8cbbdf397b7f2685fbef2d0fab448238279fb
RMD160 (NetBSD-6.0/landisk/binary/kernel/netbsd-GENERIC.gz) = e35fb6d1b917230c4b622ea52276a47e2a711d33
MD5 (NetBSD-6.0/landisk/binary/kernel/netbsd-GENERIC.gz) = b6c0c1333ca3b1a45d3a828f7d18d918
SHA1 (NetBSD-6.0/landisk/binary/kernel/netbsd-INSTALL.gz) = 3348d95ce13a6217331a1992064194aec1775db4
SHA512 (NetBSD-6.0/landisk/binary/kernel/netbsd-INSTALL.gz) = bfa4fa3ab82db17aef5f5b4b818cb8fb5d9101d6709752bd88231042ce3c2c731a452a1360fdcd77d8eb3c0cd7368de7aa359b65b7d35d4f78d8182670de4a72
RMD160 (NetBSD-6.0/landisk/binary/kernel/netbsd-INSTALL.gz) = 95d2efee62025110beb81c829712563b4495633b
MD5 (NetBSD-6.0/landisk/binary/kernel/netbsd-INSTALL.gz) = 143697e4aa570e39175812011fca30f1
SHA1 (NetBSD-6.0/landisk/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/landisk/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/landisk/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/landisk/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/landisk/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/landisk/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/landisk/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/landisk/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/landisk/binary/sets/kern-GENERIC.tgz) = 4ed0c44b9ce3362aa10429a75e3dcd4a01963055
SHA512 (NetBSD-6.0/landisk/binary/sets/kern-GENERIC.tgz) = 1764a9d7ef8a4fc9cc321cecd486f767096a6bc34eb83b3bf3fd24c0383a3df139200560a7846d23bebdc4e02feffe7252a26689b2971964b9240a26abf4acd7
RMD160 (NetBSD-6.0/landisk/binary/sets/kern-GENERIC.tgz) = 75e8fa6b9c66cec999c28d9242379df0142d0bdc
MD5 (NetBSD-6.0/landisk/binary/sets/kern-GENERIC.tgz) = 7226e2efa45822998b444dfbce3affe8
SHA1 (NetBSD-6.0/landisk/binary/sets/base.tgz) = 7cef6d1878fea6a5448eb32456056155477dbb47
SHA512 (NetBSD-6.0/landisk/binary/sets/base.tgz) = fd16dca9619d1e1a3770486594935e091bc7219b24db8bf91ec0cc8b064f2a201d13dca61b50885e1a2428512bbf510ea7f69a9b8174ac82f511f66c6a8cb70b
RMD160 (NetBSD-6.0/landisk/binary/sets/base.tgz) = 97e06530ed9a98f0d48746c33db25e2ee0460360
MD5 (NetBSD-6.0/landisk/binary/sets/base.tgz) = d0e0468ef015bcada5e688057a53efc2
SHA1 (NetBSD-6.0/landisk/binary/sets/comp.tgz) = fb19c5e935ea83f6c5679796cc9527fa3e435508
SHA512 (NetBSD-6.0/landisk/binary/sets/comp.tgz) = 0636a207ee4126ef209fee7b5fdbee6eea677cd8b640798aeab5138803fbe8754de69aeff925e3298783409009e6cf8658f9983a213425d397e90f76a93e3f7e
RMD160 (NetBSD-6.0/landisk/binary/sets/comp.tgz) = 0f4190b6173c8618f7609ba726e73cb041ac75e6
MD5 (NetBSD-6.0/landisk/binary/sets/comp.tgz) = 7317e108ba56869a21b9d7137605eaf5
SHA1 (NetBSD-6.0/landisk/binary/sets/etc.tgz) = 25f6f82b594c5d383d45e6c7ee3e0720afe0cb32
SHA512 (NetBSD-6.0/landisk/binary/sets/etc.tgz) = f2f77e86753a1193cc4c37372a303a88e6c51fd0d3d2e6ffa1927a697aab915c9fd5f1f7ac81137e9182388a6a60912b0b2d6d6975abe0029cf9206367b7454d
RMD160 (NetBSD-6.0/landisk/binary/sets/etc.tgz) = a5db723013194035deb0e61dc04f6985fcc4a51f
MD5 (NetBSD-6.0/landisk/binary/sets/etc.tgz) = 92d41bd9cd00823be1afb0d3846e10e2
SHA1 (NetBSD-6.0/landisk/binary/sets/games.tgz) = 65ef265cc562c21af9cea790c1eb0f7d1ee4ac50
SHA512 (NetBSD-6.0/landisk/binary/sets/games.tgz) = f55cb863e6bd20e52f04d643199317ffeb673d8ad7a1b1506ccb6d798d6b511766d8188b1a567ab148ad38ab8dfb4516eb624044827880a0c1f289631e9d7f88
RMD160 (NetBSD-6.0/landisk/binary/sets/games.tgz) = 9a0cfcc234841f9b3086c8e6479686a9e486fe27
MD5 (NetBSD-6.0/landisk/binary/sets/games.tgz) = 9f57cb59ad62f29327c1f1daf6d8e130
SHA1 (NetBSD-6.0/landisk/binary/sets/modules.tgz) = 0755c390f758947aaa8fa5942c3600a9c1da0155
SHA512 (NetBSD-6.0/landisk/binary/sets/modules.tgz) = d20bbd1969460b825f413f06e28280bfc549d74e82287ec6f8b7eb70db2dc8680fcb5f1b42e96e48feda3c377eda9ef4890bcb03aeb2401dc45cc79353826a5a
RMD160 (NetBSD-6.0/landisk/binary/sets/modules.tgz) = b43dea943dd41ece88282f1a858926afba42f061
MD5 (NetBSD-6.0/landisk/binary/sets/modules.tgz) = 77fe2185916c4e1df174079291a33e61
SHA1 (NetBSD-6.0/landisk/binary/sets/tests.tgz) = d5cafffe5049a81843f684c81ff1909669e6e8a4
SHA512 (NetBSD-6.0/landisk/binary/sets/tests.tgz) = b01eccf78b1fac7ca92a10e6440a80ccccbcbc90dcafc44692d149a0b719c9058de6e1a28207f06fa7db29403552df527c200a27ffda708cf49ae2f2c7033886
RMD160 (NetBSD-6.0/landisk/binary/sets/tests.tgz) = 0a2d702412e9add961586f2cd87eb2895889cdef
MD5 (NetBSD-6.0/landisk/binary/sets/tests.tgz) = 7fa1f86aabba0e9ff906164c5046418f
SHA1 (NetBSD-6.0/landisk/binary/sets/text.tgz) = 753e3063d3a42b43208bfb9ee44cadfbf777b02b
SHA512 (NetBSD-6.0/landisk/binary/sets/text.tgz) = 63c337c80b7fb440d813803b06c0b34d1a484c1e5a18f1bcf6fe90f3880bde3269c8ed7fb0485405bd9ba618069fbe680163be2b4c53c6d81ff8e263777fd51c
RMD160 (NetBSD-6.0/landisk/binary/sets/text.tgz) = 83afeda6e537627561023c5570ca05aaa2d3027c
MD5 (NetBSD-6.0/landisk/binary/sets/text.tgz) = fae31d47dbe89c926a95c602012bcd36
SHA1 (NetBSD-6.0/landisk/binary/sets/xbase.tgz) = 240739fa38b2f6bf5ff63504fc02b5e006d404fa
SHA512 (NetBSD-6.0/landisk/binary/sets/xbase.tgz) = 93ace6b1ec640cb33e586a2e3b4b53c46b34efa1c3cb40ac7bde2531a6b253a42fa92564ba203f7f5b725b9673017766ba5a7636ac32e6722a3787559be916be
RMD160 (NetBSD-6.0/landisk/binary/sets/xbase.tgz) = e76eddb131991fd9641a77469b782e1672370644
MD5 (NetBSD-6.0/landisk/binary/sets/xbase.tgz) = f0502dec4137d4ba757447285b98f09e
SHA1 (NetBSD-6.0/landisk/binary/sets/xcomp.tgz) = 06e8daf72bf753e226e3d5df7b47ef8344470c27
SHA512 (NetBSD-6.0/landisk/binary/sets/xcomp.tgz) = 101b0c365ad2209e8b4520f586a9cd60595b0dfa414c061cf24cfbf18a9455ab6a786f968d3a5eb5f06db57cd9cde4262d5031aaf6d5cdde152ac92c91a31c09
RMD160 (NetBSD-6.0/landisk/binary/sets/xcomp.tgz) = 3bfd31356e625cbb25f035db41d44d3a4523fb25
MD5 (NetBSD-6.0/landisk/binary/sets/xcomp.tgz) = 2d4a2c3e4a93d2b1e56e7641bfa730d9
SHA1 (NetBSD-6.0/landisk/binary/sets/xetc.tgz) = 06a24e45e73bf98cb87b951fe3bfe4a81a1d0793
SHA512 (NetBSD-6.0/landisk/binary/sets/xetc.tgz) = 72d35ff20eec9cb6674031e31c382ff63fea51f6cb10290a540cb83de5639145faff66119fe7f176ba97bb389b9edf4aae3838f008df9f1b111699fb8555f4ee
RMD160 (NetBSD-6.0/landisk/binary/sets/xetc.tgz) = e63035911479f2a87f20fe5f030ef31b3bee1878
MD5 (NetBSD-6.0/landisk/binary/sets/xetc.tgz) = 309875d5ca3be7fb66f2ec0ab6e2e18b
SHA1 (NetBSD-6.0/landisk/binary/sets/xfont.tgz) = 06922d4fa8af9ec61283885a5040552442f1a0dd
SHA512 (NetBSD-6.0/landisk/binary/sets/xfont.tgz) = 48790c97453934379e37f2826498b64480554cad16ee807d44e83eed4c53b324659df0cc8aa954ec6f5f5a16ae03209a0c03dcf45db3bd807011d6d16d66654a
RMD160 (NetBSD-6.0/landisk/binary/sets/xfont.tgz) = 5f1a4142f95c2fbf064d94e11afb0660d17c384f
MD5 (NetBSD-6.0/landisk/binary/sets/xfont.tgz) = 0a8a39b6c5e1ae9be657196b82c16717
SHA1 (NetBSD-6.0/landisk/binary/sets/xserver.tgz) = fc311910c92490d19bb81ff540f65370a01cfa9d
SHA512 (NetBSD-6.0/landisk/binary/sets/xserver.tgz) = 8f8f392ab9696e20149e94dc85061be26982179e63fbfaac269dcb13509d4e9ec44c8188b6f333cad0b2139703c78e1f0919b71394ac6ac2526ce7935ec2329d
RMD160 (NetBSD-6.0/landisk/binary/sets/xserver.tgz) = 86ec983a4047e7a0c146cfab94df6331f0363e89
MD5 (NetBSD-6.0/landisk/binary/sets/xserver.tgz) = 2befe5df4dce2ccb1770ac2019da8ec6
SHA1 (NetBSD-6.0/landisk/installation/misc/boot.fs) = b8bbc9b7108311040a9a3e1bad714efccbdeab21
SHA512 (NetBSD-6.0/landisk/installation/misc/boot.fs) = 8fbf69cfcb9e7e104c244a8cc2c11d0bc2e2f006eb8a5b8e97267604519fb0e415a9da6bc81a0e75fd30e2ce8fb40a4026a97cbd413de71c05caa1e5f16871fa
RMD160 (NetBSD-6.0/landisk/installation/misc/boot.fs) = 333044e352899852a8fbb7308a6009c1b5a69bcb
MD5 (NetBSD-6.0/landisk/installation/misc/boot.fs) = 47ea13afac308dfe956f2d5c16a3a285
SHA1 (NetBSD-6.0/landisk/installation/netboot) = ca38444ab6911f5b12f0797a6198f082030ba9b3
SHA512 (NetBSD-6.0/landisk/installation/netboot) = 064115997cd9a49abeb68a811406fad102a2ad325dd1e3ddac075f19e86d04d660bc4f6640483f825310e51d7cfdd90de4997c52dabc883e64f7d984444ef276
RMD160 (NetBSD-6.0/landisk/installation/netboot) = dad8b6e8b216ef75971c7805baf4b4f86ecfad60
MD5 (NetBSD-6.0/landisk/installation/netboot) = d29e773e58700dde6b5384464b517785
SHA1 (NetBSD-6.0/landisk/INSTALL.html) = 5222270ac1c6fa408d5ef0bb7e901d99e916be72
SHA512 (NetBSD-6.0/landisk/INSTALL.html) = 4011e42ffc09b225893c360690d01099ef76445c496e10c2434e50f8c97755d15038206bd2bc9bb4ee773c24e7b5bb64bf969a1e4ac4e07f4ec84c0beb21bf8d
RMD160 (NetBSD-6.0/landisk/INSTALL.html) = 48a1b2c374b38629c87e1c04e661e36182caab0a
MD5 (NetBSD-6.0/landisk/INSTALL.html) = dfa8413fc180b897d13c0964362e7b3f
SHA1 (NetBSD-6.0/landisk/INSTALL.more) = add61a5d904d34e380a4fcd69cb7fffd24143a42
SHA512 (NetBSD-6.0/landisk/INSTALL.more) = 4165166b8cc79bd589c80490352c0741edc2087e7560658ea0da993347301f94b3685babd2c40e8927fdd8ca9a58350232dad0cc1ec0c247890a283f1ca191b2
RMD160 (NetBSD-6.0/landisk/INSTALL.more) = 7416e0e529dcbd99dca03cb901e7038cc60731a5
MD5 (NetBSD-6.0/landisk/INSTALL.more) = 871f600baa29f1db802884fc26783526
SHA1 (NetBSD-6.0/landisk/INSTALL.ps) = 734d5c64f3e69a68dd0b987492851f7f915692a9
SHA512 (NetBSD-6.0/landisk/INSTALL.ps) = a34a07c314fe9a7e4f9dfc296f9e05116675f722ad0aaab757d76c31246a9182398bca5906c3882ff1b14fda9480bb674f12daa90274df9968f2c374d7a2d356
RMD160 (NetBSD-6.0/landisk/INSTALL.ps) = 88c78a031426844ab1afca9dec3ba7289e994dd3
MD5 (NetBSD-6.0/landisk/INSTALL.ps) = 446d71c39365e90bf6491895a781a4e7
SHA1 (NetBSD-6.0/landisk/INSTALL.txt) = f4d9af9565019dda406298ac72875959cdcb1b72
SHA512 (NetBSD-6.0/landisk/INSTALL.txt) = fe20c04bf5c15d95d5bcf8cafc9efd9ffb07e37ed6234c7abe8a2ff01fae1cb7aca2c4eb28a7d032212a31cae34331d0ce9fad7a3f7a01bb0a9b66e40352f5ec
RMD160 (NetBSD-6.0/landisk/INSTALL.txt) = 3739b2ac5572c87aa6e8a86c6b33023db1c5e6f1
MD5 (NetBSD-6.0/landisk/INSTALL.txt) = 7e1c46e08b5773bcd5ecc0717834c25a
SHA1 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.symbols.gz) = 6d24a14ed0376a5672de20c9c3b64137fcfe3b90
SHA512 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.symbols.gz) = 5b4957606a4a575aa8c0fea6dbb10df17ef4ac7afa8624c456cc625a0e0e2d1d86c2ca9a4fcf770949933af22cda99479bfab55d383878d0acd8f0c0a260820f
RMD160 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.symbols.gz) = bd90a974d6697d2921362ed308fef5df89a6a7f5
MD5 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.symbols.gz) = b5fa03bb653ae6c758c1374109969d01
SHA1 (NetBSD-6.0/luna68k/binary/kernel/netbsd-GENERIC.aout.gz) = 1a6b46426c3e93110e9138dc56dbf4e15391c7c4
SHA512 (NetBSD-6.0/luna68k/binary/kernel/netbsd-GENERIC.aout.gz) = fe4cf575cd9e08ccf01adec28e38311c7fda42dfd2de5d17f98be37fbac8023b9fa4225c58b5d46728af2f3885825f34068acb82f600447ed60ec54ccc5753ad
RMD160 (NetBSD-6.0/luna68k/binary/kernel/netbsd-GENERIC.aout.gz) = d1e8da51257a921f3d3c0ecec9f5a06712018302
MD5 (NetBSD-6.0/luna68k/binary/kernel/netbsd-GENERIC.aout.gz) = 1d29443ff1170766ece9a482e2468f73
SHA1 (NetBSD-6.0/luna68k/binary/kernel/netbsd-GENERIC.gz) = d36634bfc138c395323135e9c60824b7f987b588
SHA512 (NetBSD-6.0/luna68k/binary/kernel/netbsd-GENERIC.gz) = b7cdfcd815eea67d65ea003a2e56fd0a83ddfa1695749a5316566f9e59ef69dd9d78e80c54e99c9aa2db57d8e0ee9a3af43580ed40b45086b357362428e4faf1
RMD160 (NetBSD-6.0/luna68k/binary/kernel/netbsd-GENERIC.gz) = 2d36be27f36008788fee498a9c18dbe7a3d66c08
MD5 (NetBSD-6.0/luna68k/binary/kernel/netbsd-GENERIC.gz) = ca10cc06f8e60b045230888b523f71ad
SHA1 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.aout.gz) = e081ce3c03ff74c60f2a0ba80540e5b398aac926
SHA512 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.aout.gz) = a02d1a6974fe702be5e51419440ebdf00bd333b0c4b88cf1289a992bfc21be4f731edb386bdb5e559b5f1fac064e392b97a800431538bd21e76ebb2f6f1c24a6
RMD160 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.aout.gz) = 0db65548da46e8d651e8b2b9d4ee3b1f95086f61
MD5 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.aout.gz) = 22e3b4a434ae0bf2e200bd4f6f223ff9
SHA1 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.gz) = 43172b3e8115db8f41cff2170f4b5d47dd1da8e5
SHA512 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.gz) = e7e217acc802aff14a3ea852a1dd0f62088dea15ecadc3cc9b335259ecabf352028dfa5929ff139f31f5b299311a5e42e944c0979be0299501caed3592e1d86e
RMD160 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.gz) = c53f3f5d7e4150eb768b377877d8a8d71f22d156
MD5 (NetBSD-6.0/luna68k/binary/kernel/netbsd-INSTALL.gz) = 5d3869dcf1301987a12ffa512b626d19
SHA1 (NetBSD-6.0/luna68k/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/luna68k/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/luna68k/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/luna68k/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/luna68k/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/luna68k/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/luna68k/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/luna68k/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/luna68k/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/luna68k/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/luna68k/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/luna68k/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/luna68k/binary/sets/xbase.tgz) = ffc6dc889560371042dd12a5486a9a2c499b86a9
SHA512 (NetBSD-6.0/luna68k/binary/sets/xbase.tgz) = 9aa2ba4d703a13020acf8543d1efcde76b6f0f4b411be20f804709b3802e537be1e651eba469830c4fd4777a379ea9936200a3a3eb4b1a702c04bb81aafcc723
RMD160 (NetBSD-6.0/luna68k/binary/sets/xbase.tgz) = 3ecf53e7414ef859a47fe5afc108cf3bfa82fb1d
MD5 (NetBSD-6.0/luna68k/binary/sets/xbase.tgz) = a6af002ce158cf72eb5c6a7cde2c0989
SHA1 (NetBSD-6.0/luna68k/binary/sets/kern-GENERIC.tgz) = dca3d0e2fc1d884287bd283e81bc797ceeb13195
SHA512 (NetBSD-6.0/luna68k/binary/sets/kern-GENERIC.tgz) = 58388c5d51f6f09f0884904ba8b18ae63374f807312177898080267382141c631e6aa0698a55b427e5673b34fa995cb3d8cd7e77dc1036c9812fe8356f976dc7
RMD160 (NetBSD-6.0/luna68k/binary/sets/kern-GENERIC.tgz) = d9c5321bc472e64656a250e3e3bd6d72119435ce
MD5 (NetBSD-6.0/luna68k/binary/sets/kern-GENERIC.tgz) = 374856b19bd6eb0c072c4b25933f5f8d
SHA1 (NetBSD-6.0/luna68k/binary/sets/base.tgz) = bc10452b7a7a1a24dec1813d74ff78d1d82ae750
SHA512 (NetBSD-6.0/luna68k/binary/sets/base.tgz) = 5525e21ff9566b0c7ce169e2a087094e41cd3e60d4f67d8ae026c4169cc5b606cb88f600e4410570053dd9ea8202b0da0fd7293fc8c6531c722bb4109accc605
RMD160 (NetBSD-6.0/luna68k/binary/sets/base.tgz) = d763e9b40ced7122fa0ccb461386a7b7ff1dce1a
MD5 (NetBSD-6.0/luna68k/binary/sets/base.tgz) = 665d860c4af2fd73227eda7bbfbf124f
SHA1 (NetBSD-6.0/luna68k/binary/sets/comp.tgz) = c69dde9c6ef3a0ed4457ced09c7016465b0e82df
SHA512 (NetBSD-6.0/luna68k/binary/sets/comp.tgz) = bbb46b665a23fb728945c51cc19ae85da4e33da19d91ce15ed24d7832258c0cb53f75351c869aa46704cf09245403b9b28fb55624e0b8e28c2e509b653d459ba
RMD160 (NetBSD-6.0/luna68k/binary/sets/comp.tgz) = c622dbd3522567457fd890414386ebbebf6204e9
MD5 (NetBSD-6.0/luna68k/binary/sets/comp.tgz) = c0c2030d7bd68b455be0b4094a1e949a
SHA1 (NetBSD-6.0/luna68k/binary/sets/etc.tgz) = 7d1db50a753d92b90ee1eaf6b4a4f02cab03d981
SHA512 (NetBSD-6.0/luna68k/binary/sets/etc.tgz) = 86c1904b4007d3a1f4a91f30140a3c6d53af8b90c38c01fee75b679484096079d357e7a71c027464a5d48d4789d9c4a04f3f2545df907953c739a22877ae130e
RMD160 (NetBSD-6.0/luna68k/binary/sets/etc.tgz) = 52873f9a82119f52ef847e3e27e00ab4e10c68cf
MD5 (NetBSD-6.0/luna68k/binary/sets/etc.tgz) = cef84e15f32a2bf92f3657f981235991
SHA1 (NetBSD-6.0/luna68k/binary/sets/modules.tgz) = 9428e74da3601eab14416d01bc4880dcfe21a6f7
SHA512 (NetBSD-6.0/luna68k/binary/sets/modules.tgz) = ec64bcf7ebed5af30e85708133fcac6f0946aa0ba7175906b1f9ed06c300c23028ba0ba1957c1e4fe6d4c7a99877e7ed104b1439cec728db79fe9f3cad3eccb1
RMD160 (NetBSD-6.0/luna68k/binary/sets/modules.tgz) = 8060c8587dfadee593f84db43c0e63acfc259e06
MD5 (NetBSD-6.0/luna68k/binary/sets/modules.tgz) = 1477ab794d1c6494bb5a9628820c4c08
SHA1 (NetBSD-6.0/luna68k/binary/sets/tests.tgz) = 8380e48719aca55a72f3f91629e52a774eb7e4d7
SHA512 (NetBSD-6.0/luna68k/binary/sets/tests.tgz) = eced534537094e8ee83154a5319f89f4cf8e46bf8715ff6329a01f8824e231bdf726ea6f6cd4ad940d0a0a45b5b99c27eb5083e69ae915f56b2beaa7d8b63d89
RMD160 (NetBSD-6.0/luna68k/binary/sets/tests.tgz) = 05c755294af5ff529c495865595c8bfe7f5d91fb
MD5 (NetBSD-6.0/luna68k/binary/sets/tests.tgz) = 4e8df8ec4228f385e7eb7d097904a8fc
SHA1 (NetBSD-6.0/luna68k/binary/sets/text.tgz) = 19f8fb95eef7c6162c661a5262f5f41fd92f6078
SHA512 (NetBSD-6.0/luna68k/binary/sets/text.tgz) = f2a4f42b05526d21fffc8d4783885b155bdcd8183b194058e9554bb34eb47a6b0ba2d6a9779c12f17844a6bb2fc07530a7a1d7dcacbec44b437fbc08796b65bf
RMD160 (NetBSD-6.0/luna68k/binary/sets/text.tgz) = f01ee7d5dd88c612e8c8b3aae25ca93e1c70df0b
MD5 (NetBSD-6.0/luna68k/binary/sets/text.tgz) = c55b5fb2f398aaebe749d7d2a6fc8570
SHA1 (NetBSD-6.0/luna68k/binary/sets/xcomp.tgz) = 01bdf2494bf84313fb000f36606161f17cbc0114
SHA512 (NetBSD-6.0/luna68k/binary/sets/xcomp.tgz) = d4fb1fb22011bf3b178593e1a2be51626563c6ccb35f3c2db922374d9dc1dc1fdd7830df24f1f699a89c46d5a7044c221372015134682aba5a2010d5f44ef77f
RMD160 (NetBSD-6.0/luna68k/binary/sets/xcomp.tgz) = 03191d2a871d99076fe73355da23bfaf537294bf
MD5 (NetBSD-6.0/luna68k/binary/sets/xcomp.tgz) = c30eac2a665f533444de26dafbbdd77f
SHA1 (NetBSD-6.0/luna68k/binary/sets/xetc.tgz) = 3d78331b298a38fb6bd231bff59cccdedceaf71a
SHA512 (NetBSD-6.0/luna68k/binary/sets/xetc.tgz) = ecaaaaa343d9abb3d24d147826956ad64fe96338d38511b5226d3cf2cc564eb8cc39aa833c7e0875fc4378aa6b8c2f3a7ad17c8906f8c68e5adc16c3cec5b715
RMD160 (NetBSD-6.0/luna68k/binary/sets/xetc.tgz) = 7734dfa9521226f76e23b91d237109bf30855cbf
MD5 (NetBSD-6.0/luna68k/binary/sets/xetc.tgz) = 0e5ef946483a93d487b70b526477efb0
SHA1 (NetBSD-6.0/luna68k/binary/sets/xfont.tgz) = dca18181278f672451b3902ebdbf3d79ccfdedb4
SHA512 (NetBSD-6.0/luna68k/binary/sets/xfont.tgz) = 3979a9cc4b1731c8b31f44cabbe74920cb03d7e7d551743eee78442e0ba67a6faa854cdc674cd0338422aa09119ae653f2fdbd2c78dc6811c49661e812ce7fc1
RMD160 (NetBSD-6.0/luna68k/binary/sets/xfont.tgz) = 9b946721f534b2fd651e742f9061ef5d648efd3a
MD5 (NetBSD-6.0/luna68k/binary/sets/xfont.tgz) = a3b4aad9b96343e5565a441b3376f454
SHA1 (NetBSD-6.0/luna68k/binary/sets/xserver.tgz) = 859f7b47396c427f5d6c3570f6f68227131b2b34
SHA512 (NetBSD-6.0/luna68k/binary/sets/xserver.tgz) = e1eee654564f67397c0324d8f89e62b0fcb40857b423ad2fb47c04d3bca7fa833a04adef639baf3c48bc27400f4d8a548d55efc7e29dfac74a82794e5182ce2a
RMD160 (NetBSD-6.0/luna68k/binary/sets/xserver.tgz) = 1f75e4523b978323e96a0c986edbd700ae351846
MD5 (NetBSD-6.0/luna68k/binary/sets/xserver.tgz) = e50f795889ea9f0cfc49fcc6d8991ec4
SHA1 (NetBSD-6.0/mac68k/binary/kernel/netbsd-GENERICSBC.gz) = 33453c6105617ec6e41c81d71be7b04b68f26e24
SHA512 (NetBSD-6.0/mac68k/binary/kernel/netbsd-GENERICSBC.gz) = 9ea5036462094ff9ff7696cc13f9b76166f4c6058ee8e79547898866c27e96b5e31e4e27228fc7bddc69b0f11ca242113778b2ab8c6c458d16fd08e7b5148557
RMD160 (NetBSD-6.0/mac68k/binary/kernel/netbsd-GENERICSBC.gz) = aa85610097d3144646f7bdcbdfd1cd55fd248280
MD5 (NetBSD-6.0/mac68k/binary/kernel/netbsd-GENERICSBC.gz) = 4b4ce73206f5d0203aedf08b4e9e73e4
SHA1 (NetBSD-6.0/mac68k/binary/kernel/netbsd-GENERIC.gz) = 644b65114d6ffc9bef291f18e3935ca539ed29e5
SHA512 (NetBSD-6.0/mac68k/binary/kernel/netbsd-GENERIC.gz) = 8e590640af3dfd78202b83a4bc6afec869ee37b467a10b3221c08e54c3a66bddc1726cca88d09fa98598f77d8bbc6dd36c136c685bab7e75198a9b6fa89e11da
RMD160 (NetBSD-6.0/mac68k/binary/kernel/netbsd-GENERIC.gz) = 2bccecd72133b03055457567ea8eca57acc56ac6
MD5 (NetBSD-6.0/mac68k/binary/kernel/netbsd-GENERIC.gz) = ece356c2c95629fc502bb2e9382f9b4c
SHA1 (NetBSD-6.0/mac68k/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/mac68k/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/mac68k/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/mac68k/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/mac68k/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/mac68k/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/mac68k/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/mac68k/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/mac68k/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/mac68k/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/mac68k/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/mac68k/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/mac68k/binary/sets/xcomp.tgz) = e13548b8974f5d55e98caf17b2b4bbbcf2430370
SHA512 (NetBSD-6.0/mac68k/binary/sets/xcomp.tgz) = e4b13c798df04ffef88e72100e2db5af400e5ae255cd7a1ca2907736b8138a59045b6f3bd8531b5753d0b87eadb1b5f801043a929bb25d1611e4be693c1da954
RMD160 (NetBSD-6.0/mac68k/binary/sets/xcomp.tgz) = 3b1f4e26ddd57e69afd9709553c1bf70d6e29ea0
MD5 (NetBSD-6.0/mac68k/binary/sets/xcomp.tgz) = 894bb9aa05b164d9923682165d708335
SHA1 (NetBSD-6.0/mac68k/binary/sets/xetc.tgz) = d2ca34bd668e108cbbfd4b45720823663339f8bd
SHA512 (NetBSD-6.0/mac68k/binary/sets/xetc.tgz) = 431ca465f300e7eb5d4dbd44c43e412a296fb4d3129217e752f86a720e8a8bdf736c9add05a45518a4f3e6cffa32381ef61e9186d360b68a5f8c64d87e385fb5
RMD160 (NetBSD-6.0/mac68k/binary/sets/xetc.tgz) = 4c5d664679856b02b694534cd6afdf20597e6d53
MD5 (NetBSD-6.0/mac68k/binary/sets/xetc.tgz) = 72bd67556ac14ea0ce147374af0ba624
SHA1 (NetBSD-6.0/mac68k/binary/sets/xfont.tgz) = 24ca4cb77a59246b1b05e0a91fed21c63259d12b
SHA512 (NetBSD-6.0/mac68k/binary/sets/xfont.tgz) = 8dc297fb6205a995767e68e77cc21169234dfb66a86bbcd2e1828fd32efeda0a25c9f2a9b1a39b97fe8d7b24f8cd5cf6160859af6be2259876f39ee4e76c405b
RMD160 (NetBSD-6.0/mac68k/binary/sets/xfont.tgz) = d2f134959940e6667aca185f01e388c05dc6bc85
MD5 (NetBSD-6.0/mac68k/binary/sets/xfont.tgz) = 3c96c1f0a13cd7d277ca35622ac9c058
SHA1 (NetBSD-6.0/mac68k/binary/sets/kern-GENERIC.tgz) = f8cc845ee4bb3a6eaba937a9aa9ef56ee7df39a1
SHA512 (NetBSD-6.0/mac68k/binary/sets/kern-GENERIC.tgz) = 173879f5b2cee758aaec0d6baa6adfe208aa0817ce70edb8a4d821189de5e6a64dd5a4f0cd1da4b166579427f416383bc068d7e49677999e075cbe3ed40c99d8
RMD160 (NetBSD-6.0/mac68k/binary/sets/kern-GENERIC.tgz) = 2e10430ed05682c3d2e2c9496d847535a68988a8
MD5 (NetBSD-6.0/mac68k/binary/sets/kern-GENERIC.tgz) = 4c60130363f2cf09731d6dd9b98fbcb3
SHA1 (NetBSD-6.0/mac68k/binary/sets/base.tgz) = 702a0468018c02cc7ce229811a3905477da4f722
SHA512 (NetBSD-6.0/mac68k/binary/sets/base.tgz) = 871ff91422cfc76ef2e1fdf7a2052cbceaba0b7afacbf5856f2b440e834f1f4cc3759322f99bf85c4effff2a657f03a68978479d495985cf3ecf1cc3d0db4d88
RMD160 (NetBSD-6.0/mac68k/binary/sets/base.tgz) = 184263ea29fe40e2def660f30c049a1f7c80d42d
MD5 (NetBSD-6.0/mac68k/binary/sets/base.tgz) = ee8a6af6800f2a8ad6b9d6bb099bdf1d
SHA1 (NetBSD-6.0/mac68k/binary/sets/comp.tgz) = c355ad494accbc2a9cc99011c6a7f1030027b4cd
SHA512 (NetBSD-6.0/mac68k/binary/sets/comp.tgz) = 084828af876a61b296f6d34258462165cd23bdcb43ff5ac7f30d2b1ca43be31716c98de636e708ff193d8f97671eff2535d09710a66e4aa67151f326e423052c
RMD160 (NetBSD-6.0/mac68k/binary/sets/comp.tgz) = 3119965a0c66603ebb2b62b7ca53838f2a468e0b
MD5 (NetBSD-6.0/mac68k/binary/sets/comp.tgz) = 777aac0001fb61b70b6f5da36c73e8ea
SHA1 (NetBSD-6.0/mac68k/binary/sets/etc.tgz) = a279de07a8d912b05e056c846bc616c084edeab9
SHA512 (NetBSD-6.0/mac68k/binary/sets/etc.tgz) = 833a3c74e048bf2cc9f148b14a81eeb526071a1226fbafc3b501409922f4bb67b69d68cdd59cc8b83fe8fccb0afd1135e64b16683e8f42230530efab0e3ed115
RMD160 (NetBSD-6.0/mac68k/binary/sets/etc.tgz) = 11fed26b10f5d68485667c9b0eb0673b4fb17390
MD5 (NetBSD-6.0/mac68k/binary/sets/etc.tgz) = c6e0534c4470b66908a3f6e194723ffe
SHA1 (NetBSD-6.0/mac68k/binary/sets/kern-GENERICSBC.tgz) = 851fd53f568f2c855c934f975d322954de4c6b07
SHA512 (NetBSD-6.0/mac68k/binary/sets/kern-GENERICSBC.tgz) = 869f6b8f6eb4408a4f2950e8b9e10d020d14c0b829dd951d9746a9a778dbf142d53cb263cd797ec862b34a1aa367dfdbef5a633bd1181e442cce9fc0da2c8de7
RMD160 (NetBSD-6.0/mac68k/binary/sets/kern-GENERICSBC.tgz) = ff894994dc6fe3773893eae064bbd95aca6077ab
MD5 (NetBSD-6.0/mac68k/binary/sets/kern-GENERICSBC.tgz) = 0f2df7a929df022d55d5a73fe75e47a4
SHA1 (NetBSD-6.0/mac68k/binary/sets/modules.tgz) = aefbcaa2fb397cb690e38e3be240caabf9282947
SHA512 (NetBSD-6.0/mac68k/binary/sets/modules.tgz) = 83c696fccc6143c9c03e4efdf8ec65363dad51bbc48b434dc21a9949424f12db413de66abfa328b6d45f4097c59ee0324797ccc4ee7e462451170ce9ed7a81f9
RMD160 (NetBSD-6.0/mac68k/binary/sets/modules.tgz) = 44606993e2ae655bdafaf93a92cee5a85efb9e2a
MD5 (NetBSD-6.0/mac68k/binary/sets/modules.tgz) = 03bdab5790fdf19db36149fc33857b4f
SHA1 (NetBSD-6.0/mac68k/binary/sets/tests.tgz) = 56b5e85ef88683b81048d2e6136aa949e690d92a
SHA512 (NetBSD-6.0/mac68k/binary/sets/tests.tgz) = d4290b5069cb2290aac992145bcd69ff6e34b98ae2661e5cc16dd5b70362c1036c079dad7c96288f473c600e9cd1870c079268ffd08acb418ecc21708e3399ae
RMD160 (NetBSD-6.0/mac68k/binary/sets/tests.tgz) = 7d46086a6a53cd19439f6967849dd294bb27b631
MD5 (NetBSD-6.0/mac68k/binary/sets/tests.tgz) = 746ea69e63fb5af15f526f2f950ea449
SHA1 (NetBSD-6.0/mac68k/binary/sets/text.tgz) = db35814cfb2c8ecce27765b3257b885d2e6debde
SHA512 (NetBSD-6.0/mac68k/binary/sets/text.tgz) = 4110c6d47405005eafdb9fb205505f2cc40c6a2ace5b611312f4ef15ae5207a9ef5c4a42ca3b53164ec6f00a80efa5aa6d669a79a9e89da6f73fadd2dc5e437d
RMD160 (NetBSD-6.0/mac68k/binary/sets/text.tgz) = 38345e956e8a7be25706d4809bfa83ac626a3e96
MD5 (NetBSD-6.0/mac68k/binary/sets/text.tgz) = 90b4f37e590530f361f5ba22882f6184
SHA1 (NetBSD-6.0/mac68k/binary/sets/xbase.tgz) = a9988db70cf4a221f20fa81d7f2525575eff6161
SHA512 (NetBSD-6.0/mac68k/binary/sets/xbase.tgz) = 996a52c0e2a69be7ffbad4fca9e79e03cd72533d3c5059660b2bc5ebb3e982e010f53a870583d9eba702f143365508f5a8ab307d9806fda3df1f48021a3dcf4e
RMD160 (NetBSD-6.0/mac68k/binary/sets/xbase.tgz) = 10e9d44638eab39da8b579a32ebf1ae283901401
MD5 (NetBSD-6.0/mac68k/binary/sets/xbase.tgz) = 649bdf23aca348bbb0d7d3741154cbcd
SHA1 (NetBSD-6.0/mac68k/binary/sets/xserver.tgz) = 2139abe77fda0708a14f6cf7cb3787ede3369f35
SHA512 (NetBSD-6.0/mac68k/binary/sets/xserver.tgz) = db558b4f6be34da37cc971183c3a67d8d51ea26f3852b9ae53844ca5bc3fc0be405b6c30b6409d1e0d653178f4f6dc46645f37526ddd7d74be735a38beeaab09
RMD160 (NetBSD-6.0/mac68k/binary/sets/xserver.tgz) = 32d160c3aab16c5a3ed819c9c020c8189e735c9f
MD5 (NetBSD-6.0/mac68k/binary/sets/xserver.tgz) = c17de992b6f02ad51a720faaf0ac04d5
SHA1 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALL.symbols.gz) = 7ab45cb4b64d15f930d7b275acf62d70015c7d9c
SHA512 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALL.symbols.gz) = 107b76145d7484485984483a8b793bbd3fd56f0eccafc619f9b5cb119bdbd0f87ca5918257554a9623c535416368eb4dac4df8fcdfdd12ac279459b775827e56
RMD160 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALL.symbols.gz) = 2f2e4239577f923664edb12730c51e15106079a2
MD5 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALL.symbols.gz) = d53fc2a484e0add618fcdd9b236ffdd3
SHA1 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALL.gz) = 6f94e9d416e73a7019fd05338369616e459db5a3
SHA512 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALL.gz) = 8fc1096815fdcc6e987e0c69b991553ff094e226127a10bcd9510ffa0dcc746fb7416896e54ea32f18e80c5782937b8af34d20c55d8758c750ff7810ec69c203
RMD160 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALL.gz) = 6ddc97fcb3cf13544c3190956c37f34897cf35f7
MD5 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALL.gz) = 3d4a8d946ed122290abac6375a9ef863
SHA1 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALLSBC.symbols.gz) = 2672525b1b4997e3793edea3ee8b30c23cffb102
SHA512 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALLSBC.symbols.gz) = 003877e10fb0dad0b7190bbf559f190f3de2c0e1068caa5601a0a938706225e2cb7b141f8e4e767da7ed1ff15350f42792ee9a7de5e1d7149dfc233d6dbb4da9
RMD160 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALLSBC.symbols.gz) = 8f7b7ddfd1b96f5ca34f4a074cf27adf7701ceca
MD5 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALLSBC.symbols.gz) = 2aba4f14a7812f2d935c5ddd84634529
SHA1 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALLSBC.gz) = c91cbdc339088c1c32868779c918f746d4b5f4f7
SHA512 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALLSBC.gz) = 11eff94f2d5aaef7decbf6447a91f692c605a5d36acbea876e6a2501dddc681a60a89d3084ed1413c80ba3d8b23c90abfc567dc99254a9d810353992757d6284
RMD160 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALLSBC.gz) = f26b7b6f484d727ff6aa460985f23af89f72c8f7
MD5 (NetBSD-6.0/mac68k/installation/instkernel/netbsd-INSTALLSBC.gz) = 2dffa66829a868235a11ab636e793d52
SHA1 (NetBSD-6.0/mac68k/installation/miniroot/miniroot.fs.gz) = dc091b6ed92bd1269a8f76c427d606345a8f49fa
SHA512 (NetBSD-6.0/mac68k/installation/miniroot/miniroot.fs.gz) = ba5b9a96d964da2b2c0c5640a4669840352057ea799013ceae75d3d9a7782d6a4096cbc591b287b663edd27a3db54ff5a5b62a81099c2f97d21c3919b0bc7b59
RMD160 (NetBSD-6.0/mac68k/installation/miniroot/miniroot.fs.gz) = dc09fba5a4fe6ead5f59da6026018a63065ee414
MD5 (NetBSD-6.0/mac68k/installation/miniroot/miniroot.fs.gz) = ab49b728c7a2ef8c273a3ad07b5e3e6b
SHA1 (NetBSD-6.0/mac68k/installation/misc/src/Installer_1.1g.src.sea.hqx) = 5ae75a6179968ac955f38b60f57e3811a1e4a3de
SHA512 (NetBSD-6.0/mac68k/installation/misc/src/Installer_1.1g.src.sea.hqx) = 0017e59c1c000d154df2f9e868a85df1897b8a7918471dd6d468a4527b092ae2d069bbd686487b0b05ed4d61abfffaa7eb6584f5ee03b320469386e2b64c89cf
RMD160 (NetBSD-6.0/mac68k/installation/misc/src/Installer_1.1g.src.sea.hqx) = d29a9600c614d00c7af518b2f75f1d1ebebf4b6d
MD5 (NetBSD-6.0/mac68k/installation/misc/src/Installer_1.1g.src.sea.hqx) = 5d00aedc5729c2e2ffcea77e666a4adc
SHA1 (NetBSD-6.0/mac68k/installation/misc/src/Booter2.0.0-src.sea) = 5a4598fcbd39bfed29d58f9d712ed8d3408ae5c1
SHA512 (NetBSD-6.0/mac68k/installation/misc/src/Booter2.0.0-src.sea) = 4828cb2efb9b40753be27ecd1cbe235e3842b9b2b024070eb6135f10716dfae1ab7cd6d45db862c64eb5b2267b40c6660af53986d3b5fcfab03600a75c0b08f3
RMD160 (NetBSD-6.0/mac68k/installation/misc/src/Booter2.0.0-src.sea) = 9f284e11ef74ef0202d4a46ffcdafce282f3104f
MD5 (NetBSD-6.0/mac68k/installation/misc/src/Booter2.0.0-src.sea) = d9185a15e8dbcefdc26e8dceea7eaa51
SHA1 (NetBSD-6.0/mac68k/installation/misc/src/Installer_1.1h.src.sea.hqx) = 3a4d1868a5f6c6e010dd6e3d0ff7ddb63a034c9e
SHA512 (NetBSD-6.0/mac68k/installation/misc/src/Installer_1.1h.src.sea.hqx) = cafa4aadc01ce3c515945359d67df14aefb9cf36cafab93d113cc5079e0cdc06360310173febca13cf7bc8fe66eb9161f6afed97a0810ddf025bca38d327e444
RMD160 (NetBSD-6.0/mac68k/installation/misc/src/Installer_1.1h.src.sea.hqx) = 90b7070035d00fe09ac95962c4d47e1ba2f565e2
MD5 (NetBSD-6.0/mac68k/installation/misc/src/Installer_1.1h.src.sea.hqx) = 9f792b26b1a5691f13193a7e08cf6d18
SHA1 (NetBSD-6.0/mac68k/installation/misc/src/Mkfs_1.47src.sea.hqx) = 76823266114b2ff531d35c646eef705677085907
SHA512 (NetBSD-6.0/mac68k/installation/misc/src/Mkfs_1.47src.sea.hqx) = ceb5f5665acc0a9c8f273bf659e17a55dec3ffde647deb90cfa9001e43a82d5e1b4e8001175559b2f1d6327f2a80f86bfbf2798f27bbe6bbe52547a92de11d57
RMD160 (NetBSD-6.0/mac68k/installation/misc/src/Mkfs_1.47src.sea.hqx) = af8327f6e2d66bbd8a04498d27ab3f5668556cce
MD5 (NetBSD-6.0/mac68k/installation/misc/src/Mkfs_1.47src.sea.hqx) = 61facd7fdeb336f108d78feb8327555b
SHA1 (NetBSD-6.0/mac68k/installation/misc/BSD_Mac68k_Booter.bin.uue) = 432b619e0eb127a5fd70fd3dc8f2988a93211a34
SHA512 (NetBSD-6.0/mac68k/installation/misc/BSD_Mac68k_Booter.bin.uue) = ab3650cca71dc57f38f480bfa74f6bdffe2d0ccbf76072658503cd168b91f90a7fe5ba0ef4f51aa497a11684ee05d73376e059957c75773d92f894877042c0ab
RMD160 (NetBSD-6.0/mac68k/installation/misc/BSD_Mac68k_Booter.bin.uue) = 61bdb1fc795bfe9242816682fc98cfdee036e0ef
MD5 (NetBSD-6.0/mac68k/installation/misc/BSD_Mac68k_Booter.bin.uue) = aed2003c59f8e824dc30c44366439426
SHA1 (NetBSD-6.0/mac68k/installation/misc/BSD_Mac68k_Booter.bin) = 8e940cccdfe4e645ddd01738477f7f58342788e0
SHA512 (NetBSD-6.0/mac68k/installation/misc/BSD_Mac68k_Booter.bin) = 0d3e0d39aa811921b012a6fd07dbf97b63f61dd14800c61d2941f3c624db88cd0805478df900120cd20c3fcf6af11c5989c52400e5b4eefabd4bf5c62f9f2023
RMD160 (NetBSD-6.0/mac68k/installation/misc/BSD_Mac68k_Booter.bin) = 49146172c77c05a522bdf118e879f0bd933bc99c
MD5 (NetBSD-6.0/mac68k/installation/misc/BSD_Mac68k_Booter.bin) = 757b6ca3559185feafc36927ce9ab578
SHA1 (NetBSD-6.0/mac68k/installation/misc/res-edit-and-others.hqx) = fe505f1ff77f1574ddec17dd4de93f0370fe236b
SHA512 (NetBSD-6.0/mac68k/installation/misc/res-edit-and-others.hqx) = cf382e50ea9a8fd8c30c8bddfadf44b64b94216656dc623039a37c88bc642080b54002b9aca1cad25c251c719cda9fb3ad89bbf3ed645672c0040baf459e0ae9
RMD160 (NetBSD-6.0/mac68k/installation/misc/res-edit-and-others.hqx) = 19be7f536eee28b7e30cffc4ef3e8a2f8c7fe88f
MD5 (NetBSD-6.0/mac68k/installation/misc/res-edit-and-others.hqx) = 707c4953e155315abe519d5d48c54318
SHA1 (NetBSD-6.0/mac68k/installation/misc/Booter-HTML-manual-12.tar) = 94e2f09daeb119ece95abfe24b103f85f9c5d558
SHA512 (NetBSD-6.0/mac68k/installation/misc/Booter-HTML-manual-12.tar) = 3d9a743e71ee985134ab679608ffc31b197a638e302450c888f90d02fefae72b647622024b3ebb4263e57b6e58ce457890d6a0774c62a8e4a9f90d235d04c15a
RMD160 (NetBSD-6.0/mac68k/installation/misc/Booter-HTML-manual-12.tar) = ee32b1b55b0495b9b4723ff3cbc42123f5b360a2
MD5 (NetBSD-6.0/mac68k/installation/misc/Booter-HTML-manual-12.tar) = 24120554ffbac39c8edd8e939706b7c5
SHA1 (NetBSD-6.0/mac68k/installation/misc/Booter2.0.0.sea) = d5178c25cc71514816986842e39c716e6cd5ab4c
SHA512 (NetBSD-6.0/mac68k/installation/misc/Booter2.0.0.sea) = fafbf6f153e9b9bf657aac3c74d569b44ff21c1e306f0cbc4a2fcab5fbe439bd46746ff0469113bc715b69990507994ae94934d8caf82321684326f4e8b4d146
RMD160 (NetBSD-6.0/mac68k/installation/misc/Booter2.0.0.sea) = 97798bf61397243efee8480c7f2f19e63fda7d00
MD5 (NetBSD-6.0/mac68k/installation/misc/Booter2.0.0.sea) = 05b46210443c60ee28bcbe81f1ebc004
SHA1 (NetBSD-6.0/mac68k/installation/misc/BooterManual.stxt.bin) = 2859ea6089df13333b6b521d8bc858079083251c
SHA512 (NetBSD-6.0/mac68k/installation/misc/BooterManual.stxt.bin) = 1891f4c07bfb51de08fea8e3a5cd572e99897d20f6df748ae72af8da5e8191ca07d5f0dc836b39be12236286926677896e4a456a8f64b31517d09285bebed460
RMD160 (NetBSD-6.0/mac68k/installation/misc/BooterManual.stxt.bin) = 37fe1a762925b8bb8ae71177bea992aad27ee03f
MD5 (NetBSD-6.0/mac68k/installation/misc/BooterManual.stxt.bin) = fcc43fd81d5691559a5aa8ae15be8899
SHA1 (NetBSD-6.0/mac68k/installation/misc/BooterManual.stxt.bin.uue) = beb0225c706b12fc8a241025c483133c1d5e6b86
SHA512 (NetBSD-6.0/mac68k/installation/misc/BooterManual.stxt.bin.uue) = 7e98b555d7cf30da5948b115595f06a725c55cc0ef3e7a705423c600cafceca2e04966c65573805e6e553cf9f7a2faa05803ff2d6cd9deb714f4e98924e2a5d7
RMD160 (NetBSD-6.0/mac68k/installation/misc/BooterManual.stxt.bin.uue) = ed9a5860dafd3484d380f1f3a91cec366b59be9d
MD5 (NetBSD-6.0/mac68k/installation/misc/BooterManual.stxt.bin.uue) = cacb436bfbfc9a4b0995c789fee8342f
SHA1 (NetBSD-6.0/mac68k/installation/misc/Installer.sea.hqx) = c6a6c25ae1fcdfe4a0ba90616c3779068280b759
SHA512 (NetBSD-6.0/mac68k/installation/misc/Installer.sea.hqx) = 5ff75859fd5eb2fdcf75215ec6c045518ef111d0e8caecbe951dbf6f9f1129cac6948ab991949cbb3ee68d05ee8f398b07206acc9af53a5db1c4c6c7148a3a2a
RMD160 (NetBSD-6.0/mac68k/installation/misc/Installer.sea.hqx) = 36b141ff6722fd5e76fe293d1247587ae0efdd8a
MD5 (NetBSD-6.0/mac68k/installation/misc/Installer.sea.hqx) = 01bc744d539b4f2f8e4fb2761d4c77e7
SHA1 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1g.sea.bin) = 8ef8265fa77478a6741bc65ceb6f64e790c7aac2
SHA512 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1g.sea.bin) = 45324e1231e7d53816640b8867eb117294373e90463297f05e6d187b1fe719b82bc1782b24dff8ad6a71e0383eddd0f47bd15d604f6cf3bb2bf3499df887eb48
RMD160 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1g.sea.bin) = 506451703e3f00ac0a26b450d33463fd2c1e4582
MD5 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1g.sea.bin) = ea338e4327628024d670ae9eb64323db
SHA1 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1g.sea.hqx) = 8e28517a94b1b0b57a504938077aa740de7e3b80
SHA512 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1g.sea.hqx) = 6e07c3682a0bbb4f7d23c235f83f3d22d4a8813cade67ab9f76092c68f41767c00eccde01600283df36e4d222bddd2b7fec355db468d07d7db8dbf51adc9f5bb
RMD160 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1g.sea.hqx) = 4159e0226c6f58b480bd0cdc42f9d2d411521034
MD5 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1g.sea.hqx) = 2c214a1b2429702942436e92e2a408c3
SHA1 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1h.sea.bin) = 0bda2b9e6a95f1b10414823f048087ad6455e6ed
SHA512 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1h.sea.bin) = bb9ad3f95609c3f23d31a876055f2cae692d8abb853431d0945ee7e66e292fe7069dda82f5ee615ac5fe5893e0699323632de8d4c1c668d987a8172597d78386
RMD160 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1h.sea.bin) = bc7f88652ae4d51954c967dde99dc6c11494a7c8
MD5 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1h.sea.bin) = 763ff66b2234fd52ffe4ad08de5f090f
SHA1 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1h.sea.hqx) = c6a6c25ae1fcdfe4a0ba90616c3779068280b759
SHA512 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1h.sea.hqx) = 5ff75859fd5eb2fdcf75215ec6c045518ef111d0e8caecbe951dbf6f9f1129cac6948ab991949cbb3ee68d05ee8f398b07206acc9af53a5db1c4c6c7148a3a2a
RMD160 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1h.sea.hqx) = 36b141ff6722fd5e76fe293d1247587ae0efdd8a
MD5 (NetBSD-6.0/mac68k/installation/misc/Installer_1.1h.sea.hqx) = 01bc744d539b4f2f8e4fb2761d4c77e7
SHA1 (NetBSD-6.0/mac68k/installation/misc/Mkfs_1.47.sea.bin) = 9783f80336af95dfbf854278f58b577891a6442e
SHA512 (NetBSD-6.0/mac68k/installation/misc/Mkfs_1.47.sea.bin) = fb3d537f73cd7a5c05b4deeacfebe4805de1172071b97af545ef0c360c8093dd34007cdb498dad1b8fba4930e939c35429b314c575ab1e6d57fc208d72bee767
RMD160 (NetBSD-6.0/mac68k/installation/misc/Mkfs_1.47.sea.bin) = eb92d84b516d3c84a2c54fa501a1f585434639cb
MD5 (NetBSD-6.0/mac68k/installation/misc/Mkfs_1.47.sea.bin) = 0b6f626c1d8f578e01818cfc2a2e79b5
SHA1 (NetBSD-6.0/mac68k/installation/misc/Mkfs.sea.hqx) = e2286a389c29eebadaad19c9c3f3c9095b8bcb3d
SHA512 (NetBSD-6.0/mac68k/installation/misc/Mkfs.sea.hqx) = 5021b88d12bdc6545c681a3df24750375d63cac19b9e0f6a7148cb8592de6d0cc22f254b9755a96e1780541acbbdf54f0f6f9b62ff7cdab8dc94db0d247b2b87
RMD160 (NetBSD-6.0/mac68k/installation/misc/Mkfs.sea.hqx) = 14c18aceaab0593ca0f41bcce95499db3fc4f8f7
MD5 (NetBSD-6.0/mac68k/installation/misc/Mkfs.sea.hqx) = a20825a32ee72f356880169993c3ac73
SHA1 (NetBSD-6.0/mac68k/installation/misc/README) = 6f10f67afd988de6fa0a66f78ec46da15cc994f3
SHA512 (NetBSD-6.0/mac68k/installation/misc/README) = e8f6d9c3217e7ad084ebc6758a07f14a664db9a34feb2b4dfb19fce8692ca1703f2b119eabcb13cb38f629a9588dea3279bfcf0cda65260acd585514d129701d
RMD160 (NetBSD-6.0/mac68k/installation/misc/README) = 9d125cfe0be05fb26b9b5f52690b9280e282af44
MD5 (NetBSD-6.0/mac68k/installation/misc/README) = 6081e0a4521b8f229bd3c09e094d4bc0
SHA1 (NetBSD-6.0/mac68k/installation/misc/Mkfs_1.47.sea.hqx) = e2286a389c29eebadaad19c9c3f3c9095b8bcb3d
SHA512 (NetBSD-6.0/mac68k/installation/misc/Mkfs_1.47.sea.hqx) = 5021b88d12bdc6545c681a3df24750375d63cac19b9e0f6a7148cb8592de6d0cc22f254b9755a96e1780541acbbdf54f0f6f9b62ff7cdab8dc94db0d247b2b87
RMD160 (NetBSD-6.0/mac68k/installation/misc/Mkfs_1.47.sea.hqx) = 14c18aceaab0593ca0f41bcce95499db3fc4f8f7
MD5 (NetBSD-6.0/mac68k/installation/misc/Mkfs_1.47.sea.hqx) = a20825a32ee72f356880169993c3ac73
SHA1 (NetBSD-6.0/mac68k/installation/misc/NetBSD_ROM.sit.hqx) = 009c5f376751cda897cdd5707932a42ed8c4c75f
SHA512 (NetBSD-6.0/mac68k/installation/misc/NetBSD_ROM.sit.hqx) = a04d6873fe2e0d3b543a05d1c12893f2715479e5bbfeeb9657babd146610197e4eec045a428d51bce4a9f8fb90f23f81b31b35f672692de953d6df8379a7531c
RMD160 (NetBSD-6.0/mac68k/installation/misc/NetBSD_ROM.sit.hqx) = 6dcd7e0f25e19d77d38b18c22af77ef0c80769e6
MD5 (NetBSD-6.0/mac68k/installation/misc/NetBSD_ROM.sit.hqx) = 6d9984dacf90adcd3f5f4a906b72b436
SHA1 (NetBSD-6.0/mac68k/installation/misc/README.Booter) = cb2fb0233a1f5444c5c1b9648a6c167a4ce09437
SHA512 (NetBSD-6.0/mac68k/installation/misc/README.Booter) = b233f9b5f8e6b615ecbb117d1853c64a82b1ac78958a7d394133e6e5f777ca6075b1e65d74ce52d1bc9cc97257584cbc4c70a11b0865ef095e6376d2c4e46bf4
RMD160 (NetBSD-6.0/mac68k/installation/misc/README.Booter) = 2bd403db763c410b968f85cb8da32d10a7520b59
MD5 (NetBSD-6.0/mac68k/installation/misc/README.Booter) = ae0f90a35340fc51be774ebd33674a41
SHA1 (NetBSD-6.0/mac68k/installation/misc/README.Installer) = e8e4011a516b40ea99e9f1b16708fc91878f88fb
SHA512 (NetBSD-6.0/mac68k/installation/misc/README.Installer) = 791e8a5415bdfd0ac0412e964d0ed1dac9dbebfbe9d546fbd92d3aa3d9b82eececba55628ac6cbbd7dd843f62183d718e2a1ad98c10311fb76c24dd40c77af06
RMD160 (NetBSD-6.0/mac68k/installation/misc/README.Installer) = fc1b834d11c4df6677ee4a40b1e3ba2015d5ad00
MD5 (NetBSD-6.0/mac68k/installation/misc/README.Installer) = bd6be3091a274d7c68da789ad1db0eda
SHA1 (NetBSD-6.0/mac68k/installation/misc/extensions.map) = 4a5fa646db0c73433f3538e17e7d18b38679284f
SHA512 (NetBSD-6.0/mac68k/installation/misc/extensions.map) = 32db0ebf9c0e80e4856c963ef21d3626060e20ba33cd61301e4114b77a636cf248e705700444b42a4631d1c56ef09294981158e1f0b0527b795501b9cea1b8f6
RMD160 (NetBSD-6.0/mac68k/installation/misc/extensions.map) = f6811f3b103ab5c902a204dce2578e8cba6e8e9a
MD5 (NetBSD-6.0/mac68k/installation/misc/extensions.map) = 5007a814ca2442a9693969cb997b4905
SHA1 (NetBSD-6.0/mac68k/installation/misc/mode32-7.5.bin) = 6eb3cb3b988e6295f7222016d241f53c61cf347e
SHA512 (NetBSD-6.0/mac68k/installation/misc/mode32-7.5.bin) = 16f05d2792e03d3a9c9ccd9d435ca5556224bd73619d89b9d2f6ac3b282f6587b76745bbbc9383c20f11c2125b60dff138845dac4a269e459a5b76ee1d5e865c
RMD160 (NetBSD-6.0/mac68k/installation/misc/mode32-7.5.bin) = be8fc56378e697fc1f6fe9da5e8e69c787dbc434
MD5 (NetBSD-6.0/mac68k/installation/misc/mode32-7.5.bin) = 33ff34f5db2f0ba88c83ce85b1e51664
SHA1 (NetBSD-6.0/mac68k/installation/misc/mode32-7.5.hqx) = f588fece9bb911749d1038da2ec31e5ea50b2b21
SHA512 (NetBSD-6.0/mac68k/installation/misc/mode32-7.5.hqx) = 1d4a1b489419171022e120d7292f479ea0d47833b2868bccc47ea1d97c442ff1e18044f43783fbd920145a6e6aad094be796dc8db00d5d997235c8f9d8ecb7ab
RMD160 (NetBSD-6.0/mac68k/installation/misc/mode32-7.5.hqx) = 6e0bd805b5595d6d8ee3a24a63290978bd8da8cc
MD5 (NetBSD-6.0/mac68k/installation/misc/mode32-7.5.hqx) = cf14fb32d042ba270b527ec8b11f6d3e
SHA1 (NetBSD-6.0/mac68k/installation/misc/pdisk.sea.hqx) = ac3641da27c3615456310ff795b2d684eb2661ef
SHA512 (NetBSD-6.0/mac68k/installation/misc/pdisk.sea.hqx) = a5404597c2479adf6ad9a910771f8d9cff57dbc4066d73346bfde3f1380a59641855820658ff3a6c8c0fd97b242dcfd4c30e7ecde50047bbbc7a7d45e9d77e4e
RMD160 (NetBSD-6.0/mac68k/installation/misc/pdisk.sea.hqx) = a5368be868266f37bdf3459067659ef0c4186dc6
MD5 (NetBSD-6.0/mac68k/installation/misc/pdisk.sea.hqx) = 71be22598a33042751caa6dc6c0af834
SHA1 (NetBSD-6.0/mac68k/INSTALL.html) = 85efa4b8402dccb52805752ff82e87eb1259e5cd
SHA512 (NetBSD-6.0/mac68k/INSTALL.html) = dd8012db2cb801b02b808eadd473f529b58977b9d2088c878aeb7deafc214adb2e281aa3b2094e0988f23e1ede88a1528492634211f86fe73d4b6085a1a917fe
RMD160 (NetBSD-6.0/mac68k/INSTALL.html) = 01614436a73df3cae2149a607abb6c59671dcb63
MD5 (NetBSD-6.0/mac68k/INSTALL.html) = deaafd9baf0d73ad2fe1e731ca8d4391
SHA1 (NetBSD-6.0/mac68k/INSTALL.more) = 02b0da5241dfb0e94347c45aa2b59230a4c4beaa
SHA512 (NetBSD-6.0/mac68k/INSTALL.more) = e167052f3a76308dda76a45e7fe93622a3e40fb7905d2bfc3e6438007d5bee8cafa06f9bddf4fabd9801d82bc3de56fbec1892a4977d2b818ece0ee570bc7729
RMD160 (NetBSD-6.0/mac68k/INSTALL.more) = 1602b574a726ead87163bea4a121dc4d5573c321
MD5 (NetBSD-6.0/mac68k/INSTALL.more) = ab7ed66c5f64540dc7b9d036048fcfb1
SHA1 (NetBSD-6.0/mac68k/INSTALL.ps) = 999ea4a94acf0cf256157ba3b447d866b0347bfa
SHA512 (NetBSD-6.0/mac68k/INSTALL.ps) = dfaf48953a73ed32b722bc95d34a58866ea8ccbf0d9d79e1fd9769634920426826a1e5cc86b94cc2eaf39eb3365bd5977ae0ee4284af484f64bb3e92847567ce
RMD160 (NetBSD-6.0/mac68k/INSTALL.ps) = 73548f30c2c139ae5e62e98dbc54a4a80d1398c5
MD5 (NetBSD-6.0/mac68k/INSTALL.ps) = 43662b6328dcc148ea4b6121de934e7b
SHA1 (NetBSD-6.0/mac68k/INSTALL.txt) = d198c59c8fb2bcb408013794ef3d8cadcb4ad448
SHA512 (NetBSD-6.0/mac68k/INSTALL.txt) = efb0679da4c0f9b6b8f15f92579ecabb67a4e7451e156430f9a748368bf8815d563d83050c24784c80985d06d0c257d66f91a03c792ad9b65da17e408d2e732b
RMD160 (NetBSD-6.0/mac68k/INSTALL.txt) = 3ea97712841cab74a13994769fbe4ff66705b6c0
MD5 (NetBSD-6.0/mac68k/INSTALL.txt) = ed4f38a04160feaf88388c0970e6ed6e
SHA1 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC_MD.symbols.gz) = ce9bd080acdc497dd570c416befca032a1e8d823
SHA512 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC_MD.symbols.gz) = 39b5933f7f43c990c2478041d398e9764dce24bfbae9baba2f603fe574448370d9d5cef5e398613fa047648cc0120dc7e878f527bc12751c6a37fe8966cb9d2b
RMD160 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC_MD.symbols.gz) = d23e372373d18f19b55bc87635bc45f3f730a91d
MD5 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC_MD.symbols.gz) = c0e545e698824468f2b0acefaf788bea
SHA1 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC.MP.gz) = 0a065a75b49d53c192d1a79e9d216a2d9382d570
SHA512 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC.MP.gz) = dc2a00bcdf70c08d7089b775b3328f1b828c5a5962ec077d95c8ea279586c6b65c48bcde34b9432c4f7a0a7cb4dc6e8f4887ae254038919de8b4f2d2a3d2e3c3
RMD160 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC.MP.gz) = 39e88535fff406327f536ecd319c2b5785898fbf
MD5 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC.MP.gz) = 3410df9cc4df7b28a292ed3d150808ea
SHA1 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC.gz) = 597f394bd55facd236e90f6aa213a5eef4711f7c
SHA512 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC.gz) = f7cbec6767fd9ce1aa61065ebc360d4aa1d2fcb26c4457f38c409fb5465753b726c8fb37ab0448863efd69c21a8216dea27e8067cd1fa1f74299677985195181
RMD160 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC.gz) = 21e5329fc0c500af9e333fabef527385021bde0a
MD5 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC.gz) = 7da219cc304f59d1662eaf7f4dc1833f
SHA1 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC_MD.gz) = e594f013594f13f586f6789005e49f13b9f3467f
SHA512 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC_MD.gz) = bc68d5a6cf29cdc0c9ee8b9ca7610703025621f64a9b427f04b87cf53bf9e14e765785283a0a66aed1a15dd0abbe81e8fc2029c3310b715ef75cacc7eec93283
RMD160 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC_MD.gz) = 7ce6740e080b6f9b5dc8d99c70c779b0486ad916
MD5 (NetBSD-6.0/macppc/binary/kernel/netbsd-GENERIC_MD.gz) = 3f6f6c37b2c6f7b04c9595245c7224c4
SHA1 (NetBSD-6.0/macppc/binary/kernel/netbsd-INSTALL.symbols.gz) = de3656de1e94d49bb43e6f0b21fc21ee15f2170a
SHA512 (NetBSD-6.0/macppc/binary/kernel/netbsd-INSTALL.symbols.gz) = c90cb34afa99a7ceb8ba3742a0a4b7bd59da93ecb4aca1ff1a540d18d540947446467627e3df717bb880f1b0a567336316bc8a2d59026a7a8f6935eca93984ff
RMD160 (NetBSD-6.0/macppc/binary/kernel/netbsd-INSTALL.symbols.gz) = b617db2dfd9b44f855f06b777f4c69680115ed27
MD5 (NetBSD-6.0/macppc/binary/kernel/netbsd-INSTALL.symbols.gz) = c90b9ecfb302f54de790366b2e17b625
SHA1 (NetBSD-6.0/macppc/binary/kernel/netbsd-INSTALL.gz) = 661d08043304e74da8232e5d14d1f4f748762510
SHA512 (NetBSD-6.0/macppc/binary/kernel/netbsd-INSTALL.gz) = e4dec9781eea446f078df552033c52e4b8df30231ecbc14131ece1f7312c8a0f98d47b06d9b66988c5c4c2ff45e8dd57e7f2715e165c431b391127f0e5c03565
RMD160 (NetBSD-6.0/macppc/binary/kernel/netbsd-INSTALL.gz) = 99e3fd9681bd7f400de2e2206163d501b6d23d41
MD5 (NetBSD-6.0/macppc/binary/kernel/netbsd-INSTALL.gz) = 1f1b3e39dcde937788a32ef3a4f65c1b
SHA1 (NetBSD-6.0/macppc/binary/sets/games.tgz) = fbf69640b362df2446d7414ac3a13db100c289eb
SHA512 (NetBSD-6.0/macppc/binary/sets/games.tgz) = d86b41d79aea4204ccf45306937a5ec45629e37873a5956c0280fd1d54cf4a808542d097f9d978fd3277791554ce8a6fb4d0ccd46e30210549d5cce1145bb499
RMD160 (NetBSD-6.0/macppc/binary/sets/games.tgz) = fd2aed1eb51f4db522fb5163e8fde4c040b55d30
MD5 (NetBSD-6.0/macppc/binary/sets/games.tgz) = 997ae5fb42196a787df89873be321e47
SHA1 (NetBSD-6.0/macppc/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/macppc/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/macppc/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/macppc/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/macppc/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/macppc/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/macppc/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/macppc/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/macppc/binary/sets/kern-GENERIC.MP.tgz) = e11a4e1da81be3a4650be198c5f9fe7204991540
SHA512 (NetBSD-6.0/macppc/binary/sets/kern-GENERIC.MP.tgz) = 924fc04dff6c7458196d0a19ec38a9a1114ef3cf5cd557aa68563c3d1d2dee67d9daf3ec72fec95c9bad8862ceb663c1f96cad61965e65c042c2723ffacfe43e
RMD160 (NetBSD-6.0/macppc/binary/sets/kern-GENERIC.MP.tgz) = 585796f103490e30928aca2c4977a41acb990c84
MD5 (NetBSD-6.0/macppc/binary/sets/kern-GENERIC.MP.tgz) = 5b9f58a612ad00b10d4812ce9e325204
SHA1 (NetBSD-6.0/macppc/binary/sets/base.tgz) = a31efb13ffcb70f4f429ad018d0ca6f9bc9e1db5
SHA512 (NetBSD-6.0/macppc/binary/sets/base.tgz) = 9746428cdb65b283fa668f63cd33cc2c0bff75e85f4357f460da4d7c0c874bbc0369170d813fa70a1f0933fc2240da6536fa1c2c2be59c6722ab3849d92a3dc0
RMD160 (NetBSD-6.0/macppc/binary/sets/base.tgz) = c83a80159c7128c8d311c62c15776776dae0fd97
MD5 (NetBSD-6.0/macppc/binary/sets/base.tgz) = 13ea19ca37828efa708cc77efe1a18f4
SHA1 (NetBSD-6.0/macppc/binary/sets/comp.tgz) = 2608b27059b55a9d53c2e4fb0f6f3417aafa8293
SHA512 (NetBSD-6.0/macppc/binary/sets/comp.tgz) = 2f6ec81a049c6c4688fc362a341d11c100ce057760eb5832fceb039ce810abde76d035a3f8e9a23fbb0f4ec3709a8729eba0e92c0f56aed9aed0779588b44bc0
RMD160 (NetBSD-6.0/macppc/binary/sets/comp.tgz) = 566f37564d448641e6bdcbd70e87c7b9406ee78e
MD5 (NetBSD-6.0/macppc/binary/sets/comp.tgz) = 0ab55dfc92d16a84368d2f1118b6385d
SHA1 (NetBSD-6.0/macppc/binary/sets/etc.tgz) = 42a97230bebad52fff9236fe06f3ac0aab27f929
SHA512 (NetBSD-6.0/macppc/binary/sets/etc.tgz) = b574a57a33d58eebac3a048823fa845d4a403e0503b21340c9cfb05ae9a743a213aa7222530c9e20d30fd90003b56366815ebcb28d085c2d9c998a73972ce602
RMD160 (NetBSD-6.0/macppc/binary/sets/etc.tgz) = dfbd6641e45fed5e9fa425be5c8878462ff8c6fe
MD5 (NetBSD-6.0/macppc/binary/sets/etc.tgz) = aa3881b4329b45b318e6d6f0d97ac24a
SHA1 (NetBSD-6.0/macppc/binary/sets/kern-GENERIC.tgz) = 77f4b674c6ef5bf00c6d6173d2498ff7ebbc64f9
SHA512 (NetBSD-6.0/macppc/binary/sets/kern-GENERIC.tgz) = f16e4dcf74059eafc235bbe7d7b59de1071b56ade3f77308bbb646fdd19e90362635125bff5f81ca92e51502c668106067ca6b377a8aec2bff708e589380977a
RMD160 (NetBSD-6.0/macppc/binary/sets/kern-GENERIC.tgz) = 0906acf7e10f69ea769a5a58d0ffe2d1e060ba98
MD5 (NetBSD-6.0/macppc/binary/sets/kern-GENERIC.tgz) = 50cf40ab93c7505da5fe7c9bd488abd2
SHA1 (NetBSD-6.0/macppc/binary/sets/modules.tgz) = b2c51fb3ca33a8f70c00f9e7d49cd4b91fb79673
SHA512 (NetBSD-6.0/macppc/binary/sets/modules.tgz) = 2cfd8f4f53c0561b8ab8aafac2c38363750be6484c21c8d670b7afb89390875893a9b549b2e28d08cdeeb81217f7e0e0fe437e09059871d4b063ffa0057bae97
RMD160 (NetBSD-6.0/macppc/binary/sets/modules.tgz) = 9fb4d8eb95a719022a522b3e9a970cda1a416f15
MD5 (NetBSD-6.0/macppc/binary/sets/modules.tgz) = 54914b3291b652b921239631c740dcea
SHA1 (NetBSD-6.0/macppc/binary/sets/tests.tgz) = f87666428063cf6b980c1d284aab5894e2c3d2b3
SHA512 (NetBSD-6.0/macppc/binary/sets/tests.tgz) = 94474f3ed0feff6b8c10299c4e190154bf4c6d579dd3edced5de901aec60c9413a921b282bc7ae0c2dd3ec49de8162204660a7fe80393102aa7ebd3318f4124e
RMD160 (NetBSD-6.0/macppc/binary/sets/tests.tgz) = 7467da9599fb2cb270854e337b696568de24b205
MD5 (NetBSD-6.0/macppc/binary/sets/tests.tgz) = 165d0d19f6c40d8249ddd46214a68e25
SHA1 (NetBSD-6.0/macppc/binary/sets/text.tgz) = b3e0e6f977a3598fb6f8a377cf91a67aa21ff74f
SHA512 (NetBSD-6.0/macppc/binary/sets/text.tgz) = 361c67a072fbfe6055fcf8436f3a84b10d92a0cc987c818a32a9f536466b285d47ef2ba39c8f04319ac0a93e7b2353eed589e6b72c3bd32f38c12aae82ff26ed
RMD160 (NetBSD-6.0/macppc/binary/sets/text.tgz) = 3d0c5c4e3a84335431a0d6cc9b78315495e137d0
MD5 (NetBSD-6.0/macppc/binary/sets/text.tgz) = d968d8f10bd69add51020decb7cd01bf
SHA1 (NetBSD-6.0/macppc/binary/sets/xbase.tgz) = 63043b541de378d777f43d0a333d2112a98a4e8a
SHA512 (NetBSD-6.0/macppc/binary/sets/xbase.tgz) = 189d8b05121b356d4fa587c46cd4bb50df1ff89a197c983dafed9ce93293ad16e82f77d14ae3ea2c8e9f01f8a5eab4d9841274d9caa7143f4d977d921d3e1a84
RMD160 (NetBSD-6.0/macppc/binary/sets/xbase.tgz) = 3cccd55a4407632e7d7444a6919901aca686c9cb
MD5 (NetBSD-6.0/macppc/binary/sets/xbase.tgz) = 1e9175f276efb45934e2e976787ec61b
SHA1 (NetBSD-6.0/macppc/binary/sets/xcomp.tgz) = 9322a94973bf4e59542a1c3fe2d307f1dee362ff
SHA512 (NetBSD-6.0/macppc/binary/sets/xcomp.tgz) = ca37cdedb39625b92d2068c8736c2d1bdadd132edbffb4bf6ea3a4b1a21ffcd5bfdd78e5aaad874be32cff4d1b98059e142c5a2192dc1f38ffb7a64e8e180a17
RMD160 (NetBSD-6.0/macppc/binary/sets/xcomp.tgz) = 48ef59c053d4a964872a70f1350252f98423794b
MD5 (NetBSD-6.0/macppc/binary/sets/xcomp.tgz) = fc0aa962b13c0e092657bfc3d6368284
SHA1 (NetBSD-6.0/macppc/binary/sets/xetc.tgz) = 5ace04f9545e5ac27432bddcde0d42dd592540c1
SHA512 (NetBSD-6.0/macppc/binary/sets/xetc.tgz) = ccdaeb27301167072fc0c98b78633c2d52b1eb4a5bf3ed5c606a646a420c7bd73e197b73d1a928c4aa17013fca5bbbd98f06fc4baa4ca97ada3a3e89490fdc06
RMD160 (NetBSD-6.0/macppc/binary/sets/xetc.tgz) = 048fd168d184f88b2164c654594e68665dec62c4
MD5 (NetBSD-6.0/macppc/binary/sets/xetc.tgz) = 1735140072351ccaa55f8b25cd002fd6
SHA1 (NetBSD-6.0/macppc/binary/sets/xfont.tgz) = 2cea3fa057b220c72ba147f9821d33b18d8346d2
SHA512 (NetBSD-6.0/macppc/binary/sets/xfont.tgz) = d76f6d6613f96475c85427792ad339576815f1423ae7e84018944d31cb66aa6ec14f2a8886f0912963745293326812a7efb8b1d2f08a6ee27da8fbf53c3223d7
RMD160 (NetBSD-6.0/macppc/binary/sets/xfont.tgz) = 42190ebf50a1db179d8d7b1bdd1aa3e86082ac77
MD5 (NetBSD-6.0/macppc/binary/sets/xfont.tgz) = 752888a1280d52af5e6b3c79287738e4
SHA1 (NetBSD-6.0/macppc/binary/sets/xserver.tgz) = 4c9ca17aec65f27a569ee9e6ed4af95ad3b9cd21
SHA512 (NetBSD-6.0/macppc/binary/sets/xserver.tgz) = 208d60880438a476ecacab3a4ab5872007c2728cafe16022cee9c0d781ce4dd8ef95d302ac9212a1a66b825869dc35c8df096d9f7c97a97054cf963ba58418fa
RMD160 (NetBSD-6.0/macppc/binary/sets/xserver.tgz) = f0bd1020d51ed7036593840c49bfaee57e09e58c
MD5 (NetBSD-6.0/macppc/binary/sets/xserver.tgz) = c241cf0caa30e6ae203fef4c95d2d74d
SHA1 (NetBSD-6.0/macppc/installation/floppy/boot1.fs) = 053808f21cbc6acc50469ffa2a6c48471bfbf8fe
SHA512 (NetBSD-6.0/macppc/installation/floppy/boot1.fs) = 82f844d8ee09d09a5cb06ad8cf8131826ad9e8c38a289e95757d6b9664b0aff4e30743ba750dd7ea780d556b4aa619388fbbef8727c52b302e314b08a6853306
RMD160 (NetBSD-6.0/macppc/installation/floppy/boot1.fs) = 291a7e7bd8b98758f2dce412054abbb69c6cf4cb
MD5 (NetBSD-6.0/macppc/installation/floppy/boot1.fs) = 369cf1bf5a69c9220d3ab6213c0a10d1
SHA1 (NetBSD-6.0/macppc/installation/floppy/boot2.fs) = 46098cec194fe5de4ffc1b66508397916cabd968
SHA512 (NetBSD-6.0/macppc/installation/floppy/boot2.fs) = 25ed41f6712ae333101fb168cdafbdb4c4dc677e4a0c701665d02758202af2238a2a53828e833b0917ac0547ea34a1a1c95f8b9018084ff9053bbba985814b91
RMD160 (NetBSD-6.0/macppc/installation/floppy/boot2.fs) = 9e743d123b1f43953743e347218ad29f022822be
MD5 (NetBSD-6.0/macppc/installation/floppy/boot2.fs) = 05843ccdeb93f326ddad69e810303fe5
SHA1 (NetBSD-6.0/macppc/installation/ofwboot.elf) = a013bc2f2d4d39cf9e56b0d625b6c48d08bb455d
SHA512 (NetBSD-6.0/macppc/installation/ofwboot.elf) = d28569738cfad70b6d5aef80b32e9d44f38dc5038f38143bf93f78eba708ff05e9a7955c03a682833ace2cd20daf4adb1a3c36956768e5f2b08fb59a44573854
RMD160 (NetBSD-6.0/macppc/installation/ofwboot.elf) = 9a9f54b00b80d91a934e83cb17a68b15a9fe8c00
MD5 (NetBSD-6.0/macppc/installation/ofwboot.elf) = 2d4598a059853f8f7e99541afcf6e31e
SHA1 (NetBSD-6.0/macppc/installation/ofwboot.xcf) = a15eef375d155080b6809e2db46d0475baebd872
SHA512 (NetBSD-6.0/macppc/installation/ofwboot.xcf) = 4756b6eeb05754185c16ff02e9d98642ae75c3545b385a67870e58b8211feda3ce265eaf020ca5252e9c47fab64922d9af8eead98b2cff798c58af9840fbd345
RMD160 (NetBSD-6.0/macppc/installation/ofwboot.xcf) = 1654bc8a15f5d3c2bb248033cbfbc37ca23980a7
MD5 (NetBSD-6.0/macppc/installation/ofwboot.xcf) = 84cc2385de486f3e72868ab7c75c8683
SHA1 (NetBSD-6.0/macppc/INSTALL.html) = 9752ce3a0ae1964208f027404c5aa3b526efe08c
SHA512 (NetBSD-6.0/macppc/INSTALL.html) = 71dbe4d992bcea022b1aac0dbdc3cf596a7538bf3fd2de390925094bfaa5cb1f71e332a2eb419cdfa1528cb7951c92491b3b3d7b88cd9e4c40f81af0a8ea34b2
RMD160 (NetBSD-6.0/macppc/INSTALL.html) = b5e2c12fbcf7fcf56e6e9354b178188c88e772b2
MD5 (NetBSD-6.0/macppc/INSTALL.html) = 31c92f0b3c87409d1d74b17912f05333
SHA1 (NetBSD-6.0/macppc/INSTALL.more) = 7b4f4aa288bbbacca5a7f69f7fb1dde30b6a67e4
SHA512 (NetBSD-6.0/macppc/INSTALL.more) = 7ab24e82eb0b779f11e16b10982fb82d6831058b0a51b57c2abbb0c91f09daa92a7a724c18f775d551b6bc4b8bff36eb91b6a823d2f9810a478180c50f9c676e
RMD160 (NetBSD-6.0/macppc/INSTALL.more) = 7bc31a39cfc1017821c29032432a8765af7b240c
MD5 (NetBSD-6.0/macppc/INSTALL.more) = 0a198a6d31ac02b702c103117400eafb
SHA1 (NetBSD-6.0/macppc/INSTALL.ps) = 9a14912c8a58fc4e6d004dc2426c189b991d5e6a
SHA512 (NetBSD-6.0/macppc/INSTALL.ps) = 94ffb000d582ea09c0f3d4e0b6e757f0bbbe50e6793d33e3e3ec29a0da26a0130f23c778e19e681892ceab4f440dc01d37b206272d24aa96c9ef3a9a6c49ed0e
RMD160 (NetBSD-6.0/macppc/INSTALL.ps) = edde76230d345a53de93bb00af333be03971d0dc
MD5 (NetBSD-6.0/macppc/INSTALL.ps) = f275f19c885be2a1ca922cff1561b5c8
SHA1 (NetBSD-6.0/macppc/INSTALL.txt) = e7e55b4231206d5fc087d6ad0c4b8ef52a73e385
SHA512 (NetBSD-6.0/macppc/INSTALL.txt) = 51f4462c903b0948ca5634bf27a9988924cfc3a8249310f1c5f2fc3c8372e9ab916cf106fa7bd9153ccc44454a1fc298024214575f43bf938a7e8a8a18163038
RMD160 (NetBSD-6.0/macppc/INSTALL.txt) = 8213d05b6c035c0932159f098633921ab6d79247
MD5 (NetBSD-6.0/macppc/INSTALL.txt) = 319bfbce37725d2bbafa032c7891529a
SHA1 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.ecoff.gz) = c7357a28d27e6ba6fd9a1ca72b7f0d0d95c23089
SHA512 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.ecoff.gz) = 3a0fbe1a79b886094cbb8a3613d3193d2731dbace050847c6d70ee8961b4d0e0dbf2f5d666e5f6c43e5ee2871ef9c59823b5fcadbf2dba16144f8c13e778c821
RMD160 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.ecoff.gz) = 12b0c1cda3875379f2cd582b5eac8316d9fec2f4
MD5 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.ecoff.gz) = c24f3d6e9a289d321c0ee250817cd348
SHA1 (NetBSD-6.0/mipsco/binary/kernel/netbsd-GENERIC.gz) = 188a82d7aab86e27f7defbd0318c12deb06a5200
SHA512 (NetBSD-6.0/mipsco/binary/kernel/netbsd-GENERIC.gz) = e6c278b3bc3e93137e2a0ccd894420671dc3d8f880b08aa673e51b758d88667fd416b317b90e1eaa72087a8b97322ce629f538f7daf2bc7746f0209d00e3d2f4
RMD160 (NetBSD-6.0/mipsco/binary/kernel/netbsd-GENERIC.gz) = 19b32d196282fccad2907b3972d0bd8673b0b63b
MD5 (NetBSD-6.0/mipsco/binary/kernel/netbsd-GENERIC.gz) = d91b37f1b364bf92609c4cd56e738873
SHA1 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.symbols.gz) = cfae9b61b6577966f0bf0f99e4592ce788f2e047
SHA512 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.symbols.gz) = e100ef50b40411ee3886ec6b0a77ac45459c4a50b1f5b6fee1961aecbef6ee4fcc5a4a1a215eeaefcf5a47407a3d907848e09fa0299be61acb7a93da813da77f
RMD160 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.symbols.gz) = 1e65145861489b78275308a994aca38a4e60e3d6
MD5 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.symbols.gz) = 5a441172f0a1659f405e7db91dcbdbc9
SHA1 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.gz) = 517897ab45650448a730ae2d9df177e3858c7ecf
SHA512 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.gz) = c8f2a2b0e243000b5c6a134b1c822e40cd454230039ed39a147320e049a8c4b8f540ba32613a1a189455d2f14f56b2291549b3cc40a16c65baf451e5e6005874
RMD160 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.gz) = 7a428cf2e424233910e6941975b984f54639d581
MD5 (NetBSD-6.0/mipsco/binary/kernel/netbsd-INSTALL.gz) = 989821fb043e7208eb7464b9cc3d6faf
SHA1 (NetBSD-6.0/mipsco/binary/sets/games.tgz) = f56d3e5181634f6a3d7908dcb002ebcb1c5aaff0
SHA512 (NetBSD-6.0/mipsco/binary/sets/games.tgz) = 0cff50ea13e34fe338edf70da6a28af061580feb4cb0773828adefa631ebd471a96568a3b30ea0f13f3bb116856a1b6aad8a5a52443346f52e4907f4caabc5f1
RMD160 (NetBSD-6.0/mipsco/binary/sets/games.tgz) = 1538910578dbb8eee2931200fac0ccda44758e5f
MD5 (NetBSD-6.0/mipsco/binary/sets/games.tgz) = 2bd5bd0ce72df5cf92767b8c3f7cc208
SHA1 (NetBSD-6.0/mipsco/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/mipsco/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/mipsco/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/mipsco/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/mipsco/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/mipsco/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/mipsco/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/mipsco/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/mipsco/binary/sets/xbase.tgz) = d81c7fe75a90f1b2111a066e78f1f2b3e8edcd8c
SHA512 (NetBSD-6.0/mipsco/binary/sets/xbase.tgz) = c241391fcde4a685a0a2af6f843844ac2e918024c447a2b5ea2cf5300f69acc42fdadf53d3b0348f10723b786bcc0f420e68fe36228375bf3cb5140de41eb40f
RMD160 (NetBSD-6.0/mipsco/binary/sets/xbase.tgz) = 2fbf60b64c0b7ac2be72ebf998d9d7390fc40705
MD5 (NetBSD-6.0/mipsco/binary/sets/xbase.tgz) = cd23ca0f6b359c4d695d4494a147d40f
SHA1 (NetBSD-6.0/mipsco/binary/sets/xcomp.tgz) = c8f5ff5ea9f9bbe231f92d8b7f3bfcbd2897069f
SHA512 (NetBSD-6.0/mipsco/binary/sets/xcomp.tgz) = c7590075195019078b68ec5f87e36479ae2e7b7749c523df4ef21c9d881c343cda9f5c2bba897dd313329a76fe5c18e920d92827878d98ce08a7dc776352cd02
RMD160 (NetBSD-6.0/mipsco/binary/sets/xcomp.tgz) = bc60e267cb6e11b4a7305f8b7074084e179e0ff5
MD5 (NetBSD-6.0/mipsco/binary/sets/xcomp.tgz) = 8e58ee51667def3bd015f79769c0d932
SHA1 (NetBSD-6.0/mipsco/binary/sets/xfont.tgz) = c551bd13179b9508c2e3a7e19cec323e8ba21844
SHA512 (NetBSD-6.0/mipsco/binary/sets/xfont.tgz) = 764527f2f3f4686759fcae3fd093f4e00f34e1249994126221a6f9afe73d4cf8ddfbe8b8311798da2ed38c52466a44bb26ea08ad5aab1c08eebcc9019b4223f7
RMD160 (NetBSD-6.0/mipsco/binary/sets/xfont.tgz) = b3a365bedf8b4a20f3920d0e33b9a42fce288953
MD5 (NetBSD-6.0/mipsco/binary/sets/xfont.tgz) = adac8939da213a0195cc1d351f979199
SHA1 (NetBSD-6.0/mipsco/binary/sets/xserver.tgz) = 686f6e6d722c9676351ab058736f7077b4ebd2f6
SHA512 (NetBSD-6.0/mipsco/binary/sets/xserver.tgz) = 5c818f22edbeac7325228af99d6da6e44b5eb659c7c9fcc6b332071e97a0a490081eda7f9e0f22d6a3f3a589ae34179b434751a6191db5c664683d7510aecedc
RMD160 (NetBSD-6.0/mipsco/binary/sets/xserver.tgz) = 0e9f066095b505c7f062bf877147189a7a2f0649
MD5 (NetBSD-6.0/mipsco/binary/sets/xserver.tgz) = cdd2e5127838437df40af161e0c8398f
SHA1 (NetBSD-6.0/mipsco/binary/sets/kern-GENERIC.tgz) = 814522219e9ef7544ff09e8a1aeec2544e62e456
SHA512 (NetBSD-6.0/mipsco/binary/sets/kern-GENERIC.tgz) = 2e81957dd87b68e9f3b69b89d1eaf2f1cea45ea237558bc50e953d1d4987e86bd1fd7701f00971b252df35c8dfa8c006b1777c5941ee4385894680346aa3f286
RMD160 (NetBSD-6.0/mipsco/binary/sets/kern-GENERIC.tgz) = 00a21ba72b46c7d743c2255d02a5fe462ac90774
MD5 (NetBSD-6.0/mipsco/binary/sets/kern-GENERIC.tgz) = 85c821c524ccbfede1d34c6436de595e
SHA1 (NetBSD-6.0/mipsco/binary/sets/base.tgz) = 1a989a4f1f5bd6aa082d2af9c9e339530994e710
SHA512 (NetBSD-6.0/mipsco/binary/sets/base.tgz) = 0b0029d19eba066306c386aa7569a5af8ce9e2d4e12f73539cceabb3bf3eb9696f006b0fefda001cef488fe4db40a3b519928fa139fd025c8e95669a2b7fddec
RMD160 (NetBSD-6.0/mipsco/binary/sets/base.tgz) = 30de7c1cda91944523275d184e5629beb3be54f8
MD5 (NetBSD-6.0/mipsco/binary/sets/base.tgz) = dbf30b2affc7c471b1488c36fab8c383
SHA1 (NetBSD-6.0/mipsco/binary/sets/comp.tgz) = eae1cdcf3ca993f6903f7c0315b3bded0cb19954
SHA512 (NetBSD-6.0/mipsco/binary/sets/comp.tgz) = a777f285e5b72c5cd21eae29dc374956a80b738acd11f13d536513cb30a31273043c8c73554e4d227df99d4ef7d20f15bcc8fafc1af1c16718dbe6276c28f5b9
RMD160 (NetBSD-6.0/mipsco/binary/sets/comp.tgz) = c4dcbd1a3ddec8bf1dddd2ec18d6d9de13dddcb8
MD5 (NetBSD-6.0/mipsco/binary/sets/comp.tgz) = 2df422a2c23ad3357f41a02d305ca753
SHA1 (NetBSD-6.0/mipsco/binary/sets/etc.tgz) = 4035cdd82aad7dd4b19ce7b4f18f93de45b4a35a
SHA512 (NetBSD-6.0/mipsco/binary/sets/etc.tgz) = 8df26b3954bda34cd79ee7036056c166654d36d642c7abddd10caafb0f6692bd745a1f336047de01dcefc32060712559b82ea4381dbb1c3d90eb962a798e9ed0
RMD160 (NetBSD-6.0/mipsco/binary/sets/etc.tgz) = cb67e9de5fb8560ab973dd6dd877f856582c0538
MD5 (NetBSD-6.0/mipsco/binary/sets/etc.tgz) = 7cc0cebd7f9c62e5e2cd7fe61e041a2a
SHA1 (NetBSD-6.0/mipsco/binary/sets/modules.tgz) = 17b0db51acea9fb430bf4f9ee65a3d10b7e01747
SHA512 (NetBSD-6.0/mipsco/binary/sets/modules.tgz) = 4b60e6a7405c8cdda48f8c85723b4245f3af0f31c333090bc3833f203606d91cdf067a02a9a179956642b368d4f78d1da53cb48bb783a0abf24ac5a1c56681cd
RMD160 (NetBSD-6.0/mipsco/binary/sets/modules.tgz) = 34019b1b1080db5d84607337765c6fb19c103651
MD5 (NetBSD-6.0/mipsco/binary/sets/modules.tgz) = f53620978a24bc909f9555125ddc6416
SHA1 (NetBSD-6.0/mipsco/binary/sets/tests.tgz) = 528b2898dfd85f9c51f382d31f357ff470fccbed
SHA512 (NetBSD-6.0/mipsco/binary/sets/tests.tgz) = fb52dddb334a13e19dfd657aa165a2be4c9ed546e739d337141735927c74ed444c63a2ccfc7676ddc7301ade3bd9b713ac238a3cc37b8716705222856f26edde
RMD160 (NetBSD-6.0/mipsco/binary/sets/tests.tgz) = 4bf30f676d0c7cdb6838ea7807c505de82372884
MD5 (NetBSD-6.0/mipsco/binary/sets/tests.tgz) = a150e930291903d9751319a92ad116be
SHA1 (NetBSD-6.0/mipsco/binary/sets/text.tgz) = 01174401e42d87684ed569f3001994142451fe29
SHA512 (NetBSD-6.0/mipsco/binary/sets/text.tgz) = 8f2c57883df809889d5371730ff738429f147eb659895ef3bd25c23a1c78ed3baacee1e09d82630daa92b9dc5810167924537cdcb1234536acbc02269acbfd2f
RMD160 (NetBSD-6.0/mipsco/binary/sets/text.tgz) = 690f4481c50ab0bcd2a2773e762f250b60a7d088
MD5 (NetBSD-6.0/mipsco/binary/sets/text.tgz) = 0bb0dfa52358016da848c0783a4e9ccd
SHA1 (NetBSD-6.0/mipsco/binary/sets/xetc.tgz) = 1824720386f5e1fca6135a77aad15444221ffd3f
SHA512 (NetBSD-6.0/mipsco/binary/sets/xetc.tgz) = 1f752393b0913fa39f222eb8cff44c9c4b8045008459e2c2122748114a5c69c684153aed737aabceb9fde1cb4852527d6f4375be177b6f8e270aaef9a3360ab7
RMD160 (NetBSD-6.0/mipsco/binary/sets/xetc.tgz) = 346eb417698387537bd483d12d70796c968c970a
MD5 (NetBSD-6.0/mipsco/binary/sets/xetc.tgz) = d9822dfdedccbf5fbd3bf6ab9b178186
SHA1 (NetBSD-6.0/mipsco/installation/diskimage/diskimage.gz) = 7f3836f8a683b1bc03880af9a5440ce0c43d45be
SHA512 (NetBSD-6.0/mipsco/installation/diskimage/diskimage.gz) = 1c513851002df0e3d6d11e33b5b62df96547e75c300c22737be8f553613404d94dbddf036cb2a00707b8d23b00a14ab3b37fd2251fa0f04b05ab35fd13c241a8
RMD160 (NetBSD-6.0/mipsco/installation/diskimage/diskimage.gz) = 8b22ef52868c3da50f224a8663683a990816d87e
MD5 (NetBSD-6.0/mipsco/installation/diskimage/diskimage.gz) = 0264b27319b43fd4e9c8c74e0478dac4
SHA1 (NetBSD-6.0/mipsco/installation/netboot/diskimage.tgz) = 62ce9d94cbc2d187ef03f9c4c22f0d865c482500
SHA512 (NetBSD-6.0/mipsco/installation/netboot/diskimage.tgz) = 4b53601368b9a70e6391efab32547a038c5b05b39ac73c1677f8b1317ca0d0458d1178218aa5457d8e0d8545a1f2a20e65e200c3e8d8b04e395d11886e30c22c
RMD160 (NetBSD-6.0/mipsco/installation/netboot/diskimage.tgz) = 8da2d5476ab14c8f44c732fc07202a234d5b4abc
MD5 (NetBSD-6.0/mipsco/installation/netboot/diskimage.tgz) = b83ebc427977f9b30679b06765d6babe
SHA1 (NetBSD-6.0/mmeye/binary/kernel/netbsd-GENERIC.gz) = 93157fd963d5e5967b626593297490082c21b426
SHA512 (NetBSD-6.0/mmeye/binary/kernel/netbsd-GENERIC.gz) = 300868a582f1eaa960d0d211374ca563a7a5f8347239b178a368ad3c527ee8ef76ca54628fbacc01521903cda04309b666953e1136d1f1fa8ddd28a046d1182c
RMD160 (NetBSD-6.0/mmeye/binary/kernel/netbsd-GENERIC.gz) = 7da72149fd4cbd471bf3abf076a4f33c8dc0a3a5
MD5 (NetBSD-6.0/mmeye/binary/kernel/netbsd-GENERIC.gz) = 053862a7ced67c229e688793b99d2fa1
SHA1 (NetBSD-6.0/mmeye/binary/sets/games.tgz) = 37e15b0950aaaf1af36427ae5dd5866c1879b6b3
SHA512 (NetBSD-6.0/mmeye/binary/sets/games.tgz) = e674abb1b1966acbc581a8f8b3e37841f48656e66f3302077cbb2de443487912b99521336ec5dc096fa3c7769b2d7623e1d1720a079407ec1861d32159a72ebc
RMD160 (NetBSD-6.0/mmeye/binary/sets/games.tgz) = 36070071e8315d2352be1b3e563ce7ef7625d184
MD5 (NetBSD-6.0/mmeye/binary/sets/games.tgz) = 0016145b46957b95dd158b4441dfb010
SHA1 (NetBSD-6.0/mmeye/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/mmeye/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/mmeye/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/mmeye/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/mmeye/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/mmeye/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/mmeye/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/mmeye/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/mmeye/binary/sets/xbase.tgz) = 564bce6dd7c37b5e941e41d50f363c133fdf8f97
SHA512 (NetBSD-6.0/mmeye/binary/sets/xbase.tgz) = 23b3306254c789331542e63974f9d7cdc54d99ba2cd67cf6a1c97f09afa4f6cf6aa8798e37b7c38f4a55afd1572154f291fa1bf2e2a1fc87165fd6bd7f89fb80
RMD160 (NetBSD-6.0/mmeye/binary/sets/xbase.tgz) = 476fbd8301bec0795a01552728ada44204d2dbe6
MD5 (NetBSD-6.0/mmeye/binary/sets/xbase.tgz) = 9fbbbdb3acfa4ccd9f0b309ea0ed0700
SHA1 (NetBSD-6.0/mmeye/binary/sets/xcomp.tgz) = 526a278a13833073e6231e879abdec85400ca888
SHA512 (NetBSD-6.0/mmeye/binary/sets/xcomp.tgz) = d2081a7536e3d7fa27a363afd8f6eb5fc3de4606f11b4857be2f1df59934da9d3d16e0df3a6edf4cdf52c8f579e0e4c06f07de0959b5213c21be1a8fdc2ee134
RMD160 (NetBSD-6.0/mmeye/binary/sets/xcomp.tgz) = 4a767ff08e9bcf98f886849cabebd1300397256f
MD5 (NetBSD-6.0/mmeye/binary/sets/xcomp.tgz) = a23a2d8be9298497b4b6d39fda330bf4
SHA1 (NetBSD-6.0/mmeye/binary/sets/xetc.tgz) = 93a5cfbbfdc641f91ab363d4b65d1158d71c0cfc
SHA512 (NetBSD-6.0/mmeye/binary/sets/xetc.tgz) = 85a67734801e81e129a37b73ae62211459fb16d4e52a945ef21a4bb908dd7d1d775233956086dd3d59af45bcea24fbd6f54a2a9175756d242b57e07d10767548
RMD160 (NetBSD-6.0/mmeye/binary/sets/xetc.tgz) = 6fb055d0e1b6264bf1ec396236d903218832377a
MD5 (NetBSD-6.0/mmeye/binary/sets/xetc.tgz) = af954b56e624948145b91aa451d77392
SHA1 (NetBSD-6.0/mmeye/binary/sets/xfont.tgz) = 6139c05273b6feeecdfcd11cbe0da91256d12a11
SHA512 (NetBSD-6.0/mmeye/binary/sets/xfont.tgz) = bb849454ede88d39ca60d6c513c833c2b29ea548adedb9f76870839174a3daa838d38a6ad701c78bfc164e690c5ebda9c9d00d4c23ad816df9f1962313afdbf2
RMD160 (NetBSD-6.0/mmeye/binary/sets/xfont.tgz) = ea8f123f9f78b429b77ff15c924803da1da2068d
MD5 (NetBSD-6.0/mmeye/binary/sets/xfont.tgz) = a87cb6769c37ae20dbf0dd70cef93d11
SHA1 (NetBSD-6.0/mmeye/binary/sets/xserver.tgz) = 6cd4fdb0ff00ef5d89b47e68e538c86e395b2054
SHA512 (NetBSD-6.0/mmeye/binary/sets/xserver.tgz) = e3b575cab95e297a05a26a03eeff0cf843d665ebdfd103846ad544bef1d04f93aa20ceab09b1dfaae8f59d89ebb1cbef400b7521bdb41f7542f23156540aa4c0
RMD160 (NetBSD-6.0/mmeye/binary/sets/xserver.tgz) = 7e52b98276b4d097f7babe918e6768ae9d7945db
MD5 (NetBSD-6.0/mmeye/binary/sets/xserver.tgz) = a505ab4ce0801aabe21fe6038b63809f
SHA1 (NetBSD-6.0/mmeye/binary/sets/kern-GENERIC.tgz) = e14c11631e2e88e3b57b7d20fc3c645faf18fe37
SHA512 (NetBSD-6.0/mmeye/binary/sets/kern-GENERIC.tgz) = 29dfa7b68e616b6192620316e85972381e7c104070f2570a4222b22c738eddeebd3d8a49518f64adada548f3cbc31a2c501db6f186d7c2fb4178cd9187d7e289
RMD160 (NetBSD-6.0/mmeye/binary/sets/kern-GENERIC.tgz) = b4289ab1966d34a7fc19d6736c374e98c969f2f0
MD5 (NetBSD-6.0/mmeye/binary/sets/kern-GENERIC.tgz) = 6469d2d171a8a4bb99ec9a396d43c4b1
SHA1 (NetBSD-6.0/mmeye/binary/sets/base.tgz) = f26be3019cb31f22bea52c431ec65026536117d8
SHA512 (NetBSD-6.0/mmeye/binary/sets/base.tgz) = d595fa01b6176f11dd722a4c0416e83975fabb7669cfb018b364681a82c1fab6d456e35aee5c1e8f62d721708976da1ddea203ba31c782265d95dadc046080b8
RMD160 (NetBSD-6.0/mmeye/binary/sets/base.tgz) = d08e45a281fa538ba47d5e01b7ca7d2c34d05847
MD5 (NetBSD-6.0/mmeye/binary/sets/base.tgz) = 3a7f10d4e415aeeab19f10c6cd9683e6
SHA1 (NetBSD-6.0/mmeye/binary/sets/comp.tgz) = 0e37635ed1c12873ba9443f765fa8ecd7441aa43
SHA512 (NetBSD-6.0/mmeye/binary/sets/comp.tgz) = 2528bf14ced0b15c49452d705cb60cf9e33187fc759920e4e400fc7a2cbb140efb4e46a135a3c06b7c8b4ae9a2b06094e4f14b32372d3b744d0d898818ac0e4a
RMD160 (NetBSD-6.0/mmeye/binary/sets/comp.tgz) = 51bee8a0d676f8f51918b76a816ddd417cc38a6b
MD5 (NetBSD-6.0/mmeye/binary/sets/comp.tgz) = cb209fc3bf35ff197cec755c15879f27
SHA1 (NetBSD-6.0/mmeye/binary/sets/etc.tgz) = 574390c9066c5a2867c217726a60c5b0fcd954c6
SHA512 (NetBSD-6.0/mmeye/binary/sets/etc.tgz) = 29f0fdf264263ddca8a04e2a645999127bac3025daf2ee560647f4d15fbaec3d929c725256244fa561ee69d8eddf4c9439d81629894dcccba13d063ec1a0d70f
RMD160 (NetBSD-6.0/mmeye/binary/sets/etc.tgz) = 5e68a1f6a88a56e63788525987bcbef81314cfd0
MD5 (NetBSD-6.0/mmeye/binary/sets/etc.tgz) = bb1639782d6f4027a7b872732e0f819c
SHA1 (NetBSD-6.0/mmeye/binary/sets/modules.tgz) = d687589906a811d0ecf22c89bf54248326aa0bef
SHA512 (NetBSD-6.0/mmeye/binary/sets/modules.tgz) = fd536f114ba71d1ed7241a81f89af7d26a8b554b6fa12ccb16d2d4e6e23029be66fbf4d2cb4990c65102bf3206972ca00a329c6a6c8cb86ad283356aa38434bc
RMD160 (NetBSD-6.0/mmeye/binary/sets/modules.tgz) = c2bde6622408d084a8ed7fbeafce2260220c5560
MD5 (NetBSD-6.0/mmeye/binary/sets/modules.tgz) = d833b7a1b95c96a2fb4c526ddc7b6a62
SHA1 (NetBSD-6.0/mmeye/binary/sets/tests.tgz) = 28c07cc4ab9c6688c615ecfae68cdd7afeaf627b
SHA512 (NetBSD-6.0/mmeye/binary/sets/tests.tgz) = bdf6463ad028eef4bfb3feb78d43924c4c176b8195509d25d1f98cc772cd1d4edf1820497dfb1baee7dcc142d0e64e78d3203feab0636fe4d821a4cd64250459
RMD160 (NetBSD-6.0/mmeye/binary/sets/tests.tgz) = f10cf5b11bb96eb0be1c08f698897cb2c609c2f8
MD5 (NetBSD-6.0/mmeye/binary/sets/tests.tgz) = a0f6fcd15874d6439da5f852487e3539
SHA1 (NetBSD-6.0/mmeye/binary/sets/text.tgz) = 59d1600b40b17a4c0459e6b3e0a0c2cff1194270
SHA512 (NetBSD-6.0/mmeye/binary/sets/text.tgz) = 0c44d1c586ad834e2ecca5bf57e88d3315e19341deefb36e8f9584d22328fa255d9e7c26028091e8e0465aea5e68ead7fe343138f54cd219c69eaeb7facdf85f
RMD160 (NetBSD-6.0/mmeye/binary/sets/text.tgz) = 258e107b924fc698dd9ac2933f4b040374cc9291
MD5 (NetBSD-6.0/mmeye/binary/sets/text.tgz) = 241e3f6acd2a44f23f1368c4897ea43a
SHA1 (NetBSD-6.0/mmeye/INSTALL.html) = 307db90e852c93274d4d475013aa29db682ad785
SHA512 (NetBSD-6.0/mmeye/INSTALL.html) = 6a10441885d95313036134e897f41508c5823386532d4e57e33d693164d8a6a727f39aad44e3606b614cd037eed7096a2c88962b495f1915dd9db0f1165897dc
RMD160 (NetBSD-6.0/mmeye/INSTALL.html) = efd26420ebc8e2bbca8e756d05a8978b3c8f6a21
MD5 (NetBSD-6.0/mmeye/INSTALL.html) = 4074818595561d15e823f3fcf0e319ab
SHA1 (NetBSD-6.0/mmeye/INSTALL.more) = 508b142e63e52558c975e9b9e8d3ddd5f392f47a
SHA512 (NetBSD-6.0/mmeye/INSTALL.more) = da8142dcec08c7c1dd64b680e987abed58c6d8685c93c66fc4b1218f18cb64c626949d841b5ee379a09a0b8d1ae8af12a2690d8dbc4f768497f1c3e8e646f377
RMD160 (NetBSD-6.0/mmeye/INSTALL.more) = 58a81c1cc83e5033ab3a527190e7ee1fae534d76
MD5 (NetBSD-6.0/mmeye/INSTALL.more) = 248b67d9ac8e481bdfd7952cef78a14e
SHA1 (NetBSD-6.0/mmeye/INSTALL.ps) = 4bf1609d1513b6e41cfc5cb480dee72432e5a416
SHA512 (NetBSD-6.0/mmeye/INSTALL.ps) = 691a138a6af72a7e42cc03c71d6bcde5c210ec570507ee25333e1ef9702aa689be0c7d46009aea72dc1dbca93c52a46f1745c1dc985725fa5e63922c4f5927e2
RMD160 (NetBSD-6.0/mmeye/INSTALL.ps) = 4387c8bab777418797ccf13a05abc4a085b28fab
MD5 (NetBSD-6.0/mmeye/INSTALL.ps) = 98a48fdc04e24e9d6976363721d644aa
SHA1 (NetBSD-6.0/mmeye/INSTALL.txt) = 75f5b090195d57c0542212c1b5eed5e5ccd3297e
SHA512 (NetBSD-6.0/mmeye/INSTALL.txt) = 7c1310578154afee26aa66db4198dc5ba98b9f985f05c9e6a3847fe88e947e1161d541527dd1986ff3ca20ad92468483a2195013ccb8d16e89914006f1d0b566
RMD160 (NetBSD-6.0/mmeye/INSTALL.txt) = e05dd82511b132d9047eddeee16043524246867b
MD5 (NetBSD-6.0/mmeye/INSTALL.txt) = e3de50b8b8480113536fa86b62245902
SHA1 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-GENERIC.gz) = a13aee658df1b52a7aaf3029d425e922ec26d4c1
SHA512 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-GENERIC.gz) = 218d038a60dc542188f10c56e096434ebd26acddb4919ceb7ef4263ee85a3d25c60e9265683298fccfdcdb6b7d155138428f5029d37c11455d8f49077e0753cc
RMD160 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-GENERIC.gz) = 5963381f5dff4d47bd58e488fdead7f9e4886d26
MD5 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-GENERIC.gz) = 86d45589986fb0bb05dc8c7dc77278a1
SHA1 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME147.gz) = ea20eed9edbc71eca530c3280770e6f8b09fa270
SHA512 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME147.gz) = 75a7255373b228b1932477a121111add15fa0ec8b03594794c6b6e3c45055f7ca2be29923b263ffab7ab51b3c9bebc9d8280ea701916b76f581dcc18bb510584
RMD160 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME147.gz) = ea31d6a830019327b2c3435afdfd947d8464ef23
MD5 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME147.gz) = 15c5a42f970736283e7a6a0849bd86d4
SHA1 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME162.gz) = 041755caacc323b08d0d4bc7d7d50fb178374f24
SHA512 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME162.gz) = 0f9165f9574da412d25c74601c42f6678b78ae78c04b0eb0753dce9266289fb656071115c32425e79dbb88df2482971d3808095228ce56fa7124ba34a396b0d2
RMD160 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME162.gz) = d93f72e9bf72b15a7f740977f51270620e17afe6
MD5 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME162.gz) = 0c35781afea5d9df108ffe2ff82563f9
SHA1 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME167.gz) = 4ede6bb44a66a1ca0582e2bd6b38d94dc66e5c22
SHA512 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME167.gz) = 33feb4ceaa5352e85ed445bb355d997181f69cfe2b08de57a631be7c93c85cb1f42d052bd6de489f6ac631c88c77192785d170e41c0ec6315b88fd429d88f3bf
RMD160 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME167.gz) = 6768bab2219da0e3ef78324a701e0fb112c18153
MD5 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME167.gz) = 6e65873511ff8bb90d721f0c2e559bf4
SHA1 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME172.gz) = 01ee771c809080dce253e2acbce1bcdfdd46b055
SHA512 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME172.gz) = 65e0680d9ce92aed2ba08971a434f2a290ccf9f85ce3365327ec8185c5d57e1bdf08593c5ff18e3c0e6581c78ceadd570d25fb1e714edd7896fca09fae727695
RMD160 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME172.gz) = 93136f2a9773f5e243622e756ea3969c3732403d
MD5 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME172.gz) = 08a73d99398505743b3de4f5394ffdd7
SHA1 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME177.gz) = e884157c76ca076bd53f3c1752b4f95ade2eacea
SHA512 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME177.gz) = 13a0a7e5c50848593ccbf5876cf27988b138582fff9d4f140cc3b63b9d768163ddea65752fd0676614ccbb85f64d0efdd3611ff2fce31837afb9acebcdb63657
RMD160 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME177.gz) = 898154c89255fca36e0ec7051638b32cfb541dfd
MD5 (NetBSD-6.0/mvme68k/binary/kernel/netbsd-VME177.gz) = 77d2383aa5faff6ecc139af1e7dc51fa
SHA1 (NetBSD-6.0/mvme68k/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/mvme68k/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/mvme68k/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/mvme68k/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/mvme68k/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/mvme68k/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/mvme68k/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/mvme68k/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/mvme68k/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/mvme68k/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/mvme68k/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/mvme68k/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/mvme68k/binary/sets/xserver.tgz) = 1d20fd039546a66b428b854c659ea5e2ee9f8114
SHA512 (NetBSD-6.0/mvme68k/binary/sets/xserver.tgz) = 091d09049d0af5555feec02065234e281677aeb3ecbab5a19413d924a8b58dc7f64401ef68e68bc52d8b6eeb73e091d351c45805374b8bf276b0181415968561
RMD160 (NetBSD-6.0/mvme68k/binary/sets/xserver.tgz) = feef5d52278d23637d7b8ba77639bfd96918e90e
MD5 (NetBSD-6.0/mvme68k/binary/sets/xserver.tgz) = b2bd6057c374376388766fc8667eb9a9
SHA1 (NetBSD-6.0/mvme68k/binary/sets/kern-GENERIC.tgz) = b5f282108361e17368d9ca8615f3021473169abf
SHA512 (NetBSD-6.0/mvme68k/binary/sets/kern-GENERIC.tgz) = 165bec66d47652cb43981780aa068960efd933f9538145f363a559d8cc58d16d78ae5d588d98bb39795cdde762f280b36750211da2ed3a5298b0d94edf2e0bb5
RMD160 (NetBSD-6.0/mvme68k/binary/sets/kern-GENERIC.tgz) = 1ad2613634bf8c7654f87c4a54c9d3e6c0b1f349
MD5 (NetBSD-6.0/mvme68k/binary/sets/kern-GENERIC.tgz) = b4b8eee7f4f1854f8632928b5ee516de
SHA1 (NetBSD-6.0/mvme68k/binary/sets/base.tgz) = c3e2ce1f54481094d67ff2a87a6b5000d8130ad2
SHA512 (NetBSD-6.0/mvme68k/binary/sets/base.tgz) = 02c29cbe0171a6c5312fb1a468e68bb8a5a5258df936c0a31fccb7bc04c8e1293d41ce6f67de08ca5584c6208dd9050c70f99c1c3b8c35a499bf94a957808797
RMD160 (NetBSD-6.0/mvme68k/binary/sets/base.tgz) = 7920bf359b28e793c0410ce08a98025dd3ea6971
MD5 (NetBSD-6.0/mvme68k/binary/sets/base.tgz) = c9893e324841703eed47b475c2d37607
SHA1 (NetBSD-6.0/mvme68k/binary/sets/comp.tgz) = 100d0663bb2c601df55d490315c9b36b7ea700f3
SHA512 (NetBSD-6.0/mvme68k/binary/sets/comp.tgz) = 70bf8fc817821c6c5061776bac0ee9ebf8567fd19a2d98c751d1073a706290eb9870627b2f1fea650d19a876e8a5215e5e52407437d6be977777013f5f59fa8a
RMD160 (NetBSD-6.0/mvme68k/binary/sets/comp.tgz) = 497fb5a993abc08f005e1f287b8520dbace84549
MD5 (NetBSD-6.0/mvme68k/binary/sets/comp.tgz) = 24cf494b20e22c8630e6bbba17b96bc1
SHA1 (NetBSD-6.0/mvme68k/binary/sets/etc.tgz) = c6f1d70ad7bf234c79f4e15508b09c7b0c33f1d8
SHA512 (NetBSD-6.0/mvme68k/binary/sets/etc.tgz) = 13a8204695757c27b69ed52a02c55a1e2b2d6140a5dfffd8f9f24625a56c9f3bd1f5de868a99c265e5fc759cb80cb7239dbe965c44693fcce6e98e7691076d36
RMD160 (NetBSD-6.0/mvme68k/binary/sets/etc.tgz) = 7365dfc472b89a3b017cdcf41bf10f928dcf723e
MD5 (NetBSD-6.0/mvme68k/binary/sets/etc.tgz) = 1e8812db12c38683ce12950d00ded70a
SHA1 (NetBSD-6.0/mvme68k/binary/sets/kern-VME147.tgz) = 96374aa39856e13589e98f1552dd756b4f1911e9
SHA512 (NetBSD-6.0/mvme68k/binary/sets/kern-VME147.tgz) = f31cc707ced4c96b2019d7d44169e203f6df2616d09e7a3f4e2a4f0e4c509e978ba324761bdd26d1ee443e4d72e90e00effc24e537644013d5a2de95f3812574
RMD160 (NetBSD-6.0/mvme68k/binary/sets/kern-VME147.tgz) = 219ba053784c069875b7434631463f61293bab1c
MD5 (NetBSD-6.0/mvme68k/binary/sets/kern-VME147.tgz) = 5349b757ec12a84df1e102583674646d
SHA1 (NetBSD-6.0/mvme68k/binary/sets/kern-VME162.tgz) = c56ed72a7c0da9fb44d19ece17801d1b7527e05a
SHA512 (NetBSD-6.0/mvme68k/binary/sets/kern-VME162.tgz) = 12078cbbbd7b1093f956b6526643b446890b82c043b17fc97e61259f54fc81cdbd8ebc56a735eb5937bff004b63d7f291e7ea2cd0dad3bd5b07f089df2c83ab7
RMD160 (NetBSD-6.0/mvme68k/binary/sets/kern-VME162.tgz) = cfddd4246205e867729907015fc6dd16da7fe6e5
MD5 (NetBSD-6.0/mvme68k/binary/sets/kern-VME162.tgz) = 432a04306d40c933f2e04e5a332800e2
SHA1 (NetBSD-6.0/mvme68k/binary/sets/kern-VME167.tgz) = 90302a0bc14e084f2643de427e87bc5b04553ab0
SHA512 (NetBSD-6.0/mvme68k/binary/sets/kern-VME167.tgz) = 62c19ad0fb282a427beb1bf2d0d1b9a1cfc2681bcd04e3fb51ade0e793f26b52d04b5eb28e4c25524244e8c6dad724fc07de7a6544438cbd2a503972999f23ea
RMD160 (NetBSD-6.0/mvme68k/binary/sets/kern-VME167.tgz) = d84d45b735805261e014ae52746e279902ad7e9a
MD5 (NetBSD-6.0/mvme68k/binary/sets/kern-VME167.tgz) = 8d38010b19b455f324d0b08485482395
SHA1 (NetBSD-6.0/mvme68k/binary/sets/kern-VME172.tgz) = 017724bf77e35e2021308dadafe5a90e59a32498
SHA512 (NetBSD-6.0/mvme68k/binary/sets/kern-VME172.tgz) = c3ca75dd2287e6af935934a4dabba21f085889383c395c8c72636047d682541ee5669998dd5988e0127879d7b34df260dc9109c292a2c2663ca8cfbed9c03278
RMD160 (NetBSD-6.0/mvme68k/binary/sets/kern-VME172.tgz) = d8676c3325db3b3f619599f9e7a33cf3b1eb5bf1
MD5 (NetBSD-6.0/mvme68k/binary/sets/kern-VME172.tgz) = b89cd9b71b5b68285bf36d958a3b8746
SHA1 (NetBSD-6.0/mvme68k/binary/sets/kern-VME177.tgz) = f9bdc8e89488ba91385adf9f0960633ba7dbf6da
SHA512 (NetBSD-6.0/mvme68k/binary/sets/kern-VME177.tgz) = 642825fa141d2de457a0d3eca1224cdb5b087383dca6ed4f36668be6f6fe9f9e30b942ec5eb177b4733aaad40f6a0eace0f744fba3777eb356dd75d8c113785f
RMD160 (NetBSD-6.0/mvme68k/binary/sets/kern-VME177.tgz) = ef6abd72e2a71a5995c628107f473ceabe2945e6
MD5 (NetBSD-6.0/mvme68k/binary/sets/kern-VME177.tgz) = 42b79dd96c98e97639d1a9d698673996
SHA1 (NetBSD-6.0/mvme68k/binary/sets/modules.tgz) = 5479e9142d1f21bc8346cd34e9f00761e2e05f48
SHA512 (NetBSD-6.0/mvme68k/binary/sets/modules.tgz) = 094b2d20eb3dc973fc212085dcc4d110249fc71eb017ea390b7f9aeffd4d8d0aeb51d421b4c95add33a17ac5d7dae1e3c9fc654f8a4181d078dbe2bac95a96b6
RMD160 (NetBSD-6.0/mvme68k/binary/sets/modules.tgz) = 187b451690395a8e05f22294016a627e4db4539f
MD5 (NetBSD-6.0/mvme68k/binary/sets/modules.tgz) = 5d3d2aa79488e5f8e90ff7b26f80fc97
SHA1 (NetBSD-6.0/mvme68k/binary/sets/tests.tgz) = b994ffc0b63206aa7a8ab6de392ec33189894764
SHA512 (NetBSD-6.0/mvme68k/binary/sets/tests.tgz) = 714ac7fe966021f842409c135711b7adfb03c4b6a095df9527d02d1b83b14db62f1ec1a7eaaaebc68e4f12588e52fab9318c21912f20b8380a7a8b8e30e8576d
RMD160 (NetBSD-6.0/mvme68k/binary/sets/tests.tgz) = ba3293225ace42d8c47037879ba0cecd8f883aaf
MD5 (NetBSD-6.0/mvme68k/binary/sets/tests.tgz) = e32c51104c90b3b684d1d640ba79b46a
SHA1 (NetBSD-6.0/mvme68k/binary/sets/text.tgz) = 61b398a67c4c47f8633d393dda778ac5c1ed8698
SHA512 (NetBSD-6.0/mvme68k/binary/sets/text.tgz) = 29b9bb0c5aefb3d765eb6204967c3ff14f7a28b0987723c3685a2b34fa799947c876971bf524a8e43b146e9e91e37cbc89a0c4da87c47e9856bdf9d96a67f125
RMD160 (NetBSD-6.0/mvme68k/binary/sets/text.tgz) = a73c95546d018a1853c6d00f5924da860fe9e5d0
MD5 (NetBSD-6.0/mvme68k/binary/sets/text.tgz) = b0dbc4ddc47ab74ab24bdc0bede769a2
SHA1 (NetBSD-6.0/mvme68k/binary/sets/xbase.tgz) = a31e38ba081b5abb671cf5f7350799782a628712
SHA512 (NetBSD-6.0/mvme68k/binary/sets/xbase.tgz) = fa78acfbace842e0995a7f181193220f80dcae731221bf1722ece1a5bef6de846089b919beebacd7921567105f8ddae9bdaaabe0b065b62a0f18d917f03b54d9
RMD160 (NetBSD-6.0/mvme68k/binary/sets/xbase.tgz) = 1cb1d631feb356639d3b1362518ba5f09f084510
MD5 (NetBSD-6.0/mvme68k/binary/sets/xbase.tgz) = dae635ef3bd334e14be8b8b66a227353
SHA1 (NetBSD-6.0/mvme68k/binary/sets/xcomp.tgz) = fbc80a68ccbef35cab4228a5ce96b22b383d9080
SHA512 (NetBSD-6.0/mvme68k/binary/sets/xcomp.tgz) = 694780dd165f0831eac6b47ec6fd5c60589749a21e88700965a99154067b76fa374b8f7a55f26963cdba8caafe9548f66b222826cc265393e992addbf19dd468
RMD160 (NetBSD-6.0/mvme68k/binary/sets/xcomp.tgz) = 7db19d8ccd7dbda6032d9c29a65d03bf9c71644d
MD5 (NetBSD-6.0/mvme68k/binary/sets/xcomp.tgz) = 913bbabb9e401d6a31cc09a3d259b132
SHA1 (NetBSD-6.0/mvme68k/binary/sets/xetc.tgz) = 1aa27f8388f5bb0aa85eec00e225fc92736da214
SHA512 (NetBSD-6.0/mvme68k/binary/sets/xetc.tgz) = 1b524f2137ea4cc5f816623a7b042b05b1303664d244b0172799b365cb16df34f0a8a47277077055baf266c7c92d5f112830ed357fa811bdf6bd713b2f90275e
RMD160 (NetBSD-6.0/mvme68k/binary/sets/xetc.tgz) = b50ff4df41b13b9276393e871e976648d344ee7f
MD5 (NetBSD-6.0/mvme68k/binary/sets/xetc.tgz) = d7709463a7a8404b6ef93aeb60755f7f
SHA1 (NetBSD-6.0/mvme68k/binary/sets/xfont.tgz) = 4f8d5c51bdd02bdff0dca66f54e5ebd25e4d6110
SHA512 (NetBSD-6.0/mvme68k/binary/sets/xfont.tgz) = fbad852b3a697906a2ffb460f8fefe8f9aa9f2088f66423d838abc18941d497bf6f9491934de0e3359159d35c98f205b23bb9c328f832e7e62c99310067687e6
RMD160 (NetBSD-6.0/mvme68k/binary/sets/xfont.tgz) = 41c3686ef9c2cc25eff4b9923ae3163aa9c28590
MD5 (NetBSD-6.0/mvme68k/binary/sets/xfont.tgz) = 2ec6dafe4dbe645a90960ade3e86caed
SHA1 (NetBSD-6.0/mvme68k/installation/miniroot/miniroot.fs.gz) = 12df0f0fef9a077c27d02d2960d04ea3a148fa76
SHA512 (NetBSD-6.0/mvme68k/installation/miniroot/miniroot.fs.gz) = fa3eeb4f14b741311ef72000e6c283195f4513e81a35f9c0a1829dbf8795d0518355b0d54d35a17c56568ac8e6db5977c1a7d4bb902c20786f901efd9b35ec42
RMD160 (NetBSD-6.0/mvme68k/installation/miniroot/miniroot.fs.gz) = aeeeee1a475c8ef1d595c5cdff5bb3cc765bbea1
MD5 (NetBSD-6.0/mvme68k/installation/miniroot/miniroot.fs.gz) = 9bff82e8d2d57a3fad2e831272aee965
SHA1 (NetBSD-6.0/mvme68k/installation/netboot/netboot) = 588e8808e6e24e0a570f529c0dde8f02bcb74edd
SHA512 (NetBSD-6.0/mvme68k/installation/netboot/netboot) = 6e0b6bc3f5de5940811d0929dd0d5f65351402176ae9c8e44a0220b807b41e98c8363d4917b1f506a847c3510bfec3d78584eae8f9e1c337130b37da4d3c6f07
RMD160 (NetBSD-6.0/mvme68k/installation/netboot/netboot) = f1239616c5743fba614065450b1aa9ca8f4057f5
MD5 (NetBSD-6.0/mvme68k/installation/netboot/netboot) = ee5d8acc83e9f62ee902acd1da69b3d5
SHA1 (NetBSD-6.0/mvme68k/installation/netboot/sboot) = 5f45e50e16e46d9ab20ad20d929d1a01e4d60ad5
SHA512 (NetBSD-6.0/mvme68k/installation/netboot/sboot) = bcd667220dd50d44aa895c0d76b9ee906dd6e3ff72d9bb80167cf48ddb6456c5cb8c06829e66ee06cf69c99e50afa3005f23a6854fc56ea2a027b50315eace2c
RMD160 (NetBSD-6.0/mvme68k/installation/netboot/sboot) = eee81405c2dd838804a4d13219b431e93120fdae
MD5 (NetBSD-6.0/mvme68k/installation/netboot/sboot) = bd2f963054643f0ffa22af1ea8ecda92
SHA1 (NetBSD-6.0/mvme68k/installation/tapeimage/bootst) = df842ee6dc97b75a8f8b8e20996f60fa5c0e6ff9
SHA512 (NetBSD-6.0/mvme68k/installation/tapeimage/bootst) = 3a6900acfffe7118e262864abea6c325e1e69060f2b47556b8865f2f28458beb29e56d96a61a8b98520530ba3d6b5ee4e0d5499cc698497445e0d480d0b9466b
RMD160 (NetBSD-6.0/mvme68k/installation/tapeimage/bootst) = 78ea882ed8fd3cadb3b25971c7d0ad2ef159918b
MD5 (NetBSD-6.0/mvme68k/installation/tapeimage/bootst) = 720e708780805b3a7afa57bcdb9ee62b
SHA1 (NetBSD-6.0/mvme68k/installation/tapeimage/netbsd-RAMDISK.symbols.gz) = 0c590bfc2ae0c74a96ac421e16e9f6aea312d63f
SHA512 (NetBSD-6.0/mvme68k/installation/tapeimage/netbsd-RAMDISK.symbols.gz) = 1f91be15380c61c4b2eebc8e34fb6eba768dc06fe958ba03cd1a074ef76e77c490d63484834b458bef181f18a5dd84444dcb33b4ed1b35360f7df9ab727a5088
RMD160 (NetBSD-6.0/mvme68k/installation/tapeimage/netbsd-RAMDISK.symbols.gz) = 49fe1be9202c415a147b25ca21b57c4e91f3d886
MD5 (NetBSD-6.0/mvme68k/installation/tapeimage/netbsd-RAMDISK.symbols.gz) = 387b741997770fa953dd689bdc7c2dce
SHA1 (NetBSD-6.0/mvme68k/installation/tapeimage/netbsd-RAMDISK.gz) = 071feb6189682ed7fde6e75790b17ed023512e15
SHA512 (NetBSD-6.0/mvme68k/installation/tapeimage/netbsd-RAMDISK.gz) = 029ef207dc80563c3d28369d7f6b1a99dae4ca573dccae9983bf3d135cbef8aece9d69a8fafe0c989bf5fc851d5a31d61c77911f7680dc4584a9229f582359f5
RMD160 (NetBSD-6.0/mvme68k/installation/tapeimage/netbsd-RAMDISK.gz) = 6ca56e2be45d06a300f533941a582caafcbe5623
MD5 (NetBSD-6.0/mvme68k/installation/tapeimage/netbsd-RAMDISK.gz) = 16c9e88acf91351f83fcec78132304bd
SHA1 (NetBSD-6.0/mvme68k/installation/tapeimage/stboot) = 652f8b35ed92540699c4a0b92d781bab584ff9ed
SHA512 (NetBSD-6.0/mvme68k/installation/tapeimage/stboot) = 63627858e9ea5ddf22d56e9f5bbc94c53a5fa45eb877b7b540931eaee973bf82251afaf6acf6078456f9aec047eb60b7bd66e694bafaa2210b51b0e83e19fd21
RMD160 (NetBSD-6.0/mvme68k/installation/tapeimage/stboot) = 1283f83860ab4f66692d5ef01d0cdeceb40668b8
MD5 (NetBSD-6.0/mvme68k/installation/tapeimage/stboot) = a54cfad8cf02a1a31a004dbc5c0a037e
SHA1 (NetBSD-6.0/mvme68k/INSTALL.html) = e3f98f5f338b9417a3f88ccc83fda9eb47696c0e
SHA512 (NetBSD-6.0/mvme68k/INSTALL.html) = b1750f826b36c84a821126455bdba2e1116bd51cd3b404c7b0c0b80110130e81e19105cd5d04563d31824e996820e1992486827b818f3f9a01d87f2b2d4da46b
RMD160 (NetBSD-6.0/mvme68k/INSTALL.html) = d6ab4cf012544dae8696f8f41208dd27ddc1ea7b
MD5 (NetBSD-6.0/mvme68k/INSTALL.html) = cd685f3e5b2052919547bfd9c8d98d0e
SHA1 (NetBSD-6.0/mvme68k/INSTALL.more) = 82746f9fb96a2db4cec433f8bad448fccaa98eef
SHA512 (NetBSD-6.0/mvme68k/INSTALL.more) = 676952c9168c0dcaea1eef58271884f95e8ad80380e33a060e29be2da7e7b139a731369e6cc0a112779be95c412de4be19f716284a3724b41f93cc5075e93b9d
RMD160 (NetBSD-6.0/mvme68k/INSTALL.more) = 79436013f39135308106f0d08b8a0fc951773a09
MD5 (NetBSD-6.0/mvme68k/INSTALL.more) = 95f12215d3961d47f7cc70f786c331da
SHA1 (NetBSD-6.0/mvme68k/INSTALL.ps) = b2c39b95dd800c754733c8986e8f279a5895b9f9
SHA512 (NetBSD-6.0/mvme68k/INSTALL.ps) = 120009d7f5781b9c060f3852d92bc047590b3d86d6491aa700c61d935ce8bccec1ddcca64f60fe17ffffa66871ec9c9908d42b738f8e37def53c97ac3a339c6a
RMD160 (NetBSD-6.0/mvme68k/INSTALL.ps) = 1cc7939281b187b32afa1a30955dfc3751d8dc4b
MD5 (NetBSD-6.0/mvme68k/INSTALL.ps) = b0c09fe08cf758e2dea14a23111c0c2f
SHA1 (NetBSD-6.0/mvme68k/INSTALL.txt) = d25191ccd7930c3335bacbc56dfcbfd7ab04091d
SHA512 (NetBSD-6.0/mvme68k/INSTALL.txt) = 581736f1c9236b74a0321e060d27305911aa735244a5b2e54825617cd97fa5bc1523472b87b65057fb3bc0f08216c7b53e60dca0c8ead238d7ea3da16502f16a
RMD160 (NetBSD-6.0/mvme68k/INSTALL.txt) = 05b193bfebc12030dcbe3c96273582360eb132e3
MD5 (NetBSD-6.0/mvme68k/INSTALL.txt) = c895e463555bd32f7b61fa481b63213d
SHA1 (NetBSD-6.0/mvmeppc/binary/kernel/netbsd-GENERIC.gz) = 418f7d15770b902578623141540daa7987ed2659
SHA512 (NetBSD-6.0/mvmeppc/binary/kernel/netbsd-GENERIC.gz) = f8ad97d747d2ffc1de7408ae9360afe25fb82eb60f73fc62198432ecb059aa64dfdc7ed74fc0d63e70a815d12414c6c9f1d8e5a717d614bafa0c9c299548ab9a
RMD160 (NetBSD-6.0/mvmeppc/binary/kernel/netbsd-GENERIC.gz) = e3603ef7455506046e4f789e2341020d81bfb46b
MD5 (NetBSD-6.0/mvmeppc/binary/kernel/netbsd-GENERIC.gz) = 713e8e8d42cebf7dea0ae4e96670a52a
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/games.tgz) = fbf69640b362df2446d7414ac3a13db100c289eb
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/games.tgz) = d86b41d79aea4204ccf45306937a5ec45629e37873a5956c0280fd1d54cf4a808542d097f9d978fd3277791554ce8a6fb4d0ccd46e30210549d5cce1145bb499
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/games.tgz) = fd2aed1eb51f4db522fb5163e8fde4c040b55d30
MD5 (NetBSD-6.0/mvmeppc/binary/sets/games.tgz) = 997ae5fb42196a787df89873be321e47
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/mvmeppc/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/mvmeppc/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/xbase.tgz) = b7c8146146ebc15191c1bf53f51f8f980fe8c816
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/xbase.tgz) = 091f5d00414a9ad478811bfb0e953f2a204869bc5e310790886a59c11953974cf7068cd9a96c8823856ab2687ffa2ac31e97494f47338bfe31d80217382c59f1
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/xbase.tgz) = 4bd1bfdc7c1e1619de63fb83276c21039b945357
MD5 (NetBSD-6.0/mvmeppc/binary/sets/xbase.tgz) = e621806afa0fcc9cdcef0f9b3c7dfcbd
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/xcomp.tgz) = 960dd33b61e19dec92096a54791aca1cdf56ce54
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/xcomp.tgz) = 111dd50041e4008aeadb898c3d08f9995c5c13a41ccc20993b4ad51c81d7f6ee3b4dab16158ac7436d085b80e4a643b236a9c97edd3e915125a1466154538420
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/xcomp.tgz) = 0cf8b9a8d1489a32ba0b5a277d90e429c40650fb
MD5 (NetBSD-6.0/mvmeppc/binary/sets/xcomp.tgz) = 4ba4077ca2c0edf71813165fcf5321fb
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/xfont.tgz) = 4711fcbade0238d60b7b89cf0d7873b03d7ed1f7
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/xfont.tgz) = c29130f55f536f602253a7f92f073ebf6ba02baea09f8011530bbf7e03ae96594334f0c60b29a8bceaffc3eb8b152db47548e38a3c8a14abc5fe35bc5d1d7d93
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/xfont.tgz) = 3b01af0f8e34367988d559fe0fbb0acbbafb0b2b
MD5 (NetBSD-6.0/mvmeppc/binary/sets/xfont.tgz) = 6ba15ed12217b726d9129db8e942a048
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/xserver.tgz) = 05aac7e617e0cac05a88062fef29bb459da7a1d1
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/xserver.tgz) = 19fb5af5d6afd78839a22e5c0694a492047b5200c569cbf58c676ff32ec250b21edcfe2ed00b65092fdac374d1ab9eaaed6113c69bc03009cf458aedc6014913
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/xserver.tgz) = 774d2a1072b8d122790ad189a6d2ada2ad99e7cf
MD5 (NetBSD-6.0/mvmeppc/binary/sets/xserver.tgz) = 3daf1477cadf3593e9ba45e3703fe8a5
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/kern-GENERIC.tgz) = cb6794f87c6c90ffe561a49bb2b3b48a5dacb40a
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/kern-GENERIC.tgz) = 6ee4daa27b1f69c1df2bcfb09ba9efd43a3ffd07e7de964e8f6a59ab46e0f36681a410527be024a58dff6f2efcb8f7a5af5da857fc36cca6d6bb8b71a3fd7c4e
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/kern-GENERIC.tgz) = 809b466fff3437d0e9ef68e9380b08145900b9cb
MD5 (NetBSD-6.0/mvmeppc/binary/sets/kern-GENERIC.tgz) = 21a47714184c5a998e528ceac6d8acb3
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/base.tgz) = cd5e92a746545a809e765ded7acbaac7de15f633
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/base.tgz) = ae16a9e58ccadf19e57966b608b0230cdeb066f65997cf36749dd72597ad80741f9a5a8f6f9b49e7ff42fe071c07fbdcc237e076ffd7a9fc4cfe94e9aecae6ef
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/base.tgz) = e8ce9e46b09fea354214094c65f085f963f9fabb
MD5 (NetBSD-6.0/mvmeppc/binary/sets/base.tgz) = 3480b07c4c07ba28b82c27eb4e566b9b
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/comp.tgz) = f8a50dda9bf86a613d543daad62ea5aeb0455af8
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/comp.tgz) = 5a042383539c5fa72acca742a572cc185f28e2e82f9b0a05af00a57ba3ef220272f4221718cb05322b6010236af0a29254afb0e7716a7bc1543a72bdb7ab2990
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/comp.tgz) = e13123ebdbe098433cfe3c82ff350ccf23206004
MD5 (NetBSD-6.0/mvmeppc/binary/sets/comp.tgz) = 93a0b66d03af7b52d9e2394c7fcc93b4
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/etc.tgz) = 7295ed78597062a35a244940465c2148ba96ccb1
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/etc.tgz) = 28ab60b5de702a8c4369ef7a04fd88d92a33eb79c3537c7593f1e2c3b16aecfdfb03582daa36dc97d1ab147d0a217a0354906475dae397970b991e39d88905b1
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/etc.tgz) = 814c4c6297cc41c8961e92e042c1ceea8d0f7bf5
MD5 (NetBSD-6.0/mvmeppc/binary/sets/etc.tgz) = 9b00467e487f1dc4c8e6e520f531e48b
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/modules.tgz) = 07df9e6517dc7ab1f7b4d89e4c69a0f6ed5e2551
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/modules.tgz) = bdb4958d494d5cb163ab0c77759fcf2c7c826e4f026d25fd156a19c44dd337064b604d9e5fcfd2bf60140be482cdfce94ce6d382911b813291ba2bf5a0c424cc
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/modules.tgz) = 43473d4d0defd1cff47ee9f1874161538a4b82ac
MD5 (NetBSD-6.0/mvmeppc/binary/sets/modules.tgz) = b73307a25c9030b40171a0360293a564
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/tests.tgz) = 9b7fcd1db26e218ce61b5325966cbc8fdf647ec7
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/tests.tgz) = c18be2b0c91b327f620d40c6ee7890dbff28fb4a431f2fec6a870c7542e08a0545c256547371cb2039a8707ae5d5d0a5b876ab209f7ea4f2ec66e4d5f676f05b
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/tests.tgz) = 78c4ec64a4a643c239af74a89cc617bd5e959fa2
MD5 (NetBSD-6.0/mvmeppc/binary/sets/tests.tgz) = d6e2627f54fc989e87ba95e4243e933e
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/text.tgz) = 0ef97a5332a5c76de8329dd7dd9d97fe8528a83b
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/text.tgz) = b75d8259b2a0886380f796340d08c4f56ab60002e02c04b09665e87e14eed8cddd86ae483c12e236e7589291f579a516dd049be349fd3af7c63f245cd8ddc0a4
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/text.tgz) = eaad1bf719adc5b5e1645c5fc77be97c1c946ddf
MD5 (NetBSD-6.0/mvmeppc/binary/sets/text.tgz) = 22b500a7593d8b412fc9c000558444a3
SHA1 (NetBSD-6.0/mvmeppc/binary/sets/xetc.tgz) = 98938e35cc38b7007b05b049e42e888ec0a93860
SHA512 (NetBSD-6.0/mvmeppc/binary/sets/xetc.tgz) = 0e9027a638278b6102fab31d408f258e8798bfffc6400e99b434656b81bf13053605caa8c3e8d0744685d359261fd8576c267f5eeb987da4c3a054aaf17bb993
RMD160 (NetBSD-6.0/mvmeppc/binary/sets/xetc.tgz) = f71d30b8062059ead3243e21c732ebca57eb117d
MD5 (NetBSD-6.0/mvmeppc/binary/sets/xetc.tgz) = e09a8d3ed7015c6b274ef5a025041fa2
SHA1 (NetBSD-6.0/netwinder/binary/kernel/netbsd-GENERIC.gz) = e5f0ce4b52ffef57f8ecc29a7c26ddb9181a5c59
SHA512 (NetBSD-6.0/netwinder/binary/kernel/netbsd-GENERIC.gz) = e1d81a5be8964559be0b23b0d6a4ca22f95dac64abf4a1ebc7912cbab18f47ee31ab954d85155feb750926267b7b886c1eead88223b16a4c4a363f8286d5110a
RMD160 (NetBSD-6.0/netwinder/binary/kernel/netbsd-GENERIC.gz) = e270203b95b8a0f7736779454b07957e20ccdbbf
MD5 (NetBSD-6.0/netwinder/binary/kernel/netbsd-GENERIC.gz) = 18ec5abd22e842bdc99a66e283e07231
SHA1 (NetBSD-6.0/netwinder/binary/sets/games.tgz) = e1d2bb091d8202962133475bc83edb55230b9008
SHA512 (NetBSD-6.0/netwinder/binary/sets/games.tgz) = a147fe6d0419d53d5a428802d0bb6c9fa8a84a76c96d7d6d7c0d399e5af3a69240116463549a83d53387b0214ba0207796d7c10228d92a684fe7ff0ece3aec16
RMD160 (NetBSD-6.0/netwinder/binary/sets/games.tgz) = 2be515c495548ba5135ff42a75a8ebbfc35ea776
MD5 (NetBSD-6.0/netwinder/binary/sets/games.tgz) = 9059a6f2cd41cb58827709c9e9a1649b
SHA1 (NetBSD-6.0/netwinder/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/netwinder/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/netwinder/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/netwinder/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/netwinder/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/netwinder/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/netwinder/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/netwinder/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/netwinder/binary/sets/kern-GENERIC.tgz) = 3f337b136bc35343128f53eabc787cf5a5a5a6a4
SHA512 (NetBSD-6.0/netwinder/binary/sets/kern-GENERIC.tgz) = 2243a1cbdf776e683d5cb32b411adddd35e137dec2bea0e9a2d9df17972a8ebea0302328171c8fdd06da240689991c623b07a953cb8e374ace3607c0bbb6c5f7
RMD160 (NetBSD-6.0/netwinder/binary/sets/kern-GENERIC.tgz) = bb3ef1387ed7dcee67af89f3662f2eb39a23ee7b
MD5 (NetBSD-6.0/netwinder/binary/sets/kern-GENERIC.tgz) = b370234d8f18c218bc78739602325c44
SHA1 (NetBSD-6.0/netwinder/binary/sets/base.tgz) = feddaeaf51113aca08fb9dd5689745efcd3ea178
SHA512 (NetBSD-6.0/netwinder/binary/sets/base.tgz) = 1fc5531cf2b47451e0eff63df4222a4e88f7f0112d4f0b2e43009b39dd98a0550f015833d22fb0d7c0ffa9b699944fb72283715f51e5bdae97d263e167b787b4
RMD160 (NetBSD-6.0/netwinder/binary/sets/base.tgz) = 4e1e7dcd563d7e68d0cfa8824cfa25bffd17187b
MD5 (NetBSD-6.0/netwinder/binary/sets/base.tgz) = b5f660d9a7fe6f6fa951a3a7c3c0f6a5
SHA1 (NetBSD-6.0/netwinder/binary/sets/comp.tgz) = 06b1e34718e894be491327ee15f5464e91aaac32
SHA512 (NetBSD-6.0/netwinder/binary/sets/comp.tgz) = 3f6ea371d7852951a374855e3ed4396b683b752aabe0d47e0e6b1d7243b9468023ec6893076a6d77861ccc6f8ff74957bdb4675adc878fcfa620dda04f13d02b
RMD160 (NetBSD-6.0/netwinder/binary/sets/comp.tgz) = 21fb07ace630d66e088ca8fba7f3de545bf75b0c
MD5 (NetBSD-6.0/netwinder/binary/sets/comp.tgz) = 0b21dce21957ca73da5de9ebbeb3a157
SHA1 (NetBSD-6.0/netwinder/binary/sets/etc.tgz) = 6eb3f0d4ee4b86c87bd03cc4f93958a42e9079f3
SHA512 (NetBSD-6.0/netwinder/binary/sets/etc.tgz) = 2de4303f21bd502d062486b8789198bbf3e7f5068888d62d4da08e5e51d58f42ede6ea13c8a83bfbc9d0cc64cc6e11d3f2e949a3c2ad800aa78680d4046aa7d1
RMD160 (NetBSD-6.0/netwinder/binary/sets/etc.tgz) = 3a77fb3f3a4ea25db5b8b450274bade4192541c8
MD5 (NetBSD-6.0/netwinder/binary/sets/etc.tgz) = f57f45618396a67a169ab7c0e74be2d5
SHA1 (NetBSD-6.0/netwinder/binary/sets/modules.tgz) = 780745396f7dee2fcce36e4c401f9462ee620e55
SHA512 (NetBSD-6.0/netwinder/binary/sets/modules.tgz) = 9e4ffad337fd2e30a157ec7ade8f868043addcc2e8020a950cfd8e515d17bccae185b04ff15dfccebce764c1e0a8f1f722bac33e385ae294700dac0b0c3a5792
RMD160 (NetBSD-6.0/netwinder/binary/sets/modules.tgz) = af686e341bf03f36fc43b5bfa86015e3204803f9
MD5 (NetBSD-6.0/netwinder/binary/sets/modules.tgz) = eb5745d3d6289ded145654a58bf4a9e0
SHA1 (NetBSD-6.0/netwinder/binary/sets/tests.tgz) = 46a999d0afcee9103c765b57765971557cffb18d
SHA512 (NetBSD-6.0/netwinder/binary/sets/tests.tgz) = 0937f49f33fd690da609dff62adf04978e1ca36e61e3357e360e3ca622e406758a12e5909fa0aa431f914eb145625cc7dc3dc1caf09bf2793feec7db45cba00a
RMD160 (NetBSD-6.0/netwinder/binary/sets/tests.tgz) = 7d070858cc4bdbbee9bd6568acf73cc9b70147b0
MD5 (NetBSD-6.0/netwinder/binary/sets/tests.tgz) = 879591282409cebe2b0c397f9c05f417
SHA1 (NetBSD-6.0/netwinder/binary/sets/text.tgz) = bc73c2b37be55d902c10db2244412f884dcb2c49
SHA512 (NetBSD-6.0/netwinder/binary/sets/text.tgz) = cbf470fcdd95eb9bf94d5296d7d728eb537a243ed03448e134e56c6714c6528968d217c69e3ac830045f2b8b5c8c4df451fd24ff2b5b176d37fcd84ca1f07f92
RMD160 (NetBSD-6.0/netwinder/binary/sets/text.tgz) = 00aeff72db0b05b48fa40bf80b1e24c5a667fd0f
MD5 (NetBSD-6.0/netwinder/binary/sets/text.tgz) = 42470e4348951037fd056476c26dbb77
SHA1 (NetBSD-6.0/netwinder/binary/sets/xbase.tgz) = 66d3e62a116c56bfaac8ee3257ba675a845b9ab3
SHA512 (NetBSD-6.0/netwinder/binary/sets/xbase.tgz) = bfc8c00cfc6bf2fcb3d26ff9abb9e6091c5389aeac5370ae475b3c74569e3ccf05494e363544d6c805820e2c7554a1fea936abab6e3212dd4710e2463625006a
RMD160 (NetBSD-6.0/netwinder/binary/sets/xbase.tgz) = 26ff99801c9a84190cca4292c137c17305bec08c
MD5 (NetBSD-6.0/netwinder/binary/sets/xbase.tgz) = 532381165745c160613ef72bcd7e7848
SHA1 (NetBSD-6.0/netwinder/binary/sets/xcomp.tgz) = 0eed09b5c7a6e8d76c99ddaa31029f414a14d6bd
SHA512 (NetBSD-6.0/netwinder/binary/sets/xcomp.tgz) = b03064377958f9ca9252e83d2933574a5f0fedc89ac864641340f8e50a6d5fa1a074d45a80f0dcb1567ef7d23eb2bc4bec3de866c2d07912ab4122acaf062c35
RMD160 (NetBSD-6.0/netwinder/binary/sets/xcomp.tgz) = f7a208a7ac2137d4940ec2f78c0658e630ea329e
MD5 (NetBSD-6.0/netwinder/binary/sets/xcomp.tgz) = fa248543b87b553473cc67fd45e385ae
SHA1 (NetBSD-6.0/netwinder/binary/sets/xetc.tgz) = 4e2fdc95511cd90317da41fb941f7ec6f622b84e
SHA512 (NetBSD-6.0/netwinder/binary/sets/xetc.tgz) = 57a60f983fadbdb0a0ef5720c2bda405d4052d460f67976e71a34ad016d25357cd04fbdf48e866c3641dcabc55194a683931a0b36c7f55549a7998c31e42ef88
RMD160 (NetBSD-6.0/netwinder/binary/sets/xetc.tgz) = fef501d07dce370292072f65ed25dd32df41bfcb
MD5 (NetBSD-6.0/netwinder/binary/sets/xetc.tgz) = efdbfcb8e6fbf0c4cb2094e749c54b1a
SHA1 (NetBSD-6.0/netwinder/binary/sets/xfont.tgz) = b92ebefbcc4a96bdbba5540df3119bed0ee2efdf
SHA512 (NetBSD-6.0/netwinder/binary/sets/xfont.tgz) = 605e099b75b0133d811b12387461856bcf7ef7cc0dddc68490eb6ef51ba284ac2edafcc757feb0665618b119eacd2509c302f42736eb0d582ab7f17f5fdb1b23
RMD160 (NetBSD-6.0/netwinder/binary/sets/xfont.tgz) = 084a5e8e5f95debf53ba5be9e52230fe72dbb2d0
MD5 (NetBSD-6.0/netwinder/binary/sets/xfont.tgz) = 02b5344f23cdb9eb4d9cbe8b30c7707e
SHA1 (NetBSD-6.0/netwinder/binary/sets/xserver.tgz) = 570bbfb56fda998c07f8414731eaf79af5b01bd1
SHA512 (NetBSD-6.0/netwinder/binary/sets/xserver.tgz) = b70eccc78ca3e290869502760fcc8a8378db3d49c402f075e2f12e1453a7fbb68911df18993c0966ce840b837225d577344ec7088d2aea5eac78130e9dcce0ec
RMD160 (NetBSD-6.0/netwinder/binary/sets/xserver.tgz) = 5b7c5740805aebc0ea76f778b429803b25cf9128
MD5 (NetBSD-6.0/netwinder/binary/sets/xserver.tgz) = d866ffe531725f8c18bd3f88129d09db
SHA1 (NetBSD-6.0/news68k/binary/kernel/netbsd-GENERIC_TINY.gz) = c260c1b8e7017de3a9c7383ebad6833c172b745c
SHA512 (NetBSD-6.0/news68k/binary/kernel/netbsd-GENERIC_TINY.gz) = da338bee7bcaf58401ea3fa629c91fc5732016581d869f2bb8897f252c90e296773eaf35dae48667a753e7119340ac376fff713f3f87e25802fb3874879fcc65
RMD160 (NetBSD-6.0/news68k/binary/kernel/netbsd-GENERIC_TINY.gz) = 25e70f61645eea330feda79fa0e6302bedeecdc3
MD5 (NetBSD-6.0/news68k/binary/kernel/netbsd-GENERIC_TINY.gz) = df5ca7bb950327f08e460757a4b03de4
SHA1 (NetBSD-6.0/news68k/binary/kernel/netbsd-GENERIC.gz) = 96c08009db46bd9bf3e0b1b0b8ea6b96b7772396
SHA512 (NetBSD-6.0/news68k/binary/kernel/netbsd-GENERIC.gz) = 635ef8e6ee4c767afba768a982f587d6040afeb15585be8a8fcfd44e75bcd9031f8965f1bef2b7fdb955b81b2e09f0dc6a51618b48827898763cd75310669a6b
RMD160 (NetBSD-6.0/news68k/binary/kernel/netbsd-GENERIC.gz) = b9a864c14cc967d2336473c2c2c19eeeefa5b3ec
MD5 (NetBSD-6.0/news68k/binary/kernel/netbsd-GENERIC.gz) = 76b1a1ad9865766e5d7b1ceb1d374f26
SHA1 (NetBSD-6.0/news68k/binary/kernel/netbsd-INSTALL.symbols.gz) = 620ddb95366fb58c53d115681a205d73fb79bf02
SHA512 (NetBSD-6.0/news68k/binary/kernel/netbsd-INSTALL.symbols.gz) = 4eb494f02f39ee17205b28f07993fd1c9bbb872f2617aacdb173f9b458603fe284dedcde279eb0b9ffe083b01f485dc04963ec09117c31927e3e97a9b5441f6f
RMD160 (NetBSD-6.0/news68k/binary/kernel/netbsd-INSTALL.symbols.gz) = 3456f6c84edd59e0e1db531ec40b838fdd076d14
MD5 (NetBSD-6.0/news68k/binary/kernel/netbsd-INSTALL.symbols.gz) = 0898606b528a29a5c2c754d365e1fb22
SHA1 (NetBSD-6.0/news68k/binary/kernel/netbsd-INSTALL.gz) = 76b25bf6b038d7c48a55bf1f1df350717707af3f
SHA512 (NetBSD-6.0/news68k/binary/kernel/netbsd-INSTALL.gz) = 7aadf02eb4862fc3cf8a0099673414092e8a113e59dfcfda713e430422cc8f03fd4a711ec767b10905ac5feeb6823054e372da6da4f19e15c595712a75dff046
RMD160 (NetBSD-6.0/news68k/binary/kernel/netbsd-INSTALL.gz) = 4e01f28fc0dd67b8a2b16ac61c27adf4ec83c7e7
MD5 (NetBSD-6.0/news68k/binary/kernel/netbsd-INSTALL.gz) = 7ecaba69661efb53aea818bfc0df2870
SHA1 (NetBSD-6.0/news68k/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/news68k/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/news68k/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/news68k/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/news68k/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/news68k/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/news68k/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/news68k/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/news68k/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/news68k/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/news68k/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/news68k/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/news68k/binary/sets/xserver.tgz) = 1d20fd039546a66b428b854c659ea5e2ee9f8114
SHA512 (NetBSD-6.0/news68k/binary/sets/xserver.tgz) = 091d09049d0af5555feec02065234e281677aeb3ecbab5a19413d924a8b58dc7f64401ef68e68bc52d8b6eeb73e091d351c45805374b8bf276b0181415968561
RMD160 (NetBSD-6.0/news68k/binary/sets/xserver.tgz) = feef5d52278d23637d7b8ba77639bfd96918e90e
MD5 (NetBSD-6.0/news68k/binary/sets/xserver.tgz) = b2bd6057c374376388766fc8667eb9a9
SHA1 (NetBSD-6.0/news68k/binary/sets/kern-GENERIC.tgz) = 6d14a82918f258c781499c87f7ef23fca3c85c68
SHA512 (NetBSD-6.0/news68k/binary/sets/kern-GENERIC.tgz) = ed399e3bab46335dc4d2bb1f2971692d8a1d8aa22c4196bb340e005c951a3941517d64de8be8c8047a60de791ae934197b38fe02706bcde080f4062a5ffbf431
RMD160 (NetBSD-6.0/news68k/binary/sets/kern-GENERIC.tgz) = 74c42f841cfb865522b4eda052f46a7b13e491d0
MD5 (NetBSD-6.0/news68k/binary/sets/kern-GENERIC.tgz) = 6ac0ec8267f457a0d94feba55a094773
SHA1 (NetBSD-6.0/news68k/binary/sets/base.tgz) = 14763cf9124eb67c9440727bf0559d98d9aa4765
SHA512 (NetBSD-6.0/news68k/binary/sets/base.tgz) = 24bf851943de08a6e5cb869a69d8d4350f6d6c64988741dbb8691afb927fa8ccb922a7e608da48ef52913796c0e6172c8302b22248c64ce576446d26f26add71
RMD160 (NetBSD-6.0/news68k/binary/sets/base.tgz) = fc4ad357c41d200d8e12444d1f4e13956521a4d5
MD5 (NetBSD-6.0/news68k/binary/sets/base.tgz) = 5401468976f41209a9a28889d85f52be
SHA1 (NetBSD-6.0/news68k/binary/sets/comp.tgz) = 6b20fee4bcb085b5b09893416ce5eee6a96ca5bd
SHA512 (NetBSD-6.0/news68k/binary/sets/comp.tgz) = 3550045bb700ea29604590e25cd9e6305459476a837f67b4d746e62ce0facc8218aca1d03f2dd570e2590802f32079e438ad4e977a96f9a8939dd986d725d898
RMD160 (NetBSD-6.0/news68k/binary/sets/comp.tgz) = 85fb356111807dbc811b8fe2fb3d294e4544b358
MD5 (NetBSD-6.0/news68k/binary/sets/comp.tgz) = 223302daf2a8f7f52863763374438b3e
SHA1 (NetBSD-6.0/news68k/binary/sets/etc.tgz) = 575f76fc225f509cec73dbc1e6a213d4370c221f
SHA512 (NetBSD-6.0/news68k/binary/sets/etc.tgz) = 4705c91993153a30b1bf6bab9f1d0b75d9bc29ccbd698d26c44ae70fadeb4a8390298eb1e0e590f0da62748d9429e8e5927558a31d2efe0ef068a9ca353681b4
RMD160 (NetBSD-6.0/news68k/binary/sets/etc.tgz) = 7ce6e09c062f61b659f2c16138516b3629c010bf
MD5 (NetBSD-6.0/news68k/binary/sets/etc.tgz) = 234be1ba6587866294eb09b75439840b
SHA1 (NetBSD-6.0/news68k/binary/sets/kern-GENERIC_TINY.tgz) = 119015b6a9a67300144e248352a3842857ba86e9
SHA512 (NetBSD-6.0/news68k/binary/sets/kern-GENERIC_TINY.tgz) = 2eecd5ff4fd2a1ca0edc61ff92c804f5202ffedc5b6e74c5c0be372f74cf533e4b4f1b6a69625a155a33661185e1a56943947ab1c98fa8b4dc3892526781950a
RMD160 (NetBSD-6.0/news68k/binary/sets/kern-GENERIC_TINY.tgz) = 5477d4127ff479fa19aa58a3de57bbd3a4cc0b90
MD5 (NetBSD-6.0/news68k/binary/sets/kern-GENERIC_TINY.tgz) = fd4f85a1589e4907b314ae996b3aa08a
SHA1 (NetBSD-6.0/news68k/binary/sets/modules.tgz) = 1885df3fb4ea56feab816bd8dc5e0dc08fae234b
SHA512 (NetBSD-6.0/news68k/binary/sets/modules.tgz) = beeb1e94ef6f34b7525c0c2de9bfe9927aa443bfdc5be8c4a5546a27483c0ca45006d3f546d93bc0036f25804e7827ea1b8fd943e7642bc73c4b6c75710362ed
RMD160 (NetBSD-6.0/news68k/binary/sets/modules.tgz) = e3ed23f75e48c70efbdf3b5ec9c28ec41886d878
MD5 (NetBSD-6.0/news68k/binary/sets/modules.tgz) = e4db103ff1b7f68a57e8fe813b2b8d3c
SHA1 (NetBSD-6.0/news68k/binary/sets/tests.tgz) = 984686c1f0ad93ff8200e810482527b87bad3de3
SHA512 (NetBSD-6.0/news68k/binary/sets/tests.tgz) = 508704556a3bc66c9f5f88a6377d34d81edf9aaace78a4496c550021dd5a88c2ff85d5663ff2aa42c4e7a74721249b2dddabd50bf597572b016794c42d22a622
RMD160 (NetBSD-6.0/news68k/binary/sets/tests.tgz) = 06cb486210f5e4ef2ec33d863cf995780caa24df
MD5 (NetBSD-6.0/news68k/binary/sets/tests.tgz) = 60053afffd0ad1c95c2654ce6a991d21
SHA1 (NetBSD-6.0/news68k/binary/sets/text.tgz) = edd7603519c0dbdddbe9b9c8d9ddf066d62ee617
SHA512 (NetBSD-6.0/news68k/binary/sets/text.tgz) = b5c7ca241708d02df4df5a6bf43b6329cdb063a922bdf23955b3807cf813d10718cdd1a7a4c3167e0869d5f8f0a742ee275951a72fa2cbfbd32a0afa37d3e49e
RMD160 (NetBSD-6.0/news68k/binary/sets/text.tgz) = c54012faa763171bbd263ba23d4d2495e7cf2762
MD5 (NetBSD-6.0/news68k/binary/sets/text.tgz) = 2088a3dfd39fd763db42fe375a460a3e
SHA1 (NetBSD-6.0/news68k/binary/sets/xbase.tgz) = 9b8940f65ca6a841b84406a22a13971665edc464
SHA512 (NetBSD-6.0/news68k/binary/sets/xbase.tgz) = 4a7d78ad9cf1b84c183fc731bac2fce861f94bc26c7a1681ab1d62a9b761ba89538469688b24588f6c671503a1b66a1e521c2eeef2fd528d21dc0351272e56e8
RMD160 (NetBSD-6.0/news68k/binary/sets/xbase.tgz) = 85735e089a6ac5293e7438a609dd11105cd4eb25
MD5 (NetBSD-6.0/news68k/binary/sets/xbase.tgz) = b6f52c8fde2fb1a5372a12d2f321e9c5
SHA1 (NetBSD-6.0/news68k/binary/sets/xcomp.tgz) = 63a79437c028936698509a386963708082356d29
SHA512 (NetBSD-6.0/news68k/binary/sets/xcomp.tgz) = 25791832de667b9a44f341f7267f09aaa6b9a4216452ceaab8b56dc6ea5fca3be2da53868b624f47a275edf19ada59dfed40e355e2ff1f62ab96bb54448c1e00
RMD160 (NetBSD-6.0/news68k/binary/sets/xcomp.tgz) = e681ae34bd83be8dbb6945cef3a129bc1538b66a
MD5 (NetBSD-6.0/news68k/binary/sets/xcomp.tgz) = c87752020f2ad29a80262eae4bbeaece
SHA1 (NetBSD-6.0/news68k/binary/sets/xetc.tgz) = 76661bc6d6851cb2ae57f3a7f25f816a33360a2b
SHA512 (NetBSD-6.0/news68k/binary/sets/xetc.tgz) = a1994fbd97492275e9128c5fc764dc7cf5539bee26bb9d13d9f3fcc14429858e794ed8a10554570c4f70b87714f8b0ab23269ebbaf44d4fb95d0901dcc28fe81
RMD160 (NetBSD-6.0/news68k/binary/sets/xetc.tgz) = f9f8fc83764439a533110c614e1ef19ffe98361a
MD5 (NetBSD-6.0/news68k/binary/sets/xetc.tgz) = ab945fe1f43ea47433a047babda53a81
SHA1 (NetBSD-6.0/news68k/binary/sets/xfont.tgz) = 4403bc158b92ded3d27646c6c0858aa6c117cb8d
SHA512 (NetBSD-6.0/news68k/binary/sets/xfont.tgz) = 79e31eee0397803ec17a9413b50a5d501fbc3515e4080a0e9e053498c64557809cddfd5718f131d3c451a231c4a38ff4f40b50ef401af5c204a68b61df8f8ba7
RMD160 (NetBSD-6.0/news68k/binary/sets/xfont.tgz) = b7e19b89a1f67382428aeaf248710dade3b0672e
MD5 (NetBSD-6.0/news68k/binary/sets/xfont.tgz) = 8a1f532d7a481f797deb226b66d15ceb
SHA1 (NetBSD-6.0/news68k/installation/floppy/boot.fs) = baea772f5bf7116b8c489a2e75f77c98e308ac5f
SHA512 (NetBSD-6.0/news68k/installation/floppy/boot.fs) = be4dbd6dc24f56ac71f96d60b8ed0e6b4f725bae2b9d39ef1b9e5ea5fe71053f5fea04979676ff13a50685bcbce539bbc93631408c56d29a2924c79370ded347
RMD160 (NetBSD-6.0/news68k/installation/floppy/boot.fs) = ff9f3a85babe4d597e2ccb4a864eb0c611a7894f
MD5 (NetBSD-6.0/news68k/installation/floppy/boot.fs) = 98f566671f6dd720fb61e8c657bd93e8
SHA1 (NetBSD-6.0/news68k/INSTALL.html) = 8d9ab316296bfbb751b0a05557f4648ff2990af0
SHA512 (NetBSD-6.0/news68k/INSTALL.html) = 5916c04ef717b2e47ee36f93a1e3a43a27215b5c688b36520e0f160bda3816d6e4645e2e9e922d564260e8b8a60e46da67fbd3b443832032a66e76153596c4eb
RMD160 (NetBSD-6.0/news68k/INSTALL.html) = a07fa866614d3f5ecfb5642eeb354b6b9c675318
MD5 (NetBSD-6.0/news68k/INSTALL.html) = 89f43b16d907873c28b8b7b6ab8cb4a9
SHA1 (NetBSD-6.0/news68k/INSTALL.more) = db7d9b68680f5c20b525bf253e342f2775d04a71
SHA512 (NetBSD-6.0/news68k/INSTALL.more) = ab50b18effb102ac017c0cad2c054dffe8ccec3876ff9cc6a0353762b5a2b11651296f05b3cf3da76cbb10a4561bc529b5cf113141c742a96fdb73a5358e129c
RMD160 (NetBSD-6.0/news68k/INSTALL.more) = d6aacc53d5c3384704e2ff1f64a4eb9fd474effe
MD5 (NetBSD-6.0/news68k/INSTALL.more) = 4c22b00a52d8ad3cdb12f66d23a069a9
SHA1 (NetBSD-6.0/news68k/INSTALL.ps) = 021d67431fbc7f9fd378e496c25c28097b72af46
SHA512 (NetBSD-6.0/news68k/INSTALL.ps) = 224e78f8ddc0149a710511083c55a69e1eb2f37089ad15ce1fbc1c07733055eb5b34ad4c32b2803d1f3050b1c5a038ef1d429af780c0a57c8a4b41fde8d10413
RMD160 (NetBSD-6.0/news68k/INSTALL.ps) = 86b953fd3860caadf7a681e0d75486cb2ac453d2
MD5 (NetBSD-6.0/news68k/INSTALL.ps) = 4f2098f135e2d4f945dd6365a55b7bc2
SHA1 (NetBSD-6.0/news68k/INSTALL.txt) = 1491600e777e5a7a96cc05a7fc5a0ab0edf2072f
SHA512 (NetBSD-6.0/news68k/INSTALL.txt) = 0552976751403a424febde147bdbf34b7e589c0f27225072c9691e30f630e1f49be93c9e75ebbc7d0c6d06ab78419506d4c4313059023e9ae13dbd8d1f5c4445
RMD160 (NetBSD-6.0/news68k/INSTALL.txt) = 2c6df35393c4ee9d9fb7dcb58002f996c587c32a
MD5 (NetBSD-6.0/news68k/INSTALL.txt) = 9cff69046f721931b5b5ef4bd511d12c
SHA1 (NetBSD-6.0/newsmips/binary/kernel/netbsd-INSTALL.symbols.gz) = 0e0add1125bb456d3210efa5035169e1bd7ae98a
SHA512 (NetBSD-6.0/newsmips/binary/kernel/netbsd-INSTALL.symbols.gz) = 138e28fb744019ee2e89cb8ef1f3e23b5fd296b2f8c12a26245cef11aef9540f12eb2c180e5847d047d74d593e62916694c7f031fda0e3477e42ecdb79da7510
RMD160 (NetBSD-6.0/newsmips/binary/kernel/netbsd-INSTALL.symbols.gz) = 4ee95362c6f9e38a32cd9e0fe36cdc3aadb91291
MD5 (NetBSD-6.0/newsmips/binary/kernel/netbsd-INSTALL.symbols.gz) = 0d8837629ba02a89e50b6764649d03a2
SHA1 (NetBSD-6.0/newsmips/binary/kernel/netbsd-GENERIC.gz) = decd427b7283d775b0356186a50720409c1c2ecf
SHA512 (NetBSD-6.0/newsmips/binary/kernel/netbsd-GENERIC.gz) = 05df61671fb365dba6ee67d097f5e51b372720a260b0dd05913a4cf283f323836aa7b157d88eaa68e4cdfa2a7180e57e3ba8ee3cd538dffe4a6afadc1ce880c9
RMD160 (NetBSD-6.0/newsmips/binary/kernel/netbsd-GENERIC.gz) = e997c6efa2aacfb924670a53162157c8fbd93bdd
MD5 (NetBSD-6.0/newsmips/binary/kernel/netbsd-GENERIC.gz) = 1b105c1608405606b59942471102880a
SHA1 (NetBSD-6.0/newsmips/binary/kernel/netbsd-INSTALL.gz) = 42aeaa1b3dd454b7ac007db9ac4f9b43ca2bd52f
SHA512 (NetBSD-6.0/newsmips/binary/kernel/netbsd-INSTALL.gz) = f4262b07a27e2166082ed9cd4b0a247fcbd65f3aebc1c1f8e502f5fc931e130627b0906c92263d2f748d3d30d482391c6b72637c29812f98aaed2fd85830cfdf
RMD160 (NetBSD-6.0/newsmips/binary/kernel/netbsd-INSTALL.gz) = ab7e5e64d2b4761006e95f8d1605d767f6ace146
MD5 (NetBSD-6.0/newsmips/binary/kernel/netbsd-INSTALL.gz) = e23868785597388ddbae56348a08d9d7
SHA1 (NetBSD-6.0/newsmips/binary/sets/games.tgz) = f56d3e5181634f6a3d7908dcb002ebcb1c5aaff0
SHA512 (NetBSD-6.0/newsmips/binary/sets/games.tgz) = 0cff50ea13e34fe338edf70da6a28af061580feb4cb0773828adefa631ebd471a96568a3b30ea0f13f3bb116856a1b6aad8a5a52443346f52e4907f4caabc5f1
RMD160 (NetBSD-6.0/newsmips/binary/sets/games.tgz) = 1538910578dbb8eee2931200fac0ccda44758e5f
MD5 (NetBSD-6.0/newsmips/binary/sets/games.tgz) = 2bd5bd0ce72df5cf92767b8c3f7cc208
SHA1 (NetBSD-6.0/newsmips/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/newsmips/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/newsmips/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/newsmips/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/newsmips/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/newsmips/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/newsmips/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/newsmips/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/newsmips/binary/sets/kern-GENERIC.tgz) = 7b98d2f4f3acbf629e953e6b694d8f4755b8e8bd
SHA512 (NetBSD-6.0/newsmips/binary/sets/kern-GENERIC.tgz) = 5d7e2e9bce3611aee142686bab7a69abc59ddb5e1308694cc88e63645d3b411013765ffa5198cbb44fa445ae258f85cfae976c239875ea26ac4d0439cb497746
RMD160 (NetBSD-6.0/newsmips/binary/sets/kern-GENERIC.tgz) = 849d144d0090f36a2d013a9f85c9b4c472c595df
MD5 (NetBSD-6.0/newsmips/binary/sets/kern-GENERIC.tgz) = 40ecc8186e3d4328e24ef91e817c5a4e
SHA1 (NetBSD-6.0/newsmips/binary/sets/base.tgz) = c229edbeb3438aa2444933af1633c73b66f23748
SHA512 (NetBSD-6.0/newsmips/binary/sets/base.tgz) = 8ce8dce68f07ee529d0f079794502bd225475cabe7f0dd351caf01b5acf8a34d5f865eea046c06998db53a2cb23c7c36764314493d537ecad8e69c5908a07dbe
RMD160 (NetBSD-6.0/newsmips/binary/sets/base.tgz) = 6d2365c6adbf64a1b6655c7341a7fa1756878e93
MD5 (NetBSD-6.0/newsmips/binary/sets/base.tgz) = 650ad75dc13c5edd48f4d8a58759375e
SHA1 (NetBSD-6.0/newsmips/binary/sets/comp.tgz) = d6ce335f1f52b3d77282e7c08c3781202018257f
SHA512 (NetBSD-6.0/newsmips/binary/sets/comp.tgz) = 4c8cac5f9fff3b5334083d5df788b0967f62817e797206393f6b83a9cbb0b78c151408440045b92bc50ac2540e0b5d38ea854f29b6af69c1043952d472cf0a85
RMD160 (NetBSD-6.0/newsmips/binary/sets/comp.tgz) = 7985588ccbaf837e8fd044a0ea7064aa023245c4
MD5 (NetBSD-6.0/newsmips/binary/sets/comp.tgz) = ac02ccf4b35b94d87960214974f909ba
SHA1 (NetBSD-6.0/newsmips/binary/sets/etc.tgz) = 1744b2b2050df7622a70aec4c4ab39ff4766b598
SHA512 (NetBSD-6.0/newsmips/binary/sets/etc.tgz) = 933f4b9af8a1f5033c88ecb680c70b258b36267cbd3d8804aa7d2c0ce797f272a4ffc7266bb369505635e0e0ce0efef285d8fa26704ba28690f1dc6db810c42e
RMD160 (NetBSD-6.0/newsmips/binary/sets/etc.tgz) = c41f76161f0a9ada7ee8cad68d5311f70051c08e
MD5 (NetBSD-6.0/newsmips/binary/sets/etc.tgz) = adb5f50f9862a06a123d1d848dbeaef0
SHA1 (NetBSD-6.0/newsmips/binary/sets/modules.tgz) = 494b6a3b255bace06cfee70f87d2ac213b190a5f
SHA512 (NetBSD-6.0/newsmips/binary/sets/modules.tgz) = c8274f58ae9d514e72f8d82e46599c655e1b959b744e73340a6a106dce528f83687b7bd7f8e769eab60294e6962a7bc5e0d7173330710ee381219ca6f87b51c7
RMD160 (NetBSD-6.0/newsmips/binary/sets/modules.tgz) = 3536ddd91df0d4d53e6da53bf30548ac60062349
MD5 (NetBSD-6.0/newsmips/binary/sets/modules.tgz) = 4cefc39360426c683af0c3031403bd54
SHA1 (NetBSD-6.0/newsmips/binary/sets/tests.tgz) = 8c653879bccb5f7b67536fb96f92fa52cf11e69f
SHA512 (NetBSD-6.0/newsmips/binary/sets/tests.tgz) = 5cc47261708222b4fe1ca43d1dc014909df674db2c477c9d814e2af319e2c21c66555df78786ec48c08b21c4c8c1ddc8d815967ffaaceb625f61687df9ac46b1
RMD160 (NetBSD-6.0/newsmips/binary/sets/tests.tgz) = 8712fce841356834ff6036bdcfe58dbb391ffd4e
MD5 (NetBSD-6.0/newsmips/binary/sets/tests.tgz) = ccc445c6ba7fc49d4de4b0dcb26aefff
SHA1 (NetBSD-6.0/newsmips/binary/sets/text.tgz) = 766d9d6d34c79c7be29da59f543f6e029090ecff
SHA512 (NetBSD-6.0/newsmips/binary/sets/text.tgz) = d11a630ead4cf249224c0f887cf6c2ef347983e885f5657c44cc988733880ccfee735f36dd5558ca60ffffb5ea8dab042d016d3824d8e9116cda749516508624
RMD160 (NetBSD-6.0/newsmips/binary/sets/text.tgz) = 53cfd7b3389f1b71430daad46ff3be2c692690a3
MD5 (NetBSD-6.0/newsmips/binary/sets/text.tgz) = f0692f4de9721110ecf0e1ecac4bf1d9
SHA1 (NetBSD-6.0/newsmips/binary/sets/xbase.tgz) = 19fb5f83a2ef362388d812fbe3c4ccad858dd191
SHA512 (NetBSD-6.0/newsmips/binary/sets/xbase.tgz) = 4774d680e11097be0f6d8b9cdd2a296cfceefdf1c19d0e8188c742efb2603b253b46c2d01084c6bd5470e34ecc6dd81f502a97c9c69e02b9aca1af80f9d3bafb
RMD160 (NetBSD-6.0/newsmips/binary/sets/xbase.tgz) = b52fbd33962b06618a8334439f89a7114cb762a5
MD5 (NetBSD-6.0/newsmips/binary/sets/xbase.tgz) = d9c670a2392dd2f58f041b18ecd32be6
SHA1 (NetBSD-6.0/newsmips/binary/sets/xcomp.tgz) = e99076dd0244130e9ae9ac14533df11f058aab3c
SHA512 (NetBSD-6.0/newsmips/binary/sets/xcomp.tgz) = 2869e7c2d397772612ce36217da0c8f7ab0ed5b18694d0620e789854dcd2fbc6763dab632614cb9df7be4c7697065db5f68e9e400fc2068a03e38d98709399b7
RMD160 (NetBSD-6.0/newsmips/binary/sets/xcomp.tgz) = af2372e9b8ba4ed8a4d8f951b9570e411be2dcf8
MD5 (NetBSD-6.0/newsmips/binary/sets/xcomp.tgz) = cb05a7f616ecaa766d696b7ab5d9120e
SHA1 (NetBSD-6.0/newsmips/binary/sets/xetc.tgz) = 973a0c9b770b14e156ea2d999b0c155f8675e95f
SHA512 (NetBSD-6.0/newsmips/binary/sets/xetc.tgz) = 4298ae57e88d6a5853a8832cf06c6d294051708ffaff6e2fa28efd06c2a8cf99274f5d1be0bff31248d3d9d9ee7bcb2738fcc9b5a4f55a2f34246cf5e1bd3546
RMD160 (NetBSD-6.0/newsmips/binary/sets/xetc.tgz) = 4b9023581af4b378be8ce78fa812c832dae9f682
MD5 (NetBSD-6.0/newsmips/binary/sets/xetc.tgz) = 8eee7c09a6b78148e9b1260abee79e53
SHA1 (NetBSD-6.0/newsmips/binary/sets/xfont.tgz) = 389d0d14a3b5ebbdf3f49b0c5bca3efcf7155a42
SHA512 (NetBSD-6.0/newsmips/binary/sets/xfont.tgz) = e246f6527bb91eecd3190cd9eb82157db0a0d868f95acd47bb292caea573405170156e37dfb50d3ec97b00ac7c50cf0893640945575638bd9f958cfbece4bd72
RMD160 (NetBSD-6.0/newsmips/binary/sets/xfont.tgz) = eab3db1a489e432867ce806385ed12e1b25165ff
MD5 (NetBSD-6.0/newsmips/binary/sets/xfont.tgz) = 4030cc9fd13f8d3524a8c161715edc6b
SHA1 (NetBSD-6.0/newsmips/binary/sets/xserver.tgz) = 2e0e1a1280e01a64615430396697d3a4cc4aabee
SHA512 (NetBSD-6.0/newsmips/binary/sets/xserver.tgz) = e5c9d65657227d24b52db9d109c9a4993c5962c3f8171b1353709464a0ba18c99264bf3cd8dc53478692cf2a297f36bccf15519f1aa198cf1a6392e3557914da
RMD160 (NetBSD-6.0/newsmips/binary/sets/xserver.tgz) = 82a04ae9c10abf44d00acaa3f97f99150e7dc607
MD5 (NetBSD-6.0/newsmips/binary/sets/xserver.tgz) = 9f3edc37a2f786ee70d5f0e100105971
SHA1 (NetBSD-6.0/newsmips/installation/floppy/boot1.fs) = d55e9f59aac50a141dbec353c4bc9586faa8d00d
SHA512 (NetBSD-6.0/newsmips/installation/floppy/boot1.fs) = 659476e61e29371975130565ae2e343ad3bc9423a910ef0b08cffe41e87179020c5b2f42f98d351c9957655b67ded13cf6111835e441d2ddd39584bc1a57c7ba
RMD160 (NetBSD-6.0/newsmips/installation/floppy/boot1.fs) = f3becca1f4c20eadf3e77ff4bcb84990a9e543f6
MD5 (NetBSD-6.0/newsmips/installation/floppy/boot1.fs) = 992e9d201b2a7183a6f8725486e8f2d0
SHA1 (NetBSD-6.0/newsmips/installation/floppy/boot2.fs) = 2c228b57a187bbfae507d67aae5d0faef2f308ea
SHA512 (NetBSD-6.0/newsmips/installation/floppy/boot2.fs) = 72c6b63777f2171c852f15cfd2c04fd9b21a9cfc423542d0127012be6d5ca85ad15bb33a67b1fb28e1f07f2652f94ae2d2f0c0641c6e860c0ed50d9556eacc59
RMD160 (NetBSD-6.0/newsmips/installation/floppy/boot2.fs) = 495d95dd6cdcbf0a0cdb07fb8500738e306dd970
MD5 (NetBSD-6.0/newsmips/installation/floppy/boot2.fs) = 90827190455600b1692585932670fc44
SHA1 (NetBSD-6.0/newsmips/INSTALL.html) = 3bba0333ef60ff85cbb905deb86171f67a7e7d35
SHA512 (NetBSD-6.0/newsmips/INSTALL.html) = 4778fb087bc4c44b84fec903f4e00003a41cfa3675a2f712bba28b33773800ad6b7d546074787caff68fd34a885ed84e05a0183a4867bed9c3e881ab30c68c64
RMD160 (NetBSD-6.0/newsmips/INSTALL.html) = 27d6089590736239f38e1b0638f0b1662465ada6
MD5 (NetBSD-6.0/newsmips/INSTALL.html) = 5041c29c150843c89b4dfa0c1cea869e
SHA1 (NetBSD-6.0/newsmips/INSTALL.more) = 91afcc03bfa9fd0c378f7d9d22ab81cfe214f36c
SHA512 (NetBSD-6.0/newsmips/INSTALL.more) = fc99840a5b29382c2f54287ded48994162804002b6947d2fc1fe907a0674b3827c64fa52e2bd3de735f387e09627ea254820bcd142f0552374acd2c951b0edec
RMD160 (NetBSD-6.0/newsmips/INSTALL.more) = cdb0bd00e8d126b4736b77688ca2ac9bcc330283
MD5 (NetBSD-6.0/newsmips/INSTALL.more) = 0dd24b3ef2da6fb2fd0989edf90aecd5
SHA1 (NetBSD-6.0/newsmips/INSTALL.ps) = 2e236b56c944116ec48df342a4a4ab1414be54c8
SHA512 (NetBSD-6.0/newsmips/INSTALL.ps) = 22e6e7b9975ffcc75b2560ecc103ab0f3337c0e637fc0b13e73c52b8d726c5a7676f8eb60a931113d2ce9570ca78209008810495d6260b374c61aeb79bf1915a
RMD160 (NetBSD-6.0/newsmips/INSTALL.ps) = 031019f4af07c2dbb384bbb76702d2a6da747a0d
MD5 (NetBSD-6.0/newsmips/INSTALL.ps) = fe0ac420b2c0c7a4fba8bfbbc247c0fb
SHA1 (NetBSD-6.0/newsmips/INSTALL.txt) = d2abcef4ce10c3f567350ad9de6370a0f972f3f8
SHA512 (NetBSD-6.0/newsmips/INSTALL.txt) = a26df9df3950896c9fb77d728190516f1211b1630dff4969c651ad3f343acd9359a9a6251815c560a6e47071717c53b123d911776d5f5d6b8dee38c12d075381
RMD160 (NetBSD-6.0/newsmips/INSTALL.txt) = f2ef263bdb545134c7dbadb4509ff94ecf14d84d
MD5 (NetBSD-6.0/newsmips/INSTALL.txt) = fea6e65c9aac2b407af208e4f7272cf9
SHA1 (NetBSD-6.0/next68k/binary/kernel/netbsd-GENERIC.gz) = 9f78ea3f1d4a4b1c07d0a25487b20560adcb7fab
SHA512 (NetBSD-6.0/next68k/binary/kernel/netbsd-GENERIC.gz) = daae6643ef07004beaaa345b57182f2d2b69e990896734e18ec50ee925e17acf334da7a5478fa0fa38de4062fef3285582c9a4fe352759770e4cd9012d70d6f6
RMD160 (NetBSD-6.0/next68k/binary/kernel/netbsd-GENERIC.gz) = 2dcc60a3c451aaf9303c328a8471775b83ae066a
MD5 (NetBSD-6.0/next68k/binary/kernel/netbsd-GENERIC.gz) = d4502e039874ee01d7564f7572fa4789
SHA1 (NetBSD-6.0/next68k/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/next68k/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/next68k/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/next68k/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/next68k/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/next68k/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/next68k/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/next68k/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/next68k/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/next68k/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/next68k/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/next68k/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/next68k/binary/sets/xserver.tgz) = 1d20fd039546a66b428b854c659ea5e2ee9f8114
SHA512 (NetBSD-6.0/next68k/binary/sets/xserver.tgz) = 091d09049d0af5555feec02065234e281677aeb3ecbab5a19413d924a8b58dc7f64401ef68e68bc52d8b6eeb73e091d351c45805374b8bf276b0181415968561
RMD160 (NetBSD-6.0/next68k/binary/sets/xserver.tgz) = feef5d52278d23637d7b8ba77639bfd96918e90e
MD5 (NetBSD-6.0/next68k/binary/sets/xserver.tgz) = b2bd6057c374376388766fc8667eb9a9
SHA1 (NetBSD-6.0/next68k/binary/sets/kern-GENERIC.tgz) = 86b352f48720312e04260ba1dcf2f6e7e9d31e25
SHA512 (NetBSD-6.0/next68k/binary/sets/kern-GENERIC.tgz) = 602f5ac04b050c0462d21d4e50a9a8e5c794a3b4442b6df19370ab6dbd6839755bc61e2c7745f777aef417d5c1179ff4a2af52a5880d35eb20d1c5d3a580a844
RMD160 (NetBSD-6.0/next68k/binary/sets/kern-GENERIC.tgz) = f625cea32a206f3738a72b5ec0b01d8485c7eccc
MD5 (NetBSD-6.0/next68k/binary/sets/kern-GENERIC.tgz) = ff2d20b2f2b0a12231e405b582b61801
SHA1 (NetBSD-6.0/next68k/binary/sets/base.tgz) = d68bfc49aa82b76a52007a927f1a7bac692aaf18
SHA512 (NetBSD-6.0/next68k/binary/sets/base.tgz) = 4d2c20f0dcfa061e1a1176d34111225b6c8a94b0742141560ccdc986cd5311906967d06e4294a6f988bcea4cbc09d1a9607810960f2d0be35f4ab2895f68f112
RMD160 (NetBSD-6.0/next68k/binary/sets/base.tgz) = 6d98b84ec87a909545f72526f63cc27d450865fb
MD5 (NetBSD-6.0/next68k/binary/sets/base.tgz) = e40a7e78d87959c20f37bd937921383d
SHA1 (NetBSD-6.0/next68k/binary/sets/comp.tgz) = eb3082aae89eb8720e91538fd100125c9e6a717f
SHA512 (NetBSD-6.0/next68k/binary/sets/comp.tgz) = 8c695463b6ac0a403023068b670d04c7cbe50a9e3925e19896fddba7f409b2d142e9b2817bf18144d9bb55aefe2d7cd90c1242f9bdd6cea628332277d08f8520
RMD160 (NetBSD-6.0/next68k/binary/sets/comp.tgz) = 81c05df1b3c2c6a72575a325b2c507081279a587
MD5 (NetBSD-6.0/next68k/binary/sets/comp.tgz) = 11b81d6d2371e3b3123f0fc776526b80
SHA1 (NetBSD-6.0/next68k/binary/sets/etc.tgz) = d7697b36fa1c6c814b0745d496709c177b43441d
SHA512 (NetBSD-6.0/next68k/binary/sets/etc.tgz) = 9b4ed551eb36810109e30dee6273d5c18fc918686ba9cab5ffc4ab25a1763e645a179920e7d4ba56be3b15269425e38abf3237e1e6a788b369a4d26d59aa8a6d
RMD160 (NetBSD-6.0/next68k/binary/sets/etc.tgz) = beadac2eb477aca072b36fa2587e46645e908d11
MD5 (NetBSD-6.0/next68k/binary/sets/etc.tgz) = ebd0202a3d65b7a528b4854483ac669d
SHA1 (NetBSD-6.0/next68k/binary/sets/modules.tgz) = 49e37041d779f3b282c12b3f41d876e4621914f2
SHA512 (NetBSD-6.0/next68k/binary/sets/modules.tgz) = c8f22b3d92b7089aaf3b5fe563a9d0cbc7124b6e402e53b458881c123468e51b5b4f92d8e8b1a5c0f7ff19fc032bdd012db8dfd801be6330e96e0e79917df08a
RMD160 (NetBSD-6.0/next68k/binary/sets/modules.tgz) = 093e67a0796abb445f0f3b7f6bbfa42a68dff403
MD5 (NetBSD-6.0/next68k/binary/sets/modules.tgz) = dcf44be798ece88e148faaabe70e2251
SHA1 (NetBSD-6.0/next68k/binary/sets/tests.tgz) = 28b7d959516205c1231c7f756614c776fdba9576
SHA512 (NetBSD-6.0/next68k/binary/sets/tests.tgz) = b704bcbe14b260067ea704350ff9b49d49e040a6a87597211de8cea7e05039054c4f4996bd96829a9acc1a97320b3fa3f1ca2300717365e2c13fefac4e13ec11
RMD160 (NetBSD-6.0/next68k/binary/sets/tests.tgz) = b7ad8dc285efaeb99c3d1384dada9b20cc7104bb
MD5 (NetBSD-6.0/next68k/binary/sets/tests.tgz) = 45d4bf99be70595e367459256c139b65
SHA1 (NetBSD-6.0/next68k/binary/sets/text.tgz) = e079bed4ddd9e37cd0cb2f514823f81de9200669
SHA512 (NetBSD-6.0/next68k/binary/sets/text.tgz) = 3f66684ea0cdaaa433f52dc40eb6776091d19e5f53c87c3fa888ceaef29524dcd7b5c08ce8f08df94db7f039026dfe60a963d23f888bdeab70da4bf71592b122
RMD160 (NetBSD-6.0/next68k/binary/sets/text.tgz) = 692b64c1a4e521caa76fdc109175a636ef4f9d21
MD5 (NetBSD-6.0/next68k/binary/sets/text.tgz) = ea58b70eb622e2bc8fb610431f9f27c4
SHA1 (NetBSD-6.0/next68k/binary/sets/xbase.tgz) = 5e093e4987d3ef6e66dee1dbf5d2c2781a7be759
SHA512 (NetBSD-6.0/next68k/binary/sets/xbase.tgz) = 9bf75e2feb514bf47ead4151dfc44e0bd8361bd2155bdd7f5f5bda16c2ce60edef88cfbfe054161af32ea2804ccc5903b5ecc8a4270b2d19d6e864c7df776ef7
RMD160 (NetBSD-6.0/next68k/binary/sets/xbase.tgz) = 6c8fe359b16a5d6d7ab318ad8970b36b3a9d8b41
MD5 (NetBSD-6.0/next68k/binary/sets/xbase.tgz) = 4dc8195a284c6ff792f07b0279a1ffee
SHA1 (NetBSD-6.0/next68k/binary/sets/xcomp.tgz) = 6248c1a2455726a172e7d41ad3b71396d1569c27
SHA512 (NetBSD-6.0/next68k/binary/sets/xcomp.tgz) = 03272ed1984404a7ed8d6753aae6fd2797b82da0f856173bff109dd46909561af865acf81ec8007e8e95794d41e49caa4ffead9d0f5a304e4fe3a627fc62b5ca
RMD160 (NetBSD-6.0/next68k/binary/sets/xcomp.tgz) = a6fd52049b8fb32c457db212ea0b0f04ae8837f6
MD5 (NetBSD-6.0/next68k/binary/sets/xcomp.tgz) = 1e5c017075bcb0ddc800674737475134
SHA1 (NetBSD-6.0/next68k/binary/sets/xetc.tgz) = fc7edd9e51673c25587253ccfa05b06556554596
SHA512 (NetBSD-6.0/next68k/binary/sets/xetc.tgz) = 6d1f1584a7dbf8ddc3695951796b3951ef5fcae7176c93e5992aa84618b1a935d28baf14dac468cfd50e084414978e824054e08d30b3761e2467aecaa8a2d5cc
RMD160 (NetBSD-6.0/next68k/binary/sets/xetc.tgz) = 287ef7ae8028b2cdfdd081f19a387900bcb49b37
MD5 (NetBSD-6.0/next68k/binary/sets/xetc.tgz) = 9cc9fc5b70004b4e532fc745ac7c07f8
SHA1 (NetBSD-6.0/next68k/binary/sets/xfont.tgz) = 66f50766e91abe745da27688863877f1002907c7
SHA512 (NetBSD-6.0/next68k/binary/sets/xfont.tgz) = 3f1fd3be74c0971fe2217fb8e39f6ca693d07b5a272b3eff1028ee2f012ec2c04239ec90cd90846553e3a066b58fa957d0ab1d3695faaf0063a6c947165e63f4
RMD160 (NetBSD-6.0/next68k/binary/sets/xfont.tgz) = 185771ae1f0ed1ca9d8e7025f402bb41c5a14519
MD5 (NetBSD-6.0/next68k/binary/sets/xfont.tgz) = 57e25cb12fb7ae0b3cd06def9e685b7d
SHA1 (NetBSD-6.0/next68k/installation/boot) = 1678cf659734cb8f7d5d034bf1ce6aa506278920
SHA512 (NetBSD-6.0/next68k/installation/boot) = bf027e98b6c1a81300dfe506c084a9777f4f5f9b4078b35d61ea0a39ff73a5d9abe526f423164ed3c076ce7a33cedf460111142f24861c142a5b2c6485881ab9
RMD160 (NetBSD-6.0/next68k/installation/boot) = 8d3d5e3786b272a9dda7ca19185e3802639f7f6b
MD5 (NetBSD-6.0/next68k/installation/boot) = ec7ee9f5d5be157c6a9a56ac77835f47
SHA1 (NetBSD-6.0/next68k/INSTALL.html) = aaf1f7407874cf9d7bdff5b434a74e298b7dc52f
SHA512 (NetBSD-6.0/next68k/INSTALL.html) = b2d952c916578af512363fa118f81950b95b545aac62474194ef2ed4df3fc4fd9f334f7b3171335f7669c8d0992c37140daa86bab57dcb0020dfa39abfcc4ecd
RMD160 (NetBSD-6.0/next68k/INSTALL.html) = 52a08c6bd446a00228899639e98aeb8327f100ee
MD5 (NetBSD-6.0/next68k/INSTALL.html) = f08b6811f0f572819de3b455714166b7
SHA1 (NetBSD-6.0/next68k/INSTALL.more) = dced1147a6dc54b1365d13a49d0bf57d596653aa
SHA512 (NetBSD-6.0/next68k/INSTALL.more) = 5136e7a75ef6c3a39ca15112414c73c023c560dfbc09d4da60e6435112a4108c192f25b200f0a149a9eafdff3a74001630f972f58dcc2a9d18174286c906b680
RMD160 (NetBSD-6.0/next68k/INSTALL.more) = fd826a26af8b89375881dc361a7426b7841baa52
MD5 (NetBSD-6.0/next68k/INSTALL.more) = 0d480ba3bfd38cc17c62d387a740111d
SHA1 (NetBSD-6.0/next68k/INSTALL.ps) = 47541be31baefad1f76a45e45b2050cda159953f
SHA512 (NetBSD-6.0/next68k/INSTALL.ps) = 297a4fb8e5242a764b721a3bea309d123d3f71b67aec903926d5118b42831ba054242783fcbdc19520609ac06f81b733785b5952e24ae23218219046922f8f8a
RMD160 (NetBSD-6.0/next68k/INSTALL.ps) = 56423276a2db90876c067eb51b7c19f1341d261b
MD5 (NetBSD-6.0/next68k/INSTALL.ps) = ed71185d0510303cbb5d8ff681dcfaf5
SHA1 (NetBSD-6.0/next68k/INSTALL.txt) = 3d70ff39023294025c590934bdaca3fce5dd33bd
SHA512 (NetBSD-6.0/next68k/INSTALL.txt) = d0485c5d3587022d986a53345ed78f49130ef54070d0c45a7812d57400402079855d6cc72401579d1185d2dbeb5e5846efebfa50fda484cc5a1ac719ca7b706a
RMD160 (NetBSD-6.0/next68k/INSTALL.txt) = bd984eb4c13e1ec872ee2517b72c13ac7e90e090
MD5 (NetBSD-6.0/next68k/INSTALL.txt) = 0cf50370809ab3ff52dc710474d37edb
SHA1 (NetBSD-6.0/ofppc/binary/kernel/netbsd-GENERIC.gz) = 31cf98d9ea175b5ea6fe09455217e660d44f1973
SHA512 (NetBSD-6.0/ofppc/binary/kernel/netbsd-GENERIC.gz) = 22e1ac76e1fa230f69485f53ac23186a97562a1088b1a99979d9d446dcef8a863f1bd63eb6bf16ebf990b40b161fb75025890d9d52474b82cdb3d497e2056b2a
RMD160 (NetBSD-6.0/ofppc/binary/kernel/netbsd-GENERIC.gz) = 1ade658130f4e40d57cee2ab641234c1e93c3f01
MD5 (NetBSD-6.0/ofppc/binary/kernel/netbsd-GENERIC.gz) = ae8374ac0ba9ca59c6068561dcad1ba3
SHA1 (NetBSD-6.0/ofppc/binary/kernel/netbsd-INSTALL.gz) = 0b815e688d07caf3d93190562bcd53f3057ec1b6
SHA512 (NetBSD-6.0/ofppc/binary/kernel/netbsd-INSTALL.gz) = 43255a25b3044320db3d12f474a70920efe93a52aad3351933ae80bd079a8f93a866fa1c939a92d075161822066dc76ffe314feaf493a783ca51058eba9e96e0
RMD160 (NetBSD-6.0/ofppc/binary/kernel/netbsd-INSTALL.gz) = 87995ce3c896b4ffcef1eaeef9c7ff27e8c5446f
MD5 (NetBSD-6.0/ofppc/binary/kernel/netbsd-INSTALL.gz) = 008fbd36266fbb4ef45c3c289d07dbd3
SHA1 (NetBSD-6.0/ofppc/binary/sets/games.tgz) = fbf69640b362df2446d7414ac3a13db100c289eb
SHA512 (NetBSD-6.0/ofppc/binary/sets/games.tgz) = d86b41d79aea4204ccf45306937a5ec45629e37873a5956c0280fd1d54cf4a808542d097f9d978fd3277791554ce8a6fb4d0ccd46e30210549d5cce1145bb499
RMD160 (NetBSD-6.0/ofppc/binary/sets/games.tgz) = fd2aed1eb51f4db522fb5163e8fde4c040b55d30
MD5 (NetBSD-6.0/ofppc/binary/sets/games.tgz) = 997ae5fb42196a787df89873be321e47
SHA1 (NetBSD-6.0/ofppc/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/ofppc/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/ofppc/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/ofppc/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/ofppc/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/ofppc/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/ofppc/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/ofppc/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/ofppc/binary/sets/xcomp.tgz) = 960dd33b61e19dec92096a54791aca1cdf56ce54
SHA512 (NetBSD-6.0/ofppc/binary/sets/xcomp.tgz) = 111dd50041e4008aeadb898c3d08f9995c5c13a41ccc20993b4ad51c81d7f6ee3b4dab16158ac7436d085b80e4a643b236a9c97edd3e915125a1466154538420
RMD160 (NetBSD-6.0/ofppc/binary/sets/xcomp.tgz) = 0cf8b9a8d1489a32ba0b5a277d90e429c40650fb
MD5 (NetBSD-6.0/ofppc/binary/sets/xcomp.tgz) = 4ba4077ca2c0edf71813165fcf5321fb
SHA1 (NetBSD-6.0/ofppc/binary/sets/xfont.tgz) = 4711fcbade0238d60b7b89cf0d7873b03d7ed1f7
SHA512 (NetBSD-6.0/ofppc/binary/sets/xfont.tgz) = c29130f55f536f602253a7f92f073ebf6ba02baea09f8011530bbf7e03ae96594334f0c60b29a8bceaffc3eb8b152db47548e38a3c8a14abc5fe35bc5d1d7d93
RMD160 (NetBSD-6.0/ofppc/binary/sets/xfont.tgz) = 3b01af0f8e34367988d559fe0fbb0acbbafb0b2b
MD5 (NetBSD-6.0/ofppc/binary/sets/xfont.tgz) = 6ba15ed12217b726d9129db8e942a048
SHA1 (NetBSD-6.0/ofppc/binary/sets/kern-GENERIC.tgz) = 7ace360c931ac41952447fce2d6241a8393a7b3c
SHA512 (NetBSD-6.0/ofppc/binary/sets/kern-GENERIC.tgz) = 9c63f0169c8bf1282c54a5ae3b1396376b69a23b6ac881859a304a9d21aad45f64d0998eae3399cc397489a1c3e1b7f52cdce0ee806c160ae5da3287b113c00b
RMD160 (NetBSD-6.0/ofppc/binary/sets/kern-GENERIC.tgz) = b96e84c0a6d945475f6523d73cabb42ff80caa46
MD5 (NetBSD-6.0/ofppc/binary/sets/kern-GENERIC.tgz) = 2153bc142bb6f9067287344b5d3703e6
SHA1 (NetBSD-6.0/ofppc/binary/sets/base.tgz) = 48aef3545f49ee86082480f6712e6c7df6c1c82f
SHA512 (NetBSD-6.0/ofppc/binary/sets/base.tgz) = 580ad38769fc64e5f50a59f726d13d21625392459a03633c4908fbaf16e08715032dd10fbee9f4efe5261beb24169dcdaf86f11d20f2e2bfc7af22f16ce5b10c
RMD160 (NetBSD-6.0/ofppc/binary/sets/base.tgz) = 93fe246dba1352ba0bc5ef955094069e82e777cc
MD5 (NetBSD-6.0/ofppc/binary/sets/base.tgz) = 6a0ea91d58c236cd1d72963bee61dbc8
SHA1 (NetBSD-6.0/ofppc/binary/sets/comp.tgz) = 7d553fd5460fd823ee474377ad7e4e4a57796d04
SHA512 (NetBSD-6.0/ofppc/binary/sets/comp.tgz) = 17dff29a91c71c4fcabe616595144201b999b75d56033109f24dfa89e5653ea24890e2657fc007cc0bcbceb749213eb99db5d268db8c96f41f1c2ec203e951ef
RMD160 (NetBSD-6.0/ofppc/binary/sets/comp.tgz) = 79001f2a220166711cf3e076c137d9944cb72d07
MD5 (NetBSD-6.0/ofppc/binary/sets/comp.tgz) = bb52657afeeafa0a04fc52d5b96130cb
SHA1 (NetBSD-6.0/ofppc/binary/sets/etc.tgz) = b19a807890c2859c0c5dfcc27d97c7b94703ba1c
SHA512 (NetBSD-6.0/ofppc/binary/sets/etc.tgz) = 157110603a432bdf8da99f467e263037d2683372b37c2fb9e8ba7a25a5ed6e44b6d62aa3d60464ea3d0dd101d55b85ff951afde528cc0f61158892a3b379bc90
RMD160 (NetBSD-6.0/ofppc/binary/sets/etc.tgz) = 8e67db3042b34b9e0ec19853d0209996cb18cc22
MD5 (NetBSD-6.0/ofppc/binary/sets/etc.tgz) = 7c3ef92dc32ef4a24a29fd071fad94f5
SHA1 (NetBSD-6.0/ofppc/binary/sets/modules.tgz) = 976ee92c542c431c7eadaf77dc3267c54643aef9
SHA512 (NetBSD-6.0/ofppc/binary/sets/modules.tgz) = eedd03d73c348500f3df270f002d65d16cdc9f5318ecf14ebc8c4e27af6aff26701d985e5c5a47ed9f37359fb119d61ebe64cbcd726bb5ea7a0b2947aa312a8a
RMD160 (NetBSD-6.0/ofppc/binary/sets/modules.tgz) = 03d76ba89e00f988e16e0a5cd6798fca65852c82
MD5 (NetBSD-6.0/ofppc/binary/sets/modules.tgz) = 132d39b3a365d6ecd924789660a7b72c
SHA1 (NetBSD-6.0/ofppc/binary/sets/tests.tgz) = f643b7607751a3355cc6e9e8d26fcc1bfdd14a22
SHA512 (NetBSD-6.0/ofppc/binary/sets/tests.tgz) = 0a517f2c914e449c0bedb3be5e96e76c369be5be270ad8ba2aee4b96f3a3d7afce2bb56de0eef54d97b1194101eb24331dd202605eb58e648b3f79f04b44d96a
RMD160 (NetBSD-6.0/ofppc/binary/sets/tests.tgz) = aca5ea7884341e0dc0f4e06d552b908550cb7dde
MD5 (NetBSD-6.0/ofppc/binary/sets/tests.tgz) = d8c47bef7caa4afa0ea7967d42a59cfa
SHA1 (NetBSD-6.0/ofppc/binary/sets/text.tgz) = 010dbafdbd84074ed3acf0bacea61b34dad4005d
SHA512 (NetBSD-6.0/ofppc/binary/sets/text.tgz) = a8a4fe56016f47a03dd2e23cfb5368111cc7734a7a9dba9ab42081b609c00ac812ec8e73eb75b5dc2f81b847f0bf782257834e09603aac858c8efe1e6755eaf8
RMD160 (NetBSD-6.0/ofppc/binary/sets/text.tgz) = 263b72d30191910c092ab984458cd1442ba67f3a
MD5 (NetBSD-6.0/ofppc/binary/sets/text.tgz) = a6dfbc450a9aec79555f52694a2971f0
SHA1 (NetBSD-6.0/ofppc/binary/sets/xbase.tgz) = 292a1ad5d516609f5c3c5e78e83f4f200f967775
SHA512 (NetBSD-6.0/ofppc/binary/sets/xbase.tgz) = b3c082fb1fe52edc65fb728d9347d51285a761ba8bbb56cf84744f70614c2c1207a8b48c49d5ad598fc09745f3a900ca2bb08cff4381be8f007d6b89faf7c058
RMD160 (NetBSD-6.0/ofppc/binary/sets/xbase.tgz) = 954ba03beb3cc9b5064c0d9e9008358cf9c490f5
MD5 (NetBSD-6.0/ofppc/binary/sets/xbase.tgz) = 410d993f39451d776e2c9d9ca55414ac
SHA1 (NetBSD-6.0/ofppc/binary/sets/xetc.tgz) = b83db52f87a7a2ba08d22f29249f2b0d8af31028
SHA512 (NetBSD-6.0/ofppc/binary/sets/xetc.tgz) = 541ca16e25a0cd271b490773e043e6d66d9d704f4d4bd22458d5f3c4f76f23190264201a80cd57b567d8410ca2944a36e10868c4e2916300d978949d78293b33
RMD160 (NetBSD-6.0/ofppc/binary/sets/xetc.tgz) = de0ae01a1d264465ca58a678c658f75d2844d04a
MD5 (NetBSD-6.0/ofppc/binary/sets/xetc.tgz) = d60b94274347e3f9073b80da9f5dd874
SHA1 (NetBSD-6.0/ofppc/binary/sets/xserver.tgz) = e8fd963e80c60b8a85a284484d0d78463038f413
SHA512 (NetBSD-6.0/ofppc/binary/sets/xserver.tgz) = ca88d3ce0ccd0e82636ae59d4ea4acba9d29ba6fb51be52c6d8de5986d7b037e6062d00bc45390783efd3089de3eabbdbf565929936af7f46c207b705c33176f
RMD160 (NetBSD-6.0/ofppc/binary/sets/xserver.tgz) = a6b1345264d3fcddf0ff72b3005dd71053aba0f5
MD5 (NetBSD-6.0/ofppc/binary/sets/xserver.tgz) = 8150c6dfee587ce36ff31c34a1929e29
SHA1 (NetBSD-6.0/ofppc/installation/netboot/netbsd-GENERIC.gz) = 31cf98d9ea175b5ea6fe09455217e660d44f1973
SHA512 (NetBSD-6.0/ofppc/installation/netboot/netbsd-GENERIC.gz) = 22e1ac76e1fa230f69485f53ac23186a97562a1088b1a99979d9d446dcef8a863f1bd63eb6bf16ebf990b40b161fb75025890d9d52474b82cdb3d497e2056b2a
RMD160 (NetBSD-6.0/ofppc/installation/netboot/netbsd-GENERIC.gz) = 1ade658130f4e40d57cee2ab641234c1e93c3f01
MD5 (NetBSD-6.0/ofppc/installation/netboot/netbsd-GENERIC.gz) = ae8374ac0ba9ca59c6068561dcad1ba3
SHA1 (NetBSD-6.0/ofppc/installation/netboot/netbsd-INSTALL.gz) = 0b815e688d07caf3d93190562bcd53f3057ec1b6
SHA512 (NetBSD-6.0/ofppc/installation/netboot/netbsd-INSTALL.gz) = 43255a25b3044320db3d12f474a70920efe93a52aad3351933ae80bd079a8f93a866fa1c939a92d075161822066dc76ffe314feaf493a783ca51058eba9e96e0
RMD160 (NetBSD-6.0/ofppc/installation/netboot/netbsd-INSTALL.gz) = 87995ce3c896b4ffcef1eaeef9c7ff27e8c5446f
MD5 (NetBSD-6.0/ofppc/installation/netboot/netbsd-INSTALL.gz) = 008fbd36266fbb4ef45c3c289d07dbd3
SHA1 (NetBSD-6.0/ofppc/installation/netboot/ofwboot) = 56bb70527f572ae12667a4d9dc71027032602d26
SHA512 (NetBSD-6.0/ofppc/installation/netboot/ofwboot) = d137aad957d917f4db49ad9b912cf794d5ba9e52ac0f0d1926599dcac166df97c5b81292247b62d15a495134d4c9967ba777b0570701b3340021dcac293ab2d2
RMD160 (NetBSD-6.0/ofppc/installation/netboot/ofwboot) = 5a6f580e7c000543b09dc7590555ce944b7f2df1
MD5 (NetBSD-6.0/ofppc/installation/netboot/ofwboot) = 139a6e28174158a78719ab4e67de40b5
SHA1 (NetBSD-6.0/ofppc/INSTALL.html) = 70ffc237a74825aeff35c638c9dd33b10334c09e
SHA512 (NetBSD-6.0/ofppc/INSTALL.html) = 5df9959e93527b0f53d3866b15fcac4dbce2edd6a1bff082e588348ada3767bd0ca45d99089540fde08ecc73b8c8846ca7a046ada5f1f1072d6cd1e6c3268497
RMD160 (NetBSD-6.0/ofppc/INSTALL.html) = 3718a3b3de1edc4d81cc61c88146e69d8b126266
MD5 (NetBSD-6.0/ofppc/INSTALL.html) = 4a16acd4059c24dbcf1ccab96be7928c
SHA1 (NetBSD-6.0/ofppc/INSTALL.more) = 14b22fc68257222ce1d15d02554588760ce4fc62
SHA512 (NetBSD-6.0/ofppc/INSTALL.more) = e0c15cc5760ad5087d58c7659e8692b18fa805b6df85a7965ce142bb7506d3105c1dea0c7acdecfc51fa63f857dc959e25c74fa0ee871417670e013f4c2b2a1c
RMD160 (NetBSD-6.0/ofppc/INSTALL.more) = b726d0dd772cab274c9b2f6f484d5b1903151827
MD5 (NetBSD-6.0/ofppc/INSTALL.more) = 40e2acf64e90023938d03f0ff945ab0f
SHA1 (NetBSD-6.0/ofppc/INSTALL.ps) = 16a078ed4ada6fe71e1d5939bacd3a60c10cefbd
SHA512 (NetBSD-6.0/ofppc/INSTALL.ps) = 1d9815a162a77d91aebbfd11e0fe6d386c24be9e734683a6eee3cc49e059e8c4c9e559d00c3bfdbd3ceaaf885c11deb5c985ab4de41f1e1b1c9c5b037d300ec5
RMD160 (NetBSD-6.0/ofppc/INSTALL.ps) = 83f80393d6bd7845dac2c2f789b14104b5ed3b9e
MD5 (NetBSD-6.0/ofppc/INSTALL.ps) = 9ef011b3eea954489d1e03261175f278
SHA1 (NetBSD-6.0/ofppc/INSTALL.txt) = 0703c2fc0836a815f8a23c9e295482feced08667
SHA512 (NetBSD-6.0/ofppc/INSTALL.txt) = e5a59c4554e4dad54c4de8af3ef1fc051a3ed6ddcca88fe60b0f1c5e52c75507268a0cd154482fd7f8633ef0b6de5fa9052334adb93bc787e0f45db400018aea
RMD160 (NetBSD-6.0/ofppc/INSTALL.txt) = b4b1e1c493973acc61f5280eec83fd1a4b386f9e
MD5 (NetBSD-6.0/ofppc/INSTALL.txt) = 167ec8dd0f2430f3baafba98c15b0b77
SHA1 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.ecoff.gz) = c3341e74bbfc424f067743c23369e7d4de2320d8
SHA512 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.ecoff.gz) = 902237dd2bac97bbf8bc0d70e933307897e375cf760f90531ae39ff8f9abaf1c70091d61678e56448b46d060919eb93b052e0bdcba34482b12174855bf29a941
RMD160 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.ecoff.gz) = 12abc208e6d943a92255fa99934e2fbdd98b86d5
MD5 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.ecoff.gz) = 3a5c93ed6aedf91145618578c0283312
SHA1 (NetBSD-6.0/pmax/binary/kernel/netbsd-GENERIC.gz) = c673249c82581fc220eeab9db4826ae70a879f7d
SHA512 (NetBSD-6.0/pmax/binary/kernel/netbsd-GENERIC.gz) = 28c33a6cbb2cb4ad53e3e780c74a903ee93a6350da5a36c0232a57e690b80caf1053dc2bad1badd5631abeb20dbfbd223413762bf576db7363a3bd07aa91ddee
RMD160 (NetBSD-6.0/pmax/binary/kernel/netbsd-GENERIC.gz) = 1c56f35974b50c40595fb19a763e6e13a052136c
MD5 (NetBSD-6.0/pmax/binary/kernel/netbsd-GENERIC.gz) = 27f93c549aa40bae7bae47978622eb2c
SHA1 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.symbols.gz) = af7b35f9bea1761095521088a9bad8d6122565c9
SHA512 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.symbols.gz) = 506859e85d0266360f83d3a7af1deb8f9a84c8f1bbab10c1bd8b9b43bdf8ff7056629450f8aff2ed6f3424ff2034cd36e21144bceaaac6048fe54f58982dfa8f
RMD160 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.symbols.gz) = 71b182d23dfaa77e92597e840c0d5cb502b9ff7b
MD5 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.symbols.gz) = f70b79efeefcd03ef2b47a6d61b723c8
SHA1 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.gz) = f22aa49d239f1c2b49fc2779dbc71b79d6470468
SHA512 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.gz) = 82136634c9a3ea07a6eb23ad3e6d4214d796f54d406de73cd23f0da45a3872bae0f7f120ddb67aab0da05ef88feb21bb63ec1adf880469ee5d4eda2e32080732
RMD160 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.gz) = 351fe6fec563003dcb4a0045c740ba3f2fcb73ad
MD5 (NetBSD-6.0/pmax/binary/kernel/netbsd-INSTALL.gz) = 50f2b444f3462f71810f3c93b47083ff
SHA1 (NetBSD-6.0/pmax/binary/sets/games.tgz) = 27d9f98fecb3495e51939df9f107e81e527af26f
SHA512 (NetBSD-6.0/pmax/binary/sets/games.tgz) = 46bb4e10313cfffaaf4f7fa2157474ef940f94bcc9f84b46f829c7615bbf45e4e01edee7b7efb84bc28561e0712c37cfc737f9c2a0d9a73b67c80c8aa0f3a9af
RMD160 (NetBSD-6.0/pmax/binary/sets/games.tgz) = 947feea90ec2698a65d555cb78a29a001b086589
MD5 (NetBSD-6.0/pmax/binary/sets/games.tgz) = 355f00978fed8e8da6a251e591be5387
SHA1 (NetBSD-6.0/pmax/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/pmax/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/pmax/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/pmax/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/pmax/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/pmax/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/pmax/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/pmax/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/pmax/binary/sets/kern-GENERIC.tgz) = 1ae699dc77ef033860c3fcc0df198c6ec15bf312
SHA512 (NetBSD-6.0/pmax/binary/sets/kern-GENERIC.tgz) = 1a29fbe70596378ab0f543dd010c24ca327822d743152ce0405eb0e4ab268224b91d346d23b9ee1eec706b4cbe88ccdea27cb6a9b115ce9ddcdaced115a58682
RMD160 (NetBSD-6.0/pmax/binary/sets/kern-GENERIC.tgz) = 5edbc95d6df7a073b545902862a9e012a981b36b
MD5 (NetBSD-6.0/pmax/binary/sets/kern-GENERIC.tgz) = d9d9dab85e423b5953eb17bdf8072375
SHA1 (NetBSD-6.0/pmax/binary/sets/base.tgz) = a7e09afd631257929bbc0a4c6705fd56cafdda30
SHA512 (NetBSD-6.0/pmax/binary/sets/base.tgz) = 8313b9e6146645955a135e9f9c69b90688f9011ef4087a7911e842ff13802150347ac4e7e9b3f60313a06300e90e93c192910741fd427705709be5f669a45701
RMD160 (NetBSD-6.0/pmax/binary/sets/base.tgz) = 52eb2bdd47d2a4ae8ac2b780e70006dcc79f4cf6
MD5 (NetBSD-6.0/pmax/binary/sets/base.tgz) = 3b726d62e4341637cf824c141cdc362c
SHA1 (NetBSD-6.0/pmax/binary/sets/comp.tgz) = 9cc5684de31ac3b6c086f3716926eccbfaec6191
SHA512 (NetBSD-6.0/pmax/binary/sets/comp.tgz) = fbead7650935dcbea46ef3e11ef03468287b0461cbb70b51e330e641dd7e246bd602bc92935980e3e8ea8100e8ea487906763fc615e477462a9644201a7f616f
RMD160 (NetBSD-6.0/pmax/binary/sets/comp.tgz) = def96b7d151cc9405fe72c51f7ca95b552f3bd46
MD5 (NetBSD-6.0/pmax/binary/sets/comp.tgz) = 93a3bdd6f9461cc642955b749cddb59a
SHA1 (NetBSD-6.0/pmax/binary/sets/etc.tgz) = 73127a1dcbea90c9029538e87773bc7846ac2016
SHA512 (NetBSD-6.0/pmax/binary/sets/etc.tgz) = b96b30df1576538bcbaddcf471cd4d87258c06d199077d35d4ea37dfc9c3f388bd62c1ea8c3634ee501d6f28409bd966b791bf3eef8741ca2234c12e1101008a
RMD160 (NetBSD-6.0/pmax/binary/sets/etc.tgz) = 44ef8ce24cdd12aa5af0cbbf700abeef900e12e5
MD5 (NetBSD-6.0/pmax/binary/sets/etc.tgz) = 0dd976e6a6c855f02a86fc0fb624b4af
SHA1 (NetBSD-6.0/pmax/binary/sets/modules.tgz) = 9ed1e0f57d9ee14dc3850e541f0b95132a3048c2
SHA512 (NetBSD-6.0/pmax/binary/sets/modules.tgz) = 7a367e78224326f43584bb7cfa1aedd74763ccfca9d6abc8587882ecb1b8276851ead5ff1afbe9cd16accfd0d415e26db6604d3a9c3a1bac78d94107a57d033b
RMD160 (NetBSD-6.0/pmax/binary/sets/modules.tgz) = cff6f3fc309012a3f45c047ea5238f83432235a5
MD5 (NetBSD-6.0/pmax/binary/sets/modules.tgz) = 825fa6d6cac00e2629ec059fcb68008b
SHA1 (NetBSD-6.0/pmax/binary/sets/tests.tgz) = 802f716b73250eee2696cac4b6e3395833c0237d
SHA512 (NetBSD-6.0/pmax/binary/sets/tests.tgz) = f56cba629167f2f679a460679faaf566f4baf053d644a71fd70fa3041aa592aa0a53b98876ce474b48f46a582401a882161005360fdddc49246f094680b1029f
RMD160 (NetBSD-6.0/pmax/binary/sets/tests.tgz) = a9b17f97265bdfb69402190690d77306f42ecb6b
MD5 (NetBSD-6.0/pmax/binary/sets/tests.tgz) = 54dd7a96d4abee24bb0019a846d77825
SHA1 (NetBSD-6.0/pmax/binary/sets/text.tgz) = a0b59e7fb204b9db40df7edda528e8ca8b4e3fd6
SHA512 (NetBSD-6.0/pmax/binary/sets/text.tgz) = fb7c0d5cb17bb2ef1ffeefa5791bab7932a05e33cd064f279d99eebe07a52f1420aa64a8c6acf191efacbfd130509fb427e6d5d8f14c042b3c0e5e285282592b
RMD160 (NetBSD-6.0/pmax/binary/sets/text.tgz) = b0495549753b7c3a8abb342045ef4f65e7b640d5
MD5 (NetBSD-6.0/pmax/binary/sets/text.tgz) = 9f9e6d5b8f9dbb70b845a9aba07ffff4
SHA1 (NetBSD-6.0/pmax/binary/sets/xbase.tgz) = 99d83d866f2f15b6030b0533990e012ab59b1f46
SHA512 (NetBSD-6.0/pmax/binary/sets/xbase.tgz) = 238db55892d9cbb0bd5dacb225a16e694c79af2fe461f06563a65c5d2c5b3abfc823ab26f23c54bc1514568018726baab0258aac0e52ccf7615d59c2a6ef324a
RMD160 (NetBSD-6.0/pmax/binary/sets/xbase.tgz) = 16560205f7564558bc22495116ace64618834133
MD5 (NetBSD-6.0/pmax/binary/sets/xbase.tgz) = 384d33f9a8ec5c7bc8e104869e3e493a
SHA1 (NetBSD-6.0/pmax/binary/sets/xcomp.tgz) = 6b6b8d6b522b0546a13a304195765526158f6eb2
SHA512 (NetBSD-6.0/pmax/binary/sets/xcomp.tgz) = 37af51b256f03e8b4c61d4c67724cead53115ccbfda7d597f3b00a020bd8f4c1fff4f6206e5fe8246e0b3fae50c7dc1df313858670a6e1729b1155e0f97e9b23
RMD160 (NetBSD-6.0/pmax/binary/sets/xcomp.tgz) = 91d9a2cecb3955beabb71191b73d3586f2c370bd
MD5 (NetBSD-6.0/pmax/binary/sets/xcomp.tgz) = 9028e22a82276249cc188f83101af806
SHA1 (NetBSD-6.0/pmax/binary/sets/xetc.tgz) = 2171c5c0cb3143aa83fb8e55d4d98fd87d3e556d
SHA512 (NetBSD-6.0/pmax/binary/sets/xetc.tgz) = a63407a7d884ea60761ccdee9279c37b790aa08defced01410786e0e020a63b5a4e4cfd503a0ef09e555d04a98a99dbfcfb087c476070771f8b5b5a205b4495a
RMD160 (NetBSD-6.0/pmax/binary/sets/xetc.tgz) = fa4c1f7437db71fb1a3122cc366591b39b00fa07
MD5 (NetBSD-6.0/pmax/binary/sets/xetc.tgz) = be3b5f617ddd47650cf9423451320c4a
SHA1 (NetBSD-6.0/pmax/binary/sets/xfont.tgz) = ca16714159668bc20017cbb9a81fe3746171e383
SHA512 (NetBSD-6.0/pmax/binary/sets/xfont.tgz) = 077ab3a7540abb8d4039b7f017ff52dcb5aae2ba5a984dd12b04d22908863cc60db8aa67b5f772a584aabbe05a74426534eeced9f323aa07c9225c8333c41f3f
RMD160 (NetBSD-6.0/pmax/binary/sets/xfont.tgz) = 33beb97b3267941786413a7c5624d78297f5ccf6
MD5 (NetBSD-6.0/pmax/binary/sets/xfont.tgz) = c262ee8846a85c1e5b00a8ecbe754dcc
SHA1 (NetBSD-6.0/pmax/binary/sets/xserver.tgz) = 86332fccabc2c7b8e7ef7f8fa9eebc2f22c8afba
SHA512 (NetBSD-6.0/pmax/binary/sets/xserver.tgz) = e02ed6aa6333bd8941d02f15a3797f862f3fe47b5afce08d60021a06d5ed9db56e8bf802920e8b7614ff5e3127706b195ab9a678710b7b6afb6b70e0f3a7f636
RMD160 (NetBSD-6.0/pmax/binary/sets/xserver.tgz) = 4bc06df7b2012be80b61ef6b9860a4698b815353
MD5 (NetBSD-6.0/pmax/binary/sets/xserver.tgz) = 5dd0ae8362aa55b5c7ab207d771834f3
SHA1 (NetBSD-6.0/pmax/installation/diskimage/diskimage.gz) = 013da364eedff1ef317fe4933205a1f6e45a7520
SHA512 (NetBSD-6.0/pmax/installation/diskimage/diskimage.gz) = 9a415703b540ea67b99a367bf8a2dbd73c18fc1aff0ee2b3ffedb858d09cfa1b3e7bc62a654061963394f3540c50259ad343edfc41518fa444332d7ab9ea5e8e
RMD160 (NetBSD-6.0/pmax/installation/diskimage/diskimage.gz) = 3c7faef744c1c309dbad390121852ca1ad8ed0f1
MD5 (NetBSD-6.0/pmax/installation/diskimage/diskimage.gz) = 2e40475c5dfbfa75ac856a0a02f4afd7
SHA1 (NetBSD-6.0/pmax/installation/miniroot/miniroot.fs.gz) = fb7b259d7649170e5eaa31217417b6e75291529f
SHA512 (NetBSD-6.0/pmax/installation/miniroot/miniroot.fs.gz) = 3ef73219ab90c01753bf53dc325b64e26b8b41a7cd78ff1fcb85c4d27ab44b2979b6d312c17383ae9f8c3d24c968f5b896fd04a0679e64a47ea859b578022722
RMD160 (NetBSD-6.0/pmax/installation/miniroot/miniroot.fs.gz) = 4f8d0f80c84f93b4ea5d9025c4fd3227c2880641
MD5 (NetBSD-6.0/pmax/installation/miniroot/miniroot.fs.gz) = 3e2cc1da2be2c9ae099bd488393a9904
SHA1 (NetBSD-6.0/pmax/installation/netboot/diskimage.tgz) = de0f5cd5da27716499b56ae61afe5ca6a8bb0839
SHA512 (NetBSD-6.0/pmax/installation/netboot/diskimage.tgz) = 0283bc16d8fef0704b0d90bda6213d05e63271053ce59cb20d6cf0dc3a76bb8ed711e968de5a194d2d900becf83eb4fccbd82c22ed868c9a79b7cfbbc43acef7
RMD160 (NetBSD-6.0/pmax/installation/netboot/diskimage.tgz) = a46875bffd27ea9d145b5e5f2a286a49c5b93cdb
MD5 (NetBSD-6.0/pmax/installation/netboot/diskimage.tgz) = a491535a6424e97d5c52207404922ba9
SHA1 (NetBSD-6.0/pmax/INSTALL.html) = 416fb09d6569c88845b26c89d9062106799cb772
SHA512 (NetBSD-6.0/pmax/INSTALL.html) = cc06ccfbe9a109b6873a6e280baa5d401d3adb507160e145446e0c6d3c042cd5ba2f9edbe1c8d9731722d12fcc42bf40c5ff7a8e1746d096c182fe92437d2072
RMD160 (NetBSD-6.0/pmax/INSTALL.html) = c40efcbd9facf5dce4471b1adf725830cd53b934
MD5 (NetBSD-6.0/pmax/INSTALL.html) = 140966d090ff98ea58dc24cf673a5e11
SHA1 (NetBSD-6.0/pmax/INSTALL.more) = c737caf24e7e33dd11ee48d0f0b1c0ec2efe8010
SHA512 (NetBSD-6.0/pmax/INSTALL.more) = e014f63f15cfe131b38adf13500caab4ba7d43a7dce6432b2d688e916c3939ac851e5cb9af544db231cf729e5e39f223b1a73cff5ad66a92714aac71546c863d
RMD160 (NetBSD-6.0/pmax/INSTALL.more) = 3d69b81b9dacf7933a2b0e9c76568f9ccd212330
MD5 (NetBSD-6.0/pmax/INSTALL.more) = 7aae0fb416c11b3ba0558595afc3af37
SHA1 (NetBSD-6.0/pmax/INSTALL.ps) = 64b1af6366baed6f548759b6a09fdbbfbb85dbf2
SHA512 (NetBSD-6.0/pmax/INSTALL.ps) = ec6c213be8ee3af8e2720fcce970c9e8d5e220b1b94ba5f1ef1d8fd4a4c1ae773d522e077e28aadbb4c78a814ea4c06e2f1a915de8e8628f2602cd8705ff786a
RMD160 (NetBSD-6.0/pmax/INSTALL.ps) = 17eac5c26ecf0947848d6742ba0aeb54d5b7ad21
MD5 (NetBSD-6.0/pmax/INSTALL.ps) = 242d97c9cf75db873cb975d2788a24c5
SHA1 (NetBSD-6.0/pmax/INSTALL.txt) = 90f4bfa985417eff921768c47ffc495b066292f6
SHA512 (NetBSD-6.0/pmax/INSTALL.txt) = 36a7a301319525765fa1ae6c4d73b93c02b59a6f532f0dd1f96503a330daf0ba903f97df81dbd906bd54f83a83332efb28a8450d3a99ec5214abe0ed5369041f
RMD160 (NetBSD-6.0/pmax/INSTALL.txt) = 02fe56f81cd11b4c65a13da0868e887ccfd7d511
MD5 (NetBSD-6.0/pmax/INSTALL.txt) = c6a7c6addcfbd8f1b1bc708946065b52
SHA1 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL.symbols.gz) = 4a2e47a10e1c8913a04a5c574a0f09b82540b23d
SHA512 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL.symbols.gz) = 07c86d3b4d188d94141d181b1828e0548bf1e9597c6b470e7a35c787e7c830905fdc63ea4c8ea3f8310378f2e2bb55097c915d1785379d30c12861047d95af12
RMD160 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL.symbols.gz) = 70b4600b73a7d8b937790bb9d69c68e82fdf559f
MD5 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL.symbols.gz) = b0f9598d18b5ff0081a3dcc02e4ae185
SHA1 (NetBSD-6.0/prep/binary/kernel/netbsd-GENERIC.gz) = aadff547ddd7061ee0fe0c72833b691eaeae554a
SHA512 (NetBSD-6.0/prep/binary/kernel/netbsd-GENERIC.gz) = 57a25d74b982bbd519271f24456cd5de583e9e5c7e2ef7d8850d6aa64fcccddc9a2e6545c3ed7323e087d04116c2c78ace5b427b14fc3542435ba3bb9497e4b5
RMD160 (NetBSD-6.0/prep/binary/kernel/netbsd-GENERIC.gz) = 9a642cc44881b1f2b6049710d159f7a4ffa0f25d
MD5 (NetBSD-6.0/prep/binary/kernel/netbsd-GENERIC.gz) = 2a37ae4ea5cdc2702db0dba34cf7be3c
SHA1 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL.gz) = c138618a1e3ab93778033de4e66966f08e047d2b
SHA512 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL.gz) = b4045a71215ac632950cae3022d9d2d7ce9af9d225ffd8a0e775093e934cc4b0492f1e95167bcb7bc245bb9d5f3390de974158b0d0ea8146fd54bc40199ed10f
RMD160 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL.gz) = 5a2a9371e99acf25730bdd8e9826afc16257560b
MD5 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL.gz) = 3aaa4706000b77fc8078bf2dc572c746
SHA1 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL_SMALL.symbols.gz) = 1ef4af8edaba52436270f4eabc0549fd8c57767d
SHA512 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL_SMALL.symbols.gz) = fba7441719c1023b2719179ad017ac8e4c7a99499d2152ce75dc54014cfb55eecd0bb50e549ce976e1a5d58b420cd60416db364ed8633682a7c9c66a22af1329
RMD160 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL_SMALL.symbols.gz) = 428a02bccd74b8b6462a0ec5ecfebab5a9576def
MD5 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL_SMALL.symbols.gz) = 901fb77c124b0ba7923d535a65a06fa6
SHA1 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL_SMALL.gz) = ec4f1dffaaffea221d480df7dfe89d7a129dd9b1
SHA512 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL_SMALL.gz) = d5ee09d3d063a6a816b7fd0fd3fa3d0251e1b056ec75368bf557e79836afc0292274d3642d9afedbfb4893a4d4cca97e4f1d0d859db60e3a8b644d9826e57d94
RMD160 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL_SMALL.gz) = 0bfa88f9965abca2ffa342158b4e1621183887e1
MD5 (NetBSD-6.0/prep/binary/kernel/netbsd-INSTALL_SMALL.gz) = 8d52534cc5095a06286d6644f2ede176
SHA1 (NetBSD-6.0/prep/binary/sets/games.tgz) = fbf69640b362df2446d7414ac3a13db100c289eb
SHA512 (NetBSD-6.0/prep/binary/sets/games.tgz) = d86b41d79aea4204ccf45306937a5ec45629e37873a5956c0280fd1d54cf4a808542d097f9d978fd3277791554ce8a6fb4d0ccd46e30210549d5cce1145bb499
RMD160 (NetBSD-6.0/prep/binary/sets/games.tgz) = fd2aed1eb51f4db522fb5163e8fde4c040b55d30
MD5 (NetBSD-6.0/prep/binary/sets/games.tgz) = 997ae5fb42196a787df89873be321e47
SHA1 (NetBSD-6.0/prep/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/prep/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/prep/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/prep/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/prep/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/prep/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/prep/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/prep/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/prep/binary/sets/xcomp.tgz) = 960dd33b61e19dec92096a54791aca1cdf56ce54
SHA512 (NetBSD-6.0/prep/binary/sets/xcomp.tgz) = 111dd50041e4008aeadb898c3d08f9995c5c13a41ccc20993b4ad51c81d7f6ee3b4dab16158ac7436d085b80e4a643b236a9c97edd3e915125a1466154538420
RMD160 (NetBSD-6.0/prep/binary/sets/xcomp.tgz) = 0cf8b9a8d1489a32ba0b5a277d90e429c40650fb
MD5 (NetBSD-6.0/prep/binary/sets/xcomp.tgz) = 4ba4077ca2c0edf71813165fcf5321fb
SHA1 (NetBSD-6.0/prep/binary/sets/xfont.tgz) = 4711fcbade0238d60b7b89cf0d7873b03d7ed1f7
SHA512 (NetBSD-6.0/prep/binary/sets/xfont.tgz) = c29130f55f536f602253a7f92f073ebf6ba02baea09f8011530bbf7e03ae96594334f0c60b29a8bceaffc3eb8b152db47548e38a3c8a14abc5fe35bc5d1d7d93
RMD160 (NetBSD-6.0/prep/binary/sets/xfont.tgz) = 3b01af0f8e34367988d559fe0fbb0acbbafb0b2b
MD5 (NetBSD-6.0/prep/binary/sets/xfont.tgz) = 6ba15ed12217b726d9129db8e942a048
SHA1 (NetBSD-6.0/prep/binary/sets/kern-GENERIC.tgz) = 7ea9bbc043530351e4dfc3ac54ce19938a412ec4
SHA512 (NetBSD-6.0/prep/binary/sets/kern-GENERIC.tgz) = 0123955eb6ead53c99c21e1dcd5b17f62f228c35e8d17c9f540291f0022574e2ae331a9aebc2ed947cc0cabf439fa47c59e8f820eb8e5241b6446eb79c11bdee
RMD160 (NetBSD-6.0/prep/binary/sets/kern-GENERIC.tgz) = ef4531ac387b909f96b2cd3079ba8f12ecc61aa9
MD5 (NetBSD-6.0/prep/binary/sets/kern-GENERIC.tgz) = 3d31ea9358a8ba0d7cb49ae1baf77e52
SHA1 (NetBSD-6.0/prep/binary/sets/base.tgz) = 586f23dd6c50c12204650c92632bf016c9e4d57e
SHA512 (NetBSD-6.0/prep/binary/sets/base.tgz) = 712c63416a188e041d85155a36248e5580df0aace895b045b9a2a6cb68f5cb80315dd7f731e5a79139ee19d670c7278e44561336e1224c56a71f89ece1aa79ac
RMD160 (NetBSD-6.0/prep/binary/sets/base.tgz) = 11a21367c332428fc4e855764763276bae636fb3
MD5 (NetBSD-6.0/prep/binary/sets/base.tgz) = 8d8b79b420ca775f8a7f33a109e40f0a
SHA1 (NetBSD-6.0/prep/binary/sets/comp.tgz) = 2ecb14f0695750113144390d254e685c1c41331a
SHA512 (NetBSD-6.0/prep/binary/sets/comp.tgz) = 7f8528e32862466f82158d4a60e46cf04a9c451b13655d2f25e929bf9df3dcda5c90e1e491457be3a82caea35cb2cfafdd307286e82bbddc466e8b929f44c322
RMD160 (NetBSD-6.0/prep/binary/sets/comp.tgz) = 8ac4b55abc59b0089485a58d43d304c23dfa7519
MD5 (NetBSD-6.0/prep/binary/sets/comp.tgz) = 7dc818422b8bdb702d921d284f0f1246
SHA1 (NetBSD-6.0/prep/binary/sets/etc.tgz) = 56d8194e33b3ee4a1c059b1435e89a915add1275
SHA512 (NetBSD-6.0/prep/binary/sets/etc.tgz) = 4b5eb6c9222bdcc591d9e6ad9e0781fcaa6a3bb981fe7d5f4b7137ec8df902b9fb979ea419db313dc1a823dbbdc284204b46872516d8f801f18854f75da74d68
RMD160 (NetBSD-6.0/prep/binary/sets/etc.tgz) = 4b2fb8d900c22be6844c8931cf4c660954d123e4
MD5 (NetBSD-6.0/prep/binary/sets/etc.tgz) = 553a0fb84d187ff3c420d5b5dc7287eb
SHA1 (NetBSD-6.0/prep/binary/sets/modules.tgz) = 60eb72b005b0e80f397d42ed5d78fea15e867a1c
SHA512 (NetBSD-6.0/prep/binary/sets/modules.tgz) = 98ef7da109a52a1814d3a7ff1c6377fd30140cc9783a919d6b3331226b1ea3c7b168598fb3102533637e24ad48be9f265063c0a55043e6614ae5959a56ba0aba
RMD160 (NetBSD-6.0/prep/binary/sets/modules.tgz) = 214a8c4490e3ba6d6a4861d19bc88700de79a2b8
MD5 (NetBSD-6.0/prep/binary/sets/modules.tgz) = 775b2278852b97f68453f62cb4a1fabd
SHA1 (NetBSD-6.0/prep/binary/sets/tests.tgz) = baf4e3d6a0237d7eac65eca2d3e571ccde49c97d
SHA512 (NetBSD-6.0/prep/binary/sets/tests.tgz) = beea30696935343ffd119c696e826a617a189dd4e9be2a08f468207f0dfa518af4164a1c0bfb6e8d796f609888ef2e1c8d2b1481c58309cd3077cf361118052a
RMD160 (NetBSD-6.0/prep/binary/sets/tests.tgz) = 8547af5c43e56924032b7b7205c61c7dfb90c018
MD5 (NetBSD-6.0/prep/binary/sets/tests.tgz) = fad9ca518beadf4a15db6b0aace1d019
SHA1 (NetBSD-6.0/prep/binary/sets/text.tgz) = 265ca8e29442ca31af66219963a9931d7f71fcd0
SHA512 (NetBSD-6.0/prep/binary/sets/text.tgz) = 50cd104f35d738fded65600514abc46f96dbcc010f13cade2131a3aad0feb7cf1208360d5869114ca13276096a83e4960ff2f342c1c58b04c26232bc23a8d950
RMD160 (NetBSD-6.0/prep/binary/sets/text.tgz) = bf0370ee7934a80fdd3dadfe68b2124a7d1a3784
MD5 (NetBSD-6.0/prep/binary/sets/text.tgz) = 2305e5d5fe891fca62f8500eaf7e32ec
SHA1 (NetBSD-6.0/prep/binary/sets/xbase.tgz) = a9c6040c6de18dcf42371b7eb0eaa88cb005f1e6
SHA512 (NetBSD-6.0/prep/binary/sets/xbase.tgz) = 2f5640b41005ccc6b989454a60ae68f9637d701aca7e62a0c5f9965efda0bed43891a239be22ef2f837d0769ce54a649b9e357a920f007778e182d7362feb3e3
RMD160 (NetBSD-6.0/prep/binary/sets/xbase.tgz) = b171d2bf07e02360363a642a9ccf6b624bdb85cf
MD5 (NetBSD-6.0/prep/binary/sets/xbase.tgz) = f1029043dab956ff0864f197dc1c0875
SHA1 (NetBSD-6.0/prep/binary/sets/xetc.tgz) = 3d9da3ee21149017269ed1870b2c5c1096e9ed50
SHA512 (NetBSD-6.0/prep/binary/sets/xetc.tgz) = d971316b80d139e87b2f7fa47f64ecc41a6285e002986165268e551b0ab059ffa332b9e9278b1e86d60829dac01f8c7cdd3a36f047f0e4a50469046456dbce46
RMD160 (NetBSD-6.0/prep/binary/sets/xetc.tgz) = e6a7c17de4ac3a609498d8005165277e34f40b8d
MD5 (NetBSD-6.0/prep/binary/sets/xetc.tgz) = 8ad7ef8013d48e0ca196e8fd2a4fd0aa
SHA1 (NetBSD-6.0/prep/binary/sets/xserver.tgz) = 16190128b64666ba44729a72387061d80fd14c1e
SHA512 (NetBSD-6.0/prep/binary/sets/xserver.tgz) = 8f9b3a559b5b53dc2161128db89509e12bb3adc951dd51141212c5082e3afc19b6bd5d960900e7986c72a6ee664ebab7473e2119657c1f34cfebb05737655b6f
RMD160 (NetBSD-6.0/prep/binary/sets/xserver.tgz) = d62cfac8d16c1e56e866fe621b3cb5bbace44d07
MD5 (NetBSD-6.0/prep/binary/sets/xserver.tgz) = eafffc08a401b7b8e92e47968dc8faaf
SHA1 (NetBSD-6.0/prep/installation/floppy/generic_com0.fs) = 7bfb784953be9b7b783a4d7927b8795441ec6096
SHA512 (NetBSD-6.0/prep/installation/floppy/generic_com0.fs) = 59567fa24b9801e9aaba2fa0b85245b3c1a34894e6f2f58c39877bdb3053b3e21cce4cad703d60626466a2c0d2cd75715919df4a3acacefa627d19946123a840
RMD160 (NetBSD-6.0/prep/installation/floppy/generic_com0.fs) = 0614c7375a11ffba581e95ea182aef3b59f7f624
MD5 (NetBSD-6.0/prep/installation/floppy/generic_com0.fs) = b93c25c459836cca031a48a788ac7c78
SHA1 (NetBSD-6.0/prep/installation/floppy/generic.fs) = c4d629bee0746f9d60ffb5ee501e4e9e73ef3724
SHA512 (NetBSD-6.0/prep/installation/floppy/generic.fs) = 4df78c22f6a3d4ad414bb85ff6305a0d5caa97cc88375f3bdb2f3872e8103135f5716e72d564fe5fd3b7c1dbf7353f5464317ca30feb7187599186320da0a912
RMD160 (NetBSD-6.0/prep/installation/floppy/generic.fs) = 7e34800d3411adb3862a914554d3ad27f7775e02
MD5 (NetBSD-6.0/prep/installation/floppy/generic.fs) = c40503911a43d7bb98c5a5224c776666
SHA1 (NetBSD-6.0/prep/installation/floppy/sysinst_com0.fs) = 3d67fecfe7d82c9cf7479152b092bd106b87f2fd
SHA512 (NetBSD-6.0/prep/installation/floppy/sysinst_com0.fs) = 929443de2610ee9409ed76468aa9645a917bdc03366267e8483cc83bbcecb35208acc06ef0d2520d389b408b566c113f5e6b71f5568a7e1c029532979221682f
RMD160 (NetBSD-6.0/prep/installation/floppy/sysinst_com0.fs) = 51197c7d8ae21a29cf54064c3cf11f8d1eeef956
MD5 (NetBSD-6.0/prep/installation/floppy/sysinst_com0.fs) = d1e27f1504e7dfeeab0714332e89fda2
SHA1 (NetBSD-6.0/prep/installation/floppy/sysinst.fs) = 47998496e145c48564efadaf91f3ba4aede44ccd
SHA512 (NetBSD-6.0/prep/installation/floppy/sysinst.fs) = 6688642f8ad525611b7757bfaa63c958abc0cc1f0c17219df85055c7d97c20896156a543992bbcaf08bfe1bfe36e112e8c8af8610c971585dffa27ae2083d410
RMD160 (NetBSD-6.0/prep/installation/floppy/sysinst.fs) = d674830e0387841948abeb5e194ab60d9ab942fd
MD5 (NetBSD-6.0/prep/installation/floppy/sysinst.fs) = 3164ba344aa2283cc7aca4efbd627dd3
SHA1 (NetBSD-6.0/prep/installation/floppy/sysinst_small_com0.fs) = 2614d29a10f74be3e6ec6563046d2129976f3e1e
SHA512 (NetBSD-6.0/prep/installation/floppy/sysinst_small_com0.fs) = 061156a6e9e1f9130d9ae085838526f027f3f4263ed03ebc3b5a4532e75243715bc554e4a0ad307401c792f35334d79653642e0d2e1e438dbacc246cda1a92fc
RMD160 (NetBSD-6.0/prep/installation/floppy/sysinst_small_com0.fs) = 58ef2e610b17686a92958917250c4252bf2d2530
MD5 (NetBSD-6.0/prep/installation/floppy/sysinst_small_com0.fs) = e1b821660c4f9a7b22a9069631fcefcb
SHA1 (NetBSD-6.0/prep/installation/floppy/sysinst_small.fs) = 73f7d288f4026018498e79c14d362eeb5e25ef01
SHA512 (NetBSD-6.0/prep/installation/floppy/sysinst_small.fs) = b65d0794537f5dea1b2a3d6afbd08b85d6d97193a6f0c69928b2edb3fd90c1202035df882878e8f45ddca7214a1abd28e312ab6c334a0708bea0daa1a2686a94
RMD160 (NetBSD-6.0/prep/installation/floppy/sysinst_small.fs) = b40f136693485d517b455f0443614a487904776b
MD5 (NetBSD-6.0/prep/installation/floppy/sysinst_small.fs) = 3d605afd5b9a7b57949f448c43e8274c
SHA1 (NetBSD-6.0/prep/INSTALL.html) = 2674ff3a673f19a61cac4e2877a5656f053b2a3f
SHA512 (NetBSD-6.0/prep/INSTALL.html) = aabbf165a39dc0207f39cfd27508821de8e50d26892baa27daf1a9b1a01841777fa59a147baa2fc082ebe5ee1f7cafd27674ea44c3e9410179bfbafb33130b49
RMD160 (NetBSD-6.0/prep/INSTALL.html) = a818880156d01aebc4c92e2e82677fb0890e92c4
MD5 (NetBSD-6.0/prep/INSTALL.html) = c716d934eb76805dd4acbd677ef13fe5
SHA1 (NetBSD-6.0/prep/INSTALL.more) = f408ad8a753e682120f22e8763d9e20d8eeda7ed
SHA512 (NetBSD-6.0/prep/INSTALL.more) = 0a27fc0b1bc6bdf44bc2aced239066f13e352ff248d8ed50d430e676bba3132f26ba807259fda765738bc915a6765fab2fd2e97e04b0166a50cefc4c88de38d2
RMD160 (NetBSD-6.0/prep/INSTALL.more) = 5cc22227f3a244681cc352cc364650ad683c2fc1
MD5 (NetBSD-6.0/prep/INSTALL.more) = 69c4b26606cf1e071a451c5afab13cd0
SHA1 (NetBSD-6.0/prep/INSTALL.ps) = 9ab023c53af56e033541d6c27abf9b666b3f6a06
SHA512 (NetBSD-6.0/prep/INSTALL.ps) = b68f21d8587797e750a6eabe92273481a396bf68d62e300e6e07139439144660d1e8769404b109759707bccf02a97b27fde4aed219f2b760e9e1f65c0b8c7016
RMD160 (NetBSD-6.0/prep/INSTALL.ps) = baf09ab6b65aa90ca2f415e4428e21edf2d6d7a5
MD5 (NetBSD-6.0/prep/INSTALL.ps) = dd269cf99b1eb54e0f7c376c10330ec4
SHA1 (NetBSD-6.0/prep/INSTALL.txt) = 38f3b013b2b744dd7e21df3db8308dad808e45cd
SHA512 (NetBSD-6.0/prep/INSTALL.txt) = 2a42b7a45500710fd4df5f3883f8fef5aac4416c41a54c532ce358255541fbf3cd1e5ae03863f1721d845c82caab792d5c8d85feba2cd87225edc8f5606de585
RMD160 (NetBSD-6.0/prep/INSTALL.txt) = 71583c4ea3b65f1e17ccede286d6025fb6f281cf
MD5 (NetBSD-6.0/prep/INSTALL.txt) = 15bc48222a898edd8de0a70f559c4a4e
SHA1 (NetBSD-6.0/rs6000/binary/kernel/netbsd-INSTALL.symbols.gz) = e7fddbe9a6fe76bfec3ec556acf83101b6b1b27d
SHA512 (NetBSD-6.0/rs6000/binary/kernel/netbsd-INSTALL.symbols.gz) = 7a96ad902221852bdbc290707be328b8374e69d7339165e8af54a65abc17707a1177c2534306ba28e066eaca2eb1cd9bd76fcceaffef911926dacd88c22e3edb
RMD160 (NetBSD-6.0/rs6000/binary/kernel/netbsd-INSTALL.symbols.gz) = 3f5d25d432d08fa0458c087a3928d3370e60d080
MD5 (NetBSD-6.0/rs6000/binary/kernel/netbsd-INSTALL.symbols.gz) = f7adcf576fff9a965730d8cb07b677c3
SHA1 (NetBSD-6.0/rs6000/binary/kernel/netbsd-GENERIC.gz) = 3786662f4c42751a4ad0475d5b7674ba8ecf6113
SHA512 (NetBSD-6.0/rs6000/binary/kernel/netbsd-GENERIC.gz) = d0ecc5a21218668d185cc74f70ccd066dd08c39ca5e092966623857e7e0e0a9e0d350a055e8f54ed96a046e970ac6e8fbefe3aca12f8cbb238a1240858353dfb
RMD160 (NetBSD-6.0/rs6000/binary/kernel/netbsd-GENERIC.gz) = edfd5c086e4207543b08fa1125d137a436a17caa
MD5 (NetBSD-6.0/rs6000/binary/kernel/netbsd-GENERIC.gz) = d2909133467c780a0364052df6f5a906
SHA1 (NetBSD-6.0/rs6000/binary/kernel/netbsd-INSTALL.gz) = 0588aeaa722ae0065f1e5d92187e088c75000be7
SHA512 (NetBSD-6.0/rs6000/binary/kernel/netbsd-INSTALL.gz) = 53b6d6603b831ffbb5016ab99f61a20fad9900bbd76b18cd33608c499ad1d2b7a07dd3d0ea386bc27f1f9a8fe68db48691e128186cb4a810c82eca63e8e04e8c
RMD160 (NetBSD-6.0/rs6000/binary/kernel/netbsd-INSTALL.gz) = 4bbcd396759fe15f6052ec4ebf5b1ab0d1780a77
MD5 (NetBSD-6.0/rs6000/binary/kernel/netbsd-INSTALL.gz) = 2a0d673dd3b76fbdef31325f905bf6f2
SHA1 (NetBSD-6.0/rs6000/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/rs6000/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/rs6000/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/rs6000/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/rs6000/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/rs6000/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/rs6000/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/rs6000/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/rs6000/binary/sets/kern-GENERIC.tgz) = 953706a28ce3b7a195c738f805b6ecf878555e93
SHA512 (NetBSD-6.0/rs6000/binary/sets/kern-GENERIC.tgz) = a699aa0db691fd82015721bf9aab0b201574798731a795112cbe358a61f7fc3b4b1b0540abd25fc52b4c44692b81c4fe400dc6f90315d97563c32c1e2ef12a27
RMD160 (NetBSD-6.0/rs6000/binary/sets/kern-GENERIC.tgz) = 697ddaf6582c2712860f4c5abd80537b50da0059
MD5 (NetBSD-6.0/rs6000/binary/sets/kern-GENERIC.tgz) = 6d7ea95cdb07cb265eda9f09ca484ac9
SHA1 (NetBSD-6.0/rs6000/binary/sets/base.tgz) = c973cd9af8b007e754b99ead0517ab7bed7d3fa3
SHA512 (NetBSD-6.0/rs6000/binary/sets/base.tgz) = a82112dc5f53f010f6c259a543502b6a0d90259169233b2b043786cfa677ace8ed63f648dd738cc08730d6ac35233a2b1b60cd606222f399546b5f0a61488656
RMD160 (NetBSD-6.0/rs6000/binary/sets/base.tgz) = 0e84db039067d0ee045bdae56557595dfeb51935
MD5 (NetBSD-6.0/rs6000/binary/sets/base.tgz) = af2472e60e90aafe55028b419b776975
SHA1 (NetBSD-6.0/rs6000/binary/sets/comp.tgz) = 2dcac8a8c80d565b5373277deec9a67aee293370
SHA512 (NetBSD-6.0/rs6000/binary/sets/comp.tgz) = b6b94000d67080ab3ff62768c3afa85ee4a63a0ee62440648bb8c1728c6e4df9b72cc13927e012d9eee0a73792f2ce5e22d6c9d86eaa7c4a6789643177b44371
RMD160 (NetBSD-6.0/rs6000/binary/sets/comp.tgz) = fc6b3a3b6bb9613f29c50451190dba5fc7884df2
MD5 (NetBSD-6.0/rs6000/binary/sets/comp.tgz) = e7c6fcfdf054fef00feececd660789ca
SHA1 (NetBSD-6.0/rs6000/binary/sets/etc.tgz) = 302c5ba7b8efb24364748a665f5cfb824f6068b5
SHA512 (NetBSD-6.0/rs6000/binary/sets/etc.tgz) = 1ab1027b3082336f54fa330dd4d21924a42a4f76602d35bb2108229b745b305356c5f944247d0b9bfda63f66318a8ad6525eaf5af42cb3a8d0c3cf6e8b4d45b7
RMD160 (NetBSD-6.0/rs6000/binary/sets/etc.tgz) = 21418d6a16eb9e779b5e6d86cc411c9ab531453b
MD5 (NetBSD-6.0/rs6000/binary/sets/etc.tgz) = 695e526865e9cf18c91a35647fd75bf8
SHA1 (NetBSD-6.0/rs6000/binary/sets/games.tgz) = 0a7cc1be35944ade7ef1c617e0bb5f1a60a891e5
SHA512 (NetBSD-6.0/rs6000/binary/sets/games.tgz) = dacf177fc2affd7189a7cb7b166bf0b3f6b1a92cf1358f1dad2c5e364d28a7183f841befea024435b0f125cf4008741c26f98e5860ea3872d7a8c36e0e699430
RMD160 (NetBSD-6.0/rs6000/binary/sets/games.tgz) = 2620719e3f39dc3137ebb8f47fcbe482d77ff5e7
MD5 (NetBSD-6.0/rs6000/binary/sets/games.tgz) = 175c85d188b3c9b4922c4aaa7adc3dc3
SHA1 (NetBSD-6.0/rs6000/binary/sets/modules.tgz) = a710b61251bc52c73e64a0254489fd7ca7fb766b
SHA512 (NetBSD-6.0/rs6000/binary/sets/modules.tgz) = a39d57fa5428c4734bf521afdd1356df4213980a9180b4bb28731b5ff7c22b606b2b2189cfba9741cc98fa049ae01a26516c1096969f2b0b17d24e1d1112eb94
RMD160 (NetBSD-6.0/rs6000/binary/sets/modules.tgz) = ebbd79b609edee28f4c8331d5dfef0e510f67d9e
MD5 (NetBSD-6.0/rs6000/binary/sets/modules.tgz) = 825a589361390bbf9d409497e022b2ef
SHA1 (NetBSD-6.0/rs6000/binary/sets/tests.tgz) = a0eb45c7b71306f6cacafd52bf8f7f06b9d5fabc
SHA512 (NetBSD-6.0/rs6000/binary/sets/tests.tgz) = 4a95458b6a260485c6fa3d136abc9c76c6764f870d30bbf4f7c78c86233241366f84bc5cf7d7be0a3ccf49afd7f5c95e0f732d4164d6f32a01f953ac51a529de
RMD160 (NetBSD-6.0/rs6000/binary/sets/tests.tgz) = 67ec4f342cf6d25a56f4b0b62675b12df01ae445
MD5 (NetBSD-6.0/rs6000/binary/sets/tests.tgz) = 1ccae2c93acbafe1ffc9dc329c8a73de
SHA1 (NetBSD-6.0/rs6000/binary/sets/text.tgz) = ec4ec898eb442d2f8d86aaca0ce5490c7dc1722b
SHA512 (NetBSD-6.0/rs6000/binary/sets/text.tgz) = cbf687bd4d20115e4f993b08ed41dde158b39a0133af78503c41b3240e7f7aeef3a015594eeb423d6a8c7d74549646e06170eb7dee86d76919163083c81a27d9
RMD160 (NetBSD-6.0/rs6000/binary/sets/text.tgz) = e638a4bf115fc0fa2e4ef7d5399d79597762b94d
MD5 (NetBSD-6.0/rs6000/binary/sets/text.tgz) = 6e84e62deb9c45636b7047e88a6b1dce
SHA1 (NetBSD-6.0/rs6000/binary/sets/xbase.tgz) = 143ea7f8f94f52bf946db4dbf3fccb963f0006fd
SHA512 (NetBSD-6.0/rs6000/binary/sets/xbase.tgz) = 66442666aa2e6b9139ae8acf690e1c671edb34160977f786332b6f18c87d8be537160b4d8a114b48660a09cdb349890514b9cd6c3cf12a2f5c3e49e6ef070ed0
RMD160 (NetBSD-6.0/rs6000/binary/sets/xbase.tgz) = f28a5be274784cb9b35ca7c9a452e6fb1758562f
MD5 (NetBSD-6.0/rs6000/binary/sets/xbase.tgz) = ba48685231270484c0e79da384a77a28
SHA1 (NetBSD-6.0/rs6000/binary/sets/xcomp.tgz) = 1b1dda90e8376c39a124f39cb8f09824e1d88599
SHA512 (NetBSD-6.0/rs6000/binary/sets/xcomp.tgz) = c53a3e10df7ec1ddc2687ebf299a37f0b721f83df64797b64968bf1ebf2cdf7becd0ea8505a6a77864e9119c8817aa78deccac0d198b9c815f750a248fdbd5ba
RMD160 (NetBSD-6.0/rs6000/binary/sets/xcomp.tgz) = 4d57950bc653c7030f7ccae430f5b00812f106c2
MD5 (NetBSD-6.0/rs6000/binary/sets/xcomp.tgz) = 0cbd6372a9d8cc2488e23708a105f563
SHA1 (NetBSD-6.0/rs6000/binary/sets/xetc.tgz) = a1c99d1c17918fad811cf27b08625a0b1461f583
SHA512 (NetBSD-6.0/rs6000/binary/sets/xetc.tgz) = 567bf3cde35f4e54ad9875625c64895d28a5c8accbfb0d5339ca523a9192c312db7ee3c44f9fde3fb5a765839630d282fa59fbe483f5deed16811bb82d8bb3be
RMD160 (NetBSD-6.0/rs6000/binary/sets/xetc.tgz) = 263747682247c8c38b367970ff103f92d92b0fae
MD5 (NetBSD-6.0/rs6000/binary/sets/xetc.tgz) = e69bc3b1b38e787aff9d1df386ac63e6
SHA1 (NetBSD-6.0/rs6000/binary/sets/xfont.tgz) = ebe1c8e3f81f06dc9a816b1ff31240517e9b102d
SHA512 (NetBSD-6.0/rs6000/binary/sets/xfont.tgz) = 11f0db11eacf0ee00ecafa091c0f9cee84402dcf44a4467d809355d464eb698ff805acfc3da5752b167a2398a1f1080c93e3ca1ed8f08a6129ffe4e4a2402da1
RMD160 (NetBSD-6.0/rs6000/binary/sets/xfont.tgz) = 7997ed31cdf3895e1fc12b024314c1b72d94be89
MD5 (NetBSD-6.0/rs6000/binary/sets/xfont.tgz) = 9e4241f680bc0b0303a0aa682b61b1c9
SHA1 (NetBSD-6.0/rs6000/binary/sets/xserver.tgz) = e0f7bcff652a034a14b70eb608f9d48a21cff87e
SHA512 (NetBSD-6.0/rs6000/binary/sets/xserver.tgz) = 9838dce48efcde1f60ff163b1c5da7a19d414c730d97d8db7ed19ac77a52f27f2294ba24de75f15fd4d11a30a97830ca2f68b4311379505208025b2ea4913644
RMD160 (NetBSD-6.0/rs6000/binary/sets/xserver.tgz) = 9072c3c3519475c71ea42925c529ce5e323e9661
MD5 (NetBSD-6.0/rs6000/binary/sets/xserver.tgz) = 997077c1eb372bf091c82bf06f96f7b6
SHA1 (NetBSD-6.0/rs6000/installation/bootfs/boot.fs.gz) = 56e0afc45b285ce022bd561db40dcda0eb2fdca4
SHA512 (NetBSD-6.0/rs6000/installation/bootfs/boot.fs.gz) = c88fa6848032305418c173e9bca58e76545523b762e5ef4f65bc84f1f820e5813a37dcd256af8b0352f5d6f6e1c700d888f5633d1ce58664cbc5886114aa5332
RMD160 (NetBSD-6.0/rs6000/installation/bootfs/boot.fs.gz) = c575e56387f6956fd086bfeb5f85fed7a9fb4c27
MD5 (NetBSD-6.0/rs6000/installation/bootfs/boot.fs.gz) = d47804a5227b204d2bea50e243a749ae
SHA1 (NetBSD-6.0/rs6000/INSTALL.html) = 0f891193771ae2bbe138549727d4df583dc39a94
SHA512 (NetBSD-6.0/rs6000/INSTALL.html) = 84b1ae87d5befd27db8ed74276b95a91228bf60a05ecf610adacbddc7905cebbfa3909eb6fe8bed0c8f31bb92112fad398f9d4594204057892a538c16861551e
RMD160 (NetBSD-6.0/rs6000/INSTALL.html) = ca2e4dce100c234133c9da46ffb498d84e0115d3
MD5 (NetBSD-6.0/rs6000/INSTALL.html) = 031cc1f873f6c8d6f6472b991915701e
SHA1 (NetBSD-6.0/rs6000/INSTALL.more) = d86963437bc21fb0eb944b4562b79aefcec5c511
SHA512 (NetBSD-6.0/rs6000/INSTALL.more) = 130cf1977b54ee32332c7c1a2f35e9c19c088753ad8755ce48965aebc1cc19cd89f37f073749ee5f8b84223a8e8b3fc4a68993e0b22d35663c5850c419a0c84e
RMD160 (NetBSD-6.0/rs6000/INSTALL.more) = 7e35c4d92d61591cf32e7fc4f272734817b60eb8
MD5 (NetBSD-6.0/rs6000/INSTALL.more) = 4bb3c03008a5857e2c1f7b30cd76f10e
SHA1 (NetBSD-6.0/rs6000/INSTALL.ps) = 52af7c1bd8dc0f193c6ebaad3320a0d9b9c82497
SHA512 (NetBSD-6.0/rs6000/INSTALL.ps) = 44c13947238205228eecd13b48642f68179745ae8441ed58db4a3e7103c431a86c86d9ca88d0f71c656f1f0d54e2ebed8b395dbe1a9a437ccac79f91df7ff56b
RMD160 (NetBSD-6.0/rs6000/INSTALL.ps) = 545bc23ee542bce73f526bfb4184cc2b9314b0f4
MD5 (NetBSD-6.0/rs6000/INSTALL.ps) = 06416001febd58f722bad81cc2a585d8
SHA1 (NetBSD-6.0/rs6000/INSTALL.txt) = cfc54ae2bb8482e56b8b0e130285a78532b9772f
SHA512 (NetBSD-6.0/rs6000/INSTALL.txt) = 9fe957ce5896aa3ca77bf2b62a785904d22b21250e1ed6a18ca0b0e68268ba76f314948e052a0bbea78abe78e95a2d5ca51bf70b7cde2d935ab802a1e276b888
RMD160 (NetBSD-6.0/rs6000/INSTALL.txt) = aa823cc8082e6b246dc65bce012fec305a7d3ef4
MD5 (NetBSD-6.0/rs6000/INSTALL.txt) = b47464cee8c76f0b619c0a70acca22ca
SHA1 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL.symbols.gz) = d09224835f9948e426f67e0008e9e40de1a39635
SHA512 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL.symbols.gz) = b130c4fc02e1783188dcb4b5a3f8fc88f4294cd331d3beddb039364aa8557c61cf050ccaf520fbbd8badeb018e1c676b589d6eff6a17189ae8edd3dc7d6f94d6
RMD160 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL.symbols.gz) = cb79e2871b79e109038c8b7e62ec213695e9aff1
MD5 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL.symbols.gz) = 5f38dd4085224ecbd6ecae5e07a92e1d
SHA1 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-GENERIC.gz) = c315b6db09ef1d0161ad647352d9983b72da2ba9
SHA512 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-GENERIC.gz) = 27bf14f92d278fe7b4174cb265862fbd24c96a006f99e5d241f52fee928fdcaed13528f058d6e5cf78656ab7e0e17e974394a2dff8ab37664689ac8eeb457c61
RMD160 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-GENERIC.gz) = 4aa32ba67c19f07cc67a13835b3773b2bb8178d4
MD5 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-GENERIC.gz) = 87054c981c0647f65a1aa363c3ad3074
SHA1 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL.gz) = b43de7e1f292d5d1e050668babd148e3310cbe81
SHA512 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL.gz) = 0c591bfbaab13c5c0d03a50e0914a97444f87d9513ea48a96fa84e05ea1bed95a63a7ae3cf77c5c71f79cf774f1d1eef2b866c6125173e97027882bb37c38565
RMD160 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL.gz) = 3bc37d3a7d32ba229944341349eee1d7498487c5
MD5 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL.gz) = 1812ae09823196a8493b490db6d35829
SHA1 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL_KURO.symbols.gz) = 81f74ee4373718b36152d797c812097688ce515e
SHA512 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL_KURO.symbols.gz) = e26f69d2d4df43224e12a922b2d1602678db53a8f7f5823776047ed0f1106e6ffcbaf6d88a4101a75bd3b298737971a99d82b806f78e70a7039dc1e8c027a0cb
RMD160 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL_KURO.symbols.gz) = 32e0b167f67afd79e83b8b5da906e7eb9822640b
MD5 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL_KURO.symbols.gz) = 31c633e475629988bf3addc50252b158
SHA1 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL_KURO.gz) = b857802c9d1b447b451a50601e7592b6c72208a7
SHA512 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL_KURO.gz) = ef8318f98fc351491b88805dfb1150beb5bac352b18504bbe4756fb9fc54b064ee442999e08fa2a0b82306e717dd100a8a45e02814c5a03d168f33d05c34bd1c
RMD160 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL_KURO.gz) = 4add6bd73eeaf1c25a9d732adad0e24304e39b39
MD5 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-INSTALL_KURO.gz) = 61015da4f88490480f8553e59599dbd2
SHA1 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-KUROBOX.gz) = a2b339d5cb7d614c94341cd9dc8928597275fdbf
SHA512 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-KUROBOX.gz) = 44b9aa3d91de2f2e0d1c3ec42000182ad97fde311c2bebed17f21fa2165e9f2d574a2fff2330d9f9c2534e595e78bf1eb25e605158e62bf9c65fa449d4d426ef
RMD160 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-KUROBOX.gz) = 8548f8853cf766e38388c6f16da063334cb4a8a1
MD5 (NetBSD-6.0/sandpoint/binary/kernel/netbsd-KUROBOX.gz) = 9b8e45e5e31d018a614b2f9f2ec67f95
SHA1 (NetBSD-6.0/sandpoint/binary/sets/games.tgz) = fbf69640b362df2446d7414ac3a13db100c289eb
SHA512 (NetBSD-6.0/sandpoint/binary/sets/games.tgz) = d86b41d79aea4204ccf45306937a5ec45629e37873a5956c0280fd1d54cf4a808542d097f9d978fd3277791554ce8a6fb4d0ccd46e30210549d5cce1145bb499
RMD160 (NetBSD-6.0/sandpoint/binary/sets/games.tgz) = fd2aed1eb51f4db522fb5163e8fde4c040b55d30
MD5 (NetBSD-6.0/sandpoint/binary/sets/games.tgz) = 997ae5fb42196a787df89873be321e47
SHA1 (NetBSD-6.0/sandpoint/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/sandpoint/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/sandpoint/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/sandpoint/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/sandpoint/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/sandpoint/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/sandpoint/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/sandpoint/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/sandpoint/binary/sets/xbase.tgz) = b7c8146146ebc15191c1bf53f51f8f980fe8c816
SHA512 (NetBSD-6.0/sandpoint/binary/sets/xbase.tgz) = 091f5d00414a9ad478811bfb0e953f2a204869bc5e310790886a59c11953974cf7068cd9a96c8823856ab2687ffa2ac31e97494f47338bfe31d80217382c59f1
RMD160 (NetBSD-6.0/sandpoint/binary/sets/xbase.tgz) = 4bd1bfdc7c1e1619de63fb83276c21039b945357
MD5 (NetBSD-6.0/sandpoint/binary/sets/xbase.tgz) = e621806afa0fcc9cdcef0f9b3c7dfcbd
SHA1 (NetBSD-6.0/sandpoint/binary/sets/xcomp.tgz) = 960dd33b61e19dec92096a54791aca1cdf56ce54
SHA512 (NetBSD-6.0/sandpoint/binary/sets/xcomp.tgz) = 111dd50041e4008aeadb898c3d08f9995c5c13a41ccc20993b4ad51c81d7f6ee3b4dab16158ac7436d085b80e4a643b236a9c97edd3e915125a1466154538420
RMD160 (NetBSD-6.0/sandpoint/binary/sets/xcomp.tgz) = 0cf8b9a8d1489a32ba0b5a277d90e429c40650fb
MD5 (NetBSD-6.0/sandpoint/binary/sets/xcomp.tgz) = 4ba4077ca2c0edf71813165fcf5321fb
SHA1 (NetBSD-6.0/sandpoint/binary/sets/xfont.tgz) = 4711fcbade0238d60b7b89cf0d7873b03d7ed1f7
SHA512 (NetBSD-6.0/sandpoint/binary/sets/xfont.tgz) = c29130f55f536f602253a7f92f073ebf6ba02baea09f8011530bbf7e03ae96594334f0c60b29a8bceaffc3eb8b152db47548e38a3c8a14abc5fe35bc5d1d7d93
RMD160 (NetBSD-6.0/sandpoint/binary/sets/xfont.tgz) = 3b01af0f8e34367988d559fe0fbb0acbbafb0b2b
MD5 (NetBSD-6.0/sandpoint/binary/sets/xfont.tgz) = 6ba15ed12217b726d9129db8e942a048
SHA1 (NetBSD-6.0/sandpoint/binary/sets/xserver.tgz) = 05aac7e617e0cac05a88062fef29bb459da7a1d1
SHA512 (NetBSD-6.0/sandpoint/binary/sets/xserver.tgz) = 19fb5af5d6afd78839a22e5c0694a492047b5200c569cbf58c676ff32ec250b21edcfe2ed00b65092fdac374d1ab9eaaed6113c69bc03009cf458aedc6014913
RMD160 (NetBSD-6.0/sandpoint/binary/sets/xserver.tgz) = 774d2a1072b8d122790ad189a6d2ada2ad99e7cf
MD5 (NetBSD-6.0/sandpoint/binary/sets/xserver.tgz) = 3daf1477cadf3593e9ba45e3703fe8a5
SHA1 (NetBSD-6.0/sandpoint/binary/sets/kern-GENERIC.tgz) = 3120fc2c7bad39d27640482bb99222862238a9bd
SHA512 (NetBSD-6.0/sandpoint/binary/sets/kern-GENERIC.tgz) = 38c1992332ac419fa5e630965bf0bac621fbea752cdec3c147bafdff55bf8b24343913cec8e645b3170cddb071ed501781a47048be08d03c21154c7cf0e2ff6c
RMD160 (NetBSD-6.0/sandpoint/binary/sets/kern-GENERIC.tgz) = 3964f06ff1e895fe07b28c6758713c82cf0eec72
MD5 (NetBSD-6.0/sandpoint/binary/sets/kern-GENERIC.tgz) = 1ff2a1c2e3cb2d7f2fd9a270d9cae445
SHA1 (NetBSD-6.0/sandpoint/binary/sets/base.tgz) = 9e8d3d9c03cbd6db0b99c34641dab20f5835b6bb
SHA512 (NetBSD-6.0/sandpoint/binary/sets/base.tgz) = 4eb8d796b52589cc4132777dbfeb5295ef44bbe83e1e7c804540d39b98dbf66d7b0805a8e00dcab835309c9a69b3ca518288e0992e53b773d588339da9bbe0ac
RMD160 (NetBSD-6.0/sandpoint/binary/sets/base.tgz) = 48e26330269decea0489889a395af45b39ad55ff
MD5 (NetBSD-6.0/sandpoint/binary/sets/base.tgz) = 2f7c4bf56ff06bf89e9c5d49b1154805
SHA1 (NetBSD-6.0/sandpoint/binary/sets/comp.tgz) = 8f217e53da02ee1bf5445f76ccccd7262aaad680
SHA512 (NetBSD-6.0/sandpoint/binary/sets/comp.tgz) = 71a6e2221f054534477100908bb3ae686e88eeed96bc7d28920b39863d10881e6951d39011f65415d50c2cadc78874582a1d7f0505c29fbef25800003310febb
RMD160 (NetBSD-6.0/sandpoint/binary/sets/comp.tgz) = 8507052499e85b2c4d6fd985f3d0ddc099732443
MD5 (NetBSD-6.0/sandpoint/binary/sets/comp.tgz) = cb3a0690211b22d68a64e941711cc03e
SHA1 (NetBSD-6.0/sandpoint/binary/sets/etc.tgz) = 6b44f637dd5d01718b446128da95e0bb37c1682a
SHA512 (NetBSD-6.0/sandpoint/binary/sets/etc.tgz) = 8795d966d0da1b5419f50e577036167ab6e5de5589080ed0ee6903e0d3b59ae9bfd2395d7951b2b38a025acc39daf302a3d22212e0a6ba94344889d7f99d9d57
RMD160 (NetBSD-6.0/sandpoint/binary/sets/etc.tgz) = df0e4d773d5a180fba5b21b73cb6aabdaec11487
MD5 (NetBSD-6.0/sandpoint/binary/sets/etc.tgz) = c8ec3910d995970ea51634c4f126474a
SHA1 (NetBSD-6.0/sandpoint/binary/sets/kern-KUROBOX.tgz) = b4c7f1bc4058463ebb759d84f0ed8a5613cb3c44
SHA512 (NetBSD-6.0/sandpoint/binary/sets/kern-KUROBOX.tgz) = 50df0ca65f4f7ed454cc4176eb26b63df738d658536d63dfcacf8ea6def0cf3f4778d686b19ff4f5b32dd07e862f9b0d753ac34c93eedc21f47bd27a0ab5a7df
RMD160 (NetBSD-6.0/sandpoint/binary/sets/kern-KUROBOX.tgz) = a86947596f1a0b658f0cf664ddd67c05385efb56
MD5 (NetBSD-6.0/sandpoint/binary/sets/kern-KUROBOX.tgz) = 30f2e8b4824b9cedf289c6cdd46df34e
SHA1 (NetBSD-6.0/sandpoint/binary/sets/modules.tgz) = 7c4b66eea61f3cd7bebee7421f78ddabdce86b26
SHA512 (NetBSD-6.0/sandpoint/binary/sets/modules.tgz) = db987ae1a97bcaec2085bec2e33c49a53dedad2499e91db24570b50b6f83cd3387e2be0fd6f23dd0611f676133c5fb524c06a4c42c864b41acd74b88e1a2eb34
RMD160 (NetBSD-6.0/sandpoint/binary/sets/modules.tgz) = 24ff5d8cba57c1fd9aac448c121d8b5446c3d7aa
MD5 (NetBSD-6.0/sandpoint/binary/sets/modules.tgz) = 380e4c6e51bf10cf6fd605b12332badf
SHA1 (NetBSD-6.0/sandpoint/binary/sets/tests.tgz) = 45a3f3ff64bc425bfd332e018456252756b0b3af
SHA512 (NetBSD-6.0/sandpoint/binary/sets/tests.tgz) = 37a7200f8d06849b2eb8e82a19284edcba07a3445ffa40725f288d09eb4e6dba18856b97c64b24373c3c191343c64df2bb2cd5ea8b1367ccae239cb1db9755b8
RMD160 (NetBSD-6.0/sandpoint/binary/sets/tests.tgz) = ff0eaa5ae045658f2e34b26ee39f41fc03f40589
MD5 (NetBSD-6.0/sandpoint/binary/sets/tests.tgz) = 19128cd2482ccb19f4486a6090b89d41
SHA1 (NetBSD-6.0/sandpoint/binary/sets/text.tgz) = 8f4f8a5d09b63477d2a051e2d628b512363ae893
SHA512 (NetBSD-6.0/sandpoint/binary/sets/text.tgz) = e24825ec69e7b195c384e0fc83bd56a39598721ee30bde5dafc433e24fbedc2e5555a628ec1df0427cff6691f11040e244abd2f8f518421cdf53a94aca7752c6
RMD160 (NetBSD-6.0/sandpoint/binary/sets/text.tgz) = 264071e66052088a2a9c93b7d030331aa64dc522
MD5 (NetBSD-6.0/sandpoint/binary/sets/text.tgz) = 3a05d3c0c3a0074852621cc38b0afe82
SHA1 (NetBSD-6.0/sandpoint/binary/sets/xetc.tgz) = 80be15e90a6779956ecd9acecae91936d1417c2d
SHA512 (NetBSD-6.0/sandpoint/binary/sets/xetc.tgz) = fa0fd709723b8c3768f9797e34f30dc3cd8a16e3bf178beb883ccddf5041f196fe85828a32e3296c24f96a8a942927aac4587142705d7cd6ce8ffbcc6d4a2fda
RMD160 (NetBSD-6.0/sandpoint/binary/sets/xetc.tgz) = 02d7f0088ce5bda4f14607ca80de273d00a82137
MD5 (NetBSD-6.0/sandpoint/binary/sets/xetc.tgz) = e285972d692807df05e4f96ac34ef6f0
SHA1 (NetBSD-6.0/sandpoint/installation/altboot.bin) = 56c653262e9e672cd47728f06dbf1428a7a16604
SHA512 (NetBSD-6.0/sandpoint/installation/altboot.bin) = 0ff15a1e8f4fdff1c97f5539c69c4f40e1bbec0ceb6d3ae0df8e6e472fa25dd1b44398135cff0995ca21389819fba455b70a59e2af6fcc817781e09b1b11a3f5
RMD160 (NetBSD-6.0/sandpoint/installation/altboot.bin) = 7fd0c6c48de76b1fb04eb16a2cbd53531f67d78f
MD5 (NetBSD-6.0/sandpoint/installation/altboot.bin) = a2b44c7df7a4199c917096623275d178
SHA1 (NetBSD-6.0/sandpoint/installation/altboot) = 420331106c0a0f28c0cb27f39016852d65f47533
SHA512 (NetBSD-6.0/sandpoint/installation/altboot) = 7569cc668f087aa5a2af3ed0f231cdc8a0364cf7f64fcb06b08b3aab17e3bd295bcc8f2c8175f1efe73a1e7cca4ba459d35e65872fa8436d729697517dedfe29
RMD160 (NetBSD-6.0/sandpoint/installation/altboot) = ef26fc304a3f7bfa4717b03a4544207b44e2c382
MD5 (NetBSD-6.0/sandpoint/installation/altboot) = a5c881881c99da3587d7db293ce8b453
SHA1 (NetBSD-6.0/sandpoint/installation/altboot.img) = 9db4851a0e6349ca1ae912e60a329b8f250bb51b
SHA512 (NetBSD-6.0/sandpoint/installation/altboot.img) = 2bfb5c17dcfa6f1619f398a6f994f57be2f2a5e5bee0d2542bfe076f46d13567cf5519ccbea8bc97962122073bcb002309509caa06eab11014ee85766d813410
RMD160 (NetBSD-6.0/sandpoint/installation/altboot.img) = a140607e1b7ba43d76b22db739253f59234398bd
MD5 (NetBSD-6.0/sandpoint/installation/altboot.img) = f007fcb1d420ec5ab2f662013f78663a
SHA1 (NetBSD-6.0/sandpoint/INSTALL.html) = 87c1358b7dd97f4166607bc031ee8fd603d8a2c9
SHA512 (NetBSD-6.0/sandpoint/INSTALL.html) = b27b71a7b9fe4a37e2ffe20eb09588a3e3aa1337aa7936c064a35752b02ff8b1f560f053206db9ddc4b0a7a5f9163f221dba4690c54f3791fdccdf791b1893ed
RMD160 (NetBSD-6.0/sandpoint/INSTALL.html) = 790a76dee78dc46f35cf2b4f6078d699707d2adb
MD5 (NetBSD-6.0/sandpoint/INSTALL.html) = b1a454f7901f9b043c6993ba09ce7e20
SHA1 (NetBSD-6.0/sandpoint/INSTALL.more) = 60d0c93b382dfe57cc87c292f3cb53f66bc0ef7e
SHA512 (NetBSD-6.0/sandpoint/INSTALL.more) = 6b1ac0a4e00aee1bcf54d434b02b41102927bb125be551b0cf8cd25016104b3a03f3bf40506bc6076ed3af4a8128a81f733d523e5ccc3679cd8b51bba105cc27
RMD160 (NetBSD-6.0/sandpoint/INSTALL.more) = 46c78bd1d9df63eeed2f580faaabbac38f5757dd
MD5 (NetBSD-6.0/sandpoint/INSTALL.more) = f317efc426b562d069a20655d45d361a
SHA1 (NetBSD-6.0/sandpoint/INSTALL.ps) = f67cac326fc01aa5d0e47013afb174ce0962a46b
SHA512 (NetBSD-6.0/sandpoint/INSTALL.ps) = ab10cc2d28735af2a34ab85a4c84d7da346375a964a65ba9a328cd93b4b55c6badba306bd7093be08b662695d8ad82e96c4def014a7bac3f711767c9b741c33f
RMD160 (NetBSD-6.0/sandpoint/INSTALL.ps) = 5db8cd9595e21ecf5132a9bfb2574aaa01260515
MD5 (NetBSD-6.0/sandpoint/INSTALL.ps) = c36aec246e63cd311c87bdd6ff54f28a
SHA1 (NetBSD-6.0/sandpoint/INSTALL.txt) = 545b47f56ca866c03078d60d49d3276d4d000648
SHA512 (NetBSD-6.0/sandpoint/INSTALL.txt) = d7a925733a465005a83a915d9c58ded20149a6118013493b687c7b0ef2e816d8340631828ce1ec7ca13400c0d7a2dbe86c4fed363baf84ba288cb57a47384f89
RMD160 (NetBSD-6.0/sandpoint/INSTALL.txt) = 1a30cf24c59bd8394d8748102ceb9027d81d66d1
MD5 (NetBSD-6.0/sandpoint/INSTALL.txt) = 25fb074abc8d6bd5c9aec25dcf7c3667
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/kernel/netbsd-GENERIC.gz) = f23f0dec597709dddd3a0ee1d355f7beb9110b46
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/kernel/netbsd-GENERIC.gz) = c07b4b1360396b00184b77c19fda5ddb4463eef2fddf2c9ba43cbc4e7b0548e79999928db4f662046f73e282051bef9912f497c91b3757d60a1a11a520c3703f
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/kernel/netbsd-GENERIC.gz) = 22b4ec375839d277a7cddcfdf9df0a122a9cba19
MD5 (NetBSD-6.0/sbmips-mipseb/binary/kernel/netbsd-GENERIC.gz) = b4ee3e5e49f3bdf6fed5e7f9028fe2f1
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/games.tgz) = f56d3e5181634f6a3d7908dcb002ebcb1c5aaff0
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/games.tgz) = 0cff50ea13e34fe338edf70da6a28af061580feb4cb0773828adefa631ebd471a96568a3b30ea0f13f3bb116856a1b6aad8a5a52443346f52e4907f4caabc5f1
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/games.tgz) = 1538910578dbb8eee2931200fac0ccda44758e5f
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/games.tgz) = 2bd5bd0ce72df5cf92767b8c3f7cc208
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/xbase.tgz) = d81c7fe75a90f1b2111a066e78f1f2b3e8edcd8c
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/xbase.tgz) = c241391fcde4a685a0a2af6f843844ac2e918024c447a2b5ea2cf5300f69acc42fdadf53d3b0348f10723b786bcc0f420e68fe36228375bf3cb5140de41eb40f
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/xbase.tgz) = 2fbf60b64c0b7ac2be72ebf998d9d7390fc40705
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/xbase.tgz) = cd23ca0f6b359c4d695d4494a147d40f
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/xcomp.tgz) = c8f5ff5ea9f9bbe231f92d8b7f3bfcbd2897069f
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/xcomp.tgz) = c7590075195019078b68ec5f87e36479ae2e7b7749c523df4ef21c9d881c343cda9f5c2bba897dd313329a76fe5c18e920d92827878d98ce08a7dc776352cd02
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/xcomp.tgz) = bc60e267cb6e11b4a7305f8b7074084e179e0ff5
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/xcomp.tgz) = 8e58ee51667def3bd015f79769c0d932
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/xfont.tgz) = c551bd13179b9508c2e3a7e19cec323e8ba21844
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/xfont.tgz) = 764527f2f3f4686759fcae3fd093f4e00f34e1249994126221a6f9afe73d4cf8ddfbe8b8311798da2ed38c52466a44bb26ea08ad5aab1c08eebcc9019b4223f7
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/xfont.tgz) = b3a365bedf8b4a20f3920d0e33b9a42fce288953
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/xfont.tgz) = adac8939da213a0195cc1d351f979199
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/xserver.tgz) = 686f6e6d722c9676351ab058736f7077b4ebd2f6
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/xserver.tgz) = 5c818f22edbeac7325228af99d6da6e44b5eb659c7c9fcc6b332071e97a0a490081eda7f9e0f22d6a3f3a589ae34179b434751a6191db5c664683d7510aecedc
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/xserver.tgz) = 0e9f066095b505c7f062bf877147189a7a2f0649
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/xserver.tgz) = cdd2e5127838437df40af161e0c8398f
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/kern-GENERIC.tgz) = 3dee2d103b30c27015f41f351b45247d1f1067af
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/kern-GENERIC.tgz) = d9393724341c9d8036646fc768c21cb510aa4915555460753caded9c1f7a5355501d801e72ffb12e3ac1d25a57797b3b9b8c38787f7ae42906c6dbfecd0bf384
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/kern-GENERIC.tgz) = 9e22ac41f53032ad9194b1b0002c56c13211cf65
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/kern-GENERIC.tgz) = 6b9b0560a05738c70d580a25e0c5f203
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/base.tgz) = 61ec72dd34b2103b7b2f3228ee7766133a089c29
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/base.tgz) = 9aa13fa9008c18633f949ee74c5aaa5fa1d58d59459e48cf5f161d3f11e133a0cdaf7caa941a4e435076644ea799d7c9e73274267b18c3401866fdae5444f491
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/base.tgz) = ebc31ee01aa8cf119d20f5e084d536bcc610623b
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/base.tgz) = 828ed9e45442d989a4a5233b6523d82e
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/comp.tgz) = 705bd29c423e517e231920389f853990b8103339
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/comp.tgz) = 33d7feebe4ce7a3f8a4317c49f10f51e69e40374a9291c6b9e657f3e965d533702513b92dba0cba318a786f33fb5f275237be13870c093dc229d2c826a6b15ae
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/comp.tgz) = ea75f4ad366964a45cc5c8aac35cee5a56460d6a
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/comp.tgz) = e37746a52cc63d8d2992aef55484bb81
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/etc.tgz) = 9b4b768dc9d999330aa8fe804d604738fbabf650
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/etc.tgz) = 6e56a5122f29741aa02b651671d5ecdaef0adac94f52c8d550ab8a17352d3fa6755a87c13bd707d1a7f4259fef1c77e4ec85699cabb96bac5c47b82f2fdb6086
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/etc.tgz) = c2c3fbfe163cc38ff774413c1be35f0fee0b0c3f
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/etc.tgz) = 3d6d678755b01ec623ed215f7e1bbbc6
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/modules.tgz) = de0c98fc8b4bb81299edb50e3f99f222ac383dcf
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/modules.tgz) = c279717a8c4a6c66f3dff4e22d72466cf7b2c82070bba336bce5e659dc520c8c1fa1ef80f6f3154b665e0fb76dc2745ac8cf3996389209966f3976c36dc591c2
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/modules.tgz) = 80479a2131441999c8d251d822e00800e4f69b9c
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/modules.tgz) = cef9345e5430560c5955adee76e537c5
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/tests.tgz) = 92c135698d2030f4b378f18fc991c9f4efbcbf1a
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/tests.tgz) = 7fde9b55e820f169c4d1820d5b3f213f460c81bf892df6bff0ccf47ee73054c651c7277d956af77eb6f3d75648c7916be72d08dc3970601dc872662f009e9117
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/tests.tgz) = 38049cb62625ec093f26a53f6fdad2bce0b5a2ee
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/tests.tgz) = dfc74f3ea392e2c963257cb48926147c
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/text.tgz) = f8ddb7ac8fe765ad3760b3c00516f209f31ba211
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/text.tgz) = 9e8f6eb5da98c7c88d96d539e5379bf98f0d6726b84c5d4c2e1236d515d3cc31a4e1deedf1beab8335084f0c8d2795e8bef3391feff024d654877cd105254e03
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/text.tgz) = fc5441996f0f1650b41b80cdeacd377cec93c8ca
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/text.tgz) = 22ad03bdad15b336ac4f73bc31c29d78
SHA1 (NetBSD-6.0/sbmips-mipseb/binary/sets/xetc.tgz) = f89ac267632b07b88fa020f09c395a6766a2b9b1
SHA512 (NetBSD-6.0/sbmips-mipseb/binary/sets/xetc.tgz) = 59325ee391ab258e507fad56e2f38e5062c2ca15599c71ed06bd0373dfb109d61605dad949434d521d5ab9ea6139ee099919d485cb6d680d20b1bffc6881fd9a
RMD160 (NetBSD-6.0/sbmips-mipseb/binary/sets/xetc.tgz) = 6e805117acfbb8c9af9eb6f105b15c11447faaa8
MD5 (NetBSD-6.0/sbmips-mipseb/binary/sets/xetc.tgz) = 3e113c15bdfceb0a896ebfcee6e75172
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/kernel/netbsd-GENERIC.gz) = 017c9fbd7101f94f65ae9f0a41a1a028a8bd3435
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/kernel/netbsd-GENERIC.gz) = 45e2aa91e69a575f02a48dd776cc7401f37e60905ffc986634f53829aabc87d2dfc52840fc92821a3611b61e728f415ac44ab19f73db85786962df64d87b60c6
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/kernel/netbsd-GENERIC.gz) = 56ea720eb521974e26c07d74f93fcebd6e67de91
MD5 (NetBSD-6.0/sbmips-mipsel/binary/kernel/netbsd-GENERIC.gz) = e5f16d9d0838695b251b809dd42e10b9
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/games.tgz) = 27d9f98fecb3495e51939df9f107e81e527af26f
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/games.tgz) = 46bb4e10313cfffaaf4f7fa2157474ef940f94bcc9f84b46f829c7615bbf45e4e01edee7b7efb84bc28561e0712c37cfc737f9c2a0d9a73b67c80c8aa0f3a9af
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/games.tgz) = 947feea90ec2698a65d555cb78a29a001b086589
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/games.tgz) = 355f00978fed8e8da6a251e591be5387
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/xbase.tgz) = 76468fcada5a9e26652aa9f0f9f3c94829a317e9
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/xbase.tgz) = b2adcfc2dedef30b21b885c8fc757d0479541744a280a9997fbeaa9c881009ab4b0019e0001a540a8eb6db5f84db87b365e420bb2d4270af209821bdb080369c
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/xbase.tgz) = 1e0074d03d9fbb3373e0a427b25cc7097bf6d3c8
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/xbase.tgz) = a8f29ec2e3768405942f97af3dd08ee7
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/xcomp.tgz) = d8ac89f3f6ff20abb0adabc49227afdb74aa2040
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/xcomp.tgz) = 4f03213fb061dae9421747209628d06ede6bbb71a7560a9e1a54796b907f1620146a33fdce52974227a21acedcc210b99e15c4fdd36432eba60c1041a92d8cee
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/xcomp.tgz) = 6be5e72b9b57c9bdf8ed8ea127e2a73830feeecb
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/xcomp.tgz) = 59a7ea141587417be38ab1f6056cc780
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/xetc.tgz) = b9f6b1a54ffab37b11f4c6e717b481cd087fb609
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/xetc.tgz) = 72f35e65f3a90967e250a2006d500619b3824f7f5c3d7494554f153ec5cda5b28c445f28c34a1e4463f40da5855e8d4257726411796967fe37763d90cf7fe461
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/xetc.tgz) = 1169c246f91189fd0c4ae86ac27a92f5c2edda3e
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/xetc.tgz) = 8f3c52cc5af1cafaf7b0d0f884a5b537
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/xfont.tgz) = d06848571034f3b7fc975082f1de37b912acb18c
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/xfont.tgz) = 66a1013c935bee5e7e5d24325a527522306bcded7ced63790f2e58831b3ed91176a1b9510b2159be63504d5167d48d534919927ffc6e225ae2bb2487f90a9ede
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/xfont.tgz) = c9502843a5d4dd7e996e8d158c519cf7f8575cbd
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/xfont.tgz) = e812899176eac2bf65a6d949ac3f3424
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/xserver.tgz) = ff805022f409202f3346c45475c7ec0f8187218e
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/xserver.tgz) = f2bb3810a5572b519980126a3d934f63b5ab5c2fcf9fb8166c556bd3cc14cfaa151d582086bb1506ac9ac7e21d4e70a5cd5c3faed96212d8a14a0b5bafdcc4c1
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/xserver.tgz) = d5ae336337cc77f7f55f542b3457918d6fc0efe8
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/xserver.tgz) = 92447a1560887d337d22e2deed0670a6
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/kern-GENERIC.tgz) = 8c185e51bf6d6c63d23ce62fd86e5f7bfb3d3845
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/kern-GENERIC.tgz) = 6f1a341b96a6890f1bfe68a588e74639caec8c320b72bac774f03d8f7e71d9198887b791e6528a10bd201bb14537e2f83a4e81aa402adc5bd57e4ada3c8022a9
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/kern-GENERIC.tgz) = 43be2a6c07b4b366d31b2b74ce7666d0337cb84c
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/kern-GENERIC.tgz) = 1ad137c19b7c2df8bcb8bfbe053466d6
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/base.tgz) = d0664b8cdc36b3e6b0381b6a9de9c7556bd383ef
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/base.tgz) = 737f1bed2c71e1473483bec24e7e48be28fca93e0ccb7e124661569d1960fa99bfdfc7f8e86bd3a61f744593346ef767b1ae2ed0c1e2fb9c5b421dd2b8125dcb
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/base.tgz) = 130c2d1408fad9b0c2e8c612b6d720a6c92674b6
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/base.tgz) = c2623377960d1a7fe4db1ffff70882f2
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/comp.tgz) = ed63cf25383b8110ece9e4cb4752fe762b636cd5
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/comp.tgz) = f699747ae04f024400653aa5bf5be624e4cd0bd92c9086ee8f89ce9327c5adb27e7cc53f1a85904b3e0f2cbbb49e45a3098fc803e131338a3100a56f1628218e
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/comp.tgz) = ca975dc6b36da0208cc2a79c346e74fe367ba833
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/comp.tgz) = 063da88eb27db9b012142fbec1707b34
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/etc.tgz) = a6430055b2ccc0d418f11c4eb5628d091998d4f3
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/etc.tgz) = 51a2c3cbb860e042c3ecc234e1e0645489a5546ccf0c87a2925f32c01dfb3fbc1dd088b4bc947cb27c731b9645f234e13fa9e805665be34aa3fe9318c51d12a7
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/etc.tgz) = a7c1913e10b8fdc44a191159b5b6278728bd8a0a
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/etc.tgz) = 690495c3b0cbba76f83085422161887c
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/modules.tgz) = 37c4fabb3a3edebf715572a4c7deca5c2f63508f
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/modules.tgz) = 2245821ee48c5a4a4c7ad378c2d726975b18a3f04b1e4cf04e78ee0c15855502eca0673b11420af837146c9003ec6c9899b815a37aef06632b9e8a00f4f95f0d
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/modules.tgz) = 2ae1af8e754811706b0d5633f2631a306eee664a
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/modules.tgz) = 3136834e1ac671cf35958f353e33d8c0
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/tests.tgz) = d3118b0c152c61b20933e5dc38272991b808fce0
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/tests.tgz) = ce817f1900de3b09a7b063abed50f10f78042f3000f1bb031963db7eb1666fb55b42ffcf446370b4ef2541ea86739e43c9bf0771aba86f5bcd62e6181975a62b
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/tests.tgz) = 6fe992cf685d12015118ce0faf06d77b241e10d6
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/tests.tgz) = d59a11116844dc6e6a754ab8676e30f3
SHA1 (NetBSD-6.0/sbmips-mipsel/binary/sets/text.tgz) = 82712c2413930b0726af1cc8e3d7f7183e789534
SHA512 (NetBSD-6.0/sbmips-mipsel/binary/sets/text.tgz) = 2681e2fd0973fa7573dd6a1746ba7c491956a82256a8522cbb65c267402ba700da71b0ce44c65269787385e596913ba1042eca4542e5751f55beaa2679d673d7
RMD160 (NetBSD-6.0/sbmips-mipsel/binary/sets/text.tgz) = aba3f473d85e3a8fd5ae41ca25c7433a22b11f36
MD5 (NetBSD-6.0/sbmips-mipsel/binary/sets/text.tgz) = 0b2dbc9b54843ba81289bc994b3fcf7a
SHA1 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.symbols.gz) = b1a8652e56663a63c6376ecf20ccf9f8f876a9e8
SHA512 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.symbols.gz) = 3e4a01eeec83c7da8725badd449c51dcc677203373d1ca0725a157a6f9ebee86fa8ca7d66e1936420569e7152fe9bf2089764b94132aa04d61e36661ea0d1b32
RMD160 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.symbols.gz) = ae01910e3901d7adf1b925f13b4520e46bce2917
MD5 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.symbols.gz) = 14b2804f6c6f16940d4bdda28c6426e1
SHA1 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP12.ecoff.gz) = 532b94eea6032bf2867a860f78973c95b3baf210
SHA512 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP12.ecoff.gz) = a3017e2f1bc7d008d45d53f3733ed573c18de5d44c2c7a1e969f2938119a5d6996c1eb913e7ef6b28cddfb2e6da4e4ddb6aaddbb9caa6c43b90d046f2ba396db
RMD160 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP12.ecoff.gz) = bfbcc81a415674512ffdaa8ad7180d4499fd410d
MD5 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP12.ecoff.gz) = 01742badd90ac599c850ef5d7f98de49
SHA1 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP12.gz) = 1040bc406f9543efe4f2a6c198d136ef45d0e9f7
SHA512 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP12.gz) = 1bb8fe9e495603393f69f25ac5fe57d8f26e64cc1573af4088d3e2929d8043ecff721d38b0d4f3f6b6934661df6f6f2790d1d506ced6f8f21b57c1f1554e79a2
RMD160 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP12.gz) = eece122ff5df495cfee6fafeba2fb086cce6623b
MD5 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP12.gz) = 8bc4a605a8cb06bfcf835aa4a5b16411
SHA1 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP2x.ecoff.gz) = 0fa0539d5bdaa751a71d3df4a573adb90d742cb3
SHA512 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP2x.ecoff.gz) = 4e94bbc422b322b6b250a8cdf31144cb97f6349a2cd075b8720497dbae36dc9f90d0b48da549bbfd3d7908bdf078cc95e300bde365df6e204008a0af85a5e7e6
RMD160 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP2x.ecoff.gz) = ec847458dd2c3b20dcd27d5bc14ced2601c998e5
MD5 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP2x.ecoff.gz) = e74b65c6a2f141174323dfe40cc0e730
SHA1 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP2x.gz) = 2c863ba399e4aad44ef7b76dc60bcedbc112d1dc
SHA512 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP2x.gz) = 27c329497ed9ac49c096071c6250b01135b590b9d2f5d79366f53409b4abf9daeca15f323a35521b9a79f4264d5784e1d49dec269cc51ee0df52a686bf6fea88
RMD160 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP2x.gz) = 410dae1e4c4a5b3f2b9b000c413faa6ccedddd5e
MD5 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP2x.gz) = 0538c7187f0c5778a89a660749018ccc
SHA1 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP3x.gz) = 1968926bf27f4cb8f5fac47a22ed3327dfb52c43
SHA512 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP3x.gz) = 0d5d484b19505ebbebda2daff665c43c77aa9e087946daf198684f7cba944f11e2dde344930e99ef4a24c02c6fb83009115b45a0100b955b0dc8450c00fc2f38
RMD160 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP3x.gz) = 4194967cb2a542b818e0f4670e194e91c6b20635
MD5 (NetBSD-6.0/sgimips/binary/kernel/netbsd-GENERIC32_IP3x.gz) = 2d25467ddd2b88d4110211ab86f1c77b
SHA1 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.ecoff.gz) = 8bbc293222db73fb4be9015b07e05070c413de50
SHA512 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.ecoff.gz) = 7c0f743299c829bea945e1f927dce38ad44b07f012949532d1e0f73bf3f44bb5931f1772db34b8598cb46b1e8d522c1de29e537105e8f4f1c10057bebaabe33c
RMD160 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.ecoff.gz) = 4c67683e70c5c31659196601d15fc98de34dbd10
MD5 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.ecoff.gz) = e3cb87e4586bd5e1ec2602cf7767495f
SHA1 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.gz) = 66a33c0bed49bb9e1dd95cb3c79f944ee56fea31
SHA512 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.gz) = f3dfc078ef7e21741758b7e3ea76af80772a1d3218b0178ce825814bd16a895115df7122b394c58f63af6b2128e57a95064ae3fc44a5a3e56d8a03327bf3174f
RMD160 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.gz) = c60e7094e3801d730f699b7a67c4403ced988c92
MD5 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP2x.gz) = 2c406bbd7ec78cc4c97dcdc48ee476d1
SHA1 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP3x.symbols.gz) = af7722c589f536235fb6d25f27231a6a3eacd3c4
SHA512 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP3x.symbols.gz) = f7b6acf595cbb8e7a54b34b458c39c384ea0e3ce44fcd142c7f71cb08551b56624969872fde0f90614e40582bc45fa3e41be31de0de954ce43fcd34375cd5d04
RMD160 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP3x.symbols.gz) = 3b853ba98c7c4016aeec80103318c1147aa52ac6
MD5 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP3x.symbols.gz) = 001d3d08526a1d35fe6a2d66baa59ff9
SHA1 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP3x.gz) = 9fef8eeac9eedbd6ce7744b13c0e14efc835c1fd
SHA512 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP3x.gz) = d4af0b7ffc0596e209e19161bce14533610fd414350b761a58873c16858c3f44f2bc20135f7f14ffc9458dbfc029dfebb998e19f02176c9718f6a6406ea5b262
RMD160 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP3x.gz) = 420fe89a1bf2f952113c22d083512456f98738db
MD5 (NetBSD-6.0/sgimips/binary/kernel/netbsd-INSTALL32_IP3x.gz) = 7081086c1dca701908d84d74c29fc81e
SHA1 (NetBSD-6.0/sgimips/binary/sets/games.tgz) = f56d3e5181634f6a3d7908dcb002ebcb1c5aaff0
SHA512 (NetBSD-6.0/sgimips/binary/sets/games.tgz) = 0cff50ea13e34fe338edf70da6a28af061580feb4cb0773828adefa631ebd471a96568a3b30ea0f13f3bb116856a1b6aad8a5a52443346f52e4907f4caabc5f1
RMD160 (NetBSD-6.0/sgimips/binary/sets/games.tgz) = 1538910578dbb8eee2931200fac0ccda44758e5f
MD5 (NetBSD-6.0/sgimips/binary/sets/games.tgz) = 2bd5bd0ce72df5cf92767b8c3f7cc208
SHA1 (NetBSD-6.0/sgimips/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/sgimips/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/sgimips/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/sgimips/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/sgimips/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/sgimips/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/sgimips/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/sgimips/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/sgimips/binary/sets/base.tgz) = d02636d2688644bf3230c10bf072000a44cd2f5a
SHA512 (NetBSD-6.0/sgimips/binary/sets/base.tgz) = abc4fb8ba2047eb2c99070b0b7a22e29a071c44ef8ce3d896969648dfb6ead55d396e8a7c0716e34e45f51565c6e29011d0e35cb914cab4691c8672a29f24fbc
RMD160 (NetBSD-6.0/sgimips/binary/sets/base.tgz) = f91ad6342293731972f28cb6286e6926b3de10e0
MD5 (NetBSD-6.0/sgimips/binary/sets/base.tgz) = ff1da3a02630406282d40b75528def3e
SHA1 (NetBSD-6.0/sgimips/binary/sets/comp.tgz) = 27ba59fd7d5e90f91446a151f804437f8c249e8b
SHA512 (NetBSD-6.0/sgimips/binary/sets/comp.tgz) = f119dbb5c22cab49fdd8f0dec15407fa520d1d8eb02cb4550bd01f88d49378b4999e0041f6da8d4ee2bfa4712590a5345a184ead5a4b014d80c79c459bb34e3e
RMD160 (NetBSD-6.0/sgimips/binary/sets/comp.tgz) = 9cf77a31c5419a857ccf19a148eaf510a4d83aed
MD5 (NetBSD-6.0/sgimips/binary/sets/comp.tgz) = 2a2b72f824b9953fe580cb284a37c96c
SHA1 (NetBSD-6.0/sgimips/binary/sets/etc.tgz) = 6140d0c55fb88e07b633da0abfd0d5a345eb908b
SHA512 (NetBSD-6.0/sgimips/binary/sets/etc.tgz) = da25730f805732e0d4b90612dd9b5236f6fed96c75f022f4d1053495d897f7355b5d75b95f010b041795d6331285a2ab0ada3f34175b611147b295e697d54eb9
RMD160 (NetBSD-6.0/sgimips/binary/sets/etc.tgz) = 79973e6b9ad3344bb4e64ed23a1ba9bcd16fb8a1
MD5 (NetBSD-6.0/sgimips/binary/sets/etc.tgz) = c96ede97a5ce906a1fb580fc1dabbf94
SHA1 (NetBSD-6.0/sgimips/binary/sets/modules.tgz) = a206108d50c0e8a138571adf8a674334de685263
SHA512 (NetBSD-6.0/sgimips/binary/sets/modules.tgz) = bfb96bc27db44cc540d3ca143099e2574ca9a42eeea307eb3b835b624b52ebc14fead54754fb3d95916e6f4c2fecafcaedebaf28a32ec0c1e1b798caf9a7e44c
RMD160 (NetBSD-6.0/sgimips/binary/sets/modules.tgz) = 9114bd11fa6c211d72726f5e88458396165a6fe4
MD5 (NetBSD-6.0/sgimips/binary/sets/modules.tgz) = 8f298135900ed72db173ffa002c4103b
SHA1 (NetBSD-6.0/sgimips/binary/sets/tests.tgz) = b10f0716cc4a705193fe7485de5b6123153a40a0
SHA512 (NetBSD-6.0/sgimips/binary/sets/tests.tgz) = 79061c6367c51f6b0769e7ad724d440d055cefcc6ce1bf1730e602b2080af16be5a7f74a3c57391567dfb62f49588721bee5704d71039527c4269add0bd8420f
RMD160 (NetBSD-6.0/sgimips/binary/sets/tests.tgz) = 869f46c688b7d65e6f93dcf06e4d406fea799082
MD5 (NetBSD-6.0/sgimips/binary/sets/tests.tgz) = b02332c28d31942e96e416d4dbf2f6d8
SHA1 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP12.tgz) = 18e386c5766aa3391d5d29a7d55ae1eb3b69af2b
SHA512 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP12.tgz) = 8caad11aefe7fe71d346fbe3950fcaeb08a3454793e24f12c83b91c3078ef4e15f44c94cf47aa785e8b975f7403f0de2be38fa3ad3a48798e70efcdded5eed0d
RMD160 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP12.tgz) = 8e7531be01ac24bab8b898224ae33df0e84d637c
MD5 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP12.tgz) = 2f1a7076fdb72698d28a9d8032b79943
SHA1 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP2x.tgz) = 4097c0cfecd7e8766e4542ffd936ac18fb9ed32f
SHA512 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP2x.tgz) = 9595785d477c04e57b81e2dd71957dfc3309fad013b7210d93d2501e64c308b57af9a22581ac3e61d8f3ec91aa62d49bb01ed5ab1eaf90557f41c0d4d3c3096f
RMD160 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP2x.tgz) = 2479e9e6495e16753c6c0e3e8dead0c49095c245
MD5 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP2x.tgz) = 8661ec66f60bce4cb75af3d6756580c5
SHA1 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP3x.tgz) = a5a29b61f5eba57e0bcdbd6af9ecfb9a4485a90b
SHA512 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP3x.tgz) = 4a5c4ebea7ddfad9bf8f3536212eeed17f22dbd74a38b7f48bd7963c6c7c4f29da10ec6454053a441db830cd5c18a3a342b6ad22f5fbb9d97104b683b94685cd
RMD160 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP3x.tgz) = 3bae9be1ed83ab2eeb80905c1e1bcfc59a4b6f7b
MD5 (NetBSD-6.0/sgimips/binary/sets/kern-GENERIC32_IP3x.tgz) = e402bda170a73a296a773b851afa1a3b
SHA1 (NetBSD-6.0/sgimips/binary/sets/text.tgz) = 11418c012b904e8a99e91493d36fdca3f6e6f006
SHA512 (NetBSD-6.0/sgimips/binary/sets/text.tgz) = 501a9e78e9310aeb3f0c766464b5453e2718f23b5d3290428ad775977cd2a42b5444b91d7b1707ae9991b58af14f376d9d93cebdc54c74ea8d20ed06a7e0387d
RMD160 (NetBSD-6.0/sgimips/binary/sets/text.tgz) = 272adf1723ad8ab137b985946e64ca3b27055199
MD5 (NetBSD-6.0/sgimips/binary/sets/text.tgz) = 0f6d77a90f9714d54517c4b5660150fa
SHA1 (NetBSD-6.0/sgimips/binary/sets/xbase.tgz) = 0e912b46017d4afca665692603e483e95d627af5
SHA512 (NetBSD-6.0/sgimips/binary/sets/xbase.tgz) = 54c6bb297c09510019a0e21816018b7f8abbecd3e9dab5f5dd48d1cf35493b0cb0ec5ca01f1bbc9066fa1b964cf550119c1678a70d80d4471d9df73627655bb6
RMD160 (NetBSD-6.0/sgimips/binary/sets/xbase.tgz) = d8b979b953547c386cf66488db96771f3923cc2f
MD5 (NetBSD-6.0/sgimips/binary/sets/xbase.tgz) = c04dfa38a56e78bf0d075dd4d333a869
SHA1 (NetBSD-6.0/sgimips/binary/sets/xcomp.tgz) = d764bc540d10ebb805e8c976e8b300d9c6a108e0
SHA512 (NetBSD-6.0/sgimips/binary/sets/xcomp.tgz) = a6795d24491875468bdabf17698ee483c0aaeaae64772667083049c7f2b2ae82943277cd41fe870c085ae7498f6546d971362d77e62f6150f40aa23f6b92d732
RMD160 (NetBSD-6.0/sgimips/binary/sets/xcomp.tgz) = 99f0489f322b26960b51a53e24fd03faccccc512
MD5 (NetBSD-6.0/sgimips/binary/sets/xcomp.tgz) = d6e8029f965bcade40d027658ea70e14
SHA1 (NetBSD-6.0/sgimips/binary/sets/xetc.tgz) = bcbb9c22987ffa1ec653920c5013030fb974d73c
SHA512 (NetBSD-6.0/sgimips/binary/sets/xetc.tgz) = 7e2ad6276a043e98c194a03fc38f3754193c0f75bfbb79cc855bf08d7336d01763d7df89edb3ac63056a403d08e7e21b43a1b4468c34cb2969024bf5894988e2
RMD160 (NetBSD-6.0/sgimips/binary/sets/xetc.tgz) = 35c9e251920fa14e32a68f6d811ec8ef81a233b1
MD5 (NetBSD-6.0/sgimips/binary/sets/xetc.tgz) = baf2f338baf24e78da645312b33eab7b
SHA1 (NetBSD-6.0/sgimips/binary/sets/xfont.tgz) = 43b823091c3c5b340538ecf7cbd17be47e6d5692
SHA512 (NetBSD-6.0/sgimips/binary/sets/xfont.tgz) = 63a18f59b0c2ef7eab2ba7526fe2815155e476426d34e21e06d2c00fc872cc01bc241704971524c9c6410f420650a7d69914205a45a1b232413e9ccbff2d6b93
RMD160 (NetBSD-6.0/sgimips/binary/sets/xfont.tgz) = 3cee7affdad412ed072636553915fab9f0d23ca2
MD5 (NetBSD-6.0/sgimips/binary/sets/xfont.tgz) = 0be724043d9312d60c89901238982903
SHA1 (NetBSD-6.0/sgimips/binary/sets/xserver.tgz) = f604963ee4c952665400bd5742f6ac3925903291
SHA512 (NetBSD-6.0/sgimips/binary/sets/xserver.tgz) = 2f5d36f2f728aeed88a3a2e8da36acb3117018fe92f97769babe2b37670c0ad9cce319bded4a4c78d7859c12df4124fdb2faed33abaae1b4cea20098582ff99e
RMD160 (NetBSD-6.0/sgimips/binary/sets/xserver.tgz) = c18f29c81d11389a45b5c8a6226ec1e7c074582c
MD5 (NetBSD-6.0/sgimips/binary/sets/xserver.tgz) = 41d79391ddbccae7930228219015083b
SHA1 (NetBSD-6.0/sgimips/installation/diskimage/diskimage.gz) = 5acd23fd19ce75e45f071d5de6396668b89ea681
SHA512 (NetBSD-6.0/sgimips/installation/diskimage/diskimage.gz) = 007da5ebfe4af418c4f696956cb768575b656dd40297cb39dea7b03146bf48aa22fe9f288325737b6904f5e08b89c79d60a375dce4d885fc83f3bb85595507b2
RMD160 (NetBSD-6.0/sgimips/installation/diskimage/diskimage.gz) = d4fabae9c31423c0557937fd07cc796f48f392fa
MD5 (NetBSD-6.0/sgimips/installation/diskimage/diskimage.gz) = 8575ffc5db9771c5aa948ab437c80e1a
SHA1 (NetBSD-6.0/sgimips/installation/netboot/diskimage.tgz) = 075e8acd73646b2b89509714ba7ca7a001cb72a1
SHA512 (NetBSD-6.0/sgimips/installation/netboot/diskimage.tgz) = e0dc8dfb79ee590decb7c315fd4b8cf7a55a2981a3752aeddc25afd83a79fd72d6852f9823ab857077cfe40c9619dc8b537ab1227c42441b43b2fb9129a419a3
RMD160 (NetBSD-6.0/sgimips/installation/netboot/diskimage.tgz) = 4b4d8d21878765a902c28673fa2ff2fe62ccbc55
MD5 (NetBSD-6.0/sgimips/installation/netboot/diskimage.tgz) = 250ed8b78bb6bc799d92f3e82d62003a
SHA1 (NetBSD-6.0/sgimips/INSTALL.html) = 57dd965aef89b4c8190886a37ab3fff06021b8eb
SHA512 (NetBSD-6.0/sgimips/INSTALL.html) = c1e5774651642601daaedef68c0337dbf8d634abddc9eaf8d33d38358076f49867df6644d0c8c7e8e730c2d37a2a35af9c7c03dbd817f5b4eae5d7383e900190
RMD160 (NetBSD-6.0/sgimips/INSTALL.html) = a5c94dcca469b47cb176426a8cbd4c0e2df210e5
MD5 (NetBSD-6.0/sgimips/INSTALL.html) = 7c4d8de6801d46c1dac589fc304590a5
SHA1 (NetBSD-6.0/sgimips/INSTALL.more) = b9c2017ba93cbb87b7d0a88fd6912eb81a9b7ed7
SHA512 (NetBSD-6.0/sgimips/INSTALL.more) = ca9a739d8c56336ebbfffb856904a4753e8948cf8c1a4da057f212cffaba7493c07467bcd0608923d1dc3deb76ceac989cc556cd025380ad1f80109a46aa6746
RMD160 (NetBSD-6.0/sgimips/INSTALL.more) = 0e733569f0cd6acc3de8e7b3acdac89953e554d7
MD5 (NetBSD-6.0/sgimips/INSTALL.more) = a43c8c4a3aea3585e8de329c8efef129
SHA1 (NetBSD-6.0/sgimips/INSTALL.ps) = e61b0904bc556853037f1a9564486e0962a8b8e2
SHA512 (NetBSD-6.0/sgimips/INSTALL.ps) = 122ebadac3919533845714c90f87c59a65722d6e79f50bf231866f2917e6d4ebcb4a8b5da8e4ac9acbe4e333e841611ef80665b7e69427db90d4843e5f630c8b
RMD160 (NetBSD-6.0/sgimips/INSTALL.ps) = efbc102832be0cee24ce3a5810999c5ea9c4b18e
MD5 (NetBSD-6.0/sgimips/INSTALL.ps) = e86216add5a2ed163e13a17c6ff291f2
SHA1 (NetBSD-6.0/sgimips/INSTALL.txt) = 104c1ef873cfbe774ba106e437168053539b65b1
SHA512 (NetBSD-6.0/sgimips/INSTALL.txt) = 8b5ad1ef07604c0e8ac5ec2c7f43a4c483610a7ad7ac4c15a13c4151652500a21b203569bcc9ecbb53e035f8b90e1c7e3fc81d985fe3817685957865fb4924cf
RMD160 (NetBSD-6.0/sgimips/INSTALL.txt) = 7cc2ca6174d6b5e2ed23f8db3d6b62bb02c41637
MD5 (NetBSD-6.0/sgimips/INSTALL.txt) = 99bca5bfeb2112666aba79a96379899a
SHA1 (NetBSD-6.0/shared/ALL/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/shared/ALL/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/shared/ALL/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/shared/ALL/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/shared/ALL/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/shared/ALL/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/shared/ALL/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/shared/ALL/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/shared/arm/games.tgz) = e1d2bb091d8202962133475bc83edb55230b9008
SHA512 (NetBSD-6.0/shared/arm/games.tgz) = a147fe6d0419d53d5a428802d0bb6c9fa8a84a76c96d7d6d7c0d399e5af3a69240116463549a83d53387b0214ba0207796d7c10228d92a684fe7ff0ece3aec16
RMD160 (NetBSD-6.0/shared/arm/games.tgz) = 2be515c495548ba5135ff42a75a8ebbfc35ea776
MD5 (NetBSD-6.0/shared/arm/games.tgz) = 9059a6f2cd41cb58827709c9e9a1649b
SHA1 (NetBSD-6.0/shared/arm/xbase.tgz) = 875fd641f969ec00f854ca1334d084947450c740
SHA512 (NetBSD-6.0/shared/arm/xbase.tgz) = 75d0c9e9685b6b7bb706df7427c8b52f59c4fc4d0e725b02ee18024f75a546dc41de5f1a5d8552a8d2c84bdf87856e5857b0f12914801ca7b36b69536d617b4e
RMD160 (NetBSD-6.0/shared/arm/xbase.tgz) = 1a4d099d59b2bd882e99f1b2df1c7bf22b890d48
MD5 (NetBSD-6.0/shared/arm/xbase.tgz) = 5eb9c573c5a0c96701f268a3470a58b9
SHA1 (NetBSD-6.0/shared/arm/xcomp.tgz) = 8d7469a85b7433c89690cce74e6b3f4e66cf0647
SHA512 (NetBSD-6.0/shared/arm/xcomp.tgz) = 411304cf66e300a7f162128203099bb4d6eebc4a93ba18ed7caf79f30b31cc7ffcede85c4e1bad02c3f5f8a27d009fb71f3ce5862c1d8aa477f39a25c535fdaa
RMD160 (NetBSD-6.0/shared/arm/xcomp.tgz) = 822d26e0ee08b72ca8b991f0709a8c1bf0399208
MD5 (NetBSD-6.0/shared/arm/xcomp.tgz) = 5af361f0334d28119d703064278b3ed3
SHA1 (NetBSD-6.0/shared/arm/xetc.tgz) = 222d14c205f1018cc5a86eb03214d18cda2abb82
SHA512 (NetBSD-6.0/shared/arm/xetc.tgz) = fb82261996aa772fd8a9022eadebf899ee04ee70229ee4fe07758ca14937bdb34752c8218eb4c5b722b6208f719891493bbc141372fcfa503de83e0a61aa7387
RMD160 (NetBSD-6.0/shared/arm/xetc.tgz) = 511ebfe5076f361b7f7ca03a265d16052a16d259
MD5 (NetBSD-6.0/shared/arm/xetc.tgz) = 165cffbb6d2c79663198d80c82afac1e
SHA1 (NetBSD-6.0/shared/arm/xfont.tgz) = ce0e141beeffeb5b470478e9887e4127da85de3c
SHA512 (NetBSD-6.0/shared/arm/xfont.tgz) = 5d98354ff77103793d56af3192ab5f9afcde62208791976064ac33a149f994e0227dd5dd28ed544eeac5ea504e5212d5fb9e7d05e59e5a48a2fe4062e34e3106
RMD160 (NetBSD-6.0/shared/arm/xfont.tgz) = 4d09f5e82fec875ef0ea328809465e7a98f2ebfc
MD5 (NetBSD-6.0/shared/arm/xfont.tgz) = be26b3890261302420b09976a1d38144
SHA1 (NetBSD-6.0/shared/arm/xserver.tgz) = 4ccfef1d49179603286042088bec0218c6b2031d
SHA512 (NetBSD-6.0/shared/arm/xserver.tgz) = 809fc0a9065c1b9e7cd17aeb0d41cfd3158efd6c585556af01c4a3c93f0df3646a2ff77c6994449a703a408c0988fc1aa68cf34c7e297a3056d73ded57034c12
RMD160 (NetBSD-6.0/shared/arm/xserver.tgz) = 403604404261764c74f10fab774c4653fd20cc9b
MD5 (NetBSD-6.0/shared/arm/xserver.tgz) = 914f909b6878cef16a92765a8cbb52ca
SHA1 (NetBSD-6.0/shared/m68k/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/shared/m68k/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/shared/m68k/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/shared/m68k/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/shared/m68k/xbase.tgz) = ffc6dc889560371042dd12a5486a9a2c499b86a9
SHA512 (NetBSD-6.0/shared/m68k/xbase.tgz) = 9aa2ba4d703a13020acf8543d1efcde76b6f0f4b411be20f804709b3802e537be1e651eba469830c4fd4777a379ea9936200a3a3eb4b1a702c04bb81aafcc723
RMD160 (NetBSD-6.0/shared/m68k/xbase.tgz) = 3ecf53e7414ef859a47fe5afc108cf3bfa82fb1d
MD5 (NetBSD-6.0/shared/m68k/xbase.tgz) = a6af002ce158cf72eb5c6a7cde2c0989
SHA1 (NetBSD-6.0/shared/m68k/xcomp.tgz) = e13548b8974f5d55e98caf17b2b4bbbcf2430370
SHA512 (NetBSD-6.0/shared/m68k/xcomp.tgz) = e4b13c798df04ffef88e72100e2db5af400e5ae255cd7a1ca2907736b8138a59045b6f3bd8531b5753d0b87eadb1b5f801043a929bb25d1611e4be693c1da954
RMD160 (NetBSD-6.0/shared/m68k/xcomp.tgz) = 3b1f4e26ddd57e69afd9709553c1bf70d6e29ea0
MD5 (NetBSD-6.0/shared/m68k/xcomp.tgz) = 894bb9aa05b164d9923682165d708335
SHA1 (NetBSD-6.0/shared/m68k/xetc.tgz) = d2ca34bd668e108cbbfd4b45720823663339f8bd
SHA512 (NetBSD-6.0/shared/m68k/xetc.tgz) = 431ca465f300e7eb5d4dbd44c43e412a296fb4d3129217e752f86a720e8a8bdf736c9add05a45518a4f3e6cffa32381ef61e9186d360b68a5f8c64d87e385fb5
RMD160 (NetBSD-6.0/shared/m68k/xetc.tgz) = 4c5d664679856b02b694534cd6afdf20597e6d53
MD5 (NetBSD-6.0/shared/m68k/xetc.tgz) = 72bd67556ac14ea0ce147374af0ba624
SHA1 (NetBSD-6.0/shared/m68k/xfont.tgz) = 24ca4cb77a59246b1b05e0a91fed21c63259d12b
SHA512 (NetBSD-6.0/shared/m68k/xfont.tgz) = 8dc297fb6205a995767e68e77cc21169234dfb66a86bbcd2e1828fd32efeda0a25c9f2a9b1a39b97fe8d7b24f8cd5cf6160859af6be2259876f39ee4e76c405b
RMD160 (NetBSD-6.0/shared/m68k/xfont.tgz) = d2f134959940e6667aca185f01e388c05dc6bc85
MD5 (NetBSD-6.0/shared/m68k/xfont.tgz) = 3c96c1f0a13cd7d277ca35622ac9c058
SHA1 (NetBSD-6.0/shared/m68k/xserver.tgz) = 1d20fd039546a66b428b854c659ea5e2ee9f8114
SHA512 (NetBSD-6.0/shared/m68k/xserver.tgz) = 091d09049d0af5555feec02065234e281677aeb3ecbab5a19413d924a8b58dc7f64401ef68e68bc52d8b6eeb73e091d351c45805374b8bf276b0181415968561
RMD160 (NetBSD-6.0/shared/m68k/xserver.tgz) = feef5d52278d23637d7b8ba77639bfd96918e90e
MD5 (NetBSD-6.0/shared/m68k/xserver.tgz) = b2bd6057c374376388766fc8667eb9a9
SHA1 (NetBSD-6.0/shared/mipseb/games.tgz) = f56d3e5181634f6a3d7908dcb002ebcb1c5aaff0
SHA512 (NetBSD-6.0/shared/mipseb/games.tgz) = 0cff50ea13e34fe338edf70da6a28af061580feb4cb0773828adefa631ebd471a96568a3b30ea0f13f3bb116856a1b6aad8a5a52443346f52e4907f4caabc5f1
RMD160 (NetBSD-6.0/shared/mipseb/games.tgz) = 1538910578dbb8eee2931200fac0ccda44758e5f
MD5 (NetBSD-6.0/shared/mipseb/games.tgz) = 2bd5bd0ce72df5cf92767b8c3f7cc208
SHA1 (NetBSD-6.0/shared/mipseb/xbase.tgz) = d81c7fe75a90f1b2111a066e78f1f2b3e8edcd8c
SHA512 (NetBSD-6.0/shared/mipseb/xbase.tgz) = c241391fcde4a685a0a2af6f843844ac2e918024c447a2b5ea2cf5300f69acc42fdadf53d3b0348f10723b786bcc0f420e68fe36228375bf3cb5140de41eb40f
RMD160 (NetBSD-6.0/shared/mipseb/xbase.tgz) = 2fbf60b64c0b7ac2be72ebf998d9d7390fc40705
MD5 (NetBSD-6.0/shared/mipseb/xbase.tgz) = cd23ca0f6b359c4d695d4494a147d40f
SHA1 (NetBSD-6.0/shared/mipseb/xcomp.tgz) = c8f5ff5ea9f9bbe231f92d8b7f3bfcbd2897069f
SHA512 (NetBSD-6.0/shared/mipseb/xcomp.tgz) = c7590075195019078b68ec5f87e36479ae2e7b7749c523df4ef21c9d881c343cda9f5c2bba897dd313329a76fe5c18e920d92827878d98ce08a7dc776352cd02
RMD160 (NetBSD-6.0/shared/mipseb/xcomp.tgz) = bc60e267cb6e11b4a7305f8b7074084e179e0ff5
MD5 (NetBSD-6.0/shared/mipseb/xcomp.tgz) = 8e58ee51667def3bd015f79769c0d932
SHA1 (NetBSD-6.0/shared/mipseb/xfont.tgz) = c551bd13179b9508c2e3a7e19cec323e8ba21844
SHA512 (NetBSD-6.0/shared/mipseb/xfont.tgz) = 764527f2f3f4686759fcae3fd093f4e00f34e1249994126221a6f9afe73d4cf8ddfbe8b8311798da2ed38c52466a44bb26ea08ad5aab1c08eebcc9019b4223f7
RMD160 (NetBSD-6.0/shared/mipseb/xfont.tgz) = b3a365bedf8b4a20f3920d0e33b9a42fce288953
MD5 (NetBSD-6.0/shared/mipseb/xfont.tgz) = adac8939da213a0195cc1d351f979199
SHA1 (NetBSD-6.0/shared/mipseb/xserver.tgz) = 686f6e6d722c9676351ab058736f7077b4ebd2f6
SHA512 (NetBSD-6.0/shared/mipseb/xserver.tgz) = 5c818f22edbeac7325228af99d6da6e44b5eb659c7c9fcc6b332071e97a0a490081eda7f9e0f22d6a3f3a589ae34179b434751a6191db5c664683d7510aecedc
RMD160 (NetBSD-6.0/shared/mipseb/xserver.tgz) = 0e9f066095b505c7f062bf877147189a7a2f0649
MD5 (NetBSD-6.0/shared/mipseb/xserver.tgz) = cdd2e5127838437df40af161e0c8398f
SHA1 (NetBSD-6.0/shared/mipsel/games.tgz) = 27d9f98fecb3495e51939df9f107e81e527af26f
SHA512 (NetBSD-6.0/shared/mipsel/games.tgz) = 46bb4e10313cfffaaf4f7fa2157474ef940f94bcc9f84b46f829c7615bbf45e4e01edee7b7efb84bc28561e0712c37cfc737f9c2a0d9a73b67c80c8aa0f3a9af
RMD160 (NetBSD-6.0/shared/mipsel/games.tgz) = 947feea90ec2698a65d555cb78a29a001b086589
MD5 (NetBSD-6.0/shared/mipsel/games.tgz) = 355f00978fed8e8da6a251e591be5387
SHA1 (NetBSD-6.0/shared/mipsel/xbase.tgz) = 76468fcada5a9e26652aa9f0f9f3c94829a317e9
SHA512 (NetBSD-6.0/shared/mipsel/xbase.tgz) = b2adcfc2dedef30b21b885c8fc757d0479541744a280a9997fbeaa9c881009ab4b0019e0001a540a8eb6db5f84db87b365e420bb2d4270af209821bdb080369c
RMD160 (NetBSD-6.0/shared/mipsel/xbase.tgz) = 1e0074d03d9fbb3373e0a427b25cc7097bf6d3c8
MD5 (NetBSD-6.0/shared/mipsel/xbase.tgz) = a8f29ec2e3768405942f97af3dd08ee7
SHA1 (NetBSD-6.0/shared/mipsel/xcomp.tgz) = d8ac89f3f6ff20abb0adabc49227afdb74aa2040
SHA512 (NetBSD-6.0/shared/mipsel/xcomp.tgz) = 4f03213fb061dae9421747209628d06ede6bbb71a7560a9e1a54796b907f1620146a33fdce52974227a21acedcc210b99e15c4fdd36432eba60c1041a92d8cee
RMD160 (NetBSD-6.0/shared/mipsel/xcomp.tgz) = 6be5e72b9b57c9bdf8ed8ea127e2a73830feeecb
MD5 (NetBSD-6.0/shared/mipsel/xcomp.tgz) = 59a7ea141587417be38ab1f6056cc780
SHA1 (NetBSD-6.0/shared/mipsel/xetc.tgz) = b9f6b1a54ffab37b11f4c6e717b481cd087fb609
SHA512 (NetBSD-6.0/shared/mipsel/xetc.tgz) = 72f35e65f3a90967e250a2006d500619b3824f7f5c3d7494554f153ec5cda5b28c445f28c34a1e4463f40da5855e8d4257726411796967fe37763d90cf7fe461
RMD160 (NetBSD-6.0/shared/mipsel/xetc.tgz) = 1169c246f91189fd0c4ae86ac27a92f5c2edda3e
MD5 (NetBSD-6.0/shared/mipsel/xetc.tgz) = 8f3c52cc5af1cafaf7b0d0f884a5b537
SHA1 (NetBSD-6.0/shared/mipsel/xfont.tgz) = d06848571034f3b7fc975082f1de37b912acb18c
SHA512 (NetBSD-6.0/shared/mipsel/xfont.tgz) = 66a1013c935bee5e7e5d24325a527522306bcded7ced63790f2e58831b3ed91176a1b9510b2159be63504d5167d48d534919927ffc6e225ae2bb2487f90a9ede
RMD160 (NetBSD-6.0/shared/mipsel/xfont.tgz) = c9502843a5d4dd7e996e8d158c519cf7f8575cbd
MD5 (NetBSD-6.0/shared/mipsel/xfont.tgz) = e812899176eac2bf65a6d949ac3f3424
SHA1 (NetBSD-6.0/shared/mipsel/xserver.tgz) = ff805022f409202f3346c45475c7ec0f8187218e
SHA512 (NetBSD-6.0/shared/mipsel/xserver.tgz) = f2bb3810a5572b519980126a3d934f63b5ab5c2fcf9fb8166c556bd3cc14cfaa151d582086bb1506ac9ac7e21d4e70a5cd5c3faed96212d8a14a0b5bafdcc4c1
RMD160 (NetBSD-6.0/shared/mipsel/xserver.tgz) = d5ae336337cc77f7f55f542b3457918d6fc0efe8
MD5 (NetBSD-6.0/shared/mipsel/xserver.tgz) = 92447a1560887d337d22e2deed0670a6
SHA1 (NetBSD-6.0/shared/powerpc/games.tgz) = fbf69640b362df2446d7414ac3a13db100c289eb
SHA512 (NetBSD-6.0/shared/powerpc/games.tgz) = d86b41d79aea4204ccf45306937a5ec45629e37873a5956c0280fd1d54cf4a808542d097f9d978fd3277791554ce8a6fb4d0ccd46e30210549d5cce1145bb499
RMD160 (NetBSD-6.0/shared/powerpc/games.tgz) = fd2aed1eb51f4db522fb5163e8fde4c040b55d30
MD5 (NetBSD-6.0/shared/powerpc/games.tgz) = 997ae5fb42196a787df89873be321e47
SHA1 (NetBSD-6.0/shared/powerpc/xbase.tgz) = b7c8146146ebc15191c1bf53f51f8f980fe8c816
SHA512 (NetBSD-6.0/shared/powerpc/xbase.tgz) = 091f5d00414a9ad478811bfb0e953f2a204869bc5e310790886a59c11953974cf7068cd9a96c8823856ab2687ffa2ac31e97494f47338bfe31d80217382c59f1
RMD160 (NetBSD-6.0/shared/powerpc/xbase.tgz) = 4bd1bfdc7c1e1619de63fb83276c21039b945357
MD5 (NetBSD-6.0/shared/powerpc/xbase.tgz) = e621806afa0fcc9cdcef0f9b3c7dfcbd
SHA1 (NetBSD-6.0/shared/powerpc/xcomp.tgz) = 960dd33b61e19dec92096a54791aca1cdf56ce54
SHA512 (NetBSD-6.0/shared/powerpc/xcomp.tgz) = 111dd50041e4008aeadb898c3d08f9995c5c13a41ccc20993b4ad51c81d7f6ee3b4dab16158ac7436d085b80e4a643b236a9c97edd3e915125a1466154538420
RMD160 (NetBSD-6.0/shared/powerpc/xcomp.tgz) = 0cf8b9a8d1489a32ba0b5a277d90e429c40650fb
MD5 (NetBSD-6.0/shared/powerpc/xcomp.tgz) = 4ba4077ca2c0edf71813165fcf5321fb
SHA1 (NetBSD-6.0/shared/powerpc/xfont.tgz) = 4711fcbade0238d60b7b89cf0d7873b03d7ed1f7
SHA512 (NetBSD-6.0/shared/powerpc/xfont.tgz) = c29130f55f536f602253a7f92f073ebf6ba02baea09f8011530bbf7e03ae96594334f0c60b29a8bceaffc3eb8b152db47548e38a3c8a14abc5fe35bc5d1d7d93
RMD160 (NetBSD-6.0/shared/powerpc/xfont.tgz) = 3b01af0f8e34367988d559fe0fbb0acbbafb0b2b
MD5 (NetBSD-6.0/shared/powerpc/xfont.tgz) = 6ba15ed12217b726d9129db8e942a048
SHA1 (NetBSD-6.0/shared/powerpc/xserver.tgz) = 05aac7e617e0cac05a88062fef29bb459da7a1d1
SHA512 (NetBSD-6.0/shared/powerpc/xserver.tgz) = 19fb5af5d6afd78839a22e5c0694a492047b5200c569cbf58c676ff32ec250b21edcfe2ed00b65092fdac374d1ab9eaaed6113c69bc03009cf458aedc6014913
RMD160 (NetBSD-6.0/shared/powerpc/xserver.tgz) = 774d2a1072b8d122790ad189a6d2ada2ad99e7cf
MD5 (NetBSD-6.0/shared/powerpc/xserver.tgz) = 3daf1477cadf3593e9ba45e3703fe8a5
SHA1 (NetBSD-6.0/shared/sh3eb/games.tgz) = 37e15b0950aaaf1af36427ae5dd5866c1879b6b3
SHA512 (NetBSD-6.0/shared/sh3eb/games.tgz) = e674abb1b1966acbc581a8f8b3e37841f48656e66f3302077cbb2de443487912b99521336ec5dc096fa3c7769b2d7623e1d1720a079407ec1861d32159a72ebc
RMD160 (NetBSD-6.0/shared/sh3eb/games.tgz) = 36070071e8315d2352be1b3e563ce7ef7625d184
MD5 (NetBSD-6.0/shared/sh3eb/games.tgz) = 0016145b46957b95dd158b4441dfb010
SHA1 (NetBSD-6.0/shared/sh3eb/xbase.tgz) = 564bce6dd7c37b5e941e41d50f363c133fdf8f97
SHA512 (NetBSD-6.0/shared/sh3eb/xbase.tgz) = 23b3306254c789331542e63974f9d7cdc54d99ba2cd67cf6a1c97f09afa4f6cf6aa8798e37b7c38f4a55afd1572154f291fa1bf2e2a1fc87165fd6bd7f89fb80
RMD160 (NetBSD-6.0/shared/sh3eb/xbase.tgz) = 476fbd8301bec0795a01552728ada44204d2dbe6
MD5 (NetBSD-6.0/shared/sh3eb/xbase.tgz) = 9fbbbdb3acfa4ccd9f0b309ea0ed0700
SHA1 (NetBSD-6.0/shared/sh3eb/xcomp.tgz) = 526a278a13833073e6231e879abdec85400ca888
SHA512 (NetBSD-6.0/shared/sh3eb/xcomp.tgz) = d2081a7536e3d7fa27a363afd8f6eb5fc3de4606f11b4857be2f1df59934da9d3d16e0df3a6edf4cdf52c8f579e0e4c06f07de0959b5213c21be1a8fdc2ee134
RMD160 (NetBSD-6.0/shared/sh3eb/xcomp.tgz) = 4a767ff08e9bcf98f886849cabebd1300397256f
MD5 (NetBSD-6.0/shared/sh3eb/xcomp.tgz) = a23a2d8be9298497b4b6d39fda330bf4
SHA1 (NetBSD-6.0/shared/sh3eb/xetc.tgz) = 93a5cfbbfdc641f91ab363d4b65d1158d71c0cfc
SHA512 (NetBSD-6.0/shared/sh3eb/xetc.tgz) = 85a67734801e81e129a37b73ae62211459fb16d4e52a945ef21a4bb908dd7d1d775233956086dd3d59af45bcea24fbd6f54a2a9175756d242b57e07d10767548
RMD160 (NetBSD-6.0/shared/sh3eb/xetc.tgz) = 6fb055d0e1b6264bf1ec396236d903218832377a
MD5 (NetBSD-6.0/shared/sh3eb/xetc.tgz) = af954b56e624948145b91aa451d77392
SHA1 (NetBSD-6.0/shared/sh3eb/xfont.tgz) = 6139c05273b6feeecdfcd11cbe0da91256d12a11
SHA512 (NetBSD-6.0/shared/sh3eb/xfont.tgz) = bb849454ede88d39ca60d6c513c833c2b29ea548adedb9f76870839174a3daa838d38a6ad701c78bfc164e690c5ebda9c9d00d4c23ad816df9f1962313afdbf2
RMD160 (NetBSD-6.0/shared/sh3eb/xfont.tgz) = ea8f123f9f78b429b77ff15c924803da1da2068d
MD5 (NetBSD-6.0/shared/sh3eb/xfont.tgz) = a87cb6769c37ae20dbf0dd70cef93d11
SHA1 (NetBSD-6.0/shared/sh3eb/xserver.tgz) = 6cd4fdb0ff00ef5d89b47e68e538c86e395b2054
SHA512 (NetBSD-6.0/shared/sh3eb/xserver.tgz) = e3b575cab95e297a05a26a03eeff0cf843d665ebdfd103846ad544bef1d04f93aa20ceab09b1dfaae8f59d89ebb1cbef400b7521bdb41f7542f23156540aa4c0
RMD160 (NetBSD-6.0/shared/sh3eb/xserver.tgz) = 7e52b98276b4d097f7babe918e6768ae9d7945db
MD5 (NetBSD-6.0/shared/sh3eb/xserver.tgz) = a505ab4ce0801aabe21fe6038b63809f
SHA1 (NetBSD-6.0/shared/sh3el/games.tgz) = 2040b4af557a62f2ade5fc9fcfe4cab3df88c7e2
SHA512 (NetBSD-6.0/shared/sh3el/games.tgz) = 8b5071337dc844042537c24c75660e61c9709cd10cbbeab059a84da6d482260b73365ff3870f85060c130a291b6f02be174d288127ca62ce4d291a6982192317
RMD160 (NetBSD-6.0/shared/sh3el/games.tgz) = 28acfa5a131d52df7c157e2edca1dd0f20e774dc
MD5 (NetBSD-6.0/shared/sh3el/games.tgz) = d82c46dbc0021fd4694e464042936052
SHA1 (NetBSD-6.0/shared/sh3el/xcomp.tgz) = 53aa1e2a910e8eed09dc07f8b724804f53fe5e2f
SHA512 (NetBSD-6.0/shared/sh3el/xcomp.tgz) = 24645da85419528015cdaefd0c835741fa801392e795b363bdfc39558e75e7f384511c0668d12e2440114b7c543c7532aaec438048f08d35c54c9c7a748f20c0
RMD160 (NetBSD-6.0/shared/sh3el/xcomp.tgz) = 60a461ffc1f1562274110d65c72a541ee131cd21
MD5 (NetBSD-6.0/shared/sh3el/xcomp.tgz) = f1d2650a8d7a4ecd60fda8e1aaf8c09b
SHA1 (NetBSD-6.0/shared/sh3el/xetc.tgz) = 1636a6fe3e01f6d8a07a19cd204a6807c8caa4c7
SHA512 (NetBSD-6.0/shared/sh3el/xetc.tgz) = 9cea4ba4af287dd57e591d4269f2f2582010250316f323746ac56b12cefcf2590b93fbb0197c665f61b1df41dcca62d17fb6ad80d4b1feb8ac8265e91e15074f
RMD160 (NetBSD-6.0/shared/sh3el/xetc.tgz) = 5f68e3f96f7bb9ae07fccdb1d058e4b1307b4b29
MD5 (NetBSD-6.0/shared/sh3el/xetc.tgz) = da84d97230787283e215a4ee1c4053ad
SHA1 (NetBSD-6.0/shared/sh3el/xfont.tgz) = 97439ccf6840b18b1e9885f5504a401bf1a93650
SHA512 (NetBSD-6.0/shared/sh3el/xfont.tgz) = 3d15541f7d7ae2128f44355bf110415adfa4e5182829259d2c6f7a4217feff99e236d13c5bb7bbeb724ff1a09ac1f58f02bdd6eef528bc416ec3118db6fc7d45
RMD160 (NetBSD-6.0/shared/sh3el/xfont.tgz) = cdc9fcdb5fd5032803d5bba18db58f3ea6d62c85
MD5 (NetBSD-6.0/shared/sh3el/xfont.tgz) = a8466a0477c6c443eafe99e9d1c61131
SHA1 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.symbols.gz) = 4033fc14abf440fef2f4e6370dead899e4983721
SHA512 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.symbols.gz) = 17fb47f8b2731f8fe73376ab664040eefdc64690156e26acad0791871e64e5d401ea95f592a034eaf963ceebc919a6d54c06874c8be8141b049637cc55aabc84
RMD160 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.symbols.gz) = 9de247cfdaae7ee9022f6d09030b58665536dd2f
MD5 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.symbols.gz) = c89c4849f164a941bc9a17cb89ca4dc2
SHA1 (NetBSD-6.0/shark/binary/kernel/netbsd-GENERIC.aout.gz) = 98a104383326606b65c14160cf13b700c0d0becc
SHA512 (NetBSD-6.0/shark/binary/kernel/netbsd-GENERIC.aout.gz) = 60c184cc2a78026b2ae297623a540618e63a58962e578fa8ecb0e68c5e94e9c64fe910da565b6e9427e0e99aae311e8645e2083a4cd02373f08c514a29fa9bfc
RMD160 (NetBSD-6.0/shark/binary/kernel/netbsd-GENERIC.aout.gz) = c1569194e6d7d526911ecc813919db5103eba092
MD5 (NetBSD-6.0/shark/binary/kernel/netbsd-GENERIC.aout.gz) = 77f58eeb32ecae1a67c7821cfef3e1f1
SHA1 (NetBSD-6.0/shark/binary/kernel/netbsd-GENERIC.gz) = 3aec79cc4bd37f7777e326445a18e4d551f4dc23
SHA512 (NetBSD-6.0/shark/binary/kernel/netbsd-GENERIC.gz) = 46894446ed52b86e01d7fed75aa5285670d2d32a972be76a1019ea797583fff37423f4d9d40c29c394d123365b9a196fc7bb3824dc5952f6407b21198127cdba
RMD160 (NetBSD-6.0/shark/binary/kernel/netbsd-GENERIC.gz) = e34c8493497296b2d98dc0522d2d4d58b56a3546
MD5 (NetBSD-6.0/shark/binary/kernel/netbsd-GENERIC.gz) = 9ccf14a3d80b596f8e594c2e7db89224
SHA1 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.aout.gz) = 08fdda8c99f112770acb58f662e7dc7f92e211aa
SHA512 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.aout.gz) = eb7dd72ec967546883dbf9e50fc9bf770e8aced07020c03f93abdc0c9bd6b1a03ff320a3f5167f74caf732ff37495e7507628e0ea213ae8fbf209fbfc2ed2368
RMD160 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.aout.gz) = 97cd05778ad6cbef869bfd1ef9bda107db56b099
MD5 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.aout.gz) = f174a788515cbae3b497a13a1f27b68b
SHA1 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.gz) = 78fa421384bc331c7bf1194c90f45969e0b9f9ca
SHA512 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.gz) = c97fe5aa7c364929d749ee65dec8b6a7736da39e4d3366dfd8b17e67e8b5e56f9d37a0b8b4437c161602bfd265b486378ac272fc4e7917851b64faf8a9389370
RMD160 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.gz) = ef146283b89c96c7d68bef166e853a83df6a65ee
MD5 (NetBSD-6.0/shark/binary/kernel/netbsd-INSTALL.gz) = dda410ee10ec3e07d4b962e08e5895c6
SHA1 (NetBSD-6.0/shark/binary/sets/games.tgz) = e1d2bb091d8202962133475bc83edb55230b9008
SHA512 (NetBSD-6.0/shark/binary/sets/games.tgz) = a147fe6d0419d53d5a428802d0bb6c9fa8a84a76c96d7d6d7c0d399e5af3a69240116463549a83d53387b0214ba0207796d7c10228d92a684fe7ff0ece3aec16
RMD160 (NetBSD-6.0/shark/binary/sets/games.tgz) = 2be515c495548ba5135ff42a75a8ebbfc35ea776
MD5 (NetBSD-6.0/shark/binary/sets/games.tgz) = 9059a6f2cd41cb58827709c9e9a1649b
SHA1 (NetBSD-6.0/shark/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/shark/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/shark/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/shark/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/shark/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/shark/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/shark/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/shark/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/shark/binary/sets/xetc.tgz) = 222d14c205f1018cc5a86eb03214d18cda2abb82
SHA512 (NetBSD-6.0/shark/binary/sets/xetc.tgz) = fb82261996aa772fd8a9022eadebf899ee04ee70229ee4fe07758ca14937bdb34752c8218eb4c5b722b6208f719891493bbc141372fcfa503de83e0a61aa7387
RMD160 (NetBSD-6.0/shark/binary/sets/xetc.tgz) = 511ebfe5076f361b7f7ca03a265d16052a16d259
MD5 (NetBSD-6.0/shark/binary/sets/xetc.tgz) = 165cffbb6d2c79663198d80c82afac1e
SHA1 (NetBSD-6.0/shark/binary/sets/xfont.tgz) = ce0e141beeffeb5b470478e9887e4127da85de3c
SHA512 (NetBSD-6.0/shark/binary/sets/xfont.tgz) = 5d98354ff77103793d56af3192ab5f9afcde62208791976064ac33a149f994e0227dd5dd28ed544eeac5ea504e5212d5fb9e7d05e59e5a48a2fe4062e34e3106
RMD160 (NetBSD-6.0/shark/binary/sets/xfont.tgz) = 4d09f5e82fec875ef0ea328809465e7a98f2ebfc
MD5 (NetBSD-6.0/shark/binary/sets/xfont.tgz) = be26b3890261302420b09976a1d38144
SHA1 (NetBSD-6.0/shark/binary/sets/kern-GENERIC.tgz) = 436b25de68b7e666f29611394561037b31f1a9b7
SHA512 (NetBSD-6.0/shark/binary/sets/kern-GENERIC.tgz) = 64acef2068786ba3afa466eda01db892f03fb48db03ad9ee0817f6a78b8019225beac65553dcd733d9858b4c35da337cbb20a7efa7019e77178e637dd5cb25f2
RMD160 (NetBSD-6.0/shark/binary/sets/kern-GENERIC.tgz) = 83396c190ee6331e8297e4abc20d5b940b654e23
MD5 (NetBSD-6.0/shark/binary/sets/kern-GENERIC.tgz) = a00e97b04a6d5affc36743311bc2293e
SHA1 (NetBSD-6.0/shark/binary/sets/base.tgz) = fa2c393fa435744e1b106d19470278c97d9419fb
SHA512 (NetBSD-6.0/shark/binary/sets/base.tgz) = a6ec0e86858f7c9faef7b4df4f2a951fa120af290d18d06598bb82b54541fe1a36d60b47f03da404a0d97889a5ea582db0a13b4a1827acf9e1842c70c90fcc0c
RMD160 (NetBSD-6.0/shark/binary/sets/base.tgz) = 151c8a2e662926762a132b689b24943ac8b93c01
MD5 (NetBSD-6.0/shark/binary/sets/base.tgz) = 787953226e7c5d1aeb78aba04cb18579
SHA1 (NetBSD-6.0/shark/binary/sets/comp.tgz) = d37041a235ddf0a129bfa9f1d5e11d2fdb77816f
SHA512 (NetBSD-6.0/shark/binary/sets/comp.tgz) = 19427e8b7dc48941e4e54ff0b8841358a551599c43862f830a752551044f96ed39bbd8038348fb3f3bf15ee831e0638cc4a3b4f0c04037222666f017cb11b862
RMD160 (NetBSD-6.0/shark/binary/sets/comp.tgz) = 58df952b8aea84f1e0bc2658871408502eef4667
MD5 (NetBSD-6.0/shark/binary/sets/comp.tgz) = ef10bf2cdec1811b1d3a2d3376201583
SHA1 (NetBSD-6.0/shark/binary/sets/etc.tgz) = 07309c170d21a4c786ac5a12f5e792580788cd76
SHA512 (NetBSD-6.0/shark/binary/sets/etc.tgz) = 6424ecfc34615c6fd1acb909b70414806e516119f30df73b70ee7128e69060da9457043da999de6011e7556a6685091cd30f4033e0ab7ea8cb0a2c6cc02c03ec
RMD160 (NetBSD-6.0/shark/binary/sets/etc.tgz) = b2fa61062310af5511249b80505cefc966801f9b
MD5 (NetBSD-6.0/shark/binary/sets/etc.tgz) = 1b69ed72ccfb0af799b148825a53bc9c
SHA1 (NetBSD-6.0/shark/binary/sets/modules.tgz) = 3ef4de9b2730972b94688f3674b121eeabd2448d
SHA512 (NetBSD-6.0/shark/binary/sets/modules.tgz) = 7c29c66462378dc8d7e7d5bfce324b012383176d8b381e83122808920f2d88d80665547a6e2974738fb6b1268aeacc4a7cde9ef17faec3d9d5c155ae2544cd37
RMD160 (NetBSD-6.0/shark/binary/sets/modules.tgz) = ae6f3aba96b1b9b2d2a412754a42296afc85a69c
MD5 (NetBSD-6.0/shark/binary/sets/modules.tgz) = 5f374bf40599728af733192b3a546d7e
SHA1 (NetBSD-6.0/shark/binary/sets/tests.tgz) = 62e58a9b83eec5a0fb21b9fe7f5845e5315811e7
SHA512 (NetBSD-6.0/shark/binary/sets/tests.tgz) = 43d78bd32bff49e6f6c62914cd16916e41541fb446913c688d0fe4fdbcf5fff6d5605688ec165f8e856a418bba4f36c7d7233b8344c86b954ce104e963abe0a0
RMD160 (NetBSD-6.0/shark/binary/sets/tests.tgz) = 560f27cf65beada1eb6e62458b7450968218fd37
MD5 (NetBSD-6.0/shark/binary/sets/tests.tgz) = 4d39b8cb6664b022b300186cb7afa8e6
SHA1 (NetBSD-6.0/shark/binary/sets/text.tgz) = 663a9a70c81324796717d31e46d5e8c00620a132
SHA512 (NetBSD-6.0/shark/binary/sets/text.tgz) = ae6817fabcfae5fd0a9f35787e6d9af82d64a14c6d532c1a54c347c367d23c6ccc274eb813a23e4bb4dcb036b829437130400c418ffd0eb9c36858443f8d9b84
RMD160 (NetBSD-6.0/shark/binary/sets/text.tgz) = 51994c6b412ee1e6834b270482f8c0dbbc95a835
MD5 (NetBSD-6.0/shark/binary/sets/text.tgz) = c93688098447ac5ebf664bfb3362615e
SHA1 (NetBSD-6.0/shark/binary/sets/xbase.tgz) = 2782c3a99b0cdb6ea4c4b29543f6f06b24c1726f
SHA512 (NetBSD-6.0/shark/binary/sets/xbase.tgz) = 90a42af846fd7479e30789a28c7908b2b70416e7824d7f78c31b378c5e1dd38d4eb797f4430f6dd9baf051afe1750ad4bacdd3e93f2b65b2ad92945f682a1685
RMD160 (NetBSD-6.0/shark/binary/sets/xbase.tgz) = 6b4e44c69f47563964562f18c42a5bc5a99e750a
MD5 (NetBSD-6.0/shark/binary/sets/xbase.tgz) = 1c370ba00bea97b4afbef7e3bc919655
SHA1 (NetBSD-6.0/shark/binary/sets/xcomp.tgz) = 997e19373b3e9b90464e849f1d635e33235aa5f7
SHA512 (NetBSD-6.0/shark/binary/sets/xcomp.tgz) = 1e9d84d70a45aa3ea4d4bff9066594e853b9af68912e58bc933060f2b652ceea2d40231e04debed15dc569ec3d1d2d3682b3042b009c0738f17bf11979f2e86e
RMD160 (NetBSD-6.0/shark/binary/sets/xcomp.tgz) = 60b1ce3cf224d2caf6d4c9757761c8145d4bcc4f
MD5 (NetBSD-6.0/shark/binary/sets/xcomp.tgz) = 0bd79975afc5e2dbf542c8ab9363b664
SHA1 (NetBSD-6.0/shark/binary/sets/xserver.tgz) = cf7350b715c568255b8d095503eed754c3d1ff31
SHA512 (NetBSD-6.0/shark/binary/sets/xserver.tgz) = 3c88b006114891f2e3aaac95d15f25de6c8ac1dd00802097d1885f015326e926a297dcdf6c9c10a798ff73242e3130a7e359bd7f4bab45b23408a628da011725
RMD160 (NetBSD-6.0/shark/binary/sets/xserver.tgz) = e9a9b2579181cfc7974ab1586990bbb0f2470299
MD5 (NetBSD-6.0/shark/binary/sets/xserver.tgz) = 795a4afd76d016cb1868344522fb0fb4
SHA1 (NetBSD-6.0/shark/installation/instkernel/netbsd.gz) = 08fdda8c99f112770acb58f662e7dc7f92e211aa
SHA512 (NetBSD-6.0/shark/installation/instkernel/netbsd.gz) = eb7dd72ec967546883dbf9e50fc9bf770e8aced07020c03f93abdc0c9bd6b1a03ff320a3f5167f74caf732ff37495e7507628e0ea213ae8fbf209fbfc2ed2368
RMD160 (NetBSD-6.0/shark/installation/instkernel/netbsd.gz) = 97cd05778ad6cbef869bfd1ef9bda107db56b099
MD5 (NetBSD-6.0/shark/installation/instkernel/netbsd.gz) = f174a788515cbae3b497a13a1f27b68b
SHA1 (NetBSD-6.0/shark/INSTALL.html) = ee446561ae849bf3326e81939b2071c706bbbe7e
SHA512 (NetBSD-6.0/shark/INSTALL.html) = 01bd9dca202f365a64c9f523cf025abc31a7180a8b45f843d1e60cfe4682329aedc8a892b8e2a6a230bfda1803c6740b6d9fe415992f5a2206425cd875ca3e1f
RMD160 (NetBSD-6.0/shark/INSTALL.html) = c73d5cfb8e656b4634046ab9dabaa09772f8e383
MD5 (NetBSD-6.0/shark/INSTALL.html) = 4207442c91faf52d0f3778fb77943ec0
SHA1 (NetBSD-6.0/shark/INSTALL.more) = 9259e39f2fbc92e793da04efa5acb0318ee257c6
SHA512 (NetBSD-6.0/shark/INSTALL.more) = a9510586f73298c7b0b8af60cf30b86d4588b90682b73adc3f14cb862f0f7ec0f103ff62eda6d4a141a2c5a24fa13d108f08817829a00545f8400ff9785460dc
RMD160 (NetBSD-6.0/shark/INSTALL.more) = f80b0169a0a6cc36d204319f0e5ea6ac9e3c6256
MD5 (NetBSD-6.0/shark/INSTALL.more) = cc98ec10bfc9ee19abfac7bb78843b52
SHA1 (NetBSD-6.0/shark/INSTALL.ps) = d63b6262c2b64c83f763fdc888153b4d5785a5d7
SHA512 (NetBSD-6.0/shark/INSTALL.ps) = e7082fc171e77f43761b8f8dba6443f9a76103be8b964d70e845940cec689c10579d8e5a6aa80bbb008cbde61c125fe348a3db7d157235df9dd6b53d247c3b8d
RMD160 (NetBSD-6.0/shark/INSTALL.ps) = 5ae64ccead4bce193eef02a56c1883be5836e3f6
MD5 (NetBSD-6.0/shark/INSTALL.ps) = b80bf37de21a0d2a7c1891ea18ee27cc
SHA1 (NetBSD-6.0/shark/INSTALL.txt) = 21743fee95f9d668c37c80415896d05a652d8e5f
SHA512 (NetBSD-6.0/shark/INSTALL.txt) = 80bc5a7e1ef3338c29c29b51f72b5c014bb10a7e7f367912ecc1e6633c7882ad23c119733c81dbcc4d7ff5178303dc555fa42951a48d364613ff52048d8e500d
RMD160 (NetBSD-6.0/shark/INSTALL.txt) = 15c92de408750d0c47946ad7c7c7371a79e1e2c6
MD5 (NetBSD-6.0/shark/INSTALL.txt) = ba3059416376468dad953da22a4bf29e
SHA1 (NetBSD-6.0/source/sets/sharesrc.tgz) = 87a69ce98fc413c46524823e16dcd0dce768698c
SHA512 (NetBSD-6.0/source/sets/sharesrc.tgz) = aedb194ace955f8c15820ae3e6d8bdbf67992da79f8cb3d2d0a2ad409819a78c7d3921fda193a982a6f032f92113fc807e0ad2eb064a82da98b8f5ec957bc53a
RMD160 (NetBSD-6.0/source/sets/sharesrc.tgz) = 8e03f055e853b3b44d48cbafde267da9acd549f2
MD5 (NetBSD-6.0/source/sets/sharesrc.tgz) = cdf8ab923e83b705dfc8e47e47ebdecf
SHA1 (NetBSD-6.0/source/sets/gnusrc.tgz) = a5bb46c56b0f27a47dbaa3690758336bdc472292
SHA512 (NetBSD-6.0/source/sets/gnusrc.tgz) = d94945054d64c9b16ddf2b3fbcf0dc8aa41965f76e469e8da0fc822082189d87324169709f5a95aa8461209522d3cdb387f52beaada7ecefb46571b021d925bf
RMD160 (NetBSD-6.0/source/sets/gnusrc.tgz) = 562debda10c4d28bc11f9760fa7f7797f062a9fb
MD5 (NetBSD-6.0/source/sets/gnusrc.tgz) = b72569083f678e665d6658cbd2c3e587
SHA1 (NetBSD-6.0/source/sets/syssrc.tgz) = 8206633fc5ac0f890ecffdd50176768236cd4cc6
SHA512 (NetBSD-6.0/source/sets/syssrc.tgz) = 566104c4324710005c869e065dc03889e85824910520d097a88c1bd5219adc750b3ef8d0057a7d98a7d535002c4c6e04594d98b5be138b8e29d74b1f3965b045
RMD160 (NetBSD-6.0/source/sets/syssrc.tgz) = 8f26506a9ce36e2a163026804d629e30628f8c4b
MD5 (NetBSD-6.0/source/sets/syssrc.tgz) = 8ece18d74c2fbd264198467f47e8b4bb
SHA1 (NetBSD-6.0/source/sets/src.tgz) = bada9cb20d00977af0fc80b465ae9bd20648c893
SHA512 (NetBSD-6.0/source/sets/src.tgz) = 981158338e0123f59b5a38f3f5c3a4fb8142f0280e49db0af5f0753512035113887bdcc9aa3f980761b51abf625764d0e89ca4de41cb80145f5ef7239676d1b2
RMD160 (NetBSD-6.0/source/sets/src.tgz) = 90dc01e48266e240dd95962496cd2a1aa1ca90f9
MD5 (NetBSD-6.0/source/sets/src.tgz) = 58e1e661d9640ca3eb542ac3086580f3
SHA1 (NetBSD-6.0/source/sets/xsrc.tgz) = 73f98faff9d1b90c50d50d50a1f17e6880f43f12
SHA512 (NetBSD-6.0/source/sets/xsrc.tgz) = dbcf2591e42422a0ddb15529b35cd7e1f198589459806239d9efc917c5b604cd152fda76c5fd5a8e912410ee1fc25e561f7e505b63461192406a121e84fe3784
RMD160 (NetBSD-6.0/source/sets/xsrc.tgz) = 617e6308cfbb0899c230ceea2d843c197c4f4804
MD5 (NetBSD-6.0/source/sets/xsrc.tgz) = 68d1535d0baed6625f6dd95fd20758ba
SHA1 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC.MP.gz) = 496928e218106c5bc8d32bd6c736eddf69258416
SHA512 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC.MP.gz) = cc26d9ef454d86c7e95a122eb0ef5d6f9dc742b0f9718d0a798e9666974d977de44d90025f38c2947c9d161c3b52a9eb8f42aa866a0198d310f7defd7f472db8
RMD160 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC.MP.gz) = 3ccc735931179c31c8f2d4b5ce1e9f96121ff695
MD5 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC.MP.gz) = 61be9899c92c833cb52f77b6f8e6aee4
SHA1 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC.gz) = 7b45c74363d7977534a97b006e114ef4051246c0
SHA512 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC.gz) = fc19ac6cf8544def619f803ab7573083991e050b8d551e9c4967d1e327e4ebfe51137f516815152fc00dceed5879309ba4e5e9c420b088378c0323c739df8cdc
RMD160 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC.gz) = 86944c7c8fbed3cad7ddfef09833eee06231c3bd
MD5 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC.gz) = 2daad7afb8771cd536a5bd668d3e981b
SHA1 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC_SCSI3.gz) = 1138b7a86c5bb93d6b3904164e3841031880b675
SHA512 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC_SCSI3.gz) = df36f0aac760c383f970b2dd9e13a4e6e3ca28d47846c904cc07ba11b6e289fe6c8319a40fd51850e5ced7a4585d61c83f489769cfe8aac189b650116f8faeeb
RMD160 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC_SCSI3.gz) = e95d11e401563a66d9cc798db312d3f2a1b96c5f
MD5 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC_SCSI3.gz) = 0aaca233a00a9fce3265a749b9a2852d
SHA1 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC_SUN4U.gz) = 0732f8665299b419119c4507e525a6c992f20656
SHA512 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC_SUN4U.gz) = dcd45f52ade746443eeb910f331ccbce2d48eaf03f2e49c7ac75940b36e2097e41ed5873224bf664e4bd935ba17b1503a61640427a7d9ea87ecdcad39f7eee60
RMD160 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC_SUN4U.gz) = 568b09bb65717c3ae1b2f2e3a5b1c15316a3c53d
MD5 (NetBSD-6.0/sparc/binary/kernel/netbsd-GENERIC_SUN4U.gz) = 785d8668cfd1df9f6bd6af6587636e0f
SHA1 (NetBSD-6.0/sparc/binary/kernel/netbsd-KRUPS.gz) = bd5c8ef678bdeca11a32980b732803967ab0398f
SHA512 (NetBSD-6.0/sparc/binary/kernel/netbsd-KRUPS.gz) = 1c79b7d19dbab9bb132529d1bf10f6a3d9f8bdbf006d1296721c5a987812cf55a077368b75b4775e1425dda56afde6ab35f5995648187c623dcdb3154d2a050a
RMD160 (NetBSD-6.0/sparc/binary/kernel/netbsd-KRUPS.gz) = 93604d098cedbfd647a5ac6ea9409928bebc48d6
MD5 (NetBSD-6.0/sparc/binary/kernel/netbsd-KRUPS.gz) = 7108981b6236f991fca67fe58d6eae81
SHA1 (NetBSD-6.0/sparc/binary/kernel/netbsd-MRCOFFEE.gz) = 976e0a56648d3e03356d7f76426971fa523e4551
SHA512 (NetBSD-6.0/sparc/binary/kernel/netbsd-MRCOFFEE.gz) = 172bd67678acdf034d897369d1d92be25dd684dbd14f362fad6c9655f7d5265caf4b57d495c570eebc1e53a87c53374d8f552656fa66366d3ac6221ccf98a752
RMD160 (NetBSD-6.0/sparc/binary/kernel/netbsd-MRCOFFEE.gz) = b3a6897f2e3eb5acfe2d87f7d377697caeb3b007
MD5 (NetBSD-6.0/sparc/binary/kernel/netbsd-MRCOFFEE.gz) = 7ea122d02744bc1fc5e42853992c908a
SHA1 (NetBSD-6.0/sparc/binary/kernel/netbsd-MRCOFFEE_OFW.gz) = 181bf7c2d1cd731f71c5334a5d0d097749f166af
SHA512 (NetBSD-6.0/sparc/binary/kernel/netbsd-MRCOFFEE_OFW.gz) = 63f26657f4bdde1cff58858fd5ffda69867a45d8148c4b4cf13804281389f34026a09f9f228eb670ff01ed8a02e99affb67b7a170fdcdda5f8f5cb128eaa9ece
RMD160 (NetBSD-6.0/sparc/binary/kernel/netbsd-MRCOFFEE_OFW.gz) = fabe6d8566d88af5324ffa952c57f0fad3accd95
MD5 (NetBSD-6.0/sparc/binary/kernel/netbsd-MRCOFFEE_OFW.gz) = 8a5f7d38f6ddc835e528d51886df36e7
SHA1 (NetBSD-6.0/sparc/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/sparc/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/sparc/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/sparc/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/sparc/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/sparc/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/sparc/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/sparc/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC.MP.tgz) = c77b07953f63e5e21d1313e08dacf592acbddeef
SHA512 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC.MP.tgz) = 40e533f5b17404c1d1137cee7e541193f5f6ad323bc615ba5db6a1dfd4816a27eb54b26d272886dbedc9184e85b36ba1ef6e798e140f84e669475eebdd61a015
RMD160 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC.MP.tgz) = 2e067f3f9ef9697eba5185ff0d15d847c6e277a4
MD5 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC.MP.tgz) = 496e0471d0eb80e227337bdaa45db5b2
SHA1 (NetBSD-6.0/sparc/binary/sets/base.tgz) = 952c71b877521b1af63302bbe4c072b482c3723e
SHA512 (NetBSD-6.0/sparc/binary/sets/base.tgz) = 14dc66a667648834321d8bde6c4f9e07c582ac569ceda4991552119282b53690521a89cb0fc06e2c0952f75d9dbc6c9c554f922e74111f4d42c94085bd4697ec
RMD160 (NetBSD-6.0/sparc/binary/sets/base.tgz) = e57ecaa3098d2ae780c38e3e085c6f164498cd95
MD5 (NetBSD-6.0/sparc/binary/sets/base.tgz) = b9fb689a8d63084ad2d6a4089028296d
SHA1 (NetBSD-6.0/sparc/binary/sets/comp.tgz) = 59f49bbeaebf959d30414d0cacca3724359d55e8
SHA512 (NetBSD-6.0/sparc/binary/sets/comp.tgz) = a800a26ddca7c79ccda284ab3b5e815cc4b53c409ca78bb00f386348e4ae973edf66440a931d9fe2ab0f8a16e329d885c3a2f190780600ca4e205936d7d01e8d
RMD160 (NetBSD-6.0/sparc/binary/sets/comp.tgz) = 3a0cc6492eb72ee033ccd0b60c0c1698fa6b49d1
MD5 (NetBSD-6.0/sparc/binary/sets/comp.tgz) = 39c91111ea991ede7ef81fa845ac319d
SHA1 (NetBSD-6.0/sparc/binary/sets/etc.tgz) = 7cca1a9e594b2122f17b67a26d15e2051403bc68
SHA512 (NetBSD-6.0/sparc/binary/sets/etc.tgz) = 191f890e7588a17a3258e92d51a91407281c97fe84e2ce68f6fc1db48b95011b1c15f86cc3bef58dff5b167c8ee1608fb42c22b55b78df7461a60827882bb0e4
RMD160 (NetBSD-6.0/sparc/binary/sets/etc.tgz) = 39d06e86cfc8d3fb588c5b95371ece29ef11b5de
MD5 (NetBSD-6.0/sparc/binary/sets/etc.tgz) = f909a9a70e65ad480468d1de75f057b0
SHA1 (NetBSD-6.0/sparc/binary/sets/games.tgz) = 352b9089651bdd1ba05526f34fd99bf8e89db039
SHA512 (NetBSD-6.0/sparc/binary/sets/games.tgz) = bd7c551fde89b7339d6b652a2583322c662bf0d4b69ce143034253f9764cfefabc5abe45d8e8850f7b9ee2d5714cda13f26f1d516679b544c096d27cbaf87b08
RMD160 (NetBSD-6.0/sparc/binary/sets/games.tgz) = 9e71d26c198691e30f653c47b3b725ab54f1144e
MD5 (NetBSD-6.0/sparc/binary/sets/games.tgz) = f6932706a5fba3dc666106b7b84ea734
SHA1 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC_SCSI3.tgz) = e5359bdc3fc08e7fc7b5f0ffae289d8e3bbebe72
SHA512 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC_SCSI3.tgz) = 88e1c0850a758a10d7752c1c64d4d80b5fc9935eb8cd9f4869585ca42cb5706bb6c1fa38467e7ba67a28f0b3e4e12df989839b1c7f6a777785e919eda6c4e464
RMD160 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC_SCSI3.tgz) = 6486dfa05e73ea92b83ab42151f3978fb8d9cb3b
MD5 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC_SCSI3.tgz) = 8965e59f45390938dd1f4cca3b47b215
SHA1 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC.tgz) = 0803f566c7ab3cc4ac8bdb66f0e6e93e44f6780b
SHA512 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC.tgz) = ac72f82075f9cd2d7a5aaa4556f0e6efcdd6207ecfd3c6eef2773281f9088fca35169b68987c3a95f8481670fbd637caaccc48a6fea450c5334e1a5a61a009ab
RMD160 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC.tgz) = 4a8db5843d949d1d2b6c45e23313628ef2db07ae
MD5 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC.tgz) = 7c8d15c09c139fc7445d83e669004f26
SHA1 (NetBSD-6.0/sparc/binary/sets/xcomp.tgz) = 4671cf1becbd220fb0693d8dc580bf90fd558a38
SHA512 (NetBSD-6.0/sparc/binary/sets/xcomp.tgz) = a37753c7f879f4d7205dfa23488b6fda40aa098cdf5d998f3b8335c487d2bffc7d9edbd57de2f12dfce586d60f2044f1617173fa6902e821a74dacc314382b98
RMD160 (NetBSD-6.0/sparc/binary/sets/xcomp.tgz) = ed445184a03edb2cf1bf204e9cbbd04ce6f8367e
MD5 (NetBSD-6.0/sparc/binary/sets/xcomp.tgz) = 82dc6b78584e5997bcdf436f37400177
SHA1 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC_SUN4U.tgz) = 478f84d19aae7f5aee8ddfdb45eb9aa37a21a3d3
SHA512 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC_SUN4U.tgz) = b9168217a02a3da991b5b3fb069097b663dea23088a529f16ea09dc12c233bc4d63b4310fbc8e26285e6bc9cd566144141898743fc8617bd9296956432b2f9b2
RMD160 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC_SUN4U.tgz) = e9e53c86f42b75fcc12853ed2dfd5423b0968581
MD5 (NetBSD-6.0/sparc/binary/sets/kern-GENERIC_SUN4U.tgz) = ee3736e6af879f5d4e0cc369bfa579b5
SHA1 (NetBSD-6.0/sparc/binary/sets/kern-KRUPS.tgz) = bc69fd900a788cd8ce5d7413b4716f656814b804
SHA512 (NetBSD-6.0/sparc/binary/sets/kern-KRUPS.tgz) = 64c42a3748d62926aa10870c08331bce20b67f6b50afd13a19a76b232b2f9f2d1c2fa04819cf88f211badbdff0cc1684ca48ca7f1c6d8f44859257764e882842
RMD160 (NetBSD-6.0/sparc/binary/sets/kern-KRUPS.tgz) = 18df230d31af7e526b9f6fb3254061e6601a9b15
MD5 (NetBSD-6.0/sparc/binary/sets/kern-KRUPS.tgz) = 1057e644aef74d636a558c03dd9f7e7e
SHA1 (NetBSD-6.0/sparc/binary/sets/kern-MRCOFFEE.tgz) = 58b500435295d07ef47bd1702ce2f8c5fe798e8c
SHA512 (NetBSD-6.0/sparc/binary/sets/kern-MRCOFFEE.tgz) = f3657cd11910995d2bf8e4140f4df4d6829311de9d5a4fa39a980af3577035ad7cc1189066aa0d9093803727b0e1405e9e2c5bbad4db8e4e172e9d5d000c2541
RMD160 (NetBSD-6.0/sparc/binary/sets/kern-MRCOFFEE.tgz) = 91c5e94452446fb8838dfadab79d8f0c915669a5
MD5 (NetBSD-6.0/sparc/binary/sets/kern-MRCOFFEE.tgz) = 2535895e15ea9dc1c4806df090d1bc81
SHA1 (NetBSD-6.0/sparc/binary/sets/kern-MRCOFFEE_OFW.tgz) = cc966bc2458bae8bdd3995d423ae83707d0a0a38
SHA512 (NetBSD-6.0/sparc/binary/sets/kern-MRCOFFEE_OFW.tgz) = 35967055f863081ef3d16144c5324ab62c2e44942100202414961a58b8b3e1243e8a2b3e2ff4d7075df0e624d9da1ff7889f8c9dc710c888f6654d8d845233d9
RMD160 (NetBSD-6.0/sparc/binary/sets/kern-MRCOFFEE_OFW.tgz) = 5451d5da667a2a57e3af6df7119703db7a2ec8a2
MD5 (NetBSD-6.0/sparc/binary/sets/kern-MRCOFFEE_OFW.tgz) = 8cb1f508b5103088f400e7422e23a34f
SHA1 (NetBSD-6.0/sparc/binary/sets/modules.tgz) = 6519d6696e013f8a9e51d2856aac1390c0c3e69a
SHA512 (NetBSD-6.0/sparc/binary/sets/modules.tgz) = 4dd5d8721c4d0733813e1fc884e9242cbf957f33ccf92fa7a8f3255bcf1af3598de843e15ad052a696cd95065251cae8a6927816c041732a6f12823657ef22fd
RMD160 (NetBSD-6.0/sparc/binary/sets/modules.tgz) = cab574162ec0996f845835763055d6e65bb4ceb4
MD5 (NetBSD-6.0/sparc/binary/sets/modules.tgz) = a60477fd2b83a211972512c035679276
SHA1 (NetBSD-6.0/sparc/binary/sets/tests.tgz) = 816dc19db04ba2f95dcf80c62598f9fec7fa92ff
SHA512 (NetBSD-6.0/sparc/binary/sets/tests.tgz) = 80c23ac0ed623404542d3e6b6e7dc8194ae876c6a0d17b250151ce6e941e874a033dcce8afa002fbe3d875b1c79687e392fa6f527fa46ec0e3c82ae31cfd5eb9
RMD160 (NetBSD-6.0/sparc/binary/sets/tests.tgz) = 834ebf4c2126e3ec1b2226e48564ab87d7ff8da0
MD5 (NetBSD-6.0/sparc/binary/sets/tests.tgz) = 9d0c1bd587f83df1b0090f88357f8531
SHA1 (NetBSD-6.0/sparc/binary/sets/text.tgz) = 92d7911a1d761b73c8d2080e819705f901d1ce6d
SHA512 (NetBSD-6.0/sparc/binary/sets/text.tgz) = 4e8e2f2abc8cf95a4672ef492e091142b6cc4a4baafddb6fc123fa5b3cc3d67a16e61ab4a7b188dac4b8ea198d8c88c1c81ffe1eca70f9900780a8a134257c36
RMD160 (NetBSD-6.0/sparc/binary/sets/text.tgz) = 4d5ca8f6d680d3389d6b90c45875bcc4b4e962f3
MD5 (NetBSD-6.0/sparc/binary/sets/text.tgz) = c16beb242b426177d80af90e40266abf
SHA1 (NetBSD-6.0/sparc/binary/sets/xbase.tgz) = 2809df01d9a0aa85270f6f77341cf50d72bc78bc
SHA512 (NetBSD-6.0/sparc/binary/sets/xbase.tgz) = 886f26f21f95ea17ef60d28fabebf415485d13647d335b237a4ee954039c7c6dfc0d9e68c06aec52ef463b6c2b7b88cd6baf514f2c5b567b8f5421c99df12672
RMD160 (NetBSD-6.0/sparc/binary/sets/xbase.tgz) = 499a1e2a76a05fa4a0b3b53438d60b8434198695
MD5 (NetBSD-6.0/sparc/binary/sets/xbase.tgz) = 82bb74d040bf256d711afb55db6c4066
SHA1 (NetBSD-6.0/sparc/binary/sets/xetc.tgz) = 997196c30074dcc988f2dfa2a7f5dfc8c31df3db
SHA512 (NetBSD-6.0/sparc/binary/sets/xetc.tgz) = 09670a40f33c9902b5b22c4400f9e8151f1985504cc1c956df68697e3bd6a7b841391d6dd32cdbba42aebf42f4a1cdd3559c5783f79da236603dc6b5cf86d33c
RMD160 (NetBSD-6.0/sparc/binary/sets/xetc.tgz) = 50061239bbe31a4dce18784bfcece78a234e6ef5
MD5 (NetBSD-6.0/sparc/binary/sets/xetc.tgz) = 12ae04f5957291f424568bb61a138079
SHA1 (NetBSD-6.0/sparc/binary/sets/xfont.tgz) = 7e5ccc7436f81855bd5e20960adaddd9b9851fd5
SHA512 (NetBSD-6.0/sparc/binary/sets/xfont.tgz) = f263bf4a2b1bc795c1c4dcaf0db819dca08e90c51b41c144911f897a991a3575dcbbac203060d8f51842c85c6e85c4839297fb7307353a38b1a46e1b446d29a3
RMD160 (NetBSD-6.0/sparc/binary/sets/xfont.tgz) = cb721308df7413a821278143750bf95844eb4d82
MD5 (NetBSD-6.0/sparc/binary/sets/xfont.tgz) = 9de78b2d18118d7435a6c95d8a503ae4
SHA1 (NetBSD-6.0/sparc/binary/sets/xserver.tgz) = 0d3b410b9501560307d3b9b17a431be0da1de869
SHA512 (NetBSD-6.0/sparc/binary/sets/xserver.tgz) = d3dceb5ebfb5085e905e8121c952a97bf7035bbd4e60898d913fbaf58c147181ae7c97f913a122e9ce83523d05a585d1e771139192d42c44fa85e60e2bed762a
RMD160 (NetBSD-6.0/sparc/binary/sets/xserver.tgz) = fd59e7a1c949c113ec555a9ee63ab369f64559fa
MD5 (NetBSD-6.0/sparc/binary/sets/xserver.tgz) = a89a1c519c6d6759f810717fc0ec43ab
SHA1 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.aout.gz) = c2c368ba46549cb580707b799f4f0f88b26128de
SHA512 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.aout.gz) = 9fb6bef663b2fb68b9bf3591834f6e653c0b3b7f69ac8c40f2479efaa5d0641140d87623fe8eeb8a21cb0e0065ceacb57d6a5db4e1905db97f7dffa52ba8b565
RMD160 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.aout.gz) = 8fbcdca0f8214901d1fec117c567cfe93865e391
MD5 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.aout.gz) = 1cf0fdefa01fee95d494b3418ea4a0ea
SHA1 (NetBSD-6.0/sparc/installation/bootfs/boot.fs.gz) = e36798e87b9d195c7df8602883c775f8f4268448
SHA512 (NetBSD-6.0/sparc/installation/bootfs/boot.fs.gz) = acd6ad2fa9c1618b4b14d6a63679e347aad6302001e36ea63c58968252ce993df23af7e05e590b72efbcdbc1013192b1862d70d7737fc2fa804249410c9d2724
RMD160 (NetBSD-6.0/sparc/installation/bootfs/boot.fs.gz) = 986addabcfd9f0b1faccd9fed0081f88e78f2aaf
MD5 (NetBSD-6.0/sparc/installation/bootfs/boot.fs.gz) = 93526666c7879bc6391c5b7c38b24cb0
SHA1 (NetBSD-6.0/sparc/installation/bootfs/instfs.tgz) = 98f621b007b19382a0d529f2056cd8c82feb863f
SHA512 (NetBSD-6.0/sparc/installation/bootfs/instfs.tgz) = 7131f92cd02791f6d351a8ea5d8b5fe93b643ba5975522e85bf81da8f69157ebb21f80f5696beca47809a9732e79e85c7e97e07a33e830eeac2f9542685a01ce
RMD160 (NetBSD-6.0/sparc/installation/bootfs/instfs.tgz) = dc7168f774cac7e9f7eba640dc55c878ab2c16db
MD5 (NetBSD-6.0/sparc/installation/bootfs/instfs.tgz) = 80ed52c842f1d4d56d4506d293e30b57
SHA1 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.symbols.gz) = 673b10e3d23ee6e204bdba91b52b0b50304cb8e3
SHA512 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.symbols.gz) = edf95ecf1f6133e5228d4f8b6a8d004c1f172354acfa34f1235a488456a69b2fd19cd959f97a0a7d06f007347815672bfce07a3ee25ae511ce73a6e5de217267
RMD160 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.symbols.gz) = 8829961d6f2e5487af4bd4418335139d06916659
MD5 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.symbols.gz) = 75ea54ba199fcf8a56054d15e4fb3b96
SHA1 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.gz) = 8eb131ce31d568ee4fb0674f1732052bc686bb18
SHA512 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.gz) = 7345380b1e2e1140564b9886e4d2afc5dc21dacf7148a5680283abc41f7249a8e8512db23ca58533ebf5bffc588bb6a28b18c7e0eb5d50c8a5a06da4043af319
RMD160 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.gz) = 37ae0d93bd4532ff69c9187a1fd8ee3b75e97671
MD5 (NetBSD-6.0/sparc/installation/bootfs/netbsd.ram.gz) = dc1f591b5bca88a07ff8e67a1c3dc01d
SHA1 (NetBSD-6.0/sparc/installation/miniroot/miniroot.fs.gz) = 4e71664ebc1dc62f55992a935734fe61f52177cd
SHA512 (NetBSD-6.0/sparc/installation/miniroot/miniroot.fs.gz) = 3cf2607774e1aa7343ed57c78034828878665e9a041561702daf18539a0775d27768ad89d9336c77d897a0f040c71c8cefdd555c00397d26fded0b6ce72c1dc3
RMD160 (NetBSD-6.0/sparc/installation/miniroot/miniroot.fs.gz) = 39fff07e0ce17f0c62c3a399173b5a8681fa3559
MD5 (NetBSD-6.0/sparc/installation/miniroot/miniroot.fs.gz) = 0355cc648c33e3ed61ac4ff2f51edd04
SHA1 (NetBSD-6.0/sparc/installation/netboot/rootfs.tgz) = 98f621b007b19382a0d529f2056cd8c82feb863f
SHA512 (NetBSD-6.0/sparc/installation/netboot/rootfs.tgz) = 7131f92cd02791f6d351a8ea5d8b5fe93b643ba5975522e85bf81da8f69157ebb21f80f5696beca47809a9732e79e85c7e97e07a33e830eeac2f9542685a01ce
RMD160 (NetBSD-6.0/sparc/installation/netboot/rootfs.tgz) = dc7168f774cac7e9f7eba640dc55c878ab2c16db
MD5 (NetBSD-6.0/sparc/installation/netboot/rootfs.tgz) = 80ed52c842f1d4d56d4506d293e30b57
SHA1 (NetBSD-6.0/sparc/installation/netboot/boot.net) = af35e94ca3d3ca7e07084d17c7d16370c2be1949
SHA512 (NetBSD-6.0/sparc/installation/netboot/boot.net) = 42e02f39699d1ef5f76a2a07641f5405b74a0ce72df8c1306cdc1bf9e8e176b0857a4c6c1098a35bdaaebb729e887633bfc91cd7ed4316ff549b474e4b29932f
RMD160 (NetBSD-6.0/sparc/installation/netboot/boot.net) = 5f64296269674b5af481fa3a55d6268bafcdd50c
MD5 (NetBSD-6.0/sparc/installation/netboot/boot.net) = 9a2b02686d556555bf350f6eb35e07f1
SHA1 (NetBSD-6.0/sparc/installation/netboot/bootjs.net) = 9bba50438d03cdef774f7572e6260bf1feb544f4
SHA512 (NetBSD-6.0/sparc/installation/netboot/bootjs.net) = afa66b734ea6f3f9551680c5dc1af1ce43de4b385eed1c732dbf22f676e5adf1432a2048a494932650e36667405df0160b3d1cb87d4d78b5d064dfab2d366cd0
RMD160 (NetBSD-6.0/sparc/installation/netboot/bootjs.net) = b2cde57103124271b4b9332cb99a7b7e01db3477
MD5 (NetBSD-6.0/sparc/installation/netboot/bootjs.net) = b6958131d22fe5ced08a41671ddfa073
SHA1 (NetBSD-6.0/sparc/installation/tape/tapefile1.gz) = c2c368ba46549cb580707b799f4f0f88b26128de
SHA512 (NetBSD-6.0/sparc/installation/tape/tapefile1.gz) = 9fb6bef663b2fb68b9bf3591834f6e653c0b3b7f69ac8c40f2479efaa5d0641140d87623fe8eeb8a21cb0e0065ceacb57d6a5db4e1905db97f7dffa52ba8b565
RMD160 (NetBSD-6.0/sparc/installation/tape/tapefile1.gz) = 8fbcdca0f8214901d1fec117c567cfe93865e391
MD5 (NetBSD-6.0/sparc/installation/tape/tapefile1.gz) = 1cf0fdefa01fee95d494b3418ea4a0ea
SHA1 (NetBSD-6.0/sparc/installation/tape/tapefile2) = 98f621b007b19382a0d529f2056cd8c82feb863f
SHA512 (NetBSD-6.0/sparc/installation/tape/tapefile2) = 7131f92cd02791f6d351a8ea5d8b5fe93b643ba5975522e85bf81da8f69157ebb21f80f5696beca47809a9732e79e85c7e97e07a33e830eeac2f9542685a01ce
RMD160 (NetBSD-6.0/sparc/installation/tape/tapefile2) = dc7168f774cac7e9f7eba640dc55c878ab2c16db
MD5 (NetBSD-6.0/sparc/installation/tape/tapefile2) = 80ed52c842f1d4d56d4506d293e30b57
SHA1 (NetBSD-6.0/sparc/INSTALL.html) = 8213ae2b753d10e3eec349c15a7cabde6ae1c3d5
SHA512 (NetBSD-6.0/sparc/INSTALL.html) = 7a12bfc31cddfda66761861cb87fc54deb5f683863b41ff44152198ea44e9489c8cce0d4702c278e2586dd3189e5c31692a02a3f17b112bfa19f1134571ff8f9
RMD160 (NetBSD-6.0/sparc/INSTALL.html) = 77a4274595de342e22766d8c468e2a324c26fbf5
MD5 (NetBSD-6.0/sparc/INSTALL.html) = 80cd31d110f62b89b7b910ee0aecd7b8
SHA1 (NetBSD-6.0/sparc/INSTALL.more) = fce351f89d2da1fb9f709bbe890892db755f56c4
SHA512 (NetBSD-6.0/sparc/INSTALL.more) = 496bb7e73d2dda1eb01518c40ab3abc767b8fb6a8b406b07176b9e926c2f4d0e5512bfdc9c26dbfa1b3a4c5069355f6ee1bb8ad8c1d459b71011ea9034f89d17
RMD160 (NetBSD-6.0/sparc/INSTALL.more) = 89c6de032d286741f2ddde15e4b796205a8b8511
MD5 (NetBSD-6.0/sparc/INSTALL.more) = 308ae548170c8b1b5a9bf59f64b1df7f
SHA1 (NetBSD-6.0/sparc/INSTALL.ps) = 3e6e3cbbf8141043fce33e2245513a9683be0edd
SHA512 (NetBSD-6.0/sparc/INSTALL.ps) = 93b81d3ba2fd2e1fa3ef46360e92e247fa72b320af7b0e60ab35fd90cdd10d7bc3dfc03920b4a8b209ef95facc74e047630cdec03cabe06edde21e795e158e60
RMD160 (NetBSD-6.0/sparc/INSTALL.ps) = 8d4373a814dc88c7865b14a59e27731783e711e5
MD5 (NetBSD-6.0/sparc/INSTALL.ps) = ecde0171eda632b616139caf04081ec4
SHA1 (NetBSD-6.0/sparc/INSTALL.txt) = 556744433939c09c3ef5ee3e9b12c1570ebb1211
SHA512 (NetBSD-6.0/sparc/INSTALL.txt) = 1686bb5982301002cb580022760f12fe9160a4fe826d7e0fe74cc4a480c891d31099d5219678b0768c0fbc04be329f0f40511d739c201a5e43367b5a27e72545
RMD160 (NetBSD-6.0/sparc/INSTALL.txt) = cbc495189f2ce52328d87a1a73dc8452ad23e723
MD5 (NetBSD-6.0/sparc/INSTALL.txt) = 17100faaf077e73c2549484e7623550f
SHA1 (NetBSD-6.0/sparc64/binary/kernel/netbsd-INSTALL.symbols.gz) = b76da4e7c7cdcdbebd2148ee815c24554b0c4bee
SHA512 (NetBSD-6.0/sparc64/binary/kernel/netbsd-INSTALL.symbols.gz) = f31c311444a309674e2718baab015b47c0139523e7996a29ab8d3e97a941b5d74be8765571bf3239375c979fff1e963d32d2dbae19ad5c7dedcbd79a8827575f
RMD160 (NetBSD-6.0/sparc64/binary/kernel/netbsd-INSTALL.symbols.gz) = 95a34099fb1159f17619713ae37d3c6337b479c5
MD5 (NetBSD-6.0/sparc64/binary/kernel/netbsd-INSTALL.symbols.gz) = ab2e1333f6d957074854ea8f92320442
SHA1 (NetBSD-6.0/sparc64/binary/kernel/netbsd-GENERIC.UP.gz) = 961388f6e1c9522c0d8f19f0a471f14d3c35fef7
SHA512 (NetBSD-6.0/sparc64/binary/kernel/netbsd-GENERIC.UP.gz) = 7b3fc655414c6c2334e46315eb91ee562b94400f5729e0a33dcc9e9a79c431b4b5e0c124541b9666731740ce365c034718671348a494c43167eb0b73966133a2
RMD160 (NetBSD-6.0/sparc64/binary/kernel/netbsd-GENERIC.UP.gz) = e02a398cd00c0b8ab7bc3b6dbd52e767b18e3275
MD5 (NetBSD-6.0/sparc64/binary/kernel/netbsd-GENERIC.UP.gz) = 200fd2ad96dfa6ec886585e20d4b4b06
SHA1 (NetBSD-6.0/sparc64/binary/kernel/netbsd-GENERIC.gz) = bf5fe38aaa01c9b2e9963b540f1550562789da22
SHA512 (NetBSD-6.0/sparc64/binary/kernel/netbsd-GENERIC.gz) = c2b63ce56ff32ddaa96642ae15a33143a245b9239560c86e51e78db333d8bfd19e761d6cbe27ce69647a62fa4c76317670f0d22aa491a2cadf45dae18573511f
RMD160 (NetBSD-6.0/sparc64/binary/kernel/netbsd-GENERIC.gz) = 65f66c6c7f87e3be728339a1b294b4240b14b953
MD5 (NetBSD-6.0/sparc64/binary/kernel/netbsd-GENERIC.gz) = 44d6f63922847e6f53d78f10bc776e3d
SHA1 (NetBSD-6.0/sparc64/binary/kernel/netbsd-INSTALL.gz) = 1fe47cdf745a00c31a47faf1155eb0f9af2ec4fc
SHA512 (NetBSD-6.0/sparc64/binary/kernel/netbsd-INSTALL.gz) = 7a10dbbdcfdcad97fec4ef476e9b6c876a3bb80b78fb4634e3f502b043b392faaf457b90ff572cc9fcf35c8c26fc908fe035b5859ed66ea2f0db490d2638f9a4
RMD160 (NetBSD-6.0/sparc64/binary/kernel/netbsd-INSTALL.gz) = 06d3a86dbdb12e0b810888233ed79bc735ab13c7
MD5 (NetBSD-6.0/sparc64/binary/kernel/netbsd-INSTALL.gz) = 203f7a8aad11cea65388e14dec6a65f9
SHA1 (NetBSD-6.0/sparc64/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/sparc64/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/sparc64/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/sparc64/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/sparc64/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/sparc64/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/sparc64/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/sparc64/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/sparc64/binary/sets/kern-GENERIC.UP.tgz) = 643e8b15fb356856d93258e6c5d05cb14eb7d569
SHA512 (NetBSD-6.0/sparc64/binary/sets/kern-GENERIC.UP.tgz) = 26c4eb3392f80d4ba7c28038f548e59a04a84742962da3340fb49d09ef42b2e3debc1387710b3c4baa6c735370b5daaa23a6f4da91f1be6552ae4c0cd314f4de
RMD160 (NetBSD-6.0/sparc64/binary/sets/kern-GENERIC.UP.tgz) = 37e5fdfd623c179edfa002163e005518b26fab7d
MD5 (NetBSD-6.0/sparc64/binary/sets/kern-GENERIC.UP.tgz) = 70d61e4a2588eb3242e66d0cd4cf1b1e
SHA1 (NetBSD-6.0/sparc64/binary/sets/base.tgz) = cdce0eb09c71244d8362eb4e09c0bdbd92fb4900
SHA512 (NetBSD-6.0/sparc64/binary/sets/base.tgz) = 455fa6b72e5735a13b3106260b7cb5cabeefd3ca83603dd9daf3805a0abe55acc6b484ad8b4875590e5d949c35b8476cec136f79f8ecfcd7b12cbeee44d6f305
RMD160 (NetBSD-6.0/sparc64/binary/sets/base.tgz) = 13508c9d394104d0c0b460a895f796e091ca8635
MD5 (NetBSD-6.0/sparc64/binary/sets/base.tgz) = cdafa4124cc784a56444f96fb24a7a34
SHA1 (NetBSD-6.0/sparc64/binary/sets/comp.tgz) = 71040515074d030a65e0200afe6dbd6955251af6
SHA512 (NetBSD-6.0/sparc64/binary/sets/comp.tgz) = 52a075300885b0d15ac960430ac3ed1ecb4434cd0bb2de216c6d8c27239011544628ad1ccd87076ac9d40d64e0c088cc2ce0e928902f80629cea11631f30e35c
RMD160 (NetBSD-6.0/sparc64/binary/sets/comp.tgz) = 96bd6b8b5758360970d9d5b703019bf660aa56d5
MD5 (NetBSD-6.0/sparc64/binary/sets/comp.tgz) = be2c961313f29bb9c0282516867850d2
SHA1 (NetBSD-6.0/sparc64/binary/sets/etc.tgz) = e3bae514025003988d5de6ffcc14e130ee8da818
SHA512 (NetBSD-6.0/sparc64/binary/sets/etc.tgz) = 2e5d76dc66495eb24d8abd5eed70cb05cfef3190e81875f0d6e6da67af962b0de0062bc4e9758859647ebc5d4334bd9bc28a20508f8ddecc7c183de8174a4ed4
RMD160 (NetBSD-6.0/sparc64/binary/sets/etc.tgz) = 3fe88c6bd07952ad383d6ab1dbb7d6c3d5315b4a
MD5 (NetBSD-6.0/sparc64/binary/sets/etc.tgz) = 810c7b6940fd2e8049abf4318577f5fa
SHA1 (NetBSD-6.0/sparc64/binary/sets/games.tgz) = a40f5681634cff38215edeb22795d4e5f3f35324
SHA512 (NetBSD-6.0/sparc64/binary/sets/games.tgz) = ff66f16eea28f3191abed7466b21d1c1969f0556cb5a9a61e5167bb7a2c4e55ed8682222a389a04d44306b39b424be25623cf8bcd41c97139960ee48ba91199d
RMD160 (NetBSD-6.0/sparc64/binary/sets/games.tgz) = c3cc1abe005bc98e0afebadc87f0181ba1fd8d6d
MD5 (NetBSD-6.0/sparc64/binary/sets/games.tgz) = a70797dbec20b831d406952468b8376a
SHA1 (NetBSD-6.0/sparc64/binary/sets/kern-GENERIC.tgz) = c047c462dd890c992df60d0f23657638875f90c2
SHA512 (NetBSD-6.0/sparc64/binary/sets/kern-GENERIC.tgz) = 79ca275266c148fa3db47ac564e8f764c75b80a8243a0a32bbd3cd3392621b47733afdedf64f3a2103a99c03ee2f63494d892260951b85949e1383d3bf8c5673
RMD160 (NetBSD-6.0/sparc64/binary/sets/kern-GENERIC.tgz) = c0b40543a2674c1db7d554e9ad19f8e289683de8
MD5 (NetBSD-6.0/sparc64/binary/sets/kern-GENERIC.tgz) = a9346477bd0ae3d50b4b82f8f8da8531
SHA1 (NetBSD-6.0/sparc64/binary/sets/modules.tgz) = c24c622359c9e5f8a05c03adb30e56c8a9862abd
SHA512 (NetBSD-6.0/sparc64/binary/sets/modules.tgz) = f2f834fded5c1142611bd5ff4ebcec67db75f620256fb27689e4cc626785a1bdf90235965e2fc4f2ee7c0ba574e73063e7f13d7f85d291dd23395074517abb9d
RMD160 (NetBSD-6.0/sparc64/binary/sets/modules.tgz) = 6a1b6845fc699f9837a88fc3c18585162a7a0aaa
MD5 (NetBSD-6.0/sparc64/binary/sets/modules.tgz) = e8588a8eed7bb775dfb236cfc8c458d4
SHA1 (NetBSD-6.0/sparc64/binary/sets/tests.tgz) = 1103500cca9a6dcabfafd50046f15e035a1d0914
SHA512 (NetBSD-6.0/sparc64/binary/sets/tests.tgz) = cc628d649f4d16af2300baf5edbf075bc43c91b027c19dcd59b6f1be0ff1b02a3f8f97eb88db2f2bcfa93f69ef02653021f416c546dcdb0f0aa5bf4726257f79
RMD160 (NetBSD-6.0/sparc64/binary/sets/tests.tgz) = ce47518fd68d02d821bb4b4060ea3a775f64e37e
MD5 (NetBSD-6.0/sparc64/binary/sets/tests.tgz) = 2444ea500f4f8363d339a5eb58b50e27
SHA1 (NetBSD-6.0/sparc64/binary/sets/text.tgz) = 09bdcb54fb74d5ae4e43b5b74a6a35467e385e09
SHA512 (NetBSD-6.0/sparc64/binary/sets/text.tgz) = 82834b26dc0a102bdb5da4744fe2677fb50a6018b34a656d7f0813919bfe66168e37b5b47ad027ee6912fb77cceaa5a1cc33eef8269eb6a8a9f42368d01b3037
RMD160 (NetBSD-6.0/sparc64/binary/sets/text.tgz) = 376a9a61492aed41b486fa58060edcb472152739
MD5 (NetBSD-6.0/sparc64/binary/sets/text.tgz) = 69cce7d6bb4cb104cc27f75f17da32e2
SHA1 (NetBSD-6.0/sparc64/binary/sets/xbase.tgz) = 865d8aa735ba57980974089ee2e44d79c69cd036
SHA512 (NetBSD-6.0/sparc64/binary/sets/xbase.tgz) = 2b7f8d15caf6a2b831bdba3e8a2d4f78633f3350d0fc93f107c35cba746c1423c966313224422832484166e384a54b1fc2c75cba78ab138bf36e81c5468c2952
RMD160 (NetBSD-6.0/sparc64/binary/sets/xbase.tgz) = ec680d9c70f3e6729a19167932b552a727c628a5
MD5 (NetBSD-6.0/sparc64/binary/sets/xbase.tgz) = a1907a4afda2ea247a6b54b158ac97f3
SHA1 (NetBSD-6.0/sparc64/binary/sets/xcomp.tgz) = c0b90da52efa86aadb3af4b7de3773c29289c65c
SHA512 (NetBSD-6.0/sparc64/binary/sets/xcomp.tgz) = d3774cc192c6ea74391dc0323db4f7e1695f290908c1ba68f885a9e670fb2c31f9a0c831a6020cd1a98e1b93691e02fadb260b26d8c6d252e9638ebf9cc0e8e5
RMD160 (NetBSD-6.0/sparc64/binary/sets/xcomp.tgz) = 40969f50dcfbfd6948b48f819a7f84957bf14134
MD5 (NetBSD-6.0/sparc64/binary/sets/xcomp.tgz) = 681754abde2de73609efdcbc7370dbe4
SHA1 (NetBSD-6.0/sparc64/binary/sets/xetc.tgz) = 43c0a0b2db4bc88c509eb0d271a99e11f823babb
SHA512 (NetBSD-6.0/sparc64/binary/sets/xetc.tgz) = 4d28ba7d6db0b7c7cb005697eabd739544bae3e3128eb29b8ce62297b26b81d944d0992a6ca3522a51bb83ca874c4a5cf844e06a7aac0a6ab26b30838016faf2
RMD160 (NetBSD-6.0/sparc64/binary/sets/xetc.tgz) = bf885030bc65c540373a06e2b4c6257a5b634b61
MD5 (NetBSD-6.0/sparc64/binary/sets/xetc.tgz) = 006f8b77a6fb41dad2dcbf2f28efdb8f
SHA1 (NetBSD-6.0/sparc64/binary/sets/xfont.tgz) = adf52a9866ae0e4b1e871102f664dd42907d327f
SHA512 (NetBSD-6.0/sparc64/binary/sets/xfont.tgz) = a02de6961b726d4f5fc291ac48a0a59d0f991ba96fd83788e766be8591e94a64eae0fdb23262f965042348516c52a173e332b8d90b21457d9089b7478273221c
RMD160 (NetBSD-6.0/sparc64/binary/sets/xfont.tgz) = b458181e798b9b33d15ad35adfb734a0b597d042
MD5 (NetBSD-6.0/sparc64/binary/sets/xfont.tgz) = cf37180a29cdf9bd6a34a24a8608e6bd
SHA1 (NetBSD-6.0/sparc64/binary/sets/xserver.tgz) = 6ac9cf2e66f22ee56ef1d692e9bed5ae97092088
SHA512 (NetBSD-6.0/sparc64/binary/sets/xserver.tgz) = b6c611b3360d537e94339536354d0a5904137b519e4debab49a93a874429a115cee276f3550a6db23ec9caeb17af3ab162999d181189311a66d65f5230a3a99c
RMD160 (NetBSD-6.0/sparc64/binary/sets/xserver.tgz) = eb216c772571b79eb35e44285f8bed559eeb4fd7
MD5 (NetBSD-6.0/sparc64/binary/sets/xserver.tgz) = e7994f53d2b295259dfb313f86d4fe4d
SHA1 (NetBSD-6.0/sparc64/installation/miniroot/miniroot.fs.gz) = dcedff4706750585b89433f833cfe99fdd454400
SHA512 (NetBSD-6.0/sparc64/installation/miniroot/miniroot.fs.gz) = 94360d4f72150432307bd097e8fa02b18bf3d63ea095b8cecfb4de863ab367c73268116681bb94083bef6445842d4718bfdeec200f57124e95c1de8dfff96b4a
RMD160 (NetBSD-6.0/sparc64/installation/miniroot/miniroot.fs.gz) = 344981ca28094153d14b7d36321dc5e747f214cf
MD5 (NetBSD-6.0/sparc64/installation/miniroot/miniroot.fs.gz) = 8e457a7c91b800f997cdaf6a17e79bc8
SHA1 (NetBSD-6.0/sparc64/installation/misc/boot.fs.gz) = d5afbef3f3b358e77789ea7e9f22298a77415d22
SHA512 (NetBSD-6.0/sparc64/installation/misc/boot.fs.gz) = aba7c9613d7a34e3b939644d37b0a8fb6c27fa0a8aa208aa729499b367f76fc6d5e1a9a06a5a779748c2d2ebfdf9521ac2fcd484608bf64ee3fa0eee8b97fd5c
RMD160 (NetBSD-6.0/sparc64/installation/misc/boot.fs.gz) = bb8a8ed04649c3369dd5893897f7ea2ce7085739
MD5 (NetBSD-6.0/sparc64/installation/misc/boot.fs.gz) = b9aef29dae0ef2a33611cd4655ff58f0
SHA1 (NetBSD-6.0/sparc64/installation/misc/bootblk) = 9aee0c29c2eeb5c2e61aa9bfcd289e88c1fe1e97
SHA512 (NetBSD-6.0/sparc64/installation/misc/bootblk) = 294708defc60602553eef0ae08ed910845b4c89ed72f33493c5fedad3737c9ae1cc96e947a11227e52505c62e04ad6d1f8f1f292df7e7885346acb824c4ea3fe
RMD160 (NetBSD-6.0/sparc64/installation/misc/bootblk) = 28bd7d5c0fe6f33afb9627a3e3c8950ca062afa8
MD5 (NetBSD-6.0/sparc64/installation/misc/bootblk) = 16b4c89c82c34d5795a50c811941022c
SHA1 (NetBSD-6.0/sparc64/installation/misc/installboot) = d1cb90b57a258cef3671cc649d50aa7ea6f73f4e
SHA512 (NetBSD-6.0/sparc64/installation/misc/installboot) = bf100237898c168e020c365da754b8a659b7cbc01ab310fd9b77a2ab60982c67e5f28cfbc7cd152ec37bc9591c1d0402c850803d57a45591282f774b97da3e2e
RMD160 (NetBSD-6.0/sparc64/installation/misc/installboot) = 64f2ea944507466a6f5a61367e6190897cf00dc5
MD5 (NetBSD-6.0/sparc64/installation/misc/installboot) = 7436c373285d89dc76dc705c3bbba5c0
SHA1 (NetBSD-6.0/sparc64/installation/misc/instfs.tgz) = 3715a9104cdd1744835ef982a2a95525a5500c42
SHA512 (NetBSD-6.0/sparc64/installation/misc/instfs.tgz) = 17a2112d5279ffd1a5889e55cf8e7081d564c330645f8c90082e324d413568db5bb58f8281a5b8e22d566b8699c326fa1661a0eb31d4b167874ea828a283bcc6
RMD160 (NetBSD-6.0/sparc64/installation/misc/instfs.tgz) = 1065f4e43c679d182e72c654ce6a236f57350137
MD5 (NetBSD-6.0/sparc64/installation/misc/instfs.tgz) = 75c25d3de06e50ff85364600e40380f4
SHA1 (NetBSD-6.0/sparc64/installation/misc/ofwboot) = 42516022d9da6a2cd08daa28c5e60e68c6accd8b
SHA512 (NetBSD-6.0/sparc64/installation/misc/ofwboot) = 1d63340007e8aefb0f0a725f444f8cd8bd62c68045c0ba6bede194f5d5f4c05a795f728ae00b4cdf65dd82cfa163c2f954ebc2a8caa54b35a908a337f1cac736
RMD160 (NetBSD-6.0/sparc64/installation/misc/ofwboot) = 53bf739509e7c3fe2f4ef8cad8aecec6fed9fd1e
MD5 (NetBSD-6.0/sparc64/installation/misc/ofwboot) = dfafb8bbdf16b3219ff2b38d69dffb87
SHA1 (NetBSD-6.0/sparc64/installation/netboot/ofwboot.net) = 42516022d9da6a2cd08daa28c5e60e68c6accd8b
SHA512 (NetBSD-6.0/sparc64/installation/netboot/ofwboot.net) = 1d63340007e8aefb0f0a725f444f8cd8bd62c68045c0ba6bede194f5d5f4c05a795f728ae00b4cdf65dd82cfa163c2f954ebc2a8caa54b35a908a337f1cac736
RMD160 (NetBSD-6.0/sparc64/installation/netboot/ofwboot.net) = 53bf739509e7c3fe2f4ef8cad8aecec6fed9fd1e
MD5 (NetBSD-6.0/sparc64/installation/netboot/ofwboot.net) = dfafb8bbdf16b3219ff2b38d69dffb87
SHA1 (NetBSD-6.0/sparc64/INSTALL.html) = 0422ec241b04a9456f70ae28d29c7bc2be78b7f4
SHA512 (NetBSD-6.0/sparc64/INSTALL.html) = 15bd5a5a0362c4412c416767608ce7b9580ebe8777ad15fa529af5a2e658a250000fa4a8d9313e2f7f718bc904518943d0ef095d8f0ba50deb50997f6a447fe7
RMD160 (NetBSD-6.0/sparc64/INSTALL.html) = d534bc86ddea48b776c562b00bf67b686b91ea9d
MD5 (NetBSD-6.0/sparc64/INSTALL.html) = 90c6044241bca2933db37134f775bf07
SHA1 (NetBSD-6.0/sparc64/INSTALL.more) = 055a5f58c1a926fdc7d272cf573eddd0c868ef7f
SHA512 (NetBSD-6.0/sparc64/INSTALL.more) = 403ee0491131f45d867e10305a65b801f95bd8fa195bfd4e95499d63ba74851774ef14f1fcc8cbd03220f973050903279438f15cb824ae2b294ead99c7d07bcc
RMD160 (NetBSD-6.0/sparc64/INSTALL.more) = 1744bae9283a6f0e12f95e382f3bedd44bb6da14
MD5 (NetBSD-6.0/sparc64/INSTALL.more) = c42b30d10ac37a1ac1270335414e796b
SHA1 (NetBSD-6.0/sparc64/INSTALL.ps) = 60073507d2c04d97ccd9cbbf5fc51458cfd12509
SHA512 (NetBSD-6.0/sparc64/INSTALL.ps) = 86c33bcc6fb6ff78056914afa17cfbd8efbea1f2100569f4169ba49d1033e4a86ea3ce49d1c5932012712bdf49bd356dcbe59aea4f8271c822482722dc08c7ce
RMD160 (NetBSD-6.0/sparc64/INSTALL.ps) = 51b6801d68738b1606a65969c2ae3c3fa811310f
MD5 (NetBSD-6.0/sparc64/INSTALL.ps) = bfeb86c3637bf7b5af6fcb553b62ab1b
SHA1 (NetBSD-6.0/sparc64/INSTALL.txt) = 518f7cf7bfb65b40a409b78375c0276569ac5540
SHA512 (NetBSD-6.0/sparc64/INSTALL.txt) = f3e1503729f62b588c53747850b7907b92d823f702ec484925d3c0173c6dd866626ef500b8b581f72d54ce4b2aa72788a5543d98c72408a5060067b832e93a88
RMD160 (NetBSD-6.0/sparc64/INSTALL.txt) = 2716b124f107462bce88b2bc12d54d5d5aaaef81
MD5 (NetBSD-6.0/sparc64/INSTALL.txt) = 26984ffe79eab5c070bafdc74267e26a
SHA1 (NetBSD-6.0/sun2/binary/kernel/netbsd-RAMDISK.symbols.gz) = a1a418aa1a4815cd42b51d842b61002b4acded14
SHA512 (NetBSD-6.0/sun2/binary/kernel/netbsd-RAMDISK.symbols.gz) = 272b8dc02294eb259adca2d15a6ea1633066c7708a1d8461d5d299bc7ff5887510fd6f20f490a781813fd61e6f95eed92ef03ba3b4f846ba4ecfe6b7a1d91682
RMD160 (NetBSD-6.0/sun2/binary/kernel/netbsd-RAMDISK.symbols.gz) = c69407e3e5bddd21575929744fd986e7795c7382
MD5 (NetBSD-6.0/sun2/binary/kernel/netbsd-RAMDISK.symbols.gz) = c47a2dcd2bac0d4ade29e3ceba7b8b90
SHA1 (NetBSD-6.0/sun2/binary/kernel/netbsd-DISKLESS.gz) = c6cb928563370afd5d467adbd4d24a0ac37bc8c0
SHA512 (NetBSD-6.0/sun2/binary/kernel/netbsd-DISKLESS.gz) = e24a4feb31212693c9c77732473107dc01cbe19c4d23a3f50a7028079f669e66c5252b9d39904e4d1daab0af8ac020b993f36f027df256a008249a8879d848f9
RMD160 (NetBSD-6.0/sun2/binary/kernel/netbsd-DISKLESS.gz) = b3bd2f4f03f92c810f3b21d3163d4fad1383a0a9
MD5 (NetBSD-6.0/sun2/binary/kernel/netbsd-DISKLESS.gz) = 2d2987883fb76005f8b4ca0686fdb261
SHA1 (NetBSD-6.0/sun2/binary/kernel/netbsd-FOURMEG.gz) = 83451516991b23dddcd67b3d4aed3d4d5eef0af9
SHA512 (NetBSD-6.0/sun2/binary/kernel/netbsd-FOURMEG.gz) = 8ea5066dcd3ab85ca812a169d2a385befcf5717e1dca01c4d5a9daae2368498c1eedcb683aa76d912be0cb6193217c33a5819d8c597521a21fd9ccc5c4f84932
RMD160 (NetBSD-6.0/sun2/binary/kernel/netbsd-FOURMEG.gz) = f8dbe58a262e48a7a1bdddfad8bcbf4c02455f93
MD5 (NetBSD-6.0/sun2/binary/kernel/netbsd-FOURMEG.gz) = f717bd51debc97e9efb5979b5ba425e8
SHA1 (NetBSD-6.0/sun2/binary/kernel/netbsd-GENERIC.gz) = 89a4c10827c07a9100a4cf7692012221e6f7d99d
SHA512 (NetBSD-6.0/sun2/binary/kernel/netbsd-GENERIC.gz) = f53dd3eaf33dac52acfcbd6657269783ca14c1a33ea4e0cdf47fe96f3a620961e07905d5eccc2e15ff75cb0910f9e47d8a37487c7a8b455db494b5690d9ce5cf
RMD160 (NetBSD-6.0/sun2/binary/kernel/netbsd-GENERIC.gz) = dca21703d8350ff9c84920da0fe38b4e96b58c73
MD5 (NetBSD-6.0/sun2/binary/kernel/netbsd-GENERIC.gz) = 5be2117280d7f1f78689744e44903f49
SHA1 (NetBSD-6.0/sun2/binary/kernel/netbsd-INSTALL.gz) = 31619c78adb4831a8ae456942ee6a31c5bc0576b
SHA512 (NetBSD-6.0/sun2/binary/kernel/netbsd-INSTALL.gz) = 08d1e00c5013bfab18920543807657eb480d54d0270d1699a9a583ac0c8fd4df0750924727ee2723c7ad6bd1c0450c5c89378c450b412ab6d6e6c8be723f2b86
RMD160 (NetBSD-6.0/sun2/binary/kernel/netbsd-INSTALL.gz) = 3d651236509dbd65a585082b2adffa0abd47c298
MD5 (NetBSD-6.0/sun2/binary/kernel/netbsd-INSTALL.gz) = d39784e7514b56ad37120e1accca74e9
SHA1 (NetBSD-6.0/sun2/binary/kernel/netbsd-RAMDISK.gz) = abe6ff67ad153c993974749b41bbf6b2e207792a
SHA512 (NetBSD-6.0/sun2/binary/kernel/netbsd-RAMDISK.gz) = 63fcbf6f7b5582b69aa77b4dcfed2125f32b40a622b2eb533eefd92bf829513488e37c4947c44e004ec8006a7e35030fb3e500f1584922101d376e9cc67dc8aa
RMD160 (NetBSD-6.0/sun2/binary/kernel/netbsd-RAMDISK.gz) = aa2ba7c4930e6f557f809407458f12a5fe2da818
MD5 (NetBSD-6.0/sun2/binary/kernel/netbsd-RAMDISK.gz) = 10a7099f6bfc84a63eb89985f93dfab6
SHA1 (NetBSD-6.0/sun2/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/sun2/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/sun2/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/sun2/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/sun2/binary/sets/kern-DISKLESS.tgz) = a6ddacc5894ed442e82fc8ad75e4ba5a85acbdd0
SHA512 (NetBSD-6.0/sun2/binary/sets/kern-DISKLESS.tgz) = 7b0afaf8fcafd005d56142659c130fa14fed8681f0907c262b52ba367db985390c1ada5b3c22705655fb8fdbdb2577f112b2e867dae8a47d30da17db9f68dca1
RMD160 (NetBSD-6.0/sun2/binary/sets/kern-DISKLESS.tgz) = b1566d3099e8654a48e74a7f4b959d22ce72241a
MD5 (NetBSD-6.0/sun2/binary/sets/kern-DISKLESS.tgz) = 320ebbbe9568d045ad4dc2f4284e45e5
SHA1 (NetBSD-6.0/sun2/binary/sets/base.tgz) = cdfdfc1f6d5fd3fab1cd9e5546d1060a4cb16b45
SHA512 (NetBSD-6.0/sun2/binary/sets/base.tgz) = d7a0059ea7babe24d603084826c5a338b33ef3ac4ce1a6039f926692141f28e32d654656a85e995aa000742f8971b33d3e4fad8dbee8a9e6b3cac2c15b9cf01d
RMD160 (NetBSD-6.0/sun2/binary/sets/base.tgz) = 3652a11470a78517f725320750e8a61c5328510f
MD5 (NetBSD-6.0/sun2/binary/sets/base.tgz) = 3505f44ff71277ff11e5402646e110ef
SHA1 (NetBSD-6.0/sun2/binary/sets/comp.tgz) = 40c4eb1b75f3e903a92ea86d8919fd23a56e470d
SHA512 (NetBSD-6.0/sun2/binary/sets/comp.tgz) = a52435520af63ae741654d81bdd5abd611857aaa19dc757bce5c841fa13b15d878780509be22a31ed893a2a4ca1c29548c5a737a2a650a7241805b518f9c70ac
RMD160 (NetBSD-6.0/sun2/binary/sets/comp.tgz) = d51f8b1dbebe3ba6c4c3623351e647981b20f9da
MD5 (NetBSD-6.0/sun2/binary/sets/comp.tgz) = cfdfa248cfcb4e2ae4af64f3f8a4eddf
SHA1 (NetBSD-6.0/sun2/binary/sets/etc.tgz) = 5872a9f5eeb6004802bba9df4a0d4ec1dfccea8d
SHA512 (NetBSD-6.0/sun2/binary/sets/etc.tgz) = 617be23175e9c578e834e039720a0c23357cd24b22639c5641a54c5b36ccb725446cda8e3c3def6e8e73c39c1f90065c348a0ea990f7a50384ba9a65ada10e16
RMD160 (NetBSD-6.0/sun2/binary/sets/etc.tgz) = fa533eae2fafa51d8e2643dbe20be6ac179f5d05
MD5 (NetBSD-6.0/sun2/binary/sets/etc.tgz) = ea43e005a9a50dcbbf704fa27745eb78
SHA1 (NetBSD-6.0/sun2/binary/sets/games.tgz) = 97fe1f17370ab456fea7c21c66dabbaefbb8bf39
SHA512 (NetBSD-6.0/sun2/binary/sets/games.tgz) = 2dfb36fb06f40d0ce519fe3808d53b8babd8bc9ae810c4fadb09e84e93e57b88b893c04cecbaf062c4ed2257eacec30127513aaca5c91ef9234da9226f366c4f
RMD160 (NetBSD-6.0/sun2/binary/sets/games.tgz) = 6affdde9886085d8acd3bef196096dbff77fd99f
MD5 (NetBSD-6.0/sun2/binary/sets/games.tgz) = 7eeaf3572032c8e522266e0283594105
SHA1 (NetBSD-6.0/sun2/binary/sets/kern-FOURMEG.tgz) = 81e4d6b63aa15ea1f60604aa24a21d0eb4ddedf0
SHA512 (NetBSD-6.0/sun2/binary/sets/kern-FOURMEG.tgz) = 117485cb1d871f3e643c009a998998885011c1d2d826ec2db092d0e3569a21fc63618a73a803332fdd936ecb30738e76f75be3e76ee17d71b190a8bb188aa873
RMD160 (NetBSD-6.0/sun2/binary/sets/kern-FOURMEG.tgz) = 3c7e6d14f1a2a9b288f245ad0d6f115469b060c2
MD5 (NetBSD-6.0/sun2/binary/sets/kern-FOURMEG.tgz) = f9ac31754f27f2e22aff03c5ce4e166d
SHA1 (NetBSD-6.0/sun2/binary/sets/kern-GENERIC.tgz) = 05b5ac2735fe34a61df42c13670bf7a758b8c37a
SHA512 (NetBSD-6.0/sun2/binary/sets/kern-GENERIC.tgz) = d5273855067665e77291f477adc2bcf482db2ac749b078502d04bc53b103977f4f16da05acbe917a97d4bba20626dd1954132939cfcf168cf8b75f5e12fe71b6
RMD160 (NetBSD-6.0/sun2/binary/sets/kern-GENERIC.tgz) = bf5c45b36a68224af25aca91aa2eac8d213442c9
MD5 (NetBSD-6.0/sun2/binary/sets/kern-GENERIC.tgz) = baac1013c64948cd3fb2bff22a21e5c4
SHA1 (NetBSD-6.0/sun2/binary/sets/man.tgz) = 9bb0fc4bb2f60df3acbdcbd6d03d11e52227c70f
SHA512 (NetBSD-6.0/sun2/binary/sets/man.tgz) = 3bdb75b91c18982b39f2c2842d9260749491f16c6da624510c603ef0e897298f6ecd620a63a96ccf21354f6a4008b4723249635280e4e956a4668d1984faf7f5
RMD160 (NetBSD-6.0/sun2/binary/sets/man.tgz) = 3b28f01096b5070887994e84b171cd6c59f0ad94
MD5 (NetBSD-6.0/sun2/binary/sets/man.tgz) = 06cfb9c81392e3ae7b04daa01475db38
SHA1 (NetBSD-6.0/sun2/binary/sets/modules.tgz) = 0ff7c443e874d15372f107b6772b5004ee0ff6d2
SHA512 (NetBSD-6.0/sun2/binary/sets/modules.tgz) = ad3913a8edf4494515dab3a4df4a56799f7cb2cf335ca9e4cf78e61a8b92f161cf2148ea2201ce9e9a6966872c5ee749b69642205ef4fa475aae1b3671e49d69
RMD160 (NetBSD-6.0/sun2/binary/sets/modules.tgz) = ecbeaeb92cd60504c907b743ebf8bc729483eb35
MD5 (NetBSD-6.0/sun2/binary/sets/modules.tgz) = 94f595d3a8f52f8b7b202c1715896b1e
SHA1 (NetBSD-6.0/sun2/binary/sets/tests.tgz) = 4651980ab126f0fb7fa3282e401fed649129dd46
SHA512 (NetBSD-6.0/sun2/binary/sets/tests.tgz) = 4fcec0fde688d32f2f53d0c3d6ec9218a2c670f0006b421ab907098f7210481d0ef690d9e8415dd5201be799f83b303a8a2a5b4f8b35af16b311b4996a353b93
RMD160 (NetBSD-6.0/sun2/binary/sets/tests.tgz) = d260affd3d3656aa73171aca7e8f40f2f502a645
MD5 (NetBSD-6.0/sun2/binary/sets/tests.tgz) = 801c9c531d070d152c1f259dd1111f64
SHA1 (NetBSD-6.0/sun2/binary/sets/text.tgz) = b76b4bc516bd7a1747415f9bdbd6af7370cded48
SHA512 (NetBSD-6.0/sun2/binary/sets/text.tgz) = 5654c7217b9d02e327c7803de1dc67d5df4225d1399c3c0a433728fb2fc124ddf0fb2e8cc89d304e03a7c1104a1e1b6ff665b5b29825e3afb0acb43f37fa327a
RMD160 (NetBSD-6.0/sun2/binary/sets/text.tgz) = 89f288a449a1049aabef74f6033bf00611123bf7
MD5 (NetBSD-6.0/sun2/binary/sets/text.tgz) = bc500c71253fd2e2b3bb1dac2bf2b738
SHA1 (NetBSD-6.0/sun2/installation/miniroot/miniroot.fs.gz) = 74ae5c6fe766a40a9bf149885809186bdaae96ff
SHA512 (NetBSD-6.0/sun2/installation/miniroot/miniroot.fs.gz) = 4c7ab73a13c13b3b4cd7a01e12ca25d7a385691dee2b830fb94885c9eece4e62c45cdb9e3961b5426f0e979b2e0f2e951c52774231129f824787ee6daa8d8ed1
RMD160 (NetBSD-6.0/sun2/installation/miniroot/miniroot.fs.gz) = 032b6ddf1452800c8f824983c6d3c1310ba723dc
MD5 (NetBSD-6.0/sun2/installation/miniroot/miniroot.fs.gz) = 6d2927b0648b018b2726f7159e60a911
SHA1 (NetBSD-6.0/sun2/installation/netboot/bootyy) = f97ac30d2c6258791d64d408e295cc2f48e7892b
SHA512 (NetBSD-6.0/sun2/installation/netboot/bootyy) = c0aa6a81071323564d4445ae06b9e0629b32b118bf6c516e28109d3c2d7a72613b8a2005b6f968a43acc6115e3edba57f15c314b77a774cfbb977ff668dd946b
RMD160 (NetBSD-6.0/sun2/installation/netboot/bootyy) = 3de2112b92a222b55ffeff2d5930ed5eccf70a7a
MD5 (NetBSD-6.0/sun2/installation/netboot/bootyy) = 7a6fce60607e791e1c40186b0b7a5403
SHA1 (NetBSD-6.0/sun2/installation/netboot/netboot) = f0ab4e961148d2d8885a47a4fd1b2bc1350e4978
SHA512 (NetBSD-6.0/sun2/installation/netboot/netboot) = b462fb05ea093921b40a1042c15b0013a848c7b022f22523e3dfe73e0141b35982d26825812604bbfcb735f8851fafe9709979d90fb1db21ee552f0e93db6a86
RMD160 (NetBSD-6.0/sun2/installation/netboot/netboot) = c7c6416042d8ed641a8ed033541a7757492229ba
MD5 (NetBSD-6.0/sun2/installation/netboot/netboot) = 52d961060a7035f07cb1946d71736289
SHA1 (NetBSD-6.0/sun2/installation/tapeimage/MakeBootTape) = 1c24969b0bded7ed9bed7203e9c577d896d25566
SHA512 (NetBSD-6.0/sun2/installation/tapeimage/MakeBootTape) = 33eeb495736ac60dabd87faf803defa85f2e400f49b1b4078c16aff18404673a7e668676dc90eb92dc597835ccac8abe26fb68ecb22ddc5afd5afd7de8f2d680
RMD160 (NetBSD-6.0/sun2/installation/tapeimage/MakeBootTape) = 234492df364f78e2aef3c9cd8ac4496077ba2760
MD5 (NetBSD-6.0/sun2/installation/tapeimage/MakeBootTape) = 924860ba84acf16d29a70783d6e106a3
SHA1 (NetBSD-6.0/sun2/installation/tapeimage/MakeInstTape) = 75b8d9faa2e1960a48354922aef5c25c76f66ee6
SHA512 (NetBSD-6.0/sun2/installation/tapeimage/MakeInstTape) = ba61feb33064bdb843f8cd563135cc8fe28e9bbbaf8d1ac2eba2c5e1ad083daba83b47295f3c677caa014c9be7606ef6a40a6c1bc9b07e4eaed139d8299be97d
RMD160 (NetBSD-6.0/sun2/installation/tapeimage/MakeInstTape) = 198879d5058f7ee226c4c669f3a7429bb49c12ce
MD5 (NetBSD-6.0/sun2/installation/tapeimage/MakeInstTape) = 58a95592c402c3f752f4c801fa3d20a7
SHA1 (NetBSD-6.0/sun2/installation/tapeimage/tapeboot) = 1612942dd5bbca5773de6ac02723ea898ea4e172
SHA512 (NetBSD-6.0/sun2/installation/tapeimage/tapeboot) = c5b75882d4b9041edfd74cbf9dc4df8393e687503043e77af766452d780cc8a5584b916d0a792748730e9c87bd52803bbfc0a043720e611079a5cdcce6243bff
RMD160 (NetBSD-6.0/sun2/installation/tapeimage/tapeboot) = 61a36a8d1737147b7b4587353f974c87fca2b442
MD5 (NetBSD-6.0/sun2/installation/tapeimage/tapeboot) = 751ee54f544a63177c273ea6757dbdc0
SHA1 (NetBSD-6.0/sun2/INSTALL.html) = 40cf15ad864adcc8d6216b9d4220cc3eab1463ff
SHA512 (NetBSD-6.0/sun2/INSTALL.html) = 0992031a6cae198c16b08553e94277a39b34886cf0bfea343b95493439d79868cd50a1c2b9f2ce9852068b43beb7ee2a2016480c3ea452b9d62a25b71c72d567
RMD160 (NetBSD-6.0/sun2/INSTALL.html) = f7aaba481555931612488d204c9a256899f04efb
MD5 (NetBSD-6.0/sun2/INSTALL.html) = ccf725a8f39711748946cc414f688dde
SHA1 (NetBSD-6.0/sun2/INSTALL.more) = 21b6ecc61f6d15fd2b54511150b840b242996f02
SHA512 (NetBSD-6.0/sun2/INSTALL.more) = 5c43f22fe85470c7069ea524ff18570a1a3605a956157d9f662cf8b76fa6f52e4eef41e765488ae00b491830cde3b148254ed5bec6b20db0f5661d29ff0f7fce
RMD160 (NetBSD-6.0/sun2/INSTALL.more) = 9adc2008a6388ec7ce907c23ac6f1de1f941c979
MD5 (NetBSD-6.0/sun2/INSTALL.more) = 79afb0a5f30435a2731e455a85884a28
SHA1 (NetBSD-6.0/sun2/INSTALL.ps) = 37c7ffa52cf698e058442a61d937a0ad09fa0156
SHA512 (NetBSD-6.0/sun2/INSTALL.ps) = 09998f504688d7144d223f857edd4273bf00c1c50b1ec47294db2be1abdc050cbf73d0037c0b88279fb1028fde2556bbcadc2d4fbb7466834385905c46f3d688
RMD160 (NetBSD-6.0/sun2/INSTALL.ps) = cac0f152af9ddd8fd643b2578688e7db03a3deb5
MD5 (NetBSD-6.0/sun2/INSTALL.ps) = b2a094aacd3846aa4c121629daa938fb
SHA1 (NetBSD-6.0/sun2/INSTALL.txt) = 90377374ac7c61d0fb6bafdaecd790a86fe7a447
SHA512 (NetBSD-6.0/sun2/INSTALL.txt) = c37c159387cdfa0f40268a16c5a55cef0473b31f51e5c978ec948fe4964d3e9574dd9bbbd758eccb9304a9af6d8216b38aa256676a36537b480a5606b2cb1fde
RMD160 (NetBSD-6.0/sun2/INSTALL.txt) = b16dee40608a6b8e8022df13c27783e34e48279f
MD5 (NetBSD-6.0/sun2/INSTALL.txt) = b33826d2ef20420c36a3723256cd4488
SHA1 (NetBSD-6.0/sun3/binary/kernel/netbsd-DISKLESS3X.gz) = 08ec51564fa19e69828a501a60ac05c687f8471f
SHA512 (NetBSD-6.0/sun3/binary/kernel/netbsd-DISKLESS3X.gz) = d31891f04947a6295085ee899ccb5d532b5e50eb66d1c4f7f7779aae9a1c711b54e142c1d26f45ce10f6ef6815894fe545b436d40cc0ba05149e02885981f029
RMD160 (NetBSD-6.0/sun3/binary/kernel/netbsd-DISKLESS3X.gz) = a71a82ee59a431dd5c7a6d5100d84fd264ca4e4c
MD5 (NetBSD-6.0/sun3/binary/kernel/netbsd-DISKLESS3X.gz) = 261995193793314b7306c5753aed0e23
SHA1 (NetBSD-6.0/sun3/binary/kernel/netbsd-DISKLESS.gz) = 941fbb0e064253228f04d4baaa8e2db4a09b8d88
SHA512 (NetBSD-6.0/sun3/binary/kernel/netbsd-DISKLESS.gz) = 2013168c4985c258bd2f605cda56dd678c394580edf2afa1fb37fd243b748b4d50e049a2f4b314ee5b70f30d181ec684d886094258115a0cdf0e485c120a2628
RMD160 (NetBSD-6.0/sun3/binary/kernel/netbsd-DISKLESS.gz) = 6028696adecea1922e0bdb1d757a4d70725c3639
MD5 (NetBSD-6.0/sun3/binary/kernel/netbsd-DISKLESS.gz) = a49e05dd12628e1fcf1a30e8815ea72e
SHA1 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK.symbols.gz) = be5066fcab92b01e828769c3766ac902896a9dae
SHA512 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK.symbols.gz) = cc0f0b30fa4b28c2fbaf9229568530ee61e17b9ec8515242e53d4d6ac32d1f3ace313c4d138ff173a2301e208ded7389e93551a4a05d7f9a5bf536a8566c2de2
RMD160 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK.symbols.gz) = 832199c94540e3274e2376b17e23144cf99af3c7
MD5 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK.symbols.gz) = fb76fe7920f7c1e0138270f64dd90315
SHA1 (NetBSD-6.0/sun3/binary/kernel/netbsd-GENERIC.gz) = a1458c3d482f170ce26ff1b4ae12fcbe82206cd0
SHA512 (NetBSD-6.0/sun3/binary/kernel/netbsd-GENERIC.gz) = 703f16ca7fb7cddbf33e84bf7859d73ad619464c859927767e6f3e292c63b7da65ed90d20a32cb7364f425c41aff05e0a08f7ba4e7bcfaa04804f4389425d402
RMD160 (NetBSD-6.0/sun3/binary/kernel/netbsd-GENERIC.gz) = 5a542bbb7c45f6940f5c9a63deba729d2d6e8b75
MD5 (NetBSD-6.0/sun3/binary/kernel/netbsd-GENERIC.gz) = 28729a3aab91b7b81a0bb5af1378c67c
SHA1 (NetBSD-6.0/sun3/binary/kernel/netbsd-GENERIC3X.gz) = 50f6f8937a0b18960660908aea8a5cd490175077
SHA512 (NetBSD-6.0/sun3/binary/kernel/netbsd-GENERIC3X.gz) = 2798612a5e0d6d60f80dded7a22bdd6c1f5a64709b65fa33bfe8dfa873c38da0269b0a080185efe2025b5682ba9fd18bb3063a1133f8227cdc41acb03fc4bc2c
RMD160 (NetBSD-6.0/sun3/binary/kernel/netbsd-GENERIC3X.gz) = ecd36a3bb10a07cb7dacfd2a8add6b4e51eddcf1
MD5 (NetBSD-6.0/sun3/binary/kernel/netbsd-GENERIC3X.gz) = 63a3954b02444ee35ebdfc8a280c6eb8
SHA1 (NetBSD-6.0/sun3/binary/kernel/netbsd-INSTALL.gz) = 2da156899061f57aabfe77a55081e1281cc46107
SHA512 (NetBSD-6.0/sun3/binary/kernel/netbsd-INSTALL.gz) = a316b1fb791d3df671cf366c911756cd9e87735247eda75372d73075bd8ec8bfae4aba620740f86cfe8011ffe0614777d480d4979ca021912702ce5699865244
RMD160 (NetBSD-6.0/sun3/binary/kernel/netbsd-INSTALL.gz) = 51a1809d8ddf6701d556d858a18416a72afb0f37
MD5 (NetBSD-6.0/sun3/binary/kernel/netbsd-INSTALL.gz) = b4a756c7502efd5883243e26139d206a
SHA1 (NetBSD-6.0/sun3/binary/kernel/netbsd-INSTALL3X.gz) = 78f041430b2aafc6768aef55603c4427f90c44ed
SHA512 (NetBSD-6.0/sun3/binary/kernel/netbsd-INSTALL3X.gz) = be3a7f26745d4693d2393f0ee044871dca2ec554f8897b7ec54bba85ff3cd75143a548fa0880bb76810d2811018c9c3b5e8cc44fbe163588b82c0344a2f0cf81
RMD160 (NetBSD-6.0/sun3/binary/kernel/netbsd-INSTALL3X.gz) = 9d21135a1448a6fdc1fa052c56434cf809e43046
MD5 (NetBSD-6.0/sun3/binary/kernel/netbsd-INSTALL3X.gz) = 0ff6857605eea16d5d90a85b77335229
SHA1 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK.gz) = 6f90c82aa6150d861190e41dca228549cedcc458
SHA512 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK.gz) = e4e4f5901d286b164f9e4b471f6084a6ba879c91b21b4e86f44650ccf0447698a11f33008d75cb3e12673396094d91e2d715e24b596727cb97c125340fa82abe
RMD160 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK.gz) = 13e4983150a8508d64c022000d27b77c1e0da384
MD5 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK.gz) = 6ce2d84c334909c66b0d54b928fdcb2d
SHA1 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK3X.symbols.gz) = 6c31ae63d1149ae040b2f04fdcce0f41aa8095af
SHA512 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK3X.symbols.gz) = f986b22aef0685dbb454c95d9f0f6b7379a15b365f56f3a13e1357b8b2bf7fb787b09eb4b474d06dd78e6869152e6493177f911929f972a819f688af90854d1e
RMD160 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK3X.symbols.gz) = 18640447fceaf27b7ebfd19917bf40521718c441
MD5 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK3X.symbols.gz) = 639d60631052afddcfb58f64847964c8
SHA1 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK3X.gz) = 78efb67e8cb813af99a12a1c67c9238916bb1071
SHA512 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK3X.gz) = ef27cfba56aff6b13544132e71513598e68e488cb27b7038442279eaec9d6a715cf9196ce98d93acfc4a7e89cb679e5ce787f7c4b14a661a2113b1a987861a9f
RMD160 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK3X.gz) = 556c68e9415f0795d58c4f8c491eed75755a59fa
MD5 (NetBSD-6.0/sun3/binary/kernel/netbsd-RAMDISK3X.gz) = ff2a3fad6f6dd6504713e5874b3eb8cb
SHA1 (NetBSD-6.0/sun3/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/sun3/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/sun3/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/sun3/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/sun3/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/sun3/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/sun3/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/sun3/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/sun3/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/sun3/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/sun3/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/sun3/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/sun3/binary/sets/xcomp.tgz) = e13548b8974f5d55e98caf17b2b4bbbcf2430370
SHA512 (NetBSD-6.0/sun3/binary/sets/xcomp.tgz) = e4b13c798df04ffef88e72100e2db5af400e5ae255cd7a1ca2907736b8138a59045b6f3bd8531b5753d0b87eadb1b5f801043a929bb25d1611e4be693c1da954
RMD160 (NetBSD-6.0/sun3/binary/sets/xcomp.tgz) = 3b1f4e26ddd57e69afd9709553c1bf70d6e29ea0
MD5 (NetBSD-6.0/sun3/binary/sets/xcomp.tgz) = 894bb9aa05b164d9923682165d708335
SHA1 (NetBSD-6.0/sun3/binary/sets/xetc.tgz) = d2ca34bd668e108cbbfd4b45720823663339f8bd
SHA512 (NetBSD-6.0/sun3/binary/sets/xetc.tgz) = 431ca465f300e7eb5d4dbd44c43e412a296fb4d3129217e752f86a720e8a8bdf736c9add05a45518a4f3e6cffa32381ef61e9186d360b68a5f8c64d87e385fb5
RMD160 (NetBSD-6.0/sun3/binary/sets/xetc.tgz) = 4c5d664679856b02b694534cd6afdf20597e6d53
MD5 (NetBSD-6.0/sun3/binary/sets/xetc.tgz) = 72bd67556ac14ea0ce147374af0ba624
SHA1 (NetBSD-6.0/sun3/binary/sets/xfont.tgz) = 24ca4cb77a59246b1b05e0a91fed21c63259d12b
SHA512 (NetBSD-6.0/sun3/binary/sets/xfont.tgz) = 8dc297fb6205a995767e68e77cc21169234dfb66a86bbcd2e1828fd32efeda0a25c9f2a9b1a39b97fe8d7b24f8cd5cf6160859af6be2259876f39ee4e76c405b
RMD160 (NetBSD-6.0/sun3/binary/sets/xfont.tgz) = d2f134959940e6667aca185f01e388c05dc6bc85
MD5 (NetBSD-6.0/sun3/binary/sets/xfont.tgz) = 3c96c1f0a13cd7d277ca35622ac9c058
SHA1 (NetBSD-6.0/sun3/binary/sets/kern-DISKLESS.tgz) = 00b387b99dc84b6128941ae2b849feb76c7f88c8
SHA512 (NetBSD-6.0/sun3/binary/sets/kern-DISKLESS.tgz) = 0b2cf325d14f69c5905bd6d832adfdc806bec302780811c6ad50ecbb646de07d1e91dd0cf16873b0128a44688f9e810d83ae7c6a4415b8393ea0567fe77d5f32
RMD160 (NetBSD-6.0/sun3/binary/sets/kern-DISKLESS.tgz) = 71913a5ec9bab31e74ff27ab154715fd84b34a96
MD5 (NetBSD-6.0/sun3/binary/sets/kern-DISKLESS.tgz) = 6988897d69e304577b9f493c44425ddb
SHA1 (NetBSD-6.0/sun3/binary/sets/base.tgz) = 43ba28352421316629ca0c4a93709af5e5b588cd
SHA512 (NetBSD-6.0/sun3/binary/sets/base.tgz) = 1b0efd643066d4cc5d689a64ba97092091ac8b94ac5244ca7955b27aa6ce60eaec0f020390d9ed60c00f894312fce76d1ca0555ac4626a4c5c140beb57b97ba4
RMD160 (NetBSD-6.0/sun3/binary/sets/base.tgz) = 46456971e8d821c310f3ac304fcda320396916be
MD5 (NetBSD-6.0/sun3/binary/sets/base.tgz) = ed8d7947e7e5282728b94a7fbae41ea3
SHA1 (NetBSD-6.0/sun3/binary/sets/comp.tgz) = 960167513f2de6e53e0512ba9d8a8fa891d7f096
SHA512 (NetBSD-6.0/sun3/binary/sets/comp.tgz) = a6d2d94ad9a752263254736253c7be190d2baa69140ca1e40c06719dfb89dd5660fa83c72392b2c5b30c0caee058cb049029b8414fdd48017fba2615781dc08b
RMD160 (NetBSD-6.0/sun3/binary/sets/comp.tgz) = 9d995295dc5806397c31541ccd589210c0f675b2
MD5 (NetBSD-6.0/sun3/binary/sets/comp.tgz) = 3f277a5219ca41a91776606698660e31
SHA1 (NetBSD-6.0/sun3/binary/sets/etc.tgz) = be0c7f05a0122f7fa7e2bf1cf643080c5b434c1b
SHA512 (NetBSD-6.0/sun3/binary/sets/etc.tgz) = 64d9a05110471e97c99c53479a14ba17dedd7c42f1471e26e2b8e73770d94a27b77ca12ab6cc014f8c131fb6757a26a565d4b8c71e78cab8dbbd21608fa76c33
RMD160 (NetBSD-6.0/sun3/binary/sets/etc.tgz) = c4c29b032978e5a6f803c353d4d5b5ab4c58985d
MD5 (NetBSD-6.0/sun3/binary/sets/etc.tgz) = 68851b8898496b07817e828fee991233
SHA1 (NetBSD-6.0/sun3/binary/sets/kern-DISKLESS3X.tgz) = a63f54a0db8362478ef1f3fbbc18060e3834723a
SHA512 (NetBSD-6.0/sun3/binary/sets/kern-DISKLESS3X.tgz) = 5ff8b7a463fc5e0edd17f47d2c59a01f9c53f23db8e73ba7bb0e67c73cd84d2688a22aa8e6b56bdccc29a0056fafeb6c984ea6a34a01e454a93d62eb2c54b623
RMD160 (NetBSD-6.0/sun3/binary/sets/kern-DISKLESS3X.tgz) = 6ff482b8e336432022ac05a24c376d746fe7f8e8
MD5 (NetBSD-6.0/sun3/binary/sets/kern-DISKLESS3X.tgz) = 1f49b690f568492ba1d2ff5cfd76ec6c
SHA1 (NetBSD-6.0/sun3/binary/sets/kern-GENERIC.tgz) = 9eb75c58a99f7b046dab7888235bf8162b10834c
SHA512 (NetBSD-6.0/sun3/binary/sets/kern-GENERIC.tgz) = 9cdcf4835398df2124cc418b977f1cdffaa87d4edaff2c61ded06b44cbd5aa656743e74abd4b5ccd5f7232e0890f4786523d4438feed288bf77df1482fbbb9fd
RMD160 (NetBSD-6.0/sun3/binary/sets/kern-GENERIC.tgz) = 8a50ed11d968a35170c50ddfcd51c3cfeaa7dfc0
MD5 (NetBSD-6.0/sun3/binary/sets/kern-GENERIC.tgz) = 510f2e6a9de0cd18e8dfc6cce74964d2
SHA1 (NetBSD-6.0/sun3/binary/sets/kern-GENERIC3X.tgz) = fc31315a9d236278dc8b6dda1e1385680e35eb4c
SHA512 (NetBSD-6.0/sun3/binary/sets/kern-GENERIC3X.tgz) = 9de034f2f6ee9cf0c65ba4fc41bf7147205c9918263131957d8555c7d54997d98dead2108ec2fac89805d371e673bd66abb93715b61f3c7a71c9db8caf761c40
RMD160 (NetBSD-6.0/sun3/binary/sets/kern-GENERIC3X.tgz) = d1b03f8e7188324e362e4fbd7764d4218a3a98d4
MD5 (NetBSD-6.0/sun3/binary/sets/kern-GENERIC3X.tgz) = b8b84407a1e0d4e215a494aee08eafa8
SHA1 (NetBSD-6.0/sun3/binary/sets/modules.tgz) = 7c6df8904730e2b7d3ea7e98d3a88ee7450e937c
SHA512 (NetBSD-6.0/sun3/binary/sets/modules.tgz) = 95d55d7951896249f4b1b3c38ba80c75c415ffee77e3294f40f8680756cb1cd5b5c2b74740ccd66f218474f220f9603feb81bf9e6d72fa130a9ec3c595abba63
RMD160 (NetBSD-6.0/sun3/binary/sets/modules.tgz) = 485905b274e58b342035d5485507e7c018ee62f0
MD5 (NetBSD-6.0/sun3/binary/sets/modules.tgz) = 5350f1aec24cf27edd1c4d50c916ba33
SHA1 (NetBSD-6.0/sun3/binary/sets/tests.tgz) = b76bdb5e8311a0fe1f22aab836007d5cbaf92618
SHA512 (NetBSD-6.0/sun3/binary/sets/tests.tgz) = 475835a862280be0709e09a8ea5eb132d8c068c63df12dcc925ff3268250afbde228b593f66061d2b946dabc19eeebbb1a4fd313a751c4e5fd8a63ba63f2fdc4
RMD160 (NetBSD-6.0/sun3/binary/sets/tests.tgz) = 58e77c597950e782b3c75967d864ebbc961ffede
MD5 (NetBSD-6.0/sun3/binary/sets/tests.tgz) = 729752bb47500babd316c4776ba02e80
SHA1 (NetBSD-6.0/sun3/binary/sets/text.tgz) = edf2e1f8352a2c45287ecff9c473da4673473a6f
SHA512 (NetBSD-6.0/sun3/binary/sets/text.tgz) = 2098084f46640c2f45b0fb916a2123eacc2e390faaa04e3a8ea99d93f41c4ca5ba77d4192805b13a053ad8262aecccc2228448c9f048c03c96ca51869f342ebd
RMD160 (NetBSD-6.0/sun3/binary/sets/text.tgz) = 1adc9b33d9aec637e1f8e7888f07076d500b2ea6
MD5 (NetBSD-6.0/sun3/binary/sets/text.tgz) = 60c2eabb16dd9df538a98153e2dc7dc7
SHA1 (NetBSD-6.0/sun3/binary/sets/xbase.tgz) = 404b75757bcc91b281fbd5ec9b9333e6c106579d
SHA512 (NetBSD-6.0/sun3/binary/sets/xbase.tgz) = 0f1604049c7e1caac48d610304a1490d2e55dcf3f6a325139eaa57a38211836c479b1dc070cc3b8b0afcdddac7a84b679d96330f65f86015a80b7800fdca633e
RMD160 (NetBSD-6.0/sun3/binary/sets/xbase.tgz) = b1e258608f09cd175c9e15844371daff5fe0f027
MD5 (NetBSD-6.0/sun3/binary/sets/xbase.tgz) = a7ecfb2c66b224ecf2460fe5c6715f0b
SHA1 (NetBSD-6.0/sun3/binary/sets/xserver.tgz) = 88091f11f474465d1a996569643d6e22cf08f717
SHA512 (NetBSD-6.0/sun3/binary/sets/xserver.tgz) = 36de7d4b8be72d53d5f90a144d278359da55008917d2747419d3347d53989e9ceca6230fb2596d4ce6257d1bc9f171c6b9a8e738af86930adc8535e726a8e356
RMD160 (NetBSD-6.0/sun3/binary/sets/xserver.tgz) = e161f48499d8f1ca64b22d65365704f65344875a
MD5 (NetBSD-6.0/sun3/binary/sets/xserver.tgz) = 6ed6442ab169c491e991c8074512bb48
SHA1 (NetBSD-6.0/sun3/installation/miniroot/miniroot.fs.gz) = 8b01f52ec164477fb5ec72725d941e3330c7da94
SHA512 (NetBSD-6.0/sun3/installation/miniroot/miniroot.fs.gz) = fb6995a81aeb2f9fb89e096290cf0921e92b1b2be0adaf95dd2b62c89bbd047b0879231d4e5805a68bd023026ce563bf6f0fc3214ecfabc449650dee9790385f
RMD160 (NetBSD-6.0/sun3/installation/miniroot/miniroot.fs.gz) = 54d3f3d3782536c89d2b6bbf6abb793617c83925
MD5 (NetBSD-6.0/sun3/installation/miniroot/miniroot.fs.gz) = f9994dd6a5b9837915a4ff01a7c9baa2
SHA1 (NetBSD-6.0/sun3/installation/netboot/netboot) = 88a9ccc7444c4692483705ef90da4af422b33497
SHA512 (NetBSD-6.0/sun3/installation/netboot/netboot) = f0810823fbe7335cc83516a71bdf4d5ab281fc2fb71309d76cb979e741af00d81b55e4b298af07e4932ec8e70a79518ace9af7ebc3b5d8a3328555785a8c2d2b
RMD160 (NetBSD-6.0/sun3/installation/netboot/netboot) = 57b89f091bf51dfda7a3b1b3e92bdd0c7e96e958
MD5 (NetBSD-6.0/sun3/installation/netboot/netboot) = 9041ee544a0625228cbbb93e06d2961f
SHA1 (NetBSD-6.0/sun3/installation/tapeimage/MakeBootTape) = 2b66897333fbeba29a086477f6d25026b6d60e55
SHA512 (NetBSD-6.0/sun3/installation/tapeimage/MakeBootTape) = db9307323e7dc0d394fb2b8e012c86df51109357851f2c5ee7894468d4e3b54465ca5015141621fd9d28aeb7a2eef0beac987fc7faeee223f9571ff4589d1036
RMD160 (NetBSD-6.0/sun3/installation/tapeimage/MakeBootTape) = 986f20c8bab33006ea857b9f925b964ab43b9b03
MD5 (NetBSD-6.0/sun3/installation/tapeimage/MakeBootTape) = 68ea84a134fa2781294912ca85d4b1ee
SHA1 (NetBSD-6.0/sun3/installation/tapeimage/MakeInstTape) = a242866885b14ec2e624319b2fe1fc4f940a422f
SHA512 (NetBSD-6.0/sun3/installation/tapeimage/MakeInstTape) = bd5950ce181ae973784c56bed18085fdfd6fd8b1a46d83564c798c130bc265f7a811cb896be8b9f8f78f014aa1a1d37e34b307a7f806ad8f7790ce2308856b61
RMD160 (NetBSD-6.0/sun3/installation/tapeimage/MakeInstTape) = cb0330a75c5b5fa600e3789d81ff75f1c1d81078
MD5 (NetBSD-6.0/sun3/installation/tapeimage/MakeInstTape) = fdc38b08d909159925b6a4efca11a498
SHA1 (NetBSD-6.0/sun3/installation/tapeimage/tapeboot) = 276be9760458783489c93cea994dc73289ace945
SHA512 (NetBSD-6.0/sun3/installation/tapeimage/tapeboot) = 7d6f1a348f99e42960f1697105d2771228a921860b0bd78624e96efcd9dde839412511e414f824ea78f4d5c938141c96d896acb3af426fbb393bbc69825c6c57
RMD160 (NetBSD-6.0/sun3/installation/tapeimage/tapeboot) = 585b74fc55271ea82d4c2e3a38cd3bd7fc9cebd4
MD5 (NetBSD-6.0/sun3/installation/tapeimage/tapeboot) = 6e786930e4f263c8fd65c0684b10f17f
SHA1 (NetBSD-6.0/sun3/INSTALL.html) = e1f85d7f20325efe213999b96d606094b64a2c3e
SHA512 (NetBSD-6.0/sun3/INSTALL.html) = 571b45b18a5fa33d705470440bfc49b38b19f5625aa26b79241f27232a4e902c1689e6acfa563d95204a6bc5a9955565fdd915673db770f739003c8838a7b680
RMD160 (NetBSD-6.0/sun3/INSTALL.html) = 10984fb54cdde46e9dfe3567fff0a41e2f2fddfa
MD5 (NetBSD-6.0/sun3/INSTALL.html) = b221aa68f0c0063c7d4ceb493971636b
SHA1 (NetBSD-6.0/sun3/INSTALL.more) = 3f54bfd9cb004fe817d400878e005dd81f1d2850
SHA512 (NetBSD-6.0/sun3/INSTALL.more) = 6b56e0d51f7423f74aa64a269fe9a0787b0a71d54f8582242d874066df05cc6bb4840ff6d3fd2817f4919b707ff621ca00a77f32938ece1dada70d8a3e6eedef
RMD160 (NetBSD-6.0/sun3/INSTALL.more) = 40aec024ba64af07553ff888f3140e1e6793ac68
MD5 (NetBSD-6.0/sun3/INSTALL.more) = 9c67417a5a30c173b09f1a8b518032d1
SHA1 (NetBSD-6.0/sun3/INSTALL.ps) = b83169ce92b3dc6bd83a1ca5e858120f8b91db45
SHA512 (NetBSD-6.0/sun3/INSTALL.ps) = 546cc770038d3194ca5dfab82113644f6b4fbb5930b9560c14550234858d0bb0edc7fe1fe24d44413042dc7f2c64dc5e4002f3fb60b9893c76f0b80dbe02ae02
RMD160 (NetBSD-6.0/sun3/INSTALL.ps) = 1110e213ad092c715e1c5314d9c11d4e4d0cd977
MD5 (NetBSD-6.0/sun3/INSTALL.ps) = 848bd14e183958c567bdde20345d9b7b
SHA1 (NetBSD-6.0/sun3/INSTALL.txt) = f37f80d7db1daf78f66d91f7f76ac9caecc69d65
SHA512 (NetBSD-6.0/sun3/INSTALL.txt) = aaa6abd3508681ed137796f3f64187fe0e9c7ec26f0d0f81ae73a4e2853ef5aa6b70e861ba69fad5ab51d62f1b09440a0f3f033ef2443a8b12ffc2129e5efdb1
RMD160 (NetBSD-6.0/sun3/INSTALL.txt) = dfcd996574e4d266e95640301662f4928e3a343b
MD5 (NetBSD-6.0/sun3/INSTALL.txt) = 194d8eac56ca75bff60e3d8c6e003ab1
SHA1 (NetBSD-6.0/vax/binary/kernel/netbsd-GENERIC.MP.gz) = 71d44bf3bd2a39ac207a085a22a8310014e4b062
SHA512 (NetBSD-6.0/vax/binary/kernel/netbsd-GENERIC.MP.gz) = 875f45349bf522e4f63d8f1261433b3e109df5ce7c3ee1edd13e2d4526616b7863507dc305d954300308a4c650b9fa12168219d0deb6622b517442b34e10bcfa
RMD160 (NetBSD-6.0/vax/binary/kernel/netbsd-GENERIC.MP.gz) = 357c54bfe13253c5fec2c4ca6cd18909b6a3a6a4
MD5 (NetBSD-6.0/vax/binary/kernel/netbsd-GENERIC.MP.gz) = 5aa951b5c5cb5592f950fa3882b2f60f
SHA1 (NetBSD-6.0/vax/binary/kernel/netbsd-GENERIC.gz) = 9b77cf6513c4cf4fdb0081400a46c34a1c0cf04b
SHA512 (NetBSD-6.0/vax/binary/kernel/netbsd-GENERIC.gz) = ff9d18b6c5d3bcd7bf4ea6e341e77180a7c68b034c61e02dc3d540389e9690faf7a157587847b3f1c25199cee097178b0b658118daddd136aa5f5856dcec1c38
RMD160 (NetBSD-6.0/vax/binary/kernel/netbsd-GENERIC.gz) = 2ed49d64a3f42845c08a8ae30d45569492895cc3
MD5 (NetBSD-6.0/vax/binary/kernel/netbsd-GENERIC.gz) = 26548ca549cfa4f7c9be70e52a981d69
SHA1 (NetBSD-6.0/vax/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/vax/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/vax/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/vax/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/vax/binary/sets/kern-GENERIC.MP.tgz) = d4cac92dc06ec7b3c4ec9ca0d60dd1ace1814e50
SHA512 (NetBSD-6.0/vax/binary/sets/kern-GENERIC.MP.tgz) = 3b10e201e963285a8c3d9319e818510701756a3c33003a5536d273e886cc0b3c02ffe10e762d5d7b4aee2d0098d3600396afc2baafea64cc268fa032203a46f9
RMD160 (NetBSD-6.0/vax/binary/sets/kern-GENERIC.MP.tgz) = d9b4a6a9c85abeee0178536cd1b5c4f0322067a4
MD5 (NetBSD-6.0/vax/binary/sets/kern-GENERIC.MP.tgz) = 15126f47c08adefe93c7261b6eeeddec
SHA1 (NetBSD-6.0/vax/binary/sets/base.tgz) = 9b83f88ab7d06a112cfff79b6657912bddb77987
SHA512 (NetBSD-6.0/vax/binary/sets/base.tgz) = de7c42524163a5d9de7fda440cd2523834d5c46e4e992724a4a7af5e0768e5b105909e9a60e603b2a942cade3795cf4cc2bb22c7627453eda92bf662da83a16a
RMD160 (NetBSD-6.0/vax/binary/sets/base.tgz) = 3059d6e4a7d54ebc9c529f90719ca561ebbe8036
MD5 (NetBSD-6.0/vax/binary/sets/base.tgz) = d06066c8a452b8a7831de680a420ea4c
SHA1 (NetBSD-6.0/vax/binary/sets/comp.tgz) = 3ad25c223276936b26916121fe8341524e2e51a6
SHA512 (NetBSD-6.0/vax/binary/sets/comp.tgz) = 852dc3d7a01fc8e11e66723930d90e7141d974b976d8f7986f068fb5202dad396ef2762400c1e9e02b0d6c5a087435a56f65e502079eec9d9e86c24cf81782bb
RMD160 (NetBSD-6.0/vax/binary/sets/comp.tgz) = a4e74e817807d4d79a43ac450109b14474fda740
MD5 (NetBSD-6.0/vax/binary/sets/comp.tgz) = cf51957de39d3a72c7dda2633d39e705
SHA1 (NetBSD-6.0/vax/binary/sets/etc.tgz) = 21f4ff8b449498081fd6dde92b43eaca86d886ef
SHA512 (NetBSD-6.0/vax/binary/sets/etc.tgz) = c6e43ea57dd0b53f9e0a995aa872d9f7f1d8852fbe7105e125116f577d0ce2acb1e8deddc6a8dcb51e272a982cf549706b23bb26b0acedad416ceef8910a6461
RMD160 (NetBSD-6.0/vax/binary/sets/etc.tgz) = dd7a5f74812906801bf474d265a4846f069c00ef
MD5 (NetBSD-6.0/vax/binary/sets/etc.tgz) = fbb2c18ea3be98a3624f54b834399ee9
SHA1 (NetBSD-6.0/vax/binary/sets/games.tgz) = cd79679fa36349431355dcd14b8a058476f20086
SHA512 (NetBSD-6.0/vax/binary/sets/games.tgz) = f68c931765849554222cc439d5761411d36e8e95b4331d15525e73f77b31249c220b9de4c2061f5e35207e75e27d5d6ade99a7eadf2baf54fc8eeb16ac243dbe
RMD160 (NetBSD-6.0/vax/binary/sets/games.tgz) = 7da1dd4ece7e9062635f7b0766af245e2624c032
MD5 (NetBSD-6.0/vax/binary/sets/games.tgz) = 995e32bef1b920f33c5d7eb946441682
SHA1 (NetBSD-6.0/vax/binary/sets/kern-GENERIC.tgz) = 6e61cfb380e7cf45e2094ef64d26d44ee33fd086
SHA512 (NetBSD-6.0/vax/binary/sets/kern-GENERIC.tgz) = 0d35a882e1e98796099bedc0157fc30978b2e43ef0627be603036259069dc13503fee5dd57b2e9da74dd5c2cb1bb823f4bf91d98259d84b993977bae05d8c10c
RMD160 (NetBSD-6.0/vax/binary/sets/kern-GENERIC.tgz) = 3e2f321e7934e6597ea83627934e88d2214baad3
MD5 (NetBSD-6.0/vax/binary/sets/kern-GENERIC.tgz) = 2efb4b8d6a76ebb59f4ce1171e1502db
SHA1 (NetBSD-6.0/vax/binary/sets/man.tgz) = 9c2da95fb72f9fd1eaa8f7baf5fe45496c8b4615
SHA512 (NetBSD-6.0/vax/binary/sets/man.tgz) = 1a158e92b8414d92745a06fe1c721047e87fd768d7158e078c3a17de1d2634463f8f6b022a001f5f9010e14f0eedf6d5a94e4b73b4dd0ed9ac4576abbf2aeaf4
RMD160 (NetBSD-6.0/vax/binary/sets/man.tgz) = d2485fd19bffbc6103f8aa7521e52c8f652fafa2
MD5 (NetBSD-6.0/vax/binary/sets/man.tgz) = 83e81f5c82acdf7ee76ef5854eca3f5b
SHA1 (NetBSD-6.0/vax/binary/sets/modules.tgz) = 4e60e0177f0e5d6e42ee875a7c30cc612c22ca64
SHA512 (NetBSD-6.0/vax/binary/sets/modules.tgz) = 0e5926279b9534cb9bd34ff0c4d49fa4cd467966829d080ec8ab739039536f42173c2ec91aea40ca37d94625a5b395c3f55ce7936f963bd8d970d1a81fcff8af
RMD160 (NetBSD-6.0/vax/binary/sets/modules.tgz) = 54b0d690b929b4df06adc044864e76d47f09c844
MD5 (NetBSD-6.0/vax/binary/sets/modules.tgz) = aa18ce8768d9fd88206d465fd1ae155d
SHA1 (NetBSD-6.0/vax/binary/sets/tests.tgz) = ec975a37d490f45fb9e7f0f30f0ec46fc0038531
SHA512 (NetBSD-6.0/vax/binary/sets/tests.tgz) = 33b8686c97da76e031a9678103ba99a83cc4189545b4020241a83084585f0eb200c0cb44634e578c8882cd849f80151383a54511c80d1a35964a3567f8e6b314
RMD160 (NetBSD-6.0/vax/binary/sets/tests.tgz) = c27fcdf838dfc43c1527fde3fba2735065a5d12c
MD5 (NetBSD-6.0/vax/binary/sets/tests.tgz) = 3789eac68014743577dfe22503dd98a5
SHA1 (NetBSD-6.0/vax/binary/sets/text.tgz) = c7dcf12545842d5563887c983d694fa569d25ab3
SHA512 (NetBSD-6.0/vax/binary/sets/text.tgz) = 50a7ad69dc23c0d178a3f835e9bed7305cbe432a92cf947e64e28316ac94ddd7def90e91c53515d0cd3ac26c5df23d3f60932ba688ab04525d635555b6078a6a
RMD160 (NetBSD-6.0/vax/binary/sets/text.tgz) = 24ae1e1016b492c3dd575099936598d00d8ac6fa
MD5 (NetBSD-6.0/vax/binary/sets/text.tgz) = 99863a74757bc9677befe8c6ef049685
SHA1 (NetBSD-6.0/vax/binary/sets/xbase.tgz) = 35f6a1b09f246736f7c1c6049831ca0704fe53d1
SHA512 (NetBSD-6.0/vax/binary/sets/xbase.tgz) = d2d0b6a5702e4a87e4823ce7478f5c8f1527045d96040680960e3d96f98f64e40ac7263ad2ca4c4272d4d253fcb41b7e296a9d27f9ffa7dc03be6c437835878e
RMD160 (NetBSD-6.0/vax/binary/sets/xbase.tgz) = e2455fc04171cd3aef8662a36820552811c7657b
MD5 (NetBSD-6.0/vax/binary/sets/xbase.tgz) = ec8b6356f82d4ccb90e93dd78e5c741f
SHA1 (NetBSD-6.0/vax/binary/sets/xcomp.tgz) = ebadf7653d6173ccc73d1228338a4df09eb7dce2
SHA512 (NetBSD-6.0/vax/binary/sets/xcomp.tgz) = a4c4cfaaea473e617870a92bfce40fbc2e34413f31ec2159554031c4312dc1e605079a9e0fd3774967bba9cc27bf9efc97899f09959bd401726ceeeec7006790
RMD160 (NetBSD-6.0/vax/binary/sets/xcomp.tgz) = e6d1591328fcc0af7336d22a5426f72c8343ad49
MD5 (NetBSD-6.0/vax/binary/sets/xcomp.tgz) = 1cc0fa0971f06c2bd9e296db9d6d6fa5
SHA1 (NetBSD-6.0/vax/binary/sets/xetc.tgz) = 2c85388aa75c419cc5651c053b00710471a87350
SHA512 (NetBSD-6.0/vax/binary/sets/xetc.tgz) = 38c9f1abf9d6ff828c361a2c58690f70a7d707aa739f7ded9f256b414e7b78c11ff4f915a59f99884afd5ef0b874c1c1b70aa438f6bdff07e354d67d2257bf72
RMD160 (NetBSD-6.0/vax/binary/sets/xetc.tgz) = b1186ef0c975fa76d8d4d0c62e24ae86cb3d433c
MD5 (NetBSD-6.0/vax/binary/sets/xetc.tgz) = 46ca22a86d8052c1044717eb907d7874
SHA1 (NetBSD-6.0/vax/binary/sets/xfont.tgz) = e9a33e8d99378b3944fc21aa983c02707831f0bf
SHA512 (NetBSD-6.0/vax/binary/sets/xfont.tgz) = 3d09f4edf49bfa98c7817cdb1b91aa4ec902dc87c1765e9a92f143bf4918a60e011fe614a70d9dd926ca2a019a33e5f0dc2d7198205f2273e755049dd8d5a2a1
RMD160 (NetBSD-6.0/vax/binary/sets/xfont.tgz) = 9ae96f8628d6a35053f093acee972bdd8b6722cd
MD5 (NetBSD-6.0/vax/binary/sets/xfont.tgz) = 6e95a6fd030425bb19f0505b9308d20d
SHA1 (NetBSD-6.0/vax/binary/sets/xserver.tgz) = aa6c021940fae979a4a3a3cd01a138808c49621d
SHA512 (NetBSD-6.0/vax/binary/sets/xserver.tgz) = 540455f89e8dfe7d376322ab95288e96e54aa0531aab7cb96e0b1354f85ef64778cbf265bfc2fca7827879a286d80b15f9e973d62594e5bf02df4b0add15be0c
RMD160 (NetBSD-6.0/vax/binary/sets/xserver.tgz) = 84dee738690e4487d7f28cba1d3f61ab9eb12240
MD5 (NetBSD-6.0/vax/binary/sets/xserver.tgz) = e2df86b6bfa5259e72d02f40de72491b
SHA1 (NetBSD-6.0/vax/installation/bootfs/boot.fs) = c4d1340eb7d2ea1d555a46fcf2071fa2f091d1d1
SHA512 (NetBSD-6.0/vax/installation/bootfs/boot.fs) = 155e7faaf14e619309c3ca8cc163e08d6bd3cac41b4d38dc5dc4cd4736e99003a2b6a6d0ffb6cb698772f43d31f577732b9933a71fb1f6315af16dfae34d4f66
RMD160 (NetBSD-6.0/vax/installation/bootfs/boot.fs) = 52a2200dbef9e624d7ca91eb6d2110599b8010b1
MD5 (NetBSD-6.0/vax/installation/bootfs/boot.fs) = f02d1ee9291b0dd37f0dc75d0a27ad10
SHA1 (NetBSD-6.0/vax/installation/miniroot/miniroot.fs.gz) = 5aed885a0dde234caed611cca0f474dc6cf8461b
SHA512 (NetBSD-6.0/vax/installation/miniroot/miniroot.fs.gz) = 3d647213e86ccb976107853ab81ec82d339d2cc7c7bda4cc0c416f55e521c7a43a73523c7f9ff6e590531f019badcd414f1bdab4c95db0806a606065c6803ef4
RMD160 (NetBSD-6.0/vax/installation/miniroot/miniroot.fs.gz) = 6ddb63ee115a7b4a78546092ac7b8cee7c056921
MD5 (NetBSD-6.0/vax/installation/miniroot/miniroot.fs.gz) = 0f860ac5f201e72f4b22392027b25117
SHA1 (NetBSD-6.0/vax/installation/netboot/install.ram.gz) = 8f48dc649fe71a6f861a9f96e2b63679971b762c
SHA512 (NetBSD-6.0/vax/installation/netboot/install.ram.gz) = 857920e536fe08897d7e20c62c74a5606ea791d97111740222c8998ee4a1a9ae05f52ac27b5580d5f78a70f75d5a9ea9446802a85297a13411b98ebe8a84a3e1
RMD160 (NetBSD-6.0/vax/installation/netboot/install.ram.gz) = 9c94e2d64328f88eb7157d6a47e5aec15b1da2a3
MD5 (NetBSD-6.0/vax/installation/netboot/install.ram.gz) = 23d3d0b714186b3e2e7b70d5f0a1ce21
SHA1 (NetBSD-6.0/vax/installation/netboot/boot) = 738e9f280ab9874a2eb21af8627d70e2abd2ba0c
SHA512 (NetBSD-6.0/vax/installation/netboot/boot) = 934496142a69c247ce8b96ea6de19e0cbe0d9e4e8a1ae1043f4d60926c6b3deb30132f3bdc346b3422f1fea02da3a3feaf900d64a9759d265b33fe81ab3da070
RMD160 (NetBSD-6.0/vax/installation/netboot/boot) = 8a60febc408768e89ebb1a834bffaa06a72d3503
MD5 (NetBSD-6.0/vax/installation/netboot/boot) = f238dc1fafca2d7f1a83a22b6cb31091
SHA1 (NetBSD-6.0/vax/installation/netboot/install.ram.symbols.gz) = 6b57f7a857c24539639cce3ed41ed5de16ec233b
SHA512 (NetBSD-6.0/vax/installation/netboot/install.ram.symbols.gz) = e9ac802e948631231eda425e5bab76bf14bc59462613eb6c1025708d67f54c67e9b928cbf33253473605c6239842667eba5e58240d6cb53afc5c80c8546357f5
RMD160 (NetBSD-6.0/vax/installation/netboot/install.ram.symbols.gz) = 5a6ea2a19caafd034e75e9440d1e139d1af3ef0c
MD5 (NetBSD-6.0/vax/installation/netboot/install.ram.symbols.gz) = 712856f65322faccd1b116c548d40196
SHA1 (NetBSD-6.0/vax/INSTALL.html) = 8d850ffd9374c9bdc63e37fa14dc63eda02a93fe
SHA512 (NetBSD-6.0/vax/INSTALL.html) = 3c93355dbfd0cbad25eb74c8497975855706a20dafe8a1baa64d0b53801d5e02d81a93c3347c52950fdd424623cbb58c7a0aa58b7989e89116b157f9d18bef51
RMD160 (NetBSD-6.0/vax/INSTALL.html) = 42d4d65000e12d8f2cbe30f6743d84d7a040bb05
MD5 (NetBSD-6.0/vax/INSTALL.html) = bbad6392f6badb4a1d0ee2977b9df90c
SHA1 (NetBSD-6.0/vax/INSTALL.more) = 624b96b802364eb833ca077dc706d75b5f68f336
SHA512 (NetBSD-6.0/vax/INSTALL.more) = 300954b8c1f3c47a1a45d773f8c7bad3c12dbc292485b7a7206289f56584ce12befa89b8d315511c3e3f7c407de9ca068fbdaae9d75c52dff72a8f8a9b5e4a55
RMD160 (NetBSD-6.0/vax/INSTALL.more) = ab1b5238e9eb5ad1c937c75901199023c08d2434
MD5 (NetBSD-6.0/vax/INSTALL.more) = e634a224d0d67c5cc1c7dffd98156b79
SHA1 (NetBSD-6.0/vax/INSTALL.ps) = 57710ab0d4450b0692b07eb812c9c7e6d70a49d5
SHA512 (NetBSD-6.0/vax/INSTALL.ps) = 0afac140d664edbb10a41c6dd646c797b78e37c1d36aa4804653fbef64171ddf9688d7220755553ff4add8891bd97f083bde4b03c93fe579f1e9d030c83f3d07
RMD160 (NetBSD-6.0/vax/INSTALL.ps) = e84f468d6f328faa9889cf2ad76cab5846c45da0
MD5 (NetBSD-6.0/vax/INSTALL.ps) = 51b9a2a1046494b2443503636fa223da
SHA1 (NetBSD-6.0/vax/INSTALL.txt) = 0a4b816342615d9a4e60cbc99bfea81afe41d0f1
SHA512 (NetBSD-6.0/vax/INSTALL.txt) = b1853a3be6cd17449c4e5539945812b9ba87e02e07877dfe791e04c3db39ed1577e795d7d5e3b1048fc60eca5694592bb5671982c24aaa1f78e3783ede603ded
RMD160 (NetBSD-6.0/vax/INSTALL.txt) = f774566773432a691acf285d569dabcf5037ecfb
MD5 (NetBSD-6.0/vax/INSTALL.txt) = 43a9353717d9309c11fb5f4a9685942d
SHA1 (NetBSD-6.0/x68k/binary/kernel/netbsd-INSTALL.symbols.gz) = 0e3d568ade336cd6e538d567ea11a965d49588cf
SHA512 (NetBSD-6.0/x68k/binary/kernel/netbsd-INSTALL.symbols.gz) = 721a7e1ec94d9a37f37eb30b96d66b19d387090964ee93233de36771e4d106e72cdbeb4b0b17f8d490d9f64484b8a511526841c511b2d33624bd626b4388b623
RMD160 (NetBSD-6.0/x68k/binary/kernel/netbsd-INSTALL.symbols.gz) = 176ba2b94555c5bda82260a075ba295021d61e8a
MD5 (NetBSD-6.0/x68k/binary/kernel/netbsd-INSTALL.symbols.gz) = aa03bcea1e574f24c1ef0eeb364f1c8d
SHA1 (NetBSD-6.0/x68k/binary/kernel/netbsd-GENERIC.gz) = 9a7796c7abf7b4828ee7eb7125b73f5ccd25e2db
SHA512 (NetBSD-6.0/x68k/binary/kernel/netbsd-GENERIC.gz) = 58c7f85cf73d90a04c42c502a3dedd779d9f5013a58f0df8a279cf559dedb80b6bdf2a22804284fa58a7df9c1ba8a8f0454d3f76a43787a7372afb58d5d734b3
RMD160 (NetBSD-6.0/x68k/binary/kernel/netbsd-GENERIC.gz) = efce8e9537f895492b5becab7878dec07fafe7c1
MD5 (NetBSD-6.0/x68k/binary/kernel/netbsd-GENERIC.gz) = adf275bc0aec4a8ba50661637b7707e5
SHA1 (NetBSD-6.0/x68k/binary/kernel/netbsd-INSTALL.gz) = e5fee2392b742d34a50bd2236dc4ba26acdd57e2
SHA512 (NetBSD-6.0/x68k/binary/kernel/netbsd-INSTALL.gz) = 6aa21ef102224a5194fd6709b70a47788e073ad0d325dbe3b982a19a24c67b421acb003496ddf42804785b0dae9bb32dccde617e7d6d9185511205ab390cdcb3
RMD160 (NetBSD-6.0/x68k/binary/kernel/netbsd-INSTALL.gz) = 7251ea070e26aa877c85b9eabffd62cd6ee4b04c
MD5 (NetBSD-6.0/x68k/binary/kernel/netbsd-INSTALL.gz) = e5e8b74d980cbbbc0324cab3da12af0b
SHA1 (NetBSD-6.0/x68k/binary/sets/games.tgz) = 16ee93ea4d5fac2fb28fad5c32c8a3d57d011d79
SHA512 (NetBSD-6.0/x68k/binary/sets/games.tgz) = a7f49ab72f50273f66c8dcf408b57ec4668debd5537c5bd0ec59aa0f4fa1d599ba0798d1b283315c4208cbcb608925808ce003b1ade79034886edb8208f9f48d
RMD160 (NetBSD-6.0/x68k/binary/sets/games.tgz) = 1e09d99d7fe2e495763f05919472d27a4ba699df
MD5 (NetBSD-6.0/x68k/binary/sets/games.tgz) = 8a26f3e864444209088bccf21419a616
SHA1 (NetBSD-6.0/x68k/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/x68k/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/x68k/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/x68k/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/x68k/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/x68k/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/x68k/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/x68k/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/x68k/binary/sets/xcomp.tgz) = e13548b8974f5d55e98caf17b2b4bbbcf2430370
SHA512 (NetBSD-6.0/x68k/binary/sets/xcomp.tgz) = e4b13c798df04ffef88e72100e2db5af400e5ae255cd7a1ca2907736b8138a59045b6f3bd8531b5753d0b87eadb1b5f801043a929bb25d1611e4be693c1da954
RMD160 (NetBSD-6.0/x68k/binary/sets/xcomp.tgz) = 3b1f4e26ddd57e69afd9709553c1bf70d6e29ea0
MD5 (NetBSD-6.0/x68k/binary/sets/xcomp.tgz) = 894bb9aa05b164d9923682165d708335
SHA1 (NetBSD-6.0/x68k/binary/sets/xfont.tgz) = 24ca4cb77a59246b1b05e0a91fed21c63259d12b
SHA512 (NetBSD-6.0/x68k/binary/sets/xfont.tgz) = 8dc297fb6205a995767e68e77cc21169234dfb66a86bbcd2e1828fd32efeda0a25c9f2a9b1a39b97fe8d7b24f8cd5cf6160859af6be2259876f39ee4e76c405b
RMD160 (NetBSD-6.0/x68k/binary/sets/xfont.tgz) = d2f134959940e6667aca185f01e388c05dc6bc85
MD5 (NetBSD-6.0/x68k/binary/sets/xfont.tgz) = 3c96c1f0a13cd7d277ca35622ac9c058
SHA1 (NetBSD-6.0/x68k/binary/sets/kern-GENERIC.tgz) = 253e2e072ae66dd896be3d6d936c637bb96ef2c2
SHA512 (NetBSD-6.0/x68k/binary/sets/kern-GENERIC.tgz) = e2b9221ffad2c150f1529243197e7672c58a41d865dcf02a7252c1bb1f0fdb644e8c6bc3250f16f2091748d2a009badd12c0cbda2ae2b186bbfa0b6302957293
RMD160 (NetBSD-6.0/x68k/binary/sets/kern-GENERIC.tgz) = cd78ce4ed94bd540d9cfcf2d3a6d51fd83cbef95
MD5 (NetBSD-6.0/x68k/binary/sets/kern-GENERIC.tgz) = a41a26a51d64ac00f3d0b1f5657a5400
SHA1 (NetBSD-6.0/x68k/binary/sets/base.tgz) = 97339969786af5934c8db561afd33bf2fdd5cb77
SHA512 (NetBSD-6.0/x68k/binary/sets/base.tgz) = a5cb607f828570f11ae3653032d442cc191339a00037cc6c37635c726fb2b06a9bf1d2dec7e71bfb824ac11810ff1ccbff55ec69659c66112b36fd28521a304c
RMD160 (NetBSD-6.0/x68k/binary/sets/base.tgz) = f77783ac8537a39ac77014b7fd8860269a6aae0c
MD5 (NetBSD-6.0/x68k/binary/sets/base.tgz) = 752b937da621346aaab5b18ac1fc9f3a
SHA1 (NetBSD-6.0/x68k/binary/sets/comp.tgz) = 5b2f871bf928a68b7ffdf2b2f45e5ee80a8c83a0
SHA512 (NetBSD-6.0/x68k/binary/sets/comp.tgz) = d037edaa7d8b4a275a39771f2f7ffd4dd46f68a3107af0ca88f9ce205370c756904c6c0bcb7f6138fabeaf0bfb504ce541253c9d6d2892675e7e308358045047
RMD160 (NetBSD-6.0/x68k/binary/sets/comp.tgz) = b371813449f8d913bf22466d52a7a238522b8a5d
MD5 (NetBSD-6.0/x68k/binary/sets/comp.tgz) = efab43faa3de48943a5ba3b96a07741d
SHA1 (NetBSD-6.0/x68k/binary/sets/etc.tgz) = 52e6167471d4788f7f060166de8f23a31f731596
SHA512 (NetBSD-6.0/x68k/binary/sets/etc.tgz) = ce530878bf230e22786bac9ce6075958eebb52c5119b760ebc183b8f7b2fa59297797768461a44f708b9ce993c21a96e0f49b207c79208e25eea93412889babc
RMD160 (NetBSD-6.0/x68k/binary/sets/etc.tgz) = ad019f3b95e8cc9d3320909dee7dd91ff45a4da6
MD5 (NetBSD-6.0/x68k/binary/sets/etc.tgz) = 9f179baf30026ced05db113a623006b9
SHA1 (NetBSD-6.0/x68k/binary/sets/modules.tgz) = d52fa0ed8a899526add267f705735e9dcbb62a14
SHA512 (NetBSD-6.0/x68k/binary/sets/modules.tgz) = af60d178109bc5330363f041d688e46e36e1894ca0320c7a1735f5de483349a38e63b316b297a5bbf7fb282c1f69539698f7704e759a27c1c833239ec775e371
RMD160 (NetBSD-6.0/x68k/binary/sets/modules.tgz) = 5e3db9d5547a118143f5eac40b442cf27dce8a52
MD5 (NetBSD-6.0/x68k/binary/sets/modules.tgz) = 18d52b98a32e1b37501fb2056be5d356
SHA1 (NetBSD-6.0/x68k/binary/sets/tests.tgz) = 469b2b371980f0bd11407497ee2c104536867df3
SHA512 (NetBSD-6.0/x68k/binary/sets/tests.tgz) = bfb64b49fd88386e2012b827d9af754bbc75a21e63650da66a8add45b54bfc700a3d22a3c6e98e8736309f1a1015f82c366e51ac6539556f1645b28d3d018ad5
RMD160 (NetBSD-6.0/x68k/binary/sets/tests.tgz) = 8e674a86bd2551876719f842f61850d6505b290e
MD5 (NetBSD-6.0/x68k/binary/sets/tests.tgz) = 5c524dfbbb575f61ff958baa317a4808
SHA1 (NetBSD-6.0/x68k/binary/sets/text.tgz) = c8b514b008fdd482c7df7df5849c5bbcd04d289c
SHA512 (NetBSD-6.0/x68k/binary/sets/text.tgz) = 37fe61c52eaa54e8de5709bb3494efe1845a0946c65ed007ec0785902d44acbb3b971bd31b53305ce72d4d32180a56f280a4b29a2a9dcb83c443299edcfe0ea8
RMD160 (NetBSD-6.0/x68k/binary/sets/text.tgz) = d927619390ecab77ada3966395d227967cb38a43
MD5 (NetBSD-6.0/x68k/binary/sets/text.tgz) = 1854795033d3fb974637643eed84359a
SHA1 (NetBSD-6.0/x68k/binary/sets/xbase.tgz) = 012b5faed219406f267667b7d5047ea9e85ac58a
SHA512 (NetBSD-6.0/x68k/binary/sets/xbase.tgz) = 88b65e79f9801e1d3a0da5d3407754c7d48911bf86e5262fb8d02c46c5653b6fd55e6588177771ff82740b12fc84c8a19e207cee83ceee3a772e5beb182ae72b
RMD160 (NetBSD-6.0/x68k/binary/sets/xbase.tgz) = e5d43999d4316d561ee7689321c04698af4ed5b4
MD5 (NetBSD-6.0/x68k/binary/sets/xbase.tgz) = e5cdea5cf19b94dc3ef382951ef923e3
SHA1 (NetBSD-6.0/x68k/binary/sets/xetc.tgz) = bf1f9fa57ffed50beb8d7d455c95928d560bf224
SHA512 (NetBSD-6.0/x68k/binary/sets/xetc.tgz) = 3bf13d97afb20cb5e7098d9837f0a1bf471e49ec53672948373fcdb23c25c56fcb6de974a021c7e3bcb00e6069e0a68c5a84cd187e4304af129b3db66942d953
RMD160 (NetBSD-6.0/x68k/binary/sets/xetc.tgz) = d2f0715f277a6de929f1504d50ce1b5fe27327fc
MD5 (NetBSD-6.0/x68k/binary/sets/xetc.tgz) = 85342c60f299840da352a811f666392c
SHA1 (NetBSD-6.0/x68k/binary/sets/xserver.tgz) = 54b441f5b0afb06e26e816a729b259abe8b3c234
SHA512 (NetBSD-6.0/x68k/binary/sets/xserver.tgz) = a81cf87de57b646f79cbbdf5cf9a69c90bb2f97adccc1310bfaf639313c3027b04cb7591d49cf7dd07f4f6210e971656c7ae8980d8e0a1e854fbbc87ccfd91d0
RMD160 (NetBSD-6.0/x68k/binary/sets/xserver.tgz) = 0ddef8e502a8ca10a21d59aa99f9476dbac2d29f
MD5 (NetBSD-6.0/x68k/binary/sets/xserver.tgz) = e60e100d889a7377fd708a441d2a818a
SHA1 (NetBSD-6.0/x68k/installation/floppy/sysinst1.fs) = 0c52f0ff358294b5fefca38d963f7fcc3c6dfe93
SHA512 (NetBSD-6.0/x68k/installation/floppy/sysinst1.fs) = aeaba8cc7f83d83ec41a0ffef1a01203de34a89ad6028f8d5df06b2a6523b82ca44f13370f0f2f03e65cf2bb8a76532868471f25ddac4a8e53aa2f8a57f4be57
RMD160 (NetBSD-6.0/x68k/installation/floppy/sysinst1.fs) = 396eff107455609a3b7322595660741e14a57077
MD5 (NetBSD-6.0/x68k/installation/floppy/sysinst1.fs) = e1a273155ed6f80cc365bf441909a914
SHA1 (NetBSD-6.0/x68k/installation/floppy/sysinst2.fs) = 35344d834aba89ce14574026bdc0c3648c4bed0d
SHA512 (NetBSD-6.0/x68k/installation/floppy/sysinst2.fs) = 59d6206fafca21e1ad29ef12d869f7ad81855f0b1124625278db71dff739a76bc36da83a1bfa4ee920602a246f3f925e4c383ab8ad5455ae3cc19408347da8ed
RMD160 (NetBSD-6.0/x68k/installation/floppy/sysinst2.fs) = ee5f34edcf9ae13406c1db031b9d99cf2e485c4f
MD5 (NetBSD-6.0/x68k/installation/floppy/sysinst2.fs) = c5cf691f0b3380cfa9ed93158a0598a2
SHA1 (NetBSD-6.0/x68k/installation/misc/gz124x5f.lzh) = 50ef2f563b410530f5f52a857705848973a870a4
SHA512 (NetBSD-6.0/x68k/installation/misc/gz124x5f.lzh) = 7df17f64a217eaba1374a2833326824d43acbe84a8e47e558ec805053fb3d93e07619d5c89287951eb7e6961068aa8e534f74b7da87106d3c0f306bf5191acbe
RMD160 (NetBSD-6.0/x68k/installation/misc/gz124x5f.lzh) = 2ebf89a1a5ee6564bbd9bdcb5a8d2716f4bc121d
MD5 (NetBSD-6.0/x68k/installation/misc/gz124x5f.lzh) = e27b76e450d506e3cd555f8ae3a99e61
SHA1 (NetBSD-6.0/x68k/installation/misc/gzip.x) = 639b9edf903ccc07259e146ac141cc8c102bab95
SHA512 (NetBSD-6.0/x68k/installation/misc/gzip.x) = cac2274fa0b5e1ac2ef978b8a09c001cf3e3acdeaf7131e93677b4fb13d56412bcfee86fe886758866d6daffb67f54dd6fa4c2796bda49ca0e59e4d3a113e7d6
RMD160 (NetBSD-6.0/x68k/installation/misc/gzip.x) = 7c431086e8719f5e48c9d698808a1ae74a78b2d1
MD5 (NetBSD-6.0/x68k/installation/misc/gzip.x) = 69fb6553b4cf7d3825b53c8d7bf5a865
SHA1 (NetBSD-6.0/x68k/installation/misc/loadbsd.0) = 45a858903634d838d7525fff2db12467195ce1d5
SHA512 (NetBSD-6.0/x68k/installation/misc/loadbsd.0) = 7c67ae6947fb128acfe72dab897edc0d8cb4c8e82b1886337ed07b5d031e0e550ff791b470f0276ef0fb920b4d6eefdc574efec0dc04116b05b510684ad8044d
RMD160 (NetBSD-6.0/x68k/installation/misc/loadbsd.0) = 4b30a14f171cf026efa10f44dd0357cc2db91d17
MD5 (NetBSD-6.0/x68k/installation/misc/loadbsd.0) = 25b2b8aa6efadacb72cf96f497a38d5a
SHA1 (NetBSD-6.0/x68k/installation/misc/loadbsd.html) = 9cd27c39d6fa54fc2c534e4e6dfb30f86764ea49
SHA512 (NetBSD-6.0/x68k/installation/misc/loadbsd.html) = 71544bd6ce6c3139654cd55ec1ed293fee603e733fb2cde6574ace66df4ee8450182cf890c4519313ac74bc0d43919340cf277c329e6b7a66734aa33d6465593
RMD160 (NetBSD-6.0/x68k/installation/misc/loadbsd.html) = d43cd8e0a08f11a2bdd4fcf2d8489761a8dfd629
MD5 (NetBSD-6.0/x68k/installation/misc/loadbsd.html) = 3228e40e5e967fb87b75ee462388f92a
SHA1 (NetBSD-6.0/x68k/installation/misc/loadbsd.x) = 9ecad059bb18bfc8fb33c165f63a15770326d3bf
SHA512 (NetBSD-6.0/x68k/installation/misc/loadbsd.x) = 7ed4df332f2e65e5db31f95bc7635f2ea1f5b638727509ff16ee7a531a0006addf16f1da4eceb727d4148dcad8686b3dc960760d289f361fbdb2177e1089032f
RMD160 (NetBSD-6.0/x68k/installation/misc/loadbsd.x) = ead03264a128cab57ebeb615b42e3688b5641f80
MD5 (NetBSD-6.0/x68k/installation/misc/loadbsd.x) = 4b9bcac67fd51cd5bf95911a5431b49a
SHA1 (NetBSD-6.0/x68k/installation/misc/rawrite.doc) = e2cab55990a4835524655fbaf2ade5d9eee29c17
SHA512 (NetBSD-6.0/x68k/installation/misc/rawrite.doc) = d9df8ac2fe18ff8cc13ad62af9107e821c40ca41cd0ae6d3cd0dd1c55351f8b0f7bb2cdd7f0917a16738d11e0230c0ec233f2297b8603f3a619ef99b0eb1ed9a
RMD160 (NetBSD-6.0/x68k/installation/misc/rawrite.doc) = ada5298cb0d2df6d1833f32ff21cc97244376748
MD5 (NetBSD-6.0/x68k/installation/misc/rawrite.doc) = 53481a27e630ca1b583cd953d24d13ca
SHA1 (NetBSD-6.0/x68k/installation/misc/rawrite.lzh) = 575e26ad9625ca21e5f3aad4c32fbbe01312c97f
SHA512 (NetBSD-6.0/x68k/installation/misc/rawrite.lzh) = 36104c3fcd87dbedc2421541f14f54151087348f45f69508e5916c3570915f0b0fc355b3c075e4c1cc3907094abe87e14c85ff333e0064403c8230d93719ea0a
RMD160 (NetBSD-6.0/x68k/installation/misc/rawrite.lzh) = 4c0538916a9ca007aa9ba66ea42c6c46a4c563f2
MD5 (NetBSD-6.0/x68k/installation/misc/rawrite.lzh) = c15b00572229133bf9505876ea120a31
SHA1 (NetBSD-6.0/x68k/installation/misc/rawrite.x) = 30e3d4504bfbf7c55a77a19435d75bb1069ed596
SHA512 (NetBSD-6.0/x68k/installation/misc/rawrite.x) = 06d706b3581dc972d282ef9a6d254740b721a43d5f9c6325af7fb3d1fc8402071887b0ca99c720c5d02c8edcab6086b819faa69640ba4463c1713b168d432059
RMD160 (NetBSD-6.0/x68k/installation/misc/rawrite.x) = 844aa4ab7dbef030c140e543475f24fc60f0a075
MD5 (NetBSD-6.0/x68k/installation/misc/rawrite.x) = 6efe87d85560d8d6e3eec7a7b9d1545a
SHA1 (NetBSD-6.0/x68k/INSTALL.html) = 5ef0d8d03b7f3f600444752d3763c17000da1e3b
SHA512 (NetBSD-6.0/x68k/INSTALL.html) = 94181608314a1dcc1fa0c634060b28ce229883affc9f64c33d9eeae09b21f9f88149deff8d84a7f64b6b5e7cfb741b6c43d9ccdd2ccdd4c9d6e043f22452adb5
RMD160 (NetBSD-6.0/x68k/INSTALL.html) = a5e8a1d8e28859a3e08b662d2363d758baf23eac
MD5 (NetBSD-6.0/x68k/INSTALL.html) = c7154e6e818d1490ce1dbc392bb8207e
SHA1 (NetBSD-6.0/x68k/INSTALL.more) = 9771b22f8b8b9a5a3ad2aead71a06bccd63caae8
SHA512 (NetBSD-6.0/x68k/INSTALL.more) = 4f3561d62f5d3a6f1f936d476d783dca2d4caaa50e21d8571ac795a8f3a06ad10f490d02b05ae8e51ed3e6eeb7aba645215339632aa61c99b93c4c4b37ee5329
RMD160 (NetBSD-6.0/x68k/INSTALL.more) = a8994368a67cb7bd6df0232815e6c21a1a69537b
MD5 (NetBSD-6.0/x68k/INSTALL.more) = 13d4f52ad2a72fc92e24c1467463be66
SHA1 (NetBSD-6.0/x68k/INSTALL.ps) = 3dfb26b142ddfa2771a49a371bcbf72cacd74f7c
SHA512 (NetBSD-6.0/x68k/INSTALL.ps) = bc12356b462b30559d05d0ce0ccc73f07fda3f5575121bf2911db6c638375416e4a79364f7727ea1adef19ed2679e86600c960585e9ef7899ed583d630655b20
RMD160 (NetBSD-6.0/x68k/INSTALL.ps) = f46789b092425a148d38cae66d987253e12efecd
MD5 (NetBSD-6.0/x68k/INSTALL.ps) = 34b1b9c1a6ef25bb677408f0a869ed52
SHA1 (NetBSD-6.0/x68k/INSTALL.txt) = 7371dc6f22eece4157c00dfb01f59a035f305da6
SHA512 (NetBSD-6.0/x68k/INSTALL.txt) = 600911e35b9c641ee4315452cc4b534e844695af852328eec243632f462b0685acad59f35588c46d5c083f745ea5c1ab1df32e8e4f2307409dc5dd8cdafe110c
RMD160 (NetBSD-6.0/x68k/INSTALL.txt) = 8f3c307f17b591859a241414a6a0f33f592e811a
MD5 (NetBSD-6.0/x68k/INSTALL.txt) = 6bd22ee53bf58cae2454a2f8627718f9
SHA1 (NetBSD-6.0/zaurus/binary/kernel/netbsd-GENERIC.gz) = 7f4cbcd7d5390357b1873da969b0a7cca60e73c1
SHA512 (NetBSD-6.0/zaurus/binary/kernel/netbsd-GENERIC.gz) = 0c552ce725209b5686341ac0101cb7969a5a43ab8ba5d1b9a540fa07979e93921acfca844622ee00f5be4d3b06598ef4750772767b72b754e5cae22735ec24e1
RMD160 (NetBSD-6.0/zaurus/binary/kernel/netbsd-GENERIC.gz) = 2ba59423e410d54f3a829f2ce9e5bcbd43b5b840
MD5 (NetBSD-6.0/zaurus/binary/kernel/netbsd-GENERIC.gz) = 4cd87eddf3285a67e0f463d4896db002
SHA1 (NetBSD-6.0/zaurus/binary/kernel/netbsd-C700.gz) = 01cc33fb982cd5f2b188d4688147750a94863fe2
SHA512 (NetBSD-6.0/zaurus/binary/kernel/netbsd-C700.gz) = ceac21a76cc922a67cbf4d8cc83f471015d36d0896efabbf232ebec9217a3d56f8cea43f3e1c22bc527b32be5179898822807a041ef418e3fd6ff942d09dbac7
RMD160 (NetBSD-6.0/zaurus/binary/kernel/netbsd-C700.gz) = 781cc21ea042587fe9f4c0fe9305ae159f5a15e5
MD5 (NetBSD-6.0/zaurus/binary/kernel/netbsd-C700.gz) = 95bcd094f576b6f49e7a1bdbb5048a13
SHA1 (NetBSD-6.0/zaurus/binary/sets/games.tgz) = e1d2bb091d8202962133475bc83edb55230b9008
SHA512 (NetBSD-6.0/zaurus/binary/sets/games.tgz) = a147fe6d0419d53d5a428802d0bb6c9fa8a84a76c96d7d6d7c0d399e5af3a69240116463549a83d53387b0214ba0207796d7c10228d92a684fe7ff0ece3aec16
RMD160 (NetBSD-6.0/zaurus/binary/sets/games.tgz) = 2be515c495548ba5135ff42a75a8ebbfc35ea776
MD5 (NetBSD-6.0/zaurus/binary/sets/games.tgz) = 9059a6f2cd41cb58827709c9e9a1649b
SHA1 (NetBSD-6.0/zaurus/binary/sets/man.tgz) = 899b9d8242e983a5110ab755651c24f709b5af67
SHA512 (NetBSD-6.0/zaurus/binary/sets/man.tgz) = 410999db110d710f4bb498c3d7ae7adc62170c7c2f470ce189c92f6c436c5ea96149659c631ef0b5605caaa025d97dfe1c07c428d86f51239657bd6b71717b15
RMD160 (NetBSD-6.0/zaurus/binary/sets/man.tgz) = c1270f68c6d41b351877f71f0396ac34877dfc8d
MD5 (NetBSD-6.0/zaurus/binary/sets/man.tgz) = df4db2bbfa39ef6a4a215d949653d04a
SHA1 (NetBSD-6.0/zaurus/binary/sets/misc.tgz) = 982918d46b2da2693dded747147baed27160d325
SHA512 (NetBSD-6.0/zaurus/binary/sets/misc.tgz) = 54a45fa2bf5c57e3b63e23a95b6ca74f3b8e6b8f655715663ec5f83f713a9c6149b8b1ccc298dfa2eff9e81376f9f64107b35841cbd20ee62be6575539e01c81
RMD160 (NetBSD-6.0/zaurus/binary/sets/misc.tgz) = 9c9161a1e855276c6aa3fe61de655dbea3ec9a43
MD5 (NetBSD-6.0/zaurus/binary/sets/misc.tgz) = 7b0005d0512287e8a34e24c043582fb8
SHA1 (NetBSD-6.0/zaurus/binary/sets/xcomp.tgz) = 8d7469a85b7433c89690cce74e6b3f4e66cf0647
SHA512 (NetBSD-6.0/zaurus/binary/sets/xcomp.tgz) = 411304cf66e300a7f162128203099bb4d6eebc4a93ba18ed7caf79f30b31cc7ffcede85c4e1bad02c3f5f8a27d009fb71f3ce5862c1d8aa477f39a25c535fdaa
RMD160 (NetBSD-6.0/zaurus/binary/sets/xcomp.tgz) = 822d26e0ee08b72ca8b991f0709a8c1bf0399208
MD5 (NetBSD-6.0/zaurus/binary/sets/xcomp.tgz) = 5af361f0334d28119d703064278b3ed3
SHA1 (NetBSD-6.0/zaurus/binary/sets/xetc.tgz) = 222d14c205f1018cc5a86eb03214d18cda2abb82
SHA512 (NetBSD-6.0/zaurus/binary/sets/xetc.tgz) = fb82261996aa772fd8a9022eadebf899ee04ee70229ee4fe07758ca14937bdb34752c8218eb4c5b722b6208f719891493bbc141372fcfa503de83e0a61aa7387
RMD160 (NetBSD-6.0/zaurus/binary/sets/xetc.tgz) = 511ebfe5076f361b7f7ca03a265d16052a16d259
MD5 (NetBSD-6.0/zaurus/binary/sets/xetc.tgz) = 165cffbb6d2c79663198d80c82afac1e
SHA1 (NetBSD-6.0/zaurus/binary/sets/xfont.tgz) = ce0e141beeffeb5b470478e9887e4127da85de3c
SHA512 (NetBSD-6.0/zaurus/binary/sets/xfont.tgz) = 5d98354ff77103793d56af3192ab5f9afcde62208791976064ac33a149f994e0227dd5dd28ed544eeac5ea504e5212d5fb9e7d05e59e5a48a2fe4062e34e3106
RMD160 (NetBSD-6.0/zaurus/binary/sets/xfont.tgz) = 4d09f5e82fec875ef0ea328809465e7a98f2ebfc
MD5 (NetBSD-6.0/zaurus/binary/sets/xfont.tgz) = be26b3890261302420b09976a1d38144
SHA1 (NetBSD-6.0/zaurus/binary/sets/kern-C700.tgz) = 645cc6b6c3250ead1e2e121bcc179be2e9056b65
SHA512 (NetBSD-6.0/zaurus/binary/sets/kern-C700.tgz) = 833883cb8aada6c391917ec890282011d2cfa6c5068d9483ab1b8c9b74361baa093b1c6b074dd5cd89709eb67636cbba1a6be292c23b28b8ff7578194c427263
RMD160 (NetBSD-6.0/zaurus/binary/sets/kern-C700.tgz) = 0867bac200dd8bd12d1c4d225d392a175e06f24a
MD5 (NetBSD-6.0/zaurus/binary/sets/kern-C700.tgz) = d1c5388985d5693d492ec75713ef3ccd
SHA1 (NetBSD-6.0/zaurus/binary/sets/base.tgz) = fcee164dc5d4cd72f3c9841b7c7fbe295c4b3e6a
SHA512 (NetBSD-6.0/zaurus/binary/sets/base.tgz) = 910606c4dedf523e5db213165fd5a244a655a40a7c4686e7a8e80f015d8ca496a1770b1a94532e118ad7cb4d2f4cfd0abb4e07560e0507275b4a27fefa3fb8af
RMD160 (NetBSD-6.0/zaurus/binary/sets/base.tgz) = c1f5fb6f7b18486f08cacbf1ea6cef5d852a3b4d
MD5 (NetBSD-6.0/zaurus/binary/sets/base.tgz) = e646c1da6b4ca9c38607d97cbbb6871a
SHA1 (NetBSD-6.0/zaurus/binary/sets/comp.tgz) = 9af4ccb1edd8eea77f32d50aad4592c483bc41bc
SHA512 (NetBSD-6.0/zaurus/binary/sets/comp.tgz) = b3eecf648f57fcc3af7ed707bf7f584db4aff71b7755986652e80e2a11e83b7a663263800f143d4dfe668bff0dd606f7733f1ae36be132f70c888a45e210d14c
RMD160 (NetBSD-6.0/zaurus/binary/sets/comp.tgz) = ef64fdf6c3e118e4aab5264d457278d625408275
MD5 (NetBSD-6.0/zaurus/binary/sets/comp.tgz) = c22150f0e6684dc1d64fef0abf882dec
SHA1 (NetBSD-6.0/zaurus/binary/sets/etc.tgz) = 04a520449ff803fefaf80b0beaee929122d9a5bd
SHA512 (NetBSD-6.0/zaurus/binary/sets/etc.tgz) = a109dffb85cd44758b30a9fd7b5eeb0e68b43babb96984974662ede674c4e4b06cddfdb84fd4377cdcf66944506547585f2e4e1fde1dd2c6ba0c823fec5af6e7
RMD160 (NetBSD-6.0/zaurus/binary/sets/etc.tgz) = 1412ce7fbadb5bc2a7ea54662dc5af065a52fd06
MD5 (NetBSD-6.0/zaurus/binary/sets/etc.tgz) = fdcf4276a2adc8bc6b8899a0f92827b2
SHA1 (NetBSD-6.0/zaurus/binary/sets/kern-GENERIC.tgz) = 5a4db67f97cbf15dfa86f3d1bdd1e0fbf45cf2c3
SHA512 (NetBSD-6.0/zaurus/binary/sets/kern-GENERIC.tgz) = d47f1b328c845abe0a131e5fec333a5badbbde60e1abb1c2a01032128350ae33a69df720f9a5a944e93a9c29e04b669f530d7d4b803d9262edef1ce8953099d9
RMD160 (NetBSD-6.0/zaurus/binary/sets/kern-GENERIC.tgz) = 57cc4468fe4f292c4798dcaf7c7ee8b6a79c72cd
MD5 (NetBSD-6.0/zaurus/binary/sets/kern-GENERIC.tgz) = 7b6a87debe65e4d987181b2a600a0cc9
SHA1 (NetBSD-6.0/zaurus/binary/sets/modules.tgz) = 97cf73286143f2ee9d1d75d291226b14382bc3e9
SHA512 (NetBSD-6.0/zaurus/binary/sets/modules.tgz) = 819d952342aedefae4120c8b0f3aa362089528ea9a604479b4242154b52e76688353386c6c6fe7f675086997459a1038f74037840642d4d522145be09d13b01c
RMD160 (NetBSD-6.0/zaurus/binary/sets/modules.tgz) = 1c5ade56f427ccb12d516da4b60266496a0dbe7f
MD5 (NetBSD-6.0/zaurus/binary/sets/modules.tgz) = fdf54f31acaaf41ae7e04155dc2ed8e3
SHA1 (NetBSD-6.0/zaurus/binary/sets/tests.tgz) = d267f0e6d994c769d85698d5fe84dc84d3eb0aea
SHA512 (NetBSD-6.0/zaurus/binary/sets/tests.tgz) = 400bcfd917639182daf83eddf20aa476be42285c8c5fdeb187b5086c7c00b0ebf2a0aa7badb7f4e1e43e70f43c46972003a4e6b4dd95eb5b7d621e2f8a562623
RMD160 (NetBSD-6.0/zaurus/binary/sets/tests.tgz) = b8578e18377c21e4f0bc5509eddaacca4e439849
MD5 (NetBSD-6.0/zaurus/binary/sets/tests.tgz) = ae201c948e67c3c74b0f5f89dde362ce
SHA1 (NetBSD-6.0/zaurus/binary/sets/text.tgz) = 3467c94a0c6bcd54efe8849c282fa26f9f02624a
SHA512 (NetBSD-6.0/zaurus/binary/sets/text.tgz) = 0f24772a93aa15de7b2a40e237ea5efeb0f701a6c3645b7add0e478bfcfb3d3050a74d80b1328eb821591ee0464e051572f2afa6bb2935d34ae369809546cf65
RMD160 (NetBSD-6.0/zaurus/binary/sets/text.tgz) = 6acb227a392a29ef30d17e861ec681c56da75790
MD5 (NetBSD-6.0/zaurus/binary/sets/text.tgz) = b4ed717b1487f24bf1768305e0c0065d
SHA1 (NetBSD-6.0/zaurus/binary/sets/xbase.tgz) = 8b5a46ee79d9bffd6d6eb0af06ff3928029b7a92
SHA512 (NetBSD-6.0/zaurus/binary/sets/xbase.tgz) = c675ab7e92391becf93b9894115e5149b1ef4b4cb9b2f41a58a12c0009c8c1999de4aa8e239f82308c5892accac5cbbe0a3d380f30134bec29b433a5357d1cfc
RMD160 (NetBSD-6.0/zaurus/binary/sets/xbase.tgz) = 81d52af994bb130ce57e83468b7ce350f107f3bd
MD5 (NetBSD-6.0/zaurus/binary/sets/xbase.tgz) = 02cd86601f44cd4ba11e466236b04e1d
SHA1 (NetBSD-6.0/zaurus/binary/sets/xserver.tgz) = e293d1143593154074951807e155c8e87adff50a
SHA512 (NetBSD-6.0/zaurus/binary/sets/xserver.tgz) = 060d8e60ec568aebafef85ab34b527dafe11516ef5761598e2ba79e5e7994cd4f710812e857fd20bb1e09b0df5692ef1e10949273027c66ce2255e2b15f78132
RMD160 (NetBSD-6.0/zaurus/binary/sets/xserver.tgz) = 8ea704c20eb2aafb79603904423f7e137136ef4e
MD5 (NetBSD-6.0/zaurus/binary/sets/xserver.tgz) = f8d5d948e6ccb12094636c6d4106f895
SHA1 (NetBSD-6.0/zaurus/installation/kernel/netbsd-INSTALL_C700) = 08e381ea42ccb632460555771fd7fe6bae057a24
SHA512 (NetBSD-6.0/zaurus/installation/kernel/netbsd-INSTALL_C700) = de5d9f71284b987f67077a0fcf2273d11418b73108ada25e4807614752eea38d58b8d32c53e86293a0492d05a842e81f44e5e5e5bdf3f0ca76e440263becd5ff
RMD160 (NetBSD-6.0/zaurus/installation/kernel/netbsd-INSTALL_C700) = 91c741493cd043610e542d01a4cde6ed30469be3
MD5 (NetBSD-6.0/zaurus/installation/kernel/netbsd-INSTALL_C700) = 5544b1f0b4a87402fc0fc5951aa303ae
SHA1 (NetBSD-6.0/zaurus/installation/kernel/netbsd-INSTALL) = a54eadb65290c45af5e6ae4d0cb0b803640fa2b0
SHA512 (NetBSD-6.0/zaurus/installation/kernel/netbsd-INSTALL) = e78fc99a17c732908af345a40f102ae8eef248f4c5c3a4fad85a8e9d497346f86b65096e816aba26975a71780672040c148f40fc4879cf0b6f5d5ec7f6aa34af
RMD160 (NetBSD-6.0/zaurus/installation/kernel/netbsd-INSTALL) = e0bde1431f7c4b3f649b2721c0c38f02ad273bc3
MD5 (NetBSD-6.0/zaurus/installation/kernel/netbsd-INSTALL) = 02bd814abd9951fec27c269ddefb6346
SHA1 (NetBSD-6.0/zaurus/installation/zbsdmod.o) = 6f5a55fa2497fac0ae75f20df2074eaedde0aa0c
SHA512 (NetBSD-6.0/zaurus/installation/zbsdmod.o) = 46e48ce8f3e637b5e708a066847e2f7449c8134f8f8d327e3ace9116cae4fc7b77aeee136cb797e00000277c4d224dcb10d946aff983be3292690580ba666b8b
RMD160 (NetBSD-6.0/zaurus/installation/zbsdmod.o) = b96bdc9f3009ddb56d5b4b05e01e2cc3862a9e11
MD5 (NetBSD-6.0/zaurus/installation/zbsdmod.o) = 49e746ab92af22b281d4c15eecf85c10
SHA1 (NetBSD-6.0/zaurus/installation/zboot) = 142512f9c4e5a1a3762dc5b96416392a19fba367
SHA512 (NetBSD-6.0/zaurus/installation/zboot) = 30520adab63612d4cf76919b21d2262235d3e2996c5a308f050e480acee5f544aad94993e53cd1e7490ef054b2fecfab7db0483ca1f9c045287debeedd84300e
RMD160 (NetBSD-6.0/zaurus/installation/zboot) = ebd6148f2ed15ea48fd7a2f2a273136661287134
MD5 (NetBSD-6.0/zaurus/installation/zboot) = da83febbb31821d4b68c07a9c9487946
-----BEGIN PGP SIGNATURE-----
Version: GnuPG v1.4.11 (NetBSD)

iQIcBAEBAgAGBQJQfDnGAAoJEAZJc6xMSnBuq9AP/1n7mninneKOiCjD5RTtdS96
sjxa7DbX1xd8ZIL+klchZurlqqKeBdk63gALDo6VpAsSbNH45R0Ib9rzvJiwSkRK
7PHPqijtxgACFV4UclrUXWts+Px6AS8QxMfuLgRwu0KETYMnuR8Ghp9YD6xIGqaw
kG5Pfnqs2qWi9Q/70uTlmJgl3iEj/KStdDyxcmH2VlejtAGxJxqVVwQ6XpAu5z1Z
OVMmoZoReFeDiTZAydf0VzvwyfmHuPvXKfO8Z+PhkFboAUfuIKIO8DfljQ7A5OBp
oEEK1nnjeac59KWPAvI4IbXfVuC1o2rEYmLHz9vBxHKiPf9SxL4XjNbZozejXsMD
h1NWWt4q6nD4nGQaOPyR4afN0ZWluC+6cASQ+Srmr/0QJbsBwgtXuKza2vwNs0iC
IF80PL1iPjDPmf1CQnOZqJktUfv4aeAcwNs8oiq+toIlgOkIwCzFQFjflny9pRKp
/YOxwFY8izCb6pu0/3LM2CSutwDEkR7I6aAn65BFW0ia6Ty5azf9iIFOAIbzefkH
AP1CJKn36vgtcny/R9KueIw/50Hok2rrFrDqzC8mMq5j3Pletm/4equmVu27e+uc
0jTiNwCHb6ZvuoO7d1clrIEEJPEXe66Xcmj9iJzo+Ai1nhCj1BDNZnj9zHbIwPIQ
6yQgxCHP9Ney/0j1V+sF
=nHXc
-----END PGP SIGNATURE-----
EOF
	cat > expected16 << EOF
Good signature for b.gpg made Mon Sep 10 00:15:38 2012
signature     2048/RSA (Encrypt or Sign) 1b68dcfcc0596823 2004-01-12
fingerprint   d415 9deb 336d e4cc cdfa 00cd 1b68 dcfc c059 6823 
uid           Alistair Crooks <agc@alistaircrooks.com>
uid           Alistair Crooks <agc@pkgsrc.org>
uid           Alistair Crooks <agc@netbsd.org>
uid           Alistair Crooks <agc@netflix.com>

EOF
	cat >expected17 <<EOF
Good signature for a.gpg made Sun Sep  9 17:44:11 2012
signature  2048/RSA (Encrypt or Sign) 1b68dcfcc0596823 2004-01-12
fingerprint:  d415 9deb 336d e4cc cdfa 00cd 1b68 dcfc c059 6823 
uid              Alistair Crooks <agc@alistaircrooks.com>
uid              Alistair Crooks <agc@pkgsrc.org>
uid              Alistair Crooks <agc@netbsd.org>
uid              Alistair Crooks <agc@netflix.com>
encryption 2048/RSA (Encrypt or Sign) 79deb61e488eee74 2004-01-12
fingerprint:  57c0 c1e6 bf71 8845 416b 9522 79de b61e 488e ee74 

EOF
	cat >expected18 <<EOF
Good signature for gpgsigned-a.gpg made Sun Sep  9 17:43:01 2012
signature  2048/RSA (Encrypt or Sign) 1b68dcfcc0596823 2004-01-12
fingerprint:  d415 9deb 336d e4cc cdfa 00cd 1b68 dcfc c059 6823 
uid              Alistair Crooks <agc@alistaircrooks.com>
uid              Alistair Crooks <agc@pkgsrc.org>
uid              Alistair Crooks <agc@netbsd.org>
uid              Alistair Crooks <agc@netflix.com>
encryption 2048/RSA (Encrypt or Sign) 79deb61e488eee74 2004-01-12
fingerprint:  57c0 c1e6 bf71 8845 416b 9522 79de b61e 488e ee74 

EOF
	cat >expected19 <<EOF
Good signature for NetBSD-6.0_RC2_hashes.asc made Wed Sep 19 07:53:18 2012
signature  4096/RSA (Encrypt or Sign) 064973ac4c4a706e 2009-06-23
fingerprint:  ddee 2bdb 9c98 a0d1 d4fb dbf7 0649 73ac 4c4a 706e 
uid              NetBSD Security Officer <security-officer@NetBSD.org>
encryption 4096/RSA (Encrypt or Sign) 9ff2c24fdf2ce620 2009-06-23 [Expiry 2019-06-21]
fingerprint:  1915 0801 fbd8 f45d 89f2 0205 9ff2 c24f df2c e620 

EOF
	cat >expected20 <<EOF
1. tag & 0x3f
2. len

one pass (tag 4)
========
b version:3
b sig type
b hash alg
b pubkey alg
8b keyid

literal data (tag 11)
=============
b binary/text
b length
c string
L mtime
text
EOF
	cat >expected21 <<EOF
Good signature for [stdin] made Sun Sep  9 17:44:11 2012
signature  2048/RSA (Encrypt or Sign) 1b68dcfcc0596823 2004-01-12
fingerprint:  d415 9deb 336d e4cc cdfa 00cd 1b68 dcfc c059 6823 
uid              Alistair Crooks <agc@alistaircrooks.com>
uid              Alistair Crooks <agc@pkgsrc.org>
uid              Alistair Crooks <agc@netbsd.org>
uid              Alistair Crooks <agc@netflix.com>
encryption 2048/RSA (Encrypt or Sign) 79deb61e488eee74 2004-01-12
fingerprint:  57c0 c1e6 bf71 8845 416b 9522 79de b61e 488e ee74 

EOF
	cat >expected22 <<EOF
Good signature for [stdin] made Sun Sep 30 10:50:20 2012
signature     2048/RSA (Encrypt or Sign) 1b68dcfcc0596823 2004-01-12
fingerprint   d415 9deb 336d e4cc cdfa 00cd 1b68 dcfc c059 6823 
uid           Alistair Crooks <agc@alistaircrooks.com>
uid           Alistair Crooks <agc@pkgsrc.org>
uid           Alistair Crooks <agc@netbsd.org>
uid           Alistair Crooks <agc@netflix.com>

EOF
	cat >expected23 <<EOF
Good signature for [stdin] made Wed Sep 19 07:53:18 2012
signature  4096/RSA (Encrypt or Sign) 064973ac4c4a706e 2009-06-23
fingerprint:  ddee 2bdb 9c98 a0d1 d4fb dbf7 0649 73ac 4c4a 706e 
uid              NetBSD Security Officer <security-officer@NetBSD.org>
encryption 4096/RSA (Encrypt or Sign) 9ff2c24fdf2ce620 2009-06-23 [Expiry 2019-06-21]
fingerprint:  1915 0801 fbd8 f45d 89f2 0205 9ff2 c24f df2c e620 

EOF
	cat >expected24 <<EOF
Good signature for [stdin] made Mon Sep 10 00:15:38 2012
signature     2048/RSA (Encrypt or Sign) 1b68dcfcc0596823 2004-01-12
fingerprint   d415 9deb 336d e4cc cdfa 00cd 1b68 dcfc c059 6823 
uid           Alistair Crooks <agc@alistaircrooks.com>
uid           Alistair Crooks <agc@pkgsrc.org>
uid           Alistair Crooks <agc@netbsd.org>
uid           Alistair Crooks <agc@netflix.com>

EOF
	cat >expected25 <<EOF
Good signature for NetBSD-6.0_RC1_hashes.gpg made Tue Oct 16 08:12:16 2012
signature  4096/RSA (Encrypt or Sign) 064973ac4c4a706e 2009-06-23
fingerprint:  ddee 2bdb 9c98 a0d1 d4fb dbf7 0649 73ac 4c4a 706e 
uid              NetBSD Security Officer <security-officer@NetBSD.org>
encryption 4096/RSA (Encrypt or Sign) 9ff2c24fdf2ce620 2009-06-23 [Expiry 2019-06-21]
fingerprint:  1915 0801 fbd8 f45d 89f2 0205 9ff2 c24f df2c e620 

EOF
	cat >expected26 <<EOF
Good signature for [stdin] made Tue Oct 16 08:12:16 2012
signature  4096/RSA (Encrypt or Sign) 064973ac4c4a706e 2009-06-23
fingerprint:  ddee 2bdb 9c98 a0d1 d4fb dbf7 0649 73ac 4c4a 706e 
uid              NetBSD Security Officer <security-officer@NetBSD.org>
encryption 4096/RSA (Encrypt or Sign) 9ff2c24fdf2ce620 2009-06-23 [Expiry 2019-06-21]
fingerprint:  1915 0801 fbd8 f45d 89f2 0205 9ff2 c24f df2c e620 

EOF
	cat >expected27 <<EOF
Good signature for [stdin] made Mon Oct 15 09:28:54 2012
signature     4096/RSA (Encrypt or Sign) 064973ac4c4a706e 2009-06-23
fingerprint   ddee 2bdb 9c98 a0d1 d4fb dbf7 0649 73ac 4c4a 706e 
uid           NetBSD Security Officer <security-officer@NetBSD.org>

EOF
	cat >expected28 <<EOF
Good signature for NetBSD-6.0_hashes.asc made Mon Oct 15 09:28:54 2012
signature     4096/RSA (Encrypt or Sign) 064973ac4c4a706e 2009-06-23
fingerprint   ddee 2bdb 9c98 a0d1 d4fb dbf7 0649 73ac 4c4a 706e 
uid           NetBSD Security Officer <security-officer@NetBSD.org>

EOF
	cat >expected29 <<EOF
Good signature for NetBSD-6.0_RC1_hashes_ascii.gpg made Sun Sep  9 17:41:24 2012
signature  4096/RSA (Encrypt or Sign) 064973ac4c4a706e 2009-06-23
fingerprint:  ddee 2bdb 9c98 a0d1 d4fb dbf7 0649 73ac 4c4a 706e 
uid              NetBSD Security Officer <security-officer@NetBSD.org>
encryption 4096/RSA (Encrypt or Sign) 9ff2c24fdf2ce620 2009-06-23 [Expiry 2019-06-21]
fingerprint:  1915 0801 fbd8 f45d 89f2 0205 9ff2 c24f df2c e620 

EOF
	cat >expected30 <<EOF
Good signature for [stdin] made Sun Sep  9 17:41:24 2012
signature  4096/RSA (Encrypt or Sign) 064973ac4c4a706e 2009-06-23
fingerprint:  ddee 2bdb 9c98 a0d1 d4fb dbf7 0649 73ac 4c4a 706e 
uid              NetBSD Security Officer <security-officer@NetBSD.org>
encryption 4096/RSA (Encrypt or Sign) 9ff2c24fdf2ce620 2009-06-23 [Expiry 2019-06-21]
fingerprint:  1915 0801 fbd8 f45d 89f2 0205 9ff2 c24f df2c e620 

EOF
	uudecode <<EOF
begin-base64 644 expected31
UFJPRz1wClNSQ1M9cGFyc2UuYwpXQVJOUz01Ck1LTUFOPW5vCkNQUEZMQUdTKz0t
ZyAtTzAKTERGTEFHUys9LWcgLU8wCgouaW5jbHVkZSA8YnNkLnByb2cubWs+Cgp0
OiAke1BST0d9CgkuLyR7UFJPR30gZ3Bnc2lnbmVkLWEuZ3BnClBST0c9cApTUkNT
PXBhcnNlLmMKV0FSTlM9NQpNS01BTj1ubwpDUFBGTEFHUys9LWcgLU8wCkxERkxB
R1MrPS1nIC1PMAoKLmluY2x1ZGUgPGJzZC5wcm9nLm1rPgoKdDogJHtQUk9HfQoJ
Li8ke1BST0d9IGdwZ3NpZ25lZC1hLmdwZwpQUk9HPXAKU1JDUz1wYXJzZS5jCldB
Uk5TPTUKTUtNQU49bm8KQ1BQRkxBR1MrPS1nIC1PMApMREZMQUdTKz0tZyAtTzAK
Ci5pbmNsdWRlIDxic2QucHJvZy5taz4KCnQ6ICR7UFJPR30KCS4vJHtQUk9HfSBn
cGdzaWduZWQtYS5ncGcK
====
EOF
	cat >expected32 <<EOF
Good signature for b.gpg made Mon Sep 10 00:15:38 2012
signature     2048/RSA (Encrypt or Sign) 1b68dcfcc0596823 2004-01-12
fingerprint   d415 9deb 336d e4cc cdfa 00cd 1b68 dcfc c059 6823 
uid           Alistair Crooks <agc@alistaircrooks.com>
uid           Alistair Crooks <agc@pkgsrc.org>
uid           Alistair Crooks <agc@netbsd.org>
uid           Alistair Crooks <agc@netflix.com>

Good signature for b.gpg made Mon Sep 10 00:15:38 2012
signature     2048/RSA (Encrypt or Sign) 1b68dcfcc0596823 2004-01-12
fingerprint   d415 9deb 336d e4cc cdfa 00cd 1b68 dcfc c059 6823 
uid           Alistair Crooks <agc@alistaircrooks.com>
uid           Alistair Crooks <agc@pkgsrc.org>
uid           Alistair Crooks <agc@netbsd.org>
uid           Alistair Crooks <agc@netflix.com>

Good signature for b.gpg made Mon Sep 10 00:15:38 2012
signature     2048/RSA (Encrypt or Sign) 1b68dcfcc0596823 2004-01-12
fingerprint   d415 9deb 336d e4cc cdfa 00cd 1b68 dcfc c059 6823 
uid           Alistair Crooks <agc@alistaircrooks.com>
uid           Alistair Crooks <agc@pkgsrc.org>
uid           Alistair Crooks <agc@netbsd.org>
uid           Alistair Crooks <agc@netflix.com>

EOF
	uudecode <<EOF
begin-base64 644 expected33
UFJPRz1wClNSQ1M9cGFyc2UuYwpXQVJOUz01Ck1LTUFOPW5vCkNQUEZMQUdTKz0t
ZyAtTzAKTERGTEFHUys9LWcgLU8wCgouaW5jbHVkZSA8YnNkLnByb2cubWs+Cgp0
OiAke1BST0d9CgkuLyR7UFJPR30gZ3Bnc2lnbmVkLWEuZ3BnCjEuIHRhZyAmIDB4
M2YKMi4gbGVuCgpvbmUgcGFzcyAodGFnIDQpCj09PT09PT09CmIgdmVyc2lvbjoz
CmIgc2lnIHR5cGUKYiBoYXNoIGFsZwpiIHB1YmtleSBhbGcKOGIga2V5aWQKCmxp
dGVyYWwgZGF0YSAodGFnIDExKQo9PT09PT09PT09PT09CmIgYmluYXJ5L3RleHQK
YiBsZW5ndGgKYyBzdHJpbmcKTCBtdGltZQp0ZXh0ClBST0c9cApTUkNTPXBhcnNl
LmMKV0FSTlM9NQpNS01BTj1ubwpDUFBGTEFHUys9LWcgLU8wCkxERkxBR1MrPS1n
IC1PMAoKLmluY2x1ZGUgPGJzZC5wcm9nLm1rPgoKdDogJHtQUk9HfQoJLi8ke1BS
T0d9IGdwZ3NpZ25lZC1hLmdwZwo=
====
EOF
	cat >expected34 <<EOF
Good signature for det.sig made Thu Oct 18 02:12:33 2012
signature     2048/RSA (Encrypt or Sign) 1b68dcfcc0596823 2004-01-12
fingerprint   d415 9deb 336d e4cc cdfa 00cd 1b68 dcfc c059 6823 
uid           Alistair Crooks <agc@alistaircrooks.com>
uid           Alistair Crooks <agc@pkgsrc.org>
uid           Alistair Crooks <agc@netbsd.org>
uid           Alistair Crooks <agc@netflix.com>

EOF
	cat >expected35 <<EOF
To Do
=====
tests with -k sig
detached sigs
DSA

Done
====
basics
localise pgp_read_packets
fix lint
WARNS=5
lib man page
prog man page
do we do it statically linked as well?
multiple files in netpgpverify
EOF
	atf_check -s eq:0 -o file:expected16 -e empty env TZ=US/Pacific netpgpverify -k pubring.gpg -c verify b.gpg
	#atf_check -s eq:0 -o file:expected17 -e empty netpgpverify -c verify a.gpg
	#atf_check -s eq:0 -o file:expected18 -e empty netpgpverify -c verify gpgsigned-a.gpg
	#atf_check -s eq:0 -o file:expected19 -e empty netpgpverify -c verify NetBSD-6.0_RC2_hashes.asc
	atf_check -s eq:0 -o file:expected20 -e empty env TZ=US/Pacific netpgpverify -k pubring.gpg -c cat jj.asc
	#atf_check -s eq:0 -o file:expected21 -e empty netpgpverify < a.gpg
	atf_check -s eq:0 -o file:expected22 -e empty env TZ=US/Pacific netpgpverify -k pubring.gpg < jj.asc
	#atf_check -s eq:0 -o file:expected23 -e empty netpgpverify < NetBSD-6.0_RC2_hashes.asc
	atf_check -s eq:0 -o file:expected24 -e empty env TZ=US/Pacific netpgpverify -k pubring.gpg < b.gpg
	#atf_check -s eq:0 -o file:expected25 -e empty netpgpverify NetBSD-6.0_RC1_hashes.gpg
	#atf_check -s eq:0 -o file:expected26 -e empty netpgpverify < NetBSD-6.0_RC1_hashes.gpg
	atf_check -s eq:0 -o file:expected27 -e empty env TZ=US/Pacific netpgpverify -k pubring.gpg < NetBSD-6.0_hashes.asc
	atf_check -s eq:0 -o file:expected28 -e empty env TZ=US/Pacific netpgpverify -k pubring.gpg NetBSD-6.0_hashes.asc
	#atf_check -s eq:0 -o file:expected29 -e empty netpgpverify NetBSD-6.0_RC1_hashes_ascii.gpg
	#atf_check -s eq:0 -o file:expected30 -e empty netpgpverify < NetBSD-6.0_RC1_hashes_ascii.gpg
	atf_check -s eq:0 -o file:expected31 -e empty env TZ=US/Pacific netpgpverify -k pubring.gpg -c cat b.gpg b.gpg b.gpg
	atf_check -s eq:0 -o file:expected32 -e empty env TZ=US/Pacific netpgpverify -k pubring.gpg b.gpg b.gpg b.gpg
	atf_check -s eq:0 -o file:expected33 -e empty env TZ=US/Pacific netpgpverify -k pubring.gpg -c cat b.gpg jj.asc b.gpg
	atf_check -s eq:0 -o file:expected34 -e empty env TZ=US/Pacific netpgpverify -k pubring.gpg det.sig
	atf_check -s eq:0 -o file:expected35 -e empty env TZ=US/Pacific netpgpverify -c cat -k pubring.gpg det.sig
	#atf_check -s eq:0 -o file:expected46 -e empty netpgpverify -k problem-pubring.gpg NetBSD-6.0_hashes.asc

}

atf_test_case netpgpverify_dsa
netpgpverify_dsa_head() {
    atf_set "descr" "DSA signatures for netpgpverify"
}
netpgpverify_dsa_body() {
	uudecode << EOF
begin-base64 600 dsa-pubring.gpg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====
EOF
	uudecode << EOF
begin-base64 644 in1.gpg
owGbwMvMwCGoZv+8NenRnzrGNY1JzJl5hgGthk+VOfX8Ukucgl2sFHwTs1PTMnNS
dcoUDPVM9QwN9AwVjAwMjfQNTPUNzBQMLawMTawMzRQS05MVXCsKFPS4uIJDnVw8
g7RtOXMykxT0wh09QxBCSZl5XFx6mXnJOaUpqQo2ScUpesWlSSmZRXq52XZcHXEs
DIIcDGysTCBXMHBxCsBct0iSkWHa0k9iZ5i+mtQ43zvsb7nA9sWeSrvwubEbQw0P
MNe3Vuky/C+59v/fFbGv5ofnzRSY/ld6vkfIxCtxoVEnvHZVrZeeb60BAA==
====
EOF
	cat > in1.asc << EOF
-----BEGIN PGP SIGNED MESSAGE-----
Hash: SHA256

#	.NetBSD: Makefile,v 1.5.10.1 2012/05/06 18:14:16 agc Exp .

SUBDIR+=	lib .WAIT
SUBDIR+=	bin

.include <bsd.subdir.mk>
-----BEGIN PGP SIGNATURE-----
Version: GnuPG v1.4.11 (NetBSD)

iF4EAREIAAYFAlCFMdYACgkQJj/nhWLi/H7CkQEAgDQrFwPD76JC+6FnOKEz/9DP
H7WjRRMoIQNTGC3ZXRsA/1xah8eFePQDmTO1sQGnINbgX9vZ7GAFOgTjW7+tVb7H
=wtKb
-----END PGP SIGNATURE-----
EOF
	uudecode << EOF
begin-base64 644 in2.gpg
owGbwMvMwCGoZv+8NenRnzrGNU1JzJl5RgGtHj+VOfX8Ukucgl2sFHwTs1PTMnNS
dcoUDPVM9QwN9AwVjAwMjfQNTPUNzBQMLawMTawMzRQS05MVXCsKFPQUuLiCQ51c
PIO0bTlzMpMU9MIdPUMQQkmZeVxcepl5yTmlKakKNknFKXrFpUkpmUV6udl2XB1x
LAyCHAxsrEwgZzBwcQrAnFfzjeGv9GtuvQj7X/Kv5IQ8f2s9n/36yaOmu9HPs8/V
8pg+SdfaxMhwdEOjtLVOGtuaizmXnJ8oa4fN/bfL0WfP1smOn38vsLFlBQA=
====
EOF
	cat > in2.asc << EOF
-----BEGIN PGP SIGNED MESSAGE-----
Hash: SHA256

#	.NetBSD: Makefile,v 1.5.10.1 2012/05/06 18:14:16 agc Exp . 

SUBDIR+=	lib .WAIT
SUBDIR+=	bin

.include <bsd.subdir.mk>
-----BEGIN PGP SIGNATURE-----
Version: GnuPG v1.4.11 (NetBSD)

iF4EAREIAAYFAlCFSQoACgkQJj/nhWLi/H6aKAD9HCLTwY8CwiqTXrzKxHZ5lHQn
qEZbcbXjkCxlk+m/PHUA/2Whlc0t5ZtmI221LQy5inTnzpu1U75E5lJvw0YMTdXJ
=v788
-----END PGP SIGNATURE-----
EOF
	cat > expected36 << EOF
Good signature for in1.gpg made Mon Oct 22 04:45:41 2012
signature     2048/DSA 263fe78562e2fc7e 2012-10-21
fingerprint   d2e5 07b6 5d59 33d3 9c8d a618 263f e785 62e2 fc7e 
uid           David Armstrong (Test DSA key - do not use) <dsa@dsa.com>

EOF
	cat >expected37 <<EOF
Good signature for [stdin] made Mon Oct 22 04:45:41 2012
signature     2048/DSA 263fe78562e2fc7e 2012-10-21
fingerprint   d2e5 07b6 5d59 33d3 9c8d a618 263f e785 62e2 fc7e 
uid           David Armstrong (Test DSA key - do not use) <dsa@dsa.com>

EOF
	cat >expected38 <<EOF
Good signature for in1.asc made Mon Oct 22 04:45:26 2012
signature     2048/DSA 263fe78562e2fc7e 2012-10-21
fingerprint   d2e5 07b6 5d59 33d3 9c8d a618 263f e785 62e2 fc7e 
uid           David Armstrong (Test DSA key - do not use) <dsa@dsa.com>

EOF
	cat >expected39 <<EOF
Good signature for [stdin] made Mon Oct 22 04:45:26 2012
signature     2048/DSA 263fe78562e2fc7e 2012-10-21
fingerprint   d2e5 07b6 5d59 33d3 9c8d a618 263f e785 62e2 fc7e 
uid           David Armstrong (Test DSA key - do not use) <dsa@dsa.com>

EOF
	cat >expected40 <<EOF
#	.NetBSD: Makefile,v 1.5.10.1 2012/05/06 18:14:16 agc Exp .

SUBDIR+=	lib .WAIT
SUBDIR+=	bin

.include <bsd.subdir.mk>
EOF
	cat >expected41 <<EOF
#	.NetBSD: Makefile,v 1.5.10.1 2012/05/06 18:14:16 agc Exp .

SUBDIR+=	lib .WAIT
SUBDIR+=	bin

.include <bsd.subdir.mk>
EOF
	cat >expected42 <<EOF
#	.NetBSD: Makefile,v 1.5.10.1 2012/05/06 18:14:16 agc Exp .

SUBDIR+=	lib .WAIT
SUBDIR+=	bin

.include <bsd.subdir.mk>
EOF
	cat >expected43 <<EOF
#	.NetBSD: Makefile,v 1.5.10.1 2012/05/06 18:14:16 agc Exp .

SUBDIR+=	lib .WAIT
SUBDIR+=	bin

.include <bsd.subdir.mk>
EOF
	cat >expected44 <<EOF
Good signature for in2.gpg made Mon Oct 22 06:24:09 2012
signature     2048/DSA 263fe78562e2fc7e 2012-10-21
fingerprint   d2e5 07b6 5d59 33d3 9c8d a618 263f e785 62e2 fc7e 
uid           David Armstrong (Test DSA key - do not use) <dsa@dsa.com>

EOF
	cat >expected45 <<EOF
Good signature for in2.asc made Mon Oct 22 06:24:26 2012
signature     2048/DSA 263fe78562e2fc7e 2012-10-21
fingerprint   d2e5 07b6 5d59 33d3 9c8d a618 263f e785 62e2 fc7e 
uid           David Armstrong (Test DSA key - do not use) <dsa@dsa.com>

EOF
	cat >expected46 <<EOF
Ignoring unusual/reserved signature subpacket 18
Good signature for NetBSD-6.0_hashes.asc made Mon Oct 15 09:28:54 2012
signature  4096/RSA (Encrypt or Sign) 064973ac4c4a706e 2009-06-23
fingerprint:  ddee 2bdb 9c98 a0d1 d4fb dbf7 0649 73ac 4c4a 706e 
uid              NetBSD Security Officer <security-officer@NetBSD.org>
encryption 4096/RSA (Encrypt or Sign) 9ff2c24fdf2ce620 2009-06-23 [Expiry 2019-06-21]
fingerprint:  1915 0801 fbd8 f45d 89f2 0205 9ff2 c24f df2c e620 

EOF
	atf_check -s eq:0 -o file:expected36 -e empty env TZ=US/Pacific netpgpverify -k dsa-pubring.gpg in1.gpg
	atf_check -s eq:0 -o file:expected37 -e empty env TZ=US/Pacific netpgpverify -k dsa-pubring.gpg < in1.gpg
	atf_check -s eq:0 -o file:expected38 -e empty env TZ=US/Pacific netpgpverify -k dsa-pubring.gpg in1.asc
	atf_check -s eq:0 -o file:expected39 -e empty env TZ=US/Pacific netpgpverify -k dsa-pubring.gpg < in1.asc
	atf_check -s eq:0 -o file:expected40 -e empty env TZ=US/Pacific netpgpverify -k dsa-pubring.gpg -c cat in1.gpg
	atf_check -s eq:0 -o file:expected41 -e empty env TZ=US/Pacific netpgpverify -k dsa-pubring.gpg -c cat < in1.gpg
	atf_check -s eq:0 -o file:expected42 -e empty env TZ=US/Pacific netpgpverify -k dsa-pubring.gpg -c cat in1.asc
	atf_check -s eq:0 -o file:expected43 -e empty env TZ=US/Pacific netpgpverify -k dsa-pubring.gpg -c cat < in1.asc
	atf_check -s eq:0 -o file:expected44 -e empty env TZ=US/Pacific netpgpverify -k dsa-pubring.gpg in2.gpg
	atf_check -s eq:0 -o file:expected45 -e empty env TZ=US/Pacific netpgpverify -k dsa-pubring.gpg in2.asc
}

atf_init_test_cases() {
    atf_add_test_case netpgpverify_rsa
    atf_add_test_case netpgpverify_dsa
}
