/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2012 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2012 Sun Microsystems, Inc.
 */
package org.netbeans.spi.java.project.support;

import java.text.MessageFormat;
import org.netbeans.api.annotations.common.CheckForNull;
import org.netbeans.api.annotations.common.NonNull;
import org.netbeans.api.java.platform.JavaPlatform;
import org.netbeans.api.java.platform.JavaPlatformManager;
import org.netbeans.api.java.platform.Specification;
import org.openide.util.NbPreferences;
import org.openide.util.Parameters;

/**
 * Returns a preferred {@link JavaPlatform} for a new project.
 * @author Tomas Zezula
 * @since 1.46
 */
public final  class PreferredProjectPlatform {
    
    private static final String PREFERRED_PLATFORM = "preferred.platform.{0}";  //NOI18N
    private static final String PLATFORM_ANT_NAME = "platform.ant.name";    //NOI18N
    
    private PreferredProjectPlatform() {
        throw new AssertionError();
    }
    
    /**
     * Returns a preferred {@link JavaPlatform} for a new project.
     * @param platformType the platform type as specified by {@link Specification#getName()}
     * @return the preferred {@link JavaPlatform}
     */
    @CheckForNull
    public static JavaPlatform getPreferredPlatform(@NonNull final String platformType) {
        Parameters.notNull("platformType", platformType);   //NOI18N
        final String platformId = NbPreferences.forModule(PreferredProjectPlatform.class).get(
                MessageFormat.format(PREFERRED_PLATFORM, platformType),
                null);
        final JavaPlatformManager jpm = JavaPlatformManager.getDefault();
        if (platformId != null) {
            for (JavaPlatform jp : jpm.getInstalledPlatforms()) {
                if (platformId.equals(jp.getProperties().get(PLATFORM_ANT_NAME)) &&
                    platformType.equals(jp.getSpecification().getName())) {
                    return jp;
                }
            }
        }
        final JavaPlatform defaultPlatform = jpm.getDefaultPlatform();
        return platformType.equals(defaultPlatform.getSpecification().getName())?
               defaultPlatform:
               null;
    }
    
    /**
     * Sets a preferred {@link JavaPlatform} for a new project.
     * @param platform the preferred {@link JavaPlatform}
     */
    public static void setPreferredPlatform(@NonNull final JavaPlatform platform) {
        Parameters.notNull("platform", platform);   //NOI18N
        final String platformId = platform.getProperties().get(PLATFORM_ANT_NAME);
        if (platformId == null) {
            throw new IllegalArgumentException("Invalid platform, the platform has no platform.ant.name");  //NOI18N
        }
        final String platformType = platform.getSpecification().getName();
        NbPreferences.forModule(PreferredProjectPlatform.class).put(
                MessageFormat.format(PREFERRED_PLATFORM, platformType),
                platformId);
    }
}
