/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.cnd.modelimpl.csm.core;

import java.io.PrintStream;
import org.netbeans.modules.cnd.antlr.ASTVisitor;
import org.netbeans.modules.cnd.antlr.Token;
import org.netbeans.modules.cnd.antlr.collections.AST;
import org.netbeans.modules.cnd.apt.utils.APTUtils;
import org.netbeans.modules.cnd.modelimpl.parser.CsmAST;
import org.netbeans.modules.cnd.modelimpl.parser.FakeAST;
import org.netbeans.modules.cnd.modelimpl.parser.OffsetableAST;
import org.netbeans.modules.cnd.modelimpl.parser.OffsetableFakeAST;
import org.netbeans.modules.cnd.modelimpl.parser.TokenBasedAST;
import org.netbeans.modules.cnd.modelimpl.parser.TokenBasedFakeAST;
import org.netbeans.modules.cnd.modelimpl.parser.generated.CPPTokenTypes;
import org.netbeans.modules.cnd.modelimpl.textcache.NameCache;
import org.openide.util.CharSequences;

/**
 * Miscellaneous AST-related static utility functions
 * @author Vladimir Kvashin
 */
public class AstUtil {

    private AstUtil() {
    }

    public static boolean isEmpty(AST ast, boolean hasFakeChild) {
	if( isEmpty(ast) ) {
	    return true;
	}
	else {
	    return hasFakeChild ? isEmpty(ast.getFirstChild()) : false;
	}
    }

    private static boolean isEmpty(AST ast) {
	return (ast == null || ast.getType() == CPPTokenTypes.EOF);
    }

    public static boolean isElaboratedKeyword(AST ast) {
        if (ast != null) {
            return ast.getType() == CPPTokenTypes.LITERAL_struct ||
                   ast.getType() == CPPTokenTypes.LITERAL_class ||
                   ast.getType() == CPPTokenTypes.LITERAL_union ||
                   ast.getType() == CPPTokenTypes.LITERAL_enum;
        }
        return false;
    }
    
    public static CharSequence getRawNameInChildren(AST ast) {
        return getRawName(findIdToken(ast));
    }

    public static CharSequence[] toRawName(CharSequence rawName) {
        if (rawName == null) {
            return null;
        }
        String[] split = rawName.toString().split("\\."); //NOI18N
        CharSequence[] res = new CharSequence[split.length];
        for(int i = 0; i < split.length; i++) {
            res[i] = CharSequences.create(split[i]);
        }
        return res;
    }
    
    public static CharSequence getRawName(AST token) {
        StringBuilder l = new StringBuilder();
        for( ; token != null; token = token.getNextSibling() ) {
            switch( token.getType() ) {
                case CPPTokenTypes.IDENT:
                    if (l.length()>0) {
                        l.append('.');
                    }
                    l.append(AstUtil.getText(token));
                    break;
                case CPPTokenTypes.SCOPE:
                    break;
                default:
                    //TODO: process templates
                    break;
            }
        }
        return NameCache.getManager().getString(CharSequences.create(l));
    }

    private static AST findIdToken(AST ast) {
        for( AST token = ast.getFirstChild(); token != null; token = token.getNextSibling() ) {
            if( token.getType() == CPPTokenTypes.IDENT ) {
                return token;
            }
            else if( token.getType() == CPPTokenTypes.CSM_QUALIFIED_ID ) {
                return token.getFirstChild();
            }
        }
        return null;
    }

    public static CharSequence findId(AST ast) {
        return findId(ast, -1);
    }

    /**
     * Finds ID (either CPPTokenTypes.CSM_QUALIFIED_ID or CPPTokenTypes.ID)
     * in direct children of the given AST tree
     *
     * @param ast tree to search ID in
     *
     * @param limitingTokenType type of token that, if being found, stops search
     *        -1 means that there is no such token.
     *        This parameter allows, for example, searching until "}" is encountered
     * @return found id
     */
    public static CharSequence findId(AST ast, int limitingTokenType) {
	return findId(ast, limitingTokenType, false);
    }

    /**
     * Finds ID (either CPPTokenTypes.CSM_QUALIFIED_ID or CPPTokenTypes.ID)
     * in direct children of the given AST tree
     *
     * @param ast tree to search ID in
     *
     * @param limitingTokenType type of token that, if being found, stops search
     *        -1 means that there is no such token.
     *        This parameter allows, for example, searching until "}" is encountered
     * @param qualified flag indicating if full qualified id is needed
     * @return id
     */
    public static CharSequence findId(AST ast, int limitingTokenType, boolean qualified) {
        for( AST token = ast.getFirstChild(); token != null; token = token.getNextSibling() ) {
            int type = token.getType();
            if( type == limitingTokenType && limitingTokenType >= 0 ) {
                return null;
            }
            else if( type == CPPTokenTypes.IDENT ) {
                return AstUtil.getText(token);
            }
            else if( type == CPPTokenTypes.CSM_QUALIFIED_ID ) {
		if( qualified ) {
		    return AstUtil.getText(token);
		}
                AST last = getLastChild(token);
                if( last != null) {
                    if( last.getType() == CPPTokenTypes.IDENT ) {
                        return AstUtil.getText(last);
                    }
                    else {
                        AST first = token.getFirstChild();
                        if( first.getType() == CPPTokenTypes.LITERAL_OPERATOR ) {
                            StringBuilder sb = new StringBuilder(AstUtil.getText(first));
                            sb.append(' ');
                            AST next = first.getNextSibling();
                            if( next != null ) {
                                sb.append(AstUtil.getText(next));
                            }
                            return sb;
                        } else if (first.getType() == CPPTokenTypes.IDENT){
                            return AstUtil.getText(first);
                        }
                    }
                }
            }
        }
        return "";
    }

    public static CharSequence getText(AST ast) {
        if (ast instanceof FakeAST) {
            return ((FakeAST)ast).getTextID();
        } else if (ast instanceof CsmAST) {
            return ((CsmAST)ast).getTextID();
        }
        return ast.getText();
    }

    public static AST findMethodName(AST ast){
        AST type = ast.getFirstChild(); // type
        AST qn = null;
        int i = 0;
        while(type != null){
            switch(type.getType()){
                case CPPTokenTypes.LESSTHAN:
                    i++;
                    type = type.getNextSibling();
                    continue;
                case CPPTokenTypes.GREATERTHAN:
                    i--;
                    type = type.getNextSibling();
                    continue;
                case CPPTokenTypes.CSM_TYPE_BUILTIN:
                case CPPTokenTypes.CSM_TYPE_COMPOUND:
                    type = type.getNextSibling();
                    if (i == 0){
                        qn = type;
                    }
                    continue;
                case CPPTokenTypes.IDENT:
                    if (i == 0 && qn == null) {
                        qn = type;
                    }
                    type = type.getNextSibling();
                    continue;
                case CPPTokenTypes.CSM_QUALIFIED_ID:
                    if (i == 0) {
                        qn = type;
                    }
                    type = type.getNextSibling();
                    continue;
                case CPPTokenTypes.CSM_COMPOUND_STATEMENT:
                case CPPTokenTypes.CSM_COMPOUND_STATEMENT_LAZY:
                case CPPTokenTypes.CSM_TRY_CATCH_STATEMENT_LAZY:
                case CPPTokenTypes.COLON:
                case CPPTokenTypes.POINTERTO:
                    break;
                default:
                    type = type.getNextSibling();
                    continue;
            }
            break;
        }
        return qn;
    }
    
    public static AST findTypeNode(AST ast) {
        AST typeAst = AstUtil.findChildOfType(ast, CPPTokenTypes.CSM_TYPE_BUILTIN);
        if (typeAst == null) {
            typeAst = AstUtil.findChildOfType(ast, CPPTokenTypes.CSM_TYPE_COMPOUND);
            if (typeAst == null) {
                typeAst = AstUtil.findChildOfType(ast, CPPTokenTypes.CSM_TYPE_DECLTYPE);
            }
        }        
        return typeAst;
    }
    
    public static boolean isTypeNode(AST ast) {
        return ast != null && 
            (ast.getType() == CPPTokenTypes.CSM_TYPE_BUILTIN ||
             ast.getType() == CPPTokenTypes.CSM_TYPE_COMPOUND ||
             ast.getType() == CPPTokenTypes.CSM_TYPE_DECLTYPE);
    }

    public static boolean hasChildOfType(AST ast, int type) {
        for( AST token = ast.getFirstChild(); token != null; token = token.getNextSibling() ) {
            if( token.getType() == type ) {
                return true;
            }
        }
        return false;
    }

    public static AST findChildOfType(AST ast, int type) {
        return findChildOfType(ast, type, null);
    }
    
    public static AST findChildOfType(AST ast, int type, AST stopToken) {
        for( AST token = ast.getFirstChild(); token != null && token != stopToken; token = token.getNextSibling() ) {
            if( token.getType() == type ) {
                return token;
            }
        }
        return null;
    }    

    public static AST findSiblingOfType(AST ast, int type) {
        return findSiblingOfType(ast, type, null);
    }
    
    public static AST findSiblingOfType(AST ast, int type, AST stopToken) {
        for( AST token = ast; token != null && token != stopToken; token = token.getNextSibling() ) {
            if( token.getType() == type ) {
                return token;
            }
        }
        return null;
    }    

    public static AST findLastSiblingOfType(AST ast, int type) {
        AST result = null;
        for( AST token = ast; token != null; token = token.getNextSibling() ) {
            if( token.getType() == type ) {
                result = token;
            }
        }
        return result;
    }
    
    public static AST skipTokens(AST ast, int...tokens) {
        AST next = ast;
        while (next != null && arrayContains(next.getType(), tokens)) {
            next = next.getNextSibling();
        }
        return next;
    }

    public static AST getLastChild(AST token) {
        if( token == null ) {
            return null;
        }
        AST child = token.getFirstChild();
        if( child != null ) {
            while( child.getNextSibling() != null ) {
                child = child.getNextSibling();
            }
            return child;
        }
        return null;
    }

    public static AST getLastChildRecursively(AST token) {
        if( token == null ) {
            return null;
        }
        if( token.getFirstChild() == null ) {
            return token;
        }
        else {
            AST child = getLastChild(token);
            return getLastChildRecursively(child);
        }
    }
    
    public static AST getLastNonEOFChildRecursively(AST token) {
        if( token == null ) {
            return null;
        }
        AST child = token.getFirstChild();
        if(child == null) {
            return token;
        } else {
            AST lastChild = getLastNonEOFChildRecursively(child);
            while( child.getNextSibling() != null) {
                child = child.getNextSibling();
                AST lastChild2 = getLastNonEOFChildRecursively(child);
                if(lastChild2.getType() != Token.EOF_TYPE && lastChild2 instanceof CsmAST) {
                    lastChild = lastChild2;
                }
            }
            return lastChild;
        }
    }    

    public static OffsetableAST getFirstOffsetableAST(AST node) {
        if( node != null ) {
            if( node instanceof OffsetableAST ) {
                return (OffsetableAST) node;
            }
            else {
                return getFirstOffsetableAST(node.getFirstChild());
            }
        }
        return null;
    }

    public static String toString(AST ast) {
        final StringBuilder out = new StringBuilder();
        ASTVisitor impl = new ASTVisitor() {

            @Override
            public void visit(AST node) {
                print(node, out);
                for (AST node2 = node; node2 != null; node2 = node2.getNextSibling()) {
                    if (node2.getFirstChild() != null) {
                        out.append('>');
                        visit(node2.getFirstChild());
                        out.append('<');
                    }
                }
            }
        };
        impl.visit(ast);
        return out.toString();
    }

    public static void toStream(AST ast, final PrintStream ps) {
        ASTVisitor impl = new ASTVisitor() {
            @Override
            public void visit(AST node) {
		print(node, ps);
                for( AST node2 = node; node2 != null; node2 = node2.getNextSibling() ) {
                    if (node2.getFirstChild() != null) {
			ps.print('>');
                        visit(node2.getFirstChild());
			ps.print('<');
                    }
                }
            }
        };
        impl.visit(ast);
    }

    /**
     * Creates an AST with node <code>n1</code> as root and node <code>n2</code>
     * as its single child, discarding all other children and siblings of
     * both nodes. This function creates copies of nodes, original nodes
     * are not changed.
     *
     * @param n1  root node
     * @param n2  child node
     * @return AST consisting of two given nodes
     */
    public static AST createAST(AST n1, AST n2) {
        AST root = new CsmAST();
        root.initialize(n1);
        AST child = new CsmAST();
        child.initialize(n2);
        root.addChild(child);
        return root;
    }

    private static void print(AST ast, PrintStream ps) {
        ps.print('[');
        ps.print(ast.getText());
        ps.print('(');
        ps.print(ast.getType());
        ps.print(')');
        ps.print(ast.getLine());
        ps.print(':');
        ps.print(ast.getColumn());
        ps.print(']');
        //ps.print('\n');
    }
    
    private static void print(AST ast, StringBuilder out) {
        out.append('[');
        out.append(ast.getText());
        out.append('(');
        out.append(ast.getType());
        out.append(')');
        out.append(ast.getLine());
        out.append(':');
        out.append(ast.getColumn());
        out.append(']');
        //out.append('\n');
    }

    public static String getOffsetString(AST ast) {
        if (ast == null) {
            return "<null>"; // NOI18N
        }
        OffsetableAST startAst = getFirstOffsetableAST(ast);
        AST endAst = getLastChildRecursively(ast);
        if (startAst != null && endAst != null) {
            StringBuilder sb = new StringBuilder();// NOI18N
            sb.append("[").append(startAst.getLine());// NOI18N
            sb.append(":").append(startAst.getColumn());// NOI18N
            sb.append("-").append(endAst.getLine());// NOI18N
            sb.append(":").append(endAst.getColumn());// NOI18N
            sb.append("]"); //NOI18N
            return sb.toString();
        }
        return "<no csm nodes>"; // NOI18N
    }
    
    public static boolean visitAST(ASTTokenVisitor visitor, AST ast) {
        if (ast != null) {
            switch (visitor.visit(ast)) {
                case ABORT:
                    return false;
                case SKIP_SUBTREE:
                    return true;
                case CONTINUE:
                    for (AST insideToken = ast.getFirstChild(); insideToken != null; insideToken = insideToken.getNextSibling()) {
                        if (!visitAST(visitor, insideToken)) {
                            return false;
                        }
                    }                        
            }
        }
        return true;
    }       
    
    /**
     * 
     * @param ast
     * @return true if ast has macro expanded tokens
     */
    public static boolean hasExpandedTokens(AST ast) {
        ASTExpandedTokensChecker checker = new ASTExpandedTokensChecker();
        visitAST(checker, ast);
        return checker.HasExpanded();
    }    
    
    /**
     * Clones AST until stop node is reached
     * @param source
     * @param stopNode - the last AST node to be cloned
     * @return "cloned" AST
     */
    public static AST cloneAST(AST source, AST stopNode) {
        return cloneAST(source, stopNode, true);
    }
    
    /**
     * Clones AST until stop node is reached
     * @param source
     * @param stopNode
     * @param includeLast - true if stopNode should be included
     * @return "cloned" AST
     */    
    public static AST cloneAST(AST source, AST stopNode, boolean includeLast) {
        if (source == null) {
            return null;
        }
        
        AST firstClonedNode = createFakeClone(source);
        AST currentClonedAST = firstClonedNode;
        AST prevClonedAST = null;
        
        while (source != null) {
            currentClonedAST.initialize(source);
            currentClonedAST.setFirstChild(source.getFirstChild());
            if (prevClonedAST != null) {
                prevClonedAST.setNextSibling(currentClonedAST);
            }
            if (source == stopNode) {
                break;
            }
            source = source.getNextSibling();
            prevClonedAST = currentClonedAST;
            currentClonedAST = createFakeClone(source);
        }
        
        if (!includeLast) {
            if (prevClonedAST == null) {
                return null;
            } else {
                prevClonedAST.setNextSibling(null);
            }
        }
        
        return firstClonedNode;
    }
    
    public static AST createFakeClone(AST ast) {
        if (ast instanceof TokenBasedAST) {
            return new TokenBasedFakeAST();
        } else if (ast instanceof OffsetableAST) {
            return new OffsetableFakeAST();
        }
        return new FakeAST();
    }       
    
    public static interface ASTTokenVisitor {
        
        public static enum Action {
            CONTINUE,
            SKIP_SUBTREE,
            ABORT
        }
        
        /**
         * Called on enter
         * @param token 
         * @return what action to perform
         */
        Action visit(AST token);
        
    }                 
    
    public static class ASTExpandedTokensChecker implements ASTTokenVisitor {
    
        private boolean expanded;

        @Override
        public Action visit(AST token) {
            if (token instanceof TokenBasedAST) {
                TokenBasedAST tokenBasedAST = (TokenBasedAST) token;
                if (APTUtils.isMacroExpandedToken(tokenBasedAST.getToken())) {
                    expanded = true;
                    return Action.ABORT;
                }
            }
            return Action.CONTINUE;
        }        

        public boolean HasExpanded() {
            return expanded;
        }
    }    
    
    public static class ASTTokensStringizer implements ASTTokenVisitor {
        protected int numStringizedTokens = 0;
    
        protected final StringBuilder sb = new StringBuilder();

        @Override
        public Action visit(AST token) {
            if (token.getFirstChild() == null) {
                sb.append(token.getText());
                numStringizedTokens++;
            }
            return Action.CONTINUE;
        }
        
        public String getText() {
            return sb.toString();
        }

        public int getNumberOfStringizedTokens() {
            return numStringizedTokens;
        }     
    }
    
    private static boolean arrayContains(int value, int...array) {
        for (int elem : array) {
            if (value == elem) {
                return true;
            }
        }
        return false;
    }
}

 