////////////////////////////////////////////////////////////////////////////////
//
//	Pico Technology USB Device Driver
//
///	\file     PicoUsbDevice_Linux.cpp
///	\brief    Generic Pico USB device class
//
//	Copyright (c) 2007, Pico Technology.
//	All rights reserved.
//   
//	Redistribution and use in source and binary forms, with or without
//	modification, are permitted provided that the following conditions are met:
//		* Redistributions of source code must retain the above copyright
//		  notice, this list of conditions and the following disclaimer.
//		* Redistributions in binary form must reproduce the above copyright
//		  notice, this list of conditions and the following disclaimer in the
//		  documentation and/or other materials provided with the distribution.
//		* The name of Pico Technology may not be used to endorse or promote
//		  products derived from this software without specific prior written
//		  permission.
//
//	THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY ``AS IS'' AND ANY
//	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
//	DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//	ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//	THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//	Version $Id: PicoUsbDevice_Linux.cpp,v 1.30 2008/04/22 10:51:40 pei Exp $
//
////////////////////////////////////////////////////////////////////////////////


//#define DEBUG 1


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include <errno.h>
#include <string.h>
#include <pthread.h>
#include <signal.h>

#include <fcntl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <dirent.h>


#include "PicoUsbDevice_Linux.h"
#include "LinuxUsbFS.h"
#include "PicoUsbID.h"
#include "PicoPortability.h"

#define PICO_USB_INTERFACE 0
#define PICO_USB_TIMEOUT 1000


pthread_mutex_t PicoLinuxUsbDevice::handlesMutex = PTHREAD_MUTEX_INITIALIZER;
bool PicoLinuxUsbDevice::handlesInitialized = 0;
unsigned char * PicoLinuxUsbDevice::deviceHandles = NULL;

const int maxDeviceHandles = 256 / 8;


//////////////////////////////////////////////////////////////////////////
/// Performs any (OS-specific) initialisation.
/// On Linux this is an empty function.
//////////////////////////////////////////////////////////////////////////
int PicoUsbDevice::Init() {
	// Perform any necessary initialization here
	return 0;
}

//////////////////////////////////////////////////////////////////////////
/// Iterates through the currently connected USB devices to count 
/// the number of devices with the Pico VID and a matching PID.
/// Does not check DID so does not differentiate between variants
/// (example: you can count all the PS5000s but not all the PS5203s)
/// Just calls Enumerate() with length of 0.
/// <param name="product">The product ID to match</param>
/// <returns>The number of matching devices</returns>
//////////////////////////////////////////////////////////////////////////

int PicoUsbDevice::Count(unsigned short product) 
{
	return PicoUsbDevice::Enumerate(NULL, 0, product);
}


//////////////////////////////////////////////////////////////////////////
/// Iterates through the currently connected USB devices and returns their paths
/// <returns>Vector of char * containing the paths.</returns>
//////////////////////////////////////////////////////////////////////////
std::vector<char *> * PicoLinuxUsbDevice::ListUSBDevices() {

#if DEBUG
	printf("PicoLinuxUsbDevice::ListUSBDevices()\n");
#endif	
	
	DIR *busdir, *dir;
	struct dirent *entry;
	char buspath[PATH_MAX + 1];
	char dirpath[PATH_MAX + 1];
	char * filepath;
	std::vector<char *>  * deviceList = new std::vector<char *>;

	snprintf(buspath, PATH_MAX, "%s", "/dev/bus/usb");
	busdir = opendir(buspath);
	if (!busdir)
	{
		snprintf(buspath, PATH_MAX, "%s", "/proc/bus/usb");
		busdir = opendir(buspath);
		if (!busdir)
		{
			return deviceList;
		}
	}
		

	while (entry = readdir(busdir)) {
		
		if (entry->d_name[0] == '.' || !strchr("0123456789", entry->d_name[strlen(entry->d_name) - 1])) 
			continue;
		
		snprintf(dirpath, PATH_MAX, "%s/%s", buspath, entry->d_name);

		
		if (!(dir = opendir(dirpath)))
			continue;

		while (entry = readdir(dir)) {
			/* Skip anything starting with a . */
			if (entry->d_name[0] == '.')
				continue;
			filepath = new char[PATH_MAX + 1];
			snprintf(filepath, PATH_MAX, "%s/%s", dirpath, entry->d_name);
			deviceList->push_back(filepath);
		}

		closedir(dir);
		
	}
	
	closedir(busdir);

	return deviceList;
}


//////////////////////////////////////////////////////////////////////////
/// Iterates through the currently connected USB dev ices to make a list of
/// the devices with the Pico VID and a matching PID.
/// Does not check DID so does not differentiate between variants
/// (example: you can list all the PS5000s but not all the PS5203s). This also
/// means that Enumerate() will find connected devices which have not had 
/// firmware downloaded, so you should check DID in calling code.
/// <param name="list">Pointer to an array of pointers to PicoUsbDevice which 
/// the function will fill with the devices it detects</param>
/// <param name="length">The number of entries in the above array. This is the 
/// maximum number of devices that the function will return</param>
/// <param name="product">The product ID to match</param>
/// <returns>The number of matching devices which have been found. May be
/// larger than length, in which case only the first length devices appear
/// in list.</returns>
//////////////////////////////////////////////////////////////////////////
int PicoUsbDevice::Enumerate(PicoUsbDevice **list,unsigned int length,unsigned short product) {

#if DEBUG
	printf("PicoLinuxUsbDevice::Enumerate()\n");
#endif	

	int deviceCount = 0;

	std::vector<char *> * devices = PicoLinuxUsbDevice::ListUSBDevices();
	std::vector<char *>::const_iterator it;
	for (it=devices->begin(); it != devices->end(); it++)
	{
		unsigned char device_desc[18];

		int fd, ret;

		fd = open(*it, O_RDWR);
		delete[]  *it;
		if (fd < 0) {
			continue;
		}

		ret = read(fd, (void *)device_desc, 18);
		if (ret < 0) {
			close(fd);
			continue;
		}

		if (VENDOR_ID_PICO_TECHNOLOGY != device_desc[8] + device_desc[9] * 256 || 
				product != device_desc[10] + device_desc[11] * 256 ){
			close(fd);
			continue;
		}

		// We have a device, do stuff!
		if (deviceCount < length)
		{
			list[deviceCount]= (PicoUsbDevice *) new PicoLinuxUsbDevice(fd, device_desc);
			if (list[deviceCount])
				deviceCount++;
		}
	}

	devices->clear();
	delete devices;
	return deviceCount;

}

#define PICO_USB_URB_SIGNAL (SIGRTMIN + 4) 

static void UrbCompletionSigAction(int signal, siginfo_t * siginfo, void * unused)
{
	assert (signal == PICO_USB_URB_SIGNAL);
	if (!siginfo || siginfo->si_code != SI_ASYNCIO)
	{
#if DEBUG
		printf("UrbCompletionSigAction called with unknown si_code %d\n", siginfo->si_code);
#endif
		return;
	}
	linux_usbfs_urb * urb = (linux_usbfs_urb *)(siginfo->si_addr);
	assert (urb);
	UrbContext * ctx = (UrbContext *)(urb->usercontext);
	assert (ctx);
	PicoLinuxUsbDevice * dev = (PicoLinuxUsbDevice *)(ctx->dev);
	

	// Modified on 11/03/2008 by P_TANG
	// Open/Close units: Assertion failed
	
	//assert (dev);
	if (dev == NULL)
	{
#if DEBUG
	printf("PicoUsbDevice_Linux::UrbCompletionSigAction dev == NULL %x\n", dev);
#endif	
	}
	else
	{
#if DEBUG
	printf("PicoUsbDevice_Linux::UrbCompletionSigAction dev != NULL %x\n", dev);
#endif		
	//assert (siginfo->si_errno == urb->status);
	if (siginfo->si_errno != urb->status)
	{
#if DEBUG
		printf("UrbCompletionSigAction called with siginfo->si_errno (%d) != urb->status (%d)\n", siginfo->si_errno, urb->status);
#endif
		return;
	}

	// Modified on 11/03/2008 by P_TANG
#if DEBUG
	printf("UrbCompletionSigAction processing urb for device %x\n", dev);
#endif
	
	dev->ProcessUrb(ctx);
	
	}
}


//////////////////////////////////////////////////////////////////////////
/// Constructor. Only called by Enumerate()
/// <param name="newDevice">Libusb handle to new device</param> 
//////////////////////////////////////////////////////////////////////////
PicoLinuxUsbDevice::PicoLinuxUsbDevice(int newDevice, unsigned char * device_desc) {
#if DEBUG
	printf("PicoLinuxUsbDevice::PicoLinuxUsbDevice()\n");
#endif
	
	int ret;
	
	pthread_mutex_lock(&handlesMutex);
	
	if (!handlesInitialized) {
#if DEBUG
	printf("PicoLinuxUsbDevice::Init handles\n", Endpoints.size());
#endif
		handlesInitialized = true;
		deviceHandles = new unsigned char[maxDeviceHandles];
		assert(deviceHandles);
		deviceHandles[0]=0x01; // Don't use 0 as a handle
		for (int i = 1; i < maxDeviceHandles; i++) {
			deviceHandles[i] = 0;
		}
	}
	
	pthread_mutex_unlock(&handlesMutex);
	
	// Set signal handler for USB urb completion
	struct sigaction sa;
	sa.sa_sigaction = UrbCompletionSigAction;
	sa.sa_flags = SA_SIGINFO;
	sigemptyset(&(sa.sa_mask));
	
	sigaction(PICO_USB_URB_SIGNAL, &sa, NULL);
	
	// Just set up some member variables
	assert(newDevice);
	memcpy(&Descriptor, device_desc, sizeof(Descriptor));
	device=newDevice; 
	serialString=NULL;
	handle = 0;
	state = PICODEVICE_STATE_CLOSED;	
	pthread_mutex_init(&mutex,NULL);
	pthread_mutex_init(&urbMutex,NULL);
	
	
	unsigned char *tmpBuf, *bufStart, *bufEnd;
	EndpointDescriptor * ep;
	
	// Read first 8 bytes of config descriptor as per USB spec
	bufStart = (unsigned char *)malloc(8);
	ret = read(device, (void *)bufStart, 8);
	if (ret < 8) { }
	
	int totallength=(bufStart[3] << 8) | bufStart[2];
#if DEBUG
	printf("PicoLinuxUsbDevice::Descriptor starts %x %x %x %x %x %x %x %x \n", bufStart[0],  bufStart[1], 
			 bufStart[2],  bufStart[3],  bufStart[4],  bufStart[5],  bufStart[6],  bufStart[7] );
	printf("PicoLinuxUsbDevice::Total descriptor length %d\n", totallength);
#endif
	
	bufStart = (unsigned char *)realloc( (void *)bufStart, totallength);
	tmpBuf = bufStart;
	bufEnd = bufStart + totallength;
	if (!bufStart) { }
	
	// Read rest of config descriptor
	ret = read(device, (void *)(bufStart + 8), totallength - 8);
	if (ret < totallength - 8) { }
	
	// Skip to the interface descriptor
#define INTERFACE_DESCRIPTOR 0x04
#define ENDPOINT_DESCRIPTOR 0x05
	while (bufStart[1] != INTERFACE_DESCRIPTOR && &bufStart[4] < bufEnd)
		bufStart += *bufStart;
	
	// Read NumEndpoints
	int numEndpoints = bufStart[4];
	highSpeed = false;
	
	// Skip to the next endpoint descriptor
	do
	{
		bufStart += bufStart[0];
		if (&bufStart[1] >= bufEnd || &bufStart[bufStart[0] - 1] >= bufEnd)
			break;
		if (bufStart[1] != ENDPOINT_DESCRIPTOR || bufStart[0] > sizeof(EndpointDescriptor))
			continue;
		ep = new EndpointDescriptor();
#if DEBUG
	printf("PicoLinuxUsbDevice::EP is %x %x %x %x %x %x %x\n", bufStart[0],  bufStart[1], 
				 bufStart[2],  bufStart[3],  bufStart[4],  bufStart[5],  bufStart[6]);
	printf("PicoLinuxUsbDevice::Memcpy ep %d, %d bytes\n", Endpoints.size(), sizeof(EndpointDescriptor));
#endif
		memcpy(ep, bufStart, sizeof(EndpointDescriptor));
		highSpeed |= (ep->wMaxPacketSize > 64);
		Endpoints.push_back(ep);
		numEndpoints--;
	} while (numEndpoints);
		
#if DEBUG
	printf("PicoLinuxUsbDevice::Got %d Pipes\n", Endpoints.size());
#endif

	// Modified on 25/03/2008
	if (tmpBuf != NULL)
		free(tmpBuf);
	// Modified on 25/03/2008
}

//////////////////////////////////////////////////////////////////////////
/// Destructor. Closes the libusb device and frees storage
//////////////////////////////////////////////////////////////////////////
PicoLinuxUsbDevice::~PicoLinuxUsbDevice(void) {

#if DEBUG
	printf("PicoLinuxUsbDevice::~PicoLinuxUsbDevice()\n");
#endif

  // Clean up
	
	// Modified on 25/03/2008 by P_TANG
	
	//std::vector<EndpointDescriptor *>::iterator it;
	int i = 0;
	for(; i < Endpoints.size(); i ++)
		delete Endpoints[i];
	
	Endpoints.clear();
	// Modified on 25/03/2008 by P_TANG
	
	if(device) {
		Close();
		// Modified on 11/03/2008 by P_TANG
		close(device);
	}
	
#if DEBUG
	printf("PicoLinuxUsbDevice::~PicoLinuxUsbDevice(): close()\n");
#endif
	
	
	
	if(serialString) {
		free(serialString);
	}
	
#if DEBUG
	printf("PicoLinuxUsbDevice::~PicoLinuxUsbDevice(): free string\n");
#endif

	pthread_mutex_destroy(&mutex);
	pthread_mutex_destroy(&urbMutex);
	
#if DEBUG
	printf("PicoLinuxUsbDevice::~PicoLinuxUsbDevice(): free mutex\n");
#endif

}

//////////////////////////////////////////////////////////////////////////
/// Get current connection state. The state is cached by PicoUsbDevice 
/// after each USB transaction and
/// calling this function returns the cached value.
/// <returns> PICODEVICE_STATES value indicating the connection state
/// Possible values are PICODEVICE_STATE_DISCONNECTED,PICODEVICE_STATE_LOCKED,
/// PICODEVICE_STATE_CLOSED and PICODEVICE_STATE_OPEN </returns>
//////////////////////////////////////////////////////////////////////////
PICODEVICE_STATES PicoLinuxUsbDevice::GetDeviceState() {
  // Return state variable
#if DEBUG
	printf("PicoLinuxUsbDevice::GetDeviceState(), trying to get mutex\n");
#endif
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoLinuxUsbDevice::GetDeviceState()");
#endif
	
	PICODEVICE_STATES tmpState=state;
	
	pthread_mutex_unlock(&mutex);
	
#if DEBUG
	printf("=%i\n",tmpState);
#endif
	
	return tmpState;
}

//////////////////////////////////////////////////////////////////////////
/// Attempt to open the device.
/// <returns> PICODEVICE_STATES value indicating the connection state after 
/// the attempt at opening.
/// Possible values are PICODEVICE_STATE_DISCONNECTED,PICODEVICE_STATE_LOCKED,
/// PICODEVICE_STATE_CLOSED and PICODEVICE_STATE_OPEN . The latter
/// indicates that the opening was successful: for a description of
/// the others see PICODEVICE_STATES</returns>
//////////////////////////////////////////////////////////////////////////
PICODEVICE_STATES PicoLinuxUsbDevice::Open(void){
	PICODEVICE_STATES tmpState;
	int err;
	int i, j;
	
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoLinuxUsbDevice::Open()\n");
#endif
	assert(device);
	
	handle = 0;
	
	// Try to open the device
	int interface = PICO_USB_INTERFACE;
	  err = ioctl(device, IOCTL_USBFS_CLAIMINTERFACE, &interface);
	if(err < 0 && EBUSY == errno) {
		// The device is already open
#if DEBUG
		printf("PicoLinuxUsbDevice::Open: Interface already in use\n");
#endif
		state=tmpState=PICODEVICE_STATE_LOCKED;
	} else if (err == 0){
		
		
		
		
#if DEBUG
		printf("PicoLinuxUsbDevice::Open: Reset endpoints\n");
#endif
		
		err = ioctl(device, IOCTL_USBFS_RESET, NULL);

		for (int pipe = 0; pipe < Endpoints.size(); pipe++)
		{
			ioctl(device, IOCTL_USBFS_RESETEP, &((Endpoints.at(pipe))->bEndpointAddress));
		}

#if DEBUG
		printf("PicoLinuxUsbDevice::Open: Set interface\n");
#endif
		struct linux_usbfs_setinterface set_if = {PICO_USB_INTERFACE,0};
		err = ioctl(device, IOCTL_USBFS_SETINTERFACE, &set_if);
#if DEBUG
		printf("PicoLinuxUsbDevice::Open: Set interface done, returned %d, errno %d\n", err, errno);
#endif
		for (int pipe = 0; pipe < Endpoints.size(); pipe++)
		{
			ioctl(device, IOCTL_USBFS_CLEAR_HALT, &((Endpoints.at(pipe))->bEndpointAddress));
			ioctl(device, IOCTL_USBFS_CLEAR_HALT, &((Endpoints.at(pipe))->bEndpointAddress));
			ioctl(device, IOCTL_USBFS_CLEAR_HALT, &((Endpoints.at(pipe))->bEndpointAddress));
			ioctl(device, IOCTL_USBFS_CLEAR_HALT, &((Endpoints.at(pipe))->bEndpointAddress));
		}

		// Look for an available handle
		pthread_mutex_lock(&handlesMutex);
		
		for (i = 0; i < maxDeviceHandles; i++) {
			// Look for a handle location which isn't in use
			if (deviceHandles[i] != 0xFF) 
				break;
		}
		if (i == maxDeviceHandles) { 
			// We've run out of handles.
			state=tmpState=PICODEVICE_STATE_CLOSED;
		} else {
			for (j = 0; j < 8; j++) {  
				if (!(deviceHandles[i] & (1<<j))) 
					break;
			}
	
			handle = (i * 8) + j;
			deviceHandles[i] |= (1<<j);
#if DEBUG
			printf("PicoLinuxUsbDevice::Open: Issued handle %d\n", handle);
#endif
			
		// Success
			state=tmpState=PICODEVICE_STATE_OPEN;
		}
		pthread_mutex_unlock(&handlesMutex);
		
	} else {
	// Other failure
#if DEBUG
		printf("PicoLinuxUsbDevice::Open: Can't claim interface, error %d\n", err);
#endif
		state=tmpState=PICODEVICE_STATE_CLOSED;
	}

	pthread_mutex_unlock(&mutex);

	if (tmpState==PICODEVICE_STATE_OPEN)
			GetPipes();

	
#if DEBUG
	printf("PicoLinuxUsbDevice::Open: Exit\n");
#endif
	return tmpState;
}

//////////////////////////////////////////////////////////////////////////
/// Get handle for use with C API
/// <returns>The handle of the device</returns>
//////////////////////////////////////////////////////////////////////////
short PicoLinuxUsbDevice::GetHandle(void) {

#if DEBUG
	printf("PicoLinuxUsbDevice::GetHandle()\n");
#endif

	return handle;
}


//////////////////////////////////////////////////////////////////////////
/// Get Product ID. For an explanation of the ID numbers see
/// PicoUsbID.h
/// <returns>The USB Product ID of the device</returns>
//////////////////////////////////////////////////////////////////////////
int PicoLinuxUsbDevice::GetPID(void) {
#if DEBUG
	printf("PicoLinuxUsbDevice::GetPID()\n");
#endif

	return Descriptor.idProduct;
	
}

//////////////////////////////////////////////////////////////////////////
/// Get Device ID. For an explanation of the ID numbers see
/// PicoUsbID.h
/// <returns>The USB Device ID of the device</returns>
//////////////////////////////////////////////////////////////////////////
int PicoLinuxUsbDevice::GetDID(void) {
#if DEBUG
	printf("PicoLinuxUsbDevice::GetDID()\n");
#endif

	return Descriptor.bcdDevice;
}

//////////////////////////////////////////////////////////////////////////
/// Get Device Class. For some Pico products this identifies whether FW is loaded
/// <returns>The USB Device Class of the device</returns>
//////////////////////////////////////////////////////////////////////////
int PicoLinuxUsbDevice::GetDeviceClass(void) {
#if DEBUG
	printf("PicoLinuxUsbDevice::GetDeviceClass()\n");
#endif

	return Descriptor.bDeviceClass;
}

//////////////////////////////////////////////////////////////////////////
/// Get serial number. Requires Mac FW
/// <returns>The USB Serial Number of the device</returns>
//////////////////////////////////////////////////////////////////////////
const char *PicoLinuxUsbDevice::GetSerialString(void) {
	int ret, i, j;
	pthread_mutex_lock(&mutex);

#if DEBUG
	printf("PicoLinuxUsbDevice::GetSerialString()\n");
#endif

	assert(device);
	if(serialString==NULL){
#if DEBUG
		printf("PicoLinuxUsbDevice::GetSerialString(): Retrieving string from device\n");
#endif
	
		char buf[255];	/* Some devices choke on size > 255 */
	
		
		struct linux_usbfs_ctrltransfer ctrl = 
		{
			0x80, // In transfer
			0x06, // Get descriptor
			0x0300,// + Descriptor.iSerialNumber,
			0,//nb + Descriptor.iSerialNumber, // Index of string
			sizeof(buf), // Data length
			1000, // Timeout in ms
			(uint8_t *) buf // Data
		};
	
		ret = ioctl(device, IOCTL_USBFS_CONTROL, &ctrl);

#if DEBUG
		printf("PicoLinuxUsbDevice::GetSerialString: ret = %d ------ 1\n", ret);
#endif
		//printf("PicoLinuxUsbDevice::GetSerialString -- 1: buf = %s \n", buf);
	  
		if (ret < 4){
#if DEBUG
			printf("PicoLinuxUsbDevice::GetSerialString: Error getting string\n");
#endif
			serialString=(char*)malloc(21 * sizeof(char));
			for (i = 0, j = 2; i < 20 && j < ret; i++, j+=2)
				serialString[i] = buf[j];
			serialString[i+1] = 0;
		}
		else
		{
			int langid = buf[2] | (buf[3] << 8);

			struct linux_usbfs_ctrltransfer ctrl_des = 
			{
				0x80, // In transfer
				0x06, // Get descriptor
				0x0300 + Descriptor.iSerialNumber,
				langid,//nb + Descriptor.iSerialNumber, // Index of string
				sizeof(buf), // Data length
				1000, // Timeout in ms
				(uint8_t *) buf // Data
			};
			
			ret = ioctl(device, IOCTL_USBFS_CONTROL, &ctrl_des);

			if (ret < 0)
			{
#if DEBUG
			printf("PicoLinuxUsbDevice::GetSerialString: Error getting string\n");
#endif
			}

			serialString=(char*)malloc(21 * sizeof(char));
			for (i = 0, j = 2; i < 20 && j < ret; i++, j+=2)
				serialString[i] = buf[j];
			serialString[i+1] = 0;
		}
	}
	pthread_mutex_unlock(&mutex);

#if DEBUG
	printf("PicoLinuxUsbDevice::GetSerialString(): Serial is %s\n", serialString);
#endif
	
	return serialString;
}

//////////////////////////////////////////////////////////////////////////
/// Get pipes (number of endpoints)
//////////////////////////////////////////////////////////////////////////
int PicoLinuxUsbDevice::GetPipes() {
	int ret; 
#if DEBUG
	printf("PicoLinuxUsbDevice::GetPipes()\n");
#endif
#if DEBUG

	for (int pipe = 0; pipe < Endpoints.size(); pipe++)
		GetPipeInformation(pipe);
#endif

	return Endpoints.size();

}
//////////////////////////////////////////////////////////////////////////
/// Get status of a given pipe
//////////////////////////////////////////////////////////////////////////
PicoUsbDevice::PipeInformation PicoLinuxUsbDevice::GetPipeInformation(int pipeNumber) {
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoLinuxUsbDevice::GetPipeInformation(%i)\n",pipeNumber);
#endif
	PicoUsbDevice::PipeInformation info;
	// TODO: Check these (only used for debug...)
	info.direction=(Endpoints.at(pipeNumber))->bEndpointAddress & 0x80;
	info.number=(Endpoints.at(pipeNumber))->bEndpointAddress;
	info.transferType=(Endpoints.at(pipeNumber))->bmAttributes;
	info.maxPacketSize=(Endpoints.at(pipeNumber))->wMaxPacketSize;
	info.interval=(Endpoints.at(pipeNumber))->bInterval;
	
	// TODO: Fix this
	info.status = StatusOK;

#if DEBUG
	printf("PicoLinuxUsbDevice::GetPipeInformation: Direction %x\n",info.direction);
	printf("PicoLinuxUsbDevice::GetPipeInformation: Number %x\n",info.number);
	printf("PicoLinuxUsbDevice::GetPipeInformation: TransferType %x\n",info.transferType);
	printf("PicoLinuxUsbDevice::GetPipeInformation: MaxPacketSize %x\n",info.maxPacketSize);
	printf("PicoLinuxUsbDevice::GetPipeInformation: Interval %x\n",info.interval);
	printf("PicoLinuxUsbDevice::GetPipeInformation: Status %x\n",info.status);
#endif


	pthread_mutex_unlock(&mutex);
	
	return info;
}

void PicoLinuxUsbDevice::ProcessUrb(UrbContext * urbctx)
{
#if DEBUG
		printf("ProcessUrb...\n");
#endif
	pthread_mutex_lock(&urbMutex);
	urbctx->completed = 1;
	pthread_mutex_unlock(&urbMutex);


}




//////////////////////////////////////////////////////////////////////////
/// Usb Bulk transfer (read or write) on given pipe.
/// <param name="pipeNumber">The pipe to use. Note that the pipe numbers
/// start at 1 whereas Linux USB endpoint addresses start at 0.</param>
/// <param name="buf">Pointer to a location in which to place the data read
/// from the pipe. The caller must allocate sufficient storage in this location
/// for up to size bytes</param>
/// <param name="size">Pointer to the number of bytes to attempt to read.
/// Function will read as many bytes as are available up to a maximum of size
/// and place the number of bytes actually read back into size.</param>
/// <returns>PICO_RETURNS value indicating whether an error occurred. Note that 
/// failing to read the requested number of bytes is an error, and in this case 
/// size indicates the number of bytes of valid data available.</returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS PicoLinuxUsbDevice::BulkTransfer(int pipeNumber,void *buf, unsigned int *size) {
	PICO_RETURNS retVal;
	if (!size || *size == 0)
		return PICO_INCOMPLETE_USB_TRANSFER;
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoLinuxUsbDevice::BulkTransfer(Pipe %i, Size %d)\n",pipeNumber, *size);
 	long startTime = -1 * GetTickCount();
#endif
 	
 	// Any thread which will lock urbMutex must mask PICO_USB_URB_SIGNAL so it
 	// cannot be interrupted by the signal handler with the mutex locked
 	sigset_t signal_set;
 	sigfillset( &signal_set );
 	pthread_sigmask( SIG_BLOCK, &signal_set, NULL );
 	
 	// Set timeout to the default (1 second) plus 1 second for each 250kB
 	// to be transferred, otherwise large transfers may time out (example: 
 	// transferring the full 128MB buffer of a PS5204 might take up to 3 
 	// minutes over USB1.1, and we mustn't time out during this.)
 	// TODO: it would be nice if the timeout were full-/high-speed dependant.
 	struct timeval timeout, timenow;
 	gettimeofday(&timenow, NULL);
 	timeout.tv_sec = timenow.tv_sec + (PICO_USB_TIMEOUT * (1 + *size/250000)) / 1000;
 	timeout.tv_usec = timenow.tv_usec + (PICO_USB_TIMEOUT * (1 + *size/250000)) % 1000 * 1000;
 	  
 	if (timeout.tv_usec > 1000000) {
 		timeout.tv_usec -= 1000000;
 		timeout.tv_sec++;
 	  }

 	
 	int bytesSubmitted = 0, bytesSucceded = 0;
 	int UrbPacketSize = 1<<14;
 	UrbContext * urbctx;
 	struct  linux_usbfs_urb * urb;
 	std::vector<linux_usbfs_urb *> urbs;
	int requested;
	int ret, waiting;
	if (Endpoints[pipeNumber]->wMaxPacketSize < 512 && Endpoints[pipeNumber]->bmAttributes & 0x02 && highSpeed)
		UrbPacketSize = Endpoints[pipeNumber]->wMaxPacketSize;
	
	
	do {
		
		
			requested = *size - bytesSubmitted;
			if (requested > UrbPacketSize)
				requested = UrbPacketSize;
			urb = new linux_usbfs_urb();
			urbctx = new UrbContext();
			urb-> status = 0;
			urb-> start_frame = 0;
			urb-> error_count = 0;
			urb->type = 3; // Bulk
			urb->endpoint = Endpoints[pipeNumber]->bEndpointAddress;
			urb->flags = 0;
			urb->buffer = (char *)buf + bytesSubmitted;
			urb->buffer_length = requested;
			urb->signr = PICO_USB_URB_SIGNAL;
			urb->actual_length = 0;
			urb->number_of_packets = 0;
			urbctx->completed=0;
			urbctx->dev = this;
			urb->usercontext = urbctx;
			pthread_mutex_lock(&urbMutex);
#if DEBUG
	printf("PicoLinuxUsbDevice::BulkTransfer submitting an URB, endpoint %x, length %d\n",urb->endpoint, requested);
#endif
			ret = ioctl(device, IOCTL_USBFS_SUBMITURB, urb);
			
			if (ret == 0)
				urbs.push_back(urb);
			
			pthread_mutex_unlock(&urbMutex);
			bytesSubmitted += requested;
#if DEBUG
			if (ret != 0)
				printf("PicoLinuxUsbDevice::IOCTL_USBFS_SUBMITURB returned %d, errno is %d\n",ret, errno);
			
			printf("PicoLinuxUsbDevice-----------1 ::IOCTL_USBFS_SUBMITURB returned %d, errno is %d\n",ret, errno);
#endif
			
	} while (ret == 0 && bytesSubmitted < *size);
	

	std::vector<linux_usbfs_urb *>::const_iterator it;
	for (it=urbs.begin(); it != urbs.end(); it++)
	{
		pthread_mutex_lock(&urbMutex);
		urb = *it;
		while (!((UrbContext *)(urb->usercontext))->completed)
		{
			
// TODO: This is a hack to make the PS5000 not use timeouts. It should be replaced with a proper variable-timeout function
#ifndef PICO_NO_USB_TIMEOUT
			gettimeofday (&timenow, NULL);
			if (timenow.tv_sec > timeout.tv_sec)
				break;
#endif			
			pthread_mutex_unlock(&urbMutex);
			usleep(1);
			pthread_mutex_lock(&urbMutex);
		}
		pthread_mutex_unlock(&urbMutex);
		struct linux_usbfs_urb * context = NULL;
		int ret = ioctl(device, IOCTL_USBFS_REAPURBNDELAY, &context);
#if DEBUG
		if (ret != 0)
			printf("PicoLinuxUsbDevice::IOCTL_USBFS_REAPURBNDELAY returned %d, errno is %d\n",ret, errno);
#endif
		if (ret < 0) 
		{
			ioctl(device, IOCTL_USBFS_DISCARDURB, &urb);
			ioctl(device, IOCTL_USBFS_REAPURB, &context);
		}
		else
		{


			// Modified on 11/03/2008 by P_TANG 
			// Assertion failed when the units connect the host for a while

			//assert (context == urb);
			if (context != urb)
			{
#if DEBUG
				printf("PicoUsbDevice_linux::context != urb happened in URBS' loop");
#endif
			}
			else
			{

#if DEBUG		
				printf("ioctl OK = %d \n", ret);
#endif


				// Modified on 06/03/2008 by P_TANG
				// assert (urb->actual_length == urb->buffer_length);
				if (urb->actual_length != urb->buffer_length)
				{
#if DEBUG
				printf("PicoUsbDevice_Linux::urb->actual_length is: %d and urb->buffer_length is: %d \n", urb->actual_length, urb->buffer_length);
#endif

				}
				else
				{
#if DEBUG
					printf("PicoUsbDevice_Linux::Continue URBS' loop, urb->actual_length is: %d and urb->buffer_length is: %d \n", urb->actual_length, urb->buffer_length);
#endif				


					// Modified on 06/03/2008 by P_TANG			


				}
				bytesSucceded += urb->actual_length;

			}
		}

		delete urb->usercontext;
		delete urb;

	}


	urbs.clear();
	pthread_mutex_unlock(&mutex);
#if DEBUG
	startTime += GetTickCount();
	printf("Time to read %d kbytes %d ms\n", bytesSubmitted/1024, startTime);
	printf("PicoLinuxUsbDevice::BulkTransfer wanted %d bytes got %d bytes\n\n\n",*size, bytesSucceded);
#endif
	// Modified on 13/03/2008 by P_TANG
	// printf("PicoLinuxUsbDevice::BulkTransfer wanted %d bytes got %d bytes\n\n\n",*size, bytesSucceded);
	// Modified on 13/03/2008 by P_TANG
	
	
	if (bytesSucceded == *size) // We read the correct number of bytes
	{
		retVal = PICO_SUCCESS;
		*size = bytesSubmitted;
	}
	else if (bytesSucceded > 0) // We read some bytes but not enough
	{
		retVal = PICO_INCOMPLETE_USB_TRANSFER;
		*size = bytesSucceded;
	}
	else // An error occurred during the read
	{
		retVal = PICO_INCOMPLETE_USB_TRANSFER;
		state = PICODEVICE_STATE_DISCONNECTED;
		*size = 0;
	}
	return retVal;
}
//////////////////////////////////////////////////////////////////////////
/// Read bytes from given pipe.
/// <param name="pipeNumber">The pipe to read from. Note that the pipe numbers
/// start at 1 whereas Linux USB endpoint addresses start at 0.</param>
/// <param name="buf">Pointer to a location in which to place the data read
/// from the pipe. The caller must allocate sufficient storage in this location
/// for up to size bytes</param>
/// <param name="size">Pointer to the number of bytes to attempt to read.
/// Function will read as many bytes as are available up to a maximum of size
/// and place the number of bytes actually read back into size.</param>
/// <returns>PICO_RETURNS value indicating whether an error occurred. Note that 
/// failing to read the requested number of bytes is an error, and in this case 
/// size indicates the number of bytes of valid data available.</returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS PicoLinuxUsbDevice::ReadPipe(int pipeNumber,void *buf, unsigned int *size) {
	PICO_RETURNS retVal;
	retVal = BulkTransfer(pipeNumber, buf, size);
	return retVal;
}

//////////////////////////////////////////////////////////////////////////
/// Write bytes to a given pipe.
/// <param name="pipeNumber">The pipe to write to. Note that the pipe numbers
/// start at 1 whereas Linux USB endpoint addresses start at 0.</param>
/// <param name="buf">Pointer to a location in which the data to be written
/// is found</param>
/// <param name="size">Pointer to the number of bytes to attempt to write.
/// </param>
/// <returns>PICO_RETURNS value indicating whether an error occurred. </returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS PicoLinuxUsbDevice::WritePipe(int pipeNumber,void *buf, unsigned int size) {
	PICO_RETURNS retVal;
	retVal = BulkTransfer(pipeNumber, buf, &size);
	return retVal;
}

//////////////////////////////////////////////////////////////////////////
/// Flush pipe
/// Clears a stall condition
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS PicoLinuxUsbDevice::ResetPipe(int pipeNumber) {
	int ok = 0;
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoLinuxUsbDevice::ResetPipe(%i)\n",pipeNumber);
#endif

	ok = ioctl(device, IOCTL_USBFS_CLEAR_HALT, &((Endpoints.at(pipeNumber))->bEndpointAddress));
	ok &= ioctl(device, IOCTL_USBFS_CLEAR_HALT, &((Endpoints.at(pipeNumber))->bEndpointAddress));
	ok &= ioctl(device, IOCTL_USBFS_CLEAR_HALT, &((Endpoints.at(pipeNumber))->bEndpointAddress));
	ok &=ioctl(device, IOCTL_USBFS_CLEAR_HALT, &((Endpoints.at(pipeNumber))->bEndpointAddress));
	
	pthread_mutex_unlock(&mutex);
	
	return PICO_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////
/// Close USB device
//////////////////////////////////////////////////////////////////////////
void PicoLinuxUsbDevice::Close() {
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoLinuxUsbDevice::Close()\n");
#endif
	
	assert(device);
	
	// Close interface
#if DEBUG
	printf("PicoLinuxUsbDevice::Close: Closing interface\n");
#endif
	
	int interface = PICO_USB_INTERFACE;
	ioctl(device, IOCTL_USBFS_RELEASEINTERFACE, &interface);

	//usb_release_interface(device, PICO_USB_INTERFACE);
	// Close device
	
#if DEBUG
	printf("PicoLinuxUsbDevice::Close: Closing device\n");
#endif
	//close(device);	
	state=PICODEVICE_STATE_CLOSED;
	//device = NULL;
	
	// Relinquish our handle
	if (handle)
	{
	pthread_mutex_lock(&handlesMutex);
	deviceHandles[handle / 8] &= ~(1<<(handle % 8));
#if DEBUG
	printf("PicoLinuxUsbDevice::Close: Released handle %d\n", handle);
#endif
	handle = 0;
	pthread_mutex_unlock(&handlesMutex);
	}
	pthread_mutex_unlock(&mutex);
}
