/*
 * SPDX license identifier: MPL-2.0
 *
 * Copyright (C) 2011-2015, BMW AG
 *
 * This file is part of GENIVI Project DLT - Diagnostic Log and Trace.
 *
 * This Source Code Form is subject to the terms of the
 * Mozilla Public License (MPL), v. 2.0.
 * If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * For further information see http://www.genivi.org/.
 */

/*!
 * \author Lassi Marttala <lassi.lm.marttala@partner.bmw.de>
 *
 * \copyright Copyright © 2011-2015 BMW AG. \n
 * License MPL-2.0: Mozilla Public License version 2.0 http://mozilla.org/MPL/2.0/.
 *
 * \file dlt-system.h
 */

/*******************************************************************************
**                                                                            **
**  SRC-MODULE: dlt-system.h                                                  **
**                                                                            **
**  TARGET    : linux                                                         **
**                                                                            **
**  PROJECT   : DLT                                                           **
**                                                                            **
**  AUTHOR    : Lassi Marttala <lassi.lm.marttala@partner.bmw.de>             **
**                                                                            **
**  PURPOSE   :                                                               **
**                                                                            **
**  REMARKS   :                                                               **
**                                                                            **
**  PLATFORM DEPENDANT [yes/no]: yes                                          **
**                                                                            **
**  TO BE CHANGED BY USER [yes/no]: no                                        **
**                                                                            **
*******************************************************************************/

/*******************************************************************************
**                      Author Identity                                       **
********************************************************************************
**                                                                            **
** Initials     Name                       Company                            **
** --------     -------------------------  ---------------------------------- **
**  lm          Lassi Marttala             BMW                                **
*******************************************************************************/

#ifndef DLT_SYSTEM_H_
#define DLT_SYSTEM_H_

/* DLT related includes. */
#include "dlt.h"
#include "dlt_common.h"

/* Constants */
#define DEFAULT_CONF_FILE (CONFIGURATION_FILES_DIR "/dlt-system.conf")
#define DLT_SYSTEM_LOG_FILE_MAX 32
#define DLT_SYSTEM_LOG_DIRS_MAX 32
#define DLT_SYSTEM_LOG_PROCESSES_MAX 32

#define DLT_SYSTEM_MODE_OFF 0
#define DLT_SYSTEM_MODE_STARTUP 1
#define DLT_SYSTEM_MODE_REGULAR 2

#define MAX_LINE 1024

#define MAX_THREADS 8

/* Macros */
#define MALLOC_ASSERT(x) if (x == NULL) { \
        fprintf(stderr, "Out of memory\n"); \
        abort(); }

/**
 * Configuration structures.
 * Please see dlt-system.conf for explanation of all the options.
 */

/* Command line options */
typedef struct {
    char *ConfigurationFileName;
    int Daemonize;
} DltSystemCliOptions;

/* Configuration shell options */
typedef struct {
    int Enable;
} ShellOptions;

/* Configuration syslog options */
typedef struct {
    int Enable;
    char *ContextId;
    int Port;
} SyslogOptions;

/* Configuration journal options */
typedef struct {
    int Enable;
    char *ContextId;
    int CurrentBoot;
    int Follow;
    int MapLogLevels;
    int UseOriginalTimestamp;
} JournalOptions;

typedef struct {
    int Enable;
    char *ContextId;
    int TimeStartup;
    int TimeDelay;
    int TimeoutBetweenLogs;
    char *TempDir;

    /* Variable number of file transfer dirs */
    int Count;
    int Compression[DLT_SYSTEM_LOG_DIRS_MAX];
    int CompressionLevel[DLT_SYSTEM_LOG_DIRS_MAX];
    char *Directory[DLT_SYSTEM_LOG_DIRS_MAX];
} FiletransferOptions;

typedef struct {
    int Enable;

    /* Variable number of files to transfer */
    int Count;
    char *ContextId[DLT_SYSTEM_LOG_FILE_MAX];
    char *Filename[DLT_SYSTEM_LOG_FILE_MAX];
    int Mode[DLT_SYSTEM_LOG_FILE_MAX];
    int TimeDelay[DLT_SYSTEM_LOG_FILE_MAX];
} LogFileOptions;

typedef struct {
    int Enable;
    char *ContextId;

    /* Variable number of processes */
    int Count;
    char *Name[DLT_SYSTEM_LOG_PROCESSES_MAX];
    char *Filename[DLT_SYSTEM_LOG_PROCESSES_MAX];
    int Mode[DLT_SYSTEM_LOG_PROCESSES_MAX];
    int TimeDelay[DLT_SYSTEM_LOG_PROCESSES_MAX];
} LogProcessOptions;

typedef struct {
    char *ApplicationId;
    ShellOptions Shell;
    SyslogOptions Syslog;
    JournalOptions Journal;
    FiletransferOptions Filetransfer;
    LogFileOptions LogFile;
    LogProcessOptions LogProcesses;
} DltSystemConfiguration;

typedef struct {
    pthread_t threads[MAX_THREADS];
    int count;
    int shutdown;
} DltSystemThreads;

/**
 * Forward declarations for the whole application
 */

/* In dlt-system-options.c */
int read_command_line(DltSystemCliOptions *options, int argc, char *argv[]);
int read_configuration_file(DltSystemConfiguration *config, char *file_name);

/* In dlt-process-handling.c */
int daemonize();
void start_threads(DltSystemConfiguration *config);
void start_thread(DltSystemConfiguration *conf,
                  void (thread)(void *), const char *nam);
void join_threads();
void dlt_system_signal_handler(int sig);
void register_with_dlt(DltSystemConfiguration *config);

/* Threads */
void init_shell();
void syslog_thread(void *v_conf);
void filetransfer_thread(void *v_conf);
void logfile_thread(void *v_conf);
void logprocess_thread(void *v_conf);

#if defined(DLT_SYSTEMD_WATCHDOG_ENABLE)
void watchdog_thread(void *v_conf);
#endif

#if defined(DLT_SYSTEMD_JOURNAL_ENABLE)
void journal_thread(void *v_conf);
#endif

#endif /* DLT_SYSTEM_H_ */
