/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "cdw_task.h"
#include "cdw_disc.h"
#include "cdw_widgets.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_erase_disc.h"
#include "cdw_main_window.h"
#include "cdw_utils.h"
#include "cdw_erase_wizard.h"
#include "cdw_drive.h"
#include "cdw_cdrecord.h"
#include "cdw_dvd_rw_format.h"
#include "cdw_xorriso.h"
#include "cdw_ext_tools.h"
#include "cdw_processwin.h"
#include "cdw_read_disc_info.h"



static bool     cdw_erase_disc_is_proper(cdw_disc_t *disc);
static cdw_rv_t cdw_erase_disc_dispatcher(cdw_task_t *task, cdw_disc_t *disc);
static cdw_rv_t cdw_erase_disc_perform_erase(cdw_task_t *task, cdw_disc_t *disc);


/**
   \brief Check if blanking is possible, show blanking wizard and run the process

   Check if disc is ready for blanking, do other necessary checks, display
   blanking wizard and then call blank_dispatcher() function that will take
   care of blanking.

   \return CDW_OK if command finished successfully
   \return CDW_NO if some preconditions were not met
   \return CDW_CANCEL if user cancelled blanking
   \return CDW_ERROR if some problems occurred
*/
cdw_rv_t cdw_erase_disc(void)
{
	/* FIXME: if disc is erasable, but due to some problems with disc
	   file system cdio module can't read full meta information from
	   the disc, cdw will be unable to erase the disc: no meta info
	   from "broken" disc = no possibility to erase and restore it;
	   implement getting some basic information needed by erase module:
	   knowing disc type + simple disc type (+ if disc is not empty)
	   should be enough */


	cdw_disc_t *disc = cdw_disc_new();
	cdw_rv_t crv = cdw_disc_get(disc);
	if (crv != CDW_OK) {
		cdw_vdm ("WARNING: cdw_disc_get() returns %s\n", cdw_utils_get_crv_label(crv));
		cdw_disc_delete(&disc);
		return crv;
	}

	cdw_main_window_disc_info_view_display_data(disc);

	/* check if the disc in drive is of proper type */
	if (!cdw_erase_disc_is_proper(disc)) {
		cdw_disc_delete(&disc);
		return CDW_NO;
	}

	cdw_task_t *task = cdw_task_new(CDW_TASK_ERASE_DISC, disc);
	if (task == (cdw_task_t *) NULL) {
		cdw_vdm ("ERROR: failed to create a task\n");
		cdw_disc_delete(&disc);
		return CDW_ERROR;
	}

	/* the function will set task->blanking.dvd_rw_format_mode for DVD-RW */
	cdw_rv_t decision = cdw_erase_wizard(task, disc);

	cdw_rv_t retval = CDW_OK;
	if (decision == CDW_OK) {
		retval = cdw_erase_disc_perform_erase(task, disc);
	} else if (decision == CDW_CANCEL) { /* selection of blanking method was cancelled by user */
		retval = CDW_CANCEL;
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window, user
				      cancelled erasing media by pressing Escape key */
				   _("Some error occurred when attempting to erase disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		retval = CDW_ERROR;
	}
	if (retval == CDW_OK) {
		cdw_task_save_options(task);
	}
	cdw_task_delete(&task);
	cdw_disc_delete(&disc);
	/* 2TRANS: this is title of dialog window, displaying
	   messages from program blanking disc */
	after_event(_("\"Erase\" log"), 1);

	return retval;
}





cdw_rv_t cdw_erase_disc_perform_erase(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_main_window_wrefresh();

	if (task->erase.tool.id == CDW_TOOL_DVD_RW_FORMAT) {
		if (disc->type == CDW_DVD_RW
		    || disc->type == CDW_DVD_RW_SEQ
		    || disc->type == CDW_DVD_RW_RES) {

			cdw_assert (task->erase.dvd_rw_format_mode != CDW_ERASE_DVD_RW_FORMAT_NONE,
				    "ERROR: correct format mode is not set!");
		}
	}

	/* 2TRANS: this is title of process window;
	   "disc" is "optical disc" */
	cdw_processwin_create(_("Erase disc"),
			      /* 2TRANS: this is message in dialog window;
				 "disc" is "optical disc" */
			      _("Erasing disc"), false);

	cdw_rv_t command_return = cdw_erase_disc_dispatcher(task, disc);

	/* this function sets task.success according to task->tool_status,
	   and resets task->tool_status */
	cdw_rv_t tool_status = cdw_task_check_tool_status(task);
	const char *drive = cdw_drive_get_drive_fullpath();
	cdw_drive_reload_tray_with_ui_update(drive, true, task->erase.eject);

	/* after erasing disc kernel may print to console information
	   about problems with reading data from empty disc, and this
	   information will disrupt cdw ui (main window + processwin);
	   reload_tray_with_ui_update() refreshes only disc info area,
	   let's refresh main window + process window */
	cdw_main_window_wrefresh();
	cdw_processwin_force_refresh();

	if (command_return == CDW_OK && tool_status == CDW_OK) {
		/* TODO: find a way to display time of blanking */
		/* 2TRANS: this is message in dialog window:
		   disc is now erased */
		cdw_processwin_destroy(_("Erasing of disc finished successfully"), true);
		return CDW_OK;
	} else {
		/* 2TRANS: this is message in dialog window:
		   cdw called cdrecord to perform erasing, but
		   most probably it failed and disc was not erased. */
		cdw_processwin_destroy(_("Erasing of disc probably failed. Please check your disc."), true);
		return CDW_ERROR;
	}
}





/**
   \brief Check which method of blanking should be used, call appropriate blanking function

   This function checks task->tool to decide which tool should handle erasing,
   and then calls appropriate blanking function.

   \param task - variable describing current task

   \return CDW_ERROR if current disc type is not erasable by cdw
   \return other, blanking-function-specific values
*/
cdw_rv_t cdw_erase_disc_dispatcher(cdw_task_t *task, cdw_disc_t *disc)
{
	if (disc->type != CDW_DVD_RWP) {
		if (task->erase.erase_mode == CDW_ERASE_MODE_FAST) {
			/* 2TRANS: this is message in process window,
			   it says about fast mode of blanking */
			cdw_processwin_display_main_info(_("Erasing in \"fast\" mode"));
		} else {
			/* 2TRANS: this is message in process window -
			   erasing a disc will be a time-consuming task */
			cdw_processwin_display_main_info(_("Erasing in \"all\" mode (long time)"));
		}
	} else { /* no fast / all parameter for erasing DVD+RW */
		/* 2TRANS: this is message in process window;
		   "disc" is optical disc */
		cdw_processwin_display_main_info(_("Erasing of disc in progress"));
	}

	/* at this point we only have to check which tool was selected by
	   external_tools_select_tool_for_task() to do the job; the function
	   made the selection taking disc type into consideration */
	if (task->erase.tool.id == CDW_TOOL_CDRECORD) {

		cdw_sdm ("INFO: dispatching to cdrecord\n");
		/* 2TRANS: this is message in progress window, displayed when
		   cdrecord blanks disc - no information about progress of
		   erasing is provided */
		cdw_processwin_display_sub_info(_("No progress info will be displayed"));
		return cdw_cdrecord_run_task(task, disc);

	} else if (task->erase.tool.id == CDW_TOOL_DVD_RW_FORMAT) {

		cdw_sdm ("INFO: dispatching to run_command_dvd_rw_format_blank()\n");
		cdw_processwin_add_progress_bar();
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_sub_info(_("Erasing with dvd+rw-format"));
		return cdw_dvd_rw_format_run_task(task, disc);
	} else if (task->erase.tool.id == CDW_TOOL_XORRISO) {

		cdw_sdm ("INFO: dispatching to xorriso\n");
		cdw_processwin_add_progress_bar();
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_sub_info(_("Erasing with xorriso"));
		return cdw_xorriso_run_task(task, disc);
#if 0
	} else if (task->erase.tool.id == CDW_TOOL_GROWISOFS) {

		cdw_sdm ("INFO: dispatching to run_command_growisofs_blank_dvd()\n");
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_sub_info(_("Erasing with growisofs"));
		return run_command_growisofs_blank_dvd(task);
#endif
	} else {
		cdw_assert (0, "ERROR: incorrect blanking tool id: %lld\n", task->erase.tool.id);
		return CDW_ERROR;
	}
}





bool cdw_erase_disc_is_proper(cdw_disc_t *disc)
{
	if (disc->type == CDW_DISC_TYPE_UNKNOWN) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      disc is not recognized as supported by
				      currently selected tools */
				   _("Can't recognize disc in drive. Try changing tools family in Configuration -> Tools or use different disc type."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return false;
	}

	if (disc->type_erasable == CDW_TRUE) {
		return true;

	} else if (disc->type_erasable == CDW_FALSE) {

		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message dialog window, user
				      tries to erase disc that is not erasable */
				   _("Disc in drive is not erasable. Cannot erase it."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return false;
	} else { /* CDW_DISC_ERASABLE_UNKNOWN || CDW_DISC_ERASABLE_ERROR */
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Warning"),
				   /* 2TRANS: this is message dialog window, user
				      tries to erase disc that may be not erasable */
				   _("Type of disc in drive is unknown. Cannot erase it."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return true;
	}
}





void cdw_erase_init(cdw_erase_t *erase)
{
	erase->tool.id    = CDW_TOOL_NONE;
	erase->tool.label = (char *) NULL;

	return;
}





void cdw_erase_set_defaults(cdw_erase_t *erase)
{
	cdw_assert (erase, "ERROR: erase argument is NULL\n");

	erase->erase_mode = CDW_ERASE_MODE_ALL;
	erase->speed_id = 0;
	erase->eject = false;  /* Keep consistent with cdw_write_t->eject. */
	erase->dvd_rw_format_mode = CDW_ERASE_DVD_RW_FORMAT_NONE; /* it can stay as "NONE" when we use cdrecord for erasing */

	erase->erase_time = 0;  /* Not a default value, more like an initialization. */

	return;
}




void cdw_erase_copy(cdw_erase_t *dest, cdw_erase_t *src)
{
	cdw_assert (dest, "ERROR: dest argument is NULL\n");
	cdw_assert (src, "ERROR: src argument is NULL\n");

	dest->erase_mode         = src->erase_mode;
	dest->speed_id           = src->speed_id;
	dest->eject              = src->eject;
	dest->dvd_rw_format_mode = src->dvd_rw_format_mode;

	dest->tool.id    = src->tool.id;
	dest->tool.label = src->tool.label;

	dest->erase_time = src->erase_time;

	return;

}





void cdw_erase_debug_print_options(cdw_erase_t *erase)
{
	cdw_assert (erase, "ERROR: erase argument is NULL\n");

	cdw_vdm ("INFO: Erasing options:\n");

	cdw_vdm ("INFO:                  erase mode = %s\n",   erase->erase_mode == CDW_ERASE_MODE_FAST ? "fast" : "all");
	cdw_vdm ("INFO:                    speed id = %lld\n", erase->speed_id);
	cdw_vdm ("INFO:                       eject = %s\n",   erase->eject ? "true" : "false");

	cdw_vdm ("INFO:          DVD RW format mode = %s\n",
		 erase->dvd_rw_format_mode == CDW_ERASE_DVD_RW_FORMAT_NONE ? "none" :
		 erase->dvd_rw_format_mode == CDW_ERASE_DVD_RW_FORMAT_SEQ ? "seq" : "res");

	cdw_vdm ("INFO:                     tool id = %lld / %s\n",  erase->tool.id, cdw_ext_tools_get_tool_name(erase->tool.id));
	cdw_vdm ("INFO:                  tool label = %s\n",   erase->tool.label);

	cdw_vdm ("INFO:                  erase time = %zd\n",  erase->erase_time);

	return;
}
