"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerMetricsRoute = registerMetricsRoute;
var _configSchema = require("@osd/config-schema");
var _metrics = require("../../../common/constants/metrics");
var _shared = require("../../../common/constants/shared");
var _metrics_analytics_adaptor = require("../../adaptors/metrics/metrics_analytics_adaptor");
var _metrics_helper = require("../../common/metrics/metrics_helper");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerMetricsRoute(router, dataSourceEnabled) {
  const metricsAnalyticsBackend = new _metrics_analytics_adaptor.MetricsAnalyticsAdaptor();
  router.get({
    path: `${_shared.OBSERVABILITY_BASE}/stats`,
    validate: false
  }, async (context, request, response) => {
    try {
      const metrics = (0, _metrics_helper.getMetrics)();
      return response.ok({
        body: metrics
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.OBSERVABILITY_BASE}/stats`,
    validate: {
      body: _configSchema.schema.object({
        element: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const {
        element
      } = request.body;
      (0, _metrics_helper.addClickToMetric)(element);
      return response.ok();
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_shared.OBSERVABILITY_BASE}/search/indices/{dataSourceMDSId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      format: 'json',
      index: _metrics.DATA_PREPPER_INDEX_NAME
    };
    try {
      let resp;
      const dataSourceMDSId = request.params.dataSourceMDSId;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('cat.indices', params);
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('cat.indices', params);
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_shared.OBSERVABILITY_BASE}/metrics/otel/{index}/documentNames/{dataSourceMDSId?}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const dataSourceMDSId = request.params.dataSourceMDSId;
    let opensearchNotebooksClient;
    if (dataSourceEnabled && dataSourceMDSId) {
      opensearchNotebooksClient = context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
    } else {
      opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    }
    try {
      const resp = await metricsAnalyticsBackend.queryToFetchDocumentNames(opensearchNotebooksClient, request.params.index, dataSourceMDSId);
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_shared.OBSERVABILITY_BASE}/metrics/otel/{index}/{histogramSampleDocument}/{dataSourceMDSId?}`,
    validate: {
      params: _configSchema.schema.object({
        histogramSampleDocument: _configSchema.schema.string(),
        index: _configSchema.schema.string(),
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const dataSourceMDSId = request.params.dataSourceMDSId;
    let opensearchNotebooksClient;
    if (dataSourceEnabled && dataSourceMDSId) {
      opensearchNotebooksClient = context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
    } else {
      opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    }
    try {
      const resp = await metricsAnalyticsBackend.queryToFetchSampleDocument(opensearchNotebooksClient, request.params.histogramSampleDocument, request.params.index, dataSourceMDSId);
      return response.ok({
        body: resp.hits
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.OBSERVABILITY_BASE}/metrics/otel/aggregatedBinCount`,
    validate: {
      body: _configSchema.schema.object({
        min: _configSchema.schema.string(),
        max: _configSchema.schema.string(),
        startTime: _configSchema.schema.string(),
        endTime: _configSchema.schema.string(),
        documentName: _configSchema.schema.string(),
        index: _configSchema.schema.string(),
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const dataSourceMDSId = request.body.dataSourceMDSId;
    let opensearchNotebooksClient;
    if (dataSourceEnabled && dataSourceMDSId) {
      opensearchNotebooksClient = context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
    } else {
      opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    }
    try {
      const resp = await metricsAnalyticsBackend.queryToFetchBinCount(opensearchNotebooksClient, request.body.min, request.body.max, request.body.startTime, request.body.endTime, request.body.documentName, request.body.index, dataSourceMDSId);
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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