import {Range, TypeScriptCustomCommandArguments} from "tsc-ide-plugin/protocol"
import {LspProcessingContext, tryHandleCustomTsServerCommandLsp} from "tsc-ide-plugin/ide-commands"
import {languageFeatureWorker} from '@volar/language-service/lib/utils/featureWorkers';
import {ServiceContext} from "@vue/language-server";
import * as ts from "tsc-ide-plugin/tsserverlibrary.shim"
import {LanguageService} from "@volar/language-service";
import * as vscode from "vscode-languageserver"

export async function handleCustomTsServerCommand(ts: typeof import("typescript/lib/tsserverlibrary"),
                                                  getService: (uri: string) => Promise<LanguageService | undefined>,
                                                  commandName: string,
                                                  requestArguments: TypeScriptCustomCommandArguments,
                                                  cancellationToken: vscode.CancellationToken): Promise<ts.server.HandlerResponse | undefined> {
  const tsCancellationToken = {
    isCancellationRequested() {
      // TODO - this cancellation token apparently doesn't work,
      //        most likely because the cancel notification needs to be
      //        processed, but TypeScript language service is non-async,
      //        and does not give a chance to do any processing.
      return cancellationToken.isCancellationRequested
    },
    throwIfCancellationRequested() {
      if (this.isCancellationRequested()) {
        throw new ts.OperationCanceledException()
      }
    }
  }
  return tryHandleCustomTsServerCommandLsp(ts, commandName, requestArguments, {
    cancellationToken: tsCancellationToken,
    async process<T>(fileUri: string, range: Range | undefined, processor: (context: LspProcessingContext) => T): Promise<T | undefined> {
      const languageService = await getService(fileUri);
      if (!languageService) return undefined;
      return callInLanguageFeatureWorker(languageService.context, fileUri, range, tsCancellationToken, (requestContext) => {
        return processor(requestContext);
      })
    }
  });
}

function callInLanguageFeatureWorker<T>(
  context: ServiceContext<import('volar-service-typescript').Provide>,
  uri: string,
  range: Range | undefined,
  cancellationToken: ts.CancellationToken,
  callable: (requestContext: LspProcessingContext) => T
): Promise<NonNullable<Awaited<T | undefined>> | undefined> {
  return languageFeatureWorker(
    context,
    uri,
    range,
    (range, sourceMap) => range ? sourceMap.toGeneratedRanges(range, data => true) : [range],
    (service, document, range, sourceMap) => {
      if (document.languageId !== 'typescript') {
        return;
      }
      if (document.uri.includes("format.ts")) {
        return;
      }
      if (context.services["typescript"] != service) {
        return; // we don't really need full iteration
      }

      let languageService = context.inject("typescript/languageService") as ts.LanguageService;
      let program = languageService.getProgram();
      if (!program) return undefined

      const sourceFile = context.inject("typescript/sourceFile", document!!)
      if (!sourceFile) return undefined

      const reverseMapper = (targetFile: typeof sourceFile, generatedRange: Range) => {
        // sourceMap is only available for SFC sources, i.e., not for TS, since such a map would be pointless.
        if (sourceMap && sourceFile == targetFile) {
          const sourceRange = sourceMap.toSourceRange(generatedRange);
          if (sourceRange) {
            const sourceDocument = sourceMap.sourceFileDocument;
            const targetName = targetFile.fileName;
            return {
              pos: sourceDocument.offsetAt(sourceRange.start),
              end: sourceDocument.offsetAt(sourceRange.end),
              fileName: targetName.endsWith(".vue.ts") ? targetName.substring(0, targetName.length - ".ts".length) : targetName
            }
          }
        }
        return undefined;
      }

      return callable({languageService, program, sourceFile, range, cancellationToken, reverseMapper: reverseMapper});
    },
    item => item,
    items => items[0],
  );
}