//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// G4NucleiPropertiesTableAME12 class implementation
//
// Data updated to AME2012
//   "The Ame2012 atomic mass evaluation (I)"
//    by G.Audi, M.Wang, A.H.Wapstra, F.G.Kondev,
//       M.MacCormick, X.Xu, and B.Pfeiffer
//    Chinese Physics C36 p. 1287-1602, December 2012.
//   "The Ame2012 atomic mass evaluation (II)"
//    by M.Wang, G.Audi, A.H.Wapstra, F.G.Kondev,
//       M.MacCormick, X.Xu, and B.Pfeiffer
//    Chinese Physics C36 p. 1603-2014, December 2012.
//
// Author: Tatsumi Koi, SLAC - August 2016
// --------------------------------------------------------------------

#include "G4NucleiPropertiesTableAME12.hh"

#include "G4PhysicalConstants.hh"
#include "G4SystemOfUnits.hh"
#include "G4ios.hh"

G4ThreadLocal G4bool G4NucleiPropertiesTableAME12::isIntialized = false;
G4ThreadLocal G4double G4NucleiPropertiesTableAME12::electronMass[ZMax];

// Determine the table index for a Nuclide with Z protons and A nucleons.
//
G4int G4NucleiPropertiesTableAME12::GetIndex(G4int Z, G4int A)
{
  if (A > G4NucleiPropertiesTableAME12::MaxA) {
    G4Exception("G4NucleiPropertiesTableAME12::GetIndex", "PART201", EventMustBeAborted,
                "Nucleon number larger than 293");
    return -1;
  }
  if (A < 1) {
    G4Exception("G4NucleiPropertiesTableAME12::GetIndex", "Illegal arguemntPART201",
                EventMustBeAborted, " Nucleon number is negative");
    return -1;
  }
  if (Z > A) {
    G4Exception("G4NucleiPropertiesTableAME12::GetIndex", "PART201", EventMustBeAborted,
                "Nucleon number smaller than Z");
    return -1;
  }

  for (G4int i = shortTable[A - 1]; i < shortTable[A]; ++i) {
    if (indexArray[0][i] == Z) return i;
  }
  return -1;
}

G4int G4NucleiPropertiesTableAME12::MinZ(G4int A)
{
  G4int i = shortTable[A - 1];
  return indexArray[0][i];
}

G4int G4NucleiPropertiesTableAME12::MaxZ(G4int A)
{
  G4int i = shortTable[A] - 1;
  return indexArray[0][i];
}

G4double G4NucleiPropertiesTableAME12::GetNuclearMass(G4int Z, G4int A)
{
  if (!isIntialized) {
    // calculate electron mass in orbit with binding energy
    isIntialized = true;
    for (G4int iz = 1; iz < ZMax; iz += 1) {
      electronMass[iz] = iz * electron_mass_c2 - (14.4381 * std::pow(G4double(iz), 2.39)) * eV
                         - (1.55468 * 1e-6 * std::pow(G4double(iz), 5.35)) * eV;
    }
  }

  G4double nuclearMass = GetAtomicMass(Z, A) - electronMass[Z];

  if (nuclearMass < 0.0) nuclearMass = 0.0;

  return nuclearMass;
}

G4double G4NucleiPropertiesTableAME12::GetMassExcess(G4int Z, G4int A)
{
  G4int i = GetIndex(Z, A);
  if (i >= 0) {
    return MassExcess[i] * keV;
  }

  return 0.0;
}

G4double G4NucleiPropertiesTableAME12::GetBindingEnergy(G4int Z, G4int A)
{
  G4int i = GetIndex(Z, A);
  if (i >= 0) {
    return (G4double(A - Z) * MassExcess[0] + G4double(Z) * MassExcess[1] - MassExcess[i]) * keV;
  }

  return 0.0;
}

G4double G4NucleiPropertiesTableAME12::GetBetaDecayEnergy(G4int Z, G4int A)
{
  G4int i = GetIndex(Z, A);
  if (i >= 0) {
    return BetaEnergy[i] * keV;
  }

  return 0.0;
}

G4double G4NucleiPropertiesTableAME12::GetAtomicMass(G4int Z, G4int A)
{
  G4int i = GetIndex(Z, A);
  if (i >= 0) {
    return MassExcess[i] * keV + G4double(A) * amu_c2;
  }

  return 0.0;
}

G4bool G4NucleiPropertiesTableAME12::IsInTable(G4int Z, G4int A)
{
  return (Z <= A && A >= 1 && A <= 273 && Z >= 0 && Z <= 110 && GetIndex(Z, A) >= 0);
}

// clang-format off
//+------------------------------------------------------------+
//| Table of Z (number of protons) and A (number of nucleons)  |
//|        indexArray[0][ ] --> Z                              |
//|        indexArray[1][ ] --> A                              |
//+------------------------------------------------------------

const G4int G4NucleiPropertiesTableAME12::indexArray[2][G4NucleiPropertiesTableAME12::nEntries] = {
  // ---------- Z ----------
  {

   0,   1,   1,   1,   2,   3,   1,   2,   3,   1,
   2,   3,   4,   1,   2,   3,   4,   5,   1,   2,
   3,   4,   5,   2,   3,   4,   5,   6,   2,   3,
   4,   5,   6,   2,   3,   4,   5,   6,   7,   3,
   4,   5,   6,   7,   3,   4,   5,   6,   7,   8,
   3,   4,   5,   6,   7,   8,   4,   5,   6,   7,
   8,   9,   4,   5,   6,   7,   8,   9,   4,   5,
   6,   7,   8,   9,  10,   5,   6,   7,   8,   9,
  10,   5,   6,   7,   8,   9,  10,  11,   5,   6,
   7,   8,   9,  10,  11,  12,   5,   6,   7,   8,
   9,  10,  11,  12,   5,   6,   7,   8,   9,  10,
  11,  12,  13,   6,   7,   8,   9,  10,  11,  12,
  13,  14,   6,   7,   8,   9,  10,  11,  12,  13,
  14,   7,   8,   9,  10,  11,  12,  13,  14,  15,
   7,   8,   9,  10,  11,  12,  13,  14,  15,   8,
   9,  10,  11,  12,  13,  14,  15,  16,   8,   9,
  10,  11,  12,  13,  14,  15,  16,   8,   9,  10,
  11,  12,  13,  14,  15,  16,  17,   9,  10,  11,
  12,  13,  14,  15,  16,  17,   9,  10,  11,  12,
  13,  14,  15,  16,  17,  18,   9,  10,  11,  12,
  13,  14,  15,  16,  17,  18,  10,  11,  12,  13,
  14,  15,  16,  17,  18,  19,  10,  11,  12,  13,
  14,  15,  16,  17,  18,  19,  10,  11,  12,  13,
  14,  15,  16,  17,  18,  19,  20,  11,  12,  13,
  14,  15,  16,  17,  18,  19,  20,  11,  12,  13,
  14,  15,  16,  17,  18,  19,  20,  21,  11,  12,
  13,  14,  15,  16,  17,  18,  19,  20,  21,  12,
  13,  14,  15,  16,  17,  18,  19,  20,  21,  22,
  12,  13,  14,  15,  16,  17,  18,  19,  20,  21,
  22,  12,  13,  14,  15,  16,  17,  18,  19,  20,
  21,  22,  23,  13,  14,  15,  16,  17,  18,  19,
  20,  21,  22,  23,  13,  14,  15,  16,  17,  18,
  19,  20,  21,  22,  23,  24,  13,  14,  15,  16,
  17,  18,  19,  20,  21,  22,  23,  24,  14,  15,
  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,
  14,  15,  16,  17,  18,  19,  20,  21,  22,  23,
  24,  25,  26,  15,  16,  17,  18,  19,  20,  21,
  22,  23,  24,  25,  26,  15,  16,  17,  18,  19,
  20,  21,  22,  23,  24,  25,  26,  27,  16,  17,
  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,
  28,  16,  17,  18,  19,  20,  21,  22,  23,  24,
  25,  26,  27,  28,  17,  18,  19,  20,  21,  22,
  23,  24,  25,  26,  27,  28,  17,  18,  19,  20,
  21,  22,  23,  24,  25,  26,  27,  28,  18,  19,
  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,
  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,
  28,  29,  19,  20,  21,  22,  23,  24,  25,  26,
  27,  28,  29,  30,  19,  20,  21,  22,  23,  24,
  25,  26,  27,  28,  29,  30,  19,  20,  21,  22,
  23,  24,  25,  26,  27,  28,  29,  30,  31,  20,
  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,
  31,  20,  21,  22,  23,  24,  25,  26,  27,  28,
  29,  30,  31,  32,  21,  22,  23,  24,  25,  26,
  27,  28,  29,  30,  31,  32,  21,  22,  23,  24,
  25,  26,  27,  28,  29,  30,  31,  32,  33,  21,
  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,
  32,  33,  22,  23,  24,  25,  26,  27,  28,  29,
  30,  31,  32,  33,  22,  23,  24,  25,  26,  27,
  28,  29,  30,  31,  32,  33,  23,  24,  25,  26,
  27,  28,  29,  30,  31,  32,  33,  34,  23,  24,
  25,  26,  27,  28,  29,  30,  31,  32,  33,  34,
  23,  24,  25,  26,  27,  28,  29,  30,  31,  32,
  33,  34,  24,  25,  26,  27,  28,  29,  30,  31,
  32,  33,  34,  35,  24,  25,  26,  27,  28,  29,
  30,  31,  32,  33,  34,  35,  25,  26,  27,  28,
  29,  30,  31,  32,  33,  34,  35,  36,  25,  26,
  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,
  25,  26,  27,  28,  29,  30,  31,  32,  33,  34,
  35,  36,  37,  26,  27,  28,  29,  30,  31,  32,
  33,  34,  35,  36,  37,  26,  27,  28,  29,  30,
  31,  32,  33,  34,  35,  36,  37,  38,  26,  27,
  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,
  38,  27,  28,  29,  30,  31,  32,  33,  34,  35,
  36,  37,  38,  27,  28,  29,  30,  31,  32,  33,
  34,  35,  36,  37,  38,  39,  28,  29,  30,  31,
  32,  33,  34,  35,  36,  37,  38,  39,  28,  29,
  30,  31,  32,  33,  34,  35,  36,  37,  38,  39,
  40,  28,  29,  30,  31,  32,  33,  34,  35,  36,
  37,  38,  39,  40,  29,  30,  31,  32,  33,  34,
  35,  36,  37,  38,  39,  40,  29,  30,  31,  32,
  33,  34,  35,  36,  37,  38,  39,  40,  41,  29,
  30,  31,  32,  33,  34,  35,  36,  37,  38,  39,
  40,  41,  30,  31,  32,  33,  34,  35,  36,  37,
  38,  39,  40,  41,  42,  30,  31,  32,  33,  34,
  35,  36,  37,  38,  39,  40,  41,  42,  30,  31,
  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,
  42,  43,  31,  32,  33,  34,  35,  36,  37,  38,
  39,  40,  41,  42,  43,  31,  32,  33,  34,  35,
  36,  37,  38,  39,  40,  41,  42,  43,  44,  32,
  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,
  43,  44,  32,  33,  34,  35,  36,  37,  38,  39,
  40,  41,  42,  43,  44,  45,  32,  33,  34,  35,
  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,
  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,
  43,  44,  45,  46,  33,  34,  35,  36,  37,  38,
  39,  40,  41,  42,  43,  44,  45,  46,  34,  35,
  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,
  46,  47,  34,  35,  36,  37,  38,  39,  40,  41,
  42,  43,  44,  45,  46,  47,  34,  35,  36,  37,
  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,
  48,  35,  36,  37,  38,  39,  40,  41,  42,  43,
  44,  45,  46,  47,  48,  35,  36,  37,  38,  39,
  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,
  45,  46,  47,  48,  49,  36,  37,  38,  39,  40,
  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,
  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,
  46,  47,  48,  49,  50,  36,  37,  38,  39,  40,
  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,
  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,
  47,  48,  49,  50,  37,  38,  39,  40,  41,  42,
  43,  44,  45,  46,  47,  48,  49,  50,  51,  38,
  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,
  49,  50,  51,  38,  39,  40,  41,  42,  43,  44,
  45,  46,  47,  48,  49,  50,  51,  52,  38,  39,
  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
  50,  51,  52,  38,  39,  40,  41,  42,  43,  44,
  45,  46,  47,  48,  49,  50,  51,  52,  53,  39,
  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
  50,  51,  52,  53,  39,  40,  41,  42,  43,  44,
  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,
  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
  50,  51,  52,  53,  54,  40,  41,  42,  43,  44,
  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,
  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
  50,  51,  52,  53,  54,  55,  41,  42,  43,  44,
  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,
  55,  41,  42,  43,  44,  45,  46,  47,  48,  49,
  50,  51,  52,  53,  54,  55,  56,  41,  42,  43,
  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,
  54,  55,  56,  42,  43,  44,  45,  46,  47,  48,
  49,  50,  51,  52,  53,  54,  55,  56,  57,  42,
  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,
  53,  54,  55,  56,  57,  43,  44,  45,  46,  47,
  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,
  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,
  53,  54,  55,  56,  57,  58,  43,  44,  45,  46,
  47,  48,  49,  50,  51,  52,  53,  54,  55,  56,
  57,  58,  44,  45,  46,  47,  48,  49,  50,  51,
  52,  53,  54,  55,  56,  57,  58,  59,  44,  45,
  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,
  56,  57,  58,  59,  44,  45,  46,  47,  48,  49,
  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,
  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,
  54,  55,  56,  57,  58,  59,  60,  45,  46,  47,
  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,
  58,  59,  60,  45,  46,  47,  48,  49,  50,  51,
  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,
  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,
  56,  57,  58,  59,  60,  61,  46,  47,  48,  49,
  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,
  60,  61,  62,  47,  48,  49,  50,  51,  52,  53,
  54,  55,  56,  57,  58,  59,  60,  61,  62,  47,
  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,
  58,  59,  60,  61,  62,  63,  48,  49,  50,  51,
  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,
  62,  63,  48,  49,  50,  51,  52,  53,  54,  55,
  56,  57,  58,  59,  60,  61,  62,  63,  48,  49,
  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,
  60,  61,  62,  63,  64,  49,  50,  51,  52,  53,
  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
  64,  49,  50,  51,  52,  53,  54,  55,  56,  57,
  58,  59,  60,  61,  62,  63,  64,  65,  50,  51,
  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,
  62,  63,  64,  65,  50,  51,  52,  53,  54,  55,
  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,
  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,
  60,  61,  62,  63,  64,  65,  66,  51,  52,  53,
  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
  64,  65,  66,  51,  52,  53,  54,  55,  56,  57,
  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,
  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,
  62,  63,  64,  65,  66,  67,  52,  53,  54,  55,
  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,
  66,  67,  68,  52,  53,  54,  55,  56,  57,  58,
  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,
  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,
  63,  64,  65,  66,  67,  68,  69,  53,  54,  55,
  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,
  66,  67,  68,  69,  54,  55,  56,  57,  58,  59,
  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,
  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
  64,  65,  66,  67,  68,  69,  54,  55,  56,  57,
  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,
  68,  69,  70,  55,  56,  57,  58,  59,  60,  61,
  62,  63,  64,  65,  66,  67,  68,  69,  70,  55,
  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,
  66,  67,  68,  69,  70,  71,  55,  56,  57,  58,
  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,
  69,  70,  71,  56,  57,  58,  59,  60,  61,  62,
  63,  64,  65,  66,  67,  68,  69,  70,  71,  56,
  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,
  67,  68,  69,  70,  71,  72,  57,  58,  59,  60,
  61,  62,  63,  64,  65,  66,  67,  68,  69,  70,
  71,  72,  57,  58,  59,  60,  61,  62,  63,  64,
  65,  66,  67,  68,  69,  70,  71,  72,  73,  58,
  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,
  69,  70,  71,  72,  73,  58,  59,  60,  61,  62,
  63,  64,  65,  66,  67,  68,  69,  70,  71,  72,
  73,  74,  59,  60,  61,  62,  63,  64,  65,  66,
  67,  68,  69,  70,  71,  72,  73,  74,  59,  60,
  61,  62,  63,  64,  65,  66,  67,  68,  69,  70,
  71,  72,  73,  74,  75,  60,  61,  62,  63,  64,
  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,
  75,  60,  61,  62,  63,  64,  65,  66,  67,  68,
  69,  70,  71,  72,  73,  74,  75,  76,  61,  62,
  63,  64,  65,  66,  67,  68,  69,  70,  71,  72,
  73,  74,  75,  76,  61,  62,  63,  64,  65,  66,
  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,
  62,  63,  64,  65,  66,  67,  68,  69,  70,  71,
  72,  73,  74,  75,  76,  77,  62,  63,  64,  65,
  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,
  76,  77,  63,  64,  65,  66,  67,  68,  69,  70,
  71,  72,  73,  74,  75,  76,  77,  78,  63,  64,
  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,
  75,  76,  77,  78,  64,  65,  66,  67,  68,  69,
  70,  71,  72,  73,  74,  75,  76,  77,  78,  64,
  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,
  75,  76,  77,  78,  79,  65,  66,  67,  68,  69,
  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,
  75,  76,  77,  78,  79,  80,  66,  67,  68,  69,
  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
  80,  66,  67,  68,  69,  70,  71,  72,  73,  74,
  75,  76,  77,  78,  79,  80,  67,  68,  69,  70,
  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,
  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,
  77,  78,  79,  80,  68,  69,  70,  71,  72,  73,
  74,  75,  76,  77,  78,  79,  80,  81,  68,  69,
  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
  80,  81,  69,  70,  71,  72,  73,  74,  75,  76,
  77,  78,  79,  80,  81,  82,  69,  70,  71,  72,
  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,
  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
  80,  81,  82,  70,  71,  72,  73,  74,  75,  76,
  77,  78,  79,  80,  81,  82,  71,  72,  73,  74,
  75,  76,  77,  78,  79,  80,  81,  82,  71,  72,
  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,
  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,
  81,  82,  83,  71,  72,  73,  74,  75,  76,  77,
  78,  79,  80,  81,  82,  83,  72,  73,  74,  75,
  76,  77,  78,  79,  80,  81,  82,  83,  84,  72,
  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,
  83,  84,  72,  73,  74,  75,  76,  77,  78,  79,
  80,  81,  82,  83,  84,  72,  73,  74,  75,  76,
  77,  78,  79,  80,  81,  82,  83,  84,  73,  74,
  75,  76,  77,  78,  79,  80,  81,  82,  83,  84,
  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,
  83,  84,  85,  73,  74,  75,  76,  77,  78,  79,
  80,  81,  82,  83,  84,  85,  74,  75,  76,  77,
  78,  79,  80,  81,  82,  83,  84,  85,  86,  74,
  75,  76,  77,  78,  79,  80,  81,  82,  83,  84,
  85,  86,  75,  76,  77,  78,  79,  80,  81,  82,
  83,  84,  85,  86,  75,  76,  77,  78,  79,  80,
  81,  82,  83,  84,  85,  86,  75,  76,  77,  78,
  79,  80,  81,  82,  83,  84,  85,  86,  75,  76,
  77,  78,  79,  80,  81,  82,  83,  84,  85,  86,
  76,  77,  78,  79,  80,  81,  82,  83,  84,  85,
  86,  87,  76,  77,  78,  79,  80,  81,  82,  83,
  84,  85,  86,  87,  76,  77,  78,  79,  80,  81,
  82,  83,  84,  85,  86,  87,  88,  76,  77,  78,
  79,  80,  81,  82,  83,  84,  85,  86,  87,  88,
  77,  78,  79,  80,  81,  82,  83,  84,  85,  86,
  87,  88,  77,  78,  79,  80,  81,  82,  83,  84,
  85,  86,  87,  88,  78,  79,  80,  81,  82,  83,
  84,  85,  86,  87,  88,  78,  79,  80,  81,  82,
  83,  84,  85,  86,  87,  88,  89,  79,  80,  81,
  82,  83,  84,  85,  86,  87,  88,  89,  79,  80,
  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,
  79,  80,  81,  82,  83,  84,  85,  86,  87,  88,
  89,  90,  79,  80,  81,  82,  83,  84,  85,  86,
  87,  88,  89,  90,  80,  81,  82,  83,  84,  85,
  86,  87,  88,  89,  90,  80,  81,  82,  83,  84,
  85,  86,  87,  88,  89,  90,  91,  80,  81,  82,
  83,  84,  85,  86,  87,  88,  89,  90,  91,  80,
  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,
  91,  80,  81,  82,  83,  84,  85,  86,  87,  88,
  89,  90,  91,  80,  81,  82,  83,  84,  85,  86,
  87,  88,  89,  90,  91,  81,  82,  83,  84,  85,
  86,  87,  88,  89,  90,  91,  92,  81,  82,  83,
  84,  85,  86,  87,  88,  89,  90,  91,  92,  82,
  83,  84,  85,  86,  87,  88,  89,  90,  91,  92,
  93,  82,  83,  84,  85,  86,  87,  88,  89,  90,
  91,  92,  93,  83,  84,  85,  86,  87,  88,  89,
  90,  91,  92,  93,  83,  84,  85,  86,  87,  88,
  89,  90,  91,  92,  93,  83,  84,  85,  86,  87,
  88,  89,  90,  91,  92,  93,  83,  84,  85,  86,
  87,  88,  89,  90,  91,  92,  93,  84,  85,  86,
  87,  88,  89,  90,  91,  92,  93,  84,  85,  86,
  87,  88,  89,  90,  91,  92,  93,  84,  85,  86,
  87,  88,  89,  90,  91,  92,  93,  85,  86,  87,
  88,  89,  90,  91,  92,  93,  94,  85,  86,  87,
  88,  89,  90,  91,  92,  93,  94,  86,  87,  88,
  89,  90,  91,  92,  93,  94,  95,  86,  87,  88,
  89,  90,  91,  92,  93,  94,  95,  87,  88,  89,
  90,  91,  92,  93,  94,  95,  96,  87,  88,  89,
  90,  91,  92,  93,  94,  95,  96,  88,  89,  90,
  91,  92,  93,  94,  95,  96,  97,  88,  89,  90,
  91,  92,  93,  94,  95,  96,  97,  89,  90,  91,
  92,  93,  94,  95,  96,  97,  89,  90,  91,  92,
  93,  94,  95,  96,  97,  98,  90,  91,  92,  93,
  94,  95,  96,  97,  98,  90,  91,  92,  93,  94,
  95,  96,  97,  98,  99,  91,  92,  93,  94,  95,
  96,  97,  98,  99,  91,  92,  93,  94,  95,  96,
  97,  98,  99, 100,  92,  93,  94,  95,  96,  97,
  98,  99, 100,  92,  93,  94,  95,  96,  97,  98,
  99, 100,  93,  94,  95,  96,  97,  98,  99, 100,
  93,  94,  95,  96,  97,  98,  99, 100, 101,  94,
  95,  96,  97,  98,  99, 100, 101,  94,  95,  96,
  97,  98,  99, 100, 101,  95,  96,  97,  98,  99,
 100, 101, 102,  95,  96,  97,  98,  99, 100, 101,
 102,  96,  97,  98,  99, 100, 101, 102,  96,  97,
  98,  99, 100, 101, 102, 103,  96,  97,  98,  99,
 100, 101, 102, 103,  97,  98,  99, 100, 101, 102,
 103, 104,  97,  98,  99, 100, 101, 102, 103, 104,
  98,  99, 100, 101, 102, 103, 104, 105,  98,  99,
 100, 101, 102, 103, 104, 105,  99, 100, 101, 102,
 103, 104, 105,  99, 100, 101, 102, 103, 104, 105,
 106, 100, 101, 102, 103, 104, 105, 106, 100, 101,
 102, 103, 104, 105, 106, 107, 101, 102, 103, 104,
 105, 106, 107, 101, 102, 103, 104, 105, 106, 107,
 102, 103, 104, 105, 106, 107, 108, 102, 103, 104,
 105, 106, 107, 108, 103, 104, 105, 106, 107, 108,
 109, 103, 104, 105, 106, 107, 108, 109, 104, 105,
 106, 107, 108, 109, 110, 104, 105, 106, 107, 108,
 109, 110, 105, 106, 107, 108, 109, 110, 105, 106,
 107, 108, 109, 110, 106, 107, 108, 109, 110, 106,
 107, 108, 109, 110, 111, 106, 107, 108, 109, 110,
 111, 107, 108, 109, 110, 111, 107, 108, 109, 110,
 111, 108, 109, 110, 111, 112, 108, 109, 110, 111,
 112, 109, 110, 111, 112, 113, 109, 110, 111, 112,
 113, 110, 111, 112, 113, 110, 111, 112, 113, 111,
 112, 113, 111, 112, 113, 112, 113, 112, 113, 114,
 113, 114, 113, 114, 115, 114, 115, 114, 115, 116,
 115, 116, 115, 116, 117, 116, 117, 116, 117, 118,
 117, 118, 118
  },
  // --------- A -----------
  {
   1,   1,   2,   3,   3,   3,   4,   4,   4,   5,
   5,   5,   5,   6,   6,   6,   6,   6,   7,   7,
   7,   7,   7,   8,   8,   8,   8,   8,   9,   9,
   9,   9,   9,  10,  10,  10,  10,  10,  10,  11,
  11,  11,  11,  11,  12,  12,  12,  12,  12,  12,
  13,  13,  13,  13,  13,  13,  14,  14,  14,  14,
  14,  14,  15,  15,  15,  15,  15,  15,  16,  16,
  16,  16,  16,  16,  16,  17,  17,  17,  17,  17,
  17,  18,  18,  18,  18,  18,  18,  18,  19,  19,
  19,  19,  19,  19,  19,  19,  20,  20,  20,  20,
  20,  20,  20,  20,  21,  21,  21,  21,  21,  21,
  21,  21,  21,  22,  22,  22,  22,  22,  22,  22,
  22,  22,  23,  23,  23,  23,  23,  23,  23,  23,
  23,  24,  24,  24,  24,  24,  24,  24,  24,  24,
  25,  25,  25,  25,  25,  25,  25,  25,  25,  26,
  26,  26,  26,  26,  26,  26,  26,  26,  27,  27,
  27,  27,  27,  27,  27,  27,  27,  28,  28,  28,
  28,  28,  28,  28,  28,  28,  28,  29,  29,  29,
  29,  29,  29,  29,  29,  29,  30,  30,  30,  30,
  30,  30,  30,  30,  30,  30,  31,  31,  31,  31,
  31,  31,  31,  31,  31,  31,  32,  32,  32,  32,
  32,  32,  32,  32,  32,  32,  33,  33,  33,  33,
  33,  33,  33,  33,  33,  33,  34,  34,  34,  34,
  34,  34,  34,  34,  34,  34,  34,  35,  35,  35,
  35,  35,  35,  35,  35,  35,  35,  36,  36,  36,
  36,  36,  36,  36,  36,  36,  36,  36,  37,  37,
  37,  37,  37,  37,  37,  37,  37,  37,  37,  38,
  38,  38,  38,  38,  38,  38,  38,  38,  38,  38,
  39,  39,  39,  39,  39,  39,  39,  39,  39,  39,
  39,  40,  40,  40,  40,  40,  40,  40,  40,  40,
  40,  40,  40,  41,  41,  41,  41,  41,  41,  41,
  41,  41,  41,  41,  42,  42,  42,  42,  42,  42,
  42,  42,  42,  42,  42,  42,  43,  43,  43,  43,
  43,  43,  43,  43,  43,  43,  43,  43,  44,  44,
  44,  44,  44,  44,  44,  44,  44,  44,  44,  44,
  45,  45,  45,  45,  45,  45,  45,  45,  45,  45,
  45,  45,  45,  46,  46,  46,  46,  46,  46,  46,
  46,  46,  46,  46,  46,  47,  47,  47,  47,  47,
  47,  47,  47,  47,  47,  47,  47,  47,  48,  48,
  48,  48,  48,  48,  48,  48,  48,  48,  48,  48,
  48,  49,  49,  49,  49,  49,  49,  49,  49,  49,
  49,  49,  49,  49,  50,  50,  50,  50,  50,  50,
  50,  50,  50,  50,  50,  50,  51,  51,  51,  51,
  51,  51,  51,  51,  51,  51,  51,  51,  52,  52,
  52,  52,  52,  52,  52,  52,  52,  52,  52,  52,
  53,  53,  53,  53,  53,  53,  53,  53,  53,  53,
  53,  53,  54,  54,  54,  54,  54,  54,  54,  54,
  54,  54,  54,  54,  55,  55,  55,  55,  55,  55,
  55,  55,  55,  55,  55,  55,  56,  56,  56,  56,
  56,  56,  56,  56,  56,  56,  56,  56,  56,  57,
  57,  57,  57,  57,  57,  57,  57,  57,  57,  57,
  57,  58,  58,  58,  58,  58,  58,  58,  58,  58,
  58,  58,  58,  58,  59,  59,  59,  59,  59,  59,
  59,  59,  59,  59,  59,  59,  60,  60,  60,  60,
  60,  60,  60,  60,  60,  60,  60,  60,  60,  61,
  61,  61,  61,  61,  61,  61,  61,  61,  61,  61,
  61,  61,  62,  62,  62,  62,  62,  62,  62,  62,
  62,  62,  62,  62,  63,  63,  63,  63,  63,  63,
  63,  63,  63,  63,  63,  63,  64,  64,  64,  64,
  64,  64,  64,  64,  64,  64,  64,  64,  65,  65,
  65,  65,  65,  65,  65,  65,  65,  65,  65,  65,
  66,  66,  66,  66,  66,  66,  66,  66,  66,  66,
  66,  66,  67,  67,  67,  67,  67,  67,  67,  67,
  67,  67,  67,  67,  68,  68,  68,  68,  68,  68,
  68,  68,  68,  68,  68,  68,  69,  69,  69,  69,
  69,  69,  69,  69,  69,  69,  69,  69,  70,  70,
  70,  70,  70,  70,  70,  70,  70,  70,  70,  70,
  71,  71,  71,  71,  71,  71,  71,  71,  71,  71,
  71,  71,  71,  72,  72,  72,  72,  72,  72,  72,
  72,  72,  72,  72,  72,  73,  73,  73,  73,  73,
  73,  73,  73,  73,  73,  73,  73,  73,  74,  74,
  74,  74,  74,  74,  74,  74,  74,  74,  74,  74,
  74,  75,  75,  75,  75,  75,  75,  75,  75,  75,
  75,  75,  75,  76,  76,  76,  76,  76,  76,  76,
  76,  76,  76,  76,  76,  76,  77,  77,  77,  77,
  77,  77,  77,  77,  77,  77,  77,  77,  78,  78,
  78,  78,  78,  78,  78,  78,  78,  78,  78,  78,
  78,  79,  79,  79,  79,  79,  79,  79,  79,  79,
  79,  79,  79,  79,  80,  80,  80,  80,  80,  80,
  80,  80,  80,  80,  80,  80,  81,  81,  81,  81,
  81,  81,  81,  81,  81,  81,  81,  81,  81,  82,
  82,  82,  82,  82,  82,  82,  82,  82,  82,  82,
  82,  82,  83,  83,  83,  83,  83,  83,  83,  83,
  83,  83,  83,  83,  83,  84,  84,  84,  84,  84,
  84,  84,  84,  84,  84,  84,  84,  84,  85,  85,
  85,  85,  85,  85,  85,  85,  85,  85,  85,  85,
  85,  85,  86,  86,  86,  86,  86,  86,  86,  86,
  86,  86,  86,  86,  86,  87,  87,  87,  87,  87,
  87,  87,  87,  87,  87,  87,  87,  87,  87,  88,
  88,  88,  88,  88,  88,  88,  88,  88,  88,  88,
  88,  88,  89,  89,  89,  89,  89,  89,  89,  89,
  89,  89,  89,  89,  89,  89,  90,  90,  90,  90,
  90,  90,  90,  90,  90,  90,  90,  90,  90,  90,
  91,  91,  91,  91,  91,  91,  91,  91,  91,  91,
  91,  91,  91,  91,  92,  92,  92,  92,  92,  92,
  92,  92,  92,  92,  92,  92,  92,  92,  93,  93,
  93,  93,  93,  93,  93,  93,  93,  93,  93,  93,
  93,  93,  94,  94,  94,  94,  94,  94,  94,  94,
  94,  94,  94,  94,  94,  94,  95,  95,  95,  95,
  95,  95,  95,  95,  95,  95,  95,  95,  95,  95,
  95,  96,  96,  96,  96,  96,  96,  96,  96,  96,
  96,  96,  96,  96,  96,  97,  97,  97,  97,  97,
  97,  97,  97,  97,  97,  97,  97,  97,  97,  97,
  98,  98,  98,  98,  98,  98,  98,  98,  98,  98,
  98,  98,  98,  98,  98,  99,  99,  99,  99,  99,
  99,  99,  99,  99,  99,  99,  99,  99,  99,  99,
 100, 100, 100, 100, 100, 100, 100, 100, 100, 100,
 100, 100, 100, 100, 100, 101, 101, 101, 101, 101,
 101, 101, 101, 101, 101, 101, 101, 101, 101, 101,
 102, 102, 102, 102, 102, 102, 102, 102, 102, 102,
 102, 102, 102, 102, 103, 103, 103, 103, 103, 103,
 103, 103, 103, 103, 103, 103, 103, 103, 103, 104,
 104, 104, 104, 104, 104, 104, 104, 104, 104, 104,
 104, 104, 104, 105, 105, 105, 105, 105, 105, 105,
 105, 105, 105, 105, 105, 105, 105, 105, 106, 106,
 106, 106, 106, 106, 106, 106, 106, 106, 106, 106,
 106, 106, 106, 107, 107, 107, 107, 107, 107, 107,
 107, 107, 107, 107, 107, 107, 107, 107, 107, 108,
 108, 108, 108, 108, 108, 108, 108, 108, 108, 108,
 108, 108, 108, 108, 109, 109, 109, 109, 109, 109,
 109, 109, 109, 109, 109, 109, 109, 109, 109, 109,
 110, 110, 110, 110, 110, 110, 110, 110, 110, 110,
 110, 110, 110, 110, 110, 111, 111, 111, 111, 111,
 111, 111, 111, 111, 111, 111, 111, 111, 111, 111,
 112, 112, 112, 112, 112, 112, 112, 112, 112, 112,
 112, 112, 112, 112, 112, 112, 113, 113, 113, 113,
 113, 113, 113, 113, 113, 113, 113, 113, 113, 113,
 113, 114, 114, 114, 114, 114, 114, 114, 114, 114,
 114, 114, 114, 114, 114, 114, 114, 115, 115, 115,
 115, 115, 115, 115, 115, 115, 115, 115, 115, 115,
 115, 115, 115, 116, 116, 116, 116, 116, 116, 116,
 116, 116, 116, 116, 116, 116, 116, 116, 116, 117,
 117, 117, 117, 117, 117, 117, 117, 117, 117, 117,
 117, 117, 117, 117, 117, 118, 118, 118, 118, 118,
 118, 118, 118, 118, 118, 118, 118, 118, 118, 118,
 119, 119, 119, 119, 119, 119, 119, 119, 119, 119,
 119, 119, 119, 119, 119, 119, 120, 120, 120, 120,
 120, 120, 120, 120, 120, 120, 120, 120, 120, 120,
 120, 120, 121, 121, 121, 121, 121, 121, 121, 121,
 121, 121, 121, 121, 121, 121, 121, 121, 122, 122,
 122, 122, 122, 122, 122, 122, 122, 122, 122, 122,
 122, 122, 122, 122, 123, 123, 123, 123, 123, 123,
 123, 123, 123, 123, 123, 123, 123, 123, 123, 123,
 124, 124, 124, 124, 124, 124, 124, 124, 124, 124,
 124, 124, 124, 124, 124, 124, 124, 125, 125, 125,
 125, 125, 125, 125, 125, 125, 125, 125, 125, 125,
 125, 125, 125, 126, 126, 126, 126, 126, 126, 126,
 126, 126, 126, 126, 126, 126, 126, 126, 126, 126,
 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
 127, 127, 127, 127, 127, 127, 128, 128, 128, 128,
 128, 128, 128, 128, 128, 128, 128, 128, 128, 128,
 128, 128, 128, 129, 129, 129, 129, 129, 129, 129,
 129, 129, 129, 129, 129, 129, 129, 129, 129, 130,
 130, 130, 130, 130, 130, 130, 130, 130, 130, 130,
 130, 130, 130, 130, 130, 130, 131, 131, 131, 131,
 131, 131, 131, 131, 131, 131, 131, 131, 131, 131,
 131, 131, 132, 132, 132, 132, 132, 132, 132, 132,
 132, 132, 132, 132, 132, 132, 132, 132, 133, 133,
 133, 133, 133, 133, 133, 133, 133, 133, 133, 133,
 133, 133, 133, 133, 133, 134, 134, 134, 134, 134,
 134, 134, 134, 134, 134, 134, 134, 134, 134, 134,
 134, 135, 135, 135, 135, 135, 135, 135, 135, 135,
 135, 135, 135, 135, 135, 135, 135, 135, 136, 136,
 136, 136, 136, 136, 136, 136, 136, 136, 136, 136,
 136, 136, 136, 136, 137, 137, 137, 137, 137, 137,
 137, 137, 137, 137, 137, 137, 137, 137, 137, 137,
 138, 138, 138, 138, 138, 138, 138, 138, 138, 138,
 138, 138, 138, 138, 138, 138, 138, 139, 139, 139,
 139, 139, 139, 139, 139, 139, 139, 139, 139, 139,
 139, 139, 139, 140, 140, 140, 140, 140, 140, 140,
 140, 140, 140, 140, 140, 140, 140, 140, 140, 140,
 141, 141, 141, 141, 141, 141, 141, 141, 141, 141,
 141, 141, 141, 141, 141, 141, 142, 142, 142, 142,
 142, 142, 142, 142, 142, 142, 142, 142, 142, 142,
 142, 142, 142, 143, 143, 143, 143, 143, 143, 143,
 143, 143, 143, 143, 143, 143, 143, 143, 143, 143,
 144, 144, 144, 144, 144, 144, 144, 144, 144, 144,
 144, 144, 144, 144, 144, 144, 144, 145, 145, 145,
 145, 145, 145, 145, 145, 145, 145, 145, 145, 145,
 145, 145, 145, 145, 146, 146, 146, 146, 146, 146,
 146, 146, 146, 146, 146, 146, 146, 146, 146, 146,
 147, 147, 147, 147, 147, 147, 147, 147, 147, 147,
 147, 147, 147, 147, 147, 147, 148, 148, 148, 148,
 148, 148, 148, 148, 148, 148, 148, 148, 148, 148,
 148, 148, 148, 149, 149, 149, 149, 149, 149, 149,
 149, 149, 149, 149, 149, 149, 149, 149, 149, 150,
 150, 150, 150, 150, 150, 150, 150, 150, 150, 150,
 150, 150, 150, 150, 150, 150, 151, 151, 151, 151,
 151, 151, 151, 151, 151, 151, 151, 151, 151, 151,
 151, 151, 151, 152, 152, 152, 152, 152, 152, 152,
 152, 152, 152, 152, 152, 152, 152, 152, 152, 153,
 153, 153, 153, 153, 153, 153, 153, 153, 153, 153,
 153, 153, 153, 153, 153, 153, 154, 154, 154, 154,
 154, 154, 154, 154, 154, 154, 154, 154, 154, 154,
 154, 154, 155, 155, 155, 155, 155, 155, 155, 155,
 155, 155, 155, 155, 155, 155, 155, 155, 155, 156,
 156, 156, 156, 156, 156, 156, 156, 156, 156, 156,
 156, 156, 156, 156, 156, 157, 157, 157, 157, 157,
 157, 157, 157, 157, 157, 157, 157, 157, 157, 157,
 157, 157, 158, 158, 158, 158, 158, 158, 158, 158,
 158, 158, 158, 158, 158, 158, 158, 158, 159, 159,
 159, 159, 159, 159, 159, 159, 159, 159, 159, 159,
 159, 159, 159, 159, 159, 160, 160, 160, 160, 160,
 160, 160, 160, 160, 160, 160, 160, 160, 160, 160,
 160, 161, 161, 161, 161, 161, 161, 161, 161, 161,
 161, 161, 161, 161, 161, 161, 161, 161, 162, 162,
 162, 162, 162, 162, 162, 162, 162, 162, 162, 162,
 162, 162, 162, 162, 163, 163, 163, 163, 163, 163,
 163, 163, 163, 163, 163, 163, 163, 163, 163, 163,
 164, 164, 164, 164, 164, 164, 164, 164, 164, 164,
 164, 164, 164, 164, 164, 164, 165, 165, 165, 165,
 165, 165, 165, 165, 165, 165, 165, 165, 165, 165,
 165, 165, 166, 166, 166, 166, 166, 166, 166, 166,
 166, 166, 166, 166, 166, 166, 166, 166, 167, 167,
 167, 167, 167, 167, 167, 167, 167, 167, 167, 167,
 167, 167, 167, 167, 168, 168, 168, 168, 168, 168,
 168, 168, 168, 168, 168, 168, 168, 168, 168, 169,
 169, 169, 169, 169, 169, 169, 169, 169, 169, 169,
 169, 169, 169, 169, 169, 170, 170, 170, 170, 170,
 170, 170, 170, 170, 170, 170, 170, 170, 170, 170,
 171, 171, 171, 171, 171, 171, 171, 171, 171, 171,
 171, 171, 171, 171, 171, 171, 172, 172, 172, 172,
 172, 172, 172, 172, 172, 172, 172, 172, 172, 172,
 172, 173, 173, 173, 173, 173, 173, 173, 173, 173,
 173, 173, 173, 173, 173, 173, 174, 174, 174, 174,
 174, 174, 174, 174, 174, 174, 174, 174, 174, 174,
 175, 175, 175, 175, 175, 175, 175, 175, 175, 175,
 175, 175, 175, 175, 176, 176, 176, 176, 176, 176,
 176, 176, 176, 176, 176, 176, 176, 176, 177, 177,
 177, 177, 177, 177, 177, 177, 177, 177, 177, 177,
 177, 177, 178, 178, 178, 178, 178, 178, 178, 178,
 178, 178, 178, 178, 178, 178, 179, 179, 179, 179,
 179, 179, 179, 179, 179, 179, 179, 179, 179, 179,
 180, 180, 180, 180, 180, 180, 180, 180, 180, 180,
 180, 180, 180, 181, 181, 181, 181, 181, 181, 181,
 181, 181, 181, 181, 181, 181, 182, 182, 182, 182,
 182, 182, 182, 182, 182, 182, 182, 182, 183, 183,
 183, 183, 183, 183, 183, 183, 183, 183, 183, 183,
 184, 184, 184, 184, 184, 184, 184, 184, 184, 184,
 184, 184, 184, 185, 185, 185, 185, 185, 185, 185,
 185, 185, 185, 185, 185, 185, 186, 186, 186, 186,
 186, 186, 186, 186, 186, 186, 186, 186, 186, 187,
 187, 187, 187, 187, 187, 187, 187, 187, 187, 187,
 187, 187, 188, 188, 188, 188, 188, 188, 188, 188,
 188, 188, 188, 188, 188, 189, 189, 189, 189, 189,
 189, 189, 189, 189, 189, 189, 189, 189, 190, 190,
 190, 190, 190, 190, 190, 190, 190, 190, 190, 190,
 191, 191, 191, 191, 191, 191, 191, 191, 191, 191,
 191, 191, 191, 192, 192, 192, 192, 192, 192, 192,
 192, 192, 192, 192, 192, 192, 193, 193, 193, 193,
 193, 193, 193, 193, 193, 193, 193, 193, 193, 194,
 194, 194, 194, 194, 194, 194, 194, 194, 194, 194,
 194, 194, 195, 195, 195, 195, 195, 195, 195, 195,
 195, 195, 195, 195, 196, 196, 196, 196, 196, 196,
 196, 196, 196, 196, 196, 196, 197, 197, 197, 197,
 197, 197, 197, 197, 197, 197, 197, 197, 198, 198,
 198, 198, 198, 198, 198, 198, 198, 198, 198, 198,
 199, 199, 199, 199, 199, 199, 199, 199, 199, 199,
 199, 199, 200, 200, 200, 200, 200, 200, 200, 200,
 200, 200, 200, 200, 201, 201, 201, 201, 201, 201,
 201, 201, 201, 201, 201, 201, 201, 202, 202, 202,
 202, 202, 202, 202, 202, 202, 202, 202, 202, 202,
 203, 203, 203, 203, 203, 203, 203, 203, 203, 203,
 203, 203, 204, 204, 204, 204, 204, 204, 204, 204,
 204, 204, 204, 204, 205, 205, 205, 205, 205, 205,
 205, 205, 205, 205, 205, 206, 206, 206, 206, 206,
 206, 206, 206, 206, 206, 206, 206, 207, 207, 207,
 207, 207, 207, 207, 207, 207, 207, 207, 208, 208,
 208, 208, 208, 208, 208, 208, 208, 208, 208, 208,
 209, 209, 209, 209, 209, 209, 209, 209, 209, 209,
 209, 209, 210, 210, 210, 210, 210, 210, 210, 210,
 210, 210, 210, 210, 211, 211, 211, 211, 211, 211,
 211, 211, 211, 211, 211, 212, 212, 212, 212, 212,
 212, 212, 212, 212, 212, 212, 212, 213, 213, 213,
 213, 213, 213, 213, 213, 213, 213, 213, 213, 214,
 214, 214, 214, 214, 214, 214, 214, 214, 214, 214,
 214, 215, 215, 215, 215, 215, 215, 215, 215, 215,
 215, 215, 215, 216, 216, 216, 216, 216, 216, 216,
 216, 216, 216, 216, 216, 217, 217, 217, 217, 217,
 217, 217, 217, 217, 217, 217, 217, 218, 218, 218,
 218, 218, 218, 218, 218, 218, 218, 218, 218, 219,
 219, 219, 219, 219, 219, 219, 219, 219, 219, 219,
 219, 220, 220, 220, 220, 220, 220, 220, 220, 220,
 220, 220, 220, 221, 221, 221, 221, 221, 221, 221,
 221, 221, 221, 221, 222, 222, 222, 222, 222, 222,
 222, 222, 222, 222, 222, 223, 223, 223, 223, 223,
 223, 223, 223, 223, 223, 223, 224, 224, 224, 224,
 224, 224, 224, 224, 224, 224, 224, 225, 225, 225,
 225, 225, 225, 225, 225, 225, 225, 226, 226, 226,
 226, 226, 226, 226, 226, 226, 226, 227, 227, 227,
 227, 227, 227, 227, 227, 227, 227, 228, 228, 228,
 228, 228, 228, 228, 228, 228, 228, 229, 229, 229,
 229, 229, 229, 229, 229, 229, 229, 230, 230, 230,
 230, 230, 230, 230, 230, 230, 230, 231, 231, 231,
 231, 231, 231, 231, 231, 231, 231, 232, 232, 232,
 232, 232, 232, 232, 232, 232, 232, 233, 233, 233,
 233, 233, 233, 233, 233, 233, 233, 234, 234, 234,
 234, 234, 234, 234, 234, 234, 234, 235, 235, 235,
 235, 235, 235, 235, 235, 235, 235, 236, 236, 236,
 236, 236, 236, 236, 236, 236, 237, 237, 237, 237,
 237, 237, 237, 237, 237, 237, 238, 238, 238, 238,
 238, 238, 238, 238, 238, 239, 239, 239, 239, 239,
 239, 239, 239, 239, 239, 240, 240, 240, 240, 240,
 240, 240, 240, 240, 241, 241, 241, 241, 241, 241,
 241, 241, 241, 241, 242, 242, 242, 242, 242, 242,
 242, 242, 242, 243, 243, 243, 243, 243, 243, 243,
 243, 243, 244, 244, 244, 244, 244, 244, 244, 244,
 245, 245, 245, 245, 245, 245, 245, 245, 245, 246,
 246, 246, 246, 246, 246, 246, 246, 247, 247, 247,
 247, 247, 247, 247, 247, 248, 248, 248, 248, 248,
 248, 248, 248, 249, 249, 249, 249, 249, 249, 249,
 249, 250, 250, 250, 250, 250, 250, 250, 251, 251,
 251, 251, 251, 251, 251, 251, 252, 252, 252, 252,
 252, 252, 252, 252, 253, 253, 253, 253, 253, 253,
 253, 253, 254, 254, 254, 254, 254, 254, 254, 254,
 255, 255, 255, 255, 255, 255, 255, 255, 256, 256,
 256, 256, 256, 256, 256, 256, 257, 257, 257, 257,
 257, 257, 257, 258, 258, 258, 258, 258, 258, 258,
 258, 259, 259, 259, 259, 259, 259, 259, 260, 260,
 260, 260, 260, 260, 260, 260, 261, 261, 261, 261,
 261, 261, 261, 262, 262, 262, 262, 262, 262, 262,
 263, 263, 263, 263, 263, 263, 263, 264, 264, 264,
 264, 264, 264, 264, 265, 265, 265, 265, 265, 265,
 265, 266, 266, 266, 266, 266, 266, 266, 267, 267,
 267, 267, 267, 267, 267, 268, 268, 268, 268, 268,
 268, 268, 269, 269, 269, 269, 269, 269, 270, 270,
 270, 270, 270, 270, 271, 271, 271, 271, 271, 272,
 272, 272, 272, 272, 272, 273, 273, 273, 273, 273,
 273, 274, 274, 274, 274, 274, 275, 275, 275, 275,
 275, 276, 276, 276, 276, 276, 277, 277, 277, 277,
 277, 278, 278, 278, 278, 278, 279, 279, 279, 279,
 279, 280, 280, 280, 280, 281, 281, 281, 281, 282,
 282, 282, 283, 283, 283, 284, 284, 285, 285, 285,
 286, 286, 287, 287, 287, 288, 288, 289, 289, 289,
 290, 290, 291, 291, 291, 292, 292, 293, 293, 293,
 294, 294, 295
  }
};

//+----------------------+
//| Table of Mass Excess |
//+----------------------+
// unit: keV
const G4double G4NucleiPropertiesTableAME12::MassExcess[G4NucleiPropertiesTableAME12::nEntries] = {

   8071.31714,   7288.97059,  13135.72174,  14949.80611,  14931.21551,  28667      ,  24621.123  ,   2424.91561,  25323.186  ,  32892.440  ,
  11231.233  ,  11678.886  ,  37139      ,  41875.717  ,  17592.095  ,  14086.87893,  18375.034  ,  47320      ,  49135      ,  26073.126  ,
  14907.10520,  15768.999  ,  27676.550  ,  31609.681  ,  20945.804  ,   4941.671  ,  22921.577  ,  35064.269  ,  40935.896  ,  24954.902  ,
  11348.453  ,  12416.488  ,  28910.970  ,  49172.316  ,  33052.624  ,  12607.488  ,  12050.690  ,  15698.755  ,  38800.107  ,  40728.254  ,
  20177.167  ,   8667.883  ,  10650.295  ,  24303.642  ,  48919.571  ,  25077.759  ,  13369.416  ,      0.0    ,  17338.072  ,  31914.696  ,
  58340.888  ,  33659.076  ,  16562.119  ,   3125.00875,   5345.481  ,  23115.439  ,  39954.498  ,  23663.685  ,   3019.89278,   2863.41669,
   8007.457  ,  31964.410  ,  49760      ,  28957.988  ,   9873.144  ,    101.43866,   2855.605  ,  16806.810  ,  57447.132  ,  37112.271  ,
  13694.128  ,   5683.907  ,  -4737.00137,  10680.254  ,  23986.154  ,  43770.816  ,  21030.778  ,   7870.075  ,   -808.76361,   1951.701  ,
  16500.451  ,  51847.133  ,  24918.124  ,  13113.168  ,   -782.81558,    873.113  ,   5317.623  ,  25036.931  ,  58777      ,  32412.612  ,
  15855.521  ,   3332.858  ,  -1487.44434,   1752.054  ,  12929.391  ,  31828.392  ,  67133      ,  37557.610  ,  21765.110  ,   3796.168  ,
    -17.463  ,  -7041.93055,   6850.604  ,  17558.667  ,  75721      ,  45643      ,  25251.164  ,   8061.907  ,    -47.609  ,  -5731.776  ,
  -2184.637  ,  10913.515  ,  26990      ,  53590.244  ,  32038.675  ,   9283.322  ,   2793.377  ,  -8024.714  ,  -5181.518  ,   -399.939  ,
  18201      ,  33338      ,  64171      ,  38322      ,  14620.657  ,   3313.042  ,  -5154.044  ,  -9529.85249,  -5473.264  ,   6748.070  ,
  23697      ,  46938      ,  18500.402  ,   7559.527  ,  -5951.641  ,  -8417.958  , -13933.569  ,    -47.614  ,  10744.353  ,  33320      ,
  55983      ,  27347.719  ,  11363.509  ,  -2059.806  ,  -9357.817  , -13192.771  ,  -8916.165  ,   3827.331  ,  19738      ,  34733.037  ,
  18665.061  ,    479.445  ,  -6860.780  , -16214.546  , -12210.112  ,  -7140.977  ,  10973      ,  27079      ,  44451      ,  24629.633  ,
   7035.824  ,  -5517.676  , -14586.614  , -17196.747  , -12384.389  ,   -722.461  ,  17028      ,  52080      ,  32920.950  ,  11291.569  ,
   -988.315  , -15018.730  , -16850.530  , -21492.79459,  -7147.740  ,   4073.205  ,  27516      ,  39626      ,  18399.801  ,   2679.976  ,
 -10602.829  , -18204.662  , -21895.07867, -16952.451  ,  -3156.408  ,  13767      ,  48112      ,  23039.573  ,   8474.670  ,  -8883.727  ,
 -15872.452  , -24432.961  , -20200.608  , -14059.007  ,   4443      ,  21490      ,  55618      ,  30820.342  ,  12260.935  ,  -3122.337  ,
 -14954.976  , -22949.036  , -24440.54111, -19042.525  ,  -7066.124  ,  11294      ,  36999      ,  18809.658  ,   -828.807  , -11098.528  ,
 -24077.686  , -24304.874  , -26015.53353, -13334.674  ,  -2200.351  ,  21098      ,  45997      ,  23967      ,   4962.204  ,  -8468.233  ,
 -20514.326  , -26337.346  , -26585.85432, -21003.337  ,  -9384.293  ,   7042      ,  52842      ,  31289      ,   8323.347  ,  -3068.901  ,
 -19956.729  , -24548.698  , -29931.693  , -24440.087  , -18378.293  ,  -1220      ,  13851      ,  37840      ,  15639.784  ,   -219.833  ,
 -14360.399  , -24857.791  , -28846.217  , -29013.540  , -23047.410  , -11172.891  ,   4788      ,  45912      ,  20380.157  ,   5950.384  ,
 -12393.323  , -20251.028  , -30664.124  , -29522.017  , -30231.540  , -17417.065  ,  -6451.150  ,  15351      ,  53143      ,  28289      ,
   9809.563  ,  -6594.287  , -18996.105  , -26896.410  , -31761.521  , -30947.648  , -24800.199  , -13136.066  ,   3484      ,  34074      ,
  16209.859  ,  -4170.299  , -14669.556  , -26861.187  , -29798.087  , -34714.820  , -28800.754  , -22058.500  ,  -4546      ,  10666      ,
  42275      ,  20996      ,   2320.352  , -12829.273  , -23162.346  , -29800.192  , -33242.190  , -33807.19022, -27282.702  , -14172.710  ,
   2198      ,  48605      ,  27973      ,   5429.679  ,  -8074.425  , -22837.846  , -27557.813  , -35039.89464, -33535.492  , -34846.386  ,
 -20523.336  ,  -8850.386  ,  11886      ,  33888      ,  12119.668  ,  -4979.767  , -19008.577  , -27307.189  , -33067.505  , -35559.54329,
 -35137.887  , -28642.411  , -15697.537  ,    196      ,  40837      ,  16562      ,   1009.739  , -17637.746  , -24912.990  , -34422.675  ,
 -35022.026  , -38547.243  , -32121.143  , -25104.663  ,  -7620      ,   6241      ,  47944      ,  23101      ,   4679.826  , -12195.459  ,
 -24322.627  , -32009.808  , -36575.389  , -38408.815  , -36188.100  , -29321.084  , -17916.356  ,  -2301      ,  28513      ,  10442      ,
  -9204.234  , -20609.730  , -32673.255  , -35781.492  , -41468.658  , -37815.978  , -37548.570  , -24116.380  , -13637      ,   6660      ,
  37213      ,  15323      ,  -3989.589  , -18359.747  , -29770.796  , -36615.638  , -40812.152  , -41071.177  , -39009.121  , -31880.549  ,
 -19514.799  ,  -5133      ,  13431      ,  22784      ,     37      , -13810.330  , -29730.762  , -35413.924  , -43138.396  , -41760.499  ,
 -44126.996  , -37074.602  , -29473.531  , -12957      ,    587      ,  29240      ,   7405      , -10097      , -25211.011  , -35711.976  ,
 -42343.453  , -44335.630  , -44936.400  , -42005.801  , -34560.887  , -22565.442  ,  -7592      ,   9846      ,  12761      ,  -4061      ,
 -22440      , -32284.477  , -44224.759  , -44503.420  , -48491.734  , -44476.768  , -42821.654  , -29323.431  , -18416      ,    866      ,
  16477      ,  21203      ,   1146      , -16860      , -29611.491  , -41299.895  , -46561.073  , -48562.809  , -47960.953  , -45332.693  ,
 -37636.738  , -24750.727  , -10330      ,   7173      ,   8430      , -12920      , -25727.848  , -39589.224  , -44547.514  , -51430.681  ,
 -49223.856  , -50261.709  , -42627.232  , -34488.631  , -17782      ,  -4900      ,  14475      ,  -5868      , -22516.039  , -36338.529  ,
 -43228.814  , -49731.858  , -52203.685  , -51451.054  , -48243.495  , -40202.330  , -27342.143  , -12938      ,   -969      , -16543      ,
 -34263.009  , -40166.506  , -49468.845  , -51443.612  , -55418.089  , -50706.854  , -48332.293  , -33990      , -23474      ,  -3065      ,
   6791      , -11681      , -28457      , -38107.420  , -46829.991  , -51849.991  , -55285.894  , -54689.045  , -50946.734  , -42658.627  ,
 -29630.824  , -14354      ,  -5002      , -24778      , -33598.989  , -45597.053  , -49892.107  , -56933.697  , -55556.524  , -56253.867  ,
 -48009.320  , -39222.917  , -21741      ,  -7416      ,    708      , -18350      , -29976.509  , -41667.530  , -49143.620  , -55108.643  ,
 -57711.725  , -57480.637  , -54029.258  , -45335.224  , -31635.399  , -14923      ,   7927      , -13898      , -24731      , -39205.322  ,
 -46123.102  , -55281.245  , -56910.845  , -60606.422  , -56039.798  , -53906.909  , -38238      , -25579      ,  -4322      ,  -6874      ,
 -20707      , -33867.607  , -44228.389  , -52524.139  , -57486.131  , -60181.180  , -59344.948  , -56083.219  , -47308.253  , -32546      ,
 -15649      ,  -1919      , -14876      , -31112      , -40319.499  , -51834.726  , -55827.560  , -62154.467  , -59846.558  , -60228.153  ,
 -51667.134  , -42298.153  , -23492      ,  -7713      , -10302      , -25644      , -37832.015  , -47891.490  , -55525.320  , -60664.164  ,
 -62229.119  , -61156.118  , -56357.732  , -47214.955  , -33972      , -16310      ,  -4052      , -22328      , -33241.956  , -46503.876  ,
 -52967.938  , -61412.375  , -61649.720  , -64472.533  , -58344.551  , -54173.732  , -39784      , -27609      ,  -5701      ,    931      ,
 -16348      , -30506.429  , -42455.089  , -51742.122  , -58920.494  , -62897.623  , -64221.322  , -61983.834  , -56348.691  , -47134.542  ,
 -33729      , -17587      , -12566      , -25476      , -40894.961  , -48481      , -58878.048  , -61423.824  , -66745.863  , -62786.966  ,
 -61167.512  , -51986.446  , -41899      , -24582      ,  -5822      , -21993      , -35722.114  , -46887.053  , -55635.621  , -61850.999  ,
 -65512.321  , -65579.298  , -62213.111  , -56547.092  , -46921.216  , -33627      , -16171      , -33459      , -42989.034  , -54969.544  ,
 -59791.888  , -67098.478  , -65424.094  , -66003.789  , -58832.751  , -54315.496  , -39652      , -26929      , -11644      , -27982      ,
 -40967.338  , -51221.492  , -59185.198  , -65125.242  , -67263.457  , -65911.798  , -62657.328  , -56478.216  , -46937.051  , -33161      ,
  -5608      , -24536      , -36750.387  , -50067.840  , -56408.533  , -66006.285  , -66258.070  , -68899.062  , -63724.062  , -61607.033  ,
 -52025.077  , -41368      , -18481      , -33310      , -46068.530  , -55321.775  , -63742.680  , -67318.779  , -67880.067  , -66878.896  ,
 -62658.004  , -56587.225  , -46580.289  , -32928      , -14876      , -28299      , -43825.349  , -51923.721  , -63463.814  , -65567.034  ,
 -70006.846  , -67085.746  , -66978.789  , -58894.519  , -54189.441  , -38441      , -24536      , -39058      , -50169.085  , -59978.648  ,
 -65736.212  , -68417.599  , -69327.799  , -67100.654  , -63112.163  , -56434.706  , -46111.471  , -32435      , -19217      , -36310      ,
 -46919.353  , -59213.860  , -62976.372  , -69564.709  , -68910.131  , -70561.835  , -64341.835  , -61929.891  , -51425.620  , -40948      ,
 -15202      , -31000      , -44369.925  , -55406.228  , -62711.126  , -67328.775  , -70139.097  , -69906.457  , -67893.057  , -63146.507  ,
 -56502.418  , -46327.205  , -32304      , -28103      , -39784      , -54226.061  , -59782.998  , -68145.486  , -68588.256  , -72585.901  ,
 -68229.798  , -67868.180  , -59067.321  , -53940.575  , -38117      , -22617      , -36896      , -50108.152  , -58987.437  , -65593.402  ,
 -69699.335  , -71297.523  , -70952.749  , -68227.388  , -63647.585  , -56551.750  , -46082      , -31950      , -19235      , -32463      ,
 -48456      , -56006.205  , -65756.712  , -68049.617  , -73422.442  , -70860.055  , -72213.202  , -65288.249  , -62331.834  , -51915.985  ,
 -40827      , -29100      , -44246      , -54471.341  , -62558.908  , -68464.580  , -71856.965  , -73034.190  , -72169.481  , -69107.118  ,
 -64323.623  , -57218.694  , -46618.694  , -24098      , -41610      , -50975.985  , -62303.016  , -66296.639  , -73212.889  , -72291.372  ,
 -75251.950  , -70289.069  , -69014.137  , -60479.081  , -54247.638  , -38601      , -36747      , -48512      , -58789.195  , -65992.344  ,
 -71212.862  , -73916.319  , -74599.485  , -73234.805  , -70169.442  , -64830.491  , -57803.436  , -46779      , -34130      , -44497.468  ,
 -57483.234  , -63705.950  , -71862.053  , -72816.961  , -77025.909  , -73452.125  , -74179.577  , -66935.418  , -63173.941  , -52527      ,
 -41302      , -27710      , -41899      , -53432.294  , -62547.678  , -69526.594  , -73636.069  , -75917.425  , -76068.055  , -74442.277  ,
 -70802.985  , -65476.889  , -58356.889  , -47059      , -36431      , -51648.612  , -59223.667  , -69535.305  , -72214.460  , -77759.477  ,
 -75889.014  , -77893.313  , -72175.467  , -70311.480  , -61146.903  , -55517.043  , -31790      , -46199.663  , -57627.954  , -66291.686  ,
 -72533.314  , -76389.007  , -77975.651  , -77694.814  , -75456.663  , -71528.125  , -65711.726  , -58396.391  , -46947      , -25670      ,
 -42607      , -52930.718  , -65415.066  , -70103.089  , -77593.919  , -77497.277  , -80590.318  , -76187.803  , -76010.046  , -68062.962  ,
 -63943      , -52201      , -36738      , -49257.122  , -60976.434  , -69669.322  , -75340.567  , -79013.401  , -79990.031  , -79070.629  ,
 -76797.605  , -72204.524  , -65910.513  , -58410.513  , -46686      , -32410      , -44283      , -58148.427  , -65853.559  , -75947.721  ,
 -77783.084  , -82439.33541, -79758.960  , -80649.555  , -73893.476  , -71420.533  , -61021      , -54502      , -25840      , -40064      ,
 -53123.419  , -63189.143  , -72413.635  , -78575.468  , -81480.330  , -82167.33008, -81103.276  , -77842.123  , -73174.025  , -66279.675  ,
 -57509.752  , -46034      , -34456      , -49760      , -58962.142  , -70503.166  , -75632.251  , -83265.66488, -82747.010  , -84523.199  ,
 -79283.199  , -77968.652  , -69132.879  , -64110.244  , -51297      , -29580      , -44078      , -55617.906  , -66426.124  , -73891.676  ,
 -80709.521  , -84597.790  , -84880.033  , -83018.343  , -79346.533  , -73873.299  , -66884.808  , -57690.044  , -45932      , -40138      ,
 -50720      , -63884.195  , -70715.958  , -79691.285  , -82608.994  , -87921.351  , -84298.751  , -83628.329  , -76175.987  , -72686.543  ,
 -61681.314  , -54399      , -33729      , -46798      , -58992.390  , -68274.262  , -76535.784  , -81712.239  , -86208.750  , -87709.151  ,
 -84876.390  , -80625.395  , -75014.934  , -67394.848  , -58107      , -46030      , -29221      , -41330      , -55800.217  , -64000.297  ,
 -74959.250  , -79364.730  , -85948.917  , -86494.860  , -88773.558  , -82662.222  , -80172.886  , -70724.686  , -64883.792  , -51959      ,
 -36896      , -50338      , -61107.293  , -70973.964  , -77745.126  , -83652.406  , -86351.877  , -87896.158  , -86638.575  , -82208.712  ,
 -75986.253  , -68239.518  , -58796      , -46277      , -30981      , -46724      , -56232.805  , -68769.319  , -74772.524  , -82867.392  ,
 -84817.054  , -88459.630  , -86453.745  , -86807.827  , -78925.692  , -74301.201  , -62999.087  , -55070      , -40716      , -52970.338  ,
 -64135.994  , -72619.953  , -80085.915  , -84227.656  , -87122.683  , -87212.984  , -86806.320  , -83605.357  , -77216.712  , -69011.800  ,
 -59140      , -46267      , -36803      , -47599      , -61347.771  , -68562.789  , -78845.748  , -82352.842  , -87270.901  , -86369.207  ,
 -88412.843  , -84157.095  , -82583.591  , -72907.613  , -66101.047  , -52411      , -30460      , -43771      , -56158.912  , -65893.881  ,
 -75122.251  , -81211.319  , -85661.557  , -86785.057  , -87710.624  , -86020.106  , -83456.497  , -78340.606  , -69964.665  , -59597      ,
 -46631      , -38163      , -53079.678  , -61354.361  , -72929.855  , -78341.593  , -85444.610  , -85606.827  , -88793.627  , -85820.385  ,
 -86079.123  , -79686.469  , -76182.159  , -64511.635  , -55573      , -34055      , -47423.491  , -58519.121  , -68582.214  , -76127.179  ,
 -82948.417  , -85608.088  , -87543.559  , -87218.851  , -86119.303  , -82596.303  , -77805.842  , -70825.842  , -60454      , -47189      ,
 -28448      , -44311      , -54318.333  , -66425.976  , -72300.590  , -81292.514  , -83530.365  , -88114.842  , -86431.125  , -88224.765  ,
 -83175.112  , -81320.975  , -73066.415  , -67636.415  , -53896      , -38759      , -51205.403  , -62511.942  , -70655.702  , -77624.426  ,
 -82331.795  , -85968.970  , -87326.776  , -87621.838  , -85577.994  , -82181.239  , -76712.473  , -69931.122  , -61376      , -47944      ,
 -35052      , -46547      , -59830.146  , -67333.330  , -76381.971  , -79803.334  , -86189.480  , -86019.859  , -89223.846  , -85588.217  ,
 -85227.225  , -78137.959  , -74194.596  , -64312.972  , -57282.972  , -29128      , -42812      , -55561.993  , -65066.993  , -73170.750  ,
 -78887.990  , -83516.403  , -86341.057  , -87954.577  , -87410.826  , -85430.657  , -81334.373  , -75836.455  , -68614      , -60305.625  ,
 -37707      , -52358.366  , -61173.366  , -71593.762  , -76310.599  , -83570.462  , -84570.589  , -89102.900  , -86780.428  , -87930.996  ,
 -82246.534  , -79659.534  , -70693.745  , -64933.745  , -33608      , -47422      , -58457.765  , -67821.446  , -75025.099  , -80966.783  ,
 -84601.889  , -87263.637  , -88028.052  , -87485.032  , -84800.290  , -80651.995  , -74629.515  , -66969.515  , -56178      , -44106      ,
 -54064      , -65730.230  , -71825.182  , -80355.913  , -82507.185  , -88093.734  , -86955.692  , -89394.976  , -85116.322  , -83968.236  ,
 -76182.664  , -71627.041  , -59170.544  , -38610      , -50822      , -61470.962  , -69911.912  , -77343.235  , -82292.947  , -85932.515  ,
 -87850.550  , -88417.759  , -87070.828  , -84333.837  , -79640.571  , -73337.986  , -64016.090  , -52811.510  , -34790      , -46053      ,
 -58908      , -66199.683  , -76140.819  , -79775.384  , -86322.384  , -86361.788  , -89907.404  , -86942.262  , -87132.015  , -80608.012  ,
 -77353.680  , -66473.291  , -58218.688  , -28900      , -42364      , -54269      , -63720.100  , -72557.908  , -78747.902  , -83860.500  ,
 -86863.855  , -88372.517  , -88406.595  , -86990.248  , -83564.248  , -78512.227  , -70653.222  , -60536.329  , -49574      , -37297      ,
 -51353      , -59545.965  , -70762.348  , -75920.745  , -83659.318  , -85031.920  , -89524.361  , -87606.688  , -89252.551  , -84119.941  ,
 -82069.943  , -72445.330  , -65781.671  , -52646.529  , -33200      , -46193      , -56619.931  , -66672.465  , -74280.725  , -80736.351  ,
 -84999.925  , -87606.632  , -88719.883  , -88504.363  , -86487.931  , -82630.914  , -76251.054  , -67715.390  , -57673.183  , -46169.627  ,
 -42886      , -52136      , -64549.112  , -71032.441  , -80070.575  , -82828.978  , -88331.527  , -87457.761  , -90348.765  , -86470.765  ,
 -85841.983  , -77449.733  , -72229.823  , -60464.175  , -51918.586  , -37560      , -48875      , -59937.674  , -69022.535  , -76783.184  ,
 -82304.658  , -85986.510  , -88216.279  , -89253.079  , -88390.901  , -85939.518  , -80836.736  , -73587.477  , -64953.808  , -54392.537  ,
 -33810      , -44274      , -57464      , -65255.055  , -75628.711  , -79732.885  , -86321.830  , -86583.717  , -90575.805  , -87991.071  ,
 -88655.990  , -81598.964  , -77567.508  , -67063.329  , -60026.421  , -46287.280  , -40511      , -52769      , -62811.539  , -71870.352  ,
 -78767.745  , -83591.301  , -87026.789  , -89043.251  , -89365.821  , -88328.240  , -84417.076  , -78347.029  , -71119.507  , -62203.632  ,
 -51763.919  , -35387      , -49807      , -58768      , -70222.022  , -75713.421  , -83491.154  , -84930.799  , -90014.842  , -88568.397  ,
 -90557.336  , -84495.645  , -81888.569  , -72796      , -67085.889  , -54682.260  , -45959.505  , -31354      , -44749      , -55908      ,
 -66303.837  , -74229.276  , -80426.583  , -84982.661  , -88084.389  , -89536.343  , -90033.833  , -87003.403  , -82062.759  , -76337.796  ,
 -68656.747  , -59699      , -49025      , -41500      , -51456      , -64068.907  , -70739.025  , -79831.849  , -82542.653  , -88712.556  ,
 -88249.746  , -91525.995  , -86821.654  , -85268.961  , -77492.236  , -73046.723  , -62063      , -54697      , -40702      , -36170      ,
 -48382      , -59519.672  , -68897.589  , -76424.685  , -82182.050  , -86418.465  , -88943.073  , -90397.782  , -88639.564  , -85095.425  ,
 -80436.079  , -74185.336  , -66493.091  , -57621.687  , -46588      , -43790      , -57259      , -64887.678  , -75388.883  , -79553.791  ,
 -86701.720  , -87228.250  , -91652.886  , -87996.246  , -87696.780  , -80971.048  , -78079.056  , -68409.367  , -62354      , -49621      ,
 -40371      , -52564      , -62822.792  , -71408.122  , -78645.771  , -83977.117  , -87699.500  , -90065.063  , -89474.217  , -87181.217  ,
 -83765.530  , -78794.412  , -72305.051  , -64590.086  , -54967      , -44050      , -35518      , -50012      , -58815      , -70280.208  ,
 -75651.501  , -83957.354  , -85728.591  , -91098.591  , -88417.983  , -89368.186  , -83753.186  , -82172.423  , -73888.638  , -68888.638  ,
 -57687      , -49798      , -45047      , -56430      , -66182.327  , -74402.820  , -81073.825  , -85836.183  , -89197.486  , -89598.575  ,
 -88544.269  , -86251.640  , -82480.985  , -77102.331  , -70744.836  , -62270      , -52769      , -41619      , -42411      , -52173      ,
 -64616.159  , -71106.108  , -80612.373  , -83572.954  , -89941.545  , -88333.624  , -90314.440  , -86080.440  , -85354.976  , -78144.759  ,
 -74608.944  , -64543      , -57874      , -44945      , -37362      , -49509      , -60417      , -69548.078  , -77414.181  , -83430.454  ,
 -87816.391  , -89224.825  , -89172.136  , -87943.711  , -85248.694  , -81043.645  , -75654.952  , -68651      , -60286      , -50338      ,
 -34419      , -45168      , -58554      , -66200.133  , -76701.671  , -80870.217  , -88234.237  , -87621.004  , -90525.302  , -87365.715  ,
 -87661.052  , -81731.334  , -79089.775  , -70258.610  , -64916      , -53151      , -44525      , -42206      , -54222      , -64230.237  ,
 -73348.094  , -80476.840  , -85896.420  , -88256.255  , -89022.955  , -88837.185  , -87193.012  , -84087.840  , -79668.965  , -73759.484  ,
 -66658      , -58032      , -47599      , -37763      , -52015      , -60780      , -72256.802  , -77772.929  , -86015.286  , -86393.286  ,
 -90065.330  , -87911.302  , -89145.566  , -84350.454  , -82669.902  , -74973.468  , -70820.558  , -60324      , -52993      , -39197      ,
 -47441      , -58582      , -68490.514  , -76898.298  , -83471.230  , -86699.477  , -88281.659  , -88983.895  , -88321.546  , -86240.151  ,
 -82817.944  , -77896.108  , -71979.336  , -64543      , -55536      , -44786      , -44870      , -54902      , -67241.890  , -74145.950  ,
 -83362.021  , -84630.302  , -88993.745  , -87738.709  , -89860.278  , -85931.564  , -85378.907  , -78625.430  , -75533.917  , -66330.756  ,
 -60314      , -47786      , -38732      , -52210      , -63509      , -72837.892  , -80607.102  , -84629.337  , -87004.837  , -88507.142  ,
 -88696.05676, -87499.252  , -85063.288  , -81324.613  , -76287.496  , -69773.558  , -62315      , -52881      , -42141      , -45920      ,
 -61533.861  , -69883.861  , -80132.861  , -82286.266  , -87352.947  , -86936.155  , -89880.462  , -86899.743  , -87261.734  , -81627.365  ,
 -79422.905  , -71175.457  , -66596.232  , -55396      , -47506      , -33824      , -55331      , -68025.582  , -77272.128  , -81981.906  ,
 -85211.010  , -87442.784  , -88413.631  , -88058.879  , -86683.922  , -83769.256  , -79708.439  , -74300.656  , -67768.033  , -59923      ,
 -50133      , -39270      , -50263      , -62408.912  , -76543.912  , -79635.573  , -85188.185  , -85703.490  , -89278.96289, -87156.248  ,
 -88434.990  , -83723.632  , -82470.963  , -75213.483  , -71425.807  , -61628      , -55079      , -42225      , -43920      , -57464      ,
 -70874.208  , -78922.507  , -82932.060  , -85886.571  , -87643.571  , -88070.931  , -87553.588  , -85494.383  , -82418.214  , -77937.580  ,
 -72332.372  , -65407.646  , -57231      , -47236      , -36021      , -51661      , -66432.229  , -74020.541  , -82535.996  , -84059.102  ,
 -88124.303  , -86891.154  , -88950.055  , -85218.650  , -84832.889  , -78527.991  , -75646.432  , -66738.751  , -61376      , -49928      ,
 -41302      , -46528      , -60632.243  , -69689.625  , -77727.859  , -83788.959  , -86416.766  , -87581.815  , -87850.713  , -86643.508  ,
 -84616.356  , -80935.860  , -76213.609  , -70026.925  , -62857.215  , -54148      , -44293      , -32825      , -55899      , -64509.798  ,
 -74425.804  , -79545.478  , -86429.152  , -86338.914  , -88887.138  , -86037.478  , -86508.588  , -81340.426  , -79199.286  , -71180.500  ,
 -66810.889  , -56244      , -49090      , -36058      , -49788      , -60030.131  , -69304.222  , -76356.251  , -82383.396  , -86545.824  ,
 -87721.453  , -87140.904  , -85918.804  , -83201.836  , -79584.566  , -74072.847  , -68026.525  , -60119      , -51214      , -40967      ,
 -44861      , -54539      , -65696.198  , -71979.892  , -79972.231  , -82887.048  , -88261.856  , -86521.895  , -87568.736  , -83131.736  ,
 -82018.309  , -74940.483  , -71497.762  , -61749.669  , -55657      , -43668      , -34931      , -49788      , -60205.071  , -68459.027  ,
 -75644.575  , -80701.140  , -84913.969  , -87228.585  , -86950.204  , -84821.133  , -82014.648  , -77500.608  , -72380.218  , -65398.041  ,
 -57632      , -48130      , -37642      , -43939      , -56356.721  , -63595.890  , -72986.451  , -77050.330  , -83270.226  , -84318.245  ,
 -88079.178  , -84691.178  , -84253.772  , -78208.572  , -75455.934  , -66985.934  , -61782.271  , -50482.271  , -42830      , -29259      ,
 -50487      , -59904      , -68197.298  , -74477.145  , -79732.781  , -82934.609  , -85436.002  , -86016.422  , -84193.408  , -80522.920  ,
 -75933.866  , -69925.620  , -63224.224  , -54540.837  , -45382      , -34364      , -46370      , -54769.984  , -65229.639  , -70517.830  ,
 -77842.734  , -80023.821  , -84532.731  , -83788.251  , -85949.886  , -81142.059  , -78986.606  , -71313.606  , -66959.515  , -56559.515  ,
 -50120      , -37250      , -27852      , -40278      , -50627      , -60202.873  , -67673.666  , -73937.063  , -78171.430  , -81606.215  ,
 -83068.031  , -84002.139  , -82960.475  , -79516.703  , -74241.300  , -68231.300  , -60419.183  , -52168.941  , -42048      , -31093      ,
 -45280      , -56872.292  , -63270.507  , -71767.080  , -74849.652  , -80431.769  , -80750.415  , -83747.855  , -81415.932  , -81965.449  ,
 -75619.099  , -71759.503  , -62368.181  , -56570.082  , -44609.513  , -36608      , -22092      , -40939      , -51493.329  , -60055.807  ,
 -67516.176  , -72835.315  , -77066.944  , -79625.824  , -81431.837  , -81267.322  , -80651.229  , -77991.543  , -72924.037  , -66301.929  ,
 -58242.598  , -49120.105  , -39076      , -27583      , -47954.942  , -55568.953  , -64940.303  , -69046.831  , -75635.066  , -76680.726  ,
 -80925.751  , -79454.206  , -80996.206  , -77117.420  , -76085.532  , -67763.409  , -62554.917  , -51238.218  , -44322.012  , -30892      ,
 -42607      , -52018.205  , -60264.029  , -66678.390  , -72013.882  , -75444.011  , -78146.631  , -79041.934  , -79266.008  , -77544.399  ,
 -75356.575  , -70742.509  , -64196.037  , -55757.092  , -46607.806  , -35974.400  , -39002      , -47296.480  , -57593.735  , -62708.735  ,
 -70398.380  , -72535.360  , -77407.809  , -76865.073  , -79336.065  , -76299.378  , -76269.321  , -70531.280  , -67853.216  , -57991.157  ,
 -51478.988  , -38765.027  , -30199      , -43762      , -53021      , -60219.913  , -66669.913  , -71039.365  , -74375.279  , -76063.679  ,
 -77135.097  , -76440.481  , -75126.744  , -71488.858  , -67699.408  , -61662.438  , -53741.615  , -43883      , -33198      , -38815      ,
 -50254      , -56383      , -64846.847  , -68300.398  , -73679.080  , -73596.464  , -77050.464  , -74791.823  , -75763.602  , -71105.649  ,
 -69309.428  , -61945.833  , -57831.276  , -46491      , -38638      , -24640      , -34857      , -45392      , -53729      , -61225.051  ,
 -66779.471  , -70942.314  , -73385.389  , -74575.599  , -74651.975  , -74187.955  , -71622.947  , -68752.115  , -63622.582  , -58266.284  ,
 -50777.717  , -41542.306  , -30108      , -42094      , -49537      , -59057      , -63758.063  , -70148.749  , -71253.533  , -74761.968  ,
 -72887.371  , -74706.290  , -70716.290  , -70117.944  , -63599.035  , -60493.935  , -51770.574  , -46320.681  , -33422      , -36924      ,
 -46239      , -55023      , -61568.457  , -67330.270  , -70647.592  , -72559.051  , -73366.327  , -72881.928  , -71312.942  , -69142.584  ,
 -65012.292  , -60472.180  , -53989.298  , -47208      , -38423.145  , -27302      , -41759      , -52350      , -58194.398  , -65684.398  ,
 -68491.327  , -72454.527  , -71737.183  , -73705.338  , -70155.687  , -70393.674  , -64639.317  , -62604.927  , -54427.163  , -49932.125  ,
 -39715      , -32733      , -38182      , -47925      , -55415.247  , -62283.702  , -66939.908  , -70190.170  , -71817.156  , -72069.250  ,
 -71249.466  , -69154.966  , -66039.153  , -62208.978  , -56625.836  , -50502.687  , -42550.087  , -34363      , -23991      , -44870      ,
 -51568      , -60473.620  , -64163.620  , -69362.706  , -70085.148  , -72534.282  , -70090.097  , -70528.335  , -65478.335  , -64210.384  ,
 -56828.983  , -53258.450  , -43749.924  , -37867.276  , -26054      , -40008      , -48540      , -56461.520  , -62297.021  , -66677.553  ,
 -69458.297  , -70822.822  , -70762.778  , -69423.633  , -66830.570  , -63389.417  , -58736.174  , -53425.513  , -46456.710  , -38903      ,
 -29643.709  , -19710      , -44330      , -54055      , -59089.187  , -65250.220  , -67255.165  , -70688.892  , -69469.940  , -70406.159  ,
 -66186.403  , -65303.808  , -58703.194  , -56010.223  , -47212.300  , -42102.511  , -31167      , -23695      , -41088      , -49807      ,
 -56554.259  , -62207.753  , -66042.861  , -68560.782  , -69531.640  , -69166.263  , -67328.663  , -64560.163  , -60570.397  , -55838.758  ,
 -49708.604  , -42852.758  , -34444.267  , -25487      , -14737      , -47134      , -53002      , -60234.770  , -63480.441  , -67940.926  ,
 -67835.532  , -69671.423  , -66381.423  , -66064.228  , -60302.028  , -58164.901  , -50269.937  , -45931.271  , -35874.153  , -29376.914  ,
 -16931      , -42588      , -50235      , -56671.937  , -61791.501  , -65505.009  , -67460.822  , -68054.493  , -67196.536  , -65200.295  ,
 -61898.708  , -57839.128  , -52562.344  , -46315.400  , -38701.466  , -30556      , -20890.661  , -10220      , -46370      , -54530      ,
 -58694.590  , -64279.590  , -65674.649  , -68180.170  , -66040.806  , -66333.211  , -61476.483  , -59826.502  , -52831.753  , -49169.062  ,
 -39780.305  , -34000.330  , -22501      , -14503      , -43249      , -50720      , -56638.864  , -61313.864  , -64594.769  , -66379.861  ,
 -66377.306  , -65166.577  , -62727.577  , -59299.250  , -54791.409  , -49263.973  , -42534.708  , -34908.351  , -26007.442  , -16385      ,
 -48102      , -53334      , -59774      , -62076.660  , -65966.660  , -64980.767  , -65941.586  , -61903.724  , -61017.575  , -54642.370  ,
 -51818.224  , -43282.800  , -38227.878  , -27522.974  , -20472.643  ,  -7537      , -43808      , -50686      , -56486      , -60566      ,
 -63611.300  , -64898.272  , -64520.401  , -62928.809  , -60295.381  , -56442.241  , -51635.507  , -45847.601  , -38861.195  , -30643.674  ,
 -21795      , -11643      , -46929      , -54530      , -57883.483  , -62583.483  , -63070.595  , -64925.569  , -61887.902  , -61595.275  ,
 -56020.981  , -53858.983  , -46097.775  , -41888.158  , -31893.589  , -25436.794  , -13354      ,  -4792      , -43883      , -50813      ,
 -55927      , -59931.039  , -62281.039  , -63290.716  , -62544.054  , -60590.901  , -57501.125  , -53467.756  , -48351.059  , -42099.281  ,
 -34843      , -26502.151  , -17077.717  ,  -6805      , -48363      , -52723      , -58559.979  , -60060.709  , -62990.709  , -61313.348  ,
 -61581.439  , -57067.388  , -55360.552  , -48393.908  , -44892.853  , -35794.885  , -29986.887  , -18716.842  , -11058.077  , -44153      ,
 -50329      , -55597.587  , -58797.587  , -60922.642  , -61275.641  , -60377.101  , -58084.101  , -54716.889  , -50290.430  , -44917.716  ,
 -38409.165  , -30722.931  , -22077.816  , -12513      ,  -1788      , -46724      , -53663      , -56239.094  , -60109.094  , -59796.295  ,
 -60764.729  , -57307.034  , -56253.854  , -50137.665  , -47290.471  , -38917.753  , -33926.451  , -23360      , -16304.550  ,  -3752      ,
 -44032      , -50189      , -54519.391  , -57719.391  , -59211.468  , -59308.029  , -57829.979  , -55431.345  , -51720.273  , -47086.090  ,
 -41250.280  , -34303.227  , -26416.935  , -17469.939  ,  -7567.651  ,   3288      , -48009      , -51484      , -56484.065  , -57374.888  ,
 -59256.179  , -56738.135  , -56402.225  , -51329.977  , -49097.186  , -41526.314  , -37243.629  , -27379.369  , -21097.480  ,  -9368.702  ,
  -1109.439  , -43939      , -49351      , -53654      , -56254.440  , -57552.229  , -56882.585  , -55411.784  , -52396.538  , -48727.383  ,
 -43553.865  , -37438.173  , -30268.305  , -21943.150  , -12816.496  ,  -2710      , -45690      , -51949      , -53865.547  , -56945.547  ,
 -55572.136  , -55846.659  , -51740.766  , -50227.088  , -43673.096  , -39995.085  , -30868.738  , -25318.446  , -14236      ,  -6646.444  ,
 -43203      , -48652      , -52311.585  , -54696.585  , -55167.553  , -54483.841  , -52408.647  , -49632.795  , -45288.307  , -40105.182  ,
 -33394.442  , -25700.179  , -17415.479  ,  -7972.892  , -46631      , -49369.695  , -53489.695  , -53384.219  , -54578.442  , -51365.374  ,
 -50641.603  , -45062.885  , -42097.940  , -33859.444  , -28933.729  , -18396.604  , -11773.280  ,    581.278  , -42858      , -47469      ,
 -50984.778  , -52385.802  , -52883.041  , -51717.041  , -49701.725  , -46269.170  , -41949.247  , -36047.421  , -29370.436  , -21545.047  ,
 -12782.594  ,  -3324.661  , -44116      , -49693.859  , -50339.782  , -52437.678  , -50601      , -50409.378  , -45653.453  , -43543.966  ,
 -36251.884  , -31997.263  , -22326.122  , -16316.264  ,  -4794      ,   3568.780  , -41601      , -46537      , -49060.918  , -50465.352  ,
 -50359.774  , -49297.416  , -46584.825  , -43019.298  , -38079.308  , -32267.927  , -24988.580  , -16924.007  ,  -8280.333  ,   2050.323  ,
 -44600      , -46678.794  , -49781.794  , -48936.193  , -49638.565  , -45837.359  , -44361.950  , -37977.526  , -34435.511  , -25594.291  ,
 -20250.367  ,  -9262.288  ,  -1929.789  , -41088      , -44797.410  , -47405.277  , -48441.629  , -48253.454  , -46520.978  , -43549.963  ,
 -39471.633  , -34374.375  , -27871.188  , -20661.122  , -12798.747  ,  -3117.421  , -41880      , -46051.959  , -46433.249  , -48247.714  ,
 -45447.714  , -44609.073  , -39051.678  , -36168.213  , -28300.768  , -23576.717  , -13308.106  ,  -6825.554  , -39716.110  , -43286.112  ,
 -45296.112  , -46367.212  , -45811.212  , -43664.042  , -40203.311  , -35772.347  , -30189.129  , -23804.531  , -16587.263  ,  -7571.092  ,
 -36412      , -41501.557  , -42841.557  , -45707.557  , -44224.718  , -44256.643  , -39610.851  , -37338.832  , -30318.710  , -26348.678  ,
 -16873.221  , -11051.523  ,   1187.428  , -33888      , -38319.800  , -41396.446  , -43389.946  , -43822.614  , -42809.843  , -40335.553  ,
 -36688.140  , -31866.811  , -26175.551  , -19757.872  , -11541.207  ,  -2236      , -36424.210  , -38609.805  , -42510.805  , -41930.648  ,
 -43002.379  , -39174.769  , -37864.474  , -31714.852  , -28538.859  , -19886.613  , -14681.900  ,  -3126.391  ,   4092.278  , -32817      ,
 -36900.203  , -39906.282  , -41218.543  , -41221.010  , -39549.372  , -36685.050  , -33027.838  , -28117.679  , -22443.092  , -14986.905  ,
  -6383.356  ,   2832.724  , -30879      , -33612.030  , -38669.876  , -39018.876  , -41139.270  , -38351.382  , -37829.006  , -32276.833  ,
 -30211.250  , -22336.400  , -17814.995  ,  -7184.666  ,   -544.393  , -27162      , -31829      , -35617.536  , -37980.718  , -38988.462  ,
 -38456.703  , -36485.425  , -33581.954  , -29626.234  , -24602.181  , -17878.164  , -10064.807  ,  -1421.991  , -28513      , -34380.894  ,
 -35634.992  , -38709.442  , -36755.621  , -37325.237  , -32883.237  , -31370.170  , -24379      , -20416.486  , -10598.947  ,  -4563.790  ,
 -26492      , -31176.173  , -34352.054  , -36396.860  , -36710.837  , -35701.452  , -33811.138  , -30592.908  , -26282.849  , -20235.776  ,
 -13239.737  ,  -5068.760  ,   3863.753  , -23157      , -29649      , -31588.825  , -35883.945  , -34837.645  , -36292.174  , -32775.833  ,
 -32011.069  , -25872.246  , -22565.434  , -13535.485  ,  -8070.533  ,   2935.796  , -26287      , -30235.364  , -33396.044  , -34538.319  ,
 -34481.690  , -33405.930  , -31062.849  , -27477.212  , -22194.490  , -15872.832  ,  -8359.526  ,    -67.609  ,   9042.924  , -24526      ,
 -27237      , -32437.191  , -32533.791  , -34762.564  , -32213.155  , -32183.944  , -26937.491  , -24207.433  , -16036      , -11004.576  ,
   -711.531  ,   5723.462  , -25579      , -29511.593  , -31694.334  , -32796.302  , -32569.478  , -30999.617  , -28155.088  , -23713.115  ,
 -18025.544  , -11060.995  ,  -3475.827  ,   5050.256  , -22542      , -28278.795  , -29437.900  , -32646.917  , -31139.930  , -31826.765  ,
 -27496.591  , -25360.638  , -18009.031  , -13482.535  ,  -3912.629  ,   1971.040  , -20502      , -25309      , -28265.798  , -30421.954  ,
 -31140.975  , -30541.036  , -28339.875  , -24748.709  , -19687.033  , -13358.069  ,  -6344.231  ,   1476.065  , -17139      , -23837      ,
 -25821      , -29905.689  , -29581.995  , -30954.847  , -27494.847  , -26050.089  , -19369.486  , -15472.924  ,  -6721      ,  -1230.357  ,
 -20484      , -24400.204  , -27390.372  , -29094.952  , -29546.390  , -28059.394  , -25231.676  , -20797.457  , -15213.984  ,  -8823.433  ,
  -1499.839  ,   6761.337  , -18779      , -21611      , -26600.853  , -27240.185  , -29503.591  , -27047.551  , -26250.532  , -20370.675  ,
 -16954.318  ,  -8987.971  ,  -4014.338  ,   6134.822  , -15239      , -19897      , -23740.883  , -26400.883  , -27662.727  , -27178.934  ,
 -25259.399  , -21414.660  , -16524.884  , -10789.358  ,  -4072.482  ,   3601.709  ,  11841      , -13087      , -16776      , -22692.125  ,
 -24352.979  , -27345.496  , -25986.409  , -25940.221  , -20740.775  , -17924.127  , -10590.805  ,  -6274.285  ,   3092      ,   9091.143  ,
 -14690      , -19627      , -23143.484  , -25268.789  , -25760.846  , -24786.006  , -21524.237  , -17310.760  , -12162.526  ,  -6159.279  ,
    876.284  ,   8666.876  ,  -9688      , -17922      , -20650      , -24690.197  , -24345.620  , -25109.370  , -20645.575  , -18340.801  ,
 -11875.432  ,  -7982.976  ,    607.390  ,   6047.135  , -12966      , -18770      , -22287.468  , -23820.349  , -23769.717  , -21064.159  ,
 -17509.208  , -12970.252  ,  -7713.850  ,  -1309.812  ,   5838.833  ,  -9632      , -14215      , -20945.200  , -22252.843  , -23785.062  ,
 -20027.757  , -18188.360  , -12428.986  ,  -9115.396  ,  -1242.490  ,   3565.809  ,  13462      , -10805      , -16487.444  , -21033.402  ,
 -22451.523  , -20054.113  , -17145.300  , -13226.905  ,  -8634.713  ,  -2844.337  ,   3538.895  ,  11146.100  ,  -6101      , -13265.406  ,
 -16749.192  , -21748.074  , -18869.711  , -17469.152  , -12469.628  ,  -9655.227  ,  -2665.813  ,   1714.994  ,  10759.134  ,  16674.000  ,
  -2468      ,  -8644      , -13637.674  , -17614.122  , -18258.150  , -16365.570  , -12882.361  ,  -8928.826  ,  -3767.956  ,   1854.098  ,
   8844.314  ,  16536.789  ,   2329      ,  -5365      ,  -9246.048  , -14727.981  , -14791.467  , -15952.695  , -11971.655  ,  -9604.535  ,
  -3332.504  ,    460.315  ,   8789.626  ,  14059.638  ,   -624      ,  -6077.998  , -10491.300  , -11858.159  , -12432.126  , -11646.829  ,
  -8755.009  ,  -4139.817  ,    832.024  ,   7202.218  ,  13906.710  ,   2757      ,  -1551      ,  -7547.241  ,  -8117.014  , -10369.040  ,
  -8627.788  ,  -8659.243  ,  -3515.782  ,   -198.673  ,   7277.922  ,  12097.868  ,  21613.450  ,   7666      ,   1783.811  ,  -3201.643  ,
  -5230.276  ,  -6653.080  ,  -6579.162  ,  -5697.884  ,  -3552.680  ,    357.934  ,   6156.263  ,  12118.013  ,  19663.130  ,  11178      ,
   6465      ,   -180.786  ,  -1199.832  ,  -4469.603  ,  -3379.399  ,  -4319.378  ,   -958.197  ,     92.896  ,   6444.531  ,  10712.329  ,
  19485.441  ,  16208      ,   9911      ,   4416      ,   1648.580  ,   -540.130  ,  -1254.862  ,  -1168.195  ,    318.420  ,   2534.055  ,
   6030.969  ,  10921.557  ,  17868.783  ,  19859      ,  14718      ,   7480      ,   5873.992  ,   1783.984  ,   2257.527  ,    253.235  ,
   2971.341  ,   3291.363  ,   8144.502  ,  10298.604  ,  17799.838  ,  18313      ,  12240      ,   8729.962  ,   5885.328  ,   4395.928  ,
   3658.925  ,   4314.943  ,   5887.719  ,   8703.762  ,  12218.129  ,  17069.966  ,  22971      ,  23092      ,  15453      ,  13216.037  ,
   8358.805  ,   8099.083  ,   5217.844  ,   7059.470  ,   6651.455  ,  10844.119  ,  12366.823  ,  18684.447  ,  21911.973  ,  20279      ,
  16283      ,  12681.359  ,  10397.195  ,   8830.900  ,   8618.581  ,   9394.568  ,  11569.836  ,  14473.071  ,  18540.484  ,  23286.540  ,
  29277      ,  23669      ,  20819      ,  15263.461  ,  14375.747  ,  10613.564  ,  11483.181  ,  10270.534  ,  13744.058  ,  14669.306  ,
  20218      ,  22934      ,  30311      ,  24098      ,  19773.755  ,  16782.727  ,  14472.693  ,  13278.600  ,  12964.236  ,  14523.463  ,
  16938.354  ,  20376.377  ,  24483      ,  29845      ,  28671      ,  22486.265  ,  20953.026  ,  16374.031  ,  16349.762  ,  14321.577  ,
  16621.748  ,  17203.317  ,  22155      ,  24222      ,  31019      ,  32137      ,  27079      ,  23428.006  ,  20389.739  ,  18383.978  ,
  17234.808  ,  17826.697  ,  19386.117  ,  22321.032  ,  25838.886  ,  30600      ,  36766      ,  29910      ,  27711.015  ,  22445.098  ,
  21795.097  ,  18827.328  ,  20234.997  ,  19993.944  ,  23862.654  ,  25714.044  ,  31876      ,  34530      ,  30395      ,  26534.142  ,
  23820.800  ,  21994.303  ,  21638.594  ,  22310.551  ,  24340.878  ,  27377.867  ,  31587.793  ,  37549      ,  34614      ,  28747.193  ,
  27540.552  ,  23669.571  ,  24310.487  ,  23197.352  ,  26033.471  ,  27329.197  ,  32777      ,  42281      ,  37483      ,  32885.834  ,
  29685.782  ,  27179.458  ,  25851.085  ,  25806.321  ,  26832.012  ,  29022.347  ,  32562.356  ,  41684      ,  35243.465  ,  33369.073  ,
  28942.197  ,  28896.386  ,  26772.323  ,  28924.447  ,  29222.357  ,  33596.070  ,  36079.083  ,  44823      ,  39362.400  ,  35674.357  ,
  32548.508  ,  30698.209  ,  29586.778  ,  29898.383  ,  31210.899  ,  33779.829  ,  37393.536  ,  42048      ,  39510.575  ,  34516.307  ,
  33838.383  ,  30864.244  ,  32174.778  ,  31614.997  ,  35236.487  ,  36934.002  ,  42932      ,  46454      ,  42064.406  ,  38216.486  ,
  35762.849  ,  33817.544  ,  33425.957  ,  33807.514  ,  35625.328  ,  38285.812  ,  42439      ,  45986      ,  40496.954  ,  39154.419  ,
  35448.712  ,  35947.972  ,  34610.869  ,  37361      ,  38363.191  ,  43268      ,  46404      ,  49034      ,  44322.348  ,  41308.033  ,
  38733.643  ,  37490.046  ,  36920.254  ,  37949.987  ,  40052.115  ,  43263      ,  47291.952  ,  46893.271  ,  44841.190  ,  40614.449  ,
  40340.361  ,  38146.828  ,  39956.674  ,  40349.887  ,  44461      ,  46723.960  ,  53344      ,  51204      ,  47357.155  ,  44017.749  ,
  42288.896  ,  40920.654  ,  41044.874  ,  42183.829  ,  44626.243  ,  48011      ,  52704      ,  51220.992  ,  46255.198  ,  45333.950  ,
  42446.515  ,  43379.438  ,  42902.852  ,  46042      ,  47855.018  ,  53542      ,  54278      ,  49955.092  ,  47527.618  ,  45392.055  ,
  44873.466  ,  45093.467  ,  46571      ,  49247.430  ,  53188      ,  57936.868  ,  52626      ,  50894.038  ,  47309.112  ,  47456.463  ,
  46164.946  ,  48423.290  ,  49445.025  ,  54216      ,  57278      ,  56607      ,  53337      ,  50574.046  ,  49312.586  ,  48590.073  ,
  49392.190  ,  51148.445  ,  54251      ,  58246      ,  63556      ,  56803      ,  52716.372  ,  52317.758  ,  50127.186  ,  51511.976  ,
  51725.565  ,  55666      ,  57990.838  ,  64199      ,  59690      ,  56197      ,  54261.981  ,  52956.981  ,  52936.198  ,  53703.583  ,
  56034      ,  59327      ,  63863      ,  69126      ,  58621      ,  57418.575  ,  54718.575  ,  55469.876  ,  54805.418  ,  57735      ,
  59386.724  ,  64801      ,  68400      ,  62401      ,  59877      ,  57755.978  ,  57176.315  ,  57183.788  ,  58691.380  ,  60991      ,
  64748      ,  69364      ,  63202      ,  59806.815  ,  59881.141  ,  58453.841  ,  60715.692  ,  61479.351  ,  66027      ,  68966      ,
  65950      ,  63179.566  ,  61902.280  ,  61004.897  ,  61815.639  ,  63386.935  ,  66368      ,  70187      ,  75272      ,  65395.964  ,
  64995      ,  62618.605  ,  63968.605  ,  64091.940  ,  67902      ,  70188.512  ,  76115      ,  69108      ,  67154      ,  65534.451  ,
  65490.830  ,  66103.904  ,  68578.468  ,  71674      ,  75937      ,  70563      ,  67393.463  ,  68081      ,  67239.957  ,  70301      ,
  71898.728  ,  77149      ,  80621      ,  73104      ,  70751.410  ,  69850.571  ,  69725.962  ,  71177      ,  73520.540  ,  77234      ,
  81782      ,  72990.299  ,  72951.535  ,  71171.958  ,  73227      ,  74073.366  ,  78632      ,  81563      ,  76649.327  ,  75229.327  ,
  74136.327  ,  74513.546  ,  75953.919  ,  78966.824  ,  82850      ,  87728      ,  79056      ,  78535      ,  76035.247  ,  77295.247  ,
  76817.572  ,  80511      ,  82872.142  ,  88738      ,  80929      ,  79302.276  ,  79014.646  ,  79348.918  ,  81175      ,  84359.919  ,
  88577      ,  93557      ,  84393      ,  81342.027  ,  81992.151  ,  80904.351  ,  83454      ,  84724.653  ,  89873      ,  93198      ,
  84811      ,  84090.583  ,  83800.963  ,  84844.339  ,  86807.235  ,  89947.301  ,  94331      ,  99733      ,  87041      ,  87187      ,
  85487.368  ,  87457      ,  87823.929  ,  91746.599  ,  94222.685  , 100499      ,  89403      ,  88590.741  ,  88997.161  ,  90250.433  ,
  92607      ,  95867.612  , 100208      ,  92702      ,  90427      ,  91688.349  ,  91479      ,  94784      ,  96342.263  , 101799      ,
 105243      ,  93705      ,  93625      ,  94111      ,  95853      ,  98362      , 101991.017  , 106559      ,  95766      ,  96552      ,
  95612      ,  98277      ,  99149      , 103673      , 106548.108  , 113324      ,  98578      ,  98456      ,  99562      , 101321.771  ,
 104250      , 108006.228  , 113135      , 101627      , 100102      , 102103      , 102394      , 106257      , 108366.919  , 114543      ,
 103130      , 103730      , 104788      , 107112      , 110190      , 114496      , 119718      , 105011      , 106377      , 106077      ,
 109362      , 110784      , 116058      , 119563.699  , 108233      , 108691      , 110487      , 112797      , 116357      , 120901.389  ,
 126680      , 111622      , 110077      , 112738      , 113619      , 118108      , 121137.442  , 127963      , 113445      , 114075      ,
 115838      , 118767      , 122653      , 127791      , 133919      , 115476      , 117062      , 116802      , 120807      , 122831      ,
 129151      , 133649      , 119148      , 119815      , 121482      , 124590      , 129312      , 134835.736  , 122357      , 121492      ,
 124227      , 125090      , 130714      , 134679.272  , 124758      , 125990      , 127773      , 131102      , 135947      , 126580      ,
 128792      , 129007      , 133582      , 136016      , 142773      , 130018      , 130633      , 131973      , 134507      , 138383      ,
 142636      , 133714      , 133487      , 137158      , 139175      , 144617      , 135691      , 136621      , 138625      , 141618      ,
 145257      , 138285      , 141209      , 142542      , 147487      , 150352      , 141493      , 142768      , 145233      , 148172      ,
 152430      , 145599      , 146282      , 150430      , 152910      , 158893      , 147246      , 149130      , 151574      , 155133      ,
 159239      , 150260      , 153825      , 155697      , 161083      , 153241      , 154963      , 158117      , 161596      , 157534      ,
 158816      , 163638      , 158861      , 161400      , 164475      , 162225      , 166483      , 164983      , 167415      , 171062      ,
 169725      , 171613      , 170825      , 173987      , 177639      , 174722      , 179536      , 177374      , 180363      , 184587      ,
 182550      , 185030      , 183570      , 187302      , 191454      , 187921      , 193251      , 190479      , 193970      , 198932      ,
 196044      , 199266      , 201427      
};

//+----------------------------+
//| Table of Beta Decay Energy |
//+----------------------------+
//unit: keV
const G4double G4NucleiPropertiesTableAME12::BetaEnergy[G4NucleiPropertiesTableAME12::nEntries] = {

    782.347,      0    ,      0    ,     18.591, -13736    ,      0    ,  22196.208, -22898.270,      0    ,  21661.208,
   -447.653, -25460    ,      0    ,  24283.622,   3505.216,  -4288.155, -28945    ,      0    ,  23062    ,  11166.021,
   -861.893, -11907.551,      0    ,  10663.878,  16004.133, -17979.906, -12142.691,      0    ,  15980.994,  13606.450,
  -1068.035, -16494.482,      0    ,  16119.692,  20445.136,    556.797,  -3648.064, -23101.353,      0    ,  20551.087,
  11509.284,  -1982.412, -13653.347,      0    ,  23841.813,  11708.343,  13369.416, -17338.072, -14576.623,      0    ,
  24681.813,  17096.957,  13437.110,  -2220.472, -17769.959,      0    ,  16290.813,  20643.792,    156.476,  -5144.040,
 -23956.953,      0    ,  20802    ,  19084.844,   9771.705,  -2754.166, -13951.205,      0    ,  20334.862,  23418.142,
   8010.221,  10420.908, -15417.256, -13305.900,      0    ,  22740.038,  13160.703,   8678.839,  -2760.465, -14548.749,
      0    ,  26929.008,  11804.956,  13895.984,  -1655.929,  -4444.510, -19719.308,      0    ,  26365    ,  16557.091,
  12522.664,   4820.302,  -3239.498, -11177.337, -18899.001,      0    ,  29575    ,  15792.500,  17968.942,   3813.632,
   7024.467, -13892.535, -10708.063,      0    ,  30078    ,  20392    ,  17189.257,   8109.516,   5684.167,  -3547.138,
 -13098.152, -16077    ,      0    ,  21551.569,  22755.353,   6489.946,  10818.091,  -2843.196,  -4781.580, -18601    ,
 -15137    ,      0    ,  25849    ,  23701    ,  11307.616,   8467.086,   4375.808,  -4056.588, -12221.335, -16949    ,
      0    ,  28438    ,  10940.876,  13511.167,   2466.317,   5515.611, -13885.955, -10791.968, -22575    ,      0    ,
  28635    ,  15984.210,  13423.315,   7298.011,   3834.954,  -4276.606, -12743.496, -15911    ,      0    ,  16067.976,
  18185.615,   7340.225,   9353.766,  -4004.433,  -5069.136, -18114    , -16106    ,      0    ,  19821    ,  17593.809,
  12553.499,   9068.939,   2610.133,  -4812.359, -11661.928, -17750    ,      0    ,  19159    ,  21629.381,  12279.884,
  14030.415,   1831.800,   4642.264, -14345.055, -11220.945, -23443    ,      0    ,  21226    ,  15719.825,  13282.806,
   7601.833,   3690.417,  -4942.628, -13796.043, -16924    ,      0    ,  25072    ,  14564.903,  17358.397,   6988.725,
   8560.508,  -4232.352,  -6141.601, -18502    , -17046    ,      0    ,  24798    ,  18559.407,  15383.272,  11832.639,
   7994.060,   1491.505,  -5398.016, -11976.401, -18360    ,      0    ,  18189    ,  19638.465,  10269.721,  12979.158,
    227.188,   1710.660, -12680.859, -11134.323, -23299    ,      0    ,  22030    ,  19005    ,  13430.437,  12046.093,
   5823.021,    248.508,  -5582.517, -11619.044, -16426    ,      0    ,  21553    ,  22966    ,  11392.249,  16887.828,
   4591.970,   5382.994,  -5491.605,  -6061.795, -17158    , -15072    ,      0    ,  22200    ,  15859.617,  14140.566,
  10497.392,   3988.426,    167.323,  -5966.130, -11874.519, -15961    ,      0    ,  25531    ,  14429.774,  18343.707,
   7857.705,  10413.096,  -1142.107,    709.522, -12814.475, -10965.915, -21802    ,      0    ,  24853    ,  18480    ,
  16403.851,  12401.817,   7900.306,   4865.111,   -813.873,  -6147.449, -11664.133, -16620    ,      0    ,  17864    ,
  20380.157,  10499.257,  12191.631,   2936.900,   4916.733,  -5914.066,  -6742.253, -17513    , -15211    ,      0    ,
  21279    ,  18676    ,  15149.625,  10333.073,   6637.846,   3441.998,    565.000,  -6524.488, -13109.992, -16371    ,
      0    ,  20633    ,  22543    ,  13504.103,  14763.421,   4719.967,   7482.082,  -1504.403,   1310.894, -14323.050,
 -11672.950, -20736    ,      0    ,  21768    ,  17099.435,  14028.810,   8298.611,   5760.317,   2492.038,   -421.656,
  -6495.476, -12944.874, -15893    ,      0    ,  24275    ,  15552    ,  18647.485,   7275.244,   9509.686,    599.351,
   3525.217,  -6426.100,  -7016.480, -17485    , -13861    ,      0    ,  24843    ,  18421    ,  16875.285,  12127.168,
   7687.181,   4565.581,   1833.426,  -2220.715,  -6867.016, -11404.728, -15616    ,      0    ,  18071    ,  19646    ,
  11405.496,  12063.525,   3108.237,   5687.166,  -3652.680,   -267.407, -13432.191, -10479    , -20297    ,      0    ,
  21890    ,  19313    ,  14370.158,  11411.050,   6844.841,   4196.514,    259.025,  -2062.055,  -7128.572, -12365.750,
 -14382    , -18564    ,      0    ,  22747    ,  13848    ,  15920.432,   5683.162,   7724.473,  -1377.897,   2366.496,
  -7052.394,  -7601.071, -16516    , -13544    ,      0    ,  21834    ,  17503    ,  15114    ,  10500.966,   6631.477,
   1992.177,    600.770,  -2930.599,  -7444.914, -11995.445, -14974    , -17438    ,      0    ,  16823    ,  18378    ,
   9845    ,  11940.282,    278.661,   3988.314,  -4014.966,  -1655.114, -13498.222, -10908    , -19282    , -15610    ,
      0    ,  20057    ,  18006    ,  12751    ,  11688.404,   5261.178,   2001.737,   -601.856,  -2628.260,  -7695.955,
 -12886.011, -14420    , -17503    ,      0    ,  21350    ,  12808    ,  13861.376,   4958.290,   6883.167,  -2206.825,
   1037.853,  -7634.477,  -8138.601, -16706    , -12883    ,      0    ,  20344    ,  16648    ,  13822.490,   6890.284,
   6503.044,   2471.826,   -752.630,  -3207.559,  -8041.165, -12860.187, -14404    ,      0    ,  15575    ,  17720    ,
   5903.497,   9302.339,   1974.766,   3974.477,  -4711.235,  -2374.561, -14342    , -10517    , -20409    ,      0    ,
  18472    ,  16776    ,   9650    ,   8722.572,   5020.000,   3435.903,   -596.849,  -3742.311,  -8288.106, -13027.803,
 -15277    ,      0    ,  19776    ,   8821    ,  11998.064,   4295.054,   7041.591,  -1377.174,    697.343,  -8244.547,
  -8786.402, -17482    , -14325    ,      0    ,  19058    ,  11626    ,  11691.021,   7476.090,   5965.023,   2603.082,
   -231.088,  -3451.379,  -8694.034, -13699.825, -16713    ,      0    ,  21825    ,  10833    ,  14474    ,   6917.780,
   9158.143,   1629.600,   3695.577,  -4566.624,  -2132.889, -15669    , -12659    , -21257    ,      0    ,  13833    ,
  13160    ,  10360.782,   8295.751,   4961.992,   2695.049,   -836.232,  -3261.729,  -8774.967, -14762    , -16897    ,
      0    ,  12957    ,  16236    ,   9208    ,  11515.227,   3992.834,   6326.907,  -2307.909,    381.595,  -8561.019,
  -9368.981, -18806    , -15780    ,      0    ,  15342    ,  12188    ,  10059.475,   7633.830,   5138.843,   1564.956,
  -1073.001,  -4798.386,  -9142.777, -13243    , -17661    ,      0    ,  18276    ,  10914    ,  13261.920,   6464.062,
   8444.437,    237.345,   2822.813,  -6127.982,  -4170.819, -14390    , -12175    , -21909    ,      0    ,  17279    ,
  14159    ,  11948.660,   9287.033,   7178.372,   3977.129,   1323.699,  -2237.488,  -5635.144,  -9214.149, -13405    ,
 -16143    ,      0    ,  12911    ,  15419    ,   7586    ,  10397    ,   2545.776,   5322.039,  -3958.896,  -1619.455,
  -9181.066, -10088    , -17316    ,      0    ,  16171    ,  13730    ,  11164.939,   8748.568,   6215.378,   3661.322,
     66.977,  -3366.187,  -5666.019,  -9625.876, -13294    ,      0    ,  17289    ,   9530    ,  11980.510,   4822.343,
   7306.591,  -1674.385,    579.696,  -7171.039,  -4517.255, -14663    , -12723    ,      0    ,  16338    ,  12985    ,
  10254.154,   7963.706,   5940.044,   2138.215,  -1351.659,  -3254.470,  -6179.112,  -9541.165, -13776    ,      0    ,
  18928    ,  12215    ,  13317.453,   6340.694,   9597.751,    251.785,   2640.993,  -5175.000,  -2117.030,  -9581.956,
 -10657    ,      0    ,  14829    ,  12758    ,   9253.245,   8420.904,   3576.099,    561.288,  -1001.171,  -4220.892,
  -6070.780, -10006.936, -13652    ,      0    ,  13423    ,  15527    ,   8098.373,  11540.093,   2103.220,   4439.812,
  -2921.100,   -106.957,  -8084.271,  -4705.077, -15748    ,      0    ,  14522    ,  11112    ,   9809.563,   5757.564,
   2681.387,    910.200,  -2227.146,  -3988.490,  -6677.457, -10323.236, -13677    ,      0    ,  17093    ,  10610    ,
  12294.507,   3762.512,   6588.337,   -654.578,   1651.704,  -6220.000,  -2411.943, -10504.272, -10477    ,      0    ,
  15798    ,  13370    ,  11036.302,   7304.898,   4617.649,   2810.322,   -232.641,  -2013.400,  -4746.550,  -6644.089,
 -10175.212, -14023    ,      0    ,  11681    ,  14442    ,   5556.938,   8362.487,    442.770,   3997.645,  -4356.102,
   -361.618,  -8800.859,  -5126.746, -15824    ,      0    ,  14280    ,  13212    ,   8879.285,   6605.965,   4105.932,
   1598.189,   -344.774,  -2725.361,  -4579.803,  -7095.834, -10470    , -14131    ,      0    ,  13227    ,  15994    ,
   7550    ,   9750.507,   2292.905,   5372.825,  -2562.387,   1353.147,  -6924.952,  -2956.415, -10415.849, -11089    ,
      0    ,  15146    ,  10225    ,   8087.566,   5905.672,   3392.385,   1177.226,   -864.709,  -3062.363,  -4783.495,
  -7104.929, -10600.000,      0    ,  17512    ,   9366    ,  11327.031,   3993.624,   6916.250,   -921.517,   2960.578,
  -4962.881,  -1274.931,  -8535.057,  -6231.442, -15647    ,      0    ,  11765    ,  10277    ,   7203.149,   5220.518,
   2703.457,    683.166,  -1364.680,  -3065.363,  -5338.951,  -7027.056, -11025    ,      0    ,  10368    ,  12985.766,
   6222.716,   8156.103,    954.907,   4208.949,  -3573.784,    727.452,  -7244.158,  -3761.477, -10647    , -11225    ,
      0    ,  14189    ,  11534    ,   9115.384,   6978.916,   4109.475,   2281.356,    150.630,  -1625.778,  -3639.292,
  -5326.096,  -7120.000, -11298    ,      0    ,  15218    ,   7575.055,  10311.639,   2679.155,   5545.017,  -1870.463,
   2004.299,  -5717.846,  -1863.987,  -9164.578,  -5629.860,      0    ,  14410    ,  11428.291,   8663.733,   6241.628,
   3855.693,   1586.644,   -280.837,  -2238.151,  -3928.538,  -5816.399,  -7315.335, -11449    ,      0    ,  16936    ,
  10324    ,  12484.348,   4688.023,   7490.829,    -96.642,   3093.041,  -4402.515,   -177.757,  -7947.084,  -4120    ,
 -11742    ,      0    ,  12519    ,  11719.312,   8692.888,   5671.245,   3672.833,    976.630,   -919.402,  -2273.024,
  -4593.081,  -6294.012,  -7500.000, -11724    ,      0    ,  11873    ,  13865    ,   7705.132,  10094.161,   1835.364,
   4656.251,  -2680.375,    890.595,  -6756.079,  -2472.943, -10400    ,  -6519    ,      0    ,  14224    ,  13060    ,
  10065.724,   9224.492,   6161.833,   2904.862,    687.000,  -1064.054,  -3261.153,  -4668.098,  -6894.349,  -8769.923,
 -11475    ,      0    ,  15304    ,   9202    ,  11541.024,   5129.085,   7633.414,   -518.655,   1776.189,  -5240.000,
  -1314.546,  -8835.774,  -5022.634, -12813    ,      0    ,  14499    ,  11540    ,  10808.218,   7465.552,   6817.845,
   3888.269,    282.243,  -1861.690,  -3671.810,  -5473.234,  -6988.491,  -9194.764, -11758    ,      0    ,  10582    ,
  13164    ,   6831.763,   8975.327,   2917.709,   5312.357,  -3622.600,   -670.422,  -7452.342,  -3489.443, -11005.229,
  -7282    ,      0    ,  13069    ,  12194    ,   9281.872,   8261.522,   5176.454,   4496.511,   1500.401,  -2832.760,
  -4250.996,  -5610.460,  -7620.087,  -9288    , -12076    ,      0    ,  12109    ,  14470    ,   8200.081,  10958.952,
   4405.480,   6584.187,    545.944,   2278.698,  -6111.336,  -2489.336,  -9448.200,  -5840.894, -12925    ,      0    ,
  13441    ,  10769    ,   9866.671,   6771.161,   5907.281,   2699.470,   1544.281,  -1257.582,  -4429.863,  -6222.459,
  -7746.735,  -9444    , -12519    ,      0    ,  15742    ,   9509    ,  12536.515,   6003.204,   8094.868,   1949.662,
   3642.576,  -2005.885,    354.082,  -7882.135,  -4624.492, -11302.114,  -7929    ,      0    ,  12255    ,  11165.656,
   8483.959,   7465.963,   4141.741,   2895.027,     90.301,   -406.665,  -3200.963,  -6388.644,  -8204.913,  -9872    ,
 -12872    ,      0    ,  10796    ,  13748    ,   7215.018,  10282.959,   3507.094,   4918.059,   -901.694,   2043.636,
  -4255.748,  -1573.504,  -9675.978,  -6806.567, -13690    ,      0    ,  13311    ,  12388    ,   9734.969,   9228.370,
   6089.067,   4450.238,   1123.500,    925.567,  -1690.518,  -2563.609,  -5115.891,  -8375.941, -10368    , -12966    ,
      0    ,  14916    ,   8274.683,  11575.494,   5411.738,   7103.017,    162.217,   3186.800,  -2973.242,    258.738,
  -6392.653,  -3504.311, -11670.523,  -8939    ,      0    ,  13368    ,  11095.629,  10063.093,   7544.965,   6821.237,
   2659.671,   1935.471,   -324.708,  -1099.548,  -3523.000,  -4790.461,  -6980.000, -10372    , -13264    ,      0    ,
  15863    ,  10007    ,  12107.643,   5874.614,   8991.924,   2237.851,   4584.477,  -1683.717,   1793.640,  -5049.653,
  -1854.137,  -8254.560,  -5430.000, -13740    ,      0    ,  12446    ,  11306.539,   8143.760,   6968.724,   4707.369,
   3637.175,   1357.806,    295.062,  -2043.844,  -3396.755,  -5468.766,  -6781.350,  -8555    , -13432    ,      0    ,
  11495    ,  13283    ,   7503.184,   9048.641,   3421.362,   6386.146,   -169.620,   3203.986,  -3635.629,   -360.992,
  -7089.266,  -3943.363,  -9881.624,  -7030.000,      0    ,  13684    ,  12750    ,   9505.000,   8103.757,   5717.240,
   4628.414,   2824.654,   1613.520,   -543.751,  -1980.169,  -4096.283,  -5497.918,  -7223    ,  -8308    ,      0    ,
  14651    ,   8815.000,  10420.396,   4716.837,   7259.862,   1000.127,   4532.311,  -2322.473,   1150.568,  -5684.462,
  -2587.000,  -8965.789,  -5760.000,      0    ,  13814    ,  11035    ,   9363.681,   7203.653,   5941.683,   3635.106,
   2661.748,    764.415,   -543.020,  -2684.742,  -4148.295,  -6022.480,  -7660.000, -10791    ,      0    ,   9958    ,
  11666    ,   6094.952,   8530.732,   2151.272,   5586.549,  -1138.042,   2439.284,  -4278.653,  -1148.087,  -7785.571,
  -4555.623, -12456.496,      0    ,  12212    ,  10649    ,   8440.950,   7431.323,   4949.711,   3639.569,   1918.035,
    567.208,  -1346.930,  -2736.991,  -4693.266,  -6302.586,  -9321.896, -11204.580,      0    ,  11263    ,  12855    ,
   7292    ,   9941.136,   3634.565,   6547.000,     39.405,   3545.616,  -2965.142,    189.753,  -6524.003,  -3254.332,
 -10880.389,  -8254.603,      0    ,  13465    ,  11904    ,   9451    ,   8837.808,   6189.994,   5112.598,   3003.355,
   1508.662,     34.078,  -1416.347,  -3426.000,  -5052.021,  -7859.005, -10116.893, -10962    ,      0    ,  14056    ,
   8193    ,  11216.383,   5158.397,   7738.573,   1372.602,   4492.440,  -1917.673,   1645.863,  -5132.610,  -2049.998,
  -9624.613,  -6663.659, -13135.142,      0    ,  12992    ,  10427    ,  10052.534,   7608.260,   6455.626,   4263.574,
   2606.706,   1113.251,   -215.521,  -2016.432,  -3857.018,  -6379.860,  -8535.664, -10042.207, -11503.556,      0    ,
   9250    ,  12413    ,   6483.328,   9038.135,   2758.402,   5502.549,   -873.766,   2891.004,  -3878.000,   -628.781,
  -8392.250,  -5219.910, -11765.648,  -8545.589,      0    ,  11316    ,  11062    ,   9084.861,   7760.649,   5521.474,
   3681.852,   2229.769,   1036.800,   -862.178,  -2451.383,  -5102.783,  -7249.259,  -8633.669, -10561.271,      0    ,
  10463    ,  13190    ,   7791    ,  10373.656,   4104.174,   6588.945,    261.888,   3992.088,  -2584.734,    664.919,
  -7057.026,  -4031.457, -10504.178,  -7036.908, -13739.142,      0    ,  12258    ,  10042    ,   9058.813,   6897.393,
   4823.555,   3435.488,   2016.462,    322.571,  -1037.581,  -3911.164,  -6070.047,  -7227.523,  -8915.874, -10439.713,
      0    ,  14420    ,   8961    ,  11454    ,   5491.399,   7777.733,   1439.645,   5084.043,  -1446.445,   1988.939,
  -6061.691,  -2607.075,  -9092    ,  -5710    , -12403.629,  -8722.755,      0    ,  13395    ,  11159    ,  10396    ,
   7925.439,   6197.307,   4556.078,   3101.728,   1451.955,    497.490,  -3030.430,  -4940.644,  -5724.962,  -7681.049,
  -8957    , -10675    ,      0    ,   9956    ,  12613    ,   6670.118,   9092.824,   2710.804,   6169.903,   -462.810,
   3276.249,  -4704.341,  -1552.693,  -7776.725,  -4445.512, -10984    ,  -7365    , -13995    ,      0    ,  12212    ,
  11138    ,   9377.917,   7527.095,   5757.366,   4236.414,   2524.608,   1454.710,  -1758.218,  -3544.139,  -4659.346,
  -6250.743,  -7692.245,  -8871.404, -11033    ,      0    ,  13469    ,   7629    ,  10501.206,   4164.908,   7147.929,
    526.530,   4424.635,  -3656.640,   -299.466,  -6725.732,  -2891.991,  -9669.689,  -6055    , -12734    ,      0    ,
  12193    ,  10259    ,   8585.330,   7237.648,   5331.346,   3722.384,   2365.563,   -590.846,  -2293.000,  -3415.687,
  -4971.117,  -6489.361,  -7714.965,  -9623    , -10917    ,      0    ,  14494    ,   8803    ,  11466    ,   5371.293,
   8305.853,   1771.237,   5370.000,  -2680.608,    950.204,  -5615.000,  -1580.763,  -8283.785,  -5000.000, -11201    ,
  -7890    ,      0    ,  11383    ,   9752    ,   8220.493,   6671.005,   4762.358,   3361.303,    401.089,  -1054.306,
  -2292.629,  -3770.655,  -5378.654,  -6357.495,  -8474    ,  -9501    , -11150    ,      0    ,   9762    ,  12443    ,
   6489.949,   9506.265,   2960.581,   6368.591,  -1607.921,   1980.816,  -4234.000,   -725.464,  -7210.218,  -3535.815,
 -10066    ,  -6669    , -12929    ,      0    ,  12147    ,  10908    ,   9131    ,   7866.103,   6016.274,   4385.936,
   1408.435,    -52.689,  -1228.425,  -2695.017,  -4205.048,  -5388.693,  -7004    ,  -8365    ,  -9948    ,      0    ,
  10749    ,  13386    ,   7646    ,  10501.538,   4168.546,   7364.020,   -613.233,   2904.299,  -3159.587,    295.337,
  -5929.718,  -2641.560,  -8831.165,  -5343    , -11765    ,  -8626    ,      0    ,  12016    ,  10008    ,   9117.857,
   7128.746,   5419.579,   2359.836,    766.700,   -185.770,  -1644.173,  -3105.172,  -4418.875,  -5909.481,  -7102    ,
  -8626    , -10433    ,      0    ,  14252    ,   8765    ,  11477    ,   5516.127,   8242.357,    378.000,   3672.043,
  -2154.027,   1234.264,  -4795.112,  -1680.553,  -7696.434,  -4152.910, -10497    ,  -7331    , -13795    ,      0    ,
  11141    ,   9909    ,   8407.785,   6572.931,   3228.247,   1582.183,    702.236,   -662.349,  -2081.396,  -3422.207,
  -4921.836,  -5916.772,  -7436    ,  -9008    , -10749    ,      0    ,  10032    ,  12340    ,   6904.060,   9216.071,
   1268.282,   4363.443,  -1255.036,   2121.568,  -3928.713,   -552.657,  -6753.477,  -3091.513,  -9203.161,  -6017    ,
 -12529    ,  -9054    ,      0    ,  11299    ,   9329    ,   7769.210,   4022.234,   2375.500,   1502.305,    188.915,
  -1196.805,  -2435.964,  -3738.675,  -5037.117,  -6513.938,  -7459    ,  -9434    , -10740    ,      0    ,  15614    ,
   8350.000,  10249.000,   2153.406,   5066.681,   -416.792,   2944.307,  -2980.719,    361.992,  -5634.369,  -2204.461,
  -8247.448,  -4579.225, -11200    ,  -7890    , -13682    ,      0    ,  12695    ,   9246.546,   4709.778,   3229.104,
   2231.774,    970.848,   -354.752,  -1374.958,  -2914.666,  -4060.816,  -5407.784,  -6532.623,  -7845    ,  -9790    ,
 -10863    ,      0    ,  12145    ,  14135.000,   3091.660,   5552.613,    515.305,   3575.473,  -2122.715,   1278.742,
  -4711.358,  -1252.669,  -7257.479,  -3787.676,  -9798    ,  -6548    , -12855    ,      0    ,  13544    ,  13410    ,
   8048.300,   4009.553,   2954.511,   1757.000,    427.360,   -517.343,  -2059.206,  -3076.168,  -4480.634,  -5605.208,
  -6924.726,  -8177    ,  -9995    , -11215    ,      0    ,  14772    ,   7588.311,   8515.455,   1523.107,   4065.201,
  -1233.149,   2058.901,  -3731.405,   -385.760,  -6304.898,  -2881.559,  -8907.681,  -5363    , -11448    ,  -8626    ,
      0    ,  14104    ,   9057.382,   8038.234,   6061.100,   2627.807,   1165.048,    268.898,  -1207.205,  -2027.152,
  -3680.496,  -4722.252,  -6186.684,  -7169.709,  -8709    ,  -9855    , -11467    ,      0    ,   8611    ,   9916.006,
   5119.674,   6883.674,    -90.238,   2548.224,  -2849.660,    471.110,  -5168.163,  -2141.140,  -8018.786,  -4369.611,
 -10567    ,  -7154    , -13032    ,      0    ,  10242    ,   9274.091,   7052.029,   6027.146,   4162.428,   1175.629,
   -580.548,  -1222.100,  -2716.969,  -3617.269,  -5511.719,  -6046.323,  -7908    ,  -8905    , -10246    ,      0    ,
   9678    ,  11157    ,   6283.694,   7992.339,   2914.817,   5374.808,  -1739.960,   1046.841,  -4437.000,  -1113.427,
  -7077.827,  -3442.721,  -9748.093,  -6093    , -11988    ,  -8737    ,      0    ,  10417    ,   8253.957,   7185.548,
   5056.565,   4212.829,   2314.616,   -278.382,  -2129.070,  -2806.486,  -4514.039,  -5120.390,  -6982.177,  -7767    ,
  -9501    , -10489    ,      0    ,  12418    ,   7239.169,   9390.561,   4063.879,   6219.896,   1048.019,   3760.933,
  -3388.000,   -437.406,  -6045.200,  -2752.638,  -8470.000,  -5203.663, -11300.000,  -7652    , -13571    ,      0    ,
   9417    ,   8293    ,   6279.847,   5255.636,   3201.827,   2501.393,    580.420,  -1823.014,  -3670.488,  -4589.054,
  -6008.246,  -6701.396,  -8683.387,  -9158    , -11018    ,      0    ,   8400    ,  10459.654,   5288.192,   7324.904,
   2181.086,   4508.911,   -744.480,   2161.635,  -4807.827,  -2155.453,  -7673.000,  -4354.091, -10400.000,  -6440    ,
 -12869    ,  -9399    ,      0    ,  10349    ,   9576    ,   7470.793,   6263.397,   4234.368,   3434.784,   1461.816,
    934.108,  -1041.664,  -3443.772,  -5275.403,  -6010.000,  -7812.117,  -8250.242, -10121    , -10954    ,      0    ,
  11592    ,   6398.215,   8496.573,   3082.572,   5582.117,    318.646,   2997.440,  -2331.923,    549.516,  -6346.349,
  -3859.596,  -9391.323,  -5798.099, -11960.569,  -8002    , -14515    ,      0    ,  10554    ,   8562.479,   7460.368,
   5319.139,   4231.629,   2558.881,   1806.013,   -164.515,   -616.093,  -2659.686,  -5067.506,  -6622.108,  -8059.331,
  -9122.493, -10044    , -11494    ,      0    ,   7614.011,   9371.350,   4106.527,   6588.235,   1045.660,   4245.025,
  -1471.545,   1542.000,  -3878.786,  -1031.888,  -8322.123,  -5208.492, -11316.698,  -6916.206, -13431    ,      0    ,
   9412    ,   8245.824,   6414.361,   5335.492,   3430.129,   2702.620,    895.303,    224.074,  -1721.609,  -2187.824,
  -4614.066,  -6546.472,  -8438.945,  -9149.286, -10633.406,      0    ,   8295    ,  10297.255,   5115.000,   7689.645,
   2136.980,   4872.449,   -542.735,   2470.992,  -3036.687,    -30.057,  -5738.041,  -2678.064,  -9862.059,  -6512.169,
 -12713.962,  -8566    ,      0    ,   9259    ,   7199    ,   6450.000,   4369.452,   3335.913,   1688.400,   1071.418,
   -694.616,  -1313.737,  -3637.887,  -3789.450,  -6036.970,  -7920.823,  -9859    , -10684    ,      0    ,  11439    ,
   6129    ,   8464    ,   3453.551,   5378.682,    -82.616,   3454.000,  -2258.642,    971.780,  -4657.953,  -1796.221,
  -7363.595,  -4114.557, -11340    ,  -7852    , -13998    ,      0    ,  10535    ,   8337    ,   7496    ,   5554.420,
   4162.843,   2443.076,   1190.210,     76.376,   -464.021,  -2565.008,  -2870.832,  -5129.533,  -5356.298,  -7488.567,
  -9235.411, -11434    ,      0    ,   7443    ,   9520    ,   4701    ,   6390.686,   1104.784,   3508.436,  -1874.597,
   1818.919,  -3990.000,   -598.346,  -6518.909,  -3105.100,  -8723.362,  -5449.892, -12899    ,      0    ,   9315    ,
   8784    ,   6545    ,   5761.813,   3317.322,   1911.459,    807.276,   -484.399,  -1568.987,  -2170.358,  -4130.292,
  -4540.112,  -6482.882,  -6781    ,  -8785    , -11121    ,      0    ,  10591    ,   5844    ,   7490.000,   2806.930,
   3963.200,   -717.345,   1968.156,  -3549.651,    237.987,  -5754.357,  -2034.390,  -8177.764,  -4495.038, -10217    ,
  -6982    ,      0    ,   9743    ,   7490    ,   6868.455,   4656.206,   3250.262,   1626.986,    252.094,   -819.784,
  -2094.500,  -3115.813,  -3830.175,  -5583.141,  -6123.149,  -7952.600,  -8187    , -10372    ,      0    ,   6697    ,
   8906    ,   3690.000,   5199.086,    722.442,   2449.134,  -2444.184,    438.238,  -5050.000,  -1267.951,  -7381.401,
  -3570.533,  -9508.527,  -5882.648, -11813    ,      0    ,   8532    ,   7921    ,   5835.500,   4380.533,   2780.744,
   1364.524,    -60.044,  -1339.145,  -2593.062,  -3441.154,  -4653.243,  -5310.661,  -6968.803,  -7553    ,  -9260    ,
  -9933    ,      0    ,   9725    ,   5035    ,   6161.033,   2004.946,   3433.727,  -1218.952,    936.219,  -4219.755,
   -882.595,  -6600.615,  -2692.971,  -8797.922,  -5109.789, -10936    ,  -7471    ,      0    ,   8719    ,   6747    ,
   5653.494,   3835.109,   2517.920,    970.858,   -365.377,  -1837.600,  -2768.500,  -3989.765,  -4731.639,  -6130.155,
  -6855.846,  -8408.491,  -8957    , -10750    ,      0    ,   5868    ,   7233    ,   3245.671,   4460.485,   -105.394,
   1835.891,  -3290.000,   -317.196,  -5762.200,  -2137.127,  -7894.964,  -4338.665, -10057.118,  -6497.240, -12445    ,
      0    ,   7648    ,   6436    ,   5119.564,   3713.508,   1955.813,    593.672,   -857.957,  -1996.241,  -3301.587,
  -4059.580,  -5276.784,  -6246.944,  -7613.934,  -8146    ,  -9665    , -10671    ,      0    ,   8160    ,   4165    ,
   5585.000,   1395.059,   2505.521,  -2139.364,    292.405,  -4856.728,  -1649.980,  -6994.749,  -3662.691,  -9388.757,
  -5779.975, -11499    ,  -7998    ,      0    ,   7471    ,   5919    ,   4675.000,   3280.905,   1785.092,     -2.555,
  -1210.729,  -2439.000,  -3428.327,  -4507.842,  -5527.435,  -6729.266,  -7626.357,  -8900.909,  -9622    ,      0    ,
   5232    ,   6440.000,   2303    ,   3890.000,   -985.893,    960.819,  -4037.862,   -886.149,  -6375.206,  -2824.146,
  -8535.424,  -5054.923, -10704.904,  -7050.331, -12936    ,      0    ,   6878    ,   5800.000,   4080    ,   3046    ,
   1286.972,   -377.871,  -1591.591,  -2633.428,  -3853.140,  -4806.734,  -5787.906,  -6986.406,  -8217.521,  -8849    ,
 -10152    ,      0    ,   7601    ,   3354    ,   4700.000,    487.112,   1854.974,  -3037.667,   -292.628,  -5574.294,
  -2161.998,  -7761.208,  -4209.617,  -9994.569,  -6456.795, -12083    ,  -8562    ,      0    ,   6930    ,   5114    ,
   4004    ,   2350.000,   1009.677,   -746.662,  -1953.153,  -3089.776,  -4033.369,  -5116.697,  -6251.778,  -7257    ,
  -8341    ,  -9424.435, -10273    ,      0    ,   4359    ,   5837    ,   1500.730,   2930.000,  -1677.361,    268.091,
  -4514.051,  -1706.837,  -6966.644,  -3501.055,  -9097.968,  -5807.998, -11270.045,  -7658.765,      0    ,   6176    ,
   5269    ,   3200.000,   2125.054,    352.999,   -898.540,  -2293.000,  -3367.211,  -4426.459,  -5372.713,  -6508.552,
  -7686.234,  -8645.115,  -9565    , -10724    ,      0    ,   6940    ,   2576    ,   3870.000,   -312.798,    968.433,
  -3457.695,  -1053.179,  -6116.190,  -2847.193,  -8372.718,  -4991.302, -10567    ,  -7055    , -12553    ,      0    ,
   6157    ,   4330    ,   3200.000,   1492.076,     96.561,  -1478.049,  -2398.634,  -3711.072,  -4634.183,  -5835.810,
  -6947.054,  -7886.292,  -8946.996,  -9902.288, -10855    ,      0    ,   3474    ,   5000    ,    890.823,   1881.291,
  -2518.043,   -335.910,  -5072.248,  -2232.791,  -7570.872,  -4282.685,  -9864.260,  -6281.889, -11728.779,  -8259.262,
      0    ,   5412    ,   4304    ,   2600    ,   1297.789,   -669.644,  -1470.802,  -3015.246,  -3669.155,  -5173.518,
  -6115.692,  -7169.868,  -8325.155,  -9126.654, -10107    ,      0    ,   6260    ,   1916    ,   3080.000,  -1373.411,
    274.523,  -4105.893,  -1513.678,  -6553.992,  -3678.011,  -9126.348,  -5550.291, -11083    ,  -7589    ,      0    ,
   5449    ,   3660    ,   2385.000,    470.968,   -683.712,  -2075.194,  -2775.852,  -4344.488,  -5183.125,  -6710.740,
  -7694.263,  -8284.700,  -9442.587,      0    ,   2739    ,   4120.000,   -105.476,   1194.223,  -3213.068,   -723.771,
  -5578.718,  -2964.945,  -8238.496,  -4925.715, -10537.126,  -6623.324, -12354.558,      0    ,   4611    ,   3516    ,
   1401.024,    497.239,  -1166.000,  -2015.315,  -3432.555,  -4319.923,  -5901.826,  -6676.984,  -7825.389,  -8762.453,
  -9457.933,      0    ,   5578    ,    645.923,   2097.896,  -1837    ,   -191    ,  -4755.926,  -2109.487,  -7292.082,
  -4254.621,  -9671.141,  -6009.858, -11523    ,  -8362    ,      0    ,   4937    ,   2523    ,   1404.435,   -105.578,
  -1062.358,  -2712.592,  -3565.527,  -4939.989,  -5811.382,  -7279.346,  -8064.573,  -8643.674, -10330.656,      0    ,
   2079    ,   3103.000,   -845.601,    702.372,  -3801.207,  -1475.409,  -6384.424,  -3542.014,  -8841.221,  -5343.923,
 -10988.079,  -7332.499,      0    ,   3709    ,   2607.868,   1036.352,   -188.175,  -1732.476,  -2971.015,  -4078.330,
  -5097.258,  -6503.187,  -7210.066,  -7862.376,  -9681.326,      0    ,   4172    ,    381.290,   1814.465,  -2800.000,
   -838.641,  -5557.395,  -2883.465,  -7867.445,  -4724.051, -10268.611,  -6482.553,      0    ,   3570.002,   2010.000,
   1071.100,   -556.000,  -2147.169,  -3460.732,  -4430.964,  -5583.218,  -6384.599,  -7217.268,  -9016.171,      0    ,
   5089    ,   1340.000,   2866.000,  -1482.839,     31.925,  -4645.792,  -2272.019,  -7020.122,  -3970.032,  -9475.458,
  -5821.698, -12238.950,      0    ,   4432    ,   3076.646,   1993.500,    432.668,  -1012.771,  -2474.289,  -3647.414,
  -4821.329,  -5691.260,  -6417.679,  -8216.665,  -9305    ,      0    ,   2185.595,   3901.000,   -580.156,   1071.731,
  -3827.610,  -1310.295,  -6149.622,  -3175.993,  -8652.247,  -5204.713, -11555.509,  -7218.669,      0    ,   4084    ,
   3006.079,   1312.261,      2.467,  -1671.638,  -2864.323,  -3657.212,  -4910.159,  -5674.587,  -7456.187,  -8603.549,
  -9216.080,      0    ,   2733    ,   5057.847,    349.000,   2120.394,  -2787.888,   -522.376,  -5552.173,  -2065.584,
  -7874.849,  -4521.405, -10630.329,  -6640.273,      0    ,   4667    ,   3788    ,   2363.182,   1007.744,   -531.759,
  -1971.278,  -2903.471,  -3955.720,  -5024.054,  -6724.016,  -7813.358,  -8642.816,      0    ,   5868    ,   1254.097,
   3074.450,  -1953.820,    569.616,  -4442.000,  -1513.068,  -6991    ,  -3963    ,  -9817.539,  -6035.157,      0    ,
   4684    ,   3175.881,   2044.806,    313.977,  -1009.384,  -1890.315,  -3218.230,  -4310.059,  -6047.073,  -6996.039,
  -8170.977,  -8932.513,      0    ,   6493    ,   1939    ,   4295.120,  -1046.300,   1454.529,  -3516.341,   -764.764,
  -6138.823,  -3306.812,  -9029.950,  -5464.952, -11006.330,      0    ,   3949    ,   3160.680,   1142.275,    -56.628,
  -1075.761,  -2343.081,  -3585.637,  -5282.722,  -6321.658,  -7513.305,  -8291.917,  -9110.533,      0    ,   2711    ,
   5200    ,     96.600,   2228.772,  -2549.409,    -29.210,  -5246.454,  -2730.058,  -8171    ,  -5032    , -10293.044,
  -6434.994,      0    ,   3933    ,   2182.741,   1101.968,   -226.824,  -1569.861,  -2844.529,  -4441.973,  -5687.571,
  -6964.549,  -7585.169,  -8526.083,      0    ,   5737    ,   1159.106,   3209.016,  -1506.987,    686.836,  -4330.174,
  -2135.953,  -7351.608,  -4526.495,  -9569.906,  -5883.669,      0    ,   4807    ,   2957    ,   2156.155,    719.021,
   -599.939,  -2201.161,  -3591.166,  -5061.676,  -6328.964,  -7013.838,  -7820.295,      0    ,   6697    ,   1984    ,
   4085    ,   -323.694,   1372.852,  -3460.000,  -1444.758,  -6680.603,  -3896.563,  -8752    ,  -5491    ,      0    ,
   3917    ,   2990.167,   1704.580,    451.438,  -1486.996,  -2827.718,  -4434.219,  -5583.473,  -6390.551,  -7323.594,
  -8261.176,      0    ,   2832    ,   4990    ,    639.332,   2263.406,  -2456.040,   -797.019,  -5879.857,  -3416.357,
  -7966.347,  -4973.633, -10149.160,      0    ,   4657    ,   3844    ,   2660.000,   1261.844,   -483.793,  -1919.535,
  -3844.740,  -4889.775,  -5735.526,  -6716.876,  -7674.191,  -8239    ,      0    ,   3689    ,   5916    ,   1660.854,
   2992.517,  -1359.088,    -46.188,  -5199.446,  -2816.648,  -7333.321,  -4316.520,  -9366    ,  -5999    ,      0    ,
   4937    ,   3517    ,   2125.305,    492.056,   -974.840,  -3261.769,  -4213.477,  -5148.234,  -6003.247,  -7035.563,
  -7790.592,      0    ,   8234    ,   2728    ,   4040    ,   -344.578,    763.750,  -4463.795,  -2304.774,  -6465.369,
  -3892.456,  -8590.365,  -5439.745,      0    ,   5803    ,   3518    ,   1532.881,    -50.632,  -2705.559,  -3554.951,
  -4538.956,  -5256.401,  -6404.038,  -7148.645,      0    ,   4583    ,   6731    ,   1307.643,   1532.219,  -3757.306,
  -1839.397,  -5759.374,  -3313.590,  -7872.906,  -4808.299,  -9896    ,      0    ,   5682    ,   4545.959,   1418.121,
  -2397.410,  -2908.813,  -3918.395,  -4592.192,  -5790.376,  -6383.231,  -7607.205,      0    ,   7164    ,   3483.786,
   4998.882,  -2878.363,  -1400.559,  -4999.524,  -2814.402,  -6989.414,  -4380.807,  -9044.140,  -5914.866,      0    ,
   6176    ,   4993    ,   3976.448,    644.028,  -1892.580,  -3483.209,  -3953.536,  -5160.869,  -5622.054,  -6990.216,
  -7692.475,      0    ,   7694    ,   3881    ,   5481.933,     63.486,   1161.228,  -3981.039,  -2367.121,  -6272.031,
  -3792.820,  -8329.310,  -5270.012,      0    ,   5454    ,   4413.302,   1366.859,    573.967,   -785.297,  -2891.820,
  -4615.192,  -4971.841,  -6370.194,  -6704.492,      0    ,   4308    ,   5996    ,    569.773,   2252.026,  -1741.252,
     31.455,  -5143.461,  -3317.109,  -7476.596,  -4819.945,  -9515.582,      0    ,   5882    ,   4985.454,   2028.633,
   1422.804,    -73.918,   -881.278,  -2145.205,  -3910.614,  -5798.329,  -5961.751,  -7545.116,      0    ,   4713    ,
   6645    ,   1019.046,   3269.771,  -1090.204,    939.980,  -3361.181,  -1051.093,  -6351.635,  -4267.798,  -8773.112,
      0    ,   6297    ,   5495    ,   2768    ,   2188.710,    714.732,    -86.667,  -1486.616,  -2215.635,  -3496.913,
  -4890.588,  -6947.227,      0    ,   5142    ,   7238    ,   1606    ,   4090.007,   -473.543,   2004.292,  -2718.105,
   -320.022,  -4853.139,  -2154.103,  -7501.234,      0    ,   6073    ,   3510    ,   2844.634,   1489.400,    737.003,
   -656.018,  -1572.776,  -2816.043,  -3514.367,  -4851.837,  -5901    ,      0    ,   7638    ,   2237    ,   4857.232,
    259.722,   2881.239,  -1841.626,    408.015,  -4192.664,  -1522.704,  -6317.624,  -3227.527,      0    ,   3996    ,
   3601    ,   2284.165,   1566.295,    212.319,   -775.987,  -2175.268,  -2903.235,  -4067.413,  -4746.056,  -5990    ,
      0    ,   2850    ,   5555    ,    887.714,   3762.183,   -869.617,   1212.647,  -3473.524,   -925.248,  -5549    ,
  -2715    ,  -7377    ,      0    ,   4324    ,   2991.027,   2310.034,   1194.094,    314.364,  -1559.227,  -2414.892,
  -3438.023,  -4107    ,  -5362    ,      0    ,   6185    ,   1533.239,   4578.995,     24.269,   2028.185,  -2300.171,
   -581.568,  -4952    ,  -2067    ,  -6797    ,      0    ,   5058    ,   3651    ,   3038.267,   2005.761,   1149.170,
   -591.889,  -1559.420,  -2934.915,  -3517.855,  -4761    ,      0    ,   6856    ,   2199    ,   5265.917,    650.001,
   2967.769,  -1407.669,    241.053,  -3868.710,  -1851.390,  -6162    ,      0    ,   4136    ,   3861    ,   2713.342,
   1826.497,    355.709,   -671.957,  -2030.327,  -3036.988,  -4209.926,      0    ,   2934    ,   5867    ,   1206.641,
   3870.981,   -640.917,   1113.135,  -2836.119,  -1295.725,  -5448    ,      0    ,   4797    ,   4597    ,   3200.052,
   2506.324,   1328.372,     44.764,  -1025.691,  -2190.335,  -3540.009,      0    ,   6441    ,   1874.392,   4426.876,
     45.811,   2124.063,  -2152.124,   -297.909,  -4373.713,  -2483.014,      0    ,   5461    ,   3688.043,   3125.849,
   1850.300,   1111.431,   -311.606,  -1312.516,  -2568.930,  -3613.708,      0    ,   2537    ,   4994.268,    677.924,
   2974.139,  -1310.534,    559.782,  -3621.490,  -1697.515,  -5998    ,      0    ,   4389    ,   3847.920,   2453.636,
   1945.305,    391.587,   -381.556,  -1817.814,  -2660.484,  -4153    ,      0    ,   5489    ,   1342.535,   3705.706,
   -499.260,   1337.103,  -2750    ,  -1002    ,  -4905    ,  -3136    ,      0    ,   4711    ,   3014.315,   2574.390,
   1243.596,    569.792,  -1029.733,  -2102.128,  -3211    ,  -4029    ,      0    ,   2052.081,   4226.740,    274.088,
   2193.533,  -1809.846,   -393.213,  -4112    ,  -2263    ,  -6620    ,      0    ,   3847    ,   3339.406,   1728.853,
   1368.242,   -124.220,  -1138.955,  -2442.414,  -3384    ,  -4693    ,      0    ,   4965.795,    921.248,   2887.435,
   -932.923,    476.585,  -3139    ,  -1813    ,  -5687    ,      0    ,   4323    ,   2427.473,   2135.563,    518.590,
   -220.001,  -1478    ,  -2676    ,  -3941    ,  -4749    ,      0    ,   1732    ,   3584.926,   -147.351,   1291.516,
  -2258.343,  -1021.735,  -4771    ,  -3061    ,      0    ,   3270    ,   2763    ,   1261.461,    722.513,   -802.117,
  -1756.255,  -3103    ,  -3995    ,  -5310    ,      0    ,   4086    ,    398.614,   2190.573,  -1384.790,   -213.589,
  -3940    ,  -2325    ,  -6208    ,      0    ,   3493    ,   1935    ,   1305.000,     20.783,   -767.384,  -2330    ,
  -3293    ,  -4536    ,  -5263    ,      0    ,   1202    ,   2700.000,   -751.301,    664.458,  -2930    ,  -1651    ,
  -5415    ,  -3598    ,      0    ,   2524    ,   2121    ,    579.663,     -7.473,  -1507.591,  -2300    ,  -3757    ,
  -4616    ,      0    ,   3395    ,    -74.326,   1427.300,  -2261.850,   -763.659,  -4547    ,  -2939    ,      0    ,
   2770    ,   1277.287,    897.383,   -810.742,  -1571.296,  -2981    ,  -3820    ,  -5085    ,      0    ,    401    ,
   2377    ,  -1350.000,   -123.335,  -3810    ,  -2286    ,  -5926    ,      0    ,   1953    ,   1620    ,     43.621,
   -613.073,  -2474.564,  -3095    ,  -4264    ,      0    ,   3170    ,   -687    ,    841    ,  -3061    ,  -1598    ,
  -5250    ,  -3472    ,      0    ,   2352    ,    900.839,    124.609,  -1451    ,  -2344    ,  -3713    ,  -4549    ,
      0    ,     38.764,   1779.577,  -2055    ,   -846    ,  -4558    ,  -2932    ,      0    ,   1420.000,   1093.000,
   -377.219,  -1440.373,  -3012.904,  -3884    ,  -4878    ,      0    ,    521    ,   2500    ,  -1260.000,    477.674,
  -3693    ,  -2362    ,  -5866    ,      0    ,   1626    ,    287.630,   -334.272,  -1826    ,  -3185    ,  -4217    ,
  -4980    ,      0    ,   3051    ,   -650.123,   1087.800,  -2550    ,  -1270    ,  -5148    ,  -3325    ,      0    ,
    720    ,    289.620,  -1043.376,  -1962.896,  -3140.066,  -4383    ,  -5402    ,      0    ,   -147    ,   1700    ,
  -1969    ,   -367    ,  -3922.670,  -2476.086,  -6276    ,      0    ,    813    ,   -406.420,  -1253.272,  -2357    ,
  -3261    ,  -4341    ,      0    ,   2275    ,  -1261    ,    209    ,  -3304    ,  -1559    ,  -5456    ,  -3445    ,
      0    ,     80    ,   -486    ,  -1742    ,  -2509    ,  -3629    ,  -4568    ,      0    ,   -787    ,    940    ,
  -2665    ,   -872    ,  -4524    ,  -2875    ,  -6776    ,      0    ,    123    ,  -1106    ,  -1760    ,  -2928    ,
  -3756    ,  -5128    ,      0    ,   1525    ,  -2001    ,   -291    ,  -3863    ,  -2110    ,  -6176    ,      0    ,
   -600    ,  -1058    ,  -2324    ,  -3078    ,  -4306    ,  -5222    ,      0    ,  -1367    ,    300    ,  -3285    ,
  -1422    ,  -5274    ,  -3506    ,      0    ,   -457    ,  -1796    ,  -2310    ,  -3559    ,  -4545    ,  -5778    ,
      0    ,   1545    ,  -2661    ,   -881    ,  -4489    ,  -3029    ,  -6826    ,      0    ,   -630    ,  -1763    ,
  -2929    ,  -3886    ,  -5138    ,  -6128    ,      0    ,  -1587    ,    260    ,  -4005    ,  -2024    ,  -6319    ,
  -4498    ,      0    ,   -667    ,  -1667    ,  -3108    ,  -4721    ,  -5524    ,      0    ,    865    ,  -2735    ,
   -863    ,  -5624    ,  -3965    ,      0    ,  -1232    ,  -1783    ,  -3329    ,  -4846    ,      0    ,  -2211    ,
   -215    ,  -4575    ,  -2434    ,  -6756    ,      0    ,   -615    ,  -1340    ,  -2534    ,  -3876    ,  -4253    ,
      0    ,    227    ,  -3671    ,  -2017    ,  -5442    ,      0    ,   -930    ,  -2005    ,  -2993    ,  -3639    ,
      0    ,  -2924    ,  -1333    ,  -4945    ,  -2866    ,      0    ,  -1275    ,  -2465    ,  -2939    ,  -4259    ,
      0    ,   -683    ,  -4148    ,  -2480    ,  -5983    ,      0    ,  -1884    ,  -2445    ,  -3559    ,  -4106    ,
      0    ,  -3565    ,  -1871    ,  -5386    ,      0    ,  -1722    ,  -3154    ,  -3478    ,      0    ,  -1283    ,
  -4821    ,      0    ,  -2539    ,  -3075    ,      0    ,  -4258    ,      0    ,  -2431    ,  -3647    ,      0    ,
  -1888    ,      0    ,  -3161    ,  -3653    ,      0    ,  -4814    ,      0    ,  -2988    ,  -4224    ,      0    ,
  -2480    ,      0    ,  -3731    ,  -4153    ,      0    ,  -5329    ,      0    ,  -3492    ,  -4962    ,      0    ,
  -3222    ,      0    ,      0    

};

// Reduced Table of A for shorter index search.
//         The index in this table coincide with A-1
//         For each A value shortTable[A-1] has the index of 
//         the 1st occurrence in the indexArray[][]
//
const G4int G4NucleiPropertiesTableAME12::shortTable[G4NucleiPropertiesTableAME12::MaxA+1]=
{
    0,    2,    3,    6,    9,   13,   18,   23,   28,   33,
   39,   44,   50,   56,   62,   68,   75,   81,   88,   96,
  104,  113,  122,  131,  140,  149,  158,  167,  177,  186,
  196,  206,  216,  226,  237,  247,  258,  269,  280,  291,
  303,  314,  326,  338,  350,  363,  375,  388,  401,  414,
  426,  438,  450,  462,  474,  486,  499,  511,  524,  536,
  549,  562,  574,  586,  598,  610,  622,  634,  646,  658,
  670,  683,  695,  708,  721,  733,  746,  758,  771,  784,
  796,  809,  822,  835,  848,  862,  875,  889,  902,  916,
  930,  944,  958,  972,  986, 1001, 1015, 1030, 1045, 1060,
 1075, 1090, 1104, 1119, 1133, 1148, 1163, 1179, 1194, 1210,
 1225, 1240, 1256, 1271, 1287, 1303, 1319, 1335, 1350, 1366,
 1382, 1398, 1414, 1430, 1447, 1463, 1480, 1496, 1513, 1529,
 1546, 1562, 1578, 1595, 1611, 1628, 1644, 1660, 1677, 1693,
 1710, 1726, 1743, 1760, 1777, 1794, 1810, 1826, 1843, 1859,
 1876, 1893, 1909, 1926, 1942, 1959, 1975, 1992, 2008, 2025,
 2041, 2058, 2074, 2090, 2106, 2122, 2138, 2154, 2169, 2185,
 2200, 2216, 2231, 2246, 2260, 2274, 2288, 2302, 2316, 2330,
 2343, 2356, 2368, 2380, 2393, 2406, 2419, 2432, 2445, 2458,
 2470, 2483, 2496, 2509, 2522, 2534, 2546, 2558, 2570, 2582,
 2594, 2607, 2620, 2632, 2644, 2655, 2667, 2678, 2690, 2702,
 2714, 2725, 2737, 2749, 2761, 2773, 2785, 2797, 2809, 2821,
 2833, 2844, 2855, 2866, 2877, 2887, 2897, 2907, 2917, 2927,
 2937, 2947, 2957, 2967, 2977, 2987, 2996, 3006, 3015, 3025,
 3034, 3044, 3053, 3062, 3070, 3079, 3087, 3095, 3103, 3111,
 3118, 3126, 3134, 3142, 3150, 3158, 3166, 3173, 3181, 3188,
 3196, 3203, 3210, 3217, 3224, 3231, 3238, 3245, 3252, 3258,
 3264, 3269, 3275, 3281, 3286, 3291, 3296, 3301, 3306, 3311,
 3315, 3319, 3322, 3325, 3327, 3330, 3332, 3335, 3337, 3340,
 3342, 3345, 3347, 3350, 3352
};
