// Copyright 2019 the u-root Authors. All rights reserved
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package mtd

var vendors = []vendor{
	{names: []VendorName{"ZETTADEVICE", "Zetta"}, id: 0xBA},
	{names: []VendorName{"WINBOND"}, id: 0xDA},
	{names: []VendorName{"WINBOND", "NEX", "Nexcom", "Winbond (ex Nexcom) serial flashes,"}, id: 0xEF},
	// {names: []VendorName{"TI_OLD",}, id: 0x01, "TI chips from last century,"},
	{names: []VendorName{"TI", "Texas Instruments"}, id: 0x97},
	{names: []VendorName{"TENX", "Tenx", "Tenx Technologies"}, id: 0x7F7F5E},
	{names: []VendorName{"SYNCMOS", "MVC"}, id: 0x40},
	{names: []VendorName{"Micron", "ST", "SGS/Thomson", "Numonyx"}, id: 0x20},
	{names: []VendorName{"SST"}, id: 0xBF},
	{names: []VendorName{"SPANSION", "AMD"}, id: 0x01},
	{names: []VendorName{"SHARP"}, id: 0xB0},
	{names: []VendorName{"SANYO"}, id: 0x62},
	{names: []VendorName{"PMC", "Chingis Technology Corporation"}, id: 0x7F9D},
	{names: []VendorName{"NANTRONICS"}, id: 0x7F7F7F7F7F7FD5},
	{names: []VendorName{"MACRONIX", "MX", "Generalplus"}, id: 0xC2},
	// ISSI Integrated Silicon Solutions, disjoint overlap with PMC
	{names: []VendorName{"ISSI"}, id: 0xD5},
	{names: []VendorName{"INTEL"}, id: 0x89},
	{names: []VendorName{"IMT", "Integrated Memory Technologies"}, id: 0x7F1F},
	{names: []VendorName{"HYUNDAI"}, id: 0xAD},
	{names: []VendorName{"GIGADEVICE"}, id: 0xC8},
	{names: []VendorName{"FUJITSU"}, id: 0x04},
	{names: []VendorName{"FIDELIX"}, id: 0xF8},
	{names: []VendorName{"EXCEL", "Excel Semiconductor Inc. (ESI) resides in bank 5"}, id: 0x7F7F7F7F4A},
	{names: []VendorName{"EON", "EON Silicon Devices"}, id: 0x7F1C},
	{names: []VendorName{"ESMT", "Elite Semiconductor Memory Technology (ESMT) / EFST Elite Flash Storage"}, id: 0x8C},
	{names: []VendorName{"CATALYST"}, id: 0x31},
	{names: []VendorName{"BRIGHT", "Bright Microelectronics"}, id: 0xAD},
	{names: []VendorName{"Adesto", "ATMEL"}, id: 0x1F},
	{names: []VendorName{"AMIC"}, id: 0x7F37},
	{names: []VendorName{"ALLIANCE", "Alliance", "Alliance Semiconductor,"}, id: 0x52},
}

// Devices are the individual devices. Each device has a few properties
// that vary with the device, and a few that do not. For example, many
// devices have the same basic block sizes, erase patterns, and so on;
// they differ only in size. Hence we can save space if we have
// per-device data and per-family data.
// Flashrom supported a lot of oldjunk. For now, we're just
// going to worry about those things /dev/mtd can support.
// TODO: figure out what defining common data.
var (
	devices = []ChipDevice{
		{vendor: "ALLIANCE", devices: []ChipName{"AS29F002B"}, id: 0x34},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29F002T"}, id: 0xB0},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29F010"}, id: 0x04},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29F040"}, id: 0xA4},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29F200B"}, id: 0x57},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29F200T"}, id: 0x51},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29LV160B"}, id: 0x49},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29LV160T"}, id: 0xCA},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29LV400B"}, id: 0xBA},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29LV400T"}, id: 0xB9},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29LV800B"}, id: 0x5B},
		{vendor: "ALLIANCE", devices: []ChipName{"AS29LV800T"}, id: 0xDA},
		{vendor: "AMD", devices: []ChipName{"AM29DL400BT"}, id: 0x0C},
		{vendor: "AMD", devices: []ChipName{"AM29DL400BB"}, id: 0x0F},
		{vendor: "AMD", devices: []ChipName{"AM29DL800BT"}, id: 0x4A},
		{vendor: "AMD", devices: []ChipName{"AM29DL800BB"}, id: 0xCB},
		{vendor: "AMD", devices: []ChipName{"AM29F002BB"}, id: 0x34, remarks: "	/* Same as Am29F002NBB */"},
		{vendor: "AMD", devices: []ChipName{"AM29F002BT"}, id: 0xB0, remarks: "	/* Same as Am29F002NBT */"},
		{vendor: "AMD", devices: []ChipName{"AM29F004BB"}, id: 0x7B},
		{vendor: "AMD", devices: []ChipName{"AM29F004BT"}, id: 0x77},
		{vendor: "AMD", devices: []ChipName{"AM29F016D"}, id: 0xAD},
		{vendor: "AMD", devices: []ChipName{"AM29F010"}, id: 0x20, remarks: "	/* Same as Am29F010A and Am29F010B */"},
		{vendor: "AMD", devices: []ChipName{"AM29F040"}, id: 0xA4, remarks: "	/* Same as AM29F040B */"},
		{vendor: "AMD", devices: []ChipName{"AM29F080"}, id: 0xD5, remarks: "	/* Same as Am29F080B */"},
		{vendor: "AMD", devices: []ChipName{"AM29F200BB"}, id: 0x57},
		{vendor: "AMD", devices: []ChipName{"AM29F200BT"}, id: 0x51},
		{vendor: "AMD", devices: []ChipName{"AM29F400BB"}, id: 0xAB},
		{vendor: "AMD", devices: []ChipName{"AM29F400BT"}, id: 0x23},
		{vendor: "AMD", devices: []ChipName{"AM29F800BB"}, id: 0x58},
		{vendor: "AMD", devices: []ChipName{"AM29F800BT"}, id: 0xD6},
		{vendor: "AMD", devices: []ChipName{"AM29LV001BB"}, id: 0x6D},
		{vendor: "AMD", devices: []ChipName{"AM29LV001BT"}, id: 0xED},
		{vendor: "AMD", devices: []ChipName{"AM29LV010B"}, id: 0x6E, remarks: "	/* 1Mb, uniform */"},
		{vendor: "AMD", devices: []ChipName{"AM29LV002BB"}, id: 0xC2},
		{vendor: "AMD", devices: []ChipName{"AM29LV002BT"}, id: 0x40},
		{vendor: "AMD", devices: []ChipName{"AM29LV004BB"}, id: 0xB6},
		{vendor: "AMD", devices: []ChipName{"AM29LV004BT"}, id: 0xB5},
		{vendor: "AMD", devices: []ChipName{"AM29LV008BB"}, id: 0x37},
		{vendor: "AMD", devices: []ChipName{"AM29LV008BT"}, id: 0x3E},
		{vendor: "AMD", devices: []ChipName{"AM29LV040B"}, id: 0x4F},
		{vendor: "AMD", devices: []ChipName{"AM29LV080B"}, id: 0x38, remarks: "	/* Same as Am29LV081B */"},
		{vendor: "AMD", devices: []ChipName{"AM29LV200BB"}, id: 0xBF},
		{vendor: "AMD", devices: []ChipName{"AM29LV200BT"}, id: 0x3B},
		{vendor: "AMD", devices: []ChipName{"AM29LV800BB"}, id: 0x5B, remarks: "	/* Same as Am29LV800DB */"},
		{vendor: "AMD", devices: []ChipName{"AM29LV400BT"}, id: 0xB9},
		{vendor: "AMD", devices: []ChipName{"AM29LV400BB"}, id: 0xBA},
		{vendor: "AMD", devices: []ChipName{"AM29LV800BT"}, id: 0xDA, remarks: "	/* Same as Am29LV800DT */"},
		{vendor: "AMIC", devices: []ChipName{"A25L05PT"}, id: 0x2020},
		{vendor: "AMIC", devices: []ChipName{"A25L05PU"}, id: 0x2010},
		{vendor: "AMIC", devices: []ChipName{"A25L10PT"}, id: 0x2021},
		{vendor: "AMIC", devices: []ChipName{"A25L10PU"}, id: 0x2011},
		{vendor: "AMIC", devices: []ChipName{"A25L20PT"}, id: 0x2022},
		{vendor: "AMIC", devices: []ChipName{"A25L20PU"}, id: 0x2012},
		{vendor: "AMIC", devices: []ChipName{"A25L40PT"}, id: 0x2013, remarks: "	/* Datasheet says T and U have"},
		{vendor: "AMIC", devices: []ChipName{"A25L40PU"}, id: 0x2013},
		{vendor: "AMIC", devices: []ChipName{"A25L80P"}, id: 0x2014, remarks: "	/* Seems that no A25L80PT exists */"},
		{vendor: "AMIC", devices: []ChipName{"A25L16PT"}, id: 0x2025},
		{vendor: "AMIC", devices: []ChipName{"A25L16PU"}, id: 0x2015},
		{vendor: "AMIC", devices: []ChipName{"A25L512"}, id: 0x3010},
		{vendor: "AMIC", devices: []ChipName{"A25L010"}, id: 0x3011},
		{vendor: "AMIC", devices: []ChipName{"A25L020"}, id: 0x3012},
		{vendor: "AMIC", devices: []ChipName{"A25L040"}, id: 0x3013},
		{vendor: "AMIC", devices: []ChipName{"A25L080"}, id: 0x3014},
		{vendor: "AMIC", devices: []ChipName{"A25L016"}, id: 0x3015},
		{vendor: "AMIC", devices: []ChipName{"A25L032"}, id: 0x3016},
		{vendor: "AMIC", devices: []ChipName{"A25LQ16"}, id: 0x4015},
		{vendor: "AMIC", devices: []ChipName{"A25LQ032"}, id: 0x4016, remarks: "	/* Same as A25LQ32A, but the latter supports SFDP */"},
		{vendor: "AMIC", devices: []ChipName{"A25LQ64"}, id: 0x4017},
		{vendor: "AMIC", devices: []ChipName{"A29002B"}, id: 0x0, remarks: "d"},
		{vendor: "AMIC", devices: []ChipName{"A29002T"}, id: 0x8C, remarks: "	/* Same as A290021T */"},
		{vendor: "AMIC", devices: []ChipName{"A29040B"}, id: 0x86},
		{vendor: "AMIC", devices: []ChipName{"A29400T"}, id: 0xB0, remarks: "	/* Same as 294001T */"},
		{vendor: "AMIC", devices: []ChipName{"A29400U"}, id: 0x31, remarks: "	/* Same as A294001U */"},
		{vendor: "AMIC", devices: []ChipName{"A29800T"}, id: 0x0E},
		{vendor: "AMIC", devices: []ChipName{"A29800U"}, id: 0x8F},
		{vendor: "AMIC", devices: []ChipName{"A29L004T"}, id: 0x34, remarks: "	/* Same as A29L400T */"},
		{vendor: "AMIC", devices: []ChipName{"A29L004U"}, id: 0xB5, remarks: "	/* Same as A29L400U */"},
		{vendor: "AMIC", devices: []ChipName{"A29L008T"}, id: 0x1A, remarks: "	/* Same as A29L800T */"},
		{vendor: "AMIC", devices: []ChipName{"A29L008U"}, id: 0x9B, remarks: "	/* Same as A29L800U */"},
		{vendor: "AMIC", devices: []ChipName{"A29L040"}, id: 0x92},
		{vendor: "AMIC", devices: []ChipName{"A49LF040A"}, id: 0x9, remarks: "d"},
		{vendor: "ATMEL", devices: []ChipName{"AT25DF021"}, id: 0x4300},
		{vendor: "ATMEL", devices: []ChipName{"AT25DF021A"}, id: 0x4301},
		{vendor: "ATMEL", devices: []ChipName{"AT25DF041A"}, id: 0x4401},
		{vendor: "ATMEL", devices: []ChipName{"AT25DF081"}, id: 0x4502, remarks: "	/* EDI 0x00. AT25DL081 has same ID + EDI 0x0100 */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25DF081A"}, id: 0x4501, remarks: "	/* Yes, 81A has a lower number than 81 */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25DF161"}, id: 0x4602},
		{vendor: "ATMEL", devices: []ChipName{"AT25DF321"}, id: 0x4700, remarks: "	/* Same as 26DF321 */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25DF321A"}, id: 0x4701},
		{vendor: "ATMEL", devices: []ChipName{"AT25DF641"}, id: 0x4800},
		{vendor: "ATMEL", devices: []ChipName{"AT25DL161"}, id: 0x4603, remarks: "	/* EDI 0x0100 */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25DQ161"}, id: 0x8600, remarks: "	/* EDI 0x0100 */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25DQ321"}, id: 0x8700, remarks: "	/* EDI 0x0100 */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25F512"}, id: 0x60, remarks: "	/* Needs AT25F_RDID. ID from PCN and actual HW. Seems to be a relabeled AT25F1024. */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25F512A"}, id: 0x65, remarks: "	/* Needs AT25F_RDID */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25F512B"}, id: 0x6500},
		{vendor: "ATMEL", devices: []ChipName{"AT25F1024"}, id: 0x60, remarks: "	/* Needs AT25F_RDID */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25F2048"}, id: 0x63, remarks: "	/* Needs AT25F_RDID */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25F4096"}, id: 0x64, remarks: "	/* Needs AT25F_RDID */"},
		{vendor: "ATMEL", devices: []ChipName{"AT25FS010"}, id: 0x6601},
		{vendor: "ATMEL", devices: []ChipName{"AT25FS040"}, id: 0x6604},
		{vendor: "ATMEL", devices: []ChipName{"AT25SF041"}, id: 0x8401},
		{vendor: "ATMEL", devices: []ChipName{"AT25SF081"}, id: 0x8501},
		{vendor: "ATMEL", devices: []ChipName{"AT25SF161"}, id: 0x8601},
		{vendor: "ATMEL", devices: []ChipName{"AT25SL128A"}, id: 0x4218},
		{vendor: "ATMEL", devices: []ChipName{"AT26DF041"}, id: 0x4400},
		{vendor: "ATMEL", devices: []ChipName{"AT26DF081"}, id: 0x4500, remarks: "	/* guessed, no datasheet available */"},
		{vendor: "ATMEL", devices: []ChipName{"AT26DF081A"}, id: 0x4501},
		{vendor: "ATMEL", devices: []ChipName{"AT26DF161"}, id: 0x4600},
		{vendor: "ATMEL", devices: []ChipName{"AT26DF161A"}, id: 0x4601},
		{vendor: "ATMEL", devices: []ChipName{"AT26F004"}, id: 0x0400},
		{vendor: "ATMEL", devices: []ChipName{"AT29LV512"}, id: 0x3D},
		{vendor: "ATMEL", devices: []ChipName{"AT29LV010A"}, id: 0x35, remarks: "	/* Same as AT29BV010A, the latter works down to 2.7V */"},
		{vendor: "ATMEL", devices: []ChipName{"AT29LV020"}, id: 0xBA},
		{vendor: "ATMEL", devices: []ChipName{"AT29BV040A"}, id: 0xC4},
		{vendor: "ATMEL", devices: []ChipName{"AT29C040A"}, id: 0xA4},
		{vendor: "ATMEL", devices: []ChipName{"AT29C010A"}, id: 0xD5},
		{vendor: "ATMEL", devices: []ChipName{"AT29C020"}, id: 0xDA},
		{vendor: "ATMEL", devices: []ChipName{"AT29C512"}, id: 0x5D},
		{vendor: "ATMEL", devices: []ChipName{"AT45CS1282"}, id: 0x2920},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB011D"}, id: 0x2200},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB021D"}, id: 0x2300},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB041D"}, id: 0x2400},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB081D"}, id: 0x2500},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB081E	/*"}, id: 0x0100, remarks: "*/"},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB161D"}, id: 0x2600},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB161E	/*"}, id: 0x0100, remarks: "*/"},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB321C"}, id: 0x2700},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB321E	/*"}, id: 0x0100, remarks: "*/"},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB321D"}, id: 0x2701, remarks: "/* Buggy data sheet */"},
		{vendor: "ATMEL", devices: []ChipName{"AT45DB642D"}, id: 0x2800},
		{vendor: "ATMEL", devices: []ChipName{"AT49BV512"}, id: 0x03, remarks: "	/* Same as AT49F512 */"},
		{vendor: "ATMEL", devices: []ChipName{"AT49F001N"}, id: 0x05, remarks: "	/* Same as AT49F001 */"},
		{vendor: "ATMEL", devices: []ChipName{"AT49F001NT"}, id: 0x04, remarks: "	/* Same as AT49F001T */"},
		{vendor: "ATMEL", devices: []ChipName{"AT49F002N"}, id: 0x07, remarks: "	/* for AT49F002(N)  */"},
		{vendor: "ATMEL", devices: []ChipName{"AT49LH002"}, id: 0xE9},
		{vendor: "ATMEL", devices: []ChipName{"AT49LH00B4"}, id: 0xED},
		{vendor: "ATMEL", devices: []ChipName{"AT49LH004"}, id: 0xEE},
		{vendor: "ATMEL", devices: []ChipName{"AT49F002NT"}, id: 0x08, remarks: "	/* for AT49F002(N)T */"},
		{vendor: "ATMEL", devices: []ChipName{"AT49F010"}, id: 0x17, remarks: "	/* Same as AT49HF010 (some erroneous datasheets say 0x87), AT49BV010, AT49HBV010, AT49HLV010 */"},
		{vendor: "ATMEL", devices: []ChipName{"AT49F020"}, id: 0x0B},
		{vendor: "ATMEL", devices: []ChipName{"AT49F040"}, id: 0x13},
		{vendor: "ATMEL", devices: []ChipName{"AT49F080"}, id: 0x23},
		{vendor: "ATMEL", devices: []ChipName{"AT49F080T"}, id: 0x27},
		/* Bright Microelectronics has the same manufacturer ID as Hyundai... */
		{vendor: "BRIGHT", devices: []ChipName{"BM29F040"}, id: 0x40},
		{vendor: "BRIGHT", devices: []ChipName{"BM29F400B"}, id: 0xAB},
		{vendor: "BRIGHT", devices: []ChipName{"BM29F400T"}, id: 0xAD},
		{vendor: "CATALYST", devices: []ChipName{"CAT28F512"}, id: 0xB8},
		{vendor: "ESMT", devices: []ChipName{"F25L008A"}, id: 0x2014},
		{vendor: "ESMT", devices: []ChipName{"F25L32PA"}, id: 0x2016},
		{vendor: "ESMT", devices: []ChipName{"F25D08QA"}, id: 0x2534},
		{vendor: "ESMT", devices: []ChipName{"F25L16QA2S"}, id: 0x4015},
		{vendor: "ESMT", devices: []ChipName{"F25L32QA"}, id: 0x4016},
		{vendor: "ESMT", devices: []ChipName{"F25L32QA2S"}, id: 0x4116},
		{vendor: "ESMT", devices: []ChipName{"F25L64QA"}, id: 0x4117},
		{vendor: "ESMT", devices: []ChipName{"F25L128QA"}, id: 0x4118},
		{vendor: "ESMT", devices: []ChipName{"F49B002UA"}, id: 0x00},
		// /*
		//  * EN25 chips are SPI, first byte of device ID is memory type,
		//  * second byte of device ID is log(bitsize)-9.
		//  * Vendor and device ID of EN29 series are both prefixed with 0x7F, which
		//  * is the continuation code for IDs in bank 2.
		//  * Vendor ID of EN25 series is NOT prefixed with 0x7F, this results in
		//  * a collision with Mitsubishi. Mitsubishi once manufactured flash chips.
		//  * Let's hope they are not manufacturing SPI flash chips as well.
		//  */
		{vendor: "EON", devices: []ChipName{"EN25B05"}, id: 0x2010, remarks: "	/* Same as EN25P05, can be distinguished by RES/REMS: */"},
		{vendor: "EON", devices: []ChipName{"EN25P05"}, id: 0x05},
		{vendor: "EON", devices: []ChipName{"EN25B05T"}, id: 0x25},
		{vendor: "EON", devices: []ChipName{"EN25B05B"}, id: 0x95},
		{vendor: "EON", devices: []ChipName{"EN25B10"}, id: 0x2011, remarks: "	/* Same as EN25P10, can be distinguished by RES/REMS: */"},
		{vendor: "EON", devices: []ChipName{"EN25P10"}, id: 0x10},
		{vendor: "EON", devices: []ChipName{"EN25B10T"}, id: 0x40},
		{vendor: "EON", devices: []ChipName{"EN25B10B"}, id: 0x30},
		{vendor: "EON", devices: []ChipName{"EN25B20"}, id: 0x2012, remarks: "	/* Same as EN25P20, can be distinguished by RES/REMS: */"},
		{vendor: "EON", devices: []ChipName{"EN25P20"}, id: 0x11},
		{vendor: "EON", devices: []ChipName{"EN25B20T"}, id: 0x41},
		{vendor: "EON", devices: []ChipName{"EN25B20B"}, id: 0x31},
		{vendor: "EON", devices: []ChipName{"EN25B40"}, id: 0x2013, remarks: "	/* Same as EN25P40, can be distinguished by RES/REMS: */"},
		{vendor: "EON", devices: []ChipName{"EN25P40"}, id: 0x12},
		{vendor: "EON", devices: []ChipName{"EN25B40T"}, id: 0x42},
		{vendor: "EON", devices: []ChipName{"EN25B40B"}, id: 0x32},
		{vendor: "EON", devices: []ChipName{"EN25B80"}, id: 0x2014, remarks: "	/* Same as EN25P80, can be distinguished by RES/REMS: */"},
		{vendor: "EON", devices: []ChipName{"EN25P80"}, id: 0x13},
		{vendor: "EON", devices: []ChipName{"EN25B80T"}, id: 0x43},
		{vendor: "EON", devices: []ChipName{"EN25B80B"}, id: 0x33},
		{vendor: "EON", devices: []ChipName{"EN25B16"}, id: 0x2015, remarks: "	/* Same as EN25P16, can be distinguished by RES/REMS: */"},
		{vendor: "EON", devices: []ChipName{"EN25P16"}, id: 0x14},
		{vendor: "EON", devices: []ChipName{"EN25B16T"}, id: 0x44},
		{vendor: "EON", devices: []ChipName{"EN25B16B"}, id: 0x34},
		{vendor: "EON", devices: []ChipName{"EN25B32"}, id: 0x2016, remarks: "	/* Same as EN25P32, can be distinguished by RES/REMS: */"},
		{vendor: "EON", devices: []ChipName{"EN25P32"}, id: 0x15},
		{vendor: "EON", devices: []ChipName{"EN25B32T"}, id: 0x45},
		{vendor: "EON", devices: []ChipName{"EN25B32B"}, id: 0x35},
		{vendor: "EON", devices: []ChipName{"EN25B64"}, id: 0x2017, remarks: "	/* Same as EN25P64, can be distinguished by RES/REMS: */"},
		{vendor: "EON", devices: []ChipName{"EN25P64"}, id: 0x16},
		{vendor: "EON", devices: []ChipName{"EN25B64T"}, id: 0x46},
		{vendor: "EON", devices: []ChipName{"EN25B64B"}, id: 0x36},
		{vendor: "EON", devices: []ChipName{"EN25F05"}, id: 0x3110},
		{vendor: "EON", devices: []ChipName{"EN25F10"}, id: 0x3111},
		{vendor: "EON", devices: []ChipName{"EN25F20"}, id: 0x3112},
		{vendor: "EON", devices: []ChipName{"EN25F40"}, id: 0x3113},
		{vendor: "EON", devices: []ChipName{"EN25F80"}, id: 0x3114},
		{vendor: "EON", devices: []ChipName{"EN25F16"}, id: 0x3115},
		{vendor: "EON", devices: []ChipName{"EN25F32"}, id: 0x3116},
		{vendor: "EON", devices: []ChipName{"EN25F64"}, id: 0x3117},
		{vendor: "EON", devices: []ChipName{"EN25Q40"}, id: 0x3013},
		{vendor: "EON", devices: []ChipName{"EN25Q80"}, id: 0x3014},
		{vendor: "EON", devices: []ChipName{"EN25Q16"}, id: 0x3015, remarks: "	/* Same as EN25D16 */"},
		{vendor: "EON", devices: []ChipName{"EN25Q32"}, id: 0x3016, remarks: "	/* Same as EN25Q32A and EN25Q32B */"},
		{vendor: "EON", devices: []ChipName{"EN25Q64"}, id: 0x3017},
		{vendor: "EON", devices: []ChipName{"EN25Q128"}, id: 0x3018},
		{vendor: "EON", devices: []ChipName{"EN25QH16"}, id: 0x7015},
		{vendor: "EON", devices: []ChipName{"EN25QH32"}, id: 0x7016},
		{vendor: "EON", devices: []ChipName{"EN25QH64"}, id: 0x7017},
		{vendor: "EON", devices: []ChipName{"EN25QH128"}, id: 0x7018},
		{vendor: "EON", devices: []ChipName{"EN25QH256"}, id: 0x7019},
		{vendor: "EON", devices: []ChipName{"EN25S10"}, id: 0x3811},
		{vendor: "EON", devices: []ChipName{"EN25S20"}, id: 0x3812},
		{vendor: "EON", devices: []ChipName{"EN25S40"}, id: 0x3813},
		{vendor: "EON", devices: []ChipName{"EN25S80"}, id: 0x3814},
		{vendor: "EON", devices: []ChipName{"EN25S16"}, id: 0x3815},
		{vendor: "EON", devices: []ChipName{"EN25S32"}, id: 0x3816},
		{vendor: "EON", devices: []ChipName{"EN25S64"}, id: 0x3817},
		{vendor: "EON", devices: []ChipName{"EN25T80"}, id: 0x5114},
		{vendor: "EON", devices: []ChipName{"EN25T16"}, id: 0x5115},
		{vendor: "EON", devices: []ChipName{"EN29F512"}, id: 0x7F21},
		{vendor: "EON", devices: []ChipName{"EN29F010"}, id: 0x20},
		{vendor: "EON", devices: []ChipName{"EN29F040A"}, id: 0x7F04},
		{vendor: "EON", devices: []ChipName{"EN29LV010"}, id: 0x7F6E},
		{vendor: "EON", devices: []ChipName{"EN29LV040"}, id: 0x4F, remarks: "	/* Same as EN29LV040A */"},
		{vendor: "EON", devices: []ChipName{"EN29LV640B"}, id: 0xCB},
		{vendor: "EON", devices: []ChipName{"EN29LV640T"}, id: 0xC9},
		{vendor: "EON", devices: []ChipName{"EN29LV640U"}, id: 0x7E},
		{vendor: "EON", devices: []ChipName{"EN29F002T"}, id: 0x7F92, remarks: "	/* Same as EN29F002A */"},
		{vendor: "EON", devices: []ChipName{"EN29F002B"}, id: 0x7F97, remarks: "	/* Same as EN29F002AN */"},
		{vendor: "EON", devices: []ChipName{"EN29GL064HL"}, id: 0x7E0C01, remarks: "	/* Uniform Sectors, WP protects Top OR Bottom sector */"},
		{vendor: "EON", devices: []ChipName{"EN29GL064T"}, id: 0x7E1001, remarks: "	/* Same ID as EN29GL064AT */"},
		{vendor: "EON", devices: []ChipName{"EN29GL064B"}, id: 0x7E1000, remarks: "	/* Same ID as EN29GL064AB */"},
		{vendor: "EON", devices: []ChipName{"EN29GL128HL"}, id: 0x7F2101, remarks: "	/* Uniform Sectors, WP protects Top OR Bottom sector */"},
		{vendor: "EON", devices: []ChipName{"EN29GL256HL"}, id: 0x7F2201, remarks: "	/* Uniform Sectors, WP protects Top OR Bottom sector */"},
		{vendor: "EXCEL", devices: []ChipName{"ES25P40"}, id: 0x2013},
		{vendor: "EXCEL", devices: []ChipName{"ES25P80"}, id: 0x2014},
		{vendor: "EXCEL", devices: []ChipName{"ES25P16"}, id: 0x2015},
		{vendor: "FIDELIX", devices: []ChipName{"FM25M16"}, id: 0x4215},
		{vendor: "FIDELIX", devices: []ChipName{"FM25M32"}, id: 0x4216},
		{vendor: "FIDELIX", devices: []ChipName{"FM25M64"}, id: 0x4217},
		{vendor: "FIDELIX", devices: []ChipName{"FM25Q08"}, id: 0x3214},
		{vendor: "FIDELIX", devices: []ChipName{"FM25Q16"}, id: 0x3215, remarks: "	/* Same as FM25S16 (which is apparently single I/O only) */"},
		{vendor: "FIDELIX", devices: []ChipName{"FM25Q32"}, id: 0x3216},
		{vendor: "FIDELIX", devices: []ChipName{"FM25Q64"}, id: 0x3217},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29DL400BC"}, id: 0x0F},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29DL400TC"}, id: 0x0C},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29DL800BA"}, id: 0xCB},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29DL800TA"}, id: 0x4A},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F002BC"}, id: 0x34},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F002TC"}, id: 0xB0},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F004BC"}, id: 0x7B},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F004TC"}, id: 0x77},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F040C"}, id: 0xA4},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F080A"}, id: 0xD5},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F200BC"}, id: 0x57},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F200TC"}, id: 0x51},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F400BC"}, id: 0xAB},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F400TC"}, id: 0x23},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F800BA"}, id: 0x58},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29F800TA"}, id: 0xD6},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV002BC"}, id: 0xC2},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV002TC"}, id: 0x40},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV004BC"}, id: 0xB6},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV004TC"}, id: 0xB5},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV008BA"}, id: 0x37},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV008TA"}, id: 0x3E},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV080A"}, id: 0x38},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV200BC"}, id: 0xBF},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV200TC"}, id: 0x3B},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV400BC"}, id: 0xBA},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV400TC"}, id: 0xB9},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV800BA"}, id: 0x5B, remarks: "	/* Same as MBM29LV800BE */"},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV800TA"}, id: 0xDA, remarks: "	/* Same as MBM29LV800TE */"},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV160BE"}, id: 0x49, remarks: "	/* 16 b mode 0x2249 */"},
		{vendor: "FUJITSU", devices: []ChipName{"MBM29LV160TE"}, id: 0xC4, remarks: "	/* 16 b mode 0x22C4 */"},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25T80"}, id: 0x3114},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25Q512"}, id: 0x4010},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25Q10"}, id: 0x4011},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25Q20"}, id: 0x4012, remarks: "	/* Same as GD25QB */"},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25Q40"}, id: 0x4013, remarks: "	/* Same as GD25QB */"},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25Q80"}, id: 0x4014, remarks: "	/* Same as GD25Q80B (which has OTP) */"},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25Q16"}, id: 0x4015, remarks: "	/* Same as GD25Q16B (which has OTP) */"},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25Q32"}, id: 0x4016, remarks: "	/* Same as GD25Q32B */"},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25Q64"}, id: 0x4017, remarks: "	/* Same as GD25Q64B */"},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25Q128"}, id: 0x4018, remarks: "	/* GD25Q128B and GD25Q128C only, can be distinguished by SFDP */"},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25VQ21B"}, id: 0x4212},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25VQ41B"}, id: 0x4213, remarks: "/* Same as GD25VQ40C, can be distinguished by SFDP */"},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25VQ80C"}, id: 0x4214},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25VQ16C"}, id: 0x4215},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25LQ40"}, id: 0x6013},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25LQ80"}, id: 0x6014},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25LQ16"}, id: 0x6015},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25LQ32"}, id: 0x6016},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25LQ64"}, id: 0x6017, remarks: "	/* Same as GD25LQ64B (which is faster) */"},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD25LQ128"}, id: 0x6018},
		{vendor: "GIGADEVICE", devices: []ChipName{"GD29GL064CAB"}, id: 0x7E0601},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29F400T"}, id: 0x23, remarks: "	/* Same as HY29F400AT */"},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29F800B"}, id: 0x58, remarks: "	/* Same as HY29F800AB */"},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29LV800B"}, id: 0x5B},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29F040A"}, id: 0xA4},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29F400B"}, id: 0xAB, remarks: "	/* Same as HY29F400AB */"},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29F002B"}, id: 0x34},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29F002T"}, id: 0xB0},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29LV400T"}, id: 0xB9},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29LV400B"}, id: 0xBA},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29F080"}, id: 0xD5},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29F800T"}, id: 0xD6, remarks: "	/* Same as HY29F800AT */"},
		{vendor: "HYUNDAI", devices: []ChipName{"HY29LV800T"}, id: 0xDA},
		{vendor: "IMT", devices: []ChipName{"IM29F004B"}, id: 0xAE},
		{vendor: "IMT", devices: []ChipName{"IM29F004T"}, id: 0xAF},
		{vendor: "INTEL", devices: []ChipName{"28F320J5"}, id: 0x14},
		{vendor: "INTEL", devices: []ChipName{"28F640J5"}, id: 0x15},
		{vendor: "INTEL", devices: []ChipName{"28F320J3"}, id: 0x16},
		{vendor: "INTEL", devices: []ChipName{"28F640J3"}, id: 0x17},
		{vendor: "INTEL", devices: []ChipName{"28F128J3"}, id: 0x18},
		{vendor: "INTEL", devices: []ChipName{"28F256J3"}, id: 0x1D},
		{vendor: "INTEL", devices: []ChipName{"28F400T"}, id: 0x70, remarks: "	/* 28F400BV/BX/CE/CV-T */"},
		{vendor: "INTEL", devices: []ChipName{"28F400B"}, id: 0x71, remarks: "	/* 28F400BV/BX/CE/CV-B */"},
		{vendor: "INTEL", devices: []ChipName{"28F200T"}, id: 0x74, remarks: "	/* 28F200BL/BV/BX/CV-T */"},
		{vendor: "INTEL", devices: []ChipName{"28F200B"}, id: 0x75, remarks: "	/* 28F200BL/BV/BX/CV-B */"},
		{vendor: "INTEL", devices: []ChipName{"28F004T"}, id: 0x78, remarks: "	/* 28F004B5/BE/BV/BX-T */"},
		{vendor: "INTEL", devices: []ChipName{"28F004B"}, id: 0x79, remarks: "	/* 28F004B5/BE/BV/BX-B */"},
		{vendor: "INTEL", devices: []ChipName{"28F002T"}, id: 0x7C, remarks: "	/* 28F002BC/BL/BV/BX-T */"},
		{vendor: "INTEL", devices: []ChipName{"28F002B"}, id: 0x7D, remarks: "	/* 28F002BL/BV/BX-B */"},
		{vendor: "INTEL", devices: []ChipName{"28F001T"}, id: 0x94, remarks: "	/* 28F001BN/BX-T */"},
		{vendor: "INTEL", devices: []ChipName{"28F001B"}, id: 0x95, remarks: "	/* 28F001BN/BX-B */"},
		{vendor: "INTEL", devices: []ChipName{"28F008T"}, id: 0x98, remarks: "	/* 28F008BE/BV-T */"},
		{vendor: "INTEL", devices: []ChipName{"28F008B"}, id: 0x99, remarks: "	/* 28F008BE/BV-B */"},
		{vendor: "INTEL", devices: []ChipName{"28F800T"}, id: 0x9C, remarks: "	/* 28F800B5/BV/CE/CV-T */"},
		{vendor: "INTEL", devices: []ChipName{"28F800B"}, id: 0x9D, remarks: "	/* 28F800B5/BV/CE/CV-B */"},
		{vendor: "INTEL", devices: []ChipName{"28F016SV"}, id: 0xA0, remarks: "	/* 28F016SA/SV */"},
		{vendor: "INTEL", devices: []ChipName{"28F008SA"}, id: 0xA2},
		{vendor: "INTEL", devices: []ChipName{"28F008S3"}, id: 0xA6, remarks: "	/* 28F008S3/S5/SC */"},
		{vendor: "INTEL", devices: []ChipName{"28F004S3"}, id: 0xA7, remarks: "	/* 28F008S3/S5/SC */"},
		{vendor: "INTEL", devices: []ChipName{"28F016XS"}, id: 0xA8},
		{vendor: "INTEL", devices: []ChipName{"28F016S3"}, id: 0xAA, remarks: "	/* 28F016S3/S5/SC */"},
		{vendor: "INTEL", devices: []ChipName{"82802AC"}, id: 0xAC},
		{vendor: "INTEL", devices: []ChipName{"82802AB"}, id: 0xAD},
		{vendor: "INTEL", devices: []ChipName{"28F010"}, id: 0xB4},
		{vendor: "INTEL", devices: []ChipName{"28F512"}, id: 0xB8},
		{vendor: "INTEL", devices: []ChipName{"28F256A"}, id: 0xB9},
		{vendor: "INTEL", devices: []ChipName{"28F020"}, id: 0xBD},
		{vendor: "INTEL", devices: []ChipName{"28F016B3T"}, id: 0xD0, remarks: "	/* 28F016B3-T */"},
		{vendor: "INTEL", devices: []ChipName{"28F016B3B"}, id: 0xD1, remarks: "	/* 28F016B3-B */"},
		{vendor: "INTEL", devices: []ChipName{"28F008B3T"}, id: 0xD2, remarks: "	/* 28F008B3-T */"},
		{vendor: "INTEL", devices: []ChipName{"28F008B3B"}, id: 0xD3, remarks: "	/* 28F008B3-B */"},
		{vendor: "INTEL", devices: []ChipName{"28F004B3T"}, id: 0xD4, remarks: "	/* 28F004B3-T */"},
		{vendor: "INTEL", devices: []ChipName{"28F004B3B"}, id: 0xD5, remarks: "	/* 28F004B3-B */"},
		{vendor: "INTEL", devices: []ChipName{"25F160S33B8"}, id: 0x8911, remarks: "	/* Same as 25F016S33B8 */"},
		{vendor: "INTEL", devices: []ChipName{"25F320S33B8"}, id: 0x8912},
		{vendor: "INTEL", devices: []ChipName{"25F640S33B8"}, id: 0x8913},
		{vendor: "INTEL", devices: []ChipName{"25F160S33T8"}, id: 0x8915, remarks: "	/* Same as 25F016S33T8 */"},
		{vendor: "INTEL", devices: []ChipName{"25F320S33T8"}, id: 0x8916},
		{vendor: "INTEL", devices: []ChipName{"25F640S33T8"}, id: 0x8917},
		{vendor: "SHARP", devices: []ChipName{"LH28F008SA"}, id: 0xA2, remarks: "	/* Sharp chip, Intel Vendor ID */"},
		{vendor: "SHARP", devices: []ChipName{"LH28F008SC"}, id: 0xA6, remarks: "	/* Sharp chip, Intel Vendor ID */"},
		{vendor: "ISSI", devices: []ChipName{"IS25LP064"}, id: 0x6017},
		{vendor: "ISSI", devices: []ChipName{"IS25LP128"}, id: 0x6018},
		{vendor: "ISSI", devices: []ChipName{"IS25LP256"}, id: 0x6019},
		{vendor: "ISSI", devices: []ChipName{"IS25WP032"}, id: 0x7016},
		{vendor: "ISSI", devices: []ChipName{"IS25WP064"}, id: 0x7017},
		{vendor: "ISSI", devices: []ChipName{"IS25WP128"}, id: 0x7018},
		{vendor: "ISSI", devices: []ChipName{"IS25WP256"}, id: 0x7019},
		{vendor: "ISSI", devices: []ChipName{"PMC_"}, id: 0xF9},
		{vendor: "ISSI", devices: []ChipName{"PMC_"}, id: 0xF6},
		{vendor: "ISSI", devices: []ChipName{"PMC_"}, id: 0x7E1000},
		{vendor: "ISSI", devices: []ChipName{"PMC_"}, id: 0x7E1001},
		{vendor: "ISSI", devices: []ChipName{"PMC_"}, id: 0x7E0C01},
		{vendor: "ISSI", devices: []ChipName{"PMC_"}, id: 0x7E2101},
		{vendor: "ISSI", devices: []ChipName{"PMC_"}, id: 0x7E2201},
		/* Mask ROMs */
		{vendor: "MACRONIX", devices: []ChipName{"MX23L1654"}, id: 0x0515},
		{vendor: "MACRONIX", devices: []ChipName{"MX23L3254"}, id: 0x0516},
		{vendor: "MACRONIX", devices: []ChipName{"MX23L6454"}, id: 0x0517},
		{vendor: "MACRONIX", devices: []ChipName{"MX23L12854"}, id: 0x0518},
		/* MX25 chips are SPI, first byte of device ID is memory type,
		 * second byte of device ID is log(bitsize)-9.
		 * Generalplus SPI chips seem to be compatible with Macronix
		 * and use the same set of IDs. */
		{vendor: "MACRONIX", devices: []ChipName{"MX25L512"}, id: 0x2010, remarks: "	/* Same as MX25L512E, MX25V512, MX25V512C */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L1005"}, id: 0x2011, remarks: "	/* Same as MX25L1005C, MX25L1006E */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L2005"}, id: 0x2012, remarks: "	/* Same as MX25L2005C, MX25L2006E */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L4005"}, id: 0x2013, remarks: "	/* Same as MX25L4005A, MX25L4005C, MX25L4006E */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L8005"}, id: 0x2014, remarks: "	/* Same as MX25V8005, MX25L8006E, MX25L8008E, FIXME: MX25L8073E (4k 0x20) */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L1605"}, id: 0x2015, remarks: "	/* MX25L1605 (64k 0x20); MX25L1605A/MX25L1606E/MX25L1608E (4k 0x20, 64k 0x52); MX25L1605D/MX25L1608D/MX25L1673E (4k 0x20) */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L3205"}, id: 0x2016, remarks: "	/* MX25L3205, MX25L3205A (64k 0x20); MX25L3205D/MX25L3208D (4k 0x20); MX25L3206E/MX25L3208E (4k 0x20, 64k 0x52); MX25L3273E (4k 0x20, 32k 0x52) */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L6405"}, id: 0x2017, remarks: "	/* MX25L6405, MX25L6405D (64k 0x20); MX25L6406E/MX25L6408E (4k 0x20); MX25L6436E/MX25L6445E/MX25L6465E/MX25L6473E (4k 0x20, 32k 0x52) */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L12805D"}, id: 0x2018, remarks: "	/* MX25L12805D (no 32k); MX25L12865E, MX25L12835F, MX25L12845E (32k 0x52) */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L25635F"}, id: 0x2019, remarks: "	/* Same as MX25L25639F, but the latter seems to not support REMS */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L1635D"}, id: 0x2415},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L1635E"}, id: 0x2515, remarks: "	/* MX25L1635{E} */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX66L51235F"}, id: 0x201, remarks: "a	/* MX66L51235F, MX25L51245G */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25U8032E"}, id: 0x2534},
		{vendor: "MACRONIX", devices: []ChipName{"MX25U1635E"}, id: 0x2535},
		{vendor: "MACRONIX", devices: []ChipName{"MX25U3235E"}, id: 0x2536, remarks: "	/* Same as MX25U6435F */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25U6435E"}, id: 0x2537, remarks: "	/* Same as MX25U6435F */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25U12835E"}, id: 0x2538, remarks: "	/* Same as MX25U12835F */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25U25635F"}, id: 0x2539},
		{vendor: "MACRONIX", devices: []ChipName{"MX25U51245G"}, id: 0x253, remarks: "a"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L3235D"}, id: 0x5E16, remarks: "	/* MX25L3225D/MX25L3235D/MX25L3237D */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX25L6495F"}, id: 0x9517},
		{vendor: "MACRONIX", devices: []ChipName{"MX25R6435F"}, id: 0x2817},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F001B"}, id: 0x19},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F001T"}, id: 0x18},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F002B"}, id: 0x34, remarks: "	/* Same as MX29F002NB; N has reset pin n/c. */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F002T"}, id: 0xB0, remarks: "	/* Same as MX29F002NT; N has reset pin n/c. */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F004B"}, id: 0x46},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F004T"}, id: 0x45},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F022B"}, id: 0x37, remarks: "	/* Same as MX29F022NB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F022T"}, id: 0x36, remarks: "	/* Same as MX29F022NT */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F040"}, id: 0xA4, remarks: "	/* Same as MX29F040C */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F080"}, id: 0xD5},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F200B"}, id: 0x57, remarks: "	/* Same as MX29F200CB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F200T"}, id: 0x51, remarks: "	/* Same as MX29F200CT */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F400B"}, id: 0xAB, remarks: "	/* Same as MX29F400CB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F400T"}, id: 0x23, remarks: "	/* Same as MX29F400CT */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F800B"}, id: 0x58},
		{vendor: "MACRONIX", devices: []ChipName{"MX29F800T"}, id: 0xD6},
		{vendor: "MACRONIX", devices: []ChipName{"MX29GL320EB"}, id: 0x7E1A00},
		{vendor: "MACRONIX", devices: []ChipName{"MX29GL320ET"}, id: 0x7E1A01},
		{vendor: "MACRONIX", devices: []ChipName{"MX29GL320EHL"}, id: 0x7E1D00},
		{vendor: "MACRONIX", devices: []ChipName{"MX29GL640EB"}, id: 0x7E1000},
		{vendor: "MACRONIX", devices: []ChipName{"MX29GL640ET"}, id: 0x7E1001},
		{vendor: "MACRONIX", devices: []ChipName{"MX29GL640EHL"}, id: 0x7E0C01},
		{vendor: "MACRONIX", devices: []ChipName{"MX29GL128F"}, id: 0x7E2101, remarks: "/* Same as MX29GL128E */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29GL256F"}, id: 0x7E2201, remarks: "/* Same as MX29GL256E */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29GL512F"}, id: 0x7E2301},
		{vendor: "MACRONIX", devices: []ChipName{"MX68GL1G0F"}, id: 0x7E2801},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV002CB"}, id: 0x5A},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV002CT"}, id: 0x59},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV004B"}, id: 0xB6, remarks: "	/* Same as MX29LV004CB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV004T"}, id: 0xB5, remarks: "	/* Same as MX29LV004CT */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV008B"}, id: 0x37, remarks: "	/* Same as MX29LV008CB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV008T"}, id: 0x3E, remarks: "	/* Same as MX29LV008CT */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV040"}, id: 0x4F, remarks: "	/* Same as MX29LV040C */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV081"}, id: 0x38},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV128DB"}, id: 0x7A},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV128DT"}, id: 0x7E},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV160DB"}, id: 0x49, remarks: "	/* Same as MX29LV161DB/MX29LV160CB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV160DT"}, id: 0xC4, remarks: "	/* Same as MX29LV161DT/MX29LV160CT */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV320DB"}, id: 0xA8, remarks: "	/* Same as MX29LV321DB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV320DT"}, id: 0xA7, remarks: "	/* Same as MX29LV321DT */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV400B"}, id: 0xBA, remarks: "	/* Same as MX29LV400CB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV400T"}, id: 0xB9, remarks: "	/* Same as MX29LV400CT */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV640DB"}, id: 0xCB, remarks: "	/* Same as MX29LV640EB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV640DT"}, id: 0xC9, remarks: "	/* Same as MX29LV640ET */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV800B"}, id: 0x5B, remarks: "	/* Same as MX29LV800CB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29LV800T"}, id: 0xDA, remarks: "	/* Same as MX29LV800CT */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29SL402CB"}, id: 0xF1},
		{vendor: "MACRONIX", devices: []ChipName{"MX29SL402CT"}, id: 0x70},
		{vendor: "MACRONIX", devices: []ChipName{"MX29SL800CB"}, id: 0x6B, remarks: "	/* Same as MX29SL802CB */"},
		{vendor: "MACRONIX", devices: []ChipName{"MX29SL800CT"}, id: 0xEA, remarks: "	/* Same as MX29SL802CT */"},
		/* Nantronics Semiconductors is listed in JEP106AJ in bank 7, so it should have 6 continuation codes in front
		 * of the manufacturer ID of 0xD5. httpName://www.nantronicssemi.com */
		{vendor: "NANTRONICS", devices: []ChipName{"N25S10"}, id: 0x3011},
		{vendor: "NANTRONICS", devices: []ChipName{"N25S20"}, id: 0x3012},
		{vendor: "NANTRONICS", devices: []ChipName{"N25S40"}, id: 0x3013},
		{vendor: "NANTRONICS", devices: []ChipName{"N25S80"}, id: 0x3014},
		{vendor: "NANTRONICS", devices: []ChipName{"N25S16"}, id: 0x3015},
		/*
		 * Programmable Micro Corp is listed in JEP106W in bank 2, so it should
		 * have a 0x7F continuation code prefix.
		 * Apparently PMC was renamed to "Chingis Technology Corporation"httpName://www.chingistek.com which is now a
		 * subsidiary of ISSI. They continue to use the PMC manufacturer ID (instead of ISSI's) nevertheless, even for
		 * new chips with IS* model numbers.
		 */
		{vendor: "PMC", devices: []ChipName{"PM25LD256C"}, id: 0x2F},
		{vendor: "PMC", devices: []ChipName{"PM25LD512"}, id: 0x20, remarks: "	/* Same as Pm25LD512C, but the latter has more locking options. */"},
		{vendor: "PMC", devices: []ChipName{"PM25LD010"}, id: 0x21, remarks: "	/* Same as Pm25LD010C, but the latter has more locking options. */"},
		{vendor: "PMC", devices: []ChipName{"PM25LD020"}, id: 0x22, remarks: "	/* Same as Pm25LD020C, but the latter has more locking options. */"},
		{vendor: "PMC", devices: []ChipName{"PM25LQ020"}, id: 0x42},
		{vendor: "PMC", devices: []ChipName{"PM25LQ040"}, id: 0x43},
		{vendor: "PMC", devices: []ChipName{"PM25LQ080"}, id: 0x44},
		{vendor: "PMC", devices: []ChipName{"PM25LQ016"}, id: 0x45},
		{vendor: "PMC", devices: []ChipName{"PM25LQ032C"}, id: 0x46},
		{vendor: "PMC", devices: []ChipName{"PM25LV512"}, id: 0x7B, remarks: "	/* Same as Pm25LV512A */"},
		{vendor: "PMC", devices: []ChipName{"PM25LV010"}, id: 0x7C, remarks: "	/* Same as Pm25LV010A, but the former does not support RDID but RES3 only. */"},
		{vendor: "PMC", devices: []ChipName{"PM25LV020"}, id: 0x7D},
		{vendor: "PMC", devices: []ChipName{"PM25LV040"}, id: 0x7E, remarks: "	/* Same as PM25LD040(C), but the latter supports more features. */"},
		{vendor: "PMC", devices: []ChipName{"PM25LV080B"}, id: 0x13},
		{vendor: "PMC", devices: []ChipName{"PM25LV016B"}, id: 0x14},
		{vendor: "PMC", devices: []ChipName{"PM29F002T"}, id: 0x1D},
		{vendor: "PMC", devices: []ChipName{"PM29F002B"}, id: 0x2D},
		{vendor: "PMC", devices: []ChipName{"PM39LV512"}, id: 0x1B, remarks: "	/* Same as IS39LV512 */"},
		{vendor: "PMC", devices: []ChipName{"PM39F010"}, id: 0x1C, remarks: "	/* Same as Pm39LV010, IS39LV010 */"},
		{vendor: "PMC", devices: []ChipName{"PM39LV020"}, id: 0x3D},
		{vendor: "PMC", devices: []ChipName{"PM39LV040"}, id: 0x3E, remarks: "	/* Same as IS39LV040 */"},
		{vendor: "PMC", devices: []ChipName{"PM39F020"}, id: 0x4D},
		{vendor: "PMC", devices: []ChipName{"PM39F040"}, id: 0x4E},
		{vendor: "PMC", devices: []ChipName{"PM49FL002"}, id: 0x6D},
		{vendor: "PMC", devices: []ChipName{"PM49FL004"}, id: 0x6E},
		/*
		 * The Sanyo chip found so far uses SPI, first byte is manufacture code,
		 * second byte is the device code,
		 * third byte is a placeholder byte.
		 */
		{vendor: "SANYO", devices: []ChipName{"LE25FW203A"}, id: 0x1600},
		{vendor: "SANYO", devices: []ChipName{"LE25FW403A"}, id: 0x1100},
		{vendor: "SANYO", devices: []ChipName{"LE25FW106"}, id: 0x15},
		{vendor: "SANYO", devices: []ChipName{"LE25FW406"}, id: 0x07, remarks: "	/* RES2 */"},
		{vendor: "SANYO", devices: []ChipName{"LE25FW418A"}, id: 0x10, remarks: "	/* RES2 and some weird 1 byte RDID variant */"},
		{vendor: "SANYO", devices: []ChipName{"LE25FW406A"}, id: 0x1A, remarks: "	/* RES2, no datasheet */"},
		{vendor: "SANYO", devices: []ChipName{"LE25FU106B"}, id: 0x1D},
		{vendor: "SANYO", devices: []ChipName{"LE25FU206"}, id: 0x44},
		{vendor: "SANYO", devices: []ChipName{"LE25FU206A"}, id: 0x0612},
		{vendor: "SANYO", devices: []ChipName{"LE25FU406B"}, id: 0x1E, remarks: "	/* LE25FW418A without HD_READ mode option variant */"},
		{vendor: "SANYO", devices: []ChipName{"LE25FU406C"}, id: 0x0613, remarks: "	/* Also known as LE25U40CMC apparently */"},
		{vendor: "SANYO", devices: []ChipName{"LE25FW806"}, id: 0x26, remarks: "	/* RES2 and some weird 1 byte RDID variant */"},
		{vendor: "SANYO", devices: []ChipName{"LE25FW808"}, id: 0x20, remarks: "	/* RES2 and some weird 1 byte RDID variant */"},
		{vendor: "SHARP_LH28F008BJ_", devices: []ChipName{"PT"}, id: 0xEC},
		{vendor: "SHARP_LH28F008BJ_", devices: []ChipName{"PB"}, id: 0xED},
		{vendor: "SHARP_LH28F800BV_", devices: []ChipName{"BTL"}, id: 0x4B},
		{vendor: "SHARP_LH28F800BV_", devices: []ChipName{"BV"}, id: 0x4D},
		{vendor: "SHARP_LH28F800BV_", devices: []ChipName{"TV"}, id: 0x4C},
		{vendor: "SHARP", devices: []ChipName{"LHF00L02"}, id: 0xC9, remarks: "	/* Same as LHF00L06/LHF00L07 */"},
		{vendor: "SHARP", devices: []ChipName{"LHF00L04"}, id: 0xCF, remarks: "	/* Same as LHF00L03/LHF00L05 */"},
		/* Spansion was previously a joint venture of AMD and Fujitsu. */
		/* S25 chips are SPI. The first device ID byte is memory type and
		 * the second device ID byte is memory capacity. */
		{vendor: "SPANSION", devices: []ChipName{"S25FL004A"}, id: 0x0212},
		{vendor: "SPANSION", devices: []ChipName{"S25FL008A"}, id: 0x0213},
		{vendor: "SPANSION", devices: []ChipName{"S25FL016A"}, id: 0x0214},
		{vendor: "SPANSION", devices: []ChipName{"S25FL032A"}, id: 0x0215, remarks: "	/* Same as S25FL032P, but the latter supports EDI and CFI */"},
		{vendor: "SPANSION", devices: []ChipName{"S25FL064A"}, id: 0x0216, remarks: "	/* Same as S25FL064P, but the latter supports EDI and CFI */"},
		{vendor: "SPANSION", devices: []ChipName{"S25FL128"}, id: 0x2018, remarks: "	/* Same ID for various S25FL127S, S25FL128P, S25FL128S and S25FL129P (including dual-die S70FL256P) variants (EDI supported) */"},
		{vendor: "SPANSION", devices: []ChipName{"S25FL256"}, id: 0x0219},
		{vendor: "SPANSION", devices: []ChipName{"S25FL512"}, id: 0x0220},
		{vendor: "SPANSION", devices: []ChipName{"S25FL204"}, id: 0x4013},
		{vendor: "SPANSION", devices: []ChipName{"S25FL208"}, id: 0x4014},
		{vendor: "SPANSION", devices: []ChipName{"S25FL216"}, id: 0x4015, remarks: "	/* Same as S25FL216K, but the latter supports OTP, 3 status regs, quad I/O, SFDP etc. */"},
		{vendor: "SPANSION", devices: []ChipName{"S25FL132K"}, id: 0x4016},
		{vendor: "SPANSION", devices: []ChipName{"S25FL164K"}, id: 0x4017},
		/* Spansion 29GL families got a suffix indicating the process technology but share the same 3-Byte IDs. They can
		 * however be differentiated by CFI byte 45h. Some versions exist which have special top or bottom boot sectors
		 * and various WP configurations (not heeded in the table below).
		 *
		 * Suf.  Process Sector Sz  Rd Page  Wr Page  Data Width  OTP Sz   Min Size     Max Size
		 *  A     200 nm    64 kB      8 B     32 B     x8/x16     256 B   16Mb/ 2MB   64Mb/  8MB
		 *  M     230 nm    64 kB      8 B     32 B     x8/x16     256 B   32Mb/ 4MB  256Mb/ 32MB
		 *  N*    110 nm    64 kB     16 B     32 B     x8/x16     256 B   32Mb/ 4MB   64Mb/  8MB
		 *  N*    110 nm   128 kB     16 B     32 B     x8/x16     256 B  128Mb/16MB  256Mb/ 64MB
		 *  P      90 nm   128 kB     16 B     64 B     x8/x16     256 B  128Mb/16MB    2Gb/256MB
		 *  S      65 nm   128 kB     32 B    512 B     x8 only    512 B  128Mb/16MB    2Gb/256MB
		 *
		 * For the N series there are two subgroupsName: the 4 and 8MB devices (S29GL032N, S29GL064N) have 64 kB erase
		 * sectors while the bigger chips got 128 kB sectors.
		 * Each series includes multiple models varying in speedgrade, boot block configurations etc.
		 */
		{vendor: "SPANSION_S29GL016", devices: []ChipName{"1"}, id: 0xC4, remarks: "	/* Top Boot Sector, WP protects Top 2 sectors */"},
		{vendor: "SPANSION_S29GL016", devices: []ChipName{"2"}, id: 0x49, remarks: "	/* Bottom Boot Sector, WP protects Bottom 2 sectors */"},
		/* Same IDs for S29GL032A, S29GL032M, S29GL032N (variations) */
		{vendor: "SPANSION_S29GL032", devices: []ChipName{"1289"}, id: 0x7E1D00, remarks: "	/* Uniform Sectors, WP protects Top OR Bottom sector */"},
		{vendor: "SPANSION_S29GL032", devices: []ChipName{"3"}, id: 0x7E1A01, remarks: "	/* Top Boot Sector, WP protects Top 2 sectors */"},
		{vendor: "SPANSION_S29GL032", devices: []ChipName{"4"}, id: 0x7E1A00, remarks: "	/* Bottom Boot Sector, WP protects Bottom 2 sectors */"},
		/* Same IDs for S29GL064A, S29GL064M, S29GL064N, S29GL064S (variations) */
		{vendor: "SPANSION_S29GL064", devices: []ChipName{"1289"}, id: 0x7E0C01, remarks: "	/* Uniform Sectors, WP protects Top OR Bottom sector */"},
		{vendor: "SPANSION_S29GL064", devices: []ChipName{"3"}, id: 0x7E1001, remarks: "	/* Top Boot Sector, WP protects Top 2 sectors */"},
		{vendor: "SPANSION_S29GL064", devices: []ChipName{"4"}, id: 0x7E1000, remarks: "	/* Bottom Boot Sector, WP protects Bottom 2 sectors */"},
		{vendor: "SPANSION_S29GL064", devices: []ChipName{"567"}, id: 0x7E1301, remarks: "	/* x16 only, Uniform Sectors */"},
		{vendor: "SPANSION", devices: []ChipName{"S29GL128"}, id: 0x7E2101, remarks: "	/* Same ID for S29GL128M, S29GL128N, S29GL128P, S29GL128S */"},
		{vendor: "SPANSION", devices: []ChipName{"S29GL256"}, id: 0x7E2201, remarks: "	/* Same ID for S29GL256M, S29GL256N, S29GL256P, S29GL256S */"},
		{vendor: "SPANSION", devices: []ChipName{"S29GL512"}, id: 0x7E2301, remarks: "	/* Same ID for S29GL512P, S29GL512S */"},
		{vendor: "SPANSION", devices: []ChipName{"S29GL01G"}, id: 0x7E2801, remarks: "	/* Same ID for S29GL01GP, S29GL01GS */"},
		{vendor: "SPANSION", devices: []ChipName{"S70GL02G"}, id: 0x7E4801, remarks: "	/* Same ID for S70GL02GP, S70GL02GS; based on two S29GL01G dies respectively */"},
		/*
		 * SST25 chips are SPI, first byte of device ID is memory type, second
		 * byte of device ID is related to log(bitsize) at least for some chips.
		 */
		{vendor: "SST_SST25LF020", devices: []ChipName{"REMS"}, id: 0x43, remarks: "	/* REMS or RES opcode */"},
		{vendor: "SST", devices: []ChipName{"SST25WF512"}, id: 0x2501},
		{vendor: "SST", devices: []ChipName{"SST25WF010"}, id: 0x2502},
		{vendor: "SST", devices: []ChipName{"SST25WF020"}, id: 0x2503},
		{vendor: "SST", devices: []ChipName{"SST25WF040"}, id: 0x2504},
		{vendor: "SST", devices: []ChipName{"SST25WF080"}, id: 0x2505},
		/* There exist some successors to members of the SST25WF family with alphabetic suffixes. Their datasheets show
		 * a 4 byte long response w/o a vendor ID. The first byte is 0x62 that is actually Sanyo's and might be due to
		 * a collaboration in the mid 2000ies between Sanyo and SST. */
		{vendor: "SST", devices: []ChipName{"SST25WF020A"}, id: 0x1612},
		{vendor: "SST", devices: []ChipName{"SST25WF040B"}, id: 0x1613},
		{vendor: "SST", devices: []ChipName{"SST25WF080B"}, id: 0x1614},
		{vendor: "SST_SST25VF512", devices: []ChipName{"REMS"}, id: 0x48, remarks: "	/* REMS or RES opcode, same as SST25VF512A */"},
		{vendor: "SST_SST25VF010", devices: []ChipName{"REMS"}, id: 0x49, remarks: "	/* REMS or RES opcode, same as SST25VF010A */"},
		{vendor: "SST_SST25VF020", devices: []ChipName{"REMS"}, id: 0x43, remarks: "	/* REMS or RES opcode, same as SST25LF020A */"},
		{vendor: "SST", devices: []ChipName{"SST25VF020B"}, id: 0x258C},
		{vendor: "SST_SST25VF040", devices: []ChipName{"REMS"}, id: 0x44, remarks: "	/* REMS or RES opcode, same as SST25LF040A */"},
		{vendor: "SST", devices: []ChipName{"SST25VF040B"}, id: 0x258D},
		{vendor: "SST_SST25VF040B", devices: []ChipName{"REMS"}, id: 0x8D, remarks: "	/* REMS or RES opcode */"},
		{vendor: "SST_SST25VF080", devices: []ChipName{"REMS"}, id: 0x80, remarks: "	/* REMS or RES opcode, same as SST25LF080A */"},
		{vendor: "SST", devices: []ChipName{"SST25VF080B"}, id: 0x258E},
		{vendor: "SST_SST25VF080B", devices: []ChipName{"REMS"}, id: 0x8E, remarks: "	/* REMS or RES opcode */"},
		{vendor: "SST", devices: []ChipName{"SST25VF016B"}, id: 0x2541},
		{vendor: "SST", devices: []ChipName{"SST25VF032B"}, id: 0x254A},
		{vendor: "SST_SST25VF032B", devices: []ChipName{"REMS"}, id: 0x4A, remarks: "	/* REMS or RES opcode */"},
		{vendor: "SST", devices: []ChipName{"SST25VF064C"}, id: 0x254B},
		{vendor: "SST", devices: []ChipName{"SST26VF016"}, id: 0x2601},
		{vendor: "SST", devices: []ChipName{"SST26VF032"}, id: 0x2602},
		{vendor: "SST", devices: []ChipName{"SST26VF016B"}, id: 0x2641},
		{vendor: "SST", devices: []ChipName{"SST26VF032B"}, id: 0x2642},
		{vendor: "SST", devices: []ChipName{"SST26VF064B"}, id: 0x2643},
		{vendor: "SST", devices: []ChipName{"SST27SF512"}, id: 0xA4},
		{vendor: "SST", devices: []ChipName{"SST27SF010"}, id: 0xA5},
		{vendor: "SST", devices: []ChipName{"SST27SF020"}, id: 0xA6},
		{vendor: "SST", devices: []ChipName{"SST27VF010"}, id: 0xA9},
		{vendor: "SST", devices: []ChipName{"SST27VF020"}, id: 0xAA},
		{vendor: "SST", devices: []ChipName{"SST28SF040"}, id: 0x04},
		{vendor: "SST", devices: []ChipName{"SST29LE512"}, id: 0x3D, remarks: "	/* Same as SST29VE512 */"},
		{vendor: "SST", devices: []ChipName{"SST29EE512"}, id: 0x5D},
		{vendor: "SST", devices: []ChipName{"SST29EE010"}, id: 0x07},
		{vendor: "SST", devices: []ChipName{"SST29LE010"}, id: 0x08, remarks: "	/* Same as SST29VE010 */"},
		{vendor: "SST", devices: []ChipName{"SST29EE020A"}, id: 0x10, remarks: "	/* Same as SST29EE020 */"},
		{vendor: "SST", devices: []ChipName{"SST29LE020"}, id: 0x12, remarks: "	/* Same as SST29VE020 */"},
		{vendor: "SST", devices: []ChipName{"SST29SF020"}, id: 0x24},
		{vendor: "SST", devices: []ChipName{"SST29VF020"}, id: 0x25},
		{vendor: "SST", devices: []ChipName{"SST29SF040"}, id: 0x13},
		{vendor: "SST", devices: []ChipName{"SST29VF040"}, id: 0x14},
		{vendor: "SST", devices: []ChipName{"SST39SF512"}, id: 0xB4},
		{vendor: "SST", devices: []ChipName{"SST39SF010"}, id: 0xB5},
		{vendor: "SST", devices: []ChipName{"SST39SF020"}, id: 0xB6, remarks: "	/* Same as 39SF020A */"},
		{vendor: "SST", devices: []ChipName{"SST39SF040"}, id: 0xB7},
		{vendor: "SST", devices: []ChipName{"SST39VF512"}, id: 0xD4},
		{vendor: "SST", devices: []ChipName{"SST39VF010"}, id: 0xD5},
		{vendor: "SST", devices: []ChipName{"SST39VF020"}, id: 0xD6, remarks: "	/* Same as 39LF020 */"},
		{vendor: "SST", devices: []ChipName{"SST39VF040"}, id: 0xD7, remarks: "	/* Same as 39LF040 */"},
		{vendor: "SST", devices: []ChipName{"SST39VF080"}, id: 0xD8, remarks: "	/* Same as 39LF080/39VF080/39VF088 */"},
		{vendor: "SST", devices: []ChipName{"SST45VF512"}, id: 0x41, remarks: "	/* REMS, read opcode 0xFF */"},
		{vendor: "SST", devices: []ChipName{"SST45LF010"}, id: 0x42, remarks: "	/* REMS, read opcode 0xFF, 'funny' other opcodes */"},
		{vendor: "SST", devices: []ChipName{"SST45VF010"}, id: 0x45, remarks: "	/* REMS, read opcode 0xFF */"},
		{vendor: "SST", devices: []ChipName{"SST45VF020"}, id: 0x43, remarks: "	/* REMS, read opcode 0xFF */"},
		{vendor: "SST", devices: []ChipName{"SST49LF040B"}, id: 0x50},
		{vendor: "SST", devices: []ChipName{"SST49LF040"}, id: 0x51},
		{vendor: "SST", devices: []ChipName{"SST49LF020"}, id: 0x61},
		{vendor: "SST", devices: []ChipName{"SST49LF020A"}, id: 0x52},
		{vendor: "SST", devices: []ChipName{"SST49LF030A"}, id: 0x1C},
		{vendor: "SST", devices: []ChipName{"SST49LF080A"}, id: 0x5B},
		{vendor: "SST", devices: []ChipName{"SST49LF002A"}, id: 0x57},
		{vendor: "SST", devices: []ChipName{"SST49LF003A"}, id: 0x1B},
		{vendor: "SST", devices: []ChipName{"SST49LF004A"}, id: 0x60, remarks: "	/* Same as 49LF004B */"},
		{vendor: "SST", devices: []ChipName{"SST49LF008A"}, id: 0x5A},
		{vendor: "SST", devices: []ChipName{"SST49LF004C"}, id: 0x54},
		{vendor: "SST", devices: []ChipName{"SST49LF008C"}, id: 0x59},
		{vendor: "SST", devices: []ChipName{"SST49LF016C"}, id: 0x5C},
		{vendor: "SST", devices: []ChipName{"SST49LF160C"}, id: 0x4C},
		/*
		 * ST25P chips are SPI, first byte of device ID is memory type, second
		 * byte of device ID is related to log(bitsize) at least for some chips.
		 */
		{vendor: "ST", devices: []ChipName{"M25P05A"}, id: 0x2010},
		{vendor: "ST_M25P05", devices: []ChipName{"RES"}, id: 0x05},
		{vendor: "ST", devices: []ChipName{"M25P10A"}, id: 0x2011},
		{vendor: "ST_M25P10", devices: []ChipName{"RES"}, id: 0x10},
		{vendor: "ST", devices: []ChipName{"M25P20"}, id: 0x2012},
		{vendor: "ST_M25P20", devices: []ChipName{"RES"}, id: 0x11},
		{vendor: "ST", devices: []ChipName{"M25P40"}, id: 0x2013},
		{vendor: "ST_M25P40", devices: []ChipName{"RES"}, id: 0x12},
		{vendor: "ST", devices: []ChipName{"M25P80"}, id: 0x2014},
		{vendor: "ST", devices: []ChipName{"M25P16"}, id: 0x2015},
		{vendor: "ST", devices: []ChipName{"M25P32"}, id: 0x2016},
		{vendor: "ST", devices: []ChipName{"M25P64"}, id: 0x2017},
		{vendor: "ST", devices: []ChipName{"M25P128"}, id: 0x2018},
		{vendor: "ST", devices: []ChipName{"M45PE10"}, id: 0x4011},
		{vendor: "ST", devices: []ChipName{"M45PE20"}, id: 0x4012},
		{vendor: "ST", devices: []ChipName{"M45PE40"}, id: 0x4013},
		{vendor: "ST", devices: []ChipName{"M45PE80"}, id: 0x4014},
		{vendor: "ST", devices: []ChipName{"M45PE16"}, id: 0x4015},
		{vendor: "ST", devices: []ChipName{"M25PX80"}, id: 0x7114},
		{vendor: "ST", devices: []ChipName{"M25PX16"}, id: 0x7115},
		{vendor: "ST", devices: []ChipName{"M25PX32"}, id: 0x7116},
		{vendor: "ST", devices: []ChipName{"M25PX64"}, id: 0x7117},
		{vendor: "ST", devices: []ChipName{"M25PE10"}, id: 0x8011},
		{vendor: "ST", devices: []ChipName{"M25PE20"}, id: 0x8012},
		{vendor: "ST", devices: []ChipName{"M25PE40"}, id: 0x8013},
		{vendor: "ST", devices: []ChipName{"M25PE80"}, id: 0x8014},
		{vendor: "ST", devices: []ChipName{"M25PE16"}, id: 0x8015},
		{vendor: "ST", devices: []ChipName{"M50FLW040A"}, id: 0x08},
		{vendor: "ST", devices: []ChipName{"M50FLW040B"}, id: 0x28},
		{vendor: "ST", devices: []ChipName{"M50FLW080A"}, id: 0x80},
		{vendor: "ST", devices: []ChipName{"M50FLW080B"}, id: 0x81},
		{vendor: "ST", devices: []ChipName{"M50FW002"}, id: 0x29},
		{vendor: "ST", devices: []ChipName{"M50FW040"}, id: 0x2C},
		{vendor: "ST", devices: []ChipName{"M50FW080"}, id: 0x2D},
		{vendor: "ST", devices: []ChipName{"M50FW016"}, id: 0x2E},
		{vendor: "ST", devices: []ChipName{"M50LPW080"}, id: 0x2F},
		{vendor: "ST", devices: []ChipName{"M50LPW116"}, id: 0x30},
		{vendor: "ST", devices: []ChipName{"M29F002B"}, id: 0x34, remarks: "	/* Same as M29F002BB */"},
		{vendor: "ST", devices: []ChipName{"M29F002T"}, id: 0xB0, remarks: "	/* Same as M29F002BT/M29F002NT/M29F002BNT */"},
		{vendor: "ST", devices: []ChipName{"M29F040B"}, id: 0xE2, remarks: "	/* Same as M29F040 */"},
		{vendor: "ST", devices: []ChipName{"M29F080"}, id: 0xF1},
		{vendor: "ST", devices: []ChipName{"M29F200BT"}, id: 0xD3},
		{vendor: "ST", devices: []ChipName{"M29F200BB"}, id: 0xD4},
		{vendor: "ST", devices: []ChipName{"M29F400BT"}, id: 0xD5, remarks: "	/* Same as M29F400T */"},
		{vendor: "ST", devices: []ChipName{"M29F400BB"}, id: 0xD6, remarks: "	/* Same as M29F400B */"},
		{vendor: "ST", devices: []ChipName{"M29F800DB"}, id: 0x58},
		{vendor: "ST", devices: []ChipName{"M29F800DT"}, id: 0xEC},
		{vendor: "ST", devices: []ChipName{"M29W010B"}, id: 0x23},
		{vendor: "ST", devices: []ChipName{"M29W040B"}, id: 0xE3},
		{vendor: "ST", devices: []ChipName{"M29W512B"}, id: 0x27},
		{vendor: "ST", devices: []ChipName{"M28W160ECB"}, id: 0x88CF},
		{vendor: "ST", devices: []ChipName{"M28W160ECT"}, id: 0x88CE},
		{vendor: "ST", devices: []ChipName{"M28W320FCB"}, id: 0x88BB},
		{vendor: "ST", devices: []ChipName{"M28W320FCT"}, id: 0x88BA},
		{vendor: "ST", devices: []ChipName{"M28W640HCB"}, id: 0x8849},
		{vendor: "ST", devices: []ChipName{"M28W640HCT"}, id: 0x8848},
		{vendor: "ST", devices: []ChipName{"M29DW127G"}, id: 0x7E2004},
		{vendor: "ST", devices: []ChipName{"M29W128GH"}, id: 0x7E2101},
		{vendor: "ST", devices: []ChipName{"M29W128GL"}, id: 0x7E2100},
		{vendor: "ST", devices: []ChipName{"M29W160EB"}, id: 0x2249},
		{vendor: "ST", devices: []ChipName{"M29W160ET"}, id: 0x22C4},
		{vendor: "ST", devices: []ChipName{"M29W256GH"}, id: 0x7E21, remarks: "xx"},
		{vendor: "ST", devices: []ChipName{"M29W256GL"}, id: 0x7E21, remarks: "xx"},
		{vendor: "ST", devices: []ChipName{"M29W320DB"}, id: 0x88CB},
		{vendor: "ST", devices: []ChipName{"M29W320DT"}, id: 0x88CA},
		{vendor: "ST", devices: []ChipName{"M29W400FB"}, id: 0x00EF},
		{vendor: "ST", devices: []ChipName{"M29W400FT"}, id: 0x00EE},
		{vendor: "ST", devices: []ChipName{"M29W512GH"}, id: 0x7E2301},
		{vendor: "ST", devices: []ChipName{"M29W640FB"}, id: 0x22FD},
		{vendor: "ST", devices: []ChipName{"M29W640FT"}, id: 0x22ED},
		{vendor: "ST", devices: []ChipName{"M29W640GB"}, id: 0x7E1000},
		{vendor: "ST", devices: []ChipName{"M29W640GH"}, id: 0x7E0C01},
		{vendor: "ST", devices: []ChipName{"M29W640GL"}, id: 0x7E0C00},
		{vendor: "ST", devices: []ChipName{"M29W640GT"}, id: 0x7E1001},
		{vendor: "ST", devices: []ChipName{"M29W800FB"}, id: 0x225B},
		{vendor: "ST", devices: []ChipName{"M29W800FT"}, id: 0x22D7},
		{vendor: "ST", devices: []ChipName{"M58BW16FB"}, id: 0x8839},
		{vendor: "ST", devices: []ChipName{"M58BW16FT"}, id: 0x883A},
		{vendor: "ST", devices: []ChipName{"M58BW32FB"}, id: 0x8837},
		{vendor: "ST", devices: []ChipName{"M58BW32FT"}, id: 0x8838},
		{vendor: "ST", devices: []ChipName{"M58WR016KB"}, id: 0x8813},
		{vendor: "ST", devices: []ChipName{"M58WR016KT"}, id: 0x8812},
		{vendor: "ST", devices: []ChipName{"M58WR032KB"}, id: 0x8815},
		{vendor: "ST", devices: []ChipName{"M58WR032KT"}, id: 0x8814},
		{vendor: "ST", devices: []ChipName{"M58WR064KB"}, id: 0x8811},
		{vendor: "ST", devices: []ChipName{"M58WR064KT"}, id: 0x8810},
		{vendor: "ST_MT28GU01G__", devices: []ChipName{"1"}, id: 0x88B0},
		{vendor: "ST_MT28GU01G__", devices: []ChipName{"2"}, id: 0x88B1},
		{vendor: "ST_MT28GU256__", devices: []ChipName{"1"}, id: 0x8901},
		{vendor: "ST_MT28GU256__", devices: []ChipName{"2"}, id: 0x8904},
		{vendor: "ST_MT28GU512__", devices: []ChipName{"1"}, id: 0x887E},
		{vendor: "ST_MT28GU512__", devices: []ChipName{"2"}, id: 0x8881},
		{vendor: "ST_N25Q016_", devices: []ChipName{"1E"}, id: 0xBB15, remarks: "	/* N25Q016, 1.8V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q032_", devices: []ChipName{"3E"}, id: 0xBA16, remarks: "	/* N25Q032, 3.0V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q032_", devices: []ChipName{"1E"}, id: 0xBB16, remarks: "	/* N25Q032, 1.8V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q064_", devices: []ChipName{"3E"}, id: 0xBA17, remarks: "	/* N25Q064, 3.0V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q064_", devices: []ChipName{"1E"}, id: 0xBB17, remarks: "	/* N25Q064, 1.8V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q128_", devices: []ChipName{"3E"}, id: 0xBA18, remarks: "	/* N25Q128, 3.0V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q128_", devices: []ChipName{"1E"}, id: 0xBB18, remarks: "	/* N25Q128, 1.8V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q256_", devices: []ChipName{"3E"}, id: 0xBA19, remarks: "	/* N25Q256, 3.0V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q256_", devices: []ChipName{"1E"}, id: 0xBB19, remarks: "	/* N25Q256, 1.8V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q512_", devices: []ChipName{"3E"}, id: 0xBA20, remarks: "	/* N25Q512, 3.0V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q512_", devices: []ChipName{"1E"}, id: 0xBB20, remarks: "	/* N25Q512, 1.8V, (uniform sectors expected) */"},
		{vendor: "ST_N25Q00A_", devices: []ChipName{"3E"}, id: 0xBA21, remarks: "	/* N25Q00A, 3.0V, (uniform sectors expected) */"},
		{vendor: "ST", devices: []ChipName{"NP5Q032"}, id: 0xDA16, remarks: "	/* Phase-change memory (PCM), 3V */"},
		{vendor: "ST", devices: []ChipName{"NP5Q064"}, id: 0xDA17, remarks: "	/* Phase-change memory (PCM), 3V */"},
		{vendor: "ST", devices: []ChipName{"NP5Q128"}, id: 0xDA18, remarks: "	/* Phase-change memory (PCM), 3V */"},
		{vendor: "MVC", devices: []ChipName{"V29C51000T"}, id: 0x00},
		{vendor: "MVC", devices: []ChipName{"V29C51400T"}, id: 0x13},
		{vendor: "MVC", devices: []ChipName{"V29LC51000"}, id: 0x20},
		{vendor: "MVC", devices: []ChipName{"V29LC51001"}, id: 0x60},
		{vendor: "MVC", devices: []ChipName{"V29LC51002"}, id: 0x82},
		{vendor: "MVC", devices: []ChipName{"V29C51000B"}, id: 0xA0},
		{vendor: "MVC", devices: []ChipName{"V29C51400B"}, id: 0xB3},
		{vendor: "SM_MVC", devices: []ChipName{"29C51001T"}, id: 0x01, remarks: "	/* Identical chips: {F,S,V}29C51001T */"},
		{vendor: "SM_MVC", devices: []ChipName{"29C51002T"}, id: 0x02, remarks: "	/* Identical chips: {F,S,V}29C51002T */"},
		{vendor: "SM_MVC", devices: []ChipName{"29C51004T"}, id: 0x03, remarks: "	/* Identical chips: {F,S,V}29C51004T */"},
		{vendor: "SM_MVC", devices: []ChipName{"29C31004T"}, id: 0x63, remarks: "	/* Identical chips: {S,V}29C31004T */"},
		{vendor: "SM_MVC", devices: []ChipName{"29C31004B"}, id: 0x73, remarks: "	/* Identical chips: {S,V}29C31004B */"},
		{vendor: "SM_MVC", devices: []ChipName{"29C51001B"}, id: 0xA1, remarks: "	/* Identical chips: {F,S,V}29C51001B */"},
		{vendor: "SM_MVC", devices: []ChipName{"29C51002B"}, id: 0xA2, remarks: "	/* Identical chips: {F,S,V}29C51002B */"},
		{vendor: "SM_MVC", devices: []ChipName{"29C51004B"}, id: 0xA3, remarks: "	/* Identical chips: {F,S,V}29C51004B */"},
		{vendor: "TENX", devices: []ChipName{"ICE25P05"}, id: 0x01, remarks: "/* Maybe? */"},
		{vendor: "TI", devices: []ChipName{"TMS29F002RT"}, id: 0xB0},
		{vendor: "TI", devices: []ChipName{"TMS29F002RB"}, id: 0x34},
		/*
		 * W25X chips are SPI, first byte of device ID is memory type, second
		 * byte of device ID is related to log(bitsize).
		 */
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25P80"}, id: 0x2014},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25P16"}, id: 0x2015},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25P32"}, id: 0x2016},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25X10"}, id: 0x3011},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25X20"}, id: 0x3012},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25X40"}, id: 0x3013},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25X80"}, id: 0x3014},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25X16"}, id: 0x3015},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25X32"}, id: 0x3016},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25X64"}, id: 0x3017},
		{vendor: "WINBOND_NEX_W25Q40", devices: []ChipName{"V"}, id: 0x4013, remarks: "	/* W25Q40BV; W25Q40BL (2.3-3.6V) */"},
		{vendor: "WINBOND_NEX_W25Q80", devices: []ChipName{"V"}, id: 0x4014, remarks: "	/* W25Q80BV */"},
		{vendor: "WINBOND_NEX_W25Q16", devices: []ChipName{"V"}, id: 0x4015, remarks: "	/* W25Q16CV; W25Q16DV */"},
		{vendor: "WINBOND_NEX_W25Q32", devices: []ChipName{"V"}, id: 0x4016, remarks: "	/* W25Q32BV; W25Q32FV in SPI mode (default) */"},
		{vendor: "WINBOND_NEX_W25Q64", devices: []ChipName{"V"}, id: 0x4017, remarks: "	/* W25Q64BV, W25Q64CV; W25Q64FV in SPI mode (default) */"},
		{vendor: "WINBOND_NEX_W25Q128", devices: []ChipName{"V"}, id: 0x4018, remarks: "	/* W25Q128BV; W25Q128FV in SPI mode (default) */"},
		{vendor: "WINBOND_NEX_W25Q256", devices: []ChipName{"V"}, id: 0x4019, remarks: "	/* W25Q256FV or W25Q256JV_Q (QE=1) */"},
		{vendor: "WINBOND_NEX_W25Q20", devices: []ChipName{"W"}, id: 0x5012, remarks: "	/* W25Q20BW */"},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25Q40BW"}, id: 0x5013, remarks: "	/* W25Q40BW */"},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25Q80BW"}, id: 0x5014, remarks: "	/* W25Q80BW */"},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25Q40EW"}, id: 0x6013, remarks: "	/* W25Q40EW */"},
		{vendor: "WINBOND_NEX", devices: []ChipName{"W25Q80EW"}, id: 0x6014, remarks: "	/* W25Q80EW */"},
		{vendor: "WINBOND_NEX_W25Q16", devices: []ChipName{"W"}, id: 0x6015, remarks: "	/* W25Q16DW */"},
		{vendor: "WINBOND_NEX_W25Q32", devices: []ChipName{"W"}, id: 0x6016, remarks: "	/* W25Q32DW; W25Q32FV in QPI mode */"},
		{vendor: "WINBOND_NEX_W25Q64", devices: []ChipName{"W"}, id: 0x6017, remarks: "	/* W25Q64DW; W25Q64FV in QPI mode */"},
		{vendor: "WINBOND_NEX_W25Q128", devices: []ChipName{"W"}, id: 0x6018, remarks: "	/* W25Q128FW; W25Q128FV in QPI mode */"},
		{vendor: "WINBOND_NEX_W25Q128_V", devices: []ChipName{"M"}, id: 0x7018, remarks: "	/* W25Q128JVSM */"},
		{vendor: "WINBOND_NEX_W25Q256JV", devices: []ChipName{"M"}, id: 0x7019, remarks: "	/* W25Q256JV_M (QE=0) */"},
		{vendor: "WINBOND", devices: []ChipName{"W19B160BB"}, id: 0x49},
		{vendor: "WINBOND", devices: []ChipName{"W19B160BT"}, id: 0xC4},
		{vendor: "WINBOND", devices: []ChipName{"W19B320SB"}, id: 0x2A, remarks: "	/* Same as W19L320SB */"},
		{vendor: "WINBOND", devices: []ChipName{"W19B320ST"}, id: 0xBA, remarks: "	/* Same as W19L320ST */"},
		{vendor: "WINBOND", devices: []ChipName{"W19B322MB"}, id: 0x92},
		{vendor: "WINBOND", devices: []ChipName{"W19B322MT"}, id: 0x10},
		{vendor: "WINBOND", devices: []ChipName{"W19B323MB"}, id: 0x94},
		{vendor: "WINBOND", devices: []ChipName{"W19B323MT"}, id: 0x13},
		{vendor: "WINBOND", devices: []ChipName{"W19B324MB"}, id: 0x97},
		{vendor: "WINBOND", devices: []ChipName{"W19B324MT"}, id: 0x16},
		{vendor: "WINBOND", devices: []ChipName{"W29C010"}, id: 0xC1, remarks: "	/* Same as W29C010M, W29C011A, W29EE011, W29EE012, and ASD AE29F1008 */"},
		{vendor: "WINBOND", devices: []ChipName{"W29C020"}, id: 0x45, remarks: "	/* Same as W29C020C, W29C022 and ASD AE29F2008 */"},
		{vendor: "WINBOND", devices: []ChipName{"W29C040"}, id: 0x46, remarks: "	/* Same as W29C040P */"},
		{vendor: "WINBOND", devices: []ChipName{"W29C512A"}, id: 0xC8, remarks: "	/* Same as W29EE512 */"},
		{vendor: "WINBOND", devices: []ChipName{"W29GL032CHL"}, id: 0x7E1D01, remarks: "	/* Uniform Sectors, WP protects Top OR Bottom sector */"},
		{vendor: "WINBOND", devices: []ChipName{"W29GL032CB"}, id: 0x7E1A00, remarks: "	/* Top Boot Sector, WP protects Top 2 sectors */"},
		{vendor: "WINBOND", devices: []ChipName{"W29GL032CT"}, id: 0x7E1A01, remarks: "	/* Bottom Boot Sector, WP protects Bottom 2 sectors */"},
		{vendor: "WINBOND", devices: []ChipName{"W29GL064CHL"}, id: 0x7E0C01, remarks: "	/* Uniform Sectors, WP protects Top OR Bottom sector */"},
		{vendor: "WINBOND", devices: []ChipName{"W29GL064CB"}, id: 0x7E1000, remarks: "	/* Top Boot Sector, WP protects Top 2 sectors */"},
		{vendor: "WINBOND", devices: []ChipName{"W29GL064CT"}, id: 0x7E1001, remarks: "	/* Bottom Boot Sector, WP protects Bottom 2 sectors */"},
		{vendor: "WINBOND", devices: []ChipName{"W29GL128CHL"}, id: 0x7E2101, remarks: "	/* Uniform Sectors, WP protects Top OR Bottom sector */"},
		{vendor: "WINBOND", devices: []ChipName{"W29GL256HL"}, id: 0x7E2201, remarks: "	/* Same ID for W29GL0256P and W29GL0256S; uniform Sectors, WP protects Top OR Bottom sector */"},
		{vendor: "WINBOND", devices: []ChipName{"W39F010"}, id: 0xA1},
		{vendor: "WINBOND", devices: []ChipName{"W39L010"}, id: 0x31},
		{vendor: "WINBOND", devices: []ChipName{"W39L020"}, id: 0xB5},
		{vendor: "WINBOND", devices: []ChipName{"W39L040"}, id: 0xB6},
		{vendor: "WINBOND", devices: []ChipName{"W39L040A"}, id: 0xD6},
		{vendor: "WINBOND", devices: []ChipName{"W39L512"}, id: 0x38},
		{vendor: "WINBOND", devices: []ChipName{"W39V040A"}, id: 0x3D},
		{vendor: "WINBOND", devices: []ChipName{"W39V040FA"}, id: 0x34},
		{vendor: "WINBOND", devices: []ChipName{"W39V040B"}, id: 0x54, remarks: "	/* Same as W39V040FB */"},
		{vendor: "WINBOND", devices: []ChipName{"W39V040C"}, id: 0x50, remarks: "	/* Same as W39V040FC */"},
		{vendor: "WINBOND", devices: []ChipName{"W39V080A"}, id: 0xD0},
		{vendor: "WINBOND", devices: []ChipName{"W39V080FA"}, id: 0xD3},
		{vendor: "WINBOND_W39V080FA", devices: []ChipName{"DM"}, id: 0x93, remarks: "	/* W39V080FA dual mode */"},
		{vendor: "WINBOND", devices: []ChipName{"W49F002"}, id: 0x25, remarks: "	/* Same as W49F002B */"},
		{vendor: "WINBOND", devices: []ChipName{"W49F002U"}, id: 0x0B, remarks: "	/* Same as W49F002N and ASD AE49F2008 */"},
		{vendor: "WINBOND", devices: []ChipName{"W49F020"}, id: 0x8C},
		{vendor: "WINBOND", devices: []ChipName{"W49V002A"}, id: 0xB0},
		{vendor: "WINBOND", devices: []ChipName{"W49V002FA"}, id: 0x32},
		{vendor: "ZETTADEVICE", devices: []ChipName{"ZD25D20"}, id: 0x2012},
		{vendor: "ZETTADEVICE", devices: []ChipName{"ZD25D40"}, id: 0x2013},
	}
)
