// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief e+ e- > hadrons
  class BESII_2006_I717720 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESII_2006_I717720);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(Beam(), "Beams");
      declare(UnstableParticles(Cuts::abspid==411 ||
				Cuts::abspid==421 ), "UFS");
      declare(FinalState(), "FS");

      // Book histograms
      for(unsigned int ix=0;ix<4;++ix)
	book(_c[ix], "/TMP/sigma_"+toString(ix+1));
      book(_c[4], "/TMP/energy");
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // get the axis, direction of incoming positron
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      _c[4]->fill((beams.first.momentum()+beams.second.momentum()).mass());
      const FinalState& fs = apply<FinalState>(event, "FS");
      const FinalState& ufs = apply<FinalState>(event, "UFS");
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
      	nCount[p.pid()] += 1;
      	++ntotal;
      }
      // mu+mu- + photons
      if(nCount[-13]==1 and nCount[13]==1 &&
      	 ntotal==2+nCount[22])
      	vetoEvent;
      else if(nCount[-11]==1 and nCount[11]==1 &&
      	      ntotal==2+nCount[22])
      	vetoEvent;
      // everything else
      else {
      	_c[0]->fill();
	if(!ufs.particles().empty()) {
	  _c[1]->fill();
	  if(ufs.particles()[0].abspid()==421)
	    _c[2]->fill();
	  else
	    _c[3]->fill();
	}
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      double fact = crossSection()/sumOfWeights()/nanobarn;
      double energy = _c[4]->val()/ sumOfWeights();
      for(unsigned int ix=0;ix<4;++ix) {
	double sig_h = _c[ix]->val()*fact;
	double err_h = _c[ix]->err()*fact;
	unsigned int ih=2,iy=ix;
	if(ix==0) {
	  ih=1;
	  iy=1;
	}
	Scatter2D temphisto(refData(ih, 1, iy));
	Scatter2DPtr cross;
	book(cross, ih, 1, iy);
	double deltaE=1e30;
	unsigned int ipoint=100000000;
	for (size_t b = 0; b < temphisto.numPoints(); b++) {
	  double test = abs(temphisto.point(b).x()-energy);
	  if(test<deltaE) {
	    deltaE=test;
	    ipoint=b;
	  }
	}
	if(deltaE>0.01) ipoint=-1;
	for (size_t b = 0; b < temphisto.numPoints(); b++) {
	  const double x  = temphisto.point(b).x();
	  pair<double,double> ex = temphisto.point(b).xErrs();
	  if (b!=ipoint)
	    cross  ->addPoint(x, 0., ex, make_pair(0.,.0));
	  else
	    cross  ->addPoint(x, sig_h, ex, make_pair(err_h,err_h));
	}
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    CounterPtr _c[5];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESII_2006_I717720);

}
