// Copyright 2018 The Cockroach Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
// implied. See the License for the specific language governing
// permissions and limitations under the License.

package main

import (
	"bytes"
	"os"
	"path/filepath"
	"testing"

	"github.com/cockroachdb/cockroach/pkg/testutils"
)

var expectedA = `# Code generated by prereqs. DO NOT EDIT!

bin/a: a/a.go a/ignore.go a/invalid.go a/cgo.go a/a.c a/a.f a b/b.go b b/vendor/foo.com/bar/bar.go b/vendor/foo.com/bar vendor/foo.com/foo/foo.go vendor/foo.com/foo

a/a.go:
a/ignore.go:
a/invalid.go:
a/cgo.go:
a/a.c:
a/a.f:
a:
b/b.go:
b:
b/vendor/foo.com/bar/bar.go:
b/vendor/foo.com/bar:
vendor/foo.com/foo/foo.go:
vendor/foo.com/foo:
`

var expectedB = `# Code generated by prereqs. DO NOT EDIT!

bin/b: b/b.go b b/vendor/foo.com/bar/bar.go b/vendor/foo.com/bar vendor/foo.com/foo/foo.go vendor/foo.com/foo

b/b.go:
b:
b/vendor/foo.com/bar/bar.go:
b/vendor/foo.com/bar:
vendor/foo.com/foo/foo.go:
vendor/foo.com/foo:
`

var expectedFoo = `# Code generated by prereqs. DO NOT EDIT!

bin/foo: vendor/foo.com/foo/foo.go vendor/foo.com/foo

vendor/foo.com/foo/foo.go:
vendor/foo.com/foo:
`

var expectedSpecialChars = `# Code generated by prereqs. DO NOT EDIT!

bin/specialchars: specialchars/a\[\]\*\?\~ $$%\#.go specialchars

specialchars/a\[\]\*\?\~ $$%\#.go:
specialchars:
`

var expectedTestNoDeps = `# Code generated by prereqs. DO NOT EDIT!

bin/test: test

test:
`
var expectedTestWithDeps = `# Code generated by prereqs. DO NOT EDIT!

bin/test: test test/internal_test.go test/external_test.go vendor/foo.com/foo/foo.go vendor/foo.com/foo vendor/foo.com/baz/baz.go vendor/foo.com/baz

test:
test/internal_test.go:
test/external_test.go:
vendor/foo.com/foo/foo.go:
vendor/foo.com/foo:
vendor/foo.com/baz/baz.go:
vendor/foo.com/baz:
`

func TestPrereqs(t *testing.T) {
	gopath, err := filepath.Abs("testdata")
	if err != nil {
		t.Fatal(err)
	}
	buildCtx.GOPATH = gopath

	const pkg = "testdata/src/example.com"
	absPkg, err := filepath.Abs(pkg)
	if err != nil {
		t.Fatal(err)
	}
	if err := os.Chdir(pkg); err != nil {
		t.Fatal(err)
	}

	testutils.RunTrueAndFalse(t, "symlink", func(t *testing.T, symlink bool) {
		if symlink {
			tempDir, cleanup := testutils.TempDir(t)
			defer cleanup()

			link := filepath.Join(tempDir, "link")
			if err := os.Symlink(absPkg, link); err != nil {
				t.Fatal(err)
			}

			// You can't chdir into a symlink. Instead, you chdir into the physical
			// path and set the PWD to the logical path. (At least, this is the hack
			// that most shells employ, and os.Getwd respects it.)
			if err := os.Setenv("PWD", link); err != nil {
				t.Fatal(err)
			}
			if cwd, err := os.Getwd(); err != nil {
				t.Fatal(err)
			} else if cwd != link {
				t.Fatalf("failed to chdir into symlink %s (os.Getwd reported %s)", link, cwd)
			}
		}

		for _, tc := range []struct {
			path        string
			exp         string
			includeTest bool
		}{
			{path: "example.com/a", exp: expectedA},
			{path: "./b", exp: expectedB},
			{path: "./a/../b", exp: expectedB},
			{path: "example.com/a/../b", exp: expectedB},
			{path: "example.com/b", exp: expectedB},
			{path: "foo.com/foo", exp: expectedFoo},
			{path: "./vendor/foo.com/foo", exp: expectedFoo},
			{path: "example.com/vendor/foo.com/foo", exp: expectedFoo},
			{path: "example.com/specialchars", exp: expectedSpecialChars},
			{path: "example.com/test", exp: expectedTestNoDeps},
			{path: "example.com/test", exp: expectedTestWithDeps, includeTest: true},
		} {
			t.Run(tc.path, func(t *testing.T) {
				var buf bytes.Buffer
				if err := run(&buf, tc.path, tc.includeTest, ""); err != nil {
					t.Fatal(err)
				}
				if e, a := tc.exp, buf.String(); e != a {
					t.Fatalf("expected:\n%s\nactual:\n%s\n", e, a)
				}
			})
		}
	})
}
