/* Lowlevel routines, also contains main() (!)
 * Ben Lynn
 */
/*
Copyright (C) 2002 Benjamin Lynn (blynn@cs.stanford.edu)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>

#include <pthread.h>

//for select
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

//for open
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>

//for errno
#include <errno.h>

#include "button.h"
#include "handler.h"
#include "main.h"

static pthread_t timer_thread;
static int timerpipe[2];
static int quitpipe[2];

static void *timerloop()
{
    int n;
    int status;
    fd_set set;
    int maxfd;
    struct timeval tv;
    int digit_count = 0;
    char ch;

    maxfd = quitpipe[0] > timerpipe[0] ? quitpipe[0] : timerpipe[0];

    n = 0;

    FD_ZERO(&set);
    for(;;) {
	FD_SET(timerpipe[0], &set);
	FD_SET(quitpipe[0], &set);
	tv.tv_sec = delay_sec;
	tv.tv_usec = delay_usec;

	if (digit_count) {
	    status = select(maxfd + 1, &set, NULL, NULL, &tv);
	} else {
	    status = select(maxfd + 1, &set, NULL, NULL, NULL);
	}
	if (status == -1) {
	    if (errno == EINTR) { //since we're handling signals
		continue;
	    } else {
		perror("select");
		break;
	    }
	}
	
	if (FD_ISSET(quitpipe[0], &set)) {
	    //signal from parent thread to quit
	    break;
	}
	if (FD_ISSET(timerpipe[0], &set)) {
	    read(timerpipe[0], &ch, 1);
	    n *= 10;
	    n += ch - '0';
	    digit_count++;
	    if (digit_count >= string_digit_limit) {
		//send what we have
		handle_number(n);
		digit_count = 0;
		n = 0;
	    }
	} else if (digit_count) {
	    //timeout; send what we have
	    handle_number(n);
	    digit_count = 0;
	    n = 0;
	}
    }
    return NULL;
}

int main_loop()
{
    int fd;
    int i, j;
    unsigned char c = 0, c1 = 0;
    unsigned int code = 0;

    fd = open(midi_device, O_RDONLY);
    if (fd < 0) {
	printf("Unable to open %s\n", midi_device);
	return 1;
    }

    init_handler();

    pipe(quitpipe);
    pipe(timerpipe);

    pthread_create(&timer_thread, NULL, timerloop, NULL);

    for(;;) {
	//a button push is a SysEx midi event
	//with manufacturer ID = 0
	
	//wait for sysex
	do {
	    read(fd, &c, 1);
	} while (c != 0xF0);
	read(fd, &c, 1);
	if (c) continue;

	//should be 11 more bytes
	//they determine the button
	for (i=2; i<13; i++) {
	    read(fd, &c, 1);
	    if (i == 7) {
		c1 = c & 0xF;
		code = 0;
	    }
	    j = i - 8;
	    if (j >= 0 && j <= 3) {
		code += (c << (8 * j));
		if (c1 & (1 << (3 - j))) {
		    code |= 1 << (8 * j + 7);
		}
	    }
	}
	for (j=0; j<button_max; j++) {
	    if (button_code[j] == code) {
		printf("%s pressed\n", button_name[j]);
		if (j < 10) {
		    write(timerpipe[1], button_name[j], 1);
		} else {
		    handle_button(j, 0);
		}
		break;
	    } else if (speaker_code(button_code[j]) == code) {
		printf("speaker + %s pressed\n", button_name[j]);
		handle_button(j, 1);
		break;
	    }
	}
    }

    pthread_join(timer_thread, NULL);
    close(quitpipe[0]);
    close(quitpipe[1]);
    close(timerpipe[0]);
    close(timerpipe[1]);
    close(fd);
    return 0;
}
