import type { I18nextToolkitConfig } from '../types';
/**
 * Ensures that the directory for a given file path exists.
 * Creates all necessary parent directories recursively if they don't exist.
 *
 * @param filePath - The file path for which to ensure the directory exists
 *
 * @example
 * ```typescript
 * await ensureDirectoryExists('/path/to/nested/file.json')
 * // Creates /path/to/nested/ directory if it doesn't exist
 * ```
 */
export declare function ensureDirectoryExists(filePath: string): Promise<void>;
/**
 * Reads a file asynchronously and returns its content as a UTF-8 string.
 *
 * @param filePath - The path to the file to read
 * @returns Promise resolving to the file content as a string
 *
 * @example
 * ```typescript
 * const content = await readFileAsync('./config.json')
 * const config = JSON.parse(content)
 * ```
 */
export declare function readFileAsync(filePath: string): Promise<string>;
/**
 * Writes data to a file asynchronously.
 *
 * @param filePath - The path where to write the file
 * @param data - The string data to write to the file
 *
 * @example
 * ```typescript
 * const jsonData = JSON.stringify({ key: 'value' }, null, 2)
 * await writeFileAsync('./output.json', jsonData)
 * ```
 */
export declare function writeFileAsync(filePath: string, data: string): Promise<void>;
/**
 * Resolve an output template (string or function) into an actual path string.
 *
 * - If `outputTemplate` is a function, call it with (language, namespace)
 * - If it's a string, replace placeholders:
 *    - {{language}} or {{lng}} -> language
 *    - {{namespace}} -> namespace (or removed if namespace is undefined)
 * - Normalizes duplicate slashes and returns a platform-correct path.
 */
export declare function getOutputPath(outputTemplate: string | ((language: string, namespace?: string) => string) | undefined, language: string, namespace?: string): string;
/**
 * Dynamically loads a translation file, supporting .json, .js, and .ts formats.
 * @param filePath - The path to the translation file.
 * @returns The parsed content of the file, or null if not found or failed to parse.
 */
export declare function loadTranslationFile(filePath: string): Promise<Record<string, any> | null>;
export declare function loadRawJson5Content(filePath: string): Promise<string | null>;
/**
 * Serializes a translation object into a string based on the desired format.
 * For JSON5, preserves comments and formatting using JsonObjectNode.update().
 */
export declare function serializeTranslationFile(data: Record<string, any>, format?: I18nextToolkitConfig['extract']['outputFormat'], indentation?: number | string, rawContent?: string): string;
//# sourceMappingURL=file-utils.d.ts.map