/* $Id: dp_log.c,v 1.7 2004/04/03 19:57:32 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
** Portions Copyright (C) 2003-2004 Sourcefire, Inc.
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


/*  I N C L U D E S  *****************************************************/
#include "config.h"

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef SOLARIS
    #include <strings.h>
#endif
#include <errno.h>
#include <unistd.h>
#include "barnyard.h"
#include "output-plugins/op_plugbase.h"
#include "dp_plugbase.h"
#include "strlcpyu.h"
#include "util.h"

#include "dp_log.h"
#include "spool.h"

/*  P R O T O T Y P E S  ************************************************/

void LogDpSetup(char *config, DpFunctionalNode *dpfn);
int LogDpReadFileHeader(DpFunctionalNode *dpfn, SpoolFileHandle *sph);
int LogDpReadRecord(SpoolFileHandle *sph);
int LogDpProcessRecord(void *data, DpFunctionalNode *dp);
int LogDpHeaderCompare(DpFunctionalNode *dpfn, void *old_header, 
        void *new_header);
void LogDpPrintRecord(void *data);

int linktype = 0;

void LogDpInit()
{
    PluginInfo pi;

    pi.author = strdup("Martin Roesch <roesch@sourcefire.com>\n"
            "Andrew R. Baker <andrewb@snort.org>");
    pi.version = strdup("1.0");
    pi.type = strdup("log input processor");
    pi.copyright = strdup("(C) Copyright 2001, Martin Roesch"
            " & Andrew R. Baker");
    pi.description = 
        strdup("reads the log spool and reformats for output");
    pi.usage = strdup("dp_log");

    RegisterDp("dp_log", LogDpSetup, "log", &pi);

    free(pi.author);
    free(pi.version);
    free(pi.type);
    free(pi.copyright);
    free(pi.description);
    free(pi.usage);
 
    if(pv.verbose)
        LogMessage("dp_log loaded\n");
    return;
}


void LogDpSetup(char *config, DpFunctionalNode *dpfn)
{
    dpfn->type = strdup("log");
	dpfn->magic = LOG_MAGIC;
	RegisterDpReadRecord(LogDpReadRecord, dpfn);
	RegisterDpReadFileHeader(LogDpReadFileHeader, dpfn);
	RegisterDpProcessRecord(LogDpProcessRecord, dpfn);
	RegisterDpHeaderCompare(LogDpHeaderCompare, dpfn);
}

int LogDpReadFileHeader(DpFunctionalNode *dpfn, SpoolFileHandle *sph)
{
	ssize_t bytes_read;
       
    /* allocate storage space for the spool file header */
    if(!sph->header)
    {
        if(!(sph->header = calloc(1, sizeof(UnifiedLogFileHeader))))
        {
            FatalError("Out of memory reading spool file header "
                    "(wanted %u bytes)",
                    sizeof(UnifiedLogFileHeader));
        }
    }

    bytes_read = read(sph->filedes, sph->header + sph->offset, 
            sizeof(UnifiedLogFileHeader) - sph->offset);
    
    if(bytes_read + sph->offset != sizeof(UnifiedLogFileHeader))
    {
        if(bytes_read == -1)
        {
            /* Read error */
            LogMessage("ERROR: read error: %s\n", strerror(errno));
            return N_FILE_ERROR;
        }

        if(bytes_read == 0)
        {
            return N_READ_EOF;
        }

        sph->offset += bytes_read;
        return N_READ_PARTIAL;
    }

    sph->offset = 0;

    {
        UnifiedLogFileHeader *file_header;
        file_header = (UnifiedLogFileHeader *)sph->header;
        linktype = file_header->linktype;
    }
 
    /* XXX should be part of the start step */
    /* copy the header into the dp plugin */
    if(dpfn->context.file_header != NULL)
        free(dpfn->context.file_header);
    dpfn->context.file_header = SafeAlloc(sizeof(UnifiedLogFileHeader));
    memcpy(dpfn->context.file_header, sph->header, 
            sizeof(UnifiedLogFileHeader));
    
    return 0;
}
            
int LogDpHeaderCompare(DpFunctionalNode *dpfn, void *old_header, 
        void *new_header)
{
    if(memcmp(old_header, new_header, sizeof(UnifiedLogFileHeader)) != 0)
    {
        DataProcessorRestart(dpfn);
    }
    return 0;
}

/* Partial reads should rarely, if ever, happen.  Thus we should not actually
   call lseek very often 
 */

static UnifiedLog unified_log;

int LogDpReadRecord(SpoolFileHandle *sph)
{
    ssize_t bytes_read;
    ssize_t packet_len;
    int fd;
    Record *record;

    if(!sph)
        return -1;  /* Invalid args */

    fd = sph->filedes;
    record = &sph->record;

    if(sph->offset < sizeof(UnifiedLog))
    {   
        /* Read the first portion of the unified log reader */
        bytes_read = read(sph->filedes, &unified_log + sph->offset,
                sizeof(UnifiedLog) - sph->offset);
        if(bytes_read == -1)
        {
            LogMessage("ERROR: read error: %s\n", strerror(errno));
            return N_FILE_ERROR;
        }

        if(bytes_read + sph->offset != sizeof(UnifiedLog))
        {
            if(bytes_read + sph->offset == 0)
            {
                return N_READ_EOF;
            }

            sph->offset += bytes_read;
            return N_READ_PARTIAL;
        }

        /* Extract packet length */
        packet_len = unified_log.pkth.caplen;
        if(packet_len >= 0x1ffff)
        {
            LogMessage("ERROR: Invalid packet length: %u\n", packet_len);
            return N_FILE_ERROR;
        }

        /* Allocate record storage based on packet length */
        sph->record.dynamic = 1;
        sph->record.data = calloc(1, sizeof(UnifiedLog) + packet_len + 2);
        if(!sph->record.data)
        {
            FatalError("Out of memory (wanted %u bytes)\n", 
                    sizeof(UnifiedLog) + packet_len + 2);
        }
        memcpy(sph->record.data, &unified_log, sizeof(UnifiedLog));

        sph->offset = sizeof(UnifiedLog) + 2;
    }

    if(sph->offset >= sizeof(UnifiedLog) + 2)
    {
        /* In case we don't have it already */
        packet_len = unified_log.pkth.caplen;

        bytes_read = read(sph->filedes, sph->record.data + sph->offset, 
                packet_len + 2 + sizeof(UnifiedLog) - sph->offset);

        if(bytes_read == -1)
        {
            LogMessage("ERROR: read error: %s\n", strerror(errno));
            return N_FILE_ERROR;
        }
            
        if(bytes_read + sph->offset != sizeof(UnifiedLog) + 2 + packet_len)
        {
            sph->offset += bytes_read;
            return N_READ_PARTIAL;
        }

        sph->offset = 0;
        return 0;
    }

    /* Should be unreachable */
    LogMessage("Should be unreachable\n");
    return -1;
}


int LogDpProcessRecord(void *data, DpFunctionalNode *dp)
{
    if(data == NULL)
	{
#ifdef DEBUG
		printf("NULL Argument to ProcessUnifiedLogRecord\n");
#endif
		return 1;
	}
	
#ifdef DEBUG
    LogDpPrintRecord(data);
#endif

	/* Call the output plugins */
    CallOutputPlugins(dp->oList, data);

    return 0;
}

void LogDpPrintRecord(void *data)
{
	UnifiedLogRecord *record;
    if(data == NULL)
        return;

	record = (UnifiedLogRecord *)data;
    LogMessage("Log->sig_generator  = %x\n", record->log.event.sig_generator);
    LogMessage("Log->sig_id         = %x\n", record->log.event.sig_id);
    LogMessage("Log->sig_rev        = %x\n", record->log.event.sig_rev);
    LogMessage("Log->classification = %x\n", record->log.event.classification);
    LogMessage("Log->priority       = %x\n", record->log.event.priority);
    LogMessage("Log->event_id       = %x\n", record->log.event.event_id);
    LogMessage("Log->reference      = %x\n", record->log.event.event_reference);
    LogMessage("Log->ref_sec        = %x\n", record->log.event.ref_time.tv_sec);
    LogMessage("Log->ref_usec       = %x\n", record->log.event.ref_time.tv_usec);
    LogMessage("Log->flags          = 0x%08X\n", (u_int32_t) record->log.flags);
    LogMessage("Log->sec            = %x\n", record->log.pkth.ts.tv_sec);
    LogMessage("Log->usec           = %x\n", record->log.pkth.ts.tv_usec);
    LogMessage("Log->pktlen         = %x\n", record->log.pkth.pktlen);
    LogMessage("Log->caplen         = %u\n", record->log.pkth.caplen);
    LogMessage("------------------------------------------------------\n");
}
