/* $Id: dp_alert.c,v 1.6 2004/02/20 01:59:48 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
** Portions Copyright (C) 2003-2004 Sourcefire, Inc.
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

/*  I N C L U D E S  *****************************************************/
#include "config.h"

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef SOLARIS
    #include <strings.h>
#endif
#include <errno.h>
#include <unistd.h>
#include "barnyard.h"
#include "output-plugins/op_plugbase.h"
#include "dp_plugbase.h"
#include "strlcpyu.h"
#include "util.h"
#include "event.h"
#include "dp_alert.h"
#include "spool.h"

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

/*  P R O T O T Y P E S  ************************************************/
void AlertDpSetup(char *, DpFunctionalNode *);
int AlertDpReadFileHeader(DpFunctionalNode *, SpoolFileHandle *sph);
int AlertDpReadRecord(SpoolFileHandle *sph);
int AlertDpProcessRecord(void *, DpFunctionalNode *dpfn);
void AlertDpPrintRecord(void *);

/*-----------------------Local Global Variables -----------------------*/
/* this is safe since we are single threaded and can only run one instance
 * of a data processor
 */
static void *buffer;


void AlertDpInit()
{
    PluginInfo pi;

    pi.author = strdup("Martin Roesch <roesch@sourcefire.com>\n"
            "Andrew R. Baker <andrewb@snort.org>");
    pi.version = strdup("1.0");
    pi.type = strdup("alert input processor");
    pi.copyright = strdup("(C) Copyright 2001, Martin Roesch"
            " & Andrew R. Baker");
    pi.description = 
        strdup("reads the alert (event) spool and reformats for output");
    pi.usage = strdup("dp_alert");
    
    /* register the name, setup function and data type */
    RegisterDp("dp_alert", AlertDpSetup, "alert", &pi);
    
        
    free(pi.author);
    free(pi.version);
    free(pi.type);
    free(pi.copyright);
    free(pi.description);
    free(pi.usage);

    if(pv.verbose)
        LogMessage("dp_alert loaded\n");
    return;
}

/* Add the input plugin to the DataProcessors list */
void AlertDpSetup(char *config, DpFunctionalNode *dpfn)
{
    dpfn->type = strdup("alert");
	dpfn->magic = ALERT_MAGIC;
	RegisterDpReadRecord(AlertDpReadRecord, dpfn);
	RegisterDpReadFileHeader(AlertDpReadFileHeader, dpfn);
	RegisterDpProcessRecord(AlertDpProcessRecord, dpfn);
    /* allocate the global buffer */
    buffer = SafeAlloc(sizeof(UnifiedAlertRecord));
}


int AlertDpReadFileHeader(DpFunctionalNode *dpfn, SpoolFileHandle *sph)
{
    ssize_t bytes_read;

    if(!sph->header)
    {
        if(!(sph->header = calloc(1, sizeof(UnifiedAlertFileHeader))))
        {
            FatalError("Out of memory reading spool file header "
                    "(wanted %u bytes)", sizeof(UnifiedAlertFileHeader));
        }
    }

    bytes_read = read(sph->filedes, sph->header + sph->offset, 
            sizeof(UnifiedAlertFileHeader) - sph->offset);

    if(bytes_read == -1)
    {
        /* read error */
        LogMessage("ERROR: read error: %s\n", strerror(errno));
        return N_FILE_ERROR;
    }
    
    if(bytes_read + sph->offset != sizeof(UnifiedAlertFileHeader))
    {
        if(bytes_read + sph->offset == 0)
        {
            return N_READ_EOF;
        }

        sph->offset += bytes_read;
        return N_READ_PARTIAL;
    }

    sph->offset = 0;

#ifdef DEBUG
    {
        UnifiedAlertFileHeader *file_header;
        file_header = (UnifiedAlertFileHeader *)sph->header;
        printf("Opened Unified Alert File \"%s\", header:\n", sph->filepath);
        printf(" Version.major  = %d\n", file_header->version_major);
        printf(" Version.minor  = %d\n", file_header->version_minor);
        printf(" timezone       = %d\n", file_header->timezone);
        printf("====================================="
                "===============================\n"); 
    }
#endif

    /* copy the header into the dp plugin */
    if(dpfn->context.file_header != NULL)
        free(dpfn->context.file_header);
    dpfn->context.file_header = SafeAlloc(sizeof(UnifiedAlertFileHeader));
    memcpy(dpfn->context.file_header, sph->header, 
            sizeof(UnifiedAlertFileHeader));

    return 0;
}
            
/* Partial reads should rarely, if ever, happen.  Thus we should not actually
   call lseek very often 
 */
   
int AlertDpReadRecord(SpoolFileHandle *sph)
{
    ssize_t bytes_read;

    if(!sph)
        return -1;  /* Invalid args */

    bytes_read = read(sph->filedes, buffer + sph->offset, 
            sizeof(UnifiedAlertRecord) - sph->offset);

    if(bytes_read == -1)
    {
        /* Read error */
        LogMessage("ERROR: read error: %s\n", strerror(errno));
        return N_FILE_ERROR;
    }

    if(bytes_read + sph->offset != sizeof(UnifiedAlertRecord))
    {
        if(bytes_read + sph->offset == 0)
        {
            return N_READ_EOF;
        }

        sph->offset += bytes_read;
        return N_READ_PARTIAL;
    }
    
    sph->offset = 0;
    sph->record.data = buffer;
    sph->record.dynamic = 0;
    return 0;
}

int AlertDpProcessRecord(void *data, DpFunctionalNode *dpfn)
{
    if(data == NULL)
    {
#ifdef DEBUG
        printf("NULL Argument to ProcessUnifiedAlertRecord\n");
#endif
        return 1;
    }
    //AlertDpPrintRecord(data);
    

    /* Call the output plugins */
    CallOutputPlugins(dpfn->oList, data);

    return 0;
}

void AlertDpPrintRecord(void *data)
{
    struct in_addr in;
    UnifiedAlertRecord *record;

    if(data == NULL)
        return;
    
    record = (UnifiedAlertRecord *)data;
    printf("Event->sig_generator  = %d\n", record->event.sig_generator);
    printf("Event->sig_id         = %d\n", record->event.sig_id);
    printf("Event->sig_rev        = %d\n", record->event.sig_rev);
    printf("Event->classification = %d\n", record->event.classification);
    printf("Event->priority       = %d\n", record->event.priority);
    printf("Event->id             = %d\n", record->event.event_id);
    printf("Event->reference      = %d\n", record->event.event_reference);
    printf("Alert->ts.tv_sec      = %lu\n", record->ts.tv_sec);
    printf("Alert->ts.tv_usec     = %lu\n", record->ts.tv_usec);
    in.s_addr = record->sip;
    printf("Alert->sip            = %s\n", inet_ntoa(in));
    in.s_addr = record->dip;
    printf("Alert->dip            = %s\n", inet_ntoa(in));
    printf("Alert->sp             = %d\n", record->sp);
    printf("Alert->dp             = %d\n", record->dp);
    printf("Alert->protocol       = %d\n", record->protocol);
    printf("Alert->flags          = 0x%X\n", record->flags); 
    printf("------------------------------------------------------\n");
}
