/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { removeListItem } from '../util';
import { AbstractControl, isOptionsObj, pickAsyncValidators, pickValidators } from './abstract_model';
function isFormControlState(formState) {
    return typeof formState === 'object' && formState !== null &&
        Object.keys(formState).length === 2 && 'value' in formState && 'disabled' in formState;
}
export const FormControl = (class FormControl extends AbstractControl {
    constructor(
    // formState and defaultValue will only be null if T is nullable
    formState = null, validatorOrOpts, asyncValidator) {
        super(pickValidators(validatorOrOpts), pickAsyncValidators(asyncValidator, validatorOrOpts));
        /** @publicApi */
        this.defaultValue = null;
        /** @internal */
        this._onChange = [];
        /** @internal */
        this._pendingChange = false;
        this._applyFormState(formState);
        this._setUpdateStrategy(validatorOrOpts);
        this._initObservables();
        this.updateValueAndValidity({
            onlySelf: true,
            // If `asyncValidator` is present, it will trigger control status change from `PENDING` to
            // `VALID` or `INVALID`.
            // The status should be broadcasted via the `statusChanges` observable, so we set
            // `emitEvent` to `true` to allow that during the control creation process.
            emitEvent: !!this.asyncValidator
        });
        if (isOptionsObj(validatorOrOpts) &&
            (validatorOrOpts.nonNullable || validatorOrOpts.initialValueIsDefault)) {
            if (isFormControlState(formState)) {
                this.defaultValue = formState.value;
            }
            else {
                this.defaultValue = formState;
            }
        }
    }
    setValue(value, options = {}) {
        this.value = this._pendingValue = value;
        if (this._onChange.length && options.emitModelToViewChange !== false) {
            this._onChange.forEach((changeFn) => changeFn(this.value, options.emitViewToModelChange !== false));
        }
        this.updateValueAndValidity(options);
    }
    patchValue(value, options = {}) {
        this.setValue(value, options);
    }
    reset(formState = this.defaultValue, options = {}) {
        this._applyFormState(formState);
        this.markAsPristine(options);
        this.markAsUntouched(options);
        this.setValue(this.value, options);
        this._pendingChange = false;
    }
    /**  @internal */
    _updateValue() { }
    /**  @internal */
    _anyControls(condition) {
        return false;
    }
    /**  @internal */
    _allControlsDisabled() {
        return this.disabled;
    }
    registerOnChange(fn) {
        this._onChange.push(fn);
    }
    /** @internal */
    _unregisterOnChange(fn) {
        removeListItem(this._onChange, fn);
    }
    registerOnDisabledChange(fn) {
        this._onDisabledChange.push(fn);
    }
    /** @internal */
    _unregisterOnDisabledChange(fn) {
        removeListItem(this._onDisabledChange, fn);
    }
    /** @internal */
    _forEachChild(cb) { }
    /** @internal */
    _syncPendingControls() {
        if (this.updateOn === 'submit') {
            if (this._pendingDirty)
                this.markAsDirty();
            if (this._pendingTouched)
                this.markAsTouched();
            if (this._pendingChange) {
                this.setValue(this._pendingValue, { onlySelf: true, emitModelToViewChange: false });
                return true;
            }
        }
        return false;
    }
    _applyFormState(formState) {
        if (isFormControlState(formState)) {
            this.value = this._pendingValue = formState.value;
            formState.disabled ? this.disable({ onlySelf: true, emitEvent: false }) :
                this.enable({ onlySelf: true, emitEvent: false });
        }
        else {
            this.value = this._pendingValue = formState;
        }
    }
});
export const UntypedFormControl = FormControl;
export const isFormControl = (control) => control instanceof FormControl;
//# sourceMappingURL=data:application/json;base64,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