///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// ==========================================================================
//  includes
// ==========================================================================

#include "rheolef/space.h"
#include "rheolef/domain.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h"
#include "rheolef/tiny_matvec.h"
#include "rheolef/tensor.h"
#include "rheolef/piola_v1.h"
#include "rheolef/banded_level_set.h"
#include "../basis/P1pw_numbering.h"
using namespace std;
namespace rheolef { 

// ==========================================================================

space::space(const const_space_component& Vi)
  : smart_pointer<spacerep> (new_macro(spacerep((*(Vi._pV)).data(), Vi._i_comp)))
{
}
space
space::operator=(const const_space_component &Vi)
{
    *this = space(Vi);
    return *this;
}
space
pow (const space& V, space::size_type n)
{
    if (n == 0) return space();
    space T = V;
    for (space::size_type i = 2; i <= n; i++) 
        T = T*V;
    return T;
}
// ==========================================================================
// inquire: what is the number of degrees of freedom for a space
//          with given approximation and mesh ?
// ==========================================================================

/* static */
spacerep::size_type
spacerep::inquire_size(const geo& g, const numbering& numb)
{
    const size_type  map_dim        = g.map_dimension();
    const size_type* mesh_n_geo     = g.get_geo_counter();
    const size_type* mesh_n_element = g.get_element_counter();
    return numb.ndof (map_dim, mesh_n_geo, mesh_n_element);
}
spacerep::size_type
spacerep::_n_dof(size_type i_comp)
{
    return space::inquire_size (get_geo(), _numbering[i_comp]);
}
spacerep::size_type
spacerep::_n_global_dof(size_type i_comp)
{
    return space::inquire_size (get_global_geo(), _numbering[i_comp]);
}
// ==========================================================================
//  accessor
// ==========================================================================

fem_helper::valued_field_type
spacerep::get_valued_type () const
{
    return _valued;
}
string
spacerep::get_valued () const
{
    return fem_helper::valued_field_name (_valued);
}
// ==========================================================================
//  constructors
// ==========================================================================

// space();
spacerep::spacerep()
  : _g(), 
    _is_frozen(false),
    _has_locked(false),
    _b(),
    _numbering(),
    _tr_p1(),
    _tr_p1_value(),
    _name(),
    _start(1),
    _start_blocked(1),
    _start_unknown(1),
    _is_blocked(),
    _is_periodic(),
    _period_assoc(),
    _num(),
    _dof_oriented_as_element(),
    _dof_orientation_initialized(false),
    _is_locked(),
    _locked_dir(),
    _is_on_boundary_domain(false),
    _boundary_domain(),
    _global_geo(),
    _global_dof(),
    _boundary_dof(),
    _start_global_dof(),
    _valued(fem_helper::max_valued_field)
{
}
// --------------------------------------------------------------------------

// space(e);
spacerep::spacerep(const spacerep& e)
  : _g(e._g), 
    _is_frozen(e._is_frozen),
    _has_locked(e._has_locked),
    _b(e._b),
    _numbering(e._numbering),
    _tr_p1(),
    _tr_p1_value(),
    _name(e._name),
    _start(e._start),
    _start_blocked(e._start_blocked),
    _start_unknown(e._start_unknown),
    _is_blocked(e._is_blocked),
    _is_periodic(e._is_periodic),
    _period_assoc(e._period_assoc),
    _num(e._num),
    _dof_oriented_as_element(),
    _dof_orientation_initialized(false),
    _is_locked(e._is_locked),
    _locked_dir(e._locked_dir),
    _is_on_boundary_domain(e._is_on_boundary_domain),
    _boundary_domain(e._boundary_domain),
    _global_geo(e._global_geo),
    _global_dof(e._num.size()),
    _boundary_dof(e._boundary_dof.size()),
    _start_global_dof(e._start_global_dof.size()),
    _valued(fem_helper::max_valued_field)
{
  fatal_macro ("\"spacerep\" physical copy may not occur !");
}
// --------------------------------------------------------------------------

// space(g, "P1", "vector");
// space(g, "P1", "scalar"); <==> space(g, "P1")
spacerep::spacerep(const geo& g, const string& approx_name, const string& valued)
  : _g(g),
    _is_frozen(false),
    _has_locked(false),
    _b(),
    _numbering(),
    _tr_p1("P1"),
    _tr_p1_value(),
    _name(),
    _start(),
    _start_blocked(),
    _start_unknown(),
    _is_blocked(),
    _is_periodic(),
    _period_assoc(),
    _num(),
    _dof_oriented_as_element(),
    _dof_orientation_initialized(false),
    _is_locked(),
    _locked_dir(),
    _is_on_boundary_domain(false),
    _boundary_domain(),
    _global_geo(),
    _global_dof(),
    _boundary_dof(),
    _start_global_dof(),
    _valued(fem_helper::max_valued_field)
{
  check_macro(approx_name!="P1pw", 
  	"Piecewise continuous spaces can be initalized only with the" << endl <<
	"space(geo,approx,interface,region[,valued_type]) constructor");
  string basis_name = fem_helper::basis_name     (approx_name);
  string num_name   = fem_helper::numbering_name (approx_name);
  size_type dim = g.dimension();
  g.may_have_version_2();
  // n = number of components of the field
  size_type n;
  if (valued == "tensor4") { // A_ijkl with A_ijkl=A_jikl, A_ijkl=A_ijlk and A_ijkl=A_klij
     if ((_g.coordinate_system() == "rz")||(_g.coordinate_system() == "zr")) {
       n = 10; // add all the theta,theta components
     } else {
       switch (dim) {
       case 1 : n = 1; break;
       case 2 : n = 6; break;
       default : n = 21; break;
       }
     }
  } else if (valued == "tensor") {
     if ((_g.coordinate_system() == "rz")||(_g.coordinate_system() == "zr"))
       n = 4; // add the \tau_{\theta,\theta}" component
     else
       n = dim*(dim+1)/2;
  } else if (valued == "unsymmetric_tensor") {
     if ((_g.coordinate_system() == "rz")||(_g.coordinate_system() == "zr"))
       n = 5;
     else
       n = dim*dim;
  } else if (valued == "vector") {
     n = dim;
  } else if (valued == "scalar") {
     n = 1;
  } else {
     error_macro ("unsupported valued space `" << valued << "'");
  }
  _valued = fem_helper::valued_field (valued);
  _b.resize (n);
  _numbering.resize (n);
  _name.resize (n);
  _tr_p1_value.resize (n);
  _start.resize (n+1);
  _start_blocked.resize (n+1, numeric_limits<size_type>::max());
  _start_unknown.resize (n+1, numeric_limits<size_type>::max());
  _start_global_dof.resize (n+1);
  numbering numb (num_name);
  basis bi (basis_name);
  for (size_type i = 0; i < n; i++) {
      _name[i] = approx_name;
      _b[i] = bi;
      _numbering[i] = numb;
      _tr_p1_value[i].set (_b[i], _tr_p1);
  }
  _start[0] = 0;
  // counts the number of dof for each component
  for (size_type i = 0; i < n; i++) {
      size_type ndof_i = _n_dof(i);
      _start[i+1] = _start[i] + ndof_i;
  }
  size_type ndof = _start[n];
  resize(_num, ndof, numeric_limits<size_type>::max());
  resize(_period_assoc, ndof, numeric_limits<size_type>::max());
  resize(_is_blocked, ndof, false);
  resize(_is_periodic, ndof, false);
  resize(_global_dof, ndof, size_type(0));
  size_type n_global_dof_i = 0;
  if (n >= 1) {
     n_global_dof_i = _n_global_dof(0);
  }
  size_type nglobaldof = n*n_global_dof_i;
  resize(_boundary_dof, nglobaldof, size_type(0));
  _build_global_dof_mapping();
}
// --------------------------------------------------------------------------

spacerep::spacerep(const geo& g, const std::string& approx_name, 
	const domain& interface, const domain& region, const std::string& valued)
  : _g(g),
    _is_frozen(false),
    _has_locked(false),
    _b(),
    _numbering(),
    _tr_p1("P1"),
    _tr_p1_value(),
    _name(),
    _start(),
    _start_blocked(),
    _start_unknown(),
    _is_blocked(),
    _is_periodic(),
    _period_assoc(),
    _num(),
    _dof_oriented_as_element(),
    _dof_orientation_initialized(false),
    _is_locked(),
    _locked_dir(),
    _is_on_boundary_domain(false),
    _boundary_domain(),
    _global_geo(),
    _global_dof(),
    _boundary_dof(),
    _start_global_dof(),
    _valued(fem_helper::max_valued_field)
{
  check_macro(approx_name=="P1pw", "Wrong approximation " << approx_name 
	<< " for space with interfacial jump");
  string basis_name = fem_helper::basis_name     (approx_name);
  string num_name   = fem_helper::numbering_name (approx_name);
  if (valued != "tensor" && valued != "unsymmetric_tensor" &&
      valued != "vector" && valued != "scalar") {
     error_macro ("unsupported valued space `" << valued << "'");
  }
  _valued = fem_helper::valued_field (valued);
  size_type dim = g.dimension();
  g.may_have_version_2();
  // n = number of components of the field
  size_type n;
  if (valued == "tensor") {
     if ((_g.coordinate_system() == "rz")||(_g.coordinate_system() == "zr"))
       n = 4; // add the \tau_{\theta,\theta}" component
     else
       n = dim*(dim+1)/2;
  } else if (valued == "unsymmetric_tensor") {
     if ((_g.coordinate_system() == "rz")||(_g.coordinate_system() == "zr"))
       n = 5;
     else
       n = dim*dim;
  } else if (valued == "vector") {
     n = dim;
  } else {
     n = 1;
  }
  map<size_type, tiny_vector<size_type> > special_elements;
  map<size_type, size_type> node_index_on_interface;
  g.jump_interface(interface, region, special_elements, node_index_on_interface);
  const size_type  map_dim        = g.map_dimension();
  const size_type* mesh_n_geo     = g.get_geo_counter();
  const size_type* mesh_n_element = g.get_element_counter();
  size_type nb_interface_nodes=node_index_on_interface.size();
  _b.resize (n);
  _numbering.resize (n);
  _name.resize (n);
  _tr_p1_value.resize (n);
  _start.resize (n+1);
  _start_blocked.resize (n+1, numeric_limits<size_type>::max());
  _start_unknown.resize (n+1, numeric_limits<size_type>::max());
  _start_global_dof.resize (n+1);
  // TODO: avoid pointer manips & add to the numbering class a generic "initialize(geo)" method ?
  numbering_P1pw* p1pw_ptr = new_macro (numbering_P1pw);
  p1pw_ptr -> initialize (nb_interface_nodes, special_elements);
  numbering numb (p1pw_ptr);
  basis bi (basis_name);
  for (size_type i = 0; i < n; i++) {
      _name[i] = approx_name;
      _b[i] = bi;
      _numbering[i] = numb;
      _tr_p1_value[i].set (_b[i], _tr_p1);
  }
  _start[0] = 0;
  // counts the number of dof for each component
  for (size_type i = 0; i < n; i++) {
      size_type ndof_i = (_numbering[i].ndof(map_dim,mesh_n_geo,mesh_n_element));
      _start[i+1] = _start[i] + ndof_i;
  }
  size_type ndof = _start[n];
  resize(_num, ndof, numeric_limits<size_type>::max());
  resize(_period_assoc, ndof, numeric_limits<size_type>::max());
  resize(_is_blocked, ndof, false);
  resize(_is_periodic, ndof, false);
  resize(_global_dof, ndof, size_type(0));
  size_type n_global_dof_i = 0;
  if (n >= 1) {
     n_global_dof_i = _n_global_dof(0);
  }
  size_type nglobaldof = n*n_global_dof_i;
  resize(_boundary_dof, nglobaldof, size_type(0));
  _build_global_dof_mapping();
}


// --------------------------------------------------------------------------
// space(g, g["bottom"], "P1")
spacerep::spacerep(const geo& g, const domain& d, const string& approx_name)
  : _g(geo(g,d)), 
    _is_frozen(false),
    _has_locked(false),
    _b(1),
    _numbering(1),
    _tr_p1("P1"),
    _tr_p1_value(1),
    _name(1),
    _start(2),
    _start_blocked(2,numeric_limits<size_type>::max()),
    _start_unknown(2,numeric_limits<size_type>::max()),
    _is_blocked(),
    _is_periodic(),
    _period_assoc(),
    _num(),
    _dof_oriented_as_element(),
    _dof_orientation_initialized(false),
    _is_locked (),
    _locked_dir           (),
    _is_on_boundary_domain(true),
    _boundary_domain(d),
    _global_geo(g),
    _global_dof(),
    _boundary_dof(),
    _start_global_dof(2),
    _valued(fem_helper::scalar)
{
  check_macro(approx_name!="P1pw", 
  	"Piecewise continuous spaces can be initalized only with the" << endl <<
	"space(geo,approx,interface,region[,valued_type]) constructor");
  string basis_name = fem_helper::basis_name     (approx_name);
  string num_name   = fem_helper::numbering_name (approx_name);
  g.may_have_version_2();
  _name[0] = approx_name; 
  _b[0] = basis(basis_name) ; 
  _numbering[0] = numbering(num_name) ; 
  _tr_p1_value[0].set (_b[0], _tr_p1); 
  size_type ndof = _n_dof(0);
  _start[0] = 0;
  _start[1] = ndof;
  resize(_num, ndof, numeric_limits<size_type>::max()) ;
  resize(_period_assoc, ndof, numeric_limits<size_type>::max()) ;
  resize(_is_blocked, ndof, false) ;
  resize(_is_periodic, ndof, false) ;
  resize(_global_dof, ndof, numeric_limits<size_type>::max()) ;
  size_type nglobaldof = _n_global_dof(0);
  resize(_boundary_dof, nglobaldof, numeric_limits<size_type>::max());
  _build_global_dof_mapping();
}

// dirty hack for banded level set 
spacerep::spacerep(const field& phi, const string& tag, const string& approx_name)
  : _g(), 
    _is_frozen(false),
    _has_locked(false),
    _b(1),
    _numbering(1),
    _tr_p1("P1"),
    _tr_p1_value(1),
    _name(1),
    _start(2),
    _start_blocked(2,numeric_limits<size_type>::max()),
    _start_unknown(2,numeric_limits<size_type>::max()),
    _is_blocked(),
    _is_periodic(),
    _period_assoc(),
    _num(),
    _dof_oriented_as_element(),
    _dof_orientation_initialized(false),
    _is_locked (),
    _locked_dir           (),
    _is_on_boundary_domain(true),
    _boundary_domain(),
    _global_geo(phi.get_geo()),
    _global_dof(),
    _boundary_dof(),
    _start_global_dof(2),
    _valued(fem_helper::scalar)
{
  check_macro(tag=="banded_level_set", "This constructor is for banded level set only");
  banded_level_set(phi, _g, _boundary_domain);
  string basis_name = fem_helper::basis_name     (approx_name);
  string num_name   = fem_helper::numbering_name (approx_name);
  _g.may_have_version_2();
  _name[0] = approx_name; 
  _b[0] = basis(basis_name) ; 
  _numbering[0] = numbering(num_name) ; 
  _tr_p1_value[0].set (_b[0], _tr_p1); 
  size_type ndof = _n_dof(0);
  _start[0] = 0;
  _start[1] = ndof;
  resize(_num, ndof, numeric_limits<size_type>::max()) ;
  resize(_period_assoc, ndof, numeric_limits<size_type>::max()) ;
  resize(_is_blocked, ndof, false) ;
  resize(_is_periodic, ndof, false) ;
  resize(_global_dof, ndof, numeric_limits<size_type>::max()) ;
  size_type nglobaldof = _n_global_dof(0);
  resize(_boundary_dof, nglobaldof, numeric_limits<size_type>::max());
  _build_global_dof_mapping();
  block("isolated");
}
// --------------------------------------------------------------------------

// X*Y
spacerep::spacerep(const spacerep& X, const spacerep& Y)
  : _g(X.get_geo()),
    _is_frozen(true),
    _has_locked(false),
    _b             (X.n_component()  + Y.n_component()),
    _numbering     (X.n_component()  + Y.n_component()),
    _tr_p1         ("P1"),
    _tr_p1_value   (X.n_component()  + Y.n_component()),
    _name          (X._name.size() + Y._name.size()),
    _start         (X.n_component()  + Y.n_component() + 1), 
    _start_blocked (X.n_component()  + Y.n_component() + 1), 
    _start_unknown (X.n_component()  + Y.n_component() + 1), 
    _is_blocked    (X.size() + Y.size()), 
    _is_periodic   (X.size() + Y.size()), 
    _period_assoc  (X.size() + Y.size()),
    _num           (X.size() + Y.size()),
    _dof_oriented_as_element(),
    _dof_orientation_initialized(false),
    _is_locked (),
    _locked_dir(),
    _is_on_boundary_domain (X._is_on_boundary_domain),
    _boundary_domain       (X._boundary_domain),
    _global_geo            (X._global_geo),
    _global_dof    (X._global_dof.size()   + Y._global_dof.size()),
    _boundary_dof  (X._boundary_dof.size() + Y._boundary_dof.size()),
    _start_global_dof(X.n_component()  + Y.n_component() + 1),
    _valued(fem_helper::hybrid)
{
      check_macro(!X.has_locked() && !Y.has_locked(),
	  	"Cartesian space product unsupported with locked-component feature");
      X.freeze();
      Y.freeze();
      size_type nx = X.n_component();
      size_type ny = Y.n_component();
      _start[0]            = X._start           [0];
      _start_blocked[0]    = X._start_blocked   [0];
      _start_unknown[0]    = X._start_unknown   [0];
      _start_global_dof[0] = X._start_global_dof[0];
      for (size_type i = 0; i < nx; i++) {
	  _start            [i+1] = X._start[i+1];
	  _start_blocked    [i+1] = X._start_blocked[i+1];
	  _start_unknown    [i+1] = X._start_unknown[i+1];
	  _start_global_dof [i+1] = X._start_global_dof[i+1];
          _b                [i]   = X.get_basis(i);
          _numbering        [i]   = X.get_numbering(i);
          _name             [i]   = X.get_approx(i);
          _tr_p1_value      [i].set (_b[i], _tr_p1);
      }
      size_type last_x         = X._start           [nx];
      size_type last_blocked_x = X._start_blocked   [nx];
      size_type last_unknown_x = X._start_unknown   [nx];
      size_type last_global_x  = X._start_global_dof[nx];
      for (size_type i = 0; i < ny; i++) {
	  _start            [i+nx+1] = last_x         + Y._start           [i+1];
	  _start_blocked    [i+nx+1] = last_blocked_x + Y._start_blocked   [i+1];
	  _start_unknown    [i+nx+1] = last_unknown_x + Y._start_unknown   [i+1];
	  _start_global_dof [i+nx+1] = last_global_x  + Y._start_global_dof[i+1];
          _b                [i+nx]   = Y.get_basis(i) ;
          _numbering        [i+nx]   = Y.get_numbering(i);
          _name             [i+nx]   = Y.get_approx(i);
          _tr_p1_value      [i+nx].set (_b[i+nx], _tr_p1);
      }
      for (size_type i = 0; i < X.size(); i++) { 
	  _num[i]          = X.index(i);
	  _period_assoc[i] = X.period_association(i);
	  _is_blocked[i]   = X.is_blocked(i) ;
	  _is_periodic[i]  = X.is_periodic(i) ;
      }
      size_type size_x      = X.size();
      size_type n_unknown_x = X.n_unknown();
      size_type n_blocked_x = X.n_blocked();
      for (size_type i = 0; i < Y.size(); i++) { 
	  if (!Y.is_blocked(i)) {
	      _num [i+size_x] = n_unknown_x + Y.index(i); 
	  } else {
	      _num [i+size_x] = n_blocked_x + Y.index(i); 
	  }
	  _is_blocked  [i+size_x] = Y.is_blocked(i);
	  _is_periodic [i+size_x] = Y.is_periodic(i);
	  _period_assoc[i+size_x] = size_x + Y.period_association(i);
      }
      // set global dof numbering for spaces on boundary
      size_type small_size_x = X._global_dof.size();
      size_type large_size_x = X._boundary_dof.size();
      for (size_type i=0 ; i < X._global_dof.size(); i++) {
	  _global_dof [i] = X._global_dof [i];
      }
      for (size_type i=0 ; i < Y._global_dof.size(); i++) {
	  _global_dof [small_size_x + i] = large_size_x + Y._global_dof [i];
      }
      for (size_type i=0 ; i < X._boundary_dof.size(); i++) {
	  _boundary_dof [i] = X._boundary_dof [i];
      }
      for (size_type i=0 ; i < Y._boundary_dof.size(); i++) {
	  _boundary_dof [large_size_x + i] = small_size_x + Y._boundary_dof [i];
      }
      // detect vector-valued space
      size_type d = dimension();
      bool same_approx = true;
      for (size_type i_comp = 1; i_comp < n_component() && same_approx; i_comp++) {
          if (get_approx(i_comp) != get_approx(0)) same_approx = false;
      }
      _valued = fem_helper::scalar;
      fem_helper::coordinate_type sys_coord = coordinate_system_type();
      if (same_approx && n_component() > 1) {
	  if (n_component() == d) {
              _valued = fem_helper::vectorial;
          } else if ((n_component() == d*(d+1)/2 && sys_coord == fem_helper::cartesian) ||
		     (n_component() == 4 && sys_coord == fem_helper::axisymmetric_zr)   ||
		     (n_component() == 4 && sys_coord == fem_helper::axisymmetric_rz)   ) {
	      _valued = fem_helper::tensorial;
          } else if ((n_component() == d*d && sys_coord == fem_helper::cartesian)     ||
		     (n_component() == 5 && sys_coord == fem_helper::axisymmetric_zr) ||
		     (n_component() == 5 && sys_coord == fem_helper::axisymmetric_rz) ) {
	      _valued = fem_helper::unsymmetric_tensorial;
          }
      }
}
// --------------------------------------------------------------------------

// space Xi = X[i];
spacerep::spacerep(const spacerep& X, size_type i_comp)
  : _g(X.get_geo()),
    _is_frozen(false),
    _has_locked(false),
    _b             (1),
    _numbering     (1),
    _tr_p1         ("P1"),
    _tr_p1_value   (1),
    _name          (1),
    _start         (2), 
    _start_blocked (2), 
    _start_unknown (2), 
    _is_blocked     (X.size_component(i_comp)), 
    _is_periodic    (X.size_component(i_comp)), 
    _period_assoc   (X.size_component(i_comp), numeric_limits<size_type>::max()),
    _num            (X.size_component(i_comp), numeric_limits<size_type>::max()),
    _dof_oriented_as_element(),
    _dof_orientation_initialized(false),
    _is_locked (),
    _locked_dir(),
    _is_on_boundary_domain (X._is_on_boundary_domain),
    _boundary_domain       (X._boundary_domain),
    _global_geo            (X._global_geo),
    _global_dof    (X.size_component(i_comp)),
    _boundary_dof  (X.n_global_dof_component()),
    _start_global_dof(2),
    _valued(fem_helper::scalar)
{
  _b[0] = X.get_basis(i_comp);
  _numbering[0] = X.get_numbering(i_comp);
  _name[0] = X.get_approx(i_comp);
  _tr_p1_value[0].set (_b[0], _tr_p1);
  size_type ndof = X.size_component(i_comp);
  _start[0] = 0;
  _start[1] = ndof;
  _start_blocked[0] = 0;
  _start_blocked[1] = X.n_blocked_component(i_comp);
  _start_unknown[0] = 0;
  _start_unknown[1] = X.n_unknown_component(i_comp);
  for (size_type p = X.start(i_comp); p < X.start(i_comp+1); p++) {
    _is_blocked  [p - X.start(i_comp)] = X._is_blocked   [p];
    _is_periodic [p - X.start(i_comp)] = X._is_periodic  [p];
    _period_assoc[p - X.start(i_comp)] = X._period_assoc [p] - X.start(i_comp);
  }
  _build_global_dof_mapping();
  freeze();
}
// --------------------------------------------------------------------------

spacerep::~spacerep()
{
}
// ==========================================================================
//  dofs
// ==========================================================================

static
void
internal_set_dof(
  	const geo&                     g,
  	const basis&                   b,  
  	const numbering&               numb,  
	const geo_element&             K,
	tiny_vector<space::size_type>& idx,
	const geo::size_type*          mesh_n_geo,
        const geo::size_type*          mesh_n_element,
        geo::size_type                 first)
{
  typedef spacerep::size_type size_type;
  size_type ndof = b.size(K.variant());
  for (size_type i = 0; i < ndof; i++) {	
      idx[i] = first + numb.idof (mesh_n_geo, mesh_n_element, K, i);
    }
}
static
void
internal_set_dof(
  	const geo&                     g,
  	const basis&                   b,  
  	const numbering&               numb,  
	const geo_element&             K,
	tiny_vector<space::size_type>& idx,
	const geo::size_type*          mesh_n_geo,
        const geo::size_type*          mesh_n_element,
        geo::size_type                 first,
	element_constant::dof_family_type family)
{
  typedef spacerep::size_type size_type;
  if (family == element_constant::dof_family_max)
    {
    	size_type ndof = b.size(K.variant());
  	for (size_type i = 0; i < ndof; i++) {	
	      idx[i] = first + numb.idof (mesh_n_geo, mesh_n_element, K, i);
	  }
    }
  else 
    {
    	size_type ndof = b.size(K.variant(), family);
  	for (size_type i = 0; i < ndof; i++) {	
	      if (b.dof_family(K, i)==family)
	      	idx[i] = first + numb.idof (mesh_n_geo, mesh_n_element, K, i);
	  }
    }
}
void
spacerep::set_dof(
	const geo_element&      K,
	tiny_vector<size_type>& idx ,
	spacerep::size_type     i_comp) const
{
    size_type ndof = get_basis(i_comp).size(K.variant());
    idx.resize (ndof);
    internal_set_dof(get_geo(), get_basis(i_comp), get_numbering(i_comp),
		        K, idx, 
			get_geo().get_geo_counter(),
  			get_geo().get_element_counter(), 
			start(i_comp));
}
void
spacerep::set_dof(
	const geo_element&      K,
	tiny_vector<size_type>& idx ,
	spacerep::size_type     i_comp,
	element_constant::dof_family_type family) const
{
    size_type ndof = get_basis(i_comp).size(K.variant(), family);
    idx.resize (ndof);
    internal_set_dof(get_geo(), get_basis(i_comp), get_numbering(i_comp),
		        K, idx, 
			get_geo().get_geo_counter(),
  			get_geo().get_element_counter(), 
			start(i_comp),
			family);
}
// global version: (e.g. operators on boundary with global numbering)
// 	get_geo -> get_global_geo
// 	start   -> start_global_dof
void
spacerep::set_global_dof (
	const geo_element&      K, 
	tiny_vector<size_type>& idx,
	spacerep::size_type     i_comp) const
{
    size_type ndof = get_basis(i_comp).size(K.variant());
    idx.resize (ndof);
    internal_set_dof (get_global_geo(), get_basis(i_comp), get_numbering(i_comp),
		        K, idx,
  			get_global_geo().get_geo_counter(),
  			get_global_geo().get_element_counter(),
			start_global_dof(i_comp));
}
void
spacerep::set_dof(const geo_element& K, tiny_vector<size_type>& idx) const
{
    size_type ndof = 0;
    for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
	ndof += get_basis(i_comp).size(K.variant());
    }
    idx.resize (ndof);

    size_type shift = 0;
    for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
	idx.set_origin(shift);
        internal_set_dof(get_geo(), get_basis(i_comp), get_numbering(i_comp),
			K, idx, 
			get_geo().get_geo_counter(),
  			get_geo().get_element_counter(), 
			start(i_comp));
	shift += get_basis(i_comp).size(K.variant());
    }
    idx.set_origin(0);
}
void
spacerep::set_global_dof(const geo_element& K, tiny_vector<size_type>& idx) const
{
    size_type ndof = 0;
    for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
	ndof += get_basis(i_comp).size(K.variant());
    }
    idx.resize (ndof);

    size_type shift = 0;
    for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
	idx.set_origin(shift);
        internal_set_dof (get_global_geo(), get_basis(i_comp), get_numbering(i_comp),
			K, idx,
  			get_global_geo().get_geo_counter(),
  			get_global_geo().get_element_counter(),
			start_global_dof(i_comp));
	shift += get_basis(i_comp).size(K.variant());
    }
    idx.set_origin(0);
}
// ==========================================================================
//  block & freeze
// ==========================================================================

// block all dofs
void
spacerep::block (size_type i_comp) const
{
  if (_is_frozen) {
    error_macro("numbering already done -- cannot block any domain anymore");
  }
  tiny_vector<size_type> idx;
  geo::const_iterator last_side = get_geo().end(); 
  for (geo::const_iterator iter_side = get_geo().begin(); iter_side != last_side; iter_side++) {
    const geo_element& S = *iter_side ;
    set_dof (S, idx, i_comp);
    for (size_type i = 0 ; i < idx.size(); i++) {
      _is_blocked [idx(i)] = true;
    }
  }
}

// block all Lagrange dofs
void
spacerep::block_Lagrange (size_type i_comp) const
{
  if (_is_frozen) {
    error_macro("numbering already done -- cannot block any domain anymore");
  }
  tiny_vector<size_type> idx;
  geo::const_iterator last_side = get_geo().end(); 
  for (geo::const_iterator iter_side = get_geo().begin(); iter_side != last_side; iter_side++) {
    const geo_element& S = *iter_side ;
    set_dof (S, idx, i_comp, element_constant::Lagrange);
    for (size_type i = 0 ; i < idx.size(); i++) {
      _is_blocked [idx(i)] = true;
    }
  }
}

void
spacerep::block () const
{
  for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
     block (i_comp);
  }
}
void
spacerep::block_Lagrange () const
{
  for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
     block_Lagrange (i_comp);
  }
}

// block a domain
void
spacerep::block (const domain& d, size_type i_comp) const
{
  if (_is_frozen) {
    error_macro("numbering already done -- cannot block any domain anymore");
  }
  tiny_vector<size_type> idx;
  domain::const_iterator last_side = d.end(); 
  for (domain::const_iterator iter_side = d.begin(); iter_side != last_side; iter_side++) {
    const geo_element& S = *iter_side ;
    set_dof (S, idx, i_comp);
    for (size_type i = 0 ; i < idx.size(); i++) {
      _is_blocked [idx(i)] = true;
    }
  }
}
void
spacerep::block (const domain& d) const
{
  for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
     block (d, i_comp);
  }
}

void
spacerep::block_dof (size_type dof_idx) const
{
  if (_is_frozen) {
    error_macro("numbering already done -- cannot block any domain anymore");
  }
  _is_blocked [dof_idx] = true;
}
void
spacerep::unblock_dof (size_type dof_idx) const
{
  if (_is_frozen) {
    error_macro("numbering already done -- cannot block any domain anymore");
  }
  _is_blocked [dof_idx] = false;
}
// --------------------------------------------------------------------------

void
spacerep::set_periodic (const domain& d1, const domain& d2, size_type i_comp) const
{
  if (_is_frozen) {
    error_macro("numbering already done -- cannot set_periodic any pair of domains anymore");
  }
  tiny_vector<size_type> dof1;
  tiny_vector<size_type> dof2;
  list<pair<size_type, size_type> > double_periodic;
  domain::const_iterator last_side1 = d1.end(); 
  domain::const_iterator iter_side1 = d1.begin();
  domain::const_iterator iter_side2 = d2.begin();
  for (; iter_side1 != last_side1; iter_side1++, iter_side2++) {
    const geo_element& S1 = *iter_side1 ;
    const geo_element& S2 = *iter_side2 ;
    set_dof (S1, dof1, i_comp);
    set_dof (S2, dof2, i_comp);
    for (size_type i1 = 0 ; i1 < dof1.size(); i1++) {
      size_t i2;
      if (i1==0 || i1==1) { i2 = 1-i1; }
      else { i2 = dof1.size()+1-i1; }
      if (!_is_periodic[dof1(i1)])
        {
         _is_periodic  [dof1(i1)] = true;
	 if (!_is_periodic[dof2(i2)])
	         _period_assoc [dof1(i1)] = dof2(i2);
	 else // target dof is periodic (from a previous call), reach directly for its own target
	         _period_assoc [dof1(i1)] = _period_assoc [dof2(i2)];
	 	
	}
      else // dof is already mapped to another one: store it for re-examination
	double_periodic.insert(double_periodic.begin(), pair<size_type,size_type>(dof1(i1),dof2(i2)));
    }
   }
  if (double_periodic.size()!=0)
   {
      typedef list<pair<size_type, size_type> > mylist;
      mylist::const_iterator i = double_periodic.begin();
      mylist::const_iterator end = double_periodic.end();
      for ( ; i!=end; i++)
        {
	  if (!_is_periodic[(*i).second])
	         _period_assoc [(*i).first] = (*i).second;
	 else // target dof is periodic (from a previous call), reach directly for its own target
	         _period_assoc [(*i).first] = _period_assoc [(*i).second];
	}
   }
}
void
spacerep::set_periodic (const domain& d1, const domain& d2) const
{
  for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
     set_periodic (d1, d2, i_comp);
  }
}
//! On a uniform boundary, all dofs have the same value.
//! Useful e.g. for solid boundaries with translation only.
void
spacerep::set_uniform (const domain& d, size_type i_comp) const
{
  if (_is_frozen) {
    error_macro("numbering already done -- cannot set_periodic any pair of domains anymore");
  }
  tiny_vector<size_type> dof;
  domain::const_iterator last_side = d.end(); 
  domain::const_iterator iter_side = d.begin();
  bool single_dof_set = false;
  size_t single_dof = 0;
  for (; iter_side != last_side; iter_side++) {
    const geo_element& S = *iter_side ;
    set_dof (S, dof, i_comp);
    for (size_type i = 0 ; i < dof.size(); i++) {
      // the first dof encountered will not be blocked, all others will
      // be bound to it.
      if (!single_dof_set)
       {
       	single_dof = dof(i);
	single_dof_set = true;
       }
      if (dof(i)!=single_dof) 
       {
        _is_periodic  [dof(i)] = true;
        _period_assoc [dof(i)] = single_dof;
       }
    }
  }
}
// ==========================================================================
//  lock components
// ==========================================================================
void
spacerep::lock_components (const domain& d, point locked_dir) const
{
  _has_locked=true;
  check_macro(dimension()==2, "Components-locking only implemented in 2D");
  if (n_component()!=2) warning_macro("Components-locking only locks the first 2 components of vectors");
  size_type n_comp=2;
  check_macro(get_approx(0)==get_approx(1), "Components-locking only implemented when vector components are based on a the same approximation");
  if (_is_frozen) {
    error_macro("numbering already done -- cannot lock the components on any domain anymore");
  }
  resize(_is_locked, _is_blocked.size(), numeric_limits<size_type>::max());
  resize(_locked_with, _is_blocked.size(), numeric_limits<size_type>::max());
  size_type i_locked_domain=0;
  tiny_vector<size_type> idx0;
  tiny_vector<size_type> idx1;
  domain::const_iterator last_side = d.end(); 
//std::cerr << "Block: ";
  for (domain::const_iterator iter_side = d.begin(); iter_side != last_side; iter_side++) {
    const geo_element& S = *iter_side ;
    set_dof (S, idx0, 0);
    set_dof (S, idx1, 1);
    for (size_type i = 0 ; i < idx0.size(); i++) {
        _is_locked [idx0(i)] = i_locked_domain;
        _is_locked [idx1(i)] = i_locked_domain++;
	_is_blocked[idx1(i)] = true; // blocks only one of the two components
	_locked_with [idx0(i)] = idx1(i);
	_locked_with [idx1(i)] = idx0(i);
	_locked_dir.insert(_locked_dir.end(), locked_dir);
//std::cerr << idx0(i) << " & " << idx1(i) << " | ";
    }
  }
}
// --------------------------------------------------------------------------

//
// set numbering 
// modified: _is_frozen, _num[], _start_blocked[]
void
spacerep::freeze () const
{
  if (_is_frozen) return;
  _is_frozen = true;
  size_type i_blocked = 0;
  size_type i_unknown = 0;
  _start_blocked[0] = i_blocked;
  _start_unknown[0] = i_unknown;
  for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
      for (size_type dof = start(i_comp); dof < start(i_comp+1); dof++) {
         if (!is_periodic(dof)) {
             if (is_blocked(dof)) {
                _num [dof] = i_blocked++;
             } else {
                _num [dof] = i_unknown++;
	     }
         }
      }
      _start_blocked [i_comp+1] = i_blocked;
      _start_unknown [i_comp+1] = i_unknown;
  }
  for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
      for (size_type dof = start(i_comp); dof < start(i_comp+1); dof++) {
         if (is_periodic(dof)) {
            _num [dof] = _num [ _period_assoc[dof] ];
	 }
      }
  }
}
//! set _boundary_dof[] and _global_dof[]
void
spacerep::_build_global_dof_mapping ()
{
//ICI
  _start_global_dof [0] = 0;
  for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {
      _start_global_dof [i_comp+1] = _start_global_dof [i_comp] + _n_global_dof(i_comp);
  }
  if (!is_on_boundary_domain()) { // on a full geometry:
      for (size_type i_dof = 0; i_dof < size(); i_dof++) {
        _boundary_dof [i_dof] = _global_dof [i_dof] = i_dof;
      }
      return;
  }
  // else: on a boundary domain:
  // correspondance between mesh_dof on full mesh geometry
  // and domain_dof on domain
  tiny_vector<size_type> dmn_dof;
  tiny_vector<size_type> msh_dof;

  size_type K_idx = 0;
  geo::const_iterator iter_K = get_geo().begin();
  fill (_boundary_dof.begin(), _boundary_dof.end(), numeric_limits<size_type>::max());

  for (domain::const_iterator iter_S = _boundary_domain.begin();
        iter_S != _boundary_domain.end(); iter_S++, iter_K++, K_idx++) {
    const geo_element& S = *iter_S;
    const geo_element& K = *iter_K;
    for (size_type i_comp = 0; i_comp < n_component(); i_comp++) {

      set_dof (K, dmn_dof, i_comp);
      size_type first_dof = start (i_comp);
      internal_set_dof(get_global_geo(), get_basis(), get_numbering(),
	S, msh_dof,
  	get_global_geo().get_geo_counter(),
  	get_global_geo().get_element_counter(),
	first_dof);

      for (size_type iloc = 0; iloc < dmn_dof.size(); iloc++) {
        size_type i_dmn_dof = dmn_dof(iloc);
        size_type i_msh_dof = msh_dof(iloc);
        if (_boundary_dof[i_msh_dof] != numeric_limits<size_type>::max()) {
          // already set
          continue;
        }
        check_macro (i_msh_dof < global_size(), "i_msh_dof "<< i_msh_dof << " out of range 0:" << global_size()-1);
        check_macro (i_dmn_dof < size(), "i_dmn_dof "<< i_dmn_dof << " out of range 0:" << size()-1);
        _boundary_dof [i_msh_dof] = i_dmn_dof;
        _global_dof   [i_dmn_dof] = i_msh_dof;
      } // loop i
    } // loop i_comp
  } // loop S
}
// --------------------------------------------------------------------------
// get the coordinates of the dof `iloc' in local index
// --------------------------------------------------------------------------
point
spacerep::x_dof(const geo_element& K, size_type iloc) const
{
    size_type i_comp = 0;
    size_type i_hat_node = 0;
    if (n_component() == 1) {
        i_hat_node = iloc;
    } else {
        // TODO: precompute in class all sizes and start indexes
	// for all reference elements : avoid slow basis().size() calls
        size_type offset = 0;
        size_type sz;
        do {
            sz = get_basis(i_comp).size(K);
            if (iloc < offset + sz)
	        break;
	    offset += sz;
	    i_comp++;
        } while (i_comp < n_component());
        check_macro (i_comp < n_component(), "local index "<< iloc << " out of range");
        // then: offset <= iloc < offset + sz
	i_hat_node = iloc-offset;
    }
    // only P1 transformation supported : isoparametric not yet !
    basis_on_nodal_basis::const_iterator first = _tr_p1_value[i_comp].begin(K,i_hat_node);
    basis_on_nodal_basis::const_iterator last  = _tr_p1_value[i_comp].end(K,i_hat_node);
    point x;
    geo::const_iterator_node p = _g.begin_node();
    size_type d = _g.dimension();
    //geo::const_iterator_node p = get_global_geo().begin_node();
    //size_type d = get_global_geo().dimension();
    for (size_type k = 0; first != last; first++, k++) {
        const point& ak = p[K[k]];
	Float coeff = (*first);
	for (size_type i = 0; i < d; i++)
	  x[i] += coeff * ak[i];
    }
    return x;
}

// ==========================================================================
// Space on a 1D domain: returns the orientation of positive curvilinear 
// coordinates
// ==========================================================================
/*
int
spacerep::element_orientation(const geo_element& K) const
{
  if (!_element_orientation_initialized) initialize_element_orientation(); 
  return (element_positively_oriented[K.index()])?1:-1;
}
void
spacerep::initialize_element_orientation() const
{
  check_macro(_is_on_boundary_domain && (get_geo().map_dimension()==1),
  	"Only 1D boundary domains can be oriented");
  interface bc;
  get_global_geo().sort_interface(_boundary_domain, bc, element_positively_oriented);
  iter_K = bc.begin();
  last_K = bc.end();
  const geo_element& e0 = _boundary_domain.at(*iter_K++);
  const geo_element& e1 = _boundary_domain.at(*iter_K);
  previous_pos_or = (e0[1]==e1[0] || e0[1]==e1[1]);
  element_positively_oriented[(*iter_K).index()] = previous_pos_or;
  for (; iter_K!=last_K; iter_K++)
   {
  	e1 = _boundary_domain.at(*iter_K);
	if (previous_pos_or)
	{
	  previous_pos_or = (e1[0] == e0[1]);
	  check_macro( (e1[0]==e0[1])||(e1[1]==e0[1]), "error");
	}
	else
	{
	  previous_pos_or = (e1[0] == e0[0]);
	  check_macro( (e1[0]==e0[1])||(e1[1]==e0[1]), "error");
	}
	element_positively_oriented[(*iter_K).index()] = previous_pos_or;
	e0 = e1; 
   }
  _dof_orientation_initialized=true;
}
*/
// ==========================================================================
// Hermite/Argyris: returns 1 if the global dof contains the derivative along 
// outward normal of K, -1 else  
// ==========================================================================
void   
spacerep::dof_orientation(const geo_element& K, tiny_vector<int>& orientation) const
{
  if (!_dof_orientation_initialized) initialize_dof_orientation(); 
  tiny_vector<size_type> idx;
  set_global_dof (K, idx);
  orientation.resize(idx.size());
  for (size_t i=0; i<idx.size(); i++)
   {
   	size_t idof=domain_dof(idx(i));
	check_macro(_dof_oriented_as_element[idof]!=numeric_limits<size_t>::max(),
		"Error!");
   	orientation[i] = (_dof_oriented_as_element[idof]==K.index())?1:-1;
   }
}

// ==========================================================================
// Hermite/Argyris: the element of lowest index number sets the orientation
// ==========================================================================
void
spacerep::initialize_dof_orientation() const
{
  geo::const_iterator iter_K;
  geo::const_iterator last_K;
  if (_is_on_boundary_domain)
   {
     iter_K = _boundary_domain.begin();
     last_K = _boundary_domain.end();
   }
  else
   {
     iter_K = get_geo().begin();
     last_K = get_geo().end();
   }
  _dof_oriented_as_element.resize(size(), numeric_limits<size_t>::max());
  for (; iter_K!=last_K; iter_K++)
   {
	const geo_element& K=*iter_K;
  	tiny_vector<size_type> idx;
   	set_global_dof(K, idx);
	for (size_t i=0; i<idx.size(); i++)
	 {
	 	size_t i_dof = domain_dof(idx[i]); 
		if (K.index()<_dof_oriented_as_element[i_dof])
			_dof_oriented_as_element[i_dof]=K.index();
	 }
   }
  _dof_orientation_initialized=true;
}

// ==========================================================================
//  input/output
// ==========================================================================

ostream&
operator << (ostream& s, const spacerep& x)
{
  typedef spacerep::size_type size_type;

  x.freeze();
  
  s << "#!space\n\n";
  s << "space\n";
  // version number
  if (x.n_component() == 1) 
      s << "1 ";
  else 
      s << "2 ";
  s << x.size() << endl;
  s << x.get_geo().name() << endl;
  if (x.n_component() == 1) {
      s << x.get_approx(0) << endl;
  } else {
      s << x.n_component() << endl;
      for (size_type i=0;i<x.n_component();i++) {
        s << x.size_component(i) << endl;
        s << x.get_approx(i) << endl;
      }
  }
  s << endl;
  vector<size_type>::const_iterator iter_n = x._num.begin(); 
  vector<size_type>::const_iterator last_n = x._num.end();
  vector<bool>::const_iterator      iter_b = x._is_blocked.begin(); 
  vector<bool>::const_iterator      last_b = x._is_blocked.end();
  while (iter_b != last_b) {
    s << *iter_n;
    if (*iter_b) 
      s << "\tB\n";
    else
      s << endl;
    ++iter_n;
    ++iter_b;
  }
  s << endl;
  return s;
}

// --------------------------------------------------------------------------

istream&
operator >> (istream& is, spacerep& x)
{
  typedef spacerep::size_type size_type;

  bool verbose = iorheo::getverbose(clog);

  if (!scatch(is,"\nspace"))
      error_macro("input stream does not contains a space");

  size_type version;
  size_type ndof;
  is >> version >> ndof;

  string geo_name;
  is >> geo_name;
  x._g = geo(geo_name);
 
  x._is_frozen = true;

  size_type n_component;
  if (version == 1) n_component = 1; 
  else              is >> n_component;

  x._name.resize(n_component);
  x._b.resize(n_component);
  x._numbering.resize(n_component);
  x._start.resize(n_component+1);
  x._start_blocked.resize(n_component+1);

  string app_name;
  if (version == 1) {

     // one-component space format
     is >> app_name;
     x._name[0] = app_name;
     string basis_name = fem_helper::basis_name     (app_name);
     string num_name   = fem_helper::numbering_name (app_name);
     x._start [0] = 0;
     x._start [1] = ndof;
     x._b [0]         = basis(basis_name);
     x._numbering [0] = numbering(num_name);
  } else {

     // multi-component space format
     x._start [0] = 0;
     for (size_type i=0 ; i<n_component ; i++) {
        size_type size_comp;
        is >> size_comp;
        x._start [i+1] = x._start [i] + size_comp;
        is >> app_name;
        x._name[i] = app_name;
        string basis_name = fem_helper::basis_name     (app_name);
        string num_name   = fem_helper::numbering_name (app_name);
        x._b [i] = basis(basis_name);
        x._numbering [i] = numbering(num_name);
     }
  }
  x._tr_p1 = basis("P1");
  resize(x._num,        ndof, numeric_limits<size_type>::max());
  resize(x._is_blocked, ndof, false);

  x._start_blocked [0] = 0;
  for (size_type i_comp = 0; i_comp < n_component; i_comp++) {
      x._start_blocked [i_comp+1] = x._start_blocked [i_comp];
      for (size_type dof = x._start[i_comp]; dof < x._start[i_comp+1]; dof++) {
	  is >> x._num[dof];
          // skit white and tabs, but not end-of-line !
          char c;
          for (c = is.get(); c == ' ' || c == '\t'; c = is.get());
          if (c == 'B') {
	      x._is_blocked[dof] = true;
              x._start_blocked [i_comp+1]++;
	  } else {
              is.putback(c);
	  }
      }
  }
  return is;
}
ostream&
spacerep::dump (ostream& os) const
{
    os << "begin space dump" << endl;
    if (_is_frozen) {
	os << "is_frozen: true" << endl; 
    } else {
	os << "is_frozen: false" << endl; 
    }
    os << "start.size " << _start.size() << endl;
    for (size_type i = 0; i < _start.size(); i++) os << "start["<<i<<"]="<<_start[i] << endl;

    os << "start_global_dof.size " << _start_global_dof.size() << endl;
    for (size_type i = 0; i < _start_global_dof.size(); i++) os << "start_global_dof["<<i<<"]="<<_start_global_dof[i] << endl;

    os << "start_blocked.size " << _start_blocked.size() << endl;
    for (size_type i = 0; i < _start_blocked.size(); i++) os << "start_blocked["<<i<<"]="<<_start_blocked[i] << endl;

    os << "is_blocked.size " << _is_blocked.size() << endl;
    for (size_type i = 0; i < _is_blocked.size(); i++) os << "is_blocked["<<i<<"]="<<_is_blocked[i] << endl;

    os << "num.size " << _num.size() << endl;
    for (size_type i = 0; i < _num.size(); i++) os << "num["<<i<<"]="<<_num[i] << endl;

    os << "is_periodic.size " << _is_periodic.size() << endl;
    for (size_type i = 0; i < _is_periodic.size(); i++) 
    	if (_is_periodic[i]) os << "period_assoc["<<i<<"]="<<_period_assoc[i] << endl;

    os << "global_dof.size " << _global_dof.size() << endl;
    for (size_type i = 0; i < _global_dof.size(); i++) os << "global_dof["<<i<<"]="<<_global_dof[i] << endl;

    os << "boundary_dof.size " << _boundary_dof.size() << endl;
    for (size_type i = 0; i < _boundary_dof.size(); i++) os << "boundary_dof["<<i<<"]="<<_boundary_dof[i] << endl;

    os << "b.size " << _b.size() << endl;
    for (size_type i = 0; i < _b.size(); i++) {
	os << "b["<<i<<"]:"<< endl;
        _b[i].dump(os);
    }

    os << "numbering.size " << _numbering.size() << endl;
    for (size_type i = 0; i < _numbering.size(); i++) {
	os << "numbering["<<i<<"]:"<< endl;
        _numbering[i].dump(os);
    }

    os << "space geo dump" << endl;
    _g.dump(os);
    os << "end space geo:" << endl;

    if (_is_on_boundary_domain) {
	os << "is_on_boundary_domain: true" << endl; 

	os << "space global geo dump" << endl; 
        _global_geo.dump(os);
	os << "end space global geo dump" << endl; 

	os << "space domain dump" << endl; 
        _boundary_domain.dump(os);
	os << "end space domain dump" << endl; 

    } else {
	os << "is_on_boundary_domain: false" << endl; 
    }
    os << "end space dump" << endl;
    return os;
}
void
spacerep::check () const
{
    warning_macro ("checking space...");
    cerr << "size=" << size() << endl;

    cerr << "start.size " << _start.size() << endl;
    for (size_type i = 0; i < _start.size(); i++) cerr << "start["<<i<<"]="<<_start[i] << endl;

    cerr << "start_blocked.size " << _start_blocked.size() << endl;
    for (size_type i = 0; i < _start_blocked.size(); i++) cerr << "start_blocked["<<i<<"]="<<_start_blocked[i] << endl;

    cerr << "start_unknown.size " << _start_blocked.size() << endl;
    for (size_type i = 0; i < _start_blocked.size(); i++) cerr << "start_unknown["<<i<<"]="<<_start[i]-_start_blocked[i] << endl;

    cerr << "start_global_dof.size " << _start_global_dof.size() << endl;
    for (size_type i = 0; i < _start_global_dof.size(); i++) cerr << "start_global_dof["<<i<<"]="<<_start_global_dof[i] << endl;


    _g.check();
    if (_is_on_boundary_domain) {
        _boundary_domain.check();
    }
    size_type n = n_component();
    size_type n_dof = size();
    size_type n_blk = n_blocked();
    size_type n_glob_dof = global_size();

    for (size_type i = 0; i < _b.size(); i++) {
	cerr << "basis["<<i<<"]=\"" <<_b[i].name() << "\""<< endl;
    }
    if (_start_blocked.size() != n+1) {
	warning_macro ("_start_blocked.size="<<_start_blocked.size()<<"; expect "<<n+1);
    }
    if (_start_global_dof.size() != n+1) {
	warning_macro ("_start_global_dof.size="<<_start_global_dof.size()<<"; expect "<<n+1);
    }
    if (_is_blocked.size() != n_dof) {
	warning_macro ("_is_blocked.size="<<_is_blocked.size()<<"; expect "<<n_dof);
    }
    if (_num.size() != n_dof) {
	warning_macro ("_num.size="<<_num.size()<<"; expect "<<n_dof);
    }
    if (_global_dof.size() != n_dof) {
	warning_macro ("_global_dof.size="<<_global_dof.size()<<"; expect "<<n_dof);
    }
    if (_boundary_dof.size() != n_glob_dof) {
	warning_macro ("_boundary_dof.size="<<_boundary_dof.size()<<"; expect "<<n_glob_dof);
    }
    for (size_type i = 0; i < n+1; i++) {
        if (_start[i] > n_dof) {
	    warning_macro ("_start["<<i<<"]="<<_start[i]<<" is out of range 0:"<<n_dof-1);
	}
        if (_is_frozen && _start_blocked[i] > n_blk) {
	    warning_macro ("_start_blocked["<<i<<"]="<<_start_blocked[i]<<" is out of range 0:"<<n_blk-1);
	}
        if (_start_global_dof[i] > n_glob_dof) {
	    warning_macro ("_start_global_dof["<<i<<"]="<<_start_global_dof[i]<<" is out of range 0:"<<n_glob_dof-1);
	}
    }
    for (size_type k = 0; k < n; k++) {
      size_type first_b = _start_blocked[k];
      size_type last_b  = _start_blocked[k+1];
      size_type first_u = _start[k]   - _start_blocked[k];
      size_type last_u  = _start[k+1] - _start_blocked[k+1];
      size_type first_g = _start_global_dof[k];
      size_type last_g  = _start_global_dof[k+1];
      for (size_type i = _start[k]; i < _start[k+1]; i++) {
	if (_is_blocked [i] && (_num[i] < first_b || _num[i] >= last_b)) {
	    warning_macro ("_num["<<i<<"]="<<_num[i]<<" is out of range "<<first_b<<":"<<last_b-1);
	}
	if (!_is_blocked [i] && (_num[i] < first_u || _num[i] >= last_u)) {
	    warning_macro ("_num["<<i<<"]="<<_num[i]<<" is out of range "<<first_u<<":"<<last_u-1);
	}
	if (_global_dof[i]  < first_g || _global_dof[i] >= last_g) {
	    warning_macro ("_global_dof["<<i<<"]="<<_global_dof[i]<<" is out of range "
		<<first_g<<":"<<last_g-1);
	} else {
	    size_type ig = _global_dof[i];
	    if (_boundary_dof[ig] <  _start[k] || 
                _boundary_dof[ig] >= _start[k+1]) {
	        warning_macro ("_boundary_dof["<<ig<<"]="<<_boundary_dof[ig]<<" is out of range "
		    <<_start[k]<<":"<<_start[k+1]-1);
	    }
	}
      }
    }
    warning_macro ("checking space done.");
}
bool 
spacerep::operator == (const spacerep& y) const
{
    if (get_geo() != y.get_geo()) return false;
    if (get_approx() != y.get_approx()) return false;
    return true;
}
// =================================================================
// hatter and dehatter:
//  F : hat(K) --> K is a nonlinear transformation
// that depends upon the degree of the polynomial basis
// only linear transformation on e, t, T is supported yet.
// =================================================================
meshpoint 
spacerep::hatter (const point& x, size_type K_idx) const {
    const geo_element& K = _g.element (K_idx);
    switch(K.variant()) {
        case geo_element::e: {
            Float a = _g.vertex(K[0]) [0];
            Float b = _g.vertex(K[1]) [0];
            return meshpoint(K_idx, point((x[0] - a)/(b-a)));
	}	  
        case geo_element::t: {
            const point& a = _g.vertex(K[0]);
            const point& b = _g.vertex(K[1]);
            const point& c = _g.vertex(K[2]);
            point xx=x;
            Float t9 = 1/(-b[0]*c[1]+b[0]*a[1]+a[0]*c[1]+c[0]*b[1]-c[0]*a[1]-a[0]*b[1]);
            Float t11 = -a[0]+xx[0];
            Float t15 = -a[1]+xx[1];

            return meshpoint(K_idx, 
		point((-c[1]+a[1])*t9*t11-(-c[0]+a[0])*t9*t15,
                      (b[1]-a[1])*t9*t11-(b[0]-a[0])*t9*t15,
                       0));
        }
        case geo_element::T: {
            const point& a = _g.vertex(K[0]);
            const point& b = _g.vertex(K[1]);
            const point& c = _g.vertex(K[2]);
            const point& d = _g.vertex(K[3]);	
	    tensor A;
	    point ax;
	    for (size_t i = 0; i < 3; i++) {
	      ax[i]  = x[i]-a[i];
	      A(i,0) = b[i]-a[i];
	      A(i,1) = c[i]-a[i];
	      A(i,2) = d[i]-a[i];
	    }
	    tensor inv_A;
	    bool is_singular = ! invert_3x3 (A, inv_A);
            check_macro(!is_singular, "hatter: singular transformation in element: " << K.index());
	    point hat_x = inv_A*ax;
            return meshpoint(K_idx, hat_x); 
	}	  
        default : error_macro("hatter: element not yet implemented: " << K.name());
     }
}
point
spacerep::dehatter (const meshpoint& hat_xe, bool is_a_vector) const
{
    const point&       hat_x = hat_xe.hat();
    size_t             K_idx = hat_xe.element();
    const geo_element& K     = _g.element (K_idx);
    switch(K.variant()) {
        case geo_element::e: {
            Float a = _g.vertex(K[0]) [0];
            Float b = _g.vertex(K[1]) [0];
            return point((!is_a_vector?a:0)
                    + (b-a)*hat_x[0]);
	}	  
        case geo_element::t: {
            const point& a = _g.vertex(K[0]);
            const point& b = _g.vertex(K[1]);
            const point& c = _g.vertex(K[2]);
            return ((!is_a_vector)?a:point(0,0))
                    + (b-a)*hat_x[0]
	            + (c-a)*hat_x[1];
        }
        case geo_element::T: {
            const point& a = _g.vertex(K[0]);
            const point& b = _g.vertex(K[1]);
            const point& c = _g.vertex(K[2]);
            const point& d = _g.vertex(K[3]);	
            return ((!is_a_vector)?a:point(0,0))
                    + (b-a)*hat_xe.hat()[0] 
                    + (c-a)*hat_xe.hat()[1]
                    + (d-a)*hat_xe.hat()[2];
        }
        default : error_macro("dehatter: element not yet implemented: "	<< K.name());
     }
}
}// namespace rheolef
