#ifndef _RHEO_BASIS_ON_POINTSET_V2_H
#define _RHEO_BASIS_ON_POINTSET_V2_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:basis_on_pointset
NAME: @code{basis_on_pointset} - pre-evaluated polynomial basis
@cindex  polynomial basis
@clindex basis
@cindex  reference element
@clindex reference_element
SYNOPSIS:
  @noindent
  The @code{basis_on_pointset} class is able to evaluates a polynomial
  basis and its derivatives on a set of point of the reference element
  (@pxref{reference_element iclass}).
  The basis is described by the @code{basis} class (@pxref{basis class}).
  The set of points could be
  either quadrature nodes on the reference element (@pxref{quadrature iclass})
  or Lagrange nodes associated to another basis.
  For application of an integration of jump and average values of a discontuous 
  approximation on a side, the set of nodes could be defined on a specific side only.
  In all these cases, the evaluation of polynomials could be performed
  one time for all on the reference element and its result stored and reused
  for all elements of the mesh: the speedup is important, espcially for
  high order polynomials. 

AUTHOR: Pierre.Saramito@imag.fr
DATE:   4 january 2018
End:
*/

#include "rheolef/basis.h"
#include "rheolef/quadrature.h"
namespace rheolef {

// -----------------------------------------------------------------------
// basis evaluated on lattice of quadrature formulae
// -----------------------------------------------------------------------
template<class T>
class basis_on_pointset_rep {
public:

    typedef typename std::vector<T>::size_type                     size_type;
    typedef typename arma::Col<T>::const_iterator                  const_iterator;
    typedef typename std::vector<point_basic<T> >::const_iterator  const_iterator_grad;

// allocators:

    basis_on_pointset_rep ();
    basis_on_pointset_rep (const basis_on_pointset_rep<T>&);
    basis_on_pointset_rep<T>& operator= (const basis_on_pointset_rep<T>&);
#ifdef TO_CLEAN
    basis_on_pointset_rep (const quadrature<T>&  quad, const basis_basic<T>& b);
    basis_on_pointset_rep (const basis_basic<T>& nb,   const basis_basic<T>& b);
#endif // TO_CLEAN

// modifiers:

    void set (const quadrature<T>&  quad, const basis_basic<T>& b);
    void set (const basis_basic<T>& nb,   const basis_basic<T>& b);

// accessors:

    const basis_basic<T>& get_basis() const { return _b; }
    size_type pointset_size (reference_element hat_K) const;
    void restrict_on_side (reference_element tilde_L, const side_information_type& sid);

    void evaluate      (reference_element hat_K, size_type q) const;
    void evaluate_grad (reference_element hat_K, size_type q) const;

    void evaluate      (reference_element hat_K, const point_basic<T>& hat_xq) const;
    void evaluate_grad (reference_element hat_K, const point_basic<T>& hat_xq) const;

    const_iterator      begin() const;
    const_iterator      end() const;
    const_iterator_grad begin_grad() const;
    const_iterator_grad end_grad() const;

    // new interface: random accessors
    const T& value (size_type loc_idof) const;
    const point_basic<T>& grad_value (size_type loc_idof) const;

// data:
    typedef enum {
      quad_mode     = 0,
      nodal_mode    = 1,
      max_mode      = 2
    } mode_type;
protected:
    basis_basic<T>                                      _b;
    mutable mode_type                                   _mode; 
    mutable bool                                        _restricted_on_side;
    mutable bool                                        _specific_value;
    quadrature<T>                                       _quad; // when mode: on quadrature pointset
    basis_basic<T>                                      _nb;   // when mode: on nodal basis pointset
    mutable std::array<size_type,
	reference_element::max_variant>                 _basis_size;
    mutable std::array<arma::Col<T>,
	reference_element::max_variant>                 _val;
    mutable std::array<std::vector<point_basic<T> >,
	reference_element::max_variant>			_grad_val;
    mutable std::array<arma::Col<T>,
	reference_element::max_variant>                 _spec_val;
    mutable std::array<std::vector<point_basic<T> >,
	reference_element::max_variant>			_spec_grad_val;
    // sid_val [tilde_L][loc_isid][orient][shift][basis_sz*idof+q]
    mutable std::array<
              std::array<
                std::array<
                  std::array<
                    arma::Col<T>,
	            8>,
                  2>,
                8>,
              reference_element::max_variant>           _sid_val;
    mutable std::array<
              std::array<
                std::array<
                  std::array<
                    std::vector<point_basic<T> >,
	            8>,
                  2>,
                8>,
              reference_element::max_variant>           _sid_grad_val;
    mutable std::array<bool,
	               reference_element::max_variant>  _initialized;
    mutable std::array<bool,
	               reference_element::max_variant>  _grad_initialized;
    mutable std::array<bool,
	               reference_element::max_variant>  _sid_initialized;
    mutable size_type                                   _curr_K_variant;
    mutable size_type                                   _curr_q;
    mutable side_information_type                       _curr_sid;

    void _initialize     (reference_element hat_K) const;
    void _grad_initialize(reference_element hat_K) const;
    void _sid_initialize (reference_element tilde_L, const side_information_type& sid) const;
    void _sid_initialize (reference_element tilde_L) const;
};
// -----------------------------------------------------------------------
// interface with shallow copy semantic
// -----------------------------------------------------------------------
//<verbatim:
template<class T>
class basis_on_pointset: public smart_pointer<basis_on_pointset_rep<T> > {
public:

  typedef basis_on_pointset_rep<T>          rep;
  typedef smart_pointer<rep>                base;
  typedef typename rep::size_type           size_type;
  typedef typename rep::const_iterator      const_iterator;
  typedef typename rep::const_iterator_grad const_iterator_grad;

// allocators:

  basis_on_pointset ();
  basis_on_pointset (const quadrature<T>&  quad, const basis_basic<T>& b);
  basis_on_pointset (const basis_basic<T>& nb,   const basis_basic<T>& b);

// modifiers:

  void set (const quadrature<T>&  quad, const basis_basic<T>& b);
  void set (const basis_basic<T>& nb,   const basis_basic<T>& b);

// accessors:

  const basis_basic<T>& get_basis() const;
  size_type pointset_size (reference_element hat_K) const;

// reference element and pointset:

  void evaluate      (reference_element hat_K, size_type q) const;
  void evaluate_grad (reference_element hat_K, size_type q) const;

  // for a specific node instead of a pointset:
  void evaluate      (reference_element hat_K, const point_basic<T>& hat_xq) const;
  void evaluate_grad (reference_element hat_K, const point_basic<T>& hat_xq) const;

  // side trace, for DG:
  void restrict_on_side (reference_element tilde_L, const side_information_type& sid);

// accessors to values:

  const T& value (size_type loc_idof) const;
  const point_basic<T>& grad_value (size_type loc_idof) const;

  const_iterator      begin() const;
  const_iterator      end() const;
  const_iterator_grad begin_grad() const;
  const_iterator_grad end_grad() const;
};
//>verbatim:

// -----------------------------------------------------------------------
// inlined
// -----------------------------------------------------------------------
template<class T>
inline
basis_on_pointset<T>::basis_on_pointset()
 : base(new_macro(rep))
{
}
template<class T>
inline
basis_on_pointset<T>::basis_on_pointset (const quadrature<T>&  quad, const basis_basic<T>& b)
 : base(new_macro(rep))
{
  set (quad, b);
}
template<class T>
inline
basis_on_pointset<T>::basis_on_pointset (const basis_basic<T>& nb,   const basis_basic<T>& b)
 : base(new_macro(rep))
{
  set (nb, b);
}
template<class T>
inline
void
basis_on_pointset<T>::set (const quadrature<T>&  quad, const basis_basic<T>& b)
{
  base::data().set (quad, b);
}
template<class T>
inline
void
basis_on_pointset<T>::set (const basis_basic<T>& nb,   const basis_basic<T>& b)
{
  base::data().set (nb, b);
}
template<class T>
inline
const basis_basic<T>&
basis_on_pointset<T>::
get_basis() const
{
  return base::data().get_basis();
}
template<class T>
inline
typename basis_on_pointset<T>::size_type
basis_on_pointset<T>::pointset_size (reference_element hat_K) const
{
  return base::data().pointset_size(hat_K);
}
template<class T>
inline
void
basis_on_pointset<T>::evaluate (reference_element hat_K, size_type q) const
{
  base::data().evaluate (hat_K, q);
}
template<class T>
inline
void
basis_on_pointset<T>::evaluate_grad (reference_element hat_K, size_type q) const
{
  base::data().evaluate_grad (hat_K, q);
}
template<class T>
inline
void
basis_on_pointset<T>::evaluate (reference_element hat_K, const point_basic<T>& hat_xq) const
{
  base::data().evaluate (hat_K, hat_xq);
}
template<class T>
inline
void
basis_on_pointset<T>::evaluate_grad (reference_element hat_K, const point_basic<T>& hat_xq) const
{
  base::data().evaluate_grad (hat_K, hat_xq);
}
template<class T>
inline
void
basis_on_pointset<T>::restrict_on_side (reference_element tilde_L, const side_information_type& sid)
{
  base::data().restrict_on_side (tilde_L, sid);
}
template<class T>
inline
const T&
basis_on_pointset<T>::value (size_type loc_idof) const
{
  return base::data().value (loc_idof);
}
template<class T>
inline
const point_basic<T>&
basis_on_pointset<T>::grad_value (size_type loc_idof) const
{
  return base::data().grad_value (loc_idof);
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator
basis_on_pointset<T>::begin() const
{
  return base::data().begin();
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator
basis_on_pointset<T>::end() const
{
  return base::data().end();
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator_grad
basis_on_pointset<T>::begin_grad() const
{
  return base::data().begin_grad();
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator_grad
basis_on_pointset<T>::end_grad() const
{
  return base::data().end_grad();
}

}// namespace rheolef
#endif // _RHEO_BASIS_ON_POINTSET_V2_H
