/*
 * Copyright (C) 2007 Felipe Weckx <felipe.weckx@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */



#include "gbemol-playlist.h"
#include "gbemol-utils.h"

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

static GObjectClass *parent_class = NULL;

/* Playlists View Columns */
enum
{
	COLUMN_PLAYLISTS_TITLE,
	COLUMN_PLAYLISTS_ICON,
	COLUMN_PLAYLISTS_CURRENT,   /* is it the current playlist ? */
	COLUMN_PLAYLISTS_BOLD,
	COLUMN_PLAYLISTS_BOLD_SET,
	N_COLUMNS_PLAYLISTS
};

/* Songs View Columns */
enum
{
	COLUMN_SONGS_MARKUP,
	COLUMN_SONGS_POS,
	COLUMN_SONGS_ARTIST,
	COLUMN_SONGS_TITLE,
	COLUMN_SONGS_ALBUM,
	COLUMN_SONGS_TRACK,
	COLUMN_SONGS_DATE,
	COLUMN_SONGS_GENRE,
	COLUMN_SONGS_COMPOSER,
	COLUMN_SONGS_DISC,
	COLUMN_SONGS_COMMENT,
	COLUMN_SONGS_TIME,
	COLUMN_SONGS_ID,
	COLUMN_SONGS_BOLD,
	COLUMN_SONGS_BOLD_SET,
	COLUMN_SONGS_BG,
	COLUMN_SONGS_BG_SET,
	COLUMN_SONGS_FG,
	COLUMN_SONGS_FG_SET,
	N_COLUMNS_SONGS
};


struct _GbemolPlaylistPrivate {
	GbemolMpd* mpd;

	GtkUIManager *manager;
	GtkTreeView *tv_playlists;
	GtkTreeView *tv_songs;
	GtkListStore *lst_playlists;
	GtkListStore *lst_songs;
	GtkTreeSelection *sel_playlists;
	GtkTreeSelection *sel_songs;
	GtkTreeModel *fil_songs;
	GtkListStore *lst_column_search;

	GtkWidget *ent_search;
	GtkWidget *cmb_search;
	GtkWidget *btn_search;
	GtkWidget *btn_search_clear;
	GtkWidget *btn_add;
	GtkWidget *btn_replace;
	GtkWidget *btn_new_playlist;
	GtkWidget *btn_save_playlist;
	GtkWidget *btn_add_url;

	GtkWidget *mnu_songs;
	GtkWidget *mnu_item_columns;
	GtkWidget *mnu_item_mode[N_PLAYLIST_MODES];

	GtkWidget *mnu_playlists;

	GtkTooltips *tips;

	GbemolPlaylistMode mode;

	gboolean dispose_has_run;
};

/* Constructors/Destructors */
static void gbemol_playlist_class_init (GObjectClass *g_class);
static void gbemol_playlist_init (GbemolPlaylist *obj);
static void gbemol_playlist_finalize (GObject *object);
static void gbemol_playlist_dispose (GObject *object);

/* Private functions */
static gchar* gbemol_playlist_save_dialog (GbemolPlaylist *pls);
void gbemol_playlist_clear_all (GbemolPlaylist* pls);
static GList* gbemol_playlist_get_selected (GtkTreeView* view);
static void gbemol_playlist_free_selected (GList* l);
void gbemol_playlist_append_song (GbemolPlaylist* pls, mpd_Song* song);
void gbemol_playlist_populate_songs (GbemolPlaylist *pls, GList* songs);
void gbemol_playlist_populate_playlists (GbemolPlaylist* pls);
void gbemol_playlist_update_songs (GbemolPlaylist* pls);
void gbemol_playlist_remove_selected_songs (GbemolPlaylist *pls);
void gbemol_playlist_play_selected (GbemolPlaylist *pls);
void gbemol_playlist_play_last_song (GbemolPlaylist *pls);
void gbemol_playlist_remove_selected_playlist (GbemolPlaylist *pls);
void gbemol_playlist_load_selected_playlist (GbemolPlaylist *pls);
gchar* gbemol_playlist_save_dialog (GbemolPlaylist *pls);
void gbemol_playlist_columns_dialog (GbemolPlaylist* pls);

/* Callbacks */
static void on_sel_playlists_changed (GtkTreeSelection *select, GbemolPlaylist* pls);
static gboolean on_tvw_playlists_key_press_event (GtkWidget *w, GdkEventKey *event, GbemolPlaylist* pls);
static gint on_tvw_playlists_button_press_event (GtkWidget *w, GdkEventButton *event, GbemolPlaylist* pls);
static gboolean on_tvw_songs_key_press_event (GtkWidget* w, GdkEventKey* event, GbemolPlaylist* pls);
static gboolean on_tvw_songs_button_press_event (GtkWidget *w, GdkEventButton *event, GbemolPlaylist* pls);
static void on_btn_new_playlist_clicked (GtkButton *button, GbemolPlaylist* pls);
static void on_btn_save_playlist_clicked (GtkButton *button, GbemolPlaylist* pls);
static void on_btn_playlist_add_clicked (GtkWidget* b, GbemolPlaylist* pls);
static void on_btn_playlist_replace_clicked (GtkWidget* b, GbemolPlaylist* pls);
static void on_mnu_playlist_remove_activate (GtkWidget* m, GbemolPlaylist* pls);
static void on_mnu_clear_activate (GtkWidget* w, GbemolPlaylist* pls);
static void on_mnu_shuffle_activate (GtkWidget* w, GbemolPlaylist* pls);
static void on_mnu_column_activate (GtkWidget* w, GbemolPlaylist* pls);
static void on_column_clicked (GtkTreeViewColumn* column, GbemolPlaylist* pls);
static void on_mnu_mode_toggled (GtkCheckMenuItem *item, GbemolPlaylist *pls);
static void on_mnu_remove_activate (GtkWidget* w, GbemolPlaylist *pls);
static void on_btn_clear_search_activate (GtkButton* btn, GbemolPlaylist *pls);
static void on_ent_search_changed (GtkEditable* entry, GbemolPlaylist *pls);
static void on_btn_add_url_clicked (GtkButton* btn, GbemolPlaylist *pls);

static gboolean gbemol_playlist_is_visible (GtkTreeModel* model, GtkTreeIter *iter, GbemolPlaylist* pls);

static const gchar* column_songs[N_COLUMNS] = {
		NULL,
		N_("#"),
		N_("Artist"),
		N_("Title"),
		N_("Album"),
		N_("Track"),
		N_("Date"),
		N_("Genre"),
		N_("Composer"),
		N_("Disc"),
		N_("Comment"),
		N_("Time")
	};


GType
gbemol_playlist_get_type (void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (GbemolPlaylistClass),
			NULL,   /* base_init */
			NULL,   /* base_finalize */
			(GClassInitFunc) gbemol_playlist_class_init,   /* class_init */
			NULL,   /* class_finalize */
			NULL,   /* class_data */
			sizeof (GbemolPlaylist),
			0,      /* n_preallocs */
			(GInstanceInitFunc) gbemol_playlist_init    /* instance_init */
		};

		type = g_type_register_static (GTK_TYPE_VBOX,
				"GbemolPlaylist",
				&info, 0);
	}
	return type;
}

static void
gbemol_playlist_class_init (GObjectClass *g_class)
{
	parent_class = g_type_class_peek_parent (g_class);
	g_class->finalize = gbemol_playlist_finalize;
	g_class->dispose = gbemol_playlist_dispose;	
}

static void
gbemol_playlist_dispose (GObject* obj)
{
	GbemolPlaylist* self = (GbemolPlaylist *) obj;

	if (self->priv->dispose_has_run)
		return;
	
	self->priv->dispose_has_run = TRUE;

	gbemol_cfg_set_int ("playlist", "cmb_search", 
			gtk_combo_box_get_active (GTK_COMBO_BOX (self->priv->cmb_search)));

	G_OBJECT_CLASS (parent_class)->dispose (obj);

}

static void 
gbemol_playlist_finalize (GObject *obj)
{
	g_free (GBEMOL_PLAYLIST (obj)->priv);
	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static void
gbemol_playlist_init (GbemolPlaylist *obj)
{
	obj->priv = g_new0 (GbemolPlaylistPrivate, 1);
	obj->priv->mode = GBEMOL_PLAYLIST_MODE_COLUMNS;
	obj->priv->dispose_has_run = FALSE;

	/* Playlist modes names */
	obj->playlist_modes[GBEMOL_PLAYLIST_MODE_COLUMNS] = _("Columns Mode");
	obj->playlist_modes[GBEMOL_PLAYLIST_MODE_LINES] = _("Lines Mode");
}

static void
gbemol_playlist_build_menus (GbemolPlaylist *pls)
{
	GtkWidget *item, *img, *submnu;
	GSList *group = NULL;
	gint i;

	pls->priv->mnu_songs = gtk_menu_new ();

	/*** SONGS MENU ****/
	/* Remove song */
	item = gtk_image_menu_item_new_with_label (_("Remove"));
	img = gtk_image_new_from_stock (GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (pls->priv->mnu_songs), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_remove_activate), pls);
	gtk_widget_show (item);

	/* Clear songs */
	item = gtk_image_menu_item_new_from_stock (GTK_STOCK_CLEAR, NULL);
	gtk_menu_shell_append (GTK_MENU_SHELL (pls->priv->mnu_songs), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_clear_activate), pls);
	gtk_widget_show (item);

	/* Shuffle songs */
	item = gtk_image_menu_item_new_with_label (_("Shuffle"));
	img = gtk_image_new_from_stock (GTK_STOCK_REFRESH, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (pls->priv->mnu_songs), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_shuffle_activate), pls);
	gtk_widget_show (item);

	item = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (pls->priv->mnu_songs), item);
	gtk_widget_show (item);

	/* Modes Submenu */
	item = gtk_menu_item_new_with_label (_("Playlist Mode"));
	gtk_menu_shell_append (GTK_MENU_SHELL (pls->priv->mnu_songs), item);
	gtk_widget_show (item);

	submnu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), submnu);

	/* Modes */
	item = gtk_radio_menu_item_new_with_label (NULL, pls->playlist_modes[0]);
	g_object_set_data (G_OBJECT (item), "mode", GINT_TO_POINTER (0));
	g_signal_connect (G_OBJECT (item), "toggled", G_CALLBACK (on_mnu_mode_toggled), pls);
	gtk_menu_shell_append (GTK_MENU_SHELL (submnu), item);
	gtk_widget_show (item);
	pls->priv->mnu_item_mode[0] = item;
	group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (item));

	for (i = 1; i < N_PLAYLIST_MODES; i++)
	{
		item = gtk_radio_menu_item_new_with_label (group, pls->playlist_modes[i]);
		g_object_set_data (G_OBJECT (item), "mode", GINT_TO_POINTER (i));
		g_signal_connect (G_OBJECT (item), "toggled", G_CALLBACK (on_mnu_mode_toggled), pls);
		gtk_menu_shell_append (GTK_MENU_SHELL (submnu), item);
		gtk_widget_show (item);
		pls->priv->mnu_item_mode[i] = item;
		group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (item));
	}

	/* Separator */
	item = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (pls->priv->mnu_songs), item);
	gtk_widget_show (item);

	/* Columns edit */
	item = gtk_image_menu_item_new_with_label (_("Edit Columns"));
	img = gtk_image_new_from_stock (GTK_STOCK_GO_DOWN, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (pls->priv->mnu_songs), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_column_activate), pls);
	gtk_widget_show (item);
	pls->priv->mnu_item_columns = item;

	/*** PLAYLISTS MENU ***/
	pls->priv->mnu_playlists = gtk_menu_new ();

	/* Add to current playlist */
	item = gtk_image_menu_item_new_with_label (_("Enqueue"));
	img = gtk_image_new_from_stock (GTK_STOCK_ADD, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (pls->priv->mnu_playlists), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_btn_playlist_add_clicked), pls);
	gtk_widget_show (item);

	/* Replace current playlist */
	item = gtk_image_menu_item_new_with_label (_("Play"));
	img = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (pls->priv->mnu_playlists), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_btn_playlist_replace_clicked), pls);
	gtk_widget_show (item);

	/* Delete selected playlist */
	item = gtk_image_menu_item_new_with_label (_("Delete"));
	img = gtk_image_new_from_stock (GTK_STOCK_DELETE, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), img);
	gtk_menu_shell_append (GTK_MENU_SHELL (pls->priv->mnu_playlists), item);
	g_signal_connect (G_OBJECT (item), "activate", G_CALLBACK (on_mnu_playlist_remove_activate), pls);
	gtk_widget_show (item);
}

GbemolPlaylist* 
gbemol_playlist_new (GbemolMpd* mpd)
{
	GbemolPlaylist *pls;
	GtkCellRenderer* renderer;
	GtkTreeViewColumn *column;
	GtkWidget *hpnd, *btn, *hbox, *vbox, *lbl, *scr;
	GtkTreeIter iter;
	int i;
	gchar *str;

	pls = GBEMOL_PLAYLIST (g_object_new (GBEMOL_TYPE_PLAYLIST, NULL));

	pls->priv->mpd = mpd;


	gbemol_playlist_build_menus (pls);
	pls->priv->tips = gtk_tooltips_new ();

	hpnd = gtk_hpaned_new ();
	gtk_box_pack_start (GTK_BOX (pls), hpnd, TRUE, TRUE, 0);

	/* The playlists store */
	pls->priv->lst_playlists = gtk_list_store_new (N_COLUMNS_PLAYLISTS,
			G_TYPE_STRING,
			G_TYPE_STRING,
			G_TYPE_BOOLEAN,
			G_TYPE_INT,
			G_TYPE_BOOLEAN);

	pls->priv->tv_playlists = GTK_TREE_VIEW (gtk_tree_view_new ());
	gtk_tree_view_set_model (pls->priv->tv_playlists, GTK_TREE_MODEL (pls->priv->lst_playlists));
	gtk_tree_view_set_headers_visible (pls->priv->tv_playlists, FALSE);
	g_object_unref (G_OBJECT (pls->priv->lst_playlists));

	scr = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scr), GTK_SHADOW_ETCHED_IN);
	gtk_container_add (GTK_CONTAINER (scr), GTK_WIDGET (pls->priv->tv_playlists));
	
	gtk_paned_pack1 (GTK_PANED (hpnd), scr, TRUE, FALSE);

	gtk_tree_view_set_search_column (pls->priv->tv_playlists, COLUMN_PLAYLISTS_TITLE);
	pls->priv->sel_playlists = gtk_tree_view_get_selection (pls->priv->tv_playlists);
	gtk_tree_selection_set_mode (pls->priv->sel_playlists, GTK_SELECTION_BROWSE);

	/* The Icon column */
	renderer = gtk_cell_renderer_pixbuf_new ();
	column = gtk_tree_view_column_new_with_attributes (NULL, renderer, 
			"stock-id", COLUMN_PLAYLISTS_ICON,
			NULL);
	gtk_tree_view_append_column (pls->priv->tv_playlists, column);
	
	/* The playlist name column */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (NULL, renderer,
			"text", COLUMN_PLAYLISTS_TITLE,
			"weight", COLUMN_PLAYLISTS_BOLD,
			"weight-set", COLUMN_PLAYLISTS_BOLD_SET,
			NULL);	
	gtk_tree_view_append_column (pls->priv->tv_playlists, column);

	vbox = gtk_vbox_new (FALSE, 5);

	/* The search field */
	hbox = gtk_hbox_new (FALSE, 5);
	lbl = gtk_label_new (_("Search:"));
	gtk_box_pack_start (GTK_BOX (hbox), lbl, FALSE, FALSE, 0);

	pls->priv->ent_search = gtk_entry_new ();
	gtk_tooltips_set_tip (pls->priv->tips, pls->priv->ent_search, _("Enter text to be searched"), NULL);
	gtk_box_pack_start (GTK_BOX (hbox), pls->priv->ent_search, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (pls->priv->ent_search), "changed", G_CALLBACK (on_ent_search_changed), pls);

	pls->priv->cmb_search = gtk_combo_box_new ();
	gtk_box_pack_start (GTK_BOX (hbox), pls->priv->cmb_search, FALSE, FALSE, 0);

	btn = gbemol_utils_make_button (NULL, GTK_STOCK_CLEAR);
	gtk_button_set_relief (GTK_BUTTON (btn), GTK_RELIEF_NONE);
	gtk_tooltips_set_tip (pls->priv->tips, btn, _("Clear search results"), NULL);
	gtk_box_pack_start (GTK_BOX (hbox), btn, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_clear_search_activate), pls);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_clear_search_activate), pls);
	pls->priv->btn_search_clear = btn;

	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 5);

	pls->priv->lst_column_search = gtk_list_store_new (2, GTK_TYPE_STRING, G_TYPE_INT);

	/* The songs list */
	pls->priv->lst_songs = gtk_list_store_new (N_COLUMNS_SONGS,
			G_TYPE_STRING,	  /* Lines mode title, markup */
			G_TYPE_INT,	  /* Position */	
			G_TYPE_STRING,    /* Title */
			G_TYPE_STRING,    /* Artist */
			G_TYPE_STRING,    /* Album */
			G_TYPE_STRING,    /* Track */
			G_TYPE_STRING,    /* Date */
			G_TYPE_STRING,    /* Genre */
			G_TYPE_STRING,    /* Composer */
			G_TYPE_STRING,    /* Disc */
			G_TYPE_STRING,    /* Comment */
			G_TYPE_STRING,    /* Time */
			G_TYPE_INT,       /* Song id */
			G_TYPE_INT,       /* Bold */
			G_TYPE_BOOLEAN,   /* Bold set? */
			G_TYPE_STRING,	  /* Background color */
			G_TYPE_INT,	  /* Background set? */
			G_TYPE_STRING,    /* Foreground color */
			G_TYPE_INT	  /* Foreground set? */
			);


	pls->priv->fil_songs = gtk_tree_model_filter_new (GTK_TREE_MODEL (pls->priv->lst_songs), NULL);
	gtk_tree_model_filter_set_visible_func (GTK_TREE_MODEL_FILTER (pls->priv->fil_songs), 
			(GtkTreeModelFilterVisibleFunc) gbemol_playlist_is_visible, pls, NULL);
	g_object_unref (G_OBJECT (pls->priv->lst_songs));

	pls->priv->tv_songs = GTK_TREE_VIEW (gtk_tree_view_new ());
	gtk_tree_view_set_model (pls->priv->tv_songs, GTK_TREE_MODEL (pls->priv->fil_songs));
	gtk_tree_view_set_rules_hint (pls->priv->tv_songs, TRUE);
	gtk_tree_view_set_headers_clickable (pls->priv->tv_songs, TRUE);
	g_object_unref (G_OBJECT (pls->priv->fil_songs));
	gtk_widget_set_name (GTK_WIDGET (pls->priv->tv_songs), "tv_songs");

	scr = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scr), GTK_SHADOW_ETCHED_IN);
	gtk_container_add (GTK_CONTAINER (scr), GTK_WIDGET (pls->priv->tv_songs));

	gtk_box_pack_start (GTK_BOX (vbox), scr, TRUE, TRUE, 5);

	/* The bottom buttons */
	hbox = gtk_hbox_new (FALSE, 5);

	btn = gbemol_utils_make_button (NULL, GTK_STOCK_NEW);
	gtk_tooltips_set_tip (pls->priv->tips, btn, _("New Playist"), _("Will erase current playlist"));
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_new_playlist_clicked), pls);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_new_playlist_clicked), pls);
	gtk_box_pack_start (GTK_BOX (hbox), btn, FALSE, FALSE, 5);
	pls->priv->btn_new_playlist = btn;

	btn = gbemol_utils_make_button (NULL, GTK_STOCK_SAVE);
	gtk_tooltips_set_tip (pls->priv->tips, btn, _("Save Playlist"), NULL);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_save_playlist_clicked), pls);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_save_playlist_clicked), pls);
	gtk_box_pack_start (GTK_BOX (hbox), btn, FALSE, FALSE, 5);
	pls->priv->btn_save_playlist = btn;

	btn = gbemol_utils_make_button (NULL, GTK_STOCK_NETWORK);
	gtk_tooltips_set_tip (pls->priv->tips, btn, _("Add URL"), NULL);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_add_url_clicked), pls);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_add_url_clicked), pls);
	gtk_box_pack_start (GTK_BOX (hbox), btn, FALSE, FALSE, 5);
	pls->priv->btn_add_url = btn;

	pls->priv->btn_add = gbemol_utils_make_button (_("_Enqueue"), GTK_STOCK_ADD);
	gtk_tooltips_set_tip (pls->priv->tips, pls->priv->btn_add, _("Add playlist to current playlist"), NULL);
	g_signal_connect (G_OBJECT (pls->priv->btn_add), "clicked", G_CALLBACK (on_btn_playlist_add_clicked), pls);
	g_signal_connect (G_OBJECT (pls->priv->btn_add), "activate", G_CALLBACK (on_btn_playlist_add_clicked), pls);
	gtk_box_pack_end (GTK_BOX (hbox), pls->priv->btn_add, FALSE, FALSE, 5);

	pls->priv->btn_replace = gbemol_utils_make_button (_("_Play"), GTK_STOCK_MEDIA_PLAY);
	gtk_tooltips_set_tip (pls->priv->tips, pls->priv->btn_replace, _("Replace current playlist with this playlist"), NULL);
	g_signal_connect (G_OBJECT (pls->priv->btn_replace), "clicked", 
			G_CALLBACK (on_btn_playlist_replace_clicked), pls);
	g_signal_connect (G_OBJECT (pls->priv->btn_replace), "activate", 
			G_CALLBACK (on_btn_playlist_replace_clicked), pls);
	gtk_box_pack_end (GTK_BOX (hbox), pls->priv->btn_replace, FALSE, FALSE, 5);

	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 5);

	gtk_paned_pack2 (GTK_PANED (hpnd), vbox, TRUE, FALSE);

	gtk_tree_view_set_search_column (pls->priv->tv_songs, COLUMN_SONGS_TITLE);
	pls->priv->sel_songs = gtk_tree_view_get_selection (pls->priv->tv_songs);
	gtk_tree_selection_set_mode (pls->priv->sel_songs, GTK_SELECTION_MULTIPLE);

	renderer = gtk_cell_renderer_text_new ();
	g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);
	column = gtk_tree_view_column_new_with_attributes (NULL, renderer,
				"markup", COLUMN_SONGS_MARKUP,
				"foreground", COLUMN_SONGS_FG,
				"foreground-set", COLUMN_SONGS_FG_SET,
				"background", COLUMN_SONGS_BG,
				"background-set", COLUMN_SONGS_BG_SET,
				NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (pls->priv->tv_songs), column);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_reorderable (column, TRUE);
	gtk_tree_view_column_set_visible (column, FALSE);

	/* The columns for the playlist */
	for (i = 1; i < N_COLUMNS; i++) {
		renderer = gtk_cell_renderer_text_new ();
		column = gtk_tree_view_column_new_with_attributes (column_songs[i], renderer,
				"text", i,
				"weight", COLUMN_SONGS_BOLD,
				"weight-set", COLUMN_SONGS_BOLD_SET,
				"foreground", COLUMN_SONGS_FG,
				"foreground-set", COLUMN_SONGS_FG_SET,
				"background", COLUMN_SONGS_BG,
				"background-set", COLUMN_SONGS_BG_SET,
				NULL);
		gtk_tree_view_append_column (GTK_TREE_VIEW (pls->priv->tv_songs), column);
		gtk_tree_view_column_set_resizable (column, TRUE);
		gtk_tree_view_column_set_reorderable (column, TRUE);
		gtk_tree_view_column_set_clickable (column, TRUE);
		g_object_set_data (G_OBJECT (column), "id", GINT_TO_POINTER (i));
		g_signal_connect (G_OBJECT (column), "clicked", G_CALLBACK (on_column_clicked), pls);
		if (i == COLUMN_SONGS_TIME)
			g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);

		str = g_strdup_printf ("column%d", i);
		gtk_tree_view_column_set_visible (column, gbemol_cfg_get_int ("playlist", str));
		g_free (str);

		/* The search combo box */
		if (i != 1)
		{
			gtk_list_store_append (pls->priv->lst_column_search, &iter);
			gtk_list_store_set (pls->priv->lst_column_search, &iter,
					0, column_songs[i],
					1, i,
					-1);
		}


	}

	gtk_combo_box_set_model (GTK_COMBO_BOX (pls->priv->cmb_search), GTK_TREE_MODEL (pls->priv->lst_column_search));
	renderer = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (pls->priv->cmb_search), renderer, TRUE);
	gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (pls->priv->cmb_search), renderer,
				"text", 0,
				NULL);
	g_object_unref (G_OBJECT (pls->priv->lst_column_search));

	gbemol_playlist_populate_playlists (pls);
	gbemol_playlist_set_mode (pls, gbemol_cfg_get_int ("playlist", "mode"));	

	if (gbemol_mpd_is_connected (pls->priv->mpd))
		gbemol_playlist_bold_song (pls, pls->priv->mpd->status->songid);	

	gtk_combo_box_set_active (GTK_COMBO_BOX (pls->priv->cmb_search), gbemol_cfg_get_int ("playlist", "cmb_search"));
	/* Signal connecting */
	g_signal_connect (G_OBJECT (pls->priv->sel_playlists), 
			"changed", 
			G_CALLBACK (on_sel_playlists_changed),
			pls);
	g_signal_connect (G_OBJECT (pls->priv->tv_playlists), 
			"key-press-event", 
			G_CALLBACK (on_tvw_playlists_key_press_event),
			pls);
	g_signal_connect (G_OBJECT (pls->priv->tv_playlists), 
			"button-press-event", 
			G_CALLBACK (on_tvw_playlists_button_press_event),
			pls);
	g_signal_connect (G_OBJECT (pls->priv->tv_songs), 
			"key-press-event", 
			G_CALLBACK (on_tvw_songs_key_press_event),
			pls);
	g_signal_connect (G_OBJECT (pls->priv->tv_songs), 
			"button-press-event", 
			G_CALLBACK (on_tvw_songs_button_press_event),
			pls);

	gtk_widget_show_all (GTK_WIDGET (pls));
	gtk_widget_hide (pls->priv->btn_add);
	gtk_widget_hide (pls->priv->btn_replace);

	return pls;
}

void gbemol_playlist_set_sensitive (GbemolPlaylist* pls, gboolean sensitive)
{
	if (sensitive)
	{
		gbemol_playlist_update_songs (pls);
		gbemol_playlist_populate_playlists (pls);
	}
	else
	{
		gtk_list_store_clear (pls->priv->lst_playlists);
		gtk_list_store_clear (pls->priv->lst_songs);
	}

	gtk_widget_set_sensitive (GTK_WIDGET (pls->priv->tv_playlists), sensitive);
	gtk_widget_set_sensitive (GTK_WIDGET (pls->priv->tv_songs), sensitive);
	gtk_widget_set_sensitive (pls->priv->cmb_search, sensitive);
	gtk_widget_set_sensitive (pls->priv->ent_search, sensitive);
	gtk_widget_set_sensitive (pls->priv->btn_add, sensitive);
	gtk_widget_set_sensitive (pls->priv->btn_replace, sensitive);
	gtk_widget_set_sensitive (pls->priv->btn_new_playlist, sensitive);
	gtk_widget_set_sensitive (pls->priv->btn_save_playlist, sensitive);
	gtk_widget_set_sensitive (pls->priv->btn_search_clear, sensitive);
}

/* Return TRUE if the current playlist is selected */
gboolean
gbemol_playlist_current_is_selected (GbemolPlaylist* pls)
{
	GtkTreeIter iter;
	gboolean current = FALSE;
	GtkTreeModel* mdl;

	if (gtk_tree_selection_get_selected (pls->priv->sel_playlists, &mdl, &iter))
		gtk_tree_model_get (mdl, &iter, COLUMN_PLAYLISTS_CURRENT, &current, -1);

	return current;
}


void gbemol_playlist_clear_all (GbemolPlaylist* pls)
{
	gtk_list_store_clear (pls->priv->lst_playlists);
	gtk_list_store_clear (pls->priv->lst_songs);
}

void gbemol_playlist_set_mode (GbemolPlaylist* pls, GbemolPlaylistMode mode)
{
	gint i;
	GtkTreeViewColumn* column;
	gchar *str;

	switch (mode)
	{
		case GBEMOL_PLAYLIST_MODE_COLUMNS:
			for (i = 1; i < N_COLUMNS; i++)
			{
				str = g_strdup_printf ("column%d", i);
				column = gtk_tree_view_get_column (GTK_TREE_VIEW (pls->priv->tv_songs), i);
				gtk_tree_view_column_set_visible (column, gbemol_cfg_get_int ("playlist", str));
				g_free (str);
			}
			gtk_widget_show (pls->priv->mnu_item_columns);
			gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (pls->priv->tv_songs), TRUE);
			gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (pls->priv->mnu_item_mode[mode]), TRUE);
			break;

		case GBEMOL_PLAYLIST_MODE_LINES:
			for (i = 0; i < N_COLUMNS; i++)
			{
				column = gtk_tree_view_get_column (GTK_TREE_VIEW (pls->priv->tv_songs), i);
				gtk_tree_view_column_set_visible (column, FALSE);
			}
			/* Only two columns needed */
			column = gtk_tree_view_get_column (GTK_TREE_VIEW (pls->priv->tv_songs), COLUMN_SONGS_MARKUP);
			gtk_tree_view_column_set_visible (column, TRUE);

			column = gtk_tree_view_get_column (GTK_TREE_VIEW (pls->priv->tv_songs), COLUMN_SONGS_TIME);
			gtk_tree_view_column_set_visible (column, TRUE);

			gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (pls->priv->tv_songs), FALSE);
			gtk_widget_hide (pls->priv->mnu_item_columns);
			gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (pls->priv->mnu_item_mode[mode]), TRUE);
			break;
		default:
			break;
	}

	gbemol_cfg_set_int ("playlist", "mode", mode);
	pls->priv->mode = mode;

	gbemol_playlist_update_songs (pls);
}


static GList* gbemol_playlist_get_selected (GtkTreeView* view)
{
	GtkTreeModel *model;
	GtkTreeSelection *select;
	GList *rows = NULL;

	select = gtk_tree_view_get_selection (view);
	if (gtk_tree_selection_count_selected_rows (select) > 0)
	{
		rows = gtk_tree_selection_get_selected_rows (select, &model);
		return rows;
	} 
	else
		return NULL;
}

void gbemol_playlist_columns_dialog (GbemolPlaylist* pls)
{
	GtkWidget *dlg, *vbox, *chk[N_COLUMNS];
	GtkTreeViewColumn* column;
	gchar *str;
	int i;

	if (pls->priv->mode == GBEMOL_PLAYLIST_MODE_LINES)
		return;

	dlg = gtk_dialog_new_with_buttons (_("Playlist Columns"), 
			NULL,
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_OK,
			GTK_RESPONSE_ACCEPT,
			GTK_STOCK_CANCEL,
			GTK_RESPONSE_REJECT,
			NULL);

	vbox = gtk_vbox_new (TRUE, 5);

	for (i = 0; i < (N_COLUMNS-1); i++)
	{
		column = gtk_tree_view_get_column (GTK_TREE_VIEW (pls->priv->tv_songs), i+1);
		chk[i+1] = gtk_check_button_new_with_label (column_songs[i+1]);
		gtk_box_pack_start (GTK_BOX (vbox), chk[i+1], FALSE, FALSE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chk[i+1]), 
				gtk_tree_view_column_get_visible (column));
	}

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), vbox, TRUE, TRUE, 5);
	gtk_widget_show_all (vbox);

	if (gtk_dialog_run (GTK_DIALOG (dlg)) == GTK_RESPONSE_ACCEPT)
	{
		for (i = 0; i < (N_COLUMNS-1); i++)
		{
			column = gtk_tree_view_get_column (GTK_TREE_VIEW (pls->priv->tv_songs), i+1);
			gtk_tree_view_column_set_visible (column, 
					gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(chk[i+1])));
			/* Save the config */
			str = g_strdup_printf ("column%d", i+1);			
			gbemol_cfg_set_int ("playlist", str, 
					gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (chk[i+1])));
			g_free (str);
		}
	}

	gtk_widget_destroy (dlg);
}


static void gbemol_playlist_free_selected (GList* l)
{
        if (l)
        {
                g_list_foreach (l, (GFunc) gtk_tree_path_free, NULL);
                g_list_free (l);
        }
}

void gbemol_playlist_append_song (GbemolPlaylist* pls, mpd_Song* song)
{
	GtkTreeIter iter;
	gchar* str, *artist, *time;

	if (song->time == MPD_SONG_NO_TIME)
		time = g_strdup ("--:--");
	else
		time = g_strdup_printf ("%02d:%02d", song->time / 60, song->time % 60);

	if (gbemol_mpd_song_is_stream (song))
		artist = song->name;
	else
		artist = song->artist;

	gtk_list_store_append (pls->priv->lst_songs, &iter);
	gtk_list_store_set (pls->priv->lst_songs, &iter,
			COLUMN_SONGS_TIME, time,
			COLUMN_SONGS_ID, song->id, 
			-1);
	g_free (time);

	switch (pls->priv->mode)
	{
		case GBEMOL_PLAYLIST_MODE_COLUMNS:
			gtk_list_store_set (pls->priv->lst_songs, &iter,
					COLUMN_SONGS_ARTIST, artist,
					COLUMN_SONGS_TITLE, song->title,
					COLUMN_SONGS_ALBUM, song->album,			
					COLUMN_SONGS_TRACK, song->track,
					COLUMN_SONGS_DATE, song->date,
					COLUMN_SONGS_GENRE, song->genre,
					COLUMN_SONGS_COMPOSER, song->composer,
					COLUMN_SONGS_DISC, song->disc,
					COLUMN_SONGS_COMMENT, song->comment, 
					COLUMN_SONGS_POS, song->pos + 1,
					-1);
			break;
		case GBEMOL_PLAYLIST_MODE_LINES:
			if (!gbemol_mpd_song_is_stream (song))
				str = g_markup_printf_escaped 
					("<span weight=\"ultrabold\" size=\"medium\">%s</span>\n"
					 "<span size=\"small\">%s</span> - <span size=\"small\">%s</span>", 
					 song->title, song->artist, song->album);
			else
				str = g_markup_printf_escaped 
					("<span weight=\"ultrabold\" size=\"medium\">%s</span>\n"
					 "<span size=\"small\">%s</span>", 
					 song->title, artist);
			gtk_list_store_set (pls->priv->lst_songs, &iter,
					COLUMN_SONGS_MARKUP, str,
					COLUMN_SONGS_ARTIST, artist,
					COLUMN_SONGS_TITLE, song->title,
					COLUMN_SONGS_ALBUM, song->album,			
					COLUMN_SONGS_TRACK, song->track,
					COLUMN_SONGS_DATE, song->date,
					COLUMN_SONGS_GENRE, song->genre,
					COLUMN_SONGS_COMPOSER, song->composer,
					COLUMN_SONGS_DISC, song->disc,
					COLUMN_SONGS_COMMENT, song->comment, 
					COLUMN_SONGS_POS, song->pos + 1,
				       	-1);
			g_free (str);
			break;	
		default:
			break;
	}
}

void gbemol_playlist_populate_songs (GbemolPlaylist *pls, GList* songs)
{
	gint total_time=0, total_songs=0;

	gtk_list_store_clear (pls->priv->lst_songs);
	if (!songs)
		return;
	do    
	{
		total_time += ((GbemolMpdSong *)songs->data)->time;
		total_songs++;
		gbemol_playlist_append_song (pls, (GbemolMpdSong *)songs->data);
	} while ((songs = g_list_next(songs)));

}

void gbemol_playlist_populate_playlists (GbemolPlaylist* pls)
{
	GList* playlists;
	GtkTreeIter iter;
	
	gtk_list_store_clear (pls->priv->lst_playlists);

	gtk_list_store_append (pls->priv->lst_playlists, &iter);
	gtk_list_store_set (pls->priv->lst_playlists, &iter,
				COLUMN_PLAYLISTS_TITLE, _("Current Playlist"),
				COLUMN_PLAYLISTS_ICON, "gtk-justify-fill",
				COLUMN_PLAYLISTS_CURRENT, TRUE,
				COLUMN_PLAYLISTS_BOLD, PANGO_WEIGHT_ULTRABOLD,
				COLUMN_PLAYLISTS_BOLD_SET, TRUE,				
				-1);

	gtk_tree_selection_select_iter (pls->priv->sel_playlists, &iter);

	playlists = gbemol_mpd_get_playlists (pls->priv->mpd);
	if (!playlists)		
		return;

	do 
	{
		gtk_list_store_append (pls->priv->lst_playlists, &iter);
		gtk_list_store_set (pls->priv->lst_playlists, &iter,
				COLUMN_PLAYLISTS_TITLE, (gchar *)playlists->data,
				COLUMN_PLAYLISTS_CURRENT, FALSE,
				COLUMN_PLAYLISTS_ICON, "gtk-edit",
				-1);
	} while ((playlists = g_list_next(playlists)));

	gbemol_utils_char_list_free (playlists);
} 

void 
gbemol_playlist_bold_song (GbemolPlaylist *pls, gint song_id)
{
	gboolean oro, bold;
	GtkTreeIter iter;
	gint id;
	GtkTreePath* path;
	GtkListStore* lst = pls->priv->lst_songs;

	if (song_id < 0)
		return;

	oro = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (lst), &iter);
	while (oro)
	{
		gtk_tree_model_get (GTK_TREE_MODEL (lst), &iter,
				COLUMN_SONGS_ID, &id,
				-1);

		gtk_tree_model_get (GTK_TREE_MODEL (lst), &iter,
				COLUMN_SONGS_BOLD_SET, &bold,
				-1);

		if (bold)
			gtk_list_store_set (lst, &iter,
					COLUMN_SONGS_BOLD, PANGO_WEIGHT_NORMAL,
					COLUMN_SONGS_BOLD_SET, FALSE,
					COLUMN_SONGS_FG, "black",
					COLUMN_SONGS_FG_SET, FALSE,
					COLUMN_SONGS_BG, "white",
					COLUMN_SONGS_FG_SET, FALSE,
					-1);

		if (id == song_id)
		{
			gtk_list_store_set (lst, &iter,
					COLUMN_SONGS_BOLD, PANGO_WEIGHT_ULTRABOLD,
					COLUMN_SONGS_BOLD_SET, TRUE,
					COLUMN_SONGS_FG, "white",
					COLUMN_SONGS_FG_SET, TRUE,
					COLUMN_SONGS_BG, "black",
					COLUMN_SONGS_FG_SET, TRUE,
					-1);
			path = gtk_tree_model_get_path (GTK_TREE_MODEL (lst), &iter);
			gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (pls->priv->tv_songs), path, NULL, TRUE, 0.5, 0.5);
			gtk_tree_path_free (path);
		}
		oro = gtk_tree_model_iter_next (GTK_TREE_MODEL (lst), &iter);
	}
}


void gbemol_playlist_update_songs (GbemolPlaylist* pls)
{
	GList* songs;

	songs = gbemol_mpd_playlist_get_songs (pls->priv->mpd);
	gbemol_playlist_populate_songs (pls, songs);
	if (pls->priv->mpd->status)
		gbemol_playlist_bold_song (pls, pls->priv->mpd->status->songid);

	gbemol_utils_song_list_free (songs);
	gtk_tree_view_columns_autosize (GTK_TREE_VIEW (pls->priv->tv_songs));
}


void gbemol_playlist_remove_selected_songs (GbemolPlaylist *pls)
{
	GList *l;
	GtkTreeIter iter;
	gint id;
	GtkListStore* lst = pls->priv->lst_songs;
	GbemolMpd* mpd = pls->priv->mpd;

	l = gbemol_playlist_get_selected (pls->priv->tv_songs);

	if (l)
	{
		gbemol_mpd_queue_start (mpd);
		do
		{
			gtk_tree_model_get_iter (GTK_TREE_MODEL (lst), &iter, (GtkTreePath *) l->data);
			gtk_tree_model_get (GTK_TREE_MODEL (lst), &iter, COLUMN_SONGS_ID, &id, -1);
			gbemol_mpd_queue_remove_song (mpd, id);
		} while ((l = g_list_next (l)));

		gbemol_mpd_queue_finish (mpd);
		gbemol_playlist_free_selected (l);
	}

	gbemol_playlist_update_songs (pls);
}

void gbemol_playlist_play_selected (GbemolPlaylist *pls)
{
	GtkTreeIter iter;
	gint id;
	GList* l;

	l = gbemol_playlist_get_selected (pls->priv->tv_songs);

	if (l)
	{
		gtk_tree_model_get_iter (GTK_TREE_MODEL (pls->priv->fil_songs), &iter, (GtkTreePath *) l->data);
		gtk_tree_model_get (GTK_TREE_MODEL (pls->priv->fil_songs), &iter, COLUMN_SONGS_ID, &id, -1);

		gbemol_mpd_player_play_song_by_id (pls->priv->mpd, id);
		gbemol_playlist_free_selected (l);
	}
}

void gbemol_playlist_play_last_song (GbemolPlaylist *pls)
{
	gbemol_mpd_player_play_song_by_pos (pls->priv->mpd, pls->priv->mpd->status->playlistLength - 1);
}
	

void gbemol_playlist_remove_selected_playlist (GbemolPlaylist *pls)
{
	GtkTreeIter iter;
	gchar *str;
	gboolean oro;
	GtkTreeModel* model;

	if (gtk_tree_selection_get_selected (pls->priv->sel_playlists, &model, &iter))
        {
		gtk_tree_model_get (model, &iter, 
				COLUMN_PLAYLISTS_CURRENT, &oro, -1);

		if (!oro)
		{
			gtk_tree_model_get (model, &iter, 
					COLUMN_PLAYLISTS_TITLE, &str, -1);
			gbemol_mpd_delete_playlist (pls->priv->mpd, str);
			g_free (str);
		}
		gbemol_playlist_populate_playlists (pls);
	}
}

void gbemol_playlist_load_selected_playlist (GbemolPlaylist *pls)
{
	GtkTreeIter iter;
	gchar* str;
	GtkTreeModel* model;

	if (gtk_tree_selection_get_selected (pls->priv->sel_playlists, &model, &iter))
        {
		gtk_tree_model_get (model, &iter, COLUMN_PLAYLISTS_TITLE, &str, -1);
		gbemol_mpd_load_playlist (pls->priv->mpd, str);
		g_free (str);
	}
}

gchar* gbemol_playlist_save_dialog (GbemolPlaylist *pls)
{
	GtkWidget* dialog;
	GtkWidget* cmb_entry, *lbl;
	GtkListStore* model;
	gint oro;
	gchar* str;
	GtkTreeIter iter;
	GList *l;

	model = gtk_list_store_new (1, G_TYPE_STRING);

	dialog = gtk_dialog_new_with_buttons (_("Save Playlist"),
			NULL, 
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_OK,
			GTK_RESPONSE_ACCEPT,
			GTK_STOCK_CANCEL,
			GTK_RESPONSE_REJECT,
			NULL);

	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);

	lbl = gtk_label_new (_("Select playlist to overwrite or create a new one"));
	cmb_entry = gtk_combo_box_entry_new_with_model (GTK_TREE_MODEL (model), 0);

	l = gbemol_mpd_get_playlists (pls->priv->mpd);

	if (l)
	{
		do
		{
			gtk_list_store_append (model, &iter);
			gtk_list_store_set (model, &iter, 0, (gchar *)l->data, -1);
		} while ((l = g_list_next(l)));

		gbemol_utils_char_list_free (l);
	}

	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), lbl, TRUE, TRUE, 5);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), cmb_entry, 
			FALSE, FALSE, 0);
	gtk_widget_show_all (GTK_DIALOG(dialog)->vbox);

	oro = gtk_dialog_run (GTK_DIALOG (dialog));

	if (oro == GTK_RESPONSE_ACCEPT)
		str = g_strdup (gtk_combo_box_get_active_text (GTK_COMBO_BOX (cmb_entry)));
	else
		str = NULL;

	gtk_widget_destroy (dialog);

	return str;

}
gchar* gbemol_playlist_add_url_dialog (GbemolPlaylist *pls)
{
	GtkWidget* dialog;
	GtkWidget* entry, *lbl;
	gchar* str;
	gint oro;

	dialog = gtk_dialog_new_with_buttons (_("Add URL to playlist"),
			NULL, 
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_OK,
			GTK_RESPONSE_ACCEPT,
			GTK_STOCK_CANCEL,
			GTK_RESPONSE_REJECT,
			NULL);

	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);

	lbl = gtk_label_new (_("Stream URL:"));
	entry = gtk_entry_new ();

	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), lbl, TRUE, TRUE, 5);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), entry, FALSE, FALSE, 0);
	gtk_widget_show_all (GTK_DIALOG(dialog)->vbox);

	oro = gtk_dialog_run (GTK_DIALOG (dialog));

	if (oro == GTK_RESPONSE_ACCEPT)
		str = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
	else
		str = NULL;

	gtk_widget_destroy (dialog);

	return str;
}

static void 
on_sel_playlists_changed (GtkTreeSelection *select, GbemolPlaylist* pls)
{
	GtkTreeIter iter;
	GtkTreeViewColumn* column;
	GtkTreeModel* model;
        gchar *str;
	GList* songs = NULL;

	column = gtk_tree_view_get_column (pls->priv->tv_songs, COLUMN_SONGS_POS);

	if (gbemol_playlist_current_is_selected (pls))
	{
		gtk_widget_hide (pls->priv->btn_add);
		gtk_widget_hide (pls->priv->btn_replace);
		gtk_widget_show (pls->priv->btn_save_playlist);
		gtk_widget_show (pls->priv->btn_new_playlist);
		gtk_widget_show (pls->priv->btn_add_url);
		gbemol_playlist_update_songs (pls);
	}
	else
	{
		if (gtk_tree_selection_get_selected (select, &model, &iter))
		{
			gtk_tree_model_get (model, &iter, 
					COLUMN_PLAYLISTS_TITLE, &str, 
					-1);
			songs = gbemol_mpd_preview_playlist (pls->priv->mpd, str);
			g_free (str);

			gtk_widget_show (pls->priv->btn_add);
			gtk_widget_show (pls->priv->btn_replace);
			gtk_widget_hide (pls->priv->btn_save_playlist);
			gtk_widget_hide (pls->priv->btn_new_playlist);
			gtk_widget_hide (pls->priv->btn_add_url);

			gtk_tree_view_column_set_visible (column, FALSE);
			
			gbemol_playlist_populate_songs (pls, songs);
			gbemol_utils_song_list_free (songs);
		}
	}

        
}

static gboolean 
on_tvw_playlists_key_press_event (GtkWidget *w, GdkEventKey *event, GbemolPlaylist* pls)
{
	if (event->keyval == GDK_Delete)
	{
		gbemol_playlist_remove_selected_playlist (pls);
		return TRUE;
	}

	return FALSE;
}

static gint 
on_tvw_playlists_button_press_event (GtkWidget *w, GdkEventButton *event, GbemolPlaylist* pls)
{
	if (gbemol_playlist_current_is_selected (pls))
		return FALSE;

	if (event->type == GDK_2BUTTON_PRESS)
		gbemol_playlist_load_selected_playlist (pls);

	if (event->type == GDK_BUTTON_PRESS)
		if (event->button == 3)
		{
			gtk_menu_popup (GTK_MENU(pls->priv->mnu_playlists), 
						NULL, NULL, NULL, NULL,
					       	event->button, event->time);
			return TRUE;
		}

	return FALSE;

}

static gboolean 
on_tvw_songs_key_press_event (GtkWidget* w, GdkEventKey* event, GbemolPlaylist* pls)
{
	if (event->keyval == GDK_Delete)
	{
		if (gbemol_playlist_current_is_selected (pls))
			gbemol_playlist_remove_selected_songs (pls);
		return TRUE;
	}

	return FALSE;
}

static gboolean 
on_tvw_songs_button_press_event (GtkWidget *w, GdkEventButton *event, GbemolPlaylist* pls)
{
	if (event->type == GDK_BUTTON_PRESS)
		if (event->button == 3)
		{
			if (gbemol_playlist_current_is_selected (pls))
				gtk_menu_popup (GTK_MENU (pls->priv->mnu_songs),
					NULL, NULL, NULL, NULL, 
					event->button, event->time);
			return TRUE;
		}

	if (event->type == GDK_2BUTTON_PRESS)
		gbemol_playlist_play_selected (pls);

	return FALSE;
}

static void 
on_btn_new_playlist_clicked (GtkButton *button, GbemolPlaylist* pls)
{
	gbemol_mpd_player_stop (pls->priv->mpd);
	gbemol_mpd_playlist_clear (pls->priv->mpd);
}

static void 
on_btn_save_playlist_clicked (GtkButton *button, GbemolPlaylist* pls)
{
	gchar* path;
	
	path = gbemol_playlist_save_dialog (pls);

	if (path != NULL)
	{
		gbemol_mpd_save_playlist (pls->priv->mpd, path);
		gbemol_playlist_populate_playlists (pls);
		g_free (path);
	}
}

static void 
on_btn_playlist_add_clicked (GtkWidget* b, GbemolPlaylist* pls)
{

	GtkTreeIter iter;
	gchar* str;
	GtkTreeModel* mdl;

	if (gtk_tree_selection_get_selected (pls->priv->sel_playlists, &mdl, &iter))
        {
		gtk_tree_model_get (mdl, &iter, COLUMN_PLAYLISTS_TITLE, &str, -1);
		gbemol_mpd_load_playlist (pls->priv->mpd, str);
		g_free (str);
	}
}

static void 
on_btn_playlist_replace_clicked (GtkWidget* b, GbemolPlaylist* pls)
{
	GtkTreeIter iter;
	gchar* str;
	GtkTreeModel* mdl;

	if (gtk_tree_selection_get_selected (pls->priv->sel_playlists, &mdl, &iter))
        {
		gtk_tree_model_get (mdl, &iter, COLUMN_PLAYLISTS_TITLE, &str, -1);
		gbemol_mpd_playlist_clear (pls->priv->mpd);
		gbemol_mpd_load_playlist (pls->priv->mpd, str);
		g_free (str);
	}

	gbemol_mpd_player_play_song_by_pos (pls->priv->mpd, 0);
}

static void 
on_mnu_playlist_remove_activate (GtkWidget* m, GbemolPlaylist* pls)
{
	gbemol_playlist_remove_selected_playlist (pls);
}

static void 
on_mnu_clear_activate (GtkWidget* w, GbemolPlaylist* pls)
{
	gbemol_mpd_playlist_clear (pls->priv->mpd);
}

static void 
on_mnu_shuffle_activate (GtkWidget* w, GbemolPlaylist* pls)
{
	gbemol_mpd_playlist_shuffle (pls->priv->mpd);
}

static void
on_mnu_column_activate (GtkWidget* w, GbemolPlaylist* pls)
{
	gbemol_playlist_columns_dialog (pls);
}

static void on_column_clicked (GtkTreeViewColumn* column, GbemolPlaylist* pls)
{
	int old_id, new_id;
	GtkSortType old_sort, new_sort;
	GtkTreeViewColumn* c;

	new_sort = GTK_SORT_ASCENDING;
	new_id = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (column), "id"));

	gtk_tree_sortable_get_sort_column_id (GTK_TREE_SORTABLE (pls->priv->lst_songs), &old_id, &old_sort);

	/* Same column, reverse order */
	if (old_id == new_id)
		if (old_sort == GTK_SORT_ASCENDING)
			new_sort = GTK_SORT_DESCENDING;

	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (pls->priv->lst_songs), new_id, new_sort);

	c = gtk_tree_view_get_column (GTK_TREE_VIEW (pls->priv->tv_songs), old_id);
	gtk_tree_view_column_set_sort_indicator (c, FALSE);
	gtk_tree_view_column_set_sort_indicator (column, TRUE);
	gtk_tree_view_column_set_sort_order (column, new_sort);
}

static void on_mnu_mode_toggled (GtkCheckMenuItem *item, GbemolPlaylist *pls)
{
	if (gtk_check_menu_item_get_active (item))
		gbemol_playlist_set_mode (pls, GPOINTER_TO_INT (g_object_get_data(G_OBJECT (item), "mode")));
}

static void
on_btn_clear_search_activate (GtkButton* btn, GbemolPlaylist *pls)
{
	gtk_entry_set_text (GTK_ENTRY (pls->priv->ent_search), "");
}

static gboolean
gbemol_playlist_is_visible (GtkTreeModel* model, GtkTreeIter *iter, GbemolPlaylist* pls)
{
	gchar *str1, *list_str = NULL, *search_str = NULL;
	const gchar *str2, *str3;
	GtkTreeIter cmb_iter;
	gint column;
	gboolean result = TRUE;

	if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (pls->priv->cmb_search), &cmb_iter))
		return TRUE;

	gtk_tree_model_get (GTK_TREE_MODEL (pls->priv->lst_column_search), &cmb_iter, 1, &column, -1);
	gtk_tree_model_get (model, iter, column, &str1, -1);
	
	str2 = gtk_entry_get_text (GTK_ENTRY (pls->priv->ent_search));
	/* Case sensitive search? */
	if (!gbemol_cfg_get_int ("playlist", "case-sensitive"))
	{
		if (str1)
			list_str = g_utf8_casefold (str1, -1);
		if (str2)
			search_str = g_utf8_casefold (str2, -1);
	}
	else
	{
		list_str = str1;
		search_str = g_strdup (str2);
	}

	/* Const strs for comparing */
	str2 = list_str;
	str3 = search_str;

	if (str2 && str3)
		if (!g_strrstr (list_str, search_str))
			result = FALSE;

	if (str1 != list_str)
		g_free (list_str);
	g_free (str1);
	g_free (search_str);

	return result;
}
		
static void 
on_mnu_remove_activate (GtkWidget* w, GbemolPlaylist *pls)
{
	gbemol_playlist_remove_selected_songs (pls);
}

static void 
on_ent_search_changed (GtkEditable* entry, GbemolPlaylist *pls)
{
	gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (pls->priv->fil_songs));
}

static void 
on_btn_add_url_clicked (GtkButton* btn, GbemolPlaylist *pls)
{
	gchar *url;

	url = gbemol_playlist_add_url_dialog (pls);
	if (url)
		gbemol_mpd_playlist_add_song (pls->priv->mpd, url);
	g_free (url);
}


