package main

import (
	"encoding/json"
	"fmt"
	"github.com/godbus/dbus"
	"strings"
)

type daemonConnData struct {
	Path           string `json:"Path"`
	Uuid           string `json:"Uuid"`
	Id             string `json:"Id"`
	HwAddress      string `json:"HwAddress"`
	CloneedAddress string `json:"ClonedAddress"`
	Ssid           string `json:"Ssid"`
}
type daemonConnDataList []daemonConnData

const (
	connTypeWireless = "wireless"

	dbusSender  = "com.deepin.daemon.Network"
	dbusPath    = "/com/deepin/daemon/Network"
	dbusIFC     = "com.deepin.daemon.Network"
	dbusConnIFC = "com.deepin.daemon.Network.ConnectionSession"

	propConnections        = "Connections"
	propConnSections       = "AvailableSections"
	methodEditConnection   = "EditConnections"  // (uuid, devPath) --> (sessionPath)
	methodCreateConnection = "CreateConnection" // (connType, devPath) --> (sessionPath)
)

func main() {
	obj, err := newBusObject()
	if err != nil {
		fmt.Println("Failed to connect bus:", err)
		return
	}
	wireless, err := getWirelessConnections(obj)
	if err != nil {
		fmt.Println("Failed to get wireless connections:", err)
		return
	}
	fmt.Println("Wrireless:", wireless)
}

func newBusObject() (dbus.BusObject, error) {
	conn, err := dbus.SessionBus()
	if err != nil {
		return nil, err
	}
	return conn.Object(dbusSender, dbusPath), nil
}

func getWirelessConnections(obj dbus.BusObject) (daemonConnDataList, error) {
	prop := dbusIFC + "." + propConnections
	variant, err := obj.GetProperty(prop)
	if err != nil {
		fmt.Printf("Failed to get %q error: %s\n", prop, err)
		return nil, err
	}

	// fmt.Println("Signature:", variant.Signature().String())
	value, ok := variant.Value().(string)
	if !ok {
		fmt.Println("Failed to decode:", variant.String())
		return nil, fmt.Errorf("invalid prop value")
	}
	strings.TrimLeft(value, "\"")
	strings.TrimRight(value, "\"")
	// fmt.Println(value)
	var connMap = make(map[string]daemonConnDataList)
	err = json.Unmarshal([]byte(value), &connMap)
	if err != nil {
		fmt.Println("Failed to unmarshal:", err)
		return nil, err
	}
	wireless, ok := connMap[connTypeWireless]
	if !ok {
		return nil, fmt.Errorf("no wireless found")
	}
	return wireless, nil
}
