// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_FEEDBACK_REDACTION_TOOL_PII_TYPES_H_
#define COMPONENTS_FEEDBACK_REDACTION_TOOL_PII_TYPES_H_

namespace redaction {

// PII (Personally Identifiable Information) types that can be detected in the
// debug data.
// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
// Keep the order in sync with enums.xml.
enum class PIIType {
  // Indicates no PII. Mainly for testing.
  kNone = 0,
  // Android App Storage paths. The path starts with either
  // /home/root/<hash>/data/data/<package_name>/ or
  // /home/root/<hash>/data/user_de/<number>/<package_name>/, the path
  // components following <package_name>/ are app specific and will be
  // considered as PII sensitive data.
  kAndroidAppStoragePath = 1,
  // Email addresses.
  kEmail = 2,
  // GAIA (Google Accounts and ID Administration) ID. Gaia ID is a 64-bit
  // integer.
  kGaiaID = 3,
  // IPP (Internet Printing Protocol) Addresses.
  kIPPAddress = 4,
  // IP (Internet Protocol) address. Stores data in two versions: IPv4 (e.g.
  // 127.0.0.1) or IPv6 (e.g. 2001:0db8:85a3:0000:0000:8a2e:0370:7334).
  kIPAddress = 5,
  // Location information related to Cell tower object that's used by
  // ModemManager. Contains two type of data:
  // 1- The Location Area Code (LAC) for GSM and WCDMA networks of the Cell
  // tower object that's used by ModemManager.
  // 2- Cell ID as unique identifier of the cell of the Cell tower object that's
  // used by ModemManager.
  kLocationInfo = 6,
  // MAC address is a unique identifier assigned to a network interface
  // controller (NIC) for use as a network address in communications within a
  // network segment (e.g 00:00:5e:00:53:af). MAC addresses with general meaning
  // like ARP failure result MAC and Broadcast MAC won't be treated as PII
  // sensitive data and won't be included in this category.
  kMACAddress = 7,
  // Window titles that appear in UI hierarchy.
  kUIHierarchyWindowTitles = 8,
  // URLs that can appear in logs.
  kURL = 9,
  // Serial numbers, which also includes PSM identifiers.
  kSerial = 10,
  // SSID (Service Set Identifier) of wifi networks can be detected in the logs
  // provided by wpa_supplicant and shill. BSSID (Basic Service Set Identifier)
  // of a wifi service, is also categorized under this. This type also contains
  // custom names of WiFi, cellular, Ethernet, etc.
  kSSID = 11,
  // Stable identifiers. Contains information in these main categories:
  // 1- Universal Unique Identifiers (UUIDs): UUID can also be given by 'blkid',
  // 'lvs' and 'pvs' tools.
  // 2- Hashes: Hexadecimal strings of length 32, 40 and 64 are considered to be
  // hashes.
  // 3- Eche specific UID which is a base46 conversion of a 32 byte binary value
  // generated by `ash/webui/eche_app_ui/eche_uid_provider.cc`
  kStableIdentifier = 12,
  // Volume labels presented in the 'blkid' tool, and as part of removable
  // media paths shown in various logs such as cros-disks (in syslog).
  kVolumeLabel = 13,
  // Extensible Authentication Protocol (EAP) properties provided by shill.
  kEAP = 14,
  kMaxValue = kEAP,
};

}  // namespace redaction

#endif  // COMPONENTS_FEEDBACK_REDACTION_TOOL_PII_TYPES_H_
