#include "src/asio.h"
#include <iostream>
#include <iomanip>

// 十六进制打印缓冲区内容
void print_hex(const void* data, std::size_t size) {
    const unsigned char* p = static_cast<const unsigned char*>(data);
    
    for (std::size_t i = 0; i < size; ++i) {
        std::cout << std::hex << std::setw(2) << std::setfill('0') 
                  << static_cast<int>(p[i]) << " ";
        
        if ((i + 1) % 16 == 0 || i == size - 1) {
            std::cout << "  ";
            
            // 打印ASCII
            for (std::size_t j = (i / 16) * 16; j <= i; ++j) {
                unsigned char c = p[j];
                if (c >= 32 && c <= 126) {
                    std::cout << c;
                } else {
                    std::cout << ".";
                }
            }
            
            std::cout << std::endl;
        }
    }
    
    std::cout << std::dec; // 恢复十进制输出
}

int main() {
    try {
        std::cout << "流缓冲区 (streambuf) 测试程序" << std::endl;
        std::cout << "==========================" << std::endl;
        
        // 创建缓冲区
        asio::streambuf buf;
        
        std::cout << "初始状态:" << std::endl;
        std::cout << "  大小: " << buf.size() << " 字节" << std::endl;
        std::cout << "  容量: " << buf.capacity() << " 字节" << std::endl;
        std::cout << "  是否为空: " << (buf.empty() ? "是" : "否") << std::endl;
        std::cout << std::endl;
        
        // 写入一些数据
        {
            std::cout << "准备写入数据..." << std::endl;
            std::string data = "Hello, 这是一个测试消息!";
            
            // 获取写入缓冲区
            asio::mutable_buffer write_buf = buf.prepare(data.size());
            
            // 复制数据
            std::memcpy(write_buf.data(), data.data(), data.size());
            
            // 确认写入
            buf.commit(data.size());
            
            std::cout << "  已写入: " << data.size() << " 字节" << std::endl;
            std::cout << "  当前大小: " << buf.size() << " 字节" << std::endl;
            std::cout << "  当前容量: " << buf.capacity() << " 字节" << std::endl;
            std::cout << "  内容: " << buf.str() << std::endl;
            std::cout << std::endl;
        }
        
        // 以十六进制打印缓冲区内容
        {
            std::cout << "缓冲区内容 (十六进制):" << std::endl;
            asio::const_buffer data = buf.data();
            print_hex(data.data(), data.size());
            std::cout << std::endl;
        }
        
        // 读取一部分数据
        {
            std::cout << "读取一部分数据..." << std::endl;
            asio::const_buffer data = buf.data();
            std::size_t to_read = 10; // 读取前10个字节
            
            std::cout << "  读取: " << to_read << " 字节" << std::endl;
            
            // 复制数据
            std::vector<char> read_data(to_read);
            std::memcpy(read_data.data(), data.data(), to_read);
            
            // 消费数据
            buf.consume(to_read);
            
            std::cout << "  读取内容: ";
            for (char c : read_data) {
                std::cout << c;
            }
            std::cout << std::endl;
            
            std::cout << "  剩余大小: " << buf.size() << " 字节" << std::endl;
            std::cout << "  剩余内容: " << buf.str() << std::endl;
            std::cout << std::endl;
        }
        
        // 追加更多数据
        {
            std::cout << "追加更多数据..." << std::endl;
            std::string more_data = " [这是追加的更多数据]";
            
            // 获取写入缓冲区
            asio::mutable_buffer write_buf = buf.prepare(more_data.size());
            
            // 复制数据
            std::memcpy(write_buf.data(), more_data.data(), more_data.size());
            
            // 确认写入
            buf.commit(more_data.size());
            
            std::cout << "  已追加: " << more_data.size() << " 字节" << std::endl;
            std::cout << "  当前大小: " << buf.size() << " 字节" << std::endl;
            std::cout << "  内容: " << buf.str() << std::endl;
            std::cout << std::endl;
        }
        
        // 清空缓冲区
        {
            std::cout << "清空缓冲区..." << std::endl;
            
            std::cout << "  清空前大小: " << buf.size() << " 字节" << std::endl;
            buf.clear();
            std::cout << "  清空后大小: " << buf.size() << " 字节" << std::endl;
            std::cout << "  是否为空: " << (buf.empty() ? "是" : "否") << std::endl;
            std::cout << std::endl;
        }
        
        std::cout << "流缓冲区测试程序完成!" << std::endl;
    }
    catch (const std::exception& e) {
        std::cerr << "异常: " << e.what() << std::endl;
        return 1;
    }
    
    return 0;
} 