/*
 * Copyright (C) 2016-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <inttypes.h>
#include <stdarg.h>

/*
 * Info:
 *
 * Parameter:
 *	1: %rdi
 *	2: %rsi
 *	3: %rdx
 *	4: %rcx
 *	5: %r8
 *	6: %r9
 *	7: stack
 *	8: ...
 *
 * Result:
 *	%rax
 *
 * Callee-saved:
 *	%rbp
 *	%rbx
 *	%r12
 *	%r13
 *	%r14
 *	%r15
 *
 * Caller-saved:
 *	%rax
 *	%r10
 *	%r11
 *
 * Stack pointer:
 *	%rsp
 */

#include <sys/mman.h>

typedef enum jit_reg {
	JIT_RAX = 0,
	JIT_RCX = 1,
	JIT_RDX = 2,
	JIT_RBX = 3,
	JIT_RSP = 4,
	JIT_RBP = 5,
	JIT_RSI = 6,
	JIT_RDI = 7,
	JIT_R8 = 8,
	JIT_R9 = 9,
	JIT_R10 = 10,
	JIT_R11 = 11,
	JIT_R12 = 12,
	JIT_R13 = 13,
	JIT_R14 = 14,
	JIT_R15 = 15,
} jit_reg;

static const uint16_t TMP_REGS =
	(1 << JIT_RAX)
	| (1 << JIT_RCX)
	| (1 << JIT_RDX)
	| (1 << JIT_RSI)
	| (1 << JIT_RDI)
	| (1 << JIT_R8)
	| (1 << JIT_R9)
	| (1 << JIT_R10)
	| (1 << JIT_R11);
static const uint16_t VAR_REGS =
	(1 << JIT_RBX)
	| (1 << JIT_RBP)
	| (1 << JIT_R12)
	| (1 << JIT_R13)
	| (1 << JIT_R14)
	| (1 << JIT_R15);
static const uint16_t CLASS_c =
	1 << JIT_RCX;
static const uint16_t CLASS_r =
	(1 << JIT_RAX)	/* TMP_REGS */
	| (1 << JIT_RCX)
	| (1 << JIT_RDX)
	| (1 << JIT_RSI)
	| (1 << JIT_RDI)
	| (1 << JIT_R8)
	| (1 << JIT_R9)
	| (1 << JIT_R10)
	| (1 << JIT_R11)
	| (1 << JIT_RBX) /* VAR_REGS */
	| (1 << JIT_RBP)
	| (1 << JIT_R12)
	| (1 << JIT_R13)
	| (1 << JIT_R14)
	| (1 << JIT_R15);

static char jit_buf[1024*1024] __attribute__((aligned(4096)));
static char *jit_last = jit_buf;

static void
jit__patch(void *jmp, void *label)
{
	uint32_t offset;

	offset = (char *) label - (char *) jmp - 4;
	*(uint32_t *) jmp = offset;
}

static void
jit__jmp(struct jit_label *label, void *jmp)
{       
	int i;

	for (i = 0; ; i++) {
		assert(i < sizeof(label->jmp) / sizeof(label->jmp[0]));
		if (! label->jmp[i]) {
			label->jmp[i] = jmp;

			if (label->label) {
				jit__patch(label->jmp[i], label->label);
			}
			break;
		}
	}
}

static enum jit_reg
jit__alloc(uint16_t *free_regs, uint16_t regclass)
{
	int r;

	for (r = 0; r < 16; r++) {
		if (r == JIT_RCX) continue; /* %rcx is special */
		if (((*free_regs & regclass) >> r) & 1) {
			*free_regs &= ~(1 << r);
			assert(r != JIT_RSP);
			return r;
		}
	}
	r = JIT_RCX;
	if (((*free_regs & regclass) >> r) & 1) {
		*free_regs &= ~(1 << r);
		assert(r != JIT_RSP);
		return r;
	}
	assert(0); /* No free reg found. */
	return -1;
}

static enum jit_reg
jit__alloc_any(uint16_t *free_regs)
{
	return jit__alloc(free_regs, CLASS_r);
}

static void
jit__free(uint16_t *free_regs, enum jit_reg reg)
{
	if ((TMP_REGS >> reg) & 1) {
		*free_regs |= 1 << reg;
	}
}

static uint8_t *
jit__prefix(
	uint8_t *inst,
	unsigned long size,
	uint8_t reg,
	uint8_t idx,
	uint8_t base
)
{
	if (size == 2) {
		*inst++ = 0x66;
	}
	if (size == 1
	 || size == 8
	 || JIT_R8 <= reg
	 || JIT_R8 <= idx
	 || JIT_R8 <= base) {
		*inst++ = 0x40
			| ((size == 8) << 3)
			| (((reg >> 3) & 1) << 2)
			| (((idx >> 3) & 1) << 1)
			| (((base >> 3) & 1) << 0);
	}

	return inst;
}

static uint8_t *
jit__regop_base(uint8_t *inst, uint8_t op, uint8_t base)
{
	*inst++ = (3 << 6) | (op << 3) | ((base & 7) << 0);

	return inst;
}

static uint8_t *
jit__regop_off_base(
	uint8_t *inst,
	uint8_t regop,
	uint32_t off,
	uint8_t base
)
{
	if (base == JIT_RSP) {
		if (off == 0) {
			*inst++ = (0 << 6) | ((regop & 7) << 3) | (4 << 0);
			*inst++ = 0x24;
		} else if (off <= 0xff) {
			*inst++ = (1 << 6) | ((regop & 7) << 3) | (4 << 0);
			*inst++ = 0x24;
			*(uint8_t *) inst = off;
			inst += sizeof(uint8_t);
		} else {
			*inst++ = (2 << 6) | ((regop & 7) << 3) | (4 << 0);
			*inst++ = 0x24;
			*(uint32_t *) inst = off;
			inst += sizeof(uint32_t);
		}
	} else {
		if (off == 0) {
			*inst++ = (0 << 6) | ((regop & 7) << 3) | ((base & 7) << 0);
		} else if (off <= 0xff) {
			*inst++ = (1 << 6) | ((regop & 7) << 3) | ((base & 7) << 0);
			*(uint8_t *) inst = off;
			inst += sizeof(uint8_t);
		} else {
			*inst++ = (2 << 6) | ((regop & 7) << 3) | ((base & 7) << 0);
			*(uint32_t *) inst = off;
			inst += sizeof(uint32_t);
		}
	}

	return inst;
}

static uint8_t *
jit__regop_off_rip(
	uint8_t *inst,
	uint8_t regop,
	uint8_t *off
)
{
	*inst++ = (0 << 6) | ((regop & 7) << 3) | (JIT_RBP << 0);
	*(uint32_t *) inst = off - (inst + sizeof(uint32_t));
	inst += sizeof(uint32_t);

	return inst;
}

static uint8_t *
jit__inst(uint8_t *inst, uint8_t i)
{
	*inst++ = i;

	return inst;
}

#define ret(inst) \
	jit__inst(inst, 0xc3)

static uint8_t *
jit__instreg(uint8_t *inst, uint8_t i, uint8_t reg)
{
	inst = jit__prefix(inst, 1, 0, 0, reg);
	*inst++ = i | (reg & 7);

	return inst;
}

#define pushq_reg(inst, reg) \
	jit__instreg(inst, 0x50, reg)
#define popq_reg(inst, reg) \
	jit__instreg(inst, 0x58, reg)

static uint8_t *
jit__inst_regop_base(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	uint8_t regop,
	uint8_t base
)
{
	inst = jit__prefix(inst, size, regop, 0, base);
	*inst++ = i;
	inst = jit__regop_base(inst, regop, base);

	return inst;
}

#define shlq_1_base(inst, reg) \
	jit__inst_regop_base(inst, 8, 0xd1, 4, reg);
#define shrq_1_base(inst, reg) \
	jit__inst_regop_base(inst, 8, 0xd1, 5, reg);
#define sarq_1_base(inst, reg) \
	jit__inst_regop_base(inst, 8, 0xd1, 7, reg);

#define shlq_cl_base(inst, reg) \
	jit__inst_regop_base(inst, 8, 0xd3, 4, reg);
#define sarq_cl_base(inst, reg) \
	jit__inst_regop_base(inst, 8, 0xd3, 5, reg);
#define shrq_cl_base(inst, reg) \
	jit__inst_regop_base(inst, 8, 0xd3, 7, reg);

#define negq_base(inst, reg) \
	jit__inst_regop_base(inst, 8, 0xf7, 3, reg)
#define mulq_reg_raxrdx(inst, reg) \
	jit__inst_regop_base(inst, 8, 0xf7, 4, reg)
#define divq_reg_raxrdx(inst, reg) \
	jit__inst_regop_base(inst, 8, 0xf7, 7, reg)

#define addq_reg_base(inst, reg, base) \
	jit__inst_regop_base(inst, 8, 0x01, reg, base)
#define orq_reg_base(inst, reg, base) \
	jit__inst_regop_base(inst, 8, 0x09, reg, base)
#define andq_reg_base(inst, reg, base) \
	jit__inst_regop_base(inst, 8, 0x21, reg, base)
#define subq_reg_base(inst, reg, base) \
	jit__inst_regop_base(inst, 8, 0x29, reg, base)
#define xorq_reg_base(inst, reg, base) \
	jit__inst_regop_base(inst, 8, 0x31, reg, base)

static uint8_t *
jit__inst2_regop_base(
	uint8_t *inst,
	unsigned long size,
	uint8_t i0,
	uint8_t i1,
	uint8_t regop,
	uint8_t base
)
{
	inst = jit__prefix(inst, size, regop, 0, base);
	*inst++ = i0;
	*inst++ = i1;
	inst = jit__regop_base(inst, regop, base);

	return inst;
}

#define setb_base(inst, base) \
	jit__inst2_regop_base(inst, 1, 0x0f, 0x92, 0, base)
#define setae_base(inst, base) \
	jit__inst2_regop_base(inst, 1, 0x0f, 0x93, 0, base)
#define sete_base(inst, base) \
	jit__inst2_regop_base(inst, 1, 0x0f, 0x94, 0, base)
#define setne_base(inst, base) \
	jit__inst2_regop_base(inst, 1, 0x0f, 0x95, 0, base)

static uint8_t *
jit__inst_imm8_base(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	uint8_t op,
	uint8_t base,
	uint8_t imm
)
{
	inst = jit__prefix(inst, size, 0, 0, base);
	*inst++ = i;
	inst = jit__regop_base(inst, op, base);
	*inst++ = imm;

	return inst;
}

#define xorq_imm8_base(inst, imm, reg) \
	jit__inst_imm8_base(inst, 8, 0x83, 6, reg, imm)
#define cmpq_imm8_base(inst, imm, reg) \
	jit__inst_imm8_base(inst, 8, 0x83, 7, reg, imm)

#define shlq_imm8_base(inst, imm, reg) \
	jit__inst_imm8_base(inst, 8, 0xc1, 4, reg, imm)
#define shrq_imm8_base(inst, imm, reg) \
	jit__inst_imm8_base(inst, 8, 0xc1, 5, reg, imm)
#define sarq_imm8_base(inst, imm, reg) \
	jit__inst_imm8_base(inst, 8, 0xc1, 7, reg, imm)

static uint8_t *
jit__inst_imm_base(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	uint8_t op,
	uint8_t base,
	uint32_t imm
)
{
	inst = jit__prefix(inst, size, 0, 0, base);
	*inst++ = i;
	inst = jit__regop_base(inst, op, base);
	switch (size) {
	case 1:
		*(uint8_t *) inst = imm;
		inst += sizeof(uint8_t);
		break;
	case 2:
		*(uint16_t *) inst = imm;
		inst += sizeof(uint16_t);
		break;
	case 4:
	case 8:
		*(uint32_t *) inst = imm;
		inst += sizeof(uint32_t);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	return inst;
}

#define cmpl_imm_base(inst, imm, base) \
	jit__inst_imm_base(inst, 4, 0x81, 7, base, imm);

#define addq_imm_base(inst, imm, base) \
	jit__inst_imm_base(inst, 8, 0x81, 0, base, imm);
#define orq_imm_base(inst, imm, base) \
	jit__inst_imm_base(inst, 8, 0x81, 1, base, imm);
#define andq_imm_base(inst, imm, base) \
	jit__inst_imm_base(inst, 8, 0x81, 4, base, imm);
#define subq_imm_base(inst, imm, base) \
	jit__inst_imm_base(inst, 8, 0x81, 5, base, imm);
#define xorq_imm_base(inst, imm, base) \
	jit__inst_imm_base(inst, 8, 0x81, 6, base, imm);
#define cmpq_imm_base(inst, imm, base) \
	jit__inst_imm_base(inst, 8, 0x81, 7, base, imm);

#define movq_imm32_base(inst, imm, base) \
	jit__inst_imm_base(inst, 8, 0xc7, 0, base, imm)

static uint8_t *
jit__instreg_imm_base(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	uint64_t imm,
	uint8_t base
)
{
	inst = jit__prefix(inst, size, 0, 0, base);
	*inst++ = i | (base & 7);
	*(uint64_t *) inst = imm; \
	inst += sizeof(uint64_t); \

	return inst;
}

#define movq_imm64_base(inst, imm, base) \
	jit__instreg_imm_base(inst, 8, 0xb8, imm, base)

static uint8_t *
jit__inst_imm_off_base(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	uint8_t op,
	uint32_t off,
	uint8_t base,
	uint32_t imm
)
{
	assert(size <= 4);

	inst = jit__prefix(inst, size, 0, 0, base);
	*inst++ = i;
	inst = jit__regop_off_base(inst, op, off, base);
	switch (size) {
	case 1:
		*(uint8_t *) inst = imm;
		inst += sizeof(uint8_t);
		break;
	case 2:
		*(uint16_t *) inst = imm;
		inst += sizeof(uint16_t);
		break;
	case 4:
		*(uint32_t *) inst = imm;
		inst += sizeof(uint32_t);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	return inst;
}

#define movb_imm_off_base(inst, imm, off, base) \
	jit__inst_imm_off_base(inst, 1, 0xc6, 0, off, base, imm)
#define movw_imm_off_base(inst, imm, off, base) \
	jit__inst_imm_off_base(inst, 2, 0xc7, 0, off, base, imm)
#define movl_imm_off_base(inst, imm, off, base) \
	jit__inst_imm_off_base(inst, 4, 0xc7, 0, off, base, imm)

static uint8_t *
jit__inst_base_reg(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	jit_reg base,
	jit_reg reg
)
{
	inst = jit__prefix(inst, size, reg, 0, base);
	*inst++ = i;
	inst = jit__regop_base(inst, reg, base);

	return inst;
}

#define movslq_base_reg(inst, base, reg) \
	jit__inst_base_reg(inst, 8, 0x63, base, reg)
#define movl_base_reg(inst, base, reg) \
	jit__inst_base_reg(inst, 4, 0x8b, base, reg)
#define movq_base_reg(inst, base, reg) \
	jit__inst_base_reg(inst, 8, 0x8b, base, reg)

static uint8_t *
jit__inst_reg_base(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	jit_reg reg,
	jit_reg base
)
{
	return jit__inst_base_reg(inst, size, i, base, reg);
}

#define cmpq_reg_base(inst, reg, base) \
	jit__inst_reg_base(inst, 8, 0x39, reg, base)

static uint8_t *
jit__inst2_base_reg(
	uint8_t *inst,
	unsigned long size,
	uint8_t i0,
	uint8_t i1,
	jit_reg base,
	jit_reg reg
)
{
	assert(base != JIT_RSP);
	assert(reg != JIT_RSP);

	inst = jit__prefix(inst, size, reg, 0, base);
	*inst++ = i0;
	*inst++ = i1;
	inst = jit__regop_base(inst, reg, base);

	return inst;
}

#define movzbq_base_reg(inst, base, reg) \
	jit__inst2_base_reg(inst, 8, 0x0f, 0xb6, base, reg)
#define movzwq_base_reg(inst, base, reg) \
	jit__inst2_base_reg(inst, 8, 0x0f, 0xb7, base, reg)
#define movsbq_base_reg(inst, base, reg) \
	jit__inst2_base_reg(inst, 8, 0x0f, 0xbe, base, reg)
#define movswq_base_reg(inst, base, reg) \
	jit__inst2_base_reg(inst, 8, 0x0f, 0xbf, base, reg)

static uint8_t *
jit__inst_off_base_reg(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	uint32_t off,
	jit_reg base,
	jit_reg reg
)
{
	assert(base != JIT_RSP);
	assert(reg != JIT_RSP);

	inst = jit__prefix(inst, size, reg, 0, base);
	*inst++ = i;
	inst = jit__regop_off_base(inst, reg, off, base);

	return inst;
}

#define movslq_off_base_reg(inst, off, base, reg) \
	jit__inst_off_base_reg(inst, 8, 0x63, off, base, reg)
#define movl_off_base_reg(inst, off, base, reg) \
	jit__inst_off_base_reg(inst, 4, 0x8b, off, base, reg)
#define movq_off_base_reg(inst, off, base, reg) \
	jit__inst_off_base_reg(inst, 8, 0x8b, off, base, reg)

static uint8_t *
jit__inst_reg_off_base(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	jit_reg reg,
	uint32_t off,
	jit_reg base
)
{
	return jit__inst_off_base_reg(inst, size, i, off, base, reg);
}

#define movb_reg_off_base(inst, reg, off, base) \
	jit__inst_reg_off_base(inst, 1, 0x88, reg, off, base)
#define movw_reg_off_base(inst, reg, off, base) \
	jit__inst_reg_off_base(inst, 2, 0x89, reg, off, base)
#define movl_reg_off_base(inst, reg, off, base) \
	jit__inst_reg_off_base(inst, 4, 0x89, reg, off, base)
#define movq_reg_off_base(inst, reg, off, base) \
	jit__inst_reg_off_base(inst, 8, 0x89, reg, off, base)

static uint8_t *
jit__inst2_off_base_reg(
	uint8_t *inst,
	unsigned long size,
	uint8_t i0,
	uint8_t i1,
	uint32_t off,
	jit_reg base,
	jit_reg reg
)
{
	inst = jit__prefix(inst, size, reg, 0, base);
	*inst++ = i0;
	*inst++ = i1;
	inst = jit__regop_off_base(inst, reg, off, base);

	return inst;
}

#define movzbq_off_base_reg(inst, off, base, reg) \
	jit__inst2_off_base_reg(inst, 8, 0x0f, 0xb6, off, base, reg)
#define movzwq_off_base_reg(inst, off, base, reg) \
	jit__inst2_off_base_reg(inst, 8, 0x0f, 0xb7, off, base, reg)
#define movsbq_off_base_reg(inst, off, base, reg) \
	jit__inst2_off_base_reg(inst, 8, 0x0f, 0xbe, off, base, reg)
#define movswq_off_base_reg(inst, off, base, reg) \
	jit__inst2_off_base_reg(inst, 8, 0x0f, 0xbf, off, base, reg)

static uint8_t *
jit__inst_off_rip_reg(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	void *_off,
	jit_reg reg
)
{
	uint8_t *off = _off;

	inst = jit__prefix(inst, size, reg, 0, JIT_RBP);
	*inst++ = i;
	inst = jit__regop_off_rip(inst, reg, off);

	return inst;
}

#define movslq_off_rip_reg(inst, off, reg) \
	jit__inst_off_rip_reg(inst, 8, 0x63, off, reg)

#define movl_off_rip_reg(inst, off, reg) \
	jit__inst_off_rip_reg(inst, 4, 0x8b, off, reg)
#define movq_off_rip_reg(inst, off, reg) \
	jit__inst_off_rip_reg(inst, 8, 0x8b, off, reg)

static uint8_t *
jit__inst2_off_rip_reg(
	uint8_t *inst,
	unsigned long size,
	uint8_t i0,
	uint8_t i1,
	void *_off,
	jit_reg reg
)
{
	uint8_t *off = _off;

	inst = jit__prefix(inst, size, reg, 0, JIT_RBP);
	*inst++ = i0;
	*inst++ = i1;
	inst = jit__regop_off_rip(inst, reg, off);

	return inst;
}

#define movzbq_off_rip_reg(inst, off, reg) \
	jit__inst2_off_rip_reg(inst, 8, 0x0f, 0xb6, off, reg)
#define movzwq_off_rip_reg(inst, off, reg) \
	jit__inst2_off_rip_reg(inst, 8, 0x0f, 0xb7, off, reg)
#define movsbq_off_rip_reg(inst, off, reg) \
	jit__inst2_off_rip_reg(inst, 8, 0x0f, 0xbe, off, reg)
#define movswq_off_rip_reg(inst, off, reg) \
	jit__inst2_off_rip_reg(inst, 8, 0x0f, 0xbf, off, reg)

static uint8_t *
jit__inst_reg_off_rip(
	uint8_t *inst,
	unsigned long size,
	uint8_t i,
	jit_reg reg,
	void *_off
)
{
	uint8_t *off = _off;

	inst = jit__prefix(inst, size, reg, 0, JIT_RBP);
	*inst++ = i;
	inst = jit__regop_off_rip(inst, reg, off);

	return inst;
}

#define movb_reg_off_rip(inst, reg, off) \
	jit__inst_reg_off_rip(inst, 1, 0x88, reg, off)
#define movw_reg_off_rip(inst, reg, off) \
	jit__inst_reg_off_rip(inst, 2, 0x89, reg, off)
#define movl_reg_off_rip(inst, reg, off) \
	jit__inst_reg_off_rip(inst, 4, 0x89, reg, off)
#define movq_reg_off_rip(inst, reg, off) \
	jit__inst_reg_off_rip(inst, 8, 0x89, reg, off)

static uint8_t *
jit__inst_label(
	uint8_t *inst,
	uint8_t i,
	struct jit_label *label
)
{
	*inst++ = i;
	*(uint32_t *) inst = 0; /* Will be patched later. */
	jit__jmp(label, inst);
	inst += sizeof(uint32_t);

	return inst;
}

#define jmp_label(inst, label) \
	jit__inst_label(inst, 0xe9, label)

static uint8_t *
jit__inst2_label(
	uint8_t *inst,
	uint8_t i0,
	uint8_t i1,
	struct jit_label *label
)
{
	*inst++ = i0;
	*inst++ = i1;
	*(uint32_t *) inst = 0; /* Will be patched later. */
	jit__jmp(label, inst);
	inst += sizeof(uint32_t);

	return inst;
}

#define jb_label(inst, label) \
	jit__inst2_label(inst, 0x0f, 0x82, label)
#define jae_label(inst, label) \
	jit__inst2_label(inst, 0x0f, 0x83, label)
#define je_label(inst, label) \
	jit__inst2_label(inst, 0x0f, 0x84, label)
#define jne_label(inst, label) \
	jit__inst2_label(inst, 0x0f, 0x85, label)

static enum jit_reg
jit__eval(struct jit_func *func, uint16_t *free_regs, uint16_t regclass, struct jit_expr *expr)
{
	enum jit_reg reg;
	enum jit_reg rax;
	enum jit_reg rdx;
	enum jit_reg tmp;
	enum jit_reg base;
	uint32_t offset;
	uint8_t *inst;

	switch (expr->type) {
	case JIT_CONST:
		reg = jit__alloc(free_regs, regclass);

		inst = func->tail;
		if (expr->val <= 0xffffffff) {
			/* movq $const, %reg */
			inst = movq_imm32_base(inst, expr->val, reg);
		} else {
			/* movabsq $imm64, %reg */
			inst = movq_imm64_base(inst, expr->val, reg);
		}
		func->tail = inst;
		break;

	case JIT_GLOBAL:
		reg = jit__alloc(free_regs, regclass);

		inst = func->tail;
		switch (expr->sign * expr->size) {
		case -1:
			/* movsbq addr(%rip), %reg */
			inst = movsbq_off_rip_reg(inst, expr->addr, reg);
			break;
		case 1:
			/* movzbq addr(%rip), %reg */
			inst = movzbq_off_rip_reg(inst, expr->addr, reg);
			break;
		case -2:
			/* movswq addr(%rip), %reg */
			inst = movswq_off_rip_reg(inst, expr->addr, reg);
			break;
		case 2:
			/* movzwq addr(%rip), %reg */
			inst = movzwq_off_rip_reg(inst, expr->addr, reg);
			break;
		case -4:
			/* movslq addr(%rip), %reg */
			inst = movslq_off_rip_reg(inst, expr->addr, reg);
			break;
		case 4:
			/* movl addr(%rip), %reg */
			inst = movl_off_rip_reg(inst, expr->addr, reg);
			break;
		case -8:
		case 8:
			/* movq addr(%rip), %reg */
			inst = movq_off_rip_reg(inst, expr->addr, reg);
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
		func->tail = inst;
		break;

	case JIT_PARAM:
	case JIT_LOCAL:
		inst = func->tail;
		if (expr->reg != -1) {
			if ((regclass >> expr->reg) & 1) {
				reg = expr->reg;
			} else {
				reg = jit__alloc(free_regs, regclass);
				/* movq %reg, %reg */
				inst = movq_base_reg(inst,
						expr->reg, reg);
			}

		} else {
			reg = jit__alloc(free_regs, regclass);

			switch (expr->sign * expr->size) {
			case -1:
				/* movsbq offset(%rsp), %reg */
				inst = movsbq_off_base_reg(inst,
						expr->offset, JIT_RSP, reg);
				break;
			case 1:
				/* movzbq offset(%rsp), %reg */
				inst = movzbq_off_base_reg(inst,
						expr->offset, JIT_RSP, reg);
				break;
			case -2:
				/* movswq offset(%rsp), %reg */
				inst = movswq_off_base_reg(inst,
						expr->offset, JIT_RSP, reg);
				break;
			case 2:
				/* movzwq offset(%rsp), %reg */
				inst = movzwq_off_base_reg(inst,
						expr->offset, JIT_RSP, reg);
				break;
			case -4:
				/* movslq offset(%rsp), %reg */
				inst = movslq_off_base_reg(inst,
						expr->offset, JIT_RSP, reg);
				break;
			case 4:
				/* movl offset(%rsp), %reg */
				inst = movl_off_base_reg(inst,
						expr->offset, JIT_RSP, reg);
				break;
			case -8:
			case 8:
				/* movq offset(%rsp), %reg */
				inst = movq_off_base_reg(inst,
						expr->offset, JIT_RSP, reg);
				break;
			default:
				assert(0); /* Mustn't happen. */
			}
		}
		func->tail = inst;
		break;

	case JIT_STAR:
		reg = jit__alloc(free_regs, regclass);

		if (expr->op0->type == JIT_ADD
		 && expr->op0->op1->type == JIT_CONST) {
			/*
			 * *(op0 + const)
			 */
			base = jit__eval(func, free_regs, CLASS_r, expr->op0->op0);
			offset = expr->op0->op1->val;

		} else {
			base = jit__eval(func, free_regs, CLASS_r, expr->op0);
			offset = 0;
		}

		inst = func->tail;
		switch (expr->sign * expr->size) {
		case -1:
			/* movsbq offset(%base), %reg */
			inst = movsbq_off_base_reg(inst, offset, base, reg);
			break;
		case 1:
			/* movzbq offset(%base), %reg */
			inst = movzbq_off_base_reg(inst, offset, base, reg);
			break;
		case -2:
			/* movswq offset(%base), %reg */
			inst = movswq_off_base_reg(inst, offset, base, reg);
			break;
		case 2:
			/* movzwq offset(%base), %reg */
			inst = movzwq_off_base_reg(inst, offset, base, reg);
			break;
		case -4:
			/* movslq offset(%base), %reg */
			inst = movslq_off_base_reg(inst, offset, base, reg);
			break;
		case 4:
			/* movl offset(%base), %reg */
			inst = movl_off_base_reg(inst, offset, base, reg);
			break;
		case -8:
		case 8:
			/* movq offset(%base), %reg */
			inst = movq_off_base_reg(inst, offset, base, reg);
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
		func->tail = inst;

		jit__free(free_regs, base);
		break;

	case JIT_ADDR:
		reg = -1; /* FIXME */
		switch (expr->op0->type) {
		case JIT_GLOBAL:
			assert(0); /* FIXME */
		case JIT_PARAM:
			assert(0); /* FIXME */
		case JIT_LOCAL:
			assert(0); /* FIXME */
		default:
			assert(0); /* Mustn't happen. */
		}
		break;

	case JIT_CONV:
		reg = jit__eval(func, free_regs, regclass & TMP_REGS, expr->op0);

		inst = func->tail;
		switch (expr->sign * expr->size) {
		case -1:
			/* movsbq %regb, %reg */
			inst = movsbq_base_reg(inst, reg, reg);
			break;
		case 1:
			/* movzbq %regb, %reg */
			inst = movzbq_base_reg(inst, reg, reg);
			break;
		case -2:
			/* movswq %regw, %reg */
			inst = movswq_base_reg(inst, reg, reg);
			break;
		case 2:
			/* movzwq %regw, %reg */
			inst = movzwq_base_reg(inst, reg, reg);
			break;
		case -4:
			/* movslq %regd, %reg */
			inst = movslq_base_reg(inst, reg, reg);
			break;
		case 4:
			/* movl %regd, %reg */
			inst = movl_base_reg(inst, reg, reg);
			break;
		case 8:
		default:
			assert(0); /* Mustn't happen. */
		}
		func->tail = inst;
		break;

	case JIT_NOT:
	case JIT_INV:
	case JIT_NEG:
		reg = jit__eval(func, free_regs, regclass & TMP_REGS, expr->op0);

		inst = func->tail;
		switch (expr->type) {
		case JIT_NOT:
			/* cmpq $0, %reg */
			inst = cmpq_imm8_base(inst, 0x00, reg);
			/* sete %regb */
			inst = sete_base(inst, reg);
			/* movzbq %regb, %reg */
			inst = movzbq_base_reg(inst, reg, reg);
			break;
		case JIT_INV:
			/* xorq $-1, %reg */
			inst = xorq_imm8_base(inst, -1, reg);
			break;
		case JIT_NEG:
			/* negq %reg */
			inst = negq_base(inst, reg);
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		func->tail = inst;
		break;

	case JIT_ADD:
	case JIT_SUB:
	case JIT_AND:
	case JIT_OR:
	case JIT_XOR:
		reg = jit__eval(func, free_regs, regclass & TMP_REGS, expr->op0);

		if (expr->op1->type == JIT_CONST
		 && expr->op1->val <= 0xffffffff) {
			/*
			 * op0 operator const
			 */
			inst = func->tail;
			switch (expr->type) {
			case JIT_ADD:
				/* addq $imm, %reg */
				inst = addq_imm_base(inst, expr->op1->val, reg);
				break;
			case JIT_SUB:
				/* subq $imm, %reg */
				inst = subq_imm_base(inst, expr->op1->val, reg);
				break;
			case JIT_AND:
				/* andq $imm, %reg */
				inst = andq_imm_base(inst, expr->op1->val, reg);
				break;
			case JIT_OR:
				/* orq $imm, %reg */
				inst = orq_imm_base(inst, expr->op1->val, reg);
				break;
			case JIT_XOR:
				/* xorq $imm, %reg */
				inst = xorq_imm_base(inst, expr->op1->val, reg);
				break;
			default:
				assert(0); /* Cannot happen. */
			}
			func->tail = inst;

		} else {
			tmp = jit__eval(func, free_regs, CLASS_r, expr->op1);

			inst = func->tail;
			switch (expr->type) {
			case JIT_ADD:
				/* addq %tmp, %reg */
				inst = addq_reg_base(inst, tmp, reg);
				break;
			case JIT_SUB:
				/* subq %tmp, %reg */
				inst = subq_reg_base(inst, tmp, reg);
				break;
			case JIT_AND:
				/* andq %tmp, %reg */
				inst = andq_reg_base(inst, tmp, reg);
				break;
			case JIT_OR:
				/* orq %tmp, %reg */
				inst = orq_reg_base(inst, tmp, reg);
				break;
			case JIT_XOR:
				/* xorq %tmp, %reg */
				inst = xorq_reg_base(inst, tmp, reg);
				break;
			default:
				assert(0); /* Cannot happen. */
			}
			func->tail = inst;

			jit__free(free_regs, tmp);
		}
		break;

	case JIT_MUL:
		reg = jit__eval(func, free_regs, regclass & (1 << JIT_RAX), expr->op0);
		rdx = jit__alloc(free_regs, 1 << JIT_RDX);

		tmp = jit__eval(func, free_regs, CLASS_r, expr->op1);

		inst = func->tail;
		inst = mulq_reg_raxrdx(inst, tmp);
		func->tail = inst;

		jit__free(free_regs, tmp);
		jit__free(free_regs, rdx);
		break;

	case JIT_DIV:
	case JIT_REM:
		rax = jit__eval(func, free_regs, regclass & (1 << JIT_RAX), expr->op0);
		rdx = jit__alloc(free_regs, 1 << JIT_RDX);

		tmp = jit__eval(func, free_regs, CLASS_r, expr->op1);

		inst = func->tail;
		inst = xorq_reg_base(inst, rdx, rdx); /* rdx = 0 */
		inst = divq_reg_raxrdx(inst, tmp);
		func->tail = inst;

		jit__free(free_regs, tmp);
		if (expr->type == JIT_DIV) {
			reg = rax;
			jit__free(free_regs, rdx);
		} else if (expr->type == JIT_REM) {
			reg = rdx;
			jit__free(free_regs, rax);
			assert(0); /* TODO: remove this line and examine result */
		} else {
			assert(0); /* Cannot happen. */
		}
		break;

	case JIT_LSL:
	case JIT_ASR:
	case JIT_LSR:
		reg = jit__eval(func, free_regs, regclass & TMP_REGS, expr->op0);

		if (expr->op1->type == JIT_CONST
		 && expr->op1->val == 1) {
			/*
			 * op0 shift 1
			 */
			inst = func->tail;
			switch (expr->type) {
			case JIT_LSL:
				/* shlq $1, %reg */
				inst = shlq_1_base(inst, reg);
				break;
			case JIT_ASR:
				/* sarq $1, %reg */
				inst = sarq_1_base(inst, reg);
				break;
			case JIT_LSR:
				/* shrq $1, %reg */
				inst = shrq_1_base(inst, reg);
				break;
			default:
				assert(0); /* Cannot happen. */
			}
			func->tail = inst;

		} else if (expr->op1->type == JIT_CONST) {
			/*
			 * op0 shift const
			 */
			assert(expr->op1->val != 0);

			inst = func->tail;
			switch (expr->type) {
			case JIT_LSL:
				/* shlq $const, %reg */
				inst = shlq_imm8_base(inst, expr->op1->val & 0x3f, reg);
				break;
			case JIT_ASR:
				/* sarq $const, %reg */
				inst = sarq_imm8_base(inst, expr->op1->val & 0x3f, reg);
				break;
			case JIT_LSR:
				/* shrq $const, %reg */
				inst = shrq_imm8_base(inst, expr->op1->val & 0x3f, reg);
				break;
			default:
				assert(0); /* Cannot happen. */
			}
			func->tail = inst;

		} else {
			tmp = jit__eval(func, free_regs, CLASS_c, expr->op1);

			inst = func->tail;
			switch (expr->type) {
			case JIT_LSL:
				/* shlq %cl, %reg */
				inst = shlq_cl_base(inst, reg);
				break;
			case JIT_ASR:
				/* sarq %cl, %reg */
				inst = sarq_cl_base(inst, reg);
				break;
			case JIT_LSR:
				/* shrq %cl, %reg */
				inst = shrq_cl_base(inst, reg);
				break;
			default:
				assert(0); /* Cannot happen. */
			}
			func->tail = inst;

			jit__free(free_regs, tmp);
		}
		break;

	case JIT_EQ:
	case JIT_NEQ:
	case JIT_B:
	case JIT_AE:
		reg = jit__eval(func, free_regs, regclass & TMP_REGS, expr->op0);
		tmp = jit__eval(func, free_regs, CLASS_r, expr->op1);

		inst = func->tail;
		/* cmpq %tmp, %reg */
		inst = cmpq_reg_base(inst, tmp, reg);
		switch (expr->type) {
		case JIT_EQ:
			/* sete %regb */
			inst = sete_base(inst, reg);
			break;
		case JIT_NEQ:
			/* setne %regb */
			inst = setne_base(inst, reg);
			break;
		case JIT_B:
			/* setb %regb */
			inst = setb_base(inst, reg);
			break;
		case JIT_AE:
			/* setae %regb */
			inst = setae_base(inst, reg);
			break;
		default:
			assert(0); /* Cannot happen. */
		}

		/* movzbq %regb, %reg */
		inst = movzbq_base_reg(inst, reg, reg);
		func->tail = inst;

		jit__free(free_regs, tmp);
		break;

	default:
		reg = -1;
		assert(0); /* Mustn't happen. */
	}

	return reg;
}

/*
 * Store value in %reg to variable.
 */
static void
jit_st(struct jit_func *func, uint16_t *free_regs, struct jit_expr *dst, jit_reg reg)
{
	jit_reg base;
	uint32_t offset;
	uint8_t *inst;

	switch (dst->type) {
	case JIT_CONST:
		assert(0); /* Mustn't happen. */

	case JIT_GLOBAL:
		inst = func->tail;
		switch (dst->size) {
		case 1:
			/* movb %reg, addr(%rip) */
			inst = movb_reg_off_rip(inst, reg, dst->addr);
			break;
		case 2:
			/* movw %reg, addr(%rip) */
			inst = movw_reg_off_rip(inst, reg, dst->addr);
			break;
		case 4:
			/* movl %reg, addr(%rip) */
			inst = movl_reg_off_rip(inst, reg, dst->addr);
			break;
		case 8:
			/* movq %reg, addr(%rip) */
			inst = movq_reg_off_rip(inst, reg, dst->addr);
			break;
		default:
			assert(0);
		}
		func->tail = inst;
		break;

	case JIT_PARAM:
	case JIT_LOCAL:
		inst = func->tail;
		if (dst->reg != -1) {
			/* movq %reg, %reg */
			inst = movq_base_reg(inst, reg, dst->reg);

		} else {
			switch (dst->size) {
			case 1:
				/* movb %reg, offset(%rsp) */
				inst = movb_reg_off_base(inst, reg, dst->offset, JIT_RSP);
				break;
			case 2:
				/* movw %reg, offset(%rsp) */
				inst = movw_reg_off_base(inst, reg, dst->offset, JIT_RSP);
				break;
			case 4:
				/* movl %reg, offset(%rsp) */
				inst = movl_reg_off_base(inst, reg, dst->offset, JIT_RSP);
				break;
			case 8:
				/* movq %reg, addr(%rsp) */
				inst = movq_reg_off_base(inst, reg, dst->offset, JIT_RSP);
				break;
			default:
				assert(0);
			}
		}
		func->tail = inst;
		break;

	case JIT_STAR:
		if (dst->op0->type == JIT_ADD
		 && dst->op0->op1->type == JIT_CONST) {
			/*
			 * *(op0 + const) = ...
			 */
			base = jit__eval(func, free_regs, CLASS_r, dst->op0->op0);
			offset = dst->op0->op1->val;

		} else {
			base = jit__eval(func, free_regs, CLASS_r, dst->op0);
			offset = 0;
		}

		inst = func->tail;
		switch (dst->size) {
		case 1:
			/* movb %reg, offset(%base) */
			inst = movb_reg_off_base(inst, reg, offset, base);
			break;
		case 2:
			/* movw %reg, offset(%base) */
			inst = movw_reg_off_base(inst, reg, offset, base);
			break;
		case 4:
			/* movl %reg, offset(%base) */
			inst = movl_reg_off_base(inst, reg, offset, base);
			break;
		case 8:
			/* movq %reg, offset(%base) */
			inst = movq_reg_off_base(inst, reg, offset, base);
			break;
		default:
			assert(0);
		}
		func->tail = inst;

		jit__free(free_regs, base);
		break;

	case JIT_ADDR:

	case JIT_CONV:

	case JIT_NOT:
	case JIT_INV:
	case JIT_NEG:

	case JIT_ADD:
	case JIT_SUB:
	case JIT_MUL:
	case JIT_DIV:
	case JIT_REM:

	case JIT_AND:
	case JIT_OR:
	case JIT_XOR:

	case JIT_LSL:
	case JIT_ASR:
	case JIT_LSR:

	case JIT_EQ:
	case JIT_NEQ:
	case JIT_B:
	case JIT_AE:
		assert(0); /* Mustn't happen. */
	}
}

static void
jit_x86_64_assign(struct jit_func *func, struct jit_expr *res, struct jit_expr *op0)
{
	uint16_t free_regs = TMP_REGS;
	uint8_t *inst;

	if (res->type == JIT_STAR
	 && res->size <= 4
	 && op0->type == JIT_CONST) {
		/*
		 * *res->op0 = const;
		 */
		enum jit_reg base;
		uint32_t offset;

		if (res->op0->type == JIT_ADD
		 && res->op0->op1->type == JIT_CONST) {
			base = jit__eval(func, &free_regs, CLASS_r, res->op0->op0);
			offset = res->op0->op1->val;
		} else {
			base = jit__eval(func, &free_regs, CLASS_r, res->op0);
			offset = 0;
		}

		inst = func->tail;
		switch (res->size) {
		case 1:
			/* movb $imm, off(%base) */
			inst = movb_imm_off_base(inst, op0->val, offset, base);
			break;
		case 2:
			/* movw $imm, off(%base) */
			inst = movw_imm_off_base(inst, op0->val, offset, base);
			break;
		case 4:
			/* movl $imm, off(%base) */
			inst = movl_imm_off_base(inst, op0->val, offset, base);
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
		func->tail = inst;

		jit__free(&free_regs, base);

	} else {
		/*
		 * res = op0;
		 */
		enum jit_reg reg;

		/* Copy value from op0 to %reg. */
		reg = jit__eval(func, &free_regs, CLASS_r, op0);

		/* Copy value from %reg to res */
		jit_st(func, &free_regs, res, reg);

		jit__free(&free_regs, reg);
	}

	assert(free_regs == TMP_REGS);
}

static void
jit_x86_64_goto(struct jit_func *func, struct jit_label *label)
{
	uint8_t *inst;

	inst = func->tail;
	/* jmp label */
	inst = jmp_label(inst, label);
	func->tail = inst;
}

static void
jit_x86_64_ifgoto(struct jit_func *func, struct jit_expr *cond, struct jit_label *label)
{
	uint16_t free_regs = TMP_REGS;
	enum jit_reg reg;
	enum jit_reg tmp;
	uint8_t *inst;

	switch (cond->type) {
	case JIT_CONST:
		assert(0); /* Mustn't happen. */

	case JIT_NOT:
		/*
		 * if (! cond->op0) goto label;
		 */
		reg = jit__eval(func, &free_regs, CLASS_r, cond->op0);

		inst = func->tail;
		/* cmpq $0, %rax */
		inst = cmpq_imm8_base(inst, 0x00, reg);

		/* je label */
		inst = je_label(inst, label);
		func->tail = inst;

		jit__free(&free_regs, reg);
		break;

	case JIT_EQ:
	case JIT_NEQ:
	case JIT_B:
	case JIT_AE:
		reg = jit__eval(func, &free_regs, CLASS_r, cond->op0);
		tmp = jit__eval(func, &free_regs, CLASS_r, cond->op1);

		inst = func->tail;
		/* cmpq %tmp, %reg */
		inst = cmpq_reg_base(inst, tmp, reg);
		switch (cond->type) {
		case JIT_EQ:
			/* je label */
			inst = je_label(inst, label);
			break;
		case JIT_NEQ:
			/* jne label */
			inst = jne_label(inst, label);
			break;
		case JIT_B:
			/* jb label */
			inst = jb_label(inst, label);
			break;
		case JIT_AE:
			/* jae label */
			inst = jae_label(inst, label);
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		func->tail = inst;

		jit__free(&free_regs, tmp);
		jit__free(&free_regs, reg);
		break;

	default:
		/* Copy value from cond to %rax. */
		reg = jit__eval(func, &free_regs, CLASS_r, cond);

		inst = func->tail;
		/* cmpq $0, %rax */
		inst = cmpq_imm8_base(inst, 0x00, JIT_RAX);
		/* jne label */
		inst = jne_label(inst, label);
		func->tail = inst;

		jit__free(&free_regs, reg);
		break;
	}

	assert(free_regs == TMP_REGS);
}

static void
jit_x86_64_switchgoto(
	struct jit_func *func,
	struct jit_expr *expr,
	struct jit_label *def,
	int val[],
	struct jit_label *lab[]
)
{
	uint16_t free_regs = TMP_REGS;
	jit_reg reg;
	uint8_t *inst;
	unsigned int n;

	/* Get expr to %reg. */
	reg = jit__eval(func, &free_regs, CLASS_r, expr);

	inst = func->tail;
	for (n = 0; lab[n]; n++) {
		/* cmp val[n], %rax */
		inst = cmpl_imm_base(inst, val[n], reg);
		/* je lab[n] */
		inst = je_label(inst, lab[n]);
	}
	/* jmp def */
	inst = jmp_label(inst, def);
	func->tail = inst;

	jit__free(&free_regs, reg);

	assert(free_regs == TMP_REGS);
}

static void
jit_x86_64_label(struct jit_func *func, struct jit_label *label)
{
	unsigned int i;

	label->label = func->tail;

	for (i = 0;
	    i < sizeof(label->jmp) / sizeof(label->jmp[0]) && label->jmp[i];
	    i++) {
		jit__patch(label->jmp[i], label->label);
	}
}

static void
jit_x86_64_call(struct jit_func *func, struct jit_expr *res, void *_ptr, ...)
{
	uint8_t *ptr = _ptr;
	static const unsigned int reg[] = {
		JIT_RDI,
		JIT_RSI,
		JIT_RDX,
		JIT_RCX,
		JIT_R8,
		JIT_R9,
	};
	uint16_t free_regs = TMP_REGS;
	va_list arglist;
	struct jit_expr *arg;
	int32_t offset;
	signed int n;
	uint8_t *inst;

	/*
	 * Copy parameter to registers.
	 */
	va_start(arglist, _ptr);

	for (n = 0; ; n++) {
		arg = va_arg(arglist, struct jit_expr *);
		if (arg == NULL) {
			break;
		}

		assert(n < sizeof(reg) / sizeof(reg[0]));
		(void) jit__eval(func, &free_regs, 1 << reg[n], arg);
	}

	va_end(arglist);

	/*
	 * Call function.
	 */
	inst = func->tail;
	/* call func */
	offset = ptr - (inst + 5);
	*inst++ = 0xe8;
	*(uint32_t *) inst = offset;
	inst += sizeof(uint32_t);
	func->tail = inst;

	/*
	 * Free registers.
	 */
	for (n--; 0 <= n; n--) {
		jit__free(&free_regs, reg[n]);
	}

	/*
	 * Copy result to register.
	 */
	jit__alloc(&free_regs, 1 << JIT_RAX);
	if (res != NULL) {
		jit_st(func, &free_regs, res, JIT_RAX);
	}
	jit__free(&free_regs, JIT_RAX);

	assert(free_regs == TMP_REGS);
}

static void
jit_x86_64_brk(struct jit_func *func, struct jit_stmt *stmt)
{
	uint8_t *inst;

	inst = func->tail;
	*inst++ = 0xcc;
	func->tail = inst;
}

#if 1
/* Just to get the encodings... */
asm (
	"jit_encodings: .globl jit_encodings\n"

	/* enter */
	"	pushq %rax\n"
	"	pushq %rcx\n"
	"	pushq %r8\n"
	"	pushq %r9\n"
	"	movq %rsi, %rbp\n"
	"	subq $0x1234, %rsp\n"

	/* leave */
	"	addq $0x1234, %rsp\n"
	"	popq %rax\n"
	"	popq %rcx\n"
	"	popq %r8\n"
	"	popq %r9\n"
	"	ret\n"

	/* const -> reg */
	"	movabsq $0x0123456789abcdef, %rax\n"
	"	movabsq $0x0123456789abcdef, %rcx\n"
	"	movabsq $0x0123456789abcdef, %r8\n"
	"	movabsq $0x0123456789abcdef, %r9\n"
	"	movq $0x01234567, %rax\n"
	"	movq $0x01234567, %rcx\n"
	"	movq $0x01234567, %r8\n"
	"	movq $0x01234567, %r9\n"

	/* global -> reg */
	"	movsbq jit_encodings(%rip), %rax\n"
	"	movsbq jit_encodings(%rip), %rcx\n"
	"	movsbq jit_encodings(%rip), %r8\n"
	"	movsbq jit_encodings(%rip), %r9\n"
	"	movzbq jit_encodings(%rip), %rax\n"
	"	movzbq jit_encodings(%rip), %rcx\n"
	"	movzbq jit_encodings(%rip), %r8\n"
	"	movzbq jit_encodings(%rip), %r9\n"
	"	movswq jit_encodings(%rip), %rax\n"
	"	movswq jit_encodings(%rip), %rcx\n"
	"	movswq jit_encodings(%rip), %r8\n"
	"	movswq jit_encodings(%rip), %r9\n"
	"	movzwq jit_encodings(%rip), %rax\n"
	"	movzwq jit_encodings(%rip), %rcx\n"
	"	movzwq jit_encodings(%rip), %r8\n"
	"	movzwq jit_encodings(%rip), %r9\n"
	"	movslq jit_encodings(%rip), %rax\n"
	"	movslq jit_encodings(%rip), %rcx\n"
	"	movslq jit_encodings(%rip), %r8\n"
	"	movslq jit_encodings(%rip), %r9\n"
	"	movl jit_encodings(%rip), %eax\n"
	"	movl jit_encodings(%rip), %ecx\n"
	"	movl jit_encodings(%rip), %r8d\n"
	"	movl jit_encodings(%rip), %r9d\n"
	"	movq jit_encodings(%rip), %rax\n"
	"	movq jit_encodings(%rip), %rcx\n"
	"	movq jit_encodings(%rip), %r8\n"
	"	movq jit_encodings(%rip), %r9\n"

	/* local -> reg */
	"	movsbq 1024(%rsp), %rax\n"
	"	movsbq 1024(%rsp), %rcx\n"
	"	movsbq 1024(%rsp), %r8\n"
	"	movsbq 1024(%rsp), %r9\n"
	"	movzbq 1024(%rsp), %rax\n"
	"	movzbq 1024(%rsp), %rcx\n"
	"	movzbq 1024(%rsp), %r8\n"
	"	movzbq 1024(%rsp), %r9\n"
	"	movswq 1024(%rsp), %rax\n"
	"	movswq 1024(%rsp), %rcx\n"
	"	movswq 1024(%rsp), %r8\n"
	"	movswq 1024(%rsp), %r9\n"
	"	movzwq 1024(%rsp), %rax\n"
	"	movzwq 1024(%rsp), %rcx\n"
	"	movzwq 1024(%rsp), %r8\n"
	"	movzwq 1024(%rsp), %r9\n"
	"	movslq 1024(%rsp), %rax\n"
	"	movslq 1024(%rsp), %rcx\n"
	"	movslq 1024(%rsp), %r8\n"
	"	movslq 1024(%rsp), %r9\n"
	"	movl 1024(%rsp), %eax\n"
	"	movl 1024(%rsp), %ecx\n"
	"	movl 1024(%rsp), %r8d\n"
	"	movl 1024(%rsp), %r9d\n"
	"	movq 1024(%rsp), %rax\n"
	"	movq 1024(%rsp), %rcx\n"
	"	movq 1024(%rsp), %r8\n"
	"	movq 1024(%rsp), %r9\n"

	/* attr/reg -> reg */
	"	movsbq 1024(%rax), %rax\n"
	"	movsbq 1024(%rcx), %rax\n"
	"	movsbq 1024(%r8), %rax\n"
	"	movsbq 1024(%r9), %rax\n"
	"	movsbq 1024(%rax), %rcx\n"
	"	movsbq 1024(%rax), %r8\n"
	"	movsbq 1024(%rax), %r9\n"
	"	movzbq 1024(%rax), %rax\n"
	"	movzbq 1024(%rax), %rcx\n"
	"	movzbq 1024(%rax), %r8\n"
	"	movzbq 1024(%rax), %r9\n"
	"	movswq 1024(%rax), %rax\n"
	"	movswq 1024(%rax), %rcx\n"
	"	movswq 1024(%rax), %r8\n"
	"	movswq 1024(%rax), %r9\n"
	"	movzwq 1024(%rax), %rax\n"
	"	movzwq 1024(%rax), %rcx\n"
	"	movzwq 1024(%rax), %r8\n"
	"	movzwq 1024(%rax), %r9\n"
	"	movslq 1024(%rax), %rax\n"
	"	movslq 1024(%rax), %rcx\n"
	"	movslq 1024(%rax), %r8\n"
	"	movslq 1024(%rax), %r9\n"
	"	movl 1024(%rax), %eax\n"
	"	movl 1024(%rax), %ecx\n"
	"	movl 1024(%rax), %r8d\n"
	"	movl 1024(%rax), %r9d\n"
	"	movq 1024(%rax), %rax\n"
	"	movq 1024(%rax), %rcx\n"
	"	movq 1024(%rax), %r8\n"
	"	movq 1024(%rax), %r9\n"

	/* cpssp -> reg */
	"	movq %rbp, %rax\n"
	"	movq %rbp, %rcx\n"
	"	movq %rbp, %r8\n"
	"	movq %rbp, %r9\n"

	/* reg -> global */
	"	movb %al, jit_encodings(%rip)\n"
	"	movw %ax, jit_encodings(%rip)\n"
	"	movl %eax, jit_encodings(%rip)\n"
	"	movq %rax, jit_encodings(%rip)\n"
	"	movb %cl, jit_encodings(%rip)\n"
	"	movw %cx, jit_encodings(%rip)\n"
	"	movl %ecx, jit_encodings(%rip)\n"
	"	movq %rcx, jit_encodings(%rip)\n"

	/* reg -> local */
	"	movb %al, 1024(%rsp)\n"
	"	movb %cl, 1024(%rsp)\n"
	"	movb %r8b, 1024(%rsp)\n"
	"	movb %r9b, 1024(%rsp)\n"
	"	movw %ax, 1024(%rsp)\n"
	"	movw %cx, 1024(%rsp)\n"
	"	movw %r8w, 1024(%rsp)\n"
	"	movw %r9w, 1024(%rsp)\n"
	"	movl %eax, 1024(%rsp)\n"
	"	movl %ecx, 1024(%rsp)\n"
	"	movl %r8d, 1024(%rsp)\n"
	"	movl %r9d, 1024(%rsp)\n"
	"	movq %rax, 1024(%rsp)\n"
	"	movq %rcx, 1024(%rsp)\n"
	"	movq %r8, 1024(%rsp)\n"
	"	movq %r9, 1024(%rsp)\n"

	/* reg -> attr/reg */
	"	movb %al, 1024(%rax)\n"
	"	movb %al, 1024(%rcx)\n"
	"	movb %al, 1024(%r8)\n"
	"	movb %al, 1024(%r9)\n"
	"	movb %cl, 1024(%rax)\n"
	"	movb %r8b, 1024(%rax)\n"
	"	movb %r9b, 1024(%rax)\n"
	"	movw %ax, 1024(%rax)\n"
	"	movw %cx, 1024(%rax)\n"
	"	movw %r8w, 1024(%rax)\n"
	"	movw %r9w, 1024(%rax)\n"
	"	movl %eax, 1024(%rax)\n"
	"	movl %ecx, 1024(%rax)\n"
	"	movl %r8d, 1024(%rax)\n"
	"	movl %r9d, 1024(%rax)\n"
	"	movq %rax, 1024(%rax)\n"
	"	movq %rcx, 1024(%rax)\n"
	"	movq %r8, 1024(%rax)\n"
	"	movq %r9, 1024(%rax)\n"

	/* conv */
	"	movsbq %al, %rax\n"
	"	movsbq %cl, %rcx\n"
	"	movsbq %r8b, %r8\n"
	"	movsbq %r9b, %r9\n"
	"	movzbq %al, %rax\n"
	"	movzbq %cl, %rcx\n"
	"	movzbq %r8b, %r8\n"
	"	movzbq %r9b, %r9\n"
	"	movswq %ax, %rax\n"
	"	movswq %cx, %rcx\n"
	"	movswq %r8w, %r8\n"
	"	movswq %r9w, %r9\n"
	"	movzwq %ax, %rax\n"
	"	movzwq %cx, %rcx\n"
	"	movzwq %r8w, %r8\n"
	"	movzwq %r9w, %r9\n"
	"	movslq %eax, %rax\n"
	"	movslq %ecx, %rcx\n"
	"	movslq %r8d, %r8\n"
	"	movslq %r9d, %r9\n"
	"	movl %eax, %eax\n"
	"	movl %ecx, %ecx\n"
	"	movl %r8d, %r8d\n"
	"	movl %r9d, %r9d\n"

	/* not */
	"	cmpq $0, %rax\n"
	"	sete %al\n"
	"	movzbq %al, %rax\n"
	"	cmpq $0, %rcx\n"
	"	sete %cl\n"
	"	movzbq %al, %rcx\n"
	"	cmpq $0, %r8\n"
	"	sete %r8b\n"
	"	movzbq %r8b, %r8\n"
	"	cmpq $0, %r9\n"
	"	sete %r9b\n"
	"	movzbq %r9b, %r9\n"

	/* inv */
	"	xorq $-1, %rax\n"
	"	xorq $-1, %rcx\n"
	"	xorq $-1, %r8\n"
	"	xorq $-1, %r9\n"

	/* neg */
	"	negq %rax\n"
	"	negq %rcx\n"
	"	negq %r8\n"
	"	negq %r9\n"

	/* add */
	"	addq %rax, %rax\n"
	"	addq %rax, %rcx\n"
	"	addq %rax, %r8\n"
	"	addq %rax, %r9\n"
	"	addq %rax, %rax\n"
	"	addq %rcx, %rax\n"
	"	addq %r8, %rax\n"
	"	addq %r9, %rax\n"

	/* sub */
	"	subq %rax, %rax\n"
	"	subq %rax, %rcx\n"
	"	subq %rax, %r8\n"
	"	subq %rax, %r9\n"
	"	subq %rax, %rax\n"
	"	subq %rcx, %rax\n"
	"	subq %r8, %rax\n"
	"	subq %r9, %rax\n"

	/* "	mulb %rdx, %rax\n" */
	/* "	divb %rdx, %rax\n" */

	/* and */
	"	andq %rax, %rax\n"
	"	andq %rax, %rcx\n"
	"	andq %rax, %r8\n"
	"	andq %rax, %r9\n"
	"	andq %rax, %rax\n"
	"	andq %rcx, %rax\n"
	"	andq %r8, %rax\n"
	"	andq %r9, %rax\n"

	/* or */
	"	orq %rax, %rax\n"
	"	orq %rax, %rcx\n"
	"	orq %rax, %r8\n"
	"	orq %rax, %r9\n"
	"	orq %rax, %rax\n"
	"	orq %rcx, %rax\n"
	"	orq %r8, %rax\n"
	"	orq %r9, %rax\n"

	/* xor */
	"	xorq %rax, %rax\n"
	"	xorq %rax, %rcx\n"
	"	xorq %rax, %r8\n"
	"	xorq %rax, %r9\n"
	"	xorq %rax, %rax\n"
	"	xorq %rcx, %rax\n"
	"	xorq %r8, %rax\n"
	"	xorq %r9, %rax\n"

	/* lsl */
	"	shlq $1, %rax\n"
	"	shlq $1, %rcx\n"
	"	shlq $1, %r8\n"
	"	shlq $1, %r9\n"
	"	shlq $0x12, %rax\n"
	"	shlq $0x12, %rcx\n"
	"	shlq $0x12, %r8\n"
	"	shlq $0x12, %r9\n"
	"	shlq %cl, %rax\n"
	"	shlq %cl, %rcx\n"
	"	shlq %cl, %r8\n"
	"	shlq %cl, %r9\n"

	/* asr */
	"	sarq $1, %rax\n"
	"	sarq $1, %rcx\n"
	"	sarq $1, %r8\n"
	"	sarq $1, %r9\n"
	"	sarq $0x12, %rax\n"
	"	sarq $0x12, %rcx\n"
	"	sarq $0x12, %r8\n"
	"	sarq $0x12, %r9\n"
	"	sarq %cl, %rax\n"
	"	sarq %cl, %rcx\n"
	"	sarq %cl, %r8\n"
	"	sarq %cl, %r9\n"

	/* lsr */
	"	shrq $1, %rax\n"
	"	shrq $1, %rcx\n"
	"	shrq $1, %r8\n"
	"	shrq $1, %r9\n"
	"	shrq $0x12, %rax\n"
	"	shrq $0x12, %rcx\n"
	"	shrq $0x12, %r8\n"
	"	shrq $0x12, %r9\n"
	"	shrq %cl, %rax\n"
	"	shrq %cl, %rcx\n"
	"	shrq %cl, %r8\n"
	"	shrq %cl, %r9\n"

	/* eq */
	"	cmpq %rax, %rax\n"
	"	sete %al\n"
	"	movzbq %al, %rax\n"
	"	cmpq %rax, %rcx\n"
	"	sete %cl\n"
	"	movzbq %cl, %rcx\n"
	"	cmpq %rax, %r8\n"
	"	sete %r8b\n"
	"	movzbq %r8b, %r8\n"
	"	cmpq %rax, %r9\n"
	"	sete %r9b\n"
	"	movzbq %r9b, %r9\n"
	"	cmpq %rax, %rax\n"
	"	sete %al\n"
	"	movzbq %al, %rax\n"
	"	cmpq %rcx, %rax\n"
	"	sete %al\n"
	"	movzbq %al, %rax\n"

	/* neq */
	"	setne %al\n"

	/* b */
	"	setb %al\n"

	/* ae */
	"	setae %al\n"

	/* goto */
	"	jmp jit_encodings\n"

	/* ifgoto */
	"	cmpq $0, %rax\n"
	"	je jit_encodings\n"
	"	jne jit_encodings\n"

	/* switchgoto */
	"	cmpl $0x12345678, %eax\n"
	"	je jit_encodings\n"
	"	cmpl $0x12345678, %ecx\n"
	"	je jit_encodings\n"

	/* brk */
	"	int3\n"
);
#endif

void
jit__compile(struct jit_func *func)
{
	uint16_t var_regs = VAR_REGS;
	int reg;
	int off;
	struct jit_expr *p;
	struct jit_expr *l;
	uint8_t *inst;
	struct jit_stmt *stmt;

	off = 0;
	for (p = func->param_last; p; p = p->prev) {
		reg = jit__alloc_any(&var_regs);

		if (reg != -1) {
			p->reg = reg;

			/* Set offset. */
			p->offset = off;

			/* Size. */
			off += 8;

		} else {
			/* Align. */
			off += p->size - 1;
			off &= ~(p->size - 1);

			/* Set offset. */
			p->offset = off;

			/* Size. */
			off += p->size;
		}
	}
	for (l = func->local_last; l; l = l->prev) {
		reg = jit__alloc_any(&var_regs);

		if (reg != -1) {
			l->reg = reg;

			/* Set offset. */
			l->offset = off;

			/* Size. */
			off += 8;

		} else {
			/* Align. */
			off += l->size - 1;
			off &= ~(l->size - 1);

			/* Set offset. */
			l->offset = off;

			/* Size. */
			off += l->size;
		}
	}
	/* Align stack. */
	off += 8; /* For return address. */
	off += 15;
	off &= ~0xf;

	for (p = func->param_first; p; p = p->next) {
		p->offset = off - p->offset;
	}
	for (l = func->local_first; l; l = l->next) {
		l->offset = off - l->offset;
	}

	func->head = jit_last;
	func->tail = jit_last;

	inst = func->tail;
	for (p = func->param_last; p; p = p->prev) {
		if (p->reg != -1) {
			/* FIXME */
			/* push %reg */
			inst = pushq_reg(inst, p->reg);
			off -= 8;
			/* movq %rsi, %reg */
			inst = movq_base_reg(inst, JIT_RSI, p->reg);
		}
	}
	for (l = func->local_last; l; l = l->prev) {
		if (l->reg != -1) {
			/* FIXME */
			/* push %reg */
			inst = pushq_reg(inst, l->reg);
			off -= 8;
		}
	}
	off -= 8; /* For return address. */
	if (off) {
		/* subq $off, %rsp */
		inst = subq_imm_base(inst, off, JIT_RSP);
	}
	func->tail = inst;

	for (stmt = func->stmt_first; stmt; stmt = stmt->next) {
		switch (stmt->type) {
		case JIT_STMT_ASSIGN:
			jit_x86_64_assign(func,
					stmt->assign.dst, stmt->assign.src);
			break;
		case JIT_STMT_GOTO:
			jit_x86_64_goto(func,
					stmt->goto_label);
			break;
		case JIT_STMT_IFGOTO:
			jit_x86_64_ifgoto(func,
					stmt->ifgoto.cond, stmt->ifgoto.label);
			break;
		case JIT_STMT_SWITCHGOTO:
			jit_x86_64_switchgoto(func,
					stmt->switchgoto.expr,
					stmt->switchgoto.def,
					stmt->switchgoto.val,
					stmt->switchgoto.lab);
			break;
		case JIT_STMT_LABEL:
			jit_x86_64_label(func, stmt->label);
			break;
		case JIT_STMT_CALL:
			jit_x86_64_call(func, stmt->call.dst, stmt->call.func,
					stmt->call.arg[0], stmt->call.arg[1],
					stmt->call.arg[2], stmt->call.arg[3],
					stmt->call.arg[4], stmt->call.arg[5],
					NULL);
			break;
		case JIT_STMT_BRK:
			jit_x86_64_brk(func, stmt);
			break;
		}
	}

	inst = func->tail;
	if (off) {
		/* addq $off, %rsp */
		inst = addq_imm_base(inst, off, JIT_RSP);
	}
	off += 8; /* For return address. */
	for (l = func->local_first; l; l = l->next) {
		if (l->reg != -1) {
			/* FIXME */
			off -= 8;
			/* popq %reg */
			inst = popq_reg(inst, l->reg);

			jit__free(&var_regs, l->reg);
		}
	}
	for (p = func->param_first; p; p = p->next) {
		if (p->reg != -1) {
			/* FIXME */
			off -= 8;
			/* popq %reg */
			inst = popq_reg(inst, p->reg);
		}
	}
	/* ret */
	inst = ret(inst);
	func->tail = inst;

	jit_last = func->tail;
}

static uint8_t *
jit__modrm(uint8_t *inst, uint8_t *modp, uint8_t *rp, uint8_t *bp, uint32_t *offp)
{
	uint8_t m;

	m = *inst++;
	*modp = (m >> 6) & 0x3;
	*rp = (m >> 3) & 0x7;
	*bp = (m >> 0) & 0x7;
	switch ((m >> 6) & 3) {
	case 0:
		*offp = 0;
		break;
	case 1:
		*offp = *inst++;
		break;
	case 2:
		*offp = *(uint32_t *) inst;
		inst += sizeof(uint32_t);
		break;
	case 3:
		*offp = 0;
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	return inst;
}

static uint8_t *
jit__imm(uint8_t *inst, unsigned long size, uint64_t *immp)
{
	switch (size) {
	case 1:
		*immp = *(uint8_t *) inst;
		inst += sizeof(uint8_t);
		break;
	case 2:
		*immp = *(uint16_t *) inst;
		inst += sizeof(uint16_t);
		break;
	case 4:
		*immp = *(uint32_t *) inst;
		inst += sizeof(uint32_t);
		break;
	case 8:
		*immp = *(uint64_t *) inst;
		inst += sizeof(uint64_t);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	return inst;
}

static void
jit__print_size(unsigned long size)
{
	switch (size) {
	case 1:
		fprintf(stderr, "b");
		break;
	case 2:
		fprintf(stderr, "w");
		break;
	case 4:
		fprintf(stderr, "l");
		break;
	case 8:
		fprintf(stderr, "q");
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
}

static void
jit__print_reg(uint8_t r)
{
	static const char *name[16] = {
		"%rax",
		"%rcx",
		"%rdx",
		"%rbx",
		"%rsp",
		"%rbp",
		"%rsi",
		"%rdi",
		"%r8",
		"%r9",
		"%r10",
		"%r11",
		"%r12",
		"%r13",
		"%r14",
		"%r15",
	};

	assert(/* 0 <= r && */ r < sizeof(name) / sizeof(name[0]));

	fprintf(stderr, "%s", name[r]);
}

static void
jit__print_modrm(uint8_t mod, uint8_t r, uint32_t off)
{
	switch (mod) {
	case 0:
		fprintf(stderr, "(");
		jit__print_reg(r);
		fprintf(stderr, ")");
		break;
	case 1:
		fprintf(stderr, "0x%02x(", off);
		jit__print_reg(r);
		fprintf(stderr, ")");
		break;
	case 2:
		fprintf(stderr, "0x%08x(", off);
		jit__print_reg(r);
		fprintf(stderr, ")");
		break;
	case 3:
		jit__print_reg(r);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
}

static void
jit__print_cond(uint8_t c)
{
	static const char *name[] = {
		"o",
		"c",
		"e",
		"be",
		"s",
		"p",
		"l",
		"le",
	};

	if (c & 1) fprintf(stderr, "n");
	fprintf(stderr, "%s", name[c >> 1]);
}

static uint8_t *
jit__disas(uint8_t *inst)
{
	uint16_t i;
	unsigned long size;
	uint8_t rexw;
	uint8_t rexr;
	uint8_t rexb;
	uint8_t mod;
	uint8_t r;
	uint8_t b;
	uint32_t off;
	uint64_t imm;

	fprintf(stderr, "%08lx: ", (unsigned long) inst);

	size = 4;
	rexw = 0;
	rexr = 0;
	rexb = 0;
	i = *inst++;
	if (i == 0x66) {
		/* Word prefix */
		size = 2;
		i = *inst++;
	}
	if ((i & 0xf0) == 0x40) {
		/* REX prefix */
		/* FIXME */
		rexw = (i >> 3) & 1;
		rexr = (i >> 2) & 1;
		rexb = (i >> 0) & 1;
		i = *inst++;
	}
	if (i == 0x0f) {
		/* 0x0f prefix */
		i = 0x100 + *inst++;
	}
	switch (i) {
	case 0x01:
	case 0x09:
	case 0x11:
	case 0x19:
	case 0x21:
	case 0x29:
	case 0x31:
	case 0x39:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		switch (i) {
		case 0x01:
			fprintf(stderr, "add");
			break;
		case 0x09:
			fprintf(stderr, "or");
			break;
		case 0x11:
			fprintf(stderr, "adc");
			break;
		case 0x19:
			fprintf(stderr, "sbb");
			break;
		case 0x21:
			fprintf(stderr, "and");
			break;
		case 0x29:
			fprintf(stderr, "sub");
			break;
		case 0x31:
			fprintf(stderr, "xor");
			break;
		case 0x39:
			fprintf(stderr, "cmp");
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		jit__print_reg((rexr << 3) | r);
		fprintf(stderr, ", ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0x50: case 0x51: case 0x52: case 0x53:
	case 0x54: case 0x55: case 0x56: case 0x57:
		b = i & 0x7;
		fprintf(stderr, "push");
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		jit__print_reg((rexb << 3) | b);
		break;
	case 0x58: case 0x59: case 0x5a: case 0x5b:
	case 0x5c: case 0x5d: case 0x5e: case 0x5f:
		b = i & 0x7;
		fprintf(stderr, "pop");
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		jit__print_reg((rexb << 3) | b);
		break;
	case 0x63:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		fprintf(stderr, "movsl ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		fprintf(stderr, ", ");
		jit__print_reg((rexr << 3) | r);
		break;
	case 0x81:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		imm = *(uint32_t *) inst;
		inst += sizeof(uint32_t);
		switch (r) {
		case 0x0:
			fprintf(stderr, "add");
			break;
		case 0x1:
			fprintf(stderr, "or");
			break;
		case 0x2:
			fprintf(stderr, "adc");
			break;
		case 0x3:
			fprintf(stderr, "sbb");
			break;
		case 0x4:
			fprintf(stderr, "and");
			break;
		case 0x5:
			fprintf(stderr, "sub");
			break;
		case 0x6:
			fprintf(stderr, "xor");
			break;
		case 0x7:
			fprintf(stderr, "cmp");
			break;
		default:
			fprintf(stderr, "%x: Unknown\n", r);
			assert(0);
		}
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		fprintf(stderr, "$0x%08lx", imm);
		fprintf(stderr, ", ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0x83:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		imm = *(uint8_t *) inst;
		inst += sizeof(uint8_t);
		switch (r) {
		case 0x0:
			fprintf(stderr, "add");
			break;
		case 0x1:
			fprintf(stderr, "or");
			break;
		case 0x2:
			fprintf(stderr, "adc");
			break;
		case 0x3:
			fprintf(stderr, "sbb");
			break;
		case 0x4:
			fprintf(stderr, "and");
			break;
		case 0x5:
			fprintf(stderr, "sub");
			break;
		case 0x6:
			fprintf(stderr, "xor");
			break;
		case 0x7:
			fprintf(stderr, "cmp");
			break;
		default:
			fprintf(stderr, "%x: Unknown\n", r);
			assert(0);
		}
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		fprintf(stderr, "$0x%02lx", imm);
		fprintf(stderr, ", ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0x88:
		size = 1;
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		fprintf(stderr, "mov");
		jit__print_size(size);
		fprintf(stderr, " ");
		jit__print_reg((rexr << 3) | r);
		fprintf(stderr, ", ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0x89:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		fprintf(stderr, "mov");
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		jit__print_reg((rexr << 3) | r);
		fprintf(stderr, ", ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0x8b:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		fprintf(stderr, "mov");
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		fprintf(stderr, ", ");
		jit__print_reg((rexr << 3) | r);
		break;
	case 0xb8: case 0xb9: case 0xba: case 0xbb:
	case 0xbc: case 0xbd: case 0xbe: case 0xbf:
		r = i & 0x7;
		inst = jit__imm(inst, size << rexw, &imm);
		fprintf(stderr, "mov");
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		fprintf(stderr, "$0x%lx", imm);
		fprintf(stderr, ", ");
		jit__print_reg((rexr << 3) | r);
		break;
	case 0xc1:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		inst = jit__imm(inst, 1, &imm);
		switch (r) {
		case 0x4:
			fprintf(stderr, "shl");
			break;
		case 0x5:
			fprintf(stderr, "shr");
			break;
		case 0x7:
			fprintf(stderr, "sar");
			break;
		default:
			fprintf(stderr, "%x: Unknown\n", r);
			assert(0);
		}
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		fprintf(stderr, "$0x%lx", imm);
		fprintf(stderr, ", ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0xc3:
		fprintf(stderr, "ret");
		inst = NULL;
		break;
	case 0xc6:
		size = 1;
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		inst = jit__imm(inst, 1, &imm);
		switch (r) {
		case 0x0:
			fprintf(stderr, "mov");
			break;
		default:
			fprintf(stderr, "%x: Unknown\n", r);
			assert(0);
		}
		jit__print_size(size);
		fprintf(stderr, " ");
		fprintf(stderr, "$0x%lx", imm);
		fprintf(stderr, ", ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0xc7:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		inst = jit__imm(inst, size, &imm);
		switch (r) {
		case 0x0:
			fprintf(stderr, "mov");
			break;
		default:
			fprintf(stderr, "%x: Unknown\n", r);
			assert(0);
		}
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		fprintf(stderr, "$0x%lx", imm);
		fprintf(stderr, ", ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0xcc:
		fprintf(stderr, "int3");
		break;
	case 0xd1:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		imm = 1;
		switch (r) {
		case 0x4:
			fprintf(stderr, "shl");
			break;
		case 0x5:
			fprintf(stderr, "shr");
			break;
		case 0x7:
			fprintf(stderr, "sar");
			break;
		default:
			fprintf(stderr, "%x: Unknown\n", r);
			assert(0);
		}
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		fprintf(stderr, "$0x%lx", imm);
		fprintf(stderr, ", ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0xd3:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		switch (r) {
		case 0x4:
			fprintf(stderr, "shl");
			break;
		case 0x5:
			fprintf(stderr, "shr");
			break;
		case 0x7:
			fprintf(stderr, "sar");
			break;
		default:
			fprintf(stderr, "%x: Unknown\n", r);
			assert(0);
		}
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		jit__print_reg(JIT_RCX);
		fprintf(stderr, ", ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0xe8:
		imm = *(uint32_t *) inst;
		inst += sizeof(uint32_t);
		fprintf(stderr, "call ");
		fprintf(stderr, "0x%08x", (uint32_t) (long) (inst + imm));
		break;
	case 0xe9:
		imm = *(uint32_t *) inst;
		inst += sizeof(uint32_t);
		fprintf(stderr, "jmp ");
		fprintf(stderr, "0x%08x", (uint32_t) (long) (inst + imm));
		break;
	case 0xf7:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		switch (r) {
		case 0x3:
			fprintf(stderr, "neg");
			break;
		case 0x4:
			fprintf(stderr, "mul");
			break;
		case 0x7:
			fprintf(stderr, "div");
			break;
		default:
			fprintf(stderr, "%x: Unknown\n", r);
			assert(0);
		}
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		if (r == 0x4) {
			fprintf(stderr, ", ");
			jit__print_reg(JIT_RAX);
		}
		break;
	case 0x180: case 0x181: case 0x182: case 0x183:
	case 0x184: case 0x185: case 0x186: case 0x187:
	case 0x188: case 0x189: case 0x18a: case 0x18b:
	case 0x18c: case 0x18d: case 0x18e: case 0x18f:
		imm = *(uint32_t *) inst;
		inst += sizeof(uint32_t);
		fprintf(stderr, "j");
		jit__print_cond(i & 0xf);
		fprintf(stderr, " ");
		fprintf(stderr, "0x%08x", (uint32_t) (long) (inst + imm));
		break;
	case 0x190: case 0x191: case 0x192: case 0x193:
	case 0x194: case 0x195: case 0x196: case 0x197:
	case 0x198: case 0x199: case 0x19a: case 0x19b:
	case 0x19c: case 0x19d: case 0x19e: case 0x19f:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		fprintf(stderr, "set");
		jit__print_cond(i & 0xf);
		fprintf(stderr, " ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		break;
	case 0x1b6:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		fprintf(stderr, "movzb");
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		fprintf(stderr, ", ");
		jit__print_reg((rexr << 3) | r);
		break;
	case 0x1b7:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		fprintf(stderr, "movzw");
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		fprintf(stderr, ", ");
		jit__print_reg((rexr << 3) | r);
		break;
	case 0x1be:
		inst = jit__modrm(inst, &mod, &r, &b, &off);
		fprintf(stderr, "movsb");
		jit__print_size(size << rexw);
		fprintf(stderr, " ");
		jit__print_modrm(mod, (rexb << 3) | b, off);
		fprintf(stderr, ", ");
		jit__print_reg((rexr << 3) | r);
		break;
	default:
		fprintf(stderr, "%02x: Unknown\n", i);
		assert(0);
	}

	fprintf(stderr, "\n");

	return inst;
}

void
jit_x86_64_dump(void *_inst)
{
	uint8_t *inst = _inst;

	while (inst) {
		inst = jit__disas(inst);
	}
	fprintf(stderr, "\n");
}

void
jit_exec(void *_func, void *_cpssp)
{
	void (*func)(void *) = _func;

	(*func)(_cpssp);
}

void
jit_comp_reset(struct jit_comp *jit)
{
}

void
jit_comp_create(struct jit_comp *jit)
{
	int ret;

	ret = mprotect(jit_buf, sizeof(jit_buf),
			PROT_READ | PROT_WRITE | PROT_EXEC);
	assert(0 <= ret);
}

void
jit_comp_destroy(struct jit_comp *jit)
{
}
