/*
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG	0

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue-shm.h"
#include "glue-suspend.h"

#include "chip_ti_1520.h"

#define CHIP_(x) chip_ti_1520_ ## x

struct cpssp {
	/*
	 * Config
	 */

	/*
	 * Signals
	 */
	/* PCI bus */
	struct sig_pci_bus *port_pci_bus;
	struct sig_std_logic *port_int[2];
	/* Cardbus 0/1 */
	struct sig_std_logic *port_cb_power[2];
	struct sig_std_logic *port_cb_reset_hash_[2];
	struct sig_pci_bus *port_cb[2];

	int state_n_cd1[2];
	int state_n_cd2[2];
	int state_cvs1[2];
	int state_cvs2[2];

	/*
	 * State
	 */
	int powered[2];
	uint8_t int_set[2];

	/* Config Space */
	uint8_t io_en[2];
	uint8_t mem_en[2];
	uint8_t mast_en[2];
	uint8_t vga_en[2];
	uint8_t perr_en[2];
	uint8_t serr_en[2];
	uint8_t cache_line_size[2];
	uint8_t latency_timer[2];
	uint32_t base_address[2];
	uint8_t pci_bus_number[2];
	uint8_t cardbus_bus_number[2];
	uint8_t subordinate_bus_number[2];
	uint8_t cardbus_latency_timer[2];
	uint32_t mem_base[2][2];
	uint32_t mem_limit[2][2];
	uint32_t io_base[2][2];
	uint16_t io_limit[2][2];
	uint8_t interrupt_line[2];
	/* 4-14 */
	uint8_t cperren;
	uint8_t cserren;
	uint8_t isaen[2];
	uint8_t vgaen[2];
	uint8_t mabtmode;
	uint8_t crst[2];
	uint8_t intr[2];
	uint8_t prefetch0[2];
	uint8_t prefetch1[2];
	uint8_t posten[2];

	/* Subsystem Vendor ID Register */
	/* 4-15 */
	uint16_t subsystem_vendor_id[2];

	/* Subsystem Device ID Register */
	/* 4-15 */
	uint16_t subsystem_device_id[2];

	/* PC Card 16-Bit I/F Legacy-Mode Address Register */
	/* 4-15 */
	uint32_t legacy_base_address;

	/* System Control Register */
	/* 4-17 */
	uint8_t rimux[2];
	uint8_t keepclk;
	uint8_t excapower[2];
	uint8_t cb_dpar;
	uint8_t subsysrw;
	uint8_t pwrsavings[2];
	uint8_t interrogate[2];
	uint8_t delaydown;
	uint8_t delayup;
	uint8_t pwrstream;
	uint8_t socactive[2];
	uint8_t mrburstup;
	uint8_t mrburstdn;
	uint8_t reducezv[2];
	uint8_t vccprot[2];
	uint8_t cbrsvd[2];
	uint8_t smienb;
	uint8_t smistatus[2];
	uint8_t smiroute[2];
	uint8_t p2cclk;
	uint8_t intrtie;
	uint8_t ser_step;

	/* Multifunction Routing Register */
	/* 4-19 */
	uint8_t mfunc0;
	uint8_t mfunc1;
	uint8_t mfunc2;
	uint8_t mfunc3;
	uint8_t mfunc4;
	uint8_t mfunc5;
	uint8_t mfunc6;

	/* Retry Status Register */
	/* 4-21 */
	uint8_t texp_pci;
	uint8_t texp_cba;
	uint8_t texp_cbb;
	uint8_t cbretry;
	uint8_t pciretry;

	/* Card Control Register */
	/* 4-22 */
	uint8_t ifg[2];
	uint8_t spkrouten[2];
	uint8_t aud2mux[2];
	uint8_t port_sel[2];
	uint8_t zvenable[2];
	uint8_t rienb;

	/* Device Control Register */
	/* 4-23 */
	uint8_t intmode;
	uint8_t _3vcapable;
	uint8_t sktpwr_lock;

	/* Diagnostic Register */
	/* 4-24 */
	uint8_t stdzven[2];
	uint8_t csc[2];

	/* Power-Management Capabilities Register */
	/* 4-26 */
	uint8_t pme_support;

	/* Power-Management Control/Status Register */
	/* 4-27 */
	uint8_t pwr_state;
	uint8_t pme_en;
	uint8_t pmestat;

	/* Power-Management Control/Status Register Bridge Support */
	/* 4-28 */
	uint8_t b2_b3;
	uint8_t bpcc_en;

	/* General-Purpose Event Status Register */
	/* 4-29 */
	uint8_t gp0_sts;
	uint8_t gp1_sts;
	uint8_t gp2_sts;
	uint8_t gp3_sts;
	uint8_t gp4_sts;
	uint8_t vpp12_sts;
	uint8_t pwr_sts;
	uint8_t zv1_sts;
	uint8_t zv0_sts;

	/* General-Purpose Event Enable Register */
	/* 4-30 */
	uint8_t gp0_en;
	uint8_t gp1_en;
	uint8_t gp2_en;
	uint8_t gp3_en;
	uint8_t gp4_en;
	uint8_t vpp12_en;
	uint8_t pwr_en;
	uint8_t zv1_en;
	uint8_t zv0_en;

	/* General-Purpose Input Register */
	/* 4-31 */

	/* General-Purpose Output Register */
	/* 4-32 */
	uint8_t gpo0_data;
	uint8_t gpo1_data;
	uint8_t gpo2_data;
	uint8_t gpo3_data;
	uint8_t gpo4_data;

	/* Serial Registers - FIXME */
	uint8_t ser0;
	uint8_t ser1;
	uint8_t ser2;
	uint8_t ser3;

	/* Cardbus Registers */
	uint8_t pwrevent[2];
	uint8_t cd2event[2];
	uint8_t cd1event[2];
	uint8_t cstsevent[2];
	uint8_t pwrmask[2];
	uint8_t cd2mask[2];
	uint8_t cd1mask[2];
	uint8_t cstsmask[2];
	uint8_t vppctrl[2];
	uint8_t vccctrl[2];
	uint8_t stopclk[2];
	uint8_t zven[2];
	uint8_t clkctrl[2];
	uint8_t clkctrlen[2];
	uint8_t _3vcard[2];
	uint8_t _5vcard[2];
	uint8_t _16bitcard[2];
	uint8_t cbcard[2];
	uint8_t notacard[2];
	uint8_t pwrcycle[2];

	/* ExCA Registers */
	uint8_t current_exca_reg;
	uint8_t exca_regs[2][0x40];
	uint8_t page_regs[2][5];
};

static void
CHIP_(irq_update)(struct cpssp *cpssp, unsigned int func)
{
	int irq0;
	int irq1;

	/* Must obey cpssp->intr[f]! FIXME */
	/* Must obey interrupt routing! FIXME */
	/* Must obey csc[f]! FIXME */

	irq0 = (cpssp->cstsevent[0] & cpssp->cstsmask[0])
		| (cpssp->cd1event[0] & cpssp->cd1mask[0])
		| (cpssp->cd2event[0] & cpssp->cd2mask[0])
		| (cpssp->pwrevent[0] & cpssp->pwrmask[0])
		| cpssp->int_set[0];
	irq1 = (cpssp->cstsevent[1] & cpssp->cstsmask[1])
		| (cpssp->cd1event[1] & cpssp->cd1mask[1])
		| (cpssp->cd2event[1] & cpssp->cd2mask[1])
		| (cpssp->pwrevent[1] & cpssp->pwrmask[1])
		| cpssp->int_set[1];

	if (cpssp->intrtie) {
		sig_std_logic_or_set(cpssp->port_int[0], cpssp, irq0 | irq1);
		sig_std_logic_or_set(cpssp->port_int[1], cpssp, 0);
	} else {
		sig_std_logic_or_set(cpssp->port_int[0], cpssp, irq0);
		sig_std_logic_or_set(cpssp->port_int[1], cpssp, irq1);
	}
}

static void
CHIP_(interrogation)(struct cpssp *cpssp, int func)
{
	/* FIXME */
	/* update socket present state register */
	/* FIXME this forces a 3V cardbus card */
	cpssp->_3vcard[func] = 1;
	cpssp->cbcard[func] = 1;

	/* generate interrupt if not masked */
	CHIP_(irq_update)(cpssp, func);
}

static void
CHIP_(cd1_set)(struct cpssp *cpssp, unsigned int func, unsigned int val)
{
	cpssp->state_n_cd1[func] = val;
	cpssp->cd1event[func] = 1;
	CHIP_(interrogation)(cpssp, func);
}

static void
CHIP_(cd2_set)(struct cpssp *cpssp, unsigned int func, unsigned int val)
{
	cpssp->state_n_cd2[func] = val;
	cpssp->cd2event[func] = 1;
	CHIP_(interrogation)(cpssp, func);
}

static void
CHIP_(cvs1_set)(struct cpssp *cpssp, unsigned int func, unsigned int val)
{
	cpssp->state_cvs1[func] = val;
	CHIP_(interrogation)(cpssp, func);
}

static void
CHIP_(cvs2_set)(struct cpssp *cpssp, unsigned int func, unsigned int val)
{
	cpssp->state_cvs2[func] = val;
	CHIP_(interrogation)(cpssp, func);
}

static uint8_t
CHIP_(exca_mrb)(struct cpssp *cpssp, int func, int reg)
{
	uint8_t val;

	switch (reg) {
	case 0x00:
		/* ExCA Identification and Revision Register */
		/* 5-1 */
		val = 0x84; /* FIXME */
		break;
	case 0x01:
		/* ExCA Interface Status Register */
		/* 5-2 */
		val = 0; /* Reserved */
		/* BVDSTAT - FIXME */
		val |= (! cpssp->state_n_cd1[func]) << 2; /* CDETECT1 */
		val |= (! cpssp->state_n_cd2[func]) << 3; /* CDETECT2 */
		/* CARDWP - FIXME */
		val |= 1 << 5; /* READY - FIXME */
		val |= 1 << 6; /* CARDPWR - FIXME */
		break;
	default:
		val = cpssp->exca_regs[func][reg];
		break;
	}

	if (reg == 0x04
	 && (cpssp->exca_regs[func][0x1E] & 0x2) == 0) {
		/* this register can be reset when read */
		cpssp->exca_regs[func][reg] = 0x00;
	}

	return val;
}

static void
CHIP_(exca_mwb)(struct cpssp *cpssp, int func, int reg, uint8_t val)
{
	uint8_t tmp;

	assert(reg < 0x40);
	
	switch (reg) {
	case 0x00:
		tmp = val & 0x3F;
		break;
	case 0x01:
		/* read only */
		return;
	case 0x02:
		tmp = val & 0xB3;
		break;
	case 0x04:
		tmp = val & 0x0F;
		return;
	
	case 0x06:
		tmp = val & 0xDF;
		break;
	case 0x13:
	case 0x1B:
	case 0x23:
	case 0x2B:
	case 0x33:
		tmp = val & 0xAF;
		break;
	case 0x16:
		tmp = val & 0x32;
		break;
	case 0x1E:
		tmp = val & 0x1F;
		break;
	case 0x17:
	case 0x1F:
	case 0x26 ... 0x27:
	case 0x2E ... 0x2F:
	case 0x3A ... 0x3F:
		/* reserved */
		return;
	default:
		/* anything else is completely writable */
		tmp = val;
		break;
	}
	if (reg == 0x1E) {
		/* these registers are actually the same on the chip */
		cpssp->exca_regs[0][reg] = tmp;
		cpssp->exca_regs[1][reg] = tmp;
	} else if (reg == 0x04) {
		/* this can be reset by writing a 1 */
		cpssp->exca_regs[func][reg] &= ~tmp;
	} else {
		cpssp->exca_regs[func][reg] = tmp;
	}
}

static void
CHIP_(legacy_read)(
	struct cpssp *cpssp,
	uint16_t port,
	unsigned int bs,
	uint32_t *valp
)
{
	*valp = 0x00000000;
	if ((bs >> 0) & 1) {
		*valp |= cpssp->current_exca_reg << 0;
	}
	if ((bs >> 1) & 1) {
		*valp |= CHIP_(exca_mrb)(cpssp,
				(cpssp->current_exca_reg >> 6) & 1,
				(cpssp->current_exca_reg >> 0) & 0x3f) << 8;
	}
	if ((bs >> 2) & 1) {
		/* Nothing there... */
	}
	if ((bs >> 3) & 1) {
		/* Nothing there... */
	}

	if (DEBUG) fprintf(stderr, "%s: port=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, port, bs, *valp);
}

static void
CHIP_(legacy_write)(
	struct cpssp *cpssp,
	uint16_t port,
	unsigned int bs,
	uint32_t val
)
{
	if ((bs >> 0) & 1) {
		cpssp->current_exca_reg = (val >> 0) & 0xff;
	}
	if ((bs >> 1) & 1) {
		CHIP_(exca_mwb)(cpssp,
				(cpssp->current_exca_reg >> 6) & 1,
				(cpssp->current_exca_reg >> 0) & 0x3f,
				(val >> 8) & 0xff);
	}
	if ((bs >> 2) & 1) {
		/* Nothing there... */
	}
	if ((bs >> 3) & 1) {
		/* Nothing there... */
	}

	if (DEBUG) fprintf(stderr, "%s: port=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, port, bs, val);
}

static void
CHIP_(cardbus_read)(
	struct cpssp *cpssp,
	unsigned int func,
	uint32_t reg,
	unsigned int bs,
	uint32_t *valp
)
{
	switch (reg) {
	case 0x00:
		/* Socket Event Register */
		/* 6-2 */
		*valp = 0;
		if ((bs >> 0) & 1) {
			*valp |= cpssp->cstsevent[func] << 0;
			*valp |= cpssp->cd1event[func] << 1;
			*valp |= cpssp->cd2event[func] << 2;
			*valp |= cpssp->pwrevent[func] << 3;
		}
		if ((bs >> 1) & 1) {
			*valp |= 0 << 8; /* Reserved */
		}
		if ((bs >> 2) & 1) {
			*valp |= 0 << 16; /* Reserved */
		}
		if ((bs >> 3) & 1) {
			*valp |= 0 << 24; /* Reserved */
		}
		break;

	case 0x04:
		/* Socket Mask Register */
		/* 6-3 */
		*valp = 0;
		if ((bs >> 0) & 1) {
			*valp |= cpssp->cstsmask[func] << 0;
			*valp |= cpssp->cd1mask[func] << 1;
			*valp |= cpssp->cd2mask[func] << 2;
			*valp |= cpssp->pwrmask[func] << 3;
		}
		if ((bs >> 1) & 1) {
			*valp |= 0 << 8; /* Reserved */
		}
		if ((bs >> 2) & 1) {
			*valp |= 0 << 16; /* Reserved */
		}
		if ((bs >> 3) & 1) {
			*valp |= 0 << 24; /* Reserved */
		}
		break;

	case 0x08:
		/* Socket Present-State Register */
		/* 6-4 */
		*valp = 0;
		if ((bs >> 0) & 1) {
			/* CARDSTS - FIXME */
#if 0
			*valp |= cpssp->state_n_cd1[func] << 1;
			*valp |= cpssp->state_n_cd2[func] << 2;
#endif
			*valp |= cpssp->pwrcycle[func] << 3;
			*valp |= cpssp->_16bitcard[func] << 4;
			*valp |= cpssp->cbcard[func] << 5;
			/* IREQCINT - FIXME */
			*valp |= cpssp->notacard[func] << 7;
		}
		if ((bs >> 1) & 1) {
			/* DATALOST - FIXME */
			/* BADVCCREQ - FIXME */
			*valp |= cpssp->_5vcard[func] << 10;
			*valp |= cpssp->_3vcard[func] << 11;
			/* XVCARD - FIXME */
			/* YVCARD - FIXME */
			/* Bit 14-15: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Bit 24-26: Reserved */
			/* ZVSUPPORT - FIXME */
			/* 5VSOCKET - FIXME */
			/* 3VSOCKET - FIXME */
			/* XVSOCKET - FIXME */
			/* YVSOCKET - FIXME */
		}
		break;

	case 0x0c:
		/* Socket Force Event Register */
		*valp = 0x00000000; /* FIXME */
		break;

	case 0x10:
		/* Socket Control Register */
		/* 6-8 */
		*valp = 0;
		if ((bs >> 0) & 1) {
			*valp |= cpssp->vppctrl[func] << 0;
			*valp |= 0 << 3; /* Reserved */
			*valp |= cpssp->vccctrl[func] << 4;
			*valp |= cpssp->stopclk[func] << 7;
		}
		if ((bs >> 1) & 1) {
			*valp |= 0 << 8; /* Reserved */
			*valp |= cpssp->zven[func] << 9;
			/* STDZVREG - FIXME */
			/* ZV_ACTIVITY - FIXME */
			*valp |= 0 << 12; /* Reserved */
		}
		if ((bs >> 2) & 1) {
			*valp |= 0 << 16; /* Reserved */
		}
		if ((bs >> 3) & 1) {
			*valp |= 0 << 24; /* Reserved */
		}
		break;

	case 0x20:
		/* Socket Power-Management */
		/* 6-9 */
		if ((bs >> 0) & 1) {
			*valp |= cpssp->clkctrl[func] << 0;
			/* Bit 1-7: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			*valp |= cpssp->clkctrlen[func] << 16;
			/* Bit 17-23: Reserved */
		}
		if ((bs >> 3) & 1) {
			/* SKTMODE - FIXME */
			/* SKTACCES - FIXME */
			/* Bit 26-31: Reserved */
		}
		break;

	default:
		/* Reserved */
		*valp = 0;
		break;
	}

	if (DEBUG) fprintf(stderr, "%s: func=%u, reg=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, func, reg, bs, *valp);
}

static void
CHIP_(cardbus_write)(
	struct cpssp *cpssp,
	unsigned int func,
	uint32_t reg,
	unsigned int bs,
	uint32_t val
)
{
	switch (reg) {
	case 0x00:
		/* Socket Event Register */
		/* 6-2 */
		if ((bs >> 0) & 1) {
			cpssp->pwrevent[func] &= ~((val >> 3) & 1);
			cpssp->cd2event[func] &= ~((val >> 2) & 1);
			cpssp->cd1event[func] &= ~((val >> 1) & 1);
			cpssp->cstsevent[func] &= ~((val >> 0) & 1);
			CHIP_(irq_update)(cpssp, func);
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x04:
		/* Socket Mask Register */
		/* 6-3 */
		if ((bs >> 0) & 1) {
			cpssp->pwrmask[func] = (val >> 3) & 1;
			cpssp->cd2mask[func] = (val >> 2) & 1;
			cpssp->cd1mask[func] = (val >> 1) & 1;
			cpssp->cstsmask[func] = (val >> 0) & 1;
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x08:
		/* Socket Present State */
		/* 6-4 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}
		if ((bs >> 1) & 1) {
			/* Read-only */
		}
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x0C:
		/* Socket Force Event */
		/* 6-6 */
		if ((bs >> 0) & 1) {
			cpssp->cstsevent[func] = (val >> 0) & 1;
			cpssp->cd1event[func] = (val >> 1) & 1;
			cpssp->cd2event[func] = (val >> 2) & 1;
			cpssp->pwrevent[func] = (val >> 3) & 1;
			CHIP_(irq_update)(cpssp, func);

			cpssp->_16bitcard[func] |= (val >> 4) & 1;
			cpssp->cbcard[func] |= (val >> 5) & 1;
			cpssp->notacard[func] |= (val >> 7) & 1;
		}
		if ((bs >> 1) & 1) {
			/* FIXME */
			if (val & (1 << 14)) {
				/* Force card re-interrogation */
				CHIP_(interrogation)(cpssp, func);
			}
		}
		if ((bs >> 2) & 1) {
			/* FIXME */
		}
		if ((bs >> 3) & 1) {
			/* FIXME */
		}
		break;

	case 0x10:
		/* Socket Control Register */
		/* 6-8 */
		if ((bs >> 0) & 1) {
			int power_old;
			int power_new;

			power_old = (cpssp->vppctrl[func] != 0)
				| (cpssp->vccctrl[func] != 0);

			cpssp->vppctrl[func] = (val >> 0) & 0x7;
			/* Bit 3: Reserved */
			cpssp->vccctrl[func] = (val >> 4) & 0x7;
			cpssp->stopclk[func] = (val >> 7) & 1;

			power_new = (cpssp->vppctrl[func] != 0)
				| (cpssp->vccctrl[func] != 0);

			cpssp->pwrevent[func] = power_old ^ power_new;
			cpssp->pwrcycle[func] = power_new;
			cpssp->powered[func] = power_new;
			sig_std_logic_or_set(cpssp->port_cb_power[func],
					cpssp, power_new);
			CHIP_(irq_update)(cpssp, func);
		}
		if ((bs >> 1) & 1) {
			/* Bit 8: Reserved */
			cpssp->zven[func] = (val >> 9) & 1;
			/* Bit 10: Read-only */
			/* Bit 11: Read-only */
			/* Bit 12-15: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x20:
		/* Socket Power-Management */
		/* 6-9 */
		if ((bs >> 0) & 1) {
			cpssp->clkctrl[func] = (val >> 0) & 1;
			/* Bit 1-7: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			cpssp->clkctrlen[func] = (val >> 16) & 1;
			/* Bit 17-23: Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Bit 24: SKTMODE - Read-only */
			/* Bit 25: SKTMODE - Read-only */
			/* Bit 26-31: Reserved */
		}
		break;

	default:
		/* Reserved */
		break;
	}

	if (DEBUG) fprintf(stderr, "%s: func=%u, reg=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, func, reg, bs, val);
}

static void
CHIP_(exca_read)(
	struct cpssp *cpssp,
	unsigned int func,
	uint32_t reg,
	unsigned int bs,
	uint32_t *valp
)
{
	if ((bs >> 0) & 1) {
		*valp &= ~(0xff << 0);
		*valp |= CHIP_(exca_mrb)(cpssp, func, reg + 0) << 0;
	}
	if ((bs >> 1) & 1) {
		*valp &= ~(0xff << 8);
		*valp |= CHIP_(exca_mrb)(cpssp, func, reg + 1) << 8;
	}
	if ((bs >> 2) & 1) {
		*valp &= ~(0xff << 16);
		*valp |= CHIP_(exca_mrb)(cpssp, func, reg + 2) << 16;
	}
	if ((bs >> 3) & 1) {
		*valp &= ~(0xff << 24);
		*valp |= CHIP_(exca_mrb)(cpssp, func, reg + 3) << 24;
	}

	if (DEBUG) fprintf(stderr, "%s: func=%u, reg=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, func, reg, bs, *valp);
}

static void
CHIP_(exca_write)(
	struct cpssp *cpssp,
	unsigned int func,
	uint32_t reg,
	unsigned int bs,
	uint32_t val
)
{
	if ((bs >> 0) & 1) {
		CHIP_(exca_mwb)(cpssp, func, reg + 0, (val >> 0) & 0xff);
	}
	if ((bs >> 1) & 1) {
		CHIP_(exca_mwb)(cpssp, func, reg + 1, (val >> 8) & 0xff);
	}
	if ((bs >> 2) & 1) {
		CHIP_(exca_mwb)(cpssp, func, reg + 2, (val >> 16) & 0xff);
	}
	if ((bs >> 3) & 1) {
		CHIP_(exca_mwb)(cpssp, func, reg + 3, (val >> 24) & 0xff);
	}

	if (DEBUG) fprintf(stderr, "%s: func=%u, reg=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, func, reg, bs, val);
}

static void
CHIP_(page_read)(
	struct cpssp *cpssp,
	unsigned int func,
	uint32_t reg,
	unsigned int bs,
	uint32_t *valp
)
{
	if ((bs >> 0) & 1) {
		*valp &= ~(0xff << 0);
		*valp |= cpssp->page_regs[func][reg + 0] << 0;
	}
	if (reg == 0) {
		if ((bs >> 1) & 1) {
			*valp &= ~(0xff << 8);
			*valp |= cpssp->page_regs[func][reg + 1] << 8;
		}
		if ((bs >> 2) & 1) {
			*valp &= ~(0xff << 16);
			*valp |= cpssp->page_regs[func][reg + 2] << 16;
		}
		if ((bs >> 3) & 1) {
			*valp &= ~(0xff << 24);
			*valp |= cpssp->page_regs[func][reg + 3] << 24;
		}
	}

	if (DEBUG) fprintf(stderr, "%s: func=%u, reg=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, func, reg, bs, *valp);
}

static void
CHIP_(page_write)(
	struct cpssp *cpssp,
	unsigned int func,
	uint32_t reg,
	unsigned int bs,
	uint32_t val
)
{
	if ((bs >> 0) & 1) {
		cpssp->page_regs[func][reg + 0] = (val >> 0) & 0xff;
	}
	if (reg == 0) {
		if ((bs >> 1) & 1) {
			cpssp->page_regs[func][reg + 1] = (val >> 8) & 0xff;
		}
		if ((bs >> 2) & 1) {
			cpssp->page_regs[func][reg + 2] = (val >> 16) & 0xff;
		}
		if ((bs >> 3) & 1) {
			cpssp->page_regs[func][reg + 3] = (val >> 24) & 0xff;
		}
	}

	if (DEBUG) fprintf(stderr, "%s: func=%u, reg=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, func, reg, bs, val);
}

static void
CHIP_(chip_read)(
	struct cpssp *cpssp,
	unsigned int func,
	uint32_t *valp,
	uint32_t addr,
	unsigned int bs
)
{
	unsigned int offset;

	offset = addr & 0xfff;
	
	if (offset < 0x800) {
		/* CardBus Registers */
		CHIP_(cardbus_read)(cpssp, func, offset & 0x7ff, bs, valp);

	} else if (0x800 <= offset && offset < 0x840) {
		/* ExCA Compatibility Registers */
		CHIP_(exca_read)(cpssp, func, offset & 0x3f, bs, valp);

	} else if (0x840 <= offset && offset < 0x848) {
		/* Memory Window Page Registers */
		CHIP_(page_read)(cpssp, func, offset & 0x7, bs, valp);

	} else {
		/* Nothing there... */
	}
}

static void
CHIP_(chip_write)(
	struct cpssp *cpssp,
	unsigned int func,
	uint32_t val,
	uint32_t addr,
	unsigned int bs
)
{
	unsigned int offset;
	
	assert((addr & 3) == 0);
	offset = addr & 0xfff;
	
	if (offset < 0x800) {
		/* CardBus Registers */
		CHIP_(cardbus_write)(cpssp, func, offset & 0x7ff, bs, val);

	} else if (0x800 <= offset && offset < 0x840) {
		/* ExCA Compatibility Registers */
		CHIP_(exca_write)(cpssp, func, offset & 0x3f, bs, val);

	} else if (0x840 <= offset && offset < 0x848) {
		/* Memory Window Page Registers */
		CHIP_(page_write)(cpssp, func, offset & 0x7, bs, val);

	} else {
		/* Nothing there... */
	}
}

static int
CHIP_(chip_c0r)(
	struct cpssp *cpssp,
	int f,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	*valp = 0x00000000;

	switch (addr) {
	case 0x00:
		/* Vendor ID Register */
		*valp |= 0x104c << 0;

		/* Device ID Register */
		*valp |= 0xac55 << 16;
		break;

	case 0x04:
		/* Command Register */
		*valp |= cpssp->io_en[f] << 0;
		*valp |= cpssp->mem_en[f] << 1;
		*valp |= cpssp->mast_en[f] << 2;
		*valp |= 0 << 3; /* Special Cycles */
		*valp |= 0 << 4; /* Memory Write-and-Invalidate */
		*valp |= cpssp->vga_en[f] << 5;
		*valp |= cpssp->perr_en[f] << 6;
		*valp |= 0 << 7; /* Stepping Control */
		*valp |= cpssp->serr_en[f] << 6;
		*valp |= 0 << 9; /* Fast Back-to-Back */
		*valp |= 0 << 10; /* Reserved */
		*valp |= 0 << 11; /* Reserved */
		*valp |= 0 << 12; /* Reserved */
		*valp |= 0 << 13; /* Reserved */
		*valp |= 0 << 14; /* Reserved */
		*valp |= 0 << 15; /* Reserved */

		/* Status Register */
		*valp |= 0 << 16; /* Reserved */
		*valp |= 0 << 17; /* Reserved */
		*valp |= 0 << 18; /* Reserved */
		*valp |= 0 << 19; /* Reserved */
		*valp |= 1 << 20; /* Capabilities List */
		*valp |= 0 << 21; /* 66-MHz Capable */
		*valp |= 0 << 22; /* User-defined Feature Support */
		*valp |= 0 << 23; /* Fast Back-to-Back Capable */
		*valp |= 0 << 24; /* Data Parity Error Detected - FIXME */
		*valp |= 0b01 << 25; /* DEVSEL Timing */
		*valp |= 0 << 27; /* Signaled Target Abort - FIXME */
		*valp |= 0 << 28; /* Received Target Abort - FIXME */
		*valp |= 0 << 29; /* Received Master Abort - FIXME */
		*valp |= 0 << 30; /* Signaled System Error - FIXME */
		*valp |= 0 << 31; /* Detected Parity Error - FIXME */
		break;

	case 0x08:
		/* Revision ID Register */
		*valp |= 0x01 << 0;

		/* Class Code Register */
		*valp |= 0x060700 << 8; /* Bridge/CardBus */
		break;

	case 0x0c:
		/* Cache Line Size Register */
		*valp |= cpssp->cache_line_size[f] << 0;

		/* Latency Timer Register */
		*valp |= cpssp->latency_timer[f] << 8;

		/* Header Type Register */
		*valp |= 0x82 << 16;

		/* BIST Register */
		*valp |= 0x00 << 24;
		break;

	case 0x10:
		/* CardBus Socket/ExCA Base-Address Register */
		/* 4-7 */
		*valp |= cpssp->base_address[f] << 0;
		break;

	case 0x14:
		/* Capability Pointer Register */
		/* 4-7 */
		*valp |= 0xa0 << 0;

		/* Reserved */

		/* Secondary Status Register */
		/* 4-8 */
		*valp |= 0b00000 << 16; /* Reserved */
		*valp |= 0 << 21; /* CB66MHZ */
		*valp |= 0 << 22; /* CB_UDF */
		*valp |= 0 << 23; /* CBFBB_CAP */
		*valp |= 0 << 24; /* CB_DPAR - FIXME */
		*valp |= 0b01 << 25; /* CB_SPEED */
		*valp |= 0 << 27; /* SIG_CBTA - FIXME */
		*valp |= 0 << 28; /* REC_CBTA - FIXME */
		*valp |= 0 << 29; /* CBMABORT - FIXME */
		*valp |= 0 << 30; /* CBSERR - FIXME */
		*valp |= 0 << 31; /* CBPARITY - FIXME */
		break;

	case 0x18:
		/* PCI Bus Number Register */
		*valp |= cpssp->pci_bus_number[f] << 0;

		/* CardBus Bus Number Register */
		*valp |= cpssp->cardbus_bus_number[f] << 8;

		/* Subordinate Bus Number Register */
		*valp |= cpssp->subordinate_bus_number[f] << 16;

		/* Cardbus Latency Timer Register */
		*valp |= cpssp->cardbus_latency_timer[f] << 24;
		break;

	case 0x1c:
		/* Memory Base 0 Register */
		/* 4-10 */
		*valp |= cpssp->mem_base[f][0] << 0;
		break;

	case 0x20:
		/* Memory Limit 0 Register */
		/* 4-11 */
		*valp |= cpssp->mem_limit[f][0] << 0;
		break;

	case 0x24:
		/* Memory Base 1 Register */
		/* 4-10 */
		*valp |= cpssp->mem_base[f][1] << 0;
		break;

	case 0x28:
		/* Memory Limit 1 Register */
		/* 4-11 */
		*valp |= cpssp->mem_limit[f][1] << 0;
		break;

	case 0x2c:
		/* I/O Base 0 Register */
		/* 4-11 */
		*valp |= cpssp->io_base[f][0] << 0;
		break;

	case 0x30:
		/* I/O Limit 0 Register */
		/* 4-12 */
		*valp |= cpssp->io_limit[f][0] << 0;
		*valp |= 0x0000 << 16;
		break;

	case 0x34:
		/* I/O Base 1 Register */
		/* 4-11 */
		*valp |= cpssp->io_base[f][1] << 0;
		break;

	case 0x38:
		/* I/O Limit 1 Register */
		/* 4-12 */
		*valp |= cpssp->io_limit[f][1] << 0;
		*valp |= 0x0000 << 16;
		break;

	case 0x3c:
		/* Interrupt Line Register */
		/* 4-12 */
		*valp |= cpssp->interrupt_line[f] << 0;

		/* Interrupt Pin Register */
		/* 4-13 */
		switch (f) {
		case 0:
			*valp |= 0x01 << 8;
			break;
		case 1:
			if (cpssp->intrtie) {
				*valp |= 0x01 << 8;
			} else {
				*valp |= 0x02 << 8;
			}
			break;
		default: assert(0); /* Cannot happen. */
		}

		/* Bridge Control Register */
		/* 4-14 */
		*valp |= cpssp->cperren << 16;
		*valp |= cpssp->cserren << 17;
		*valp |= cpssp->isaen[f] << 18;
		*valp |= cpssp->vgaen[f] << 19;
		*valp |= 0 << 20; /* Reserved */
		*valp |= cpssp->mabtmode << 21;
		*valp |= cpssp->crst[f] << 22;
		*valp |= cpssp->intr[f] << 23;
		*valp |= cpssp->prefetch0[f] << 24;
		*valp |= cpssp->prefetch1[f] << 25;
		*valp |= cpssp->posten[f] << 26;
		*valp |= 0b00000 << 27; /* Reserved */
		break;

	case 0x40:
		/* Subsystem Vendor ID Register */
		/* 4-15 */
		*valp |= cpssp->subsystem_vendor_id[f] << 0;

		/* Subsystem Device ID Register */
		/* 4-15 */
		*valp |= cpssp->subsystem_device_id[f] << 16;
		break;

	case 0x44:
		/* PC Card 16-Bit I/F Legacy-Mode Base Address Register */
		/* 4-15 */
		*valp |= 1 << 0; /* Hardwired '1' */
		*valp |= cpssp->legacy_base_address << 0;
		break;

	case 0x48 ... 0x7c:
		/* Reserved */
		break;

	case 0x80:
		/* System Control Register */
		/* 4-17 */
		*valp |= cpssp->rimux[f] << 0;
		*valp |= cpssp->keepclk << 1;
		*valp |= cpssp->excapower[f] << 2;
		*valp |= 0b0 << 3; /* Reserved */
		*valp |= cpssp->cb_dpar << 4;
		*valp |= cpssp->subsysrw << 5;
		*valp |= cpssp->pwrsavings[f] << 6;
		*valp |= 0b0 << 7; /* Reserved */
		*valp |= cpssp->interrogate[f] << 8;
		*valp |= cpssp->delaydown << 9;
		*valp |= cpssp->delayup << 10;
		*valp |= cpssp->pwrstream << 11;
		*valp |= 0b1 << 12; /* Reserved */
		*valp |= cpssp->socactive[f] << 13;
		*valp |= cpssp->mrburstup << 14;
		*valp |= cpssp->mrburstdn << 15;
		*valp |= 0b0100 << 16; /* Reserved */
		*valp |= cpssp->reducezv[f] << 20;
		*valp |= cpssp->vccprot[f] << 21;
		*valp |= cpssp->cbrsvd[f] << 22;
		*valp |= 0b0 << 23; /* Reserved */
		*valp |= cpssp->smienb << 24;
		*valp |= cpssp->smistatus[f] << 25;
		*valp |= cpssp->smiroute[f] << 26;
		*valp |= cpssp->p2cclk << 27;
		*valp |= 0b0 << 28; /* Reserved */
		*valp |= cpssp->intrtie << 29;
		*valp |= cpssp->ser_step << 30;
		break;

	case 0x84 ... 0x88:
		/* Reserved */
		break;

	case 0x8c:
		/* Multifunction Routing Register */
		/* 4-19 */
		*valp |= cpssp->mfunc0 << 0;
		*valp |= cpssp->mfunc1 << 4;
		*valp |= cpssp->mfunc2 << 8;
		*valp |= cpssp->mfunc3 << 12;
		*valp |= cpssp->mfunc4 << 16;
		*valp |= cpssp->mfunc5 << 20;
		*valp |= cpssp->mfunc6 << 24;
		*valp |= 0b0000 << 28; /* Reserved */
		break;

	case 0x90:
		/* Retry Status Register */
		/* 4-21 */
		*valp |= 0b0 << 0; /* Reserved */
		*valp |= cpssp->texp_pci << 1;
		*valp |= 0b0 << 2; /* Reserved */
		*valp |= cpssp->texp_cba << 3;
		*valp |= 0b0 << 4; /* Reserved */
		*valp |= cpssp->texp_cbb << 5;
		*valp |= cpssp->cbretry << 6;
		*valp |= cpssp->pciretry << 7;

		/* Card Control Register */
		/* 4-22 */
		*valp |= cpssp->ifg[f] << 8;
		*valp |= cpssp->spkrouten[f] << 9;
		*valp |= cpssp->aud2mux[f] << 10;
		*valp |= 0b00 << 11; /* Reserved */
		*valp |= cpssp->port_sel[f] << 13;
		*valp |= cpssp->zvenable[f] << 14;
		*valp |= cpssp->rienb << 15;

		/* Device Control Register */
		/* 4-23 */
		*valp |= 0b0 << 16; /* Reserved */
		*valp |= cpssp->intmode << 17;
		*valp |= 0b0 << 19; /* Test */
		*valp |= 0b0 << 20; /* Reserved */
		*valp |= 0b1 << 21; /* Test */
		*valp |= cpssp->_3vcapable << 22;
		*valp |= cpssp->sktpwr_lock << 23;

		/* Diagnostic Register */
		/* 4-24 */
		*valp |= cpssp->stdzven[f] << 24;
		*valp |= 0b0 << 25; /* DIAG1 - FIXME */
		*valp |= 0b0 << 26; /* DIAG2 - FIXME */
		*valp |= 0b0 << 27; /* DIAG3 - FIXME */
		*valp |= 0b0 << 28; /* DIAG4 - FIXME */
		*valp |= cpssp->csc[f] << 29;
		*valp |= 0b1 << 30; /* Reserved */
		*valp |= 0b0 << 31; /* TRUE_VAL - FIXME */
		break;

	case 0x94 ... 0x9c:
		/* Reserved */
		break;

	case 0xa0:
		/* Capability ID Register */
		/* 4-25 */
		*valp |= 0x01 << 0; /* Read-only */

		/* Next-Item Pointer Register */
		/* 4-25 */
		*valp |= 0x00 << 8; /* Read-only */

		/* Power-Management Capabilities Register */
		/* 4-26 */
		*valp |= 0b010 << 16; /* VERSION */
		*valp |= 0b0 << 19; /* PMECLK */
		*valp |= 0b1 << 20; /* AUX_PWR */
		*valp |= 0b1 << 21; /* DSI */
		*valp |= 0b000 << 22; /* Reserved */
		*valp |= 0b1 << 25; /* D1_SUPPORT */
		*valp |= 0b1 << 26; /* D2_SUPPORT */
		*valp |= 0b1111 << 27; /* PME_SUPPORT */
		*valp |= cpssp->pme_support << 31;
		break;

	case 0xa4:
		/* Power-Management Control/Status Register */
		/* 4-27 */
		*valp |= cpssp->pwr_state << 0;
		*valp |= 0b000000 << 2; /* Reserved */
		*valp |= cpssp->pme_en << 8;
		*valp |= 0b0000 << 9; /* DATASEL */
		*valp |= 0b00 << 13; /* DATASCALE */
		*valp |= cpssp->pmestat << 15;

		/* Power-Management Control/Status Register Bridge Support */
		/* 4-28 */
		*valp |= 0b000000 << 16; /* Reserved */
		*valp |= cpssp->b2_b3 << 22;
		*valp |= cpssp->bpcc_en << 23;

		/* Power-Management Data Register */
		/* 4-28 */
		*valp |= 0x00 << 24; /* Not used. */
		break;

	case 0xa8:
		/* General-Purpose Event Status Register */
		/* 4-29 */
		*valp |= cpssp->gp0_sts << 0;
		*valp |= cpssp->gp1_sts << 1;
		*valp |= cpssp->gp2_sts << 2;
		*valp |= cpssp->gp3_sts << 3;
		*valp |= cpssp->gp4_sts << 4;
		*valp |= 0b000 << 5; /* Reserved */
		*valp |= cpssp->vpp12_sts << 8;
		*valp |= 0b00 << 9; /* Reserved */
		*valp |= cpssp->pwr_sts << 11;
		*valp |= 0b00 << 12; /* Reserved */
		*valp |= cpssp->zv1_sts << 14;
		*valp |= cpssp->zv0_sts << 15;

		/* General-Purpose Event Enable Register */
		/* 4-30 */
		*valp |= cpssp->gp0_en << 16;
		*valp |= cpssp->gp1_en << 17;
		*valp |= cpssp->gp2_en << 18;
		*valp |= cpssp->gp3_en << 19;
		*valp |= cpssp->gp4_en << 20;
		*valp |= 0b000 << 21; /* Reserved */
		*valp |= cpssp->vpp12_en << 24;
		*valp |= 0b00 << 9; /* Reserved */
		*valp |= cpssp->pwr_en << 27;
		*valp |= 0b00 << 12; /* Reserved */
		*valp |= cpssp->zv1_en << 30;
		*valp |= cpssp->zv0_en << 31;
		break;

	case 0xac:
		/* General-Purpose Input Register */
		/* 4-31 */
		*valp |= 0b0 << 0; /* GPI0_DATA - FIXME */
		*valp |= 0b0 << 1; /* GPI1_DATA - FIXME */
		*valp |= 0b0 << 2; /* GPI2_DATA - FIXME */
		*valp |= 0b0 << 3; /* GPI3_DATA - FIXME */
		*valp |= 0b0 << 4; /* GPI4_DATA - FIXME */
		*valp |= 0b00000000000 << 5; /* Reserved */

		/* General-Purpose Output Register */
		/* 4-32 */
		*valp |= cpssp->gpo0_data << 16;
		*valp |= cpssp->gpo1_data << 17;
		*valp |= cpssp->gpo2_data << 18;
		*valp |= cpssp->gpo3_data << 19;
		*valp |= cpssp->gpo4_data << 20;
		*valp |= 0b00000000000 << 21; /* Reserved */
		break;

	case 0xb0:
		/* Serial Registers - FIXME */
		*valp |= cpssp->ser0 << 0;
		*valp |= cpssp->ser1 << 8;
		*valp |= cpssp->ser2 << 16;
		*valp |= cpssp->ser3 << 24;
		break;

	case 0xb4 ... 0xfc:
		/* Reserved */
		break;

	default: assert(0); /* Cannot happen. */
	}

	if (DEBUG) fprintf(stderr, "%s: addr=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, addr, bs, *valp);

	return 0;
}

static int
CHIP_(chip_c0w)(
	struct cpssp *cpssp,
	int f,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	switch (addr) {
	case 0x00:
		/* Vendor ID Register */
		/* Read-only */

		/* Device ID Register */
		/* Read-only */
		break;

	case 0x04:
		/* Command Register */
		if ((bs >> 0) & 1) {
			cpssp->io_en[f] = (val >> 0) & 1;
			cpssp->mem_en[f] = (val >> 1) & 1;
			cpssp->mast_en[f] = (val >> 2) & 1;
			/* Bit 3: SPEACIAL - Hardwired to '0' */
			/* Bit 4: MWI_EN - Hardwired to '0' */
			cpssp->vga_en[f] = (val >> 5) & 1;
			cpssp->perr_en[f] = (val >> 6) & 1;
			/* Bit 7: STEP_EN - Hardwired to '0' */
		}
		if ((bs >> 1) & 1) {
			cpssp->serr_en[f] = (val >> 8) & 1;
			/* Bit 9: FBB_EN - Hardwired to '0' */
			/* Bit 10-15: Reserved */
		}

		/* Status Register */
		if ((bs >> 2) & 1) {
			/* Bit 0-3: Reserved */
			/* Bit 4: CAP_LIST - Hardwired to '0' */
			/* Bit 5: 66MHZ - Hardwired to '0' */
			/* Bit 6: UDF - Hardwired to '0' */
			/* Bit 7: FBB_CAP - Hardwired to '0' */
		}
		if ((bs >> 3) & 1) {
			/* Bit 8: DATA_PAR - FIXME */
			/* Bit 9-10: PCI_SPEED - Hardwired to '01' */
			/* Bit 11: TABT_SIG - FIXME */
			/* Bit 12: TABT_REC - FIXME */
			/* Bit 13: MABORT - FIXME */
			/* Bit 14: SYS_ERR - FIXME */
			/* Bit 15: PAR_ERR - FIXME */
		}
		break;

	case 0x08:
		/* Revision ID Register */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		/* Class Code Register */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x0c:
		/* Cache Line Size Register */
		if ((bs >> 0) & 1) {
			cpssp->cache_line_size[f] = (val >> 0) & 0xff;
		}

		/* Latency Timer Register */
		if ((bs >> 1) & 1) {
			cpssp->latency_timer[f] = (val >> 8) & 0xff;
		}

		/* Header Type Register */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}

		/* BIST Register */
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x10: {
		/* CardBus Socket/ExCA Base-Address Register */
		/* 4-7 */
		uint32_t oaddr;
		uint32_t naddr;

		oaddr = cpssp->base_address[f];
		if ((bs >> 0) & 1) {
			cpssp->base_address[f] &= ~(0xff << 0);
			cpssp->base_address[f] |= val & (0x00 << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->base_address[f] &= ~(0xff << 8);
			cpssp->base_address[f] |= val & (0xf0 << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->base_address[f] &= ~(0xff << 16);
			cpssp->base_address[f] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->base_address[f] &= ~(0xff << 24);
			cpssp->base_address[f] |= val & (0xff << 24);
		}
		naddr = cpssp->base_address[f];
		if (oaddr != naddr) {
			/* Remap old and new regions */
			sig_pci_bus_unmap(cpssp->port_pci_bus, cpssp,
					oaddr, 0x1000);
			sig_pci_bus_unmap(cpssp->port_pci_bus, cpssp,
					naddr, 0x1000);
		}
		break;
	    }
	case 0x14:
		/* Capability Pointer Register */
		/* 4-7 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}
		
		/* Reserved */
		if ((bs >> 1) & 1) {
			/* Reserved */
		}

		/* Secondary Status Register */
		/* 4-8 */
		if ((bs >> 2) & 1) {
			/* Bit 0-4: Reserved */
			/* CB66MHZ: Read-only */
			/* CB_UDF: Read-only */
			/* CBFBB_CAP: Read-only */
		}
		if ((bs >> 3) & 1) {
			/* CB_DPAR: FIXME */
			/* CB_SPEED: Read-only */
			/* SIG_CBTA: FIXME */
			/* REC_CBTA: FIXME */
			/* CBMABORT: FIXME */
			/* CBSERR: FIXME */
			/* CBPARITY: FIXME */
		}
		break;

	case 0x18:
		/* PCI Bus Number Register */
		if ((bs >> 0) & 1) {
			cpssp->pci_bus_number[f] = (val >> 0) & 0xff;
		}

		/* CardBus Bus Number Register */
		if ((bs >> 1) & 1) {
			cpssp->cardbus_bus_number[f] = (val >> 8) & 0xff;
		}

		/* Subordinate Bus Number Register */
		if ((bs >> 2) & 1) {
			cpssp->subordinate_bus_number[f] = (val >> 16) & 0xff;
		}

		/* Cardbus Latency Timer Register */
		if ((bs >> 3) & 1) {
			cpssp->cardbus_latency_timer[f] = (val >> 24) & 0xff;
		}
		break;

	case 0x1c:
		/* Memory Base 0 Register */
		/* 4-10 */
		if ((bs >> 0) & 1) {
			cpssp->mem_base[f][0] &= ~(0xff << 0);
			cpssp->mem_base[f][0] |= val & (0x00 << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->mem_base[f][0] &= ~(0xff << 8);
			cpssp->mem_base[f][0] |= val & (0xf0 << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->mem_base[f][0] &= ~(0xff << 16);
			cpssp->mem_base[f][0] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->mem_base[f][0] &= ~(0xff << 24);
			cpssp->mem_base[f][0] |= val & (0xff << 24);
		}
#if 0 /* FIXME */
		if (obase != nbase) {
			sig_pci_bus_unmap(cpssp->port_pci_bus,
					cpssp,
					naddr,
					SZWINDOW_0(cpssp, func));
		}
#endif
		break;

	case 0x20:
		/* Memory Limit 0 Register */
		/* 4-11 */
		if ((bs >> 0) & 1) {
			cpssp->mem_limit[f][0] &= ~(0xff << 0);
			cpssp->mem_limit[f][0] |= val & (0x00 << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->mem_limit[f][0] &= ~(0xff << 8);
			cpssp->mem_limit[f][0] |= val & (0xf0 << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->mem_limit[f][0] &= ~(0xff << 16);
			cpssp->mem_limit[f][0] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->mem_limit[f][0] &= ~(0xff << 24);
			cpssp->mem_limit[f][0] |= val & (0xff << 24);
		}
#if 0 /* FIXME */
		if (obase != nbase) {
			sig_pci_bus_unmap(cpssp->port_pci_bus,
					cpssp,
					naddr,
					SZWINDOW_0(cpssp, func));
		}
#endif
		break;

	case 0x24:
		/* Memory Base 1 Register */
		/* 4-10 */
		if ((bs >> 0) & 1) {
			cpssp->mem_base[f][1] &= ~(0xff << 0);
			cpssp->mem_base[f][1] |= val & (0x00 << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->mem_base[f][1] &= ~(0xff << 8);
			cpssp->mem_base[f][1] |= val & (0xf0 << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->mem_base[f][1] &= ~(0xff << 16);
			cpssp->mem_base[f][1] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->mem_base[f][1] &= ~(0xff << 24);
			cpssp->mem_base[f][1] |= val & (0xff << 24);
		}
#if 0 /* FIXME */
		if (obase != nbase) {
			sig_pci_bus_unmap(cpssp->port_pci_bus,
					cpssp,
					naddr,
					SZWINDOW_0(cpssp, func));
		}
#endif
		break;

	case 0x28:
		/* Memory Limit 1 Register */
		/* 4-11 */
		if ((bs >> 0) & 1) {
			cpssp->mem_limit[f][1] &= ~(0xff << 0);
			cpssp->mem_limit[f][1] |= val & (0x00 << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->mem_limit[f][1] &= ~(0xff << 8);
			cpssp->mem_limit[f][1] |= val & (0xf0 << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->mem_limit[f][1] &= ~(0xff << 16);
			cpssp->mem_limit[f][1] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->mem_limit[f][1] &= ~(0xff << 24);
			cpssp->mem_limit[f][1] |= val & (0xff << 24);
		}
#if 0 /* FIXME */
		if (obase != nbase) {
			sig_pci_bus_unmap(cpssp->port_pci_bus,
					cpssp,
					naddr,
					SZWINDOW_0(cpssp, func));
		}
#endif
		break;

	case 0x2c:
		/* I/O Base 0 Register */
		/* 4-11 */
		if ((bs >> 0) & 1) {
			cpssp->io_base[f][0] &= ~(0xff << 0);
			cpssp->io_base[f][0] |= val & (0xfc << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->io_base[f][0] &= ~(0xff << 8);
			cpssp->io_base[f][0] |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->io_base[f][0] &= ~(0xff << 16);
			cpssp->io_base[f][0] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->io_base[f][0] &= ~(0xff << 24);
			cpssp->io_base[f][0] |= val & (0xff << 24);
		}
		break;

	case 0x30:
		/* I/O Limit 0 Register */
		/* 4-12 */
		if ((bs >> 0) & 1) {
			cpssp->io_limit[f][0] &= ~(0xff << 0);
			cpssp->io_limit[f][0] |= val & (0xfc << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->io_limit[f][0] &= ~(0xff << 8);
			cpssp->io_limit[f][0] |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			/* Hardwired to '0'. */
		}
		if ((bs >> 3) & 1) {
			/* Hardwired to '0'. */
		}
		break;

	case 0x34:
		/* I/O Base 1 Register */
		/* 4-11 */
		if ((bs >> 0) & 1) {
			cpssp->io_base[f][1] &= ~(0xff << 0);
			cpssp->io_base[f][1] |= val & (0xfc << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->io_base[f][1] &= ~(0xff << 8);
			cpssp->io_base[f][1] |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->io_base[f][1] &= ~(0xff << 16);
			cpssp->io_base[f][1] |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->io_base[f][1] &= ~(0xff << 24);
			cpssp->io_base[f][1] |= val & (0xff << 24);
		}
		break;

	case 0x38:
		/* I/O Limit 1 Register */
		/* 4-12 */
		if ((bs >> 0) & 1) {
			cpssp->io_limit[f][1] &= ~(0xff << 0);
			cpssp->io_limit[f][1] |= val & (0xfc << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->io_limit[f][1] &= ~(0xff << 8);
			cpssp->io_limit[f][1] |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			/* Hardwired to '0'. */
		}
		if ((bs >> 3) & 1) {
			/* Hardwired to '0'. */
		}
		break;

	case 0x3c:
		/* Interrupt Line Register */
		/* 4-12 */
		if ((bs >> 0) & 1) {
			cpssp->interrupt_line[f] = (val >> 0) & 0xff;
		}

		/* Interrupt Pin Register */
		/* 4-13 */
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		/* Bridge Control Register */
		/* 4-14 */
		if ((bs >> 2) & 1) {
			cpssp->cperren = (val >> 16) & 1;
			cpssp->cserren = (val >> 17) & 1;
			cpssp->isaen[f] = (val >> 18) & 1;
			cpssp->vgaen[f] = (val >> 19) & 1;
			/* Bit 4: Reserved */
			cpssp->mabtmode = (val >> 21) & 1;
			cpssp->crst[f] = (val >> 22) & 1; /* FIXME */
			cpssp->intr[f] = (val >> 23) & 1;
		}
		if ((bs >> 3) & 1) {
			cpssp->prefetch0[f] = (val >> 24) & 1;
			cpssp->prefetch1[f] = (val >> 25) & 1;
			cpssp->posten[f] = (val >> 26) & 1;
			/* Bit 11-15: Reserved */
		}
		break;

	case 0x40:
		/* Subsystem Vendor ID Register */
		/* 4-15 */
		if ((bs >> 0) & 1) {
			if (! cpssp->subsysrw) {
				cpssp->subsystem_vendor_id[f] &= ~(0xff << 0);
				cpssp->subsystem_vendor_id[f] |= val & (0xff << 0);
			} else {
				/* Read-only */
			}
		}
		if ((bs >> 1) & 1) {
			if (! cpssp->subsysrw) {
				cpssp->subsystem_vendor_id[f] &= ~(0xff << 8);
				cpssp->subsystem_vendor_id[f] |= val & (0xff << 8);
			} else {
				/* Read-only */
			}
		}

		/* Subsystem Device ID Register */
		/* 4-15 */
		if ((bs >> 2) & 1) {
			if (! cpssp->subsysrw) {
				cpssp->subsystem_device_id[f] &= ~(0xff << 0);
				cpssp->subsystem_device_id[f] |= (val >> 16) & (0xff << 0);
			} else {
				/* Read-only */
			}
		}
		if ((bs >> 3) & 1) {
			if (! cpssp->subsysrw) {
				cpssp->subsystem_device_id[f] &= ~(0xff << 8);
				cpssp->subsystem_device_id[f] |= (val >> 16) & (0xff << 8);
			} else {
				/* Read-only */
			}
		}
		break;

	case 0x44:
		/* PC Card 16-Bit I/F Legacy-Mode Base Address Register */
		/* 4-15 */
		if ((bs >> 0) & 1) {
			cpssp->legacy_base_address &= ~(0xff << 0);
			cpssp->legacy_base_address |= val & (0xfe << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->legacy_base_address &= ~(0xff << 8);
			cpssp->legacy_base_address |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->legacy_base_address &= ~(0xff << 16);
			cpssp->legacy_base_address |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->legacy_base_address &= ~(0xff << 24);
			cpssp->legacy_base_address |= val & (0xff << 24);
		}
		break;

	case 0x48 ... 0x7c:
		/* Reserved */
		break;

	case 0x80:
		/* System Control Register */
		/* 4-17 */
		if ((bs >> 0) & 1) {
			cpssp->rimux[f] |= (val >> 0) & 0b1;
			cpssp->keepclk |= (val >> 1) & 0b1;
			cpssp->excapower[f] |= (val >> 2) & 0b1;
			/* Bit 3: Reserved */
			cpssp->cb_dpar |= (val >> 4) & 0b1;
			cpssp->subsysrw |= (val >> 5) & 0b1;
			cpssp->pwrsavings[f] |= (val >> 6) & 0b1;
			/* Bit 7: Reserved */
		}
		if ((bs >> 1) & 1) {
			cpssp->interrogate[f] |= (val >> 8) & 0b1;
			cpssp->delaydown |= (val >> 9) & 0b1;
			cpssp->delayup |= (val >> 10) & 0b1;
			cpssp->pwrstream |= (val >> 11) & 0b1;
			/* Bit 12: Reserved */
			cpssp->socactive[f] |= (val >> 13) & 0b1;
			cpssp->mrburstup |= (val >> 14) & 0b1;
			cpssp->mrburstdn |= (val >> 15) & 0b1;
		}
		if ((bs >> 2) & 1) {
			/* Bit 16-19: Reserved */
			cpssp->reducezv[f] |= (val >> 20) & 0b1;
			cpssp->vccprot[f] |= (val >> 21) & 0b1;
			cpssp->cbrsvd[f] |= (val >> 22) & 0b1;
			/* Bit 23: Reserved */
		}
		if ((bs >> 3) & 1) {
			cpssp->smienb = (val >> 24) & 0b1;
			cpssp->smistatus[f] &= ~((val >> 25) & 0b1);
			cpssp->smiroute[f] = (val >> 26) & 0b1;
			cpssp->p2cclk = (val >> 27) & 0b1;
			/* Bit 28: Reserved */
			cpssp->intrtie = (val >> 29) & 0b1;
			cpssp->ser_step = (val >> 30) & 0b11;
		}

	case 0x84 ... 0x88:
		/* Reserved */
		break;

	case 0x8c:
		/* Multifunction Routing Register */
		/* 4-19 */
		if ((bs >> 0) & 1) {
			cpssp->mfunc0 = (val >> 0) & 0b1111;
			cpssp->mfunc1 = (val >> 4) & 0b1111;
		}
		if ((bs >> 1) & 1) {
			cpssp->mfunc2 = (val >> 8) & 0b1111;
			cpssp->mfunc3 = (val >> 12) & 0b1111;
		}
		if ((bs >> 2) & 1) {
			cpssp->mfunc4 = (val >> 16) & 0b1111;
			cpssp->mfunc5 = (val >> 20) & 0b1111;
		}
		if ((bs >> 3) & 1) {
			cpssp->mfunc6 = (val >> 24) & 0b1111;
			/* Bit 28-31: Reserved */
		}
		break;

	case 0x90:
		/* Retry Status Register */
		/* 4-21 */
		if ((bs >> 0) & 1) {
			/* Bit 0: Reserved */
			cpssp->texp_pci &= ~((val >> 1) & 1);
			/* Bit 2: Reserved */
			cpssp->texp_cba &= ~((val >> 3) & 1);
			/* Bit 4: Reserved */
			cpssp->texp_cbb &= ~((val >> 5) & 1);
			cpssp->cbretry = (val >> 6) & 1;
			cpssp->pciretry = (val >> 7) & 1;
		}

		/* Card Control Register */
		/* 4-22 */
		if ((bs >> 1) & 1) {
			cpssp->ifg[f] &= ~((val >> 8) & 1);
			cpssp->spkrouten[f] = (val >> 9) & 1;
			cpssp->aud2mux[f] = (val >> 10) & 1;
			/* Bit 3-4: Reserved */
			cpssp->port_sel[f] = (val >> 13) & 1;
			cpssp->zvenable[f] = (val >> 14) & 1;
			cpssp->rienb = (val >> 15) & 1;
		}

		/* Device Control Register */
		/* 4-23 */
		if ((bs >> 2) & 1) {
			/* Bit 0: Reserved */
			cpssp->intmode = (val >> 17) & 0b11;
			/* Bit 3: Test */
			/* Bit 4: Reserved */
			/* Bit 5: Test */
			cpssp->_3vcapable = (val >> 22) & 0b1;
			cpssp->sktpwr_lock = (val >> 23) & 0b1;
		}

		/* Diagnostic Register */
		/* 4-24 */
		if ((bs >> 3) & 1) {
			cpssp->stdzven[f] = (val >> 24) & 0b1;
			/* DIAG1 - FIXME */
			/* DIAG2 - FIXME */
			/* DIAG3 - FIXME */
			/* DIAG4 - FIXME */
			cpssp->csc[f] = (val >> 29) & 0b1;
			/* Bit 6: Reserved */
			/* TRUE_VAL - FIXME */
		}
		break;

	case 0x94 ... 0x9c:
		/* Reserved */
		break;

	case 0xa0:
		/* Capability ID Register */
		/* 4-25 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}

		/* Next-Item Pointer Register */
		/* 4-25 */
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		/* Power-Management Capabilities Register */
		/* 4-26 */
		if ((bs >> 2) & 1) {
			/* Bit 0-5: Read-only */
			/* Bit 6-7: Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Bit 8: Reserved */
			/* Bit 9-14: Read-only */
			cpssp->pme_support = (val >> 31) & 0b1;
		}
		break;

	case 0xa4:
		/* Power-Management Control/Status Register */
		/* 4-27 */
		if ((bs >> 0) & 1) {
			cpssp->pwr_state = (val >> 0) & 0b11;
			/* Bit 2-7: Reserved */
		}
		if ((bs >> 1) & 1) {
			cpssp->pme_en = (val >> 8) & 0b1;
			/* Bit 9-14: Read-only */
			cpssp->pmestat = (val >> 15) & 0b1;
		}

		/* Power-Management Control/Status Register Bridge Support */
		/* 4-28 */
		if ((bs >> 2) & 1) {
			/* Bit 0-5: Reserved */
			cpssp->b2_b3 = (val >> 22) & 0b1;
			cpssp->bpcc_en = (val >> 23) & 0b1;
		}

		/* Power-Management Data Register */
		/* 4-28 */
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0xa8:
		/* General-Purpose Event Status Register */
		/* 4-29 */
		if ((bs >> 0) & 1) {
			cpssp->gp0_sts &= ~((val >> 0) & 1);
			cpssp->gp1_sts &= ~((val >> 1) & 1);
			cpssp->gp2_sts &= ~((val >> 2) & 1);
			cpssp->gp3_sts &= ~((val >> 3) & 1);
			cpssp->gp4_sts &= ~((val >> 4) & 1);
			/* Bit 5-7: Reserved */
		}
		if ((bs >> 1) & 1) {
			cpssp->vpp12_sts &= ~((val >> 8) & 1);
			/* Bit 9-10: Reserved */
			cpssp->pwr_sts &= ~((val >> 11) & 1);
			/* Bit 12-13: Reserved */
			cpssp->zv1_sts &= ~((val >> 14) & 1);
			cpssp->zv0_sts &= ~((val >> 15) & 1);
		}

		/* General-Purpose Event Enable Register */
		/* 4-30 */
		if ((bs >> 2) & 1) {
			cpssp->gp0_en = (val >> 16) & 1;
			cpssp->gp1_en = (val >> 17) & 1;
			cpssp->gp2_en = (val >> 18) & 1;
			cpssp->gp3_en = (val >> 19) & 1;
			cpssp->gp4_en = (val >> 20) & 1;
			/* Bit 5-7: Reserved */
		}
		if ((bs >> 3) & 1) {
			cpssp->vpp12_en = (val >> 24) & 1;
			/* Bit 9-10: Reserved */
			cpssp->pwr_en = (val >> 27) & 1;
			/* Bit 12-13: Reserved */
			cpssp->zv1_en = (val >> 30) & 1;
			cpssp->zv0_en = (val >> 31) & 1;
		}
		break;

	case 0xac:
		/* General-Purpose Input Register */
		/* 4-31 */
		if ((bs >> 0) & 1) {
			/* Bit 0-4: Read-only */
			/* Bit 5-7: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}

		/* General-Purpose Output Register */
		/* 4-32 */
		if ((bs >> 2) & 1) {
			cpssp->gpo0_data = (val >> 16) & 0b1;
			cpssp->gpo1_data = (val >> 17) & 0b1;
			cpssp->gpo2_data = (val >> 18) & 0b1;
			cpssp->gpo3_data = (val >> 19) & 0b1;
			cpssp->gpo4_data = (val >> 20) & 0b1;
			/* Bit 5-7: Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0xb0:
		/* Serial Registers - FIXME */
		if ((bs >> 0) & 1) {
			cpssp->ser0 = (val >> 0) & 0xff;
		}
		if ((bs >> 1) & 1) {
			cpssp->ser1 = (val >> 8) & 0xff;
		}
		if ((bs >> 2) & 1) {
			cpssp->ser2 = (val >> 16) & 0xff;
		}
		if ((bs >> 3) & 1) {
			cpssp->ser3 = (val >> 24) & 0xff;
		}
		break;

	case 0xb4 ... 0xfc:
		/* Reserved */
		break;

	default: assert(0); /* Cannot happen. */
	}

	if (DEBUG) fprintf(stderr, "%s: addr=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, addr, bs, val);
	
	return 0;
}

/* -------------------------------------------------------------------- */
/* Forwarder                                                            */
/* -------------------------------------------------------------------- */

enum region {
	REGION_NONE,
	REGION_CHIP_LEGACY,
	REGION_CHIP_REGISTER0,
	REGION_CHIP_REGISTER1,
	REGION_CB0,
	REGION_CB1,
	REGION_PCI,
};

static int
CHIP_(forward_c0r)(
	struct cpssp *cpssp,
	int orig,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	int f;

	assert(! (addr & 3));

	f = (addr >> 8) & 0x7;
	if (f & 0x6) {
		return -1;
	}
	addr &= 0xff;

	return CHIP_(chip_c0r)(cpssp, f, addr, bs, valp);
}

static int
CHIP_(forward_c0w)(
	struct cpssp *cpssp,
	int orig,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	int f;

	assert(! (addr & 3));

	f = (addr >> 8) & 0x7;
	if (f & 0x6) {
		return -1;
	}
	addr &= 0xff;

	return CHIP_(chip_c0w)(cpssp, f, addr, bs, val);
}

static int
CHIP_(forward_c1r)(
	struct cpssp *cpssp,
	int orig,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	unsigned int bus;
	unsigned int dev;

	bus = (addr >> 16) & 0xff;

	if (bus == cpssp->cardbus_bus_number[0]) {
		dev = (addr >> 11) & 0x1f;
		addr &= 0x7fc;
		if (! cpssp->powered[0]) {
			return 1;
		} else {
			return sig_pci_bus_c0r(cpssp->port_cb[0], cpssp,
					(1 << (11 + dev)) | addr, bs, valp);
		}
	} else if (bus == cpssp->cardbus_bus_number[1]) {
		dev = (addr >> 11) & 0x1f;
		addr &= 0x7fc;
		if (! cpssp->powered[1]) {
			return 1;
		} else {
			return sig_pci_bus_c0r(cpssp->port_cb[1], cpssp,
					(1 << (11 + dev)) | addr, bs, valp);
		}
	} else if (bus <= cpssp->subordinate_bus_number[0]) {
		if (! cpssp->powered[0]) {
			return 1;
		} else {
			return sig_pci_bus_c1r(cpssp->port_cb[0], cpssp,
					addr, bs, valp);
		}
	} else if (bus <= cpssp->subordinate_bus_number[1]) {
		if (! cpssp->powered[1]) {
			return 1;
		} else {
			return sig_pci_bus_c1r(cpssp->port_cb[1], cpssp,
					addr, bs, valp);
		}
	}

	return -1;
}

static int
CHIP_(forward_c1w)(
	struct cpssp *cpssp,
	int orig,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	unsigned int bus;
	unsigned int dev;

	bus = (addr >> 16) & 0xff;

	if (bus == cpssp->cardbus_bus_number[0]) {
		dev = (addr >> 11) & 0x1f;
		addr &= 0x7fc;
		if (! cpssp->powered[0]) {
			return 1;
		} else {
			return sig_pci_bus_c0w(cpssp->port_cb[0], cpssp,
					(1 << (11 + dev)) | addr, bs, val);
		}
	} else if (bus == cpssp->cardbus_bus_number[1]) {
		dev = (addr >> 11) & 0x1f;
		addr &= 0x7fc;
		if (! cpssp->powered[1]) {
			return 1;
		} else {
			return sig_pci_bus_c0w(cpssp->port_cb[1], cpssp,
					(1 << (11 + dev)) | addr, bs, val);
		}
	} else if (bus <= cpssp->subordinate_bus_number[0]) {
		if (! cpssp->powered[0]) {
			return 1;
		} else {
			return sig_pci_bus_c1w(cpssp->port_cb[0], cpssp,
					addr, bs, val);
		}
	} else if (bus <= cpssp->subordinate_bus_number[1]) {
		if (! cpssp->powered[1]) {
			return 1;
		} else {
			return sig_pci_bus_c1w(cpssp->port_cb[1], cpssp,
					addr, bs, val);
		}
	}

	return -1;
}

static enum region
CHIP_(region_io)(struct cpssp *cpssp, int orig, uint32_t addr)
{
	enum region region;

	if (cpssp->legacy_base_address <= addr
	 && addr < cpssp->legacy_base_address + 4) { /* FIXME */
		/* Forward to Chip's Legacy Registers */
		region = REGION_CHIP_LEGACY;

	} else if (cpssp->io_en[0]
		&& (cpssp->io_base[0][0] != 0
		 || cpssp->io_limit[0][0] != 0)
		&& (addr & 0xffff0000) == (cpssp->io_base[0][0] & 0xffff0000)
		&& (cpssp->io_base[0][0] & 0x0000ffff) <= (addr & 0x0000ffff)
		&& (addr & 0x0000ffff) <= cpssp->io_limit[0][0]) {
		/* Forward to CardBus0 */
		region = REGION_CB0;

	} else if (cpssp->io_en[0]
		&& (cpssp->io_base[0][1] != 0
		 || cpssp->io_limit[0][1] != 0)
		&& (addr & 0xffff0000) == (cpssp->io_base[0][1] & 0xffff0000)
		&& (cpssp->io_base[0][1] & 0x0000ffff) <= (addr & 0x0000ffff)
		&& (addr & 0x0000ffff) <= cpssp->io_limit[0][1]) {
		/* Forward to CardBus0 */
		region = REGION_CB0;

	} else if (cpssp->io_en[1]
		&& (cpssp->io_base[1][0] != 0
		 || cpssp->io_limit[1][0] != 0)
		&& (addr & 0xffff0000) == (cpssp->io_base[1][0] & 0xffff0000)
		&& (cpssp->io_base[1][0] & 0x0000ffff) <= (addr & 0x0000ffff)
		&& (addr & 0x0000ffff) <= cpssp->io_limit[1][0]) {
		/* Forward to CardBus0 */
		region = REGION_CB1;

	} else if (cpssp->io_en[1]
		&& (cpssp->io_base[1][1] != 0
		 || cpssp->io_limit[1][1] != 0)
		&& (addr & 0xffff0000) == (cpssp->io_base[1][1] & 0xffff0000)
		&& (cpssp->io_base[1][1] & 0x0000ffff) <= (addr & 0x0000ffff)
		&& (addr & 0x0000ffff) <= cpssp->io_limit[1][1]) {
		/* Forward to CardBus0 */
		region = REGION_CB1;

	} else {
		/* Forward to PCI Bus */
		region = REGION_PCI;
	}

	if (region == REGION_PCI
	 && orig == -1) {
		region = REGION_NONE;

	} else if (region == REGION_CB0
		&& (orig == 0
		 || ! cpssp->powered[0])) {
		region = REGION_NONE;

	} else if (region == REGION_CB1
		&& (orig == 1
		 || ! cpssp->powered[1])) {
		region = REGION_NONE;
	}

	return region;
}

static int
CHIP_(forward_ior)(
	struct cpssp *cpssp,
	int orig,
	uint32_t pa,
	unsigned int bs,
	uint32_t *valp
)
{
	int ret;

	switch (CHIP_(region_io)(cpssp, orig, pa)) {
	case REGION_CHIP_LEGACY:
		CHIP_(legacy_read)(cpssp, pa, bs, valp);
		ret = 0;
		break;
	case REGION_CB0:
		ret = sig_pci_bus_ior(cpssp->port_cb[0], cpssp, pa, bs, valp);
		break;
	case REGION_CB1:
		ret = sig_pci_bus_ior(cpssp->port_cb[1], cpssp, pa, bs, valp);
		break;
	default:
		ret = -1;
		break;
	}

	if (ret == 0
	 && DEBUG) fprintf(stderr, "%s: orig=%d, pa=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, orig, pa, bs, *valp);

	return ret;
}

static int
CHIP_(forward_iow)(
	struct cpssp *cpssp,
	int orig,
	uint32_t pa,
	unsigned int bs,
	uint32_t val
)
{
	int ret;

	switch (CHIP_(region_io)(cpssp, orig, pa)) {
	case REGION_CHIP_LEGACY:
		CHIP_(legacy_write)(cpssp, pa, bs, val);
		ret = 0;
		break;
	case REGION_CB0:
		ret = sig_pci_bus_iow(cpssp->port_cb[0], cpssp, pa, bs, val);
		break;
	case REGION_CB1:
		ret = sig_pci_bus_iow(cpssp->port_cb[1], cpssp, pa, bs, val);
		break;
	default:
		ret = -1;
		break;
	}

	if (ret == 0
	 && DEBUG) fprintf(stderr, "%s: orig=%d, pa=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, orig, pa, bs, val);

	return ret;
}

static enum region
CHIP_(region_mem)(struct cpssp *cpssp, int orig, uint32_t addr)
{
	enum region region;

	if ((cpssp->mem_en[0])
		&& cpssp->base_address[0] == (addr & ~0xfff)) {
		/* Forward to Chip Register Bank 0 */
		region = REGION_CHIP_REGISTER0;

	} else if ((cpssp->mem_en[1])
		&& cpssp->base_address[1] == (addr & ~0xfff)) {
		/* Forward to Chip Register Bank 1 */
		region = REGION_CHIP_REGISTER1;

	} else if (cpssp->mem_en[0]
		&& (cpssp->mem_base[0][0] != 0
		 || cpssp->mem_limit[0][0] != 0)
		&& cpssp->mem_base[0][0] <= (addr & ~0xfff)
		 && (addr & ~0xfff) <= cpssp->mem_limit[0][0]) {
		/* Forward to CardBus0 */
		region = REGION_CB0;

	} else if (cpssp->mem_en[0]
		&& (cpssp->mem_base[0][1] != 0
		 || cpssp->mem_limit[0][1] != 0)
		&& cpssp->mem_base[0][1] <= (addr & ~0xfff)
		 && (addr & ~0xfff) <= cpssp->mem_limit[0][1]) {
		/* Forward to CardBus0 */
		region = REGION_CB0;

	} else if (cpssp->mem_en[1]
		&& (cpssp->mem_base[1][0] != 0
		 || cpssp->mem_limit[1][0] != 0)
		&& cpssp->mem_base[1][0] <= (addr & ~0xfff)
		 && (addr & ~0xfff) <= cpssp->mem_limit[1][0]) {
		/* Forward to CardBus1 */
		region = REGION_CB1;

	} else if (cpssp->mem_en[1]
		&& (cpssp->mem_base[1][1] != 0
		 || cpssp->mem_limit[1][1] != 0)
		&& cpssp->mem_base[1][1] <= (addr & ~0xfff)
		 && (addr & ~0xfff) <= cpssp->mem_limit[1][1]) {
		/* Forward to CardBus1 */
		region = REGION_CB1;

	} else {
		/* Forward to PCI Bus */
		region = REGION_PCI;
	}

	if (region == REGION_PCI
	 && orig == -1) {
		region = REGION_NONE;

	} else if (region == REGION_CB0
		&& (orig == 0
		 || ! cpssp->powered[0])) {
		region = REGION_NONE;

	} else if (region == REGION_CB1
		&& (orig == 1
		 || ! cpssp->powered[1])) {
		region = REGION_NONE;
	}

	return region;
}

static int
CHIP_(forward_mr)(
	struct cpssp *cpssp,
	int orig,
	uint32_t pa,
	unsigned int bs,
	uint32_t *valp
)
{
	int ret;

	switch (CHIP_(region_mem)(cpssp, orig, pa)) {
	case REGION_CHIP_REGISTER0:
		CHIP_(chip_read)(cpssp, 0, valp, pa, bs);
		ret = 0;
		break;
	case REGION_CHIP_REGISTER1:
		CHIP_(chip_read)(cpssp, 1, valp, pa, bs);
		ret = 0;
		break;
	case REGION_CB0:
		ret = sig_pci_bus_mr(cpssp->port_cb[0], cpssp, pa, bs, valp);
		break;
	case REGION_CB1:
		ret = sig_pci_bus_mr(cpssp->port_cb[1], cpssp, pa, bs, valp);
		break;
	case REGION_PCI:
		ret = sig_pci_bus_mr(cpssp->port_pci_bus, cpssp, pa, bs, valp);
		break;
	default:
		ret = -1;
		break;
	}

	if (ret == 0
	 && DEBUG) fprintf(stderr, "%s: orig=%d, pa=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, orig, pa, bs, *valp);

	return ret;
}

static int
CHIP_(forward_mw)(
	struct cpssp *cpssp,
	int orig,
	uint32_t pa,
	unsigned int bs,
	uint32_t val
)
{
	int ret;

	switch (CHIP_(region_mem)(cpssp, orig, pa)) {
	case REGION_CHIP_REGISTER0:
		CHIP_(chip_write)(cpssp, 0, val, pa, bs);
		ret = 0;
		break;
	case REGION_CHIP_REGISTER1:
		CHIP_(chip_write)(cpssp, 1, val, pa, bs);
		ret = 0;
		break;
	case REGION_CB0:
		ret = sig_pci_bus_mw(cpssp->port_cb[0], cpssp, pa, bs, val);
		break;
	case REGION_CB1:
		ret = sig_pci_bus_mw(cpssp->port_cb[1], cpssp, pa, bs, val);
		break;
	case REGION_PCI:
		ret = sig_pci_bus_mw(cpssp->port_pci_bus, cpssp, pa, bs, val);
		break;
	default:
		ret = -1;
		break;
	}

	if (ret == 0
	 && DEBUG) fprintf(stderr, "%s: orig=%d, pa=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, orig, pa, bs, val);

	return ret;
}

/* -------------------------------------------------------------------- */
/* PCI Bus Interface                                                    */
/* -------------------------------------------------------------------- */

static int
CHIP_(c0r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return CHIP_(forward_c0r)(cpssp, -1, addr, bs, valp);
}

static int
CHIP_(c0w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return CHIP_(forward_c0w)(cpssp, -1, addr, bs, val);
}

static int
CHIP_(c1r)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return CHIP_(forward_c1r)(cpssp, -1, addr, bs, valp);
}

static int
CHIP_(c1w)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return CHIP_(forward_c1w)(cpssp, -1, addr, bs, val);
}

static int
CHIP_(ior)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	
	return CHIP_(forward_ior)(cpssp, -1, addr, bs, valp);
}

static int
CHIP_(iow)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	
	return CHIP_(forward_iow)(cpssp, -1, addr, bs, val);
}

static int
CHIP_(mr)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	
	return CHIP_(forward_mr)(cpssp, -1, addr, bs, valp);
}

static int
CHIP_(mw)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	
	return CHIP_(forward_mw)(cpssp, -1, addr, bs, val);
}

static int
CHIP_(map)(
	struct cpssp *cpssp,
	uint32_t addr,
	char **haddr_p
)
{
	unsigned int func;
	
	for (func = 0; func < 2; func++) {
		if (! cpssp->mem_en[func]) {
			continue;
		}
		if (cpssp->base_address[func] == (addr & ~0xfff)) {
			*haddr_p = NULL;
			return 0;
		}
		if ((cpssp->mem_base[func][0] != 0
		  || cpssp->mem_limit[func][0] != 0)
		 && cpssp->mem_base[func][0] <= (addr & ~0xfff)
		  && (addr & ~0xfff) <= cpssp->mem_limit[func][0]) {
			*haddr_p = NULL;
			return 0;
		}
		if ((cpssp->mem_base[func][1] != 0
		  || cpssp->mem_limit[func][1] != 0)
		 && cpssp->mem_base[func][1] <= (addr & ~0xfff)
		  && (addr & ~0xfff) <= cpssp->mem_limit[func][1]) {
			*haddr_p = NULL;
			return 0;
		}
	}
	
	return 1;
}

static int
CHIP_(map_r)(
	void *_cpssp,
	uint32_t addr,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = _cpssp;

	/* FIXME */
	*cfp = CHIP_(mr);
	*csp = cpssp;
	return CHIP_(map)(cpssp, addr, haddr_p);
}

static int
CHIP_(map_w)(
	void *_cpssp,
	uint32_t addr,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = _cpssp;

	/* FIXME */
	*cfp = CHIP_(mw);
	*csp = cpssp;
	return CHIP_(map)(cpssp, addr, haddr_p);
}

/* -------------------------------------------------------------------- */
/* Cardbus 0 Interface                                                  */
/* -------------------------------------------------------------------- */

static int
CHIP_(cb0_read)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return CHIP_(forward_mr)(cpssp, 0, addr, bs, valp);
}

static int
CHIP_(cb0_write)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return CHIP_(forward_mw)(cpssp, 0, addr, bs, val);
}

static int
CHIP_(cb0_map_r)(
	void *_cpssp,
	uint32_t pa,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp,
	char **haddr_p
)
{
	/* FIXME */
	return -1;
}

static int
CHIP_(cb0_map_w)(
	void *_cpssp,
	uint32_t pa,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp,
	char **haddr_p
)
{
	/* FIXME */
	return -1;
}

static void
CHIP_(cb0_irq_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (val) {
		cpssp->ifg[0] = 1;
	}
	cpssp->int_set[0] = val;
	CHIP_(irq_update)(cpssp, 0);
}

static void
CHIP_(cb0_cd1_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	CHIP_(cd1_set)(cpssp, 0, val);
}

static void
CHIP_(cb0_cd2_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	CHIP_(cd2_set)(cpssp, 0, val);
}

static void
CHIP_(cb0_cvs1_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	CHIP_(cvs1_set)(cpssp, 0, val);
}

static void
CHIP_(cb0_cvs2_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	CHIP_(cvs2_set)(cpssp, 0, val);
}

/* -------------------------------------------------------------------- */
/* Cardbus 1 Interface                                                  */
/* -------------------------------------------------------------------- */

static int
CHIP_(cb1_read)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return CHIP_(forward_mr)(cpssp, 1, addr, bs, valp);
}

static int
CHIP_(cb1_write)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return CHIP_(forward_mw)(cpssp, 1, addr, bs, val);
}

static int
CHIP_(cb1_map_r)(
	void *_cpssp,
	uint32_t pa,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp,
	char **haddr_p
)
{
	/* FIXME */
	return -1;
}

static int
CHIP_(cb1_map_w)(
	void *_cpssp,
	uint32_t pa,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp,
	char **haddr_p
)
{
	/* FIXME */
	return -1;
}

static void
CHIP_(cb1_irq_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (val) {
		cpssp->ifg[1] = 1;
	}
	cpssp->int_set[1] = val;
	CHIP_(irq_update)(cpssp, 1);
}

static void
CHIP_(cb1_cd1_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	CHIP_(cd1_set)(cpssp, 1, val);
}

static void
CHIP_(cb1_cd2_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	CHIP_(cd2_set)(cpssp, 1, val);
}

static void
CHIP_(cb1_cvs1_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	CHIP_(cvs1_set)(cpssp, 1, val);
}

static void
CHIP_(cb1_cvs2_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	CHIP_(cvs2_set)(cpssp, 1, val);
}

/* -------------------------------------------------------------------- */
/* Power/Reset Interface                                                */
/* -------------------------------------------------------------------- */

static void
CHIP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (! val) {
		/* Power off CardBus devices. */
		cpssp->powered[0] = 0;
		sig_std_logic_or_set(cpssp->port_cb_power[0], cpssp, 0);
		cpssp->powered[1] = 0;
		sig_std_logic_or_set(cpssp->port_cb_power[1], cpssp, 0);
	}

	/* More to do... */
	/* FIXME */
}

static void
CHIP_(n_reset_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	int f;
	
	for (f = 0; f < 2; f++) {
		/* Config Space Registers */
		cpssp->io_en[f] = 0;
		cpssp->mem_en[f] = 0;
		cpssp->mast_en[f] = 0;
		cpssp->vga_en[f] = 0;
		cpssp->perr_en[f] = 0;
		cpssp->serr_en[f] = 0;
		cpssp->cache_line_size[f] = 0;
		cpssp->latency_timer[f] = 0;
		cpssp->pci_bus_number[f] = 0;
		cpssp->cardbus_bus_number[f] = 0;
		cpssp->subordinate_bus_number[f] = 0;
		cpssp->cardbus_latency_timer[f] = 0;
		cpssp->mem_base[f][0] = 0;
		cpssp->mem_base[f][1] = 0;
		cpssp->mem_limit[f][0] = 0;
		cpssp->mem_limit[f][1] = 0;
		cpssp->io_base[f][0] = 0;
		cpssp->io_base[f][1] = 0;
		cpssp->io_limit[f][0] = 0;
		cpssp->io_limit[f][1] = 0;
		cpssp->interrupt_line[f] = 0;

		/* Bridge Control Register */
		/* 4-14 */
		cpssp->cperren = 0;
		cpssp->cserren = 0;
		cpssp->isaen[f] = 0;
		cpssp->vgaen[f] = 0;
		cpssp->mabtmode = 0;
		cpssp->crst[f] = 1;
		cpssp->intr[f] = 0;
		cpssp->prefetch0[f] = 1;
		cpssp->prefetch1[f] = 1;
		cpssp->posten[f] = 0;

		/* Subsystem Vendor ID Register */
		/* 4-15 */
		cpssp->subsystem_vendor_id[f] = 0;

		/* Subsystem Device ID Register */
		/* 4-15 */
		cpssp->subsystem_device_id[f] = 0;

		/* PC Card 16-Bit I/F Legacy-Mode Base Address Register */
		cpssp->legacy_base_address = 0;

		cpssp->rimux[f] = 0;
		cpssp->keepclk = 0;
		cpssp->excapower[f] = 0;
		cpssp->cb_dpar = 0;
		cpssp->subsysrw = 1;
		cpssp->pwrsavings[f] = 1;
		cpssp->interrogate[f] = 0;
		cpssp->delaydown = 0;
		cpssp->delayup = 0;
		cpssp->pwrstream = 0;
		cpssp->socactive[f] = 0;
		cpssp->mrburstup = 0;
		cpssp->mrburstdn = 1;
		cpssp->reducezv[f] = 0;
		cpssp->vccprot[f] = 0;
		cpssp->cbrsvd[f] = 1;
		cpssp->smienb = 0;
		cpssp->smistatus[f] = 0;
		cpssp->smiroute[f] = 0;
		cpssp->p2cclk = 0;
		cpssp->intrtie = 0;
		cpssp->ser_step = 0;

		/* Multifunction Routing Register */
		/* 4-19 */
		cpssp->mfunc0 = 0;
		cpssp->mfunc1 = 0;
		cpssp->mfunc2 = 0;
		cpssp->mfunc3 = 1;
		cpssp->mfunc4 = 0;
		cpssp->mfunc5 = 0;
		cpssp->mfunc6 = 0;

		/* Retry Status Register */
		/* 4-21 */
		cpssp->texp_pci = 0;
		cpssp->texp_cba = 0;
		cpssp->texp_cbb = 0;
		cpssp->cbretry = 0;
		cpssp->pciretry = 0;

		/* Card Control Register */
		/* 4-22 */
		cpssp->ifg[f] = 0;
		cpssp->spkrouten[f] = 0;
		cpssp->aud2mux[f] = 0;
		cpssp->port_sel[f] = 0;
		cpssp->zvenable[f] = 0;
		cpssp->rienb = 0;

		/* Device Control Register */
		/* 4-23 */
		cpssp->intmode = 0b00;
		cpssp->_3vcapable = 1;
		cpssp->sktpwr_lock = 0;

		/* Diagnostic Register */
		/* 4-24 */
		cpssp->stdzven[f] = 0;
		cpssp->csc[f] = 1;

		/* Power-Management Capabilities Register */
		/* 4-25 */
		cpssp->pme_support = 1;

		/* Power-Management Control/Status Register */
		/* 4-27 */
		cpssp->pwr_state = 0;
		cpssp->pme_en = 0;
		cpssp->pmestat = 0;

		/* Power-Management Control/Status Register Bridge Support */
		/* 4-28 */
		cpssp->b2_b3 = 1;
		cpssp->bpcc_en = 1;

		/* General-Purpose Event Status Register */
		/* 4-29 */
		cpssp->gp0_sts = 0;
		cpssp->gp1_sts = 0;
		cpssp->gp2_sts = 0;
		cpssp->gp3_sts = 0;
		cpssp->gp4_sts = 0;
		cpssp->vpp12_sts = 0;
		cpssp->pwr_sts = 0;
		cpssp->zv1_sts = 0;
		cpssp->zv0_sts = 0;

		/* General-Purpose Event Enable Register */
		/* 4-30 */
		cpssp->gp0_en = 0;
		cpssp->gp1_en = 0;
		cpssp->gp2_en = 0;
		cpssp->gp3_en = 0;
		cpssp->gp4_en = 0;
		cpssp->vpp12_en = 0;
		cpssp->pwr_en = 0;
		cpssp->zv1_en = 0;
		cpssp->zv0_en = 0;

		/* General-Purpose Input Register */
		/* 4-31 */

		/* General-Purpose Output Register */
		/* 4-32 */
		cpssp->gpo0_data = 0;
		cpssp->gpo1_data = 0;
		cpssp->gpo2_data = 0;
		cpssp->gpo3_data = 0;
		cpssp->gpo4_data = 0;

		/* Serial Registers - FIXME */
		cpssp->ser0 = 0;
		cpssp->ser1 = 0;
		cpssp->ser2 = 0;
		cpssp->ser3 = 0;

		/* Cardbus Registers */
		cpssp->pwrevent[f] = 0;
		cpssp->cd2event[f] = 0;
		cpssp->cd1event[f] = 0;
		cpssp->cstsevent[f] = 0;
		cpssp->pwrmask[f] = 0;
		cpssp->cd2mask[f] = 0;
		cpssp->cd1mask[f] = 0;
		cpssp->cstsmask[f] = 0;
		cpssp->vppctrl[f] = 0;
		cpssp->vccctrl[f] = 0;
		cpssp->stopclk[f] = 0;
		cpssp->zven[f] = 0;
		cpssp->clkctrl[f] = 0;
		cpssp->clkctrlen[f] = 0;
		cpssp->_3vcard[f] = 0;
		cpssp->_5vcard[f] = 0;
		cpssp->_16bitcard[f] = 0;
		cpssp->cbcard[f] = 0;
		cpssp->notacard[f] = 0;
		cpssp->pwrcycle[f] = 0;

		/* ExCA Registers */
		memset(cpssp->exca_regs[f], 0,
				sizeof(cpssp->exca_regs[f]));
		memset(cpssp->page_regs[f], 0,
				sizeof(cpssp->page_regs[f]));
	}
}
	
void *
CHIP_(create)(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_power,
	struct sig_std_logic *port_reset_hash_,
	/* PCI bus */
	struct sig_pci_idsel *port_idsel,
	struct sig_pci_bus *port_pci_bus,
	struct sig_std_logic *port_intA,
	struct sig_std_logic *port_intB,
	/* Cardbus 0 */
	struct sig_std_logic *port_cb0_power,
	struct sig_std_logic *port_cb0_reset_hash_,
	struct sig_pci_bus *port_cb0,
	struct sig_std_logic *port_cb0_int,
	struct sig_std_logic *port_cb0_cd1,
	struct sig_std_logic *port_cb0_cd2,
	struct sig_std_logic *port_cb0_cvs1,
	struct sig_std_logic *port_cb0_cvs2,
	/* Cardbus 1 */
	struct sig_std_logic *port_cb1_power,
	struct sig_std_logic *port_cb1_reset_hash_,
	struct sig_pci_bus *port_cb1,
	struct sig_std_logic *port_cb1_int,
	struct sig_std_logic *port_cb1_cd1,
	struct sig_std_logic *port_cb1_cd2,
	struct sig_std_logic *port_cb1_cvs1,
	struct sig_std_logic *port_cb1_cvs2
)
{
	static const struct sig_std_logic_funcs power_funcs = {
		.boolean_or_set = CHIP_(power_set),
	};
	static const struct sig_std_logic_funcs reset_hash__funcs = {
		.boolean_or_set = CHIP_(n_reset_set),
	};
	static const struct sig_pci_idsel_funcs idsel_funcs = {
		.c0r =		CHIP_(c0r),
		.c0w =		CHIP_(c0w),
	};
	static const struct sig_pci_bus_funcs pci_bus_funcs = {
		.c1r =		CHIP_(c1r),
		.c1w =		CHIP_(c1w),

		.ior =		CHIP_(ior),
		.iow =		CHIP_(iow),
		.mr =		CHIP_(mr),
		.mw =		CHIP_(mw),
		.map_r =	CHIP_(map_r),
		.map_w =	CHIP_(map_w),
	}; 
	static const struct sig_pci_bus_funcs cb0_funcs = {
		.mr =		CHIP_(cb0_read),
		.mw =		CHIP_(cb0_write),
		.map_r =	CHIP_(cb0_map_r),
		.map_w =	CHIP_(cb0_map_w),
	};
	static const struct sig_std_logic_funcs cb0_int_funcs = {
		.boolean_or_set = CHIP_(cb0_irq_set),
	};
	static const struct sig_std_logic_funcs cb0_cd1_funcs = {
		.boolean_or_set = CHIP_(cb0_cd1_set),
	};
	static const struct sig_std_logic_funcs cb0_cd2_funcs = {
		.boolean_or_set = CHIP_(cb0_cd2_set),
	};
	static const struct sig_std_logic_funcs cb0_cvs1_funcs = {
		.boolean_or_set = CHIP_(cb0_cvs1_set),
	};
	static const struct sig_std_logic_funcs cb0_cvs2_funcs = {
		.boolean_or_set = CHIP_(cb0_cvs2_set),
	};
	static const struct sig_pci_bus_funcs cb1_funcs = {
		.mr =		CHIP_(cb1_read),
		.mw =		CHIP_(cb1_write),
		.map_r =	CHIP_(cb1_map_r),
		.map_w =	CHIP_(cb1_map_w),
	};
	static const struct sig_std_logic_funcs cb1_int_funcs = {
		.boolean_or_set = CHIP_(cb1_irq_set),
	};
	static const struct sig_std_logic_funcs cb1_cd1_funcs = {
		.boolean_or_set = CHIP_(cb1_cd1_set),
	};
	static const struct sig_std_logic_funcs cb1_cd2_funcs = {
		.boolean_or_set = CHIP_(cb1_cd2_set),
	};
	static const struct sig_std_logic_funcs cb1_cvs1_funcs = {
		.boolean_or_set = CHIP_(cb1_cvs1_set),
	};
	static const struct sig_std_logic_funcs cb1_cvs2_funcs = {
		.boolean_or_set = CHIP_(cb1_cvs2_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->powered[0] = 0;
	cpssp->powered[1] = 0;

	/* Out */
	cpssp->port_int[0] = port_intA;
	sig_std_logic_connect_out(port_intA, cpssp, SIG_STD_LOGIC_L);

	cpssp->port_int[1] = port_intB;
	sig_std_logic_connect_out(port_intB, cpssp, SIG_STD_LOGIC_L);

	cpssp->port_cb_power[0] = port_cb0_power;
	sig_std_logic_connect_out(port_cb0_power, cpssp, SIG_STD_LOGIC_0);

	cpssp->port_cb_reset_hash_[0] = port_cb0_reset_hash_;

	cpssp->port_cb_power[1] = port_cb1_power;
	sig_std_logic_connect_out(port_cb1_power, cpssp, SIG_STD_LOGIC_0);

	cpssp->port_cb_reset_hash_[1] = port_cb1_reset_hash_;

	/* Pullup resistors */
	sig_std_logic_connect_out(port_cb0_cd1, cpssp, SIG_STD_LOGIC_H);
	sig_std_logic_connect_out(port_cb0_cd2, cpssp, SIG_STD_LOGIC_H);
	sig_std_logic_connect_out(port_cb0_cvs1, cpssp, SIG_STD_LOGIC_H);
	sig_std_logic_connect_out(port_cb0_cvs2, cpssp, SIG_STD_LOGIC_H);
	sig_std_logic_connect_out(port_cb1_cd1, cpssp, SIG_STD_LOGIC_H);
	sig_std_logic_connect_out(port_cb1_cd2, cpssp, SIG_STD_LOGIC_H);
	sig_std_logic_connect_out(port_cb1_cvs1, cpssp, SIG_STD_LOGIC_H);
	sig_std_logic_connect_out(port_cb1_cvs2, cpssp, SIG_STD_LOGIC_H);

	/* Call */
	sig_pci_idsel_connect(port_idsel, cpssp, &idsel_funcs);

	cpssp->port_pci_bus = port_pci_bus;
	sig_pci_bus_connect(port_pci_bus, cpssp, &pci_bus_funcs);

	cpssp->port_cb[0] = port_cb0;
	sig_pci_bus_connect(port_cb0, cpssp, &cb0_funcs);

	cpssp->port_cb[1] = port_cb1;
	sig_pci_bus_connect(port_cb1, cpssp, &cb1_funcs);

	/* In */
	sig_std_logic_connect_in(port_power, cpssp, &power_funcs);

	sig_std_logic_connect_in(port_reset_hash_, cpssp, &reset_hash__funcs);

	sig_std_logic_connect_in(port_cb0_int, cpssp, &cb0_int_funcs);

	sig_std_logic_connect_in(port_cb0_cd1, cpssp, &cb0_cd1_funcs);
	sig_std_logic_connect_in(port_cb0_cd2, cpssp, &cb0_cd2_funcs);
	sig_std_logic_connect_in(port_cb0_cvs1, cpssp, &cb0_cvs1_funcs);
	sig_std_logic_connect_in(port_cb0_cvs2, cpssp, &cb0_cvs2_funcs);

	sig_std_logic_connect_in(port_cb1_int, cpssp, &cb1_int_funcs);

	sig_std_logic_connect_in(port_cb1_cd1, cpssp, &cb1_cd1_funcs);
	sig_std_logic_connect_in(port_cb1_cd2, cpssp, &cb1_cd2_funcs);
	sig_std_logic_connect_in(port_cb1_cvs1, cpssp, &cb1_cvs1_funcs);
	sig_std_logic_connect_in(port_cb1_cvs2, cpssp, &cb1_cvs2_funcs);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fp);
}
