package storage

import (
	"context"
	"crypto/md5"
	"io"
	"log"
	"path/filepath"
	"sort"
	"strings"

	"github.com/pkg/errors"

	"github.com/google/go-cloud/blob"
	ucloudBlob "github.com/myml/cloud-blob/ucloud"
	upyunBlob "github.com/myml/cloud-blob/upyun"
)

const (
	// 最新版本
	VersionLatest = "latest"
)

var (
	ErrNotFound = errors.New("key or version does not Found")
)

func NewStorage(token TokenSource, hwID string) (*Storage, error) {
	backend, err := Backend(token)
	if err != nil {
		return nil, errors.Wrap(err, "Backend")
	}
	var bucket *blob.Bucket
	switch backend.Service {
	case "ucloud":
		auth := &deepinSyncAuth{
			cloud:       backend.Service,
			hwID:        hwID,
			tokenSource: token,
		}
		bucket = blob.NewBucket(ucloudBlob.OpenUcloudBucket(backend.Bucket, auth))
	case "upyun":
		auth := &deepinSyncAuth{
			hwID:        hwID,
			cloud:       backend.Service,
			tokenSource: token,
		}
		bucket = blob.NewBucket(upyunBlob.OpenUpyunBucket(backend.Bucket, auth))
	default:
		return nil, errors.New("Unknown service")
	}
	return &Storage{prefix: backend.PathPrefix, Bucket: bucket}, nil
}

// 设置值
//
// NewVersion为false时,覆写旧值
// NewVersion为true时,创建新版本,并返回版本号
func (s *Storage) Set(ctx context.Context, key string, value []byte, opts *SetOptions) (version string, err error) {
	if opts != nil && opts.NewVersion {
		version, err = s.NewVersion()
		if err != nil {
			return "", errors.Wrap(err, "NewVersion")
		}
	}
	md5sum := md5.Sum(value)
	err = s.Bucket.WriteAll(ctx, s.ToPath(key, version), value, &blob.WriterOptions{
		ContentMD5: md5sum[:],
	})
	if err != nil {
		return "", errors.Wrap(err, "WriteAll")
	}
	return version, nil
}

// 获取值
//
// Version选项指定获取版本
// Version为空时,获取无版本值
// Version为"latest"获取最新版本值
func (s *Storage) Get(ctx context.Context, key string, opts *GetOptions) (*GetResult, error) {
	var version string
	if opts != nil {
		version = opts.Version
	}
	if version == VersionLatest {
		var err error
		version, err = s.Latest(ctx, key)
		if err != nil {
			return nil, errors.Wrap(err, "Latest")
		}
	}
	b, err := s.Bucket.ReadAll(ctx, s.ToPath(key, version))
	if err != nil {
		if blob.IsNotExist(err) {
			return nil, ErrNotFound
		}
		return nil, errors.Wrap(err, "ReadAll")
	}
	return &GetResult{Content: b, Version: version}, nil
}

// 获取最新版本号
//
func (s *Storage) Latest(ctx context.Context, key string) (string, error) {
	list, err := s.List(ctx, &ListOptions{Prefix: key})
	if err != nil {
		return "", errors.Wrap(err, "List")
	}
	var listObjs []*ListObject
	for {
		obj, err := list.Next(ctx)
		if err != nil {
			if err == io.EOF {
				break
			}
			return "", errors.Wrap(err, "Next")
		}
		if obj == nil {
			break
		}
		log.Println(obj, "key", key)
		if obj.Key == key {
			listObjs = append(listObjs, obj)
		}
	}
	if len(listObjs) == 0 {
		return "", errors.Wrap(err, "Not found")
	}
	// 按版本从高到低排序
	sort.Slice(listObjs, func(i, j int) bool {
		return strings.Compare(listObjs[i].Version, listObjs[j].Version) > 0
	})
	return listObjs[0].Version, nil
}

// 删除键
//
// Version选项指定删除版本
// Version为空删除键值所有版本
func (s *Storage) Remove(ctx context.Context, key string, opts *DeleteOptions) error {
	if opts != nil && len(opts.Version) > 0 {
		return s.Bucket.Delete(ctx, s.ToPath(key, opts.Version))
	}
	list, err := s.List(ctx, &ListOptions{Prefix: key})
	if err != nil {
		return errors.Wrap(err, "List")
	}
	var count int
	for {
		obj, err := list.Next(ctx)
		if err != nil {
			if err == io.EOF {
				break
			}
			return errors.Wrap(err, "Next")
		}
		if obj == nil {
			break
		}
		if obj.Key == key {
			err = s.Bucket.Delete(ctx, s.ToPath(key, obj.Version))
			if err != nil {
				return errors.Wrap(err, "Delete")
			}
			count++
		}
	}
	if count == 0 {
		return errors.New("Not Found")
	}
	return nil
}

// 键列表
//
// Prefix可指定前缀
func (s *Storage) List(ctx context.Context, opts *ListOptions) (*ListIterator, error) {
	var prefix string
	if opts != nil {
		prefix = opts.Prefix
	}
	//	list, err := s.Bucket.List(ctx, &blob.ListOptions{Prefix: s.ToPath(prefix, "")})
	//	if err != nil {
	//		return nil, errors.Wrap(err, "List")
	//	}
	list := s.Bucket.List(&blob.ListOptions{Prefix: s.ToPath(prefix, "")})
	return &ListIterator{blobList: list, prefix: s.prefix}, nil
}

// 转换路径
func (s *Storage) ToPath(key, version string) string {
	str := filepath.Join(s.prefix, key)
	if len(version) > 0 {
		str += "." + version
	}
	log.Println(str)
	return str
}
