# -.- coding: utf-8 -.-
#
# Activity Log Manager
#
# Copyright © 2011 Stefano Candori <stefano.candori@gmail.com>
# Copyright © 2011 Seif Lotfy <seif@lotfy.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import gobject
import gio
import gtk
import pango
import logging
import time

from datetime import date

from zeitgeist.datamodel import Event, Subject, Interpretation
from remote import ZeitgeistInterface
ZG = ZeitgeistInterface()

__all__ = ["Window"]

log = logging.getLogger("Activity Log Manager")

class Window(gtk.Window):

    def __init__(self):
        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
        self.connect("destroy", gtk.main_quit)
        self.zg = ZG
        self.is_incognito = self.zg.get_incognito()
        self.incognito_disabled_text = _("Logging stopped")
        self.incognito_enabled_text = _("Logging active")
        self.set_border_width(6)
        self._set_up_ui()
        self.show_all()

    def _set_up_ui(self):
        self.set_title(_("Activity Log Manager"))
        self.set_default_size(600, 400)
        self.set_position(gtk.WIN_POS_CENTER_ALWAYS)
        
        self._notebook = gtk.Notebook()
        self._notebook.set_properties("tab-fill", True)
        self._notebook.set_properties("tab-expand", True)
        self._set_up_history_tab()
        self._set_up_applications_tab()
        self._set_up_file_tab()

        self.pix_indicator_disabled = gtk.image_new_from_stock(gtk.STOCK_NO,
                                      gtk.ICON_SIZE_LARGE_TOOLBAR)
        self.pix_indicator_enabled = gtk.image_new_from_stock(gtk.STOCK_YES,
                                     gtk.ICON_SIZE_LARGE_TOOLBAR)
        
        self.incognito_bar = IncognitoBar()
        if not self.zg.get_incognito():
            self.incognito_bar.set_text(self.incognito_enabled_text)
            self.incognito_bar.set_image(self.pix_indicator_enabled)
            self.incognito_bar.set_in_incognito(False)
        else:
            self.incognito_bar.set_text(self.incognito_disabled_text)
            self.incognito_bar.set_image(self.pix_indicator_disabled)
            self.incognito_bar.set_in_incognito(True)

        self.incognito_bar.connect('clicked', self._on_incognito_toggled)
        
        general_vbox = gtk.VBox(False)
        hbox = gtk.HBox()
        hbox.pack_end(self.incognito_bar, False, False)
        general_vbox.pack_end(hbox, False, False)
        general_vbox.pack_start(self._notebook)
        self.add(general_vbox)
        
        for child in self._notebook:
           self._notebook.child_set(child, "tab-expand", True)
           self._notebook.child_set(child, "tab-fill", True)

    def _set_up_history_tab(self):
        self.history_tab = HistoryTab()
        self._notebook.append_page(self.history_tab, gtk.Label(_("History")))

    def _set_up_applications_tab(self):
        self.applications_tab = ApplicationsTab()
        self._notebook.append_page(self.applications_tab,
            gtk.Label(_("Applications")))

    def _set_up_file_tab(self):
        self.file_tab = FileTab()
        self._notebook.append_page(self.file_tab, gtk.Label(_("Files")))

    def _on_incognito_toggled(self, *args):
        self.is_incognito = not self.is_incognito
        if self.is_incognito:
            self.incognito_bar.set_text(self.incognito_disabled_text)
            self.incognito_bar.set_image(self.pix_indicator_disabled)
            self.incognito_bar.set_in_incognito(True)
        else:
            self.incognito_bar.set_text(self.incognito_enabled_text)
            self.incognito_bar.set_image(self.pix_indicator_enabled)
            self.incognito_bar.set_in_incognito(False)
        self.zg.set_incognito(self.is_incognito)
            


class HistoryTab(gtk.VBox):

    _past_records = {
        900: _("15 minutes"),
        3600: _("hour"),
        7200: _("2 hours"),
        21600: _("6 hours"),
        86400: _("day"),
        604800: _("week"),
    }
    
    _custom_time_range = {
        0: None, #start
        1: None  #end
    }
    

    def __init__(self):
        gtk.VBox.__init__(self, spacing=4)
        self.zg = ZG
        self._set_up_ui()
        
    def _set_up_ui(self):
        self.set_border_width(10)
        label = gtk.Label()
        label.set_markup("<b>%s</b>" % _("Forget my activities for the last..."))
        label.set_alignment(0.0, 0.5)
        self.pack_start(label, False, False, 6)
        
        self._past_records_box = gtk.HBox(False, 6)
        self.pack_start(self._past_records_box, False, False, 6)
        
        self._combo_box = gtk.combo_box_new_text()
        temp_records = self._past_records.keys()
        temp_records.sort()
        for record in temp_records:
            self._combo_box.append_text(self._past_records[record])
        self._combo_box.set_active(0)
        
        self.clear_button = gtk.Button(stock=gtk.STOCK_DELETE)
        self.clear_button.connect("clicked", self._on_clear_button_clicked)
        self._past_records_box.pack_start(self._combo_box, False)
        self._past_records_box.pack_start(self.clear_button, False)

        separator = gtk.HSeparator()
        self.pack_start(separator, False, True)
        
        label = gtk.Label()
        label.set_markup("<b>%s</b>" % \
            _("Forget my activities in the following timerange:"))
        label.set_alignment(0.0, 0.5)
        self.pack_start(label, False, False, 6)
        
        markup = '<span size="smaller" font-weight="bold" >%s</span>'
        from_label = gtk.Label()
        from_label.set_markup(markup % _("From:"))
        from_label.set_alignment(0.0, 0.5)
        to_label = gtk.Label()
        to_label.set_markup(markup % _("To:"))
        to_label.set_alignment(0.0, 0.5)
        
        self.from_entry = gtk.Entry()
        self.from_entry.set_width_chars(30)
        self.from_button = gtk.Button()
        arrow = gtk.Arrow(gtk.ARROW_DOWN, gtk.SHADOW_IN)
        self.from_button.add(arrow)
        
        self.from_entry.set_editable(False)
        self.from_button.connect("clicked", self.show_calendar, 0)
        
        self.to_entry = gtk.Entry()
        self.to_entry.set_width_chars(30)
        self.to_button = gtk.Button()
        arrow = gtk.Arrow(gtk.ARROW_DOWN, gtk.SHADOW_IN)
        self.to_button.add(arrow)
        
        self.to_entry.set_editable(False)
        self.to_button.connect("clicked", self.show_calendar, 1)

        hbox = gtk.HBox(False)
        hbox.pack_start(self.from_entry, False)
        hbox.pack_start(self.from_button, False)
        vbox_from = gtk.VBox(False, 5)
        vbox_from.pack_start(from_label, False)
        vbox_from.pack_start(hbox, False)
        
        hbox = gtk.HBox(False)
        hbox.pack_start(self.to_entry, False)
        hbox.pack_start(self.to_button, False)
        vbox_to = gtk.VBox(False, 5)
        vbox_to.pack_start(to_label, False)
        vbox_to.pack_end(hbox, False)
        
        self.clear_button2 = gtk.Button(stock=gtk.STOCK_DELETE)
        self.clear_button2.connect("clicked", self._on_clear_button2_clicked)
        self.clear_button2.set_sensitive(False)
        align = gtk.Alignment(xalign=0.0)
        align.add(self.clear_button2)
        
        self._error_label = gtk.Label()
        
        hbox_delete = gtk.HBox(False, 4)
        hbox_delete.pack_start(align, False)
        hbox_delete.pack_start(self._error_label, False, padding=10)
        
        self.pack_start(vbox_from, False)
        self.pack_start(vbox_to, False)
        self.pack_start(hbox_delete, False, padding=10)
        
        #used for discriminate if the user is choosing the from or to date entry
        self._n_current_entry = -1
        
        class _Calendar(gtk.Dialog):
            def __init__(self):
                gtk.Dialog.__init__(self)
                self.calendar = gtk.Calendar()
                self.vbox.pack_start(self.calendar, False, False)
                self.set_decorated(False)
                self.set_position(gtk.WIN_POS_NONE)
                self.set_property("skip-taskbar-hint", True)
                self.connect("focus-out-event", self._on_focus_out)
                
            def _on_focus_out(self, *args):
                self.hide()
        
        self._calendar = _Calendar()
        self._calendar.calendar.connect("day-selected-double-click", 
                                        self._on_date_selected)
        
    def show_calendar(self, button, from_to):
        self._calendar.show_all()
        
        self._n_current_entry = from_to
        if from_to == 0:
            self._current_entry = self.from_entry
            entry_pos = self.from_entry.get_allocation()
        else:
            self._current_entry = self.to_entry
            entry_pos = self.to_entry.get_allocation()
            
        parent_pos = self.get_parent_window().get_position()
        self._calendar.move(parent_pos[0] + entry_pos.x,
                            parent_pos[1] + entry_pos.y + entry_pos.height)
    
    def _on_date_selected(self, calendar):
        year, month, day = calendar.get_date()
        selected_date = date(year, month + 1, day)
        str_date = selected_date.strftime("%A, %d %B %Y")
        
        self._error_label.hide()
        
        self._custom_time_range[self._n_current_entry] = selected_date
        self._current_entry.set_text(str_date)
        
        self.clear_button2.set_sensitive(True)
        check_flag = True
        for v in self._custom_time_range.values():
            if v is None: 
                self.clear_button2.set_sensitive(False)
                check_flag = False
                break
                
        if check_flag and self._custom_time_range[1]  <= self._custom_time_range[0]:
            markup = "<span color='red' size='smaller' weight='bold'>%s</span>"
            self._error_label.set_markup(markup % 
                _("Please select a regular time interval"))
            self._error_label.show()
            self.clear_button2.set_sensitive(False)

        self._calendar.hide()

    def _on_clear_button_clicked(self, button):
        diff = self._past_records.keys()
        diff.sort()
        end = time.time() * 1000
        key = diff[self._combo_box.get_active()]
        start = end - key * 1000
        text = _("Are you sure you want to delete your activity" \
               " history for the last <b>%s</b>?") % self._past_records[key]
        self._delete_events((start, end), text)

    def _on_clear_button2_clicked(self, button):
        start = time.mktime(self._custom_time_range[0].timetuple()) * 1000
        end = time.mktime(self._custom_time_range[1].timetuple()) * 1000 + 86399999
        
        start_str = self._custom_time_range[0].strftime("%A, %d %B %Y")
        end_str = self._custom_time_range[1].strftime("%A, %d %B %Y")
        text = _("Are you sure you want to delete your activity "\
               "history from <b>%s</b> to <b>%s</b>?") % (start_str, end_str)
        self._delete_events((start, end), text)
        
    def _delete_events(self, timerange, text):
        dialog = gtk.MessageDialog(self.parent.parent.parent, 
                                   gtk.DIALOG_DESTROY_WITH_PARENT, 
                                   gtk.MESSAGE_WARNING, gtk.BUTTONS_YES_NO)
        dialog.set_markup(text)
        dialog.set_title(_("Remove History"))
        dialog.show_all()
        self.parent.parent.parent.set_sensitive(False)
        
        def _callback(dialog, _id):
            dialog.destroy()
            def callback(ids):
                md = gtk.MessageDialog(self.parent.parent.parent, 
                     gtk.DIALOG_DESTROY_WITH_PARENT, gtk.MESSAGE_INFO, 
                     gtk.BUTTONS_CLOSE, _("Deleted events: %i") % len(ids))
                md.show_all()
                def _close(dialog, _id):
                    md.destroy()
                    self.parent.parent.parent.set_sensitive(True)
                    
                md.connect("response", _close)
                
            if id == gtk.RESPONSE_YES:
                self.zg.remove_history(timerange, callback)
            else:
                self.parent.parent.parent.set_sensitive(True)
                dialog.destroy()
                
        dialog.connect("response", _callback)
    

class ApplicationsTab(gtk.VBox):
    
    def __init__(self):
        gtk.VBox.__init__(self, spacing=10)
        self.app_chooser_dialog = ApplicationsChooserDialog()
        self.zg = ZG
        self._set_up_ui()
        self.buttons = {}

    def _set_up_ui(self):
        self.set_border_width(10)
        
        self.treeview = _ApplicationsTreeView()
        button_box = gtk.VButtonBox()
        button_box.set_layout(gtk.BUTTONBOX_START)
        
        self.add_button = gtk.Button(stock=gtk.STOCK_ADD)
        self.add_button.connect("clicked", self._on_add_application)
        button_box.pack_start(self.add_button)
        
        self.delete_button = gtk.Button(stock=gtk.STOCK_DELETE)
        self.delete_button.connect("clicked", self._on_delete_application)
        button_box.pack_start(self.delete_button)
        
        label = gtk.Label()
        label.set_markup("<b>%s</b>" % \
            _("Do not log the activity of these applications:"))
        align = gtk.Alignment(xalign=0, yalign=1)
        align.add(label)
        self.pack_start(align, False)
        
        general_hbox = gtk.HBox(False, 5)
        general_hbox.pack_start(self.treeview)
        general_hbox.pack_start(button_box, False)
        self.pack_start(general_hbox)
        
        self._applications = [t[4:] for t in self.zg.all_templates.iterkeys() \
            if t.startswith("app-")]
        self.populate_view(self._applications)
    
    def _is_duplicate(self, model, path, iter, user_data):
        if model.get_value(iter, 0) == user_data:
            self.duplicate = True
            return True

    def _on_add_application(self, button):
        self.duplicate = False   
        response = self.app_chooser_dialog.run()
        if response == gtk.RESPONSE_OK:
            app = self.app_chooser_dialog.treeview.get_selected_app()
            self.treeview.store.foreach(self._is_duplicate, app[0])
            if app is not None and not self.duplicate:
                self.treeview.store.append(app)
                e = Event()
                e.actor = "application://"+app[3]
                self.zg.add_blacklist_template("app-"+app[3], e)
                
        self.app_chooser_dialog.hide()

    def _on_delete_application(self, button):
        app = self.treeview.remove_selected_app()[1]
        self.zg.remove_blacklist_template("app-"+app)

    #
    # INTERFACE METHODS
    #
    #FIXME better name?
    def populate_view(self, app_list):
        """Populates the applications list.
        
        Every application in app_list parameter should be in the form:
        application://*.desktop
        """
        for app in app_list:
            try:
                app_info =  gio.unix.DesktopAppInfo(app)
                app_name = app_info.get_name()
                self.duplicate = False
                self.treeview.store.foreach(self._is_duplicate, app_name)
                if app_info is not None and not self.duplicate:
                    pix = self.treeview._get_pixbuf_from_gio_icon(app_info.get_icon())
                    self.treeview.store.append([app_name, app_info.get_description(), pix, app_info.get_id()])
            except RuntimeError:
                log.warning(_("Failed to load %s") % app)
                continue

class FileTab(gtk.VBox):
    
    mime_dict = {
        _("Audio"): Interpretation.AUDIO.uri,
        _("Video"): Interpretation.VIDEO.uri,
        _("Image"): Interpretation.IMAGE.uri,
        _("Text"): Interpretation.DOCUMENT.uri,
        _("Presentation"): Interpretation.DOCUMENT.PRESENTATION.uri,
        _("Spreadsheet"): Interpretation.DOCUMENT.SPREADSHEET.uri, 
        _("Instant Messaging"): Interpretation.IMMESSAGE.uri,
        _("E-mail"): Interpretation.EMAIL.uri,
        _("Website"): Interpretation.WEBSITE.uri
    }
    
    inverse_mime_dict = dict((v, k) for (k, v) in {1:2, 3:4}.iteritems())
     
    blacklist_dict = {
    Interpretation.AUDIO.uri : False,
    Interpretation.VIDEO.uri : False,
    Interpretation.IMAGE.uri : False,
    Interpretation.DOCUMENT.uri : False,
    Interpretation.DOCUMENT.PRESENTATION.uri : False,
    Interpretation.DOCUMENT.SPREADSHEET.uri : False,
    Interpretation.IMMESSAGE.uri : False,
    Interpretation.EMAIL.uri : False,
    Interpretation.WEBSITE.uri: False
    }
    
    dirs_blacklist_dict = {}
    
    def __init__(self):
        gtk.VBox.__init__(self, spacing=10)
        self.zg = ZG
        self._set_up_ui()

    def _set_up_ui(self):
        self.set_border_width(10)
        
        label = gtk.Label()
        label.set_markup("<b>%s</b>" % \
            _("Prevent logging for the following file types:"))
        align = gtk.Alignment(xalign=0, yalign=1)
        align.add(label)
        self.pack_start(align, False)
        
        checkbox_table = gtk.Table(3,3)
        i = 0
        self.button_list = []
        for t in sorted(self.mime_dict.keys()):
            button = gtk.CheckButton(t)
            self.button_list.append(button)
            if "interpretation-%s" % t.lower() in self.zg.all_templates:
                button.set_active(True)
            button.connect("toggled", self._on_ckeckbox_toggled, t)
            if i == 3: checkbox_table.attach(button, 0, 1, 1, 2)
            elif i == 7: checkbox_table.attach(button, 1, 2, 2, 3)
            else: checkbox_table.attach(button, i%3, (i%3) + 1 , i%4, (i%4) + 1)
            i += 1
        
        self.pack_start(checkbox_table, False)

        label = gtk.Label()
        label.set_markup("<b>%s</b>" % \
            _("Prevent logging files in the following folders:"))
        align = gtk.Alignment(xalign=0, yalign=1)
        align.add(label)
        self.pack_start(align, False)
        
        file_chooser_hbox = gtk.HBox(False, 5)
        file_chooser_dialog = gtk.FileChooserDialog(
            _("Select a directory to blacklist"),
            action = gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER,
            buttons = (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL,
                gtk.RESPONSE_CANCEL))
        self.file_chooser = gtk.FileChooserButton(file_chooser_dialog)
        file_chooser_hbox.pack_start(self.file_chooser, True)
        add_button = gtk.Button(stock=gtk.STOCK_ADD)
        add_button.connect("clicked", self._on_add_blacklist_dir)
        #file_chooser_hbox.pack_end(add_button, False)
        #self.pack_start(file_chooser_hbox, False)
        
        #Store fields:
        #1) Directory's Path
        self.store = gtk.ListStore(str)
        self.treeview = gtk.TreeView(self.store)
        self.treeview.set_headers_visible(False)
        
        column_pix_name = gtk.TreeViewColumn('Name')
        self.treeview.append_column(column_pix_name)
        pix_rend = gtk.CellRendererPixbuf()
        column_pix_name.pack_start(pix_rend, False)
        pix_rend.set_property("stock-id", gtk.STOCK_DIRECTORY)
        name_rend = gtk.CellRendererText()
        column_pix_name.pack_start(name_rend)
        column_pix_name.add_attribute(name_rend, "text", 0)
        column_pix_name.set_resizable(True)
        column_pix_name.set_max_width(350)
        
        scroll = gtk.ScrolledWindow()
        scroll.add(self.treeview)
        scroll.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
        scroll.set_shadow_type(gtk.SHADOW_IN)
        scroll.set_border_width(1)
        
        hbox_other = gtk.HBox(False, 5)
        button_box = gtk.VButtonBox()
        button_box.set_layout(gtk.BUTTONBOX_START)
        remove_button = gtk.Button(stock=gtk.STOCK_DELETE)
        remove_button.connect("clicked", self._on_remove_blacklist_dir)
        #hbox_other.pack_start(scroll, True)
        #button_box.pack_start(remove_button)
        #hbox_other.pack_start(button_box, False)
        
        #self.pack_start(hbox_other, True)
        
        hbox = gtk.HBox(False, 6)
        self.pack_start(hbox)
        
        vbox = gtk.VBox()
        hbox.pack_start(vbox)
        vbox.pack_start(file_chooser_hbox, False, False)
        vbox.pack_start(scroll, True, True)
        
        vbox = gtk.VBox()
        hbox.pack_start(vbox, False, False)
        vbox.pack_start(add_button, False, False)
        vbox.pack_start(remove_button, False, False)
        
        for t in self.zg.all_templates.iterkeys():
            if t.startswith("dir-"):
                self.store.append([t[4:]])
    
    def _on_ckeckbox_toggled(self, checkbox, label):
        is_active = checkbox.get_active()
        self.blacklist_dict[self.mime_dict[label]] = is_active
        if is_active:
            e = Event()
            s = Subject()
            s.interpretation = self.mime_dict[label]
            e.subjects = [s]
            self.zg.add_blacklist_template("interpretation-"+label.lower(), e)
        else:
            self.zg.remove_blacklist_template("interpretation-"+label.lower())

    def _on_add_blacklist_dir(self, button):
        def is_duplicate(model, path, iter, user_data):
            if model.get_value(iter, 0) == user_data:
                self.duplicate = True
                return True
            
        self.duplicate = False
        filename = self.file_chooser.get_filename()
        self.store.foreach(is_duplicate, filename)
        if not self.duplicate:
            self.dirs_blacklist_dict[filename] = True
            self.store.append([filename])
            e = Event()
            s = Subject()
            s.uri = "file://" + filename + "/*"
            e.subjects = [s]
            self.zg.add_blacklist_template("dir-"+filename, e)

    def _on_remove_blacklist_dir(self, button):
        selection = self.treeview.get_selection()
        model,_iter = selection.get_selected()
        if _iter is not None:
            directory = model[_iter][0]
            self.zg.remove_blacklist_template("dir-"+directory)
            return model.remove(_iter), directory

    #
    # INTERFACE METHODS
    #
    def populate_view(self, other_dir_list):
        """Populates the "Other files" treeview.
        
        Every items in other_dir_list parameter should be in the form:
        file://*
        """
        #other directories population
        for dir_ in other_dir_list:
            if not dir_.startswith("file://"):
                log.warning(_("%s should be in form file://*") % dir_)
                continue
            else:
                dir_ = dir_[7:]
            if not os.path.exists(dir_):
                log.warning(_("%s doesn't exists.") % dir_)
                continue
            self.dirs_blacklist_dict[dir_] = True
            self.store.append([dir_,])
    
    def get_blacklisted_items(self):
        """Returns a list of blacklisted driectories and MIMETypes. 
        
        Every directory in the list it's in the form: file://*
        """
        return ["file://" + key for key, value in \
               self.dirs_blacklist_dict.iteritems() if value], \
               [key for key, value in self.blacklist_dict.iteritems() if value]

class _ApplicationsTreeView(gtk.VBox):
    
    def __init__(self):
        gtk.VBox.__init__(self)
        self._icon_theme = gtk.icon_theme_get_default()
        #Store fields:
        #1) Name
        #2) Description
        #3) Pixbuf
        #4) Desktop file
        self.store = gtk.ListStore(str, str, gtk.gdk.Pixbuf, str)
        self.treeview = gtk.TreeView(self.store)
        self.treeview.set_headers_visible(False)
        self.treeview.set_rules_hint(True)
        self._set_up_ui()

    def _set_up_ui(self):
        column_pix_name = gtk.TreeViewColumn(_('Name'))
        self.treeview.append_column(column_pix_name)
        pix_rend = gtk.CellRendererPixbuf()
        column_pix_name.pack_start(pix_rend)
        column_pix_name.add_attribute(pix_rend, "pixbuf", 2)
        name_rend = gtk.CellRendererText()
        column_pix_name.pack_start(name_rend)
        column_pix_name.add_attribute(name_rend, "text", 0)
        column_pix_name.set_resizable(True)
        column_pix_name.set_max_width(350)
        
        column_desc = gtk.TreeViewColumn(_('Description'))
        self.treeview.append_column(column_desc)
        desc_rend = gtk.CellRendererText()
        desc_rend.set_property("ellipsize", pango.ELLIPSIZE_END)
        column_desc.pack_start(desc_rend)
        column_desc.add_attribute(desc_rend, "text", 1)
        
        scroll = gtk.ScrolledWindow()
        scroll.add(self.treeview)
        scroll.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
        scroll.set_shadow_type(gtk.SHADOW_IN)
        scroll.set_border_width(1)
        self.pack_start(scroll)
    
    def _get_pixbuf_from_gio_icon(self, gio_icon):
        icon_info = self._icon_theme.lookup_by_gicon(gio_icon, 32, gtk.ICON_LOOKUP_FORCE_SVG)
        if icon_info is not None:
            try:
                pix = icon_info.load_icon()
            except gio.Error:
                return None
        else:
            pix = None
        return pix
    
    def _populate_view(self):
        for app in gio.app_info_get_all():
            if app.should_show():
                icon = app.get_icon()
                if icon is not None:
                    pix = self._get_pixbuf_from_gio_icon(icon)
                else:
                    pix = None
                self.store.append([app.get_name(), app.get_description(), pix, app.get_id()])

    def get_selected_app(self):
        selection = self.treeview.get_selection()
        model,_iter = selection.get_selected()
        if _iter is not None:
            return model.get(_iter, 0, 1, 2, 3)
        return None
    
    def remove_selected_app(self):
        selection = self.treeview.get_selection()
        model,_iter = selection.get_selected()
        if _iter is not None:
            app = model[_iter][3]
            return model.remove(_iter), app

class ApplicationsChooserDialog(gtk.Dialog):
    
    def __init__(self):
        gtk.Dialog.__init__(self)
        self.set_title(_("Select Application"))
        self.set_destroy_with_parent(True)
        self.set_size_request(600, 400)
        self.set_skip_taskbar_hint(True)
        self._set_up_ui()

    def _set_up_ui(self):
        self.treeview = _ApplicationsTreeView()
        area = self.get_content_area()
        area.pack_start(self.treeview)
        self.add_buttons(gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
        
        self.treeview._populate_view()
        self.treeview.show_all()

class IncognitoBar(gtk.Button):
    '''A class used to display messages in a non-intrusive bar'''
    
    def __init__(self):
        gtk.Button.__init__(self)
        self.message_label = gtk.Label()
        self.message_image = gtk.Image()
        
        self.set_relief(gtk.RELIEF_NONE)
        self.markup = '<span size="smaller" font-weight="bold">%s</span>'
        
        self.message_hbox = gtk.HBox()
        #self.message_hbox.pack_start(gtk.Label())
        self.message_hbox.pack_end(self.message_image, False, False, 3)
        self.message_hbox.pack_end(self.message_label, False, False, 3)
        #self.message_hbox.pack_start(gtk.Label())
        self.add(self.message_hbox)

    def set_text(self, text):
        if text is not None:
            self.message_label.set_markup(self.markup % text)
        
    def set_image(self, pix):
        if pix is not None:
            for i in self.message_hbox:
                self.message_hbox.remove(i)
            
            self.message_image = pix
            #self.message_hbox.pack_start(gtk.Label())
            self.message_hbox.pack_end(self.message_image, False, False, 3)
            self.message_hbox.pack_end(self.message_label, False, False, 3)
            #self.message_hbox.pack_start(gtk.Label())
        self.show_all()
    
    def set_in_incognito(self, in_incognito):
        pass
