# rat_tabbed --
#
# Create a tabbed notebook display. This code is a slightly modified
# example from the following book:
# ----------------------------------------------------------------------
#  Effective Tcl/Tk Programming
#    Mark Harrison, DSC Communications Corp.
#    Michael McLennan, Bell Labs Innovations for Lucent Technologies
#    Addison-Wesley Professional Computing Series
# ======================================================================
#  Copyright (c) 1996-1997  Lucent Technologies Inc. and Mark Harrison
# ======================================================================

package provide rat_tabbed 1.0

namespace eval rat_tabbed {
    variable nbInfo
    variable tnInfo

    namespace export create page display

    global tk_version
    if [info exists tk_version] {
	option add *TkRat.tabMargin 6 widgetDefault
    }
}


proc rat_tabbed::nb_create {win} {
    variable nbInfo

    frame $win -class TkRat
    pack propagate $win 0

    set nbInfo($win-count) 0
    set nbInfo($win-pages) ""
    set nbInfo($win-current) ""
    return $win
}

proc rat_tabbed::nb_page {win name} {
    variable nbInfo

    set page "$win.page[incr nbInfo($win-count)]"
    lappend nbInfo($win-pages) $page
    set nbInfo($win-page-$name) $page

    frame $page

    if {$nbInfo($win-count) == 1} {
        after idle [list rat_tabbed::nb_display $win $name]
    }
    return $page
}

proc rat_tabbed::nb_display {win name} {
    variable nbInfo

    set page ""
    if {[info exists nbInfo($win-page-$name)]} {
        set page $nbInfo($win-page-$name)
    } elseif {[winfo exists $win.page$name]} {
        set page $win.page$name
    }
    if {$page == ""} {
        error "bad notebook page \"$name\""
    }

    rat_tabbed::nb_fix_size $win

    if {$nbInfo($win-current) != ""} {
        pack forget $nbInfo($win-current)
    }
    pack $page -side left -anchor n
    set nbInfo($win-current) $page
}

proc rat_tabbed::nb_fix_size {win} {
    variable nbInfo

    update idletasks

    set maxw 0
    set maxh 0
    foreach page $nbInfo($win-pages) {
        set w [winfo reqwidth $page]
        if {$w > $maxw} {
            set maxw $w
        }
        set h [winfo reqheight $page]
        if {$h > $maxh} {
            set maxh $h
        }
    }
    set bd [$win cget -borderwidth]
    set maxw [expr $maxw+2*$bd]
    set maxh [expr $maxh+2*$bd]
    $win configure -width $maxw -height $maxh
}

proc rat_tabbed::create {win} {
    variable tnInfo

    frame $win -class TkRat
    canvas $win.tabs -highlightthickness 0
    pack $win.tabs -fill x

    rat_tabbed::nb_create $win.notebook
    pack $win.notebook -expand yes -fill both

    set tnInfo($win-tabs) ""
    set tnInfo($win-current) ""
    set tnInfo($win-pending) ""
    return $win
}

proc rat_tabbed::page {win name {cmd {}}} {
    variable tnInfo

    set page [rat_tabbed::nb_page $win.notebook $name]
    lappend tnInfo($win-tabs) $name
    set tnInfo($win-cmd-$name) $cmd

    if {$tnInfo($win-pending) == ""} {
        set id [after idle [list rat_tabbed::refresh $win]]
        set tnInfo($win-pending) $id
    }
    return $page
}

proc rat_tabbed::refresh {win} {
    global propNormFont
    variable tnInfo

    $win.tabs delete all

    set margin [option get $win tabMargin Margin]
    set color [option get $win disabledForeground Color]
    set x 2
    set maxh 0

    foreach name $tnInfo($win-tabs) {
        set id [$win.tabs create text \
            [expr $x+$margin+2] [expr -0.5*$margin] \
            -anchor sw -text $name -font $propNormFont \
            -tags [list $name]]

        set bbox [$win.tabs bbox $id]
        set wd [expr [lindex $bbox 2]-[lindex $bbox 0]]
        set ht [expr [lindex $bbox 3]-[lindex $bbox 1]]
        if {$ht > $maxh} {
            set maxh $ht
        }

        $win.tabs create polygon 0 0  $x 0 \
            [expr $x+$margin] [expr -$ht-$margin] \
            [expr $x+$margin+$wd] [expr -$ht-$margin] \
            [expr $x+$wd+2*$margin] 0 \
            2000 0  2000 10  0 10 \
            -outline black -fill $color \
            -tags [list $name tab tab-$name]

        $win.tabs raise $id

        $win.tabs bind $name <ButtonPress-1> \
            [list rat_tabbed::display $win $name]

        set x [expr $x+$wd+2*$margin]
    }
    set height [expr $maxh+2*$margin]
    $win.tabs move all 0 $height

    $win.tabs configure -width $x -height [expr $height+4]

    if {$tnInfo($win-current) != ""} {
        rat_tabbed::display $win $tnInfo($win-current)
    } else {
        rat_tabbed::display $win [lindex $tnInfo($win-tabs) 0]
    }
    set tnInfo($win-pending) ""
}

proc rat_tabbed::display {win name} {
    variable tnInfo

    rat_tabbed::nb_display $win.notebook $name

    set normal [option get $win disabledForeground Color]
    $win.tabs itemconfigure tab -fill $normal

    set active [option get $win background Color]
    $win.tabs itemconfigure tab-$name -fill $active
    $win.tabs raise $name

    set tnInfo($win-current) $name
    if { "" != $tnInfo($win-cmd-$name)} {
	uplevel #0 $tnInfo($win-cmd-$name)
    }
}
