/*
 * ratStdFolder.c --
 *
 *      This file contains code which implements standard c-client folders.
 *	This means ONLY filefolders at this moment.
 *
 * TkRat software and its included text is Copyright 1996-2000 by
 * Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratStdFolder.h"

/*
 * We use this structure to keep a list of open connections
 */

typedef struct Connection {
    MAILSTREAM *stream;		/* Handler to c-client entity */
    int *errorFlagPtr;		/* Address of flag to set on hard errors */
    RatStdFolderType type;	/* Type of folder */
    char *host;			/* Host we are connected to */
    char *user;			/* User we are connected as */
    unsigned long port;		/* Port we are connected to */
    int refcount;		/* references count */
    int closing;		/* True if this connection is unused and
				   waiting to be closed */
    Tcl_TimerToken token;	/* Timer token for closing timer */
    struct Connection *next;	/* Struct linkage */
    StdFolderInfo *stdPtr;	/* StdFolder pointer */
} Connection;


/*
 * Remember if we must initialize the package
 */
static int initialize = 1;

/*
 * List of open connections
 */
static Connection *connListPtr = NULL;

/*
 * The values below are used to catch calls to mm_log. That is when you
 * want to handle the message internally.
 */
static RatLogLevel logLevel;
static char *logMessage = NULL;
static int logIgnore = 0;

/*
 * These variables are used by mm_login
 */
static char loginUser[MAILTMPLEN];
static char loginPassword[MAILTMPLEN];
static char loginHost[MAILTMPLEN];
static int loginStore;

/*
 * List of flag names
 */
static char *stdFlagNames[] = {
    RAT_SEEN_STR,
    RAT_DELETED_STR,
    RAT_FLAGGED_STR,
    RAT_ANSWERED_STR,
    RAT_DRAFT_STR,
    NULL
};

/*
 * The types of folders. This list is closely related to RatStdFolderType
 * in ratStdFolder.h
 */
static char *ratStdTypeNames[] = {"unix", "imap", "pop3", "mh", "mbx"};

/*
 * This is used to build a list of found mailboxes when listing
 */
typedef struct Mailbox {
    char *name;			/* The name of this mailbox */
    char *spec;			/* The specification to send to c-client */
    long attributes;		/* The attributes from c-client */
    struct Mailbox *next;	/* Pointer to the next mailbox on this level */
    struct Mailbox *child;	/* Pointer to subfolders */
} Mailbox;
static Mailbox *mailboxListPtr = NULL;

/*
 * Used to store search results
 */
long *searchResultPtr = NULL;
int searchResultSize = 0;
int searchResultNum = 0;

/*
 * Used to store status results
 */
MAILSTATUS stdStatus;

/*
 * File handler of debugging file
 */
static FILE *debugFile = NULL;

/*
 * Procedures private to this module.
 */
void Std_StreamClose(Tcl_Interp *interp, MAILSTREAM *stream);
static RatInitProc Std_InitProc;
static RatCloseProc Std_CloseProc;
static RatUpdateProc Std_UpdateProc;
static RatInsertProc Std_InsertProc;
static RatSetFlagProc Std_SetFlagProc;
static RatGetFlagProc Std_GetFlagProc;
static Tcl_TimerProc CloseConnection;
static Tcl_CmdProc StdJudgeFolder;
static Tcl_ObjCmdProc StdImportCmd;
static Connection *FindConn(MAILSTREAM *stream);
static void StdImportBuildResult(Tcl_Interp *interp, Tcl_Obj *listPtr,
	Tcl_Obj **defv, Mailbox *mPtr, RatStdFolderType type, char *user);


/*
 *----------------------------------------------------------------------
 *
 * RatStdFolderInit --
 *
 *      Initializes the file folder command.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	the result area.
 *
 * Side effects:
 *	The C-client library is initialized and the apropriate mail drivers
 *	are linked.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatStdFolderInit(Tcl_Interp *interp)
{
    mail_link(&unixdriver);
    mail_link(&mmdfdriver);
    mail_link(&imapdriver);
    mail_link(&pop3driver);
    mail_link(&mhdriver);
    mail_link(&mbxdriver);
    mail_link(&dummydriver);
    auth_link(&auth_log);
    auth_link(&auth_md5);
    Tcl_CreateCommand(interp, "RatImapCreateFolder", StdJudgeFolder,
	    (ClientData)1, NULL);
    Tcl_CreateCommand(interp, "RatImapDeleteFolder", StdJudgeFolder,
	    (ClientData)0, NULL);
    Tcl_CreateObjCommand(interp, "RatImport", StdImportCmd, NULL, NULL);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_StreamOpen --
 *
 *      Opens a standard c-client mailstream. This function handles
 *	caching of passwords and connections.
 *
 * Results:
 *	The mail stream.
 *
 * Side effects:
 *	The caches may be modified.
 *
 *----------------------------------------------------------------------
 */

MAILSTREAM*
Std_StreamOpen(Tcl_Interp *interp, char *name, char *user, long options,
	int *errorFlagPtr)
{
    MAILSTREAM *stream = NULL;
    Connection *connPtr = NULL;
    char *host = NULL, *cPtr, *pPtr, *filename;
    RatStdFolderType type = RAT_UNIX;
    unsigned int port = 0;
    Tcl_DString ds;
    int debug;

    if ('{' == name[0]) {
	RatStrNCpy(loginUser, user, sizeof(loginUser));
	host = cpystr(name+1);
	for (cPtr = host;
		*cPtr && ':' != *cPtr && '/' != *cPtr && '}' != *cPtr;
		cPtr++);
	if (':' == *cPtr) {
	    port = atoi(cPtr+1);
	}
	for (pPtr = cPtr; *pPtr && '/' != *pPtr && '}' != *pPtr; pPtr++);
	if ('/' == *pPtr
		&& ('p' == pPtr[1] || 'P' == pPtr[1])
		&& ('o' == pPtr[2] || 'O' == pPtr[2])
		&& ('p' == pPtr[3] || 'P' == pPtr[3])
		&& '3' == pPtr[4]) {
	    type = RAT_POP;
	    if (0 == port) port = 110;
	} else {
	    type = RAT_IMAP;
	    if (0 == port) port = 143;
	}
	*cPtr = '\0';
	for (connPtr = connListPtr; connPtr; connPtr = connPtr->next) {
	    if ((connPtr->closing || options & OP_HALFOPEN)
		    && connPtr->port == port
		    && !strcmp(host, connPtr->host)
		    && !strcmp(connPtr->user, user)) {
		break;
	    }
	}
	if (connPtr) {
	    stream = connPtr->stream;
	    connPtr->refcount++;
	    if (connPtr->token) {
		Tcl_DeleteTimerHandler(connPtr->token);
	    }
	    connPtr->closing = 0;
	}
	name = RatUtf8toMutf7(name);
    }
    if (stream && options & OP_HALFOPEN) {
	if (host) {
	    ckfree(host);
	}
	return stream;
    }
    Tcl_GetBooleanFromObj(interp,
	    Tcl_GetVar2Ex(timerInterp, "option", "debug_cclient",
	    TCL_GLOBAL_ONLY), &debug);
    if (debug) {
	options |= OP_DEBUG;
	filename = Tcl_GetVar2(interp, "option", "debug_file", TCL_GLOBAL_ONLY);
	filename = Tcl_TranslateFileName(interp, filename, &ds);
	if (filename) {
	    debugFile = fopen(filename, "a");
	    if (debugFile) {
		fchmod(fileno(debugFile), 0600);
	    }
	    Tcl_DStringFree(&ds);
	}
    }
    loginPassword[0] = '\0';
    stream = mail_open(stream, name, options);
    logIgnore = 0;
    if (stream && !connPtr) {
	if (!strcmp(stream->dtb->name, "mbx")) {
	    type = RAT_MBX;
	}
	connPtr = (Connection*)ckalloc(sizeof(Connection));
	connPtr->stream = stream;
	connPtr->errorFlagPtr = errorFlagPtr;
	connPtr->type = type;
	connPtr->host = cpystr(host ? host : "");
	connPtr->user = cpystr(user ? user : "");
	connPtr->port = port;
	connPtr->refcount = 1;
	connPtr->closing = 0;
	connPtr->stdPtr = NULL;
	connPtr->next = connListPtr;
	connListPtr = connPtr;
    }
    if (loginPassword[0] != '\0') {
	RatCachePassword(timerInterp, loginHost, port, loginUser,
		ratStdTypeNames[type], loginPassword, loginStore);
	memset(loginPassword, 0, strlen(loginPassword));
    }
    if (host) {
	ckfree(host);
    }
    return stream;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_StreamClose --
 *
 *      Closes a standard c-client mailstream. This function handles
 *	caching of passwords and connections.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The caches may be modified.
 *
 *----------------------------------------------------------------------
 */

void
Std_StreamClose(Tcl_Interp *interp, MAILSTREAM *stream)
{
    Connection *connPtr;

    for (connPtr = connListPtr;
	    connPtr && stream != connPtr->stream;
	    connPtr = connPtr->next);
    if (connPtr) {
	int timeout, doCache;

	if (--connPtr->refcount) {
	    return;
	}
	Tcl_GetBoolean(timerInterp,
		Tcl_GetVar2(timerInterp, "option", "cache_conn",
		TCL_GLOBAL_ONLY), &doCache);
	if (doCache && RAT_IMAP == connPtr->type) {
	    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option",
		    "cache_conn_timeout", TCL_GLOBAL_ONLY), &timeout);
	    connPtr->closing = 1;
	    if (timeout) {
		connPtr->token = Tcl_CreateTimerHandler(timeout*1000,
			CloseConnection, (ClientData)connPtr);
	    } else {
		connPtr->token = NULL;
	    }
	} else {
	    CloseConnection((ClientData)connPtr);
	}
    } else {
	logIgnore = 1;
	mail_close_full(stream, NIL);
	logIgnore = 0;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * OpenStdFolder --
 *
 *      Opens a standard c-client folder and if it is a filefolder and
 *	is of an incompatible format (unfortunately generated by an older
 *	version of this program) we convert it.
 *
 * Results:
 *	The mail stream.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

MAILSTREAM*
OpenStdFolder(Tcl_Interp *interp, char *name, char *prot, char *user,
	void *voidPtr)
{
    MAILSTREAM *stream = NULL;
    RatStdFolderType type;
    StdFolderInfo *stdPtr = (StdFolderInfo*)voidPtr;
    Tcl_DString dsBuf;
    char *host, *cPtr;
    struct stat sbuf;
    int i, isNotYet = 0, dsBufUse = 0;

    if ( '{' == name[0] ) {
	for (i=1; '}' != name[i] && '/' != name[i]; i++);
	host = (char*)ckalloc(i);
	RatStrNCpy(host, &name[1], i);

	if (!strcasecmp(prot, "pop3")) {
	    type = RAT_POP;
	} else {
	    type = RAT_IMAP;
	}
    } else {
	host = NULL;
	if ('#' == name[0]) {
	    type = RAT_MH;
	} else {
	    type = RAT_UNIX;
	    name = Tcl_UtfToExternalDString(NULL, name, -1, &dsBuf);
	    dsBufUse = 1;
	}
    }
    if ('/' == name[0] && stat(name, &sbuf) && ENOENT == errno) {
	/*
	 * This entry points to a file which doesn't exist. Check if the
	 * directory exists and in that case we accept.
	 */
	for (cPtr = name+strlen(name); '/' != *cPtr; cPtr--);
	*cPtr = '\0';
	if (stat(name, &sbuf) || !S_ISDIR(sbuf.st_mode)) {
	    *cPtr = '/';
	    Tcl_AppendResult(interp, "Failed to open std mailbox \"",
		    name, "\"", (char *) NULL);
	    return NULL;
	}
	isNotYet = 1;
	*cPtr = '/';
    } else {
	logLevel = RAT_BABBLE;
	stream = Std_StreamOpen(interp, name, user, 0, NULL);
	if (logLevel > RAT_WARN) {
	    if (host) {
		ckfree(host);
	    }
	    Tcl_SetResult(interp, logMessage, TCL_VOLATILE);
	    return NULL;
	}
	if (NIL == stream) {
	    if (host) {
		ckfree(host);
	    }
	    Tcl_AppendResult(interp, "Failed to open std mailbox \"",
		    name, "\"", (char *) NULL);
	    return NULL;
	}
	if (!strcmp(stream->dtb->name, "mbx")) {
	    type = RAT_MBX;
	}
    }
    if (stdPtr) {
	stdPtr->stream = stream;
	stdPtr->referenceCount = 1;
	stdPtr->exists = isNotYet ? 0 : stream->nmsgs;
	stdPtr->isNotYet = isNotYet;
	stdPtr->origName = cpystr(name);
	stdPtr->type = type;
	stdPtr->host = host;
	stdPtr->user = loginUser ? cpystr(loginUser) : NULL;
    } else if (host && *host) {
	ckfree(host);
    }
    if (dsBufUse) {
	Tcl_DStringFree(&dsBuf);
    }
    return stream;
}


/*
 *----------------------------------------------------------------------
 *
 * RatStdFolderCreate --
 *
 *      Creates a std folder entity.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	the result area.
 *
 * Side effects:
 *	A std folder is created.
 *
 *
 *----------------------------------------------------------------------
 */

RatFolderInfo*
RatStdFolderCreate(Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[])
{
    RatFolderInfo *infoPtr;
    StdFolderInfo *stdPtr;
    MAILSTREAM *stream = NULL;
    Connection *connPtr;
    char buf[32];
    int i;

    /*
     * Now it is time to initialize things
     */
    if (initialize) {
	env_parameters(SET_LOCALHOST, currentHost);
	initialize = 0;
    }
    if (objc != 4 && objc != 6) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " ", Tcl_GetString(objv[1]),
		" flags name [user prot]\"", (char *) NULL);
	return (RatFolderInfo *) NULL;
    }

    stdPtr = (StdFolderInfo *) ckalloc(sizeof(*stdPtr));
    stdPtr->isNotYet = 0;

    if (6 == objc) {
	stream = OpenStdFolder(interp, Tcl_GetString(objv[3]),
		Tcl_GetString(objv[5]), Tcl_GetString(objv[4]), stdPtr);
    } else {
	stream = OpenStdFolder(interp, Tcl_GetString(objv[3]),NULL,NULL,stdPtr);
    }
    if (!stream && !stdPtr->isNotYet) {
	ckfree(stdPtr);
	return NULL;
    }

    infoPtr = (RatFolderInfo *) ckalloc(sizeof(*infoPtr)); 

    infoPtr->name = cpystr(Tcl_GetString(objv[3]));
    infoPtr->type = "std";
    infoPtr->size = -1;
    if (stdPtr->isNotYet) {
	infoPtr->number = 0;
	infoPtr->recent = 0;
	infoPtr->unseen = 0;
    } else {
	infoPtr->number = stream->nmsgs;
	infoPtr->recent = stream->recent;
	infoPtr->unseen = 0;
	if (stream->nmsgs) {
	    sprintf(buf, "1:%ld", stream->nmsgs);
	    mail_fetchfast_full(stream, buf, NIL);
	    for (i = 1; i <= stream->nmsgs; i++)
		  if (!mail_elt (stream,i)->seen) infoPtr->unseen++; 
	}
	connPtr = FindConn(stream);
	connPtr->stdPtr = stdPtr;
    }
    infoPtr->initProc = Std_InitProc;
    infoPtr->closeProc = Std_CloseProc;
    infoPtr->updateProc = Std_UpdateProc;
    infoPtr->insertProc = Std_InsertProc;
    infoPtr->setFlagProc = Std_SetFlagProc;
    infoPtr->getFlagProc = Std_GetFlagProc;
    infoPtr->infoProc = Std_InfoProc;
    infoPtr->setInfoProc = Std_SetInfoProc;
    infoPtr->createProc = Std_CreateProc;
    infoPtr->syncProc = NULL;
    infoPtr->private = (ClientData) stdPtr;

    return infoPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_InitProc --
 *
 *      See the documentation for initProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for initProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static void
Std_InitProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;
    MessageInfo *msgPtr;
    int i, j, start, end;

    if (-1 == index) {
       start = 0;
       end = infoPtr->number;
    } else {
       start = index;
       end = start+1;
    }
    for (i=start; i<end; i++) {
	msgPtr = (MessageInfo*)ckalloc(sizeof(MessageInfo));
	msgPtr->folderInfoPtr = infoPtr;
	msgPtr->name[0] = '\0';
	msgPtr->type = RAT_CCLIENT_MESSAGE;
	msgPtr->bodyInfoPtr = NULL;
	msgPtr->msgNo = i;
	msgPtr->fromMe = RAT_ISME_UNKOWN;
	msgPtr->toMe = RAT_ISME_UNKOWN;
	msgPtr->clientData = NULL;
	for (j=0; j<RAT_FOLDER_END; j++) {
	    msgPtr->info[j] = NULL;
	}
	infoPtr->privatePtr[i] = (ClientData)msgPtr;
    }
    RatStdMsgStructInit(infoPtr, interp, index, stdPtr->stream, stdPtr->type,
	    stdPtr->host, stdPtr->user);
}

/*
 *----------------------------------------------------------------------
 *
 * CloseStdFolder --
 *
 *      See the documentation for closeProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for closeProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
void
CloseStdFolder(Tcl_Interp *interp, MAILSTREAM *stream)
{
    Std_StreamClose(interp, stream);
}


/*
 *----------------------------------------------------------------------
 *
 * Std_CloseProc --
 *
 *      See the documentation for closeProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for closeProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Std_CloseProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int expunge)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;
    MessageInfo *msgPtr;
    int i, j;

    if (stdPtr->isNotYet) {
	ckfree(stdPtr);
	return TCL_OK;
    }
    if (stdPtr->stream) {
	if (expunge) {
	    mail_expunge(stdPtr->stream);
	}
	Std_StreamClose(interp, stdPtr->stream);
    }
    if (0 == --stdPtr->referenceCount) {
	if (stdPtr->host) {
	    ckfree(stdPtr->host);
	}
	ckfree(stdPtr->origName);
	for (i=0; i<infoPtr->number; i++) {
	    if (NULL == infoPtr->msgCmdPtr[i]) {
		msgPtr = (MessageInfo*)infoPtr->privatePtr[i];
		if (msgPtr) {
		    for (j=0; j<RAT_FOLDER_END; j++) {
			if (msgPtr->info[j]) {
			    Tcl_DecrRefCount(msgPtr->info[j]);
			    msgPtr->info[j] = NULL;
			}
		    }
		    ckfree(msgPtr->clientData);
		    ckfree(infoPtr->privatePtr[i]);
		}
	    }
	}
	if (stdPtr->user) {
	    ckfree(stdPtr->user);
	}
	ckfree(stdPtr);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_UpdateProc --
 *
 *      See the documentation for updateProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for updateProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Std_UpdateProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, RatUpdateType mode)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;
    int numNew = 0, oldExists, i;
    MESSAGECACHE *elt;
    char sequence[16];

    if (stdPtr->isNotYet) {
	Connection *connPtr;
	struct stat sbuf;

	if (stat(stdPtr->origName, &sbuf)) {
	    return 0;
	}
	logLevel = RAT_BABBLE;
	stdPtr->stream = Std_StreamOpen(interp, stdPtr->origName, NULL, 0,NULL);
	if (logLevel > RAT_WARN) {
	    Tcl_SetResult(interp, logMessage, TCL_VOLATILE);
	    return 0;
	}
	if (NIL == stdPtr->stream) {
	    Tcl_AppendResult(interp, "Failed to open std mailbox \"",
		    stdPtr->origName, "\"", (char *) NULL);
	    return 0;
	}
	stdPtr->exists = stdPtr->stream->nmsgs;
	connPtr = FindConn(stdPtr->stream);
	connPtr->stdPtr = stdPtr;
	stdPtr->isNotYet = 0;
	numNew = stdPtr->exists;
    } else if (stdPtr->stream) {
	if (RAT_SYNC == mode) {
	    MESSAGECACHE *cachePtr;
	    MessageInfo *msgPtr;
	    int i, offset = 0;

	    if (infoPtr->number) {
		for (i=0; i<infoPtr->number; i++) {
		    cachePtr = mail_elt(stdPtr->stream, i+1);
		    if (cachePtr->deleted) {
			if (-1 != infoPtr->size) {
			    infoPtr->size -= cachePtr->rfc822_size;
			}
			if (infoPtr->msgCmdPtr[i]) {
			    RatMessageDelete(interp, infoPtr->msgCmdPtr[i]);
			}
			offset++;
		    } else if (offset) {
			infoPtr->msgCmdPtr[i-offset] = infoPtr->msgCmdPtr[i];
			infoPtr->privatePtr[i-offset] = infoPtr->privatePtr[i];
			if (infoPtr->privatePtr[i]) {
			    msgPtr = (MessageInfo*)infoPtr->privatePtr[i];
			    msgPtr->msgNo = i - offset;
			}
		    }
		}
		for (i=infoPtr->number-offset; i<infoPtr->number; i++) {
		    infoPtr->msgCmdPtr[i] = NULL;
		    infoPtr->privatePtr[i] = NULL;
		}
	    }
	    mail_expunge(stdPtr->stream);
	    numNew = stdPtr->exists - (infoPtr->number - offset);

	} else if (RAT_CHECKPOINT == mode) {
	    oldExists = infoPtr->number;
	    mail_check(stdPtr->stream);
	    numNew = stdPtr->exists-oldExists;
	} else {
	    oldExists = infoPtr->number;
	    if (T != mail_ping(stdPtr->stream)) {
		char buf[1024];
		stdPtr->stream = NIL;
		snprintf(buf, sizeof(buf), "%s close 1", infoPtr->cmdName);
		Tcl_GlobalEval(interp, buf);
		Tcl_SetResult(interp, "Mailbox stream died", TCL_STATIC);
		Tcl_SetErrorCode(interp, "C_CLIENT", "streamdied", NULL);
		return -1;
	    }
	    numNew = stdPtr->exists-oldExists;
	}
    } else {
	return 0;
    }
    if (numNew) {
	sprintf(sequence, "%d:%d", stdPtr->exists-numNew+1, stdPtr->exists);
	mail_fetchfast_full(stdPtr->stream, sequence, NIL);
    }
    infoPtr->number = stdPtr->exists;
    infoPtr->recent = stdPtr->stream->recent;
    for (i = 1,infoPtr->unseen=0; i <= stdPtr->stream->nmsgs; i++) {
	if (!mail_elt(stdPtr->stream,i)->seen) infoPtr->unseen++;
    }
    return numNew;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_InsertProc --
 *
 *      See the documentation for insertProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for insertProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Std_InsertProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int argc,
	char *argv[])
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;
    char flags[128], date[128];
    Tcl_CmdInfo cmdInfo;
    Tcl_DString ds;
    STRING string;
    int i;

    if (stdPtr->isNotYet) {
	Connection *connPtr;
	int perm;

	Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "permissions",
		TCL_GLOBAL_ONLY), &perm);
	close(open(stdPtr->origName, O_CREAT, perm));
	logLevel = RAT_BABBLE;
	stdPtr->stream = Std_StreamOpen(interp, stdPtr->origName, NULL, 0,NULL);
	if (logLevel > RAT_WARN) {
	    Tcl_SetResult(interp, logMessage, TCL_VOLATILE);
	    return TCL_ERROR;
	}
	if (stdPtr->stream) {
	    stdPtr->exists = stdPtr->stream->nmsgs;
	    connPtr = FindConn(stdPtr->stream);
	    connPtr->stdPtr = stdPtr;
	    stdPtr->isNotYet = 0;
	}
    }
    if (NIL == stdPtr->stream) {
	Tcl_AppendResult(interp, "Failed to open std mailbox \"",
		argv[2], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    Tcl_DStringInit(&ds);
    for (i=0; i<argc; i++) {
	Tcl_GetCommandInfo(interp, argv[i], &cmdInfo);
	RatMessageGet(interp, (MessageInfo*)cmdInfo.objClientData,
		      &ds, flags, sizeof(flags), date, sizeof(date));
	INIT(&string,mail_string,Tcl_DStringValue(&ds),Tcl_DStringLength(&ds));
	RatPurgeFlags(flags);
	if (!mail_append_full(stdPtr->stream, RatUtf8toMutf7(stdPtr->origName),
			      flags, date, &string)){
	    Tcl_SetResult(interp, "mail_append failed", TCL_STATIC);
	    return TCL_ERROR;
	}
	Tcl_DStringSetLength(&ds, 0);
	if (!stdPtr->exists) {
	    if (T != mail_ping(stdPtr->stream)) {
		char buf[1024];
		Tcl_DStringFree(&ds);
		snprintf(buf, sizeof(buf), "%s close", infoPtr->cmdName);
		Tcl_GlobalEval(interp, buf);
		Tcl_SetResult(interp, "Mailbox stream died", TCL_STATIC);
		Tcl_SetErrorCode(interp, "C_CLIENT", "streamdied", NULL);
		return TCL_ERROR;
	    }
	}
    }
    Tcl_DStringFree(&ds);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_SetFlagProc --
 *
 *      See the documentation for setFlagProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for setFlagProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Std_SetFlagProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index,
	RatFlag flag, int value)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;
    MessageInfo *msgPtr = (MessageInfo*)infoPtr->privatePtr[index];
    MESSAGECACHE *cachePtr;
    char sequence[8];
    int wasseen;

    if (!stdPtr->stream || stdPtr->stream->rdonly || NULL==stdFlagNames[flag]) {
	return TCL_OK;
    }

    if (msgPtr->info[RAT_FOLDER_STATUS]) {
	Tcl_DecrRefCount(msgPtr->info[RAT_FOLDER_STATUS]);
	msgPtr->info[RAT_FOLDER_STATUS] = NULL;
    }

    cachePtr = mail_elt(stdPtr->stream, index+1);
    wasseen = cachePtr->seen;
    sprintf(sequence, "%d", index+1);
    if (value) {
	mail_setflag_full(stdPtr->stream, sequence, stdFlagNames[flag], NIL);
    } else {
	mail_clearflag_full(stdPtr->stream, sequence, stdFlagNames[flag], NIL);
    }
    (void)mail_fetchenvelope(stdPtr->stream, index+1);
    cachePtr = mail_elt(stdPtr->stream, index+1);
    switch (flag) {
	case RAT_SEEN:	   
		if (wasseen != value) {
		    if (wasseen) {
			infoPtr->unseen++;
		    } else {
			infoPtr->unseen--;
		    }
		}
		cachePtr->seen = value; break;
		break;
	case RAT_DELETED:  cachePtr->deleted = value; break;
	case RAT_FLAGGED:  cachePtr->flagged = value; break;
	case RAT_ANSWERED: cachePtr->answered = value; break;
	case RAT_DRAFT:	   cachePtr->draft = value; break;
	case RAT_RECENT:   cachePtr->recent = value; break;
    }
    infoPtr->recent = stdPtr->stream->recent;
    if (logLevel > RAT_WARN) {
	Tcl_SetResult(interp, logMessage, TCL_VOLATILE);
	return TCL_ERROR;
    } else {
	return TCL_OK;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * Std_GetFlagProc --
 *
 *      See the documentation for getFlagProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for getFlagProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Std_GetFlagProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index,
	RatFlag flag)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;
    MESSAGECACHE *cachePtr;
    char sequence[8];
    int value = 0;

    if (!stdPtr->stream) return 0;

    sprintf(sequence, "%d", index+1);
    logLevel = RAT_BABBLE;
    (void)mail_fetchstructure_full(stdPtr->stream, index+1, NIL, NIL);
    cachePtr = mail_elt(stdPtr->stream, index+1);
    switch (flag) {
    case RAT_SEEN:	value = cachePtr->seen; break;
    case RAT_DELETED:	value = cachePtr->deleted; break;
    case RAT_FLAGGED:	value = cachePtr->flagged; break;
    case RAT_ANSWERED:	value = cachePtr->answered; break;
    case RAT_DRAFT:	value = cachePtr->draft; break;
    case RAT_RECENT:	value = cachePtr->recent; break;
    }
    return value;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_InfoProc --
 *
 *      See the documentation for infoProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for infoProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */

Tcl_Obj*
Std_InfoProc(Tcl_Interp *interp, ClientData clientData, RatFolderInfoType type,
	int index)
{
    RatFolderInfo *infoPtr = (RatFolderInfo*)clientData;

    return Std_GetInfoProc(interp, (ClientData)infoPtr->privatePtr[index],
	    type, 0);
}


/*
 *----------------------------------------------------------------------
 *
 * Std_SetInfoProc --
 *
 *      See the documentation for setInfoProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for setInfoProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */

void
Std_SetInfoProc(Tcl_Interp *interp, ClientData clientData,
	RatFolderInfoType type, int index, Tcl_Obj *oPtr)
{
    RatFolderInfo *infoPtr = (RatFolderInfo*)clientData;
    MessageInfo *msgPtr = (MessageInfo*)infoPtr->privatePtr[index];

    if (msgPtr->info[type]) {
	Tcl_DecrRefCount(msgPtr->info[type]);
    }
    msgPtr->info[type] = oPtr;
    if (oPtr) {
	Tcl_IncrRefCount(oPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Std_CreateProc --
 *
 *      See the documentation for createProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for createProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
char*
Std_CreateProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;

    return RatStdMessageCreate(interp, infoPtr, stdPtr->stream, index,
	    stdPtr->type, stdPtr->host, stdPtr->user);
}


/*
 *----------------------------------------------------------------------
 *
 * StdImportCmd --
 *
 *      Import folders (via mm_list)
 *
 * Results:
 *	The folders found are returned as a list
 *
 * Side effects:
 *	RatLogin may be called.
 *
 *
 *----------------------------------------------------------------------
 */
static int
StdImportCmd(ClientData dummy, Tcl_Interp *interp, int objc,
	Tcl_Obj *const objv[])
{
    char spec[1024], buf[1024], ref[1024], *user = NULL, *pattern;
    MAILSTREAM *stream = NULL;
    int i, defc, subscribed = 0, delimiter;
    Tcl_Obj *oPtr, *lPtr, **defv;
    RatStdFolderType type;

    if (objc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " def\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Build search specification
     */
    ref[0] = '\0';
    Tcl_ListObjGetElements(interp, objv[1], &defc, &defv);
    if (!strcasecmp("dir", Tcl_GetString(defv[1]))) {
	type = RAT_UNIX;
	RatStrNCpy(spec, Tcl_GetString(defv[5]), sizeof(spec));
	snprintf(ref, sizeof(ref), "%s/",
		Tcl_GetVar2(interp, "env", "HOME",TCL_GLOBAL_ONLY));
	delimiter = '/';
    } else if (!strcasecmp("mh", Tcl_GetString(defv[1])) ) {
	type = RAT_MH;
	strcpy(spec, "#mh");
	RatStrNCpy(spec+3, Tcl_GetString(defv[5]), sizeof(spec)-3);
	delimiter = '/';
    } else if (!strcasecmp("imap", Tcl_GetString(defv[1]))
    	    || !strcasecmp("dis", Tcl_GetString(defv[1])) ) {
	char *host, *mbox, *port;
	NAMESPACE ***nsPtrPtrPtr;

	if (!strcasecmp("imap", Tcl_GetString(defv[1]))) {
	    type = RAT_IMAP;
	} else {
	    type = RAT_DIS;
	}
	host = Tcl_GetString(defv[5]);
	user = Tcl_GetString(defv[6]);
	mbox = Tcl_GetString(defv[7]);
	port = Tcl_GetString(defv[8]);
	snprintf(spec, sizeof(spec), "{%s%s%s}%s", host, (*port ? ":" : ""),
		(*port ? port : ""), mbox);
	stream  = Std_StreamOpen(interp, spec, user, OP_HALFOPEN, NULL);
	if (NIL == stream) {
	    Tcl_ResetResult(interp);
	    return TCL_OK;
	}
	if (strlen(mbox)) {
	    nsPtrPtrPtr = (NAMESPACE***)mail_parameters(stream,GET_NAMESPACE,0);
	    if (*nsPtrPtrPtr) {
		delimiter = (**nsPtrPtrPtr)->delimiter;
	    } else {
		/* I guess a '.' is the best choice here since that is
		 * what cyrus uses, uwimap supports the NAMESPACE extension
		 * and should have been handled by the above clause. */
		delimiter = '.';
	    }
	} else {
	    delimiter = '\0';
	}
    } else {
	Tcl_SetResult(interp, "Illegal protocol in import", TCL_STATIC);
	return TCL_ERROR;
    }

    /*
     * Parse flags
     */
    Tcl_ListObjIndex(interp, objv[1], 2, &lPtr);
    Tcl_ListObjLength(interp, lPtr, &i);
    for (i--; i>=0; i--) {
	Tcl_ListObjIndex(interp, lPtr, i, &oPtr);
	if (!strcmp("subscribed", Tcl_GetString(oPtr))) {
	    subscribed = 1;
	    break;
	}
    }

    /*
     * Do actual search
     */
    pattern = Tcl_GetString(defv[4]);
    if (!delimiter || spec[strlen(spec)-1] == delimiter) {
	snprintf(buf, sizeof(buf), "%s%s", spec, pattern);
    } else {
	snprintf(buf, sizeof(buf), "%s%c%s", spec, delimiter, pattern);
    }
    if (subscribed) {
	mail_lsub(stream, ref, buf);
    } else {
	mail_list(stream, ref, buf);
    }
    if (stream) {
	Std_StreamClose(interp, stream);
    }

    /*
     * Build result
     */
    lPtr = Tcl_NewListObj(0, NULL);
    StdImportBuildResult(interp, lPtr, defv, mailboxListPtr, type, user);
    mailboxListPtr = NULL;
    Tcl_SetObjResult(interp, lPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * StdImportBuildResult --
 *
 *      Recursive function which builds the result of StdImportCmd
 *
 * Results:
 *	None
 *
 * Side effects:
 *	The resulting list is left in the listPtr list object
 *
 *
 *----------------------------------------------------------------------
 */

static void
StdImportBuildResult(Tcl_Interp *interp, Tcl_Obj *listPtr, Tcl_Obj **defv,
	Mailbox *mPtr, RatStdFolderType type, char *user)
{
    Tcl_Obj *vPtr[10], *ePtr;
    Mailbox *nPtr;
    int vc;

    for (; mPtr; mPtr = nPtr) {
	if (mPtr->child) {
	    vPtr[vc=0] = Tcl_NewStringObj("s", 1);
	    vPtr[++vc] = Tcl_NewStringObj(mPtr->name, -1);
	    vPtr[++vc] = Tcl_NewListObj(0, NULL);
	    StdImportBuildResult(interp, vPtr[vc], defv,mPtr->child,type,user);
	    Tcl_ListObjAppendElement(interp, listPtr,Tcl_NewListObj(++vc,vPtr));
	}
	if (0 == (mPtr->attributes & LATT_NOSELECT)) {
	    vPtr[vc=0] = Tcl_NewStringObj(mPtr->name, -1);
	    switch(type) {
		case RAT_UNIX:
		    vPtr[++vc] = Tcl_NewStringObj("file", 4);
		    break;
		case RAT_MBX:
		    vPtr[++vc] = Tcl_NewStringObj("mbx", 4);
		    break;
		case RAT_MH:
		    vPtr[++vc] = Tcl_NewStringObj("mh", 2);
		    break;
		case RAT_IMAP:
		    vPtr[++vc] = Tcl_NewStringObj("imap", 4);
		    break;
		case RAT_DIS:
		    vPtr[++vc] = Tcl_NewStringObj("dis", 3);
		    RatDisPrepareDir(interp, mPtr->name, mPtr->spec, user,
				     "imap");
		    break;
		case RAT_POP:
		    /* Can not happen */
		    break;
	    }
	    vPtr[++vc] = defv[3];
	    vPtr[++vc] = Tcl_NewStringObj(mPtr->spec, -1);
	    if (RAT_IMAP == type || RAT_DIS == type) {
		vPtr[++vc] = defv[6];
	    }
	    ePtr = Tcl_NewListObj(++vc, vPtr);
	    vPtr[vc=0] = Tcl_NewStringObj("v", 1);
	    vPtr[++vc] = ePtr;
	    Tcl_ListObjAppendElement(interp,listPtr,Tcl_NewListObj(++vc,vPtr));
	}
	nPtr = mPtr->next;
	ckfree(mPtr);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * CloseConnection --
 *
 *      Closes a connection.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The connection list is modified.
 *
 *
 *----------------------------------------------------------------------
 */
static void
CloseConnection(ClientData clientData)
{
    Connection **connPtrPtr, *connPtr = (Connection*)clientData;

    logIgnore = 1;
    mail_close_full(connPtr->stream, NIL);
    logIgnore = 0;
    for (connPtrPtr = &connListPtr; *connPtrPtr != connPtr;
	    connPtrPtr = &(*connPtrPtr)->next);
    *connPtrPtr = connPtr->next;
    ckfree(connPtr->host);
    ckfree(connPtr->user);
    ckfree(connPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * StdJudgeFolder --
 *
 *      Create or delete c-client folders
 *
 * Results:
 *	A standard Tcl-result
 *
 * Side effects:
 *	A folder may be created or deleted
 *
 *
 *----------------------------------------------------------------------
 */
static int
StdJudgeFolder(ClientData op, Tcl_Interp *interp, int argc, char *argv[])
{
    MAILSTREAM *stream;
    char *name;

    if (3 != argc) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" folder user\"", (char *) NULL);
	return TCL_ERROR;
    }

    stream = Std_StreamOpen(interp, argv[1], argv[2], OP_HALFOPEN, NULL);
    if (!stream) {
	Tcl_SetResult(interp, "Failed to open stream to server", TCL_STATIC);
	return TCL_ERROR;
    }
    name = RatUtf8toMutf7(argv[1]);
    if (op) {
	logIgnore = 1;
	(void)mail_create(stream, name);
	logIgnore = 0;
    } else {
	(void)mail_delete(stream, name);
    }
    Std_StreamClose(interp, stream);

    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * AppendToIMAP --
 *
 *      Append the given message to an IMAP folder
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The specified folder will be modified
 *
 *
 *----------------------------------------------------------------------
 */

void
AppendToIMAP(Tcl_Interp *interp, char *mailboxSpec, char *user, char *flags,
	     char *date, char *msg, int length)
{
    char *mailbox;
    MAILSTREAM *stream;
    STRING msgString;

    mailbox = RatLindex(interp, mailboxSpec, 0);
    if (NULL == (stream = Std_StreamOpen(interp, mailbox, user, 0, NULL))) {
	return;
    }

    INIT(&msgString, mail_string, msg, length);
    mail_append_full(stream, RatUtf8toMutf7(mailbox), flags, date, &msgString);

    Std_StreamClose(interp, stream);
}

/*
 *----------------------------------------------------------------------
 *
 * FindConn --
 *
 *      Find the connection pointer for the stream
 *
 * Results:
 *	A connection pointer (or NULL)
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */

static Connection*
FindConn(MAILSTREAM *stream)
{
    Connection *connPtr;

    for (connPtr=connListPtr;
	    connPtr && connPtr->stream != stream;
	    connPtr=connPtr->next);
    return connPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * mm_*
 *
 *	The functions below are called from the C-client library. They
 *	are docuemnted in Internal.DOC.
 *
 *----------------------------------------------------------------------
 */
void mm_searched (MAILSTREAM *stream,unsigned long number)
{
    if (searchResultSize == searchResultNum) {
	searchResultSize += 1024;
	searchResultPtr = (long*)ckrealloc(searchResultPtr,
		searchResultSize*sizeof(long));
    }
    searchResultPtr[searchResultNum++] = number;
}


void mm_exists (MAILSTREAM *stream,unsigned long number)
{
    Connection *connPtr = FindConn(stream);

    if (connPtr && connPtr->stdPtr) {
	connPtr->stdPtr->exists = (int)number;
    }
}


void mm_expunged (MAILSTREAM *stream,unsigned long number)
{
    Connection *connPtr = FindConn(stream);

    if (connPtr && connPtr->stdPtr) {
	connPtr->stdPtr->exists -= 1;
    }
}


void mm_mailbox (char *string)
{
}


void mm_bboard (char *string)
{
}


void mm_notify (MAILSTREAM *stream,char *string,long errflg)
{
    long flag = errflg;

    if (flag == BYE) {
	Connection *connPtr = FindConn(stream);
	if (connPtr && connPtr->errorFlagPtr) {
	    *connPtr->errorFlagPtr = 1;
	}
    }
    mm_log(string, flag);
}


void mm_log (char *string,long errflg)
{
    if (logIgnore) {
	return;
    }

    switch(errflg) {
    case NIL:	logLevel = RAT_BABBLE; break;
    case PARSE:	logLevel = RAT_PARSE; break;
    case WARN:	logLevel = RAT_WARN; break;
    case BYE:	logLevel = RAT_FATAL; break;
    case ERROR:	/* fallthrough */
    default:	logLevel = RAT_ERROR; break;
    }

    if (logMessage) {
	ckfree(logMessage);
    }
    logMessage = cpystr(string);
    RatLog(timerInterp, logLevel, string, RATLOG_NOWAIT);
}


void mm_dlog (char *string)
{
    if (debugFile) {
	fprintf(debugFile, "%s\n", string);
	fflush(debugFile);
    }
    RatLog(timerInterp, RAT_BABBLE, string, RATLOG_TIME);
}


void mm_login (NETMBX *mbPtr, char *user, char *pwd, long trial)
{
    char **largv, buf[1024], *pw;
    Tcl_DString ds;
    int largc;

    /*
     * Check for cached entry
     */
    if ((pw = RatGetCachedPassword(timerInterp, mbPtr->host, mbPtr->port,
	    loginUser, mbPtr->service))) {
	RatStrNCpy(user, loginUser, MAILTMPLEN);
	RatStrNCpy(pwd, pw, MAILTMPLEN);
	return;
    }
    Tcl_DStringInit(&ds);
    Tcl_DStringAppendElement(&ds, "RatLogin");
    Tcl_DStringAppendElement(&ds, mbPtr->host);
    sprintf(buf, "%ld", trial);
    Tcl_DStringAppendElement(&ds, buf);
    Tcl_DStringAppendElement(&ds, loginUser?loginUser:"");
    Tcl_DStringAppendElement(&ds, mbPtr->service?mbPtr->service:"");
    sprintf(buf, "%ld", mbPtr->port);
    Tcl_DStringAppendElement(&ds, buf);
    if (TCL_OK != Tcl_Eval(timerInterp, Tcl_DStringValue(&ds))
	    || TCL_OK != Tcl_SplitList(timerInterp,
		    Tcl_GetStringResult(timerInterp), &largc,&largv)
	    || 3 != largc) {
	Tcl_DStringFree(&ds);
	pwd[0] = '\0';
	return;
    }
    RatStrNCpy(user, largv[0], MAILTMPLEN);
    RatStrNCpy(pwd, largv[1], MAILTMPLEN);
    if ('\0' != largv[0][0]) {
	RatStrNCpy(loginUser, largv[0], MAILTMPLEN);
	RatStrNCpy(loginPassword, largv[1], MAILTMPLEN);
	RatStrNCpy(loginHost, mbPtr->host, MAILTMPLEN);
	Tcl_GetBoolean(timerInterp, largv[2], &loginStore);
    } else {
	/* User pressed cancel */
	loginStore = 0;
	logIgnore = 1;
    }
    Tcl_DStringFree(&ds);
    ckfree(largv);
}


void mm_critical (MAILSTREAM *stream)
{
}


void mm_nocritical (MAILSTREAM *stream)
{
}


long mm_diskerror (MAILSTREAM *stream,long errcode,long serious)
{
    char buf[64];

    sprintf(buf, "Disk error: %ld", errcode);
    RatLog(timerInterp, RAT_FATAL, buf, RATLOG_TIME);
    return 1;
}


void mm_fatal (char *string)
{
    RatLog(timerInterp, RAT_FATAL, string, RATLOG_TIME);
}

void mm_flags (MAILSTREAM *stream,unsigned long number)
{
}


void
mm_list(MAILSTREAM *stream, int delimiter, char *spec, long attributes)
{
    Mailbox **mPtrPtr = &mailboxListPtr, *nPtr;
    char *name, *s, *e;

    /*
     * Create new Mailbox structure
     */
    if (!delimiter || (NULL == (name = strrchr(spec, delimiter)))) {
	if ((name = strchr(spec, '}'))) {
	    name++;
	} else {
	    name = spec;
	}
    } else {
       name++;
    }
    if (!*name && !(attributes & LATT_NOSELECT)) {
	return;
    }

    /*
     * First find the right level
     */
    if ((s = strchr(spec, '}'))) {
	s++;
    } else {
	s = spec;
    }
    for (; delimiter && (e = strchr(s, delimiter)); *e = delimiter, s = e+1) {
	*e = '\0';
	if (!strlen(s)) {
	    continue;
	}
	while (*mPtrPtr && 0 > strcmp((*mPtrPtr)->name, s)) {
	    mPtrPtr = &(*mPtrPtr)->next;
	}
	if (!*mPtrPtr || strcmp((*mPtrPtr)->name, s)) {
	    nPtr = (Mailbox*)ckalloc(sizeof(Mailbox)+strlen(s)*3+1);
	    nPtr->name = (char*)nPtr+sizeof(Mailbox);
	    strcpy(nPtr->name, RatMutf7toUtf8(s));
	    nPtr->spec = NULL;
	    nPtr->attributes = LATT_NOSELECT;
	    nPtr->next = *mPtrPtr;
	    nPtr->child = NULL;
	    *mPtrPtr = nPtr;
	    mPtrPtr = &nPtr->child;
	} else {
	    mPtrPtr = &(*mPtrPtr)->child;
	}
    }

    if (attributes & LATT_NOSELECT) {
	return;
    }

    /*
     * Find location and link it
     */
    while (*mPtrPtr && 0 > strcmp((*mPtrPtr)->name, name)) {
	mPtrPtr = &(*mPtrPtr)->next;
    }

    /*
     * Create actual folder entry
     */
    nPtr = (Mailbox*)ckalloc(sizeof(Mailbox)+strlen(name)*3+strlen(spec)*3+2);
    nPtr->name = (char*)nPtr+sizeof(Mailbox);
    strcpy(nPtr->name, RatMutf7toUtf8(name));
    nPtr->spec = nPtr->name+strlen(nPtr->name)+1;
    strcpy(nPtr->spec, RatMutf7toUtf8(spec));
    nPtr->attributes = attributes;
    nPtr->next = *mPtrPtr;
    nPtr->child = NULL;
    *mPtrPtr = nPtr;
}


void
mm_lsub (MAILSTREAM *stream, int delimiter, char *name, long attributes)
{
    mm_list(stream, delimiter, name, attributes);
}


void mm_status (MAILSTREAM *stream, char *mailbox, MAILSTATUS *status)
{
    memcpy(&stdStatus, status, sizeof(MAILSTATUS));
}

#ifdef MEM_DEBUG
void ratStdFolderCleanup()
{
    if (logMessage) {
	ckfree(logMessage);
    }
}
#endif /* MEM_DEBUG */
