/*
   CIPE - encrypted IP over UDP tunneling

   device.c - the net device driver

   Copyright 1996 Olaf Titz <olaf@bigred.inka.de>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version
   2 of the License, or (at your option) any later version.
*/
/* $Id: device.c,v 1.22 1999/04/25 16:42:59 olaf Exp $ */

#include "cipe.h"
#include "version.h"
#include <stddef.h>
#include <linux/if.h>
#include <linux/if_arp.h>
#include <linux/sched.h>

#ifdef LINUX_21
#include <asm/uaccess.h>
#include <linux/rtnetlink.h>
#else
#define register_netdevice register_netdev
#define unregister_netdevice unregister_netdev
#endif

/*** Globals ***/

static const char driver_version[]=VERSION;

struct cipe_ctrl **cipe_ctrls = NULL;
#ifdef NO_DYNDEV
int cipe_maxdev = 4;            /* changeable via insmod */
#else
int cipe_maxdev = 100;          /* changeable via insmod */
#endif
#ifdef DEBUG
int cipe_debug = DEB_CALL;      /* changeable via insmod */
#endif

/* clear all potentially sensitive info and stats */
static void cipe_zero_c(struct cipe *c)
{
    memset(&(c->peeraddr), 0,
           offsetof(struct cipe, udp_prot)-offsetof(struct cipe, peeraddr));
    /* reset these to sensible values */
    c->tmo_keyxchg = 10*HZ;
    c->tmo_keylife = 10*60*HZ;
}

/* weak but fast PRNG, used for padding only */
static __u32 prnseed;
void cipe_prnpad(unsigned char *buf, int len)
{
    while (len>0) {
	prnseed=prnseed*0x01001001+1;
	if (len>=2) {
	    *(__u16 *)buf=prnseed>>16;
	    len-=2; buf+=2;
	} else {
	    *buf=(prnseed>>24)^jiffies; return;
	}
    }
}


/*** IOCTL handlers ***/

static int cipe_getpar(struct device *dev, struct siocgifcippar *parm)
{
    DEVTOCIPE(dev,c,-ENODEV);

    parm->sockshost=c->sockshost;
    parm->socksport=c->socksport;
    parm->tmo_keyxchg=c->tmo_keyxchg/HZ;
    parm->tmo_keylife=c->tmo_keylife/HZ;
    parm->mayclear=c->mayclear;
    parm->cttl=c->cttl;
    return 0;
}

static int cipe_setpar(struct device *dev, struct siocsifcippar *parm)
{
    DEVTOCIPE(dev,c,-ENODEV);

    if (parm->sockshost)
	c->sockshost=parm->sockshost;
    if (parm->socksport)
	c->socksport=parm->socksport;
    if (parm->tmo_keyxchg>10*60*HZ)
	return -EINVAL;
    if (parm->tmo_keyxchg)
	c->tmo_keyxchg=parm->tmo_keyxchg*HZ;
    if (parm->tmo_keylife>24*60*60*HZ)
	return -EINVAL;
    if (parm->tmo_keylife)
	c->tmo_keylife=parm->tmo_keylife*HZ;
    c->mayclear=parm->mayclear;
    c->cttl=parm->cttl;
    c->dev->hard_header_len=cipexhdrl+
	((c->sockshost) ? sizeof(struct sockshdr) : 0);
    return 0;
}

static int cipe_setkey(struct device *dev, struct siocsifcipkey *parm)
{
    DEVTOCIPE(dev,c,-ENODEV);

    dprintk(DEB_KXC, (KERN_INFO "%s: setkey %d\n", dev->name, parm->which));
    switch (parm->which) {
    case KEY_STATIC:
	ExpandUserKey(parm->thekey, c->key_e);
	InvertKey(c->key_e, c->key_d);
	c->havekey=1;
	break;
    case KEY_SEND:
	ExpandUserKey(parm->thekey, c->skey_e);
	c->timeskey=jiffies+c->tmo_keylife;
	c->cntskey=0;
	c->haveskey=1;
	break;
    case KEY_RECV:
	ExpandUserKey(parm->thekey, c->rkey_d);
	InvertKey(c->rkey_d, c->rkey_d);
	c->timerkey=jiffies+2*c->tmo_keylife; /* allow for fuzz */
	c->cntrkey=0;
	c->haverkey=1;
	break;
    case KEY_STATIC+KEY_INVAL:
	c->havekey=c->haveskey=c->haverkey=0;
	memset(&(c->key_e), 0, sizeof(c->key_e));
	memset(&(c->key_d), 0, sizeof(c->key_d));
	break;
    case KEY_SEND+KEY_INVAL:
	c->haveskey=0;
	memset(&(c->skey_e), 0, sizeof(c->skey_e));
	c->timeskey=jiffies+c->tmo_keyxchg;
	break;
    case KEY_RECV+KEY_INVAL:
	c->haverkey=0;
	memset(&(c->rkey_d), 0, sizeof(c->rkey_d));
	c->timerkey=jiffies+c->tmo_keyxchg;
	break;
    default:
	return -EINVAL;
    }
    return 0;
}

static int cipe_alloc_dev(int n);
static void cipe_unalloc_dev(int n);

#ifndef NO_DYNDEV
static int cipe_hasowner(int n)
{
    struct task_struct *p=current;
    pid_t pid=cipe_ctrls[n]->cipe.owner;
    if (!pid)
        return 0;
    do {
        if (p->pid==pid)
            return 1;
        p=p->next_task;
    } while (p!=current);
    return 0;
}
#endif

#ifdef LINUX_21
/* In 2.1 the ioctl operations are run under lock. Beware of deadlocks. */
#define cipe_alloc_LOCK()       0
#define cipe_alloc_UNLOCK()
#else
static struct semaphore cipe_alloc_sem=MUTEX;
#define cipe_alloc_LOCK()       down_interruptible(&cipe_alloc_sem)
#define cipe_alloc_UNLOCK()     up(&cipe_alloc_sem)
#endif

static int cipe_alloc(struct device *dev, struct siocsifcipall *parm)
{
#ifdef NO_DYNDEV
    return -ENOSYS;
#else
    int n=parm->num;
    int e;
    if (n>=cipe_maxdev)
        return -EINVAL;
    if ((e=cipe_alloc_LOCK()))
        return e;
    if (n>=0) {
        if (cipe_ctrls[n]) {
            if (cipe_ctrls[n]->cipe.sock || cipe_hasowner(n))
                e=-EBUSY;
            else
                cipe_ctrls[n]->cipe.owner=current->pid;
        } else {
            e=cipe_alloc_dev(n);
        }
    } else {
        e=-EMFILE;
        for (n=0; n<cipe_maxdev; ++n) {
            if (!cipe_ctrls[n]) {
                e=cipe_alloc_dev(n);
                break;
            }
            if (!cipe_hasowner(n)) {
                cipe_ctrls[n]->cipe.owner=current->pid;
                e=0;
                break;
            }
        }
    }
    if (!e) {
        parm->num=n;
        strncpy(parm->name, cipe_ctrls[n]->name, sizeof(parm->name)-1);
        parm->name[sizeof(parm->name)-1]='\0';
    }
    cipe_alloc_UNLOCK();
    return e;
#endif
}

static int cipe_unalloc(struct device *dev, struct siocsifcipall *parm)
{
#ifdef NO_DYNDEV
    return -ENOSYS;
#else
    int e;
    if (parm->num<0 || parm->num>=cipe_maxdev)
        return -EINVAL;
    if ((e=cipe_alloc_LOCK()))
        return e;
    if (cipe_ctrls[parm->num]->cipe.sock) {
        e=-EBUSY;
    } else {
        if (parm->num>0)
            cipe_unalloc_dev(parm->num);
    }
    cipe_alloc_UNLOCK();
    return e;
#endif
}


/*** Device operation handlers ***/

int cipe_dev_ioctl(struct device *dev, struct ifreq *ifr, int cmd)
{
    int e;

#ifdef LINUX_21

    if (!capable(CAP_NET_ADMIN))
	return -EPERM;

#define doioctl(nam,fun,str) {                                          \
    struct str parm;                                                    \
    dprintk(DEB_CALL, (KERN_INFO "%s: " nam "\n", dev->name));          \
    if ((e=copy_from_user((void*)&parm,(void*)ifr->ifr_data,            \
                          sizeof(parm)))<0)                             \
        return e;                                                       \
    if (parm.magic!=VERSION_MAGIC) {                                    \
        printk(KERN_WARNING "%s: ciped version mismatch", dev->name);   \
        return -EINVAL; }                                               \
    if ((e=fun(dev, &parm))<0)                                          \
        return e;                                                       \
    if ((e=copy_to_user((void*)ifr->ifr_data, (void*)&parm,             \
                        sizeof(parm)))<0)                               \
        return e;                                                       \
    return 0; }

#else

    if (!suser())
	return -EPERM;

#define doioctl(nam,fun,str) {                                              \
    struct str parm;                                                        \
    dprintk(DEB_CALL, (KERN_INFO "%s: " nam "\n", dev->name));              \
    if ((e=verify_area(VERIFY_READ, ifr->ifr_data, sizeof(parm)))<0)        \
        return e;                                                           \
    memcpy_fromfs((void*)&parm, (void*)ifr->ifr_data, sizeof(parm));        \
    if (parm.magic!=VERSION_MAGIC) {                                        \
        printk(KERN_WARNING "%s: ciped version mismatch", dev->name);       \
        return -EINVAL; }                                                   \
    if ((e=fun(dev, &parm))<0)                                              \
        return e;                                                           \
    if ((e=verify_area(VERIFY_WRITE, ifr->ifr_data, sizeof(parm)))<0)       \
        return e;                                                           \
    memcpy_tofs((void*)ifr->ifr_data, (void*)&parm, sizeof(parm));          \
    return 0; }

#endif

    switch (cmd) {
    case SIOCGIFCIPPAR:
	doioctl("getpar", cipe_getpar, siocgifcippar);
    case SIOCSIFCIPPAR:
	doioctl("setpar", cipe_setpar, siocsifcippar);
    case SIOCSIFCIPKEY:
	doioctl("setkey", cipe_setkey, siocsifcipkey);
    case SIOCSIFCIPATT:
	doioctl("attach", cipe_attach, siocsifcipatt);
    case SIOCSIFCIPALL:
	doioctl("alloc", cipe_alloc, siocsifcipall);
    case SIOCSIFCIPUNA:
	doioctl("unalloc", cipe_unalloc, siocsifcipall);
    default:
	return -EINVAL;
    }

#undef doioctl
}

int cipe_dev_open(struct device *dev)
{
    DEVTOCIPE(dev,c,-ENODEV);
    if (!c->sock)
	return -ENXIO;
    dprintk(DEB_CALL, (KERN_INFO "%s: opened\n", dev->name));
    return 0;
}

void cipe_close(struct cipe *c)
{
    cipe_zero_c(c);
    dprintk(DEB_CALL, (KERN_INFO "%s: closed\n", c->dev->name));
}

int cipe_dev_close(struct device *dev)
{
    struct cipe *c = (struct cipe*)(dev->priv);
    if ((!c) || (c->magic!=CIPE_MAGIC)) {
	printk(KERN_WARNING "%s: cipe_dev_close(): no valid struct\n",
               dev->name);
	return 0;
    }
    if (c->sock) {
	dprintk(DEB_CALL, (KERN_INFO "%s: closing\n", c->dev->name));
	/* Tell the attached socket we're going down */
	c->sock->shutdown=SHUTDOWN_MASK;
	c->sock->zapped=1;
	c->sock->err=ENXIO;
	c->sock->error_report(c->sock);
    } else {
	cipe_close(c);
    }
    return 0;
}

struct enet_statistics *cipe_get_stats(struct device *dev)
{
    DEVTOCIPE(dev,c,NULL);
    return &(c->stat);
}


/*** Initialization and finalization stuff ***/

#ifndef LINUX_21
static inline void dev_init_buffers(struct device *dev)
{
    int i;
    for (i = 0; i < DEV_NUMBUFFS; i++)  {
        skb_queue_head_init(&dev->buffs[i]);
    }
}
#endif

static int cipe_init_dev(struct device *dev)
{
    struct cipe *c = (struct cipe*)(dev->priv);
    if (!c)
	return -ENODEV;

    memset(c, 0, sizeof(struct cipe)); /* zero the device struct along */
    c->magic       = CIPE_MAGIC;
    c->dev         = dev;
    cipe_zero_c(c);

    /* Device parameters. */
    /* Procedural */
    dev->open                   = cipe_dev_open;
    dev->stop                   = cipe_dev_close;
    dev->hard_start_xmit        = cipe_xmit;
    dev->get_stats              = cipe_get_stats;
    dev->hard_header           	= NULL;
    dev->rebuild_header         = NULL;
    dev->do_ioctl               = cipe_dev_ioctl;

    /* "Hardware" */
    dev->type		        = ARPHRD_TUNNEL;
    dev->hard_header_len 	= cipexhdrl;
    dev->mtu		        = ETH_DATA_LEN
                                     -sizeof(struct sockshdr)
                                     -cipehdrlen
                                     -cipefootlen;

    dev->addr_len		= 4;            /* Dummy? */
    dev->tx_queue_len	        = 100; /* matches ethernet */

#ifdef LINUX_21
    dev->iflink         = -1;
#else
    dev->family		= AF_INET;
    dev->pa_alen	= 4;
    dev->metric         = 1;
#endif
    dev_init_buffers(dev);

    /* New-style flags */
    dev->flags		= IFF_POINTOPOINT|IFF_NOTRAILERS|IFF_NOARP;

    return 0;
}

static int cipe_alloc_dev(int n)
{
    int e=0;
    struct cipe_ctrl *cc;

    dprintk(DEB_CALL, (KERN_INFO DEVNAME ": cipe_alloc_dev %d\n", n));
    if (!(cc=kmalloc(sizeof(struct cipe_ctrl), GFP_KERNEL))) {
        cipe_ctrls[n]=NULL;
        return -ENOMEM;
    }

    memset((void *)cc, 0, sizeof(struct cipe_ctrl));
    sprintf(cc->name, DEVNAME "%d", n);
    cc->dev.name      = cc->name;
    cc->dev.base_addr = n; /* dummy */
    cc->dev.priv      = (void*)&(cc->cipe);
    cc->dev.next      = NULL;
    cc->dev.init      = cipe_init_dev; /* called by register_netdevice */

    e=register_netdevice(&(cc->dev));
    if (e<0) {
	kfree(cc);
	printk(KERN_ERR
	       "%s: register_netdevice() failed\n", cc->name);
        cc=NULL;
    } else {
        cc->cipe.owner=current->pid;
    }
    cipe_ctrls[n]=cc;
    return e;
}

static void cipe_unalloc_dev(int n)
{
    struct cipe_ctrl *cc=cipe_ctrls[n];
    if (!cc)
	return;
    dprintk(DEB_CALL, (KERN_INFO DEVNAME ": cipe_unalloc_dev %d\n", n));
    if (cc->cipe.magic!=CIPE_MAGIC) {
        printk(KERN_WARNING DEVNAME ": Ouch: cipe_unalloc_dev() wrong struct\n");
        return;
    }
    unregister_netdevice(&(cc->dev));
    cipe_ctrls[n]=NULL;
    kfree(cc);
}


int init_module(void)
{
    /* sanity check on insmod-provided data */
    if (cipe_maxdev<1)  cipe_maxdev=1;
    if (cipe_maxdev>100) cipe_maxdev=100;

#ifdef DEBUG
    printk(KERN_INFO
	   DEVNAME ": CIPE driver vers %s (c) Olaf Titz 1996-1998, %d channels, debug=%d\n",
	   driver_version, cipe_maxdev, cipe_debug);
#else
    printk(KERN_INFO
	   DEVNAME ": CIPE driver vers %s (c) Olaf Titz 1996-1998, %d channels\n",
	   driver_version, cipe_maxdev);
#endif

    prnseed=~jiffies;
    cipe_ctrls = (struct cipe_ctrl **) kmalloc(sizeof(void*)*cipe_maxdev,
					       GFP_KERNEL);
    if (!cipe_ctrls) {
	printk(KERN_ERR
	       DEVNAME ": failed to allocate master control structure\n");
	return -ENOMEM;
    }
    memset(cipe_ctrls, 0, sizeof(void*)*cipe_maxdev);
#ifdef NO_DYNDEV
    {
        int i, e;
        for (i=0; i<cipe_maxdev; ++i)
            if ((e=cipe_alloc_dev(i)))
                return e;
        return 0;
    }
#else
    return cipe_alloc_dev(0);
#endif
}

void cleanup_module(void)
{
    int i;
    for (i=0; i<cipe_maxdev; ++i)
	cipe_unalloc_dev(i);
    kfree(cipe_ctrls);
}

