# arch-tag: 0abcff18-da5d-49f0-a03e-1d866a86b5cd
# Copyright (C) 2004 Canonical Ltd.
#               Author: David Allouche <david@canonical.com>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Twisted process handling

The name of this module, "knotted", was suggested by Andrew Bennetts
as a way to avoid the name clash with the"twisted" top-level package::

    <spiv> I'd say "knotted" would be good.
    <spiv> And then you can tell people to "get knotted" ;)

I disclaim all responsibility for the bad taste of Twisted developers.
"""

import Queue
from twisted.internet import protocol
from pybaz import errors
import commandline


__all__ = ['TwistedSpawningStrategy']


import logging
logging = logging.getLogger('pybaz.knotted')


def make_exec_problem(data, command, args, expected, chdir):
    status, signal, error = data
    proc = DummyProcess(command, args, expected, chdir, status, signal, error)
    return errors.ExecProblem(proc)


class TwistedSpawningStrategy(commandline.SpawningStrategy):

    def __init__(self, command, logger):
        self._command = command
        self._logger = logger

    def _spawn(self, args, chdir):
        if __debug__:
            from twisted.python import threadable
            assert not threadable.isInIOThread()
        from twisted.internet import reactor
        # The queue size must be unlimited, otherwise the blocking
        # put() in the reactor could cause a deadlock.
        queue = Queue.Queue(0) # unlimited queue
        process_protocol = ProcessProtocol(queue)
        argv = [self._command]
        argv.extend(args)
        logging.debug("%#x spawning args=%r, chdir=%r", id(queue), argv, chdir)
        reactor.callFromThread(
            reactor.spawnProcess,
            process_protocol, self._command, args=argv, env=None, path=chdir)
        return queue

    def status_cmd(self, args, chdir, expected):
        queue = self._spawn(args, chdir)
        logging.debug("%#x await termination", id(queue))
        while True:
            msg = queue.get(block=True)
            logging.debug("%#x event %.60r", id(queue), msg)
            if msg[0] == 'start':
                self._logger.log_command(self._command, args)
            elif msg[0] == 'output':
                text = ''.join(msg[1])
                self._logger.log_output(text)
            elif msg[0] == 'error':
                text = ''.join(msg[1])
                self._logger.log_error(text)
            elif msg[0] == 'terminate':
                if msg[1] in expected and msg[2] is None:
                    return msg[1]
                else:
                    raise make_exec_problem(
                        msg[1:], self._command, args, expected, chdir)
            else:
                raise AssertionError('bad message: %r', msg)

    def sequence_cmd(self, args, chdir, expected):
        queue = self._spawn(args, chdir)
        return SequenceCmd(
            queue, self._command, args, chdir, expected, self._logger)

    def status_text_cmd(self, args, chdir, expected):
        queue = self._spawn(args, chdir)
        seq = SequenceCmd(
            queue, self._command, args, chdir, expected, self._logger)
        seq.strip = False
        text = ''.join(seq)
        return seq.status, text

class SequenceCmd(object):

    def __init__(self, queue, command, args, chdir, expected, logger):
        self._queue = queue
        self._command = command
        self._logger = logger
        self._args = args
        self._chdir = chdir
        self._expected = expected
        self._buffer = []
        self._status = None
        self.finished = False
        self.strip = True

    def __iter__(self):
        return self

    def next(self):
        msg = None
        while True:
            if self.finished:
                raise StopIteration
            if self._buffer:
                line = self._buffer.pop(0)
                if self.strip:
                    return line.rstrip('\n')
                else:
                    return line
            if msg is None:
                logging.debug("%#x read output", id(self._queue))
            msg = self._queue.get(block=True)
            logging.debug("%#x event %.60r", id(self._queue), msg)
            if msg[0] == 'start':
                self._logger.log_command(self._command, self._args)
            elif msg[0] == 'output':
                self._buffer = msg[1] + self._buffer
            elif msg[0] == 'error':
                text = ''.join(msg[1])
                self._logger.log_error(text)
            elif msg[0] == 'terminate':
                self.finished = True
                if msg[1] in self._expected and msg[2] is None:
                    self._status = msg[1]
                else:
                    raise make_exec_problem(
                        msg[1:], self._command, self._args,
                        self._expected, self._chdir)
            else:
                raise AssertionError('bad message: %r', msg)

    def _get_status(self):
        if self._status is None:
            raise AttributeError, "no status, process has not finished"
        return self._status
    status = property(_get_status)


class DummyProcess(object):
    """Dummy object providing the same interface as `forkexec.ChildProcess`."""

    def __init__(self, command, args, expected, chdir, status, signal, error):
        self.command = command
        self.args = args
        self.chdir = chdir
        self.expected = expected
        self.error = ''.join(error)
        self.status = status
        self.signal = signal


class ProcessProtocol(protocol.ProcessProtocol):

    def __init__(self, queue):
        self.__queue = queue
        self.__out_buffer = str()
        self.__err_buffer = str()
        self.error_lines = []
        self.status = None

    def connectionMade(self):
        """The process has started."""
        self.__queue.put(('start',), block=True)

    def __send_output(self, lines):
        self.__queue.put(('output', lines), block=True)

    def __send_error(self, lines):
        self.__queue.put(('error', lines), block=True)

    def outReceived(self, data):
        """The process has produced data on standard output."""
        data = self.__out_buffer + data
        lines = data.splitlines(True)
        if lines[-1].endswith('\n'):
            self.__out_buffer = str()
            complete_lines = lines
        else:
            self.__out_buffer = lines[-1]
            complete_lines = lines[:-1]
        self.__send_output(complete_lines)

    def errReceived(self, data):
        """The process has produced data on standard error."""
        data = self.__err_buffer + data
        lines = data.splitlines(True)
        if lines[-1].endswith('\n'):
            self.__err_buffer = str()
            complete_lines = lines
        else:
            self.__err_buffer = lines[-1]
            complete_lines = lines[:-1]
        self.__send_error(complete_lines)
        self.error_lines.extend(complete_lines)

    def outConnectionLost(self):
        """The process has closed standard output."""
        if self.__out_buffer:
            self.__send_output([self.__out_buffer])
            self.__out_buffer = str()

    def errConnectionLost(self):
        """The process has closed standard error."""
        if self.__err_buffer:
            self.__send_error([self.__err_buffer])
            self.error_lines.append(self.__err_buffer)
            self.__err_buffer = str()

    def processEnded(self, reason):
        """The process has terminated."""
        signal = reason.value.signal
        status = reason.value.exitCode
        msg = ('terminate', status, signal, self.error_lines)
        self.__queue.put(msg, block=True)
