# arch-tag: 5b9da267-93df-418e-bf4a-47fb9ec7f6de
# Copyright (C) 2004 Canonical Ltd.
#               Author: David Allouche <david@canonical.com>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""PyArch specific process spawning
"""

import sys
import os
import threading
from pybaz.errors import ExecProblem
import commandline

import logging
logging = logging.getLogger('pybaz.forkexec')


### Internal use constant

class StdoutType(object):
    def __repr__(self):
        return 'pybaz.backends.forkexec.STDOUT'

STDOUT = StdoutType()


### Spawning strategy

class PyArchSpawningStrategy(commandline.SpawningStrategy):

    def __init__(self, command, logger):
        self._command = command
        self._logger = logger

    def sequence_cmd(self, args, chdir=None, expected=(0,), stderr_too=False):
        return _pyarch_sequence_cmd(
            self._command, args, chdir, expected, self._logger,
            stderr_too=stderr_too)

    def status_cmd(self, args, chdir, expected):
        return exec_safe(
            self._command, args, expected=expected, chdir=chdir,
            logger=self._logger)

    def status_text_cmd(self, args, chdir, expected):
        return exec_safe_status_stdout(
            self._command, args, expected=expected, chdir=chdir,
            logger=self._logger)


class _pyarch_sequence_cmd(object):
    def __init__(self, command, args, chdir, expected, logger, stderr_too):
        if stderr_too:
            stderr = STDOUT
        else:
            stderr = None
        self._proc = exec_safe_iter_stdout(
            command, args, stderr=stderr, chdir=chdir, expected=expected,
            logger=logger)
    def __iter__(self):
        return self
    def next(self):
        line = self._proc.next()
        return line.rstrip('\n')
    def _get_finished(self):
        return self._proc.finished
    finished = property(_get_finished)
    def _get_status(self):
        return self._proc.status
    status = property(_get_status)


### Process handling commands

def exec_safe(program, args = [], stdout = None, stdin = None,
              stderr = None, expected = 0, chdir = None, logger = None):
    """Fork/exec a process and and raises an exception if the program
    died with a signal or returned an error code other than expected.
    This function will always wait."""
    proc = ChildProcess(program, args, expected, chdir, logger)
    proc.spawn(stdout, stdin, stderr)
    proc.wait()
    return proc.status


def exec_safe_status_stdout(program, args = [], expected = 0,
                            chdir = None, logger = None):
    """Run the specified program and return a tuple of two items:
    1. exit status of the program;
    2. output of the program as a single string.
    """
    read_end, write_end = os.pipe()
    proc = ChildProcess(program, args, expected, chdir, logger)
    proc.spawn(stdout=write_end)
    os.close(write_end)
    fd = os.fdopen(read_end, 'r')
    output = fd.read()
    proc.wait()
    fd.close()
    return (proc.status, output)


class exec_safe_iter_stdout(object):

    """Iterator over the output of a child process.

    Fork/exec a process with its output piped. Each iteration will
    cause a iteration of the process output pipe. The pipe is properly
    closed whenever the output is exhausted or the iterator is
    deleted.

    If the output is exhausted, the process exit status is checked and
    an ExecError exception will be raised for abnormal or unexpected
    exit status.
    """

    def __init__(self, program, args=[], stdin=None, stderr=None,
                 expected=0, delsignal=None, chdir=None, logger=None):
        self.delsignal = delsignal
        self._read_file, self._pid = None, None
        read_end, write_end = os.pipe()
        if stderr == STDOUT:
            stderr = write_end
        self.proc = ChildProcess(program, args, expected, chdir, logger)
        try:
            self.proc.spawn(stdout=write_end, stdin=stdin, stderr=stderr,
                            closefds=[read_end])
            self._pid = self.proc.pid
        except:
            os.close(write_end)
            self.errlog = '<deleted>'
            os.close(read_end)
            raise
        self._read_file = os.fdopen(read_end, 'r', 0)
        os.close(write_end)

    def __del__(self):
        """Destructor. If needed, close the pipe and wait the child process.

        If child process has already been joined, it means the iterator was
        deleted before being exhausted. It is assumed the return status is not
        cared about.
        """
        if self._pid is not None:
            if self._read_file is not None:
                self._read_file.close()
            # Some buggy programs (e.g. ls -R) do not properly
            # terminate when their output pipe is broken. They
            # must be killed by an appropriate signal before
            # waiting. SIGINT should be okay most of the time.
            if self.delsignal:
                os.kill(self.proc.pid, self.delsignal)
            self.proc.wait_nocheck()

    def _get_finished(self):
        """Whether the underlying process has terminated."""
        return self.proc.status is not None or self.proc.signal is not None
    finished = property(_get_finished)

    def _get_status(self):
        """Exit status of the underlying process.

        Raises ValueError if the process has not yet finished. (Hm...
        should be AttributeError). Can be None if the process was
        killed by a signal.
        """
        if self.proc.status is None:
            raise ValueError, "no status, process has not finished"
        return self.proc.status
    status = property(_get_status)

    def close(self):
        """Close the pipe and wait the child process.

        If the output is not exhausted yet, you should be prepared to
        handle the error condition caused by breaking the pipe.
        """
        self._read_file.close()
        try:
            self.proc.wait()
            self._pid = None
            return self.proc.status
        except ExecProblem:
            self._pid = None
            raise

    def __iter__(self):
        """Iterator. Return self."""
        return self

    def next(self):
        """Iteration method. Iterate on the pipe file.

        Close the pipe and wait the child process once output is exhausted.

        Use `file.readline` instead of `file.next` because we want
        maximal responsiveness to incremental output. The pipe
        mechanism itself provides buffering.
        """
        try:
            line = self._read_file.readline()
        except ValueError:
            if self._pid is None: raise StopIteration
            else: raise
        if line:
            return line
        else:
            self.close()
            raise StopIteration


### Low-level process handling

class ChildProcess:

    """Description of a child process, for error handling."""

    def __init__(self, command, args, expected=0, chdir=None, logger=None):
        """Create a child process descriptor.

        The child process must have already been created. The
        `command`, `args` and `expected` parameters are used for error
        reporting.

        :param command: name of the child process executable
        :type command: str
        :param args: child process arguments (argv)
        :type args: sequence of str
        :param expected: valid exit status values
        :type expected: int or sequence of int
        """
        self.pid = None
        if not isinstance(command, str):
            raise TypeError(
                "command must be a string, but was: %r" % (command,))
        self.command = command
        args = tuple(args)
        for arg in args:
            if not isinstance(arg, str):
                raise TypeError(
                    "args must be a sequence of strings, but was %r" % (args,))
        self.args = args
        if isinstance(expected, int): expected = (expected,)
        self.expected = expected
        self.signal = None
        self.status = None
        self.error = None
        if chdir is not None and not isinstance(chdir, str):
            raise TypeError(
                "chdir must be a string or None, but was: %r" % (chdir,))
        self.chdir = chdir
        self._logger = logger
        self._outlog = None
        self._errlog = None
        logging.debug('ChildProcess, %r, %r, expected=%r, chdir=%r',
                      self.command, self.args, self.expected, self.chdir)

    def spawn(self, stdout=None, stdin=None, stderr=None, closefds=[]):
        """Fork, setup file descriptors, and exec child process.

        The `stdout`, `stdin` and `stderr` file-like objects can be
        raw file descriptors (ints) or file-like objects with a
        ``fileno`` method returning a file descriptor.

        When building a pipe, one side of a pipe is used as `stdout`,
        `stdin` or `stderr`, and the other is included in the
        `closefds` list, so the child process will be able to detect a
        broken pipe.

        :param stdin: use as standard input of child process, defaults
            to ``/dev/null``.
        :type stdin: file-like object with a ``fileno`` method or raw
            file descriptor (``int``).
        :param stdout: use as standard output of child process,
            defaults to internal pipe or ``/dev/null``.
        :type stdout: file-like object with a ``fileno`` method or raw
            file descriptor (``int``). If a logger was specified,
            defaults to a pipe for logging, if no logger was
            specified, defaults to ``/dev/null``.
        :param stderr: use as standard error of child process.
            Defaults to a pipe for error reporting (see `ExecProblem`)
            and logging.
        :type stderr: file-like object with a ``fileno`` method or raw
            file descriptor (``int``).
        :param closefds: file descriptors to close in the child process.
        :type closefds: iterable of int
        """
        __pychecker__ = 'no-moddefvalue'
        if self.pid is not None:
            raise ValueError, "child process was already spawned"
        if self._logger is not None:
            self._logger.log_command(self.command, self.args)
            if stdout is None:
                self._outlog = stdout = StringOutput()
        if stderr is None:
            self._errlog = stderr = StringOutput()
        self.pid = os.fork()
        if self.pid:
            return # the parent process, nothing more to do
        try:
            source_fds = [stdin, stdout, stderr]
            closefds = list(closefds)
            for i in range(3):
                if source_fds[i] is None:
                    source_fds[i] = getnull()
                if hasattr(source_fds[i], 'fileno'):
                    source_fds[i] = source_fds[i].fileno()
            # multiple dup2 can step their own toes if a source fd is smaller
            # than its target fd so start by duplicating low fds
            for i in range(1, 3):
                if source_fds[i] is not None:
                    while source_fds[i] < i:
                        closefds.append(source_fds[i])
                        source_fds[i] = os.dup(source_fds[i])
            # must close before dup2, because closefds may contain
            # values in the range 0..2
            for fd in closefds:
                os.close(fd)
            # finally, move the given fds to their final location
            for dest, source in enumerate(source_fds):
                if source is not None and source != dest:
                    os.dup2(source, dest)
                    if source not in source_fds[dest+1:]:
                        os.close(source)

            if self.chdir:
                os.chdir(self.chdir)
            os.execvp(self.command, (self.command,) + self.args)
        except:
            sys.excepthook(*sys.exc_info())
        os._exit(255)

    def wait_nocheck(self):
        """Wait for process and return exit result.

        This (internally used) variant of `wait` is useful when you
        want to wait for a child process, but not raise an exception
        if it was terminated abnormally. Typically, that's what you
        want if you killed the child process.

        :return: second element of the tuple returned by os.wait()
        """
        if self._outlog is not None:
            self._logger.log_output(self._outlog.read())
        if self._errlog is not None:
            self.error = self._errlog.read()
            if self._logger is not None:
                self._logger.log_error(self.error)
        return os.waitpid(self.pid, 0)[1]

    def wait(self):
        """Wait for process and check its termination status.

        If the process exited, set ``self.status`` to its exit status.

        if the process was terminated by a signal, set ``self.signal``
        to the value of this signal.

        :raise ExecProblem: process was killed by a signal or exit
            status is not is ``self.expected``
        """
        result = self.wait_nocheck()
        if os.WIFSIGNALED(result):
            self.signal = os.WTERMSIG(result)
        if os.WIFEXITED(result):
            self.status = os.WEXITSTATUS(result)
        if not os.WIFEXITED(result):
            raise ExecProblem(self)
        if os.WEXITSTATUS(result) not in self.expected:
            raise ExecProblem(self)


nulldev = None

def getnull():
    """Return a file object of /dev/null/ opened  for writing."""
    global nulldev
    if not nulldev:
        nulldev = open("/dev/null", "w+")
    return nulldev


threadcount = 0

class StringOutput(object):

    """Memory buffer storing a pipe output asynchronously."""

    def __init__(self):
        read_end, self.write_end = os.pipe()
        self.readfile = os.fdopen(read_end, 'r', 0)
        self.thread = threading.Thread(target=self.__run)
        self.thread.setDaemon(True)
        self.thread.start()

    def _close_write_end(self):
        if self.write_end is not None:
            os.close(self.write_end)
            self.write_end = None

    def _join(self):
        #global threadcount
        if self.thread is not None:
            self.thread.join()
            self.thread = None
            #threadcount -= 1
            #print " %s  joined (%d)" % (self, threadcount)

    def __del__(self):
        #print "%s\tdeleting" % self
        self._close_write_end()
        self.readfile.close()
        self._join()

    def __run(self):
        #global threadcount
        #threadcount += 1
        self.data = self.readfile.read()

    def fileno(self):
        return self.write_end

    def read(self):
        #print "%s\treading" % self
        self._close_write_end()
        self._join()
        self.readfile.close()
        return self.data
