/************************************************************************/
/* Module : actionarea.c					        */
/* Purpose: provides an action area for the help window           	*/
/* Mod. By: Keith R. Davis				                */
/* Date   : 12/8/95					                */
/* Notes  : Copyright(c) 1996-98 Mutiny Bay Software	                */
/*          Copyright(c) 1995 ipvr stuttgart and thomas harrer          */
/*          modified source to change look & feel of help window and to */
/*          eliminate redundant pixmap button code                      */
/************************************************************************/

#include <Xm/Form.h>
#include <Xm/PushB.h>

#include "actionarea.h"
#include "helpp.h"		/* for malloc functions.        */
#include "contexthelp.h"
#include "imageutil.h"		/* imageutil module header	*/
#include "buttonpixmaps.h"      /* help window button pixmaps   */

#ifdef MODULE_TEST
#include <stdio.h>
#include <Xm/PanedW.h>
#endif /* MODULE_TEST  */

#ifdef TOOLTIP
#define SHOWLBL 0
#else
#define SHOWLBL 1
#endif

/* -------------------------------------------------------------------- *
**  module identification
** -------------------------------------------------------------------- */
#ifdef RCSID
static char rcsid [] =
    "$Id: actionarea.c,v 1.18 1995/06/28 12:59:30 thomas Exp $";
#endif /* RCSID */

/* -------------------------------------------------------------------- *
**  actions
** -------------------------------------------------------------------- */

/* global data  */
actionrec_t* 	actionrec = NULL;
int		number_of_actions = 0;

/* -------------------------------------------------------------------- *
**  procedure-name:	create_actionarea
**
**  purpose:		creates buttons with desired bindings
** -------------------------------------------------------------------- *
**  args:		parent widget, 
**  return type:	returns the action area form.
**  precondition:	must be child of manager widget.
**  postcondition:	action area is created and managed.
**  error handling.:	-
** -------------------------------------------------------------------- */
Widget
create_actionarea (/* i  */ Widget 	parent, 
		   /* i  */ action_t* 	actions, 
		   /* i  */ int 	num_actions)
{
    Widget action_area, widget;
    int i;
    char *button_names[] = {"close", "index", "history", "back", "up", "down"}; 

    if (actionrec != NULL)
	checked_free (actionrec);
    number_of_actions = num_actions;
    checked_malloc (actionrec, number_of_actions, actionrec_t);

    /* setup the button bar widget */
    action_area = XtCreateManagedWidget("action_area", xmFormWidgetClass, 
					      parent, NULL, 0);

    /* create the buttons.  */
    for (i = 0; i < num_actions; i++) {
	char** xpm = close_xpm;
	
	switch (actions[i].action_type) {
	  case action_dismiss:
	    xpm = close_xpm;
	    break;
	  case action_history:
	    xpm = history_xpm;
	    break;
	  case action_back:
	    xpm = back_xpm;
	    break;
	  case action_up:
	    xpm = up_xpm;
	    break;
	  case action_down:
	    xpm = down_xpm;
	    break;
	  case action_index:
	    xpm = index_xpm;
	    break;
	}

	if(i == 0){
	  widget = XtVaCreateManagedWidget(button_names[i],
					   xmPushButtonWidgetClass,
					   action_area,
					   XmNtopAttachment, XmATTACH_FORM,
					   XmNtopOffset, 3,
					   XmNbottomAttachment, XmATTACH_NONE,
					   XmNleftAttachment, XmATTACH_FORM,
					   XmNleftOffset, 3,
					   XmNrightAttachment, XmATTACH_NONE,
					   NULL);
	}
	else {
	  widget = XtVaCreateManagedWidget(button_names[i],
					   xmPushButtonWidgetClass,
					   action_area,
					   XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET,
					   XmNtopWidget, actionrec[0].w,
					   XmNbottomAttachment,
					   XmATTACH_OPPOSITE_WIDGET,
					   XmNbottomWidget, actionrec[0].w,
					   XmNleftAttachment, XmATTACH_WIDGET,
					   XmNleftWidget, actionrec[i - 1].w,
					   XmNrightAttachment, XmATTACH_NONE, 
					   NULL);
	}

	/* set button image */
	InstallLabeledPixmap(widget, xpm, SHOWLBL);

	/* set context help */
	context_help (widget, cx_help_toolbar);

	/* add the actions activate callback.  */
        if (actions[i].callback != NULL)
            XtAddCallback (widget, XmNactivateCallback,
			   actions[i].callback, 
			   (XtPointer) actions[i].action_type);

	/* record the button.  */
	actionrec[i].action_type = actions[i].action_type;
	actionrec[i].w = widget;
    }

    XtManageChild (action_area);

    return action_area;
}

/* -------------------------------------------------------------------- *
**  procedure-name:	actions_set_sensitive
**
**  purpose:		sets all actions of a specified action_type
**			to the sensitivity state.
** -------------------------------------------------------------------- *
**  args:		action id (only those are available that are
**			declared in actions.h)
**  precondition:	actions must be initialized (else nothing is done).
**  postcondition:	sensitivity state is changed.
** -------------------------------------------------------------------- */
void
actions_set_sensitive (/* i  */ int	action_type,
		       /* i  */ Boolean	sensitive)
{
    if (actionrec != NULL) {

	int i;
	for (i = 0; i < number_of_actions; i++) {
	    if (actionrec[i].action_type == action_type) {
		XtSetSensitive (actionrec[i].w, sensitive);
	    }
	}
    }
}

/* -------------------------------------------------------------------- */
/*  module test:	actionarea                                      */
/* -------------------------------------------------------------------- */

#ifdef MODULE_TEST
static void 
actionarea_test_cb (Widget w, XtPointer call_data, XtPointer client_data)
{
    printf ("button %d pressed\n", (int) call_data);
    return;
}

#define actionarea_test main
void 
actionarea_test (int argc, char* argv[])
{
    
    /* local data  */
    Widget	toplevel, pane, action_area;
    Display* 	display;
    XtAppContext app_context;
    char	class[] = "Actionarea";

    String 	fallbacks [] = {
	"*Background:  #bfbfbfbfbfbf"
    };

    action_t	actions[] =  {
        {action_dismiss, (XtCallbackProc) actionarea_test_cb, 1},
	{action_history, (XtCallbackProc) actionarea_test_cb, 2},
	{action_back,    (XtCallbackProc) actionarea_test_cb, 3},
	{action_up,      (XtCallbackProc) actionarea_test_cb, 4},
	{action_down,    (XtCallbackProc) actionarea_test_cb, 5},
    };
    /* 
     * initialize the toolkit, create a toplevel shell.
     * toplevel and display are public for all modules. 
     */
    toplevel = XtVaAppInitialize 
	(&app_context, class, NULL, 0,
	 &argc, argv, fallbacks, NULL);
    display = XtDisplay (toplevel);

    pane = XtVaCreateWidget 
	("pane", xmPanedWindowWidgetClass, toplevel,
	 XmNresizePolicy,    		XmRESIZE_ANY,
	 XmNresizable,       		True,
	 NULL);
    
    action_area = create_actionarea (pane, actions, 5);

    XtManageChild (pane);
    XtRealizeWidget (toplevel);
    XtAppMainLoop (app_context);

}
#endif /* MODULE_TEST  */


