############################################################################
##
## Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##
##
## $Id: Zone.py,v 1.65 2004/07/20 17:36:36 bazsi Exp $
##
## Author  : Bazsi
## Auditor : kisza
## Last audited version: 1.11
## Notes:
##
############################################################################

"""Module defining Zone and related classes.

This module defines 'Zone' and related classes. Zones are the basis in
access control in Zorp.
"""

from Zorp import *
from Domain import InetDomain, Inet6Domain
from SockAddr import htonl
from Cache import ShiftCache
from traceback import print_exc
import types

#labelset_class = LabelSet;
#zones = {}
root_zone = None
zone_cache_shift_threshold = 1000

class AbstractZone:
	"""Base zone class.
	
	A zone is the basis of access control in Zorp. It encapsulates an
	address range (for example an IPv4 subnet). Parsing and representing
	an address is done by the Domain class and derivates.
	
	Zones are organized into two different hierarchies. The first is
	based on the address it encapsulates. In this case a zone is a child
	of some other zone, if the parent is containing it. (For example
	0.0.0.0/0 contains 192.168.0.0/24) This hierarchy is used when
	searching the containing zone of a given network address.
	
	The other hierarchy is based on administrative decisions.
	Administrative children inherit security attributes (permitted
	set of services etc.) from their parents.
	
	The RootZone class serves two purposes:
	
	  1. it is the base class for all Zone-like classes, implementing
	     interfaces for access control, and hierarchy travelsal.
	  
	  2. an instance of this class is an address domain independent root
	     of the Zone address hierarchy and delegates searches to
	     the appropriate Zone.
	  
	Attributes
	
	  name -- a unique name of this zone
	  
	  inbound_services -- mapping indexed by service name, containing an
	                      item for each permitted inbound service
	  
	  outbound_services -- similar to inbound_services, but used for outbound
	                       services.

	  admin_parent      -- parent of this zone in the administrative hierarchy
	  
	  umbrella          -- true for umbrella zones (ie. zones that do
	                       not inherit security attributes from their
	                       administrative parents)
	
	"""
	def __init__(self, name, inbound_services=None, outbound_services=None, admin_parent=None, umbrella=0, inherit_name=FALSE):
		"""Constructor to initialize a RootZone instance.
		
		This constructor is primarily called by derived classes to
		initialize basic Zone data structures.
		
		Arguments
		
		  self -- this instance
		  
		  name -- name of this zone
		  
		  inbound_services -- an array of allowed inbound service names
		  
		  outbound_services -- an array of allowed outbound service names
		  
		  admin_parent -- administrative parent name
		  
		  umbrella -- this is an umbrella zone
		  
		  domain  -- address domain
		""" 
		
		if Globals.zones.has_key(name):
			raise ZoneException, 'Zone %s duplicate name' % name
		Globals.zones[name] = self
		self.name = name
		self.admin_children = []
		self.inherit_name = inherit_name
		self.domain = None
		self.cached_name = None
		self.umbrella = umbrella
		if admin_parent:
			self.admin_parent = Globals.zones[admin_parent]
			self.admin_parent.addAdminChild(self)
		else:
			self.admin_parent = None

                self.inbound_services = {}
                self.outbound_services = {}
                if outbound_services != None:
                        for i in outbound_services:
                                self.outbound_services[i] = 1
				## LOG ##
				# This message reports that this service is an allowed outbound service in that zone.
				##
				log(None, CORE_DEBUG, 5, "Outbound service; zone='%s', service='%s'", (self.name, i))
        
                if inbound_services != None:
                        for i in inbound_services:
                                self.inbound_services[i] = 1
				## LOG ##
				# This message reports that this service is an allowed inbound service in that zone.
				##
				log(None, CORE_DEBUG, 5, "Inbound service; zone='%s', service='%s'", (self.name, i))

	def addAdminChild(self, child):
		"""Function to add an administrative child
		
		This function adds 'child' to the set of administrative children.
		
		Arguments
		
		  self -- this instance
		  
		  child -- child zone add
		  
		"""
		child.setAdminParent(self)
		self.admin_children.append(child)

	def setAdminParent(self, parent):
		"""Function to set administrative parent of this zone.
		
		This function sets the administrative parent of this Zone.
		
		Arguments
		
		  self -- this instance
		  
		  parent -- parent Zone
		"""
		self.admin_parent = parent
		
	def iterAdminChildren(self, fn, parm = None):
		"""Function to iterate over the set of administrative children.
		
		This function iterates over the set of administrative
		children calling the function 'fn' for each item, with
		parameters 'parm', 'self' and the item.

		The callback fn may delete items from admin_children, this
		function uses a local copy of that array.
		
		Arguments
		
		  self -- this instance
		  
		  fn -- function to call
		  
		  parm -- opaque object passed to fn
		
		""" 
		for x in self.admin_children[:]:
			fn(parm, self, x)

	def __str__(self):
		"""Overridden operator to return the textual representation of self.
		
		Called by the Python core to format the object contents when it is 
		written.
		
		Arguments
		
		  self -- this instance
		"""
		return "RootZone(%s)" % self.name

        def isInboundServicePermitted(self, session):
                """Function to do inbound access control check.
                
                This function is called when a session is connecting to a server to
                check whether it is permitted.
                
                Arguments
                
                  self -- this instance
                  
                  session -- session that should be checked
                  
                Returns
                
                  Z_ACCEPT if the service is permitted, Z_REJECT otherwise
                """
                
                if self.inbound_services.has_key(session.service.name) or self.inbound_services.has_key('*'):
                        return Z_ACCEPT
                elif self.admin_parent and not self.umbrella:
               		return self.admin_parent.isInboundServicePermitted(session)
               		
                return Z_REJECT
                
        def isOutboundServicePermitted(self, session):
                """Function to do outbound access control check.
                
                This function is called when an incoming connection is
		detected to check whether it is allowable.
                
                Arguments
                
                  self -- this instance
                  
                  session -- session that should be checked
                  
                Returns
                
                  Z_ACCEPT if the service is permitted, Z_REJECT otherwise
                """
                if self.outbound_services.has_key(session.service.name) or self.outbound_services.has_key('*'):
                        return Z_ACCEPT
                elif self.admin_parent and not self.umbrella:
               		return self.admin_parent.isOutboundServicePermitted(session)
               		
                return Z_REJECT

	def getName(self):
		if self.cached_name:
			return self.cached_name
		if self.inherit_name:
			self.cached_name = self.admin_parent.getName()
		else:
			self.cached_name = self.name
		return self.cached_name


class RootZone(AbstractZone):

	def __init__(self, name):
		AbstractZone.__init__(self, name)
		self.domains = []
		self.cache = ShiftCache('zone', zone_cache_shift_threshold)

	def addDomain(self, domain_zone):
		self.domains.append(domain_zone)
		
	def findDomain(self, address):
		"""Function to find the root Zone of the given address domain.
		
		This function finds the first child in self which uses
		*domain* as address domain.
		
		Arguments
		
		  self -- this instance
		  
		  domain -- class implementing address range specifics (for
			    example InetDomain for IPv4)
		"""
		
		for x in self.domains:
			if x.isMatchingAddress(address):
				return x
			
			
		# we never return self, because RootZone is not a real zone
		raise ZoneException, "No root Zone found for address domain %s" % (domain, )

	def addZone(self, zone):
		domain_root = self.findDomain(zone.domain)
		domain_root.addZone(zone)

	def findZone(self, address):
		"""Function to find the most specific Zone for address.
		
		This function searches the address hierarchy for the most
		specific Zone containing 'address'.
		
		Arguments
		
		  self -- this instance
		  
		  address -- address we are trying to find (should be
		             derived from SockAddr)
		
		"""

		# FIXME: this is address family specific, should be separated
		# to its own method
		hash = address.ip

                zone = None
		try:
			zone = self.cache.lookup(hash)
		except KeyError:
        		domain_root = self.findDomain(address)
	        	if domain_root:
                		zone = domain_root.findZone(address)
	                	self.cache.store(hash, zone)
	        	
        	if not zone:
	                raise ZoneException, str(address)

		return zone
		
class Zone(AbstractZone):
	"""General Zone class using a real address range.
		
	This class differs from RootZone in that it uses a real address
	domain (for IPv4 InetDomain is used), unlike RootZone which
	is a general wrapper for all address types (IPv4, IPv6, SPX etc.)
	
	"""
	def __init__(self, name, addr, inbound_services = None, outbound_services = None, admin_parent = None, umbrella = 0, domain = None, inherit_name = FALSE):
		"""Constructor to initialize a Zone instance.
		
		This class initializes a Zone instance by calling the
		inherited constructor, and setting local attributes. 
		
		Arguments
		
		  self -- this instance
		  
		  name -- name of this zone
		  
		  addr -- a string representing an address range
		          interpreted by the domain class (last argument),
		          *or* a list of strings representing multiple
		          address ranges.
		  
		  inbound_services  -- set of permitted inbound services as described by RootZone
		  
		  outbound_services -- set of permitted outbound services as described by RootZone
		  
		  admin_parent -- name of the administrative parent 
		  
		  umbrella  -- TRUE if this zone is an umbrella zone
		  
		  domain -- address domain class parsing 'addr' and performing 
		            address comparisons for IPv4 addresses it should 
		            be InetDomain
		
		Notes
		
		  If 'addr' is a list of addresses (like ['192.168.1.1',
		  '192.168.1.5']), several subzones are automatically
		  created with administrative parent set to self. This way
		  you can define members with additional privilege easily.
		
		"""
		
		AbstractZone.__init__(self, name, inbound_services, outbound_services, admin_parent, umbrella, inherit_name)
		
		if domain is None:
			raise ValueError, "domain must be defined"
		
		self.domain = domain
		if type(addr) == types.ListType:
			if len(addr) == 1:
				addr = addr[0]
			else:
				i = 0
				while i < len(addr):
	                        	self.subZone("%s-#%u" % (name, i), addr[i], admin_parent=self.name, domain=domain)
	                        	i = i + 1
				addr = None
		
		if addr:
			self.address = domain(addr)
			root_zone.addZone(self)
		else:
			self.address = None

	def subZone(self, name, addr, admin_parent, domain):
		"""Function to create a subzone if multiple addresses are specified.

		This function is called to create a subzone if multiple
		addresses are specified. It is called by the Zone constructor.
		"""
		return Zone(name, addr, admin_parent=admin_parent, domain=domain, inherit_name=TRUE)


	def __str__(self):
		"""Format the Zone as string.

		This function is called by the Python core when this object
		is used as string.
		"""
		return "Zone(%s, %s)" % (self.getName(), self.address)


class InetZone(Zone):
	"""A class inherited from Zone using the InetDomain address type.

	This is a simple Zone class using InetDomain as its address
	type.
	
	"""
	def __init__(self, name, addr, inbound_services = None, outbound_services = None, admin_parent = None, umbrella = 0, inherit_name = FALSE):
		"""Constructor to initialize an InetZone instance

		This constructor initializes an InetZone object instance,
		and sets its attributes based on arguments.

		Arguments

		  self -- this instance
		  
		  name -- name of this zone
		  
		  addr -- a string representing an address range,
		          interpreted by the domain class (last argument),
		          *or* a list of strings representing multiple
		          address ranges.
		  
		  inbound_services  -- set of permitted inbound services as described by RootZone
		  
		  outbound_services -- set of permitted outbound services as described by RootZone
		  
		  admin_parent -- name of the administrative parent 
		  
		  umbrella  -- TRUE if this zone is an umbrella zone
		  
		"""
		Zone.__init__(self, name, addr, inbound_services, outbound_services, admin_parent, umbrella, InetDomain)

class InetRootZone(InetZone):
	def __init__(self, name):
		global root_zone

		InetZone.__init__(self, name, None)
		root_zone.addDomain(self)
		self.zones = []
		self.mask_hashes = [None]*33;

	def isMatchingAddress(self, addr):
		if type(addr) == types.ClassType and (addr is InetDomain):
			return TRUE
		try:
			if addr.family == AF_INET:
				return TRUE
		except AttributeError:
			pass
		return FALSE

	def addZone(self, zone):
		bits = zone.address.mask_bits
		ip = zone.address.ip
		if not self.mask_hashes[bits]:
			self.mask_hashes[bits] = {}
		if self.mask_hashes[bits].has_key(ip):
			raise ZoneException, "Zone with duplicate IP range, %s" % (zone.address)
		self.mask_hashes[bits][zone.address.ip] = zone

	def findZone(self, addr):
		"""Function to find the most specific containing Zone of 'addr'

		This function returns the most specific Zone containing
		'addr'

		Arguments

		  self -- this instance

		  addr -- address to look up
		"""
		ip = addr.ip
		i = 32
		best = None
		while i >= 0:
			h = self.mask_hashes[i]
			if h:
				try:
					m = htonl(((1 << i) - 1) << (32 - i))
				except OverflowError:
					m = htonl(0x7fffffff << 1)

				try:
					best = h[ip & m]
					break
				except KeyError:
					pass
			i = i - 1
		return best

#class Inet6Zone(Zone):
#	"""A class inherited from Zone using the Inet6Domain address type.
#
#	This is a simple Zone class using Inet6Domain as its address
#	type.
#	
#	"""
#	def __init__(self, name, addr, inbound_services = None, outbound_services = None, admin_parent = None, umbrella = 0):
#		"""Constructor to initialize an Inet6Zone instance
#
#		This constructor initializes an Inet6Zone object instance,
#		and sets its attributes based on arguments.
#
#		Arguments
#
#		  self -- this instance
#		  
#		  name -- name of this zone
#		  
#		  addr -- a string representing an address range,
#		          interpreted by the domain class (last argument),
#		          *or* a list of strings representing multiple
#		          address ranges.
#		  
#		  inbound_services  -- set of permitted inbound services as described by RootZone
#		  
#		  outbound_services -- set of permitted outbound services as described by RootZone
#		  
#		  admin_parent -- name of the administrative parent 
#		  
#		  umbrella  -- TRUE if this zone is an umbrella zone
#		  
#		"""
#		Zone.__init__(self, name, addr, inbound_services, outbound_services, admin_parent, umbrella, Inet6Domain)


root_zone = RootZone("root")
inet_root_zone = InetRootZone("inet_root")
