#ifndef _INCLUDED_CATALOGINFO_H_
#define _INCLUDED_CATALOGINFO_H_

#include <vector>
#include <string>
#include <iostream>

#include <glibmm/ustring.h>
#include <glibmm/refptr.h>
#include <gdkmm/pixbuf.h>
#include <gtkmm/widget.h>

#include "playlist.h"
//#include "musiccatalog.h"
#include "facontext.h"
#include "musicdb.h"


class CatalogInfo {
public:
    virtual ~CatalogInfo() {}
    static void init(FAContext *context, Gtk::Widget *widget);
    
    //Item Details
    /** Returns own url (If any) */
    virtual std::string& item_url() { return m_url; } 
    //Returns text to describe item
    virtual const Glib::ustring& item_text() { return m_text; } 
    /** Returns item data, so it can be split into seperate columns */
    virtual Metadata& item_info() { return m_data; } 
    //Returns Icon to represent item
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_nothing; } 
    
    //Item Maintance
    /** Returns False If Item No Longer In Library */
    virtual bool refresh_item_data() { return true; }
 
    //Item Properties
    /** Returns False If Item Fixed, e.g. My Streams, ... */
    virtual bool can_be_removed_changed() { return false; } 
    /** Returns True if URL is for a playlist */
    virtual bool is_playlist() { return false; } 
    
    //Child Items
    /** Returns URLs for all child children - or self */
    virtual void get_tracks(std::vector<std::string>&) {} 
    virtual CatalogInfo* child_item(uint32_t item_num) { CatalogInfo* nothing = new CatalogInfo; return nothing; } //Returns Child item_num, 0 = first item - Whoever calls this should delete returned item
    virtual uint32_t child_count() { return 0; } //Counts Child Items
    virtual bool should_have_children() { return false; } //Returns True if item can have children
    
    //Different Views
    virtual CatalogInfo* album_list_parent() { CatalogInfo* nothing = new CatalogInfo; return nothing; } //Returns A Catalog Item With Children To Display In AlbumList
    virtual CatalogInfo* track_list_parent() { CatalogInfo* nothing = new CatalogInfo; return nothing; } //Returns A Catalog Item With Children To Display In TrackList

protected:  
    //Static Variables, Shared Between All Items
//  static MusicCatalog *m_catalog;
    static MusicCollection *m_collection;

    static Glib::RefPtr<Gdk::Pixbuf> m_pic_nothing;
    static Glib::RefPtr<Gdk::Pixbuf> m_pic_Catalog;
    static Glib::RefPtr<Gdk::Pixbuf> m_pic_AllItems;
    static Glib::RefPtr<Gdk::Pixbuf> m_pic_UnCatItems;
    static Glib::RefPtr<Gdk::Pixbuf> m_pic_Artist;
    static Glib::RefPtr<Gdk::Pixbuf> m_pic_Album;
    static Glib::RefPtr<Gdk::Pixbuf> m_pic_Track;
    static Glib::RefPtr<Gdk::Pixbuf> m_pic_Playlist;
    static Glib::RefPtr<Gdk::Pixbuf> m_pic_Stream;
    static Glib::RefPtr<Gdk::Pixbuf> m_pic_CD;

    std::string m_url;
    Metadata    m_data;
    Glib::ustring m_text;
 
    //Copies Of Protected MusicCatalog Functions - Used By Various Child Classes
//  bool FindArtistList(const std::string& artistName, std::vector<ArtistList*>::iterator& it) const;
//  bool FindAlbumList(const std::string& albumName, std::vector<AlbumList*>* albumList, std::vector<AlbumList*>::iterator& it) const;
//  bool CaseCompare(const std::string &s1, const std::string &s2) const;
};

    

class CatalogRoot : public CatalogInfo {
public:
    CatalogRoot() { m_text = "Library"; }
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_Catalog; };
    
    virtual CatalogInfo* child_item(uint32_t item_num); 
    virtual uint32_t child_count(); 
    virtual bool should_have_children() { return true; }
};


class TrackListInfo : public CatalogInfo {
public:
    virtual ~TrackListInfo() { delete m_parent; }
    TrackListInfo(CatalogInfo *parent) { m_parent = parent; }
    
    virtual bool refresh_item_data();
    
    virtual CatalogInfo* child_item(uint32_t item_num);
    virtual uint32_t child_count();
    virtual bool should_have_children() { return true; }
    
private:
    std::vector<std::string> m_all_tracks;
    CatalogInfo *m_parent;
};

class AllTracksInfo : public CatalogInfo {
public:
    AllTracksInfo() { m_text = "All Tracks"; } 
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_AllItems; };
    
    virtual bool refresh_item_data();
    
    virtual void get_tracks(std::vector<std::string>&);
    virtual CatalogInfo* child_item(uint32_t item_num);
    virtual uint32_t child_count();
    virtual bool should_have_children() { return true; }

private:
    std::vector<std::string> m_urls;
};

class MyMusicInfo : public AllTracksInfo {
public:
    MyMusicInfo() { m_text = "My Music"; } 
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_Catalog; };

    virtual bool refresh_item_data();
    
    virtual CatalogInfo* child_item(uint32_t item_num);
    virtual uint32_t child_count();
    virtual bool should_have_children() { return true; }

private:
    std::vector<std::string> m_artists;
};

class MyPlaylistsInfo : public CatalogInfo {
public:
    MyPlaylistsInfo() { m_text = "My Playlists"; } 
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_Playlist; };

    virtual bool refresh_item_data();
    
    virtual CatalogInfo* child_item(uint32_t item_num);
    virtual uint32_t child_count();
    virtual bool should_have_children() { return true; }
private:
    std::vector<std::string> m_urls;
};

class MyStreamsInfo : public CatalogInfo {
public:
    MyStreamsInfo() { m_text = "My Streams"; } 
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_Stream; };

    virtual bool refresh_item_data();
    
    virtual void get_tracks(std::vector<std::string>&);

    virtual CatalogInfo* child_item(uint32_t item_num);
    virtual uint32_t child_count();
    virtual bool should_have_children() { return true; }
private:
    std::vector<std::string> m_urls;
};

class CDInfo : public CatalogInfo {
public:
    CDInfo() { m_text = "CD Audio"; } 
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_CD; };
    
    /* TODO: Support CDs
       virtual void get_tracks(std::vector<std::string>&);
       virtual CatalogInfo child_item(uint32_t item_num);
       virtual uint32_t child_count();*/
    virtual bool should_have_children() { return true; }
};

class UnsortedTracksInfo : public CatalogInfo {
public:
    UnsortedTracksInfo() { m_text = "Uncatagorized Tracks"; } 
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_UnCatItems; };

    virtual bool refresh_item_data();
    
    virtual void get_tracks(std::vector<std::string>&);
    virtual CatalogInfo* child_item(uint32_t item_num);
    virtual uint32_t child_count();
    virtual bool should_have_children() { return true; }
private:
    std::vector<std::string> m_urls;
};

class ArtistInfo : public CatalogInfo { 
public:
    ArtistInfo(const std::string& name);
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_Artist; }
    
    virtual bool refresh_item_data();
 
    virtual bool can_be_removed_changed() { return true; }
    
    virtual void get_tracks(std::vector<std::string>&);
    virtual CatalogInfo* child_item(uint32_t item_num);
    virtual uint32_t child_count();
    virtual bool should_have_children() { return true; }
    
    virtual CatalogInfo* album_list_parent() { return new ArtistInfo(m_name); }
    virtual CatalogInfo* track_list_parent() { return new TrackListInfo(new ArtistInfo(m_name)); }

private:
    std::string m_name;
    //std::vector<AlbumList *> *m_albums;
    std::vector<std::string> m_albums;
};

class AlbumInfo : public CatalogInfo {
public:
    AlbumInfo(const std::string& artist, const std::string& name);

    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_Album; }
    virtual bool refresh_item_data();
    virtual bool can_be_removed_changed() { return true; }

    virtual void get_tracks(std::vector<std::string>&);
    virtual CatalogInfo* child_item(uint32_t item_num);
    virtual uint32_t child_count();
    virtual bool should_have_children() { return true; }
    
    virtual CatalogInfo* track_list_parent() { return new TrackListInfo(new AlbumInfo(m_artist, m_name)); }
    
private:
    std::string m_artist;
    std::string m_name;
    std::vector<std::string> m_tracks;
    //std::vector<PlaylistItem *> *m_tracks;
};

class TrackInfo : public CatalogInfo {
public:
    TrackInfo(const std::string& url, bool extended_title = false)
        : m_full_text(extended_title) { m_url = url; }
    
    virtual const Glib::ustring& item_text();
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_Track; }
    
    virtual bool refresh_item_data();
 
    virtual bool can_be_removed_changed() { return true; }
    
    virtual void get_tracks(std::vector<std::string>&);
    virtual Metadata& item_info();
    
    virtual CatalogInfo* track_list_parent() { return new TrackListInfo(new TrackInfo(m_url)); }
    
protected:
    //std::string m_url;
    //MetaData *m_data;
    //Metadata    m_data;
    bool m_full_text;
};

class StreamInfo : public TrackInfo {
public:
    StreamInfo(const std::string& url);
    
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_Stream; }
    virtual CatalogInfo* track_list_parent() { return new TrackListInfo(new StreamInfo(m_url)); }
};

class PlaylistInfo : public CatalogInfo {
public:
    PlaylistInfo(const std::string& url);
    
    virtual std::string& item_url() { return m_url; }
    virtual Glib::RefPtr<Gdk::Pixbuf> item_icon() { return m_pic_Playlist; }
    virtual bool refresh_item_data();
    virtual bool can_be_removed_changed() { return true; }
    virtual bool is_playlist() { return true; }
    virtual void get_tracks(std::vector<std::string>&);
private:
    //std::string m_url;
};


/**
 *  Represent a directory in the tree 
 */
class FilesystemInfo : public CatalogInfo 
{
public:
    FilesystemInfo(const std::string& url);

private:
};

#endif
/* arch-tag: c3b47bc0-2403-4373-b3d8-9484a6133e04 */
