/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <X11/Intrinsic.h>	/* Intrinsics Definitions */
#include <X11/StringDefs.h>	/* Standard Name-String definitions */
#include <X11/Shell.h>     	/* Shell Definitions */
#include <X11/Xatom.h>

#include <X11/Xaw/Command.h>	/* Athena Command Widget */
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>

#include "types.h"
#include "xwave.h"
#include "xwave_widget.h"
#include "misc.h"

/* MD is global in xwave.c */

extern Main_Data *MD;
extern AppResources app_resources;

static void popdown_action(Widget w, XtPointer client_data, 
			   XtPointer call_data);

static void pop_reparent(Widget w,XtPointer client_data,XEvent *event, 
			 Boolean *flg); 
static void delpop_action(Widget w,XtPointer client_data,XEvent *event, 
			  Boolean *flg);

static void exit_action(Widget w, XtPointer client_data, XtPointer call_data);
static void yes_callback(Widget w, XtPointer client_data, XtPointer call_data);
static void (*yes_return)(XtPointer client_data1,XtPointer client_data2,
			  XtPointer call_data);
static void no_callback(Widget w, XtPointer client_data, XtPointer call_data);
static void (*no_return)(XtPointer client_data1,XtPointer client_data2,
			 XtPointer call_data);

static int gcd(int a, int b);



static Atom delwin=None,protocols=None;

static XtPointer yes_client_data1,yes_client_data2;
static XtPointer no_client_data1,no_client_data2;
static XtPointer yes_call_data,no_call_data;

Next_Wave *get_actual_nw(Main_Data *md)
{
    Next_Wave *nw=md->nw;
    while (nw!=NULL) {
	if (nw->wd==md->wd) return(nw);
	nw=nw->nw;
    }
    return(NULL);
}

char *new_name()
{
    static char *name;
    static int count=0;
    
    if ((name=calloc(strlen(app_resources.tdir)+
		     strlen(app_resources.default_newname)+
		     strlen(app_resources.default_ext)+10,1))==NULL) 
      
      return(NULL);
    
    sprintf(name,"%s/%s_%i%s",app_resources.tdir,app_resources.default_newname,
	    count++,app_resources.default_ext+1);
    return(name);
}


/* taken from sox7 */
int gcd(int a, int b)
{
    if (b==0) return a;
    else return gcd(b, a % b);
}

int lcm(int a, int b)
{
    return ((long)a * (long)b) / (long)gcd(a, b);
}

void wd2af(Wave_Data *wd,Audio_File *af)
{
    af->name=NULL;
    af->bps=wd->res;
    af->channels=wd->channels;
    af->freq=wd->freq;
    af->length=wd->length;
    af->comp=wd->comp;
    af->type=wd->type;
    af->headoffs=wd->headoffs;
    af->fd=wd->fd;
}

void af2wd(Audio_File af,Wave_Data *wd)
{
    wd->res=af.bps;
    wd->freq=af.freq;
    wd->channels=af.channels;
    wd->length=af.length;
    wd->bpspl=(wd->res*wd->channels)/8;
    wd->tlength=wd->length/wd->bpspl;
    wd->type=af.type;
    wd->comp=af.comp;
    wd->headoffs=af.headoffs;
    wd->fd=af.fd;
}

void wd2wd(Wave_Data *wd1,Wave_Data *wd2)
{
    wd1->res=wd2->res;
    wd1->channels=wd2->channels;
    wd1->bpspl=wd2->bpspl;
    wd1->freq=wd2->freq;
    wd1->length=wd2->length;
    wd1->markbeg=wd2->markbeg;
    wd1->marklength=wd2->marklength;
    wd1->tlength=wd2->tlength;
    wd1->fd=wd2->fd;
    wd1->headoffs=wd2->headoffs;
    wd1->peak_l=wd2->peak_l;
    wd1->peak_r=wd2->peak_r;
    wd1->ind_peak_l=wd2->ind_peak_l;
    wd1->ind_peak_r=wd2->ind_peak_r;
    wd1->type=wd2->type;
    wd1->comp=wd2->comp;
    wd1->oldtype=wd2->oldtype;
    wd1->oldcomp=wd2->oldcomp;
    wd1->inmem=wd2->inmem;
    wd1->ismark=wd2->ismark;
    wd1->isplay=wd2->isplay;
}

void window_order(Widget w,int mode)
{
    XWindowChanges changes;
    Display *dpy=XtDisplay(w);
    
    changes.stack_mode=mode;
    XReconfigureWMWindow(dpy,XtWindow(w),DefaultScreen(dpy),
			 CWStackMode,&changes);
}


Widget FindWmShell(Widget w)
{
    Widget wm_shell;
    for (wm_shell=w; wm_shell; wm_shell=XtParent(wm_shell)) {
	if (XtIsWMShell(wm_shell))
	  return(wm_shell);
    }
    return(None);
}


void short_fname(Widget w,char *fname)
/* write the text (fname) in a label widget, short the text, if it will not 
 * fit 
 */ 
{
    XFontStruct *font;
    Dimension width;
    int dim;
    
    XtVaGetValues(w,XtNfont,&font,XtNwidth,&width,NULL);
    dim=XTextWidth(font,fname,strlen(fname));
    if (dim>width) {
	char *fn=XtMalloc(strlen(fname)),*fp;
	int i=strlen(fname);
	int pdim;
	strcpy(fn,"...");
	pdim=XTextWidth(font,fn,strlen(fn));
	while (TRUE) {
	    fp=fname+i;
	    dim=XTextWidth(font,fp,strlen(fp));
	    if (dim+pdim>width) break;
	    i--;
	}
	fp+=2;		/* now the directory name fits in the label */
	strcat(fn,fp);
	XtVaSetValues(w,XtNlabel,fn,NULL);
	XtFree(fn); 
    } else 
      XtVaSetValues(w,XtNlabel,fname,NULL);
}

bool file_exists(char *fname)
{
    struct stat buffer;
    
    if (lstat(fname,&buffer)==-1) 
      return(FALSE);
    return(TRUE);
}

bool can_write(char *filename) 
{
    struct stat buffer;
    char link_buf1[MAX_NAMELENGTH];
    char link_buf2[MAX_NAMELENGTH];
    char fname[MAX_NAMELENGTH];
    uid_t myuid;
    gid_t mygid;
    
    strcpy(fname,filename);
    if ((lstat(fname,&buffer)==-1)) {
	int i=strlen(fname);
	while ((fname[i]!='/')&(i>0)) i--;
	fname[i+1]=0;
	if ((lstat(fname,&buffer)==-1)) 
	  return(FALSE);
    }
    myuid=getuid();
    mygid=getgid();
    if (S_ISLNK(buffer.st_mode)) {
	bool statdone=TRUE;
	int lcount;
	
	strcpy(link_buf1,fname);
	while(True) {
	    lcount=readlink(link_buf1,link_buf2,MAX_NAMELENGTH);
	    if (lcount!=-1)  {
		link_buf2[lcount]=0;
		if (strcmp(link_buf1,link_buf2)==0) {
		    statdone=FALSE;
		    break;
		}
		strcpy(link_buf1,link_buf2);
		if (lstat(link_buf2,&buffer)==-1)  {
		    statdone=FALSE;
		    break;
		}
		if (!S_ISLNK(buffer.st_mode)) {
		    statdone=TRUE;
		    break;
		}
	    } else statdone=FALSE;
	}
    }
    if ((buffer.st_uid == myuid)&&(S_IWUSR & buffer.st_mode)) return(TRUE);
    if ((buffer.st_gid == mygid)&&(S_IWGRP & buffer.st_mode)) return(TRUE);
    if (S_IWOTH & buffer.st_mode) return(TRUE);
    return(FALSE);
}

char *get_actual_name(Wave_Data *wd)
{
    char *name;
    char *actual_name;
    
    name=strrchr(wd->name,'/');
    actual_name=malloc(strlen(app_resources.tdir)+strlen(name)+MAX_NUMLENGTH);
    if (actual_name!=NULL) {
	sprintf(actual_name,"%s%s_%i~",app_resources.tdir,name,
		wd->actual_no);
    }
    return(actual_name);
}

char *get_wd(void)
{
    static char *name;
    int i=MAX_NAMELENGTH;
    extern int errno;
    
    while(True) {
	if ((name=calloc(i,1))==NULL) return(NULL);
	if (getcwd(name,i)==NULL) {
	    if (errno==ERANGE) {
		free(name);
		i+=MAX_NAMELENGTH;
	    } else {
		free(name);
		return(NULL);
	    }
	} else break;
    }
    return(name);
}


char *get_abs_name(char *name)
{
    static char *current=NULL;
    char *dirname=NULL;
    char *filename=NULL;
    int fn_length=0,dir_length;
    
    if ((current=get_wd())==NULL) return(NULL);
    
    filename=strrchr(name,'/');
    if (filename!=NULL) {
	filename++;
	fn_length=strlen(filename);
    } else filename=name;
    dir_length=strlen(name)-fn_length;
    
    if (fn_length==0) {
	if ((dirname=calloc(strlen(current)+1,1))==NULL) {
	    free(current);
	    return(NULL);
	}
	strcat(dirname,current);
    } else {
	if (strncmp(name,"/",1)!=0) {
	    /* first char is not '/' */
	    if ((dirname=calloc(strlen(current)+dir_length+2,1))==NULL) {
		free(current);
		return(NULL);
	    }
	    sprintf(dirname,"%s/",current);
	    strncat(dirname,name,dir_length);
	} else {
	    if ((dirname=calloc(dir_length+1,1))==NULL) {
		free(current);
		return(NULL);
	    }
	    strncat(dirname,name,dir_length);
	}
    }
    
    if (chdir(dirname)==-1) {
	free(dirname);
	free(current);
	return(NULL);
    }
    free(dirname);
    dirname=get_wd();
    chdir(current);
    free(current);
    if (dirname==NULL) return(NULL);
    
    current=calloc(strlen(dirname)+strlen(filename)+2,1);
    if (current==NULL) {
	free(dirname);
	return(NULL);
    }
    sprintf(current,"%s/%s",dirname,filename);
    free(dirname);
    return(current);
}

Cursor create_cursor(Widget w,char *bits,int width, int height, int x, int y,
                     char *mbits,int mwidth,int mheight) 
{
    Pixmap cB,cmB;
    Cursor create;
    XColor colors[2];
    Display *dpy=XtDisplay(w);
    Window win = DefaultRootWindow(dpy);
    Screen *scr=XtScreen(w);
    
    colors[0].pixel=BlackPixelOfScreen(scr),
      colors[1].pixel=WhitePixelOfScreen(scr);
    XQueryColors(dpy, DefaultColormapOfScreen(scr),
		 colors, XtNumber(colors));
    
    cB = XCBFD(dpy,win,(char *)bits,width,height);
    cmB = XCBFD(dpy,win,(char *)mbits,mwidth,mheight);
    
    create= XCreatePixmapCursor(dpy,cB,cmB,&colors[0],&colors[1],x,y);
    
    return create;
}

char *itoa(int zahl)
{
    static char buf[MAX_NUMLENGTH];
    sprintf(buf, "%i", zahl);
    return buf;
}

void popdown_action(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget popup=(Widget) client_data;
    XtDestroyWidget(popup);
}

void warn_popup (Widget w,char *message)
{
    Widget popup,form,label,button;
    Position x,y;
    Dimension width,height;
    static XtAccelerators accel;
    
    XtVaGetValues(w,XtNwidth, &width, XtNheight, &height, NULL);
    XtTranslateCoords(w,(Position)(width/2),(Position)(height/2),&x,&y);
    
    popup=XtVaCreatePopupShell("wn_shell",transientShellWidgetClass,w,
			       XtNx,x,XtNy,y,NULL);
    
    form  = MW ("wn_main_form",formWidgetClass,popup,NULL);
    
    label = MW ("wn_message_label",labelWidgetClass,form,
		XtNborderWidth,0,
		XtNlabel,message,
		NULL);
    
    accel = XtParseAcceleratorTable("#override\n\
<Key>Return: set() notify() unset()\n");
    
    button = MW ("wn_ok_btn",commandWidgetClass,form,
		 XtNfromVert,label,
		 XtNaccelerators, accel,
		 XtNtop,XawChainTop,XtNbottom,XawChainTop,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    XtAddCallback(button,XtNcallback,popdown_action,(XtPointer) popup);    
    XtVaGetValues(label,XtNwidth, &width, NULL);
    XtVaGetValues(button,XtNwidth, &height, NULL);
    XtVaSetValues(button,XtNhorizDistance, (width/2)-(height/2), NULL);
    
    XtSetKeyboardFocus(MD->mw->form,button);      
    XtInstallAccelerators(button, button);
    
    XtRealizeWidget(popup);
    XtVaGetValues(popup, XtNwidth,&width,XtNheight,&height,NULL);
    XtVaSetValues(popup, XtNminWidth,width,XtNmaxWidth,width,\
XtNminHeight,height,XtNmaxHeight,height,NULL);
    XtPopup(popup,XtGrabExclusive);
}

static void yes_callback(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget popup=(Widget) client_data;
    XtDestroyWidget(popup);
    XFlush(XtDisplay(w));
    if (yes_return!=NULL)
      yes_return(yes_client_data1,yes_client_data2,yes_call_data);
}

static void no_callback(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget popup=(Widget) client_data;
    XtDestroyWidget(popup);
    XFlush(XtDisplay(w));
    if (no_return!=NULL)
      no_return(no_client_data1,no_client_data2,no_call_data);
}


void bool_popup (void (*yes_call) ,XtPointer y_client_data1,
		 XtPointer y_client_data2, XtPointer y_call_data,
		 void (*no_call) ,XtPointer n_client_data1,
		 XtPointer n_client_data2, XtPointer n_call_data,
		 Widget parent,char *message)
{
    Widget popup,form,label,ybutton,nbutton;
    Position x,y;
    Dimension width,width1,height;
    
    yes_return=yes_call;
    yes_client_data1=y_client_data1;
    yes_client_data2=y_client_data2;
    yes_call_data=y_call_data;
    
    no_return=no_call;
    no_client_data1=n_client_data1;
    no_client_data2=n_client_data2;
    no_call_data=n_call_data;
    
    
    XtVaGetValues(parent,XtNwidth, &width, XtNheight, &height, NULL);
    XtTranslateCoords(parent,(Position)(width/2),(Position)(height/2),&x,&y);
    
    popup=XtVaCreatePopupShell("bo_shell",transientShellWidgetClass,parent,
			       XtNx,x,XtNy,y,NULL);
    
    form  = MW ("bo_main_form",formWidgetClass,popup,NULL);
    
    label = MW ("bo_message_label",labelWidgetClass,form,
		XtNborderWidth,0,
		XtNlabel,message,
		NULL);
    
    ybutton = MW ("bo_yes_btn",commandWidgetClass,form,
		  XtNfromVert,label,
		  XtNtop,XawChainTop,XtNbottom,XawChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    XtAddCallback(ybutton,XtNcallback,yes_callback,(XtPointer) popup);    
    
    nbutton = MW ("bo_no_btn",commandWidgetClass,form,
		  XtNfromVert,label,XtNfromHoriz,ybutton,
		  XtNtop,XawChainTop,XtNbottom,XawChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    XtAddCallback(nbutton,XtNcallback,no_callback,(XtPointer) popup);    
    
    XtVaGetValues(label,XtNwidth, &height, NULL);
    XtVaGetValues(ybutton,XtNwidth, &width, NULL);
    XtVaGetValues(nbutton,XtNwidth, &width1, NULL);
    XtVaSetValues(ybutton,XtNhorizDistance, (height/2)-((width+width1)/2),NULL);
    
    XtRealizeWidget(popup);
    XtVaGetValues(popup, XtNwidth,&width,XtNheight,&height,NULL);
    XtVaSetValues(popup, XtNminWidth,width,XtNmaxWidth,width,\
XtNminHeight,height,XtNmaxHeight,height,NULL);
    XtPopup(popup,XtGrabExclusive);
}


void exit_action(Widget w, XtPointer client_data, XtPointer call_data)
{
    XtDestroyApplicationContext(XtWidgetToApplicationContext(w));
    exit(0);
}

void initwarn_popup (Widget w,char *message)
{
    Widget form,label,button;
    Dimension width,height;
    static XtAccelerators accel;
    
    form  = MW ("res_main_form",formWidgetClass,w,NULL);
    
    label = MW ("res_message_label",labelWidgetClass,form,
		XtNborderWidth,0,
		XtNlabel,message,
		NULL);
    
    accel = XtParseAcceleratorTable("#override\n\
<Key>Return: set() notify() unset()\n");
    
    button = MW ("res_ok_btn",commandWidgetClass,form,
		 XtNlabel,"OK",
		 XtNfromVert,label,
		 XtNaccelerators, accel,
		 XtNtop,XawChainTop,XtNbottom,XawChainTop,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    XtAddCallback(button,XtNcallback,exit_action,(XtPointer) w);
    XtVaGetValues(label,XtNwidth, &width, NULL);
    XtVaGetValues(button,XtNwidth, &height, NULL);
    XtVaSetValues(button,XtNhorizDistance, (width/2)-(height/2), NULL);
    
    XtSetKeyboardFocus(form,button);      
    XtInstallAccelerators(button, button);
    
    XtRealizeWidget(w);
    
    XtVaGetValues(w, XtNwidth,&width,XtNheight,&height,NULL);
    XtVaSetValues(w, XtNminWidth,width,XtNmaxWidth,width,
		  XtNminHeight,height,XtNmaxHeight,height,NULL);
    
    XtAppMainLoop(XtWidgetToApplicationContext(w));
}

void popup_centered(Widget w)
{
    Position x, y;
    Display *dpy;
    Dimension width,width1, height,height1;
    
    XtRealizeWidget(w);
    
    dpy=XtDisplay(w);
    
    XtTranslateCoords(MD->mw->form,0,0,&x,&y);
    XtVaGetValues(MD->mw->form, XtNwidth,&width,XtNheight,&height,NULL);
    XtVaGetValues(w, XtNwidth,&width1,XtNheight,&height1,NULL);
    XtVaSetValues(w, XtNminWidth,width1,XtNmaxWidth,width1,\
XtNminHeight,height1,XtNmaxHeight,height1,NULL);
    
    if (width>width1) {
	width-=width1;
	x+=width/2;
    } else {
	width1-=width;
	x-=width1/2;
    }
    if (height>height1) {
	height-=height1;
	y+=height/2;
    } else {
	height1-=height;
	y-=height1/2;
    }
    
    XtVaSetValues(w, XtNx,x, XtNy,y,NULL);
    
    
    delwin=XInternAtom(dpy, "WM_DELETE_WINDOW", FALSE);
    protocols = XInternAtom(dpy, "WM_PROTOCOLS", FALSE);
    if (delwin==None || protocols==None)
      perror("Couldn't create ATOMS");
    
    XtAddEventHandler(w, StructureNotifyMask, FALSE,
		      (XtEventHandler)pop_reparent, (XtPointer) NULL);
    
    
    XtPopup(w,XtGrabExclusive);
}


static void pop_reparent(Widget w,XtPointer client_data,XEvent *event, Boolean *flg)
{
    if (event->type!=ReparentNotify) return;
    
    XtAddRawEventHandler(w, 0, True,
			 (XtEventHandler)delpop_action, (XtPointer) NULL);
    
    XChangeProperty(XtDisplay(w), XtWindow(w),
		    protocols,XA_ATOM, 32, PropModeReplace,
		    (unsigned char *) &delwin, 1);
}

static void delpop_action(Widget w,XtPointer client_data,XEvent *event, Boolean *flg)
{
    XClientMessageEvent *cevent=(XClientMessageEvent*)event;
    
    if (cevent->type!=ClientMessage) return;
    if (cevent->message_type==protocols && cevent->data.l[0]==delwin) {
	XtPopdown(w);
	return;
    }
}


int ConvertColor(Widget w, char *color_name)
{
    XrmValue from, to;
    
    from.size = strlen(color_name) + 1;  
    from.addr = color_name;
    
    /*
     * This conversion accepts a colorname from rgb.txt, or a #rrrgggbbb 
     * rgb color definition, or the special toolkit strings "XtDefaultForeground" 
     * and "XtDefaultBackground".
     */
    
    XtConvert(w, XtRString, (XrmValuePtr) &from, XtRPixel, (XrmValuePtr) &to);
    if (to.addr == NULL) {
	return(-1);
    }
    
    return( (int) *((Pixel *) to.addr) );
}

void setWmWindowName(Widget w, char *name)
{
    Display *dpy=XtDisplay(w);
    Atom a_name=None;
    
    if (name==NULL) return;
    a_name=XInternAtom(dpy, "WM_NAME", FALSE);
    if (a_name!=None) {
	XChangeProperty(dpy, XtWindow(FindWmShell(w)), a_name,
			XA_STRING, 8, PropModeReplace,
			(unsigned char *) name, strlen(name));
    }
}

