// $Id: Graph.cc,v 1.5 2003/02/20 15:19:56 flaterco Exp $
/*  Graph  Abstract superclass for all graphs.

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.hh"

Graph::Graph (unsigned xsize, unsigned ysize) {
  myxsize = xsize;
  myysize = ysize;
  clockmode = 0;
}

unsigned Graph::xsize() {
  return myxsize;
}

unsigned Graph::ysize() {
  return myysize;
}

struct Graph::tideline *Graph::findnextsunevent (struct tideline *first,
Timestamp now, Timestamp endt, Timestamp &nextsunevent) {
  while (first) {
    if (first->t_out > now &&
    (first->etype == Station::sunrise || first->etype == Station::sunset)) {
      nextsunevent = first->t_out;
      return first;
    }
    first = first->next;
  }
  nextsunevent = endt + Interval(DAYSECONDS);
  return NULL;
}

double Graph::linterp (double lo, double hi, double saturation) {
  return lo + saturation * (hi - lo);
}

unsigned char Graph::linterp (unsigned char lo, unsigned char hi,
		       double saturation) {
  return (unsigned char) linterp ((double)lo, (double)hi, saturation);
}

unsigned short Graph::linterp (unsigned short lo, unsigned short hi,
		       double saturation) {
  return (unsigned short) linterp ((double)lo, (double)hi, saturation);
}

  // How to translate a tide depth to a y-coordinate
#define xlate(y) linterp (ymax, ymin, (((double)y) - valmin) / (valmax - valmin))

void Graph::drawDepth (double ymax, double ymin, double valmax, double valmin,
unsigned linestep, int &mindepth, int &maxdepth, unsigned labelwidth) {
  int depth;
  double ytide;
  for (depth = 0; depth <= valmax; depth += linestep) {
    ytide = xlate(depth);
    // Leave room for 3 lines of text at top, 3 lines of text plus
    // tick marks at bottom.
    if (ytide - fontHeight()/2 - depthLineVerticalMargin() <= (fontHeight()+fontMargin()) * 3)
      break;
    if (ytide + fontHeight()/2 + depthLineVerticalMargin() >= myysize - (fontHeight()+fontMargin()) * 3 - hourTickLen())
      continue;
    maxdepth = depth;
    if (depth < mindepth) // In case one loop is never executed.
      mindepth = depth;
    drawHorizontalLine (labelwidth, myxsize-1, ytide, Colors::foreground);
  }
  for (depth = -linestep; depth >= valmin; depth -= linestep) {
    ytide = xlate(depth);
    // Leave room for 3 lines of text at top, 3 lines of text plus
    // tick marks at bottom.
    if (ytide - fontHeight()/2 - depthLineVerticalMargin() <= (fontHeight()+fontMargin()) * 3)
      continue;
    if (ytide + fontHeight()/2 + depthLineVerticalMargin() >= myysize - (fontHeight()+fontMargin()) * 3 - hourTickLen())
      break;
    mindepth = depth;
    if (depth > maxdepth) // In case one loop is never executed.
      maxdepth = depth;
    drawHorizontalLine (labelwidth, myxsize-1, ytide, Colors::foreground);
  }
}

void Graph::drawFunkyLine (double prevytide, double ytide, double nextytide,
Settings *settings, int x, Colors::colorchoice c) {
  double dy, yleft, yright;

  // The fix for line slope breaks down when the slope gets nasty,
  // so switch to a more conservative strategy when that happens.
  // Line width becomes 1 no matter what.

#define dohalfline(yy) {                                          \
  double lw;                                                      \
  if (fabs(dy) < slopelimit)                                      \
    lw = (1.0 + (M_SQRT2 - 1.0) * fabs(dy)) * settings->lw / 2.0; \
  else                                                            \
    lw = (fabs(dy) + settings->lw) / 2.0;                         \
  if (dy < 0.0)                                                   \
    lw = -lw;                                                     \
  yy = ytide - lw;                                                \
}

  dy = ytide - prevytide;
  dohalfline (yleft);
  dy = ytide - nextytide;
  dohalfline (yright);

  // Fix degenerate cases.
  if (ytide > yleft && ytide > yright) {
    if (yleft > yright)
      yleft = ytide + settings->lw / 2.0;
    else
      yright = ytide + settings->lw / 2.0;
  } else if (ytide < yleft && ytide < yright) {
    if (yleft < yright)
      yleft = ytide - settings->lw / 2.0;
    else
      yright = ytide - settings->lw / 2.0;
  }
  drawVerticalLine (x, yleft, yright, c);
}

void Graph::clearGraph (Timestamp start, Timestamp endt, Interval increment,
Station *sr, struct Graph::tideline *first) {
  assert (sr);
  Settings *settings = sr->context->settings;
  // Clear the graph by laying down a background of days and nights.
  int sunisup = 1;
  if (!(sr->coordinates.isNull()) && settings->ns == "n")
    sunisup = sun_is_up (start, sr->coordinates);
  int x;
  struct tideline *t;
  Timestamp loopt (start);
  Timestamp nextsunevent;
  findnextsunevent (first, loopt, endt, nextsunevent);
  for (x=0; x<myxsize; x++, loopt += increment) {
    while (loopt >= nextsunevent) {
      t = findnextsunevent (first, nextsunevent,
        endt, nextsunevent);
      if (t) {
        switch (t->etype) {
        case Station::sunrise:
          sunisup = 0;
          break;
        case Station::sunset:
          sunisup = 1;
          break;
        default:
          assert (0);
	}
      } else
        sunisup = !sunisup;
    }
    Colors::colorchoice c = (sunisup ? Colors::daytime : Colors::nighttime);
    drawVerticalLine (x, 0, myysize-1, c);
  }
}

void Graph::drawX (int x, double y) {
  drawVerticalLine (x, y-4, y+4, Colors::foreground);
  drawHorizontalLine (x-4, x+4, y, Colors::foreground);
}

void Graph::drawTides (Station *sr, Timestamp now, Angle *angle) {
  assert (sr);
  Settings *settings = sr->context->settings;
  assert (sr->aspect > 0.0);

  Interval increment = (max (1, (long)(aspmagnum / (double)myysize /
    (aspectfudge() * sr->aspect) + 0.5)));
  Timestamp start = now - increment * startpos();

  Timestamp endt = start + increment * xsize(); // end is a keyword
  struct tideline *t;

  // now is the "now" of the graph (the time to show).
  // rightnow is the real "now" (the actual wallclock time).
  Timestamp rightnow ((time_t)(time(NULL)));

  // First get a list of the relevant tide events.  We need to go back
  // to synchronize internal and real timestamps.  We'll also draw one
  // extra timestamp on either side, and make sure to stack up at least
  // one max event and one min for clock mode to use.
  // Furthermore, we make sure to get a previous event that is a max or
  // min event for the sake of the tide clock icon (see kludge below).
  struct tideline *first = new tideline;
  first->tm = start;
  do
    sr->predictExactTideEvent (first->tm, Station::backward,
      first->t_out, first->etype, first->etype_desc_long,
      first->etype_desc_short, first->pv);
  while (first->t_out >= start ||
        ((angle) && (!(isMaxMinEvent (first->etype)))));
  struct tideline *prev = first;
  nextmax = nextmin = NULL;
  while (prev->t_out <= endt || (!nextmax) || (!nextmin)) {
    struct tideline *next = new tideline;
    next->tm = prev->tm;
    sr->predictExactTideEvent (next->tm, Station::forward,
      next->t_out, next->etype, next->etype_desc_long,
      next->etype_desc_short, next->pv);
    if (next->etype == prev->etype)
      sr->predictExactTideEvent (next->tm, Station::forward,
        next->t_out, next->etype, next->etype_desc_long,
        next->etype_desc_short, next->pv);
    // Stack up for clock mode.
    if (next->t_out > now) {
      switch (next->etype) {
      case Station::max:
        if (!nextmax)
          nextmax = next;
        break;
      case Station::min:
        if (!nextmin)
          nextmin = next;
        break;
      default:
        ;
      }
    }
    prev->next = next;
    prev = next;
  }
  prev->next = NULL;

  clearGraph (start, endt, increment, sr, first);

  int x;
  double ytide;
  Timestamp loopt;

  // Figure constants.
  double ymin = margin * (double)myysize;
  double ymax = (double)myysize - ymin;
  double valmin = sr->minLevel().val();
  double valmax = sr->maxLevel().val();
  if (valmin >= valmax)
    barf (ABSURD_OFFSETS, sr->name);
  double midwl = xlate(0);

  // Figure increment for depth lines.  Unlike XTide 1, this will decrease
  // the increment if more vertical space becomes available.
  unsigned linestep = 1, prevstep = 1, prevmult = 10;
  while (xlate(0.0) - xlate((double)linestep) < fontHeight() + fontMargin()) {
    switch (prevmult) {
    case 10:
      prevmult = 2;
      linestep = prevstep * prevmult;
      break;
    case 2:
      prevmult = 5;
      linestep = prevstep * prevmult;
      break;
    case 5:
      prevmult = 10;
      prevstep = linestep = prevstep * prevmult;
      break;
    default:
      assert (0);
    }
  }

  // More figuring.
  Dstr u;
  sr->minLevel().Units().shortname(u);
  int mindepth=10000000, maxdepth=-10000000;
  unsigned labellen = 2 + u.length() + (unsigned) (floor (max (max (log10
    (fabs (valmin)), log10 (fabs (valmax))), 0)));
  unsigned labelwidth = fontWidth() * labellen + depthLabelLeftMargin() +
    depthLabelRightMargin();

  // Draw depth lines now?
  if (settings->tl == "n")
    drawDepth (ymax, ymin, valmax, valmin, linestep, mindepth, maxdepth,
    labelwidth);

  // Draw the actual tides.
  double prevval, prevytide;
  double val = sr->predictApproximate(start-increment).val();
  ytide = xlate(val);
  double nextval = sr->predictApproximate(start).val();
  double nextytide = xlate (nextval);
  // loopt is actually 1 step ahead of x
  for (x=0, loopt=start+increment; x<myxsize; x++, loopt += increment) {
    prevval = val;
    prevytide = ytide;
    val = nextval;
    ytide = nextytide;
    nextval = sr->predictApproximate(loopt).val();
    nextytide = xlate(nextval);

    if (sr->isCurrent) {
      Colors::colorchoice c = (val > 0.0 ?
        Colors::flood : Colors::ebb);
      if (settings->nf == "n")
        drawVerticalLine (x, midwl, ytide, c);
      else
        drawFunkyLine (prevytide, ytide, nextytide, settings, x, c);
    } else {
      Colors::colorchoice c = (prevval < val ?
        Colors::flood : Colors::ebb);
      if (settings->nf == "n")
        drawVerticalLine (x, myysize, ytide, c);
      else
        drawFunkyLine (prevytide, ytide, nextytide, settings, x, c);
    }
  }

  // Draw depth lines later?
  if (settings->tl != "n")
    drawDepth (ymax, ymin, valmax, valmin, linestep, mindepth, maxdepth,
    labelwidth);

  // Height axis.
  int depth;
  for (depth = mindepth; depth <= maxdepth; depth += linestep) {
    Dstr dlabel;
    dlabel += depth;
    dlabel += " ";
    dlabel += u;
    while (dlabel.length() < labellen)
      dlabel *= " ";
    double adj = 0.0;
    if (fontHeight() > 1)
      adj = (double)(fontHeight()) / 2.0;
    drawString (depthLabelLeftMargin(), xlate(depth)-adj, dlabel);
  }

  // Figure increment for time axis.
  unsigned timestep = 1;
  if (HOURSECONDS * timestep / increment < fontWidth() * 2)
    timestep = 2;
  if (HOURSECONDS * timestep / increment < fontWidth() * 2)
    timestep = 3;
  if (HOURSECONDS * timestep / increment < fontWidth() * 2)
    timestep = 4;
  if (HOURSECONDS * timestep / increment < fontWidth() * 2)
    timestep = 6;
  if (HOURSECONDS * timestep / increment < fontWidth() * 2)
    timestep = 12;
  if (HOURSECONDS * timestep / increment < fontWidth() * 2)
    timestep = 24;

  // Time axis.
  loopt = start;
  for (loopt.prev_hour(sr->timeZone,settings); loopt < endt + Interval(HOURSECONDS);
  loopt.inc_hour(sr->timeZone,settings)) {
    if (loopt.gethour(sr->timeZone,settings) % timestep == 0) {
      x = (int)((loopt - start) / increment + 0.5);
      drawHourTick (x, Colors::foreground);
      Dstr ts;
      loopt.printhour (ts, sr->timeZone, settings);
      labelHourTick (x, ts);
    }
  }
  /* Make tick marks for day boundaries thicker */
  /* This is not guaranteed to coincide with an hour transition! */
  loopt = start;
  for (loopt.prev_day(sr->timeZone,settings); loopt < endt + Interval(HOURSECONDS);
  loopt.inc_day(sr->timeZone,settings)) {
    x = (int)((loopt - start) / increment + 0.5);
    drawHourTick (x-1, Colors::foreground);
    drawHourTick (x, Colors::foreground);
    drawHourTick (x+1, Colors::foreground);
  }

  if (clockmode) {

    // Write current time
    Dstr ts;
    rightnow.printtime (ts, sr->timeZone, settings);
    centerStringOnLine (myxsize/2, 0, ts);

    // Write next max
    centerStringOnLine (myxsize/2, 1, nextmax->etype_desc_long);
    nextmax->t_out.printtime (ts, sr->timeZone, settings);
    centerStringOnLine (myxsize/2, 2, ts);

    // Write next min
    centerStringOnLine (myxsize/2, -3, nextmin->etype_desc_long);
    nextmin->t_out.printtime (ts, sr->timeZone, settings);
    centerStringOnLine (myxsize/2, -2, ts);

  } else {

    drawTitleLine (sr->name);

    // Put timestamps for timestampable events.
    t = first;
    while (t) {
      Dstr ts;
      x = (int)((t->t_out - start) / increment + 0.5);
      switch (t->etype) {
      case Station::max:
      case Station::min:
	t->t_out.printdate (ts, sr->timeZone, settings);
	drawTimestampLabel (x, 1, ts);
	t->t_out.printtime (ts, sr->timeZone, settings);
	drawTimestampLabel (x, 2, ts);
	break;

      case Station::moonrise:
      case Station::moonset:
      case Station::slackrise:
      case Station::slackfall:
      case Station::markrise:
      case Station::markfall:
      case Station::newmoon:
      case Station::firstquarter:
      case Station::fullmoon:
      case Station::lastquarter:
        drawHourTick (x, Colors::mark);
	t->t_out.printtime (ts, sr->timeZone, settings);
	drawTimestampLabel (x, -2, ts);

        // Pre-2.6:  only draw mark/slack crossings at the bottom;
        // include date.
	// t->t_out.printdate (ts, sr->timeZone, settings);
	// drawTimestampLabel (x, -3, ts);

        // 2.6:  replace the date with the description and draw more
        // sun/moon events.  Try to be smart about when to use long
        // descriptions or short.
        if (!is_banner() && (sr->isCurrent || sr->markLevel))
	  drawTimestampLabel (x, -3, t->etype_desc_short);
        else
	  drawTimestampLabel (x, -3, t->etype_desc_long);

	break;
      default:
	;
      }
      t = t->next;
    }
  }

  // Extra lines.
  if (sr->markLevel) {
    ytide = xlate(sr->markLevel->val());
    drawHorizontalLine (labelwidth, myxsize-1, ytide, Colors::mark);
  }
  if (settings->el != "n") {
    drawHorizontalLine (labelwidth, myxsize-1, midwl, Colors::datum);
    ytide = (ymax + ymin) / 2.0;
    drawHorizontalLine (labelwidth, myxsize-1, ytide, Colors::msl);
  }

  // X marks the current time.
  if (rightnow >= start && rightnow < endt) {
    x = (int)((rightnow - start) / increment + 0.5);
    ytide = xlate(sr->predictApproximate(rightnow).val());
    drawX (x, ytide);

    if (angle) {
      // Kludge for tide clock -- estimate angle of hand on analog tide
      // clock.
      t = first;
      while (!(isMaxMinEvent (t->etype))) {
	t = t->next;
	assert (t);
      }
      struct tideline *prev = NULL;
      while (t->t_out <= rightnow) {
        prev = t;
        t = t->next;
        while (!(isMaxMinEvent (t->etype))) {
          t = t->next;
          assert (t);
        }
        assert (t);
      }
      assert (prev);
      assert (prev->t_out <= rightnow && t->t_out > rightnow &&
              isMaxMinEvent (prev->etype) && isMaxMinEvent (t->etype));
      double temp = (rightnow - prev->t_out) / (t->t_out - prev->t_out);
      temp *= 180.0;
      if (prev->etype == Station::min)
        temp += 180.0;
      (*angle) = Angle (Angle::DEGREES, temp);
    }
  }

  // Free up temp storage.
  while (first) {
    t = first->next;
    free (first);
    first = t;
  }
}

void Graph::drawHorizontalLine (int xlo, int xhi, double y,
				Colors::colorchoice c) {
  drawHorizontalLine (xlo, xhi, (int)(y+0.5), c);
}

void Graph::drawString (int x, double y, const Dstr &s) {
  drawString (x, (int)(y+0.5), s);
}

void Graph::centerString (int x, int y, const Dstr &s) {
  drawString (x-(int)s.length()*fontWidth()/2, y, s);
}

void Graph::centerString (int x, double y, const Dstr &s) {
  drawString (x-(int)s.length()*fontWidth()/2, y, s);
}

void Graph::centerStringOnLine (int x, int line, const Dstr &s) {
  int y;
  if (line >= 0)
    y = line * (fontHeight()+fontMargin());
  else
    y = myysize+(fontHeight()+fontMargin())*line-hourTickLen();
  centerString (x, y, s);
}

unsigned Graph::fontMargin() {
  // This used to be 1 until fontHeight was increased to accommodate
  // Latin1 characters.
  return 0;
}

unsigned Graph::depthLabelLeftMargin() {
  return 2;
}

unsigned Graph::depthLabelRightMargin() {
  return 2;
}

unsigned Graph::depthLineVerticalMargin() {
  return 2;
}

unsigned Graph::hourTickLen() {
  return hourticklen;
}

double Graph::aspectfudge() {
  return 1.0;
}

void Graph::setPixel (int x, int y, Colors::colorchoice c,
		       double saturation) {
  if (saturation >= 0.5)
    setPixel (x, y, c);
}

void
Graph::drawHourTick (int x, Colors::colorchoice c) {
  drawVerticalLine (x, (int)myysize-1, (int)myysize-hourTickLen(), c);
}

void
Graph::drawVerticalLine (int x, int y1, int y2, Colors::colorchoice c) {
  int ylo, yhi;
  if (y1 < y2) {
    ylo = y1; yhi = y2;
  } else {
    ylo = y2; yhi = y1;
  }
  int i;
  for (i=ylo; i<=yhi; i++)
    setPixel (x, i, c);
}

void Graph::drawVerticalLine (int x, double y1, double y2,
                                 Colors::colorchoice c) {
  double ylo, yhi;
  if (y1 < y2) {
    ylo = y1; yhi = y2;
  } else {
    ylo = y2; yhi = y1;
  }
  int ylo2, yhi2;
  ylo2 = (int) ceil (ylo);
  yhi2 = (int) floor (yhi);
  if (ylo2 < yhi2)
    drawVerticalLine (x, ylo2, yhi2-1, c);
  // What if they both fall within the same pixel:  ylo2 > yhi2
  if (ylo2 > yhi2) {
    assert (yhi2 == ylo2 - 1);
    double saturation = yhi - ylo;
    setPixel (x, yhi2, c, saturation);
  } else {
    // The normal case.
    if (ylo < (double)ylo2) {
      double saturation = (double)ylo2 - ylo;
      setPixel (x, ylo2-1, c, saturation);
    }
    if (yhi > (double)yhi2) {
      double saturation = yhi - (double)yhi2;
      setPixel (x, yhi2, c, saturation);
    }
  }
}

void
Graph::drawHorizontalLine (int xlo, int xhi, int y,
			      Colors::colorchoice c) {
  int i;
  for (i=xlo; i<=xhi; i++)
    setPixel (i, y, c);
}

unsigned
Graph::startpos() {
  return nowposition;
}

void
Graph::labelHourTick (int x, Dstr &ts) {
  centerStringOnLine (x, -1, ts);
}

void Graph::drawTimestampLabel (int x, int line, Dstr &ts) {
  centerStringOnLine (x, line, ts);
}

void Graph::drawTitleLine (Dstr &title) {
  centerStringOnLine (myxsize/2, 0, title);
}

int Graph::is_banner () {
  return 0;
}
